# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Configuration management for dockpycli.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

import os
from pathlib import Path
from typing import Literal

import tomli
from pydantic import BaseModel, Field, field_validator

from dockpycore.logging import get_logger


__all__ = ["CLIConfig", "get_config_path", "load_config"]

logger = get_logger(__name__)

OutputFormat = Literal["table", "json", "yaml", "plain"]
LogLevel = Literal["DEBUG", "INFO", "WARNING", "ERROR", "CRITICAL"]


class CLIConfig(BaseModel):
    """CLI configuration with validation.

    Configuration is loaded from multiple sources with precedence:
    1. CLI flags (highest priority)
    2. Environment variables (DOCKPY_*)
    3. Config file (~/.dockpycli/config.toml) (lowest priority)

    Attributes:
        output_format: Output format (table, json, yaml, plain)
        no_color: Disable colored output
        debug: Enable debug logging
        docker_socket: Path to Docker Unix socket
        api_version: Docker API version
        timeout: Request timeout in seconds
        log_level: Logging level
        config_file: Path to config file (for reference)
    """

    output_format: OutputFormat = Field(default="table", description="Output format")
    no_color: bool = Field(default=False, description="Disable colored output")
    debug: bool = Field(default=False, description="Enable debug logging")
    docker_socket: str = Field(
        default="/var/run/docker.sock",
        description="Path to Docker Unix socket",
    )
    api_version: str = Field(default="v1.43", description="Docker API version")
    timeout: float = Field(default=60.0, description="Request timeout in seconds")
    log_level: LogLevel = Field(default="INFO", description="Logging level")
    config_file: str | None = Field(default=None, description="Path to config file")

    @field_validator("timeout")
    @classmethod
    def validate_timeout(cls, v: float) -> float:
        """Validate timeout is positive."""
        if v <= 0:
            msg = "Timeout must be positive"
            raise ValueError(msg)
        return v

    @field_validator("docker_socket")
    @classmethod
    def validate_socket_path(cls, v: str) -> str:
        """Validate socket path is not empty."""
        if not v:
            msg = "Docker socket path cannot be empty"
            raise ValueError(msg)
        return v

    model_config = {
        "frozen": True,
        "extra": "forbid",
    }


def get_config_path() -> Path:
    """Get default config file path.

    Returns:
        Path to ~/.dockpycli/config.toml
    """
    home = Path.home()
    config_dir = home / ".dockpycli"
    return config_dir / "config.toml"


def load_config(
    config_file: str | None = None,
    output_format: str | None = None,
    no_color: bool | None = None,
    debug: bool | None = None,
    docker_socket: str | None = None,
    api_version: str | None = None,
    timeout: float | None = None,
    log_level: str | None = None,
) -> CLIConfig:
    """Load configuration from multiple sources.

    Precedence order:
    1. CLI flags (function arguments)
    2. Environment variables (DOCKPY_*)
    3. Config file (~/.dockpycli/config.toml)
    4. Defaults

    Args:
        config_file: Path to config file (overrides default)
        output_format: Output format (overrides all)
        no_color: Disable colors (overrides all)
        debug: Enable debug (overrides all)
        docker_socket: Docker socket path (overrides all)
        api_version: API version (overrides all)
        timeout: Request timeout (overrides all)
        log_level: Log level (overrides all)

    Returns:
        CLIConfig instance with merged configuration
    """
    # Start with defaults
    config_dict: dict[str, str | bool | float | None] = {}

    # Load from config file (lowest priority)
    file_path = Path(config_file) if config_file else get_config_path()
    if file_path.exists():
        try:
            with file_path.open("rb") as f:
                file_config = tomli.load(f)
                config_dict.update(file_config.get("cli", {}))
                logger.debug("config_loaded_from_file", path=str(file_path), config_dict=config_dict)
        except Exception as e:
            logger.warning("config_file_load_failed", path=str(file_path), error=str(e))

    # Load from environment variables (medium priority)
    env_config = {
        "output_format": os.getenv("DOCKPY_OUTPUT_FORMAT"),
        "no_color": os.getenv("DOCKPY_NO_COLOR", "").lower() in ("1", "true", "yes") or None,
        "debug": os.getenv("DOCKPY_DEBUG", "").lower() in ("1", "true", "yes") or None,
        "docker_socket": os.getenv("DOCKPY_DOCKER_SOCKET"),
        "api_version": os.getenv("DOCKPY_API_VERSION"),
        "timeout": os.getenv("DOCKPY_TIMEOUT"),
        "log_level": os.getenv("DOCKPY_LOG_LEVEL"),
    }

    # Update with env vars (only non-None values)
    for key, value in env_config.items():
        logger.debug("env_config", key=key, value=value)
        if value is not None and value != "":
            if key == "timeout":
                try:
                    config_dict[key] = float(value)
                except ValueError:
                    logger.warning("invalid_timeout_env", value=value)
            else:
                config_dict[key] = value

    # Override with CLI flags (highest priority)
    if output_format is not None:
        config_dict["output_format"] = output_format
    if no_color is not None:
        config_dict["no_color"] = no_color
    if debug is not None:
        config_dict["debug"] = debug
    if docker_socket is not None:
        config_dict["docker_socket"] = docker_socket
    if api_version is not None:
        config_dict["api_version"] = api_version
    if timeout is not None:
        config_dict["timeout"] = timeout
    if log_level is not None:
        config_dict["log_level"] = log_level

    # Set config_file path for reference
    config_dict["config_file"] = str(file_path) if file_path.exists() else None

    # Create config instance
    try:
        logger.debug("config_dict", config_dict=config_dict)
        config = CLIConfig(**config_dict)
        logger.debug("config_loaded", config=config.model_dump())
        return config
    except Exception as e:
        logger.error("config_validation_failed", error=str(e), config=config_dict)
        # Return default config on validation failure
        return CLIConfig()
