# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Debug CLI entry point for dpdbg command.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

import sys

import click
from rich.console import Console

from dockpycli.config import load_config
from dockpycore.logging import configure_logging, get_logger


__all__ = ["debug_cli"]

logger = get_logger(__name__)


# Debug Click app
debug_cli = click.Group(name="dpdbg", help="DockPy Docker CLI (Debug Mode)")


@debug_cli.command()
def info() -> None:
    """Show debug information."""
    config = load_config(debug=True)
    configure_logging(level="DEBUG", format="human")

    console = Console()
    console.print("[bold]Debug Information[/bold]")
    console.print(f"Config: {config.model_dump()}")


@debug_cli.command()
def ping() -> None:
    """Test Docker daemon connection."""
    import asyncio

    from dockpysdk import AsyncDockerClient

    async def _ping() -> None:
        config = load_config(debug=True)
        configure_logging(level="DEBUG", format="human")

        async with AsyncDockerClient(
            socket_path=config.docker_socket,
            api_version=config.api_version,
            timeout=config.timeout,
        ) as client:
            result = await client.ping()
            console = Console()
            if result:
                console.print("[green]✓ Docker daemon is accessible[/green]")
            else:
                console.print("[red]✗ Docker daemon is not accessible[/red]")
                sys.exit(1)

    asyncio.run(_ping())


def main() -> None:
    """Main entry point for dpdbg command."""
    try:
        debug_cli()
    except KeyboardInterrupt:
        console = Console()
        console.print("\n[yellow]Interrupted by user[/yellow]")
        sys.exit(130)
    except Exception as e:
        console = Console()
        console.print(f"[red]Error:[/red] {e}", style="red")
        if "--debug" in sys.argv:
            import traceback

            console.print(traceback.format_exc())
        sys.exit(1)


if __name__ == "__main__":
    main()
