# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""CLI-specific exceptions for user-facing error messages.

These exceptions should NOT be used in SDK or core packages.
They are only for CLI command error handling.

Author: A M (am@bbdevs.com)

Created At: 18 Nov 2025
"""

from __future__ import annotations

from dockpycore.exceptions import DockerError


__all__ = [
    "CLIError",
    "CommandNotFoundError",
    "InvalidArgumentError",
    "OutputFormatError",
]


class CLIError(DockerError):
    """Base exception for CLI-specific errors with exit codes.

    This is the base class for all CLI-specific exceptions.
    These exceptions provide user-friendly error messages and exit codes.
    """

    def __init__(self, message: str, exit_code: int = 1) -> None:
        """Initialize CLI error.

        Args:
            message: User-friendly error message
            exit_code: Exit code for the CLI process (default: 1)
        """
        super().__init__(message)
        self.exit_code = exit_code

    def format_for_user(self) -> str:
        """Format error message for user display.

        Returns:
            Formatted error message with "Error:" prefix
        """
        return f"Error: {self.message}"


class CommandNotFoundError(CLIError):
    """Raised when a CLI command is not found.

    This exception is raised when the user tries to execute
    a command that doesn't exist.
    """

    def __init__(self, command: str) -> None:
        """Initialize command not found error.

        Args:
            command: Name of the command that was not found
        """
        super().__init__(f"Unknown command: {command}", exit_code=2)
        self.command = command


class InvalidArgumentError(CLIError):
    """Raised when CLI argument is invalid.

    This exception is raised when a user provides an invalid
    argument to a CLI command.
    """

    def __init__(self, argument: str, reason: str) -> None:
        """Initialize invalid argument error.

        Args:
            argument: Name of the invalid argument
            reason: Explanation of why the argument is invalid
        """
        super().__init__(f"Invalid argument '{argument}': {reason}", exit_code=2)
        self.argument = argument
        self.reason = reason


class OutputFormatError(CLIError):
    """Raised when output format is not supported.

    This exception is raised when the user specifies an output
    format that is not supported by the CLI.
    """

    def __init__(self, format_name: str) -> None:
        """Initialize output format error.

        Args:
            format_name: Name of the unsupported format
        """
        super().__init__(f"Unsupported output format: {format_name}", exit_code=2)
        self.format_name = format_name
