# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Main CLI entry point for dp command.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

import sys
from pathlib import Path

import click
from rich.console import Console

from dockpycli.config import CLIConfig, load_config
from dockpycore.logging import configure_logging, get_logger


__all__ = ["cli"]

# Configure logging early with INFO level (will be reconfigured if --debug is used)
configure_logging(level="INFO", format="human")

logger = get_logger(__name__)


# Global options context
class GlobalContext:
    """Global context for CLI options."""

    def __init__(self) -> None:
        """Initialize global context."""
        self.config: CLIConfig | None = None


pass_global_context = click.make_pass_decorator(GlobalContext, ensure=True)


# Global Click app with context
@click.group(name="dp", help="DockPy Docker CLI")
@click.option(
    "--debug",
    is_flag=True,
    default=None,
    help="Enable debug logging",
)
@click.option(
    "--format",
    "output_format",
    type=click.Choice(["table", "json", "yaml", "plain"], case_sensitive=False),
    default=None,
    help="Output format (table, json, yaml, plain)",
)
@click.option(
    "--no-color",
    is_flag=True,
    default=None,
    help="Disable colored output",
)
@click.option(
    "--config",
    "config_file",
    type=click.Path(exists=True, path_type=Path),
    default=None,
    help="Path to config file",
)
@click.pass_context
def cli(
    ctx: click.Context,
    debug: bool | None,
    output_format: str | None,
    no_color: bool | None,
    config_file: Path | None,
) -> None:
    """DockPy Docker CLI - Main entry point."""
    # Load configuration
    config = load_config(
        config_file=str(config_file) if config_file else None,
        output_format=output_format,
        no_color=no_color,
        debug=debug,
    )

    # Reconfigure logging with proper level if debug is enabled
    if config.debug:
        configure_logging(level="DEBUG", format="human")

    # Store config in context
    ctx.ensure_object(GlobalContext)
    ctx.obj.config = config


@cli.command()
def version() -> None:
    """Show version information."""
    from dockpycli import __version__  # noqa: PLC0415

    console = Console()
    console.print(f"dockpycli version {__version__}")


@cli.group()
def container() -> None:
    """Container management commands."""
    pass


# Import and register container commands
# These imports are intentionally placed here to avoid circular dependencies
from dockpycli.commands.container.cli import (  # noqa: E402
    container_logs,
    container_port,
    container_prune,
    container_stats,
    container_top,
    create_container,
    exec_container,
    inspect_container,
    list_containers,
    pause_container,
    remove_container,
    rename_container,
    restart_container,
    run_container,
    start_container,
    stop_container,
    unpause_container,
    wait_container,
)


# Register container commands
container.add_command(list_containers, name="ls")
container.add_command(list_containers, name="list")
container.add_command(create_container, name="create")
container.add_command(run_container, name="run")
container.add_command(start_container, name="start")
container.add_command(stop_container, name="stop")
container.add_command(restart_container, name="restart")
container.add_command(pause_container, name="pause")
container.add_command(unpause_container, name="unpause")
container.add_command(remove_container, name="rm")
container.add_command(remove_container, name="remove")
container.add_command(inspect_container, name="inspect")
container.add_command(container_logs, name="logs")
container.add_command(container_port, name="port")
container.add_command(container_prune, name="prune")
container.add_command(container_stats, name="stats")
container.add_command(container_top, name="top")
container.add_command(exec_container, name="exec")
container.add_command(wait_container, name="wait")
container.add_command(rename_container, name="rename")


@cli.group()
def image() -> None:
    """Image management commands."""
    pass


# Import and register image commands
# These imports are intentionally placed here to avoid circular dependencies
from dockpycli.commands.image.cli import (  # noqa: E402
    build_image,
    image_history,
    inspect_image,
    list_images,
    prune_images,
    pull_image,
    push_image,
    remove_image,
    search_images,
    tag_image,
)


# Register image commands
image.add_command(list_images, name="ls")
image.add_command(list_images, name="list")
image.add_command(pull_image, name="pull")
image.add_command(push_image, name="push")
image.add_command(tag_image, name="tag")
image.add_command(remove_image, name="rm")
image.add_command(remove_image, name="remove")
image.add_command(inspect_image, name="inspect")
image.add_command(image_history, name="history")
image.add_command(build_image, name="build")
image.add_command(prune_images, name="prune")
image.add_command(search_images, name="search")


@cli.group()
def system() -> None:
    """System information commands."""
    pass


# Import and register system commands
# These imports are intentionally placed here to avoid circular dependencies
from dockpycli.commands.system.cli import (  # noqa: E402
    system_df,
    system_info,
    system_ping,
    system_version,
)


# Register system commands
system.add_command(system_info, name="info")
system.add_command(system_df, name="df")
system.add_command(system_version, name="version")
system.add_command(system_ping, name="ping")


@cli.group()
def compose() -> None:
    """Docker Compose commands."""
    pass


# Import and register compose commands
# These imports are intentionally placed here to avoid circular dependencies
from dockpycli.commands.compose.cli import (  # noqa: E402
    compose_build,
    compose_config,
    compose_down,
    compose_kill,
    compose_logs,
    compose_ls,
    compose_pause,
    compose_ps,
    compose_pull,
    compose_restart,
    compose_rm,
    compose_start,
    compose_stop,
    compose_unpause,
    compose_up,
)


# Register compose commands
compose.add_command(compose_up, name="up")
compose.add_command(compose_down, name="down")
compose.add_command(compose_ps, name="ps")
compose.add_command(compose_ls, name="ls")
compose.add_command(compose_ls, name="list")
compose.add_command(compose_logs, name="logs")
compose.add_command(compose_config, name="config")
compose.add_command(compose_build, name="build")
compose.add_command(compose_start, name="start")
compose.add_command(compose_stop, name="stop")
compose.add_command(compose_restart, name="restart")
compose.add_command(compose_pull, name="pull")
compose.add_command(compose_pause, name="pause")
compose.add_command(compose_unpause, name="unpause")
compose.add_command(compose_kill, name="kill")
compose.add_command(compose_rm, name="rm")
compose.add_command(compose_rm, name="remove")


# Add command groups to main CLI
cli.add_command(container)
cli.add_command(image)
cli.add_command(system)
cli.add_command(compose)


def main() -> None:
    """Main entry point for dp command."""
    try:
        cli()
    except KeyboardInterrupt:
        console = Console()
        console.print("\n[yellow]Interrupted by user[/yellow]")
        sys.exit(130)
    except Exception as e:
        console = Console()
        console.print(f"[red]Error:[/red] {e}", style="red")
        sys.exit(1)


if __name__ == "__main__":
    main()
