# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Output formatters for compose commands.

Author: A M (am@bbdevs.com)
Created At: 08 Nov 2025
"""

from __future__ import annotations

from typing import Any

from rich.table import Table

from dockpycli.output.styles import get_status_style


__all__ = [
    "compose_service_to_dict",
    "format_compose_config",
    "format_compose_logs",
    "format_compose_ps_table",
]

logger = None  # Will be set if needed


def compose_service_to_dict(service: dict[str, Any]) -> dict[str, Any]:
    """Convert compose service to dictionary.

    Args:
        service: Service information dict

    Returns:
        Dictionary representation
    """
    return {
        "name": service.get("name", ""),
        "status": service.get("status", ""),
        "ports": service.get("ports", ""),
        "image": service.get("image", ""),
        "command": service.get("command", ""),
    }


def format_compose_ps_table(services: list[dict[str, Any]]) -> Table:
    """Create Rich table for compose ps (matching docker compose ps format).

    Args:
        services: List of service information dictionaries

    Returns:
        Rich Table instance
    """
    table = Table(
        show_header=True,
        header_style="bold magenta",
        box=None,
        padding=(0, 1),
    )

    table.add_column("NAME", style="blue")
    table.add_column("COMMAND", style="cyan")
    table.add_column("SERVICE", style="green")
    table.add_column("STATUS", style="yellow")
    table.add_column("PORTS", style="white")

    for service in services:
        name = service.get("name", "")
        command = service.get("command", "")
        service_name = service.get("service", "")
        status = service.get("status", "")
        ports = service.get("ports", "")

        # Get status style
        status_style = get_status_style(status)

        table.add_row(
            name,
            command,
            service_name,
            f"[{status_style}]{status}[/]",
            ports,
        )

    return table


def format_compose_logs(logs: list[dict[str, Any]], service_names: list[str] | None = None) -> str:
    """Format compose logs with service names.

    Args:
        logs: List of log entries with service, stream, message
        service_names: List of service names to filter (None = all)

    Returns:
        Formatted log string
    """
    if not logs:
        return ""

    formatted_lines: list[str] = []

    for entry in logs:
        service = entry.get("service", "")
        stream = entry.get("stream", "stdout")
        message = entry.get("message", "")

        # Filter by service if requested
        if service_names and service not in service_names:
            continue

        # Format: service_name | message
        prefix = f"[{service}]" if service else ""
        formatted_lines.append(f"{prefix} {message}")

    return "\n".join(formatted_lines)


def format_compose_config(config: dict[str, Any]) -> str:
    """Format compose config for display.

    Args:
        config: Merged compose configuration dictionary

    Returns:
        Formatted YAML string
    """
    from io import StringIO

    from ruyaml import YAML

    yaml = YAML()
    yaml.default_flow_style = False
    yaml.sort_keys = False

    stream = StringIO()
    yaml.dump(config, stream)
    return stream.getvalue()
