# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Container-specific output formatting.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

from typing import Any

from rich.table import Table

from dockpycli.output.styles import get_status_style
from dockpycli.utils.helpers import format_size, format_time, truncate_id


# Import helpers locally to avoid circular imports
def _get_format_ports():
    """Get format_ports function to avoid circular import."""
    from dockpycli.commands.container.helpers import format_ports  # noqa: PLC0415

    return format_ports


def _get_format_command():
    """Get format_command function to avoid circular import."""
    from dockpycli.commands.container.helpers import format_command  # noqa: PLC0415

    return format_command


__all__ = [
    "container_to_dict",
    "format_container_plain",
    "format_container_stats_table",
    "format_container_table",
    "format_port_table",
    "format_top_table",
]


def container_to_dict(container: Any) -> dict[str, Any]:  # Container type
    """Convert Container model to dictionary for formatting.

    Args:
        container: Container model instance

    Returns:
        Dictionary representation
    """
    return {
        "Id": container.id,
        "Image": container.image,
        "Command": _get_format_command()(None),  # Command not in Container model
        "Created": format_time(container.created) if container.created else "",
        "Status": container.status,
        "Ports": _get_format_ports()(container.ports),
        "Names": container.name,
    }


def format_container_plain(containers: list[Any]) -> str:  # list[Container]
    """Format containers as plain tab-separated text.

    Args:
        containers: List of Container model instances

    Returns:
        Plain text string with tab-separated values
    """
    from dockpycli.output.formatters import format_plain_table  # noqa: PLC0415

    headers = ["CONTAINER ID", "IMAGE", "COMMAND", "CREATED", "STATUS", "PORTS", "NAMES"]
    rows = []

    for container in containers:
        container_id = truncate_id(container.id, 12)
        image = container.image or ""
        command = container.command or ""
        created = format_time(container.created) if container.created else ""
        status = container.status or ""
        ports = _get_format_ports()(container.ports)
        names = container.name or ""

        rows.append([container_id, image, command, created, status, ports, names])

    return format_plain_table(headers, rows)


def format_container_table(containers: list[Any], no_trunc: bool = False) -> Table:  # list[Container]
    """Create Rich table for containers (matching docker ps format).

    Args:
        containers: List of Container model instances
        no_trunc: If True, don't truncate IDs and names

    Returns:
        Rich Table instance
    """
    table = Table(
        show_header=True,
        header_style="bold magenta",
        box=None,
        padding=(0, 1),
    )

    table.add_column("CONTAINER ID", style="cyan", no_wrap=True)
    table.add_column("IMAGE", style="blue")
    table.add_column("COMMAND", style="yellow")
    table.add_column("CREATED", style="white")
    table.add_column("STATUS", style="green")
    table.add_column("PORTS", style="white")
    table.add_column("NAMES", style="magenta")

    for container in containers:
        container_id = container.id if no_trunc else truncate_id(container.id, 12)
        image = container.image
        command = container.command
        created = format_time(container.created) if container.created else ""
        status = container.status
        ports = _get_format_ports()(container.ports)
        names = container.name

        # Get status style
        status_lower = status.lower()
        status_style = get_status_style(status_lower)

        table.add_row(
            container_id,
            image,
            command,
            created,
            f"[{status_style}]{status}[/]",
            ports,
            names,
        )

    return table


def format_container_stats_table(
    container_id: str,
    container_name: str,
    stats: Any,  # ContainerStats
) -> Table:
    """Create Rich table for container stats.

    Args:
        container_id: Container ID
        container_name: Container name
        stats: ContainerStats model instance

    Returns:
        Rich Table instance
    """
    table = Table(
        show_header=True,
        header_style="bold magenta",
        box=None,
        padding=(0, 1),
    )

    table.add_column("CONTAINER", style="cyan", no_wrap=True)
    table.add_column("CPU %", style="green")
    table.add_column("MEM USAGE / LIMIT", style="yellow")
    table.add_column("MEM %", style="yellow")
    table.add_column("NET I/O", style="blue")
    table.add_column("BLOCK I/O", style="magenta")
    table.add_column("PIDS", style="white")

    # Format memory
    mem_usage = format_size(stats.memory_usage)
    mem_limit = format_size(stats.memory_limit)
    mem_usage_limit = f"{mem_usage} / {mem_limit}"

    # Format network I/O
    net_rx = format_size(stats.network_rx_bytes)
    net_tx = format_size(stats.network_tx_bytes)
    net_io = f"{net_rx} / {net_tx}"

    # Format block I/O
    block_read = format_size(stats.block_read_bytes)
    block_write = format_size(stats.block_write_bytes)
    block_io = f"{block_read} / {block_write}"

    # Format CPU percentage
    cpu_pct = f"{stats.cpu_percent:.2f}%"

    # Format memory percentage
    mem_pct = f"{stats.memory_percent:.2f}%"

    container_display = truncate_id(container_id, 12)
    if container_name:
        container_display = f"{container_display} ({container_name})"

    table.add_row(
        container_display,
        cpu_pct,
        mem_usage_limit,
        mem_pct,
        net_io,
        block_io,
        str(stats.pids),
    )

    return table


def format_port_table(ports: dict[str, list[dict[str, Any]]]) -> Table:
    """Create Rich table for container port mappings.

    Args:
        ports: Port mappings from Container model

    Returns:
        Rich Table instance
    """
    table = Table(
        show_header=True,
        header_style="bold magenta",
        box=None,
        padding=(0, 1),
    )

    table.add_column("CONTAINER PORT", style="cyan", no_wrap=True)
    table.add_column("HOST IP", style="blue")
    table.add_column("HOST PORT", style="yellow")

    # Handle dict format: {"80/tcp": [{"HostPort": "8080", "HostIp": "0.0.0.0"}]}
    if isinstance(ports, dict):
        for container_port, bindings in ports.items():
            if bindings:
                for binding in bindings:
                    host_ip = binding.get("HostIp", "0.0.0.0")
                    host_port = binding.get("HostPort", "")
                    table.add_row(container_port, host_ip, host_port or "-")
            else:
                # Port exposed but not bound
                table.add_row(container_port, "-", "-")
    elif isinstance(ports, list):
        # Handle list format: [{"PrivatePort": 80, "PublicPort": 8080, "Type": "tcp"}]
        for port_info in ports:
            private_port = port_info.get("PrivatePort", "")
            public_port = port_info.get("PublicPort", "")
            port_type = port_info.get("Type", "tcp")
            host_ip = port_info.get("IP", "0.0.0.0")

            container_port = f"{private_port}/{port_type}" if private_port else "-"
            table.add_row(container_port, host_ip, str(public_port) if public_port else "-")

    if not table.rows:
        # No ports exposed
        table.add_row("-", "-", "-")

    return table


def format_top_table(titles: list[str], processes: list[list[str]]) -> Table:
    """Create Rich table for container top (running processes).

    Args:
        titles: Column titles from Docker API
        processes: List of process rows (each row is a list of values)

    Returns:
        Rich Table instance
    """
    table = Table(
        show_header=True,
        header_style="bold magenta",
        box=None,
        padding=(0, 1),
    )

    # Add columns based on titles
    for title in titles:
        table.add_column(title.upper(), style="cyan" if title == "PID" else "white")

    # Add process rows
    for process_row in processes:
        table.add_row(*process_row)

    if not table.rows:
        # No processes
        table.add_row("(no processes)")

    return table
