# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Output formatters for different formats.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

# import orjson as json
import json
from typing import Any, Protocol

from rich.table import Table
from ruyaml import YAML

from dockpycli.config import CLIConfig, OutputFormat
from dockpycli.output.console import get_console
from dockpycore.logging import get_logger


__all__ = [
    "JSONFormatter",
    "OutputFormatter",
    "PlainFormatter",
    "TableFormatter",
    "YAMLFormatter",
    "format_plain_table",
    "get_formatter",
]


def format_plain_table(headers: list[str], rows: list[list[str]]) -> str:
    """Format data as plain tab-separated table.

    Args:
        headers: Column headers
        rows: Data rows (each row is a list of strings)

    Returns:
        Tab-separated table string

    Example:
        >>> headers = ["ID", "NAME", "STATUS"]
        >>> rows = [["abc123", "nginx", "running"], ["def456", "redis", "running"]]
        >>> print(format_plain_table(headers, rows))
        ID      NAME    STATUS
        abc123  nginx   running
        def456  redis   running
    """
    lines = ["\t".join(headers)]
    for row in rows:
        lines.append("\t".join(str(cell) for cell in row))
    return "\n".join(lines)


class OutputFormatter(Protocol):
    """Protocol for output formatters.

    All formatters must implement this interface.
    """

    def __init__(self, config: CLIConfig) -> None:
        """Initialize output formatter.

        Args:
            config: CLI configuration
        """
        self.logger = get_logger(__name__)
        self.config = config
        self.console = get_console(config)

    def format(self, data: Any) -> str:
        """Format data for output.

        Args:
            data: Data to format (dict, list, etc.)

        Returns:
            Formatted string
        """
        ...

    def to_json(self, data: Any) -> str:
        """Convert data to JSON string.

        Args:
            data: Data to convert

        Returns:
            JSON string
        """
        data = json.dumps(data, indent=2, default=str)
        self.logger.debug("json_str", json_str=data)
        return data


class TableFormatter(OutputFormatter):
    """Rich table formatter (default)."""

    def __init__(self, config: CLIConfig) -> None:
        """Initialize table formatter.

        Args:
            config: CLI configuration
        """
        super().__init__(config)

    def format(self, data: Any) -> str:
        """Format data as Rich table.

        Args:
            data: Data to format (dict, list, Table, etc.)

        Returns:
            Empty string (table is printed directly)
        """
        if isinstance(data, Table):
            self.console.print(data)
            return ""
        if isinstance(data, (list, dict)):
            # For now, just print as JSON if not a Table
            # Phase 3.2+ will add proper table formatting
            json_str = self.to_json(data)
            self.console.print_json(json_str)
            return ""
        self.console.print(str(data))
        return ""


class JSONFormatter(OutputFormatter):
    """JSON formatter for machine-readable output."""

    def __init__(self, config: CLIConfig) -> None:
        """Initialize JSON formatter.

        Args:
            config: CLI configuration
        """
        super().__init__(config)

    def format(self, data: Any) -> str:
        """Format data as JSON.

        Args:
            data: Data to format

        Returns:
            JSON string
        """
        json_str = self.to_json(data)
        if not self.config.no_color:
            self.console.print_json(json_str)
        return json_str


class YAMLFormatter(OutputFormatter):
    """YAML formatter for human-readable structured output."""

    def __init__(self, config: CLIConfig) -> None:
        """Initialize YAML formatter.

        Args:
            config: CLI configuration
        """
        super().__init__(config)

        # Create YAML instance for dumping
        self._yaml = YAML()
        self._yaml.default_flow_style = False
        self._yaml.sort_keys = False

    def format(self, data: Any) -> str:
        """Format data as YAML.

        Args:
            data: Data to format

        Returns:
            YAML string
        """
        from io import StringIO  # noqa: PLC0415

        stream = StringIO()
        self._yaml.dump(data, stream)
        yaml_str = stream.getvalue()
        self.console.print(yaml_str)
        return yaml_str


class PlainFormatter(OutputFormatter):
    """Plain text formatter for piping and scripting.

    Outputs tab-separated values (TSV) format.
    """

    def __init__(self, config: CLIConfig) -> None:
        """Initialize plain formatter.

        Args:
            config: CLI configuration
        """
        super().__init__(config)

    def format(self, data: Any) -> str:
        """Format data as plain text (tab-separated).

        Args:
            data: Data to format (supports str, Table, list, dict)

        Returns:
            Plain text string
        """
        # If it's already a string (from plain formatter), just print it
        if isinstance(data, str):
            self.console.print(data)
            return data

        # If it's a Table, we can't extract data easily, fall back to JSON
        if isinstance(data, Table):
            # Convert to JSON as fallback
            return self.to_json(data)

        # If it's a list or dict, convert to JSON
        if isinstance(data, (list, dict)):
            json_str = self.to_json(data)
            self.console.print(json_str)
            return json_str

        # Default: convert to string
        result = str(data)
        self.console.print(result)
        return result


def get_formatter(config: CLIConfig) -> OutputFormatter:
    """Get formatter for output format.

    Args:
        config: CLI configuration

    Returns:
        OutputFormatter instance
    """
    format_map: dict[OutputFormat, type[OutputFormatter]] = {
        "table": TableFormatter,
        "json": JSONFormatter,
        "yaml": YAMLFormatter,
        "plain": PlainFormatter,
    }

    formatter_class = format_map.get(config.output_format, TableFormatter)
    return formatter_class(config)
