# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Image-specific output formatting.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

from typing import Any

from rich.table import Table

from dockpycli.commands.image.helpers import format_image_size
from dockpycli.utils.helpers import format_time, truncate_id


__all__ = [
    "format_image_history_table",
    "format_image_plain",
    "format_image_table",
    "image_to_dict",
]


def image_to_dict(image: Any) -> dict[str, Any]:  # Image type
    """Convert Image model to dictionary for formatting.

    Args:
        image: Image model instance

    Returns:
        Dictionary representation
    """
    # Parse repository and tag
    if image.repo_tags:
        repo_tag = image.repo_tags[0]
        if ":" in repo_tag:
            repository, tag = repo_tag.split(":", 1)
        else:
            repository = repo_tag
            tag = "<none>"
    else:
        repository = "<none>"
        tag = "<none>"

    return {
        "Id": image.id,
        "Repository": repository,
        "Tag": tag,
        "Created": format_time(image.created) if image.created else "",
        "Size": format_image_size(image.size),
        "VirtualSize": format_image_size(image.virtual_size),
    }


def format_image_plain(images: list[Any]) -> str:  # list[Image]
    """Format images as plain tab-separated text.

    Args:
        images: List of Image model instances

    Returns:
        Plain text string with tab-separated values
    """
    from dockpycli.output.formatters import format_plain_table  # noqa: PLC0415

    headers = ["REPOSITORY", "TAG", "IMAGE ID", "CREATED", "SIZE"]
    rows = []

    for image in images:
        # Parse repository and tag
        if image.repo_tags:
            repo_tag = image.repo_tags[0]
            if ":" in repo_tag:
                repository, tag = repo_tag.split(":", 1)
            else:
                repository = repo_tag
                tag = "<none>"
        else:
            repository = "<none>"
            tag = "<none>"

        # Format image ID (short version)
        image_id = truncate_id(image.id, 12)

        # Format created time
        created = format_time(image.created) if image.created else ""

        # Format size
        size = format_image_size(image.size)

        rows.append([repository, tag, image_id, created, size])

    return format_plain_table(headers, rows)


def format_image_table(images: list[Any], no_trunc: bool = False) -> Table:  # list[Image]
    """Create Rich table for images (matching docker images format).

    Args:
        images: List of Image model instances
        no_trunc: Don't truncate output

    Returns:
        Rich Table instance
    """
    table = Table(
        show_header=True,
        header_style="bold magenta",
        box=None,
        padding=(0, 1),
    )

    table.add_column("REPOSITORY", style="blue")
    table.add_column("TAG", style="yellow")
    table.add_column("IMAGE ID", style="cyan", no_wrap=True)
    table.add_column("CREATED", style="white")
    table.add_column("SIZE", style="green")

    for image in images:
        # Parse repository and tag
        if image.repo_tags:
            repo_tag = image.repo_tags[0]
            if ":" in repo_tag:
                repository, tag = repo_tag.split(":", 1)
            else:
                repository = repo_tag
                tag = "<none>"
        else:
            repository = "<none>"
            tag = "<none>"

        # Format image ID
        image_id = image.id if no_trunc else truncate_id(image.id, 12)

        # Format created time
        created = format_time(image.created) if image.created else ""

        # Format size
        size = format_image_size(image.size)

        table.add_row(
            repository,
            tag,
            image_id,
            created,
            size,
        )

    return table


def format_image_history_table(history: list[dict[str, Any]], no_trunc: bool = False, quiet: bool = False) -> Table:
    """Create Rich table for image history.

    Args:
        history: List of history entries from Docker API
        no_trunc: Don't truncate output
        quiet: Only show image IDs

    Returns:
        Rich Table instance
    """
    table = Table(
        show_header=True,
        header_style="bold magenta",
        box=None,
        padding=(0, 1),
    )

    if quiet:
        table.add_column("IMAGE", style="cyan", no_wrap=True)
    else:
        table.add_column("IMAGE", style="cyan", no_wrap=True)
        table.add_column("CREATED", style="white")
        table.add_column("CREATED BY", style="yellow")
        table.add_column("SIZE", style="green")
        table.add_column("COMMENT", style="blue")

    for entry in history:
        image_id = entry.get("Id", "")
        if not no_trunc and image_id:
            image_id = truncate_id(image_id, 12)

        if quiet:
            table.add_row(image_id)
        else:
            created = format_time(entry.get("Created", 0)) if entry.get("Created") else ""
            created_by = entry.get("CreatedBy", "")
            if not no_trunc and len(created_by) > 50:
                created_by = created_by[:47] + "..."
            size = format_image_size(entry.get("Size", 0))
            comment = entry.get("Comment", "")

            table.add_row(
                image_id,
                created,
                created_by,
                size,
                comment,
            )

    return table
