# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""System-specific output formatting.

Author: A M (am@bbdevs.com)

Created At: 21 Nov 2025
"""

from __future__ import annotations

from typing import Any

from rich.table import Table

from dockpycli.output.formatters import format_plain_table
from dockpycli.utils.helpers import format_size


__all__ = [
    "format_system_df_plain",
    "format_system_df_table",
    "format_system_info_plain",
    "format_system_info_table",
    "format_system_version_plain",
    "format_system_version_table",
]


def format_system_info_table(info: dict[str, Any]) -> Table:
    """Create Rich table for system info.

    Args:
        info: System information dictionary from Docker API

    Returns:
        Rich Table instance
    """
    table = Table(title="Docker System Information", show_header=True, header_style="bold magenta")

    table.add_column("Property", style="cyan", no_wrap=True)
    table.add_column("Value", style="green")

    # Core information
    table.add_row("Containers", str(info.get("Containers", 0)))
    table.add_row("Running", str(info.get("ContainersRunning", 0)))
    table.add_row("Paused", str(info.get("ContainersPaused", 0)))
    table.add_row("Stopped", str(info.get("ContainersStopped", 0)))
    table.add_row("Images", str(info.get("Images", 0)))
    table.add_row("Driver", str(info.get("Driver", "unknown")))
    table.add_row("Operating System", str(info.get("OperatingSystem", "unknown")))
    table.add_row("Architecture", str(info.get("Architecture", "unknown")))
    table.add_row("Kernel Version", str(info.get("KernelVersion", "unknown")))
    table.add_row("Docker Root Dir", str(info.get("DockerRootDir", "unknown")))

    # Memory and CPU
    if "MemTotal" in info:
        table.add_row("Total Memory", format_size(info["MemTotal"]))
    if "NCPU" in info:
        table.add_row("CPUs", str(info["NCPU"]))

    return table


def format_system_info_plain(info: dict[str, Any]) -> str:
    """Format system info as plain tab-separated text.

    Args:
        info: System information dictionary from Docker API

    Returns:
        Plain text string with tab-separated values
    """
    headers = ["PROPERTY", "VALUE"]
    rows = [
        ["Containers", str(info.get("Containers", 0))],
        ["Running", str(info.get("ContainersRunning", 0))],
        ["Paused", str(info.get("ContainersPaused", 0))],
        ["Stopped", str(info.get("ContainersStopped", 0))],
        ["Images", str(info.get("Images", 0))],
        ["Driver", str(info.get("Driver", "unknown"))],
        ["Operating System", str(info.get("OperatingSystem", "unknown"))],
        ["Architecture", str(info.get("Architecture", "unknown"))],
        ["Kernel Version", str(info.get("KernelVersion", "unknown"))],
        ["Docker Root Dir", str(info.get("DockerRootDir", "unknown"))],
    ]

    if "MemTotal" in info:
        rows.append(["Total Memory", format_size(info["MemTotal"])])
    if "NCPU" in info:
        rows.append(["CPUs", str(info["NCPU"])])

    return format_plain_table(headers, rows)


def format_system_df_table(df_data: dict[str, Any]) -> Table:
    """Create Rich table for disk usage (docker df format).

    Args:
        df_data: Disk usage dictionary from Docker API

    Returns:
        Rich Table instance
    """
    table = Table(title="Docker Disk Usage", show_header=True, header_style="bold magenta")

    table.add_column("Type", style="cyan", no_wrap=True)
    table.add_column("Total Count", style="green", justify="right")
    table.add_column("Size", style="yellow", justify="right")
    table.add_column("Reclaimable", style="blue", justify="right")

    # Images
    images = df_data.get("Images", [])
    images_size = sum(img.get("Size", 0) for img in images)
    images_shared_size = sum(img.get("SharedSize", 0) for img in images)
    images_reclaimable = images_size - images_shared_size
    table.add_row("Images", str(len(images)), format_size(images_size), format_size(images_reclaimable))

    # Containers
    containers = df_data.get("Containers", [])
    containers_size = sum(ctr.get("Size", 0) for ctr in containers)
    table.add_row("Containers", str(len(containers)), format_size(containers_size), format_size(containers_size))

    # Volumes
    volumes = df_data.get("Volumes", [])
    volumes_size = sum(vol.get("Size", 0) for vol in volumes)
    table.add_row("Volumes", str(len(volumes)), format_size(volumes_size), format_size(volumes_size))

    # Build Cache
    build_cache = df_data.get("BuildCache", [])
    build_cache_size = sum(cache.get("Size", 0) for cache in build_cache)
    table.add_row("Build Cache", str(len(build_cache)), format_size(build_cache_size), format_size(build_cache_size))

    # Total
    total_size = images_size + containers_size + volumes_size + build_cache_size
    total_reclaimable = images_reclaimable + containers_size + volumes_size + build_cache_size
    table.add_row("TOTAL", "-", format_size(total_size), format_size(total_reclaimable), style="bold")

    return table


def format_system_df_plain(df_data: dict[str, Any]) -> str:
    """Format disk usage as plain tab-separated text.

    Args:
        df_data: Disk usage dictionary from Docker API

    Returns:
        Plain text string with tab-separated values
    """
    headers = ["TYPE", "TOTAL COUNT", "SIZE", "RECLAIMABLE"]

    images = df_data.get("Images", [])
    images_size = sum(img.get("Size", 0) for img in images)
    images_shared_size = sum(img.get("SharedSize", 0) for img in images)
    images_reclaimable = images_size - images_shared_size

    containers = df_data.get("Containers", [])
    containers_size = sum(ctr.get("Size", 0) for ctr in containers)

    volumes = df_data.get("Volumes", [])
    volumes_size = sum(vol.get("Size", 0) for vol in volumes)

    build_cache = df_data.get("BuildCache", [])
    build_cache_size = sum(cache.get("Size", 0) for cache in build_cache)

    total_size = images_size + containers_size + volumes_size + build_cache_size
    total_reclaimable = images_reclaimable + containers_size + volumes_size + build_cache_size

    rows = [
        ["Images", str(len(images)), format_size(images_size), format_size(images_reclaimable)],
        ["Containers", str(len(containers)), format_size(containers_size), format_size(containers_size)],
        ["Volumes", str(len(volumes)), format_size(volumes_size), format_size(volumes_size)],
        ["Build Cache", str(len(build_cache)), format_size(build_cache_size), format_size(build_cache_size)],
        ["TOTAL", "-", format_size(total_size), format_size(total_reclaimable)],
    ]

    return format_plain_table(headers, rows)


def format_system_version_table(version: dict[str, Any]) -> Table:
    """Create Rich table for version information.

    Args:
        version: Version information dictionary from Docker API

    Returns:
        Rich Table instance
    """
    table = Table(title="Docker Version Information", show_header=True, header_style="bold magenta")

    table.add_column("Property", style="cyan", no_wrap=True)
    table.add_column("Value", style="green")

    table.add_row("Version", str(version.get("Version", "unknown")))
    table.add_row("API Version", str(version.get("ApiVersion", "unknown")))
    table.add_row("Min API Version", str(version.get("MinAPIVersion", "unknown")))
    table.add_row("Git Commit", str(version.get("GitCommit", "unknown")))
    table.add_row("Go Version", str(version.get("GoVersion", "unknown")))
    table.add_row("OS/Arch", str(version.get("Os", "unknown")) + "/" + str(version.get("Arch", "unknown")))
    table.add_row("Build Time", str(version.get("BuildTime", "unknown")))

    if "KernelVersion" in version:
        table.add_row("Kernel Version", str(version["KernelVersion"]))
    if "Experimental" in version:
        table.add_row("Experimental", str(version["Experimental"]))

    return table


def format_system_version_plain(version: dict[str, Any]) -> str:
    """Format version information as plain tab-separated text.

    Args:
        version: Version information dictionary from Docker API

    Returns:
        Plain text string with tab-separated values
    """
    headers = ["PROPERTY", "VALUE"]
    rows = [
        ["Version", str(version.get("Version", "unknown"))],
        ["API Version", str(version.get("ApiVersion", "unknown"))],
        ["Min API Version", str(version.get("MinAPIVersion", "unknown"))],
        ["Git Commit", str(version.get("GitCommit", "unknown"))],
        ["Go Version", str(version.get("GoVersion", "unknown"))],
        ["OS/Arch", str(version.get("Os", "unknown")) + "/" + str(version.get("Arch", "unknown"))],
        ["Build Time", str(version.get("BuildTime", "unknown"))],
    ]

    if "KernelVersion" in version:
        rows.append(["Kernel Version", str(version["KernelVersion"])])
    if "Experimental" in version:
        rows.append(["Experimental", str(version["Experimental"])])

    return format_plain_table(headers, rows)
