# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Rich table rendering matching Docker CLI format.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

from typing import Any

from rich.table import Table

from dockpycli.output.styles import get_status_style


__all__ = ["create_container_table", "create_image_table"]


def create_container_table(containers: list[dict[str, Any]]) -> Table:
    """Create Rich table for containers (matching docker ps format).

    Args:
        containers: List of container dictionaries

    Returns:
        Rich Table instance
    """
    table = Table(
        show_header=True,
        header_style="bold magenta",
        box=None,
        padding=(0, 1),
    )

    table.add_column("CONTAINER ID", style="cyan", no_wrap=True)
    table.add_column("IMAGE", style="blue")
    table.add_column("COMMAND", style="yellow")
    table.add_column("CREATED", style="white")
    table.add_column("STATUS", style="green")
    table.add_column("PORTS", style="white")
    table.add_column("NAMES", style="magenta")

    for container in containers:
        container_id = container.get("Id", "")[:12]
        image = container.get("Image", "")
        command = container.get("Command", "")
        created = container.get("Created", "")
        status = container.get("Status", "")
        ports = container.get("Ports", "")
        names = container.get("Names", "")

        # Get status style
        status_lower = status.lower()
        status_style = get_status_style(status_lower)

        table.add_row(
            container_id,
            image,
            command,
            created,
            f"[{status_style}]{status}[/]",
            ports,
            names,
        )

    return table


def create_image_table(images: list[dict[str, Any]]) -> Table:
    """Create Rich table for images (matching docker images format).

    Args:
        images: List of image dictionaries

    Returns:
        Rich Table instance
    """
    table = Table(
        show_header=True,
        header_style="bold magenta",
        box=None,
        padding=(0, 1),
    )

    table.add_column("REPOSITORY", style="blue")
    table.add_column("TAG", style="cyan")
    table.add_column("IMAGE ID", style="yellow", no_wrap=True)
    table.add_column("CREATED", style="white")
    table.add_column("SIZE", style="green")

    for image in images:
        repository = image.get("Repository", "")
        tag = image.get("Tag", "")
        image_id = image.get("Id", "")[:12]
        created = image.get("Created", "")
        size = image.get("Size", "")

        table.add_row(repository, tag, image_id, created, size)

    return table
