# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Async bridge utilities for Click commands.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

import asyncio
from collections.abc import Callable
from functools import wraps
from typing import Any, TypeVar


__all__ = ["async_command"]

T = TypeVar("T")


def async_command(func: Callable[..., Any]) -> Callable[..., Any]:
    """Decorator to bridge sync Click command to async function.

    This decorator wraps an async function and runs it with asyncio.run(),
    making it compatible with Click's sync command interface.

    Args:
        func: Async function to wrap

    Returns:
        Sync wrapper function

    Example:
        @click.command()
        @async_command
        async def list_containers():
            async with AsyncDockerClient() as client:
                containers = await client.containers.list()
                # format and display
    """
    if not asyncio.iscoroutinefunction(func):
        msg = "async_command decorator can only be used on async functions"
        raise TypeError(msg)

    @wraps(func)
    def wrapper(*args: Any, **kwargs: Any) -> Any:
        """Sync wrapper that runs async function."""
        try:
            return asyncio.run(func(*args, **kwargs))
        except KeyboardInterrupt:
            # Handle Ctrl+C gracefully
            return None
        except Exception:
            # Re-raise to let Click handle it
            raise

    return wrapper
