# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Automatic error message formatting with colors.

Author: A M (am@bbdevs.com)

Created At: 18 Nov 2025
"""

from __future__ import annotations


__all__ = ["format_error", "format_info", "format_warning"]


def format_error(message: str, details: str | None = None, hint: str | None = None) -> str:
    """Format error message with automatic coloring.

    Args:
        message: Main error message
        details: Optional details (shown in dim)
        hint: Optional hint for the user (shown in yellow)

    Returns:
        Formatted error string with Rich markup

    Example:
        >>> format_error("Container not found", details="ID: abc123")
        '[red]Error:[/red] Container not found\\n[dim]Details:[/dim] ID: abc123'
    """
    parts = [f"[red]Error:[/red] {message}"]

    if details:
        parts.append(f"[dim]Details:[/dim] {details}")

    if hint:
        parts.append(f"[yellow]Hint:[/yellow] {hint}")

    return "\n".join(parts)


def format_warning(message: str, details: str | None = None) -> str:
    """Format warning message with automatic coloring.

    Args:
        message: Main warning message
        details: Optional details (shown in dim)

    Returns:
        Formatted warning string with Rich markup

    Example:
        >>> format_warning("Deprecated option used")
        '[yellow]Warning:[/yellow] Deprecated option used'
    """
    parts = [f"[yellow]Warning:[/yellow] {message}"]

    if details:
        parts.append(f"[dim]Details:[/dim] {details}")

    return "\n".join(parts)


def format_info(message: str, details: str | None = None) -> str:
    """Format info message with automatic coloring.

    Args:
        message: Main info message
        details: Optional details (shown in dim)

    Returns:
        Formatted info string with Rich markup

    Example:
        >>> format_info("Operation completed successfully")
        '[green]Info:[/green] Operation completed successfully'
    """
    parts = [f"[green]Info:[/green] {message}"]

    if details:
        parts.append(f"[dim]Details:[/dim] {details}")

    return "\n".join(parts)
