# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Common utilities for CLI operations.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

from datetime import datetime


__all__ = ["format_size", "format_time", "truncate_id"]


def truncate_id(container_id: str, length: int = 12) -> str:
    """Truncate container/image ID to short format (Docker CLI convention).

    Strips "sha256:" prefix if present and returns first N characters.

    Args:
        container_id: Full container/image ID (may include "sha256:" prefix)
        length: Length to truncate to (default: 12)

    Returns:
        Truncated ID string without prefix

    Example:
        >>> truncate_id("sha256:abc123def456...", 12)
        "abc123def456"
        >>> truncate_id("abc123def456...", 12)
        "abc123def456"
    """
    if not container_id:
        return ""

    # Strip "sha256:" prefix if present
    if container_id.startswith("sha256:"):
        container_id = container_id[7:]

    return container_id[:length]


def format_size(size_bytes: int | float) -> str:
    """Format size in bytes to human-readable format.

    Args:
        size_bytes: Size in bytes

    Returns:
        Formatted size string (e.g., "1.5 MB", "500 KB")

    Example:
        >>> format_size(1024 * 1024)
        "1.0 MB"
        >>> format_size(500)
        "500 B"
    """
    if size_bytes < 1024:
        return f"{size_bytes} B"
    if size_bytes < 1024 * 1024:
        return f"{size_bytes / 1024:.1f} KB"
    if size_bytes < 1024 * 1024 * 1024:
        return f"{size_bytes / (1024 * 1024):.1f} MB"
    return f"{size_bytes / (1024 * 1024 * 1024):.1f} GB"


def format_time(timestamp: int | str | datetime) -> str:
    """Format timestamp to human-readable relative time.

    Args:
        timestamp: Unix timestamp, ISO string, or datetime object

    Returns:
        Formatted time string (e.g., "2 hours ago", "5 minutes ago")

    Example:
        >>> format_time(datetime.now() - timedelta(hours=2))
        "2 hours ago"
    """
    if isinstance(timestamp, str):
        try:
            dt = datetime.fromisoformat(timestamp.replace("Z", "+00:00"))
        except ValueError:
            return str(timestamp)
    elif isinstance(timestamp, int):
        dt = datetime.fromtimestamp(timestamp)
    elif isinstance(timestamp, datetime):
        dt = timestamp
    else:
        return str(timestamp)

    now = datetime.now(dt.tzinfo) if dt.tzinfo else datetime.now()
    delta = now - dt

    if delta.total_seconds() < 60:
        return "just now"
    if delta.total_seconds() < 3600:
        minutes = int(delta.total_seconds() / 60)
        return f"{minutes} minute{'s' if minutes != 1 else ''} ago"
    if delta.total_seconds() < 86400:
        hours = int(delta.total_seconds() / 3600)
        return f"{hours} hour{'s' if hours != 1 else ''} ago"
    days = int(delta.total_seconds() / 86400)
    return f"{days} day{'s' if days != 1 else ''} ago"
