# Copyright 2025 BBDevs
# Licensed under the Apache License, Version 2.0

"""Input validation utilities.

Author: A M (am@bbdevs.com)

Created At: 08 Nov 2025
"""

from __future__ import annotations

import re


__all__ = [
    "ValidationError",
    "validate_container_name",
    "validate_image_tag",
    "validate_port_mapping",
]


class ValidationError(ValueError):
    """Validation error for invalid input."""

    pass


def validate_container_name(name: str) -> str:
    """Validate container name.

    Container names must:
    - Start with a letter or number
    - Contain only letters, numbers, underscores, periods, hyphens
    - Be 1-64 characters long

    Args:
        name: Container name to validate

    Returns:
        Validated name

    Raises:
        ValidationError: If name is invalid

    Example:
        >>> validate_container_name("my-container")
        "my-container"
        >>> validate_container_name("invalid name")
        ValidationError: Container name contains invalid characters
    """
    if not name:
        msg = "Container name cannot be empty"
        raise ValidationError(msg)

    if len(name) > 64:
        msg = "Container name must be 64 characters or less"
        raise ValidationError(msg)

    # Docker container name pattern
    pattern = r"^[a-zA-Z0-9][a-zA-Z0-9_.-]*$"
    if not re.match(pattern, name):
        msg = "Container name contains invalid characters. Use letters, numbers, underscores, periods, or hyphens"
        raise ValidationError(msg)

    return name


def validate_image_tag(tag: str) -> str:
    """Validate image tag.

    Image tags must:
    - Be valid Docker image reference format
    - Not contain invalid characters

    Args:
        tag: Image tag to validate

    Returns:
        Validated tag

    Raises:
        ValidationError: If tag is invalid

    Example:
        >>> validate_image_tag("nginx:latest")
        "nginx:latest"
        >>> validate_image_tag("invalid tag")
        ValidationError: Invalid image tag format
    """
    if not tag:
        msg = "Image tag cannot be empty"
        raise ValidationError(msg)

    # Basic Docker image tag validation
    # Format: [registry/]repository[:tag]
    pattern = r"^[a-zA-Z0-9][a-zA-Z0-9._/-]*(?::[a-zA-Z0-9._-]+)?$"
    if not re.match(pattern, tag):
        msg = "Invalid image tag format. Use format: [registry/]repository[:tag]"
        raise ValidationError(msg)

    return tag


def validate_port_mapping(port_mapping: str) -> tuple[int, int]:
    """Validate port mapping format.

    Port mapping format: "host_port:container_port" or "host_port:container_port/protocol"

    Args:
        port_mapping: Port mapping string (e.g., "8080:80" or "8080:80/tcp")

    Returns:
        Tuple of (host_port, container_port)

    Raises:
        ValidationError: If port mapping is invalid

    Example:
        >>> validate_port_mapping("8080:80")
        (8080, 80)
        >>> validate_port_mapping("8080:80/tcp")
        (8080, 80)
        >>> validate_port_mapping("invalid")
        ValidationError: Invalid port mapping format
    """
    if not port_mapping:
        msg = "Port mapping cannot be empty"
        raise ValidationError(msg)

    # Remove protocol if present
    port_str = port_mapping.split("/")[0]

    # Parse host_port:container_port
    parts = port_str.split(":")
    if len(parts) != 2:
        msg = "Invalid port mapping format. Use format: host_port:container_port"
        raise ValidationError(msg)

    try:
        host_port = int(parts[0])
        container_port = int(parts[1])
    except ValueError as e:
        msg = "Port numbers must be integers"
        raise ValidationError(msg) from e

    # Validate port range
    if not (1 <= host_port <= 65535):
        msg = "Host port must be between 1 and 65535"
        raise ValidationError(msg)

    if not (1 <= container_port <= 65535):
        msg = "Container port must be between 1 and 65535"
        raise ValidationError(msg)

    return (host_port, container_port)
