import requests
import os
from urllib.parse import urljoin
from docx.types import DocxClient
from docx import utils, error
from docx.types import DocxRequest
from requests.exceptions import HTTPError, RequestException

from docx.constants import BASE_URL


class APIRequestor:
    def __init__(self, client: DocxClient):
        self.api_key = client.api_key or utils.default_api_key()
        self.base_url = client.base_url or BASE_URL

    def _build_url(self, endpoint: str) -> str:
        return urljoin(self.base_url, f"api/{endpoint.lstrip('/')}")

    def _prepare_request_raw(self, options: DocxRequest):
        url = self._build_url(options.url)

        default_headers = {
            "accept": "application/json",
            "x-api-key": self.api_key
        }

        if options.headers:
            default_headers.update(options.headers)

        files = None
        if options.files:
            file = open(options.files, "rb")
            files = {
                "file": (file.name, file, "image/jpeg")
            }

        return url, default_headers, files

    def request_raw(self, options: DocxRequest):
        url, headers, files = self._prepare_request_raw(options)

        try:
            response = requests.request(
                options.method,
                url,
                headers=headers,
                files=files
                )
            response.raise_for_status()
            return response
        except HTTPError as http_err:
            if response.status_code == 401:
                raise error.AuthenticationError(
                    "Authentication failed. Please check your API key."
                    ) from http_err
            else:
                raise error.APIError(
                    f"HTTP error occurred: {http_err}"
                    ) from http_err
        except RequestException as req_err:
            raise error.APIError(
                f"An error occurred while making the request: {req_err}"
                ) from req_err
        finally:
            if files:
                files['file'][1].close()

    # @overload
    # def request(self, method: str, url: str, **kwargs):
    #     options = DocxRequest(method=method, url=url, **kwargs)
    #     return self.request(options)

    def request(self, options: DocxRequest):
        response = self.request_raw(options)
        return response.json()


# Usage example
if __name__ == "__main__":
    client = DocxClient(api_key=os.getenv("DOCX_API_KEY"))
    file_path = "D:\\Docs\\work\\docx-sdk-1\\test\\IMG_2046.jpg"
    requestor = APIRequestor(client)
    try:
        response = requestor.request(
            DocxRequest(
                method="POST",
                url="/v1/document/upload/attribute-extraction-sync",
                files=file_path
            )
        )
        print("Response:")
        print(response)
    except error.APIError as e:
        print("An error occurred:", e)
