# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['docx2python']

package_data = \
{'': ['*']}

install_requires = \
['conventional-with-data>=0.1.0,<0.2.0',
 'lxml>=4.9.2,<5.0.0',
 'pre-commit>=2.21.0,<3.0.0',
 'tox-poetry-installer[poetry]>=0.10.0,<0.11.0']

setup_kwargs = {
    'name': 'docx2python',
    'version': '2.5.0',
    'description': 'Extract content from docx files',
    'long_description': '# docx2python\n\nExtract docx headers, footers, text, footnotes, endnotes, properties, and images to a Python object.\n\n`README_DOCX_FILE_STRUCTURE.md` may help if you\'d like to extend docx2python.\n\nFor a summary of what\'s new in docx2python 2, scroll down to **New in docx2python Version 2**\n\nThe code is an expansion/contraction of [python-docx2txt](https://github.com/ankushshah89/python-docx2txt) (Copyright (c) 2015 Ankush Shah). The original code is mostly gone, but some of the bones may still be here.\n\n__shared features__:\n* extracts text from docx files\n* extracts images from docx files\n\n__additions:__\n* extracts footnotes and endnotes\n* converts bullets and numbered lists to ascii with indentation\n* converts hyperlinks to ``<a href="http:/...">link text</a>``\n* retains some structure of the original file (more below)\n* extracts document properties (creator, lastModifiedBy, etc.)\n* inserts image placeholders in text (``\'----image1.jpg----\'``)\n* inserts plain text footnote and endnote references in text (``\'----footnote1----\'``)\n* (optionally) retains font size, font color, bold, italics, and underscore as html\n* extract math equations\n* extract user selections from checkboxes and dropdown menus\n\n__subtractions:__\n* no command-line interface\n* will only work with Python 3.8+\n\n\n## Installation\n```bash\npip install docx2python\n```\n\n## Use\n\n``` python\nfrom docx2python import docx2python\n\n# extract docx content\nwith docx2python(\'path/to/file.docx\') as docx_content:\n    print(docx_content.text)\n\ndocx_content = docx2python(\'path/to/file.docx\')\nprint(docx_content.text)\ndocx_content.close()\n\n# extract docx content, write images to image_directory\nwith docx2python(\'path/to/file.docx\', \'path/to/image_directory\') as docx_content:\n    print(docx_content.text)\n\n# extract docx content with basic font styles converted to html\nwith docx2python(\'path/to/file.docx\', html=True) as docx_content:\n    print(docx_content.text)\n```\n\n`docx2python` opens a zipfile object and (lazily) reads it. Use context management (`with ... as`) to close this zipfile object or explicitly close with `docx_content.close()`.\n\nNote on html feature:\n* supports ``<i>``italic, ``<b>``bold, ``<u>``underline, ``<s>``strike, ``<sup>``superscript, ``<sub>``subscript, ``<span style="font-variant: small-caps">``small caps, ``<span style="text-transform:uppercase">``all caps, ``<span style="background-color: yellow">``highlighted, ``<span style="font-size:32">``font size, ``<span style="color:#ff0000">``colored text.\n* hyperlinks will always be exported as html (``<a href="http:/...">link text</a>``), even if ``html=False``, because I couldn\'t think of a more canonical representation.\n* every tag open in a paragraph will be closed in that paragraph (and, where appropriate, reopened in the next paragraph). If two subsequenct paragraphs are bold, they will be returned as `<b>paragraph a</b>`, `<b>paragraph b</b>`. This is intentional to make  each paragraph its own entity.\n* if you specify `html=True`, `&`, `>` and `<` in your docx text will be encoded as `&amp`, `&gt;` and `&lt;`\n\n## Return Value\n\nFunction `docx2python` returns a DocxContent instance with several attributes.\n\n__header__ - contents of the docx headers in the return format described herein\n\n__footer__ - contents of the docx footers in the return format described herein\n\n__body__ - contents of the docx in the return format described herein\n\n__footnotes__ - contents of the docx in the return format described herein\n\n__endnotes__ - contents of the docx in the return format described herein\n\n__document__ - header  + body + footer (read only)\n\n__text__ - all docx text as one string, similar to what you\'d get from `python-docx2txt`\n\n__properties__ - docx property names mapped to values (e.g., `{"lastModifiedBy": "Shay Hill"}`)\n\n__images__ - image names mapped to images in binary format. Write to filesystem with\n\n```\nfor name, image in result.images.items():\n    with open(name, \'wb\') as image_destination:\n        write(image_destination, image)\n\n# or\n\nwith docx2python(\'path/to/file.docx\', \'path/to/image/directory\') as docx_content:\n    ...\n\n# or\n\nwith docx2python(\'path/to/file.docx\') as docx_content:\n    docx_content.save_images(\'path/to/image/directory\')\n\n```\n\n__docx_reader__ - a DocxReader (see `docx_reader.py`) instance with several methods for extracting xml portions.\n\n\n## Arguments\n\n    def docx2python(\n        docx_filename: str | Path | BytesIO,\n        image_folder: str | None = None,\n        html: bool = False,\n        paragraph_styles: bool = False,\n        extract_image: bool | None = None,\n        duplicate_merged_cells: bool = False\n    ) -> DocxContent:\n        """\n        Unzip a docx file and extract contents.\n\n        :param docx_filename: path to a docx file\n        :param image_folder: optionally specify an image folder\n            (images in docx will be copied to this folder)\n        :param html: bool, extract some formatting as html\n        :param paragraph_styles: prepend the paragraphs style (if any, else "") to each\n            paragraph. This will only be useful with ``*_runs`` attributes.\n        :param duplicate_merged_cells: bool, duplicate merged cells to return a mxn\n            nested list for each table (default False)\n        :return: DocxContent object\n        """\n\n\n## Return Format\n\nSome structure will be maintained. Text will be returned in a nested list, with paragraphs always at depth 4 (i.e., `output.body[i][j][k][l]` will be a paragraph).\n\nIf your docx has no tables, output.body will appear as one a table with all content in one cell:\n\n```python\n[  # document\n    [  # table\n        [  # row\n            [  # cell\n                "Paragraph 1",\n                "Paragraph 2",\n                "-- bulleted list",\n                "-- continuing bulleted list",\n                "1)  numbered list",\n                "2)  continuing numbered list"\n                "    a)  sublist",\n                "        i)  sublist of sublist",\n                "3)  keeps track of indention levels",\n                "    a)  resets sublist counters"\n            ]\n        ]\n     ]\n ]\n```\n\nTable cells will appear as table cells. Text outside tables will appear as table cells.\n\n\nA docx document can be tables within tables within tables. Docx2Python flattens most of this to more easily navigate\nwithin the content.\n\n## Working with output\n\nThis package provides several documented helper functions in [the ``docx2python.iterators`` module](https://docx2python.readthedocs.io/en/latest/docx2python.html#module-iterators). Here are a few recipes possible with these functions:\n\n```python\nfrom docx2python.iterators import enum_cells\n\ndef remove_empty_paragraphs(tables):\n    for (i, j, k), cell in enum_cells(tables):\n        tables[i][j][k] = [x for x in cell if x]\n```\n\n```\n>>> tables = [[[[\'a\', \'b\'], [\'a\', \'\', \'d\', \'\']]]]\n>>> remove_empty_paragraphs(tables)\n    [[[[\'a\', \'b\'], [\'a\', \'d\']]]]\n```\n\n```python\nfrom docx2python.iterators import enum_at_depth\n\ndef html_map(tables) -> str:\n    """Create an HTML map of document contents.\n\n    Render this in a browser to visually search for data.\n\n    :tables: value could come from, e.g.,\n        * docx_to_text_output.document\n        * docx_to_text_output.body\n    """\n\n    # prepend index tuple to each paragraph\n    for (i, j, k, l), paragraph in enum_at_depth(tables, 4):\n        tables[i][j][k][l] = " ".join([str((i, j, k, l)), paragraph])\n\n    # wrap each paragraph in <pre> tags\n    for (i, j, k), cell in enum_at_depth(tables, 3):\n        tables[i][j][k] = "".join(["<pre>{x}</pre>".format(x) for x in cell])\n\n    # wrap each cell in <td> tags\n    for (i, j), row in enum_at_depth(tables, 2):\n        tables[i][j] = "".join(["<td>{x}</td>".format(x) for x in row])\n\n    # wrap each row in <tr> tags\n    for (i,), table in enum_at_depth(tables, 1):\n        tables[i] = "".join("<tr>{x}</tr>".format(x) for x in table)\n\n    # wrap each table in <table> tags\n    tables = "".join([\'<table border="1">{x}</table>\'.format(x) for x in tables])\n\n    return ["<html><body>"] + tables + ["</body></html>"]\n```\n\n```\n>>> tables = [[[[\'a\', \'b\'], [\'a\', \'d\']]]]\n>>> html_map(tables)\n<html>\n    <body>\n        <table border="1">\n            <tr>\n                <td>\n                    \'(0, 0, 0, 0) a\'\n                    \'(0, 0, 0, 1) b\'\n                </td>\n                <td>\n                    \'(0, 0, 1, 0) a\'\n                    \'(0, 0, 1, 1) d\'\n                </td>\n            </tr>\n        </table>\n    </body>\n</html>\n```\n\n[See helper functions.](https://docx2python.readthedocs.io/en/latest/index.html)\n\nSome fine print about checkboxes:\n\nMS Word has checkboxes that can be checked any time, and others that can only be checked when the form is locked.\nThe previous print as. ``\\u2610`` (open checkbox) or ``\\u2612`` (crossed checkbox). Which this module, the latter will\ntoo. I gave checkboxes a bailout value of ``----checkbox failed----`` if the xml doesn\'t look like I expect it to,\nbecause I don\'t have several-thousand test files with checkboxes (as I did with most of the other form elements).\nCheckboxes *should* work, but please let me know if you encounter any that do not.\n\n# New in docx2python Version 2\n\n## merge consecutive runs with identical formatting\n\nMS Word will break up text runs arbitrarily, often in the middle of a word.\n\n\n    <w:r>\n        <w:t>work to im</w:t>\n    </w:r>\n    <w:r>\n        <w:t>prove docx2python</w:t>\n    </w:r>\n\nThis makes things like algorithmic search-and-replace problematic. Docx2python does not currently write docx files,\nbut I often use docx templates with placeholders (e.g., `#CATEGORY_NAME#`) then replace those placeholders with data.\nThis won\'t work if your placeholders are broken up (e.g, `#CAT`, `E`, `GORY_NAME#`).\n\nDocx2python v1 merges such runs together when exporting text. Docx2python v2 will merge such runs in the XML as a\npre-processing step. This will allow saving such "repaired" XML later on.\n\n## merge consecutive links with identical hrefs\n\nMS Word will break up links, giving each link a different `rId`, even when these `rIds` point to the same address.\n\n    <w:hyperlink r:id="rId13">  # rID13 points to https://github.com/ShayHill/docx2python\n        <w:r>\n            <w:t>docx2py</w:t>\n        </w:r>\n    </w:hyperlink>\n    <w:hyperlink r:id="rId14">  # rID14 ALSO points to https://github.com/ShayHill/docx2python\n        <w:r>\n            <w:t>thon</w:t>\n        </w:r>\n    </w:hyperlink>\n\nThis is similar to the broken-up runs, but the cause is a little deeper in. Docx2python v1 makes a mess of these.\n\n    <a href="https://github.com/ShayHill/docx2python">docx2py</a>\n    <a href="https://github.com/ShayHill/docx2python">thon</a>\n\nDocx2python v2 will merge such links together in the XML as a pre-processing step. As above, this will allow saving\nsuch "repaired" XML later on.\n\n## correctly handle nested paragraphs\n\nMS Word will nest paragraphs\n\n    <w:p>\n        <w:r>\n            <w:t>text</w:t>\n        </w:r>\n        <w:p>  # paragraph inside a paragraph\n            <w:r>\n                <w:t>text</w:t>\n            </w:r>\n        </w:p>\n        <w:r>\n            <w:t>text</w:t>\n        </w:r>\n    </w:p>\n\nI haven\'t been able to create such a paragraph, but I\'ve found a few files that have them. Docx2pyhon v1 will omit\nclosing html tags when a new paragraph is opened before the old paragraph is closed.\n\n    <b>outer par bold text\n\n    <i>This text is in nested par (not bold)</i>\n\n    outer par bold text</b>\n\nDocx2python v2 will correctly handle such cases, but this will require substantial internal changes to the way\ndocx2python opens and closes paragraphs.\n\n    <b>outer par bold text</b>\n\n    <i>This text is in nested par (not bold)</i>\n\n    </b>outer par bold text</b>\n\n## paragraph styles\n\nThe internal changes allow for easy access to paragraph styles (e.g., `Heading 1`). Docx2python v1 ignores these, even\nwith `html=True`. Docx2python v2 will capture paragraph styles.\n\n    <h1>h1 is a paragraph style<b>bold is a run style</b></h1>\n\n## export xml\n\nTo allow above-described light editing (e.g., search and replace), docx2python v2 will give the user access to\n\n    1. extracted xml files\n    2. the functions used to write these files to a docx\n\nThe user can only go so far with this. A docx file is built from folders full of xml files. None of these xml\nfiles are self contained. But search and replace is enough to make document templates (documents with placeholders for\ndata), and that\'s pretty useful in itself.\n\n## expose some intermediate functionality\n\nNavigating through XML is straightforward with `lxml`. It is a separate step to take whatever you find and bring it\n*out* of the XML. For instance, you may want to iterate over a document, looking for paragraphs with a particular\nformat, then pull the text out of those paragraphs. Docx2python v1 did not separate or expose "iter the document" and\n"pull the content". Docx2python v2 separates and exposes these steps. This will allow easier extension.\n\nSee the `docx_reader.py` module and simple examples in the `utilities.py` module.\n\n## see utilities.py for examples of major new features.\n',
    'author': 'Shay Hill',
    'author_email': 'shay_public@hotmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
