# Dolfi Python SDK

[![PyPI version](https://badge.fury.io/py/dolfi-python-sdk.svg)](https://badge.fury.io/py/dolfi-python-sdk)
[![Python Support](https://img.shields.io/pypi/pyversions/dolfi-python-sdk.svg)](https://pypi.org/project/dolfi-python-sdk)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)

An easy-to-use Python SDK for the Dolfi API that provides intelligent web search and content scraping capabilities.

## Features

- **🔍 AI-Powered Search**: Natural language web search with AI-generated summaries
- **🕷️ Web Scraping**: Extract content from web pages in markdown or text format
- **🌐 Multi-language Support**: Search and get answers in 20+ languages
- **⚡ Async Support**: Both synchronous and asynchronous client implementations
- **🔒 Type Safety**: Full type hints and Pydantic model validation
- **🛡️ Error Handling**: Comprehensive exception handling with detailed error messages

## Installation

```bash
pip install dolfi-python-sdk
```

## Quick Start

### Search Example

```python
from dolfi_sdk import DolfiClient

# Initialize the client
client = DolfiClient(api_key="your-api-key-here")

# Perform a search
response = client.search(
    query="What are the latest developments in AI?",
    max_results=5,
    include_answer=True
)

print(f"AI Answer: {response.answer}")
for result in response.results:
    print(f"- {result.title}: {result.url}")
```

### Scraping Example

```python
# Scrape content from web pages
response = client.scrape(
    urls=["https://example.com", "https://another-site.com"],
    format="markdown",
    ignore_links=False
)

for result in response.results:
    if result.status == 200:
        print(f"Content from {result.url}:")
        print(result.formatted_content[:500] + "...")
```

### Async Usage

```python
import asyncio
from dolfi_sdk import AsyncDolfiClient

async def main():
    async with AsyncDolfiClient(api_key="your-api-key-here") as client:
        # Async search
        search_response = await client.search("Python programming best practices")
        
        # Async scraping
        scrape_response = await client.scrape("https://python.org")
        
        print(f"Found {search_response.num_of_results} results")
        print(f"Scraped {len(scrape_response.results)} pages")

asyncio.run(main())
```

## API Reference

### DolfiClient

The main synchronous client for interacting with the Dolfi API.

#### Methods

##### `search(query, **kwargs) -> SearchResponse`

Perform an intelligent web search with optional AI-generated answers.

**Parameters:**
- `query` (str): Natural language search query (max 400 characters)
- `max_results` (int, optional): Maximum number of results (1-20, default: 5)
- `time_range` (str, optional): Filter by time period ("day", "month", "year")
- `search_language` (str, optional): Language code for search (default: "en")
- `include_answer` (bool, optional): Include AI-generated summary (default: True)
- `answer_language` (str, optional): Language code for AI answer (default: "en")
- `answer_instruction` (str, optional): Custom instructions for AI answer formatting

##### `scrape(urls, **kwargs) -> ScrapeResponse`

Scrape content from one or more web pages.

**Parameters:**
- `urls` (str | List[str]): Single URL or list of URLs to scrape
- `format` (str, optional): Output format ("markdown" or "text", default: "markdown")
- `ignore_links` (bool, optional): Exclude hyperlinks from output (default: False)
- `ignore_images` (bool, optional): Exclude images from output (default: False)
- `mobile` (bool, optional): Use mobile version of websites (default: False)

### AsyncDolfiClient

Asynchronous client with the same interface as `DolfiClient`, but all methods are `async`.

### Response Models

#### SearchResponse
- `id`: Unique identifier for the search
- `query`: Original search query
- `num_of_results`: Total number of results found
- `answer`: AI-generated summary answer
- `search_words`: Optimized search keywords used
- `results`: List of `SearchResultItem` objects
- `turn_around`: Processing time in seconds

#### ScrapeResponse
- `id`: Unique identifier for the scraping operation
- `request_format`: Requested output format
- `results`: List of `ScrapeResponseItem` objects
- `turn_around`: Processing time in seconds

### Supported Languages

The SDK supports 20+ languages for both search and AI answers:

`zh` (Chinese), `en` (English), `hi` (Hindi), `es` (Spanish), `ar` (Arabic), `fr` (French), `bn` (Bengali), `pt` (Portuguese), `ru` (Russian), `ja` (Japanese), `pa` (Punjabi), `de` (German), `jv` (Javanese), `ko` (Korean), `vi` (Vietnamese), `te` (Telugu), `mr` (Marathi), `tr` (Turkish), `ta` (Tamil), `ur` (Urdu)

## Error Handling

The SDK provides specific exception types for different error scenarios:

```python
from dolfi_sdk import (
    DolfiAuthenticationError,
    DolfiValidationError, 
    DolfiAPIError,
    DolfiConnectionError,
    DolfiTimeoutError
)

try:
    response = client.search("example query")
except DolfiAuthenticationError:
    print("Invalid API key")
except DolfiValidationError as e:
    print(f"Invalid request: {e.message}")
except DolfiAPIError as e:
    print(f"API error: {e.message} (Status: {e.status_code})")
except DolfiConnectionError:
    print("Connection failed")
except DolfiTimeoutError:
    print("Request timed out")
```

## Configuration

### Environment Variables

You can set your API key as an environment variable:

```bash
export DOLFI_API_KEY="your-api-key-here"
```

```python
import os
from dolfi_sdk import DolfiClient

client = DolfiClient(api_key=os.getenv("DOLFI_API_KEY"))
```

### Custom Base URL

For enterprise or custom deployments:

```python
client = DolfiClient(
    api_key="your-api-key",
    base_url="https://your-custom-api.com",
    timeout=60
)
```

## Contributing

We welcome contributions! Please see our [Contributing Guide](CONTRIBUTING.md) for details.

## License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.

## Support

- 📧 Email: support@dolfi.com
- 📖 Documentation: https://docs.dolfi.com
- 🐛 Issues: https://github.com/dolfi/dolfi-python-sdk/issues

## Changelog

See [CHANGELOG.md](CHANGELOG.md) for a history of changes to this package.