"""
Dominus SDK Endpoints

Gateway URL for authentication (JWT minting).
Orchestrator URL for all service calls.

Configuration:
    Set DOMINUS_GATEWAY_URL to override the gateway URL.
    Set DOMINUS_BASE_URL to override the orchestrator URL.
    Example: export DOMINUS_BASE_URL=http://localhost:5000

Usage:
    from dominus.config.endpoints import GATEWAY_URL, BASE_URL, get_gateway_url, get_base_url
"""
import os

# Default URLs
_DEFAULT_GATEWAY_URL = "https://dominus-gateway-production-775398158805.us-east4.run.app"
_DEFAULT_BASE_URL = "https://dominus-orchestrator-production-775398158805.us-east4.run.app"

# Gateway URL for JWT operations (can be overridden via DOMINUS_GATEWAY_URL)
GATEWAY_URL = os.environ.get("DOMINUS_GATEWAY_URL", _DEFAULT_GATEWAY_URL)

# Base URL for service calls (can be overridden via DOMINUS_BASE_URL environment variable)
BASE_URL = os.environ.get("DOMINUS_BASE_URL", _DEFAULT_BASE_URL)

# Legacy aliases (all point to orchestrator now) - DEPRECATED
SOVEREIGN_URL = BASE_URL
ARCHITECT_URL = BASE_URL
ORCHESTRATOR_URL = BASE_URL
WARDEN_URL = BASE_URL


def get_gateway_url() -> str:
    """
    Get the dominus-gateway base URL for JWT operations.

    Returns the value of DOMINUS_GATEWAY_URL environment variable if set,
    otherwise returns the default production URL.
    """
    return os.environ.get("DOMINUS_GATEWAY_URL", _DEFAULT_GATEWAY_URL)


def get_base_url() -> str:
    """
    Get the dominus-orchestrator base URL for service calls.

    Returns the value of DOMINUS_BASE_URL environment variable if set,
    otherwise returns the default production URL.
    """
    return os.environ.get("DOMINUS_BASE_URL", _DEFAULT_BASE_URL)


# DEPRECATED - use get_base_url()
def get_sovereign_url(environment: str = None) -> str:
    """Deprecated: Use get_base_url() instead."""
    return BASE_URL


def get_architect_url(environment: str = None) -> str:
    """Deprecated: Use get_base_url() instead."""
    return BASE_URL


def get_service_url(service: str, environment: str = None) -> str:
    """Deprecated: Use get_base_url() instead. All services are now consolidated."""
    return BASE_URL
