/*!

dss_obj.hpp: a C++ layer for DSS Extensions/C-API, Obj interface
Copyright (c) 2021-2022 Paulo Meira

Version: 0.5.0 (2022-07)

**POTENTIAL BREAKING CHANGES UNTIL VERSION 1.0**

*/

#pragma once
#ifndef DSS_CPP_OBJ_API
#define DSS_CPP_OBJ_API
#include "dss_common.hpp"

namespace dss { namespace obj {

#ifdef DSS_CAPI_NAMESPACE
using namespace dss::capi;
#endif

namespace detail {
    void obj_set_val(void *ptr, int32_t idx, int32_t value)
    {
        Obj_SetInt32(ptr, idx, value);
    }

    void obj_set_val(void *ptr, int32_t idx, double value)
    {
        Obj_SetFloat64(ptr, idx, value);
    }

    void obj_set_val(void *ptr, int32_t idx, bool value)
    {
        Obj_SetInt32(ptr, idx, value);
    }

    void obj_set_val(void *ptr, int32_t idx, const string &value)
    {
        Obj_SetString(ptr, idx, value.c_str());
    }

    void obj_set_val(void *ptr, int32_t idx, const char* value)
    {
        Obj_SetString(ptr, idx, value);
    }

    void obj_get_array(double** ResultPtr, int32_t* ResultCount, void *obj, int32_t Index)
    {
        Obj_GetFloat64Array(ResultPtr, ResultCount, obj, Index);
    }

    void obj_get_array(int32_t** ResultPtr, int32_t* ResultCount, void *obj, int32_t Index)
    {
        Obj_GetInt32Array(ResultPtr, ResultCount, obj, Index);
    }

    void obj_get_array(char*** ResultPtr, int32_t* ResultCount, void *obj, int32_t Index)
    {
        Obj_GetStringArray(ResultPtr, ResultCount, obj, Index);
    }

    void obj_get_array(void*** ResultPtr, int32_t* ResultCount, void *obj, int32_t Index)
    {
        Obj_GetObjectArray(ResultPtr, ResultCount, obj, Index);
    }

    void dispose_array(double** ptr, int32_t)
    {
        DSS_Dispose_PDouble(ptr);
    }

    void dispose_array(int32_t** ptr, int32_t)
    {
        DSS_Dispose_PInteger(ptr);
    }

    void dispose_array(char*** ptr, int32_t cnt)
    {
        DSS_Dispose_PPAnsiChar(ptr, cnt);
    }

    void dispose_array(void*** ptr, int32_t)
    {
        DSS_Dispose_PPointer(ptr);
    }

    void obj_set_array(void *obj, int32_t Index, double* Value, int32_t ValueCount)
    {
        Obj_SetFloat64Array(obj, Index, Value, ValueCount);
    }

    void obj_set_array(void *obj, int32_t Index, int32_t* Value, int32_t ValueCount)
    {
        Obj_SetInt32Array(obj, Index, Value, ValueCount);
    }

    void obj_set_array(void *obj, int32_t Index, const char** Value, int32_t ValueCount)
    {
        Obj_SetStringArray(obj, Index, Value, ValueCount);
    }

    void obj_set_array(void *obj, int32_t Index, void **Value, int32_t ValueCount)
    {
        Obj_SetObjectArray(obj, Index, Value, ValueCount);
    }

    void obj_set_val(void *obj, int32_t Index, complex Value)
    {
        Obj_SetFloat64Array(obj, Index, (double*)&Value, 2);
    }


    void batch_set_val(void **ptr, int32_t cnt, int32_t idx, int32_t value)
    {
        Batch_Int32(ptr, cnt, idx, BatchOperation_Set, value);
    }

    void batch_set_val(void **ptr, int32_t cnt, int32_t idx, double value)
    {
        Batch_Float64(ptr, cnt, idx, BatchOperation_Set, value);
    }

    void batch_set_val(void **ptr, int32_t cnt, int32_t idx, bool value)
    {
        Batch_Int32(ptr, cnt, idx, BatchOperation_Set, value);
    }

    void batch_set_val(void **ptr, int32_t cnt, int32_t idx, const string &value)
    {
        Batch_SetString(ptr, cnt, idx, value.c_str());
    }

    void batch_set_val(void **ptr, int32_t cnt, int32_t idx, const char* value)
    {
        Batch_SetString(ptr, cnt, idx, value);
    }

    void batch_set_val(void **ptr, int32_t cnt, int32_t idx, complex value)
    {
        void **ptr_end = ptr + cnt;
        while (ptr != ptr_end)
        {
            obj_set_val(*ptr, idx, value);
            ++ptr;
        }
    }

    void batch_op(void **ptr, int32_t cnt, int32_t idx, int32_t op, double value)
    {
        Batch_Float64(ptr, cnt, idx, op, value);
    }

    void batch_op(void **ptr, int32_t cnt, int32_t idx, int32_t op, int32_t value)
    {
        Batch_Int32(ptr, cnt, idx, op, value);
    }

    void batch_get_val(double** ResultPtr, int32_t* ResultCount, void **batch, int32_t batchSize, int32_t Index)
    {
        Batch_GetFloat64(ResultPtr, ResultCount, batch, batchSize, Index);
    }

    void batch_get_val(int32_t** ResultPtr, int32_t* ResultCount, void **batch, int32_t batchSize, int32_t Index)
    {
        Batch_GetInt32(ResultPtr, ResultCount, batch, batchSize, Index);
    }

    void batch_get_val(char*** ResultPtr, int32_t* ResultCount, void **batch, int32_t batchSize, int32_t Index)
    {
        Batch_GetString(ResultPtr, ResultCount, batch, batchSize, Index);
    }

    void batch_get_val(void*** ResultPtr, int32_t* ResultCount, void **batch, int32_t batchSize, int32_t Index)
    {
        Batch_GetObject(ResultPtr, ResultCount, batch, batchSize, Index);
    }
} // namespace detail

class DSSObj
{
public:
    APIUtil *api_util;
    void *ptr;

    DSSObj(APIUtil *util=nullptr, void *ptr_=nullptr): api_util(util), ptr(ptr_)
    {
    }

    // virtual ~DSSObj()
    // {
    // }

protected:
    void check_for_error()
    {
        api_util->check_for_error();
    }

    void set_string(int32_t index, const string &value)
    {
        Obj_SetString(ptr, index, value.c_str());
    }

    void set_string(int32_t index, const char *value)
    {
        Obj_SetString(ptr, index, value);
    }

    string get_prop_string(int32_t index)
    {
        char* sc = Obj_GetString(ptr, index);
        string res(sc);
        DSS_Dispose_String(sc);
        return res;
    }

    complex get_complex(int32_t index)
    {
        complex cres;
        double *res = (double*) &cres;
        int32_t cnt[2] = {2, 2};
        Obj_GetFloat64Array(&res, cnt, ptr, index);
        if (cnt[0] != 2)
        {
            throw std::runtime_error("Unexpected number of elements return for complex value!");
        }
        return cres;
    }

    void set_complex(int32_t index, complex value)
    {
        Obj_SetFloat64Array(ptr, index, (double*)(&value), 2);
    }

    void set_string_array(int32_t index, strings &value)
    {
        std::vector<const char*> ptrs(value.size(), nullptr);
        for (size_t i = 0; i < value.size(); ++i)
        {
            ptrs[i] = value[i].c_str();
        }
        Obj_SetStringArray(ptr, index, &ptrs[0], int32_t(value.size()));
        check_for_error();
    }

    void set_obj(int32_t index, DSSObj &value)
    {
        Obj_SetObject(ptr, index, value.ptr);
    }

    template <typename T>
    T get_obj(int32_t index)
    {
        T o{api_util};
        o.ptr = Obj_GetObject(ptr, index);
        return o;
    }

    template <typename T = VectorXd>
    T get_array(int32_t index)
    {
        T res;
        int32_t data_cnt[2] = {0, 0};

        if constexpr (std::is_same<string, typename T::value_type>::value)
        {
            char **data_ptr = nullptr;
            detail::obj_get_array(&data_ptr, data_cnt, ptr, index);
            res.resize(data_cnt[0]);
            for (size_t j = 0; j < data_cnt[0]; ++j)
            {
                res[j] = data_ptr[j];
            }
            detail::dispose_array(&data_ptr, data_cnt[0]);
        }
        else if constexpr (std::is_convertible<typename T::value_type*, DSSObj*>::value)
        {
            void **data_ptr = nullptr;
            detail::obj_get_array(&data_ptr, data_cnt, ptr, index);
            res.reserve(data_cnt[0]);
            for (size_t j = 0; j < data_cnt[0]; ++j)
            {
                res.emplace_back(api_util, data_ptr[j]);
            }
            detail::dispose_array(&data_ptr, data_cnt[0]);
        }
        else
        {
            typename T::value_type *data_ptr = nullptr;
            if constexpr(std::is_enum<typename T::value_type>::value)
            {
                detail::obj_get_array((int32_t**)&data_ptr, data_cnt, ptr, index);
            }
            else
            {
                detail::obj_get_array(&data_ptr, data_cnt, ptr, index);
            }
            res.resize(data_cnt[0]);
            memcpy(&res[0], data_ptr, sizeof(typename T::value_type) * data_cnt[0]);
            if constexpr(std::is_enum<typename T::value_type>::value)
            {
                detail::dispose_array((int32_t**)&data_ptr, data_cnt[0]);
            }
            else
            {
                detail::dispose_array(&data_ptr, data_cnt[0]);
            }
        }
        check_for_error();
        return res;
    }

    template <typename T = VectorXd>
    static void set_array(void* ptr, int32_t index, T value)
    {
        if constexpr (std::is_same<string, typename T::value_type>::value)
        {
            std::vector<const char*> prepvalue(value.size());
            for (size_t i = 0; i < value.size(); ++i)
            {
                prepvalue[i] = value[i].c_str();
            }
            detail::obj_set_array(ptr, index, &prepvalue[0], value.size());
        }
        else if constexpr (std::is_convertible<typename T::value_type*, DSSObj*>::value)
        {
            std::vector<void*> prepvalue(value.size());
            for (size_t i = 0; i < value.size(); ++i)
            {
                prepvalue[i] = value[i].ptr;
            }
            detail::obj_set_array(ptr, index, &prepvalue[0], value.size());
        }
        else if constexpr (std::is_enum<typename T::value_type>::value)
        {
            detail::obj_set_array(ptr, index, (int32_t*)&value[0], value.size());
        }
        else
        {
            detail::obj_set_array(ptr, index, &value[0], value.size());
        }
    }

    template <typename T = VectorXd>
    void set_array(int32_t index, T value)
    {
        set_array<T>(ptr, index, value);
        check_for_error();
    }

    friend DSSBatch;
};

template<typename T>
class BatchArrayProxy;

class DSSBatch
{
public:
    APIUtil *api_util;
    void **pointer;
    int32_t count[2];

    ///
    /// Create a batch of all elements in the DSS class
    ///
    DSSBatch(APIUtil *util, int32_t cls_idx): api_util(util), count{0, 0}, pointer(nullptr)
    {
        Batch_CreateByClass(api_util->ctx, &pointer, count, cls_idx);
        check_for_error();
    }

    ///
    /// Create a batch of all elements that match an integer property value
    ///
    DSSBatch(APIUtil *util, int32_t cls_idx, int32_t prop_idx, int32_t prop_value): api_util(util), count{0, 0}, pointer(nullptr)
    {
        Batch_CreateByInt32Property(api_util->ctx, &pointer, count, cls_idx, prop_idx, prop_value);
        check_for_error();
    }

    ///
    /// Create a batch of all elements that match a regular expression
    ///
    DSSBatch(APIUtil *util, int32_t cls_idx, const char* regexp): api_util(util), count{0, 0}, pointer(nullptr)
    {
        Batch_CreateByRegExp(api_util->ctx, &pointer, count, cls_idx, regexp);
        check_for_error();
    }

    ///
    /// Create an uninitialized batch instance
    ///
    DSSBatch(APIUtil *util): api_util(util), count{0, 0}, pointer(nullptr)
    {
    }

    ~DSSBatch()
    {
        Batch_Dispose(pointer);
    }

    strings name()
    {
        strings result;
        result.reserve(count[0]);
        for (size_t i = 0; i < count[0]; ++i)
        {
            result.emplace_back(Obj_GetName(pointer[i]));
        }
        return result;
    }
protected:
    inline void check_for_error()
    {
        api_util->check_for_error();
    }

    bools get_batch_bool(int32_t index)
    {
        bools res;
        int32_t res_cnt[2] = {count[0], count[0]};
        res.resize(count[0]);
        int32_t *pres = &res[0];
        Batch_GetInt32(&pres, res_cnt, pointer, count[0], index);
        return res;
    }

    std::vector<complex> get_batch_complex(int32_t index)
    {
        void** it = pointer;
        void** it_end = pointer + count[0];
        std::vector<complex> res;
        res.resize(count[0]);
        std::vector<complex>::iterator res_it = res.begin();
        int32_t res_cnt[2] = {2, 2};
        while (it != it_end)
        {
            double *pres_it = (double*)&(*res_it);
            Obj_GetFloat64Array(&pres_it, res_cnt, *it, index);
            ++it;
            ++res_it;
        }
        return res;
    }

    template <typename T = VectorXd>
    std::vector<T> get_batch_valarray(int32_t index)
    {
        std::vector<T> res;
        res.resize(count[0]);
        int32_t data_cnt[2] = {0, 0};

        if constexpr (std::is_same<string, typename T::value_type>::value)
        {
            char **data_ptr = nullptr;
            for (size_t i = 0; i < count[0]; ++i)
            {
                detail::obj_get_array(&data_ptr, data_cnt, pointer[i], index);
                res[i].resize(data_cnt[0]);
                auto &resi = res[i];
                for (size_t j = 0; j < data_cnt[0]; ++j)
                {
                    resi[j] = data_ptr[j];
                }
            }
            detail::dispose_array(&data_ptr, data_cnt[0]);
        }
        else if constexpr (std::is_convertible<typename T::value_type*, DSSObj*>::value)
        {
            char **data_ptr = nullptr;
            for (size_t i = 0; i < count[0]; ++i)
            {
                detail::obj_get_array(&data_ptr, data_cnt, pointer[i], index);
                res[i].reserve(data_cnt[0]);
                auto &resi = res[i];
                for (size_t j = 0; j < data_cnt[0]; ++j)
                {
                    resi.emplace_back(api_util, data_ptr[j]);
                }
            }
            detail::dispose_array(&data_ptr, data_cnt[0]);
        }
        else
        {
            typename T::value_type *data_ptr = nullptr;
            for (size_t i = 0; i < count[0]; ++i)
            {
                detail::obj_get_array(&data_ptr, data_cnt, pointer[i], index);
                res[i].resize(data_cnt[0]);
                memcpy(&res[i][0], data_ptr, sizeof(typename T::value_type) * data_cnt[0]);
            }
            detail::dispose_array(&data_ptr, data_cnt[0]);
        }

        return res;
    }

    void set_batch_complex_for_each(int32_t index, std::vector<complex> &values)
    {
        if (values.size() != count[0])
        {
            throw std::runtime_error("Number of elements provided must match the number of objects in the batch.");
        }
        for (size_t i = 0; i < values.size(); ++i)
        {
            Obj_SetFloat64Array(*(pointer + i), index, (double*)(&values[i]), 2);
        }
        check_for_error();
    }

    template <typename T>
    void set_batch_val(int32_t index, const T& value)
    {
        if constexpr (std::is_enum<T>::value)
        {
            detail::batch_set_val(pointer, count[0], index, int32_t(value));
        }
        else if constexpr (std::is_scalar<T>::value || std::is_same<complex, T>::value)
        {
            detail::batch_set_val(pointer, count[0], index, value);
        }
        else
        {
            void** it = pointer;
            void** it_end = pointer + count[0];

            while (it != it_end)
            {
                if constexpr (std::is_same<string, T>::value)
                {
                    detail::obj_set_val(*it, index, value.c_str());
                }
                else if constexpr (std::is_convertible<T*, DSSObj*>::value)
                {
                    detail::obj_set_val(*it, index, value.ptr);
                }
                else
                {
                    DSSObj::set_array<T>(*it, index, value);
                }
                ++it;
            }
        }
        check_for_error();
    }

protected:

    template <typename T = VectorXd>
    static T get_batch_val(int32_t index, DSSBatch &batch) //TODO: reuse code
    {
        T res;
        int32_t data_cnt[2] = {0, 0};

        if constexpr (std::is_same<string, typename T::value_type>::value)
        {
            char **data_ptr = nullptr;
            detail::batch_get_val(&data_ptr, data_cnt, batch.pointer, batch.count[0], index);
            res.resize(data_cnt[0]);
            for (size_t j = 0; j < data_cnt[0]; ++j)
            {
                res[j] = data_ptr[j];
            }
            detail::dispose_array(&data_ptr, data_cnt[0]);
        }
        else if constexpr (std::is_convertible<typename T::value_type*, DSSObj*>::value)
        {
            void **data_ptr = nullptr;
            detail::batch_get_val(&data_ptr, data_cnt, batch.pointer, batch.count[0], index);
            res.reserve(data_cnt[0]);
            for (size_t j = 0; j < data_cnt[0]; ++j)
            {
                res.emplace_back(batch.api_util, data_ptr[j]);
            }
            detail::dispose_array(&data_ptr, data_cnt[0]);
        }
        else
        {
            typename T::value_type *data_ptr = nullptr;
            if constexpr(std::is_enum<typename T::value_type>::value)
            {
                detail::batch_get_val((int32_t**)&data_ptr, data_cnt, batch.pointer, batch.count[0], index);
            }
            else
            {
                detail::batch_get_val(&data_ptr, data_cnt, batch.pointer, batch.count[0], index);
            }
            res.resize(data_cnt[0]);
            memcpy(&res[0], data_ptr, sizeof(typename T::value_type) * data_cnt[0]);
            if constexpr(std::is_enum<typename T::value_type>::value)
            {
                detail::dispose_array((int32_t**)&data_ptr, data_cnt[0]);
            }
            else
            {
                detail::dispose_array(&data_ptr, data_cnt[0]);
            }
        }
        batch.check_for_error();
        return res;
    }

    template <typename T = VectorXd>
    T get_batch_val(int32_t index) //TODO: reuse code
    {
        return get_batch_val<T>(index, *this);
    }

    template <typename T>
    void set_batch_val_for_each(int32_t index, typename T::iterator v, typename T::iterator v_end)
    {
        const int32_t cnt = v_end - v;
        void** it = pointer;
        void** it_end = pointer + count[0];
        if (cnt != count[0])
        {
            throw std::runtime_error("Number of elements provided must match the number of objects in the batch.");
        }

        while (it != it_end)
        {
            if constexpr (std::is_same<string, typename T::value_type>::value)
            {
                detail::obj_set_val(*it, index, (*v).c_str());
            }
            else if constexpr (std::is_convertible<typename T::value_type*, DSSObj*>::value)
            {
                detail::obj_set_val(*it, index, (*v).ptr());
            }
            else if constexpr (std::is_enum<typename T::value_type>::value)
            {
                detail::obj_set_val(*it, index, int32_t(*v));
            }
            else if constexpr (std::is_scalar<typename T::value_type>::value || std::is_same<complex, typename T::value_type>::value)
            {
                detail::obj_set_val(*it, index, *v);
            }
            else
            {
                DSSObj::set_array<typename T::value_type>(*it, index, *v);
            }
            ++it;
            ++v;
        }
        check_for_error();
    }
    template<typename T>
    friend class BatchArrayProxy;
};

template<typename T>
class BatchArrayProxy
{
public:
    DSSBatch &batch;
    int32_t idx;

    BatchArrayProxy(DSSBatch &batch_, int32_t prop_idx): batch(batch_), idx(prop_idx)
    {
    }

    template<typename VectorT>
    VectorT to_array()
    {
        return DSSBatch::get_batch_val<VectorT>(idx, batch);
    }

    template<typename VectorT>
    operator VectorT()
    {
        return to_array<VectorT>();
    }

    BatchArrayProxy<T>& operator+(T other)
    {
        batch_op(batch.pointer, batch.count[0], idx, BatchOperation_Increment, other);
        return *this;
    }

    BatchArrayProxy<T>& operator-(T other)
    {
        batch_op(batch.pointer, batch.count[0], idx, BatchOperation_Increment, -other);
        return *this;
    }

    BatchArrayProxy<T>& operator*(T other)
    {
        batch_op(batch.pointer, batch.count[0], idx, BatchOperation_Multiply, other);
        return *this;
    }

    BatchArrayProxy<T>& operator/(T other)
    {
        batch_op(batch.pointer, batch.count[0], idx, BatchOperation_Multiply, 1 / other);
        return *this;
    }

    BatchArrayProxy<T>& operator=(T other)
    {
        batch_op(batch.pointer, batch.count[0], idx, BatchOperation_Set, other);
        return *this;
    }

    BatchArrayProxy<T>& operator+=(T other)
    {
        batch_op(batch.pointer, batch.count[0], idx, BatchOperation_Increment, other);
        return *this;
    }

    BatchArrayProxy<T>& operator-=(T other)
    {
        batch_op(batch.pointer, batch.count[0], idx, BatchOperation_Increment, -other);
        return *this;
    }

    BatchArrayProxy<T>& operator*=(T other)
    {
        batch_op(batch.pointer, batch.count[0], idx, BatchOperation_Multiply, other);
        return *this;
    }

    BatchArrayProxy<T>& operator/=(T other)
    {
        batch_op(batch.pointer, batch.count[0], idx, BatchOperation_Multiply, 1 / other);
        return *this;
    }
};


typedef BatchArrayProxy<double> BatchFloat64ArrayProxy;
typedef BatchArrayProxy<int32_t> BatchInt32ArrayProxy;


// Global enumerations

///
/// Earth Model (DSS enumeration)
///
enum class EarthModel: int32_t
{
    Carson = 1, ///< Carson
    FullCarson = 2, ///< FullCarson
    Deri = 3 ///< Deri
};


///
/// Line Type (DSS enumeration)
///
enum class LineType: int32_t
{
    oh = 1, ///< oh
    ug = 2, ///< ug
    ug_ts = 3, ///< ug_ts
    ug_cn = 4, ///< ug_cn
    swt_ldbrk = 5, ///< swt_ldbrk
    swt_fuse = 6, ///< swt_fuse
    swt_sect = 7, ///< swt_sect
    swt_rec = 8, ///< swt_rec
    swt_disc = 9, ///< swt_disc
    swt_brk = 10, ///< swt_brk
    swt_elbow = 11 ///< swt_elbow
};


///
/// Dimension Units (DSS enumeration)
///
enum class DimensionUnits: int32_t
{
    none = 0, ///< none
    mi = 1, ///< mi
    kft = 2, ///< kft
    km = 3, ///< km
    m = 4, ///< m
    ft = 5, ///< ft
    inch = 6, ///< in
    cm = 7, ///< cm
    mm = 8, ///< mm
    meter = 4, ///< meter
    miles = 1 ///< miles
};


///
/// Scan Type (DSS enumeration)
///
enum class ScanType: int32_t
{
    none = -1, ///< None
    Zero = 0, ///< Zero
    Positive = 1 ///< Positive
};


///
/// Sequence Type (DSS enumeration)
///
enum class SequenceType: int32_t
{
    Negative = -1, ///< Negative
    Zero = 0, ///< Zero
    Positive = 1 ///< Positive
};


///
/// Connection (DSS enumeration)
///
enum class Connection: int32_t
{
    wye = 0, ///< wye
    delta = 1, ///< delta
    y = 0, ///< y
    ln = 0, ///< ln
    ll = 1 ///< ll
};


///
/// Core Type (DSS enumeration)
///
enum class CoreType: int32_t
{
    shell = 0, ///< shell
    one_phase = 1, ///< 1-phase
    three_leg = 3, ///< 3-leg
    four_leg = 4, ///< 4-leg
    five_leg = 5, ///< 5-leg
    core_1_phase = 9 ///< core-1-phase
};


///
/// Phase Sequence (DSS enumeration)
///
enum class PhaseSequence: int32_t
{
    Lag = 0, ///< Lag
    Lead = 1, ///< Lead
    ANSI = 0, ///< ANSI
    Euro = 1 ///< Euro
};


///
/// Load Solution Model (DSS enumeration)
///
enum class LoadSolutionModel: int32_t
{
    PowerFlow = 1, ///< PowerFlow
    Admittance = 2 ///< Admittance
};


///
/// Random Type (DSS enumeration)
///
enum class RandomType: int32_t
{
    none = 0, ///< None
    Gaussian = 1, ///< Gaussian
    Uniform = 2, ///< Uniform
    LogNormal = 3 ///< LogNormal
};


///
/// Control Mode (DSS enumeration)
///
enum class ControlMode: int32_t
{
    Off = -1, ///< Off
    Static = 0, ///< Static
    Event = 1, ///< Event
    Time = 2, ///< Time
    MultiRate = 3 ///< MultiRate
};


///
/// Solution Mode (DSS enumeration)
///
enum class SolutionMode: int32_t
{
    Snap = 0, ///< Snap
    Daily = 1, ///< Daily
    Yearly = 2, ///< Yearly
    M1 = 3, ///< M1
    LD1 = 4, ///< LD1
    PeakDay = 5, ///< PeakDay
    DutyCycle = 6, ///< DutyCycle
    Direct = 7, ///< Direct
    MF = 8, ///< MF
    FaultStudy = 9, ///< FaultStudy
    M2 = 10, ///< M2
    M3 = 11, ///< M3
    LD2 = 12, ///< LD2
    AutoAdd = 13, ///< AutoAdd
    Dynamic = 14, ///< Dynamic
    Harmonic = 15, ///< Harmonic
    Time = 16, ///< Time
    HarmonicT = 17, ///< HarmonicT
    Snapshot = 0 ///< Snapshot
};


///
/// Solution Algorithm (DSS enumeration)
///
enum class SolutionAlgorithm: int32_t
{
    Normal = 0, ///< Normal
    Newton = 1 ///< Newton
};


///
/// Circuit Model (DSS enumeration)
///
enum class CircuitModel: int32_t
{
    Multiphase = 0, ///< Multiphase
    Positive = 1 ///< Positive
};


///
/// AutoAdd Device Type (DSS enumeration)
///
enum class AutoAddDeviceType: int32_t
{
    Generator = 1, ///< Generator
    Capacitor = 2 ///< Capacitor
};


///
/// Load Shape Class (DSS enumeration)
///
enum class LoadShapeClass: int32_t
{
    none = -1, ///< None
    Daily = 0, ///< Daily
    Yearly = 1, ///< Yearly
    Duty = 2 ///< Duty
};


///
/// Monitored Phase (DSS enumeration)
///
enum class MonitoredPhase: int32_t
{
    min = -3, ///< min
    max = -2, ///< max
    avg = -1 ///< avg
};


class LineCode: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 1;
    struct Properties
    {
        enum {
            nphases = 1,
            r1 = 2,
            x1 = 3,
            r0 = 4,
            x0 = 5,
            C1 = 6,
            C0 = 7,
            units = 8,
            rmatrix = 9,
            xmatrix = 10,
            cmatrix = 11,
            baseFreq = 12,
            normamps = 13,
            emergamps = 14,
            faultrate = 15,
            pctperm = 16,
            repair = 17,
            Kron = 18,
            Rg = 19,
            Xg = 20,
            rho = 21,
            neutral = 22,
            B1 = 23,
            B0 = 24,
            Seasons = 25,
            Ratings = 26,
            LineType = 27,
            like = 28,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    LineCode(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    LineCode(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the LineCode element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    LineCode(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the LineCode element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    LineCode& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    LineCode& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of phases in the line this line code data represents.  Setting this property reinitializes the line code.  Impedance matrix is reset for default symmetrical component.
    ///
    /// DSS property name: nphases, DSS property index: 1
    ///
    int32_t nphases()
    {
        return Obj_GetInt32(ptr, Properties::nphases);
    }

    LineCode& nphases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::nphases, value);
        return *this;
    }

    /// Positive-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Rmatrix.
    ///
    /// DSS property name: r1, DSS property index: 2
    ///
    double r1()
    {
        return Obj_GetFloat64(ptr, Properties::r1);
    }

    LineCode& r1(double value)
    {
        Obj_SetFloat64(ptr, Properties::r1, value);
        return *this;
    }

    /// Positive-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Xmatrix
    ///
    /// DSS property name: x1, DSS property index: 3
    ///
    double x1()
    {
        return Obj_GetFloat64(ptr, Properties::x1);
    }

    LineCode& x1(double value)
    {
        Obj_SetFloat64(ptr, Properties::x1, value);
        return *this;
    }

    /// Zero-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.
    ///
    /// DSS property name: r0, DSS property index: 4
    ///
    double r0()
    {
        return Obj_GetFloat64(ptr, Properties::r0);
    }

    LineCode& r0(double value)
    {
        Obj_SetFloat64(ptr, Properties::r0, value);
        return *this;
    }

    /// Zero-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.
    ///
    /// DSS property name: x0, DSS property index: 5
    ///
    double x0()
    {
        return Obj_GetFloat64(ptr, Properties::x0);
    }

    LineCode& x0(double value)
    {
        Obj_SetFloat64(ptr, Properties::x0, value);
        return *this;
    }

    /// Positive-sequence capacitance, nf per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Cmatrix and B1.
    ///
    /// DSS property name: C1, DSS property index: 6
    ///
    double C1()
    {
        return Obj_GetFloat64(ptr, Properties::C1);
    }

    LineCode& C1(double value)
    {
        Obj_SetFloat64(ptr, Properties::C1, value);
        return *this;
    }

    /// Zero-sequence capacitance, nf per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also B0.
    ///
    /// DSS property name: C0, DSS property index: 7
    ///
    double C0()
    {
        return Obj_GetFloat64(ptr, Properties::C0);
    }

    LineCode& C0(double value)
    {
        Obj_SetFloat64(ptr, Properties::C0, value);
        return *this;
    }

    /// One of (ohms per ...) {none|mi|km|kft|m|me|ft|in|cm}.  Default is none; assumes units agree with length unitsgiven in Line object
    ///
    /// DSS property name: units, DSS property index: 8
    ///
    DimensionUnits units()
    {
        return DimensionUnits(Obj_GetInt32(ptr, Properties::units));
    }

    LineCode& units(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::units, value);
        return *this;
    }

    LineCode& units(DimensionUnits value)
    {
        Obj_SetInt32(ptr, Properties::units, int32_t(value));
        return *this;
    }

    LineCode& units(const string &value)
    {
        set_string(Properties::units, value);
        return *this;
    }

    LineCode& units(const char *value)
    {
        set_string(Properties::units, value);
        return *this;
    }

    /// One of (ohms per ...) {none|mi|km|kft|m|me|ft|in|cm}.  Default is none; assumes units agree with length unitsgiven in Line object
    ///
    /// DSS property name: units, DSS property index: 8
    ///
    string units_str()
    {
        return get_prop_string(Properties::units);
    }

    /// One of (ohms per ...) {none|mi|km|kft|m|me|ft|in|cm}.  Default is none; assumes units agree with length unitsgiven in Line object
    ///
    /// DSS property name: units, DSS property index: 8
    ///
    LineCode& units_str(const string &value)
    {
        set_string(Properties::units, value);
        return *this;
    }

    /// Resistance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration.  For balanced line models, you may use the standard symmetrical component data definition instead.
    ///
    /// DSS property name: rmatrix, DSS property index: 9
    ///
    VectorXd rmatrix()
    {
        return get_array<VectorXd>(Properties::rmatrix);
    }

    LineCode& rmatrix(VectorXd &value)
    {
        set_array<VectorXd>(Properties::rmatrix, value);
        return *this;
    }

    /// Reactance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration.  For balanced line models, you may use the standard symmetrical component data definition instead.
    ///
    /// DSS property name: xmatrix, DSS property index: 10
    ///
    VectorXd xmatrix()
    {
        return get_array<VectorXd>(Properties::xmatrix);
    }

    LineCode& xmatrix(VectorXd &value)
    {
        set_array<VectorXd>(Properties::xmatrix, value);
        return *this;
    }

    /// Nodal Capacitance matrix, lower triangle, nf per unit length.Order of the matrix is the number of phases. May be used to specify the shunt capacitance of any line configuration.  For balanced line models, you may use the standard symmetrical component data definition instead.
    ///
    /// DSS property name: cmatrix, DSS property index: 11
    ///
    VectorXd cmatrix()
    {
        return get_array<VectorXd>(Properties::cmatrix);
    }

    LineCode& cmatrix(VectorXd &value)
    {
        set_array<VectorXd>(Properties::cmatrix, value);
        return *this;
    }

    /// Frequency at which impedances are specified.
    ///
    /// DSS property name: baseFreq, DSS property index: 12
    ///
    double baseFreq()
    {
        return Obj_GetFloat64(ptr, Properties::baseFreq);
    }

    LineCode& baseFreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::baseFreq, value);
        return *this;
    }

    /// Normal ampere limit on line.  This is the so-called Planning Limit. It may also be the value above which load will have to be dropped in a contingency.  Usually about 75% - 80% of the emergency (one-hour) rating.
    ///
    /// DSS property name: normamps, DSS property index: 13
    ///
    double normamps()
    {
        return Obj_GetFloat64(ptr, Properties::normamps);
    }

    LineCode& normamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::normamps, value);
        return *this;
    }

    /// Emergency ampere limit on line (usually one-hour rating).
    ///
    /// DSS property name: emergamps, DSS property index: 14
    ///
    double emergamps()
    {
        return Obj_GetFloat64(ptr, Properties::emergamps);
    }

    LineCode& emergamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::emergamps, value);
        return *this;
    }

    /// Number of faults per unit length per year.
    ///
    /// DSS property name: faultrate, DSS property index: 15
    ///
    double faultrate()
    {
        return Obj_GetFloat64(ptr, Properties::faultrate);
    }

    LineCode& faultrate(double value)
    {
        Obj_SetFloat64(ptr, Properties::faultrate, value);
        return *this;
    }

    /// Percentage of the faults that become permanent.
    ///
    /// DSS property name: pctperm, DSS property index: 16
    ///
    double pctperm()
    {
        return Obj_GetFloat64(ptr, Properties::pctperm);
    }

    LineCode& pctperm(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctperm, value);
        return *this;
    }

    /// Hours to repair.
    ///
    /// DSS property name: repair, DSS property index: 17
    ///
    double repair()
    {
        return Obj_GetFloat64(ptr, Properties::repair);
    }

    LineCode& repair(double value)
    {
        Obj_SetFloat64(ptr, Properties::repair, value);
        return *this;
    }

    /// Kron = Y/N. Default=N.  Perform Kron reduction on the impedance matrix after it is formed, reducing order by 1. Eliminates the conductor designated by the "Neutral=" property. Do this after the R, X, and C matrices are defined. Ignored for symmetrical components. May be issued more than once to eliminate more than one conductor by resetting the Neutral property after the previous invoking of this property. Generally, you do not want to do a Kron reduction on the matrix if you intend to solve at a frequency other than the base frequency and exploit the Rg and Xg values.
    ///
    /// DSS property name: Kron, DSS property index: 18
    ///
    LineCode& Kron(bool value)
    {
        Obj_SetInt32(ptr, Properties::Kron, value);
        return *this;
    }

    /// Carson earth return resistance per unit length used to compute impedance values at base frequency.  For making better frequency adjustments. Default is 0.01805 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.
    ///
    /// DSS property name: Rg, DSS property index: 19
    ///
    double Rg()
    {
        return Obj_GetFloat64(ptr, Properties::Rg);
    }

    LineCode& Rg(double value)
    {
        Obj_SetFloat64(ptr, Properties::Rg, value);
        return *this;
    }

    /// Carson earth return reactance per unit length used to compute impedance values at base frequency.  For making better frequency adjustments. Default value is 0.155081 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.
    ///
    /// DSS property name: Xg, DSS property index: 20
    ///
    double Xg()
    {
        return Obj_GetFloat64(ptr, Properties::Xg);
    }

    LineCode& Xg(double value)
    {
        Obj_SetFloat64(ptr, Properties::Xg, value);
        return *this;
    }

    /// Default=100 meter ohms.  Earth resitivity used to compute earth correction factor.
    ///
    /// DSS property name: rho, DSS property index: 21
    ///
    double rho()
    {
        return Obj_GetFloat64(ptr, Properties::rho);
    }

    LineCode& rho(double value)
    {
        Obj_SetFloat64(ptr, Properties::rho, value);
        return *this;
    }

    /// Designates which conductor is the "neutral" conductor that will be eliminated by Kron reduction. Default is the last conductor (nphases value). After Kron reduction is set to 0. Subsequent issuing of Kron=Yes will not do anything until this property is set to a legal value. Applies only to LineCodes defined by R, X, and C matrix.
    ///
    /// DSS property name: neutral, DSS property index: 22
    ///
    int32_t neutral()
    {
        return Obj_GetInt32(ptr, Properties::neutral);
    }

    LineCode& neutral(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::neutral, value);
        return *this;
    }

    /// Alternate way to specify C1. MicroS per unit length
    ///
    /// DSS property name: B1, DSS property index: 23
    ///
    double B1()
    {
        return Obj_GetFloat64(ptr, Properties::B1);
    }

    LineCode& B1(double value)
    {
        Obj_SetFloat64(ptr, Properties::B1, value);
        return *this;
    }

    /// Alternate way to specify C0. MicroS per unit length
    ///
    /// DSS property name: B0, DSS property index: 24
    ///
    double B0()
    {
        return Obj_GetFloat64(ptr, Properties::B0);
    }

    LineCode& B0(double value)
    {
        Obj_SetFloat64(ptr, Properties::B0, value);
        return *this;
    }

    /// Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.
    ///
    /// DSS property name: Seasons, DSS property index: 25
    ///
    int32_t Seasons()
    {
        return Obj_GetInt32(ptr, Properties::Seasons);
    }

    LineCode& Seasons(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Seasons, value);
        return *this;
    }

    /// An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
    /// multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.
    ///
    /// DSS property name: Ratings, DSS property index: 26
    ///
    VectorXd Ratings()
    {
        return get_array<VectorXd>(Properties::Ratings);
    }

    LineCode& Ratings(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Ratings, value);
        return *this;
    }

    /// Code designating the type of line.
    /// One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW
    ///
    /// OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.
    ///
    /// DSS property name: LineType, DSS property index: 27
    ///
    LineType linetype()
    {
        return LineType(Obj_GetInt32(ptr, Properties::LineType));
    }

    LineCode& linetype(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::LineType, value);
        return *this;
    }

    LineCode& linetype(LineType value)
    {
        Obj_SetInt32(ptr, Properties::LineType, int32_t(value));
        return *this;
    }

    LineCode& linetype(const string &value)
    {
        set_string(Properties::LineType, value);
        return *this;
    }

    LineCode& linetype(const char *value)
    {
        set_string(Properties::LineType, value);
        return *this;
    }

    /// Code designating the type of line.
    /// One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW
    ///
    /// OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.
    ///
    /// DSS property name: LineType, DSS property index: 27
    ///
    string linetype_str()
    {
        return get_prop_string(Properties::LineType);
    }

    /// Code designating the type of line.
    /// One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW
    ///
    /// OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.
    ///
    /// DSS property name: LineType, DSS property index: 27
    ///
    LineCode& linetype_str(const string &value)
    {
        set_string(Properties::LineType, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 28
    ///
    LineCode& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 28
    ///
    LineCode& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class LoadShape: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 2;
    struct Properties
    {
        enum {
            npts = 1,
            interval = 2,
            mult = 3,
            hour = 4,
            mean = 5,
            stddev = 6,
            csvfile = 7,
            sngfile = 8,
            dblfile = 9,
            action = 10,
            qmult = 11,
            UseActual = 12,
            Pmax = 13,
            Qmax = 14,
            sinterval = 15,
            minterval = 16,
            Pbase = 17,
            Qbase = 18,
            Pmult = 19,
            PQCSVFile = 20,
            MemoryMapping = 21,
            like = 22,
        };
    };

    // Class-specific enumerations

        ///
        /// LoadShape: Action (DSS enumeration for LoadShape)
        ///
    enum class LoadShapeAction: int32_t
    {
        Normalize = 0, ///< Normalize
        DblSave = 1, ///< DblSave
        SngSave = 2 ///< SngSave
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    LoadShape(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    LoadShape(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the LoadShape element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    LoadShape(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the LoadShape element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    LoadShape& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    LoadShape& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Max number of points to expect in load shape vectors. This gets reset to the number of multiplier values found (in files only) if less than specified.
    ///
    /// DSS property name: npts, DSS property index: 1
    ///
    int32_t npts()
    {
        return Obj_GetInt32(ptr, Properties::npts);
    }

    LoadShape& npts(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::npts, value);
        return *this;
    }

    /// Time interval for fixed interval data, hrs. Default = 1. If Interval = 0 then time data (in hours) may be at either regular or  irregular intervals and time value must be specified using either the Hour property or input files. Then values are interpolated when Interval=0, but not for fixed interval data.
    ///
    /// See also "sinterval" and "minterval".
    ///
    /// DSS property name: interval, DSS property index: 2
    ///
    double interval()
    {
        return Obj_GetFloat64(ptr, Properties::interval);
    }

    LoadShape& interval(double value)
    {
        Obj_SetFloat64(ptr, Properties::interval, value);
        return *this;
    }

    /// Array of multiplier values for active power (P) or other key value (such as pu V for Vsource).
    ///
    /// You can also use the syntax:
    ///
    /// mult = (file=filename)     !for text file one value per line
    /// mult = (dblfile=filename)  !for packed file of doubles
    /// mult = (sngfile=filename)  !for packed file of singles
    /// mult = (file=MyCSVFile.CSV, col=3, header=yes)  !for multicolumn CSV files
    ///
    /// Note: this property will reset Npts if the  number of values in the files are fewer.
    ///
    /// Same as Pmult
    ///
    /// DSS property name: mult, DSS property index: 3
    ///
    VectorXd mult()
    {
        return get_array<VectorXd>(Properties::mult);
    }

    LoadShape& mult(VectorXd &value)
    {
        set_array<VectorXd>(Properties::mult, value);
        return *this;
    }

    /// Array of hour values. Only necessary to define for variable interval data (Interval=0). If you set Interval>0 to denote fixed interval data, DO NOT USE THIS PROPERTY. You can also use the syntax:
    /// hour = (file=filename)     !for text file one value per line
    /// hour = (dblfile=filename)  !for packed file of doubles
    /// hour = (sngfile=filename)  !for packed file of singles
    ///
    /// DSS property name: hour, DSS property index: 4
    ///
    VectorXd hour()
    {
        return get_array<VectorXd>(Properties::hour);
    }

    LoadShape& hour(VectorXd &value)
    {
        set_array<VectorXd>(Properties::hour, value);
        return *this;
    }

    /// Mean of the active power multipliers.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently. Used for Monte Carlo load simulations.
    ///
    /// DSS property name: mean, DSS property index: 5
    ///
    double mean()
    {
        return Obj_GetFloat64(ptr, Properties::mean);
    }

    LoadShape& mean(double value)
    {
        Obj_SetFloat64(ptr, Properties::mean, value);
        return *this;
    }

    /// Standard deviation of active power multipliers.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently.Is overwritten if you subsequently read in a curve
    ///
    /// Used for Monte Carlo load simulations.
    ///
    /// DSS property name: stddev, DSS property index: 6
    ///
    double stddev()
    {
        return Obj_GetFloat64(ptr, Properties::stddev);
    }

    LoadShape& stddev(double value)
    {
        Obj_SetFloat64(ptr, Properties::stddev, value);
        return *this;
    }

    /// Switch input of active power load curve data to a CSV text file containing (hour, mult) points, or simply (mult) values for fixed time interval data, one per line. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: csvfile, DSS property index: 7
    ///
    string csvfile()
    {
        return get_prop_string(Properties::csvfile);
    }

    LoadShape& csvfile(const string &value)
    {
        set_string(Properties::csvfile, value);
        return *this;
    }

    LoadShape& csvfile(const char* value)
    {
        set_string(Properties::csvfile, value);
        return *this;
    }

    /// Switch input of active power load curve data to a binary file of singles containing (hour, mult) points, or simply (mult) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: sngfile, DSS property index: 8
    ///
    string sngfile()
    {
        return get_prop_string(Properties::sngfile);
    }

    LoadShape& sngfile(const string &value)
    {
        set_string(Properties::sngfile, value);
        return *this;
    }

    LoadShape& sngfile(const char* value)
    {
        set_string(Properties::sngfile, value);
        return *this;
    }

    /// Switch input of active power load curve data to a binary file of doubles containing (hour, mult) points, or simply (mult) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: dblfile, DSS property index: 9
    ///
    string dblfile()
    {
        return get_prop_string(Properties::dblfile);
    }

    LoadShape& dblfile(const string &value)
    {
        set_string(Properties::dblfile, value);
        return *this;
    }

    LoadShape& dblfile(const char* value)
    {
        set_string(Properties::dblfile, value);
        return *this;
    }

    /// {NORMALIZE | DblSave | SngSave} After defining load curve data, setting action=normalize will modify the multipliers so that the peak is 1.0. The mean and std deviation are recomputed.
    ///
    /// Setting action=DblSave or SngSave will cause the present mult and qmult values to be written to either a packed file of double or single. The filename is the loadshape name. The mult array will have a "_P" appended on the file name and the qmult array, if it exists, will have "_Q" appended.
    ///
    /// DSS property name: action, DSS property index: 10
    ///
    LoadShape& action(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::action, value);
        return *this;
    }

    /// {NORMALIZE | DblSave | SngSave} After defining load curve data, setting action=normalize will modify the multipliers so that the peak is 1.0. The mean and std deviation are recomputed.
    ///
    /// Setting action=DblSave or SngSave will cause the present mult and qmult values to be written to either a packed file of double or single. The filename is the loadshape name. The mult array will have a "_P" appended on the file name and the qmult array, if it exists, will have "_Q" appended.
    ///
    /// DSS property name: action, DSS property index: 10
    ///
    LoadShape& action(LoadShapeAction value)
    {
        Obj_SetInt32(ptr, Properties::action, int32_t(value));
        return *this;
    }

    /// {NORMALIZE | DblSave | SngSave} After defining load curve data, setting action=normalize will modify the multipliers so that the peak is 1.0. The mean and std deviation are recomputed.
    ///
    /// Setting action=DblSave or SngSave will cause the present mult and qmult values to be written to either a packed file of double or single. The filename is the loadshape name. The mult array will have a "_P" appended on the file name and the qmult array, if it exists, will have "_Q" appended.
    ///
    /// DSS property name: action, DSS property index: 10
    ///
    LoadShape& action(const string &value)
    {
        set_string(Properties::action, value);
        return *this;
    }

    /// {NORMALIZE | DblSave | SngSave} After defining load curve data, setting action=normalize will modify the multipliers so that the peak is 1.0. The mean and std deviation are recomputed.
    ///
    /// Setting action=DblSave or SngSave will cause the present mult and qmult values to be written to either a packed file of double or single. The filename is the loadshape name. The mult array will have a "_P" appended on the file name and the qmult array, if it exists, will have "_Q" appended.
    ///
    /// DSS property name: action, DSS property index: 10
    ///
    LoadShape& action(const char *value)
    {
        set_string(Properties::action, value);
        return *this;
    }

    /// Array of multiplier values for reactive power (Q).  You can also use the syntax:
    /// qmult = (file=filename)     !for text file one value per line
    /// qmult = (dblfile=filename)  !for packed file of doubles
    /// qmult = (sngfile=filename)  !for packed file of singles
    /// qmult = (file=MyCSVFile.CSV, col=4, header=yes)  !for multicolumn CSV files
    ///
    /// DSS property name: qmult, DSS property index: 11
    ///
    VectorXd qmult()
    {
        return get_array<VectorXd>(Properties::qmult);
    }

    LoadShape& qmult(VectorXd &value)
    {
        set_array<VectorXd>(Properties::qmult, value);
        return *this;
    }

    /// {Yes | No* | True | False*} If true, signifies to Load, Generator, Vsource, or other objects to use the return value as the actual kW, kvar, kV, or other value rather than a multiplier. Nominally for AMI Load data but may be used for other functions.
    ///
    /// DSS property name: UseActual, DSS property index: 12
    ///
    bool UseActual()
    {
        return Obj_GetInt32(ptr, Properties::UseActual) != 0;
    }

    LoadShape& UseActual(bool value)
    {
        Obj_SetInt32(ptr, Properties::UseActual, value);
        return *this;
    }

    /// kW value at the time of max power. Is automatically set upon reading in a loadshape. Use this property to override the value automatically computed or to retrieve the value computed.
    ///
    /// DSS property name: Pmax, DSS property index: 13
    ///
    double Pmax()
    {
        return Obj_GetFloat64(ptr, Properties::Pmax);
    }

    LoadShape& Pmax(double value)
    {
        Obj_SetFloat64(ptr, Properties::Pmax, value);
        return *this;
    }

    /// kvar value at the time of max kW power. Is automatically set upon reading in a loadshape. Use this property to override the value automatically computed or to retrieve the value computed.
    ///
    /// DSS property name: Qmax, DSS property index: 14
    ///
    double Qmax()
    {
        return Obj_GetFloat64(ptr, Properties::Qmax);
    }

    LoadShape& Qmax(double value)
    {
        Obj_SetFloat64(ptr, Properties::Qmax, value);
        return *this;
    }

    /// Specify fixed interval in SECONDS. Alternate way to specify Interval property.
    ///
    /// DSS property name: sinterval, DSS property index: 15
    ///
    double sinterval()
    {
        return Obj_GetFloat64(ptr, Properties::sinterval);
    }

    LoadShape& sinterval(double value)
    {
        Obj_SetFloat64(ptr, Properties::sinterval, value);
        return *this;
    }

    /// Specify fixed interval in MINUTES. Alternate way to specify Interval property.
    ///
    /// DSS property name: minterval, DSS property index: 16
    ///
    double minterval()
    {
        return Obj_GetFloat64(ptr, Properties::minterval);
    }

    LoadShape& minterval(double value)
    {
        Obj_SetFloat64(ptr, Properties::minterval, value);
        return *this;
    }

    /// Base P value for normalization. Default is zero, meaning the peak will be used.
    ///
    /// DSS property name: Pbase, DSS property index: 17
    ///
    double Pbase()
    {
        return Obj_GetFloat64(ptr, Properties::Pbase);
    }

    LoadShape& Pbase(double value)
    {
        Obj_SetFloat64(ptr, Properties::Pbase, value);
        return *this;
    }

    /// Base Q value for normalization. Default is zero, meaning the peak will be used.
    ///
    /// DSS property name: Qbase, DSS property index: 18
    ///
    double Qbase()
    {
        return Obj_GetFloat64(ptr, Properties::Qbase);
    }

    LoadShape& Qbase(double value)
    {
        Obj_SetFloat64(ptr, Properties::Qbase, value);
        return *this;
    }

    /// Synonym for "mult".
    ///
    /// DSS property name: Pmult, DSS property index: 19
    ///
    VectorXd Pmult()
    {
        return get_array<VectorXd>(Properties::Pmult);
    }

    LoadShape& Pmult(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Pmult, value);
        return *this;
    }

    /// Switch input to a CSV text file containing (active, reactive) power (P, Q) multiplier pairs, one per row.
    /// If the interval=0, there should be 3 items on each line: (hour, Pmult, Qmult)
    ///
    /// DSS property name: PQCSVFile, DSS property index: 20
    ///
    string PQCSVFile()
    {
        return get_prop_string(Properties::PQCSVFile);
    }

    LoadShape& PQCSVFile(const string &value)
    {
        set_string(Properties::PQCSVFile, value);
        return *this;
    }

    LoadShape& PQCSVFile(const char* value)
    {
        set_string(Properties::PQCSVFile, value);
        return *this;
    }

    /// {Yes | No* | True | False*} Enables the memory mapping functionality for dealing with large amounts of load shapes.
    /// By defaul is False. Use it to accelerate the model loading when the containing a large number of load shapes.
    ///
    /// DSS property name: MemoryMapping, DSS property index: 21
    ///
    bool MemoryMapping()
    {
        return Obj_GetInt32(ptr, Properties::MemoryMapping) != 0;
    }

    LoadShape& MemoryMapping(bool value)
    {
        Obj_SetInt32(ptr, Properties::MemoryMapping, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 22
    ///
    LoadShape& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 22
    ///
    LoadShape& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class TShape: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 3;
    struct Properties
    {
        enum {
            npts = 1,
            interval = 2,
            temp = 3,
            hour = 4,
            mean = 5,
            stddev = 6,
            csvfile = 7,
            sngfile = 8,
            dblfile = 9,
            sinterval = 10,
            minterval = 11,
            action = 12,
            like = 13,
        };
    };

    // Class-specific enumerations

        ///
        /// TShape: Action (DSS enumeration for TShape)
        ///
    enum class TShapeAction: int32_t
    {
        DblSave = 0, ///< DblSave
        SngSave = 1 ///< SngSave
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    TShape(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    TShape(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the TShape element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    TShape(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the TShape element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    TShape& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    TShape& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Max number of points to expect in temperature shape vectors. This gets reset to the number of Temperature values found if less than specified.
    ///
    /// DSS property name: npts, DSS property index: 1
    ///
    int32_t npts()
    {
        return Obj_GetInt32(ptr, Properties::npts);
    }

    TShape& npts(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::npts, value);
        return *this;
    }

    /// Time interval for fixed interval data, hrs. Default = 1. If Interval = 0 then time data (in hours) may be at irregular intervals and time value must be specified using either the Hour property or input files. Then values are interpolated when Interval=0, but not for fixed interval data.
    ///
    /// See also "sinterval" and "minterval".
    ///
    /// DSS property name: interval, DSS property index: 2
    ///
    double interval()
    {
        return Obj_GetFloat64(ptr, Properties::interval);
    }

    TShape& interval(double value)
    {
        Obj_SetFloat64(ptr, Properties::interval, value);
        return *this;
    }

    /// Array of temperature values.  Units should be compatible with the object using the data. You can also use the syntax:
    /// Temp = (file=filename)     !for text file one value per line
    /// Temp = (dblfile=filename)  !for packed file of doubles
    /// Temp = (sngfile=filename)  !for packed file of singles
    ///
    /// Note: this property will reset Npts if the  number of values in the files are fewer.
    ///
    /// DSS property name: temp, DSS property index: 3
    ///
    VectorXd temp()
    {
        return get_array<VectorXd>(Properties::temp);
    }

    TShape& temp(VectorXd &value)
    {
        set_array<VectorXd>(Properties::temp, value);
        return *this;
    }

    /// Array of hour values. Only necessary to define this property for variable interval data. If the data are fixed interval, do not use this property. You can also use the syntax:
    /// hour = (file=filename)     !for text file one value per line
    /// hour = (dblfile=filename)  !for packed file of doubles
    /// hour = (sngfile=filename)  !for packed file of singles
    ///
    /// DSS property name: hour, DSS property index: 4
    ///
    VectorXd hour()
    {
        return get_array<VectorXd>(Properties::hour);
    }

    TShape& hour(VectorXd &value)
    {
        set_array<VectorXd>(Properties::hour, value);
        return *this;
    }

    /// Mean of the temperature curve values.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently. Used for Monte Carlo load simulations.
    ///
    /// DSS property name: mean, DSS property index: 5
    ///
    double mean()
    {
        return Obj_GetFloat64(ptr, Properties::mean);
    }

    TShape& mean(double value)
    {
        Obj_SetFloat64(ptr, Properties::mean, value);
        return *this;
    }

    /// Standard deviation of the temperatures.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently.Is overwritten if you subsequently read in a curve
    ///
    /// Used for Monte Carlo load simulations.
    ///
    /// DSS property name: stddev, DSS property index: 6
    ///
    double stddev()
    {
        return Obj_GetFloat64(ptr, Properties::stddev);
    }

    TShape& stddev(double value)
    {
        Obj_SetFloat64(ptr, Properties::stddev, value);
        return *this;
    }

    /// Switch input of  temperature curve data to a csv file containing (hour, Temp) points, or simply (Temp) values for fixed time interval data, one per line. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: csvfile, DSS property index: 7
    ///
    string csvfile()
    {
        return get_prop_string(Properties::csvfile);
    }

    TShape& csvfile(const string &value)
    {
        set_string(Properties::csvfile, value);
        return *this;
    }

    TShape& csvfile(const char* value)
    {
        set_string(Properties::csvfile, value);
        return *this;
    }

    /// Switch input of  temperature curve data to a binary file of singles containing (hour, Temp) points, or simply (Temp) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: sngfile, DSS property index: 8
    ///
    string sngfile()
    {
        return get_prop_string(Properties::sngfile);
    }

    TShape& sngfile(const string &value)
    {
        set_string(Properties::sngfile, value);
        return *this;
    }

    TShape& sngfile(const char* value)
    {
        set_string(Properties::sngfile, value);
        return *this;
    }

    /// Switch input of  temperature curve data to a binary file of doubles containing (hour, Temp) points, or simply (Temp) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: dblfile, DSS property index: 9
    ///
    string dblfile()
    {
        return get_prop_string(Properties::dblfile);
    }

    TShape& dblfile(const string &value)
    {
        set_string(Properties::dblfile, value);
        return *this;
    }

    TShape& dblfile(const char* value)
    {
        set_string(Properties::dblfile, value);
        return *this;
    }

    /// Specify fixed interval in SECONDS. Alternate way to specify Interval property.
    ///
    /// DSS property name: sinterval, DSS property index: 10
    ///
    double sinterval()
    {
        return Obj_GetFloat64(ptr, Properties::sinterval);
    }

    TShape& sinterval(double value)
    {
        Obj_SetFloat64(ptr, Properties::sinterval, value);
        return *this;
    }

    /// Specify fixed interval in MINUTES. Alternate way to specify Interval property.
    ///
    /// DSS property name: minterval, DSS property index: 11
    ///
    double minterval()
    {
        return Obj_GetFloat64(ptr, Properties::minterval);
    }

    TShape& minterval(double value)
    {
        Obj_SetFloat64(ptr, Properties::minterval, value);
        return *this;
    }

    /// {DblSave | SngSave} After defining temperature curve data... Setting action=DblSave or SngSave will cause the present "Temp" values to be written to either a packed file of double or single. The filename is the Tshape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    TShape& action(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::action, value);
        return *this;
    }

    /// {DblSave | SngSave} After defining temperature curve data... Setting action=DblSave or SngSave will cause the present "Temp" values to be written to either a packed file of double or single. The filename is the Tshape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    TShape& action(TShapeAction value)
    {
        Obj_SetInt32(ptr, Properties::action, int32_t(value));
        return *this;
    }

    /// {DblSave | SngSave} After defining temperature curve data... Setting action=DblSave or SngSave will cause the present "Temp" values to be written to either a packed file of double or single. The filename is the Tshape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    TShape& action(const string &value)
    {
        set_string(Properties::action, value);
        return *this;
    }

    /// {DblSave | SngSave} After defining temperature curve data... Setting action=DblSave or SngSave will cause the present "Temp" values to be written to either a packed file of double or single. The filename is the Tshape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    TShape& action(const char *value)
    {
        set_string(Properties::action, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 13
    ///
    TShape& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 13
    ///
    TShape& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class PriceShape: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 4;
    struct Properties
    {
        enum {
            npts = 1,
            interval = 2,
            price = 3,
            hour = 4,
            mean = 5,
            stddev = 6,
            csvfile = 7,
            sngfile = 8,
            dblfile = 9,
            sinterval = 10,
            minterval = 11,
            action = 12,
            like = 13,
        };
    };

    // Class-specific enumerations

        ///
        /// PriceShape: Action (DSS enumeration for PriceShape)
        ///
    enum class PriceShapeAction: int32_t
    {
        DblSave = 0, ///< DblSave
        SngSave = 1 ///< SngSave
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    PriceShape(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    PriceShape(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the PriceShape element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    PriceShape(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the PriceShape element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    PriceShape& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    PriceShape& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Max number of points to expect in price shape vectors. This gets reset to the number of Price values found if less than specified.
    ///
    /// DSS property name: npts, DSS property index: 1
    ///
    int32_t npts()
    {
        return Obj_GetInt32(ptr, Properties::npts);
    }

    PriceShape& npts(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::npts, value);
        return *this;
    }

    /// Time interval for fixed interval data, hrs. Default = 1. If Interval = 0 then time data (in hours) may be at irregular intervals and time value must be specified using either the Hour property or input files. Then values are interpolated when Interval=0, but not for fixed interval data.
    ///
    /// See also "sinterval" and "minterval".
    ///
    /// DSS property name: interval, DSS property index: 2
    ///
    double interval()
    {
        return Obj_GetFloat64(ptr, Properties::interval);
    }

    PriceShape& interval(double value)
    {
        Obj_SetFloat64(ptr, Properties::interval, value);
        return *this;
    }

    /// Array of Price values.  Units should be compatible with the object using the data. You can also use the syntax:
    /// Price = (file=filename)     !for text file one value per line
    /// Price = (dblfile=filename)  !for packed file of doubles
    /// Price = (sngfile=filename)  !for packed file of singles
    ///
    /// Note: this property will reset Npts if the  number of values in the files are fewer.
    ///
    /// DSS property name: price, DSS property index: 3
    ///
    VectorXd price()
    {
        return get_array<VectorXd>(Properties::price);
    }

    PriceShape& price(VectorXd &value)
    {
        set_array<VectorXd>(Properties::price, value);
        return *this;
    }

    /// Array of hour values. Only necessary to define this property for variable interval data. If the data are fixed interval, do not use this property. You can also use the syntax:
    /// hour = (file=filename)     !for text file one value per line
    /// hour = (dblfile=filename)  !for packed file of doubles
    /// hour = (sngfile=filename)  !for packed file of singles
    ///
    /// DSS property name: hour, DSS property index: 4
    ///
    VectorXd hour()
    {
        return get_array<VectorXd>(Properties::hour);
    }

    PriceShape& hour(VectorXd &value)
    {
        set_array<VectorXd>(Properties::hour, value);
        return *this;
    }

    /// Mean of the Price curve values.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently. Used for Monte Carlo load simulations.
    ///
    /// DSS property name: mean, DSS property index: 5
    ///
    double mean()
    {
        return Obj_GetFloat64(ptr, Properties::mean);
    }

    PriceShape& mean(double value)
    {
        Obj_SetFloat64(ptr, Properties::mean, value);
        return *this;
    }

    /// Standard deviation of the Prices.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently.Is overwritten if you subsequently read in a curve
    ///
    /// Used for Monte Carlo load simulations.
    ///
    /// DSS property name: stddev, DSS property index: 6
    ///
    double stddev()
    {
        return Obj_GetFloat64(ptr, Properties::stddev);
    }

    PriceShape& stddev(double value)
    {
        Obj_SetFloat64(ptr, Properties::stddev, value);
        return *this;
    }

    /// Switch input of  Price curve data to a csv file containing (hour, Price) points, or simply (Price) values for fixed time interval data, one per line. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: csvfile, DSS property index: 7
    ///
    string csvfile()
    {
        return get_prop_string(Properties::csvfile);
    }

    PriceShape& csvfile(const string &value)
    {
        set_string(Properties::csvfile, value);
        return *this;
    }

    PriceShape& csvfile(const char* value)
    {
        set_string(Properties::csvfile, value);
        return *this;
    }

    /// Switch input of  Price curve data to a binary file of singles containing (hour, Price) points, or simply (Price) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: sngfile, DSS property index: 8
    ///
    string sngfile()
    {
        return get_prop_string(Properties::sngfile);
    }

    PriceShape& sngfile(const string &value)
    {
        set_string(Properties::sngfile, value);
        return *this;
    }

    PriceShape& sngfile(const char* value)
    {
        set_string(Properties::sngfile, value);
        return *this;
    }

    /// Switch input of  Price curve data to a binary file of doubles containing (hour, Price) points, or simply (Price) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: dblfile, DSS property index: 9
    ///
    string dblfile()
    {
        return get_prop_string(Properties::dblfile);
    }

    PriceShape& dblfile(const string &value)
    {
        set_string(Properties::dblfile, value);
        return *this;
    }

    PriceShape& dblfile(const char* value)
    {
        set_string(Properties::dblfile, value);
        return *this;
    }

    /// Specify fixed interval in SECONDS. Alternate way to specify Interval property.
    ///
    /// DSS property name: sinterval, DSS property index: 10
    ///
    double sinterval()
    {
        return Obj_GetFloat64(ptr, Properties::sinterval);
    }

    PriceShape& sinterval(double value)
    {
        Obj_SetFloat64(ptr, Properties::sinterval, value);
        return *this;
    }

    /// Specify fixed interval in MINUTES. Alternate way to specify Interval property.
    ///
    /// DSS property name: minterval, DSS property index: 11
    ///
    double minterval()
    {
        return Obj_GetFloat64(ptr, Properties::minterval);
    }

    PriceShape& minterval(double value)
    {
        Obj_SetFloat64(ptr, Properties::minterval, value);
        return *this;
    }

    /// {DblSave | SngSave} After defining Price curve data... Setting action=DblSave or SngSave will cause the present "Price" values to be written to either a packed file of double or single. The filename is the PriceShape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    PriceShape& action(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::action, value);
        return *this;
    }

    /// {DblSave | SngSave} After defining Price curve data... Setting action=DblSave or SngSave will cause the present "Price" values to be written to either a packed file of double or single. The filename is the PriceShape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    PriceShape& action(PriceShapeAction value)
    {
        Obj_SetInt32(ptr, Properties::action, int32_t(value));
        return *this;
    }

    /// {DblSave | SngSave} After defining Price curve data... Setting action=DblSave or SngSave will cause the present "Price" values to be written to either a packed file of double or single. The filename is the PriceShape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    PriceShape& action(const string &value)
    {
        set_string(Properties::action, value);
        return *this;
    }

    /// {DblSave | SngSave} After defining Price curve data... Setting action=DblSave or SngSave will cause the present "Price" values to be written to either a packed file of double or single. The filename is the PriceShape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    PriceShape& action(const char *value)
    {
        set_string(Properties::action, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 13
    ///
    PriceShape& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 13
    ///
    PriceShape& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class XYcurve: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 5;
    struct Properties
    {
        enum {
            npts = 1,
            Points = 2,
            Yarray = 3,
            Xarray = 4,
            csvfile = 5,
            sngfile = 6,
            dblfile = 7,
            x = 8,
            y = 9,
            Xshift = 10,
            Yshift = 11,
            Xscale = 12,
            Yscale = 13,
            like = 14,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    XYcurve(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    XYcurve(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the XYcurve element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    XYcurve(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the XYcurve element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    XYcurve& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    XYcurve& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Max number of points to expect in curve. This could get reset to the actual number of points defined if less than specified.
    ///
    /// DSS property name: npts, DSS property index: 1
    ///
    int32_t npts()
    {
        return Obj_GetInt32(ptr, Properties::npts);
    }

    XYcurve& npts(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::npts, value);
        return *this;
    }

    /// One way to enter the points in a curve. Enter x and y values as one array in the order [x1, y1, x2, y2, ...]. For example:
    ///
    /// Points=[1,100 2,200 3, 300]
    ///
    /// Values separated by commas or white space. Zero fills arrays if insufficient number of values.
    ///
    /// DSS property name: Points, DSS property index: 2
    ///
    VectorXd Points()
    {
        return get_array<VectorXd>(Properties::Points);
    }

    XYcurve& Points(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Points, value);
        return *this;
    }

    /// Alternate way to enter Y values. Enter an array of Y values corresponding to the X values.  You can also use the syntax:
    /// Yarray = (file=filename)     !for text file one value per line
    /// Yarray = (dblfile=filename)  !for packed file of doubles
    /// Yarray = (sngfile=filename)  !for packed file of singles
    ///
    /// Note: this property will reset Npts to a smaller value if the  number of values in the files are fewer.
    ///
    /// DSS property name: Yarray, DSS property index: 3
    ///
    VectorXd Yarray()
    {
        return get_array<VectorXd>(Properties::Yarray);
    }

    XYcurve& Yarray(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Yarray, value);
        return *this;
    }

    /// Alternate way to enter X values. Enter an array of X values corresponding to the Y values.  You can also use the syntax:
    /// Xarray = (file=filename)     !for text file one value per line
    /// Xarray = (dblfile=filename)  !for packed file of doubles
    /// Xarray = (sngfile=filename)  !for packed file of singles
    ///
    /// Note: this property will reset Npts to a smaller value if the  number of values in the files are fewer.
    ///
    /// DSS property name: Xarray, DSS property index: 4
    ///
    VectorXd Xarray()
    {
        return get_array<VectorXd>(Properties::Xarray);
    }

    XYcurve& Xarray(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Xarray, value);
        return *this;
    }

    /// Switch input of  X-Y curve data to a CSV file containing X, Y points one per line. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: csvfile, DSS property index: 5
    ///
    string csvfile()
    {
        return get_prop_string(Properties::csvfile);
    }

    XYcurve& csvfile(const string &value)
    {
        set_string(Properties::csvfile, value);
        return *this;
    }

    XYcurve& csvfile(const char* value)
    {
        set_string(Properties::csvfile, value);
        return *this;
    }

    /// Switch input of  X-Y curve data to a binary file of SINGLES containing X, Y points packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: sngfile, DSS property index: 6
    ///
    string sngfile()
    {
        return get_prop_string(Properties::sngfile);
    }

    XYcurve& sngfile(const string &value)
    {
        set_string(Properties::sngfile, value);
        return *this;
    }

    XYcurve& sngfile(const char* value)
    {
        set_string(Properties::sngfile, value);
        return *this;
    }

    /// Switch input of  X-Y  curve data to a binary file of DOUBLES containing X, Y points packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: dblfile, DSS property index: 7
    ///
    string dblfile()
    {
        return get_prop_string(Properties::dblfile);
    }

    XYcurve& dblfile(const string &value)
    {
        set_string(Properties::dblfile, value);
        return *this;
    }

    XYcurve& dblfile(const char* value)
    {
        set_string(Properties::dblfile, value);
        return *this;
    }

    /// Enter a value and then retrieve the interpolated Y value from the Y property. On input shifted then scaled to original curve. Scaled then shifted on output.
    ///
    /// DSS property name: x, DSS property index: 8
    ///
    double x()
    {
        return Obj_GetFloat64(ptr, Properties::x);
    }

    XYcurve& x(double value)
    {
        Obj_SetFloat64(ptr, Properties::x, value);
        return *this;
    }

    /// Enter a value and then retrieve the interpolated X value from the X property. On input shifted then scaled to original curve. Scaled then shifted on output.
    ///
    /// DSS property name: y, DSS property index: 9
    ///
    double y()
    {
        return Obj_GetFloat64(ptr, Properties::y);
    }

    XYcurve& y(double value)
    {
        Obj_SetFloat64(ptr, Properties::y, value);
        return *this;
    }

    /// Shift X property values (in/out) by this amount of offset. Default = 0. Does not change original definition of arrays.
    ///
    /// DSS property name: Xshift, DSS property index: 10
    ///
    double Xshift()
    {
        return Obj_GetFloat64(ptr, Properties::Xshift);
    }

    XYcurve& Xshift(double value)
    {
        Obj_SetFloat64(ptr, Properties::Xshift, value);
        return *this;
    }

    /// Shift Y property values (in/out) by this amount of offset. Default = 0. Does not change original definition of arrays.
    ///
    /// DSS property name: Yshift, DSS property index: 11
    ///
    double Yshift()
    {
        return Obj_GetFloat64(ptr, Properties::Yshift);
    }

    XYcurve& Yshift(double value)
    {
        Obj_SetFloat64(ptr, Properties::Yshift, value);
        return *this;
    }

    /// Scale X property values (in/out) by this factor. Default = 1.0. Does not change original definition of arrays.
    ///
    /// DSS property name: Xscale, DSS property index: 12
    ///
    double Xscale()
    {
        return Obj_GetFloat64(ptr, Properties::Xscale);
    }

    XYcurve& Xscale(double value)
    {
        Obj_SetFloat64(ptr, Properties::Xscale, value);
        return *this;
    }

    /// Scale Y property values (in/out) by this factor. Default = 1.0. Does not change original definition of arrays.
    ///
    /// DSS property name: Yscale, DSS property index: 13
    ///
    double Yscale()
    {
        return Obj_GetFloat64(ptr, Properties::Yscale);
    }

    XYcurve& Yscale(double value)
    {
        Obj_SetFloat64(ptr, Properties::Yscale, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 14
    ///
    XYcurve& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 14
    ///
    XYcurve& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class GrowthShape: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 6;
    struct Properties
    {
        enum {
            npts = 1,
            year = 2,
            mult = 3,
            csvfile = 4,
            sngfile = 5,
            dblfile = 6,
            like = 7,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    GrowthShape(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    GrowthShape(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the GrowthShape element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    GrowthShape(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the GrowthShape element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    GrowthShape& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    GrowthShape& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of points to expect in subsequent vector.
    ///
    /// DSS property name: npts, DSS property index: 1
    ///
    int32_t npts()
    {
        return Obj_GetInt32(ptr, Properties::npts);
    }

    GrowthShape& npts(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::npts, value);
        return *this;
    }

    /// Array of year values, or a text file spec, corresponding to the multipliers. Enter only those years where the growth changes. May be any integer sequence -- just so it is consistent. See help on Mult.
    ///
    /// DSS property name: year, DSS property index: 2
    ///
    VectorXd year()
    {
        return get_array<VectorXd>(Properties::year);
    }

    GrowthShape& year(VectorXd &value)
    {
        set_array<VectorXd>(Properties::year, value);
        return *this;
    }

    /// Array of growth multiplier values, or a text file spec, corresponding to the year values. Enter the multiplier by which you would multiply the previous year's load to get the present year's.
    ///
    /// Examples:
    ///
    ///   Year = [1, 2, 5]   Mult=[1.05, 1.025, 1.02].
    ///   Year= (File=years.txt) Mult= (file=mults.txt).
    ///
    /// Text files contain one value per line.
    ///
    /// DSS property name: mult, DSS property index: 3
    ///
    VectorXd mult()
    {
        return get_array<VectorXd>(Properties::mult);
    }

    GrowthShape& mult(VectorXd &value)
    {
        set_array<VectorXd>(Properties::mult, value);
        return *this;
    }

    /// Switch input of growth curve data to a csv file containing (year, mult) points, one per line.
    ///
    /// DSS property name: csvfile, DSS property index: 4
    ///
    string csvfile()
    {
        return get_prop_string(Properties::csvfile);
    }

    GrowthShape& csvfile(const string &value)
    {
        set_string(Properties::csvfile, value);
        return *this;
    }

    GrowthShape& csvfile(const char* value)
    {
        set_string(Properties::csvfile, value);
        return *this;
    }

    /// Switch input of growth curve data to a binary file of singles containing (year, mult) points, packed one after another.
    ///
    /// DSS property name: sngfile, DSS property index: 5
    ///
    string sngfile()
    {
        return get_prop_string(Properties::sngfile);
    }

    GrowthShape& sngfile(const string &value)
    {
        set_string(Properties::sngfile, value);
        return *this;
    }

    GrowthShape& sngfile(const char* value)
    {
        set_string(Properties::sngfile, value);
        return *this;
    }

    /// Switch input of growth curve data to a binary file of doubles containing (year, mult) points, packed one after another.
    ///
    /// DSS property name: dblfile, DSS property index: 6
    ///
    string dblfile()
    {
        return get_prop_string(Properties::dblfile);
    }

    GrowthShape& dblfile(const string &value)
    {
        set_string(Properties::dblfile, value);
        return *this;
    }

    GrowthShape& dblfile(const char* value)
    {
        set_string(Properties::dblfile, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 7
    ///
    GrowthShape& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 7
    ///
    GrowthShape& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class TCC_Curve: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 7;
    struct Properties
    {
        enum {
            npts = 1,
            C_array = 2,
            T_array = 3,
            like = 4,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    TCC_Curve(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    TCC_Curve(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the TCC_Curve element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    TCC_Curve(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the TCC_Curve element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    TCC_Curve& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    TCC_Curve& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of points to expect in time-current arrays.
    ///
    /// DSS property name: npts, DSS property index: 1
    ///
    int32_t npts()
    {
        return Obj_GetInt32(ptr, Properties::npts);
    }

    TCC_Curve& npts(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::npts, value);
        return *this;
    }

    /// Array of current (or voltage) values corresponding to time values (see help on T_Array).
    ///
    /// DSS property name: C_array, DSS property index: 2
    ///
    VectorXd C_array()
    {
        return get_array<VectorXd>(Properties::C_array);
    }

    TCC_Curve& C_array(VectorXd &value)
    {
        set_array<VectorXd>(Properties::C_array, value);
        return *this;
    }

    /// Array of time values in sec. Typical array syntax:
    /// t_array = (1, 2, 3, 4, ...)
    ///
    /// Can also substitute a file designation:
    /// t_array =  (file=filename)
    ///
    /// The specified file has one value per line.
    ///
    /// DSS property name: T_array, DSS property index: 3
    ///
    VectorXd T_array()
    {
        return get_array<VectorXd>(Properties::T_array);
    }

    TCC_Curve& T_array(VectorXd &value)
    {
        set_array<VectorXd>(Properties::T_array, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 4
    ///
    TCC_Curve& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 4
    ///
    TCC_Curve& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Spectrum: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 8;
    struct Properties
    {
        enum {
            NumHarm = 1,
            harmonic = 2,
            pctmag = 3,
            angle = 4,
            CSVFile = 5,
            like = 6,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Spectrum(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Spectrum(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Spectrum element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Spectrum(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Spectrum element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Spectrum& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Spectrum& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of frequencies in this spectrum. (See CSVFile)
    ///
    /// DSS property name: NumHarm, DSS property index: 1
    ///
    int32_t NumHarm()
    {
        return Obj_GetInt32(ptr, Properties::NumHarm);
    }

    Spectrum& NumHarm(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::NumHarm, value);
        return *this;
    }

    /// Array of harmonic values. You can also use the syntax
    /// harmonic = (file=filename)     !for text file one value per line
    /// harmonic = (dblfile=filename)  !for packed file of doubles
    /// harmonic = (sngfile=filename)  !for packed file of singles
    ///
    /// DSS property name: harmonic, DSS property index: 2
    ///
    VectorXd harmonic()
    {
        return get_array<VectorXd>(Properties::harmonic);
    }

    Spectrum& harmonic(VectorXd &value)
    {
        set_array<VectorXd>(Properties::harmonic, value);
        return *this;
    }

    /// Array of magnitude values, assumed to be in PERCENT. You can also use the syntax
    /// %mag = (file=filename)     !for text file one value per line
    /// %mag = (dblfile=filename)  !for packed file of doubles
    /// %mag = (sngfile=filename)  !for packed file of singles
    ///
    /// DSS property name: %mag, DSS property index: 3
    ///
    VectorXd pctmag()
    {
        return get_array<VectorXd>(Properties::pctmag);
    }

    Spectrum& pctmag(VectorXd &value)
    {
        set_array<VectorXd>(Properties::pctmag, value);
        return *this;
    }

    /// Array of phase angle values, degrees.You can also use the syntax
    /// angle = (file=filename)     !for text file one value per line
    /// angle = (dblfile=filename)  !for packed file of doubles
    /// angle = (sngfile=filename)  !for packed file of singles
    ///
    /// DSS property name: angle, DSS property index: 4
    ///
    VectorXd angle()
    {
        return get_array<VectorXd>(Properties::angle);
    }

    Spectrum& angle(VectorXd &value)
    {
        set_array<VectorXd>(Properties::angle, value);
        return *this;
    }

    /// File of spectrum points with (harmonic, magnitude-percent, angle-degrees) values, one set of 3 per line, in CSV format. If fewer than NUMHARM frequencies found in the file, NUMHARM is set to the smaller value.
    ///
    /// DSS property name: CSVFile, DSS property index: 5
    ///
    string CSVFile()
    {
        return get_prop_string(Properties::CSVFile);
    }

    Spectrum& CSVFile(const string &value)
    {
        set_string(Properties::CSVFile, value);
        return *this;
    }

    Spectrum& CSVFile(const char* value)
    {
        set_string(Properties::CSVFile, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 6
    ///
    Spectrum& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 6
    ///
    Spectrum& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class WireData: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 9;
    struct Properties
    {
        enum {
            Rdc = 1,
            Rac = 2,
            Runits = 3,
            GMRac = 4,
            GMRunits = 5,
            radius = 6,
            radunits = 7,
            normamps = 8,
            emergamps = 9,
            diam = 10,
            Seasons = 11,
            Ratings = 12,
            Capradius = 13,
            like = 14,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    WireData(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    WireData(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the WireData element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    WireData(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the WireData element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    WireData& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    WireData& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// dc Resistance, ohms per unit length (see Runits). Defaults to Rac/1.02 if not specified.
    ///
    /// DSS property name: Rdc, DSS property index: 1
    ///
    double Rdc()
    {
        return Obj_GetFloat64(ptr, Properties::Rdc);
    }

    WireData& Rdc(double value)
    {
        Obj_SetFloat64(ptr, Properties::Rdc, value);
        return *this;
    }

    /// Resistance at 60 Hz per unit length. Defaults to 1.02*Rdc if not specified.
    ///
    /// DSS property name: Rac, DSS property index: 2
    ///
    double Rac()
    {
        return Obj_GetFloat64(ptr, Properties::Rac);
    }

    WireData& Rac(double value)
    {
        Obj_SetFloat64(ptr, Properties::Rac, value);
        return *this;
    }

    /// Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: Runits, DSS property index: 3
    ///
    DimensionUnits Runits()
    {
        return DimensionUnits(Obj_GetInt32(ptr, Properties::Runits));
    }

    WireData& Runits(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Runits, value);
        return *this;
    }

    WireData& Runits(DimensionUnits value)
    {
        Obj_SetInt32(ptr, Properties::Runits, int32_t(value));
        return *this;
    }

    WireData& Runits(const string &value)
    {
        set_string(Properties::Runits, value);
        return *this;
    }

    WireData& Runits(const char *value)
    {
        set_string(Properties::Runits, value);
        return *this;
    }

    /// Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: Runits, DSS property index: 3
    ///
    string Runits_str()
    {
        return get_prop_string(Properties::Runits);
    }

    /// Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: Runits, DSS property index: 3
    ///
    WireData& Runits_str(const string &value)
    {
        set_string(Properties::Runits, value);
        return *this;
    }

    /// GMR at 60 Hz. Defaults to .7788*radius if not specified.
    ///
    /// DSS property name: GMRac, DSS property index: 4
    ///
    double GMRac()
    {
        return Obj_GetFloat64(ptr, Properties::GMRac);
    }

    WireData& GMRac(double value)
    {
        Obj_SetFloat64(ptr, Properties::GMRac, value);
        return *this;
    }

    /// Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: GMRunits, DSS property index: 5
    ///
    DimensionUnits GMRunits()
    {
        return DimensionUnits(Obj_GetInt32(ptr, Properties::GMRunits));
    }

    WireData& GMRunits(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::GMRunits, value);
        return *this;
    }

    WireData& GMRunits(DimensionUnits value)
    {
        Obj_SetInt32(ptr, Properties::GMRunits, int32_t(value));
        return *this;
    }

    WireData& GMRunits(const string &value)
    {
        set_string(Properties::GMRunits, value);
        return *this;
    }

    WireData& GMRunits(const char *value)
    {
        set_string(Properties::GMRunits, value);
        return *this;
    }

    /// Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: GMRunits, DSS property index: 5
    ///
    string GMRunits_str()
    {
        return get_prop_string(Properties::GMRunits);
    }

    /// Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: GMRunits, DSS property index: 5
    ///
    WireData& GMRunits_str(const string &value)
    {
        set_string(Properties::GMRunits, value);
        return *this;
    }

    /// Outside radius of conductor. Defaults to GMR/0.7788 if not specified.
    ///
    /// DSS property name: radius, DSS property index: 6
    ///
    double radius()
    {
        return Obj_GetFloat64(ptr, Properties::radius);
    }

    WireData& radius(double value)
    {
        Obj_SetFloat64(ptr, Properties::radius, value);
        return *this;
    }

    /// Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: radunits, DSS property index: 7
    ///
    DimensionUnits radunits()
    {
        return DimensionUnits(Obj_GetInt32(ptr, Properties::radunits));
    }

    WireData& radunits(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::radunits, value);
        return *this;
    }

    WireData& radunits(DimensionUnits value)
    {
        Obj_SetInt32(ptr, Properties::radunits, int32_t(value));
        return *this;
    }

    WireData& radunits(const string &value)
    {
        set_string(Properties::radunits, value);
        return *this;
    }

    WireData& radunits(const char *value)
    {
        set_string(Properties::radunits, value);
        return *this;
    }

    /// Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: radunits, DSS property index: 7
    ///
    string radunits_str()
    {
        return get_prop_string(Properties::radunits);
    }

    /// Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: radunits, DSS property index: 7
    ///
    WireData& radunits_str(const string &value)
    {
        set_string(Properties::radunits, value);
        return *this;
    }

    /// Normal ampacity, amperes. Defaults to Emergency amps/1.5 if not specified.
    ///
    /// DSS property name: normamps, DSS property index: 8
    ///
    double normamps()
    {
        return Obj_GetFloat64(ptr, Properties::normamps);
    }

    WireData& normamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::normamps, value);
        return *this;
    }

    /// Emergency ampacity, amperes. Defaults to 1.5 * Normal Amps if not specified.
    ///
    /// DSS property name: emergamps, DSS property index: 9
    ///
    double emergamps()
    {
        return Obj_GetFloat64(ptr, Properties::emergamps);
    }

    WireData& emergamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::emergamps, value);
        return *this;
    }

    /// Diameter; Alternative method for entering radius.
    ///
    /// DSS property name: diam, DSS property index: 10
    ///
    double diam()
    {
        return Obj_GetFloat64(ptr, Properties::diam);
    }

    WireData& diam(double value)
    {
        Obj_SetFloat64(ptr, Properties::diam, value);
        return *this;
    }

    /// Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.
    ///
    /// DSS property name: Seasons, DSS property index: 11
    ///
    int32_t Seasons()
    {
        return Obj_GetInt32(ptr, Properties::Seasons);
    }

    WireData& Seasons(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Seasons, value);
        return *this;
    }

    /// An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
    /// multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.
    ///
    /// DSS property name: Ratings, DSS property index: 12
    ///
    VectorXd Ratings()
    {
        return get_array<VectorXd>(Properties::Ratings);
    }

    WireData& Ratings(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Ratings, value);
        return *this;
    }

    /// Equivalent conductor radius for capacitance calcs. Specify this for bundled conductors. Defaults to same value as radius. Define Diam or Radius property first.
    ///
    /// DSS property name: Capradius, DSS property index: 13
    ///
    double Capradius()
    {
        return Obj_GetFloat64(ptr, Properties::Capradius);
    }

    WireData& Capradius(double value)
    {
        Obj_SetFloat64(ptr, Properties::Capradius, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 14
    ///
    WireData& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 14
    ///
    WireData& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class CNData: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 10;
    struct Properties
    {
        enum {
            k = 1,
            DiaStrand = 2,
            GmrStrand = 3,
            Rstrand = 4,
            EpsR = 5,
            InsLayer = 6,
            DiaIns = 7,
            DiaCable = 8,
            Rdc = 9,
            Rac = 10,
            Runits = 11,
            GMRac = 12,
            GMRunits = 13,
            radius = 14,
            radunits = 15,
            normamps = 16,
            emergamps = 17,
            diam = 18,
            Seasons = 19,
            Ratings = 20,
            Capradius = 21,
            like = 22,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    CNData(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    CNData(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the CNData element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    CNData(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the CNData element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    CNData& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    CNData& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of concentric neutral strands; default is 2
    ///
    /// DSS property name: k, DSS property index: 1
    ///
    int32_t k()
    {
        return Obj_GetInt32(ptr, Properties::k);
    }

    CNData& k(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::k, value);
        return *this;
    }

    /// Diameter of a concentric neutral strand; same units as core conductor radius; no default.
    ///
    /// DSS property name: DiaStrand, DSS property index: 2
    ///
    double DiaStrand()
    {
        return Obj_GetFloat64(ptr, Properties::DiaStrand);
    }

    CNData& DiaStrand(double value)
    {
        Obj_SetFloat64(ptr, Properties::DiaStrand, value);
        return *this;
    }

    /// Geometric mean radius of a concentric neutral strand; same units as core conductor GMR; defaults to 0.7788 * CN strand radius.
    ///
    /// DSS property name: GmrStrand, DSS property index: 3
    ///
    double GmrStrand()
    {
        return Obj_GetFloat64(ptr, Properties::GmrStrand);
    }

    CNData& GmrStrand(double value)
    {
        Obj_SetFloat64(ptr, Properties::GmrStrand, value);
        return *this;
    }

    /// AC resistance of a concentric neutral strand; same units as core conductor resistance; no default.
    ///
    /// DSS property name: Rstrand, DSS property index: 4
    ///
    double Rstrand()
    {
        return Obj_GetFloat64(ptr, Properties::Rstrand);
    }

    CNData& Rstrand(double value)
    {
        Obj_SetFloat64(ptr, Properties::Rstrand, value);
        return *this;
    }

    /// Insulation layer relative permittivity; default is 2.3.
    ///
    /// DSS property name: EpsR, DSS property index: 5
    ///
    double EpsR()
    {
        return Obj_GetFloat64(ptr, Properties::EpsR);
    }

    CNData& EpsR(double value)
    {
        Obj_SetFloat64(ptr, Properties::EpsR, value);
        return *this;
    }

    /// Insulation layer thickness; same units as radius; no default. With DiaIns, establishes inner radius for capacitance calculation.
    ///
    /// DSS property name: InsLayer, DSS property index: 6
    ///
    double InsLayer()
    {
        return Obj_GetFloat64(ptr, Properties::InsLayer);
    }

    CNData& InsLayer(double value)
    {
        Obj_SetFloat64(ptr, Properties::InsLayer, value);
        return *this;
    }

    /// Diameter over insulation layer; same units as radius; no default. Establishes outer radius for capacitance calculation.
    ///
    /// DSS property name: DiaIns, DSS property index: 7
    ///
    double DiaIns()
    {
        return Obj_GetFloat64(ptr, Properties::DiaIns);
    }

    CNData& DiaIns(double value)
    {
        Obj_SetFloat64(ptr, Properties::DiaIns, value);
        return *this;
    }

    /// Diameter over cable; same units as radius; no default.
    ///
    /// DSS property name: DiaCable, DSS property index: 8
    ///
    double DiaCable()
    {
        return Obj_GetFloat64(ptr, Properties::DiaCable);
    }

    CNData& DiaCable(double value)
    {
        Obj_SetFloat64(ptr, Properties::DiaCable, value);
        return *this;
    }

    /// dc Resistance, ohms per unit length (see Runits). Defaults to Rac/1.02 if not specified.
    ///
    /// DSS property name: Rdc, DSS property index: 9
    ///
    double Rdc()
    {
        return Obj_GetFloat64(ptr, Properties::Rdc);
    }

    CNData& Rdc(double value)
    {
        Obj_SetFloat64(ptr, Properties::Rdc, value);
        return *this;
    }

    /// Resistance at 60 Hz per unit length. Defaults to 1.02*Rdc if not specified.
    ///
    /// DSS property name: Rac, DSS property index: 10
    ///
    double Rac()
    {
        return Obj_GetFloat64(ptr, Properties::Rac);
    }

    CNData& Rac(double value)
    {
        Obj_SetFloat64(ptr, Properties::Rac, value);
        return *this;
    }

    /// Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: Runits, DSS property index: 11
    ///
    DimensionUnits Runits()
    {
        return DimensionUnits(Obj_GetInt32(ptr, Properties::Runits));
    }

    CNData& Runits(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Runits, value);
        return *this;
    }

    CNData& Runits(DimensionUnits value)
    {
        Obj_SetInt32(ptr, Properties::Runits, int32_t(value));
        return *this;
    }

    CNData& Runits(const string &value)
    {
        set_string(Properties::Runits, value);
        return *this;
    }

    CNData& Runits(const char *value)
    {
        set_string(Properties::Runits, value);
        return *this;
    }

    /// Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: Runits, DSS property index: 11
    ///
    string Runits_str()
    {
        return get_prop_string(Properties::Runits);
    }

    /// Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: Runits, DSS property index: 11
    ///
    CNData& Runits_str(const string &value)
    {
        set_string(Properties::Runits, value);
        return *this;
    }

    /// GMR at 60 Hz. Defaults to .7788*radius if not specified.
    ///
    /// DSS property name: GMRac, DSS property index: 12
    ///
    double GMRac()
    {
        return Obj_GetFloat64(ptr, Properties::GMRac);
    }

    CNData& GMRac(double value)
    {
        Obj_SetFloat64(ptr, Properties::GMRac, value);
        return *this;
    }

    /// Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: GMRunits, DSS property index: 13
    ///
    DimensionUnits GMRunits()
    {
        return DimensionUnits(Obj_GetInt32(ptr, Properties::GMRunits));
    }

    CNData& GMRunits(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::GMRunits, value);
        return *this;
    }

    CNData& GMRunits(DimensionUnits value)
    {
        Obj_SetInt32(ptr, Properties::GMRunits, int32_t(value));
        return *this;
    }

    CNData& GMRunits(const string &value)
    {
        set_string(Properties::GMRunits, value);
        return *this;
    }

    CNData& GMRunits(const char *value)
    {
        set_string(Properties::GMRunits, value);
        return *this;
    }

    /// Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: GMRunits, DSS property index: 13
    ///
    string GMRunits_str()
    {
        return get_prop_string(Properties::GMRunits);
    }

    /// Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: GMRunits, DSS property index: 13
    ///
    CNData& GMRunits_str(const string &value)
    {
        set_string(Properties::GMRunits, value);
        return *this;
    }

    /// Outside radius of conductor. Defaults to GMR/0.7788 if not specified.
    ///
    /// DSS property name: radius, DSS property index: 14
    ///
    double radius()
    {
        return Obj_GetFloat64(ptr, Properties::radius);
    }

    CNData& radius(double value)
    {
        Obj_SetFloat64(ptr, Properties::radius, value);
        return *this;
    }

    /// Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: radunits, DSS property index: 15
    ///
    DimensionUnits radunits()
    {
        return DimensionUnits(Obj_GetInt32(ptr, Properties::radunits));
    }

    CNData& radunits(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::radunits, value);
        return *this;
    }

    CNData& radunits(DimensionUnits value)
    {
        Obj_SetInt32(ptr, Properties::radunits, int32_t(value));
        return *this;
    }

    CNData& radunits(const string &value)
    {
        set_string(Properties::radunits, value);
        return *this;
    }

    CNData& radunits(const char *value)
    {
        set_string(Properties::radunits, value);
        return *this;
    }

    /// Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: radunits, DSS property index: 15
    ///
    string radunits_str()
    {
        return get_prop_string(Properties::radunits);
    }

    /// Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: radunits, DSS property index: 15
    ///
    CNData& radunits_str(const string &value)
    {
        set_string(Properties::radunits, value);
        return *this;
    }

    /// Normal ampacity, amperes. Defaults to Emergency amps/1.5 if not specified.
    ///
    /// DSS property name: normamps, DSS property index: 16
    ///
    double normamps()
    {
        return Obj_GetFloat64(ptr, Properties::normamps);
    }

    CNData& normamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::normamps, value);
        return *this;
    }

    /// Emergency ampacity, amperes. Defaults to 1.5 * Normal Amps if not specified.
    ///
    /// DSS property name: emergamps, DSS property index: 17
    ///
    double emergamps()
    {
        return Obj_GetFloat64(ptr, Properties::emergamps);
    }

    CNData& emergamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::emergamps, value);
        return *this;
    }

    /// Diameter; Alternative method for entering radius.
    ///
    /// DSS property name: diam, DSS property index: 18
    ///
    double diam()
    {
        return Obj_GetFloat64(ptr, Properties::diam);
    }

    CNData& diam(double value)
    {
        Obj_SetFloat64(ptr, Properties::diam, value);
        return *this;
    }

    /// Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.
    ///
    /// DSS property name: Seasons, DSS property index: 19
    ///
    int32_t Seasons()
    {
        return Obj_GetInt32(ptr, Properties::Seasons);
    }

    CNData& Seasons(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Seasons, value);
        return *this;
    }

    /// An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
    /// multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.
    ///
    /// DSS property name: Ratings, DSS property index: 20
    ///
    VectorXd Ratings()
    {
        return get_array<VectorXd>(Properties::Ratings);
    }

    CNData& Ratings(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Ratings, value);
        return *this;
    }

    /// Equivalent conductor radius for capacitance calcs. Specify this for bundled conductors. Defaults to same value as radius. Define Diam or Radius property first.
    ///
    /// DSS property name: Capradius, DSS property index: 21
    ///
    double Capradius()
    {
        return Obj_GetFloat64(ptr, Properties::Capradius);
    }

    CNData& Capradius(double value)
    {
        Obj_SetFloat64(ptr, Properties::Capradius, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 22
    ///
    CNData& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 22
    ///
    CNData& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class TSData: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 11;
    struct Properties
    {
        enum {
            DiaShield = 1,
            TapeLayer = 2,
            TapeLap = 3,
            EpsR = 4,
            InsLayer = 5,
            DiaIns = 6,
            DiaCable = 7,
            Rdc = 8,
            Rac = 9,
            Runits = 10,
            GMRac = 11,
            GMRunits = 12,
            radius = 13,
            radunits = 14,
            normamps = 15,
            emergamps = 16,
            diam = 17,
            Seasons = 18,
            Ratings = 19,
            Capradius = 20,
            like = 21,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    TSData(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    TSData(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the TSData element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    TSData(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the TSData element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    TSData& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    TSData& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Diameter over tape shield; same units as radius; no default.
    ///
    /// DSS property name: DiaShield, DSS property index: 1
    ///
    double DiaShield()
    {
        return Obj_GetFloat64(ptr, Properties::DiaShield);
    }

    TSData& DiaShield(double value)
    {
        Obj_SetFloat64(ptr, Properties::DiaShield, value);
        return *this;
    }

    /// Tape shield thickness; same units as radius; no default.
    ///
    /// DSS property name: TapeLayer, DSS property index: 2
    ///
    double TapeLayer()
    {
        return Obj_GetFloat64(ptr, Properties::TapeLayer);
    }

    TSData& TapeLayer(double value)
    {
        Obj_SetFloat64(ptr, Properties::TapeLayer, value);
        return *this;
    }

    /// Tape Lap in percent; default 20.0
    ///
    /// DSS property name: TapeLap, DSS property index: 3
    ///
    double TapeLap()
    {
        return Obj_GetFloat64(ptr, Properties::TapeLap);
    }

    TSData& TapeLap(double value)
    {
        Obj_SetFloat64(ptr, Properties::TapeLap, value);
        return *this;
    }

    /// Insulation layer relative permittivity; default is 2.3.
    ///
    /// DSS property name: EpsR, DSS property index: 4
    ///
    double EpsR()
    {
        return Obj_GetFloat64(ptr, Properties::EpsR);
    }

    TSData& EpsR(double value)
    {
        Obj_SetFloat64(ptr, Properties::EpsR, value);
        return *this;
    }

    /// Insulation layer thickness; same units as radius; no default. With DiaIns, establishes inner radius for capacitance calculation.
    ///
    /// DSS property name: InsLayer, DSS property index: 5
    ///
    double InsLayer()
    {
        return Obj_GetFloat64(ptr, Properties::InsLayer);
    }

    TSData& InsLayer(double value)
    {
        Obj_SetFloat64(ptr, Properties::InsLayer, value);
        return *this;
    }

    /// Diameter over insulation layer; same units as radius; no default. Establishes outer radius for capacitance calculation.
    ///
    /// DSS property name: DiaIns, DSS property index: 6
    ///
    double DiaIns()
    {
        return Obj_GetFloat64(ptr, Properties::DiaIns);
    }

    TSData& DiaIns(double value)
    {
        Obj_SetFloat64(ptr, Properties::DiaIns, value);
        return *this;
    }

    /// Diameter over cable; same units as radius; no default.
    ///
    /// DSS property name: DiaCable, DSS property index: 7
    ///
    double DiaCable()
    {
        return Obj_GetFloat64(ptr, Properties::DiaCable);
    }

    TSData& DiaCable(double value)
    {
        Obj_SetFloat64(ptr, Properties::DiaCable, value);
        return *this;
    }

    /// dc Resistance, ohms per unit length (see Runits). Defaults to Rac/1.02 if not specified.
    ///
    /// DSS property name: Rdc, DSS property index: 8
    ///
    double Rdc()
    {
        return Obj_GetFloat64(ptr, Properties::Rdc);
    }

    TSData& Rdc(double value)
    {
        Obj_SetFloat64(ptr, Properties::Rdc, value);
        return *this;
    }

    /// Resistance at 60 Hz per unit length. Defaults to 1.02*Rdc if not specified.
    ///
    /// DSS property name: Rac, DSS property index: 9
    ///
    double Rac()
    {
        return Obj_GetFloat64(ptr, Properties::Rac);
    }

    TSData& Rac(double value)
    {
        Obj_SetFloat64(ptr, Properties::Rac, value);
        return *this;
    }

    /// Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: Runits, DSS property index: 10
    ///
    DimensionUnits Runits()
    {
        return DimensionUnits(Obj_GetInt32(ptr, Properties::Runits));
    }

    TSData& Runits(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Runits, value);
        return *this;
    }

    TSData& Runits(DimensionUnits value)
    {
        Obj_SetInt32(ptr, Properties::Runits, int32_t(value));
        return *this;
    }

    TSData& Runits(const string &value)
    {
        set_string(Properties::Runits, value);
        return *this;
    }

    TSData& Runits(const char *value)
    {
        set_string(Properties::Runits, value);
        return *this;
    }

    /// Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: Runits, DSS property index: 10
    ///
    string Runits_str()
    {
        return get_prop_string(Properties::Runits);
    }

    /// Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: Runits, DSS property index: 10
    ///
    TSData& Runits_str(const string &value)
    {
        set_string(Properties::Runits, value);
        return *this;
    }

    /// GMR at 60 Hz. Defaults to .7788*radius if not specified.
    ///
    /// DSS property name: GMRac, DSS property index: 11
    ///
    double GMRac()
    {
        return Obj_GetFloat64(ptr, Properties::GMRac);
    }

    TSData& GMRac(double value)
    {
        Obj_SetFloat64(ptr, Properties::GMRac, value);
        return *this;
    }

    /// Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: GMRunits, DSS property index: 12
    ///
    DimensionUnits GMRunits()
    {
        return DimensionUnits(Obj_GetInt32(ptr, Properties::GMRunits));
    }

    TSData& GMRunits(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::GMRunits, value);
        return *this;
    }

    TSData& GMRunits(DimensionUnits value)
    {
        Obj_SetInt32(ptr, Properties::GMRunits, int32_t(value));
        return *this;
    }

    TSData& GMRunits(const string &value)
    {
        set_string(Properties::GMRunits, value);
        return *this;
    }

    TSData& GMRunits(const char *value)
    {
        set_string(Properties::GMRunits, value);
        return *this;
    }

    /// Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: GMRunits, DSS property index: 12
    ///
    string GMRunits_str()
    {
        return get_prop_string(Properties::GMRunits);
    }

    /// Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: GMRunits, DSS property index: 12
    ///
    TSData& GMRunits_str(const string &value)
    {
        set_string(Properties::GMRunits, value);
        return *this;
    }

    /// Outside radius of conductor. Defaults to GMR/0.7788 if not specified.
    ///
    /// DSS property name: radius, DSS property index: 13
    ///
    double radius()
    {
        return Obj_GetFloat64(ptr, Properties::radius);
    }

    TSData& radius(double value)
    {
        Obj_SetFloat64(ptr, Properties::radius, value);
        return *this;
    }

    /// Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: radunits, DSS property index: 14
    ///
    DimensionUnits radunits()
    {
        return DimensionUnits(Obj_GetInt32(ptr, Properties::radunits));
    }

    TSData& radunits(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::radunits, value);
        return *this;
    }

    TSData& radunits(DimensionUnits value)
    {
        Obj_SetInt32(ptr, Properties::radunits, int32_t(value));
        return *this;
    }

    TSData& radunits(const string &value)
    {
        set_string(Properties::radunits, value);
        return *this;
    }

    TSData& radunits(const char *value)
    {
        set_string(Properties::radunits, value);
        return *this;
    }

    /// Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: radunits, DSS property index: 14
    ///
    string radunits_str()
    {
        return get_prop_string(Properties::radunits);
    }

    /// Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: radunits, DSS property index: 14
    ///
    TSData& radunits_str(const string &value)
    {
        set_string(Properties::radunits, value);
        return *this;
    }

    /// Normal ampacity, amperes. Defaults to Emergency amps/1.5 if not specified.
    ///
    /// DSS property name: normamps, DSS property index: 15
    ///
    double normamps()
    {
        return Obj_GetFloat64(ptr, Properties::normamps);
    }

    TSData& normamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::normamps, value);
        return *this;
    }

    /// Emergency ampacity, amperes. Defaults to 1.5 * Normal Amps if not specified.
    ///
    /// DSS property name: emergamps, DSS property index: 16
    ///
    double emergamps()
    {
        return Obj_GetFloat64(ptr, Properties::emergamps);
    }

    TSData& emergamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::emergamps, value);
        return *this;
    }

    /// Diameter; Alternative method for entering radius.
    ///
    /// DSS property name: diam, DSS property index: 17
    ///
    double diam()
    {
        return Obj_GetFloat64(ptr, Properties::diam);
    }

    TSData& diam(double value)
    {
        Obj_SetFloat64(ptr, Properties::diam, value);
        return *this;
    }

    /// Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.
    ///
    /// DSS property name: Seasons, DSS property index: 18
    ///
    int32_t Seasons()
    {
        return Obj_GetInt32(ptr, Properties::Seasons);
    }

    TSData& Seasons(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Seasons, value);
        return *this;
    }

    /// An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
    /// multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.
    ///
    /// DSS property name: Ratings, DSS property index: 19
    ///
    VectorXd Ratings()
    {
        return get_array<VectorXd>(Properties::Ratings);
    }

    TSData& Ratings(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Ratings, value);
        return *this;
    }

    /// Equivalent conductor radius for capacitance calcs. Specify this for bundled conductors. Defaults to same value as radius. Define Diam or Radius property first.
    ///
    /// DSS property name: Capradius, DSS property index: 20
    ///
    double Capradius()
    {
        return Obj_GetFloat64(ptr, Properties::Capradius);
    }

    TSData& Capradius(double value)
    {
        Obj_SetFloat64(ptr, Properties::Capradius, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 21
    ///
    TSData& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 21
    ///
    TSData& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class LineSpacing: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 12;
    struct Properties
    {
        enum {
            nconds = 1,
            nphases = 2,
            x = 3,
            h = 4,
            units = 5,
            like = 6,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    LineSpacing(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    LineSpacing(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the LineSpacing element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    LineSpacing(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the LineSpacing element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    LineSpacing& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    LineSpacing& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of wires in this geometry. Default is 3. Triggers memory allocations. Define first!
    ///
    /// DSS property name: nconds, DSS property index: 1
    ///
    int32_t nconds()
    {
        return Obj_GetInt32(ptr, Properties::nconds);
    }

    LineSpacing& nconds(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::nconds, value);
        return *this;
    }

    /// Number of retained phase conductors. If less than the number of wires, list the retained phase coordinates first.
    ///
    /// DSS property name: nphases, DSS property index: 2
    ///
    int32_t nphases()
    {
        return Obj_GetInt32(ptr, Properties::nphases);
    }

    LineSpacing& nphases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::nphases, value);
        return *this;
    }

    /// Array of wire X coordinates.
    ///
    /// DSS property name: x, DSS property index: 3
    ///
    VectorXd x()
    {
        return get_array<VectorXd>(Properties::x);
    }

    LineSpacing& x(VectorXd &value)
    {
        set_array<VectorXd>(Properties::x, value);
        return *this;
    }

    /// Array of wire Heights.
    ///
    /// DSS property name: h, DSS property index: 4
    ///
    VectorXd h()
    {
        return get_array<VectorXd>(Properties::h);
    }

    LineSpacing& h(VectorXd &value)
    {
        set_array<VectorXd>(Properties::h, value);
        return *this;
    }

    /// Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined
    ///
    /// DSS property name: units, DSS property index: 5
    ///
    DimensionUnits units()
    {
        return DimensionUnits(Obj_GetInt32(ptr, Properties::units));
    }

    LineSpacing& units(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::units, value);
        return *this;
    }

    LineSpacing& units(DimensionUnits value)
    {
        Obj_SetInt32(ptr, Properties::units, int32_t(value));
        return *this;
    }

    LineSpacing& units(const string &value)
    {
        set_string(Properties::units, value);
        return *this;
    }

    LineSpacing& units(const char *value)
    {
        set_string(Properties::units, value);
        return *this;
    }

    /// Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined
    ///
    /// DSS property name: units, DSS property index: 5
    ///
    string units_str()
    {
        return get_prop_string(Properties::units);
    }

    /// Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined
    ///
    /// DSS property name: units, DSS property index: 5
    ///
    LineSpacing& units_str(const string &value)
    {
        set_string(Properties::units, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 6
    ///
    LineSpacing& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 6
    ///
    LineSpacing& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class LineGeometry: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 13;
    struct Properties
    {
        enum {
            nconds = 1,
            nphases = 2,
            cond = 3,
            wire = 4,
            x = 5,
            h = 6,
            units = 7,
            normamps = 8,
            emergamps = 9,
            reduce = 10,
            spacing = 11,
            wires = 12,
            cncable = 13,
            tscable = 14,
            cncables = 15,
            tscables = 16,
            Seasons = 17,
            Ratings = 18,
            LineType = 19,
            like = 20,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    LineGeometry(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    LineGeometry(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the LineGeometry element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    LineGeometry(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the LineGeometry element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    LineGeometry& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    LineGeometry& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of conductors in this geometry. Default is 3. Triggers memory allocations. Define first!
    ///
    /// DSS property name: nconds, DSS property index: 1
    ///
    int32_t nconds()
    {
        return Obj_GetInt32(ptr, Properties::nconds);
    }

    LineGeometry& nconds(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::nconds, value);
        return *this;
    }

    /// Number of phases. Default =3; All other conductors are considered neutrals and might be reduced out.
    ///
    /// DSS property name: nphases, DSS property index: 2
    ///
    int32_t nphases()
    {
        return Obj_GetInt32(ptr, Properties::nphases);
    }

    LineGeometry& nphases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::nphases, value);
        return *this;
    }

    /// Set this = number of the conductor you wish to define. Default is 1.
    ///
    /// DSS property name: cond, DSS property index: 3
    ///
    int32_t cond()
    {
        return Obj_GetInt32(ptr, Properties::cond);
    }

    LineGeometry& cond(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::cond, value);
        return *this;
    }

    /// Code from WireData. MUST BE PREVIOUSLY DEFINED. no default.
    /// Specifies use of Overhead Line parameter calculation,
    /// Unless Tape Shield cable previously assigned to phases, and this wire is a neutral.
    ///
    /// DSS property name: wire, DSS property index: 4
    ///
    strings wire()
    {
        return get_array<strings>(Properties::wire);
    }

    LineGeometry& wire(strings &value)
    {
        set_array<strings>(Properties::wire, value);
        return *this;
    }

    LineGeometry& wire(std::vector<dss::obj::WireData> &value)
    {
        set_array<std::vector<dss::obj::WireData>>(Properties::wire, value);
        return *this;
    }

    /// Code from WireData. MUST BE PREVIOUSLY DEFINED. no default.
    /// Specifies use of Overhead Line parameter calculation,
    /// Unless Tape Shield cable previously assigned to phases, and this wire is a neutral.
    ///
    /// DSS property name: wire, DSS property index: 4
    ///
    std::vector<dss::obj::WireData> wire_obj()
    {
        return get_array<std::vector<dss::obj::WireData>>(Properties::wire);
    }

    LineGeometry& wire_obj(std::vector<dss::obj::WireData> &value)
    {
        set_array<std::vector<dss::obj::WireData>>(Properties::wire, value);
        return *this;
    }

    /// x coordinate.
    ///
    /// DSS property name: x, DSS property index: 5
    ///
    VectorXd x()
    {
        return get_array<VectorXd>(Properties::x);
    }

    LineGeometry& x(VectorXd &value)
    {
        set_array<VectorXd>(Properties::x, value);
        return *this;
    }

    /// Height of conductor.
    ///
    /// DSS property name: h, DSS property index: 6
    ///
    VectorXd h()
    {
        return get_array<VectorXd>(Properties::h);
    }

    LineGeometry& h(VectorXd &value)
    {
        set_array<VectorXd>(Properties::h, value);
        return *this;
    }

    /// Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined
    ///
    /// DSS property name: units, DSS property index: 7
    ///
    DimensionUnits units()
    {
        return DimensionUnits(Obj_GetInt32(ptr, Properties::units));
    }

    LineGeometry& units(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::units, value);
        return *this;
    }

    LineGeometry& units(DimensionUnits value)
    {
        Obj_SetInt32(ptr, Properties::units, int32_t(value));
        return *this;
    }

    LineGeometry& units(const string &value)
    {
        set_string(Properties::units, value);
        return *this;
    }

    LineGeometry& units(const char *value)
    {
        set_string(Properties::units, value);
        return *this;
    }

    /// Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined
    ///
    /// DSS property name: units, DSS property index: 7
    ///
    string units_str()
    {
        return get_prop_string(Properties::units);
    }

    /// Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined
    ///
    /// DSS property name: units, DSS property index: 7
    ///
    LineGeometry& units_str(const string &value)
    {
        set_string(Properties::units, value);
        return *this;
    }

    /// Normal ampacity, amperes for the line. Defaults to first conductor if not specified.
    ///
    /// DSS property name: normamps, DSS property index: 8
    ///
    double normamps()
    {
        return Obj_GetFloat64(ptr, Properties::normamps);
    }

    LineGeometry& normamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::normamps, value);
        return *this;
    }

    /// Emergency ampacity, amperes. Defaults to first conductor if not specified.
    ///
    /// DSS property name: emergamps, DSS property index: 9
    ///
    double emergamps()
    {
        return Obj_GetFloat64(ptr, Properties::emergamps);
    }

    LineGeometry& emergamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::emergamps, value);
        return *this;
    }

    /// {Yes | No} Default = no. Reduce to Nphases (Kron Reduction). Reduce out neutrals.
    ///
    /// DSS property name: reduce, DSS property index: 10
    ///
    bool reduce()
    {
        return Obj_GetInt32(ptr, Properties::reduce) != 0;
    }

    LineGeometry& reduce(bool value)
    {
        Obj_SetInt32(ptr, Properties::reduce, value);
        return *this;
    }

    /// Reference to a LineSpacing for use in a line constants calculation.
    /// Alternative to x, h, and units. MUST BE PREVIOUSLY DEFINED.
    /// Must match "nconds" as previously defined for this geometry.
    /// Must be used in conjunction with the Wires property.
    ///
    /// DSS property name: spacing, DSS property index: 11
    ///
    string spacing()
    {
        return get_prop_string(Properties::spacing);
    }

    LineGeometry& spacing(const string &value)
    {
        set_string(Properties::spacing, value);
        return *this;
    }

    LineGeometry& spacing(dss::obj::LineSpacing &value)
    {
        set_obj(Properties::spacing, value);
        return *this;
    }

    /// Reference to a LineSpacing for use in a line constants calculation.
    /// Alternative to x, h, and units. MUST BE PREVIOUSLY DEFINED.
    /// Must match "nconds" as previously defined for this geometry.
    /// Must be used in conjunction with the Wires property.
    ///
    /// DSS property name: spacing, DSS property index: 11
    ///
    dss::obj::LineSpacing spacing_obj()
    {
        return get_obj<dss::obj::LineSpacing>(Properties::spacing);
    }

    LineGeometry& spacing_obj(dss::obj::LineSpacing &value)
    {
        set_obj(Properties::spacing, value);
        return *this;
    }

    /// Array of WireData names for use in a line constants calculation.
    /// Alternative to individual wire inputs. ALL MUST BE PREVIOUSLY DEFINED.
    /// Must match "nconds" as previously defined for this geometry,
    /// unless TSData or CNData were previously assigned to phases, and these wires are neutrals.
    /// Must be used in conjunction with the Spacing property.
    ///
    /// DSS property name: wires, DSS property index: 12
    ///
    strings wires()
    {
        return get_array<strings>(Properties::wires);
    }

    LineGeometry& wires(strings &value)
    {
        set_array<strings>(Properties::wires, value);
        return *this;
    }

    LineGeometry& wires(std::vector<dss::obj::WireData> &value)
    {
        set_array<std::vector<dss::obj::WireData>>(Properties::wires, value);
        return *this;
    }

    /// Array of WireData names for use in a line constants calculation.
    /// Alternative to individual wire inputs. ALL MUST BE PREVIOUSLY DEFINED.
    /// Must match "nconds" as previously defined for this geometry,
    /// unless TSData or CNData were previously assigned to phases, and these wires are neutrals.
    /// Must be used in conjunction with the Spacing property.
    ///
    /// DSS property name: wires, DSS property index: 12
    ///
    std::vector<dss::obj::WireData> wires_obj()
    {
        return get_array<std::vector<dss::obj::WireData>>(Properties::wires);
    }

    LineGeometry& wires_obj(std::vector<dss::obj::WireData> &value)
    {
        set_array<std::vector<dss::obj::WireData>>(Properties::wires, value);
        return *this;
    }

    /// Code from CNData. MUST BE PREVIOUSLY DEFINED. no default.
    /// Specifies use of Concentric Neutral cable parameter calculation.
    ///
    /// DSS property name: cncable, DSS property index: 13
    ///
    strings cncable()
    {
        return get_array<strings>(Properties::cncable);
    }

    LineGeometry& cncable(strings &value)
    {
        set_array<strings>(Properties::cncable, value);
        return *this;
    }

    LineGeometry& cncable(std::vector<dss::obj::CNData> &value)
    {
        set_array<std::vector<dss::obj::CNData>>(Properties::cncable, value);
        return *this;
    }

    /// Code from CNData. MUST BE PREVIOUSLY DEFINED. no default.
    /// Specifies use of Concentric Neutral cable parameter calculation.
    ///
    /// DSS property name: cncable, DSS property index: 13
    ///
    std::vector<dss::obj::CNData> cncable_obj()
    {
        return get_array<std::vector<dss::obj::CNData>>(Properties::cncable);
    }

    LineGeometry& cncable_obj(std::vector<dss::obj::CNData> &value)
    {
        set_array<std::vector<dss::obj::CNData>>(Properties::cncable, value);
        return *this;
    }

    /// Code from TSData. MUST BE PREVIOUSLY DEFINED. no default.
    /// Specifies use of Tape Shield cable parameter calculation.
    ///
    /// DSS property name: tscable, DSS property index: 14
    ///
    strings tscable()
    {
        return get_array<strings>(Properties::tscable);
    }

    LineGeometry& tscable(strings &value)
    {
        set_array<strings>(Properties::tscable, value);
        return *this;
    }

    LineGeometry& tscable(std::vector<dss::obj::TSData> &value)
    {
        set_array<std::vector<dss::obj::TSData>>(Properties::tscable, value);
        return *this;
    }

    /// Code from TSData. MUST BE PREVIOUSLY DEFINED. no default.
    /// Specifies use of Tape Shield cable parameter calculation.
    ///
    /// DSS property name: tscable, DSS property index: 14
    ///
    std::vector<dss::obj::TSData> tscable_obj()
    {
        return get_array<std::vector<dss::obj::TSData>>(Properties::tscable);
    }

    LineGeometry& tscable_obj(std::vector<dss::obj::TSData> &value)
    {
        set_array<std::vector<dss::obj::TSData>>(Properties::tscable, value);
        return *this;
    }

    /// Array of CNData names for cable parameter calculation.
    /// All must be previously defined, and match "nphases" for this geometry.
    /// You can later define "nconds-nphases" wires for bare neutral conductors.
    ///
    /// DSS property name: cncables, DSS property index: 15
    ///
    strings cncables()
    {
        return get_array<strings>(Properties::cncables);
    }

    LineGeometry& cncables(strings &value)
    {
        set_array<strings>(Properties::cncables, value);
        return *this;
    }

    LineGeometry& cncables(std::vector<dss::obj::CNData> &value)
    {
        set_array<std::vector<dss::obj::CNData>>(Properties::cncables, value);
        return *this;
    }

    /// Array of CNData names for cable parameter calculation.
    /// All must be previously defined, and match "nphases" for this geometry.
    /// You can later define "nconds-nphases" wires for bare neutral conductors.
    ///
    /// DSS property name: cncables, DSS property index: 15
    ///
    std::vector<dss::obj::CNData> cncables_obj()
    {
        return get_array<std::vector<dss::obj::CNData>>(Properties::cncables);
    }

    LineGeometry& cncables_obj(std::vector<dss::obj::CNData> &value)
    {
        set_array<std::vector<dss::obj::CNData>>(Properties::cncables, value);
        return *this;
    }

    /// Array of TSData names for cable parameter calculation.
    /// All must be previously defined, and match "nphases" for this geometry.
    /// You can later define "nconds-nphases" wires for bare neutral conductors.
    ///
    /// DSS property name: tscables, DSS property index: 16
    ///
    strings tscables()
    {
        return get_array<strings>(Properties::tscables);
    }

    LineGeometry& tscables(strings &value)
    {
        set_array<strings>(Properties::tscables, value);
        return *this;
    }

    LineGeometry& tscables(std::vector<dss::obj::TSData> &value)
    {
        set_array<std::vector<dss::obj::TSData>>(Properties::tscables, value);
        return *this;
    }

    /// Array of TSData names for cable parameter calculation.
    /// All must be previously defined, and match "nphases" for this geometry.
    /// You can later define "nconds-nphases" wires for bare neutral conductors.
    ///
    /// DSS property name: tscables, DSS property index: 16
    ///
    std::vector<dss::obj::TSData> tscables_obj()
    {
        return get_array<std::vector<dss::obj::TSData>>(Properties::tscables);
    }

    LineGeometry& tscables_obj(std::vector<dss::obj::TSData> &value)
    {
        set_array<std::vector<dss::obj::TSData>>(Properties::tscables, value);
        return *this;
    }

    /// Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property. Defaults to first conductor if not specified.
    ///
    /// DSS property name: Seasons, DSS property index: 17
    ///
    int32_t Seasons()
    {
        return Obj_GetInt32(ptr, Properties::Seasons);
    }

    LineGeometry& Seasons(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Seasons, value);
        return *this;
    }

    /// An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
    /// multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.Defaults to first conductor if not specified.
    ///
    /// DSS property name: Ratings, DSS property index: 18
    ///
    VectorXd Ratings()
    {
        return get_array<VectorXd>(Properties::Ratings);
    }

    LineGeometry& Ratings(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Ratings, value);
        return *this;
    }

    /// Code designating the type of line.
    /// One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW
    ///
    /// OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.
    ///
    /// DSS property name: LineType, DSS property index: 19
    ///
    LineType linetype()
    {
        return LineType(Obj_GetInt32(ptr, Properties::LineType));
    }

    LineGeometry& linetype(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::LineType, value);
        return *this;
    }

    LineGeometry& linetype(LineType value)
    {
        Obj_SetInt32(ptr, Properties::LineType, int32_t(value));
        return *this;
    }

    LineGeometry& linetype(const string &value)
    {
        set_string(Properties::LineType, value);
        return *this;
    }

    LineGeometry& linetype(const char *value)
    {
        set_string(Properties::LineType, value);
        return *this;
    }

    /// Code designating the type of line.
    /// One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW
    ///
    /// OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.
    ///
    /// DSS property name: LineType, DSS property index: 19
    ///
    string linetype_str()
    {
        return get_prop_string(Properties::LineType);
    }

    /// Code designating the type of line.
    /// One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW
    ///
    /// OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.
    ///
    /// DSS property name: LineType, DSS property index: 19
    ///
    LineGeometry& linetype_str(const string &value)
    {
        set_string(Properties::LineType, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 20
    ///
    LineGeometry& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 20
    ///
    LineGeometry& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class XfmrCode: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 14;
    struct Properties
    {
        enum {
            phases = 1,
            windings = 2,
            wdg = 3,
            conn = 4,
            kV = 5,
            kVA = 6,
            tap = 7,
            pctR = 8,
            Rneut = 9,
            Xneut = 10,
            conns = 11,
            kVs = 12,
            kVAs = 13,
            taps = 14,
            Xhl = 15,
            Xht = 16,
            Xlt = 17,
            Xscarray = 18,
            thermal = 19,
            n = 20,
            m = 21,
            flrise = 22,
            hsrise = 23,
            pctloadloss = 24,
            pctnoloadloss = 25,
            normhkVA = 26,
            emerghkVA = 27,
            MaxTap = 28,
            MinTap = 29,
            NumTaps = 30,
            pctimag = 31,
            ppm_antifloat = 32,
            pctRs = 33,
            X12 = 34,
            X13 = 35,
            X23 = 36,
            RdcOhms = 37,
            Seasons = 38,
            Ratings = 39,
            like = 40,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    XfmrCode(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    XfmrCode(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the XfmrCode element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    XfmrCode(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the XfmrCode element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    XfmrCode& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    XfmrCode& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of phases this transformer. Default is 3.
    ///
    /// DSS property name: phases, DSS property index: 1
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    XfmrCode& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Number of windings, this transformers. (Also is the number of terminals) Default is 2. This property triggers memory allocation for the Transformer and will cause other properties to revert to default values.
    ///
    /// DSS property name: windings, DSS property index: 2
    ///
    int32_t windings()
    {
        return Obj_GetInt32(ptr, Properties::windings);
    }

    XfmrCode& windings(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::windings, value);
        return *this;
    }

    /// Set this = to the number of the winding you wish to define.  Then set the values for this winding.  Repeat for each winding.  Alternatively, use the array collections (buses, kvas, etc.) to define the windings.  Note: reactances are BETWEEN pairs of windings; they are not the property of a single winding.
    ///
    /// DSS property name: wdg, DSS property index: 3
    ///
    int32_t wdg()
    {
        return Obj_GetInt32(ptr, Properties::wdg);
    }

    XfmrCode& wdg(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::wdg, value);
        return *this;
    }

    /// Connection of this winding. Default is "wye" with the neutral solidly grounded.
    ///
    /// DSS property name: conn, DSS property index: 4
    ///
    std::vector<Connection> conn()
    {
        return get_array<std::vector<Connection>>(Properties::conn);
    }

    XfmrCode& conn(std::vector<int32_t> &value)
    {
        set_array<std::vector<int32_t>>(Properties::conn, value);
        return *this;
    }

    XfmrCode& conn(strings &value)
    {
        set_array<strings>(Properties::conn, value);
        return *this;
    }

    /// Connection of this winding. Default is "wye" with the neutral solidly grounded.
    ///
    /// DSS property name: conn, DSS property index: 4
    ///
    strings conn_str()
    {
        return get_array<strings>(Properties::conn);
    }

    XfmrCode& conn_str(strings &value)
    {
        conn(value);
        return *this;
    }

    /// For 2-or 3-phase, enter phase-phase kV rating.  Otherwise, kV rating of the actual winding
    ///
    /// DSS property name: kV, DSS property index: 5
    ///
    VectorXd kV()
    {
        return get_array<VectorXd>(Properties::kV);
    }

    XfmrCode& kV(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kV, value);
        return *this;
    }

    /// Base kVA rating of the winding. Side effect: forces change of max normal and emerg kva ratings.If 2-winding transformer, forces other winding to same value. When winding 1 is defined, all other windings are defaulted to the same rating and the first two winding resistances are defaulted to the %loadloss value.
    ///
    /// DSS property name: kVA, DSS property index: 6
    ///
    VectorXd kVA()
    {
        return get_array<VectorXd>(Properties::kVA);
    }

    XfmrCode& kVA(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kVA, value);
        return *this;
    }

    /// Per unit tap that this winding is normally on.
    ///
    /// DSS property name: tap, DSS property index: 7
    ///
    VectorXd tap()
    {
        return get_array<VectorXd>(Properties::tap);
    }

    XfmrCode& tap(VectorXd &value)
    {
        set_array<VectorXd>(Properties::tap, value);
        return *this;
    }

    /// Percent resistance this winding.  (half of total for a 2-winding).
    ///
    /// DSS property name: %R, DSS property index: 8
    ///
    VectorXd pctR()
    {
        return get_array<VectorXd>(Properties::pctR);
    }

    XfmrCode& pctR(VectorXd &value)
    {
        set_array<VectorXd>(Properties::pctR, value);
        return *this;
    }

    /// Default = -1. Neutral resistance of wye (star)-connected winding in actual ohms.If entered as a negative value, the neutral is assumed to be open, or floating.
    ///
    /// DSS property name: Rneut, DSS property index: 9
    ///
    VectorXd Rneut()
    {
        return get_array<VectorXd>(Properties::Rneut);
    }

    XfmrCode& Rneut(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Rneut, value);
        return *this;
    }

    /// Neutral reactance of wye(star)-connected winding in actual ohms.  May be + or -.
    ///
    /// DSS property name: Xneut, DSS property index: 10
    ///
    VectorXd Xneut()
    {
        return get_array<VectorXd>(Properties::Xneut);
    }

    XfmrCode& Xneut(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Xneut, value);
        return *this;
    }

    /// Use this to specify all the Winding connections at once using an array. Example:
    ///
    /// New Transformer.T1 buses="Hibus, lowbus" ~ conns=(delta, wye)
    ///
    /// DSS property name: conns, DSS property index: 11
    ///
    std::vector<Connection> conns()
    {
        return get_array<std::vector<Connection>>(Properties::conns);
    }

    XfmrCode& conns(std::vector<int32_t> &value)
    {
        set_array<std::vector<int32_t>>(Properties::conns, value);
        return *this;
    }

    XfmrCode& conns(strings &value)
    {
        set_array<strings>(Properties::conns, value);
        return *this;
    }

    /// Use this to specify all the Winding connections at once using an array. Example:
    ///
    /// New Transformer.T1 buses="Hibus, lowbus" ~ conns=(delta, wye)
    ///
    /// DSS property name: conns, DSS property index: 11
    ///
    strings conns_str()
    {
        return get_array<strings>(Properties::conns);
    }

    XfmrCode& conns_str(strings &value)
    {
        conns(value);
        return *this;
    }

    /// Use this to specify the kV ratings of all windings at once using an array. Example:
    ///
    /// New Transformer.T1 buses="Hibus, lowbus"
    /// ~ conns=(delta, wye)
    /// ~ kvs=(115, 12.47)
    ///
    /// See kV= property for voltage rules.
    ///
    /// DSS property name: kVs, DSS property index: 12
    ///
    VectorXd kVs()
    {
        return get_array<VectorXd>(Properties::kVs);
    }

    XfmrCode& kVs(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kVs, value);
        return *this;
    }

    /// Use this to specify the kVA ratings of all windings at once using an array.
    ///
    /// DSS property name: kVAs, DSS property index: 13
    ///
    VectorXd kVAs()
    {
        return get_array<VectorXd>(Properties::kVAs);
    }

    XfmrCode& kVAs(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kVAs, value);
        return *this;
    }

    /// Use this to specify the normal p.u. tap of all windings at once using an array.
    ///
    /// DSS property name: taps, DSS property index: 14
    ///
    VectorXd taps()
    {
        return get_array<VectorXd>(Properties::taps);
    }

    XfmrCode& taps(VectorXd &value)
    {
        set_array<VectorXd>(Properties::taps, value);
        return *this;
    }

    /// Use this to specify the percent reactance, H-L (winding 1 to winding 2).  Use for 2- or 3-winding transformers. On the kva base of winding 1.
    ///
    /// DSS property name: Xhl, DSS property index: 15
    ///
    double Xhl()
    {
        return Obj_GetFloat64(ptr, Properties::Xhl);
    }

    XfmrCode& Xhl(double value)
    {
        Obj_SetFloat64(ptr, Properties::Xhl, value);
        return *this;
    }

    /// Use this to specify the percent reactance, H-T (winding 1 to winding 3).  Use for 3-winding transformers only. On the kVA base of winding 1.
    ///
    /// DSS property name: Xht, DSS property index: 16
    ///
    double Xht()
    {
        return Obj_GetFloat64(ptr, Properties::Xht);
    }

    XfmrCode& Xht(double value)
    {
        Obj_SetFloat64(ptr, Properties::Xht, value);
        return *this;
    }

    /// Use this to specify the percent reactance, L-T (winding 2 to winding 3).  Use for 3-winding transformers only. On the kVA base of winding 1.
    ///
    /// DSS property name: Xlt, DSS property index: 17
    ///
    double Xlt()
    {
        return Obj_GetFloat64(ptr, Properties::Xlt);
    }

    XfmrCode& Xlt(double value)
    {
        Obj_SetFloat64(ptr, Properties::Xlt, value);
        return *this;
    }

    /// Use this to specify the percent reactance between all pairs of windings as an array. All values are on the kVA base of winding 1.  The order of the values is as follows:
    ///
    /// (x12 13 14... 23 24.. 34 ..)
    ///
    /// There will be n(n-1)/2 values, where n=number of windings.
    ///
    /// DSS property name: Xscarray, DSS property index: 18
    ///
    VectorXd Xscarray()
    {
        return get_array<VectorXd>(Properties::Xscarray);
    }

    XfmrCode& Xscarray(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Xscarray, value);
        return *this;
    }

    /// Thermal time constant of the transformer in hours.  Typically about 2.
    ///
    /// DSS property name: thermal, DSS property index: 19
    ///
    double thermal()
    {
        return Obj_GetFloat64(ptr, Properties::thermal);
    }

    XfmrCode& thermal(double value)
    {
        Obj_SetFloat64(ptr, Properties::thermal, value);
        return *this;
    }

    /// n Exponent for thermal properties in IEEE C57.  Typically 0.8.
    ///
    /// DSS property name: n, DSS property index: 20
    ///
    double n()
    {
        return Obj_GetFloat64(ptr, Properties::n);
    }

    XfmrCode& n(double value)
    {
        Obj_SetFloat64(ptr, Properties::n, value);
        return *this;
    }

    /// m Exponent for thermal properties in IEEE C57.  Typically 0.9 - 1.0
    ///
    /// DSS property name: m, DSS property index: 21
    ///
    double m()
    {
        return Obj_GetFloat64(ptr, Properties::m);
    }

    XfmrCode& m(double value)
    {
        Obj_SetFloat64(ptr, Properties::m, value);
        return *this;
    }

    /// Temperature rise, deg C, for full load.  Default is 65.
    ///
    /// DSS property name: flrise, DSS property index: 22
    ///
    double flrise()
    {
        return Obj_GetFloat64(ptr, Properties::flrise);
    }

    XfmrCode& flrise(double value)
    {
        Obj_SetFloat64(ptr, Properties::flrise, value);
        return *this;
    }

    /// Hot spot temperature rise, deg C.  Default is 15.
    ///
    /// DSS property name: hsrise, DSS property index: 23
    ///
    double hsrise()
    {
        return Obj_GetFloat64(ptr, Properties::hsrise);
    }

    XfmrCode& hsrise(double value)
    {
        Obj_SetFloat64(ptr, Properties::hsrise, value);
        return *this;
    }

    /// Percent load loss at full load. The %R of the High and Low windings (1 and 2) are adjusted to agree at rated kVA loading.
    ///
    /// DSS property name: %loadloss, DSS property index: 24
    ///
    double pctloadloss()
    {
        return Obj_GetFloat64(ptr, Properties::pctloadloss);
    }

    XfmrCode& pctloadloss(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctloadloss, value);
        return *this;
    }

    /// Percent no load losses at rated excitatation voltage. Default is 0. Converts to a resistance in parallel with the magnetizing impedance in each winding.
    ///
    /// DSS property name: %noloadloss, DSS property index: 25
    ///
    double pctnoloadloss()
    {
        return Obj_GetFloat64(ptr, Properties::pctnoloadloss);
    }

    XfmrCode& pctnoloadloss(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctnoloadloss, value);
        return *this;
    }

    /// Normal maximum kVA rating of H winding (winding 1).  Usually 100% - 110% ofmaximum nameplate rating, depending on load shape. Defaults to 110% of kVA rating of Winding 1.
    ///
    /// DSS property name: normhkVA, DSS property index: 26
    ///
    double normhkVA()
    {
        return Obj_GetFloat64(ptr, Properties::normhkVA);
    }

    XfmrCode& normhkVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::normhkVA, value);
        return *this;
    }

    /// Emergency (contingency)  kVA rating of H winding (winding 1).  Usually 140% - 150% ofmaximum nameplate rating, depending on load shape. Defaults to 150% of kVA rating of Winding 1.
    ///
    /// DSS property name: emerghkVA, DSS property index: 27
    ///
    double emerghkVA()
    {
        return Obj_GetFloat64(ptr, Properties::emerghkVA);
    }

    XfmrCode& emerghkVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::emerghkVA, value);
        return *this;
    }

    /// Max per unit tap for the active winding.  Default is 1.10
    ///
    /// DSS property name: MaxTap, DSS property index: 28
    ///
    VectorXd MaxTap()
    {
        return get_array<VectorXd>(Properties::MaxTap);
    }

    XfmrCode& MaxTap(VectorXd &value)
    {
        set_array<VectorXd>(Properties::MaxTap, value);
        return *this;
    }

    /// Min per unit tap for the active winding.  Default is 0.90
    ///
    /// DSS property name: MinTap, DSS property index: 29
    ///
    VectorXd MinTap()
    {
        return get_array<VectorXd>(Properties::MinTap);
    }

    XfmrCode& MinTap(VectorXd &value)
    {
        set_array<VectorXd>(Properties::MinTap, value);
        return *this;
    }

    /// Total number of taps between min and max tap.  Default is 32.
    ///
    /// DSS property name: NumTaps, DSS property index: 30
    ///
    VectorXi NumTaps()
    {
        return get_array<VectorXi>(Properties::NumTaps);
    }

    XfmrCode& NumTaps(VectorXi &value)
    {
        set_array<VectorXi>(Properties::NumTaps, value);
        return *this;
    }

    /// Percent magnetizing current. Default=0.0. Magnetizing branch is in parallel with windings in each phase. Also, see "ppm_antifloat".
    ///
    /// DSS property name: %imag, DSS property index: 31
    ///
    double pctimag()
    {
        return Obj_GetFloat64(ptr, Properties::pctimag);
    }

    XfmrCode& pctimag(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctimag, value);
        return *this;
    }

    /// Default=1 ppm.  Parts per million of transformer winding VA rating connected to ground to protect against accidentally floating a winding without a reference. If positive then the effect is adding a very large reactance to ground.  If negative, then a capacitor.
    ///
    /// DSS property name: ppm_antifloat, DSS property index: 32
    ///
    double ppm_antifloat()
    {
        return Obj_GetFloat64(ptr, Properties::ppm_antifloat);
    }

    XfmrCode& ppm_antifloat(double value)
    {
        Obj_SetFloat64(ptr, Properties::ppm_antifloat, value);
        return *this;
    }

    /// Use this property to specify all the winding %resistances using an array. Example:
    ///
    /// New Transformer.T1 buses="Hibus, lowbus" ~ %Rs=(0.2  0.3)
    ///
    /// DSS property name: %Rs, DSS property index: 33
    ///
    VectorXd pctRs()
    {
        return get_array<VectorXd>(Properties::pctRs);
    }

    XfmrCode& pctRs(VectorXd &value)
    {
        set_array<VectorXd>(Properties::pctRs, value);
        return *this;
    }

    /// Alternative to XHL for specifying the percent reactance from winding 1 to winding 2.  Use for 2- or 3-winding transformers. Percent on the kVA base of winding 1.
    ///
    /// DSS property name: X12, DSS property index: 34
    ///
    double X12()
    {
        return Obj_GetFloat64(ptr, Properties::X12);
    }

    XfmrCode& X12(double value)
    {
        Obj_SetFloat64(ptr, Properties::X12, value);
        return *this;
    }

    /// Alternative to XHT for specifying the percent reactance from winding 1 to winding 3.  Use for 3-winding transformers only. Percent on the kVA base of winding 1.
    ///
    /// DSS property name: X13, DSS property index: 35
    ///
    double X13()
    {
        return Obj_GetFloat64(ptr, Properties::X13);
    }

    XfmrCode& X13(double value)
    {
        Obj_SetFloat64(ptr, Properties::X13, value);
        return *this;
    }

    /// Alternative to XLT for specifying the percent reactance from winding 2 to winding 3.Use for 3-winding transformers only. Percent on the kVA base of winding 1.
    ///
    /// DSS property name: X23, DSS property index: 36
    ///
    double X23()
    {
        return Obj_GetFloat64(ptr, Properties::X23);
    }

    XfmrCode& X23(double value)
    {
        Obj_SetFloat64(ptr, Properties::X23, value);
        return *this;
    }

    /// Winding dc resistance in OHMS. Useful for GIC analysis. From transformer test report. Defaults to 85% of %R property
    ///
    /// DSS property name: RdcOhms, DSS property index: 37
    ///
    VectorXd RdcOhms()
    {
        return get_array<VectorXd>(Properties::RdcOhms);
    }

    XfmrCode& RdcOhms(VectorXd &value)
    {
        set_array<VectorXd>(Properties::RdcOhms, value);
        return *this;
    }

    /// Defines the number of ratings to be defined for the transfomer, to be used only when defining seasonal ratings using the "Ratings" property.
    ///
    /// DSS property name: Seasons, DSS property index: 38
    ///
    int32_t Seasons()
    {
        return Obj_GetInt32(ptr, Properties::Seasons);
    }

    XfmrCode& Seasons(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Seasons, value);
        return *this;
    }

    /// An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
    /// multiple ratings to change during a QSTS simulation to evaluate different ratings in transformers.
    ///
    /// DSS property name: Ratings, DSS property index: 39
    ///
    VectorXd Ratings()
    {
        return get_array<VectorXd>(Properties::Ratings);
    }

    XfmrCode& Ratings(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Ratings, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 40
    ///
    XfmrCode& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 40
    ///
    XfmrCode& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Line: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 15;
    struct Properties
    {
        enum {
            bus1 = 1,
            bus2 = 2,
            linecode = 3,
            length = 4,
            phases = 5,
            r1 = 6,
            x1 = 7,
            r0 = 8,
            x0 = 9,
            C1 = 10,
            C0 = 11,
            rmatrix = 12,
            xmatrix = 13,
            cmatrix = 14,
            Switch = 15,
            Rg = 16,
            Xg = 17,
            rho = 18,
            geometry = 19,
            units = 20,
            spacing = 21,
            wires = 22,
            EarthModel = 23,
            cncables = 24,
            tscables = 25,
            B1 = 26,
            B0 = 27,
            Seasons = 28,
            Ratings = 29,
            LineType = 30,
            normamps = 31,
            emergamps = 32,
            faultrate = 33,
            pctperm = 34,
            repair = 35,
            basefreq = 36,
            enabled = 37,
            like = 38,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Line(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Line(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Line element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Line(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Line element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Line& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Line& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name of bus to which first terminal is connected.
    /// Example:
    /// bus1=busname   (assumes all terminals connected in normal phase order)
    /// bus1=busname.3.1.2.0 (specify terminal to node connections explicitly)
    ///
    /// DSS property name: bus1, DSS property index: 1
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    Line& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    Line& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Name of bus to which 2nd terminal is connected.
    ///
    /// DSS property name: bus2, DSS property index: 2
    ///
    string bus2()
    {
        return get_prop_string(Properties::bus2);
    }

    Line& bus2(const string &value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    Line& bus2(const char* value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    /// Name of linecode object describing line impedances.
    /// If you use a line code, you do not need to specify the impedances here. The line code must have been PREVIOUSLY defined. The values specified last will prevail over those specified earlier (left-to-right sequence of properties).  You can subsequently change the number of phases if symmetrical component quantities are specified.If no line code or impedance data are specified, the line object defaults to 336 MCM ACSR on 4 ft spacing.
    ///
    /// DSS property name: linecode, DSS property index: 3
    ///
    string linecode()
    {
        return get_prop_string(Properties::linecode);
    }

    Line& linecode(const string &value)
    {
        set_string(Properties::linecode, value);
        return *this;
    }

    Line& linecode(dss::obj::LineCode &value)
    {
        set_obj(Properties::linecode, value);
        return *this;
    }

    /// Name of linecode object describing line impedances.
    /// If you use a line code, you do not need to specify the impedances here. The line code must have been PREVIOUSLY defined. The values specified last will prevail over those specified earlier (left-to-right sequence of properties).  You can subsequently change the number of phases if symmetrical component quantities are specified.If no line code or impedance data are specified, the line object defaults to 336 MCM ACSR on 4 ft spacing.
    ///
    /// DSS property name: linecode, DSS property index: 3
    ///
    dss::obj::LineCode linecode_obj()
    {
        return get_obj<dss::obj::LineCode>(Properties::linecode);
    }

    Line& linecode_obj(dss::obj::LineCode &value)
    {
        set_obj(Properties::linecode, value);
        return *this;
    }

    /// Length of line. Default is 1.0. If units do not match the impedance data, specify "units" property.
    ///
    /// DSS property name: length, DSS property index: 4
    ///
    double length()
    {
        return Obj_GetFloat64(ptr, Properties::length);
    }

    Line& length(double value)
    {
        Obj_SetFloat64(ptr, Properties::length, value);
        return *this;
    }

    /// Number of phases, this line.
    ///
    /// DSS property name: phases, DSS property index: 5
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    Line& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Positive-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Rmatrix.
    ///
    /// DSS property name: r1, DSS property index: 6
    ///
    double r1()
    {
        return Obj_GetFloat64(ptr, Properties::r1);
    }

    Line& r1(double value)
    {
        Obj_SetFloat64(ptr, Properties::r1, value);
        return *this;
    }

    /// Positive-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.  See also Xmatrix
    ///
    /// DSS property name: x1, DSS property index: 7
    ///
    double x1()
    {
        return Obj_GetFloat64(ptr, Properties::x1);
    }

    Line& x1(double value)
    {
        Obj_SetFloat64(ptr, Properties::x1, value);
        return *this;
    }

    /// Zero-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.
    ///
    /// DSS property name: r0, DSS property index: 8
    ///
    double r0()
    {
        return Obj_GetFloat64(ptr, Properties::r0);
    }

    Line& r0(double value)
    {
        Obj_SetFloat64(ptr, Properties::r0, value);
        return *this;
    }

    /// Zero-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.
    ///
    /// DSS property name: x0, DSS property index: 9
    ///
    double x0()
    {
        return Obj_GetFloat64(ptr, Properties::x0);
    }

    Line& x0(double value)
    {
        Obj_SetFloat64(ptr, Properties::x0, value);
        return *this;
    }

    /// Positive-sequence capacitance, nf per unit length.  Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Cmatrix and B1.
    ///
    /// DSS property name: C1, DSS property index: 10
    ///
    double C1()
    {
        return Obj_GetFloat64(ptr, Properties::C1);
    }

    Line& C1(double value)
    {
        Obj_SetFloat64(ptr, Properties::C1, value);
        return *this;
    }

    /// Zero-sequence capacitance, nf per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.See also B0.
    ///
    /// DSS property name: C0, DSS property index: 11
    ///
    double C0()
    {
        return Obj_GetFloat64(ptr, Properties::C0);
    }

    Line& C0(double value)
    {
        Obj_SetFloat64(ptr, Properties::C0, value);
        return *this;
    }

    /// Resistance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration. Using any of Rmatrix, Xmatrix, Cmatrix forces program to use the matrix values for line impedance definition. For balanced line models, you may use the standard symmetrical component data definition instead.
    ///
    /// DSS property name: rmatrix, DSS property index: 12
    ///
    VectorXd rmatrix()
    {
        return get_array<VectorXd>(Properties::rmatrix);
    }

    Line& rmatrix(VectorXd &value)
    {
        set_array<VectorXd>(Properties::rmatrix, value);
        return *this;
    }

    /// Reactance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration. Using any of Rmatrix, Xmatrix, Cmatrix forces program to use the matrix values for line impedance definition.  For balanced line models, you may use the standard symmetrical component data definition instead.
    ///
    /// DSS property name: xmatrix, DSS property index: 13
    ///
    VectorXd xmatrix()
    {
        return get_array<VectorXd>(Properties::xmatrix);
    }

    Line& xmatrix(VectorXd &value)
    {
        set_array<VectorXd>(Properties::xmatrix, value);
        return *this;
    }

    /// Nodal Capacitance matrix, lower triangle, nf per unit length.Order of the matrix is the number of phases. May be used to specify the shunt capacitance of any line configuration. Using any of Rmatrix, Xmatrix, Cmatrix forces program to use the matrix values for line impedance definition.  For balanced line models, you may use the standard symmetrical component data definition instead.
    ///
    /// DSS property name: cmatrix, DSS property index: 14
    ///
    VectorXd cmatrix()
    {
        return get_array<VectorXd>(Properties::cmatrix);
    }

    Line& cmatrix(VectorXd &value)
    {
        set_array<VectorXd>(Properties::cmatrix, value);
        return *this;
    }

    /// {y/n | T/F}  Default= no/false.  Designates this line as a switch for graphics and algorithmic purposes.
    /// SIDE EFFECT: Sets r1 = 1.0; x1 = 1.0; r0 = 1.0; x0 = 1.0; c1 = 1.1 ; c0 = 1.0;  length = 0.001; You must reset if you want something different.
    ///
    /// DSS property name: Switch, DSS property index: 15
    ///
    bool Switch()
    {
        return Obj_GetInt32(ptr, Properties::Switch) != 0;
    }

    Line& Switch(bool value)
    {
        Obj_SetInt32(ptr, Properties::Switch, value);
        return *this;
    }

    /// Carson earth return resistance per unit length used to compute impedance values at base frequency. Default is 0.01805 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.
    ///
    /// DSS property name: Rg, DSS property index: 16
    ///
    double Rg()
    {
        return Obj_GetFloat64(ptr, Properties::Rg);
    }

    Line& Rg(double value)
    {
        Obj_SetFloat64(ptr, Properties::Rg, value);
        return *this;
    }

    /// Carson earth return reactance per unit length used to compute impedance values at base frequency.  For making better frequency adjustments. Default is 0.155081 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.
    ///
    /// DSS property name: Xg, DSS property index: 17
    ///
    double Xg()
    {
        return Obj_GetFloat64(ptr, Properties::Xg);
    }

    Line& Xg(double value)
    {
        Obj_SetFloat64(ptr, Properties::Xg, value);
        return *this;
    }

    /// Default=100 meter ohms.  Earth resitivity used to compute earth correction factor. Overrides Line geometry definition if specified.
    ///
    /// DSS property name: rho, DSS property index: 18
    ///
    double rho()
    {
        return Obj_GetFloat64(ptr, Properties::rho);
    }

    Line& rho(double value)
    {
        Obj_SetFloat64(ptr, Properties::rho, value);
        return *this;
    }

    /// Geometry code for LineGeometry Object. Supercedes any previous definition of line impedance. Line constants are computed for each frequency change or rho change. CAUTION: may alter number of phases. You cannot subsequently change the number of phases unless you change how the line impedance is defined.
    ///
    /// DSS property name: geometry, DSS property index: 19
    ///
    string geometry()
    {
        return get_prop_string(Properties::geometry);
    }

    Line& geometry(const string &value)
    {
        set_string(Properties::geometry, value);
        return *this;
    }

    Line& geometry(dss::obj::LineGeometry &value)
    {
        set_obj(Properties::geometry, value);
        return *this;
    }

    /// Geometry code for LineGeometry Object. Supercedes any previous definition of line impedance. Line constants are computed for each frequency change or rho change. CAUTION: may alter number of phases. You cannot subsequently change the number of phases unless you change how the line impedance is defined.
    ///
    /// DSS property name: geometry, DSS property index: 19
    ///
    dss::obj::LineGeometry geometry_obj()
    {
        return get_obj<dss::obj::LineGeometry>(Properties::geometry);
    }

    Line& geometry_obj(dss::obj::LineGeometry &value)
    {
        set_obj(Properties::geometry, value);
        return *this;
    }

    /// Length Units = {none | mi|kft|km|m|Ft|in|cm } Default is None - assumes length units match impedance units.
    ///
    /// DSS property name: units, DSS property index: 20
    ///
    DimensionUnits units()
    {
        return DimensionUnits(Obj_GetInt32(ptr, Properties::units));
    }

    Line& units(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::units, value);
        return *this;
    }

    Line& units(DimensionUnits value)
    {
        Obj_SetInt32(ptr, Properties::units, int32_t(value));
        return *this;
    }

    Line& units(const string &value)
    {
        set_string(Properties::units, value);
        return *this;
    }

    Line& units(const char *value)
    {
        set_string(Properties::units, value);
        return *this;
    }

    /// Length Units = {none | mi|kft|km|m|Ft|in|cm } Default is None - assumes length units match impedance units.
    ///
    /// DSS property name: units, DSS property index: 20
    ///
    string units_str()
    {
        return get_prop_string(Properties::units);
    }

    /// Length Units = {none | mi|kft|km|m|Ft|in|cm } Default is None - assumes length units match impedance units.
    ///
    /// DSS property name: units, DSS property index: 20
    ///
    Line& units_str(const string &value)
    {
        set_string(Properties::units, value);
        return *this;
    }

    /// Reference to a LineSpacing for use in a line constants calculation.
    /// Must be used in conjunction with the Wires property.
    /// Specify this before the wires property.
    ///
    /// DSS property name: spacing, DSS property index: 21
    ///
    string spacing()
    {
        return get_prop_string(Properties::spacing);
    }

    Line& spacing(const string &value)
    {
        set_string(Properties::spacing, value);
        return *this;
    }

    Line& spacing(dss::obj::LineSpacing &value)
    {
        set_obj(Properties::spacing, value);
        return *this;
    }

    /// Reference to a LineSpacing for use in a line constants calculation.
    /// Must be used in conjunction with the Wires property.
    /// Specify this before the wires property.
    ///
    /// DSS property name: spacing, DSS property index: 21
    ///
    dss::obj::LineSpacing spacing_obj()
    {
        return get_obj<dss::obj::LineSpacing>(Properties::spacing);
    }

    Line& spacing_obj(dss::obj::LineSpacing &value)
    {
        set_obj(Properties::spacing, value);
        return *this;
    }

    /// Array of WireData names for use in an overhead line constants calculation.
    /// Must be used in conjunction with the Spacing property.
    /// Specify the Spacing first, and "ncond" wires.
    /// May also be used to specify bare neutrals with cables, using "ncond-nphase" wires.
    ///
    /// DSS property name: wires, DSS property index: 22
    ///
    strings wires()
    {
        return get_array<strings>(Properties::wires);
    }

    Line& wires(strings &value)
    {
        set_array<strings>(Properties::wires, value);
        return *this;
    }

    Line& wires(std::vector<dss::obj::WireData> &value)
    {
        set_array<std::vector<dss::obj::WireData>>(Properties::wires, value);
        return *this;
    }

    /// Array of WireData names for use in an overhead line constants calculation.
    /// Must be used in conjunction with the Spacing property.
    /// Specify the Spacing first, and "ncond" wires.
    /// May also be used to specify bare neutrals with cables, using "ncond-nphase" wires.
    ///
    /// DSS property name: wires, DSS property index: 22
    ///
    std::vector<dss::obj::WireData> wires_obj()
    {
        return get_array<std::vector<dss::obj::WireData>>(Properties::wires);
    }

    Line& wires_obj(std::vector<dss::obj::WireData> &value)
    {
        set_array<std::vector<dss::obj::WireData>>(Properties::wires, value);
        return *this;
    }

    /// One of {Carson | FullCarson | Deri}. Default is the global value established with the Set EarthModel command. See the Options Help on EarthModel option. This is used to override the global value for this line. This option applies only when the "geometry" property is used.
    ///
    /// DSS property name: EarthModel, DSS property index: 23
    ///
    EarthModel earthmodel()
    {
        return EarthModel(Obj_GetInt32(ptr, Properties::EarthModel));
    }

    Line& earthmodel(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::EarthModel, value);
        return *this;
    }

    Line& earthmodel(EarthModel value)
    {
        Obj_SetInt32(ptr, Properties::EarthModel, int32_t(value));
        return *this;
    }

    Line& earthmodel(const string &value)
    {
        set_string(Properties::EarthModel, value);
        return *this;
    }

    Line& earthmodel(const char *value)
    {
        set_string(Properties::EarthModel, value);
        return *this;
    }

    /// One of {Carson | FullCarson | Deri}. Default is the global value established with the Set EarthModel command. See the Options Help on EarthModel option. This is used to override the global value for this line. This option applies only when the "geometry" property is used.
    ///
    /// DSS property name: EarthModel, DSS property index: 23
    ///
    string earthmodel_str()
    {
        return get_prop_string(Properties::EarthModel);
    }

    /// One of {Carson | FullCarson | Deri}. Default is the global value established with the Set EarthModel command. See the Options Help on EarthModel option. This is used to override the global value for this line. This option applies only when the "geometry" property is used.
    ///
    /// DSS property name: EarthModel, DSS property index: 23
    ///
    Line& earthmodel_str(const string &value)
    {
        set_string(Properties::EarthModel, value);
        return *this;
    }

    /// Array of CNData names for use in a cable constants calculation.
    /// Must be used in conjunction with the Spacing property.
    /// Specify the Spacing first, using "nphases" cncables.
    /// You may later specify "nconds-nphases" wires for separate neutrals
    ///
    /// DSS property name: cncables, DSS property index: 24
    ///
    strings cncables()
    {
        return get_array<strings>(Properties::cncables);
    }

    Line& cncables(strings &value)
    {
        set_array<strings>(Properties::cncables, value);
        return *this;
    }

    Line& cncables(std::vector<dss::obj::CNData> &value)
    {
        set_array<std::vector<dss::obj::CNData>>(Properties::cncables, value);
        return *this;
    }

    /// Array of CNData names for use in a cable constants calculation.
    /// Must be used in conjunction with the Spacing property.
    /// Specify the Spacing first, using "nphases" cncables.
    /// You may later specify "nconds-nphases" wires for separate neutrals
    ///
    /// DSS property name: cncables, DSS property index: 24
    ///
    std::vector<dss::obj::CNData> cncables_obj()
    {
        return get_array<std::vector<dss::obj::CNData>>(Properties::cncables);
    }

    Line& cncables_obj(std::vector<dss::obj::CNData> &value)
    {
        set_array<std::vector<dss::obj::CNData>>(Properties::cncables, value);
        return *this;
    }

    /// Array of TSData names for use in a cable constants calculation.
    /// Must be used in conjunction with the Spacing property.
    /// Specify the Spacing first, using "nphases" tscables.
    /// You may later specify "nconds-nphases" wires for separate neutrals
    ///
    /// DSS property name: tscables, DSS property index: 25
    ///
    strings tscables()
    {
        return get_array<strings>(Properties::tscables);
    }

    Line& tscables(strings &value)
    {
        set_array<strings>(Properties::tscables, value);
        return *this;
    }

    Line& tscables(std::vector<dss::obj::TSData> &value)
    {
        set_array<std::vector<dss::obj::TSData>>(Properties::tscables, value);
        return *this;
    }

    /// Array of TSData names for use in a cable constants calculation.
    /// Must be used in conjunction with the Spacing property.
    /// Specify the Spacing first, using "nphases" tscables.
    /// You may later specify "nconds-nphases" wires for separate neutrals
    ///
    /// DSS property name: tscables, DSS property index: 25
    ///
    std::vector<dss::obj::TSData> tscables_obj()
    {
        return get_array<std::vector<dss::obj::TSData>>(Properties::tscables);
    }

    Line& tscables_obj(std::vector<dss::obj::TSData> &value)
    {
        set_array<std::vector<dss::obj::TSData>>(Properties::tscables, value);
        return *this;
    }

    /// Alternate way to specify C1. MicroS per unit length
    ///
    /// DSS property name: B1, DSS property index: 26
    ///
    double B1()
    {
        return Obj_GetFloat64(ptr, Properties::B1);
    }

    Line& B1(double value)
    {
        Obj_SetFloat64(ptr, Properties::B1, value);
        return *this;
    }

    /// Alternate way to specify C0. MicroS per unit length
    ///
    /// DSS property name: B0, DSS property index: 27
    ///
    double B0()
    {
        return Obj_GetFloat64(ptr, Properties::B0);
    }

    Line& B0(double value)
    {
        Obj_SetFloat64(ptr, Properties::B0, value);
        return *this;
    }

    /// Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.
    ///
    /// DSS property name: Seasons, DSS property index: 28
    ///
    int32_t Seasons()
    {
        return Obj_GetInt32(ptr, Properties::Seasons);
    }

    Line& Seasons(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Seasons, value);
        return *this;
    }

    /// An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
    /// multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.
    ///
    /// DSS property name: Ratings, DSS property index: 29
    ///
    VectorXd Ratings()
    {
        return get_array<VectorXd>(Properties::Ratings);
    }

    Line& Ratings(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Ratings, value);
        return *this;
    }

    /// Code designating the type of line.
    /// One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW
    ///
    /// OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.
    ///
    /// DSS property name: LineType, DSS property index: 30
    ///
    LineType linetype()
    {
        return LineType(Obj_GetInt32(ptr, Properties::LineType));
    }

    Line& linetype(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::LineType, value);
        return *this;
    }

    Line& linetype(LineType value)
    {
        Obj_SetInt32(ptr, Properties::LineType, int32_t(value));
        return *this;
    }

    Line& linetype(const string &value)
    {
        set_string(Properties::LineType, value);
        return *this;
    }

    Line& linetype(const char *value)
    {
        set_string(Properties::LineType, value);
        return *this;
    }

    /// Code designating the type of line.
    /// One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW
    ///
    /// OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.
    ///
    /// DSS property name: LineType, DSS property index: 30
    ///
    string linetype_str()
    {
        return get_prop_string(Properties::LineType);
    }

    /// Code designating the type of line.
    /// One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW
    ///
    /// OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.
    ///
    /// DSS property name: LineType, DSS property index: 30
    ///
    Line& linetype_str(const string &value)
    {
        set_string(Properties::LineType, value);
        return *this;
    }

    /// Normal rated current.
    ///
    /// DSS property name: normamps, DSS property index: 31
    ///
    double normamps()
    {
        return Obj_GetFloat64(ptr, Properties::normamps);
    }

    Line& normamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::normamps, value);
        return *this;
    }

    /// Maximum or emerg current.
    ///
    /// DSS property name: emergamps, DSS property index: 32
    ///
    double emergamps()
    {
        return Obj_GetFloat64(ptr, Properties::emergamps);
    }

    Line& emergamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::emergamps, value);
        return *this;
    }

    /// Failure rate PER UNIT LENGTH per year. Length must be same units as LENGTH property. Default is 0.1 fault per unit length per year.
    ///
    /// DSS property name: faultrate, DSS property index: 33
    ///
    double faultrate()
    {
        return Obj_GetFloat64(ptr, Properties::faultrate);
    }

    Line& faultrate(double value)
    {
        Obj_SetFloat64(ptr, Properties::faultrate, value);
        return *this;
    }

    /// Percent of failures that become permanent. Default is 20.
    ///
    /// DSS property name: pctperm, DSS property index: 34
    ///
    double pctperm()
    {
        return Obj_GetFloat64(ptr, Properties::pctperm);
    }

    Line& pctperm(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctperm, value);
        return *this;
    }

    /// Hours to repair. Default is 3 hr.
    ///
    /// DSS property name: repair, DSS property index: 35
    ///
    double repair()
    {
        return Obj_GetFloat64(ptr, Properties::repair);
    }

    Line& repair(double value)
    {
        Obj_SetFloat64(ptr, Properties::repair, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 36
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Line& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 37
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Line& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 38
    ///
    Line& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 38
    ///
    Line& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Vsource: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 16;
    struct Properties
    {
        enum {
            bus1 = 1,
            basekv = 2,
            pu = 3,
            angle = 4,
            frequency = 5,
            phases = 6,
            MVAsc3 = 7,
            MVAsc1 = 8,
            x1r1 = 9,
            x0r0 = 10,
            Isc3 = 11,
            Isc1 = 12,
            R1 = 13,
            X1 = 14,
            R0 = 15,
            X0 = 16,
            ScanType = 17,
            Sequence = 18,
            bus2 = 19,
            Z1 = 20,
            Z0 = 21,
            Z2 = 22,
            puZ1 = 23,
            puZ0 = 24,
            puZ2 = 25,
            baseMVA = 26,
            Yearly = 27,
            Daily = 28,
            Duty = 29,
            Model = 30,
            puZideal = 31,
            spectrum = 32,
            basefreq = 33,
            enabled = 34,
            like = 35,
        };
    };

    // Class-specific enumerations

        ///
        /// VSource: Model (DSS enumeration for Vsource)
        ///
    enum class VSourceModel: int32_t
    {
        Thevenin = 0, ///< Thevenin
        Ideal = 1 ///< Ideal
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Vsource(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Vsource(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Vsource element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Vsource(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Vsource element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Vsource& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Vsource& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name of bus to which the main terminal (1) is connected.
    /// bus1=busname
    /// bus1=busname.1.2.3
    ///
    /// The VSOURCE object is a two-terminal voltage source (thevenin equivalent). Bus2 defaults to Bus1 with all phases connected to ground (node 0) unless previously specified. This is a Yg connection. If you want something different, define the Bus2 property ezplicitly.
    ///
    /// DSS property name: bus1, DSS property index: 1
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    Vsource& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    Vsource& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Base Source kV, usually phase-phase (L-L) unless you are making a positive-sequence model or 1-phase modelin which case, it will be phase-neutral (L-N) kV.
    ///
    /// DSS property name: basekv, DSS property index: 2
    ///
    double basekv()
    {
        return Obj_GetFloat64(ptr, Properties::basekv);
    }

    Vsource& basekv(double value)
    {
        Obj_SetFloat64(ptr, Properties::basekv, value);
        return *this;
    }

    /// Per unit of the base voltage that the source is actually operating at.
    /// "pu=1.05"
    ///
    /// DSS property name: pu, DSS property index: 3
    ///
    double pu()
    {
        return Obj_GetFloat64(ptr, Properties::pu);
    }

    Vsource& pu(double value)
    {
        Obj_SetFloat64(ptr, Properties::pu, value);
        return *this;
    }

    /// Phase angle in degrees of first phase: e.g.,Angle=10.3
    ///
    /// DSS property name: angle, DSS property index: 4
    ///
    double angle()
    {
        return Obj_GetFloat64(ptr, Properties::angle);
    }

    Vsource& angle(double value)
    {
        Obj_SetFloat64(ptr, Properties::angle, value);
        return *this;
    }

    /// Source frequency.  Defaults to system default base frequency.
    ///
    /// DSS property name: frequency, DSS property index: 5
    ///
    double frequency()
    {
        return Obj_GetFloat64(ptr, Properties::frequency);
    }

    Vsource& frequency(double value)
    {
        Obj_SetFloat64(ptr, Properties::frequency, value);
        return *this;
    }

    /// Number of phases.  Defaults to 3.
    ///
    /// DSS property name: phases, DSS property index: 6
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    Vsource& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// MVA Short circuit, 3-phase fault. Default = 2000. Z1 is determined by squaring the base kv and dividing by this value. For single-phase source, this value is not used.
    ///
    /// DSS property name: MVAsc3, DSS property index: 7
    ///
    double MVAsc3()
    {
        return Obj_GetFloat64(ptr, Properties::MVAsc3);
    }

    Vsource& MVAsc3(double value)
    {
        Obj_SetFloat64(ptr, Properties::MVAsc3, value);
        return *this;
    }

    /// MVA Short Circuit, 1-phase fault. Default = 2100. The "single-phase impedance", Zs, is determined by squaring the base kV and dividing by this value. Then Z0 is determined by Z0 = 3Zs - 2Z1.  For 1-phase sources, Zs is used directly. Use X0R0 to define X/R ratio for 1-phase source.
    ///
    /// DSS property name: MVAsc1, DSS property index: 8
    ///
    double MVAsc1()
    {
        return Obj_GetFloat64(ptr, Properties::MVAsc1);
    }

    Vsource& MVAsc1(double value)
    {
        Obj_SetFloat64(ptr, Properties::MVAsc1, value);
        return *this;
    }

    /// Positive-sequence  X/R ratio. Default = 4.
    ///
    /// DSS property name: x1r1, DSS property index: 9
    ///
    double x1r1()
    {
        return Obj_GetFloat64(ptr, Properties::x1r1);
    }

    Vsource& x1r1(double value)
    {
        Obj_SetFloat64(ptr, Properties::x1r1, value);
        return *this;
    }

    /// Zero-sequence X/R ratio.Default = 3.
    ///
    /// DSS property name: x0r0, DSS property index: 10
    ///
    double x0r0()
    {
        return Obj_GetFloat64(ptr, Properties::x0r0);
    }

    Vsource& x0r0(double value)
    {
        Obj_SetFloat64(ptr, Properties::x0r0, value);
        return *this;
    }

    /// Alternate method of defining the source impedance.
    /// 3-phase short circuit current, amps.  Default is 10000.
    ///
    /// DSS property name: Isc3, DSS property index: 11
    ///
    double Isc3()
    {
        return Obj_GetFloat64(ptr, Properties::Isc3);
    }

    Vsource& Isc3(double value)
    {
        Obj_SetFloat64(ptr, Properties::Isc3, value);
        return *this;
    }

    /// Alternate method of defining the source impedance.
    /// single-phase short circuit current, amps.  Default is 10500.
    ///
    /// DSS property name: Isc1, DSS property index: 12
    ///
    double Isc1()
    {
        return Obj_GetFloat64(ptr, Properties::Isc1);
    }

    Vsource& Isc1(double value)
    {
        Obj_SetFloat64(ptr, Properties::Isc1, value);
        return *this;
    }

    /// Alternate method of defining the source impedance.
    /// Positive-sequence resistance, ohms.  Default is 1.65.
    ///
    /// DSS property name: R1, DSS property index: 13
    ///
    double R1()
    {
        return Obj_GetFloat64(ptr, Properties::R1);
    }

    Vsource& R1(double value)
    {
        Obj_SetFloat64(ptr, Properties::R1, value);
        return *this;
    }

    /// Alternate method of defining the source impedance.
    /// Positive-sequence reactance, ohms.  Default is 6.6.
    ///
    /// DSS property name: X1, DSS property index: 14
    ///
    double X1()
    {
        return Obj_GetFloat64(ptr, Properties::X1);
    }

    Vsource& X1(double value)
    {
        Obj_SetFloat64(ptr, Properties::X1, value);
        return *this;
    }

    /// Alternate method of defining the source impedance.
    /// Zero-sequence resistance, ohms.  Default is 1.9.
    ///
    /// DSS property name: R0, DSS property index: 15
    ///
    double R0()
    {
        return Obj_GetFloat64(ptr, Properties::R0);
    }

    Vsource& R0(double value)
    {
        Obj_SetFloat64(ptr, Properties::R0, value);
        return *this;
    }

    /// Alternate method of defining the source impedance.
    /// Zero-sequence reactance, ohms.  Default is 5.7.
    ///
    /// DSS property name: X0, DSS property index: 16
    ///
    double X0()
    {
        return Obj_GetFloat64(ptr, Properties::X0);
    }

    Vsource& X0(double value)
    {
        Obj_SetFloat64(ptr, Properties::X0, value);
        return *this;
    }

    /// {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.
    ///
    /// DSS property name: ScanType, DSS property index: 17
    ///
    ScanType scantype()
    {
        return ScanType(Obj_GetInt32(ptr, Properties::ScanType));
    }

    Vsource& scantype(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::ScanType, value);
        return *this;
    }

    Vsource& scantype(ScanType value)
    {
        Obj_SetInt32(ptr, Properties::ScanType, int32_t(value));
        return *this;
    }

    Vsource& scantype(const string &value)
    {
        set_string(Properties::ScanType, value);
        return *this;
    }

    Vsource& scantype(const char *value)
    {
        set_string(Properties::ScanType, value);
        return *this;
    }

    /// {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.
    ///
    /// DSS property name: ScanType, DSS property index: 17
    ///
    string scantype_str()
    {
        return get_prop_string(Properties::ScanType);
    }

    /// {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.
    ///
    /// DSS property name: ScanType, DSS property index: 17
    ///
    Vsource& scantype_str(const string &value)
    {
        set_string(Properties::ScanType, value);
        return *this;
    }

    /// {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence.
    ///
    /// DSS property name: Sequence, DSS property index: 18
    ///
    SequenceType Sequence()
    {
        return SequenceType(Obj_GetInt32(ptr, Properties::Sequence));
    }

    Vsource& Sequence(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Sequence, value);
        return *this;
    }

    Vsource& Sequence(SequenceType value)
    {
        Obj_SetInt32(ptr, Properties::Sequence, int32_t(value));
        return *this;
    }

    Vsource& Sequence(const string &value)
    {
        set_string(Properties::Sequence, value);
        return *this;
    }

    Vsource& Sequence(const char *value)
    {
        set_string(Properties::Sequence, value);
        return *this;
    }

    /// {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence.
    ///
    /// DSS property name: Sequence, DSS property index: 18
    ///
    string Sequence_str()
    {
        return get_prop_string(Properties::Sequence);
    }

    /// {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence.
    ///
    /// DSS property name: Sequence, DSS property index: 18
    ///
    Vsource& Sequence_str(const string &value)
    {
        set_string(Properties::Sequence, value);
        return *this;
    }

    /// Name of bus to which 2nd terminal is connected.
    /// bus2=busname
    /// bus2=busname.1.2.3
    ///
    /// Default is Bus1.0.0.0 (grounded wye connection)
    ///
    /// DSS property name: bus2, DSS property index: 19
    ///
    string bus2()
    {
        return get_prop_string(Properties::bus2);
    }

    Vsource& bus2(const string &value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    Vsource& bus2(const char* value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    /// Positive-sequence equivalent source impedance, ohms, as a 2-element array representing a complex number. Example:
    ///
    /// Z1=[1, 2]  ! represents 1 + j2
    ///
    /// If defined, Z1, Z2, and Z0 are used to define the impedance matrix of the VSOURCE. Z1 MUST BE DEFINED TO USE THIS OPTION FOR DEFINING THE MATRIX.
    ///
    /// Side Effect: Sets Z2 and Z0 to same values unless they were previously defined.
    ///
    /// DSS property name: Z1, DSS property index: 20
    ///
    complex Z1()
    {
        return get_complex(Properties::Z1);
    }
    Vsource& Z1(complex value)
    {
        set_complex(Properties::Z1, value);
        return *this;
    }

    /// Zero-sequence equivalent source impedance, ohms, as a 2-element array representing a complex number. Example:
    ///
    /// Z0=[3, 4]  ! represents 3 + j4
    ///
    /// Used to define the impedance matrix of the VSOURCE if Z1 is also specified.
    ///
    /// Note: Z0 defaults to Z1 if it is not specifically defined.
    ///
    /// DSS property name: Z0, DSS property index: 21
    ///
    complex Z0()
    {
        return get_complex(Properties::Z0);
    }
    Vsource& Z0(complex value)
    {
        set_complex(Properties::Z0, value);
        return *this;
    }

    /// Negative-sequence equivalent source impedance, ohms, as a 2-element array representing a complex number. Example:
    ///
    /// Z2=[1, 2]  ! represents 1 + j2
    ///
    /// Used to define the impedance matrix of the VSOURCE if Z1 is also specified.
    ///
    /// Note: Z2 defaults to Z1 if it is not specifically defined. If Z2 is not equal to Z1, the impedance matrix is asymmetrical.
    ///
    /// DSS property name: Z2, DSS property index: 22
    ///
    complex Z2()
    {
        return get_complex(Properties::Z2);
    }
    Vsource& Z2(complex value)
    {
        set_complex(Properties::Z2, value);
        return *this;
    }

    /// 2-element array: e.g., [1  2]. An alternate way to specify Z1. See Z1 property. Per-unit positive-sequence impedance on base of Vsource BasekV and BaseMVA.
    ///
    /// DSS property name: puZ1, DSS property index: 23
    ///
    complex puZ1()
    {
        return get_complex(Properties::puZ1);
    }
    Vsource& puZ1(complex value)
    {
        set_complex(Properties::puZ1, value);
        return *this;
    }

    /// 2-element array: e.g., [1  2]. An alternate way to specify Z0. See Z0 property. Per-unit zero-sequence impedance on base of Vsource BasekV and BaseMVA.
    ///
    /// DSS property name: puZ0, DSS property index: 24
    ///
    complex puZ0()
    {
        return get_complex(Properties::puZ0);
    }
    Vsource& puZ0(complex value)
    {
        set_complex(Properties::puZ0, value);
        return *this;
    }

    /// 2-element array: e.g., [1  2]. An alternate way to specify Z2. See Z2 property. Per-unit negative-sequence impedance on base of Vsource BasekV and BaseMVA.
    ///
    /// DSS property name: puZ2, DSS property index: 25
    ///
    complex puZ2()
    {
        return get_complex(Properties::puZ2);
    }
    Vsource& puZ2(complex value)
    {
        set_complex(Properties::puZ2, value);
        return *this;
    }

    /// Default value is 100. Base used to convert values specifiied with puZ1, puZ0, and puZ2 properties to ohms on kV base specified by BasekV property.
    ///
    /// DSS property name: baseMVA, DSS property index: 26
    ///
    double baseMVA()
    {
        return Obj_GetFloat64(ptr, Properties::baseMVA);
    }

    Vsource& baseMVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::baseMVA, value);
        return *this;
    }

    /// LOADSHAPE object to use for the per-unit voltage for YEARLY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.
    ///
    /// Must be previously defined as a LOADSHAPE object.
    ///
    /// Is set to the Daily load shape when Daily is defined.  The daily load shape is repeated in this case. Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.
    ///
    /// DSS property name: Yearly, DSS property index: 27
    ///
    string Yearly()
    {
        return get_prop_string(Properties::Yearly);
    }

    Vsource& Yearly(const string &value)
    {
        set_string(Properties::Yearly, value);
        return *this;
    }

    Vsource& Yearly(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Yearly, value);
        return *this;
    }

    /// LOADSHAPE object to use for the per-unit voltage for YEARLY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.
    ///
    /// Must be previously defined as a LOADSHAPE object.
    ///
    /// Is set to the Daily load shape when Daily is defined.  The daily load shape is repeated in this case. Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.
    ///
    /// DSS property name: Yearly, DSS property index: 27
    ///
    dss::obj::LoadShape Yearly_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::Yearly);
    }

    Vsource& Yearly_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Yearly, value);
        return *this;
    }

    /// LOADSHAPE object to use for the per-unit voltage for DAILY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.
    ///
    /// Must be previously defined as a LOADSHAPE object.
    ///
    /// Sets Yearly curve if it is not already defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.
    ///
    /// DSS property name: Daily, DSS property index: 28
    ///
    string Daily()
    {
        return get_prop_string(Properties::Daily);
    }

    Vsource& Daily(const string &value)
    {
        set_string(Properties::Daily, value);
        return *this;
    }

    Vsource& Daily(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Daily, value);
        return *this;
    }

    /// LOADSHAPE object to use for the per-unit voltage for DAILY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.
    ///
    /// Must be previously defined as a LOADSHAPE object.
    ///
    /// Sets Yearly curve if it is not already defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.
    ///
    /// DSS property name: Daily, DSS property index: 28
    ///
    dss::obj::LoadShape Daily_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::Daily);
    }

    Vsource& Daily_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Daily, value);
        return *this;
    }

    /// LOADSHAPE object to use for the per-unit voltage for DUTYCYCLE-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.
    ///
    /// Must be previously defined as a LOADSHAPE object.
    ///
    /// Defaults to Daily load shape when Daily is defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.
    ///
    /// DSS property name: Duty, DSS property index: 29
    ///
    string Duty()
    {
        return get_prop_string(Properties::Duty);
    }

    Vsource& Duty(const string &value)
    {
        set_string(Properties::Duty, value);
        return *this;
    }

    Vsource& Duty(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Duty, value);
        return *this;
    }

    /// LOADSHAPE object to use for the per-unit voltage for DUTYCYCLE-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.
    ///
    /// Must be previously defined as a LOADSHAPE object.
    ///
    /// Defaults to Daily load shape when Daily is defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.
    ///
    /// DSS property name: Duty, DSS property index: 29
    ///
    dss::obj::LoadShape Duty_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::Duty);
    }

    Vsource& Duty_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Duty, value);
        return *this;
    }

    /// {Thevenin* | Ideal}  Specifies whether the Vsource is to be considered a Thevenin short circuit model or a quasi-ideal voltage source. If Thevenin, the Vsource uses the impedances defined for all calculations. If "Ideal", the model uses a small impedance on the diagonal of the impedance matrix for the fundamental base frequency power flow only. Then switches to actual Thevenin model for other frequencies.
    ///
    /// DSS property name: Model, DSS property index: 30
    ///
    VSourceModel Model()
    {
        return VSourceModel(Obj_GetInt32(ptr, Properties::Model));
    }

    Vsource& Model(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Model, value);
        return *this;
    }

    Vsource& Model(VSourceModel value)
    {
        Obj_SetInt32(ptr, Properties::Model, int32_t(value));
        return *this;
    }

    Vsource& Model(const string &value)
    {
        set_string(Properties::Model, value);
        return *this;
    }

    Vsource& Model(const char *value)
    {
        set_string(Properties::Model, value);
        return *this;
    }

    /// {Thevenin* | Ideal}  Specifies whether the Vsource is to be considered a Thevenin short circuit model or a quasi-ideal voltage source. If Thevenin, the Vsource uses the impedances defined for all calculations. If "Ideal", the model uses a small impedance on the diagonal of the impedance matrix for the fundamental base frequency power flow only. Then switches to actual Thevenin model for other frequencies.
    ///
    /// DSS property name: Model, DSS property index: 30
    ///
    string Model_str()
    {
        return get_prop_string(Properties::Model);
    }

    /// {Thevenin* | Ideal}  Specifies whether the Vsource is to be considered a Thevenin short circuit model or a quasi-ideal voltage source. If Thevenin, the Vsource uses the impedances defined for all calculations. If "Ideal", the model uses a small impedance on the diagonal of the impedance matrix for the fundamental base frequency power flow only. Then switches to actual Thevenin model for other frequencies.
    ///
    /// DSS property name: Model, DSS property index: 30
    ///
    Vsource& Model_str(const string &value)
    {
        set_string(Properties::Model, value);
        return *this;
    }

    /// 2-element array: e.g., [1  2]. The pu impedance to use for the quasi-ideal voltage source model. Should be a very small impedances. Default is [1e-6, 0.001]. Per-unit impedance on base of Vsource BasekV and BaseMVA. If too small, solution may not work. Be sure to check the voltage values and powers.
    ///
    /// DSS property name: puZideal, DSS property index: 31
    ///
    complex puZideal()
    {
        return get_complex(Properties::puZideal);
    }
    Vsource& puZideal(complex value)
    {
        set_complex(Properties::puZideal, value);
        return *this;
    }

    /// Name of harmonic spectrum for this source.  Default is "defaultvsource", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 32
    ///
    string spectrum()
    {
        return get_prop_string(Properties::spectrum);
    }

    Vsource& spectrum(const string &value)
    {
        set_string(Properties::spectrum, value);
        return *this;
    }

    Vsource& spectrum(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Name of harmonic spectrum for this source.  Default is "defaultvsource", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 32
    ///
    dss::obj::Spectrum spectrum_obj()
    {
        return get_obj<dss::obj::Spectrum>(Properties::spectrum);
    }

    Vsource& spectrum_obj(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 33
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Vsource& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 34
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Vsource& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 35
    ///
    Vsource& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 35
    ///
    Vsource& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Isource: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 17;
    struct Properties
    {
        enum {
            bus1 = 1,
            amps = 2,
            angle = 3,
            frequency = 4,
            phases = 5,
            scantype = 6,
            sequence = 7,
            Yearly = 8,
            Daily = 9,
            Duty = 10,
            Bus2 = 11,
            spectrum = 12,
            basefreq = 13,
            enabled = 14,
            like = 15,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Isource(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Isource(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Isource element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Isource(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Isource element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Isource& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Isource& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name of bus to which source is connected.
    /// bus1=busname
    /// bus1=busname.1.2.3
    ///
    /// DSS property name: bus1, DSS property index: 1
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    Isource& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    Isource& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Magnitude of current source, each phase, in Amps.
    ///
    /// DSS property name: amps, DSS property index: 2
    ///
    double amps()
    {
        return Obj_GetFloat64(ptr, Properties::amps);
    }

    Isource& amps(double value)
    {
        Obj_SetFloat64(ptr, Properties::amps, value);
        return *this;
    }

    /// Phase angle in degrees of first phase: e.g.,Angle=10.3.
    /// Phase shift between phases is assumed 120 degrees when number of phases <= 3
    ///
    /// DSS property name: angle, DSS property index: 3
    ///
    double angle()
    {
        return Obj_GetFloat64(ptr, Properties::angle);
    }

    Isource& angle(double value)
    {
        Obj_SetFloat64(ptr, Properties::angle, value);
        return *this;
    }

    /// Source frequency.  Defaults to  circuit fundamental frequency.
    ///
    /// DSS property name: frequency, DSS property index: 4
    ///
    double frequency()
    {
        return Obj_GetFloat64(ptr, Properties::frequency);
    }

    Isource& frequency(double value)
    {
        Obj_SetFloat64(ptr, Properties::frequency, value);
        return *this;
    }

    /// Number of phases.  Defaults to 3. For 3 or less, phase shift is 120 degrees.
    ///
    /// DSS property name: phases, DSS property index: 5
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    Isource& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.
    ///
    /// DSS property name: scantype, DSS property index: 6
    ///
    ScanType scantype()
    {
        return ScanType(Obj_GetInt32(ptr, Properties::scantype));
    }

    Isource& scantype(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::scantype, value);
        return *this;
    }

    Isource& scantype(ScanType value)
    {
        Obj_SetInt32(ptr, Properties::scantype, int32_t(value));
        return *this;
    }

    Isource& scantype(const string &value)
    {
        set_string(Properties::scantype, value);
        return *this;
    }

    Isource& scantype(const char *value)
    {
        set_string(Properties::scantype, value);
        return *this;
    }

    /// {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.
    ///
    /// DSS property name: scantype, DSS property index: 6
    ///
    string scantype_str()
    {
        return get_prop_string(Properties::scantype);
    }

    /// {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.
    ///
    /// DSS property name: scantype, DSS property index: 6
    ///
    Isource& scantype_str(const string &value)
    {
        set_string(Properties::scantype, value);
        return *this;
    }

    /// {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence.
    ///
    /// DSS property name: sequence, DSS property index: 7
    ///
    SequenceType sequence()
    {
        return SequenceType(Obj_GetInt32(ptr, Properties::sequence));
    }

    Isource& sequence(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::sequence, value);
        return *this;
    }

    Isource& sequence(SequenceType value)
    {
        Obj_SetInt32(ptr, Properties::sequence, int32_t(value));
        return *this;
    }

    Isource& sequence(const string &value)
    {
        set_string(Properties::sequence, value);
        return *this;
    }

    Isource& sequence(const char *value)
    {
        set_string(Properties::sequence, value);
        return *this;
    }

    /// {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence.
    ///
    /// DSS property name: sequence, DSS property index: 7
    ///
    string sequence_str()
    {
        return get_prop_string(Properties::sequence);
    }

    /// {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence.
    ///
    /// DSS property name: sequence, DSS property index: 7
    ///
    Isource& sequence_str(const string &value)
    {
        set_string(Properties::sequence, value);
        return *this;
    }

    /// LOADSHAPE object to use for the per-unit current for YEARLY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual Amp.
    ///
    /// Must be previously defined as a LOADSHAPE object.
    ///
    /// Is set to the Daily load shape when Daily is defined.  The daily load shape is repeated in this case. Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.
    ///
    /// DSS property name: Yearly, DSS property index: 8
    ///
    string Yearly()
    {
        return get_prop_string(Properties::Yearly);
    }

    Isource& Yearly(const string &value)
    {
        set_string(Properties::Yearly, value);
        return *this;
    }

    Isource& Yearly(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Yearly, value);
        return *this;
    }

    /// LOADSHAPE object to use for the per-unit current for YEARLY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual Amp.
    ///
    /// Must be previously defined as a LOADSHAPE object.
    ///
    /// Is set to the Daily load shape when Daily is defined.  The daily load shape is repeated in this case. Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.
    ///
    /// DSS property name: Yearly, DSS property index: 8
    ///
    dss::obj::LoadShape Yearly_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::Yearly);
    }

    Isource& Yearly_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Yearly, value);
        return *this;
    }

    /// LOADSHAPE object to use for the per-unit current for DAILY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual A.
    ///
    /// Must be previously defined as a LOADSHAPE object.
    ///
    /// Sets Yearly curve if it is not already defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.
    ///
    /// DSS property name: Daily, DSS property index: 9
    ///
    string Daily()
    {
        return get_prop_string(Properties::Daily);
    }

    Isource& Daily(const string &value)
    {
        set_string(Properties::Daily, value);
        return *this;
    }

    Isource& Daily(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Daily, value);
        return *this;
    }

    /// LOADSHAPE object to use for the per-unit current for DAILY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual A.
    ///
    /// Must be previously defined as a LOADSHAPE object.
    ///
    /// Sets Yearly curve if it is not already defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.
    ///
    /// DSS property name: Daily, DSS property index: 9
    ///
    dss::obj::LoadShape Daily_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::Daily);
    }

    Isource& Daily_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Daily, value);
        return *this;
    }

    /// LOADSHAPE object to use for the per-unit current for DUTYCYCLE-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual A.
    ///
    /// Must be previously defined as a LOADSHAPE object.
    ///
    /// Defaults to Daily load shape when Daily is defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.
    ///
    /// DSS property name: Duty, DSS property index: 10
    ///
    string Duty()
    {
        return get_prop_string(Properties::Duty);
    }

    Isource& Duty(const string &value)
    {
        set_string(Properties::Duty, value);
        return *this;
    }

    Isource& Duty(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Duty, value);
        return *this;
    }

    /// LOADSHAPE object to use for the per-unit current for DUTYCYCLE-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual A.
    ///
    /// Must be previously defined as a LOADSHAPE object.
    ///
    /// Defaults to Daily load shape when Daily is defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.
    ///
    /// DSS property name: Duty, DSS property index: 10
    ///
    dss::obj::LoadShape Duty_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::Duty);
    }

    Isource& Duty_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Duty, value);
        return *this;
    }

    /// Name of bus to which 2nd terminal is connected.
    /// bus2=busname
    /// bus2=busname.1.2.3
    ///
    /// Default is Bus1.0.0.0 (grounded-wye connection)
    ///
    /// DSS property name: Bus2, DSS property index: 11
    ///
    string Bus2()
    {
        return get_prop_string(Properties::Bus2);
    }

    Isource& Bus2(const string &value)
    {
        set_string(Properties::Bus2, value);
        return *this;
    }

    Isource& Bus2(const char* value)
    {
        set_string(Properties::Bus2, value);
        return *this;
    }

    /// Harmonic spectrum assumed for this source.  Default is "default".
    ///
    /// DSS property name: spectrum, DSS property index: 12
    ///
    string spectrum()
    {
        return get_prop_string(Properties::spectrum);
    }

    Isource& spectrum(const string &value)
    {
        set_string(Properties::spectrum, value);
        return *this;
    }

    Isource& spectrum(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Harmonic spectrum assumed for this source.  Default is "default".
    ///
    /// DSS property name: spectrum, DSS property index: 12
    ///
    dss::obj::Spectrum spectrum_obj()
    {
        return get_obj<dss::obj::Spectrum>(Properties::spectrum);
    }

    Isource& spectrum_obj(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 13
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Isource& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 14
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Isource& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 15
    ///
    Isource& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 15
    ///
    Isource& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class VCCS: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 18;
    struct Properties
    {
        enum {
            bus1 = 1,
            phases = 2,
            prated = 3,
            vrated = 4,
            ppct = 5,
            bp1 = 6,
            bp2 = 7,
            filter = 8,
            fsample = 9,
            rmsmode = 10,
            imaxpu = 11,
            vrmstau = 12,
            irmstau = 13,
            spectrum = 14,
            basefreq = 15,
            enabled = 16,
            like = 17,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    VCCS(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    VCCS(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the VCCS element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    VCCS(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the VCCS element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    VCCS& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    VCCS& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name of bus to which source is connected.
    /// bus1=busname
    /// bus1=busname.1.2.3
    ///
    /// DSS property name: bus1, DSS property index: 1
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    VCCS& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    VCCS& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Number of phases.  Defaults to 1.
    ///
    /// DSS property name: phases, DSS property index: 2
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    VCCS& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Total rated power, in Watts.
    ///
    /// DSS property name: prated, DSS property index: 3
    ///
    double prated()
    {
        return Obj_GetFloat64(ptr, Properties::prated);
    }

    VCCS& prated(double value)
    {
        Obj_SetFloat64(ptr, Properties::prated, value);
        return *this;
    }

    /// Rated line-to-line voltage, in Volts
    ///
    /// DSS property name: vrated, DSS property index: 4
    ///
    double vrated()
    {
        return Obj_GetFloat64(ptr, Properties::vrated);
    }

    VCCS& vrated(double value)
    {
        Obj_SetFloat64(ptr, Properties::vrated, value);
        return *this;
    }

    /// Steady-state operating output, in percent of rated.
    ///
    /// DSS property name: ppct, DSS property index: 5
    ///
    double ppct()
    {
        return Obj_GetFloat64(ptr, Properties::ppct);
    }

    VCCS& ppct(double value)
    {
        Obj_SetFloat64(ptr, Properties::ppct, value);
        return *this;
    }

    /// XYCurve defining the input piece-wise linear block.
    ///
    /// DSS property name: bp1, DSS property index: 6
    ///
    string bp1()
    {
        return get_prop_string(Properties::bp1);
    }

    VCCS& bp1(const string &value)
    {
        set_string(Properties::bp1, value);
        return *this;
    }

    VCCS& bp1(dss::obj::XYcurve &value)
    {
        set_obj(Properties::bp1, value);
        return *this;
    }

    /// XYCurve defining the input piece-wise linear block.
    ///
    /// DSS property name: bp1, DSS property index: 6
    ///
    dss::obj::XYcurve bp1_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::bp1);
    }

    VCCS& bp1_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::bp1, value);
        return *this;
    }

    /// XYCurve defining the output piece-wise linear block.
    ///
    /// DSS property name: bp2, DSS property index: 7
    ///
    string bp2()
    {
        return get_prop_string(Properties::bp2);
    }

    VCCS& bp2(const string &value)
    {
        set_string(Properties::bp2, value);
        return *this;
    }

    VCCS& bp2(dss::obj::XYcurve &value)
    {
        set_obj(Properties::bp2, value);
        return *this;
    }

    /// XYCurve defining the output piece-wise linear block.
    ///
    /// DSS property name: bp2, DSS property index: 7
    ///
    dss::obj::XYcurve bp2_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::bp2);
    }

    VCCS& bp2_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::bp2, value);
        return *this;
    }

    /// XYCurve defining the digital filter coefficients (x numerator, y denominator).
    ///
    /// DSS property name: filter, DSS property index: 8
    ///
    string filter()
    {
        return get_prop_string(Properties::filter);
    }

    VCCS& filter(const string &value)
    {
        set_string(Properties::filter, value);
        return *this;
    }

    VCCS& filter(dss::obj::XYcurve &value)
    {
        set_obj(Properties::filter, value);
        return *this;
    }

    /// XYCurve defining the digital filter coefficients (x numerator, y denominator).
    ///
    /// DSS property name: filter, DSS property index: 8
    ///
    dss::obj::XYcurve filter_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::filter);
    }

    VCCS& filter_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::filter, value);
        return *this;
    }

    /// Sample frequency [Hz} for the digital filter.
    ///
    /// DSS property name: fsample, DSS property index: 9
    ///
    double fsample()
    {
        return Obj_GetFloat64(ptr, Properties::fsample);
    }

    VCCS& fsample(double value)
    {
        Obj_SetFloat64(ptr, Properties::fsample, value);
        return *this;
    }

    /// True if only Hz is used to represent a phase-locked loop (PLL), ignoring the BP1, BP2 and time-domain transformations. Default is no.
    ///
    /// DSS property name: rmsmode, DSS property index: 10
    ///
    bool rmsmode()
    {
        return Obj_GetInt32(ptr, Properties::rmsmode) != 0;
    }

    VCCS& rmsmode(bool value)
    {
        Obj_SetInt32(ptr, Properties::rmsmode, value);
        return *this;
    }

    /// Maximum output current in per-unit of rated; defaults to 1.1
    ///
    /// DSS property name: imaxpu, DSS property index: 11
    ///
    double imaxpu()
    {
        return Obj_GetFloat64(ptr, Properties::imaxpu);
    }

    VCCS& imaxpu(double value)
    {
        Obj_SetFloat64(ptr, Properties::imaxpu, value);
        return *this;
    }

    /// Time constant in sensing Vrms for the PLL; defaults to 0.0015
    ///
    /// DSS property name: vrmstau, DSS property index: 12
    ///
    double vrmstau()
    {
        return Obj_GetFloat64(ptr, Properties::vrmstau);
    }

    VCCS& vrmstau(double value)
    {
        Obj_SetFloat64(ptr, Properties::vrmstau, value);
        return *this;
    }

    /// Time constant in producing Irms from the PLL; defaults to 0.0015
    ///
    /// DSS property name: irmstau, DSS property index: 13
    ///
    double irmstau()
    {
        return Obj_GetFloat64(ptr, Properties::irmstau);
    }

    VCCS& irmstau(double value)
    {
        Obj_SetFloat64(ptr, Properties::irmstau, value);
        return *this;
    }

    /// Harmonic spectrum assumed for this source.  Default is "default".
    ///
    /// DSS property name: spectrum, DSS property index: 14
    ///
    string spectrum()
    {
        return get_prop_string(Properties::spectrum);
    }

    VCCS& spectrum(const string &value)
    {
        set_string(Properties::spectrum, value);
        return *this;
    }

    VCCS& spectrum(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Harmonic spectrum assumed for this source.  Default is "default".
    ///
    /// DSS property name: spectrum, DSS property index: 14
    ///
    dss::obj::Spectrum spectrum_obj()
    {
        return get_obj<dss::obj::Spectrum>(Properties::spectrum);
    }

    VCCS& spectrum_obj(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 15
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    VCCS& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 16
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    VCCS& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 17
    ///
    VCCS& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 17
    ///
    VCCS& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Load: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 19;
    struct Properties
    {
        enum {
            phases = 1,
            bus1 = 2,
            kV = 3,
            kW = 4,
            pf = 5,
            model = 6,
            yearly = 7,
            daily = 8,
            duty = 9,
            growth = 10,
            conn = 11,
            kvar = 12,
            Rneut = 13,
            Xneut = 14,
            status = 15,
            cls = 16,
            Vminpu = 17,
            Vmaxpu = 18,
            Vminnorm = 19,
            Vminemerg = 20,
            xfkVA = 21,
            allocationfactor = 22,
            kVA = 23,
            pctmean = 24,
            pctstddev = 25,
            CVRwatts = 26,
            CVRvars = 27,
            kwh = 28,
            kwhdays = 29,
            Cfactor = 30,
            CVRcurve = 31,
            NumCust = 32,
            ZIPV = 33,
            pctSeriesRL = 34,
            RelWeight = 35,
            Vlowpu = 36,
            puXharm = 37,
            XRharm = 38,
            spectrum = 39,
            basefreq = 40,
            enabled = 41,
            like = 42,
        };
    };

    // Class-specific enumerations

        ///
        /// Load: Model (DSS enumeration for Load)
        ///
    enum class LoadModel: int32_t
    {
        ConstantPQ = 1, ///< Constant PQ
        ConstantZ = 2, ///< Constant Z
        Motor = 3, ///< Motor (constant P, quadratic Q)
        CVR = 4, ///< CVR (linear P, quadratic Q)
        ConstantI = 5, ///< Constant I
        ConstantP_fixedQ = 6, ///< Constant P, fixed Q
        ConstantP_fixedX = 7, ///< Constant P, fixed X
        ZIPV = 8 ///< ZIPV
    };


        ///
        /// Load: Status (DSS enumeration for Load)
        ///
    enum class LoadStatus: int32_t
    {
        Variable = 0, ///< Variable
        Fixed = 1, ///< Fixed
        Exempt = 2 ///< Exempt
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Load(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Load(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Load element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Load(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Load element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Load& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Load& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of Phases, this load.  Load is evenly divided among phases.
    ///
    /// DSS property name: phases, DSS property index: 1
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    Load& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Bus to which the load is connected.  May include specific node specification.
    ///
    /// DSS property name: bus1, DSS property index: 2
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    Load& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    Load& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Nominal rated (1.0 per unit) voltage, kV, for load. For 2- and 3-phase loads, specify phase-phase kV. Otherwise, specify actual kV across each branch of the load. If wye (star), specify phase-neutral kV. If delta or phase-phase connected, specify phase-phase kV.
    ///
    /// DSS property name: kV, DSS property index: 3
    ///
    double kV()
    {
        return Obj_GetFloat64(ptr, Properties::kV);
    }

    Load& kV(double value)
    {
        Obj_SetFloat64(ptr, Properties::kV, value);
        return *this;
    }

    /// Total base kW for the load.  Normally, you would enter the maximum kW for the load for the first year and allow it to be adjusted by the load shapes, growth shapes, and global load multiplier.
    ///
    /// Legal ways to define base load:
    /// kW, PF
    /// kW, kvar
    /// kVA, PF
    /// XFKVA * Allocationfactor, PF
    /// kWh/(kWhdays*24) * Cfactor, PF
    ///
    /// DSS property name: kW, DSS property index: 4
    ///
    double kW()
    {
        return Obj_GetFloat64(ptr, Properties::kW);
    }

    Load& kW(double value)
    {
        Obj_SetFloat64(ptr, Properties::kW, value);
        return *this;
    }

    /// Load power factor.  Enter negative for leading powerfactor (when kW and kvar have opposite signs.)
    ///
    /// DSS property name: pf, DSS property index: 5
    ///
    double pf()
    {
        return Obj_GetFloat64(ptr, Properties::pf);
    }

    Load& pf(double value)
    {
        Obj_SetFloat64(ptr, Properties::pf, value);
        return *this;
    }

    /// Integer code for the model to use for load variation with voltage. Valid values are:
    ///
    /// 1:Standard constant P+jQ load. (Default)
    /// 2:Constant impedance load.
    /// 3:Const P, Quadratic Q (like a motor).
    /// 4:Nominal Linear P, Quadratic Q (feeder mix). Use this with CVRfactor.
    /// 5:Constant Current Magnitude
    /// 6:Const P, Fixed Q
    /// 7:Const P, Fixed Impedance Q
    /// 8:ZIPV (7 values)
    ///
    /// For Types 6 and 7, only the P is modified by load multipliers.
    ///
    /// DSS property name: model, DSS property index: 6
    ///
    LoadModel model()
    {
        return LoadModel(Obj_GetInt32(ptr, Properties::model));
    }

    Load& model(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::model, value);
        return *this;
    }

    Load& model(LoadModel value)
    {
        Obj_SetInt32(ptr, Properties::model, int32_t(value));
        return *this;
    }

    /// LOADSHAPE object to use for yearly simulations.  Must be previously defined as a Loadshape object. Is set to the Daily load shape  when Daily is defined.  The daily load shape is repeated in this case. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. The default is no variation.
    ///
    /// DSS property name: yearly, DSS property index: 7
    ///
    string yearly()
    {
        return get_prop_string(Properties::yearly);
    }

    Load& yearly(const string &value)
    {
        set_string(Properties::yearly, value);
        return *this;
    }

    Load& yearly(dss::obj::LoadShape &value)
    {
        set_obj(Properties::yearly, value);
        return *this;
    }

    /// LOADSHAPE object to use for yearly simulations.  Must be previously defined as a Loadshape object. Is set to the Daily load shape  when Daily is defined.  The daily load shape is repeated in this case. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. The default is no variation.
    ///
    /// DSS property name: yearly, DSS property index: 7
    ///
    dss::obj::LoadShape yearly_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::yearly);
    }

    Load& yearly_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::yearly, value);
        return *this;
    }

    /// LOADSHAPE object to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Default is no variation (constant) if not defined. Side effect: Sets Yearly load shape if not already defined.
    ///
    /// DSS property name: daily, DSS property index: 8
    ///
    string daily()
    {
        return get_prop_string(Properties::daily);
    }

    Load& daily(const string &value)
    {
        set_string(Properties::daily, value);
        return *this;
    }

    Load& daily(dss::obj::LoadShape &value)
    {
        set_obj(Properties::daily, value);
        return *this;
    }

    /// LOADSHAPE object to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Default is no variation (constant) if not defined. Side effect: Sets Yearly load shape if not already defined.
    ///
    /// DSS property name: daily, DSS property index: 8
    ///
    dss::obj::LoadShape daily_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::daily);
    }

    Load& daily_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::daily, value);
        return *this;
    }

    /// LOADSHAPE object to use for duty cycle simulations.  Must be previously defined as a Loadshape object.  Typically would have time intervals less than 1 hr. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.Set to NONE to reset to no loadahape. Set Status=Fixed to ignore Loadshape designation.  Defaults to Daily curve If not specified.
    ///
    /// DSS property name: duty, DSS property index: 9
    ///
    string duty()
    {
        return get_prop_string(Properties::duty);
    }

    Load& duty(const string &value)
    {
        set_string(Properties::duty, value);
        return *this;
    }

    Load& duty(dss::obj::LoadShape &value)
    {
        set_obj(Properties::duty, value);
        return *this;
    }

    /// LOADSHAPE object to use for duty cycle simulations.  Must be previously defined as a Loadshape object.  Typically would have time intervals less than 1 hr. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.Set to NONE to reset to no loadahape. Set Status=Fixed to ignore Loadshape designation.  Defaults to Daily curve If not specified.
    ///
    /// DSS property name: duty, DSS property index: 9
    ///
    dss::obj::LoadShape duty_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::duty);
    }

    Load& duty_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::duty, value);
        return *this;
    }

    /// Characteristic  to use for growth factors by years.  Must be previously defined as a Growthshape object. Defaults to circuit default growth factor (see Set Growth command).
    ///
    /// DSS property name: growth, DSS property index: 10
    ///
    string growth()
    {
        return get_prop_string(Properties::growth);
    }

    Load& growth(const string &value)
    {
        set_string(Properties::growth, value);
        return *this;
    }

    Load& growth(dss::obj::GrowthShape &value)
    {
        set_obj(Properties::growth, value);
        return *this;
    }

    /// Characteristic  to use for growth factors by years.  Must be previously defined as a Growthshape object. Defaults to circuit default growth factor (see Set Growth command).
    ///
    /// DSS property name: growth, DSS property index: 10
    ///
    dss::obj::GrowthShape growth_obj()
    {
        return get_obj<dss::obj::GrowthShape>(Properties::growth);
    }

    Load& growth_obj(dss::obj::GrowthShape &value)
    {
        set_obj(Properties::growth, value);
        return *this;
    }

    /// ={wye or LN | delta or LL}.  Default is wye.
    ///
    /// DSS property name: conn, DSS property index: 11
    ///
    Connection conn()
    {
        return Connection(Obj_GetInt32(ptr, Properties::conn));
    }

    Load& conn(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::conn, value);
        return *this;
    }

    Load& conn(Connection value)
    {
        Obj_SetInt32(ptr, Properties::conn, int32_t(value));
        return *this;
    }

    Load& conn(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    Load& conn(const char *value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// ={wye or LN | delta or LL}.  Default is wye.
    ///
    /// DSS property name: conn, DSS property index: 11
    ///
    string conn_str()
    {
        return get_prop_string(Properties::conn);
    }

    /// ={wye or LN | delta or LL}.  Default is wye.
    ///
    /// DSS property name: conn, DSS property index: 11
    ///
    Load& conn_str(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// Specify the base kvar for specifying load as kW & kvar.  Assumes kW has been already defined.  Alternative to specifying the power factor.  Side effect:  the power factor and kVA is altered to agree.
    ///
    /// DSS property name: kvar, DSS property index: 12
    ///
    double kvar()
    {
        return Obj_GetFloat64(ptr, Properties::kvar);
    }

    Load& kvar(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvar, value);
        return *this;
    }

    /// Default is -1. Neutral resistance of wye (star)-connected load in actual ohms. If entered as a negative value, the neutral can be open, or floating, or it can be connected to node 0 (ground), which is the usual default. If >=0 be sure to explicitly specify the node connection for the neutral, or last, conductor. Otherwise, the neutral impedance will be shorted to ground.
    ///
    /// DSS property name: Rneut, DSS property index: 13
    ///
    double Rneut()
    {
        return Obj_GetFloat64(ptr, Properties::Rneut);
    }

    Load& Rneut(double value)
    {
        Obj_SetFloat64(ptr, Properties::Rneut, value);
        return *this;
    }

    /// Neutral reactance of wye(star)-connected load in actual ohms.  May be + or -.
    ///
    /// DSS property name: Xneut, DSS property index: 14
    ///
    double Xneut()
    {
        return Obj_GetFloat64(ptr, Properties::Xneut);
    }

    Load& Xneut(double value)
    {
        Obj_SetFloat64(ptr, Properties::Xneut, value);
        return *this;
    }

    /// ={Variable | Fixed | Exempt}.  Default is variable. If Fixed, no load multipliers apply;  however, growth multipliers do apply.  All multipliers apply to Variable loads.  Exempt loads are not modified by the global load multiplier, such as in load duration curves, etc.  Daily multipliers do apply, so setting this property to Exempt is a good way to represent industrial load that stays the same day-after-day for the period study.
    ///
    /// DSS property name: status, DSS property index: 15
    ///
    LoadStatus status()
    {
        return LoadStatus(Obj_GetInt32(ptr, Properties::status));
    }

    Load& status(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::status, value);
        return *this;
    }

    Load& status(LoadStatus value)
    {
        Obj_SetInt32(ptr, Properties::status, int32_t(value));
        return *this;
    }

    Load& status(const string &value)
    {
        set_string(Properties::status, value);
        return *this;
    }

    Load& status(const char *value)
    {
        set_string(Properties::status, value);
        return *this;
    }

    /// ={Variable | Fixed | Exempt}.  Default is variable. If Fixed, no load multipliers apply;  however, growth multipliers do apply.  All multipliers apply to Variable loads.  Exempt loads are not modified by the global load multiplier, such as in load duration curves, etc.  Daily multipliers do apply, so setting this property to Exempt is a good way to represent industrial load that stays the same day-after-day for the period study.
    ///
    /// DSS property name: status, DSS property index: 15
    ///
    string status_str()
    {
        return get_prop_string(Properties::status);
    }

    /// ={Variable | Fixed | Exempt}.  Default is variable. If Fixed, no load multipliers apply;  however, growth multipliers do apply.  All multipliers apply to Variable loads.  Exempt loads are not modified by the global load multiplier, such as in load duration curves, etc.  Daily multipliers do apply, so setting this property to Exempt is a good way to represent industrial load that stays the same day-after-day for the period study.
    ///
    /// DSS property name: status, DSS property index: 15
    ///
    Load& status_str(const string &value)
    {
        set_string(Properties::status, value);
        return *this;
    }

    /// An arbitrary integer number representing the class of load so that load values may be segregated by load value. Default is 1; not used internally.
    ///
    /// DSS property name: class, DSS property index: 16
    ///
    int32_t cls()
    {
        return Obj_GetInt32(ptr, Properties::cls);
    }

    Load& cls(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::cls, value);
        return *this;
    }

    /// Default = 0.95.  Minimum per unit voltage for which the MODEL is assumed to apply. Lower end of normal voltage range.Below this value, the load model reverts to a constant impedance model that matches the model at the transition voltage. See also "Vlowpu" which causes the model to match Model=2 below the transition voltage.
    ///
    /// DSS property name: Vminpu, DSS property index: 17
    ///
    double Vminpu()
    {
        return Obj_GetFloat64(ptr, Properties::Vminpu);
    }

    Load& Vminpu(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vminpu, value);
        return *this;
    }

    /// Default = 1.05.  Maximum per unit voltage for which the MODEL is assumed to apply. Above this value, the load model reverts to a constant impedance model.
    ///
    /// DSS property name: Vmaxpu, DSS property index: 18
    ///
    double Vmaxpu()
    {
        return Obj_GetFloat64(ptr, Properties::Vmaxpu);
    }

    Load& Vmaxpu(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vmaxpu, value);
        return *this;
    }

    /// Minimum per unit voltage for load EEN evaluations, Normal limit.  Default = 0, which defaults to system "vminnorm" property (see Set Command under Executive).  If this property is specified, it ALWAYS overrides the system specification. This allows you to have different criteria for different loads. Set to zero to revert to the default system value.
    ///
    /// DSS property name: Vminnorm, DSS property index: 19
    ///
    double Vminnorm()
    {
        return Obj_GetFloat64(ptr, Properties::Vminnorm);
    }

    Load& Vminnorm(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vminnorm, value);
        return *this;
    }

    /// Minimum per unit voltage for load UE evaluations, Emergency limit.  Default = 0, which defaults to system "vminemerg" property (see Set Command under Executive).  If this property is specified, it ALWAYS overrides the system specification. This allows you to have different criteria for different loads. Set to zero to revert to the default system value.
    ///
    /// DSS property name: Vminemerg, DSS property index: 20
    ///
    double Vminemerg()
    {
        return Obj_GetFloat64(ptr, Properties::Vminemerg);
    }

    Load& Vminemerg(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vminemerg, value);
        return *this;
    }

    /// Default = 0.0.  Rated kVA of service transformer for allocating loads based on connected kVA at a bus. Side effect:  kW, PF, and kvar are modified. See help on kVA.
    ///
    /// DSS property name: xfkVA, DSS property index: 21
    ///
    double xfkVA()
    {
        return Obj_GetFloat64(ptr, Properties::xfkVA);
    }

    Load& xfkVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::xfkVA, value);
        return *this;
    }

    /// Default = 0.5.  Allocation factor for allocating loads based on connected kVA at a bus. Side effect:  kW, PF, and kvar are modified by multiplying this factor times the XFKVA (if > 0).
    ///
    /// DSS property name: allocationfactor, DSS property index: 22
    ///
    double allocationfactor()
    {
        return Obj_GetFloat64(ptr, Properties::allocationfactor);
    }

    Load& allocationfactor(double value)
    {
        Obj_SetFloat64(ptr, Properties::allocationfactor, value);
        return *this;
    }

    /// Specify base Load in kVA (and power factor)
    ///
    /// Legal ways to define base load:
    /// kW, PF
    /// kW, kvar
    /// kVA, PF
    /// XFKVA * Allocationfactor, PF
    /// kWh/(kWhdays*24) * Cfactor, PF
    ///
    /// DSS property name: kVA, DSS property index: 23
    ///
    double kVA()
    {
        return Obj_GetFloat64(ptr, Properties::kVA);
    }

    Load& kVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::kVA, value);
        return *this;
    }

    /// Percent mean value for load to use for monte carlo studies if no loadshape is assigned to this load. Default is 50.
    ///
    /// DSS property name: %mean, DSS property index: 24
    ///
    double pctmean()
    {
        return Obj_GetFloat64(ptr, Properties::pctmean);
    }

    Load& pctmean(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctmean, value);
        return *this;
    }

    /// Percent Std deviation value for load to use for monte carlo studies if no loadshape is assigned to this load. Default is 10.
    ///
    /// DSS property name: %stddev, DSS property index: 25
    ///
    double pctstddev()
    {
        return Obj_GetFloat64(ptr, Properties::pctstddev);
    }

    Load& pctstddev(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctstddev, value);
        return *this;
    }

    /// Percent reduction in active power (watts) per 1% reduction in voltage from 100% rated. Default=1.
    ///  Typical values range from 0.4 to 0.8. Applies to Model=4 only.
    ///  Intended to represent conservation voltage reduction or voltage optimization measures.
    ///
    /// DSS property name: CVRwatts, DSS property index: 26
    ///
    double CVRwatts()
    {
        return Obj_GetFloat64(ptr, Properties::CVRwatts);
    }

    Load& CVRwatts(double value)
    {
        Obj_SetFloat64(ptr, Properties::CVRwatts, value);
        return *this;
    }

    /// Percent reduction in reactive power (vars) per 1% reduction in voltage from 100% rated. Default=2.
    ///  Typical values range from 2 to 3. Applies to Model=4 only.
    ///  Intended to represent conservation voltage reduction or voltage optimization measures.
    ///
    /// DSS property name: CVRvars, DSS property index: 27
    ///
    double CVRvars()
    {
        return Obj_GetFloat64(ptr, Properties::CVRvars);
    }

    Load& CVRvars(double value)
    {
        Obj_SetFloat64(ptr, Properties::CVRvars, value);
        return *this;
    }

    /// kWh billed for this period. Default is 0. See help on kVA and Cfactor and kWhDays.
    ///
    /// DSS property name: kwh, DSS property index: 28
    ///
    double kwh()
    {
        return Obj_GetFloat64(ptr, Properties::kwh);
    }

    Load& kwh(double value)
    {
        Obj_SetFloat64(ptr, Properties::kwh, value);
        return *this;
    }

    /// Length of kWh billing period in days (24 hr days). Default is 30. Average demand is computed using this value.
    ///
    /// DSS property name: kwhdays, DSS property index: 29
    ///
    double kwhdays()
    {
        return Obj_GetFloat64(ptr, Properties::kwhdays);
    }

    Load& kwhdays(double value)
    {
        Obj_SetFloat64(ptr, Properties::kwhdays, value);
        return *this;
    }

    /// Factor relating average kW to peak kW. Default is 4.0. See kWh and kWhdays. See kVA.
    ///
    /// DSS property name: Cfactor, DSS property index: 30
    ///
    double Cfactor()
    {
        return Obj_GetFloat64(ptr, Properties::Cfactor);
    }

    Load& Cfactor(double value)
    {
        Obj_SetFloat64(ptr, Properties::Cfactor, value);
        return *this;
    }

    /// Default is NONE. Curve describing both watt and var factors as a function of time. Refers to a LoadShape object with both Mult and Qmult defined. Define a Loadshape to agree with yearly or daily curve according to the type of analysis being done. If NONE, the CVRwatts and CVRvars factors are used and assumed constant.
    ///
    /// DSS property name: CVRcurve, DSS property index: 31
    ///
    string CVRcurve()
    {
        return get_prop_string(Properties::CVRcurve);
    }

    Load& CVRcurve(const string &value)
    {
        set_string(Properties::CVRcurve, value);
        return *this;
    }

    Load& CVRcurve(dss::obj::LoadShape &value)
    {
        set_obj(Properties::CVRcurve, value);
        return *this;
    }

    /// Default is NONE. Curve describing both watt and var factors as a function of time. Refers to a LoadShape object with both Mult and Qmult defined. Define a Loadshape to agree with yearly or daily curve according to the type of analysis being done. If NONE, the CVRwatts and CVRvars factors are used and assumed constant.
    ///
    /// DSS property name: CVRcurve, DSS property index: 31
    ///
    dss::obj::LoadShape CVRcurve_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::CVRcurve);
    }

    Load& CVRcurve_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::CVRcurve, value);
        return *this;
    }

    /// Number of customers, this load. Default is 1.
    ///
    /// DSS property name: NumCust, DSS property index: 32
    ///
    int32_t NumCust()
    {
        return Obj_GetInt32(ptr, Properties::NumCust);
    }

    Load& NumCust(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::NumCust, value);
        return *this;
    }

    /// Array of 7 coefficients:
    ///
    ///  First 3 are ZIP weighting factors for real power (should sum to 1)
    ///  Next 3 are ZIP weighting factors for reactive power (should sum to 1)
    ///  Last 1 is cut-off voltage in p.u. of base kV; load is 0 below this cut-off
    ///  No defaults; all coefficients must be specified if using model=8.
    ///
    /// DSS property name: ZIPV, DSS property index: 33
    ///
    VectorXd ZIPV()
    {
        return get_array<VectorXd>(Properties::ZIPV);
    }

    Load& ZIPV(VectorXd &value)
    {
        set_array<VectorXd>(Properties::ZIPV, value);
        return *this;
    }

    /// Percent of load that is series R-L for Harmonic studies. Default is 50. Remainder is assumed to be parallel R and L. This can have a significant impact on the amount of damping observed in Harmonics solutions.
    ///
    /// DSS property name: %SeriesRL, DSS property index: 34
    ///
    double pctSeriesRL()
    {
        return Obj_GetFloat64(ptr, Properties::pctSeriesRL);
    }

    Load& pctSeriesRL(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctSeriesRL, value);
        return *this;
    }

    /// Relative weighting factor for reliability calcs. Default = 1. Used to designate high priority loads such as hospitals, etc.
    ///
    /// Is multiplied by number of customers and load kW during reliability calcs.
    ///
    /// DSS property name: RelWeight, DSS property index: 35
    ///
    double RelWeight()
    {
        return Obj_GetFloat64(ptr, Properties::RelWeight);
    }

    Load& RelWeight(double value)
    {
        Obj_SetFloat64(ptr, Properties::RelWeight, value);
        return *this;
    }

    /// Default = 0.50.  Per unit voltage at which the model switches to same as constant Z model (model=2). This allows more consistent convergence at very low voltaes due to opening switches or solving for fault situations.
    ///
    /// DSS property name: Vlowpu, DSS property index: 36
    ///
    double Vlowpu()
    {
        return Obj_GetFloat64(ptr, Properties::Vlowpu);
    }

    Load& Vlowpu(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vlowpu, value);
        return *this;
    }

    /// Special reactance, pu (based on kVA, kV properties), for the series impedance branch in the load model for HARMONICS analysis. Generally used to represent motor load blocked rotor reactance. If not specified (that is, set =0, the default value), the series branch is computed from the percentage of the nominal load at fundamental frequency specified by the %SERIESRL property.
    ///
    /// Applies to load model in HARMONICS mode only.
    ///
    /// A typical value would be approximately 0.20 pu based on kVA * %SeriesRL / 100.0.
    ///
    /// DSS property name: puXharm, DSS property index: 37
    ///
    double puXharm()
    {
        return Obj_GetFloat64(ptr, Properties::puXharm);
    }

    Load& puXharm(double value)
    {
        Obj_SetFloat64(ptr, Properties::puXharm, value);
        return *this;
    }

    /// X/R ratio of the special harmonics mode reactance specified by the puXHARM property at fundamental frequency. Default is 6.
    ///
    /// DSS property name: XRharm, DSS property index: 38
    ///
    double XRharm()
    {
        return Obj_GetFloat64(ptr, Properties::XRharm);
    }

    Load& XRharm(double value)
    {
        Obj_SetFloat64(ptr, Properties::XRharm, value);
        return *this;
    }

    /// Name of harmonic current spectrum for this load.  Default is "defaultload", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 39
    ///
    string spectrum()
    {
        return get_prop_string(Properties::spectrum);
    }

    Load& spectrum(const string &value)
    {
        set_string(Properties::spectrum, value);
        return *this;
    }

    Load& spectrum(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Name of harmonic current spectrum for this load.  Default is "defaultload", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 39
    ///
    dss::obj::Spectrum spectrum_obj()
    {
        return get_obj<dss::obj::Spectrum>(Properties::spectrum);
    }

    Load& spectrum_obj(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 40
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Load& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 41
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Load& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 42
    ///
    Load& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 42
    ///
    Load& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Transformer: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 20;
    struct Properties
    {
        enum {
            phases = 1,
            windings = 2,
            wdg = 3,
            bus = 4,
            conn = 5,
            kV = 6,
            kVA = 7,
            tap = 8,
            pctR = 9,
            Rneut = 10,
            Xneut = 11,
            buses = 12,
            conns = 13,
            kVs = 14,
            kVAs = 15,
            taps = 16,
            XHL = 17,
            XHT = 18,
            XLT = 19,
            Xscarray = 20,
            thermal = 21,
            n = 22,
            m = 23,
            flrise = 24,
            hsrise = 25,
            pctloadloss = 26,
            pctnoloadloss = 27,
            normhkVA = 28,
            emerghkVA = 29,
            sub = 30,
            MaxTap = 31,
            MinTap = 32,
            NumTaps = 33,
            subname = 34,
            pctimag = 35,
            ppm_antifloat = 36,
            pctRs = 37,
            bank = 38,
            XfmrCode = 39,
            XRConst = 40,
            X12 = 41,
            X13 = 42,
            X23 = 43,
            LeadLag = 44,
            WdgCurrents = 45,
            Core = 46,
            RdcOhms = 47,
            Seasons = 48,
            Ratings = 49,
            normamps = 50,
            emergamps = 51,
            faultrate = 52,
            pctperm = 53,
            repair = 54,
            basefreq = 55,
            enabled = 56,
            like = 57,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Transformer(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Transformer(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Transformer element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Transformer(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Transformer element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Transformer& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Transformer& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of phases this transformer. Default is 3.
    ///
    /// DSS property name: phases, DSS property index: 1
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    Transformer& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Number of windings, this transformers. (Also is the number of terminals) Default is 2. This property triggers memory allocation for the Transformer and will cause other properties to revert to default values.
    ///
    /// DSS property name: windings, DSS property index: 2
    ///
    int32_t windings()
    {
        return Obj_GetInt32(ptr, Properties::windings);
    }

    Transformer& windings(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::windings, value);
        return *this;
    }

    /// Set this = to the number of the winding you wish to define.  Then set the values for this winding.  Repeat for each winding.  Alternatively, use the array collections (buses, kVAs, etc.) to define the windings.  Note: reactances are BETWEEN pairs of windings; they are not the property of a single winding.
    ///
    /// DSS property name: wdg, DSS property index: 3
    ///
    int32_t wdg()
    {
        return Obj_GetInt32(ptr, Properties::wdg);
    }

    Transformer& wdg(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::wdg, value);
        return *this;
    }

    /// Bus connection spec for this winding.
    ///
    /// DSS property name: bus, DSS property index: 4
    ///
    strings bus()
    {
        return get_array<strings>(Properties::bus);
    }

    Transformer& bus(strings &value)
    {
        set_array<strings>(Properties::bus, value);
        return *this;
    }

    /// Connection of this winding {wye*, Delta, LN, LL}. Default is "wye" with the neutral solidly grounded.
    ///
    /// DSS property name: conn, DSS property index: 5
    ///
    std::vector<Connection> conn()
    {
        return get_array<std::vector<Connection>>(Properties::conn);
    }

    Transformer& conn(std::vector<int32_t> &value)
    {
        set_array<std::vector<int32_t>>(Properties::conn, value);
        return *this;
    }

    Transformer& conn(strings &value)
    {
        set_array<strings>(Properties::conn, value);
        return *this;
    }

    /// Connection of this winding {wye*, Delta, LN, LL}. Default is "wye" with the neutral solidly grounded.
    ///
    /// DSS property name: conn, DSS property index: 5
    ///
    strings conn_str()
    {
        return get_array<strings>(Properties::conn);
    }

    Transformer& conn_str(strings &value)
    {
        conn(value);
        return *this;
    }

    /// For 2-or 3-phase, enter phase-phase kV rating.  Otherwise, kV rating of the actual winding
    ///
    /// DSS property name: kV, DSS property index: 6
    ///
    VectorXd kV()
    {
        return get_array<VectorXd>(Properties::kV);
    }

    Transformer& kV(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kV, value);
        return *this;
    }

    /// Base kVA rating of the winding. Side effect: forces change of max normal and emerg kVA ratings.If 2-winding transformer, forces other winding to same value. When winding 1 is defined, all other windings are defaulted to the same rating and the first two winding resistances are defaulted to the %loadloss value.
    ///
    /// DSS property name: kVA, DSS property index: 7
    ///
    VectorXd kVA()
    {
        return get_array<VectorXd>(Properties::kVA);
    }

    Transformer& kVA(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kVA, value);
        return *this;
    }

    /// Per unit tap that this winding is on.
    ///
    /// DSS property name: tap, DSS property index: 8
    ///
    VectorXd tap()
    {
        return get_array<VectorXd>(Properties::tap);
    }

    Transformer& tap(VectorXd &value)
    {
        set_array<VectorXd>(Properties::tap, value);
        return *this;
    }

    /// Percent resistance this winding.  (half of total for a 2-winding).
    ///
    /// DSS property name: %R, DSS property index: 9
    ///
    VectorXd pctR()
    {
        return get_array<VectorXd>(Properties::pctR);
    }

    Transformer& pctR(VectorXd &value)
    {
        set_array<VectorXd>(Properties::pctR, value);
        return *this;
    }

    /// Default = -1. Neutral resistance of wye (star)-connected winding in actual ohms. If entered as a negative value, the neutral is assumed to be open, or floating. To solidly ground the neutral, connect the neutral conductor to Node 0 in the Bus property spec for this winding. For example: Bus=MyBusName.1.2.3.0, which is generally the default connection.
    ///
    /// DSS property name: Rneut, DSS property index: 10
    ///
    VectorXd Rneut()
    {
        return get_array<VectorXd>(Properties::Rneut);
    }

    Transformer& Rneut(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Rneut, value);
        return *this;
    }

    /// Neutral reactance of wye(star)-connected winding in actual ohms.  May be + or -.
    ///
    /// DSS property name: Xneut, DSS property index: 11
    ///
    VectorXd Xneut()
    {
        return get_array<VectorXd>(Properties::Xneut);
    }

    Transformer& Xneut(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Xneut, value);
        return *this;
    }

    /// Use this to specify all the bus connections at once using an array. Example:
    ///
    /// New Transformer.T1 buses="Hibus, lowbus"
    ///
    /// DSS property name: buses, DSS property index: 12
    ///
    strings buses()
    {
        return get_array<strings>(Properties::buses);
    }

    Transformer& buses(strings &value)
    {
        set_array<strings>(Properties::buses, value);
        return *this;
    }

    /// Use this to specify all the Winding connections at once using an array. Example:
    ///
    /// New Transformer.T1 buses="Hibus, lowbus" ~ conns=(delta, wye)
    ///
    /// DSS property name: conns, DSS property index: 13
    ///
    std::vector<Connection> conns()
    {
        return get_array<std::vector<Connection>>(Properties::conns);
    }

    Transformer& conns(std::vector<int32_t> &value)
    {
        set_array<std::vector<int32_t>>(Properties::conns, value);
        return *this;
    }

    Transformer& conns(strings &value)
    {
        set_array<strings>(Properties::conns, value);
        return *this;
    }

    /// Use this to specify all the Winding connections at once using an array. Example:
    ///
    /// New Transformer.T1 buses="Hibus, lowbus" ~ conns=(delta, wye)
    ///
    /// DSS property name: conns, DSS property index: 13
    ///
    strings conns_str()
    {
        return get_array<strings>(Properties::conns);
    }

    Transformer& conns_str(strings &value)
    {
        conns(value);
        return *this;
    }

    /// Use this to specify the kV ratings of all windings at once using an array. Example:
    ///
    /// New Transformer.T1 buses="Hibus, lowbus"
    /// ~ conns=(delta, wye)
    /// ~ kvs=(115, 12.47)
    ///
    /// See kV= property for voltage rules.
    ///
    /// DSS property name: kVs, DSS property index: 14
    ///
    VectorXd kVs()
    {
        return get_array<VectorXd>(Properties::kVs);
    }

    Transformer& kVs(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kVs, value);
        return *this;
    }

    /// Use this to specify the kVA ratings of all windings at once using an array.
    ///
    /// DSS property name: kVAs, DSS property index: 15
    ///
    VectorXd kVAs()
    {
        return get_array<VectorXd>(Properties::kVAs);
    }

    Transformer& kVAs(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kVAs, value);
        return *this;
    }

    /// Use this to specify the p.u. tap of all windings at once using an array.
    ///
    /// DSS property name: taps, DSS property index: 16
    ///
    VectorXd taps()
    {
        return get_array<VectorXd>(Properties::taps);
    }

    Transformer& taps(VectorXd &value)
    {
        set_array<VectorXd>(Properties::taps, value);
        return *this;
    }

    /// Use this to specify the percent reactance, H-L (winding 1 to winding 2).  Use for 2- or 3-winding transformers. On the kVA base of winding 1. See also X12.
    ///
    /// DSS property name: XHL, DSS property index: 17
    ///
    double XHL()
    {
        return Obj_GetFloat64(ptr, Properties::XHL);
    }

    Transformer& XHL(double value)
    {
        Obj_SetFloat64(ptr, Properties::XHL, value);
        return *this;
    }

    /// Use this to specify the percent reactance, H-T (winding 1 to winding 3).  Use for 3-winding transformers only. On the kVA base of winding 1. See also X13.
    ///
    /// DSS property name: XHT, DSS property index: 18
    ///
    double XHT()
    {
        return Obj_GetFloat64(ptr, Properties::XHT);
    }

    Transformer& XHT(double value)
    {
        Obj_SetFloat64(ptr, Properties::XHT, value);
        return *this;
    }

    /// Use this to specify the percent reactance, L-T (winding 2 to winding 3).  Use for 3-winding transformers only. On the kVA base of winding 1.  See also X23.
    ///
    /// DSS property name: XLT, DSS property index: 19
    ///
    double XLT()
    {
        return Obj_GetFloat64(ptr, Properties::XLT);
    }

    Transformer& XLT(double value)
    {
        Obj_SetFloat64(ptr, Properties::XLT, value);
        return *this;
    }

    /// Use this to specify the percent reactance between all pairs of windings as an array. All values are on the kVA base of winding 1.  The order of the values is as follows:
    ///
    /// (x12 13 14... 23 24.. 34 ..)
    ///
    /// There will be n(n-1)/2 values, where n=number of windings.
    ///
    /// DSS property name: Xscarray, DSS property index: 20
    ///
    VectorXd Xscarray()
    {
        return get_array<VectorXd>(Properties::Xscarray);
    }

    Transformer& Xscarray(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Xscarray, value);
        return *this;
    }

    /// Thermal time constant of the transformer in hours.  Typically about 2.
    ///
    /// DSS property name: thermal, DSS property index: 21
    ///
    double thermal()
    {
        return Obj_GetFloat64(ptr, Properties::thermal);
    }

    Transformer& thermal(double value)
    {
        Obj_SetFloat64(ptr, Properties::thermal, value);
        return *this;
    }

    /// n Exponent for thermal properties in IEEE C57.  Typically 0.8.
    ///
    /// DSS property name: n, DSS property index: 22
    ///
    double n()
    {
        return Obj_GetFloat64(ptr, Properties::n);
    }

    Transformer& n(double value)
    {
        Obj_SetFloat64(ptr, Properties::n, value);
        return *this;
    }

    /// m Exponent for thermal properties in IEEE C57.  Typically 0.9 - 1.0
    ///
    /// DSS property name: m, DSS property index: 23
    ///
    double m()
    {
        return Obj_GetFloat64(ptr, Properties::m);
    }

    Transformer& m(double value)
    {
        Obj_SetFloat64(ptr, Properties::m, value);
        return *this;
    }

    /// Temperature rise, deg C, for full load.  Default is 65.
    ///
    /// DSS property name: flrise, DSS property index: 24
    ///
    double flrise()
    {
        return Obj_GetFloat64(ptr, Properties::flrise);
    }

    Transformer& flrise(double value)
    {
        Obj_SetFloat64(ptr, Properties::flrise, value);
        return *this;
    }

    /// Hot spot temperature rise, deg C.  Default is 15.
    ///
    /// DSS property name: hsrise, DSS property index: 25
    ///
    double hsrise()
    {
        return Obj_GetFloat64(ptr, Properties::hsrise);
    }

    Transformer& hsrise(double value)
    {
        Obj_SetFloat64(ptr, Properties::hsrise, value);
        return *this;
    }

    /// Percent load loss at full load. The %R of the High and Low windings (1 and 2) are adjusted to agree at rated kVA loading.
    ///
    /// DSS property name: %loadloss, DSS property index: 26
    ///
    double pctloadloss()
    {
        return Obj_GetFloat64(ptr, Properties::pctloadloss);
    }

    Transformer& pctloadloss(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctloadloss, value);
        return *this;
    }

    /// Percent no load losses at rated excitatation voltage. Default is 0. Converts to a resistance in parallel with the magnetizing impedance in each winding.
    ///
    /// DSS property name: %noloadloss, DSS property index: 27
    ///
    double pctnoloadloss()
    {
        return Obj_GetFloat64(ptr, Properties::pctnoloadloss);
    }

    Transformer& pctnoloadloss(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctnoloadloss, value);
        return *this;
    }

    /// Normal maximum kVA rating of H winding (winding 1).  Usually 100% - 110% ofmaximum nameplate rating, depending on load shape. Defaults to 110% of kVA rating of Winding 1.
    ///
    /// DSS property name: normhkVA, DSS property index: 28
    ///
    double normhkVA()
    {
        return Obj_GetFloat64(ptr, Properties::normhkVA);
    }

    Transformer& normhkVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::normhkVA, value);
        return *this;
    }

    /// Emergency (contingency)  kVA rating of H winding (winding 1).  Usually 140% - 150% ofmaximum nameplate rating, depending on load shape. Defaults to 150% of kVA rating of Winding 1.
    ///
    /// DSS property name: emerghkVA, DSS property index: 29
    ///
    double emerghkVA()
    {
        return Obj_GetFloat64(ptr, Properties::emerghkVA);
    }

    Transformer& emerghkVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::emerghkVA, value);
        return *this;
    }

    /// ={Yes|No}  Designates whether this transformer is to be considered a substation.Default is No.
    ///
    /// DSS property name: sub, DSS property index: 30
    ///
    bool sub()
    {
        return Obj_GetInt32(ptr, Properties::sub) != 0;
    }

    Transformer& sub(bool value)
    {
        Obj_SetInt32(ptr, Properties::sub, value);
        return *this;
    }

    /// Max per unit tap for the active winding.  Default is 1.10
    ///
    /// DSS property name: MaxTap, DSS property index: 31
    ///
    VectorXd MaxTap()
    {
        return get_array<VectorXd>(Properties::MaxTap);
    }

    Transformer& MaxTap(VectorXd &value)
    {
        set_array<VectorXd>(Properties::MaxTap, value);
        return *this;
    }

    /// Min per unit tap for the active winding.  Default is 0.90
    ///
    /// DSS property name: MinTap, DSS property index: 32
    ///
    VectorXd MinTap()
    {
        return get_array<VectorXd>(Properties::MinTap);
    }

    Transformer& MinTap(VectorXd &value)
    {
        set_array<VectorXd>(Properties::MinTap, value);
        return *this;
    }

    /// Total number of taps between min and max tap.  Default is 32 (16 raise and 16 lower taps about the neutral position). The neutral position is not counted.
    ///
    /// DSS property name: NumTaps, DSS property index: 33
    ///
    VectorXi NumTaps()
    {
        return get_array<VectorXi>(Properties::NumTaps);
    }

    Transformer& NumTaps(VectorXi &value)
    {
        set_array<VectorXi>(Properties::NumTaps, value);
        return *this;
    }

    /// Substation Name. Optional. Default is null. If specified, printed on plots
    ///
    /// DSS property name: subname, DSS property index: 34
    ///
    string subname()
    {
        return get_prop_string(Properties::subname);
    }

    Transformer& subname(const string &value)
    {
        set_string(Properties::subname, value);
        return *this;
    }

    Transformer& subname(const char* value)
    {
        set_string(Properties::subname, value);
        return *this;
    }

    /// Percent magnetizing current. Default=0.0. Magnetizing branch is in parallel with windings in each phase. Also, see "ppm_antifloat".
    ///
    /// DSS property name: %imag, DSS property index: 35
    ///
    double pctimag()
    {
        return Obj_GetFloat64(ptr, Properties::pctimag);
    }

    Transformer& pctimag(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctimag, value);
        return *this;
    }

    /// Default=1 ppm.  Parts per million of transformer winding VA rating connected to ground to protect against accidentally floating a winding without a reference. If positive then the effect is adding a very large reactance to ground.  If negative, then a capacitor.
    ///
    /// DSS property name: ppm_antifloat, DSS property index: 36
    ///
    double ppm_antifloat()
    {
        return Obj_GetFloat64(ptr, Properties::ppm_antifloat);
    }

    Transformer& ppm_antifloat(double value)
    {
        Obj_SetFloat64(ptr, Properties::ppm_antifloat, value);
        return *this;
    }

    /// Use this property to specify all the winding %resistances using an array. Example:
    ///
    /// New Transformer.T1 buses="Hibus, lowbus" ~ %Rs=(0.2  0.3)
    ///
    /// DSS property name: %Rs, DSS property index: 37
    ///
    VectorXd pctRs()
    {
        return get_array<VectorXd>(Properties::pctRs);
    }

    Transformer& pctRs(VectorXd &value)
    {
        set_array<VectorXd>(Properties::pctRs, value);
        return *this;
    }

    /// Name of the bank this transformer is part of, for CIM, MultiSpeak, and other interfaces.
    ///
    /// DSS property name: bank, DSS property index: 38
    ///
    string bank()
    {
        return get_prop_string(Properties::bank);
    }

    Transformer& bank(const string &value)
    {
        set_string(Properties::bank, value);
        return *this;
    }

    Transformer& bank(const char* value)
    {
        set_string(Properties::bank, value);
        return *this;
    }

    /// Name of a library entry for transformer properties. The named XfmrCode must already be defined.
    ///
    /// DSS property name: XfmrCode, DSS property index: 39
    ///
    string XfmrCode()
    {
        return get_prop_string(Properties::XfmrCode);
    }

    Transformer& XfmrCode(const string &value)
    {
        set_string(Properties::XfmrCode, value);
        return *this;
    }

    Transformer& XfmrCode(dss::obj::XfmrCode &value)
    {
        set_obj(Properties::XfmrCode, value);
        return *this;
    }

    /// Name of a library entry for transformer properties. The named XfmrCode must already be defined.
    ///
    /// DSS property name: XfmrCode, DSS property index: 39
    ///
    dss::obj::XfmrCode XfmrCode_obj()
    {
        return get_obj<dss::obj::XfmrCode>(Properties::XfmrCode);
    }

    Transformer& XfmrCode_obj(dss::obj::XfmrCode &value)
    {
        set_obj(Properties::XfmrCode, value);
        return *this;
    }

    /// ={Yes|No} Default is NO. Signifies whether or not the X/R is assumed contant for harmonic studies.
    ///
    /// DSS property name: XRConst, DSS property index: 40
    ///
    bool XRConst()
    {
        return Obj_GetInt32(ptr, Properties::XRConst) != 0;
    }

    Transformer& XRConst(bool value)
    {
        Obj_SetInt32(ptr, Properties::XRConst, value);
        return *this;
    }

    /// Alternative to XHL for specifying the percent reactance from winding 1 to winding 2.  Use for 2- or 3-winding transformers. Percent on the kVA base of winding 1.
    ///
    /// DSS property name: X12, DSS property index: 41
    ///
    double X12()
    {
        return Obj_GetFloat64(ptr, Properties::X12);
    }

    Transformer& X12(double value)
    {
        Obj_SetFloat64(ptr, Properties::X12, value);
        return *this;
    }

    /// Alternative to XHT for specifying the percent reactance from winding 1 to winding 3.  Use for 3-winding transformers only. Percent on the kVA base of winding 1.
    ///
    /// DSS property name: X13, DSS property index: 42
    ///
    double X13()
    {
        return Obj_GetFloat64(ptr, Properties::X13);
    }

    Transformer& X13(double value)
    {
        Obj_SetFloat64(ptr, Properties::X13, value);
        return *this;
    }

    /// Alternative to XLT for specifying the percent reactance from winding 2 to winding 3.Use for 3-winding transformers only. Percent on the kVA base of winding 1.
    ///
    /// DSS property name: X23, DSS property index: 43
    ///
    double X23()
    {
        return Obj_GetFloat64(ptr, Properties::X23);
    }

    Transformer& X23(double value)
    {
        Obj_SetFloat64(ptr, Properties::X23, value);
        return *this;
    }

    /// {Lead | Lag (default) | ANSI (default) | Euro } Designation in mixed Delta-wye connections the relationship between HV to LV winding. Default is ANSI 30 deg lag, e.g., Dy1 of Yd1 vector group. To get typical European Dy11 connection, specify either "lead" or "Euro"
    ///
    /// DSS property name: LeadLag, DSS property index: 44
    ///
    PhaseSequence LeadLag()
    {
        return PhaseSequence(Obj_GetInt32(ptr, Properties::LeadLag));
    }

    Transformer& LeadLag(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::LeadLag, value);
        return *this;
    }

    Transformer& LeadLag(PhaseSequence value)
    {
        Obj_SetInt32(ptr, Properties::LeadLag, int32_t(value));
        return *this;
    }

    Transformer& LeadLag(const string &value)
    {
        set_string(Properties::LeadLag, value);
        return *this;
    }

    Transformer& LeadLag(const char *value)
    {
        set_string(Properties::LeadLag, value);
        return *this;
    }

    /// {Lead | Lag (default) | ANSI (default) | Euro } Designation in mixed Delta-wye connections the relationship between HV to LV winding. Default is ANSI 30 deg lag, e.g., Dy1 of Yd1 vector group. To get typical European Dy11 connection, specify either "lead" or "Euro"
    ///
    /// DSS property name: LeadLag, DSS property index: 44
    ///
    string LeadLag_str()
    {
        return get_prop_string(Properties::LeadLag);
    }

    /// {Lead | Lag (default) | ANSI (default) | Euro } Designation in mixed Delta-wye connections the relationship between HV to LV winding. Default is ANSI 30 deg lag, e.g., Dy1 of Yd1 vector group. To get typical European Dy11 connection, specify either "lead" or "Euro"
    ///
    /// DSS property name: LeadLag, DSS property index: 44
    ///
    Transformer& LeadLag_str(const string &value)
    {
        set_string(Properties::LeadLag, value);
        return *this;
    }

    /// (Read only) Makes winding currents available via return on query (? Transformer.TX.WdgCurrents). Order: Phase 1, Wdg 1, Wdg 2, ..., Phase 2 ...
    ///
    /// DSS property name: WdgCurrents, DSS property index: 45
    ///
    string WdgCurrents()
    {
        // []
        // StringSilentROFunction
        return get_prop_string(Properties::WdgCurrents);
    }

    /// {Shell*|5-leg|3-Leg|1-phase|core-1-phase|4-leg} Core Type. Used for GIC analysis
    ///
    /// DSS property name: Core, DSS property index: 46
    ///
    CoreType Core()
    {
        return CoreType(Obj_GetInt32(ptr, Properties::Core));
    }

    Transformer& Core(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Core, value);
        return *this;
    }

    Transformer& Core(CoreType value)
    {
        Obj_SetInt32(ptr, Properties::Core, int32_t(value));
        return *this;
    }

    Transformer& Core(const string &value)
    {
        set_string(Properties::Core, value);
        return *this;
    }

    Transformer& Core(const char *value)
    {
        set_string(Properties::Core, value);
        return *this;
    }

    /// {Shell*|5-leg|3-Leg|1-phase|core-1-phase|4-leg} Core Type. Used for GIC analysis
    ///
    /// DSS property name: Core, DSS property index: 46
    ///
    string Core_str()
    {
        return get_prop_string(Properties::Core);
    }

    /// {Shell*|5-leg|3-Leg|1-phase|core-1-phase|4-leg} Core Type. Used for GIC analysis
    ///
    /// DSS property name: Core, DSS property index: 46
    ///
    Transformer& Core_str(const string &value)
    {
        set_string(Properties::Core, value);
        return *this;
    }

    /// Winding dc resistance in OHMS. Useful for GIC analysis. From transformer test report. Defaults to 85% of %R property
    ///
    /// DSS property name: RdcOhms, DSS property index: 47
    ///
    VectorXd RdcOhms()
    {
        return get_array<VectorXd>(Properties::RdcOhms);
    }

    Transformer& RdcOhms(VectorXd &value)
    {
        set_array<VectorXd>(Properties::RdcOhms, value);
        return *this;
    }

    /// Defines the number of ratings to be defined for the transfomer, to be used only when defining seasonal ratings using the "Ratings" property.
    ///
    /// DSS property name: Seasons, DSS property index: 48
    ///
    int32_t Seasons()
    {
        return Obj_GetInt32(ptr, Properties::Seasons);
    }

    Transformer& Seasons(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Seasons, value);
        return *this;
    }

    /// An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
    /// multiple ratings to change during a QSTS simulation to evaluate different ratings in transformers. Is given in kVA
    ///
    /// DSS property name: Ratings, DSS property index: 49
    ///
    VectorXd Ratings()
    {
        return get_array<VectorXd>(Properties::Ratings);
    }

    Transformer& Ratings(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Ratings, value);
        return *this;
    }

    /// Normal rated current.
    ///
    /// DSS property name: normamps, DSS property index: 50
    ///
    double normamps()
    {
        return Obj_GetFloat64(ptr, Properties::normamps);
    }

    Transformer& normamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::normamps, value);
        return *this;
    }

    /// Maximum or emerg current.
    ///
    /// DSS property name: emergamps, DSS property index: 51
    ///
    double emergamps()
    {
        return Obj_GetFloat64(ptr, Properties::emergamps);
    }

    Transformer& emergamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::emergamps, value);
        return *this;
    }

    /// Failure rate per year.
    ///
    /// DSS property name: faultrate, DSS property index: 52
    ///
    double faultrate()
    {
        return Obj_GetFloat64(ptr, Properties::faultrate);
    }

    Transformer& faultrate(double value)
    {
        Obj_SetFloat64(ptr, Properties::faultrate, value);
        return *this;
    }

    /// Percent of failures that become permanent.
    ///
    /// DSS property name: pctperm, DSS property index: 53
    ///
    double pctperm()
    {
        return Obj_GetFloat64(ptr, Properties::pctperm);
    }

    Transformer& pctperm(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctperm, value);
        return *this;
    }

    /// Hours to repair.
    ///
    /// DSS property name: repair, DSS property index: 54
    ///
    double repair()
    {
        return Obj_GetFloat64(ptr, Properties::repair);
    }

    Transformer& repair(double value)
    {
        Obj_SetFloat64(ptr, Properties::repair, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 55
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Transformer& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 56
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Transformer& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 57
    ///
    Transformer& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 57
    ///
    Transformer& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Capacitor: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 22;
    struct Properties
    {
        enum {
            bus1 = 1,
            bus2 = 2,
            phases = 3,
            kvar = 4,
            kv = 5,
            conn = 6,
            cmatrix = 7,
            cuf = 8,
            R = 9,
            XL = 10,
            Harm = 11,
            Numsteps = 12,
            states = 13,
            normamps = 14,
            emergamps = 15,
            faultrate = 16,
            pctperm = 17,
            repair = 18,
            basefreq = 19,
            enabled = 20,
            like = 21,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Capacitor(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Capacitor(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Capacitor element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Capacitor(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Capacitor element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Capacitor& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Capacitor& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name of first bus of 2-terminal capacitor. Examples:
    /// bus1=busname
    /// bus1=busname.1.2.3
    ///
    /// If only one bus specified, Bus2 will default to this bus, Node 0, and the capacitor will be a Yg shunt bank.
    ///
    /// DSS property name: bus1, DSS property index: 1
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    Capacitor& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    Capacitor& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Name of 2nd bus. Defaults to all phases connected to first bus, node 0, (Shunt Wye Connection) except when Bus2 explicitly specified.
    ///
    /// Not necessary to specify for delta (LL) connection.
    ///
    /// DSS property name: bus2, DSS property index: 2
    ///
    string bus2()
    {
        return get_prop_string(Properties::bus2);
    }

    Capacitor& bus2(const string &value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    Capacitor& bus2(const char* value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    /// Number of phases.
    ///
    /// DSS property name: phases, DSS property index: 3
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    Capacitor& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Total kvar, if one step, or ARRAY of kvar ratings for each step.  Evenly divided among phases. See rules for NUMSTEPS.
    ///
    /// DSS property name: kvar, DSS property index: 4
    ///
    VectorXd kvar()
    {
        return get_array<VectorXd>(Properties::kvar);
    }

    Capacitor& kvar(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kvar, value);
        return *this;
    }

    /// For 2, 3-phase, kV phase-phase. Otherwise specify actual can rating.
    ///
    /// DSS property name: kv, DSS property index: 5
    ///
    double kv()
    {
        return Obj_GetFloat64(ptr, Properties::kv);
    }

    Capacitor& kv(double value)
    {
        Obj_SetFloat64(ptr, Properties::kv, value);
        return *this;
    }

    /// ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN
    ///
    /// DSS property name: conn, DSS property index: 6
    ///
    Connection conn()
    {
        return Connection(Obj_GetInt32(ptr, Properties::conn));
    }

    Capacitor& conn(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::conn, value);
        return *this;
    }

    Capacitor& conn(Connection value)
    {
        Obj_SetInt32(ptr, Properties::conn, int32_t(value));
        return *this;
    }

    Capacitor& conn(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    Capacitor& conn(const char *value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN
    ///
    /// DSS property name: conn, DSS property index: 6
    ///
    string conn_str()
    {
        return get_prop_string(Properties::conn);
    }

    /// ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN
    ///
    /// DSS property name: conn, DSS property index: 6
    ///
    Capacitor& conn_str(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// Nodal cap. matrix, lower triangle, microfarads, of the following form:
    ///
    /// cmatrix="c11 | -c21 c22 | -c31 -c32 c33"
    ///
    /// All steps are assumed the same if this property is used.
    ///
    /// DSS property name: cmatrix, DSS property index: 7
    ///
    VectorXd cmatrix()
    {
        return get_array<VectorXd>(Properties::cmatrix);
    }

    Capacitor& cmatrix(VectorXd &value)
    {
        set_array<VectorXd>(Properties::cmatrix, value);
        return *this;
    }

    /// ARRAY of Capacitance, each phase, for each step, microfarads.
    /// See Rules for NumSteps.
    ///
    /// DSS property name: cuf, DSS property index: 8
    ///
    VectorXd cuf()
    {
        return get_array<VectorXd>(Properties::cuf);
    }

    Capacitor& cuf(VectorXd &value)
    {
        set_array<VectorXd>(Properties::cuf, value);
        return *this;
    }

    /// ARRAY of series resistance in each phase (line), ohms. Default is 0.0
    ///
    /// DSS property name: R, DSS property index: 9
    ///
    VectorXd R()
    {
        return get_array<VectorXd>(Properties::R);
    }

    Capacitor& R(VectorXd &value)
    {
        set_array<VectorXd>(Properties::R, value);
        return *this;
    }

    /// ARRAY of series inductive reactance(s) in each phase (line) for filter, ohms at base frequency. Use this OR "h" property to define filter. Default is 0.0.
    ///
    /// DSS property name: XL, DSS property index: 10
    ///
    VectorXd XL()
    {
        return get_array<VectorXd>(Properties::XL);
    }

    Capacitor& XL(VectorXd &value)
    {
        set_array<VectorXd>(Properties::XL, value);
        return *this;
    }

    /// ARRAY of harmonics to which each step is tuned. Zero is interpreted as meaning zero reactance (no filter). Default is zero.
    ///
    /// DSS property name: Harm, DSS property index: 11
    ///
    VectorXd Harm()
    {
        return get_array<VectorXd>(Properties::Harm);
    }

    Capacitor& Harm(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Harm, value);
        return *this;
    }

    /// Number of steps in this capacitor bank. Default = 1. Forces reallocation of the capacitance, reactor, and states array.  Rules: If this property was previously =1, the value in the kvar property is divided equally among the steps. The kvar property does not need to be reset if that is accurate.  If the Cuf or Cmatrix property was used previously, all steps are set to the value of the first step. The states property is set to all steps on. All filter steps are set to the same harmonic. If this property was previously >1, the arrays are reallocated, but no values are altered. You must SUBSEQUENTLY assign all array properties.
    ///
    /// DSS property name: Numsteps, DSS property index: 12
    ///
    int32_t Numsteps()
    {
        return Obj_GetInt32(ptr, Properties::Numsteps);
    }

    Capacitor& Numsteps(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Numsteps, value);
        return *this;
    }

    /// ARRAY of integers {1|0} states representing the state of each step (on|off). Defaults to 1 when reallocated (on). Capcontrol will modify this array as it turns steps on or off.
    ///
    /// DSS property name: states, DSS property index: 13
    ///
    VectorXi states()
    {
        return get_array<VectorXi>(Properties::states);
    }

    Capacitor& states(VectorXi &value)
    {
        set_array<VectorXi>(Properties::states, value);
        return *this;
    }

    /// Normal rated current.
    ///
    /// DSS property name: normamps, DSS property index: 14
    ///
    double normamps()
    {
        return Obj_GetFloat64(ptr, Properties::normamps);
    }

    Capacitor& normamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::normamps, value);
        return *this;
    }

    /// Maximum or emerg current.
    ///
    /// DSS property name: emergamps, DSS property index: 15
    ///
    double emergamps()
    {
        return Obj_GetFloat64(ptr, Properties::emergamps);
    }

    Capacitor& emergamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::emergamps, value);
        return *this;
    }

    /// Failure rate per year.
    ///
    /// DSS property name: faultrate, DSS property index: 16
    ///
    double faultrate()
    {
        return Obj_GetFloat64(ptr, Properties::faultrate);
    }

    Capacitor& faultrate(double value)
    {
        Obj_SetFloat64(ptr, Properties::faultrate, value);
        return *this;
    }

    /// Percent of failures that become permanent.
    ///
    /// DSS property name: pctperm, DSS property index: 17
    ///
    double pctperm()
    {
        return Obj_GetFloat64(ptr, Properties::pctperm);
    }

    Capacitor& pctperm(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctperm, value);
        return *this;
    }

    /// Hours to repair.
    ///
    /// DSS property name: repair, DSS property index: 18
    ///
    double repair()
    {
        return Obj_GetFloat64(ptr, Properties::repair);
    }

    Capacitor& repair(double value)
    {
        Obj_SetFloat64(ptr, Properties::repair, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 19
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Capacitor& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 20
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Capacitor& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 21
    ///
    Capacitor& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 21
    ///
    Capacitor& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Reactor: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 23;
    struct Properties
    {
        enum {
            bus1 = 1,
            bus2 = 2,
            phases = 3,
            kvar = 4,
            kv = 5,
            conn = 6,
            Rmatrix = 7,
            Xmatrix = 8,
            Parallel = 9,
            R = 10,
            X = 11,
            Rp = 12,
            Z1 = 13,
            Z2 = 14,
            Z0 = 15,
            Z = 16,
            RCurve = 17,
            LCurve = 18,
            LmH = 19,
            normamps = 20,
            emergamps = 21,
            faultrate = 22,
            pctperm = 23,
            repair = 24,
            basefreq = 25,
            enabled = 26,
            like = 27,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Reactor(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Reactor(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Reactor element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Reactor(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Reactor element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Reactor& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Reactor& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name of first bus. Examples:
    /// bus1=busname
    /// bus1=busname.1.2.3
    ///
    /// Bus2 property will default to this bus, node 0, unless previously specified. Only Bus1 need be specified for a Yg shunt reactor.
    ///
    /// DSS property name: bus1, DSS property index: 1
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    Reactor& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    Reactor& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Name of 2nd bus. Defaults to all phases connected to first bus, node 0, (Shunt Wye Connection) except when Bus2 is specifically defined.
    ///
    /// Not necessary to specify for delta (LL) connection
    ///
    /// DSS property name: bus2, DSS property index: 2
    ///
    string bus2()
    {
        return get_prop_string(Properties::bus2);
    }

    Reactor& bus2(const string &value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    Reactor& bus2(const char* value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    /// Number of phases.
    ///
    /// DSS property name: phases, DSS property index: 3
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    Reactor& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Total kvar, all phases.  Evenly divided among phases. Only determines X. Specify R separately
    ///
    /// DSS property name: kvar, DSS property index: 4
    ///
    double kvar()
    {
        return Obj_GetFloat64(ptr, Properties::kvar);
    }

    Reactor& kvar(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvar, value);
        return *this;
    }

    /// For 2, 3-phase, kV phase-phase. Otherwise specify actual coil rating.
    ///
    /// DSS property name: kv, DSS property index: 5
    ///
    double kv()
    {
        return Obj_GetFloat64(ptr, Properties::kv);
    }

    Reactor& kv(double value)
    {
        Obj_SetFloat64(ptr, Properties::kv, value);
        return *this;
    }

    /// ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN. If Delta, then only one terminal.
    ///
    /// DSS property name: conn, DSS property index: 6
    ///
    Connection conn()
    {
        return Connection(Obj_GetInt32(ptr, Properties::conn));
    }

    Reactor& conn(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::conn, value);
        return *this;
    }

    Reactor& conn(Connection value)
    {
        Obj_SetInt32(ptr, Properties::conn, int32_t(value));
        return *this;
    }

    Reactor& conn(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    Reactor& conn(const char *value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN. If Delta, then only one terminal.
    ///
    /// DSS property name: conn, DSS property index: 6
    ///
    string conn_str()
    {
        return get_prop_string(Properties::conn);
    }

    /// ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN. If Delta, then only one terminal.
    ///
    /// DSS property name: conn, DSS property index: 6
    ///
    Reactor& conn_str(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// Resistance matrix, lower triangle, ohms at base frequency. Order of the matrix is the number of phases. Mutually exclusive to specifying parameters by kvar or X.
    ///
    /// DSS property name: Rmatrix, DSS property index: 7
    ///
    VectorXd Rmatrix()
    {
        return get_array<VectorXd>(Properties::Rmatrix);
    }

    Reactor& Rmatrix(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Rmatrix, value);
        return *this;
    }

    /// Reactance matrix, lower triangle, ohms at base frequency. Order of the matrix is the number of phases. Mutually exclusive to specifying parameters by kvar or X.
    ///
    /// DSS property name: Xmatrix, DSS property index: 8
    ///
    VectorXd Xmatrix()
    {
        return get_array<VectorXd>(Properties::Xmatrix);
    }

    Reactor& Xmatrix(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Xmatrix, value);
        return *this;
    }

    /// {Yes | No}  Default=No. Indicates whether Rmatrix and Xmatrix are to be considered in parallel. Default is series. For other models, specify R and Rp.
    ///
    /// DSS property name: Parallel, DSS property index: 9
    ///
    bool Parallel()
    {
        return Obj_GetInt32(ptr, Properties::Parallel) != 0;
    }

    Reactor& Parallel(bool value)
    {
        Obj_SetInt32(ptr, Properties::Parallel, value);
        return *this;
    }

    /// Resistance (in series with reactance), each phase, ohms. This property applies to REACTOR specified by either kvar or X. See also help on Z.
    ///
    /// DSS property name: R, DSS property index: 10
    ///
    double R()
    {
        return Obj_GetFloat64(ptr, Properties::R);
    }

    Reactor& R(double value)
    {
        Obj_SetFloat64(ptr, Properties::R, value);
        return *this;
    }

    /// Reactance, each phase, ohms at base frequency. See also help on Z and LmH properties.
    ///
    /// DSS property name: X, DSS property index: 11
    ///
    double X()
    {
        return Obj_GetFloat64(ptr, Properties::X);
    }

    Reactor& X(double value)
    {
        Obj_SetFloat64(ptr, Properties::X, value);
        return *this;
    }

    /// Resistance in parallel with R and X (the entire branch). Assumed infinite if not specified.
    ///
    /// DSS property name: Rp, DSS property index: 12
    ///
    double Rp()
    {
        return Obj_GetFloat64(ptr, Properties::Rp);
    }

    Reactor& Rp(double value)
    {
        Obj_SetFloat64(ptr, Properties::Rp, value);
        return *this;
    }

    /// Positive-sequence impedance, ohms, as a 2-element array representing a complex number. Example:
    ///
    /// Z1=[1, 2]  ! represents 1 + j2
    ///
    /// If defined, Z1, Z2, and Z0 are used to define the impedance matrix of the REACTOR. Z1 MUST BE DEFINED TO USE THIS OPTION FOR DEFINING THE MATRIX.
    ///
    /// Side Effect: Sets Z2 and Z0 to same values unless they were previously defined.
    ///
    /// DSS property name: Z1, DSS property index: 13
    ///
    complex Z1()
    {
        return get_complex(Properties::Z1);
    }
    Reactor& Z1(complex value)
    {
        set_complex(Properties::Z1, value);
        return *this;
    }

    /// Negative-sequence impedance, ohms, as a 2-element array representing a complex number. Example:
    ///
    /// Z2=[1, 2]  ! represents 1 + j2
    ///
    /// Used to define the impedance matrix of the REACTOR if Z1 is also specified.
    ///
    /// Note: Z2 defaults to Z1 if it is not specifically defined. If Z2 is not equal to Z1, the impedance matrix is asymmetrical.
    ///
    /// DSS property name: Z2, DSS property index: 14
    ///
    complex Z2()
    {
        return get_complex(Properties::Z2);
    }
    Reactor& Z2(complex value)
    {
        set_complex(Properties::Z2, value);
        return *this;
    }

    /// Zer0-sequence impedance, ohms, as a 2-element array representing a complex number. Example:
    ///
    /// Z0=[3, 4]  ! represents 3 + j4
    ///
    /// Used to define the impedance matrix of the REACTOR if Z1 is also specified.
    ///
    /// Note: Z0 defaults to Z1 if it is not specifically defined.
    ///
    /// DSS property name: Z0, DSS property index: 15
    ///
    complex Z0()
    {
        return get_complex(Properties::Z0);
    }
    Reactor& Z0(complex value)
    {
        set_complex(Properties::Z0, value);
        return *this;
    }

    /// Alternative way of defining R and X properties. Enter a 2-element array representing R +jX in ohms. Example:
    ///
    /// Z=[5  10]   ! equivalent to R=5  X=10
    ///
    /// DSS property name: Z, DSS property index: 16
    ///
    complex Z()
    {
        return get_complex(Properties::Z);
    }
    Reactor& Z(complex value)
    {
        set_complex(Properties::Z, value);
        return *this;
    }

    /// Name of XYCurve object, previously defined, describing per-unit variation of phase resistance, R, vs. frequency. Applies to resistance specified by R or Z property. If actual values are not known, R often increases by approximately the square root of frequency.
    ///
    /// DSS property name: RCurve, DSS property index: 17
    ///
    string RCurve()
    {
        return get_prop_string(Properties::RCurve);
    }

    Reactor& RCurve(const string &value)
    {
        set_string(Properties::RCurve, value);
        return *this;
    }

    Reactor& RCurve(dss::obj::XYcurve &value)
    {
        set_obj(Properties::RCurve, value);
        return *this;
    }

    /// Name of XYCurve object, previously defined, describing per-unit variation of phase resistance, R, vs. frequency. Applies to resistance specified by R or Z property. If actual values are not known, R often increases by approximately the square root of frequency.
    ///
    /// DSS property name: RCurve, DSS property index: 17
    ///
    dss::obj::XYcurve RCurve_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::RCurve);
    }

    Reactor& RCurve_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::RCurve, value);
        return *this;
    }

    /// Name of XYCurve object, previously defined, describing per-unit variation of phase inductance, L=X/w, vs. frequency. Applies to reactance specified by X, LmH, Z, or kvar property.L generally decreases somewhat with frequency above the base frequency, approaching a limit at a few kHz.
    ///
    /// DSS property name: LCurve, DSS property index: 18
    ///
    string LCurve()
    {
        return get_prop_string(Properties::LCurve);
    }

    Reactor& LCurve(const string &value)
    {
        set_string(Properties::LCurve, value);
        return *this;
    }

    Reactor& LCurve(dss::obj::XYcurve &value)
    {
        set_obj(Properties::LCurve, value);
        return *this;
    }

    /// Name of XYCurve object, previously defined, describing per-unit variation of phase inductance, L=X/w, vs. frequency. Applies to reactance specified by X, LmH, Z, or kvar property.L generally decreases somewhat with frequency above the base frequency, approaching a limit at a few kHz.
    ///
    /// DSS property name: LCurve, DSS property index: 18
    ///
    dss::obj::XYcurve LCurve_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::LCurve);
    }

    Reactor& LCurve_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::LCurve, value);
        return *this;
    }

    /// Inductance, mH. Alternate way to define the reactance, X, property.
    ///
    /// DSS property name: LmH, DSS property index: 19
    ///
    double LmH()
    {
        return Obj_GetFloat64(ptr, Properties::LmH);
    }

    Reactor& LmH(double value)
    {
        Obj_SetFloat64(ptr, Properties::LmH, value);
        return *this;
    }

    /// Normal rated current.
    ///
    /// DSS property name: normamps, DSS property index: 20
    ///
    double normamps()
    {
        return Obj_GetFloat64(ptr, Properties::normamps);
    }

    Reactor& normamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::normamps, value);
        return *this;
    }

    /// Maximum or emerg current.
    ///
    /// DSS property name: emergamps, DSS property index: 21
    ///
    double emergamps()
    {
        return Obj_GetFloat64(ptr, Properties::emergamps);
    }

    Reactor& emergamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::emergamps, value);
        return *this;
    }

    /// Failure rate per year.
    ///
    /// DSS property name: faultrate, DSS property index: 22
    ///
    double faultrate()
    {
        return Obj_GetFloat64(ptr, Properties::faultrate);
    }

    Reactor& faultrate(double value)
    {
        Obj_SetFloat64(ptr, Properties::faultrate, value);
        return *this;
    }

    /// Percent of failures that become permanent.
    ///
    /// DSS property name: pctperm, DSS property index: 23
    ///
    double pctperm()
    {
        return Obj_GetFloat64(ptr, Properties::pctperm);
    }

    Reactor& pctperm(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctperm, value);
        return *this;
    }

    /// Hours to repair.
    ///
    /// DSS property name: repair, DSS property index: 24
    ///
    double repair()
    {
        return Obj_GetFloat64(ptr, Properties::repair);
    }

    Reactor& repair(double value)
    {
        Obj_SetFloat64(ptr, Properties::repair, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 25
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Reactor& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 26
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Reactor& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 27
    ///
    Reactor& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 27
    ///
    Reactor& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class CapControl: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 24;
    struct Properties
    {
        enum {
            element = 1,
            terminal = 2,
            capacitor = 3,
            type = 4,
            PTratio = 5,
            CTratio = 6,
            ONsetting = 7,
            OFFsetting = 8,
            Delay = 9,
            VoltOverride = 10,
            Vmax = 11,
            Vmin = 12,
            DelayOFF = 13,
            DeadTime = 14,
            CTPhase = 15,
            PTPhase = 16,
            VBus = 17,
            EventLog = 18,
            UserModel = 19,
            UserData = 20,
            pctMinkvar = 21,
            Reset = 22,
            basefreq = 23,
            enabled = 24,
            like = 25,
        };
    };

    // Class-specific enumerations

        ///
        /// CapControl: Type (DSS enumeration for CapControl)
        ///
    enum class CapControlType: int32_t
    {
        Current = 0, ///< Current
        Voltage = 1, ///< Voltage
        kvar = 2, ///< kvar
        Time = 3, ///< Time
        PowerFactor = 4 ///< PowerFactor
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    CapControl(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    CapControl(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the CapControl element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    CapControl(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the CapControl element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    CapControl& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    CapControl& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Full object name of the circuit element, typically a line or transformer, to which the capacitor control's PT and/or CT are connected.There is no default; must be specified.
    ///
    /// DSS property name: element, DSS property index: 1
    ///
    string element()
    {
        return get_prop_string(Properties::element);
    }

    CapControl& element(const string &value)
    {
        set_string(Properties::element, value);
        return *this;
    }

    CapControl& element(dss::obj::DSSObj &value)
    {
        set_obj(Properties::element, value);
        return *this;
    }

    /// Full object name of the circuit element, typically a line or transformer, to which the capacitor control's PT and/or CT are connected.There is no default; must be specified.
    ///
    /// DSS property name: element, DSS property index: 1
    ///
    dss::obj::DSSObj element_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::element);
    }

    CapControl& element_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::element, value);
        return *this;
    }

    /// Number of the terminal of the circuit element to which the CapControl is connected. 1 or 2, typically.  Default is 1.
    ///
    /// DSS property name: terminal, DSS property index: 2
    ///
    int32_t terminal()
    {
        return Obj_GetInt32(ptr, Properties::terminal);
    }

    CapControl& terminal(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::terminal, value);
        return *this;
    }

    /// Name of Capacitor element which the CapControl controls. No Default; Must be specified.Do not specify the full object name; "Capacitor" is assumed for the object class.  Example:
    ///
    /// Capacitor=cap1
    ///
    /// DSS property name: capacitor, DSS property index: 3
    ///
    string capacitor()
    {
        return get_prop_string(Properties::capacitor);
    }

    CapControl& capacitor(const string &value)
    {
        set_string(Properties::capacitor, value);
        return *this;
    }

    CapControl& capacitor(dss::obj::Capacitor &value)
    {
        set_obj(Properties::capacitor, value);
        return *this;
    }

    /// Name of Capacitor element which the CapControl controls. No Default; Must be specified.Do not specify the full object name; "Capacitor" is assumed for the object class.  Example:
    ///
    /// Capacitor=cap1
    ///
    /// DSS property name: capacitor, DSS property index: 3
    ///
    dss::obj::Capacitor capacitor_obj()
    {
        return get_obj<dss::obj::Capacitor>(Properties::capacitor);
    }

    CapControl& capacitor_obj(dss::obj::Capacitor &value)
    {
        set_obj(Properties::capacitor, value);
        return *this;
    }

    /// {Current | voltage | kvar | PF | time } Control type.  Specify the ONsetting and OFFsetting appropriately with the type of control. (See help for ONsetting)
    ///
    /// DSS property name: type, DSS property index: 4
    ///
    CapControlType type()
    {
        return CapControlType(Obj_GetInt32(ptr, Properties::type));
    }

    CapControl& type(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::type, value);
        return *this;
    }

    CapControl& type(CapControlType value)
    {
        Obj_SetInt32(ptr, Properties::type, int32_t(value));
        return *this;
    }

    CapControl& type(const string &value)
    {
        set_string(Properties::type, value);
        return *this;
    }

    CapControl& type(const char *value)
    {
        set_string(Properties::type, value);
        return *this;
    }

    /// {Current | voltage | kvar | PF | time } Control type.  Specify the ONsetting and OFFsetting appropriately with the type of control. (See help for ONsetting)
    ///
    /// DSS property name: type, DSS property index: 4
    ///
    string type_str()
    {
        return get_prop_string(Properties::type);
    }

    /// {Current | voltage | kvar | PF | time } Control type.  Specify the ONsetting and OFFsetting appropriately with the type of control. (See help for ONsetting)
    ///
    /// DSS property name: type, DSS property index: 4
    ///
    CapControl& type_str(const string &value)
    {
        set_string(Properties::type, value);
        return *this;
    }

    /// Ratio of the PT that converts the monitored voltage to the control voltage. Default is 60.  If the capacitor is Wye, the 1st phase line-to-neutral voltage is monitored.  Else, the line-to-line voltage (1st - 2nd phase) is monitored.
    ///
    /// DSS property name: PTratio, DSS property index: 5
    ///
    double PTratio()
    {
        return Obj_GetFloat64(ptr, Properties::PTratio);
    }

    CapControl& PTratio(double value)
    {
        Obj_SetFloat64(ptr, Properties::PTratio, value);
        return *this;
    }

    /// Ratio of the CT from line amps to control ampere setting for current and kvar control types.
    ///
    /// DSS property name: CTratio, DSS property index: 6
    ///
    double CTratio()
    {
        return Obj_GetFloat64(ptr, Properties::CTratio);
    }

    CapControl& CTratio(double value)
    {
        Obj_SetFloat64(ptr, Properties::CTratio, value);
        return *this;
    }

    /// Value at which the control arms to switch the capacitor ON (or ratchet up a step).
    ///
    /// Type of Control:
    ///
    /// Current: Line Amps / CTratio
    /// Voltage: Line-Neutral (or Line-Line for delta) Volts / PTratio
    /// kvar:    Total kvar, all phases (3-phase for pos seq model). This is directional.
    /// PF:      Power Factor, Total power in monitored terminal. Negative for Leading.
    /// Time:    Hrs from Midnight as a floating point number (decimal). 7:30am would be entered as 7.5.
    ///
    /// DSS property name: ONsetting, DSS property index: 7
    ///
    double ONsetting()
    {
        return Obj_GetFloat64(ptr, Properties::ONsetting);
    }

    CapControl& ONsetting(double value)
    {
        Obj_SetFloat64(ptr, Properties::ONsetting, value);
        return *this;
    }

    /// Value at which the control arms to switch the capacitor OFF. (See help for ONsetting)For Time control, is OK to have Off time the next day ( < On time)
    ///
    /// DSS property name: OFFsetting, DSS property index: 8
    ///
    double OFFsetting()
    {
        return Obj_GetFloat64(ptr, Properties::OFFsetting);
    }

    CapControl& OFFsetting(double value)
    {
        Obj_SetFloat64(ptr, Properties::OFFsetting, value);
        return *this;
    }

    /// Time delay, in seconds, from when the control is armed before it sends out the switching command to turn ON.  The control may reset before the action actually occurs. This is used to determine which capacity control will act first. Default is 15.  You may specify any floating point number to achieve a model of whatever condition is necessary.
    ///
    /// DSS property name: Delay, DSS property index: 9
    ///
    double Delay()
    {
        return Obj_GetFloat64(ptr, Properties::Delay);
    }

    CapControl& Delay(double value)
    {
        Obj_SetFloat64(ptr, Properties::Delay, value);
        return *this;
    }

    /// {Yes | No}  Default is No.  Switch to indicate whether VOLTAGE OVERRIDE is to be considered. Vmax and Vmin must be set to reasonable values if this property is Yes.
    ///
    /// DSS property name: VoltOverride, DSS property index: 10
    ///
    bool VoltOverride()
    {
        return Obj_GetInt32(ptr, Properties::VoltOverride) != 0;
    }

    CapControl& VoltOverride(bool value)
    {
        Obj_SetInt32(ptr, Properties::VoltOverride, value);
        return *this;
    }

    /// Maximum voltage, in volts.  If the voltage across the capacitor divided by the PTRATIO is greater than this voltage, the capacitor will switch OFF regardless of other control settings. Default is 126 (goes with a PT ratio of 60 for 12.47 kV system).
    ///
    /// DSS property name: Vmax, DSS property index: 11
    ///
    double Vmax()
    {
        return Obj_GetFloat64(ptr, Properties::Vmax);
    }

    CapControl& Vmax(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vmax, value);
        return *this;
    }

    /// Minimum voltage, in volts.  If the voltage across the capacitor divided by the PTRATIO is less than this voltage, the capacitor will switch ON regardless of other control settings. Default is 115 (goes with a PT ratio of 60 for 12.47 kV system).
    ///
    /// DSS property name: Vmin, DSS property index: 12
    ///
    double Vmin()
    {
        return Obj_GetFloat64(ptr, Properties::Vmin);
    }

    CapControl& Vmin(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vmin, value);
        return *this;
    }

    /// Time delay, in seconds, for control to turn OFF when present state is ON. Default is 15.
    ///
    /// DSS property name: DelayOFF, DSS property index: 13
    ///
    double DelayOFF()
    {
        return Obj_GetFloat64(ptr, Properties::DelayOFF);
    }

    CapControl& DelayOFF(double value)
    {
        Obj_SetFloat64(ptr, Properties::DelayOFF, value);
        return *this;
    }

    /// Dead time after capacitor is turned OFF before it can be turned back ON. Default is 300 sec.
    ///
    /// DSS property name: DeadTime, DSS property index: 14
    ///
    double DeadTime()
    {
        return Obj_GetFloat64(ptr, Properties::DeadTime);
    }

    CapControl& DeadTime(double value)
    {
        Obj_SetFloat64(ptr, Properties::DeadTime, value);
        return *this;
    }

    /// Number of the phase being monitored for CURRENT control or one of {AVG | MAX | MIN} for all phases. Default=1. If delta or L-L connection, enter the first or the two phases being monitored [1-2, 2-3, 3-1]. Must be less than the number of phases. Does not apply to kvar control which uses all phases by default.
    ///
    /// DSS property name: CTPhase, DSS property index: 15
    ///
    int32_t CTPhase()
    {
        return Obj_GetInt32(ptr, Properties::CTPhase);
    }

    CapControl& CTPhase(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::CTPhase, value);
        return *this;
    }

    CapControl& CTPhase(MonitoredPhase value)
    {
        Obj_SetInt32(ptr, Properties::CTPhase, int32_t(value));
        return *this;
    }

    CapControl& CTPhase(const string &value)
    {
        set_string(Properties::CTPhase, value);
        return *this;
    }

    CapControl& CTPhase(const char *value)
    {
        set_string(Properties::CTPhase, value);
        return *this;
    }

    /// Number of the phase being monitored for CURRENT control or one of {AVG | MAX | MIN} for all phases. Default=1. If delta or L-L connection, enter the first or the two phases being monitored [1-2, 2-3, 3-1]. Must be less than the number of phases. Does not apply to kvar control which uses all phases by default.
    ///
    /// DSS property name: CTPhase, DSS property index: 15
    ///
    string CTPhase_str()
    {
        return get_prop_string(Properties::CTPhase);
    }

    /// Number of the phase being monitored for CURRENT control or one of {AVG | MAX | MIN} for all phases. Default=1. If delta or L-L connection, enter the first or the two phases being monitored [1-2, 2-3, 3-1]. Must be less than the number of phases. Does not apply to kvar control which uses all phases by default.
    ///
    /// DSS property name: CTPhase, DSS property index: 15
    ///
    CapControl& CTPhase_str(const string &value)
    {
        set_string(Properties::CTPhase, value);
        return *this;
    }

    /// Number of the phase being monitored for VOLTAGE control or one of {AVG | MAX | MIN} for all phases. Default=1. If delta or L-L connection, enter the first or the two phases being monitored [1-2, 2-3, 3-1]. Must be less than the number of phases. Does not apply to kvar control which uses all phases by default.
    ///
    /// DSS property name: PTPhase, DSS property index: 16
    ///
    int32_t PTPhase()
    {
        return Obj_GetInt32(ptr, Properties::PTPhase);
    }

    CapControl& PTPhase(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::PTPhase, value);
        return *this;
    }

    CapControl& PTPhase(MonitoredPhase value)
    {
        Obj_SetInt32(ptr, Properties::PTPhase, int32_t(value));
        return *this;
    }

    CapControl& PTPhase(const string &value)
    {
        set_string(Properties::PTPhase, value);
        return *this;
    }

    CapControl& PTPhase(const char *value)
    {
        set_string(Properties::PTPhase, value);
        return *this;
    }

    /// Number of the phase being monitored for VOLTAGE control or one of {AVG | MAX | MIN} for all phases. Default=1. If delta or L-L connection, enter the first or the two phases being monitored [1-2, 2-3, 3-1]. Must be less than the number of phases. Does not apply to kvar control which uses all phases by default.
    ///
    /// DSS property name: PTPhase, DSS property index: 16
    ///
    string PTPhase_str()
    {
        return get_prop_string(Properties::PTPhase);
    }

    /// Number of the phase being monitored for VOLTAGE control or one of {AVG | MAX | MIN} for all phases. Default=1. If delta or L-L connection, enter the first or the two phases being monitored [1-2, 2-3, 3-1]. Must be less than the number of phases. Does not apply to kvar control which uses all phases by default.
    ///
    /// DSS property name: PTPhase, DSS property index: 16
    ///
    CapControl& PTPhase_str(const string &value)
    {
        set_string(Properties::PTPhase, value);
        return *this;
    }

    /// Name of bus to use for voltage override function. Default is bus at monitored terminal. Sometimes it is useful to monitor a bus in another location to emulate various DMS control algorithms.
    ///
    /// DSS property name: VBus, DSS property index: 17
    ///
    string VBus()
    {
        return get_prop_string(Properties::VBus);
    }

    CapControl& VBus(const string &value)
    {
        set_string(Properties::VBus, value);
        return *this;
    }

    CapControl& VBus(const char* value)
    {
        set_string(Properties::VBus, value);
        return *this;
    }

    /// {Yes/True* | No/False} Default is YES for CapControl. Log control actions to Eventlog.
    ///
    /// DSS property name: EventLog, DSS property index: 18
    ///
    bool EventLog()
    {
        return Obj_GetInt32(ptr, Properties::EventLog) != 0;
    }

    CapControl& EventLog(bool value)
    {
        Obj_SetInt32(ptr, Properties::EventLog, value);
        return *this;
    }

    /// Name of DLL containing user-written CapControl model, overriding the default model.  Set to "none" to negate previous setting.
    ///
    /// DSS property name: UserModel, DSS property index: 19
    ///
    string UserModel()
    {
        return get_prop_string(Properties::UserModel);
    }

    CapControl& UserModel(const string &value)
    {
        set_string(Properties::UserModel, value);
        return *this;
    }

    CapControl& UserModel(const char* value)
    {
        set_string(Properties::UserModel, value);
        return *this;
    }

    /// String (in quotes or parentheses if necessary) that gets passed to the user-written CapControl model Edit function for defining the data required for that model.
    ///
    /// DSS property name: UserData, DSS property index: 20
    ///
    string UserData()
    {
        return get_prop_string(Properties::UserData);
    }

    CapControl& UserData(const string &value)
    {
        set_string(Properties::UserData, value);
        return *this;
    }

    CapControl& UserData(const char* value)
    {
        set_string(Properties::UserData, value);
        return *this;
    }

    /// For PF control option, min percent of total bank kvar at which control will close capacitor switch. Default = 50.
    ///
    /// DSS property name: pctMinkvar, DSS property index: 21
    ///
    double pctMinkvar()
    {
        return Obj_GetFloat64(ptr, Properties::pctMinkvar);
    }

    CapControl& pctMinkvar(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctMinkvar, value);
        return *this;
    }

    /// {Yes | No} If Yes, forces Reset of this CapControl.
    ///
    /// DSS property name: Reset, DSS property index: 22
    ///
    CapControl& Reset(bool value)
    {
        Obj_SetInt32(ptr, Properties::Reset, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 23
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    CapControl& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 24
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    CapControl& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 25
    ///
    CapControl& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 25
    ///
    CapControl& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Fault: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 25;
    struct Properties
    {
        enum {
            bus1 = 1,
            bus2 = 2,
            phases = 3,
            r = 4,
            pctstddev = 5,
            Gmatrix = 6,
            ONtime = 7,
            temporary = 8,
            MinAmps = 9,
            normamps = 10,
            emergamps = 11,
            faultrate = 12,
            pctperm = 13,
            repair = 14,
            basefreq = 15,
            enabled = 16,
            like = 17,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Fault(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Fault(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Fault element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Fault(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Fault element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Fault& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Fault& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name of first bus. Examples:
    ///
    /// bus1=busname
    /// bus1=busname.1.2.3
    ///
    /// Bus2 automatically defaults to busname.0,0,0 unless it was previously defined.
    ///
    /// DSS property name: bus1, DSS property index: 1
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    Fault& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    Fault& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Name of 2nd bus of the 2-terminal Fault object. Defaults to all phases connected to first bus, node 0, if not specified. (Shunt Wye Connection to ground reference)
    ///
    /// That is, the Fault defaults to a ground fault unless otherwise specified.
    ///
    /// DSS property name: bus2, DSS property index: 2
    ///
    string bus2()
    {
        return get_prop_string(Properties::bus2);
    }

    Fault& bus2(const string &value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    Fault& bus2(const char* value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    /// Number of Phases. Default is 1.
    ///
    /// DSS property name: phases, DSS property index: 3
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    Fault& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Resistance, each phase, ohms. Default is 0.0001. Assumed to be Mean value if gaussian random mode.Max value if uniform mode.  A Fault is actually a series resistance that defaults to a wye connection to ground on the second terminal.  You may reconnect the 2nd terminal to achieve whatever connection.  Use the Gmatrix property to specify an arbitrary conductance matrix.
    ///
    /// DSS property name: r, DSS property index: 4
    ///
    double r()
    {
        return Obj_GetFloat64(ptr, Properties::r);
    }

    Fault& r(double value)
    {
        Obj_SetFloat64(ptr, Properties::r, value);
        return *this;
    }

    /// Percent standard deviation in resistance to assume for Monte Carlo fault (MF) solution mode for GAUSSIAN distribution. Default is 0 (no variation from mean).
    ///
    /// DSS property name: %stddev, DSS property index: 5
    ///
    double pctstddev()
    {
        return Obj_GetFloat64(ptr, Properties::pctstddev);
    }

    Fault& pctstddev(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctstddev, value);
        return *this;
    }

    /// Use this to specify a nodal conductance (G) matrix to represent some arbitrary resistance network. Specify in lower triangle form as usual for DSS matrices.
    ///
    /// DSS property name: Gmatrix, DSS property index: 6
    ///
    VectorXd Gmatrix()
    {
        return get_array<VectorXd>(Properties::Gmatrix);
    }

    Fault& Gmatrix(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Gmatrix, value);
        return *this;
    }

    /// Time (sec) at which the fault is established for time varying simulations. Default is 0.0 (on at the beginning of the simulation)
    ///
    /// DSS property name: ONtime, DSS property index: 7
    ///
    double ONtime()
    {
        return Obj_GetFloat64(ptr, Properties::ONtime);
    }

    Fault& ONtime(double value)
    {
        Obj_SetFloat64(ptr, Properties::ONtime, value);
        return *this;
    }

    /// {Yes | No} Default is No.  Designate whether the fault is temporary.  For Time-varying simulations, the fault will be removed if the current through the fault drops below the MINAMPS criteria.
    ///
    /// DSS property name: temporary, DSS property index: 8
    ///
    bool temporary()
    {
        return Obj_GetInt32(ptr, Properties::temporary) != 0;
    }

    Fault& temporary(bool value)
    {
        Obj_SetInt32(ptr, Properties::temporary, value);
        return *this;
    }

    /// Minimum amps that can sustain a temporary fault. Default is 5.
    ///
    /// DSS property name: MinAmps, DSS property index: 9
    ///
    double MinAmps()
    {
        return Obj_GetFloat64(ptr, Properties::MinAmps);
    }

    Fault& MinAmps(double value)
    {
        Obj_SetFloat64(ptr, Properties::MinAmps, value);
        return *this;
    }

    /// Normal rated current.
    ///
    /// DSS property name: normamps, DSS property index: 10
    ///
    double normamps()
    {
        return Obj_GetFloat64(ptr, Properties::normamps);
    }

    Fault& normamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::normamps, value);
        return *this;
    }

    /// Maximum or emerg current.
    ///
    /// DSS property name: emergamps, DSS property index: 11
    ///
    double emergamps()
    {
        return Obj_GetFloat64(ptr, Properties::emergamps);
    }

    Fault& emergamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::emergamps, value);
        return *this;
    }

    /// Failure rate per year.
    ///
    /// DSS property name: faultrate, DSS property index: 12
    ///
    double faultrate()
    {
        return Obj_GetFloat64(ptr, Properties::faultrate);
    }

    Fault& faultrate(double value)
    {
        Obj_SetFloat64(ptr, Properties::faultrate, value);
        return *this;
    }

    /// Percent of failures that become permanent.
    ///
    /// DSS property name: pctperm, DSS property index: 13
    ///
    double pctperm()
    {
        return Obj_GetFloat64(ptr, Properties::pctperm);
    }

    Fault& pctperm(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctperm, value);
        return *this;
    }

    /// Hours to repair.
    ///
    /// DSS property name: repair, DSS property index: 14
    ///
    double repair()
    {
        return Obj_GetFloat64(ptr, Properties::repair);
    }

    Fault& repair(double value)
    {
        Obj_SetFloat64(ptr, Properties::repair, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 15
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Fault& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 16
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Fault& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 17
    ///
    Fault& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 17
    ///
    Fault& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Generator: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 26;
    struct Properties
    {
        enum {
            phases = 1,
            bus1 = 2,
            kv = 3,
            kW = 4,
            pf = 5,
            kvar = 6,
            model = 7,
            Vminpu = 8,
            Vmaxpu = 9,
            yearly = 10,
            daily = 11,
            duty = 12,
            dispmode = 13,
            dispvalue = 14,
            conn = 15,
            status = 16,
            cls = 17,
            Vpu = 18,
            maxkvar = 19,
            minkvar = 20,
            pvfactor = 21,
            forceon = 22,
            kVA = 23,
            MVA = 24,
            Xd = 25,
            Xdp = 26,
            Xdpp = 27,
            H = 28,
            D = 29,
            UserModel = 30,
            UserData = 31,
            ShaftModel = 32,
            ShaftData = 33,
            DutyStart = 34,
            debugtrace = 35,
            Balanced = 36,
            XRdp = 37,
            UseFuel = 38,
            FuelkWh = 39,
            pctFuel = 40,
            pctReserve = 41,
            Refuel = 42,
            spectrum = 43,
            basefreq = 44,
            enabled = 45,
            like = 46,
        };
    };

    // Class-specific enumerations

        ///
        /// Generator: Dispatch Mode (DSS enumeration for Generator)
        ///
    enum class GeneratorDispatchMode: int32_t
    {
        Default = 0, ///< Default
        LoadLevel = 1, ///< LoadLevel
        Price = 2 ///< Price
    };


        ///
        /// Generator: Status (DSS enumeration for Generator)
        ///
    enum class GeneratorStatus: int32_t
    {
        Variable = 0, ///< Variable
        Fixed = 1 ///< Fixed
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Generator(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Generator(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Generator element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Generator(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Generator element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Generator& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Generator& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of Phases, this Generator.  Power is evenly divided among phases.
    ///
    /// DSS property name: phases, DSS property index: 1
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    Generator& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Bus to which the Generator is connected.  May include specific node specification.
    ///
    /// DSS property name: bus1, DSS property index: 2
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    Generator& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    Generator& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Nominal rated (1.0 per unit) voltage, kV, for Generator. For 2- and 3-phase Generators, specify phase-phase kV. Otherwise, for phases=1 or phases>3, specify actual kV across each branch of the Generator. If wye (star), specify phase-neutral kV. If delta or phase-phase connected, specify phase-phase kV.
    ///
    /// DSS property name: kv, DSS property index: 3
    ///
    double kv()
    {
        return Obj_GetFloat64(ptr, Properties::kv);
    }

    Generator& kv(double value)
    {
        Obj_SetFloat64(ptr, Properties::kv, value);
        return *this;
    }

    /// Total base kW for the Generator.  A positive value denotes power coming OUT of the element,
    /// which is the opposite of a load. This value is modified depending on the dispatch mode. Unaffected by the global load multiplier and growth curves. If you want there to be more generation, you must add more generators or change this value.
    ///
    /// DSS property name: kW, DSS property index: 4
    ///
    double kW()
    {
        return Obj_GetFloat64(ptr, Properties::kW);
    }

    Generator& kW(double value)
    {
        Obj_SetFloat64(ptr, Properties::kW, value);
        return *this;
    }

    /// Generator power factor. Default is 0.80. Enter negative for leading powerfactor (when kW and kvar have opposite signs.)
    /// A positive power factor for a generator signifies that the generator produces vars
    /// as is typical for a synchronous generator.  Induction machines would be
    /// specified with a negative power factor.
    ///
    /// DSS property name: pf, DSS property index: 5
    ///
    double pf()
    {
        return Obj_GetFloat64(ptr, Properties::pf);
    }

    Generator& pf(double value)
    {
        Obj_SetFloat64(ptr, Properties::pf, value);
        return *this;
    }

    /// Specify the base kvar.  Alternative to specifying the power factor.  Side effect:  the power factor value is altered to agree based on present value of kW.
    ///
    /// DSS property name: kvar, DSS property index: 6
    ///
    double kvar()
    {
        return Obj_GetFloat64(ptr, Properties::kvar);
    }

    Generator& kvar(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvar, value);
        return *this;
    }

    /// Integer code for the model to use for generation variation with voltage. Valid values are:
    ///
    /// 1:Generator injects a constant kW at specified power factor.
    /// 2:Generator is modeled as a constant admittance.
    /// 3:Const kW, constant kV.  Somewhat like a conventional transmission power flow P-V generator.
    /// 4:Const kW, Fixed Q (Q never varies)
    /// 5:Const kW, Fixed Q(as a constant reactance)
    /// 6:Compute load injection from User-written Model.(see usage of Xd, Xdp)
    /// 7:Constant kW, kvar, but current-limited below Vminpu. Approximates a simple inverter. See also Balanced.
    ///
    /// DSS property name: model, DSS property index: 7
    ///
    int32_t model()
    {
        return Obj_GetInt32(ptr, Properties::model);
    }

    Generator& model(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::model, value);
        return *this;
    }

    /// Default = 0.90.  Minimum per unit voltage for which the Model is assumed to apply. Below this value, the load model reverts to a constant impedance model. For model 7, the current is limited to the value computed for constant power at Vminpu.
    ///
    /// DSS property name: Vminpu, DSS property index: 8
    ///
    double Vminpu()
    {
        return Obj_GetFloat64(ptr, Properties::Vminpu);
    }

    Generator& Vminpu(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vminpu, value);
        return *this;
    }

    /// Default = 1.10.  Maximum per unit voltage for which the Model is assumed to apply. Above this value, the load model reverts to a constant impedance model.
    ///
    /// DSS property name: Vmaxpu, DSS property index: 9
    ///
    double Vmaxpu()
    {
        return Obj_GetFloat64(ptr, Properties::Vmaxpu);
    }

    Generator& Vmaxpu(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vmaxpu, value);
        return *this;
    }

    /// Dispatch shape to use for yearly simulations.  Must be previously defined as a Loadshape object. If this is not specified, a constant value is assumed (no variation). If the generator is assumed to be ON continuously, specify Status=FIXED, or designate a curve that is 1.0 per unit at all times. Set to NONE to reset to no loadahape. Nominally for 8760 simulations.  If there are fewer points in the designated shape than the number of points in the solution, the curve is repeated.
    ///
    /// DSS property name: yearly, DSS property index: 10
    ///
    string yearly()
    {
        return get_prop_string(Properties::yearly);
    }

    Generator& yearly(const string &value)
    {
        set_string(Properties::yearly, value);
        return *this;
    }

    Generator& yearly(dss::obj::LoadShape &value)
    {
        set_obj(Properties::yearly, value);
        return *this;
    }

    /// Dispatch shape to use for yearly simulations.  Must be previously defined as a Loadshape object. If this is not specified, a constant value is assumed (no variation). If the generator is assumed to be ON continuously, specify Status=FIXED, or designate a curve that is 1.0 per unit at all times. Set to NONE to reset to no loadahape. Nominally for 8760 simulations.  If there are fewer points in the designated shape than the number of points in the solution, the curve is repeated.
    ///
    /// DSS property name: yearly, DSS property index: 10
    ///
    dss::obj::LoadShape yearly_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::yearly);
    }

    Generator& yearly_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::yearly, value);
        return *this;
    }

    /// Dispatch shape to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically.  If generator is assumed to be ON continuously, specify Status=FIXED, or designate a Loadshape objectthat is 1.0 perunit for all hours. Set to NONE to reset to no loadahape.
    ///
    /// DSS property name: daily, DSS property index: 11
    ///
    string daily()
    {
        return get_prop_string(Properties::daily);
    }

    Generator& daily(const string &value)
    {
        set_string(Properties::daily, value);
        return *this;
    }

    Generator& daily(dss::obj::LoadShape &value)
    {
        set_obj(Properties::daily, value);
        return *this;
    }

    /// Dispatch shape to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically.  If generator is assumed to be ON continuously, specify Status=FIXED, or designate a Loadshape objectthat is 1.0 perunit for all hours. Set to NONE to reset to no loadahape.
    ///
    /// DSS property name: daily, DSS property index: 11
    ///
    dss::obj::LoadShape daily_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::daily);
    }

    Generator& daily_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::daily, value);
        return *this;
    }

    /// Load shape to use for duty cycle dispatch simulations such as for wind generation. Must be previously defined as a Loadshape object. Typically would have time intervals less than 1 hr -- perhaps, in seconds. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.
    ///
    /// DSS property name: duty, DSS property index: 12
    ///
    string duty()
    {
        return get_prop_string(Properties::duty);
    }

    Generator& duty(const string &value)
    {
        set_string(Properties::duty, value);
        return *this;
    }

    Generator& duty(dss::obj::LoadShape &value)
    {
        set_obj(Properties::duty, value);
        return *this;
    }

    /// Load shape to use for duty cycle dispatch simulations such as for wind generation. Must be previously defined as a Loadshape object. Typically would have time intervals less than 1 hr -- perhaps, in seconds. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.
    ///
    /// DSS property name: duty, DSS property index: 12
    ///
    dss::obj::LoadShape duty_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::duty);
    }

    Generator& duty_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::duty, value);
        return *this;
    }

    /// {Default* | Loadlevel | Price } Default = Default. Dispatch mode. In default mode, gen is either always on or follows dispatch curve as specified. Otherwise, the gen comes on when either the global default load level (Loadshape "default") or the price level exceeds the dispatch value.
    ///
    /// DSS property name: dispmode, DSS property index: 13
    ///
    GeneratorDispatchMode dispmode()
    {
        return GeneratorDispatchMode(Obj_GetInt32(ptr, Properties::dispmode));
    }

    Generator& dispmode(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::dispmode, value);
        return *this;
    }

    Generator& dispmode(GeneratorDispatchMode value)
    {
        Obj_SetInt32(ptr, Properties::dispmode, int32_t(value));
        return *this;
    }

    Generator& dispmode(const string &value)
    {
        set_string(Properties::dispmode, value);
        return *this;
    }

    Generator& dispmode(const char *value)
    {
        set_string(Properties::dispmode, value);
        return *this;
    }

    /// {Default* | Loadlevel | Price } Default = Default. Dispatch mode. In default mode, gen is either always on or follows dispatch curve as specified. Otherwise, the gen comes on when either the global default load level (Loadshape "default") or the price level exceeds the dispatch value.
    ///
    /// DSS property name: dispmode, DSS property index: 13
    ///
    string dispmode_str()
    {
        return get_prop_string(Properties::dispmode);
    }

    /// {Default* | Loadlevel | Price } Default = Default. Dispatch mode. In default mode, gen is either always on or follows dispatch curve as specified. Otherwise, the gen comes on when either the global default load level (Loadshape "default") or the price level exceeds the dispatch value.
    ///
    /// DSS property name: dispmode, DSS property index: 13
    ///
    Generator& dispmode_str(const string &value)
    {
        set_string(Properties::dispmode, value);
        return *this;
    }

    /// Dispatch value.
    /// If = 0.0 (default) then Generator follow dispatch curves, if any.
    /// If > 0  then Generator is ON only when either the price signal (in Price dispatch mode) exceeds this value or the active circuit load multiplier * "default" loadshape value * the default yearly growth factor exceeds this value.  Then the generator follows dispatch curves (duty, daily, or yearly), if any (see also Status).
    ///
    /// DSS property name: dispvalue, DSS property index: 14
    ///
    double dispvalue()
    {
        return Obj_GetFloat64(ptr, Properties::dispvalue);
    }

    Generator& dispvalue(double value)
    {
        Obj_SetFloat64(ptr, Properties::dispvalue, value);
        return *this;
    }

    /// ={wye|LN|delta|LL}.  Default is wye.
    ///
    /// DSS property name: conn, DSS property index: 15
    ///
    Connection conn()
    {
        return Connection(Obj_GetInt32(ptr, Properties::conn));
    }

    Generator& conn(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::conn, value);
        return *this;
    }

    Generator& conn(Connection value)
    {
        Obj_SetInt32(ptr, Properties::conn, int32_t(value));
        return *this;
    }

    Generator& conn(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    Generator& conn(const char *value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// ={wye|LN|delta|LL}.  Default is wye.
    ///
    /// DSS property name: conn, DSS property index: 15
    ///
    string conn_str()
    {
        return get_prop_string(Properties::conn);
    }

    /// ={wye|LN|delta|LL}.  Default is wye.
    ///
    /// DSS property name: conn, DSS property index: 15
    ///
    Generator& conn_str(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// ={Fixed | Variable*}.  If Fixed, then dispatch multipliers do not apply. The generator is alway at full power when it is ON.  Default is Variable  (follows curves).
    ///
    /// DSS property name: status, DSS property index: 16
    ///
    GeneratorStatus status()
    {
        return GeneratorStatus(Obj_GetInt32(ptr, Properties::status));
    }

    Generator& status(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::status, value);
        return *this;
    }

    Generator& status(GeneratorStatus value)
    {
        Obj_SetInt32(ptr, Properties::status, int32_t(value));
        return *this;
    }

    Generator& status(const string &value)
    {
        set_string(Properties::status, value);
        return *this;
    }

    Generator& status(const char *value)
    {
        set_string(Properties::status, value);
        return *this;
    }

    /// ={Fixed | Variable*}.  If Fixed, then dispatch multipliers do not apply. The generator is alway at full power when it is ON.  Default is Variable  (follows curves).
    ///
    /// DSS property name: status, DSS property index: 16
    ///
    string status_str()
    {
        return get_prop_string(Properties::status);
    }

    /// ={Fixed | Variable*}.  If Fixed, then dispatch multipliers do not apply. The generator is alway at full power when it is ON.  Default is Variable  (follows curves).
    ///
    /// DSS property name: status, DSS property index: 16
    ///
    Generator& status_str(const string &value)
    {
        set_string(Properties::status, value);
        return *this;
    }

    /// An arbitrary integer number representing the class of Generator so that Generator values may be segregated by class.
    ///
    /// DSS property name: class, DSS property index: 17
    ///
    int32_t cls()
    {
        return Obj_GetInt32(ptr, Properties::cls);
    }

    Generator& cls(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::cls, value);
        return *this;
    }

    /// Per Unit voltage set point for Model = 3  (typical power flow model).  Default is 1.0.
    ///
    /// DSS property name: Vpu, DSS property index: 18
    ///
    double Vpu()
    {
        return Obj_GetFloat64(ptr, Properties::Vpu);
    }

    Generator& Vpu(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vpu, value);
        return *this;
    }

    /// Maximum kvar limit for Model = 3.  Defaults to twice the specified load kvar.  Always reset this if you change PF or kvar properties.
    ///
    /// DSS property name: maxkvar, DSS property index: 19
    ///
    double maxkvar()
    {
        return Obj_GetFloat64(ptr, Properties::maxkvar);
    }

    Generator& maxkvar(double value)
    {
        Obj_SetFloat64(ptr, Properties::maxkvar, value);
        return *this;
    }

    /// Minimum kvar limit for Model = 3. Enter a negative number if generator can absorb vars. Defaults to negative of Maxkvar.  Always reset this if you change PF or kvar properties.
    ///
    /// DSS property name: minkvar, DSS property index: 20
    ///
    double minkvar()
    {
        return Obj_GetFloat64(ptr, Properties::minkvar);
    }

    Generator& minkvar(double value)
    {
        Obj_SetFloat64(ptr, Properties::minkvar, value);
        return *this;
    }

    /// Deceleration factor for P-V generator model (Model=3).  Default is 0.1. If the circuit converges easily, you may want to use a higher number such as 1.0. Use a lower number if solution diverges. Use Debugtrace=yes to create a file that will trace the convergence of a generator model.
    ///
    /// DSS property name: pvfactor, DSS property index: 21
    ///
    double pvfactor()
    {
        return Obj_GetFloat64(ptr, Properties::pvfactor);
    }

    Generator& pvfactor(double value)
    {
        Obj_SetFloat64(ptr, Properties::pvfactor, value);
        return *this;
    }

    /// {Yes | No}  Forces generator ON despite requirements of other dispatch modes. Stays ON until this property is set to NO, or an internal algorithm cancels the forced ON state.
    ///
    /// DSS property name: forceon, DSS property index: 22
    ///
    bool forceon()
    {
        return Obj_GetInt32(ptr, Properties::forceon) != 0;
    }

    Generator& forceon(bool value)
    {
        Obj_SetInt32(ptr, Properties::forceon, value);
        return *this;
    }

    /// kVA rating of electrical machine. Defaults to 1.2* kW if not specified. Applied to machine or inverter definition for Dynamics mode solutions.
    ///
    /// DSS property name: kVA, DSS property index: 23
    ///
    double kVA()
    {
        return Obj_GetFloat64(ptr, Properties::kVA);
    }

    Generator& kVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::kVA, value);
        return *this;
    }

    /// MVA rating of electrical machine.  Alternative to using kVA=.
    ///
    /// DSS property name: MVA, DSS property index: 24
    ///
    double MVA()
    {
        return Obj_GetFloat64(ptr, Properties::MVA);
    }

    Generator& MVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::MVA, value);
        return *this;
    }

    /// Per unit synchronous reactance of machine. Presently used only for Thevinen impedance for power flow calcs of user models (model=6). Typically use a value 0.4 to 1.0. Default is 1.0
    ///
    /// DSS property name: Xd, DSS property index: 25
    ///
    double Xd()
    {
        return Obj_GetFloat64(ptr, Properties::Xd);
    }

    Generator& Xd(double value)
    {
        Obj_SetFloat64(ptr, Properties::Xd, value);
        return *this;
    }

    /// Per unit transient reactance of the machine.  Used for Dynamics mode and Fault studies.  Default is 0.27.For user models, this value is used for the Thevinen/Norton impedance for Dynamics Mode.
    ///
    /// DSS property name: Xdp, DSS property index: 26
    ///
    double Xdp()
    {
        return Obj_GetFloat64(ptr, Properties::Xdp);
    }

    Generator& Xdp(double value)
    {
        Obj_SetFloat64(ptr, Properties::Xdp, value);
        return *this;
    }

    /// Per unit subtransient reactance of the machine.  Used for Harmonics. Default is 0.20.
    ///
    /// DSS property name: Xdpp, DSS property index: 27
    ///
    double Xdpp()
    {
        return Obj_GetFloat64(ptr, Properties::Xdpp);
    }

    Generator& Xdpp(double value)
    {
        Obj_SetFloat64(ptr, Properties::Xdpp, value);
        return *this;
    }

    /// Per unit mass constant of the machine.  MW-sec/MVA.  Default is 1.0.
    ///
    /// DSS property name: H, DSS property index: 28
    ///
    double H()
    {
        return Obj_GetFloat64(ptr, Properties::H);
    }

    Generator& H(double value)
    {
        Obj_SetFloat64(ptr, Properties::H, value);
        return *this;
    }

    /// Damping constant.  Usual range is 0 to 4. Default is 1.0.  Adjust to get damping
    ///
    /// DSS property name: D, DSS property index: 29
    ///
    double D()
    {
        return Obj_GetFloat64(ptr, Properties::D);
    }

    Generator& D(double value)
    {
        Obj_SetFloat64(ptr, Properties::D, value);
        return *this;
    }

    /// Name of DLL containing user-written model, which computes the terminal currents for Dynamics studies, overriding the default model.  Set to "none" to negate previous setting.
    ///
    /// DSS property name: UserModel, DSS property index: 30
    ///
    string UserModel()
    {
        return get_prop_string(Properties::UserModel);
    }

    Generator& UserModel(const string &value)
    {
        set_string(Properties::UserModel, value);
        return *this;
    }

    Generator& UserModel(const char* value)
    {
        set_string(Properties::UserModel, value);
        return *this;
    }

    /// String (in quotes or parentheses) that gets passed to user-written model for defining the data required for that model.
    ///
    /// DSS property name: UserData, DSS property index: 31
    ///
    string UserData()
    {
        return get_prop_string(Properties::UserData);
    }

    Generator& UserData(const string &value)
    {
        set_string(Properties::UserData, value);
        return *this;
    }

    Generator& UserData(const char* value)
    {
        set_string(Properties::UserData, value);
        return *this;
    }

    /// Name of user-written DLL containing a Shaft model, which models the prime mover and determines the power on the shaft for Dynamics studies. Models additional mass elements other than the single-mass model in the DSS default model. Set to "none" to negate previous setting.
    ///
    /// DSS property name: ShaftModel, DSS property index: 32
    ///
    string ShaftModel()
    {
        return get_prop_string(Properties::ShaftModel);
    }

    Generator& ShaftModel(const string &value)
    {
        set_string(Properties::ShaftModel, value);
        return *this;
    }

    Generator& ShaftModel(const char* value)
    {
        set_string(Properties::ShaftModel, value);
        return *this;
    }

    /// String (in quotes or parentheses) that gets passed to user-written shaft dynamic model for defining the data for that model.
    ///
    /// DSS property name: ShaftData, DSS property index: 33
    ///
    string ShaftData()
    {
        return get_prop_string(Properties::ShaftData);
    }

    Generator& ShaftData(const string &value)
    {
        set_string(Properties::ShaftData, value);
        return *this;
    }

    Generator& ShaftData(const char* value)
    {
        set_string(Properties::ShaftData, value);
        return *this;
    }

    /// Starting time offset [hours] into the duty cycle shape for this generator, defaults to 0
    ///
    /// DSS property name: DutyStart, DSS property index: 34
    ///
    double DutyStart()
    {
        return Obj_GetFloat64(ptr, Properties::DutyStart);
    }

    Generator& DutyStart(double value)
    {
        Obj_SetFloat64(ptr, Properties::DutyStart, value);
        return *this;
    }

    /// {Yes | No }  Default is no.  Turn this on to capture the progress of the generator model for each iteration.  Creates a separate file for each generator named "GEN_name.CSV".
    ///
    /// DSS property name: debugtrace, DSS property index: 35
    ///
    bool debugtrace()
    {
        return Obj_GetInt32(ptr, Properties::debugtrace) != 0;
    }

    Generator& debugtrace(bool value)
    {
        Obj_SetInt32(ptr, Properties::debugtrace, value);
        return *this;
    }

    /// {Yes | No*} Default is No.  For Model=7, force balanced current only for 3-phase generators. Force zero- and negative-sequence to zero.
    ///
    /// DSS property name: Balanced, DSS property index: 36
    ///
    bool Balanced()
    {
        return Obj_GetInt32(ptr, Properties::Balanced) != 0;
    }

    Generator& Balanced(bool value)
    {
        Obj_SetInt32(ptr, Properties::Balanced, value);
        return *this;
    }

    /// Default is 20. X/R ratio for Xdp property for FaultStudy and Dynamic modes.
    ///
    /// DSS property name: XRdp, DSS property index: 37
    ///
    double XRdp()
    {
        return Obj_GetFloat64(ptr, Properties::XRdp);
    }

    Generator& XRdp(double value)
    {
        Obj_SetFloat64(ptr, Properties::XRdp, value);
        return *this;
    }

    /// {Yes | *No}. Activates the use of fuel for the operation of the generator. When the fuel level reaches the reserve level, the generator stops until it gets refueled. By default, the generator is connected to a continuous fuel supply, Use this mode to mimic dependency on fuel level for different generation technologies.
    ///
    /// DSS property name: UseFuel, DSS property index: 38
    ///
    bool UseFuel()
    {
        return Obj_GetInt32(ptr, Properties::UseFuel) != 0;
    }

    Generator& UseFuel(bool value)
    {
        Obj_SetInt32(ptr, Properties::UseFuel, value);
        return *this;
    }

    /// {*0}Is the nominal level of fuel for the generator (kWh). It only applies if UseFuel = Yes/True
    ///
    /// DSS property name: FuelkWh, DSS property index: 39
    ///
    double FuelkWh()
    {
        return Obj_GetFloat64(ptr, Properties::FuelkWh);
    }

    Generator& FuelkWh(double value)
    {
        Obj_SetFloat64(ptr, Properties::FuelkWh, value);
        return *this;
    }

    /// It is a number between 0 and 100 representing the current amount of fuel avaiable in percentage of FuelkWh. It only applies if UseFuel = Yes/True
    ///
    /// DSS property name: %Fuel, DSS property index: 40
    ///
    double pctFuel()
    {
        return Obj_GetFloat64(ptr, Properties::pctFuel);
    }

    Generator& pctFuel(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctFuel, value);
        return *this;
    }

    /// It is a number between 0 and 100 representing the reserve level in percentage of FuelkWh. It only applies if UseFuel = Yes/True
    ///
    /// DSS property name: %Reserve, DSS property index: 41
    ///
    double pctReserve()
    {
        return Obj_GetFloat64(ptr, Properties::pctReserve);
    }

    Generator& pctReserve(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctReserve, value);
        return *this;
    }

    /// It is a boolean value (Yes/True, No/False) that can be used to manually refuel the generator when needed. It only applies if UseFuel = Yes/True
    ///
    /// DSS property name: Refuel, DSS property index: 42
    ///
    Generator& Refuel(bool value)
    {
        Obj_SetInt32(ptr, Properties::Refuel, value);
        return *this;
    }

    /// Name of harmonic voltage or current spectrum for this generator. Voltage behind Xd" for machine - default. Current injection for inverter. Default value is "default", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 43
    ///
    string spectrum()
    {
        return get_prop_string(Properties::spectrum);
    }

    Generator& spectrum(const string &value)
    {
        set_string(Properties::spectrum, value);
        return *this;
    }

    Generator& spectrum(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Name of harmonic voltage or current spectrum for this generator. Voltage behind Xd" for machine - default. Current injection for inverter. Default value is "default", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 43
    ///
    dss::obj::Spectrum spectrum_obj()
    {
        return get_obj<dss::obj::Spectrum>(Properties::spectrum);
    }

    Generator& spectrum_obj(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 44
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Generator& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 45
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Generator& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 46
    ///
    Generator& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 46
    ///
    Generator& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class GenDispatcher: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 27;
    struct Properties
    {
        enum {
            Element = 1,
            Terminal = 2,
            kWLimit = 3,
            kWBand = 4,
            kvarlimit = 5,
            GenList = 6,
            Weights = 7,
            basefreq = 8,
            enabled = 9,
            like = 10,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    GenDispatcher(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    GenDispatcher(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the GenDispatcher element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    GenDispatcher(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the GenDispatcher element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    GenDispatcher& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    GenDispatcher& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Full object name of the circuit element, typically a line or transformer, which the control is monitoring. There is no default; must be specified.
    ///
    /// DSS property name: Element, DSS property index: 1
    ///
    string Element()
    {
        return get_prop_string(Properties::Element);
    }

    GenDispatcher& Element(const string &value)
    {
        set_string(Properties::Element, value);
        return *this;
    }

    GenDispatcher& Element(dss::obj::DSSObj &value)
    {
        set_obj(Properties::Element, value);
        return *this;
    }

    /// Full object name of the circuit element, typically a line or transformer, which the control is monitoring. There is no default; must be specified.
    ///
    /// DSS property name: Element, DSS property index: 1
    ///
    dss::obj::DSSObj Element_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::Element);
    }

    GenDispatcher& Element_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::Element, value);
        return *this;
    }

    /// Number of the terminal of the circuit element to which the GenDispatcher control is connected. 1 or 2, typically.  Default is 1. Make sure you have the direction on the power matching the sign of kWLimit.
    ///
    /// DSS property name: Terminal, DSS property index: 2
    ///
    int32_t Terminal()
    {
        return Obj_GetInt32(ptr, Properties::Terminal);
    }

    GenDispatcher& Terminal(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Terminal, value);
        return *this;
    }

    /// kW Limit for the monitored element. The generators are dispatched to hold the power in band.
    ///
    /// DSS property name: kWLimit, DSS property index: 3
    ///
    double kWLimit()
    {
        return Obj_GetFloat64(ptr, Properties::kWLimit);
    }

    GenDispatcher& kWLimit(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWLimit, value);
        return *this;
    }

    /// Bandwidth (kW) of the dead band around the target limit.No dispatch changes are attempted if the power in the monitored terminal stays within this band.
    ///
    /// DSS property name: kWBand, DSS property index: 4
    ///
    double kWBand()
    {
        return Obj_GetFloat64(ptr, Properties::kWBand);
    }

    GenDispatcher& kWBand(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWBand, value);
        return *this;
    }

    /// Max kvar to be delivered through the element.  Uses same dead band as kW.
    ///
    /// DSS property name: kvarlimit, DSS property index: 5
    ///
    double kvarlimit()
    {
        return Obj_GetFloat64(ptr, Properties::kvarlimit);
    }

    GenDispatcher& kvarlimit(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvarlimit, value);
        return *this;
    }

    /// Array list of generators to be dispatched.  If not specified, all generators in the circuit are assumed dispatchable.
    ///
    /// DSS property name: GenList, DSS property index: 6
    ///
    strings GenList()
    {
        return get_array<strings>(Properties::GenList);
    }

    GenDispatcher& GenList(strings &value)
    {
        set_array<strings>(Properties::GenList, value);
        return *this;
    }

    /// GenDispatcher.Weights
    ///
    /// DSS property name: Weights, DSS property index: 7
    ///
    VectorXd Weights()
    {
        return get_array<VectorXd>(Properties::Weights);
    }

    GenDispatcher& Weights(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Weights, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 8
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    GenDispatcher& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 9
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    GenDispatcher& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 10
    ///
    GenDispatcher& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 10
    ///
    GenDispatcher& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Storage: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 28;
    struct Properties
    {
        enum {
            phases = 1,
            bus1 = 2,
            kv = 3,
            conn = 4,
            kW = 5,
            kvar = 6,
            pf = 7,
            kVA = 8,
            pctCutin = 9,
            pctCutout = 10,
            EffCurve = 11,
            VarFollowInverter = 12,
            kvarMax = 13,
            kvarMaxAbs = 14,
            WattPriority = 15,
            PFPriority = 16,
            pctPminNoVars = 17,
            pctPminkvarMax = 18,
            kWrated = 19,
            pctkWrated = 20,
            kWhrated = 21,
            kWhstored = 22,
            pctstored = 23,
            pctreserve = 24,
            State = 25,
            pctDischarge = 26,
            pctCharge = 27,
            pctEffCharge = 28,
            pctEffDischarge = 29,
            pctIdlingkW = 30,
            pctR = 31,
            pctX = 32,
            model = 33,
            Vminpu = 34,
            Vmaxpu = 35,
            Balanced = 36,
            LimitCurrent = 37,
            yearly = 38,
            daily = 39,
            duty = 40,
            DispMode = 41,
            DischargeTrigger = 42,
            ChargeTrigger = 43,
            TimeChargeTrig = 44,
            cls = 45,
            DynaDLL = 46,
            DynaData = 47,
            UserModel = 48,
            UserData = 49,
            debugtrace = 50,
            spectrum = 51,
            basefreq = 52,
            enabled = 53,
            like = 54,
        };
    };

    // Class-specific enumerations

        ///
        /// Storage: State (DSS enumeration for Storage)
        ///
    enum class StorageState: int32_t
    {
        Charging = -1, ///< Charging
        Idling = 0, ///< Idling
        Discharging = 1 ///< Discharging
    };


        ///
        /// Storage: Dispatch Mode (DSS enumeration for Storage)
        ///
    enum class StorageDispatchMode: int32_t
    {
        Default = 0, ///< Default
        LoadLevel = 1, ///< LoadLevel
        Price = 2, ///< Price
        External = 3, ///< External
        Follow = 4 ///< Follow
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Storage(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Storage(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Storage element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Storage(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Storage element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Storage& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Storage& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of Phases, this Storage element.  Power is evenly divided among phases.
    ///
    /// DSS property name: phases, DSS property index: 1
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    Storage& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Bus to which the Storage element is connected.  May include specific node specification.
    ///
    /// DSS property name: bus1, DSS property index: 2
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    Storage& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    Storage& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Nominal rated (1.0 per unit) voltage, kV, for Storage element. For 2- and 3-phase Storage elements, specify phase-phase kV. Otherwise, specify actual kV across each branch of the Storage element.
    ///
    /// If wye (star), specify phase-neutral kV.
    ///
    /// If delta or phase-phase connected, specify phase-phase kV.
    ///
    /// DSS property name: kv, DSS property index: 3
    ///
    double kv()
    {
        return Obj_GetFloat64(ptr, Properties::kv);
    }

    Storage& kv(double value)
    {
        Obj_SetFloat64(ptr, Properties::kv, value);
        return *this;
    }

    /// ={wye|LN|delta|LL}.  Default is wye.
    ///
    /// DSS property name: conn, DSS property index: 4
    ///
    Connection conn()
    {
        return Connection(Obj_GetInt32(ptr, Properties::conn));
    }

    Storage& conn(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::conn, value);
        return *this;
    }

    Storage& conn(Connection value)
    {
        Obj_SetInt32(ptr, Properties::conn, int32_t(value));
        return *this;
    }

    Storage& conn(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    Storage& conn(const char *value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// ={wye|LN|delta|LL}.  Default is wye.
    ///
    /// DSS property name: conn, DSS property index: 4
    ///
    string conn_str()
    {
        return get_prop_string(Properties::conn);
    }

    /// ={wye|LN|delta|LL}.  Default is wye.
    ///
    /// DSS property name: conn, DSS property index: 4
    ///
    Storage& conn_str(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// Get/set the requested kW value. Final kW is subjected to the inverter ratings. A positive value denotes power coming OUT of the element, which is the opposite of a Load element. A negative value indicates the Storage element is in Charging state. This value is modified internally depending on the dispatch mode.
    ///
    /// DSS property name: kW, DSS property index: 5
    ///
    double kW()
    {
        return Obj_GetFloat64(ptr, Properties::kW);
    }

    Storage& kW(double value)
    {
        Obj_SetFloat64(ptr, Properties::kW, value);
        return *this;
    }

    /// Get/set the requested kvar value. Final kvar is subjected to the inverter ratings. Sets inverter to operate in constant kvar mode.
    ///
    /// DSS property name: kvar, DSS property index: 6
    ///
    double kvar()
    {
        return Obj_GetFloat64(ptr, Properties::kvar);
    }

    Storage& kvar(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvar, value);
        return *this;
    }

    /// Get/set the requested PF value. Final PF is subjected to the inverter ratings. Sets inverter to operate in constant PF mode. Nominally, the power factor for discharging (acting as a generator). Default is 1.0.
    ///
    /// Enter negative for leading power factor (when kW and kvar have opposite signs.)
    ///
    /// A positive power factor signifies kw and kvar at the same direction.
    ///
    /// DSS property name: pf, DSS property index: 7
    ///
    double pf()
    {
        return Obj_GetFloat64(ptr, Properties::pf);
    }

    Storage& pf(double value)
    {
        Obj_SetFloat64(ptr, Properties::pf, value);
        return *this;
    }

    /// Indicates the inverter nameplate capability (in kVA). Used as the base for Dynamics mode and Harmonics mode values.
    ///
    /// DSS property name: kVA, DSS property index: 8
    ///
    double kVA()
    {
        return Obj_GetFloat64(ptr, Properties::kVA);
    }

    Storage& kVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::kVA, value);
        return *this;
    }

    /// Cut-in power as a percentage of inverter kVA rating. It is the minimum DC power necessary to turn the inverter ON when it is OFF. Must be greater than or equal to %CutOut. Defaults to 2 for PVSystems and 0 for Storage elements which means that the inverter state will be always ON for this element.
    ///
    /// DSS property name: %Cutin, DSS property index: 9
    ///
    double pctCutin()
    {
        return Obj_GetFloat64(ptr, Properties::pctCutin);
    }

    Storage& pctCutin(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctCutin, value);
        return *this;
    }

    /// Cut-out power as a percentage of inverter kVA rating. It is the minimum DC power necessary to keep the inverter ON. Must be less than or equal to %CutIn. Defaults to 0, which means that, once ON, the inverter state will be always ON for this element.
    ///
    /// DSS property name: %Cutout, DSS property index: 10
    ///
    double pctCutout()
    {
        return Obj_GetFloat64(ptr, Properties::pctCutout);
    }

    Storage& pctCutout(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctCutout, value);
        return *this;
    }

    /// An XYCurve object, previously defined, that describes the PER UNIT efficiency vs PER UNIT of rated kVA for the inverter. Power at the AC side of the inverter is discounted by the multiplier obtained from this curve.
    ///
    /// DSS property name: EffCurve, DSS property index: 11
    ///
    string EffCurve()
    {
        return get_prop_string(Properties::EffCurve);
    }

    Storage& EffCurve(const string &value)
    {
        set_string(Properties::EffCurve, value);
        return *this;
    }

    Storage& EffCurve(dss::obj::XYcurve &value)
    {
        set_obj(Properties::EffCurve, value);
        return *this;
    }

    /// An XYCurve object, previously defined, that describes the PER UNIT efficiency vs PER UNIT of rated kVA for the inverter. Power at the AC side of the inverter is discounted by the multiplier obtained from this curve.
    ///
    /// DSS property name: EffCurve, DSS property index: 11
    ///
    dss::obj::XYcurve EffCurve_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::EffCurve);
    }

    Storage& EffCurve_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::EffCurve, value);
        return *this;
    }

    /// Boolean variable (Yes|No) or (True|False). Defaults to False, which indicates that the reactive power generation/absorption does not respect the inverter status.When set to True, the reactive power generation/absorption will cease when the inverter status is off, due to DC kW dropping below %CutOut.  The reactive power generation/absorption will begin again when the DC kW is above %CutIn.  When set to False, the Storage will generate/absorb reactive power regardless of the status of the inverter.
    ///
    /// DSS property name: VarFollowInverter, DSS property index: 12
    ///
    bool VarFollowInverter()
    {
        return Obj_GetInt32(ptr, Properties::VarFollowInverter) != 0;
    }

    Storage& VarFollowInverter(bool value)
    {
        Obj_SetInt32(ptr, Properties::VarFollowInverter, value);
        return *this;
    }

    /// Indicates the maximum reactive power GENERATION (un-signed numerical variable in kvar) for the inverter. Defaults to kVA rating of the inverter.
    ///
    /// DSS property name: kvarMax, DSS property index: 13
    ///
    double kvarMax()
    {
        return Obj_GetFloat64(ptr, Properties::kvarMax);
    }

    Storage& kvarMax(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvarMax, value);
        return *this;
    }

    /// Indicates the maximum reactive power ABSORPTION (un-signed numerical variable in kvar) for the inverter. Defaults to kvarMax.
    ///
    /// DSS property name: kvarMaxAbs, DSS property index: 14
    ///
    double kvarMaxAbs()
    {
        return Obj_GetFloat64(ptr, Properties::kvarMaxAbs);
    }

    Storage& kvarMaxAbs(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvarMaxAbs, value);
        return *this;
    }

    /// {Yes/No*/True/False} Set inverter to watt priority instead of the default var priority.
    ///
    /// DSS property name: WattPriority, DSS property index: 15
    ///
    bool WattPriority()
    {
        return Obj_GetInt32(ptr, Properties::WattPriority) != 0;
    }

    Storage& WattPriority(bool value)
    {
        Obj_SetInt32(ptr, Properties::WattPriority, value);
        return *this;
    }

    /// If set to true, priority is given to power factor and WattPriority is neglected. It works only if operating in either constant PF or constant kvar modes. Defaults to False.
    ///
    /// DSS property name: PFPriority, DSS property index: 16
    ///
    bool PFPriority()
    {
        return Obj_GetInt32(ptr, Properties::PFPriority) != 0;
    }

    Storage& PFPriority(bool value)
    {
        Obj_SetInt32(ptr, Properties::PFPriority, value);
        return *this;
    }

    /// Minimum active power as percentage of kWrated under which there is no vars production/absorption. Defaults to 0 (disabled).
    ///
    /// DSS property name: %PminNoVars, DSS property index: 17
    ///
    double pctPminNoVars()
    {
        return Obj_GetFloat64(ptr, Properties::pctPminNoVars);
    }

    Storage& pctPminNoVars(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctPminNoVars, value);
        return *this;
    }

    /// Minimum active power as percentage of kWrated that allows the inverter to produce/absorb reactive power up to its maximum reactive power, which can be either kvarMax or kvarMaxAbs, depending on the current operation quadrant. Defaults to 0 (disabled).
    ///
    /// DSS property name: %PminkvarMax, DSS property index: 18
    ///
    double pctPminkvarMax()
    {
        return Obj_GetFloat64(ptr, Properties::pctPminkvarMax);
    }

    Storage& pctPminkvarMax(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctPminkvarMax, value);
        return *this;
    }

    /// kW rating of power output. Base for Loadshapes when DispMode=Follow. Sets kVA property if it has not been specified yet. Defaults to 25.
    ///
    /// DSS property name: kWrated, DSS property index: 19
    ///
    double kWrated()
    {
        return Obj_GetFloat64(ptr, Properties::kWrated);
    }

    Storage& kWrated(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWrated, value);
        return *this;
    }

    /// Upper limit on active power as a percentage of kWrated. Defaults to 100 (disabled).
    ///
    /// DSS property name: %kWrated, DSS property index: 20
    ///
    double pctkWrated()
    {
        return Obj_GetFloat64(ptr, Properties::pctkWrated);
    }

    Storage& pctkWrated(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctkWrated, value);
        return *this;
    }

    /// Rated Storage capacity in kWh. Default is 50.
    ///
    /// DSS property name: kWhrated, DSS property index: 21
    ///
    double kWhrated()
    {
        return Obj_GetFloat64(ptr, Properties::kWhrated);
    }

    Storage& kWhrated(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWhrated, value);
        return *this;
    }

    /// Present amount of energy stored, kWh. Default is same as kWhrated.
    ///
    /// DSS property name: kWhstored, DSS property index: 22
    ///
    double kWhstored()
    {
        return Obj_GetFloat64(ptr, Properties::kWhstored);
    }

    Storage& kWhstored(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWhstored, value);
        return *this;
    }

    /// Present amount of energy stored, % of rated kWh. Default is 100.
    ///
    /// DSS property name: %stored, DSS property index: 23
    ///
    double pctstored()
    {
        return Obj_GetFloat64(ptr, Properties::pctstored);
    }

    Storage& pctstored(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctstored, value);
        return *this;
    }

    /// Percentage of rated kWh Storage capacity to be held in reserve for normal operation. Default = 20.
    /// This is treated as the minimum energy discharge level unless there is an emergency. For emergency operation set this property lower. Cannot be less than zero.
    ///
    /// DSS property name: %reserve, DSS property index: 24
    ///
    double pctreserve()
    {
        return Obj_GetFloat64(ptr, Properties::pctreserve);
    }

    Storage& pctreserve(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctreserve, value);
        return *this;
    }

    /// {IDLING | CHARGING | DISCHARGING}  Get/Set present operational state. In DISCHARGING mode, the Storage element acts as a generator and the kW property is positive. The element continues discharging at the scheduled output power level until the Storage reaches the reserve value. Then the state reverts to IDLING. In the CHARGING state, the Storage element behaves like a Load and the kW property is negative. The element continues to charge until the max Storage kWh is reached and then switches to IDLING state. In IDLING state, the element draws the idling losses plus the associated inverter losses.
    ///
    /// DSS property name: State, DSS property index: 25
    ///
    StorageState State()
    {
        return StorageState(Obj_GetInt32(ptr, Properties::State));
    }

    Storage& State(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::State, value);
        return *this;
    }

    Storage& State(StorageState value)
    {
        Obj_SetInt32(ptr, Properties::State, int32_t(value));
        return *this;
    }

    Storage& State(const string &value)
    {
        set_string(Properties::State, value);
        return *this;
    }

    Storage& State(const char *value)
    {
        set_string(Properties::State, value);
        return *this;
    }

    /// {IDLING | CHARGING | DISCHARGING}  Get/Set present operational state. In DISCHARGING mode, the Storage element acts as a generator and the kW property is positive. The element continues discharging at the scheduled output power level until the Storage reaches the reserve value. Then the state reverts to IDLING. In the CHARGING state, the Storage element behaves like a Load and the kW property is negative. The element continues to charge until the max Storage kWh is reached and then switches to IDLING state. In IDLING state, the element draws the idling losses plus the associated inverter losses.
    ///
    /// DSS property name: State, DSS property index: 25
    ///
    string State_str()
    {
        return get_prop_string(Properties::State);
    }

    /// {IDLING | CHARGING | DISCHARGING}  Get/Set present operational state. In DISCHARGING mode, the Storage element acts as a generator and the kW property is positive. The element continues discharging at the scheduled output power level until the Storage reaches the reserve value. Then the state reverts to IDLING. In the CHARGING state, the Storage element behaves like a Load and the kW property is negative. The element continues to charge until the max Storage kWh is reached and then switches to IDLING state. In IDLING state, the element draws the idling losses plus the associated inverter losses.
    ///
    /// DSS property name: State, DSS property index: 25
    ///
    Storage& State_str(const string &value)
    {
        set_string(Properties::State, value);
        return *this;
    }

    /// Discharge rate (output power) in percentage of rated kW. Default = 100.
    ///
    /// DSS property name: %Discharge, DSS property index: 26
    ///
    double pctDischarge()
    {
        return Obj_GetFloat64(ptr, Properties::pctDischarge);
    }

    Storage& pctDischarge(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctDischarge, value);
        return *this;
    }

    /// Charging rate (input power) in percentage of rated kW. Default = 100.
    ///
    /// DSS property name: %Charge, DSS property index: 27
    ///
    double pctCharge()
    {
        return Obj_GetFloat64(ptr, Properties::pctCharge);
    }

    Storage& pctCharge(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctCharge, value);
        return *this;
    }

    /// Percentage efficiency for CHARGING the Storage element. Default = 90.
    ///
    /// DSS property name: %EffCharge, DSS property index: 28
    ///
    double pctEffCharge()
    {
        return Obj_GetFloat64(ptr, Properties::pctEffCharge);
    }

    Storage& pctEffCharge(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctEffCharge, value);
        return *this;
    }

    /// Percentage efficiency for DISCHARGING the Storage element. Default = 90.
    ///
    /// DSS property name: %EffDischarge, DSS property index: 29
    ///
    double pctEffDischarge()
    {
        return Obj_GetFloat64(ptr, Properties::pctEffDischarge);
    }

    Storage& pctEffDischarge(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctEffDischarge, value);
        return *this;
    }

    /// Percentage of rated kW consumed by idling losses. Default = 1.
    ///
    /// DSS property name: %IdlingkW, DSS property index: 30
    ///
    double pctIdlingkW()
    {
        return Obj_GetFloat64(ptr, Properties::pctIdlingkW);
    }

    Storage& pctIdlingkW(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctIdlingkW, value);
        return *this;
    }

    /// Equivalent percentage internal resistance, ohms. Default is 0. Placed in series with internal voltage source for harmonics and dynamics modes. Use a combination of %IdlingkW, %EffCharge and %EffDischarge to account for losses in power flow modes.
    ///
    /// DSS property name: %R, DSS property index: 31
    ///
    double pctR()
    {
        return Obj_GetFloat64(ptr, Properties::pctR);
    }

    Storage& pctR(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctR, value);
        return *this;
    }

    /// Equivalent percentage internal reactance, ohms. Default is 50%. Placed in series with internal voltage source for harmonics and dynamics modes. (Limits fault current to 2 pu.
    ///
    /// DSS property name: %X, DSS property index: 32
    ///
    double pctX()
    {
        return Obj_GetFloat64(ptr, Properties::pctX);
    }

    Storage& pctX(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctX, value);
        return *this;
    }

    /// Integer code (default=1) for the model to be used for power output variation with voltage. Valid values are:
    ///
    /// 1:Storage element injects/absorbs a CONSTANT power.
    /// 2:Storage element is modeled as a CONSTANT IMPEDANCE.
    /// 3:Compute load injection from User-written Model.
    ///
    /// DSS property name: model, DSS property index: 33
    ///
    int32_t model()
    {
        return Obj_GetInt32(ptr, Properties::model);
    }

    Storage& model(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::model, value);
        return *this;
    }

    /// Default = 0.90.  Minimum per unit voltage for which the Model is assumed to apply. Below this value, the load model reverts to a constant impedance model.
    ///
    /// DSS property name: Vminpu, DSS property index: 34
    ///
    double Vminpu()
    {
        return Obj_GetFloat64(ptr, Properties::Vminpu);
    }

    Storage& Vminpu(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vminpu, value);
        return *this;
    }

    /// Default = 1.10.  Maximum per unit voltage for which the Model is assumed to apply. Above this value, the load model reverts to a constant impedance model.
    ///
    /// DSS property name: Vmaxpu, DSS property index: 35
    ///
    double Vmaxpu()
    {
        return Obj_GetFloat64(ptr, Properties::Vmaxpu);
    }

    Storage& Vmaxpu(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vmaxpu, value);
        return *this;
    }

    /// {Yes | No*} Default is No. Force balanced current only for 3-phase Storage. Forces zero- and negative-sequence to zero.
    ///
    /// DSS property name: Balanced, DSS property index: 36
    ///
    bool Balanced()
    {
        return Obj_GetInt32(ptr, Properties::Balanced) != 0;
    }

    Storage& Balanced(bool value)
    {
        Obj_SetInt32(ptr, Properties::Balanced, value);
        return *this;
    }

    /// Limits current magnitude to Vminpu value for both 1-phase and 3-phase Storage similar to Generator Model 7. For 3-phase, limits the positive-sequence current but not the negative-sequence.
    ///
    /// DSS property name: LimitCurrent, DSS property index: 37
    ///
    bool LimitCurrent()
    {
        return Obj_GetInt32(ptr, Properties::LimitCurrent) != 0;
    }

    Storage& LimitCurrent(bool value)
    {
        Obj_SetInt32(ptr, Properties::LimitCurrent, value);
        return *this;
    }

    /// Dispatch shape to use for yearly simulations.  Must be previously defined as a Loadshape object. If this is not specified, the Daily dispatch shape, if any, is repeated during Yearly solution modes. In the default dispatch mode, the Storage element uses this loadshape to trigger State changes.
    ///
    /// DSS property name: yearly, DSS property index: 38
    ///
    string yearly()
    {
        return get_prop_string(Properties::yearly);
    }

    Storage& yearly(const string &value)
    {
        set_string(Properties::yearly, value);
        return *this;
    }

    Storage& yearly(dss::obj::LoadShape &value)
    {
        set_obj(Properties::yearly, value);
        return *this;
    }

    /// Dispatch shape to use for yearly simulations.  Must be previously defined as a Loadshape object. If this is not specified, the Daily dispatch shape, if any, is repeated during Yearly solution modes. In the default dispatch mode, the Storage element uses this loadshape to trigger State changes.
    ///
    /// DSS property name: yearly, DSS property index: 38
    ///
    dss::obj::LoadShape yearly_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::yearly);
    }

    Storage& yearly_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::yearly, value);
        return *this;
    }

    /// Dispatch shape to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically.  In the default dispatch mode, the Storage element uses this loadshape to trigger State changes.
    ///
    /// DSS property name: daily, DSS property index: 39
    ///
    string daily()
    {
        return get_prop_string(Properties::daily);
    }

    Storage& daily(const string &value)
    {
        set_string(Properties::daily, value);
        return *this;
    }

    Storage& daily(dss::obj::LoadShape &value)
    {
        set_obj(Properties::daily, value);
        return *this;
    }

    /// Dispatch shape to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically.  In the default dispatch mode, the Storage element uses this loadshape to trigger State changes.
    ///
    /// DSS property name: daily, DSS property index: 39
    ///
    dss::obj::LoadShape daily_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::daily);
    }

    Storage& daily_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::daily, value);
        return *this;
    }

    /// Load shape to use for duty cycle dispatch simulations such as for solar ramp rate studies. Must be previously defined as a Loadshape object.
    ///
    /// Typically would have time intervals of 1-5 seconds.
    ///
    /// Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.
    ///
    /// DSS property name: duty, DSS property index: 40
    ///
    string duty()
    {
        return get_prop_string(Properties::duty);
    }

    Storage& duty(const string &value)
    {
        set_string(Properties::duty, value);
        return *this;
    }

    Storage& duty(dss::obj::LoadShape &value)
    {
        set_obj(Properties::duty, value);
        return *this;
    }

    /// Load shape to use for duty cycle dispatch simulations such as for solar ramp rate studies. Must be previously defined as a Loadshape object.
    ///
    /// Typically would have time intervals of 1-5 seconds.
    ///
    /// Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.
    ///
    /// DSS property name: duty, DSS property index: 40
    ///
    dss::obj::LoadShape duty_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::duty);
    }

    Storage& duty_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::duty, value);
        return *this;
    }

    /// {DEFAULT | FOLLOW | EXTERNAL | LOADLEVEL | PRICE } Default = "DEFAULT". Dispatch mode.
    ///
    /// In DEFAULT mode, Storage element state is triggered to discharge or charge at the specified rate by the loadshape curve corresponding to the solution mode.
    ///
    /// In FOLLOW mode the kW output of the Storage element follows the active loadshape multiplier until Storage is either exhausted or full. The element discharges for positive values and charges for negative values.  The loadshape is based on rated kW.
    ///
    /// In EXTERNAL mode, Storage element state is controlled by an external Storagecontroller2. This mode is automatically set if this Storage element is included in the element list of a StorageController element.
    ///
    /// For the other two dispatch modes, the Storage element state is controlled by either the global default Loadlevel value or the price level.
    ///
    /// DSS property name: DispMode, DSS property index: 41
    ///
    StorageDispatchMode DispMode()
    {
        return StorageDispatchMode(Obj_GetInt32(ptr, Properties::DispMode));
    }

    Storage& DispMode(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::DispMode, value);
        return *this;
    }

    Storage& DispMode(StorageDispatchMode value)
    {
        Obj_SetInt32(ptr, Properties::DispMode, int32_t(value));
        return *this;
    }

    Storage& DispMode(const string &value)
    {
        set_string(Properties::DispMode, value);
        return *this;
    }

    Storage& DispMode(const char *value)
    {
        set_string(Properties::DispMode, value);
        return *this;
    }

    /// {DEFAULT | FOLLOW | EXTERNAL | LOADLEVEL | PRICE } Default = "DEFAULT". Dispatch mode.
    ///
    /// In DEFAULT mode, Storage element state is triggered to discharge or charge at the specified rate by the loadshape curve corresponding to the solution mode.
    ///
    /// In FOLLOW mode the kW output of the Storage element follows the active loadshape multiplier until Storage is either exhausted or full. The element discharges for positive values and charges for negative values.  The loadshape is based on rated kW.
    ///
    /// In EXTERNAL mode, Storage element state is controlled by an external Storagecontroller2. This mode is automatically set if this Storage element is included in the element list of a StorageController element.
    ///
    /// For the other two dispatch modes, the Storage element state is controlled by either the global default Loadlevel value or the price level.
    ///
    /// DSS property name: DispMode, DSS property index: 41
    ///
    string DispMode_str()
    {
        return get_prop_string(Properties::DispMode);
    }

    /// {DEFAULT | FOLLOW | EXTERNAL | LOADLEVEL | PRICE } Default = "DEFAULT". Dispatch mode.
    ///
    /// In DEFAULT mode, Storage element state is triggered to discharge or charge at the specified rate by the loadshape curve corresponding to the solution mode.
    ///
    /// In FOLLOW mode the kW output of the Storage element follows the active loadshape multiplier until Storage is either exhausted or full. The element discharges for positive values and charges for negative values.  The loadshape is based on rated kW.
    ///
    /// In EXTERNAL mode, Storage element state is controlled by an external Storagecontroller2. This mode is automatically set if this Storage element is included in the element list of a StorageController element.
    ///
    /// For the other two dispatch modes, the Storage element state is controlled by either the global default Loadlevel value or the price level.
    ///
    /// DSS property name: DispMode, DSS property index: 41
    ///
    Storage& DispMode_str(const string &value)
    {
        set_string(Properties::DispMode, value);
        return *this;
    }

    /// Dispatch trigger value for discharging the Storage.
    /// If = 0.0 the Storage element state is changed by the State command or by a StorageController2 object.
    /// If <> 0  the Storage element state is set to DISCHARGING when this trigger level is EXCEEDED by either the specified Loadshape curve value or the price signal or global Loadlevel value, depending on dispatch mode. See State property.
    ///
    /// DSS property name: DischargeTrigger, DSS property index: 42
    ///
    double DischargeTrigger()
    {
        return Obj_GetFloat64(ptr, Properties::DischargeTrigger);
    }

    Storage& DischargeTrigger(double value)
    {
        Obj_SetFloat64(ptr, Properties::DischargeTrigger, value);
        return *this;
    }

    /// Dispatch trigger value for charging the Storage.
    ///
    /// If = 0.0 the Storage element state is changed by the State command or StorageController2 object.
    ///
    /// If <> 0  the Storage element state is set to CHARGING when this trigger level is GREATER than either the specified Loadshape curve value or the price signal or global Loadlevel value, depending on dispatch mode. See State property.
    ///
    /// DSS property name: ChargeTrigger, DSS property index: 43
    ///
    double ChargeTrigger()
    {
        return Obj_GetFloat64(ptr, Properties::ChargeTrigger);
    }

    Storage& ChargeTrigger(double value)
    {
        Obj_SetFloat64(ptr, Properties::ChargeTrigger, value);
        return *this;
    }

    /// Time of day in fractional hours (0230 = 2.5) at which Storage element will automatically go into charge state. Default is 2.0.  Enter a negative time value to disable this feature.
    ///
    /// DSS property name: TimeChargeTrig, DSS property index: 44
    ///
    double TimeChargeTrig()
    {
        return Obj_GetFloat64(ptr, Properties::TimeChargeTrig);
    }

    Storage& TimeChargeTrig(double value)
    {
        Obj_SetFloat64(ptr, Properties::TimeChargeTrig, value);
        return *this;
    }

    /// An arbitrary integer number representing the class of Storage element so that Storage values may be segregated by class.
    ///
    /// DSS property name: class, DSS property index: 45
    ///
    int32_t cls()
    {
        return Obj_GetInt32(ptr, Properties::cls);
    }

    Storage& cls(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::cls, value);
        return *this;
    }

    /// Name of DLL containing user-written dynamics model, which computes the terminal currents for Dynamics-mode simulations, overriding the default model.  Set to "none" to negate previous setting. This DLL has a simpler interface than the UserModel DLL and is only used for Dynamics mode.
    ///
    /// DSS property name: DynaDLL, DSS property index: 46
    ///
    string DynaDLL()
    {
        return get_prop_string(Properties::DynaDLL);
    }

    Storage& DynaDLL(const string &value)
    {
        set_string(Properties::DynaDLL, value);
        return *this;
    }

    Storage& DynaDLL(const char* value)
    {
        set_string(Properties::DynaDLL, value);
        return *this;
    }

    /// String (in quotes or parentheses if necessary) that gets passed to the user-written dynamics model Edit function for defining the data required for that model.
    ///
    /// DSS property name: DynaData, DSS property index: 47
    ///
    string DynaData()
    {
        return get_prop_string(Properties::DynaData);
    }

    Storage& DynaData(const string &value)
    {
        set_string(Properties::DynaData, value);
        return *this;
    }

    Storage& DynaData(const char* value)
    {
        set_string(Properties::DynaData, value);
        return *this;
    }

    /// Name of DLL containing user-written model, which computes the terminal currents for both power flow and dynamics, overriding the default model.  Set to "none" to negate previous setting.
    ///
    /// DSS property name: UserModel, DSS property index: 48
    ///
    string UserModel()
    {
        return get_prop_string(Properties::UserModel);
    }

    Storage& UserModel(const string &value)
    {
        set_string(Properties::UserModel, value);
        return *this;
    }

    Storage& UserModel(const char* value)
    {
        set_string(Properties::UserModel, value);
        return *this;
    }

    /// String (in quotes or parentheses) that gets passed to user-written model for defining the data required for that model.
    ///
    /// DSS property name: UserData, DSS property index: 49
    ///
    string UserData()
    {
        return get_prop_string(Properties::UserData);
    }

    Storage& UserData(const string &value)
    {
        set_string(Properties::UserData, value);
        return *this;
    }

    Storage& UserData(const char* value)
    {
        set_string(Properties::UserData, value);
        return *this;
    }

    /// {Yes | No }  Default is no.  Turn this on to capture the progress of the Storage model for each iteration.  Creates a separate file for each Storage element named "Storage_name.CSV".
    ///
    /// DSS property name: debugtrace, DSS property index: 50
    ///
    bool debugtrace()
    {
        return Obj_GetInt32(ptr, Properties::debugtrace) != 0;
    }

    Storage& debugtrace(bool value)
    {
        Obj_SetInt32(ptr, Properties::debugtrace, value);
        return *this;
    }

    /// Name of harmonic voltage or current spectrum for this Storage element. Current injection is assumed for inverter. Default value is "default", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 51
    ///
    string spectrum()
    {
        return get_prop_string(Properties::spectrum);
    }

    Storage& spectrum(const string &value)
    {
        set_string(Properties::spectrum, value);
        return *this;
    }

    Storage& spectrum(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Name of harmonic voltage or current spectrum for this Storage element. Current injection is assumed for inverter. Default value is "default", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 51
    ///
    dss::obj::Spectrum spectrum_obj()
    {
        return get_obj<dss::obj::Spectrum>(Properties::spectrum);
    }

    Storage& spectrum_obj(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 52
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Storage& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 53
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Storage& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 54
    ///
    Storage& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 54
    ///
    Storage& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class StorageController: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 29;
    struct Properties
    {
        enum {
            Element = 1,
            Terminal = 2,
            MonPhase = 3,
            kWTarget = 4,
            kWTargetLow = 5,
            pctkWBand = 6,
            kWBand = 7,
            pctkWBandLow = 8,
            kWBandLow = 9,
            ElementList = 10,
            Weights = 11,
            ModeDischarge = 12,
            ModeCharge = 13,
            TimeDischargeTrigger = 14,
            TimeChargeTrigger = 15,
            pctRatekW = 16,
            pctRateCharge = 17,
            pctReserve = 18,
            kWhTotal = 19,
            kWTotal = 20,
            kWhActual = 21,
            kWActual = 22,
            kWneed = 23,
            Yearly = 24,
            Daily = 25,
            Duty = 26,
            EventLog = 27,
            InhibitTime = 28,
            Tup = 29,
            TFlat = 30,
            Tdn = 31,
            kWThreshold = 32,
            DispFactor = 33,
            ResetLevel = 34,
            Seasons = 35,
            SeasonTargets = 36,
            SeasonTargetsLow = 37,
            basefreq = 38,
            enabled = 39,
            like = 40,
        };
    };

    // Class-specific enumerations

        ///
        /// StorageController: Discharge mode (DSS enumeration for StorageController)
        ///
    enum class StorageControllerDischargemode: int32_t
    {
        Peakshave = 5, ///< Peakshave
        Follow = 1, ///< Follow
        Support = 3, ///< Support
        Loadshape = 2, ///< Loadshape
        Time = 4, ///< Time
        Schedule = 6, ///< Schedule
        I_Peakshave = 8 ///< I-Peakshave
    };


        ///
        /// StorageController: Charge mode (DSS enumeration for StorageController)
        ///
    enum class StorageControllerChargemode: int32_t
    {
        Loadshape = 2, ///< Loadshape
        Time = 4, ///< Time
        PeakshaveLow = 7, ///< PeakshaveLow
        I_PeakshaveLow = 9 ///< I-PeakshaveLow
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    StorageController(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    StorageController(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the StorageController element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    StorageController(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the StorageController element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    StorageController& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    StorageController& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Full object name of the circuit element, typically a line or transformer, which the control is monitoring. There is no default; Must be specified.
    ///
    /// DSS property name: Element, DSS property index: 1
    ///
    string Element()
    {
        return get_prop_string(Properties::Element);
    }

    StorageController& Element(const string &value)
    {
        set_string(Properties::Element, value);
        return *this;
    }

    StorageController& Element(dss::obj::DSSObj &value)
    {
        set_obj(Properties::Element, value);
        return *this;
    }

    /// Full object name of the circuit element, typically a line or transformer, which the control is monitoring. There is no default; Must be specified.
    ///
    /// DSS property name: Element, DSS property index: 1
    ///
    dss::obj::DSSObj Element_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::Element);
    }

    StorageController& Element_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::Element, value);
        return *this;
    }

    /// Number of the terminal of the circuit element to which the StorageController2 control is connected. 1 or 2, typically.  Default is 1. Make sure to select the proper direction on the power for the respective dispatch mode.
    ///
    /// DSS property name: Terminal, DSS property index: 2
    ///
    int32_t Terminal()
    {
        return Obj_GetInt32(ptr, Properties::Terminal);
    }

    StorageController& Terminal(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Terminal, value);
        return *this;
    }

    /// Number of the phase being monitored or one of {AVG | MAX | MIN} for all phases. Default=MAX. Must be less than the number of phases. Used in PeakShave, Follow, Support and I-PeakShave discharging modes and in PeakShaveLow, I-PeakShaveLow charging modes. For modes based on active power measurements, the value used by the control is the monitored one multiplied by the number of phases of the monitored element.
    ///
    /// DSS property name: MonPhase, DSS property index: 3
    ///
    int32_t MonPhase()
    {
        return Obj_GetInt32(ptr, Properties::MonPhase);
    }

    StorageController& MonPhase(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::MonPhase, value);
        return *this;
    }

    StorageController& MonPhase(MonitoredPhase value)
    {
        Obj_SetInt32(ptr, Properties::MonPhase, int32_t(value));
        return *this;
    }

    StorageController& MonPhase(const string &value)
    {
        set_string(Properties::MonPhase, value);
        return *this;
    }

    StorageController& MonPhase(const char *value)
    {
        set_string(Properties::MonPhase, value);
        return *this;
    }

    /// Number of the phase being monitored or one of {AVG | MAX | MIN} for all phases. Default=MAX. Must be less than the number of phases. Used in PeakShave, Follow, Support and I-PeakShave discharging modes and in PeakShaveLow, I-PeakShaveLow charging modes. For modes based on active power measurements, the value used by the control is the monitored one multiplied by the number of phases of the monitored element.
    ///
    /// DSS property name: MonPhase, DSS property index: 3
    ///
    string MonPhase_str()
    {
        return get_prop_string(Properties::MonPhase);
    }

    /// Number of the phase being monitored or one of {AVG | MAX | MIN} for all phases. Default=MAX. Must be less than the number of phases. Used in PeakShave, Follow, Support and I-PeakShave discharging modes and in PeakShaveLow, I-PeakShaveLow charging modes. For modes based on active power measurements, the value used by the control is the monitored one multiplied by the number of phases of the monitored element.
    ///
    /// DSS property name: MonPhase, DSS property index: 3
    ///
    StorageController& MonPhase_str(const string &value)
    {
        set_string(Properties::MonPhase, value);
        return *this;
    }

    /// kW/kamps target for Discharging. The Storage element fleet is dispatched to try to hold the power/current in band at least until the Storage is depleted. The selection of power or current depends on the Discharge mode (PeakShave->kW, I-PeakShave->kamps).
    ///
    /// DSS property name: kWTarget, DSS property index: 4
    ///
    double kWTarget()
    {
        return Obj_GetFloat64(ptr, Properties::kWTarget);
    }

    StorageController& kWTarget(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWTarget, value);
        return *this;
    }

    /// kW/kamps target for Charging. The Storage element fleet is dispatched to try to hold the power/current in band at least until the Storage is fully charged. The selection of power or current depends on the charge mode (PeakShavelow->kW, I-PeakShavelow->kamps).
    ///
    /// DSS property name: kWTargetLow, DSS property index: 5
    ///
    double kWTargetLow()
    {
        return Obj_GetFloat64(ptr, Properties::kWTargetLow);
    }

    StorageController& kWTargetLow(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWTargetLow, value);
        return *this;
    }

    /// Bandwidth (% of Target kW/kamps) of the dead band around the kW/kamps target value. Default is 2% (+/-1%).No dispatch changes are attempted if the power in the monitored terminal stays within this band.
    ///
    /// DSS property name: %kWBand, DSS property index: 6
    ///
    double pctkWBand()
    {
        return Obj_GetFloat64(ptr, Properties::pctkWBand);
    }

    StorageController& pctkWBand(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctkWBand, value);
        return *this;
    }

    /// Alternative way of specifying the bandwidth. (kW/kamps) of the dead band around the kW/kamps target value. Default is 2% of kWTarget (+/-1%).No dispatch changes are attempted if the power in the monitored terminal stays within this band.
    ///
    /// DSS property name: kWBand, DSS property index: 7
    ///
    double kWBand()
    {
        return Obj_GetFloat64(ptr, Properties::kWBand);
    }

    StorageController& kWBand(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWBand, value);
        return *this;
    }

    /// Bandwidth (% of kWTargetLow) of the dead band around the kW/kamps low target value. Default is 2% (+/-1%).No charging is attempted if the power in the monitored terminal stays within this band.
    ///
    /// DSS property name: %kWBandLow, DSS property index: 8
    ///
    double pctkWBandLow()
    {
        return Obj_GetFloat64(ptr, Properties::pctkWBandLow);
    }

    StorageController& pctkWBandLow(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctkWBandLow, value);
        return *this;
    }

    /// Alternative way of specifying the bandwidth. (kW/kamps) of the dead band around the kW/kamps low target value. Default is 2% of kWTargetLow (+/-1%).No charging is attempted if the power in the monitored terminal stays within this band.
    ///
    /// DSS property name: kWBandLow, DSS property index: 9
    ///
    double kWBandLow()
    {
        return Obj_GetFloat64(ptr, Properties::kWBandLow);
    }

    StorageController& kWBandLow(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWBandLow, value);
        return *this;
    }

    /// Array list of Storage elements to be controlled.  If not specified, all Storage elements in the circuit not presently dispatched by another controller are assumed dispatched by this controller.
    ///
    /// DSS property name: ElementList, DSS property index: 10
    ///
    strings ElementList()
    {
        return get_array<strings>(Properties::ElementList);
    }

    StorageController& ElementList(strings &value)
    {
        set_array<strings>(Properties::ElementList, value);
        return *this;
    }

    /// Array of proportional weights corresponding to each Storage element in the ElementList. The needed kW or kvar to get back to center band is dispatched to each Storage element according to these weights. Default is to set all weights to 1.0.
    ///
    /// DSS property name: Weights, DSS property index: 11
    ///
    VectorXd Weights()
    {
        return get_array<VectorXd>(Properties::Weights);
    }

    StorageController& Weights(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Weights, value);
        return *this;
    }

    /// {PeakShave* | Follow | Support | Loadshape | Time | Schedule | I-PeakShave} Mode of operation for the DISCHARGE FUNCTION of this controller.
    ///
    /// In PeakShave mode (Default), the control attempts to discharge Storage to keep power in the monitored element below the kWTarget.
    ///
    /// In Follow mode, the control is triggered by time and resets the kWTarget value to the present monitored element power. It then attempts to discharge Storage to keep power in the monitored element below the new kWTarget. See TimeDischargeTrigger.
    ///
    /// In Support mode, the control operates oppositely of PeakShave mode: Storage is discharged to keep kW power output up near the target.
    ///
    /// In Loadshape mode, both charging and discharging precisely follows the per unit loadshape. Storage is discharged when the loadshape value is positive.
    ///
    /// In Time mode, the Storage discharge is turned on at the specified %RatekW at the specified discharge trigger time in fractional hours.
    ///
    /// In Schedule mode, the Tup, TFlat, and Tdn properties specify the up ramp duration, flat duration, and down ramp duration for the schedule. The schedule start time is set by TimeDischargeTrigger and the rate of discharge for the flat part is determined by %RatekW.
    ///
    /// In I-PeakShave mode, the control attempts to discharge Storage to keep current in the monitored element below the target given in k-amps (thousands of amps), when this control mode is active, the property kWTarget will be expressed in k-amps.
    ///
    /// DSS property name: ModeDischarge, DSS property index: 12
    ///
    StorageControllerDischargemode ModeDischarge()
    {
        return StorageControllerDischargemode(Obj_GetInt32(ptr, Properties::ModeDischarge));
    }

    StorageController& ModeDischarge(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::ModeDischarge, value);
        return *this;
    }

    StorageController& ModeDischarge(StorageControllerDischargemode value)
    {
        Obj_SetInt32(ptr, Properties::ModeDischarge, int32_t(value));
        return *this;
    }

    StorageController& ModeDischarge(const string &value)
    {
        set_string(Properties::ModeDischarge, value);
        return *this;
    }

    StorageController& ModeDischarge(const char *value)
    {
        set_string(Properties::ModeDischarge, value);
        return *this;
    }

    /// {PeakShave* | Follow | Support | Loadshape | Time | Schedule | I-PeakShave} Mode of operation for the DISCHARGE FUNCTION of this controller.
    ///
    /// In PeakShave mode (Default), the control attempts to discharge Storage to keep power in the monitored element below the kWTarget.
    ///
    /// In Follow mode, the control is triggered by time and resets the kWTarget value to the present monitored element power. It then attempts to discharge Storage to keep power in the monitored element below the new kWTarget. See TimeDischargeTrigger.
    ///
    /// In Support mode, the control operates oppositely of PeakShave mode: Storage is discharged to keep kW power output up near the target.
    ///
    /// In Loadshape mode, both charging and discharging precisely follows the per unit loadshape. Storage is discharged when the loadshape value is positive.
    ///
    /// In Time mode, the Storage discharge is turned on at the specified %RatekW at the specified discharge trigger time in fractional hours.
    ///
    /// In Schedule mode, the Tup, TFlat, and Tdn properties specify the up ramp duration, flat duration, and down ramp duration for the schedule. The schedule start time is set by TimeDischargeTrigger and the rate of discharge for the flat part is determined by %RatekW.
    ///
    /// In I-PeakShave mode, the control attempts to discharge Storage to keep current in the monitored element below the target given in k-amps (thousands of amps), when this control mode is active, the property kWTarget will be expressed in k-amps.
    ///
    /// DSS property name: ModeDischarge, DSS property index: 12
    ///
    string ModeDischarge_str()
    {
        return get_prop_string(Properties::ModeDischarge);
    }

    /// {PeakShave* | Follow | Support | Loadshape | Time | Schedule | I-PeakShave} Mode of operation for the DISCHARGE FUNCTION of this controller.
    ///
    /// In PeakShave mode (Default), the control attempts to discharge Storage to keep power in the monitored element below the kWTarget.
    ///
    /// In Follow mode, the control is triggered by time and resets the kWTarget value to the present monitored element power. It then attempts to discharge Storage to keep power in the monitored element below the new kWTarget. See TimeDischargeTrigger.
    ///
    /// In Support mode, the control operates oppositely of PeakShave mode: Storage is discharged to keep kW power output up near the target.
    ///
    /// In Loadshape mode, both charging and discharging precisely follows the per unit loadshape. Storage is discharged when the loadshape value is positive.
    ///
    /// In Time mode, the Storage discharge is turned on at the specified %RatekW at the specified discharge trigger time in fractional hours.
    ///
    /// In Schedule mode, the Tup, TFlat, and Tdn properties specify the up ramp duration, flat duration, and down ramp duration for the schedule. The schedule start time is set by TimeDischargeTrigger and the rate of discharge for the flat part is determined by %RatekW.
    ///
    /// In I-PeakShave mode, the control attempts to discharge Storage to keep current in the monitored element below the target given in k-amps (thousands of amps), when this control mode is active, the property kWTarget will be expressed in k-amps.
    ///
    /// DSS property name: ModeDischarge, DSS property index: 12
    ///
    StorageController& ModeDischarge_str(const string &value)
    {
        set_string(Properties::ModeDischarge, value);
        return *this;
    }

    /// {Loadshape | Time* | PeakShaveLow | I-PeakShaveLow} Mode of operation for the CHARGE FUNCTION of this controller.
    ///
    /// In Loadshape mode, both charging and discharging precisely follows the per unit loadshape. Storage is charged when the loadshape value is negative.
    ///
    /// In Time mode, the Storage charging FUNCTION is triggered at the specified %RateCharge at the specified charge trigger time in fractional hours.
    ///
    /// In PeakShaveLow mode, the charging operation will charge the Storage fleet when the power at amonitored element is below a specified KW target (kWTarget_low). The Storage will charge as much power as necessary to keep the power within the deadband around kWTarget_low.
    ///
    /// In I-PeakShaveLow mode, the charging operation will charge the Storage fleet when the current (Amps) at amonitored element is below a specified amps target (kWTarget_low). The Storage will charge as much power as necessary to keep the amps within the deadband around kWTarget_low. When this control mode is active, the property kWTarget_low will be expressed in k-amps and all the other parameters will be adjusted to match the amps (current) control criteria.
    ///
    /// DSS property name: ModeCharge, DSS property index: 13
    ///
    StorageControllerChargemode ModeCharge()
    {
        return StorageControllerChargemode(Obj_GetInt32(ptr, Properties::ModeCharge));
    }

    StorageController& ModeCharge(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::ModeCharge, value);
        return *this;
    }

    StorageController& ModeCharge(StorageControllerChargemode value)
    {
        Obj_SetInt32(ptr, Properties::ModeCharge, int32_t(value));
        return *this;
    }

    StorageController& ModeCharge(const string &value)
    {
        set_string(Properties::ModeCharge, value);
        return *this;
    }

    StorageController& ModeCharge(const char *value)
    {
        set_string(Properties::ModeCharge, value);
        return *this;
    }

    /// {Loadshape | Time* | PeakShaveLow | I-PeakShaveLow} Mode of operation for the CHARGE FUNCTION of this controller.
    ///
    /// In Loadshape mode, both charging and discharging precisely follows the per unit loadshape. Storage is charged when the loadshape value is negative.
    ///
    /// In Time mode, the Storage charging FUNCTION is triggered at the specified %RateCharge at the specified charge trigger time in fractional hours.
    ///
    /// In PeakShaveLow mode, the charging operation will charge the Storage fleet when the power at amonitored element is below a specified KW target (kWTarget_low). The Storage will charge as much power as necessary to keep the power within the deadband around kWTarget_low.
    ///
    /// In I-PeakShaveLow mode, the charging operation will charge the Storage fleet when the current (Amps) at amonitored element is below a specified amps target (kWTarget_low). The Storage will charge as much power as necessary to keep the amps within the deadband around kWTarget_low. When this control mode is active, the property kWTarget_low will be expressed in k-amps and all the other parameters will be adjusted to match the amps (current) control criteria.
    ///
    /// DSS property name: ModeCharge, DSS property index: 13
    ///
    string ModeCharge_str()
    {
        return get_prop_string(Properties::ModeCharge);
    }

    /// {Loadshape | Time* | PeakShaveLow | I-PeakShaveLow} Mode of operation for the CHARGE FUNCTION of this controller.
    ///
    /// In Loadshape mode, both charging and discharging precisely follows the per unit loadshape. Storage is charged when the loadshape value is negative.
    ///
    /// In Time mode, the Storage charging FUNCTION is triggered at the specified %RateCharge at the specified charge trigger time in fractional hours.
    ///
    /// In PeakShaveLow mode, the charging operation will charge the Storage fleet when the power at amonitored element is below a specified KW target (kWTarget_low). The Storage will charge as much power as necessary to keep the power within the deadband around kWTarget_low.
    ///
    /// In I-PeakShaveLow mode, the charging operation will charge the Storage fleet when the current (Amps) at amonitored element is below a specified amps target (kWTarget_low). The Storage will charge as much power as necessary to keep the amps within the deadband around kWTarget_low. When this control mode is active, the property kWTarget_low will be expressed in k-amps and all the other parameters will be adjusted to match the amps (current) control criteria.
    ///
    /// DSS property name: ModeCharge, DSS property index: 13
    ///
    StorageController& ModeCharge_str(const string &value)
    {
        set_string(Properties::ModeCharge, value);
        return *this;
    }

    /// Default time of day (hr) for initiating Discharging of the fleet. During Follow or Time mode discharging is triggered at a fixed time each day at this hour. If Follow mode, Storage will be discharged to attempt to hold the load at or below the power level at the time of triggering. In Time mode, the discharge is based on the %RatekW property value. Set this to a negative value to ignore. Default is 12.0 for Follow mode; otherwise it is -1 (ignored).
    ///
    /// DSS property name: TimeDischargeTrigger, DSS property index: 14
    ///
    double TimeDischargeTrigger()
    {
        return Obj_GetFloat64(ptr, Properties::TimeDischargeTrigger);
    }

    StorageController& TimeDischargeTrigger(double value)
    {
        Obj_SetFloat64(ptr, Properties::TimeDischargeTrigger, value);
        return *this;
    }

    /// Default time of day (hr) for initiating charging in Time control mode. Set this to a negative value to ignore. Default is 2.0.  (0200).When this value is >0 the Storage fleet is set to charging at this time regardless of other control criteria to make sure Storage is topped off for the next discharge cycle.
    ///
    /// DSS property name: TimeChargeTrigger, DSS property index: 15
    ///
    double TimeChargeTrigger()
    {
        return Obj_GetFloat64(ptr, Properties::TimeChargeTrigger);
    }

    StorageController& TimeChargeTrigger(double value)
    {
        Obj_SetFloat64(ptr, Properties::TimeChargeTrigger, value);
        return *this;
    }

    /// Sets the kW discharge rate in % of rated capacity for each element of the fleet. Applies to TIME control mode, SCHEDULE mode, or anytime discharging is triggered by time.
    ///
    /// DSS property name: %RatekW, DSS property index: 16
    ///
    double pctRatekW()
    {
        return Obj_GetFloat64(ptr, Properties::pctRatekW);
    }

    StorageController& pctRatekW(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctRatekW, value);
        return *this;
    }

    /// Sets the kW charging rate in % of rated capacity for each element of the fleet. Applies to TIME control mode and anytime charging mode is entered due to a time trigger.
    ///
    /// DSS property name: %RateCharge, DSS property index: 17
    ///
    double pctRateCharge()
    {
        return Obj_GetFloat64(ptr, Properties::pctRateCharge);
    }

    StorageController& pctRateCharge(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctRateCharge, value);
        return *this;
    }

    /// Use this property to change the % reserve for each Storage element under control of this controller. This might be used, for example, to allow deeper discharges of Storage or in case of emergency operation to use the remainder of the Storage element.
    ///
    /// DSS property name: %Reserve, DSS property index: 18
    ///
    double pctReserve()
    {
        return Obj_GetFloat64(ptr, Properties::pctReserve);
    }

    StorageController& pctReserve(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctReserve, value);
        return *this;
    }

    /// (Read only). Total rated kWh energy Storage capacity of Storage elements controlled by this controller.
    ///
    /// DSS property name: kWhTotal, DSS property index: 19
    ///
    double kWhTotal()
    {
        return Obj_GetFloat64(ptr, Properties::kWhTotal);
    }

    StorageController& kWhTotal(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWhTotal, value);
        return *this;
    }

    /// (Read only). Total rated kW power capacity of Storage elements controlled by this controller.
    ///
    /// DSS property name: kWTotal, DSS property index: 20
    ///
    double kWTotal()
    {
        return Obj_GetFloat64(ptr, Properties::kWTotal);
    }

    StorageController& kWTotal(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWTotal, value);
        return *this;
    }

    /// (Read only). Actual kWh stored of all controlled Storage elements.
    ///
    /// DSS property name: kWhActual, DSS property index: 21
    ///
    double kWhActual()
    {
        return Obj_GetFloat64(ptr, Properties::kWhActual);
    }

    StorageController& kWhActual(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWhActual, value);
        return *this;
    }

    /// (Read only). Actual kW output of all controlled Storage elements.
    ///
    /// DSS property name: kWActual, DSS property index: 22
    ///
    double kWActual()
    {
        return Obj_GetFloat64(ptr, Properties::kWActual);
    }

    StorageController& kWActual(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWActual, value);
        return *this;
    }

    /// (Read only). KW needed to meet target.
    ///
    /// DSS property name: kWneed, DSS property index: 23
    ///
    double kWneed()
    {
        return Obj_GetFloat64(ptr, Properties::kWneed);
    }

    StorageController& kWneed(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWneed, value);
        return *this;
    }

    /// Dispatch loadshape object, If any, for Yearly solution Mode.
    ///
    /// DSS property name: Yearly, DSS property index: 24
    ///
    string Yearly()
    {
        return get_prop_string(Properties::Yearly);
    }

    StorageController& Yearly(const string &value)
    {
        set_string(Properties::Yearly, value);
        return *this;
    }

    StorageController& Yearly(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Yearly, value);
        return *this;
    }

    /// Dispatch loadshape object, If any, for Yearly solution Mode.
    ///
    /// DSS property name: Yearly, DSS property index: 24
    ///
    dss::obj::LoadShape Yearly_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::Yearly);
    }

    StorageController& Yearly_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Yearly, value);
        return *this;
    }

    /// Dispatch loadshape object, If any, for Daily solution mode.
    ///
    /// DSS property name: Daily, DSS property index: 25
    ///
    string Daily()
    {
        return get_prop_string(Properties::Daily);
    }

    StorageController& Daily(const string &value)
    {
        set_string(Properties::Daily, value);
        return *this;
    }

    StorageController& Daily(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Daily, value);
        return *this;
    }

    /// Dispatch loadshape object, If any, for Daily solution mode.
    ///
    /// DSS property name: Daily, DSS property index: 25
    ///
    dss::obj::LoadShape Daily_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::Daily);
    }

    StorageController& Daily_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Daily, value);
        return *this;
    }

    /// Dispatch loadshape object, If any, for Dutycycle solution mode.
    ///
    /// DSS property name: Duty, DSS property index: 26
    ///
    string Duty()
    {
        return get_prop_string(Properties::Duty);
    }

    StorageController& Duty(const string &value)
    {
        set_string(Properties::Duty, value);
        return *this;
    }

    StorageController& Duty(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Duty, value);
        return *this;
    }

    /// Dispatch loadshape object, If any, for Dutycycle solution mode.
    ///
    /// DSS property name: Duty, DSS property index: 26
    ///
    dss::obj::LoadShape Duty_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::Duty);
    }

    StorageController& Duty_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Duty, value);
        return *this;
    }

    /// {Yes/True | No/False} Default is No. Log control actions to Eventlog.
    ///
    /// DSS property name: EventLog, DSS property index: 27
    ///
    bool EventLog()
    {
        return Obj_GetInt32(ptr, Properties::EventLog) != 0;
    }

    StorageController& EventLog(bool value)
    {
        Obj_SetInt32(ptr, Properties::EventLog, value);
        return *this;
    }

    /// Hours (integer) to inhibit Discharging after going into Charge mode. Default is 5.
    ///
    /// DSS property name: InhibitTime, DSS property index: 28
    ///
    int32_t InhibitTime()
    {
        return Obj_GetInt32(ptr, Properties::InhibitTime);
    }

    StorageController& InhibitTime(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::InhibitTime, value);
        return *this;
    }

    /// Duration, hrs, of upramp part for SCHEDULE mode. Default is 0.25.
    ///
    /// DSS property name: Tup, DSS property index: 29
    ///
    double Tup()
    {
        return Obj_GetFloat64(ptr, Properties::Tup);
    }

    StorageController& Tup(double value)
    {
        Obj_SetFloat64(ptr, Properties::Tup, value);
        return *this;
    }

    /// Duration, hrs, of flat part for SCHEDULE mode. Default is 2.0.
    ///
    /// DSS property name: TFlat, DSS property index: 30
    ///
    double TFlat()
    {
        return Obj_GetFloat64(ptr, Properties::TFlat);
    }

    StorageController& TFlat(double value)
    {
        Obj_SetFloat64(ptr, Properties::TFlat, value);
        return *this;
    }

    /// Duration, hrs, of downramp part for SCHEDULE mode. Default is 0.25.
    ///
    /// DSS property name: Tdn, DSS property index: 31
    ///
    double Tdn()
    {
        return Obj_GetFloat64(ptr, Properties::Tdn);
    }

    StorageController& Tdn(double value)
    {
        Obj_SetFloat64(ptr, Properties::Tdn, value);
        return *this;
    }

    /// Threshold, kW, for Follow mode. kW has to be above this value for the Storage element to be dispatched on. Defaults to 75% of the kWTarget value. Must reset this property after setting kWTarget if you want a different value.
    ///
    /// DSS property name: kWThreshold, DSS property index: 32
    ///
    double kWThreshold()
    {
        return Obj_GetFloat64(ptr, Properties::kWThreshold);
    }

    StorageController& kWThreshold(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWThreshold, value);
        return *this;
    }

    /// Defaults to 1 (disabled). Set to any value between 0 and 1 to enable this parameter.
    ///
    /// Use this parameter to reduce the amount of power requested by the controller in each control iteration. It can be useful when maximum control iterations are exceeded due to numerical instability such as fleet being set to charging and idling in subsequent control iterations (check the Eventlog).
    ///
    /// DSS property name: DispFactor, DSS property index: 33
    ///
    double DispFactor()
    {
        return Obj_GetFloat64(ptr, Properties::DispFactor);
    }

    StorageController& DispFactor(double value)
    {
        Obj_SetFloat64(ptr, Properties::DispFactor, value);
        return *this;
    }

    /// The level of charge required for allowing the storage to discharge again after reaching the reserve storage level. After reaching this level, the storage control  will not allow the storage device to discharge, forcing the storage to charge. Once the storage reaches thislevel, the storage will be able to discharge again. This value is a number between 0.2 and 1
    ///
    /// DSS property name: ResetLevel, DSS property index: 34
    ///
    double ResetLevel()
    {
        return Obj_GetFloat64(ptr, Properties::ResetLevel);
    }

    StorageController& ResetLevel(double value)
    {
        Obj_SetFloat64(ptr, Properties::ResetLevel, value);
        return *this;
    }

    /// With this property the user can specify the number of targets to be used by the controller using the list given at "SeasonTargets"/"SeasonTargetsLow", which can be used to dynamically adjust the storage controller during a QSTS simulation. The default value is 1. This property needs to be defined before defining SeasonTargets/SeasonTargetsLow.
    ///
    /// DSS property name: Seasons, DSS property index: 35
    ///
    int32_t Seasons()
    {
        return Obj_GetInt32(ptr, Properties::Seasons);
    }

    StorageController& Seasons(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Seasons, value);
        return *this;
    }

    /// An array of doubles specifying the targets to be used during a QSTS simulation. These targets will take effect only if SeasonRating=true. The number of targets cannot exceed the number of seasons defined at the SeasonSignal.The difference between the targets defined at SeasonTargets and SeasonTargetsLow is that SeasonTargets applies to discharging modes, while SeasonTargetsLow applies to charging modes.
    ///
    /// DSS property name: SeasonTargets, DSS property index: 36
    ///
    VectorXd SeasonTargets()
    {
        return get_array<VectorXd>(Properties::SeasonTargets);
    }

    StorageController& SeasonTargets(VectorXd &value)
    {
        set_array<VectorXd>(Properties::SeasonTargets, value);
        return *this;
    }

    /// An array of doubles specifying the targets to be used during a QSTS simulation. These targets will take effect only if SeasonRating=true. The number of targets cannot exceed the number of seasons defined at the SeasonSignal.The difference between the targets defined at SeasonTargets and SeasonTargetsLow is that SeasonTargets applies to discharging modes, while SeasonTargetsLow applies to charging modes.
    ///
    /// DSS property name: SeasonTargetsLow, DSS property index: 37
    ///
    VectorXd SeasonTargetsLow()
    {
        return get_array<VectorXd>(Properties::SeasonTargetsLow);
    }

    StorageController& SeasonTargetsLow(VectorXd &value)
    {
        set_array<VectorXd>(Properties::SeasonTargetsLow, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 38
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    StorageController& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 39
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    StorageController& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 40
    ///
    StorageController& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 40
    ///
    StorageController& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Relay: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 30;
    struct Properties
    {
        enum {
            MonitoredObj = 1,
            MonitoredTerm = 2,
            SwitchedObj = 3,
            SwitchedTerm = 4,
            type = 5,
            Phasecurve = 6,
            Groundcurve = 7,
            PhaseTrip = 8,
            GroundTrip = 9,
            TDPhase = 10,
            TDGround = 11,
            PhaseInst = 12,
            GroundInst = 13,
            Reset = 14,
            Shots = 15,
            RecloseIntervals = 16,
            Delay = 17,
            Overvoltcurve = 18,
            Undervoltcurve = 19,
            kvbase = 20,
            pctPickup47 = 21,
            BaseAmps46 = 22,
            pctPickup46 = 23,
            isqt46 = 24,
            Variable = 25,
            overtrip = 26,
            undertrip = 27,
            Breakertime = 28,
            action = 29,
            Z1mag = 30,
            Z1ang = 31,
            Z0mag = 32,
            Z0ang = 33,
            Mphase = 34,
            Mground = 35,
            EventLog = 36,
            DebugTrace = 37,
            DistReverse = 38,
            Normal = 39,
            State = 40,
            DOC_TiltAngleLow = 41,
            DOC_TiltAngleHigh = 42,
            DOC_TripSettingLow = 43,
            DOC_TripSettingHigh = 44,
            DOC_TripSettingMag = 45,
            DOC_DelayInner = 46,
            DOC_PhaseCurveInner = 47,
            DOC_PhaseTripInner = 48,
            DOC_TDPhaseInner = 49,
            basefreq = 50,
            enabled = 51,
            like = 52,
        };
    };

    // Class-specific enumerations

        ///
        /// Relay: Type (DSS enumeration for Relay)
        ///
    enum class RelayType: int32_t
    {
        Current = 0, ///< Current
        Voltage = 1, ///< Voltage
        ReversePower = 3, ///< ReversePower
        relay46 = 4, ///< 46
        relay47 = 5, ///< 47
        Generic = 6, ///< Generic
        Distance = 7, ///< Distance
        TD21 = 8, ///< TD21
        DOC = 9 ///< DOC
    };


        ///
        /// Relay: Action (DSS enumeration for Relay)
        ///
    enum class RelayAction: int32_t
    {
        close = 2, ///< close
        open = 1, ///< open
        trip = 1 ///< trip
    };


        ///
        /// Relay: State (DSS enumeration for Relay)
        ///
    enum class RelayState: int32_t
    {
        closed = 2, ///< closed
        open = 1, ///< open
        trip = 1 ///< trip
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Relay(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Relay(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Relay element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Relay(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Relay element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Relay& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Relay& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Full object name of the circuit element, typically a line, transformer, load, or generator, to which the relay's PT and/or CT are connected. This is the "monitored" element. There is no default; must be specified.
    ///
    /// DSS property name: MonitoredObj, DSS property index: 1
    ///
    string MonitoredObj()
    {
        return get_prop_string(Properties::MonitoredObj);
    }

    Relay& MonitoredObj(const string &value)
    {
        set_string(Properties::MonitoredObj, value);
        return *this;
    }

    Relay& MonitoredObj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::MonitoredObj, value);
        return *this;
    }

    /// Full object name of the circuit element, typically a line, transformer, load, or generator, to which the relay's PT and/or CT are connected. This is the "monitored" element. There is no default; must be specified.
    ///
    /// DSS property name: MonitoredObj, DSS property index: 1
    ///
    dss::obj::DSSObj MonitoredObj_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::MonitoredObj);
    }

    Relay& MonitoredObj_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::MonitoredObj, value);
        return *this;
    }

    /// Number of the terminal of the circuit element to which the Relay is connected. 1 or 2, typically.  Default is 1.
    ///
    /// DSS property name: MonitoredTerm, DSS property index: 2
    ///
    int32_t MonitoredTerm()
    {
        return Obj_GetInt32(ptr, Properties::MonitoredTerm);
    }

    Relay& MonitoredTerm(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::MonitoredTerm, value);
        return *this;
    }

    /// Name of circuit element switch that the Relay controls. Specify the full object name.Defaults to the same as the Monitored element. This is the "controlled" element.
    ///
    /// DSS property name: SwitchedObj, DSS property index: 3
    ///
    string SwitchedObj()
    {
        return get_prop_string(Properties::SwitchedObj);
    }

    Relay& SwitchedObj(const string &value)
    {
        set_string(Properties::SwitchedObj, value);
        return *this;
    }

    Relay& SwitchedObj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::SwitchedObj, value);
        return *this;
    }

    /// Name of circuit element switch that the Relay controls. Specify the full object name.Defaults to the same as the Monitored element. This is the "controlled" element.
    ///
    /// DSS property name: SwitchedObj, DSS property index: 3
    ///
    dss::obj::DSSObj SwitchedObj_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::SwitchedObj);
    }

    Relay& SwitchedObj_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::SwitchedObj, value);
        return *this;
    }

    /// Number of the terminal of the controlled element in which the switch is controlled by the Relay. 1 or 2, typically.  Default is 1.
    ///
    /// DSS property name: SwitchedTerm, DSS property index: 4
    ///
    int32_t SwitchedTerm()
    {
        return Obj_GetInt32(ptr, Properties::SwitchedTerm);
    }

    Relay& SwitchedTerm(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::SwitchedTerm, value);
        return *this;
    }

    /// One of a legal relay type:
    ///   Current
    ///   Voltage
    ///   Reversepower
    ///   46 (neg seq current)
    ///   47 (neg seq voltage)
    ///   Generic (generic over/under relay)
    ///   Distance
    ///   TD21
    ///   DOC (directional overcurrent)
    ///
    /// Default is overcurrent relay (Current) Specify the curve and pickup settings appropriate for each type. Generic relays monitor PC Element Control variables and trip on out of over/under range in definite time.
    ///
    /// DSS property name: type, DSS property index: 5
    ///
    RelayType type()
    {
        return RelayType(Obj_GetInt32(ptr, Properties::type));
    }

    Relay& type(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::type, value);
        return *this;
    }

    Relay& type(RelayType value)
    {
        Obj_SetInt32(ptr, Properties::type, int32_t(value));
        return *this;
    }

    Relay& type(const string &value)
    {
        set_string(Properties::type, value);
        return *this;
    }

    Relay& type(const char *value)
    {
        set_string(Properties::type, value);
        return *this;
    }

    /// One of a legal relay type:
    ///   Current
    ///   Voltage
    ///   Reversepower
    ///   46 (neg seq current)
    ///   47 (neg seq voltage)
    ///   Generic (generic over/under relay)
    ///   Distance
    ///   TD21
    ///   DOC (directional overcurrent)
    ///
    /// Default is overcurrent relay (Current) Specify the curve and pickup settings appropriate for each type. Generic relays monitor PC Element Control variables and trip on out of over/under range in definite time.
    ///
    /// DSS property name: type, DSS property index: 5
    ///
    string type_str()
    {
        return get_prop_string(Properties::type);
    }

    /// One of a legal relay type:
    ///   Current
    ///   Voltage
    ///   Reversepower
    ///   46 (neg seq current)
    ///   47 (neg seq voltage)
    ///   Generic (generic over/under relay)
    ///   Distance
    ///   TD21
    ///   DOC (directional overcurrent)
    ///
    /// Default is overcurrent relay (Current) Specify the curve and pickup settings appropriate for each type. Generic relays monitor PC Element Control variables and trip on out of over/under range in definite time.
    ///
    /// DSS property name: type, DSS property index: 5
    ///
    Relay& type_str(const string &value)
    {
        set_string(Properties::type, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the phase trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored). For overcurrent relay, multiplying the current values in the curve by the "phasetrip" value gives the actual current.
    ///
    /// DSS property name: Phasecurve, DSS property index: 6
    ///
    string Phasecurve()
    {
        return get_prop_string(Properties::Phasecurve);
    }

    Relay& Phasecurve(const string &value)
    {
        set_string(Properties::Phasecurve, value);
        return *this;
    }

    Relay& Phasecurve(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::Phasecurve, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the phase trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored). For overcurrent relay, multiplying the current values in the curve by the "phasetrip" value gives the actual current.
    ///
    /// DSS property name: Phasecurve, DSS property index: 6
    ///
    dss::obj::TCC_Curve Phasecurve_obj()
    {
        return get_obj<dss::obj::TCC_Curve>(Properties::Phasecurve);
    }

    Relay& Phasecurve_obj(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::Phasecurve, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the ground trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored).For overcurrent relay, multiplying the current values in the curve by the "groundtrip" valuw gives the actual current.
    ///
    /// DSS property name: Groundcurve, DSS property index: 7
    ///
    string Groundcurve()
    {
        return get_prop_string(Properties::Groundcurve);
    }

    Relay& Groundcurve(const string &value)
    {
        set_string(Properties::Groundcurve, value);
        return *this;
    }

    Relay& Groundcurve(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::Groundcurve, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the ground trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored).For overcurrent relay, multiplying the current values in the curve by the "groundtrip" valuw gives the actual current.
    ///
    /// DSS property name: Groundcurve, DSS property index: 7
    ///
    dss::obj::TCC_Curve Groundcurve_obj()
    {
        return get_obj<dss::obj::TCC_Curve>(Properties::Groundcurve);
    }

    Relay& Groundcurve_obj(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::Groundcurve, value);
        return *this;
    }

    /// Multiplier or actual phase amps for the phase TCC curve.  Defaults to 1.0.
    ///
    /// DSS property name: PhaseTrip, DSS property index: 8
    ///
    double PhaseTrip()
    {
        return Obj_GetFloat64(ptr, Properties::PhaseTrip);
    }

    Relay& PhaseTrip(double value)
    {
        Obj_SetFloat64(ptr, Properties::PhaseTrip, value);
        return *this;
    }

    /// Multiplier or actual ground amps (3I0) for the ground TCC curve.  Defaults to 1.0.
    ///
    /// DSS property name: GroundTrip, DSS property index: 9
    ///
    double GroundTrip()
    {
        return Obj_GetFloat64(ptr, Properties::GroundTrip);
    }

    Relay& GroundTrip(double value)
    {
        Obj_SetFloat64(ptr, Properties::GroundTrip, value);
        return *this;
    }

    /// Time dial for Phase trip curve. Multiplier on time axis of specified curve. Default=1.0.
    ///
    /// DSS property name: TDPhase, DSS property index: 10
    ///
    double TDPhase()
    {
        return Obj_GetFloat64(ptr, Properties::TDPhase);
    }

    Relay& TDPhase(double value)
    {
        Obj_SetFloat64(ptr, Properties::TDPhase, value);
        return *this;
    }

    /// Time dial for Ground trip curve. Multiplier on time axis of specified curve. Default=1.0.
    ///
    /// DSS property name: TDGround, DSS property index: 11
    ///
    double TDGround()
    {
        return Obj_GetFloat64(ptr, Properties::TDGround);
    }

    Relay& TDGround(double value)
    {
        Obj_SetFloat64(ptr, Properties::TDGround, value);
        return *this;
    }

    /// Actual  amps (Current relay) or kW (reverse power relay) for instantaneous phase trip which is assumed to happen in 0.01 sec + Delay Time. Default is 0.0, which signifies no inst trip. Use this value for specifying the Reverse Power threshold (kW) for reverse power relays.
    ///
    /// DSS property name: PhaseInst, DSS property index: 12
    ///
    double PhaseInst()
    {
        return Obj_GetFloat64(ptr, Properties::PhaseInst);
    }

    Relay& PhaseInst(double value)
    {
        Obj_SetFloat64(ptr, Properties::PhaseInst, value);
        return *this;
    }

    /// Actual  amps for instantaneous ground trip which is assumed to happen in 0.01 sec + Delay Time.Default is 0.0, which signifies no inst trip.
    ///
    /// DSS property name: GroundInst, DSS property index: 13
    ///
    double GroundInst()
    {
        return Obj_GetFloat64(ptr, Properties::GroundInst);
    }

    Relay& GroundInst(double value)
    {
        Obj_SetFloat64(ptr, Properties::GroundInst, value);
        return *this;
    }

    /// Reset time in sec for relay.  Default is 15. If this much time passes between the last pickup event, and the relay has not locked out, the operation counter resets.
    ///
    /// DSS property name: Reset, DSS property index: 14
    ///
    double Reset()
    {
        return Obj_GetFloat64(ptr, Properties::Reset);
    }

    Relay& Reset(double value)
    {
        Obj_SetFloat64(ptr, Properties::Reset, value);
        return *this;
    }

    /// Number of shots to lockout.  Default is 4. This is one more than the number of reclose intervals.
    ///
    /// DSS property name: Shots, DSS property index: 15
    ///
    int32_t Shots()
    {
        return Obj_GetInt32(ptr, Properties::Shots);
    }

    Relay& Shots(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Shots, value);
        return *this;
    }

    /// Array of reclose intervals. If none, specify "NONE". Default for overcurrent relay is (0.5, 2.0, 2.0) seconds. Default for a voltage relay is (5.0). In a voltage relay, this is  seconds after restoration of voltage that the reclose occurs. Reverse power relay is one shot to lockout, so this is ignored.  A locked out relay must be closed manually (set action=close).
    ///
    /// DSS property name: RecloseIntervals, DSS property index: 16
    ///
    VectorXd RecloseIntervals()
    {
        return get_array<VectorXd>(Properties::RecloseIntervals);
    }

    Relay& RecloseIntervals(VectorXd &value)
    {
        set_array<VectorXd>(Properties::RecloseIntervals, value);
        return *this;
    }

    /// Trip time delay (sec) for DEFINITE TIME relays. Default is 0.0 for current and voltage relays.  If >0 then this value is used instead of curves.  Used by Generic, RevPower, 46 and 47 relays. Defaults to 0.1 s for these relays.
    ///
    /// DSS property name: Delay, DSS property index: 17
    ///
    double Delay()
    {
        return Obj_GetFloat64(ptr, Properties::Delay);
    }

    Relay& Delay(double value)
    {
        Obj_SetFloat64(ptr, Properties::Delay, value);
        return *this;
    }

    /// TCC Curve object to use for overvoltage relay.  Curve is assumed to be defined with per unit voltage values. Voltage base should be defined for the relay. Default is none (ignored).
    ///
    /// DSS property name: Overvoltcurve, DSS property index: 18
    ///
    string Overvoltcurve()
    {
        return get_prop_string(Properties::Overvoltcurve);
    }

    Relay& Overvoltcurve(const string &value)
    {
        set_string(Properties::Overvoltcurve, value);
        return *this;
    }

    Relay& Overvoltcurve(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::Overvoltcurve, value);
        return *this;
    }

    /// TCC Curve object to use for overvoltage relay.  Curve is assumed to be defined with per unit voltage values. Voltage base should be defined for the relay. Default is none (ignored).
    ///
    /// DSS property name: Overvoltcurve, DSS property index: 18
    ///
    dss::obj::TCC_Curve Overvoltcurve_obj()
    {
        return get_obj<dss::obj::TCC_Curve>(Properties::Overvoltcurve);
    }

    Relay& Overvoltcurve_obj(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::Overvoltcurve, value);
        return *this;
    }

    /// TCC Curve object to use for undervoltage relay.  Curve is assumed to be defined with per unit voltage values. Voltage base should be defined for the relay. Default is none (ignored).
    ///
    /// DSS property name: Undervoltcurve, DSS property index: 19
    ///
    string Undervoltcurve()
    {
        return get_prop_string(Properties::Undervoltcurve);
    }

    Relay& Undervoltcurve(const string &value)
    {
        set_string(Properties::Undervoltcurve, value);
        return *this;
    }

    Relay& Undervoltcurve(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::Undervoltcurve, value);
        return *this;
    }

    /// TCC Curve object to use for undervoltage relay.  Curve is assumed to be defined with per unit voltage values. Voltage base should be defined for the relay. Default is none (ignored).
    ///
    /// DSS property name: Undervoltcurve, DSS property index: 19
    ///
    dss::obj::TCC_Curve Undervoltcurve_obj()
    {
        return get_obj<dss::obj::TCC_Curve>(Properties::Undervoltcurve);
    }

    Relay& Undervoltcurve_obj(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::Undervoltcurve, value);
        return *this;
    }

    /// Voltage base (kV) for the relay. Specify line-line for 3 phase devices); line-neutral for 1-phase devices.  Relay assumes the number of phases of the monitored element.  Default is 0.0, which results in assuming the voltage values in the "TCC" curve are specified in actual line-to-neutral volts.
    ///
    /// DSS property name: kvbase, DSS property index: 20
    ///
    double kvbase()
    {
        return Obj_GetFloat64(ptr, Properties::kvbase);
    }

    Relay& kvbase(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvbase, value);
        return *this;
    }

    /// Percent voltage pickup for 47 relay (Neg seq voltage). Default is 2. Specify also base voltage (kvbase) and delay time value.
    ///
    /// DSS property name: 47%Pickup, DSS property index: 21
    ///
    double pctPickup47()
    {
        return Obj_GetFloat64(ptr, Properties::pctPickup47);
    }

    Relay& pctPickup47(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctPickup47, value);
        return *this;
    }

    /// Base current, Amps, for 46 relay (neg seq current).  Used for establishing pickup and per unit I-squared-t.
    ///
    /// DSS property name: 46BaseAmps, DSS property index: 22
    ///
    double BaseAmps46()
    {
        return Obj_GetFloat64(ptr, Properties::BaseAmps46);
    }

    Relay& BaseAmps46(double value)
    {
        Obj_SetFloat64(ptr, Properties::BaseAmps46, value);
        return *this;
    }

    /// Percent pickup current for 46 relay (neg seq current).  Default is 20.0.   When current exceeds this value * BaseAmps, I-squared-t calc starts.
    ///
    /// DSS property name: 46%Pickup, DSS property index: 23
    ///
    double pctPickup46()
    {
        return Obj_GetFloat64(ptr, Properties::pctPickup46);
    }

    Relay& pctPickup46(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctPickup46, value);
        return *this;
    }

    /// Negative Sequence I-squared-t trip value for 46 relay (neg seq current).  Default is 1 (trips in 1 sec for 1 per unit neg seq current).  Should be 1 to 99.
    ///
    /// DSS property name: 46isqt, DSS property index: 24
    ///
    double isqt46()
    {
        return Obj_GetFloat64(ptr, Properties::isqt46);
    }

    Relay& isqt46(double value)
    {
        Obj_SetFloat64(ptr, Properties::isqt46, value);
        return *this;
    }

    /// Name of variable in PC Elements being monitored.  Only applies to Generic relay.
    ///
    /// DSS property name: Variable, DSS property index: 25
    ///
    string Variable()
    {
        return get_prop_string(Properties::Variable);
    }

    Relay& Variable(const string &value)
    {
        set_string(Properties::Variable, value);
        return *this;
    }

    Relay& Variable(const char* value)
    {
        set_string(Properties::Variable, value);
        return *this;
    }

    /// Trip setting (high value) for Generic relay variable.  Relay trips in definite time if value of variable exceeds this value.
    ///
    /// DSS property name: overtrip, DSS property index: 26
    ///
    double overtrip()
    {
        return Obj_GetFloat64(ptr, Properties::overtrip);
    }

    Relay& overtrip(double value)
    {
        Obj_SetFloat64(ptr, Properties::overtrip, value);
        return *this;
    }

    /// Trip setting (low value) for Generic relay variable.  Relay trips in definite time if value of variable is less than this value.
    ///
    /// DSS property name: undertrip, DSS property index: 27
    ///
    double undertrip()
    {
        return Obj_GetFloat64(ptr, Properties::undertrip);
    }

    Relay& undertrip(double value)
    {
        Obj_SetFloat64(ptr, Properties::undertrip, value);
        return *this;
    }

    /// Fixed delay time (sec) added to relay time. Default is 0.0. Designed to represent breaker time or some other delay after a trip decision is made.Use Delay property for setting a fixed trip time delay.Added to trip time of current and voltage relays. Could use in combination with inst trip value to obtain a definite time overcurrent relay.
    ///
    /// DSS property name: Breakertime, DSS property index: 28
    ///
    double Breakertime()
    {
        return Obj_GetFloat64(ptr, Properties::Breakertime);
    }

    Relay& Breakertime(double value)
    {
        Obj_SetFloat64(ptr, Properties::Breakertime, value);
        return *this;
    }

    /// DEPRECATED. See "State" property
    ///
    /// DSS property name: action, DSS property index: 29
    ///
    RelayAction action()
    {
        return RelayAction(Obj_GetInt32(ptr, Properties::action));
    }

    Relay& action(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::action, value);
        return *this;
    }

    Relay& action(RelayAction value)
    {
        Obj_SetInt32(ptr, Properties::action, int32_t(value));
        return *this;
    }

    Relay& action(const string &value)
    {
        set_string(Properties::action, value);
        return *this;
    }

    Relay& action(const char *value)
    {
        set_string(Properties::action, value);
        return *this;
    }

    /// DEPRECATED. See "State" property
    ///
    /// DSS property name: action, DSS property index: 29
    ///
    string action_str()
    {
        return get_prop_string(Properties::action);
    }

    /// DEPRECATED. See "State" property
    ///
    /// DSS property name: action, DSS property index: 29
    ///
    Relay& action_str(const string &value)
    {
        set_string(Properties::action, value);
        return *this;
    }

    /// Positive sequence reach impedance in primary ohms for Distance and TD21 functions. Default=0.7
    ///
    /// DSS property name: Z1mag, DSS property index: 30
    ///
    double Z1mag()
    {
        return Obj_GetFloat64(ptr, Properties::Z1mag);
    }

    Relay& Z1mag(double value)
    {
        Obj_SetFloat64(ptr, Properties::Z1mag, value);
        return *this;
    }

    /// Positive sequence reach impedance angle in degrees for Distance and TD21 functions. Default=64.0
    ///
    /// DSS property name: Z1ang, DSS property index: 31
    ///
    double Z1ang()
    {
        return Obj_GetFloat64(ptr, Properties::Z1ang);
    }

    Relay& Z1ang(double value)
    {
        Obj_SetFloat64(ptr, Properties::Z1ang, value);
        return *this;
    }

    /// Zero sequence reach impedance in primary ohms for Distance and TD21 functions. Default=2.1
    ///
    /// DSS property name: Z0mag, DSS property index: 32
    ///
    double Z0mag()
    {
        return Obj_GetFloat64(ptr, Properties::Z0mag);
    }

    Relay& Z0mag(double value)
    {
        Obj_SetFloat64(ptr, Properties::Z0mag, value);
        return *this;
    }

    /// Zero sequence reach impedance angle in degrees for Distance and TD21 functions. Default=68.0
    ///
    /// DSS property name: Z0ang, DSS property index: 33
    ///
    double Z0ang()
    {
        return Obj_GetFloat64(ptr, Properties::Z0ang);
    }

    Relay& Z0ang(double value)
    {
        Obj_SetFloat64(ptr, Properties::Z0ang, value);
        return *this;
    }

    /// Phase reach multiplier in per-unit for Distance and TD21 functions. Default=0.7
    ///
    /// DSS property name: Mphase, DSS property index: 34
    ///
    double Mphase()
    {
        return Obj_GetFloat64(ptr, Properties::Mphase);
    }

    Relay& Mphase(double value)
    {
        Obj_SetFloat64(ptr, Properties::Mphase, value);
        return *this;
    }

    /// Ground reach multiplier in per-unit for Distance and TD21 functions. Default=0.7
    ///
    /// DSS property name: Mground, DSS property index: 35
    ///
    double Mground()
    {
        return Obj_GetFloat64(ptr, Properties::Mground);
    }

    Relay& Mground(double value)
    {
        Obj_SetFloat64(ptr, Properties::Mground, value);
        return *this;
    }

    /// {Yes/True* | No/False} Default is Yes for Relay. Write trips, reclose and reset events to EventLog.
    ///
    /// DSS property name: EventLog, DSS property index: 36
    ///
    bool EventLog()
    {
        return Obj_GetInt32(ptr, Properties::EventLog) != 0;
    }

    Relay& EventLog(bool value)
    {
        Obj_SetInt32(ptr, Properties::EventLog, value);
        return *this;
    }

    /// {Yes/True* | No/False} Default is No for Relay. Write extra details to Eventlog.
    ///
    /// DSS property name: DebugTrace, DSS property index: 37
    ///
    bool DebugTrace()
    {
        return Obj_GetInt32(ptr, Properties::DebugTrace) != 0;
    }

    Relay& DebugTrace(bool value)
    {
        Obj_SetInt32(ptr, Properties::DebugTrace, value);
        return *this;
    }

    /// {Yes/True* | No/False} Default is No; reverse direction for distance and td21 types.
    ///
    /// DSS property name: DistReverse, DSS property index: 38
    ///
    bool DistReverse()
    {
        return Obj_GetInt32(ptr, Properties::DistReverse) != 0;
    }

    Relay& DistReverse(bool value)
    {
        Obj_SetInt32(ptr, Properties::DistReverse, value);
        return *this;
    }

    /// {Open | Closed} Normal state of the relay. The relay reverts to this state for reset, change of mode, etc. Defaults to "State" if not specifically declared.
    ///
    /// DSS property name: Normal, DSS property index: 39
    ///
    RelayState Normal()
    {
        return RelayState(Obj_GetInt32(ptr, Properties::Normal));
    }

    Relay& Normal(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Normal, value);
        return *this;
    }

    Relay& Normal(RelayState value)
    {
        Obj_SetInt32(ptr, Properties::Normal, int32_t(value));
        return *this;
    }

    Relay& Normal(const string &value)
    {
        set_string(Properties::Normal, value);
        return *this;
    }

    Relay& Normal(const char *value)
    {
        set_string(Properties::Normal, value);
        return *this;
    }

    /// {Open | Closed} Normal state of the relay. The relay reverts to this state for reset, change of mode, etc. Defaults to "State" if not specifically declared.
    ///
    /// DSS property name: Normal, DSS property index: 39
    ///
    string Normal_str()
    {
        return get_prop_string(Properties::Normal);
    }

    /// {Open | Closed} Normal state of the relay. The relay reverts to this state for reset, change of mode, etc. Defaults to "State" if not specifically declared.
    ///
    /// DSS property name: Normal, DSS property index: 39
    ///
    Relay& Normal_str(const string &value)
    {
        set_string(Properties::Normal, value);
        return *this;
    }

    /// {Open | Closed} Actual state of the relay. Upon setting, immediately forces state of the relay, overriding the Relay control. Simulates manual control on relay. Defaults to Closed. "Open" causes the controlled element to open and lock out. "Closed" causes the controlled element to close and the relay to reset to its first operation.
    ///
    /// DSS property name: State, DSS property index: 40
    ///
    RelayState State()
    {
        return RelayState(Obj_GetInt32(ptr, Properties::State));
    }

    Relay& State(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::State, value);
        return *this;
    }

    Relay& State(RelayState value)
    {
        Obj_SetInt32(ptr, Properties::State, int32_t(value));
        return *this;
    }

    Relay& State(const string &value)
    {
        set_string(Properties::State, value);
        return *this;
    }

    Relay& State(const char *value)
    {
        set_string(Properties::State, value);
        return *this;
    }

    /// {Open | Closed} Actual state of the relay. Upon setting, immediately forces state of the relay, overriding the Relay control. Simulates manual control on relay. Defaults to Closed. "Open" causes the controlled element to open and lock out. "Closed" causes the controlled element to close and the relay to reset to its first operation.
    ///
    /// DSS property name: State, DSS property index: 40
    ///
    string State_str()
    {
        return get_prop_string(Properties::State);
    }

    /// {Open | Closed} Actual state of the relay. Upon setting, immediately forces state of the relay, overriding the Relay control. Simulates manual control on relay. Defaults to Closed. "Open" causes the controlled element to open and lock out. "Closed" causes the controlled element to close and the relay to reset to its first operation.
    ///
    /// DSS property name: State, DSS property index: 40
    ///
    Relay& State_str(const string &value)
    {
        set_string(Properties::State, value);
        return *this;
    }

    /// Tilt angle for lower current magnitudes. Default is 90.
    ///
    /// DSS property name: DOC_TiltAngleLow, DSS property index: 41
    ///
    double DOC_TiltAngleLow()
    {
        return Obj_GetFloat64(ptr, Properties::DOC_TiltAngleLow);
    }

    Relay& DOC_TiltAngleLow(double value)
    {
        Obj_SetFloat64(ptr, Properties::DOC_TiltAngleLow, value);
        return *this;
    }

    /// Tilt angle for higher current magnitudes. Default is 90.
    ///
    /// DSS property name: DOC_TiltAngleHigh, DSS property index: 42
    ///
    double DOC_TiltAngleHigh()
    {
        return Obj_GetFloat64(ptr, Properties::DOC_TiltAngleHigh);
    }

    Relay& DOC_TiltAngleHigh(double value)
    {
        Obj_SetFloat64(ptr, Properties::DOC_TiltAngleHigh, value);
        return *this;
    }

    /// Trip setting for lower current magnitude.  Default is 0.
    ///
    /// DSS property name: DOC_TripSettingLow, DSS property index: 43
    ///
    double DOC_TripSettingLow()
    {
        return Obj_GetFloat64(ptr, Properties::DOC_TripSettingLow);
    }

    Relay& DOC_TripSettingLow(double value)
    {
        Obj_SetFloat64(ptr, Properties::DOC_TripSettingLow, value);
        return *this;
    }

    /// Trip setting for higher current magnitude.  Default is -1 (deactivated). To activate, set a positive value. Must be greater than "DOC_TripSettingLow".
    ///
    /// DSS property name: DOC_TripSettingHigh, DSS property index: 44
    ///
    double DOC_TripSettingHigh()
    {
        return Obj_GetFloat64(ptr, Properties::DOC_TripSettingHigh);
    }

    Relay& DOC_TripSettingHigh(double value)
    {
        Obj_SetFloat64(ptr, Properties::DOC_TripSettingHigh, value);
        return *this;
    }

    /// Trip setting for current magnitude (define a circle for the relay characteristics). Default is -1 (deactivated). To activate, set a positive value.
    ///
    /// DSS property name: DOC_TripSettingMag, DSS property index: 45
    ///
    double DOC_TripSettingMag()
    {
        return Obj_GetFloat64(ptr, Properties::DOC_TripSettingMag);
    }

    Relay& DOC_TripSettingMag(double value)
    {
        Obj_SetFloat64(ptr, Properties::DOC_TripSettingMag, value);
        return *this;
    }

    /// Trip time delay (sec) for operation in inner zone for DOC relay, defined when "DOC_TripSettingMag" or "DOC_TripSettingHigh" are activate. Default is -1.0 (deactivated), meaning that the relay characteristic is insensitive in the inner zone (no trip). Set to 0 for instantaneous trip and >0 for a definite time delay. If "DOC_PhaseCurveInner" is specified, time delay from curve is utilized instead.
    ///
    /// DSS property name: DOC_DelayInner, DSS property index: 46
    ///
    double DOC_DelayInner()
    {
        return Obj_GetFloat64(ptr, Properties::DOC_DelayInner);
    }

    Relay& DOC_DelayInner(double value)
    {
        Obj_SetFloat64(ptr, Properties::DOC_DelayInner, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the phase trip for operation in inner zone for DOC relay. Must have been previously defined as a TCC_Curve object. Default is none (ignored). Multiplying the current values in the curve by the "DOC_PhaseTripInner" value gives the actual current.
    ///
    /// DSS property name: DOC_PhaseCurveInner, DSS property index: 47
    ///
    double DOC_PhaseCurveInner()
    {
        return Obj_GetFloat64(ptr, Properties::DOC_PhaseCurveInner);
    }

    Relay& DOC_PhaseCurveInner(double value)
    {
        Obj_SetFloat64(ptr, Properties::DOC_PhaseCurveInner, value);
        return *this;
    }

    /// Multiplier for the "DOC_PhaseCurveInner" TCC curve.  Defaults to 1.0.
    ///
    /// DSS property name: DOC_PhaseTripInner, DSS property index: 48
    ///
    double DOC_PhaseTripInner()
    {
        return Obj_GetFloat64(ptr, Properties::DOC_PhaseTripInner);
    }

    Relay& DOC_PhaseTripInner(double value)
    {
        Obj_SetFloat64(ptr, Properties::DOC_PhaseTripInner, value);
        return *this;
    }

    /// Time dial for "DOC_PhaseCurveInner" TCC curve. Multiplier on time axis of specified curve. Default=1.0.
    ///
    /// DSS property name: DOC_TDPhaseInner, DSS property index: 49
    ///
    string DOC_TDPhaseInner()
    {
        return get_prop_string(Properties::DOC_TDPhaseInner);
    }

    Relay& DOC_TDPhaseInner(const string &value)
    {
        set_string(Properties::DOC_TDPhaseInner, value);
        return *this;
    }

    Relay& DOC_TDPhaseInner(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::DOC_TDPhaseInner, value);
        return *this;
    }

    /// Time dial for "DOC_PhaseCurveInner" TCC curve. Multiplier on time axis of specified curve. Default=1.0.
    ///
    /// DSS property name: DOC_TDPhaseInner, DSS property index: 49
    ///
    dss::obj::TCC_Curve DOC_TDPhaseInner_obj()
    {
        return get_obj<dss::obj::TCC_Curve>(Properties::DOC_TDPhaseInner);
    }

    Relay& DOC_TDPhaseInner_obj(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::DOC_TDPhaseInner, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 50
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Relay& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 51
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Relay& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 52
    ///
    Relay& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 52
    ///
    Relay& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Recloser: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 31;
    struct Properties
    {
        enum {
            MonitoredObj = 1,
            MonitoredTerm = 2,
            SwitchedObj = 3,
            SwitchedTerm = 4,
            NumFast = 5,
            PhaseFast = 6,
            PhaseDelayed = 7,
            GroundFast = 8,
            GroundDelayed = 9,
            PhaseTrip = 10,
            GroundTrip = 11,
            PhaseInst = 12,
            GroundInst = 13,
            Reset = 14,
            Shots = 15,
            RecloseIntervals = 16,
            Delay = 17,
            Action = 18,
            TDPhFast = 19,
            TDGrFast = 20,
            TDPhDelayed = 21,
            TDGrDelayed = 22,
            Normal = 23,
            State = 24,
            basefreq = 25,
            enabled = 26,
            like = 27,
        };
    };

    // Class-specific enumerations

        ///
        /// Recloser: Action (DSS enumeration for Recloser)
        ///
    enum class RecloserAction: int32_t
    {
        close = 2, ///< close
        open = 1, ///< open
        trip = 1 ///< trip
    };


        ///
        /// Recloser: State (DSS enumeration for Recloser)
        ///
    enum class RecloserState: int32_t
    {
        closed = 2, ///< closed
        open = 1, ///< open
        trip = 1 ///< trip
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Recloser(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Recloser(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Recloser element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Recloser(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Recloser element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Recloser& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Recloser& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Full object name of the circuit element, typically a line, transformer, load, or generator, to which the Recloser's PT and/or CT are connected. This is the "monitored" element. There is no default; must be specified.
    ///
    /// DSS property name: MonitoredObj, DSS property index: 1
    ///
    string MonitoredObj()
    {
        return get_prop_string(Properties::MonitoredObj);
    }

    Recloser& MonitoredObj(const string &value)
    {
        set_string(Properties::MonitoredObj, value);
        return *this;
    }

    Recloser& MonitoredObj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::MonitoredObj, value);
        return *this;
    }

    /// Full object name of the circuit element, typically a line, transformer, load, or generator, to which the Recloser's PT and/or CT are connected. This is the "monitored" element. There is no default; must be specified.
    ///
    /// DSS property name: MonitoredObj, DSS property index: 1
    ///
    dss::obj::DSSObj MonitoredObj_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::MonitoredObj);
    }

    Recloser& MonitoredObj_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::MonitoredObj, value);
        return *this;
    }

    /// Number of the terminal of the circuit element to which the Recloser is connected. 1 or 2, typically.  Default is 1.
    ///
    /// DSS property name: MonitoredTerm, DSS property index: 2
    ///
    int32_t MonitoredTerm()
    {
        return Obj_GetInt32(ptr, Properties::MonitoredTerm);
    }

    Recloser& MonitoredTerm(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::MonitoredTerm, value);
        return *this;
    }

    /// Name of circuit element switch that the Recloser controls. Specify the full object name.Defaults to the same as the Monitored element. This is the "controlled" element.
    ///
    /// DSS property name: SwitchedObj, DSS property index: 3
    ///
    string SwitchedObj()
    {
        return get_prop_string(Properties::SwitchedObj);
    }

    Recloser& SwitchedObj(const string &value)
    {
        set_string(Properties::SwitchedObj, value);
        return *this;
    }

    Recloser& SwitchedObj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::SwitchedObj, value);
        return *this;
    }

    /// Name of circuit element switch that the Recloser controls. Specify the full object name.Defaults to the same as the Monitored element. This is the "controlled" element.
    ///
    /// DSS property name: SwitchedObj, DSS property index: 3
    ///
    dss::obj::DSSObj SwitchedObj_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::SwitchedObj);
    }

    Recloser& SwitchedObj_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::SwitchedObj, value);
        return *this;
    }

    /// Number of the terminal of the controlled element in which the switch is controlled by the Recloser. 1 or 2, typically.  Default is 1.
    ///
    /// DSS property name: SwitchedTerm, DSS property index: 4
    ///
    int32_t SwitchedTerm()
    {
        return Obj_GetInt32(ptr, Properties::SwitchedTerm);
    }

    Recloser& SwitchedTerm(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::SwitchedTerm, value);
        return *this;
    }

    /// Number of Fast (fuse saving) operations.  Default is 1. (See "Shots")
    ///
    /// DSS property name: NumFast, DSS property index: 5
    ///
    int32_t NumFast()
    {
        return Obj_GetInt32(ptr, Properties::NumFast);
    }

    Recloser& NumFast(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::NumFast, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the Phase Fast trip.  Must have been previously defined as a TCC_Curve object. Default is "A". Multiplying the current values in the curve by the "phasetrip" value gives the actual current.
    ///
    /// DSS property name: PhaseFast, DSS property index: 6
    ///
    string PhaseFast()
    {
        return get_prop_string(Properties::PhaseFast);
    }

    Recloser& PhaseFast(const string &value)
    {
        set_string(Properties::PhaseFast, value);
        return *this;
    }

    Recloser& PhaseFast(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::PhaseFast, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the Phase Fast trip.  Must have been previously defined as a TCC_Curve object. Default is "A". Multiplying the current values in the curve by the "phasetrip" value gives the actual current.
    ///
    /// DSS property name: PhaseFast, DSS property index: 6
    ///
    dss::obj::TCC_Curve PhaseFast_obj()
    {
        return get_obj<dss::obj::TCC_Curve>(Properties::PhaseFast);
    }

    Recloser& PhaseFast_obj(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::PhaseFast, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the Phase Delayed trip.  Must have been previously defined as a TCC_Curve object. Default is "D".Multiplying the current values in the curve by the "phasetrip" value gives the actual current.
    ///
    /// DSS property name: PhaseDelayed, DSS property index: 7
    ///
    string PhaseDelayed()
    {
        return get_prop_string(Properties::PhaseDelayed);
    }

    Recloser& PhaseDelayed(const string &value)
    {
        set_string(Properties::PhaseDelayed, value);
        return *this;
    }

    Recloser& PhaseDelayed(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::PhaseDelayed, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the Phase Delayed trip.  Must have been previously defined as a TCC_Curve object. Default is "D".Multiplying the current values in the curve by the "phasetrip" value gives the actual current.
    ///
    /// DSS property name: PhaseDelayed, DSS property index: 7
    ///
    dss::obj::TCC_Curve PhaseDelayed_obj()
    {
        return get_obj<dss::obj::TCC_Curve>(Properties::PhaseDelayed);
    }

    Recloser& PhaseDelayed_obj(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::PhaseDelayed, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the Ground Fast trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored). Multiplying the current values in the curve by the "groundtrip" value gives the actual current.
    ///
    /// DSS property name: GroundFast, DSS property index: 8
    ///
    string GroundFast()
    {
        return get_prop_string(Properties::GroundFast);
    }

    Recloser& GroundFast(const string &value)
    {
        set_string(Properties::GroundFast, value);
        return *this;
    }

    Recloser& GroundFast(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::GroundFast, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the Ground Fast trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored). Multiplying the current values in the curve by the "groundtrip" value gives the actual current.
    ///
    /// DSS property name: GroundFast, DSS property index: 8
    ///
    dss::obj::TCC_Curve GroundFast_obj()
    {
        return get_obj<dss::obj::TCC_Curve>(Properties::GroundFast);
    }

    Recloser& GroundFast_obj(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::GroundFast, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the Ground Delayed trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored).Multiplying the current values in the curve by the "groundtrip" value gives the actual current.
    ///
    /// DSS property name: GroundDelayed, DSS property index: 9
    ///
    string GroundDelayed()
    {
        return get_prop_string(Properties::GroundDelayed);
    }

    Recloser& GroundDelayed(const string &value)
    {
        set_string(Properties::GroundDelayed, value);
        return *this;
    }

    Recloser& GroundDelayed(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::GroundDelayed, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the Ground Delayed trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored).Multiplying the current values in the curve by the "groundtrip" value gives the actual current.
    ///
    /// DSS property name: GroundDelayed, DSS property index: 9
    ///
    dss::obj::TCC_Curve GroundDelayed_obj()
    {
        return get_obj<dss::obj::TCC_Curve>(Properties::GroundDelayed);
    }

    Recloser& GroundDelayed_obj(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::GroundDelayed, value);
        return *this;
    }

    /// Multiplier or actual phase amps for the phase TCC curve.  Defaults to 1.0.
    ///
    /// DSS property name: PhaseTrip, DSS property index: 10
    ///
    double PhaseTrip()
    {
        return Obj_GetFloat64(ptr, Properties::PhaseTrip);
    }

    Recloser& PhaseTrip(double value)
    {
        Obj_SetFloat64(ptr, Properties::PhaseTrip, value);
        return *this;
    }

    /// Multiplier or actual ground amps (3I0) for the ground TCC curve.  Defaults to 1.0.
    ///
    /// DSS property name: GroundTrip, DSS property index: 11
    ///
    double GroundTrip()
    {
        return Obj_GetFloat64(ptr, Properties::GroundTrip);
    }

    Recloser& GroundTrip(double value)
    {
        Obj_SetFloat64(ptr, Properties::GroundTrip, value);
        return *this;
    }

    /// Actual amps for instantaneous phase trip which is assumed to happen in 0.01 sec + Delay Time. Default is 0.0, which signifies no inst trip.
    ///
    /// DSS property name: PhaseInst, DSS property index: 12
    ///
    double PhaseInst()
    {
        return Obj_GetFloat64(ptr, Properties::PhaseInst);
    }

    Recloser& PhaseInst(double value)
    {
        Obj_SetFloat64(ptr, Properties::PhaseInst, value);
        return *this;
    }

    /// Actual amps for instantaneous ground trip which is assumed to happen in 0.01 sec + Delay Time.Default is 0.0, which signifies no inst trip.
    ///
    /// DSS property name: GroundInst, DSS property index: 13
    ///
    double GroundInst()
    {
        return Obj_GetFloat64(ptr, Properties::GroundInst);
    }

    Recloser& GroundInst(double value)
    {
        Obj_SetFloat64(ptr, Properties::GroundInst, value);
        return *this;
    }

    /// Reset time in sec for Recloser.  Default is 15.
    ///
    /// DSS property name: Reset, DSS property index: 14
    ///
    double Reset()
    {
        return Obj_GetFloat64(ptr, Properties::Reset);
    }

    Recloser& Reset(double value)
    {
        Obj_SetFloat64(ptr, Properties::Reset, value);
        return *this;
    }

    /// Total Number of fast and delayed shots to lockout.  Default is 4. This is one more than the number of reclose intervals.
    ///
    /// DSS property name: Shots, DSS property index: 15
    ///
    int32_t Shots()
    {
        return Obj_GetInt32(ptr, Properties::Shots);
    }

    Recloser& Shots(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Shots, value);
        return *this;
    }

    /// Array of reclose intervals.  Default for Recloser is (0.5, 2.0, 2.0) seconds. A locked out Recloser must be closed manually (action=close).
    ///
    /// DSS property name: RecloseIntervals, DSS property index: 16
    ///
    VectorXd RecloseIntervals()
    {
        return get_array<VectorXd>(Properties::RecloseIntervals);
    }

    Recloser& RecloseIntervals(VectorXd &value)
    {
        set_array<VectorXd>(Properties::RecloseIntervals, value);
        return *this;
    }

    /// Fixed delay time (sec) added to Recloser trip time. Default is 0.0. Used to represent breaker time or any other delay.
    ///
    /// DSS property name: Delay, DSS property index: 17
    ///
    double Delay()
    {
        return Obj_GetFloat64(ptr, Properties::Delay);
    }

    Recloser& Delay(double value)
    {
        Obj_SetFloat64(ptr, Properties::Delay, value);
        return *this;
    }

    /// DEPRECATED. See "State" property
    ///
    /// DSS property name: Action, DSS property index: 18
    ///
    RecloserAction Action()
    {
        return RecloserAction(Obj_GetInt32(ptr, Properties::Action));
    }

    Recloser& Action(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Action, value);
        return *this;
    }

    Recloser& Action(RecloserAction value)
    {
        Obj_SetInt32(ptr, Properties::Action, int32_t(value));
        return *this;
    }

    Recloser& Action(const string &value)
    {
        set_string(Properties::Action, value);
        return *this;
    }

    Recloser& Action(const char *value)
    {
        set_string(Properties::Action, value);
        return *this;
    }

    /// DEPRECATED. See "State" property
    ///
    /// DSS property name: Action, DSS property index: 18
    ///
    string Action_str()
    {
        return get_prop_string(Properties::Action);
    }

    /// DEPRECATED. See "State" property
    ///
    /// DSS property name: Action, DSS property index: 18
    ///
    Recloser& Action_str(const string &value)
    {
        set_string(Properties::Action, value);
        return *this;
    }

    /// Time dial for Phase Fast trip curve. Multiplier on time axis of specified curve. Default=1.0.
    ///
    /// DSS property name: TDPhFast, DSS property index: 19
    ///
    double TDPhFast()
    {
        return Obj_GetFloat64(ptr, Properties::TDPhFast);
    }

    Recloser& TDPhFast(double value)
    {
        Obj_SetFloat64(ptr, Properties::TDPhFast, value);
        return *this;
    }

    /// Time dial for Ground Fast trip curve. Multiplier on time axis of specified curve. Default=1.0.
    ///
    /// DSS property name: TDGrFast, DSS property index: 20
    ///
    double TDGrFast()
    {
        return Obj_GetFloat64(ptr, Properties::TDGrFast);
    }

    Recloser& TDGrFast(double value)
    {
        Obj_SetFloat64(ptr, Properties::TDGrFast, value);
        return *this;
    }

    /// Time dial for Phase Delayed trip curve. Multiplier on time axis of specified curve. Default=1.0.
    ///
    /// DSS property name: TDPhDelayed, DSS property index: 21
    ///
    double TDPhDelayed()
    {
        return Obj_GetFloat64(ptr, Properties::TDPhDelayed);
    }

    Recloser& TDPhDelayed(double value)
    {
        Obj_SetFloat64(ptr, Properties::TDPhDelayed, value);
        return *this;
    }

    /// Time dial for Ground Delayed trip curve. Multiplier on time axis of specified curve. Default=1.0.
    ///
    /// DSS property name: TDGrDelayed, DSS property index: 22
    ///
    double TDGrDelayed()
    {
        return Obj_GetFloat64(ptr, Properties::TDGrDelayed);
    }

    Recloser& TDGrDelayed(double value)
    {
        Obj_SetFloat64(ptr, Properties::TDGrDelayed, value);
        return *this;
    }

    /// {Open | Closed} Normal state of the recloser. The recloser reverts to this state for reset, change of mode, etc. Defaults to "State" if not specificallt declared.
    ///
    /// DSS property name: Normal, DSS property index: 23
    ///
    RecloserState Normal()
    {
        return RecloserState(Obj_GetInt32(ptr, Properties::Normal));
    }

    Recloser& Normal(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Normal, value);
        return *this;
    }

    Recloser& Normal(RecloserState value)
    {
        Obj_SetInt32(ptr, Properties::Normal, int32_t(value));
        return *this;
    }

    Recloser& Normal(const string &value)
    {
        set_string(Properties::Normal, value);
        return *this;
    }

    Recloser& Normal(const char *value)
    {
        set_string(Properties::Normal, value);
        return *this;
    }

    /// {Open | Closed} Normal state of the recloser. The recloser reverts to this state for reset, change of mode, etc. Defaults to "State" if not specificallt declared.
    ///
    /// DSS property name: Normal, DSS property index: 23
    ///
    string Normal_str()
    {
        return get_prop_string(Properties::Normal);
    }

    /// {Open | Closed} Normal state of the recloser. The recloser reverts to this state for reset, change of mode, etc. Defaults to "State" if not specificallt declared.
    ///
    /// DSS property name: Normal, DSS property index: 23
    ///
    Recloser& Normal_str(const string &value)
    {
        set_string(Properties::Normal, value);
        return *this;
    }

    /// {Open | Closed} Actual state of the recloser. Upon setting, immediately forces state of the recloser, overriding the Recloser control. Simulates manual control on recloser. Defaults to Closed. "Open" causes the controlled element to open and lock out. "Closed" causes the controlled element to close and the recloser to reset to its first operation.
    ///
    /// DSS property name: State, DSS property index: 24
    ///
    RecloserState State()
    {
        return RecloserState(Obj_GetInt32(ptr, Properties::State));
    }

    Recloser& State(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::State, value);
        return *this;
    }

    Recloser& State(RecloserState value)
    {
        Obj_SetInt32(ptr, Properties::State, int32_t(value));
        return *this;
    }

    Recloser& State(const string &value)
    {
        set_string(Properties::State, value);
        return *this;
    }

    Recloser& State(const char *value)
    {
        set_string(Properties::State, value);
        return *this;
    }

    /// {Open | Closed} Actual state of the recloser. Upon setting, immediately forces state of the recloser, overriding the Recloser control. Simulates manual control on recloser. Defaults to Closed. "Open" causes the controlled element to open and lock out. "Closed" causes the controlled element to close and the recloser to reset to its first operation.
    ///
    /// DSS property name: State, DSS property index: 24
    ///
    string State_str()
    {
        return get_prop_string(Properties::State);
    }

    /// {Open | Closed} Actual state of the recloser. Upon setting, immediately forces state of the recloser, overriding the Recloser control. Simulates manual control on recloser. Defaults to Closed. "Open" causes the controlled element to open and lock out. "Closed" causes the controlled element to close and the recloser to reset to its first operation.
    ///
    /// DSS property name: State, DSS property index: 24
    ///
    Recloser& State_str(const string &value)
    {
        set_string(Properties::State, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 25
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Recloser& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 26
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Recloser& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 27
    ///
    Recloser& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 27
    ///
    Recloser& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Fuse: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 32;
    struct Properties
    {
        enum {
            MonitoredObj = 1,
            MonitoredTerm = 2,
            SwitchedObj = 3,
            SwitchedTerm = 4,
            FuseCurve = 5,
            RatedCurrent = 6,
            Delay = 7,
            Action = 8,
            Normal = 9,
            State = 10,
            basefreq = 11,
            enabled = 12,
            like = 13,
        };
    };

    // Class-specific enumerations

        ///
        /// Fuse: Action (DSS enumeration for Fuse)
        ///
    enum class FuseAction: int32_t
    {
        close = 2, ///< close
        open = 1 ///< open
    };


        ///
        /// Fuse: State (DSS enumeration for Fuse)
        ///
    enum class FuseState: int32_t
    {
        closed = 2, ///< closed
        open = 1 ///< open
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Fuse(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Fuse(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Fuse element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Fuse(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Fuse element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Fuse& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Fuse& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Full object name of the circuit element, typically a line, transformer, load, or generator, to which the Fuse is connected. This is the "monitored" element. There is no default; must be specified.
    ///
    /// DSS property name: MonitoredObj, DSS property index: 1
    ///
    string MonitoredObj()
    {
        return get_prop_string(Properties::MonitoredObj);
    }

    Fuse& MonitoredObj(const string &value)
    {
        set_string(Properties::MonitoredObj, value);
        return *this;
    }

    Fuse& MonitoredObj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::MonitoredObj, value);
        return *this;
    }

    /// Full object name of the circuit element, typically a line, transformer, load, or generator, to which the Fuse is connected. This is the "monitored" element. There is no default; must be specified.
    ///
    /// DSS property name: MonitoredObj, DSS property index: 1
    ///
    dss::obj::DSSObj MonitoredObj_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::MonitoredObj);
    }

    Fuse& MonitoredObj_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::MonitoredObj, value);
        return *this;
    }

    /// Number of the terminal of the circuit element to which the Fuse is connected. 1 or 2, typically.  Default is 1.
    ///
    /// DSS property name: MonitoredTerm, DSS property index: 2
    ///
    int32_t MonitoredTerm()
    {
        return Obj_GetInt32(ptr, Properties::MonitoredTerm);
    }

    Fuse& MonitoredTerm(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::MonitoredTerm, value);
        return *this;
    }

    /// Name of circuit element switch that the Fuse controls. Specify the full object name.Defaults to the same as the Monitored element. This is the "controlled" element.
    ///
    /// DSS property name: SwitchedObj, DSS property index: 3
    ///
    string SwitchedObj()
    {
        return get_prop_string(Properties::SwitchedObj);
    }

    Fuse& SwitchedObj(const string &value)
    {
        set_string(Properties::SwitchedObj, value);
        return *this;
    }

    Fuse& SwitchedObj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::SwitchedObj, value);
        return *this;
    }

    /// Name of circuit element switch that the Fuse controls. Specify the full object name.Defaults to the same as the Monitored element. This is the "controlled" element.
    ///
    /// DSS property name: SwitchedObj, DSS property index: 3
    ///
    dss::obj::DSSObj SwitchedObj_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::SwitchedObj);
    }

    Fuse& SwitchedObj_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::SwitchedObj, value);
        return *this;
    }

    /// Number of the terminal of the controlled element in which the switch is controlled by the Fuse. 1 or 2, typically.  Default is 1.  Assumes all phases of the element have a fuse of this type.
    ///
    /// DSS property name: SwitchedTerm, DSS property index: 4
    ///
    int32_t SwitchedTerm()
    {
        return Obj_GetInt32(ptr, Properties::SwitchedTerm);
    }

    Fuse& SwitchedTerm(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::SwitchedTerm, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the fuse blowing.  Must have been previously defined as a TCC_Curve object. Default is "Tlink". Multiplying the current values in the curve by the "RatedCurrent" value gives the actual current.
    ///
    /// DSS property name: FuseCurve, DSS property index: 5
    ///
    string FuseCurve()
    {
        return get_prop_string(Properties::FuseCurve);
    }

    Fuse& FuseCurve(const string &value)
    {
        set_string(Properties::FuseCurve, value);
        return *this;
    }

    Fuse& FuseCurve(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::FuseCurve, value);
        return *this;
    }

    /// Name of the TCC Curve object that determines the fuse blowing.  Must have been previously defined as a TCC_Curve object. Default is "Tlink". Multiplying the current values in the curve by the "RatedCurrent" value gives the actual current.
    ///
    /// DSS property name: FuseCurve, DSS property index: 5
    ///
    dss::obj::TCC_Curve FuseCurve_obj()
    {
        return get_obj<dss::obj::TCC_Curve>(Properties::FuseCurve);
    }

    Fuse& FuseCurve_obj(dss::obj::TCC_Curve &value)
    {
        set_obj(Properties::FuseCurve, value);
        return *this;
    }

    /// Multiplier or actual phase amps for the phase TCC curve.  Defaults to 1.0.
    ///
    /// DSS property name: RatedCurrent, DSS property index: 6
    ///
    double RatedCurrent()
    {
        return Obj_GetFloat64(ptr, Properties::RatedCurrent);
    }

    Fuse& RatedCurrent(double value)
    {
        Obj_SetFloat64(ptr, Properties::RatedCurrent, value);
        return *this;
    }

    /// Fixed delay time (sec) added to Fuse blowing time determined from the TCC curve. Default is 0.0. Used to represent fuse clearing time or any other delay.
    ///
    /// DSS property name: Delay, DSS property index: 7
    ///
    double Delay()
    {
        return Obj_GetFloat64(ptr, Properties::Delay);
    }

    Fuse& Delay(double value)
    {
        Obj_SetFloat64(ptr, Properties::Delay, value);
        return *this;
    }

    /// DEPRECATED. See "State" property.
    ///
    /// DSS property name: Action, DSS property index: 8
    ///
    Fuse& Action(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Action, value);
        return *this;
    }

    /// DEPRECATED. See "State" property.
    ///
    /// DSS property name: Action, DSS property index: 8
    ///
    Fuse& Action(FuseAction value)
    {
        Obj_SetInt32(ptr, Properties::Action, int32_t(value));
        return *this;
    }

    /// DEPRECATED. See "State" property.
    ///
    /// DSS property name: Action, DSS property index: 8
    ///
    Fuse& Action(const string &value)
    {
        set_string(Properties::Action, value);
        return *this;
    }

    /// DEPRECATED. See "State" property.
    ///
    /// DSS property name: Action, DSS property index: 8
    ///
    Fuse& Action(const char *value)
    {
        set_string(Properties::Action, value);
        return *this;
    }

    /// ARRAY of strings {Open | Closed} representing the Normal state of the fuse in each phase of the controlled element. The fuse reverts to this state for reset, change of mode, etc. Defaults to "State" if not specifically declared.
    ///
    /// DSS property name: Normal, DSS property index: 9
    ///
    std::vector<FuseState> Normal()
    {
        return get_array<std::vector<FuseState>>(Properties::Normal);
    }

    Fuse& Normal(std::vector<int32_t> &value)
    {
        set_array<std::vector<int32_t>>(Properties::Normal, value);
        return *this;
    }

    Fuse& Normal(strings &value)
    {
        set_array<strings>(Properties::Normal, value);
        return *this;
    }

    /// ARRAY of strings {Open | Closed} representing the Normal state of the fuse in each phase of the controlled element. The fuse reverts to this state for reset, change of mode, etc. Defaults to "State" if not specifically declared.
    ///
    /// DSS property name: Normal, DSS property index: 9
    ///
    strings Normal_str()
    {
        return get_array<strings>(Properties::Normal);
    }

    Fuse& Normal_str(strings &value)
    {
        Normal(value);
        return *this;
    }

    /// ARRAY of strings {Open | Closed} representing the Actual state of the fuse in each phase of the controlled element. Upon setting, immediately forces state of fuse(s). Simulates manual control on Fuse. Defaults to Closed for all phases.
    ///
    /// DSS property name: State, DSS property index: 10
    ///
    std::vector<FuseState> State()
    {
        return get_array<std::vector<FuseState>>(Properties::State);
    }

    Fuse& State(std::vector<int32_t> &value)
    {
        set_array<std::vector<int32_t>>(Properties::State, value);
        return *this;
    }

    Fuse& State(strings &value)
    {
        set_array<strings>(Properties::State, value);
        return *this;
    }

    /// ARRAY of strings {Open | Closed} representing the Actual state of the fuse in each phase of the controlled element. Upon setting, immediately forces state of fuse(s). Simulates manual control on Fuse. Defaults to Closed for all phases.
    ///
    /// DSS property name: State, DSS property index: 10
    ///
    strings State_str()
    {
        return get_array<strings>(Properties::State);
    }

    Fuse& State_str(strings &value)
    {
        State(value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 11
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Fuse& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 12
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Fuse& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 13
    ///
    Fuse& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 13
    ///
    Fuse& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class SwtControl: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 33;
    struct Properties
    {
        enum {
            SwitchedObj = 1,
            SwitchedTerm = 2,
            Action = 3,
            Lock = 4,
            Delay = 5,
            Normal = 6,
            State = 7,
            Reset = 8,
            basefreq = 9,
            enabled = 10,
            like = 11,
        };
    };

    // Class-specific enumerations

        ///
        /// SwtControl: Action (DSS enumeration for SwtControl)
        ///
    enum class SwtControlAction: int32_t
    {
        close = 2, ///< close
        open = 1 ///< open
    };


        ///
        /// SwtControl: State (DSS enumeration for SwtControl)
        ///
    enum class SwtControlState: int32_t
    {
        closed = 2, ///< closed
        open = 1 ///< open
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    SwtControl(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    SwtControl(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the SwtControl element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    SwtControl(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the SwtControl element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    SwtControl& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    SwtControl& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name of circuit element switch that the SwtControl operates. Specify the full object class and name.
    ///
    /// DSS property name: SwitchedObj, DSS property index: 1
    ///
    string SwitchedObj()
    {
        return get_prop_string(Properties::SwitchedObj);
    }

    SwtControl& SwitchedObj(const string &value)
    {
        set_string(Properties::SwitchedObj, value);
        return *this;
    }

    SwtControl& SwitchedObj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::SwitchedObj, value);
        return *this;
    }

    /// Name of circuit element switch that the SwtControl operates. Specify the full object class and name.
    ///
    /// DSS property name: SwitchedObj, DSS property index: 1
    ///
    dss::obj::DSSObj SwitchedObj_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::SwitchedObj);
    }

    SwtControl& SwitchedObj_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::SwitchedObj, value);
        return *this;
    }

    /// Terminal number of the controlled element switch. 1 or 2, typically.  Default is 1.
    ///
    /// DSS property name: SwitchedTerm, DSS property index: 2
    ///
    int32_t SwitchedTerm()
    {
        return Obj_GetInt32(ptr, Properties::SwitchedTerm);
    }

    SwtControl& SwitchedTerm(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::SwitchedTerm, value);
        return *this;
    }

    /// {Open | Close}  After specified delay time, and if not locked, causes the controlled switch to open or close.
    ///
    /// DSS property name: Action, DSS property index: 3
    ///
    SwtControlAction Action()
    {
        return SwtControlAction(Obj_GetInt32(ptr, Properties::Action));
    }

    SwtControl& Action(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Action, value);
        return *this;
    }

    SwtControl& Action(SwtControlAction value)
    {
        Obj_SetInt32(ptr, Properties::Action, int32_t(value));
        return *this;
    }

    SwtControl& Action(const string &value)
    {
        set_string(Properties::Action, value);
        return *this;
    }

    SwtControl& Action(const char *value)
    {
        set_string(Properties::Action, value);
        return *this;
    }

    /// {Open | Close}  After specified delay time, and if not locked, causes the controlled switch to open or close.
    ///
    /// DSS property name: Action, DSS property index: 3
    ///
    string Action_str()
    {
        return get_prop_string(Properties::Action);
    }

    /// {Open | Close}  After specified delay time, and if not locked, causes the controlled switch to open or close.
    ///
    /// DSS property name: Action, DSS property index: 3
    ///
    SwtControl& Action_str(const string &value)
    {
        set_string(Properties::Action, value);
        return *this;
    }

    /// {Yes | No} Delayed action. Sends CTRL_LOCK or CTRL_UNLOCK message to control queue. After delay time, controlled switch is locked in its present open / close state or unlocked. Switch will not respond to either manual (Action) or automatic (COM interface) control or internal OpenDSS Reset when locked.
    ///
    /// DSS property name: Lock, DSS property index: 4
    ///
    bool Lock()
    {
        return Obj_GetInt32(ptr, Properties::Lock) != 0;
    }

    SwtControl& Lock(bool value)
    {
        Obj_SetInt32(ptr, Properties::Lock, value);
        return *this;
    }

    /// Operating time delay (sec) of the switch. Defaults to 120.
    ///
    /// DSS property name: Delay, DSS property index: 5
    ///
    double Delay()
    {
        return Obj_GetFloat64(ptr, Properties::Delay);
    }

    SwtControl& Delay(double value)
    {
        Obj_SetFloat64(ptr, Properties::Delay, value);
        return *this;
    }

    /// {Open | Closed] Normal state of the switch. If not Locked, the switch reverts to this state for reset, change of mode, etc. Defaults to first Action or State specified if not specifically declared.
    ///
    /// DSS property name: Normal, DSS property index: 6
    ///
    SwtControlState Normal()
    {
        return SwtControlState(Obj_GetInt32(ptr, Properties::Normal));
    }

    SwtControl& Normal(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Normal, value);
        return *this;
    }

    SwtControl& Normal(SwtControlState value)
    {
        Obj_SetInt32(ptr, Properties::Normal, int32_t(value));
        return *this;
    }

    SwtControl& Normal(const string &value)
    {
        set_string(Properties::Normal, value);
        return *this;
    }

    SwtControl& Normal(const char *value)
    {
        set_string(Properties::Normal, value);
        return *this;
    }

    /// {Open | Closed] Normal state of the switch. If not Locked, the switch reverts to this state for reset, change of mode, etc. Defaults to first Action or State specified if not specifically declared.
    ///
    /// DSS property name: Normal, DSS property index: 6
    ///
    string Normal_str()
    {
        return get_prop_string(Properties::Normal);
    }

    /// {Open | Closed] Normal state of the switch. If not Locked, the switch reverts to this state for reset, change of mode, etc. Defaults to first Action or State specified if not specifically declared.
    ///
    /// DSS property name: Normal, DSS property index: 6
    ///
    SwtControl& Normal_str(const string &value)
    {
        set_string(Properties::Normal, value);
        return *this;
    }

    /// {Open | Closed] Present state of the switch. Upon setting, immediately forces state of switch.
    ///
    /// DSS property name: State, DSS property index: 7
    ///
    SwtControlState State()
    {
        return SwtControlState(Obj_GetInt32(ptr, Properties::State));
    }

    SwtControl& State(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::State, value);
        return *this;
    }

    SwtControl& State(SwtControlState value)
    {
        Obj_SetInt32(ptr, Properties::State, int32_t(value));
        return *this;
    }

    SwtControl& State(const string &value)
    {
        set_string(Properties::State, value);
        return *this;
    }

    SwtControl& State(const char *value)
    {
        set_string(Properties::State, value);
        return *this;
    }

    /// {Open | Closed] Present state of the switch. Upon setting, immediately forces state of switch.
    ///
    /// DSS property name: State, DSS property index: 7
    ///
    string State_str()
    {
        return get_prop_string(Properties::State);
    }

    /// {Open | Closed] Present state of the switch. Upon setting, immediately forces state of switch.
    ///
    /// DSS property name: State, DSS property index: 7
    ///
    SwtControl& State_str(const string &value)
    {
        set_string(Properties::State, value);
        return *this;
    }

    /// {Yes | No} If Yes, forces Reset of switch to Normal state and removes Lock independently of any internal reset command for mode change, etc.
    ///
    /// DSS property name: Reset, DSS property index: 8
    ///
    SwtControl& Reset(bool value)
    {
        Obj_SetInt32(ptr, Properties::Reset, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 9
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    SwtControl& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 10
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    SwtControl& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 11
    ///
    SwtControl& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 11
    ///
    SwtControl& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class PVSystem: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 34;
    struct Properties
    {
        enum {
            phases = 1,
            bus1 = 2,
            kv = 3,
            irradiance = 4,
            Pmpp = 5,
            pctPmpp = 6,
            Temperature = 7,
            pf = 8,
            conn = 9,
            kvar = 10,
            kVA = 11,
            pctCutin = 12,
            pctCutout = 13,
            EffCurve = 14,
            PTCurve = 15,
            pctR = 16,
            pctX = 17,
            model = 18,
            Vminpu = 19,
            Vmaxpu = 20,
            Balanced = 21,
            LimitCurrent = 22,
            yearly = 23,
            daily = 24,
            duty = 25,
            Tyearly = 26,
            Tdaily = 27,
            Tduty = 28,
            cls = 29,
            UserModel = 30,
            UserData = 31,
            debugtrace = 32,
            VarFollowInverter = 33,
            DutyStart = 34,
            WattPriority = 35,
            PFPriority = 36,
            pctPminNoVars = 37,
            pctPminkvarMax = 38,
            kvarMax = 39,
            kvarMaxAbs = 40,
            spectrum = 41,
            basefreq = 42,
            enabled = 43,
            like = 44,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    PVSystem(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    PVSystem(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the PVSystem element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    PVSystem(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the PVSystem element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    PVSystem& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    PVSystem& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of Phases, this PVSystem element.  Power is evenly divided among phases.
    ///
    /// DSS property name: phases, DSS property index: 1
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    PVSystem& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Bus to which the PVSystem element is connected.  May include specific node specification.
    ///
    /// DSS property name: bus1, DSS property index: 2
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    PVSystem& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    PVSystem& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Nominal rated (1.0 per unit) voltage, kV, for PVSystem element. For 2- and 3-phase PVSystem elements, specify phase-phase kV. Otherwise, specify actual kV across each branch of the PVSystem element. If 1-phase wye (star or LN), specify phase-neutral kV. If 1-phase delta or phase-phase connected, specify phase-phase kV.
    ///
    /// DSS property name: kv, DSS property index: 3
    ///
    double kv()
    {
        return Obj_GetFloat64(ptr, Properties::kv);
    }

    PVSystem& kv(double value)
    {
        Obj_SetFloat64(ptr, Properties::kv, value);
        return *this;
    }

    /// Get/set the present irradiance value in kW/sq-m. Used as base value for shape multipliers. Generally entered as peak value for the time period of interest and the yearly, daily, and duty load shape objects are defined as per unit multipliers (just like Loads/Generators).
    ///
    /// DSS property name: irradiance, DSS property index: 4
    ///
    double irradiance()
    {
        return Obj_GetFloat64(ptr, Properties::irradiance);
    }

    PVSystem& irradiance(double value)
    {
        Obj_SetFloat64(ptr, Properties::irradiance, value);
        return *this;
    }

    /// Get/set the rated max power of the PV array for 1.0 kW/sq-m irradiance and a user-selected array temperature. The P-TCurve should be defined relative to the selected array temperature.
    ///
    /// DSS property name: Pmpp, DSS property index: 5
    ///
    double Pmpp()
    {
        return Obj_GetFloat64(ptr, Properties::Pmpp);
    }

    PVSystem& Pmpp(double value)
    {
        Obj_SetFloat64(ptr, Properties::Pmpp, value);
        return *this;
    }

    /// Upper limit on active power as a percentage of Pmpp.
    ///
    /// DSS property name: %Pmpp, DSS property index: 6
    ///
    double pctPmpp()
    {
        return Obj_GetFloat64(ptr, Properties::pctPmpp);
    }

    PVSystem& pctPmpp(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctPmpp, value);
        return *this;
    }

    /// Get/set the present Temperature. Used as fixed value corresponding to PTCurve property. A multiplier is obtained from the Pmpp-Temp curve and applied to the nominal Pmpp from the irradiance to determine the net array output.
    ///
    /// DSS property name: Temperature, DSS property index: 7
    ///
    double Temperature()
    {
        return Obj_GetFloat64(ptr, Properties::Temperature);
    }

    PVSystem& Temperature(double value)
    {
        Obj_SetFloat64(ptr, Properties::Temperature, value);
        return *this;
    }

    /// Nominally, the power factor for the output power. Default is 1.0. Setting this property will cause the inverter to operate in constant power factor mode.Enter negative when kW and kvar have opposite signs.
    /// A positive power factor signifies that the PVSystem element produces vars
    /// as is typical for a generator.
    ///
    /// DSS property name: pf, DSS property index: 8
    ///
    double pf()
    {
        return Obj_GetFloat64(ptr, Properties::pf);
    }

    PVSystem& pf(double value)
    {
        Obj_SetFloat64(ptr, Properties::pf, value);
        return *this;
    }

    /// ={wye|LN|delta|LL}.  Default is wye.
    ///
    /// DSS property name: conn, DSS property index: 9
    ///
    Connection conn()
    {
        return Connection(Obj_GetInt32(ptr, Properties::conn));
    }

    PVSystem& conn(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::conn, value);
        return *this;
    }

    PVSystem& conn(Connection value)
    {
        Obj_SetInt32(ptr, Properties::conn, int32_t(value));
        return *this;
    }

    PVSystem& conn(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    PVSystem& conn(const char *value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// ={wye|LN|delta|LL}.  Default is wye.
    ///
    /// DSS property name: conn, DSS property index: 9
    ///
    string conn_str()
    {
        return get_prop_string(Properties::conn);
    }

    /// ={wye|LN|delta|LL}.  Default is wye.
    ///
    /// DSS property name: conn, DSS property index: 9
    ///
    PVSystem& conn_str(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// Get/set the present kvar value.  Setting this property forces the inverter to operate in constant kvar mode.
    ///
    /// DSS property name: kvar, DSS property index: 10
    ///
    double kvar()
    {
        return Obj_GetFloat64(ptr, Properties::kvar);
    }

    PVSystem& kvar(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvar, value);
        return *this;
    }

    /// kVA rating of inverter. Used as the base for Dynamics mode and Harmonics mode values.
    ///
    /// DSS property name: kVA, DSS property index: 11
    ///
    double kVA()
    {
        return Obj_GetFloat64(ptr, Properties::kVA);
    }

    PVSystem& kVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::kVA, value);
        return *this;
    }

    /// % cut-in power -- % of kVA rating of inverter. When the inverter is OFF, the power from the array must be greater than this for the inverter to turn on.
    ///
    /// DSS property name: %Cutin, DSS property index: 12
    ///
    double pctCutin()
    {
        return Obj_GetFloat64(ptr, Properties::pctCutin);
    }

    PVSystem& pctCutin(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctCutin, value);
        return *this;
    }

    /// % cut-out power -- % of kVA rating of inverter. When the inverter is ON, the inverter turns OFF when the power from the array drops below this value.
    ///
    /// DSS property name: %Cutout, DSS property index: 13
    ///
    double pctCutout()
    {
        return Obj_GetFloat64(ptr, Properties::pctCutout);
    }

    PVSystem& pctCutout(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctCutout, value);
        return *this;
    }

    /// An XYCurve object, previously defined, that describes the PER UNIT efficiency vs PER UNIT of rated kVA for the inverter. Inverter output power is discounted by the multiplier obtained from this curve.
    ///
    /// DSS property name: EffCurve, DSS property index: 14
    ///
    string EffCurve()
    {
        return get_prop_string(Properties::EffCurve);
    }

    PVSystem& EffCurve(const string &value)
    {
        set_string(Properties::EffCurve, value);
        return *this;
    }

    PVSystem& EffCurve(dss::obj::XYcurve &value)
    {
        set_obj(Properties::EffCurve, value);
        return *this;
    }

    /// An XYCurve object, previously defined, that describes the PER UNIT efficiency vs PER UNIT of rated kVA for the inverter. Inverter output power is discounted by the multiplier obtained from this curve.
    ///
    /// DSS property name: EffCurve, DSS property index: 14
    ///
    dss::obj::XYcurve EffCurve_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::EffCurve);
    }

    PVSystem& EffCurve_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::EffCurve, value);
        return *this;
    }

    /// An XYCurve object, previously defined, that describes the PV array PER UNIT Pmpp vs Temperature curve. Temperature units must agree with the Temperature property and the Temperature shapes used for simulations. The Pmpp values are specified in per unit of the Pmpp value for 1 kW/sq-m irradiance. The value for the temperature at which Pmpp is defined should be 1.0. The net array power is determined by the irradiance * Pmpp * f(Temperature)
    ///
    /// DSS property name: P-TCurve, DSS property index: 15
    ///
    string PTCurve()
    {
        return get_prop_string(Properties::PTCurve);
    }

    PVSystem& PTCurve(const string &value)
    {
        set_string(Properties::PTCurve, value);
        return *this;
    }

    PVSystem& PTCurve(dss::obj::XYcurve &value)
    {
        set_obj(Properties::PTCurve, value);
        return *this;
    }

    /// An XYCurve object, previously defined, that describes the PV array PER UNIT Pmpp vs Temperature curve. Temperature units must agree with the Temperature property and the Temperature shapes used for simulations. The Pmpp values are specified in per unit of the Pmpp value for 1 kW/sq-m irradiance. The value for the temperature at which Pmpp is defined should be 1.0. The net array power is determined by the irradiance * Pmpp * f(Temperature)
    ///
    /// DSS property name: P-TCurve, DSS property index: 15
    ///
    dss::obj::XYcurve PTCurve_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::PTCurve);
    }

    PVSystem& PTCurve_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::PTCurve, value);
        return *this;
    }

    /// Equivalent percent internal resistance, ohms. Default is 50%. Placed in series with internal voltage source for harmonics and dynamics modes. (Limits fault current to about 2 pu if not current limited -- see LimitCurrent)
    ///
    /// DSS property name: %R, DSS property index: 16
    ///
    double pctR()
    {
        return Obj_GetFloat64(ptr, Properties::pctR);
    }

    PVSystem& pctR(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctR, value);
        return *this;
    }

    /// Equivalent percent internal reactance, ohms. Default is 0%. Placed in series with internal voltage source for harmonics and dynamics modes.
    ///
    /// DSS property name: %X, DSS property index: 17
    ///
    double pctX()
    {
        return Obj_GetFloat64(ptr, Properties::pctX);
    }

    PVSystem& pctX(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctX, value);
        return *this;
    }

    /// Integer code (default=1) for the model to use for power output variation with voltage. Valid values are:
    ///
    /// 1:PVSystem element injects a CONSTANT kW at specified power factor.
    /// 2:PVSystem element is modeled as a CONSTANT ADMITTANCE.
    /// 3:Compute load injection from User-written Model.
    ///
    /// DSS property name: model, DSS property index: 18
    ///
    int32_t model()
    {
        return Obj_GetInt32(ptr, Properties::model);
    }

    PVSystem& model(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::model, value);
        return *this;
    }

    /// Default = 0.90.  Minimum per unit voltage for which the Model is assumed to apply. Below this value, the load model reverts to a constant impedance model except for Dynamics model. In Dynamics mode, the current magnitude is limited to the value the power flow would compute for this voltage.
    ///
    /// DSS property name: Vminpu, DSS property index: 19
    ///
    double Vminpu()
    {
        return Obj_GetFloat64(ptr, Properties::Vminpu);
    }

    PVSystem& Vminpu(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vminpu, value);
        return *this;
    }

    /// Default = 1.10.  Maximum per unit voltage for which the Model is assumed to apply. Above this value, the load model reverts to a constant impedance model.
    ///
    /// DSS property name: Vmaxpu, DSS property index: 20
    ///
    double Vmaxpu()
    {
        return Obj_GetFloat64(ptr, Properties::Vmaxpu);
    }

    PVSystem& Vmaxpu(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vmaxpu, value);
        return *this;
    }

    /// {Yes | No*} Default is No.  Force balanced current only for 3-phase PVSystems. Forces zero- and negative-sequence to zero.
    ///
    /// DSS property name: Balanced, DSS property index: 21
    ///
    bool Balanced()
    {
        return Obj_GetInt32(ptr, Properties::Balanced) != 0;
    }

    PVSystem& Balanced(bool value)
    {
        Obj_SetInt32(ptr, Properties::Balanced, value);
        return *this;
    }

    /// Limits current magnitude to Vminpu value for both 1-phase and 3-phase PVSystems similar to Generator Model 7. For 3-phase, limits the positive-sequence current but not the negative-sequence.
    ///
    /// DSS property name: LimitCurrent, DSS property index: 22
    ///
    bool LimitCurrent()
    {
        return Obj_GetInt32(ptr, Properties::LimitCurrent) != 0;
    }

    PVSystem& LimitCurrent(bool value)
    {
        Obj_SetInt32(ptr, Properties::LimitCurrent, value);
        return *this;
    }

    /// Dispatch shape to use for yearly simulations.  Must be previously defined as a Loadshape object. If this is not specified, the Daily dispatch shape, if any, is repeated during Yearly solution modes. In the default dispatch mode, the PVSystem element uses this loadshape to trigger State changes.
    ///
    /// DSS property name: yearly, DSS property index: 23
    ///
    string yearly()
    {
        return get_prop_string(Properties::yearly);
    }

    PVSystem& yearly(const string &value)
    {
        set_string(Properties::yearly, value);
        return *this;
    }

    PVSystem& yearly(dss::obj::LoadShape &value)
    {
        set_obj(Properties::yearly, value);
        return *this;
    }

    /// Dispatch shape to use for yearly simulations.  Must be previously defined as a Loadshape object. If this is not specified, the Daily dispatch shape, if any, is repeated during Yearly solution modes. In the default dispatch mode, the PVSystem element uses this loadshape to trigger State changes.
    ///
    /// DSS property name: yearly, DSS property index: 23
    ///
    dss::obj::LoadShape yearly_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::yearly);
    }

    PVSystem& yearly_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::yearly, value);
        return *this;
    }

    /// Dispatch shape to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically.  In the default dispatch mode, the PVSystem element uses this loadshape to trigger State changes.
    ///
    /// DSS property name: daily, DSS property index: 24
    ///
    string daily()
    {
        return get_prop_string(Properties::daily);
    }

    PVSystem& daily(const string &value)
    {
        set_string(Properties::daily, value);
        return *this;
    }

    PVSystem& daily(dss::obj::LoadShape &value)
    {
        set_obj(Properties::daily, value);
        return *this;
    }

    /// Dispatch shape to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically.  In the default dispatch mode, the PVSystem element uses this loadshape to trigger State changes.
    ///
    /// DSS property name: daily, DSS property index: 24
    ///
    dss::obj::LoadShape daily_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::daily);
    }

    PVSystem& daily_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::daily, value);
        return *this;
    }

    /// Load shape to use for duty cycle dispatch simulations such as for solar ramp rate studies. Must be previously defined as a Loadshape object. Typically would have time intervals of 1-5 seconds. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.
    ///
    /// DSS property name: duty, DSS property index: 25
    ///
    string duty()
    {
        return get_prop_string(Properties::duty);
    }

    PVSystem& duty(const string &value)
    {
        set_string(Properties::duty, value);
        return *this;
    }

    PVSystem& duty(dss::obj::LoadShape &value)
    {
        set_obj(Properties::duty, value);
        return *this;
    }

    /// Load shape to use for duty cycle dispatch simulations such as for solar ramp rate studies. Must be previously defined as a Loadshape object. Typically would have time intervals of 1-5 seconds. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.
    ///
    /// DSS property name: duty, DSS property index: 25
    ///
    dss::obj::LoadShape duty_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::duty);
    }

    PVSystem& duty_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::duty, value);
        return *this;
    }

    /// Temperature shape to use for yearly simulations.  Must be previously defined as a TShape object. If this is not specified, the Daily dispatch shape, if any, is repeated during Yearly solution modes. The PVSystem element uses this TShape to determine the Pmpp from the Pmpp vs T curve. Units must agree with the Pmpp vs T curve.
    ///
    /// DSS property name: Tyearly, DSS property index: 26
    ///
    string Tyearly()
    {
        return get_prop_string(Properties::Tyearly);
    }

    PVSystem& Tyearly(const string &value)
    {
        set_string(Properties::Tyearly, value);
        return *this;
    }

    PVSystem& Tyearly(dss::obj::TShape &value)
    {
        set_obj(Properties::Tyearly, value);
        return *this;
    }

    /// Temperature shape to use for yearly simulations.  Must be previously defined as a TShape object. If this is not specified, the Daily dispatch shape, if any, is repeated during Yearly solution modes. The PVSystem element uses this TShape to determine the Pmpp from the Pmpp vs T curve. Units must agree with the Pmpp vs T curve.
    ///
    /// DSS property name: Tyearly, DSS property index: 26
    ///
    dss::obj::TShape Tyearly_obj()
    {
        return get_obj<dss::obj::TShape>(Properties::Tyearly);
    }

    PVSystem& Tyearly_obj(dss::obj::TShape &value)
    {
        set_obj(Properties::Tyearly, value);
        return *this;
    }

    /// Temperature shape to use for daily simulations.  Must be previously defined as a TShape object of 24 hrs, typically.  The PVSystem element uses this TShape to determine the Pmpp from the Pmpp vs T curve. Units must agree with the Pmpp vs T curve.
    ///
    /// DSS property name: Tdaily, DSS property index: 27
    ///
    string Tdaily()
    {
        return get_prop_string(Properties::Tdaily);
    }

    PVSystem& Tdaily(const string &value)
    {
        set_string(Properties::Tdaily, value);
        return *this;
    }

    PVSystem& Tdaily(dss::obj::TShape &value)
    {
        set_obj(Properties::Tdaily, value);
        return *this;
    }

    /// Temperature shape to use for daily simulations.  Must be previously defined as a TShape object of 24 hrs, typically.  The PVSystem element uses this TShape to determine the Pmpp from the Pmpp vs T curve. Units must agree with the Pmpp vs T curve.
    ///
    /// DSS property name: Tdaily, DSS property index: 27
    ///
    dss::obj::TShape Tdaily_obj()
    {
        return get_obj<dss::obj::TShape>(Properties::Tdaily);
    }

    PVSystem& Tdaily_obj(dss::obj::TShape &value)
    {
        set_obj(Properties::Tdaily, value);
        return *this;
    }

    /// Temperature shape to use for duty cycle dispatch simulations such as for solar ramp rate studies. Must be previously defined as a TShape object. Typically would have time intervals of 1-5 seconds. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat. The PVSystem model uses this TShape to determine the Pmpp from the Pmpp vs T curve. Units must agree with the Pmpp vs T curve.
    ///
    /// DSS property name: Tduty, DSS property index: 28
    ///
    string Tduty()
    {
        return get_prop_string(Properties::Tduty);
    }

    PVSystem& Tduty(const string &value)
    {
        set_string(Properties::Tduty, value);
        return *this;
    }

    PVSystem& Tduty(dss::obj::TShape &value)
    {
        set_obj(Properties::Tduty, value);
        return *this;
    }

    /// Temperature shape to use for duty cycle dispatch simulations such as for solar ramp rate studies. Must be previously defined as a TShape object. Typically would have time intervals of 1-5 seconds. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat. The PVSystem model uses this TShape to determine the Pmpp from the Pmpp vs T curve. Units must agree with the Pmpp vs T curve.
    ///
    /// DSS property name: Tduty, DSS property index: 28
    ///
    dss::obj::TShape Tduty_obj()
    {
        return get_obj<dss::obj::TShape>(Properties::Tduty);
    }

    PVSystem& Tduty_obj(dss::obj::TShape &value)
    {
        set_obj(Properties::Tduty, value);
        return *this;
    }

    /// An arbitrary integer number representing the class of PVSystem element so that PVSystem values may be segregated by class.
    ///
    /// DSS property name: class, DSS property index: 29
    ///
    int32_t cls()
    {
        return Obj_GetInt32(ptr, Properties::cls);
    }

    PVSystem& cls(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::cls, value);
        return *this;
    }

    /// Name of DLL containing user-written model, which computes the terminal currents for Dynamics studies, overriding the default model.  Set to "none" to negate previous setting.
    ///
    /// DSS property name: UserModel, DSS property index: 30
    ///
    string UserModel()
    {
        return get_prop_string(Properties::UserModel);
    }

    PVSystem& UserModel(const string &value)
    {
        set_string(Properties::UserModel, value);
        return *this;
    }

    PVSystem& UserModel(const char* value)
    {
        set_string(Properties::UserModel, value);
        return *this;
    }

    /// String (in quotes or parentheses) that gets passed to user-written model for defining the data required for that model.
    ///
    /// DSS property name: UserData, DSS property index: 31
    ///
    string UserData()
    {
        return get_prop_string(Properties::UserData);
    }

    PVSystem& UserData(const string &value)
    {
        set_string(Properties::UserData, value);
        return *this;
    }

    PVSystem& UserData(const char* value)
    {
        set_string(Properties::UserData, value);
        return *this;
    }

    /// {Yes | No }  Default is no.  Turn this on to capture the progress of the PVSystem model for each iteration.  Creates a separate file for each PVSystem element named "PVSystem_name.CSV".
    ///
    /// DSS property name: debugtrace, DSS property index: 32
    ///
    bool debugtrace()
    {
        return Obj_GetInt32(ptr, Properties::debugtrace) != 0;
    }

    PVSystem& debugtrace(bool value)
    {
        Obj_SetInt32(ptr, Properties::debugtrace, value);
        return *this;
    }

    /// Boolean variable (Yes|No) or (True|False). Defaults to False which indicates that the reactive power generation/absorption does not respect the inverter status.When set to True, the PVSystem reactive power generation/absorption will cease when the inverter status is off, due to panel kW dropping below %Cutout.  The reactive power generation/absorption will begin again when the panel kW is above %Cutin.  When set to False, the PVSystem will generate/absorb reactive power regardless of the status of the inverter.
    ///
    /// DSS property name: VarFollowInverter, DSS property index: 33
    ///
    bool VarFollowInverter()
    {
        return Obj_GetInt32(ptr, Properties::VarFollowInverter) != 0;
    }

    PVSystem& VarFollowInverter(bool value)
    {
        Obj_SetInt32(ptr, Properties::VarFollowInverter, value);
        return *this;
    }

    /// Starting time offset [hours] into the duty cycle shape for this PVSystem, defaults to 0
    ///
    /// DSS property name: DutyStart, DSS property index: 34
    ///
    double DutyStart()
    {
        return Obj_GetFloat64(ptr, Properties::DutyStart);
    }

    PVSystem& DutyStart(double value)
    {
        Obj_SetFloat64(ptr, Properties::DutyStart, value);
        return *this;
    }

    /// {Yes/No*/True/False} Set inverter to watt priority instead of the default var priority
    ///
    /// DSS property name: WattPriority, DSS property index: 35
    ///
    bool WattPriority()
    {
        return Obj_GetInt32(ptr, Properties::WattPriority) != 0;
    }

    PVSystem& WattPriority(bool value)
    {
        Obj_SetInt32(ptr, Properties::WattPriority, value);
        return *this;
    }

    /// {Yes/No*/True/False} Set inverter to operate with PF priority when in constant PF mode. If "Yes", value assigned to "WattPriority" is neglected. If controlled by an InvControl with either Volt-Var or DRC or both functions activated, PF priority is neglected and "WattPriority" is considered. Default = No.
    ///
    /// DSS property name: PFPriority, DSS property index: 36
    ///
    bool PFPriority()
    {
        return Obj_GetInt32(ptr, Properties::PFPriority) != 0;
    }

    PVSystem& PFPriority(bool value)
    {
        Obj_SetInt32(ptr, Properties::PFPriority, value);
        return *this;
    }

    /// Minimum active power as percentage of Pmpp under which there is no vars production/absorption.
    ///
    /// DSS property name: %PminNoVars, DSS property index: 37
    ///
    double pctPminNoVars()
    {
        return Obj_GetFloat64(ptr, Properties::pctPminNoVars);
    }

    PVSystem& pctPminNoVars(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctPminNoVars, value);
        return *this;
    }

    /// Minimum active power as percentage of Pmpp that allows the inverter to produce/absorb reactive power up to its kvarMax or kvarMaxAbs.
    ///
    /// DSS property name: %PminkvarMax, DSS property index: 38
    ///
    double pctPminkvarMax()
    {
        return Obj_GetFloat64(ptr, Properties::pctPminkvarMax);
    }

    PVSystem& pctPminkvarMax(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctPminkvarMax, value);
        return *this;
    }

    /// Indicates the maximum reactive power GENERATION (un-signed numerical variable in kvar) for the inverter (as an un-signed value). Defaults to kVA rating of the inverter.
    ///
    /// DSS property name: kvarMax, DSS property index: 39
    ///
    double kvarMax()
    {
        return Obj_GetFloat64(ptr, Properties::kvarMax);
    }

    PVSystem& kvarMax(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvarMax, value);
        return *this;
    }

    /// Indicates the maximum reactive power ABSORPTION (un-signed numerical variable in kvar) for the inverter (as an un-signed value). Defaults to kVA rating of the inverter.
    ///
    /// DSS property name: kvarMaxAbs, DSS property index: 40
    ///
    double kvarMaxAbs()
    {
        return Obj_GetFloat64(ptr, Properties::kvarMaxAbs);
    }

    PVSystem& kvarMaxAbs(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvarMaxAbs, value);
        return *this;
    }

    /// Name of harmonic voltage or current spectrum for this PVSystem element. A harmonic voltage source is assumed for the inverter. Default value is "default", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 41
    ///
    string spectrum()
    {
        return get_prop_string(Properties::spectrum);
    }

    PVSystem& spectrum(const string &value)
    {
        set_string(Properties::spectrum, value);
        return *this;
    }

    PVSystem& spectrum(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Name of harmonic voltage or current spectrum for this PVSystem element. A harmonic voltage source is assumed for the inverter. Default value is "default", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 41
    ///
    dss::obj::Spectrum spectrum_obj()
    {
        return get_obj<dss::obj::Spectrum>(Properties::spectrum);
    }

    PVSystem& spectrum_obj(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 42
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    PVSystem& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 43
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    PVSystem& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 44
    ///
    PVSystem& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 44
    ///
    PVSystem& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class UPFC: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 35;
    struct Properties
    {
        enum {
            bus1 = 1,
            bus2 = 2,
            refkv = 3,
            pf = 4,
            frequency = 5,
            phases = 6,
            Xs = 7,
            Tol1 = 8,
            Mode = 9,
            VpqMax = 10,
            LossCurve = 11,
            VHLimit = 12,
            VLLimit = 13,
            CLimit = 14,
            refkv2 = 15,
            kvarLimit = 16,
            spectrum = 17,
            basefreq = 18,
            enabled = 19,
            like = 20,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    UPFC(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    UPFC(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the UPFC element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    UPFC(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the UPFC element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    UPFC& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    UPFC& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name of bus to which the input terminal (1) is connected.
    /// bus1=busname.1.3
    /// bus1=busname.1.2.3
    ///
    /// DSS property name: bus1, DSS property index: 1
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    UPFC& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    UPFC& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Name of bus to which the output terminal (2) is connected.
    /// bus2=busname.1.2
    /// bus2=busname.1.2.3
    ///
    /// DSS property name: bus2, DSS property index: 2
    ///
    string bus2()
    {
        return get_prop_string(Properties::bus2);
    }

    UPFC& bus2(const string &value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    UPFC& bus2(const char* value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    /// Base Voltage expected at the output of the UPFC
    ///
    /// "refkv=0.24"
    ///
    /// DSS property name: refkv, DSS property index: 3
    ///
    double refkv()
    {
        return Obj_GetFloat64(ptr, Properties::refkv);
    }

    UPFC& refkv(double value)
    {
        Obj_SetFloat64(ptr, Properties::refkv, value);
        return *this;
    }

    /// Power factor target at the input terminal.
    ///
    /// DSS property name: pf, DSS property index: 4
    ///
    double pf()
    {
        return Obj_GetFloat64(ptr, Properties::pf);
    }

    UPFC& pf(double value)
    {
        Obj_SetFloat64(ptr, Properties::pf, value);
        return *this;
    }

    /// UPFC working frequency.  Defaults to system default base frequency.
    ///
    /// DSS property name: frequency, DSS property index: 5
    ///
    double frequency()
    {
        return Obj_GetFloat64(ptr, Properties::frequency);
    }

    UPFC& frequency(double value)
    {
        Obj_SetFloat64(ptr, Properties::frequency, value);
        return *this;
    }

    /// Number of phases.  Defaults to 1 phase (2 terminals, 1 conductor per terminal).
    ///
    /// DSS property name: phases, DSS property index: 6
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    UPFC& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Reactance of the series transformer of the UPFC, ohms (default=0.7540 ... 2 mH)
    ///
    /// DSS property name: Xs, DSS property index: 7
    ///
    double Xs()
    {
        return Obj_GetFloat64(ptr, Properties::Xs);
    }

    UPFC& Xs(double value)
    {
        Obj_SetFloat64(ptr, Properties::Xs, value);
        return *this;
    }

    /// Tolerance in pu for the series PI controller
    /// Tol1=0.02 is the format used to define 2% tolerance (Default=2%)
    ///
    /// DSS property name: Tol1, DSS property index: 8
    ///
    double Tol1()
    {
        return Obj_GetFloat64(ptr, Properties::Tol1);
    }

    UPFC& Tol1(double value)
    {
        Obj_SetFloat64(ptr, Properties::Tol1, value);
        return *this;
    }

    /// Integer used to define the control mode of the UPFC:
    ///
    /// 0 = Off,
    /// 1 = Voltage regulator,
    /// 2 = Phase angle regulator,
    /// 3 = Dual mode
    /// 4 = It is a control mode where the user can set two different set points to create a secure GAP, these references must be defined in the parameters RefkV and RefkV2. The only restriction when setting these values is that RefkV must be higher than RefkV2.
    /// 5 = In this mode the user can define the same GAP using two set points as in control mode 4. The only difference between mode 5 and mode 4 is that in mode 5, the UPFC controller performs dual control actions just as in control mode 3
    ///
    /// DSS property name: Mode, DSS property index: 9
    ///
    int32_t Mode()
    {
        return Obj_GetInt32(ptr, Properties::Mode);
    }

    UPFC& Mode(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Mode, value);
        return *this;
    }

    /// Maximum voltage (in volts) delivered by the series voltage source (Default = 24 V)
    ///
    /// DSS property name: VpqMax, DSS property index: 10
    ///
    double VpqMax()
    {
        return Obj_GetFloat64(ptr, Properties::VpqMax);
    }

    UPFC& VpqMax(double value)
    {
        Obj_SetFloat64(ptr, Properties::VpqMax, value);
        return *this;
    }

    /// Name of the XYCurve for describing the losses behavior as a function of the voltage at the input of the UPFC
    ///
    /// DSS property name: LossCurve, DSS property index: 11
    ///
    string LossCurve()
    {
        return get_prop_string(Properties::LossCurve);
    }

    UPFC& LossCurve(const string &value)
    {
        set_string(Properties::LossCurve, value);
        return *this;
    }

    UPFC& LossCurve(dss::obj::XYcurve &value)
    {
        set_obj(Properties::LossCurve, value);
        return *this;
    }

    /// Name of the XYCurve for describing the losses behavior as a function of the voltage at the input of the UPFC
    ///
    /// DSS property name: LossCurve, DSS property index: 11
    ///
    dss::obj::XYcurve LossCurve_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::LossCurve);
    }

    UPFC& LossCurve_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::LossCurve, value);
        return *this;
    }

    /// High limit for the voltage at the input of the UPFC, if the voltage is above this value the UPFC turns off. This value is specified in Volts (default 300 V)
    ///
    /// DSS property name: VHLimit, DSS property index: 12
    ///
    double VHLimit()
    {
        return Obj_GetFloat64(ptr, Properties::VHLimit);
    }

    UPFC& VHLimit(double value)
    {
        Obj_SetFloat64(ptr, Properties::VHLimit, value);
        return *this;
    }

    /// low limit for the voltage at the input of the UPFC, if voltage is below this value the UPFC turns off. This value is specified in Volts (default 125 V)
    ///
    /// DSS property name: VLLimit, DSS property index: 13
    ///
    double VLLimit()
    {
        return Obj_GetFloat64(ptr, Properties::VLLimit);
    }

    UPFC& VLLimit(double value)
    {
        Obj_SetFloat64(ptr, Properties::VLLimit, value);
        return *this;
    }

    /// Current Limit for the UPFC, if the current passing through the UPFC is higher than this value the UPFC turns off. This value is specified in Amps (Default 265 A)
    ///
    /// DSS property name: CLimit, DSS property index: 14
    ///
    double CLimit()
    {
        return Obj_GetFloat64(ptr, Properties::CLimit);
    }

    UPFC& CLimit(double value)
    {
        Obj_SetFloat64(ptr, Properties::CLimit, value);
        return *this;
    }

    /// Base Voltage expected at the output of the UPFC for control modes 4 and 5.
    ///
    /// This reference must be lower than refkv, see control modes 4 and 5 for details
    ///
    /// DSS property name: refkv2, DSS property index: 15
    ///
    double refkv2()
    {
        return Obj_GetFloat64(ptr, Properties::refkv2);
    }

    UPFC& refkv2(double value)
    {
        Obj_SetFloat64(ptr, Properties::refkv2, value);
        return *this;
    }

    /// Maximum amount of reactive power (kvar) that can be absorved by the UPFC (Default = 5)
    ///
    /// DSS property name: kvarLimit, DSS property index: 16
    ///
    double kvarLimit()
    {
        return Obj_GetFloat64(ptr, Properties::kvarLimit);
    }

    UPFC& kvarLimit(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvarLimit, value);
        return *this;
    }

    /// Name of harmonic spectrum for this source.  Default is "defaultUPFC", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 17
    ///
    string spectrum()
    {
        return get_prop_string(Properties::spectrum);
    }

    UPFC& spectrum(const string &value)
    {
        set_string(Properties::spectrum, value);
        return *this;
    }

    UPFC& spectrum(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Name of harmonic spectrum for this source.  Default is "defaultUPFC", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 17
    ///
    dss::obj::Spectrum spectrum_obj()
    {
        return get_obj<dss::obj::Spectrum>(Properties::spectrum);
    }

    UPFC& spectrum_obj(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 18
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    UPFC& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 19
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    UPFC& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 20
    ///
    UPFC& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 20
    ///
    UPFC& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class UPFCControl: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 36;
    struct Properties
    {
        enum {
            UPFCList = 1,
            basefreq = 2,
            enabled = 3,
            like = 4,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    UPFCControl(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    UPFCControl(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the UPFCControl element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    UPFCControl(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the UPFCControl element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    UPFCControl& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    UPFCControl& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// The list of all the UPFC devices to be controlled by this controller, If left empty, this control will apply for all UPFCs in the model.
    ///
    /// DSS property name: UPFCList, DSS property index: 1
    ///
    strings UPFCList()
    {
        return get_array<strings>(Properties::UPFCList);
    }

    UPFCControl& UPFCList(strings &value)
    {
        set_array<strings>(Properties::UPFCList, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 2
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    UPFCControl& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 3
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    UPFCControl& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 4
    ///
    UPFCControl& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 4
    ///
    UPFCControl& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class ESPVLControl: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 37;
    struct Properties
    {
        enum {
            Element = 1,
            Terminal = 2,
            Type = 3,
            kWBand = 4,
            kvarlimit = 5,
            LocalControlList = 6,
            LocalControlWeights = 7,
            PVSystemList = 8,
            PVSystemWeights = 9,
            StorageList = 10,
            StorageWeights = 11,
            basefreq = 12,
            enabled = 13,
            like = 14,
        };
    };

    // Class-specific enumerations

        ///
        /// ESPVLControl: Type (DSS enumeration for ESPVLControl)
        ///
    enum class ESPVLControlType: int32_t
    {
        SystemController = 1, ///< SystemController
        LocalController = 2 ///< LocalController
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    ESPVLControl(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    ESPVLControl(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the ESPVLControl element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    ESPVLControl(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the ESPVLControl element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    ESPVLControl& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    ESPVLControl& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Full object name of the circuit element, typically a line or transformer, which the control is monitoring. There is no default; must be specified.
    ///
    /// DSS property name: Element, DSS property index: 1
    ///
    string Element()
    {
        return get_prop_string(Properties::Element);
    }

    ESPVLControl& Element(const string &value)
    {
        set_string(Properties::Element, value);
        return *this;
    }

    ESPVLControl& Element(dss::obj::DSSObj &value)
    {
        set_obj(Properties::Element, value);
        return *this;
    }

    /// Full object name of the circuit element, typically a line or transformer, which the control is monitoring. There is no default; must be specified.
    ///
    /// DSS property name: Element, DSS property index: 1
    ///
    dss::obj::DSSObj Element_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::Element);
    }

    ESPVLControl& Element_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::Element, value);
        return *this;
    }

    /// Number of the terminal of the circuit element to which the ESPVLControl control is connected. 1 or 2, typically.  Default is 1. Make sure you have the direction on the power matching the sign of kWLimit.
    ///
    /// DSS property name: Terminal, DSS property index: 2
    ///
    int32_t Terminal()
    {
        return Obj_GetInt32(ptr, Properties::Terminal);
    }

    ESPVLControl& Terminal(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Terminal, value);
        return *this;
    }

    /// Type of controller.  1= System Controller; 2= Local controller.
    ///
    /// DSS property name: Type, DSS property index: 3
    ///
    ESPVLControlType Type()
    {
        return ESPVLControlType(Obj_GetInt32(ptr, Properties::Type));
    }

    ESPVLControl& Type(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Type, value);
        return *this;
    }

    ESPVLControl& Type(ESPVLControlType value)
    {
        Obj_SetInt32(ptr, Properties::Type, int32_t(value));
        return *this;
    }

    ESPVLControl& Type(const string &value)
    {
        set_string(Properties::Type, value);
        return *this;
    }

    ESPVLControl& Type(const char *value)
    {
        set_string(Properties::Type, value);
        return *this;
    }

    /// Type of controller.  1= System Controller; 2= Local controller.
    ///
    /// DSS property name: Type, DSS property index: 3
    ///
    string Type_str()
    {
        return get_prop_string(Properties::Type);
    }

    /// Type of controller.  1= System Controller; 2= Local controller.
    ///
    /// DSS property name: Type, DSS property index: 3
    ///
    ESPVLControl& Type_str(const string &value)
    {
        set_string(Properties::Type, value);
        return *this;
    }

    /// Bandwidth (kW) of the dead band around the target limit.No dispatch changes are attempted if the power in the monitored terminal stays within this band.
    ///
    /// DSS property name: kWBand, DSS property index: 4
    ///
    double kWBand()
    {
        return Obj_GetFloat64(ptr, Properties::kWBand);
    }

    ESPVLControl& kWBand(double value)
    {
        Obj_SetFloat64(ptr, Properties::kWBand, value);
        return *this;
    }

    /// Max kvar to be delivered through the element.  Uses same dead band as kW.
    ///
    /// DSS property name: kvarlimit, DSS property index: 5
    ///
    double kvarlimit()
    {
        return Obj_GetFloat64(ptr, Properties::kvarlimit);
    }

    ESPVLControl& kvarlimit(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvarlimit, value);
        return *this;
    }

    /// Array list of ESPVLControl local controller objects to be dispatched by System Controller. If not specified, all ESPVLControl devices with type=local in the circuit not attached to another controller are assumed to be part of this controller's fleet.
    ///
    /// DSS property name: LocalControlList, DSS property index: 6
    ///
    strings LocalControlList()
    {
        return get_array<strings>(Properties::LocalControlList);
    }

    ESPVLControl& LocalControlList(strings &value)
    {
        set_array<strings>(Properties::LocalControlList, value);
        return *this;
    }

    /// Array of proportional weights corresponding to each ESPVLControl local controller in the LocalControlList.
    ///
    /// DSS property name: LocalControlWeights, DSS property index: 7
    ///
    VectorXd LocalControlWeights()
    {
        return get_array<VectorXd>(Properties::LocalControlWeights);
    }

    ESPVLControl& LocalControlWeights(VectorXd &value)
    {
        set_array<VectorXd>(Properties::LocalControlWeights, value);
        return *this;
    }

    /// Array list of PVSystem objects to be dispatched by a Local Controller.
    ///
    /// DSS property name: PVSystemList, DSS property index: 8
    ///
    strings PVSystemList()
    {
        return get_array<strings>(Properties::PVSystemList);
    }

    ESPVLControl& PVSystemList(strings &value)
    {
        set_array<strings>(Properties::PVSystemList, value);
        return *this;
    }

    /// Array of proportional weights corresponding to each PVSystem in the PVSystemList.
    ///
    /// DSS property name: PVSystemWeights, DSS property index: 9
    ///
    VectorXd PVSystemWeights()
    {
        return get_array<VectorXd>(Properties::PVSystemWeights);
    }

    ESPVLControl& PVSystemWeights(VectorXd &value)
    {
        set_array<VectorXd>(Properties::PVSystemWeights, value);
        return *this;
    }

    /// Array list of Storage objects to be dispatched by Local Controller.
    ///
    /// DSS property name: StorageList, DSS property index: 10
    ///
    strings StorageList()
    {
        return get_array<strings>(Properties::StorageList);
    }

    ESPVLControl& StorageList(strings &value)
    {
        set_array<strings>(Properties::StorageList, value);
        return *this;
    }

    /// Array of proportional weights corresponding to each Storage object in the StorageControlList.
    ///
    /// DSS property name: StorageWeights, DSS property index: 11
    ///
    VectorXd StorageWeights()
    {
        return get_array<VectorXd>(Properties::StorageWeights);
    }

    ESPVLControl& StorageWeights(VectorXd &value)
    {
        set_array<VectorXd>(Properties::StorageWeights, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 12
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    ESPVLControl& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 13
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    ESPVLControl& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 14
    ///
    ESPVLControl& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 14
    ///
    ESPVLControl& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class IndMach012: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 38;
    struct Properties
    {
        enum {
            phases = 1,
            bus1 = 2,
            kv = 3,
            kW = 4,
            pf = 5,
            conn = 6,
            kVA = 7,
            H = 8,
            D = 9,
            puRs = 10,
            puXs = 11,
            puRr = 12,
            puXr = 13,
            puXm = 14,
            Slip = 15,
            MaxSlip = 16,
            SlipOption = 17,
            Yearly = 18,
            Daily = 19,
            Duty = 20,
            Debugtrace = 21,
            spectrum = 22,
            basefreq = 23,
            enabled = 24,
            like = 25,
        };
    };

    // Class-specific enumerations

        ///
        /// IndMach012: Slip Option (DSS enumeration for IndMach012)
        ///
    enum class IndMach012SlipOption: int32_t
    {
        VariableSlip = 0, ///< VariableSlip
        FixedSlip = 1 ///< FixedSlip
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    IndMach012(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    IndMach012(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the IndMach012 element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    IndMach012(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the IndMach012 element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    IndMach012& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    IndMach012& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of Phases, this Induction Machine.
    ///
    /// DSS property name: phases, DSS property index: 1
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    IndMach012& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Bus to which the Induction Machine is connected.  May include specific node specification.
    ///
    /// DSS property name: bus1, DSS property index: 2
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    IndMach012& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    IndMach012& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Nominal rated (1.0 per unit) voltage, kV. For 2- and 3-phase machines, specify phase-phase kV. Otherwise, specify actual kV across each branch of the machine. If wye (star), specify phase-neutral kV. If delta or phase-phase connected, specify phase-phase kV.
    ///
    /// DSS property name: kv, DSS property index: 3
    ///
    double kv()
    {
        return Obj_GetFloat64(ptr, Properties::kv);
    }

    IndMach012& kv(double value)
    {
        Obj_SetFloat64(ptr, Properties::kv, value);
        return *this;
    }

    /// Shaft Power, kW, for the Induction Machine.  A positive value denotes power for a load.
    /// Negative value denotes an induction generator.
    ///
    /// DSS property name: kW, DSS property index: 4
    ///
    double kW()
    {
        return Obj_GetFloat64(ptr, Properties::kW);
    }

    IndMach012& kW(double value)
    {
        Obj_SetFloat64(ptr, Properties::kW, value);
        return *this;
    }

    /// [Read Only] Present power factor for the machine.
    ///
    /// DSS property name: pf, DSS property index: 5
    ///
    double pf()
    {
        return Obj_GetFloat64(ptr, Properties::pf);
    }

    IndMach012& pf(double value)
    {
        Obj_SetFloat64(ptr, Properties::pf, value);
        return *this;
    }

    /// Connection of stator: Delta or Wye. Default is Delta.
    ///
    /// DSS property name: conn, DSS property index: 6
    ///
    Connection conn()
    {
        return Connection(Obj_GetInt32(ptr, Properties::conn));
    }

    IndMach012& conn(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::conn, value);
        return *this;
    }

    IndMach012& conn(Connection value)
    {
        Obj_SetInt32(ptr, Properties::conn, int32_t(value));
        return *this;
    }

    IndMach012& conn(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    IndMach012& conn(const char *value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// Connection of stator: Delta or Wye. Default is Delta.
    ///
    /// DSS property name: conn, DSS property index: 6
    ///
    string conn_str()
    {
        return get_prop_string(Properties::conn);
    }

    /// Connection of stator: Delta or Wye. Default is Delta.
    ///
    /// DSS property name: conn, DSS property index: 6
    ///
    IndMach012& conn_str(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// Rated kVA for the machine.
    ///
    /// DSS property name: kVA, DSS property index: 7
    ///
    double kVA()
    {
        return Obj_GetFloat64(ptr, Properties::kVA);
    }

    IndMach012& kVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::kVA, value);
        return *this;
    }

    /// Per unit mass constant of the machine.  MW-sec/MVA.  Default is 1.0.
    ///
    /// DSS property name: H, DSS property index: 8
    ///
    double H()
    {
        return Obj_GetFloat64(ptr, Properties::H);
    }

    IndMach012& H(double value)
    {
        Obj_SetFloat64(ptr, Properties::H, value);
        return *this;
    }

    /// Damping constant.  Usual range is 0 to 4. Default is 1.0.  Adjust to get damping in Dynamics mode,
    ///
    /// DSS property name: D, DSS property index: 9
    ///
    double D()
    {
        return Obj_GetFloat64(ptr, Properties::D);
    }

    IndMach012& D(double value)
    {
        Obj_SetFloat64(ptr, Properties::D, value);
        return *this;
    }

    /// Per unit stator resistance. Default is 0.0053.
    ///
    /// DSS property name: puRs, DSS property index: 10
    ///
    double puRs()
    {
        return Obj_GetFloat64(ptr, Properties::puRs);
    }

    IndMach012& puRs(double value)
    {
        Obj_SetFloat64(ptr, Properties::puRs, value);
        return *this;
    }

    /// Per unit stator leakage reactance. Default is 0.106.
    ///
    /// DSS property name: puXs, DSS property index: 11
    ///
    double puXs()
    {
        return Obj_GetFloat64(ptr, Properties::puXs);
    }

    IndMach012& puXs(double value)
    {
        Obj_SetFloat64(ptr, Properties::puXs, value);
        return *this;
    }

    /// Per unit rotor  resistance. Default is 0.007.
    ///
    /// DSS property name: puRr, DSS property index: 12
    ///
    double puRr()
    {
        return Obj_GetFloat64(ptr, Properties::puRr);
    }

    IndMach012& puRr(double value)
    {
        Obj_SetFloat64(ptr, Properties::puRr, value);
        return *this;
    }

    /// Per unit rotor leakage reactance. Default is 0.12.
    ///
    /// DSS property name: puXr, DSS property index: 13
    ///
    double puXr()
    {
        return Obj_GetFloat64(ptr, Properties::puXr);
    }

    IndMach012& puXr(double value)
    {
        Obj_SetFloat64(ptr, Properties::puXr, value);
        return *this;
    }

    /// Per unit magnetizing reactance.Default is 4.0.
    ///
    /// DSS property name: puXm, DSS property index: 14
    ///
    double puXm()
    {
        return Obj_GetFloat64(ptr, Properties::puXm);
    }

    IndMach012& puXm(double value)
    {
        Obj_SetFloat64(ptr, Properties::puXm, value);
        return *this;
    }

    /// Initial slip value. Default is 0.007
    ///
    /// DSS property name: Slip, DSS property index: 15
    ///
    double Slip()
    {
        return Obj_GetFloat64(ptr, Properties::Slip);
    }

    IndMach012& Slip(double value)
    {
        Obj_SetFloat64(ptr, Properties::Slip, value);
        return *this;
    }

    /// Max slip value to allow. Default is 0.1. Set this before setting slip.
    ///
    /// DSS property name: MaxSlip, DSS property index: 16
    ///
    double MaxSlip()
    {
        return Obj_GetFloat64(ptr, Properties::MaxSlip);
    }

    IndMach012& MaxSlip(double value)
    {
        Obj_SetFloat64(ptr, Properties::MaxSlip, value);
        return *this;
    }

    /// Option for slip model. One of {fixedslip | variableslip*  }
    ///
    /// DSS property name: SlipOption, DSS property index: 17
    ///
    IndMach012SlipOption SlipOption()
    {
        return IndMach012SlipOption(Obj_GetInt32(ptr, Properties::SlipOption));
    }

    IndMach012& SlipOption(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::SlipOption, value);
        return *this;
    }

    IndMach012& SlipOption(IndMach012SlipOption value)
    {
        Obj_SetInt32(ptr, Properties::SlipOption, int32_t(value));
        return *this;
    }

    IndMach012& SlipOption(const string &value)
    {
        set_string(Properties::SlipOption, value);
        return *this;
    }

    IndMach012& SlipOption(const char *value)
    {
        set_string(Properties::SlipOption, value);
        return *this;
    }

    /// Option for slip model. One of {fixedslip | variableslip*  }
    ///
    /// DSS property name: SlipOption, DSS property index: 17
    ///
    string SlipOption_str()
    {
        return get_prop_string(Properties::SlipOption);
    }

    /// Option for slip model. One of {fixedslip | variableslip*  }
    ///
    /// DSS property name: SlipOption, DSS property index: 17
    ///
    IndMach012& SlipOption_str(const string &value)
    {
        set_string(Properties::SlipOption, value);
        return *this;
    }

    /// LOADSHAPE object to use for yearly simulations.  Must be previously defined as a Loadshape object. Is set to the Daily load shape  when Daily is defined.  The daily load shape is repeated in this case. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. The default is no variation.
    ///
    /// DSS property name: Yearly, DSS property index: 18
    ///
    string Yearly()
    {
        return get_prop_string(Properties::Yearly);
    }

    IndMach012& Yearly(const string &value)
    {
        set_string(Properties::Yearly, value);
        return *this;
    }

    IndMach012& Yearly(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Yearly, value);
        return *this;
    }

    /// LOADSHAPE object to use for yearly simulations.  Must be previously defined as a Loadshape object. Is set to the Daily load shape  when Daily is defined.  The daily load shape is repeated in this case. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. The default is no variation.
    ///
    /// DSS property name: Yearly, DSS property index: 18
    ///
    dss::obj::LoadShape Yearly_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::Yearly);
    }

    IndMach012& Yearly_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Yearly, value);
        return *this;
    }

    /// LOADSHAPE object to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Default is no variation (constant) if not defined. Side effect: Sets Yearly load shape if not already defined.
    ///
    /// DSS property name: Daily, DSS property index: 19
    ///
    string Daily()
    {
        return get_prop_string(Properties::Daily);
    }

    IndMach012& Daily(const string &value)
    {
        set_string(Properties::Daily, value);
        return *this;
    }

    IndMach012& Daily(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Daily, value);
        return *this;
    }

    /// LOADSHAPE object to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Default is no variation (constant) if not defined. Side effect: Sets Yearly load shape if not already defined.
    ///
    /// DSS property name: Daily, DSS property index: 19
    ///
    dss::obj::LoadShape Daily_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::Daily);
    }

    IndMach012& Daily_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Daily, value);
        return *this;
    }

    /// LOADSHAPE object to use for duty cycle simulations.  Must be previously defined as a Loadshape object.  Typically would have time intervals less than 1 hr. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.Set to NONE to reset to no loadahape. Set Status=Fixed to ignore Loadshape designation.  Defaults to Daily curve If not specified.
    ///
    /// DSS property name: Duty, DSS property index: 20
    ///
    string Duty()
    {
        return get_prop_string(Properties::Duty);
    }

    IndMach012& Duty(const string &value)
    {
        set_string(Properties::Duty, value);
        return *this;
    }

    IndMach012& Duty(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Duty, value);
        return *this;
    }

    /// LOADSHAPE object to use for duty cycle simulations.  Must be previously defined as a Loadshape object.  Typically would have time intervals less than 1 hr. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.Set to NONE to reset to no loadahape. Set Status=Fixed to ignore Loadshape designation.  Defaults to Daily curve If not specified.
    ///
    /// DSS property name: Duty, DSS property index: 20
    ///
    dss::obj::LoadShape Duty_obj()
    {
        return get_obj<dss::obj::LoadShape>(Properties::Duty);
    }

    IndMach012& Duty_obj(dss::obj::LoadShape &value)
    {
        set_obj(Properties::Duty, value);
        return *this;
    }

    /// [Yes | No*] Write DebugTrace file.
    ///
    /// DSS property name: Debugtrace, DSS property index: 21
    ///
    bool Debugtrace()
    {
        return Obj_GetInt32(ptr, Properties::Debugtrace) != 0;
    }

    IndMach012& Debugtrace(bool value)
    {
        Obj_SetInt32(ptr, Properties::Debugtrace, value);
        return *this;
    }

    /// Name of harmonic voltage or current spectrum for this IndMach012. Voltage behind Xd" for machine - default. Current injection for inverter. Default value is "default", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 22
    ///
    string spectrum()
    {
        return get_prop_string(Properties::spectrum);
    }

    IndMach012& spectrum(const string &value)
    {
        set_string(Properties::spectrum, value);
        return *this;
    }

    IndMach012& spectrum(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Name of harmonic voltage or current spectrum for this IndMach012. Voltage behind Xd" for machine - default. Current injection for inverter. Default value is "default", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 22
    ///
    dss::obj::Spectrum spectrum_obj()
    {
        return get_obj<dss::obj::Spectrum>(Properties::spectrum);
    }

    IndMach012& spectrum_obj(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 23
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    IndMach012& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 24
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    IndMach012& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 25
    ///
    IndMach012& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 25
    ///
    IndMach012& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class GICsource: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 39;
    struct Properties
    {
        enum {
            Volts = 1,
            angle = 2,
            frequency = 3,
            phases = 4,
            EN = 5,
            EE = 6,
            Lat1 = 7,
            Lon1 = 8,
            Lat2 = 9,
            Lon2 = 10,
            spectrum = 11,
            basefreq = 12,
            enabled = 13,
            like = 14,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    GICsource(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    GICsource(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the GICsource element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    GICsource(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the GICsource element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    GICsource& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    GICsource& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Voltage magnitude, in volts, of the GIC voltage induced across the associated line. When specified, induced voltage is assumed defined by Voltage and Angle properties.
    ///
    /// Specify this value
    ///
    /// OR
    ///
    /// EN, EE, lat1, lon1, lat2, lon2.
    ///
    /// Not both!!  Last one entered will take precedence. Assumed identical in each phase of the Line object.
    ///
    /// DSS property name: Volts, DSS property index: 1
    ///
    double Volts()
    {
        return Obj_GetFloat64(ptr, Properties::Volts);
    }

    GICsource& Volts(double value)
    {
        Obj_SetFloat64(ptr, Properties::Volts, value);
        return *this;
    }

    /// Phase angle in degrees of first phase. Default=0.0.  See Voltage property
    ///
    /// DSS property name: angle, DSS property index: 2
    ///
    double angle()
    {
        return Obj_GetFloat64(ptr, Properties::angle);
    }

    GICsource& angle(double value)
    {
        Obj_SetFloat64(ptr, Properties::angle, value);
        return *this;
    }

    /// Source frequency.  Defaults to  0.1 Hz. So GICSource=0 at power frequency.
    ///
    /// DSS property name: frequency, DSS property index: 3
    ///
    double frequency()
    {
        return Obj_GetFloat64(ptr, Properties::frequency);
    }

    GICsource& frequency(double value)
    {
        Obj_SetFloat64(ptr, Properties::frequency, value);
        return *this;
    }

    /// Number of phases.  Defaults to 3. All three phases are assumed in phase (zero sequence)
    ///
    /// DSS property name: phases, DSS property index: 4
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    GICsource& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Northward Electric field (V/km). If specified, Voltage and Angle are computed from EN, EE, lat and lon values.
    ///
    /// DSS property name: EN, DSS property index: 5
    ///
    double EN()
    {
        return Obj_GetFloat64(ptr, Properties::EN);
    }

    GICsource& EN(double value)
    {
        Obj_SetFloat64(ptr, Properties::EN, value);
        return *this;
    }

    /// Eastward Electric field (V/km).  If specified, Voltage and Angle are computed from EN, EE, lat and lon values.
    ///
    /// DSS property name: EE, DSS property index: 6
    ///
    double EE()
    {
        return Obj_GetFloat64(ptr, Properties::EE);
    }

    GICsource& EE(double value)
    {
        Obj_SetFloat64(ptr, Properties::EE, value);
        return *this;
    }

    /// Latitude of Bus1 of the line(degrees)
    ///
    /// DSS property name: Lat1, DSS property index: 7
    ///
    double Lat1()
    {
        return Obj_GetFloat64(ptr, Properties::Lat1);
    }

    GICsource& Lat1(double value)
    {
        Obj_SetFloat64(ptr, Properties::Lat1, value);
        return *this;
    }

    /// Longitude of Bus1 of the line (degrees)
    ///
    /// DSS property name: Lon1, DSS property index: 8
    ///
    double Lon1()
    {
        return Obj_GetFloat64(ptr, Properties::Lon1);
    }

    GICsource& Lon1(double value)
    {
        Obj_SetFloat64(ptr, Properties::Lon1, value);
        return *this;
    }

    /// Latitude of Bus2 of the line (degrees)
    ///
    /// DSS property name: Lat2, DSS property index: 9
    ///
    double Lat2()
    {
        return Obj_GetFloat64(ptr, Properties::Lat2);
    }

    GICsource& Lat2(double value)
    {
        Obj_SetFloat64(ptr, Properties::Lat2, value);
        return *this;
    }

    /// Longitude of Bus2 of the line (degrees)
    ///
    /// DSS property name: Lon2, DSS property index: 10
    ///
    double Lon2()
    {
        return Obj_GetFloat64(ptr, Properties::Lon2);
    }

    GICsource& Lon2(double value)
    {
        Obj_SetFloat64(ptr, Properties::Lon2, value);
        return *this;
    }

    /// Not used.
    ///
    /// DSS property name: spectrum, DSS property index: 11
    ///
    string spectrum()
    {
        return get_prop_string(Properties::spectrum);
    }

    GICsource& spectrum(const string &value)
    {
        set_string(Properties::spectrum, value);
        return *this;
    }

    GICsource& spectrum(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Not used.
    ///
    /// DSS property name: spectrum, DSS property index: 11
    ///
    dss::obj::Spectrum spectrum_obj()
    {
        return get_obj<dss::obj::Spectrum>(Properties::spectrum);
    }

    GICsource& spectrum_obj(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Not used.
    ///
    /// DSS property name: basefreq, DSS property index: 12
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    GICsource& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 13
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    GICsource& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 14
    ///
    GICsource& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 14
    ///
    GICsource& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class AutoTrans: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 40;
    struct Properties
    {
        enum {
            phases = 1,
            windings = 2,
            wdg = 3,
            bus = 4,
            conn = 5,
            kV = 6,
            kVA = 7,
            tap = 8,
            pctR = 9,
            Rdcohms = 10,
            Core = 11,
            buses = 12,
            conns = 13,
            kVs = 14,
            kVAs = 15,
            taps = 16,
            XHX = 17,
            XHT = 18,
            XXT = 19,
            XSCarray = 20,
            thermal = 21,
            n = 22,
            m = 23,
            flrise = 24,
            hsrise = 25,
            pctloadloss = 26,
            pctnoloadloss = 27,
            normhkVA = 28,
            emerghkVA = 29,
            sub = 30,
            MaxTap = 31,
            MinTap = 32,
            NumTaps = 33,
            subname = 34,
            pctimag = 35,
            ppm_antifloat = 36,
            pctRs = 37,
            XRConst = 38,
            LeadLag = 39,
            WdgCurrents = 40,
            normamps = 41,
            emergamps = 42,
            faultrate = 43,
            pctperm = 44,
            repair = 45,
            basefreq = 46,
            enabled = 47,
            like = 48,
        };
    };

    // Class-specific enumerations

        ///
        /// AutoTrans: Connection (DSS enumeration for AutoTrans)
        ///
    enum class AutoTransConnection: int32_t
    {
        wye = 0, ///< wye
        delta = 1, ///< delta
        series = 2, ///< series
        y = 0, ///< y
        ln = 0, ///< ln
        ll = 1 ///< ll
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    AutoTrans(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    AutoTrans(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the AutoTrans element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    AutoTrans(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the AutoTrans element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    AutoTrans& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    AutoTrans& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of phases this AutoTrans. Default is 3.
    ///
    /// DSS property name: phases, DSS property index: 1
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    AutoTrans& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Number of windings, this AutoTranss. (Also is the number of terminals) Default is 2. This property triggers memory allocation for the AutoTrans and will cause other properties to revert to default values.
    ///
    /// DSS property name: windings, DSS property index: 2
    ///
    int32_t windings()
    {
        return Obj_GetInt32(ptr, Properties::windings);
    }

    AutoTrans& windings(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::windings, value);
        return *this;
    }

    /// Set this = to the number of the winding you wish to define.  Then set the values for this winding.  Winding 1 is always the Series winding. Winding 2 is always Common winding (wye connected). Repeat for each winding.  Alternatively, use the array collections (buses, kVAs, etc.) to define the windings.  Note: reactances are BETWEEN pairs of windings; they are not the property of a single winding.
    ///
    /// DSS property name: wdg, DSS property index: 3
    ///
    int32_t wdg()
    {
        return Obj_GetInt32(ptr, Properties::wdg);
    }

    AutoTrans& wdg(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::wdg, value);
        return *this;
    }

    /// Bus connection spec for this winding.
    ///
    /// DSS property name: bus, DSS property index: 4
    ///
    strings bus()
    {
        return get_array<strings>(Properties::bus);
    }

    AutoTrans& bus(strings &value)
    {
        set_array<strings>(Properties::bus, value);
        return *this;
    }

    /// Connection of this winding {Series, wye*, Delta, LN, LL }. Default is "wye" with the neutral solidly grounded.
    /// For AutoTrans, Winding 1 is always Series and Winding 2 (the Common winding) is always Wye.
    /// If only 2 windings, no need to specify connections.
    ///
    /// DSS property name: conn, DSS property index: 5
    ///
    std::vector<AutoTransConnection> conn()
    {
        return get_array<std::vector<AutoTransConnection>>(Properties::conn);
    }

    AutoTrans& conn(std::vector<int32_t> &value)
    {
        set_array<std::vector<int32_t>>(Properties::conn, value);
        return *this;
    }

    AutoTrans& conn(strings &value)
    {
        set_array<strings>(Properties::conn, value);
        return *this;
    }

    /// Connection of this winding {Series, wye*, Delta, LN, LL }. Default is "wye" with the neutral solidly grounded.
    /// For AutoTrans, Winding 1 is always Series and Winding 2 (the Common winding) is always Wye.
    /// If only 2 windings, no need to specify connections.
    ///
    /// DSS property name: conn, DSS property index: 5
    ///
    strings conn_str()
    {
        return get_array<strings>(Properties::conn);
    }

    AutoTrans& conn_str(strings &value)
    {
        conn(value);
        return *this;
    }

    /// For 2-or 3-phase, enter phase-phase kV rating.  Otherwise, kV rating of the actual winding. Specify H terminal kV rating for Series winding.
    ///
    /// DSS property name: kV, DSS property index: 6
    ///
    VectorXd kV()
    {
        return get_array<VectorXd>(Properties::kV);
    }

    AutoTrans& kV(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kV, value);
        return *this;
    }

    /// Base kVA rating of the winding. Side effect: forces change of max normal and emerg kVA ratings.If 2-winding AutoTrans, forces other winding to same value. When winding 1 is defined, all other windings are defaulted to the same rating and the first two winding resistances are defaulted to the %loadloss value.
    ///
    /// DSS property name: kVA, DSS property index: 7
    ///
    VectorXd kVA()
    {
        return get_array<VectorXd>(Properties::kVA);
    }

    AutoTrans& kVA(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kVA, value);
        return *this;
    }

    /// Per unit tap that this winding is on.
    ///
    /// DSS property name: tap, DSS property index: 8
    ///
    VectorXd tap()
    {
        return get_array<VectorXd>(Properties::tap);
    }

    AutoTrans& tap(VectorXd &value)
    {
        set_array<VectorXd>(Properties::tap, value);
        return *this;
    }

    /// Percent ac resistance this winding.  This value is for the power flow model.Is derived from the full load losses in the transformer test report.
    ///
    /// DSS property name: %R, DSS property index: 9
    ///
    VectorXd pctR()
    {
        return get_array<VectorXd>(Properties::pctR);
    }

    AutoTrans& pctR(VectorXd &value)
    {
        set_array<VectorXd>(Properties::pctR, value);
        return *this;
    }

    /// Winding dc resistance in OHMS. Specify this for GIC analysis. From transformer test report (divide by number of phases). Defaults to 85% of %R property (the ac value that includes stray losses).
    ///
    /// DSS property name: Rdcohms, DSS property index: 10
    ///
    VectorXd Rdcohms()
    {
        return get_array<VectorXd>(Properties::Rdcohms);
    }

    AutoTrans& Rdcohms(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Rdcohms, value);
        return *this;
    }

    /// {Shell*|5-leg|3-Leg|1-phase|core-1-phase|4-leg} Core Type. Used for GIC analysis in auxiliary programs. Not used inside OpenDSS.
    ///
    /// DSS property name: Core, DSS property index: 11
    ///
    CoreType Core()
    {
        return CoreType(Obj_GetInt32(ptr, Properties::Core));
    }

    AutoTrans& Core(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Core, value);
        return *this;
    }

    AutoTrans& Core(CoreType value)
    {
        Obj_SetInt32(ptr, Properties::Core, int32_t(value));
        return *this;
    }

    AutoTrans& Core(const string &value)
    {
        set_string(Properties::Core, value);
        return *this;
    }

    AutoTrans& Core(const char *value)
    {
        set_string(Properties::Core, value);
        return *this;
    }

    /// {Shell*|5-leg|3-Leg|1-phase|core-1-phase|4-leg} Core Type. Used for GIC analysis in auxiliary programs. Not used inside OpenDSS.
    ///
    /// DSS property name: Core, DSS property index: 11
    ///
    string Core_str()
    {
        return get_prop_string(Properties::Core);
    }

    /// {Shell*|5-leg|3-Leg|1-phase|core-1-phase|4-leg} Core Type. Used for GIC analysis in auxiliary programs. Not used inside OpenDSS.
    ///
    /// DSS property name: Core, DSS property index: 11
    ///
    AutoTrans& Core_str(const string &value)
    {
        set_string(Properties::Core, value);
        return *this;
    }

    /// Use this to specify all the bus connections at once using an array. Example:
    ///
    /// New AutoTrans.T1 buses=[Hbus, Xbus]
    ///
    /// DSS property name: buses, DSS property index: 12
    ///
    strings buses()
    {
        return get_array<strings>(Properties::buses);
    }

    AutoTrans& buses(strings &value)
    {
        set_array<strings>(Properties::buses, value);
        return *this;
    }

    /// Use this to specify all the Winding connections at once using an array. Example:
    ///
    /// New AutoTrans.T1 buses=[Hbus, Xbus] ~ conns=(series, wye)
    ///
    /// DSS property name: conns, DSS property index: 13
    ///
    std::vector<AutoTransConnection> conns()
    {
        return get_array<std::vector<AutoTransConnection>>(Properties::conns);
    }

    AutoTrans& conns(std::vector<int32_t> &value)
    {
        set_array<std::vector<int32_t>>(Properties::conns, value);
        return *this;
    }

    AutoTrans& conns(strings &value)
    {
        set_array<strings>(Properties::conns, value);
        return *this;
    }

    /// Use this to specify all the Winding connections at once using an array. Example:
    ///
    /// New AutoTrans.T1 buses=[Hbus, Xbus] ~ conns=(series, wye)
    ///
    /// DSS property name: conns, DSS property index: 13
    ///
    strings conns_str()
    {
        return get_array<strings>(Properties::conns);
    }

    AutoTrans& conns_str(strings &value)
    {
        conns(value);
        return *this;
    }

    /// Use this to specify the kV ratings of all windings at once using an array. Example:
    ///
    /// New AutoTrans.T1 buses=[Hbus, Xbus]
    /// ~ conns=(series, wye)
    /// ~ kvs=(115, 12.47)
    ///
    /// See kV= property for voltage rules.
    ///
    /// DSS property name: kVs, DSS property index: 14
    ///
    VectorXd kVs()
    {
        return get_array<VectorXd>(Properties::kVs);
    }

    AutoTrans& kVs(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kVs, value);
        return *this;
    }

    /// Use this to specify the kVA ratings of all windings at once using an array.
    ///
    /// DSS property name: kVAs, DSS property index: 15
    ///
    VectorXd kVAs()
    {
        return get_array<VectorXd>(Properties::kVAs);
    }

    AutoTrans& kVAs(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kVAs, value);
        return *this;
    }

    /// Use this to specify the p.u. tap of all windings at once using an array.
    ///
    /// DSS property name: taps, DSS property index: 16
    ///
    VectorXd taps()
    {
        return get_array<VectorXd>(Properties::taps);
    }

    AutoTrans& taps(VectorXd &value)
    {
        set_array<VectorXd>(Properties::taps, value);
        return *this;
    }

    /// Use this to specify the percent reactance, H-L (winding 1 to winding 2).  Use for 2- or 3-winding AutoTranss. On the kVA base of winding 1(H-X).
    ///
    /// DSS property name: XHX, DSS property index: 17
    ///
    double XHX()
    {
        return Obj_GetFloat64(ptr, Properties::XHX);
    }

    AutoTrans& XHX(double value)
    {
        Obj_SetFloat64(ptr, Properties::XHX, value);
        return *this;
    }

    /// Use this to specify the percent reactance, H-T (winding 1 to winding 3).  Use for 3-winding AutoTranss only. On the kVA base of winding 1(H-X).
    ///
    /// DSS property name: XHT, DSS property index: 18
    ///
    double XHT()
    {
        return Obj_GetFloat64(ptr, Properties::XHT);
    }

    AutoTrans& XHT(double value)
    {
        Obj_SetFloat64(ptr, Properties::XHT, value);
        return *this;
    }

    /// Use this to specify the percent reactance, L-T (winding 2 to winding 3).  Use for 3-winding AutoTranss only. On the kVA base of winding 1(H-X).
    ///
    /// DSS property name: XXT, DSS property index: 19
    ///
    double XXT()
    {
        return Obj_GetFloat64(ptr, Properties::XXT);
    }

    AutoTrans& XXT(double value)
    {
        Obj_SetFloat64(ptr, Properties::XXT, value);
        return *this;
    }

    /// Use this to specify the percent reactance between all pairs of windings as an array. All values are on the kVA base of winding 1.  The order of the values is as follows:
    ///
    /// (x12 13 14... 23 24.. 34 ..)
    ///
    /// There will be n(n-1)/2 values, where n=number of windings.
    ///
    /// DSS property name: XSCarray, DSS property index: 20
    ///
    VectorXd XSCarray()
    {
        return get_array<VectorXd>(Properties::XSCarray);
    }

    AutoTrans& XSCarray(VectorXd &value)
    {
        set_array<VectorXd>(Properties::XSCarray, value);
        return *this;
    }

    /// Thermal time constant of the AutoTrans in hours.  Typically about 2.
    ///
    /// DSS property name: thermal, DSS property index: 21
    ///
    double thermal()
    {
        return Obj_GetFloat64(ptr, Properties::thermal);
    }

    AutoTrans& thermal(double value)
    {
        Obj_SetFloat64(ptr, Properties::thermal, value);
        return *this;
    }

    /// n Exponent for thermal properties in IEEE C57.  Typically 0.8.
    ///
    /// DSS property name: n, DSS property index: 22
    ///
    double n()
    {
        return Obj_GetFloat64(ptr, Properties::n);
    }

    AutoTrans& n(double value)
    {
        Obj_SetFloat64(ptr, Properties::n, value);
        return *this;
    }

    /// m Exponent for thermal properties in IEEE C57.  Typically 0.9 - 1.0
    ///
    /// DSS property name: m, DSS property index: 23
    ///
    double m()
    {
        return Obj_GetFloat64(ptr, Properties::m);
    }

    AutoTrans& m(double value)
    {
        Obj_SetFloat64(ptr, Properties::m, value);
        return *this;
    }

    /// Temperature rise, deg C, for full load.  Default is 65.
    ///
    /// DSS property name: flrise, DSS property index: 24
    ///
    double flrise()
    {
        return Obj_GetFloat64(ptr, Properties::flrise);
    }

    AutoTrans& flrise(double value)
    {
        Obj_SetFloat64(ptr, Properties::flrise, value);
        return *this;
    }

    /// Hot spot temperature rise, deg C.  Default is 15.
    ///
    /// DSS property name: hsrise, DSS property index: 25
    ///
    double hsrise()
    {
        return Obj_GetFloat64(ptr, Properties::hsrise);
    }

    AutoTrans& hsrise(double value)
    {
        Obj_SetFloat64(ptr, Properties::hsrise, value);
        return *this;
    }

    /// Percent load loss at full load. The %R of the High and Low windings (1 and 2) are adjusted to agree at rated kVA loading.
    ///
    /// DSS property name: %loadloss, DSS property index: 26
    ///
    double pctloadloss()
    {
        return Obj_GetFloat64(ptr, Properties::pctloadloss);
    }

    AutoTrans& pctloadloss(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctloadloss, value);
        return *this;
    }

    /// Percent no load losses at rated excitatation voltage. Default is 0. Converts to a resistance in parallel with the magnetizing impedance in each winding.
    ///
    /// DSS property name: %noloadloss, DSS property index: 27
    ///
    double pctnoloadloss()
    {
        return Obj_GetFloat64(ptr, Properties::pctnoloadloss);
    }

    AutoTrans& pctnoloadloss(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctnoloadloss, value);
        return *this;
    }

    /// Normal maximum kVA rating of H winding (winding 1+2).  Usually 100% - 110% ofmaximum nameplate rating, depending on load shape. Defaults to 110% of kVA rating of Winding 1.
    ///
    /// DSS property name: normhkVA, DSS property index: 28
    ///
    double normhkVA()
    {
        return Obj_GetFloat64(ptr, Properties::normhkVA);
    }

    AutoTrans& normhkVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::normhkVA, value);
        return *this;
    }

    /// Emergency (contingency)  kVA rating of H winding (winding 1+2).  Usually 140% - 150% ofmaximum nameplate rating, depending on load shape. Defaults to 150% of kVA rating of Winding 1.
    ///
    /// DSS property name: emerghkVA, DSS property index: 29
    ///
    double emerghkVA()
    {
        return Obj_GetFloat64(ptr, Properties::emerghkVA);
    }

    AutoTrans& emerghkVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::emerghkVA, value);
        return *this;
    }

    /// ={Yes|No}  Designates whether this AutoTrans is to be considered a substation.Default is No.
    ///
    /// DSS property name: sub, DSS property index: 30
    ///
    bool sub()
    {
        return Obj_GetInt32(ptr, Properties::sub) != 0;
    }

    AutoTrans& sub(bool value)
    {
        Obj_SetInt32(ptr, Properties::sub, value);
        return *this;
    }

    /// Max per unit tap for the active winding.  Default is 1.10
    ///
    /// DSS property name: MaxTap, DSS property index: 31
    ///
    VectorXd MaxTap()
    {
        return get_array<VectorXd>(Properties::MaxTap);
    }

    AutoTrans& MaxTap(VectorXd &value)
    {
        set_array<VectorXd>(Properties::MaxTap, value);
        return *this;
    }

    /// Min per unit tap for the active winding.  Default is 0.90
    ///
    /// DSS property name: MinTap, DSS property index: 32
    ///
    VectorXd MinTap()
    {
        return get_array<VectorXd>(Properties::MinTap);
    }

    AutoTrans& MinTap(VectorXd &value)
    {
        set_array<VectorXd>(Properties::MinTap, value);
        return *this;
    }

    /// Total number of taps between min and max tap.  Default is 32 (16 raise and 16 lower taps about the neutral position). The neutral position is not counted.
    ///
    /// DSS property name: NumTaps, DSS property index: 33
    ///
    VectorXi NumTaps()
    {
        return get_array<VectorXi>(Properties::NumTaps);
    }

    AutoTrans& NumTaps(VectorXi &value)
    {
        set_array<VectorXi>(Properties::NumTaps, value);
        return *this;
    }

    /// Substation Name. Optional. Default is null. If specified, printed on plots
    ///
    /// DSS property name: subname, DSS property index: 34
    ///
    string subname()
    {
        return get_prop_string(Properties::subname);
    }

    AutoTrans& subname(const string &value)
    {
        set_string(Properties::subname, value);
        return *this;
    }

    AutoTrans& subname(const char* value)
    {
        set_string(Properties::subname, value);
        return *this;
    }

    /// Percent magnetizing current. Default=0.0. Magnetizing branch is in parallel with windings in each phase. Also, see "ppm_antifloat".
    ///
    /// DSS property name: %imag, DSS property index: 35
    ///
    double pctimag()
    {
        return Obj_GetFloat64(ptr, Properties::pctimag);
    }

    AutoTrans& pctimag(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctimag, value);
        return *this;
    }

    /// Default=1 ppm.  Parts per million of AutoTrans winding VA rating connected to ground to protect against accidentally floating a winding without a reference. If positive then the effect is adding a very large reactance to ground.  If negative, then a capacitor.
    ///
    /// DSS property name: ppm_antifloat, DSS property index: 36
    ///
    double ppm_antifloat()
    {
        return Obj_GetFloat64(ptr, Properties::ppm_antifloat);
    }

    AutoTrans& ppm_antifloat(double value)
    {
        Obj_SetFloat64(ptr, Properties::ppm_antifloat, value);
        return *this;
    }

    /// Use this property to specify all the winding ac %resistances using an array. Example:
    ///
    /// New AutoTrans.T1 buses=[Hibus, lowbus] ~ %Rs=(0.2  0.3)
    ///
    /// DSS property name: %Rs, DSS property index: 37
    ///
    VectorXd pctRs()
    {
        return get_array<VectorXd>(Properties::pctRs);
    }

    AutoTrans& pctRs(VectorXd &value)
    {
        set_array<VectorXd>(Properties::pctRs, value);
        return *this;
    }

    /// ={Yes|No} Default is NO. Signifies whether or not the X/R is assumed contant for harmonic studies.
    ///
    /// DSS property name: XRConst, DSS property index: 38
    ///
    bool XRConst()
    {
        return Obj_GetInt32(ptr, Properties::XRConst) != 0;
    }

    AutoTrans& XRConst(bool value)
    {
        Obj_SetInt32(ptr, Properties::XRConst, value);
        return *this;
    }

    /// {Lead | Lag (default) | ANSI (default) | Euro } Designation in mixed Delta-wye connections the relationship between HV to LV winding. Default is ANSI 30 deg lag, e.g., Dy1 of Yd1 vector group. To get typical European Dy11 connection, specify either "lead" or "Euro"
    ///
    /// DSS property name: LeadLag, DSS property index: 39
    ///
    PhaseSequence LeadLag()
    {
        return PhaseSequence(Obj_GetInt32(ptr, Properties::LeadLag));
    }

    AutoTrans& LeadLag(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::LeadLag, value);
        return *this;
    }

    AutoTrans& LeadLag(PhaseSequence value)
    {
        Obj_SetInt32(ptr, Properties::LeadLag, int32_t(value));
        return *this;
    }

    AutoTrans& LeadLag(const string &value)
    {
        set_string(Properties::LeadLag, value);
        return *this;
    }

    AutoTrans& LeadLag(const char *value)
    {
        set_string(Properties::LeadLag, value);
        return *this;
    }

    /// {Lead | Lag (default) | ANSI (default) | Euro } Designation in mixed Delta-wye connections the relationship between HV to LV winding. Default is ANSI 30 deg lag, e.g., Dy1 of Yd1 vector group. To get typical European Dy11 connection, specify either "lead" or "Euro"
    ///
    /// DSS property name: LeadLag, DSS property index: 39
    ///
    string LeadLag_str()
    {
        return get_prop_string(Properties::LeadLag);
    }

    /// {Lead | Lag (default) | ANSI (default) | Euro } Designation in mixed Delta-wye connections the relationship between HV to LV winding. Default is ANSI 30 deg lag, e.g., Dy1 of Yd1 vector group. To get typical European Dy11 connection, specify either "lead" or "Euro"
    ///
    /// DSS property name: LeadLag, DSS property index: 39
    ///
    AutoTrans& LeadLag_str(const string &value)
    {
        set_string(Properties::LeadLag, value);
        return *this;
    }

    /// (Read only) Makes winding currents available via return on query (? AutoTrans.TX.WdgCurrents). Order: Phase 1, Wdg 1, Wdg 2, ..., Phase 2 ...
    ///
    /// DSS property name: WdgCurrents, DSS property index: 40
    ///
    string WdgCurrents()
    {
        // []
        // StringSilentROFunction
        return get_prop_string(Properties::WdgCurrents);
    }

    /// Normal rated current.
    ///
    /// DSS property name: normamps, DSS property index: 41
    ///
    double normamps()
    {
        return Obj_GetFloat64(ptr, Properties::normamps);
    }

    AutoTrans& normamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::normamps, value);
        return *this;
    }

    /// Maximum or emerg current.
    ///
    /// DSS property name: emergamps, DSS property index: 42
    ///
    double emergamps()
    {
        return Obj_GetFloat64(ptr, Properties::emergamps);
    }

    AutoTrans& emergamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::emergamps, value);
        return *this;
    }

    /// Failure rate per year.
    ///
    /// DSS property name: faultrate, DSS property index: 43
    ///
    double faultrate()
    {
        return Obj_GetFloat64(ptr, Properties::faultrate);
    }

    AutoTrans& faultrate(double value)
    {
        Obj_SetFloat64(ptr, Properties::faultrate, value);
        return *this;
    }

    /// Percent of failures that become permanent.
    ///
    /// DSS property name: pctperm, DSS property index: 44
    ///
    double pctperm()
    {
        return Obj_GetFloat64(ptr, Properties::pctperm);
    }

    AutoTrans& pctperm(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctperm, value);
        return *this;
    }

    /// Hours to repair.
    ///
    /// DSS property name: repair, DSS property index: 45
    ///
    double repair()
    {
        return Obj_GetFloat64(ptr, Properties::repair);
    }

    AutoTrans& repair(double value)
    {
        Obj_SetFloat64(ptr, Properties::repair, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 46
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    AutoTrans& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 47
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    AutoTrans& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 48
    ///
    AutoTrans& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 48
    ///
    AutoTrans& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class RegControl: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 21;
    struct Properties
    {
        enum {
            transformer = 1,
            winding = 2,
            vreg = 3,
            band = 4,
            ptratio = 5,
            CTprim = 6,
            R = 7,
            X = 8,
            bus = 9,
            delay = 10,
            reversible = 11,
            revvreg = 12,
            revband = 13,
            revR = 14,
            revX = 15,
            tapdelay = 16,
            debugtrace = 17,
            maxtapchange = 18,
            inversetime = 19,
            tapwinding = 20,
            vlimit = 21,
            PTphase = 22,
            revThreshold = 23,
            revDelay = 24,
            revNeutral = 25,
            EventLog = 26,
            RemotePTRatio = 27,
            TapNum = 28,
            Reset = 29,
            LDC_Z = 30,
            rev_Z = 31,
            Cogen = 32,
            basefreq = 33,
            enabled = 34,
            like = 35,
        };
    };

    // Class-specific enumerations

        ///
        /// RegControl: Phase Selection (DSS enumeration for RegControl)
        ///
    enum class RegControlPhaseSelection: int32_t
    {
        min = -3, ///< min
        max = -2 ///< max
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    RegControl(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    RegControl(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the RegControl element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    RegControl(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the RegControl element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    RegControl& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    RegControl& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name of Transformer or AutoTrans element to which the RegControl is connected. Do not specify the full object name; "Transformer" or "AutoTrans" is assumed for the object class.  Example:
    ///
    /// Transformer=Xfmr1
    ///
    /// DSS property name: transformer, DSS property index: 1
    ///
    string transformer()
    {
        return get_prop_string(Properties::transformer);
    }

    RegControl& transformer(const string &value)
    {
        set_string(Properties::transformer, value);
        return *this;
    }

    RegControl& transformer(dss::obj::DSSObj &value)
    {
        set_obj(Properties::transformer, value);
        return *this;
    }

    /// Name of Transformer or AutoTrans element to which the RegControl is connected. Do not specify the full object name; "Transformer" or "AutoTrans" is assumed for the object class.  Example:
    ///
    /// Transformer=Xfmr1
    ///
    /// DSS property name: transformer, DSS property index: 1
    ///
    dss::obj::DSSObj transformer_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::transformer);
    }

    RegControl& transformer_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::transformer, value);
        return *this;
    }

    /// Number of the winding of the transformer element that the RegControl is monitoring. 1 or 2, typically.  Side Effect: Sets TAPWINDING property to the same winding.
    ///
    /// DSS property name: winding, DSS property index: 2
    ///
    int32_t winding()
    {
        return Obj_GetInt32(ptr, Properties::winding);
    }

    RegControl& winding(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::winding, value);
        return *this;
    }

    /// Voltage regulator setting, in VOLTS, for the winding being controlled.  Multiplying this value times the ptratio should yield the voltage across the WINDING of the controlled transformer. Default is 120.0
    ///
    /// DSS property name: vreg, DSS property index: 3
    ///
    double vreg()
    {
        return Obj_GetFloat64(ptr, Properties::vreg);
    }

    RegControl& vreg(double value)
    {
        Obj_SetFloat64(ptr, Properties::vreg, value);
        return *this;
    }

    /// Bandwidth in VOLTS for the controlled bus (see help for ptratio property).  Default is 3.0
    ///
    /// DSS property name: band, DSS property index: 4
    ///
    double band()
    {
        return Obj_GetFloat64(ptr, Properties::band);
    }

    RegControl& band(double value)
    {
        Obj_SetFloat64(ptr, Properties::band, value);
        return *this;
    }

    /// Ratio of the PT that converts the controlled winding voltage to the regulator control voltage. Default is 60.  If the winding is Wye, the line-to-neutral voltage is used.  Else, the line-to-line voltage is used. SIDE EFFECT: Also sets RemotePTRatio property.
    ///
    /// DSS property name: ptratio, DSS property index: 5
    ///
    double ptratio()
    {
        return Obj_GetFloat64(ptr, Properties::ptratio);
    }

    RegControl& ptratio(double value)
    {
        Obj_SetFloat64(ptr, Properties::ptratio, value);
        return *this;
    }

    /// Rating, in Amperes, of the primary CT rating for which the line amps convert to control rated amps.The typical default secondary ampere rating is 0.2 Amps (check with manufacturer specs). Current at which the LDC voltages match the R and X settings.
    ///
    /// DSS property name: CTprim, DSS property index: 6
    ///
    double CTprim()
    {
        return Obj_GetFloat64(ptr, Properties::CTprim);
    }

    RegControl& CTprim(double value)
    {
        Obj_SetFloat64(ptr, Properties::CTprim, value);
        return *this;
    }

    /// R setting on the line drop compensator in the regulator, expressed in VOLTS.
    ///
    /// DSS property name: R, DSS property index: 7
    ///
    double R()
    {
        return Obj_GetFloat64(ptr, Properties::R);
    }

    RegControl& R(double value)
    {
        Obj_SetFloat64(ptr, Properties::R, value);
        return *this;
    }

    /// X setting on the line drop compensator in the regulator, expressed in VOLTS.
    ///
    /// DSS property name: X, DSS property index: 8
    ///
    double X()
    {
        return Obj_GetFloat64(ptr, Properties::X);
    }

    RegControl& X(double value)
    {
        Obj_SetFloat64(ptr, Properties::X, value);
        return *this;
    }

    /// Name of a bus (busname.nodename) in the system to use as the controlled bus instead of the bus to which the transformer winding is connected or the R and X line drop compensator settings.  Do not specify this value if you wish to use the line drop compensator settings.  Default is null string. Assumes the base voltage for this bus is the same as the transformer winding base specified above. Note: This bus (1-phase) WILL BE CREATED by the regulator control upon SOLVE if not defined by some other device. You can specify the node of the bus you wish to sample (defaults to 1). If specified, the RegControl is redefined as a 1-phase device since only one voltage is used.
    ///
    /// DSS property name: bus, DSS property index: 9
    ///
    string bus()
    {
        return get_prop_string(Properties::bus);
    }

    RegControl& bus(const string &value)
    {
        set_string(Properties::bus, value);
        return *this;
    }

    RegControl& bus(const char* value)
    {
        set_string(Properties::bus, value);
        return *this;
    }

    /// Time delay, in seconds, from when the voltage goes out of band to when the tap changing begins. This is used to determine which regulator control will act first. Default is 15.  You may specify any floating point number to achieve a model of whatever condition is necessary.
    ///
    /// DSS property name: delay, DSS property index: 10
    ///
    double delay()
    {
        return Obj_GetFloat64(ptr, Properties::delay);
    }

    RegControl& delay(double value)
    {
        Obj_SetFloat64(ptr, Properties::delay, value);
        return *this;
    }

    /// {Yes |No*} Indicates whether or not the regulator can be switched to regulate in the reverse direction. Default is No.Typically applies only to line regulators and not to LTC on a substation transformer.
    ///
    /// DSS property name: reversible, DSS property index: 11
    ///
    bool reversible()
    {
        return Obj_GetInt32(ptr, Properties::reversible) != 0;
    }

    RegControl& reversible(bool value)
    {
        Obj_SetInt32(ptr, Properties::reversible, value);
        return *this;
    }

    /// Voltage setting in volts for operation in the reverse direction.
    ///
    /// DSS property name: revvreg, DSS property index: 12
    ///
    double revvreg()
    {
        return Obj_GetFloat64(ptr, Properties::revvreg);
    }

    RegControl& revvreg(double value)
    {
        Obj_SetFloat64(ptr, Properties::revvreg, value);
        return *this;
    }

    /// Bandwidth for operating in the reverse direction.
    ///
    /// DSS property name: revband, DSS property index: 13
    ///
    double revband()
    {
        return Obj_GetFloat64(ptr, Properties::revband);
    }

    RegControl& revband(double value)
    {
        Obj_SetFloat64(ptr, Properties::revband, value);
        return *this;
    }

    /// R line drop compensator setting for reverse direction.
    ///
    /// DSS property name: revR, DSS property index: 14
    ///
    double revR()
    {
        return Obj_GetFloat64(ptr, Properties::revR);
    }

    RegControl& revR(double value)
    {
        Obj_SetFloat64(ptr, Properties::revR, value);
        return *this;
    }

    /// X line drop compensator setting for reverse direction.
    ///
    /// DSS property name: revX, DSS property index: 15
    ///
    double revX()
    {
        return Obj_GetFloat64(ptr, Properties::revX);
    }

    RegControl& revX(double value)
    {
        Obj_SetFloat64(ptr, Properties::revX, value);
        return *this;
    }

    /// Delay in sec between tap changes. Default is 2. This is how long it takes between changes after the first change.
    ///
    /// DSS property name: tapdelay, DSS property index: 16
    ///
    double tapdelay()
    {
        return Obj_GetFloat64(ptr, Properties::tapdelay);
    }

    RegControl& tapdelay(double value)
    {
        Obj_SetFloat64(ptr, Properties::tapdelay, value);
        return *this;
    }

    /// {Yes | No* }  Default is no.  Turn this on to capture the progress of the regulator model for each control iteration.  Creates a separate file for each RegControl named "REG_name.CSV".
    ///
    /// DSS property name: debugtrace, DSS property index: 17
    ///
    bool debugtrace()
    {
        return Obj_GetInt32(ptr, Properties::debugtrace) != 0;
    }

    RegControl& debugtrace(bool value)
    {
        Obj_SetInt32(ptr, Properties::debugtrace, value);
        return *this;
    }

    /// Maximum allowable tap change per control iteration in STATIC control mode.  Default is 16.
    ///
    /// Set this to 1 to better approximate actual control action.
    ///
    /// Set this to 0 to fix the tap in the current position.
    ///
    /// DSS property name: maxtapchange, DSS property index: 18
    ///
    int32_t maxtapchange()
    {
        return Obj_GetInt32(ptr, Properties::maxtapchange);
    }

    RegControl& maxtapchange(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::maxtapchange, value);
        return *this;
    }

    /// {Yes | No* } Default is no.  The time delay is adjusted inversely proportional to the amount the voltage is outside the band down to 10%.
    ///
    /// DSS property name: inversetime, DSS property index: 19
    ///
    bool inversetime()
    {
        return Obj_GetInt32(ptr, Properties::inversetime) != 0;
    }

    RegControl& inversetime(bool value)
    {
        Obj_SetInt32(ptr, Properties::inversetime, value);
        return *this;
    }

    /// Winding containing the actual taps, if different than the WINDING property. Defaults to the same winding as specified by the WINDING property.
    ///
    /// DSS property name: tapwinding, DSS property index: 20
    ///
    int32_t tapwinding()
    {
        return Obj_GetInt32(ptr, Properties::tapwinding);
    }

    RegControl& tapwinding(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::tapwinding, value);
        return *this;
    }

    /// Voltage Limit for bus to which regulated winding is connected (e.g. first customer). Default is 0.0. Set to a value greater then zero to activate this function.
    ///
    /// DSS property name: vlimit, DSS property index: 21
    ///
    double vlimit()
    {
        return Obj_GetFloat64(ptr, Properties::vlimit);
    }

    RegControl& vlimit(double value)
    {
        Obj_SetFloat64(ptr, Properties::vlimit, value);
        return *this;
    }

    /// For multi-phase transformers, the number of the phase being monitored or one of { MAX | MIN} for all phases. Default=1. Must be less than or equal to the number of phases. Ignored for regulated bus.
    ///
    /// DSS property name: PTphase, DSS property index: 22
    ///
    int32_t PTphase()
    {
        return Obj_GetInt32(ptr, Properties::PTphase);
    }

    RegControl& PTphase(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::PTphase, value);
        return *this;
    }

    RegControl& PTphase(RegControlPhaseSelection value)
    {
        Obj_SetInt32(ptr, Properties::PTphase, int32_t(value));
        return *this;
    }

    RegControl& PTphase(const string &value)
    {
        set_string(Properties::PTphase, value);
        return *this;
    }

    RegControl& PTphase(const char *value)
    {
        set_string(Properties::PTphase, value);
        return *this;
    }

    /// For multi-phase transformers, the number of the phase being monitored or one of { MAX | MIN} for all phases. Default=1. Must be less than or equal to the number of phases. Ignored for regulated bus.
    ///
    /// DSS property name: PTphase, DSS property index: 22
    ///
    string PTphase_str()
    {
        return get_prop_string(Properties::PTphase);
    }

    /// For multi-phase transformers, the number of the phase being monitored or one of { MAX | MIN} for all phases. Default=1. Must be less than or equal to the number of phases. Ignored for regulated bus.
    ///
    /// DSS property name: PTphase, DSS property index: 22
    ///
    RegControl& PTphase_str(const string &value)
    {
        set_string(Properties::PTphase, value);
        return *this;
    }

    /// kW reverse power threshold for reversing the direction of the regulator. Default is 100.0 kw.
    ///
    /// DSS property name: revThreshold, DSS property index: 23
    ///
    double revThreshold()
    {
        return Obj_GetFloat64(ptr, Properties::revThreshold);
    }

    RegControl& revThreshold(double value)
    {
        Obj_SetFloat64(ptr, Properties::revThreshold, value);
        return *this;
    }

    /// Time Delay in seconds (s) for executing the reversing action once the threshold for reversing has been exceeded. Default is 60 s.
    ///
    /// DSS property name: revDelay, DSS property index: 24
    ///
    double revDelay()
    {
        return Obj_GetFloat64(ptr, Properties::revDelay);
    }

    RegControl& revDelay(double value)
    {
        Obj_SetFloat64(ptr, Properties::revDelay, value);
        return *this;
    }

    /// {Yes | No*} Default is no. Set this to Yes if you want the regulator to go to neutral in the reverse direction or in cogen operation.
    ///
    /// DSS property name: revNeutral, DSS property index: 25
    ///
    bool revNeutral()
    {
        return Obj_GetInt32(ptr, Properties::revNeutral) != 0;
    }

    RegControl& revNeutral(bool value)
    {
        Obj_SetInt32(ptr, Properties::revNeutral, value);
        return *this;
    }

    /// {Yes/True* | No/False} Default is YES for regulator control. Log control actions to Eventlog.
    ///
    /// DSS property name: EventLog, DSS property index: 26
    ///
    bool EventLog()
    {
        return Obj_GetInt32(ptr, Properties::EventLog) != 0;
    }

    RegControl& EventLog(bool value)
    {
        Obj_SetInt32(ptr, Properties::EventLog, value);
        return *this;
    }

    /// When regulating a bus (the Bus= property is set), the PT ratio required to convert actual voltage at the remote bus to control voltage. Is initialized to PTratio property. Set this property after setting PTratio.
    ///
    /// DSS property name: RemotePTRatio, DSS property index: 27
    ///
    double RemotePTRatio()
    {
        return Obj_GetFloat64(ptr, Properties::RemotePTRatio);
    }

    RegControl& RemotePTRatio(double value)
    {
        Obj_SetFloat64(ptr, Properties::RemotePTRatio, value);
        return *this;
    }

    /// An integer number indicating the tap position that the controlled transformer winding tap position is currently at, or is being set to.  If being set, and the value is outside the range of the transformer min or max tap, then set to the min or max tap position as appropriate. Default is 0
    ///
    /// DSS property name: TapNum, DSS property index: 28
    ///
    int32_t TapNum()
    {
        return Obj_GetInt32(ptr, Properties::TapNum);
    }

    RegControl& TapNum(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::TapNum, value);
        return *this;
    }

    /// {Yes | No} If Yes, forces Reset of this RegControl.
    ///
    /// DSS property name: Reset, DSS property index: 29
    ///
    RegControl& Reset(bool value)
    {
        Obj_SetInt32(ptr, Properties::Reset, value);
        return *this;
    }

    /// Z value for Beckwith LDC_Z control option. Volts adjustment at rated control current.
    ///
    /// DSS property name: LDC_Z, DSS property index: 30
    ///
    double LDC_Z()
    {
        return Obj_GetFloat64(ptr, Properties::LDC_Z);
    }

    RegControl& LDC_Z(double value)
    {
        Obj_SetFloat64(ptr, Properties::LDC_Z, value);
        return *this;
    }

    /// Reverse Z value for Beckwith LDC_Z control option.
    ///
    /// DSS property name: rev_Z, DSS property index: 31
    ///
    double rev_Z()
    {
        return Obj_GetFloat64(ptr, Properties::rev_Z);
    }

    RegControl& rev_Z(double value)
    {
        Obj_SetFloat64(ptr, Properties::rev_Z, value);
        return *this;
    }

    /// {Yes|No*} Default is No. The Cogen feature is activated. Continues looking forward if power reverses, but switches to reverse-mode LDC, vreg and band values.
    ///
    /// DSS property name: Cogen, DSS property index: 32
    ///
    bool Cogen()
    {
        return Obj_GetInt32(ptr, Properties::Cogen) != 0;
    }

    RegControl& Cogen(bool value)
    {
        Obj_SetInt32(ptr, Properties::Cogen, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 33
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    RegControl& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 34
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    RegControl& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 35
    ///
    RegControl& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 35
    ///
    RegControl& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class InvControl: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 41;
    struct Properties
    {
        enum {
            DERList = 1,
            Mode = 2,
            CombiMode = 3,
            vvc_curve1 = 4,
            hysteresis_offset = 5,
            voltage_curvex_ref = 6,
            avgwindowlen = 7,
            voltwatt_curve = 8,
            DbVMin = 9,
            DbVMax = 10,
            ArGraLowV = 11,
            ArGraHiV = 12,
            DynReacavgwindowlen = 13,
            deltaQ_Factor = 14,
            VoltageChangeTolerance = 15,
            VarChangeTolerance = 16,
            VoltwattYAxis = 17,
            RateofChangeMode = 18,
            LPFTau = 19,
            RiseFallLimit = 20,
            deltaP_Factor = 21,
            EventLog = 22,
            RefReactivePower = 23,
            ActivePChangeTolerance = 24,
            monVoltageCalc = 25,
            monBus = 26,
            MonBusesVbase = 27,
            voltwattCH_curve = 28,
            wattpf_curve = 29,
            wattvar_curve = 30,
            PVSystemList = 31,
            Vsetpoint = 32,
            basefreq = 33,
            enabled = 34,
            like = 35,
        };
    };

    // Class-specific enumerations

        ///
        /// InvControl: Control Mode (DSS enumeration for InvControl)
        ///
    enum class InvControlControlMode: int32_t
    {
        Voltvar = 1, ///< Voltvar
        VoltWatt = 2, ///< VoltWatt
        DynamicReaccurr = 3, ///< DynamicReaccurr
        WattPF = 4, ///< WattPF
        Wattvar = 5, ///< Wattvar
        AVR = 6 ///< AVR
    };


        ///
        /// InvControl: Combi Mode (DSS enumeration for InvControl)
        ///
    enum class InvControlCombiMode: int32_t
    {
        VV_VW = 1, ///< VV_VW
        VV_DRC = 2 ///< VV_DRC
    };


        ///
        /// InvControl: Voltage Curve X Ref (DSS enumeration for InvControl)
        ///
    enum class InvControlVoltageCurveXRef: int32_t
    {
        Rated = 0, ///< Rated
        Avg = 1, ///< Avg
        RAvg = 2 ///< RAvg
    };


        ///
        /// InvControl: Volt-watt Y-Axis (DSS enumeration for InvControl)
        ///
    enum class InvControlVoltWattYAxis: int32_t
    {
        PAvailablePU = 0, ///< PAvailablePU
        PMPPPU = 1, ///< PMPPPU
        PctPMPPPU = 2, ///< PctPMPPPU
        KVARatingPU = 3 ///< KVARatingPU
    };


        ///
        /// InvControl: Rate-of-change Mode (DSS enumeration for InvControl)
        ///
    enum class InvControlRateOfChangeMode: int32_t
    {
        Inactive = 0, ///< Inactive
        LPF = 1, ///< LPF
        RiseFall = 2 ///< RiseFall
    };


        ///
        /// InvControl: Reactive Power Reference (DSS enumeration for InvControl)
        ///
    enum class InvControlReactivePowerReference: int32_t
    {
        VARAVAL = 0, ///< VARAVAL
        VARMAX = 1 ///< VARMAX
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    InvControl(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    InvControl(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the InvControl element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    InvControl(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the InvControl element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    InvControl& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    InvControl& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Array list of PVSystem and/or Storage elements to be controlled. If not specified, all PVSystem and Storage in the circuit are assumed to be controlled by this control.
    ///
    /// No capability of hierarchical control between two controls for a single element is implemented at this time.
    ///
    /// DSS property name: DERList, DSS property index: 1
    ///
    strings DERList()
    {
        return get_array<strings>(Properties::DERList);
    }

    InvControl& DERList(strings &value)
    {
        set_array<strings>(Properties::DERList, value);
        return *this;
    }

    /// Smart inverter function in which the InvControl will control the PC elements specified in DERList, according to the options below:
    ///
    /// Must be one of: {VOLTVAR* | VOLTWATT | DYNAMICREACCURR | WATTPF | WATTVAR}
    /// if the user desires to use modes simultaneously, then set the CombiMode property. Setting the Mode to any valid value disables combination mode.
    ///
    /// In volt-var mode (Default). This mode attempts to CONTROL the vars, according to one or two volt-var curves, depending on the monitored voltages, present active power output, and the capabilities of the PVSystem/Storage.
    ///
    /// In volt-watt mode. This mode attempts to LIMIT the watts, according to one defined volt-watt curve, depending on the monitored voltages and the capabilities of the PVSystem/Storage.
    ///
    /// In dynamic reactive current mode. This mode attempts to increasingly counter deviations by CONTROLLING vars, depending on the monitored voltages, present active power output, and the capabilities of the of the PVSystem/Storage.
    ///
    /// In watt-pf mode. This mode attempts to CONTROL the vars, according to a watt-pf curve, depending on the present active power output, and the capabilities of the PVSystem/Storage.
    ///
    /// In watt-var mode. This mode attempts to CONTROL the vars, according to a watt-var curve, depending on the present active power output, and the capabilities of the PVSystem/Storage.
    ///
    /// DSS property name: Mode, DSS property index: 2
    ///
    InvControlControlMode Mode()
    {
        return InvControlControlMode(Obj_GetInt32(ptr, Properties::Mode));
    }

    InvControl& Mode(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Mode, value);
        return *this;
    }

    InvControl& Mode(InvControlControlMode value)
    {
        Obj_SetInt32(ptr, Properties::Mode, int32_t(value));
        return *this;
    }

    InvControl& Mode(const string &value)
    {
        set_string(Properties::Mode, value);
        return *this;
    }

    InvControl& Mode(const char *value)
    {
        set_string(Properties::Mode, value);
        return *this;
    }

    /// Smart inverter function in which the InvControl will control the PC elements specified in DERList, according to the options below:
    ///
    /// Must be one of: {VOLTVAR* | VOLTWATT | DYNAMICREACCURR | WATTPF | WATTVAR}
    /// if the user desires to use modes simultaneously, then set the CombiMode property. Setting the Mode to any valid value disables combination mode.
    ///
    /// In volt-var mode (Default). This mode attempts to CONTROL the vars, according to one or two volt-var curves, depending on the monitored voltages, present active power output, and the capabilities of the PVSystem/Storage.
    ///
    /// In volt-watt mode. This mode attempts to LIMIT the watts, according to one defined volt-watt curve, depending on the monitored voltages and the capabilities of the PVSystem/Storage.
    ///
    /// In dynamic reactive current mode. This mode attempts to increasingly counter deviations by CONTROLLING vars, depending on the monitored voltages, present active power output, and the capabilities of the of the PVSystem/Storage.
    ///
    /// In watt-pf mode. This mode attempts to CONTROL the vars, according to a watt-pf curve, depending on the present active power output, and the capabilities of the PVSystem/Storage.
    ///
    /// In watt-var mode. This mode attempts to CONTROL the vars, according to a watt-var curve, depending on the present active power output, and the capabilities of the PVSystem/Storage.
    ///
    /// DSS property name: Mode, DSS property index: 2
    ///
    string Mode_str()
    {
        return get_prop_string(Properties::Mode);
    }

    /// Smart inverter function in which the InvControl will control the PC elements specified in DERList, according to the options below:
    ///
    /// Must be one of: {VOLTVAR* | VOLTWATT | DYNAMICREACCURR | WATTPF | WATTVAR}
    /// if the user desires to use modes simultaneously, then set the CombiMode property. Setting the Mode to any valid value disables combination mode.
    ///
    /// In volt-var mode (Default). This mode attempts to CONTROL the vars, according to one or two volt-var curves, depending on the monitored voltages, present active power output, and the capabilities of the PVSystem/Storage.
    ///
    /// In volt-watt mode. This mode attempts to LIMIT the watts, according to one defined volt-watt curve, depending on the monitored voltages and the capabilities of the PVSystem/Storage.
    ///
    /// In dynamic reactive current mode. This mode attempts to increasingly counter deviations by CONTROLLING vars, depending on the monitored voltages, present active power output, and the capabilities of the of the PVSystem/Storage.
    ///
    /// In watt-pf mode. This mode attempts to CONTROL the vars, according to a watt-pf curve, depending on the present active power output, and the capabilities of the PVSystem/Storage.
    ///
    /// In watt-var mode. This mode attempts to CONTROL the vars, according to a watt-var curve, depending on the present active power output, and the capabilities of the PVSystem/Storage.
    ///
    /// DSS property name: Mode, DSS property index: 2
    ///
    InvControl& Mode_str(const string &value)
    {
        set_string(Properties::Mode, value);
        return *this;
    }

    /// Combination of smart inverter functions in which the InvControl will control the PC elements in DERList, according to the options below:
    ///
    /// Must be a combination of the following: {VV_VW | VV_DRC}. Default is to not set this property, in which case the single control mode in Mode is active.
    ///
    /// In combined VV_VW mode, both volt-var and volt-watt control modes are active simultaneously.  See help individually for volt-var mode and volt-watt mode in Mode property.
    /// Note that the PVSystem/Storage will attempt to achieve both the volt-watt and volt-var set-points based on the capabilities of the inverter in the PVSystem/Storage (kVA rating, etc), any limits set on maximum active power,
    ///
    /// In combined VV_DRC, both the volt-var and the dynamic reactive current modes are simultaneously active.
    ///
    /// DSS property name: CombiMode, DSS property index: 3
    ///
    InvControlCombiMode CombiMode()
    {
        return InvControlCombiMode(Obj_GetInt32(ptr, Properties::CombiMode));
    }

    InvControl& CombiMode(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::CombiMode, value);
        return *this;
    }

    InvControl& CombiMode(InvControlCombiMode value)
    {
        Obj_SetInt32(ptr, Properties::CombiMode, int32_t(value));
        return *this;
    }

    InvControl& CombiMode(const string &value)
    {
        set_string(Properties::CombiMode, value);
        return *this;
    }

    InvControl& CombiMode(const char *value)
    {
        set_string(Properties::CombiMode, value);
        return *this;
    }

    /// Combination of smart inverter functions in which the InvControl will control the PC elements in DERList, according to the options below:
    ///
    /// Must be a combination of the following: {VV_VW | VV_DRC}. Default is to not set this property, in which case the single control mode in Mode is active.
    ///
    /// In combined VV_VW mode, both volt-var and volt-watt control modes are active simultaneously.  See help individually for volt-var mode and volt-watt mode in Mode property.
    /// Note that the PVSystem/Storage will attempt to achieve both the volt-watt and volt-var set-points based on the capabilities of the inverter in the PVSystem/Storage (kVA rating, etc), any limits set on maximum active power,
    ///
    /// In combined VV_DRC, both the volt-var and the dynamic reactive current modes are simultaneously active.
    ///
    /// DSS property name: CombiMode, DSS property index: 3
    ///
    string CombiMode_str()
    {
        return get_prop_string(Properties::CombiMode);
    }

    /// Combination of smart inverter functions in which the InvControl will control the PC elements in DERList, according to the options below:
    ///
    /// Must be a combination of the following: {VV_VW | VV_DRC}. Default is to not set this property, in which case the single control mode in Mode is active.
    ///
    /// In combined VV_VW mode, both volt-var and volt-watt control modes are active simultaneously.  See help individually for volt-var mode and volt-watt mode in Mode property.
    /// Note that the PVSystem/Storage will attempt to achieve both the volt-watt and volt-var set-points based on the capabilities of the inverter in the PVSystem/Storage (kVA rating, etc), any limits set on maximum active power,
    ///
    /// In combined VV_DRC, both the volt-var and the dynamic reactive current modes are simultaneously active.
    ///
    /// DSS property name: CombiMode, DSS property index: 3
    ///
    InvControl& CombiMode_str(const string &value)
    {
        set_string(Properties::CombiMode, value);
        return *this;
    }

    /// Required for VOLTVAR mode.
    ///
    /// Name of the XYCurve object containing the volt-var curve. The positive values of the y-axis of the volt-var curve represent values in pu of the provided base reactive power. The negative values of the y-axis are values in pu of the absorbed base reactive power.
    /// Provided and absorbed base reactive power values are defined in the RefReactivePower property
    ///
    /// Units for the x-axis are per-unit voltage, which may be in per unit of the rated voltage for the PVSystem/Storage, or may be in per unit of the average voltage at the terminals over a user-defined number of prior solutions.
    ///
    /// DSS property name: vvc_curve1, DSS property index: 4
    ///
    string vvc_curve1()
    {
        return get_prop_string(Properties::vvc_curve1);
    }

    InvControl& vvc_curve1(const string &value)
    {
        set_string(Properties::vvc_curve1, value);
        return *this;
    }

    InvControl& vvc_curve1(dss::obj::XYcurve &value)
    {
        set_obj(Properties::vvc_curve1, value);
        return *this;
    }

    /// Required for VOLTVAR mode.
    ///
    /// Name of the XYCurve object containing the volt-var curve. The positive values of the y-axis of the volt-var curve represent values in pu of the provided base reactive power. The negative values of the y-axis are values in pu of the absorbed base reactive power.
    /// Provided and absorbed base reactive power values are defined in the RefReactivePower property
    ///
    /// Units for the x-axis are per-unit voltage, which may be in per unit of the rated voltage for the PVSystem/Storage, or may be in per unit of the average voltage at the terminals over a user-defined number of prior solutions.
    ///
    /// DSS property name: vvc_curve1, DSS property index: 4
    ///
    dss::obj::XYcurve vvc_curve1_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::vvc_curve1);
    }

    InvControl& vvc_curve1_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::vvc_curve1, value);
        return *this;
    }

    /// Required for VOLTVAR mode, and defaults to 0.
    ///
    /// for the times when the terminal voltage is decreasing, this is the off-set in per-unit voltage of a curve whose shape is the same as vvc_curve. It is offset by a certain negative value of per-unit voltage, which is defined by the base quantity for the x-axis of the volt-var curve (see help for voltage_curvex_ref)
    ///
    /// if the PVSystem/Storage terminal voltage has been increasing, and has not changed directions, utilize vvc_curve1 for the volt-var response.
    ///
    /// if the PVSystem/Storage terminal voltage has been increasing and changes directions and begins to decrease, then move from utilizing vvc_curve1 to a volt-var curve of the same shape, but offset by a certain per-unit voltage value.
    ///
    /// Maintain the same per-unit available var output level (unless head-room has changed due to change in active power or kva rating of PVSystem/Storage).  Per-unit var values remain the same for this internally constructed second curve (hysteresis curve).
    ///
    /// if the terminal voltage has been decreasing and changes directions and begins to increase , then move from utilizing the offset curve, back to the vvc_curve1 for volt-var response, but stay at the same per-unit available vars output level.
    ///
    /// DSS property name: hysteresis_offset, DSS property index: 5
    ///
    double hysteresis_offset()
    {
        return Obj_GetFloat64(ptr, Properties::hysteresis_offset);
    }

    InvControl& hysteresis_offset(double value)
    {
        Obj_SetFloat64(ptr, Properties::hysteresis_offset, value);
        return *this;
    }

    /// Required for VOLTVAR and VOLTWATT modes, and defaults to rated.  Possible values are: {rated|avg|ravg}.
    ///
    /// Defines whether the x-axis values (voltage in per unit) for vvc_curve1 and the volt-watt curve corresponds to:
    ///
    /// rated. The rated voltage for the PVSystem/Storage object (1.0 in the volt-var curve equals rated voltage).
    ///
    /// avg. The average terminal voltage recorded over a certain number of prior power-flow solutions.
    /// with the avg setting, 1.0 per unit on the x-axis of the volt-var curve(s) corresponds to the average voltage.
    /// from a certain number of prior intervals.  See avgwindowlen parameter.
    ///
    /// ravg. Same as avg, with the exception that the avgerage terminal voltage is divided by the rated voltage.
    ///
    /// DSS property name: voltage_curvex_ref, DSS property index: 6
    ///
    InvControlVoltageCurveXRef voltage_curvex_ref()
    {
        return InvControlVoltageCurveXRef(Obj_GetInt32(ptr, Properties::voltage_curvex_ref));
    }

    InvControl& voltage_curvex_ref(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::voltage_curvex_ref, value);
        return *this;
    }

    InvControl& voltage_curvex_ref(InvControlVoltageCurveXRef value)
    {
        Obj_SetInt32(ptr, Properties::voltage_curvex_ref, int32_t(value));
        return *this;
    }

    InvControl& voltage_curvex_ref(const string &value)
    {
        set_string(Properties::voltage_curvex_ref, value);
        return *this;
    }

    InvControl& voltage_curvex_ref(const char *value)
    {
        set_string(Properties::voltage_curvex_ref, value);
        return *this;
    }

    /// Required for VOLTVAR and VOLTWATT modes, and defaults to rated.  Possible values are: {rated|avg|ravg}.
    ///
    /// Defines whether the x-axis values (voltage in per unit) for vvc_curve1 and the volt-watt curve corresponds to:
    ///
    /// rated. The rated voltage for the PVSystem/Storage object (1.0 in the volt-var curve equals rated voltage).
    ///
    /// avg. The average terminal voltage recorded over a certain number of prior power-flow solutions.
    /// with the avg setting, 1.0 per unit on the x-axis of the volt-var curve(s) corresponds to the average voltage.
    /// from a certain number of prior intervals.  See avgwindowlen parameter.
    ///
    /// ravg. Same as avg, with the exception that the avgerage terminal voltage is divided by the rated voltage.
    ///
    /// DSS property name: voltage_curvex_ref, DSS property index: 6
    ///
    string voltage_curvex_ref_str()
    {
        return get_prop_string(Properties::voltage_curvex_ref);
    }

    /// Required for VOLTVAR and VOLTWATT modes, and defaults to rated.  Possible values are: {rated|avg|ravg}.
    ///
    /// Defines whether the x-axis values (voltage in per unit) for vvc_curve1 and the volt-watt curve corresponds to:
    ///
    /// rated. The rated voltage for the PVSystem/Storage object (1.0 in the volt-var curve equals rated voltage).
    ///
    /// avg. The average terminal voltage recorded over a certain number of prior power-flow solutions.
    /// with the avg setting, 1.0 per unit on the x-axis of the volt-var curve(s) corresponds to the average voltage.
    /// from a certain number of prior intervals.  See avgwindowlen parameter.
    ///
    /// ravg. Same as avg, with the exception that the avgerage terminal voltage is divided by the rated voltage.
    ///
    /// DSS property name: voltage_curvex_ref, DSS property index: 6
    ///
    InvControl& voltage_curvex_ref_str(const string &value)
    {
        set_string(Properties::voltage_curvex_ref, value);
        return *this;
    }

    /// Required for VOLTVAR mode and VOLTWATT mode, and defaults to 0 seconds (0s).
    ///
    /// Sets the length of the averaging window over which the average PVSystem/Storage terminal voltage is calculated.
    ///
    /// Units are indicated by appending s, m, or h to the integer value.
    ///
    /// The averaging window will calculate the average PVSystem/Storage terminal voltage over the specified period of time, up to and including the last power flow solution.
    ///
    /// Note, if the solution stepsize is larger than the window length, then the voltage will be assumed to have been constant over the time-frame specified by the window length.
    ///
    /// DSS property name: avgwindowlen, DSS property index: 7
    ///
    int32_t avgwindowlen()
    {
        return Obj_GetInt32(ptr, Properties::avgwindowlen);
    }

    InvControl& avgwindowlen(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::avgwindowlen, value);
        return *this;
    }

    /// Required for VOLTWATT mode.
    ///
    /// Name of the XYCurve object containing the volt-watt curve.
    ///
    /// Units for the x-axis are per-unit voltage, which may be in per unit of the rated voltage for the PVSystem/Storage, or may be in per unit of the average voltage at the terminals over a user-defined number of prior solutions.
    ///
    /// Units for the y-axis are either in one of the options described in the VoltwattYAxis property.
    ///
    /// DSS property name: voltwatt_curve, DSS property index: 8
    ///
    string voltwatt_curve()
    {
        return get_prop_string(Properties::voltwatt_curve);
    }

    InvControl& voltwatt_curve(const string &value)
    {
        set_string(Properties::voltwatt_curve, value);
        return *this;
    }

    InvControl& voltwatt_curve(dss::obj::XYcurve &value)
    {
        set_obj(Properties::voltwatt_curve, value);
        return *this;
    }

    /// Required for VOLTWATT mode.
    ///
    /// Name of the XYCurve object containing the volt-watt curve.
    ///
    /// Units for the x-axis are per-unit voltage, which may be in per unit of the rated voltage for the PVSystem/Storage, or may be in per unit of the average voltage at the terminals over a user-defined number of prior solutions.
    ///
    /// Units for the y-axis are either in one of the options described in the VoltwattYAxis property.
    ///
    /// DSS property name: voltwatt_curve, DSS property index: 8
    ///
    dss::obj::XYcurve voltwatt_curve_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::voltwatt_curve);
    }

    InvControl& voltwatt_curve_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::voltwatt_curve, value);
        return *this;
    }

    /// Required for the dynamic reactive current mode (DYNAMICREACCURR), and defaults to 0.95 per-unit voltage (referenced to the PVSystem/Storage object rated voltage or a windowed average value).
    ///
    /// This parameter is the minimum voltage that defines the voltage dead-band within which no reactive power is allowed to be generated.
    ///
    /// DSS property name: DbVMin, DSS property index: 9
    ///
    double DbVMin()
    {
        return Obj_GetFloat64(ptr, Properties::DbVMin);
    }

    InvControl& DbVMin(double value)
    {
        Obj_SetFloat64(ptr, Properties::DbVMin, value);
        return *this;
    }

    /// Required for the dynamic reactive current mode (DYNAMICREACCURR), and defaults to 1.05 per-unit voltage (referenced to the PVSystem object rated voltage or a windowed average value).
    ///
    /// This parameter is the maximum voltage that defines the voltage dead-band within which no reactive power is allowed to be generated.
    ///
    /// DSS property name: DbVMax, DSS property index: 10
    ///
    double DbVMax()
    {
        return Obj_GetFloat64(ptr, Properties::DbVMax);
    }

    InvControl& DbVMax(double value)
    {
        Obj_SetFloat64(ptr, Properties::DbVMax, value);
        return *this;
    }

    /// Required for the dynamic reactive current mode (DYNAMICREACCURR), and defaults to 0.1
    ///
    /// This is a gradient, expressed in unit-less terms of %/%, to establish the ratio by which percentage capacitive reactive power production is increased as the  percent delta-voltage decreases below DbVMin.
    ///
    /// Percent delta-voltage is defined as the present PVSystem/Storage terminal voltage minus the moving average voltage, expressed as a percentage of the rated voltage for the PVSystem/Storage object.
    ///
    /// Note, the moving average voltage for the dynamic reactive current mode is different than the moving average voltage for the volt-watt and volt-var modes.
    ///
    /// DSS property name: ArGraLowV, DSS property index: 11
    ///
    double ArGraLowV()
    {
        return Obj_GetFloat64(ptr, Properties::ArGraLowV);
    }

    InvControl& ArGraLowV(double value)
    {
        Obj_SetFloat64(ptr, Properties::ArGraLowV, value);
        return *this;
    }

    /// Required for the dynamic reactive current mode (DYNAMICREACCURR), and defaults to 0.1
    ///
    /// This is a gradient, expressed in unit-less terms of %/%, to establish the ratio by which percentage inductive reactive power production is increased as the  percent delta-voltage decreases above DbVMax.
    ///
    /// Percent delta-voltage is defined as the present PVSystem/Storage terminal voltage minus the moving average voltage, expressed as a percentage of the rated voltage for the PVSystem/Storage object.
    ///
    /// Note, the moving average voltage for the dynamic reactive current mode is different than the mmoving average voltage for the volt-watt and volt-var modes.
    ///
    /// DSS property name: ArGraHiV, DSS property index: 12
    ///
    double ArGraHiV()
    {
        return Obj_GetFloat64(ptr, Properties::ArGraHiV);
    }

    InvControl& ArGraHiV(double value)
    {
        Obj_SetFloat64(ptr, Properties::ArGraHiV, value);
        return *this;
    }

    /// Required for the dynamic reactive current mode (DYNAMICREACCURR), and defaults to 1 seconds (1s). do not use a value smaller than 1.0
    ///
    /// Sets the length of the averaging window over which the average PVSystem/Storage terminal voltage is calculated for the dynamic reactive current mode.
    ///
    /// Units are indicated by appending s, m, or h to the integer value.
    ///
    /// Typically this will be a shorter averaging window than the volt-var and volt-watt averaging window.
    ///
    /// The averaging window will calculate the average PVSystem/Storage terminal voltage over the specified period of time, up to and including the last power flow solution.  Note, if the solution stepsize is larger than the window length, then the voltage will be assumed to have been constant over the time-frame specified by the window length.
    ///
    /// DSS property name: DynReacavgwindowlen, DSS property index: 13
    ///
    int32_t DynReacavgwindowlen()
    {
        return Obj_GetInt32(ptr, Properties::DynReacavgwindowlen);
    }

    InvControl& DynReacavgwindowlen(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::DynReacavgwindowlen, value);
        return *this;
    }

    /// Required for the VOLTVAR and DYNAMICREACCURR modes.  Defaults to -1.0.
    ///
    /// Defining -1.0, OpenDSS takes care internally of delta_Q itself. It tries to improve convergence as well as speed up process
    ///
    /// Sets the maximum change (in per unit) from the prior var output level to the desired var output level during each control iteration.
    ///
    ///
    /// if numerical instability is noticed in solutions such as var sign changing from one control iteration to the next and voltages oscillating between two values with some separation, this is an indication of numerical instability (use the EventLog to diagnose).
    ///
    /// if the maximum control iterations are exceeded, and no numerical instability is seen in the EventLog of via monitors, then try increasing the value of this parameter to reduce the number of control iterations needed to achieve the control criteria, and move to the power flow solution.
    ///
    /// DSS property name: deltaQ_Factor, DSS property index: 14
    ///
    double deltaQ_Factor()
    {
        return Obj_GetFloat64(ptr, Properties::deltaQ_Factor);
    }

    InvControl& deltaQ_Factor(double value)
    {
        Obj_SetFloat64(ptr, Properties::deltaQ_Factor, value);
        return *this;
    }

    /// Defaults to 0.0001 per-unit voltage.  This parameter should only be modified by advanced users of the InvControl.
    ///
    /// Tolerance in pu of the control loop convergence associated to the monitored voltage in pu. This value is compared with the difference of the monitored voltage in pu of the current and previous control iterations of the control loop
    ///
    /// This voltage tolerance value plus the var/watt tolerance value (VarChangeTolerance/ActivePChangeTolerance) determine, together, when to stop control iterations by the InvControl.
    ///
    /// If an InvControl is controlling more than one PVSystem/Storage, each PVSystem/Storage has this quantity calculated independently, and so an individual PVSystem/Storage may reach the tolerance within different numbers of control iterations.
    ///
    /// DSS property name: VoltageChangeTolerance, DSS property index: 15
    ///
    double VoltageChangeTolerance()
    {
        return Obj_GetFloat64(ptr, Properties::VoltageChangeTolerance);
    }

    InvControl& VoltageChangeTolerance(double value)
    {
        Obj_SetFloat64(ptr, Properties::VoltageChangeTolerance, value);
        return *this;
    }

    /// Required for VOLTVAR and DYNAMICREACCURR modes.  Defaults to 0.025 per unit of the base provided or absorbed reactive power described in the RefReactivePower property This parameter should only be modified by advanced users of the InvControl.
    ///
    /// Tolerance in pu of the convergence of the control loop associated with reactive power. For the same control iteration, this value is compared to the difference, as an absolute value (without sign), between the desired reactive power value in pu and the output reactive power in pu of the controlled element.
    ///
    /// This reactive power tolerance value plus the voltage tolerance value (VoltageChangeTolerance) determine, together, when to stop control iterations by the InvControl.
    ///
    /// If an InvControl is controlling more than one PVSystem/Storage, each PVSystem/Storage has this quantity calculated independently, and so an individual PVSystem/Storage may reach the tolerance within different numbers of control iterations.
    ///
    /// DSS property name: VarChangeTolerance, DSS property index: 16
    ///
    double VarChangeTolerance()
    {
        return Obj_GetFloat64(ptr, Properties::VarChangeTolerance);
    }

    InvControl& VarChangeTolerance(double value)
    {
        Obj_SetFloat64(ptr, Properties::VarChangeTolerance, value);
        return *this;
    }

    /// Required for VOLTWATT mode.  Must be one of: {PMPPPU* | PAVAILABLEPU| PCTPMPPPU | KVARATINGPU}.  The default is PMPPPU.
    ///
    /// Units for the y-axis of the volt-watt curve while in volt-watt mode.
    ///
    /// When set to PMPPPU. The y-axis corresponds to the value in pu of Pmpp property of the PVSystem.
    ///
    /// When set to PAVAILABLEPU. The y-axis corresponds to the value in pu of the available active power of the PVSystem.
    ///
    /// When set to PCTPMPPPU. The y-axis corresponds to the value in pu of the power Pmpp multiplied by 1/100 of the %Pmpp property of the PVSystem.
    ///
    /// When set to KVARATINGPU. The y-axis corresponds to the value in pu of the kVA property of the PVSystem.
    ///
    /// DSS property name: VoltwattYAxis, DSS property index: 17
    ///
    InvControlVoltWattYAxis VoltwattYAxis()
    {
        return InvControlVoltWattYAxis(Obj_GetInt32(ptr, Properties::VoltwattYAxis));
    }

    InvControl& VoltwattYAxis(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::VoltwattYAxis, value);
        return *this;
    }

    InvControl& VoltwattYAxis(InvControlVoltWattYAxis value)
    {
        Obj_SetInt32(ptr, Properties::VoltwattYAxis, int32_t(value));
        return *this;
    }

    InvControl& VoltwattYAxis(const string &value)
    {
        set_string(Properties::VoltwattYAxis, value);
        return *this;
    }

    InvControl& VoltwattYAxis(const char *value)
    {
        set_string(Properties::VoltwattYAxis, value);
        return *this;
    }

    /// Required for VOLTWATT mode.  Must be one of: {PMPPPU* | PAVAILABLEPU| PCTPMPPPU | KVARATINGPU}.  The default is PMPPPU.
    ///
    /// Units for the y-axis of the volt-watt curve while in volt-watt mode.
    ///
    /// When set to PMPPPU. The y-axis corresponds to the value in pu of Pmpp property of the PVSystem.
    ///
    /// When set to PAVAILABLEPU. The y-axis corresponds to the value in pu of the available active power of the PVSystem.
    ///
    /// When set to PCTPMPPPU. The y-axis corresponds to the value in pu of the power Pmpp multiplied by 1/100 of the %Pmpp property of the PVSystem.
    ///
    /// When set to KVARATINGPU. The y-axis corresponds to the value in pu of the kVA property of the PVSystem.
    ///
    /// DSS property name: VoltwattYAxis, DSS property index: 17
    ///
    string VoltwattYAxis_str()
    {
        return get_prop_string(Properties::VoltwattYAxis);
    }

    /// Required for VOLTWATT mode.  Must be one of: {PMPPPU* | PAVAILABLEPU| PCTPMPPPU | KVARATINGPU}.  The default is PMPPPU.
    ///
    /// Units for the y-axis of the volt-watt curve while in volt-watt mode.
    ///
    /// When set to PMPPPU. The y-axis corresponds to the value in pu of Pmpp property of the PVSystem.
    ///
    /// When set to PAVAILABLEPU. The y-axis corresponds to the value in pu of the available active power of the PVSystem.
    ///
    /// When set to PCTPMPPPU. The y-axis corresponds to the value in pu of the power Pmpp multiplied by 1/100 of the %Pmpp property of the PVSystem.
    ///
    /// When set to KVARATINGPU. The y-axis corresponds to the value in pu of the kVA property of the PVSystem.
    ///
    /// DSS property name: VoltwattYAxis, DSS property index: 17
    ///
    InvControl& VoltwattYAxis_str(const string &value)
    {
        set_string(Properties::VoltwattYAxis, value);
        return *this;
    }

    /// Required for VOLTWATT and VOLTVAR mode.  Must be one of: {INACTIVE* | LPF | RISEFALL }.  The default is INACTIVE.
    ///
    /// Auxiliary option that aims to limit the changes of the desired reactive power and the active power limit between time steps, the alternatives are listed below:
    ///
    /// INACTIVE. It indicates there is no limit on rate of change imposed for either active or reactive power output.
    ///
    /// LPF. A low-pass RC filter is applied to the desired reactive power and/or the active power limit to determine the output power as a function of a time constant defined in the LPFTau property.
    ///
    /// RISEFALL. A rise and fall limit in the change of active and/or reactive power expressed in terms of pu power per second, defined in the RiseFallLimit, is applied to the desired reactive power and/or the active power limit.
    ///
    /// DSS property name: RateofChangeMode, DSS property index: 18
    ///
    InvControlRateOfChangeMode RateofChangeMode()
    {
        return InvControlRateOfChangeMode(Obj_GetInt32(ptr, Properties::RateofChangeMode));
    }

    InvControl& RateofChangeMode(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::RateofChangeMode, value);
        return *this;
    }

    InvControl& RateofChangeMode(InvControlRateOfChangeMode value)
    {
        Obj_SetInt32(ptr, Properties::RateofChangeMode, int32_t(value));
        return *this;
    }

    InvControl& RateofChangeMode(const string &value)
    {
        set_string(Properties::RateofChangeMode, value);
        return *this;
    }

    InvControl& RateofChangeMode(const char *value)
    {
        set_string(Properties::RateofChangeMode, value);
        return *this;
    }

    /// Required for VOLTWATT and VOLTVAR mode.  Must be one of: {INACTIVE* | LPF | RISEFALL }.  The default is INACTIVE.
    ///
    /// Auxiliary option that aims to limit the changes of the desired reactive power and the active power limit between time steps, the alternatives are listed below:
    ///
    /// INACTIVE. It indicates there is no limit on rate of change imposed for either active or reactive power output.
    ///
    /// LPF. A low-pass RC filter is applied to the desired reactive power and/or the active power limit to determine the output power as a function of a time constant defined in the LPFTau property.
    ///
    /// RISEFALL. A rise and fall limit in the change of active and/or reactive power expressed in terms of pu power per second, defined in the RiseFallLimit, is applied to the desired reactive power and/or the active power limit.
    ///
    /// DSS property name: RateofChangeMode, DSS property index: 18
    ///
    string RateofChangeMode_str()
    {
        return get_prop_string(Properties::RateofChangeMode);
    }

    /// Required for VOLTWATT and VOLTVAR mode.  Must be one of: {INACTIVE* | LPF | RISEFALL }.  The default is INACTIVE.
    ///
    /// Auxiliary option that aims to limit the changes of the desired reactive power and the active power limit between time steps, the alternatives are listed below:
    ///
    /// INACTIVE. It indicates there is no limit on rate of change imposed for either active or reactive power output.
    ///
    /// LPF. A low-pass RC filter is applied to the desired reactive power and/or the active power limit to determine the output power as a function of a time constant defined in the LPFTau property.
    ///
    /// RISEFALL. A rise and fall limit in the change of active and/or reactive power expressed in terms of pu power per second, defined in the RiseFallLimit, is applied to the desired reactive power and/or the active power limit.
    ///
    /// DSS property name: RateofChangeMode, DSS property index: 18
    ///
    InvControl& RateofChangeMode_str(const string &value)
    {
        set_string(Properties::RateofChangeMode, value);
        return *this;
    }

    /// Not required. Defaults to 0 seconds.
    ///
    /// Filter time constant of the LPF option of the RateofChangeMode property. The time constant will cause the low-pass filter to achieve 95% of the target value in 3 time constants.
    ///
    /// DSS property name: LPFTau, DSS property index: 19
    ///
    double LPFTau()
    {
        return Obj_GetFloat64(ptr, Properties::LPFTau);
    }

    InvControl& LPFTau(double value)
    {
        Obj_SetFloat64(ptr, Properties::LPFTau, value);
        return *this;
    }

    /// Not required.  Defaults to no limit (-1). Must be -1 (no limit) or a positive value.
    ///
    /// Limit in power in pu per second used by the RISEFALL option of the RateofChangeMode property.The base value for this ramp is defined in the RefReactivePower property and/or in VoltwattYAxis.
    ///
    /// DSS property name: RiseFallLimit, DSS property index: 20
    ///
    double RiseFallLimit()
    {
        return Obj_GetFloat64(ptr, Properties::RiseFallLimit);
    }

    InvControl& RiseFallLimit(double value)
    {
        Obj_SetFloat64(ptr, Properties::RiseFallLimit, value);
        return *this;
    }

    /// Required for the VOLTWATT modes.  Defaults to -1.0.
    ///
    /// Defining -1.0, OpenDSS takes care internally of delta_P itself. It tries to improve convergence as well as speed up process
    ///
    /// Defining between 0.05 and 1.0, it sets the maximum change (in unit of the y-axis) from the prior active power output level to the desired active power output level during each control iteration.
    ///
    ///
    /// If numerical instability is noticed in solutions such as active power changing substantially from one control iteration to the next and/or voltages oscillating between two values with some separation, this is an indication of numerical instability (use the EventLog to diagnose).
    ///
    /// If the maximum control iterations are exceeded, and no numerical instability is seen in the EventLog of via monitors, then try increasing the value of this parameter to reduce the number of control iterations needed to achieve the control criteria, and move to the power flow solution.
    ///
    /// DSS property name: deltaP_Factor, DSS property index: 21
    ///
    double deltaP_Factor()
    {
        return Obj_GetFloat64(ptr, Properties::deltaP_Factor);
    }

    InvControl& deltaP_Factor(double value)
    {
        Obj_SetFloat64(ptr, Properties::deltaP_Factor, value);
        return *this;
    }

    /// {Yes/True* | No/False} Default is YES for InvControl. Log control actions to Eventlog.
    ///
    /// DSS property name: EventLog, DSS property index: 22
    ///
    bool EventLog()
    {
        return Obj_GetInt32(ptr, Properties::EventLog) != 0;
    }

    InvControl& EventLog(bool value)
    {
        Obj_SetInt32(ptr, Properties::EventLog, value);
        return *this;
    }

    /// Required for any mode that has VOLTVAR, DYNAMICREACCURR and WATTVAR. Defaults to VARAVAL.
    ///
    /// Defines the base reactive power for both the provided and absorbed reactive power, according to one of the following options:
    ///
    /// VARAVAL. The base values for the provided and absorbed reactive power are equal to the available reactive power.
    ///
    /// VARMAX: The base values of the provided and absorbed reactive power are equal to the value defined in the kvarMax and kvarMaxAbs properties, respectively.
    ///
    /// DSS property name: RefReactivePower, DSS property index: 23
    ///
    InvControlReactivePowerReference RefReactivePower()
    {
        return InvControlReactivePowerReference(Obj_GetInt32(ptr, Properties::RefReactivePower));
    }

    InvControl& RefReactivePower(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::RefReactivePower, value);
        return *this;
    }

    InvControl& RefReactivePower(InvControlReactivePowerReference value)
    {
        Obj_SetInt32(ptr, Properties::RefReactivePower, int32_t(value));
        return *this;
    }

    InvControl& RefReactivePower(const string &value)
    {
        set_string(Properties::RefReactivePower, value);
        return *this;
    }

    InvControl& RefReactivePower(const char *value)
    {
        set_string(Properties::RefReactivePower, value);
        return *this;
    }

    /// Required for any mode that has VOLTVAR, DYNAMICREACCURR and WATTVAR. Defaults to VARAVAL.
    ///
    /// Defines the base reactive power for both the provided and absorbed reactive power, according to one of the following options:
    ///
    /// VARAVAL. The base values for the provided and absorbed reactive power are equal to the available reactive power.
    ///
    /// VARMAX: The base values of the provided and absorbed reactive power are equal to the value defined in the kvarMax and kvarMaxAbs properties, respectively.
    ///
    /// DSS property name: RefReactivePower, DSS property index: 23
    ///
    string RefReactivePower_str()
    {
        return get_prop_string(Properties::RefReactivePower);
    }

    /// Required for any mode that has VOLTVAR, DYNAMICREACCURR and WATTVAR. Defaults to VARAVAL.
    ///
    /// Defines the base reactive power for both the provided and absorbed reactive power, according to one of the following options:
    ///
    /// VARAVAL. The base values for the provided and absorbed reactive power are equal to the available reactive power.
    ///
    /// VARMAX: The base values of the provided and absorbed reactive power are equal to the value defined in the kvarMax and kvarMaxAbs properties, respectively.
    ///
    /// DSS property name: RefReactivePower, DSS property index: 23
    ///
    InvControl& RefReactivePower_str(const string &value)
    {
        set_string(Properties::RefReactivePower, value);
        return *this;
    }

    /// Required for VOLTWATT. Default is 0.01
    ///
    /// Tolerance in pu of the convergence of the control loop associated with active power. For the same control iteration, this value is compared to the difference between the active power limit in pu resulted from the convergence process and the one resulted from the volt-watt function.
    ///
    /// This reactive power tolerance value plus the voltage tolerance value (VoltageChangeTolerance) determine, together, when to stop control iterations by the InvControl.
    ///
    /// If an InvControl is controlling more than one PVSystem/Storage, each PVSystem/Storage has this quantity calculated independently, and so an individual PVSystem/Storage may reach the tolerance within different numbers of control iterations.
    ///
    /// DSS property name: ActivePChangeTolerance, DSS property index: 24
    ///
    double ActivePChangeTolerance()
    {
        return Obj_GetFloat64(ptr, Properties::ActivePChangeTolerance);
    }

    InvControl& ActivePChangeTolerance(double value)
    {
        Obj_SetFloat64(ptr, Properties::ActivePChangeTolerance, value);
        return *this;
    }

    /// Number of the phase being monitored or one of {AVG | MAX | MIN} for all phases. Default=AVG.
    ///
    /// DSS property name: monVoltageCalc, DSS property index: 25
    ///
    int32_t monVoltageCalc()
    {
        return Obj_GetInt32(ptr, Properties::monVoltageCalc);
    }

    InvControl& monVoltageCalc(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::monVoltageCalc, value);
        return *this;
    }

    InvControl& monVoltageCalc(MonitoredPhase value)
    {
        Obj_SetInt32(ptr, Properties::monVoltageCalc, int32_t(value));
        return *this;
    }

    InvControl& monVoltageCalc(const string &value)
    {
        set_string(Properties::monVoltageCalc, value);
        return *this;
    }

    InvControl& monVoltageCalc(const char *value)
    {
        set_string(Properties::monVoltageCalc, value);
        return *this;
    }

    /// Number of the phase being monitored or one of {AVG | MAX | MIN} for all phases. Default=AVG.
    ///
    /// DSS property name: monVoltageCalc, DSS property index: 25
    ///
    string monVoltageCalc_str()
    {
        return get_prop_string(Properties::monVoltageCalc);
    }

    /// Number of the phase being monitored or one of {AVG | MAX | MIN} for all phases. Default=AVG.
    ///
    /// DSS property name: monVoltageCalc, DSS property index: 25
    ///
    InvControl& monVoltageCalc_str(const string &value)
    {
        set_string(Properties::monVoltageCalc, value);
        return *this;
    }

    /// Name of monitored bus used by the voltage-dependente control modes. Default is bus of the controlled PVSystem/Storage or Storage.
    ///
    /// DSS property name: monBus, DSS property index: 26
    ///
    strings monBus()
    {
        return get_array<strings>(Properties::monBus);
    }

    InvControl& monBus(strings &value)
    {
        set_array<strings>(Properties::monBus, value);
        return *this;
    }

    /// Array list of rated voltages of the buses and their nodes presented in the monBus property. This list may have different line-to-line and/or line-to-ground voltages.
    ///
    /// DSS property name: MonBusesVbase, DSS property index: 27
    ///
    VectorXd MonBusesVbase()
    {
        return get_array<VectorXd>(Properties::MonBusesVbase);
    }

    InvControl& MonBusesVbase(VectorXd &value)
    {
        set_array<VectorXd>(Properties::MonBusesVbase, value);
        return *this;
    }

    /// Required for VOLTWATT mode for Storage element in CHARGING state.
    ///
    /// The name of an XYCurve object that describes the variation in active power output (in per unit of maximum active power outut for the Storage).
    ///
    /// Units for the x-axis are per-unit voltage, which may be in per unit of the rated voltage for the Storage, or may be in per unit of the average voltage at the terminals over a user-defined number of prior solutions.
    ///
    /// Units for the y-axis are either in: (1) per unit of maximum active power output capability of the Storage, or (2) maximum available active power output capability (defined by the parameter: VoltwattYAxis), corresponding to the terminal voltage (x-axis value in per unit).
    ///
    /// No default -- must be specified for VOLTWATT mode for Storage element in CHARGING state.
    ///
    /// DSS property name: voltwattCH_curve, DSS property index: 28
    ///
    string voltwattCH_curve()
    {
        return get_prop_string(Properties::voltwattCH_curve);
    }

    InvControl& voltwattCH_curve(const string &value)
    {
        set_string(Properties::voltwattCH_curve, value);
        return *this;
    }

    InvControl& voltwattCH_curve(dss::obj::XYcurve &value)
    {
        set_obj(Properties::voltwattCH_curve, value);
        return *this;
    }

    /// Required for VOLTWATT mode for Storage element in CHARGING state.
    ///
    /// The name of an XYCurve object that describes the variation in active power output (in per unit of maximum active power outut for the Storage).
    ///
    /// Units for the x-axis are per-unit voltage, which may be in per unit of the rated voltage for the Storage, or may be in per unit of the average voltage at the terminals over a user-defined number of prior solutions.
    ///
    /// Units for the y-axis are either in: (1) per unit of maximum active power output capability of the Storage, or (2) maximum available active power output capability (defined by the parameter: VoltwattYAxis), corresponding to the terminal voltage (x-axis value in per unit).
    ///
    /// No default -- must be specified for VOLTWATT mode for Storage element in CHARGING state.
    ///
    /// DSS property name: voltwattCH_curve, DSS property index: 28
    ///
    dss::obj::XYcurve voltwattCH_curve_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::voltwattCH_curve);
    }

    InvControl& voltwattCH_curve_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::voltwattCH_curve, value);
        return *this;
    }

    /// Required for WATTPF mode.
    ///
    /// Name of the XYCurve object containing the watt-pf curve.
    /// The positive values of the y-axis are positive power factor values. The negative values of the the y-axis are negative power factor values. When positive, the output reactive power has the same direction of the output active power, and when negative, it has the opposite direction.
    /// Units for the x-axis are per-unit output active power, and the base active power is the Pmpp for PVSystem and kWrated for Storage.
    ///
    /// The y-axis represents the power factor and the reference is power factor equal to 0.
    ///
    /// For example, if the user wants to define the following XY coordinates: (0, 0.9); (0.2, 0.9); (0.5, -0.9); (1, -0.9).
    /// Try to plot them considering the y-axis reference equal to unity power factor.
    ///
    /// The user needs to translate this curve into a plot in which the y-axis reference is equal to 0 power factor.It means that two new XY coordinates need to be included, in this case they are: (0.35, 1); (0.35, -1).
    /// Try to plot them considering the y-axis reference equal to 0 power factor.
    /// The discontinity in 0.35pu is not a problem since var is zero for either power factor equal to 1 or -1.
    ///
    /// DSS property name: wattpf_curve, DSS property index: 29
    ///
    string wattpf_curve()
    {
        return get_prop_string(Properties::wattpf_curve);
    }

    InvControl& wattpf_curve(const string &value)
    {
        set_string(Properties::wattpf_curve, value);
        return *this;
    }

    InvControl& wattpf_curve(dss::obj::XYcurve &value)
    {
        set_obj(Properties::wattpf_curve, value);
        return *this;
    }

    /// Required for WATTPF mode.
    ///
    /// Name of the XYCurve object containing the watt-pf curve.
    /// The positive values of the y-axis are positive power factor values. The negative values of the the y-axis are negative power factor values. When positive, the output reactive power has the same direction of the output active power, and when negative, it has the opposite direction.
    /// Units for the x-axis are per-unit output active power, and the base active power is the Pmpp for PVSystem and kWrated for Storage.
    ///
    /// The y-axis represents the power factor and the reference is power factor equal to 0.
    ///
    /// For example, if the user wants to define the following XY coordinates: (0, 0.9); (0.2, 0.9); (0.5, -0.9); (1, -0.9).
    /// Try to plot them considering the y-axis reference equal to unity power factor.
    ///
    /// The user needs to translate this curve into a plot in which the y-axis reference is equal to 0 power factor.It means that two new XY coordinates need to be included, in this case they are: (0.35, 1); (0.35, -1).
    /// Try to plot them considering the y-axis reference equal to 0 power factor.
    /// The discontinity in 0.35pu is not a problem since var is zero for either power factor equal to 1 or -1.
    ///
    /// DSS property name: wattpf_curve, DSS property index: 29
    ///
    dss::obj::XYcurve wattpf_curve_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::wattpf_curve);
    }

    InvControl& wattpf_curve_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::wattpf_curve, value);
        return *this;
    }

    /// Required for WATTVAR mode.
    ///
    /// Name of the XYCurve object containing the watt-var curve. The positive values of the y-axis of the watt-var curve represent values in pu of the provided base reactive power. The negative values of the y-axis are values in pu of the absorbed base reactive power.
    /// Provided and absorbed base reactive power values are defined in the RefReactivePower property.
    ///
    /// Units for the x-axis are per-unit output active power, and the base active power is the Pmpp for PVSystem and kWrated for Storage.
    ///
    /// DSS property name: wattvar_curve, DSS property index: 30
    ///
    string wattvar_curve()
    {
        return get_prop_string(Properties::wattvar_curve);
    }

    InvControl& wattvar_curve(const string &value)
    {
        set_string(Properties::wattvar_curve, value);
        return *this;
    }

    InvControl& wattvar_curve(dss::obj::XYcurve &value)
    {
        set_obj(Properties::wattvar_curve, value);
        return *this;
    }

    /// Required for WATTVAR mode.
    ///
    /// Name of the XYCurve object containing the watt-var curve. The positive values of the y-axis of the watt-var curve represent values in pu of the provided base reactive power. The negative values of the y-axis are values in pu of the absorbed base reactive power.
    /// Provided and absorbed base reactive power values are defined in the RefReactivePower property.
    ///
    /// Units for the x-axis are per-unit output active power, and the base active power is the Pmpp for PVSystem and kWrated for Storage.
    ///
    /// DSS property name: wattvar_curve, DSS property index: 30
    ///
    dss::obj::XYcurve wattvar_curve_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::wattvar_curve);
    }

    InvControl& wattvar_curve_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::wattvar_curve, value);
        return *this;
    }

    /// Deprecated, use DERList instead.
    ///
    /// DSS property name: PVSystemList, DSS property index: 31
    ///
    strings PVSystemList()
    {
        return get_array<strings>(Properties::PVSystemList);
    }

    InvControl& PVSystemList(strings &value)
    {
        set_array<strings>(Properties::PVSystemList, value);
        return *this;
    }

    /// Required for Active Voltage Regulation (AVR).
    ///
    /// DSS property name: Vsetpoint, DSS property index: 32
    ///
    double Vsetpoint()
    {
        return Obj_GetFloat64(ptr, Properties::Vsetpoint);
    }

    InvControl& Vsetpoint(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vsetpoint, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 33
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    InvControl& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 34
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    InvControl& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 35
    ///
    InvControl& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 35
    ///
    InvControl& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class ExpControl: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 42;
    struct Properties
    {
        enum {
            PVSystemList = 1,
            Vreg = 2,
            Slope = 3,
            VregTau = 4,
            Qbias = 5,
            VregMin = 6,
            VregMax = 7,
            QmaxLead = 8,
            QmaxLag = 9,
            EventLog = 10,
            DeltaQ_factor = 11,
            PreferQ = 12,
            Tresponse = 13,
            DERList = 14,
            basefreq = 15,
            enabled = 16,
            like = 17,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    ExpControl(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    ExpControl(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the ExpControl element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    ExpControl(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the ExpControl element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    ExpControl& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    ExpControl& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Array list of PVSystems to be controlled.
    ///
    /// If not specified, all PVSystems in the circuit are assumed to be controlled by this ExpControl.
    ///
    /// DSS property name: PVSystemList, DSS property index: 1
    ///
    strings PVSystemList()
    {
        return get_array<strings>(Properties::PVSystemList);
    }

    ExpControl& PVSystemList(strings &value)
    {
        set_array<strings>(Properties::PVSystemList, value);
        return *this;
    }

    /// Per-unit voltage at which reactive power is zero; defaults to 1.0.
    ///
    /// This may dynamically self-adjust when VregTau > 0, limited by VregMin and VregMax.If imput as 0, Vreg will be initialized from a snapshot solution with no inverter Q.The equilibrium point of reactive power is also affected by Qbias
    ///
    /// DSS property name: Vreg, DSS property index: 2
    ///
    double Vreg()
    {
        return Obj_GetFloat64(ptr, Properties::Vreg);
    }

    ExpControl& Vreg(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vreg, value);
        return *this;
    }

    /// Per-unit reactive power injection / per-unit voltage deviation from Vreg; defaults to 50.
    ///
    /// Unlike InvControl, base reactive power is constant at the inverter kva rating.
    ///
    /// DSS property name: Slope, DSS property index: 3
    ///
    double Slope()
    {
        return Obj_GetFloat64(ptr, Properties::Slope);
    }

    ExpControl& Slope(double value)
    {
        Obj_SetFloat64(ptr, Properties::Slope, value);
        return *this;
    }

    /// Time constant for adaptive Vreg. Defaults to 1200 seconds.
    ///
    /// When the control injects or absorbs reactive power due to a voltage deviation from the Q=0 crossing of the volt-var curve, the Q=0 crossing will move toward the actual terminal voltage with this time constant. Over time, the effect is to gradually bring inverter reactive power to zero as the grid voltage changes due to non-solar effects. If zero, then Vreg stays fixed. IEEE1547-2018 requires adjustability from 300s to 5000s
    ///
    /// DSS property name: VregTau, DSS property index: 4
    ///
    double VregTau()
    {
        return Obj_GetFloat64(ptr, Properties::VregTau);
    }

    ExpControl& VregTau(double value)
    {
        Obj_SetFloat64(ptr, Properties::VregTau, value);
        return *this;
    }

    /// Equilibrium per-unit reactive power when V=Vreg; defaults to 0.
    ///
    /// Enter > 0 for lagging (capacitive) bias, < 0 for leading (inductive) bias.
    ///
    /// DSS property name: Qbias, DSS property index: 5
    ///
    double Qbias()
    {
        return Obj_GetFloat64(ptr, Properties::Qbias);
    }

    ExpControl& Qbias(double value)
    {
        Obj_SetFloat64(ptr, Properties::Qbias, value);
        return *this;
    }

    /// Lower limit on adaptive Vreg; defaults to 0.95 per-unit
    ///
    /// DSS property name: VregMin, DSS property index: 6
    ///
    double VregMin()
    {
        return Obj_GetFloat64(ptr, Properties::VregMin);
    }

    ExpControl& VregMin(double value)
    {
        Obj_SetFloat64(ptr, Properties::VregMin, value);
        return *this;
    }

    /// Upper limit on adaptive Vreg; defaults to 1.05 per-unit
    ///
    /// DSS property name: VregMax, DSS property index: 7
    ///
    double VregMax()
    {
        return Obj_GetFloat64(ptr, Properties::VregMax);
    }

    ExpControl& VregMax(double value)
    {
        Obj_SetFloat64(ptr, Properties::VregMax, value);
        return *this;
    }

    /// Limit on leading (inductive) reactive power injection, in per-unit of base kva; defaults to 0.44.For Category A inverters per P1547/D7, set this value to 0.25.
    ///
    /// Regardless of QmaxLead, the reactive power injection is still limited by dynamic headroom when actual real power output exceeds 0%
    ///
    /// DSS property name: QmaxLead, DSS property index: 8
    ///
    double QmaxLead()
    {
        return Obj_GetFloat64(ptr, Properties::QmaxLead);
    }

    ExpControl& QmaxLead(double value)
    {
        Obj_SetFloat64(ptr, Properties::QmaxLead, value);
        return *this;
    }

    /// Limit on lagging (capacitive) reactive power injection, in per-unit of base kva; defaults to 0.44.
    ///
    /// For Category A inverters per P1547/D7, set this value to 0.25.Regardless of QmaxLag, the reactive power injection is still limited by dynamic headroom when actual real power output exceeds 0%
    ///
    /// DSS property name: QmaxLag, DSS property index: 9
    ///
    double QmaxLag()
    {
        return Obj_GetFloat64(ptr, Properties::QmaxLag);
    }

    ExpControl& QmaxLag(double value)
    {
        Obj_SetFloat64(ptr, Properties::QmaxLag, value);
        return *this;
    }

    /// {Yes/True* | No/False} Default is No for ExpControl. Log control actions to Eventlog.
    ///
    /// DSS property name: EventLog, DSS property index: 10
    ///
    bool EventLog()
    {
        return Obj_GetInt32(ptr, Properties::EventLog) != 0;
    }

    ExpControl& EventLog(bool value)
    {
        Obj_SetInt32(ptr, Properties::EventLog, value);
        return *this;
    }

    /// Convergence parameter; Defaults to 0.7.
    ///
    /// Sets the maximum change (in per unit) from the prior var output level to the desired var output level during each control iteration. If numerical instability is noticed in solutions such as var sign changing from one control iteration to the next and voltages oscillating between two values with some separation, this is an indication of numerical instability (use the EventLog to diagnose). If the maximum control iterations are exceeded, and no numerical instability is seen in the EventLog of via monitors, then try increasing the value of this parameter to reduce the number of control iterations needed to achieve the control criteria, and move to the power flow solution.
    ///
    /// DSS property name: DeltaQ_factor, DSS property index: 11
    ///
    double DeltaQ_factor()
    {
        return Obj_GetFloat64(ptr, Properties::DeltaQ_factor);
    }

    ExpControl& DeltaQ_factor(double value)
    {
        Obj_SetFloat64(ptr, Properties::DeltaQ_factor, value);
        return *this;
    }

    /// {Yes/True* | No/False} Default is No for ExpControl.
    ///
    /// Curtails real power output as needed to meet the reactive power requirement. IEEE1547-2018 requires Yes, but the default is No for backward compatibility of OpenDSS models.
    ///
    /// DSS property name: PreferQ, DSS property index: 12
    ///
    bool PreferQ()
    {
        return Obj_GetInt32(ptr, Properties::PreferQ) != 0;
    }

    ExpControl& PreferQ(bool value)
    {
        Obj_SetInt32(ptr, Properties::PreferQ, value);
        return *this;
    }

    /// Open-loop response time for changes in Q.
    ///
    /// The value of Q reaches 90% of the target change within Tresponse, which corresponds to a low-pass filter having tau = Tresponse / 2.3026. The behavior is similar to LPFTAU in InvControl, but here the response time is input instead of the time constant. IEEE1547-2018 default is 10s for Catagory A and 5s for Category B, adjustable from 1s to 90s for both categories. However, the default is 0 for backward compatibility of OpenDSS models.
    ///
    /// DSS property name: Tresponse, DSS property index: 13
    ///
    double Tresponse()
    {
        return Obj_GetFloat64(ptr, Properties::Tresponse);
    }

    ExpControl& Tresponse(double value)
    {
        Obj_SetFloat64(ptr, Properties::Tresponse, value);
        return *this;
    }

    /// Alternative to PVSystemList for CIM export and import.
    ///
    /// However, storage is not actually implemented yet. Use fully qualified PVSystem names.
    ///
    /// DSS property name: DERList, DSS property index: 14
    ///
    strings DERList()
    {
        return get_array<strings>(Properties::DERList);
    }

    ExpControl& DERList(strings &value)
    {
        set_array<strings>(Properties::DERList, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 15
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    ExpControl& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 16
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    ExpControl& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 17
    ///
    ExpControl& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 17
    ///
    ExpControl& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class GICLine: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 43;
    struct Properties
    {
        enum {
            bus1 = 1,
            bus2 = 2,
            Volts = 3,
            Angle = 4,
            frequency = 5,
            phases = 6,
            R = 7,
            X = 8,
            C = 9,
            EN = 10,
            EE = 11,
            Lat1 = 12,
            Lon1 = 13,
            Lat2 = 14,
            Lon2 = 15,
            spectrum = 16,
            basefreq = 17,
            enabled = 18,
            like = 19,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    GICLine(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    GICLine(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the GICLine element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    GICLine(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the GICLine element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    GICLine& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    GICLine& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name of bus to which the main terminal (1) is connected.
    /// bus1=busname
    /// bus1=busname.1.2.3
    ///
    /// DSS property name: bus1, DSS property index: 1
    ///
    string bus1()
    {
        return get_prop_string(Properties::bus1);
    }

    GICLine& bus1(const string &value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    GICLine& bus1(const char* value)
    {
        set_string(Properties::bus1, value);
        return *this;
    }

    /// Name of bus to which 2nd terminal is connected.
    /// bus2=busname
    /// bus2=busname.1.2.3
    ///
    /// No Default; must be specified.
    ///
    /// DSS property name: bus2, DSS property index: 2
    ///
    string bus2()
    {
        return get_prop_string(Properties::bus2);
    }

    GICLine& bus2(const string &value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    GICLine& bus2(const char* value)
    {
        set_string(Properties::bus2, value);
        return *this;
    }

    /// Voltage magnitude, in volts, of the GIC voltage induced across this line. When spedified, voltage source is assumed defined by Voltage and Angle properties.
    ///
    /// Specify this value
    ///
    /// OR
    ///
    /// EN, EE, lat1, lon1, lat2, lon2.
    ///
    /// Not both!!  Last one entered will take precedence. Assumed identical in each phase of the Line object.
    ///
    /// DSS property name: Volts, DSS property index: 3
    ///
    double Volts()
    {
        return Obj_GetFloat64(ptr, Properties::Volts);
    }

    GICLine& Volts(double value)
    {
        Obj_SetFloat64(ptr, Properties::Volts, value);
        return *this;
    }

    /// Phase angle in degrees of first phase. Default=0.0.  See Voltage property
    ///
    /// DSS property name: Angle, DSS property index: 4
    ///
    double Angle()
    {
        return Obj_GetFloat64(ptr, Properties::Angle);
    }

    GICLine& Angle(double value)
    {
        Obj_SetFloat64(ptr, Properties::Angle, value);
        return *this;
    }

    /// Source frequency.  Defaults to 0.1 Hz.
    ///
    /// DSS property name: frequency, DSS property index: 5
    ///
    double frequency()
    {
        return Obj_GetFloat64(ptr, Properties::frequency);
    }

    GICLine& frequency(double value)
    {
        Obj_SetFloat64(ptr, Properties::frequency, value);
        return *this;
    }

    /// Number of phases.  Defaults to 3.
    ///
    /// DSS property name: phases, DSS property index: 6
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    GICLine& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Resistance of line, ohms of impedance in series with GIC voltage source.
    ///
    /// DSS property name: R, DSS property index: 7
    ///
    double R()
    {
        return Obj_GetFloat64(ptr, Properties::R);
    }

    GICLine& R(double value)
    {
        Obj_SetFloat64(ptr, Properties::R, value);
        return *this;
    }

    /// Reactance at base frequency, ohms. Default = 0.0. This value is generally not important for GIC studies but may be used if desired.
    ///
    /// DSS property name: X, DSS property index: 8
    ///
    double X()
    {
        return Obj_GetFloat64(ptr, Properties::X);
    }

    GICLine& X(double value)
    {
        Obj_SetFloat64(ptr, Properties::X, value);
        return *this;
    }

    /// Value of line blocking capacitance in microfarads. Default = 0.0, implying that there is no line blocking capacitor.
    ///
    /// DSS property name: C, DSS property index: 9
    ///
    double C()
    {
        return Obj_GetFloat64(ptr, Properties::C);
    }

    GICLine& C(double value)
    {
        Obj_SetFloat64(ptr, Properties::C, value);
        return *this;
    }

    /// Northward Electric field (V/km). If specified, Voltage and Angle are computed from EN, EE, lat and lon values.
    ///
    /// DSS property name: EN, DSS property index: 10
    ///
    double EN()
    {
        return Obj_GetFloat64(ptr, Properties::EN);
    }

    GICLine& EN(double value)
    {
        Obj_SetFloat64(ptr, Properties::EN, value);
        return *this;
    }

    /// Eastward Electric field (V/km).  If specified, Voltage and Angle are computed from EN, EE, lat and lon values.
    ///
    /// DSS property name: EE, DSS property index: 11
    ///
    double EE()
    {
        return Obj_GetFloat64(ptr, Properties::EE);
    }

    GICLine& EE(double value)
    {
        Obj_SetFloat64(ptr, Properties::EE, value);
        return *this;
    }

    /// Latitude of Bus1 (degrees)
    ///
    /// DSS property name: Lat1, DSS property index: 12
    ///
    double Lat1()
    {
        return Obj_GetFloat64(ptr, Properties::Lat1);
    }

    GICLine& Lat1(double value)
    {
        Obj_SetFloat64(ptr, Properties::Lat1, value);
        return *this;
    }

    /// Longitude of Bus1 (degrees)
    ///
    /// DSS property name: Lon1, DSS property index: 13
    ///
    double Lon1()
    {
        return Obj_GetFloat64(ptr, Properties::Lon1);
    }

    GICLine& Lon1(double value)
    {
        Obj_SetFloat64(ptr, Properties::Lon1, value);
        return *this;
    }

    /// Latitude of Bus2 (degrees)
    ///
    /// DSS property name: Lat2, DSS property index: 14
    ///
    double Lat2()
    {
        return Obj_GetFloat64(ptr, Properties::Lat2);
    }

    GICLine& Lat2(double value)
    {
        Obj_SetFloat64(ptr, Properties::Lat2, value);
        return *this;
    }

    /// Longitude of Bus2 (degrees)
    ///
    /// DSS property name: Lon2, DSS property index: 15
    ///
    double Lon2()
    {
        return Obj_GetFloat64(ptr, Properties::Lon2);
    }

    GICLine& Lon2(double value)
    {
        Obj_SetFloat64(ptr, Properties::Lon2, value);
        return *this;
    }

    /// Inherited Property for all PCElements. Name of harmonic spectrum for this source.  Default is "defaultvsource", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 16
    ///
    string spectrum()
    {
        return get_prop_string(Properties::spectrum);
    }

    GICLine& spectrum(const string &value)
    {
        set_string(Properties::spectrum, value);
        return *this;
    }

    GICLine& spectrum(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Inherited Property for all PCElements. Name of harmonic spectrum for this source.  Default is "defaultvsource", which is defined when the DSS starts.
    ///
    /// DSS property name: spectrum, DSS property index: 16
    ///
    dss::obj::Spectrum spectrum_obj()
    {
        return get_obj<dss::obj::Spectrum>(Properties::spectrum);
    }

    GICLine& spectrum_obj(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Inherited Property for all PCElements. Base frequency for specification of reactance value.
    ///
    /// DSS property name: basefreq, DSS property index: 17
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    GICLine& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 18
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    GICLine& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 19
    ///
    GICLine& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 19
    ///
    GICLine& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class GICTransformer: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 44;
    struct Properties
    {
        enum {
            BusH = 1,
            BusNH = 2,
            BusX = 3,
            BusNX = 4,
            phases = 5,
            Type = 6,
            R1 = 7,
            R2 = 8,
            KVLL1 = 9,
            KVLL2 = 10,
            MVA = 11,
            VarCurve = 12,
            pctR1 = 13,
            pctR2 = 14,
            K = 15,
            normamps = 16,
            emergamps = 17,
            faultrate = 18,
            pctperm = 19,
            repair = 20,
            basefreq = 21,
            enabled = 22,
            like = 23,
        };
    };

    // Class-specific enumerations

        ///
        /// GICTransformer: Type (DSS enumeration for GICTransformer)
        ///
    enum class GICTransformerType: int32_t
    {
        GSU = 1, ///< GSU
        Auto = 2, ///< Auto
        YY = 3 ///< YY
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    GICTransformer(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    GICTransformer(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the GICTransformer element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    GICTransformer(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the GICTransformer element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    GICTransformer& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    GICTransformer& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name of High-side(H) bus. Examples:
    /// BusH=busname
    /// BusH=busname.1.2.3
    ///
    /// DSS property name: BusH, DSS property index: 1
    ///
    string BusH()
    {
        return get_prop_string(Properties::BusH);
    }

    GICTransformer& BusH(const string &value)
    {
        set_string(Properties::BusH, value);
        return *this;
    }

    GICTransformer& BusH(const char* value)
    {
        set_string(Properties::BusH, value);
        return *this;
    }

    /// Name of Neutral bus for H, or first, winding. Defaults to all phases connected to H-side bus, node 0, if not specified and transformer type is either GSU or YY. (Shunt Wye Connection to ground reference)For Auto, this is automatically set to the X bus.
    ///
    /// DSS property name: BusNH, DSS property index: 2
    ///
    string BusNH()
    {
        return get_prop_string(Properties::BusNH);
    }

    GICTransformer& BusNH(const string &value)
    {
        set_string(Properties::BusNH, value);
        return *this;
    }

    GICTransformer& BusNH(const char* value)
    {
        set_string(Properties::BusNH, value);
        return *this;
    }

    /// Name of Low-side(X) bus, if type=Auto or YY.
    ///
    /// DSS property name: BusX, DSS property index: 3
    ///
    string BusX()
    {
        return get_prop_string(Properties::BusX);
    }

    GICTransformer& BusX(const string &value)
    {
        set_string(Properties::BusX, value);
        return *this;
    }

    GICTransformer& BusX(const char* value)
    {
        set_string(Properties::BusX, value);
        return *this;
    }

    /// Name of Neutral bus for X, or Second, winding. Defaults to all phases connected to X-side bus, node 0, if not specified. (Shunt Wye Connection to ground reference)
    ///
    /// DSS property name: BusNX, DSS property index: 4
    ///
    string BusNX()
    {
        return get_prop_string(Properties::BusNX);
    }

    GICTransformer& BusNX(const string &value)
    {
        set_string(Properties::BusNX, value);
        return *this;
    }

    GICTransformer& BusNX(const char* value)
    {
        set_string(Properties::BusNX, value);
        return *this;
    }

    /// Number of Phases. Default is 3.
    ///
    /// DSS property name: phases, DSS property index: 5
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    GICTransformer& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Type of transformer: {GSU* | Auto | YY}. Default is GSU.
    ///
    /// DSS property name: Type, DSS property index: 6
    ///
    GICTransformerType Type()
    {
        return GICTransformerType(Obj_GetInt32(ptr, Properties::Type));
    }

    GICTransformer& Type(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Type, value);
        return *this;
    }

    GICTransformer& Type(GICTransformerType value)
    {
        Obj_SetInt32(ptr, Properties::Type, int32_t(value));
        return *this;
    }

    GICTransformer& Type(const string &value)
    {
        set_string(Properties::Type, value);
        return *this;
    }

    GICTransformer& Type(const char *value)
    {
        set_string(Properties::Type, value);
        return *this;
    }

    /// Type of transformer: {GSU* | Auto | YY}. Default is GSU.
    ///
    /// DSS property name: Type, DSS property index: 6
    ///
    string Type_str()
    {
        return get_prop_string(Properties::Type);
    }

    /// Type of transformer: {GSU* | Auto | YY}. Default is GSU.
    ///
    /// DSS property name: Type, DSS property index: 6
    ///
    GICTransformer& Type_str(const string &value)
    {
        set_string(Properties::Type, value);
        return *this;
    }

    /// Resistance, each phase, ohms for H winding, (Series winding, if Auto). Default is 0.0001. If
    ///
    /// DSS property name: R1, DSS property index: 7
    ///
    double R1()
    {
        return Obj_GetFloat64(ptr, Properties::R1);
    }

    GICTransformer& R1(double value)
    {
        Obj_SetFloat64(ptr, Properties::R1, value);
        return *this;
    }

    /// Resistance, each phase, ohms for X winding, (Common winding, if Auto). Default is 0.0001.
    ///
    /// DSS property name: R2, DSS property index: 8
    ///
    double R2()
    {
        return Obj_GetFloat64(ptr, Properties::R2);
    }

    GICTransformer& R2(double value)
    {
        Obj_SetFloat64(ptr, Properties::R2, value);
        return *this;
    }

    /// Optional. kV LL rating for H winding (winding 1). Default is 500. Required if you are going to export vars for power flow analysis or enter winding resistances in percent.
    ///
    /// DSS property name: KVLL1, DSS property index: 9
    ///
    double KVLL1()
    {
        return Obj_GetFloat64(ptr, Properties::KVLL1);
    }

    GICTransformer& KVLL1(double value)
    {
        Obj_SetFloat64(ptr, Properties::KVLL1, value);
        return *this;
    }

    /// Optional. kV LL rating for X winding (winding 2). Default is 138. Required if you are going to export vars for power flow analysis or enter winding resistances in percent..
    ///
    /// DSS property name: KVLL2, DSS property index: 10
    ///
    double KVLL2()
    {
        return Obj_GetFloat64(ptr, Properties::KVLL2);
    }

    GICTransformer& KVLL2(double value)
    {
        Obj_SetFloat64(ptr, Properties::KVLL2, value);
        return *this;
    }

    /// Optional. MVA Rating assumed Transformer. Default is 100. Used for computing vars due to GIC and winding resistances if kV and MVA ratings are specified.
    ///
    /// DSS property name: MVA, DSS property index: 11
    ///
    double MVA()
    {
        return Obj_GetFloat64(ptr, Properties::MVA);
    }

    GICTransformer& MVA(double value)
    {
        Obj_SetFloat64(ptr, Properties::MVA, value);
        return *this;
    }

    /// Optional. XYCurve object name. Curve is expected as TOTAL pu vars vs pu GIC amps/phase. Vars are in pu of the MVA property. No Default value. Required only if you are going to export vars for power flow analysis. See K property.
    ///
    /// DSS property name: VarCurve, DSS property index: 12
    ///
    string VarCurve()
    {
        return get_prop_string(Properties::VarCurve);
    }

    GICTransformer& VarCurve(const string &value)
    {
        set_string(Properties::VarCurve, value);
        return *this;
    }

    GICTransformer& VarCurve(dss::obj::XYcurve &value)
    {
        set_obj(Properties::VarCurve, value);
        return *this;
    }

    /// Optional. XYCurve object name. Curve is expected as TOTAL pu vars vs pu GIC amps/phase. Vars are in pu of the MVA property. No Default value. Required only if you are going to export vars for power flow analysis. See K property.
    ///
    /// DSS property name: VarCurve, DSS property index: 12
    ///
    dss::obj::XYcurve VarCurve_obj()
    {
        return get_obj<dss::obj::XYcurve>(Properties::VarCurve);
    }

    GICTransformer& VarCurve_obj(dss::obj::XYcurve &value)
    {
        set_obj(Properties::VarCurve, value);
        return *this;
    }

    /// Optional. Percent Resistance, each phase, for H winding (1), (Series winding, if Auto). Default is 0.2.
    ///
    /// Alternative way to enter R1 value. It is the actual resistances in ohmns that matter. MVA and kV should be specified.
    ///
    /// DSS property name: %R1, DSS property index: 13
    ///
    double pctR1()
    {
        return Obj_GetFloat64(ptr, Properties::pctR1);
    }

    GICTransformer& pctR1(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctR1, value);
        return *this;
    }

    /// Optional. Percent Resistance, each phase, for X winding (2), (Common winding, if Auto). Default is 0.2.
    ///
    /// Alternative way to enter R2 value. It is the actual resistances in ohms that matter. MVA and kV should be specified.
    ///
    /// DSS property name: %R2, DSS property index: 14
    ///
    double pctR2()
    {
        return Obj_GetFloat64(ptr, Properties::pctR2);
    }

    GICTransformer& pctR2(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctR2, value);
        return *this;
    }

    /// Mvar K factor. Default way to convert GIC Amps in H winding (winding 1) to Mvar. Default is 2.2. Commonly-used simple multiplier for estimating Mvar losses for power flow analysis.
    ///
    /// Mvar = K * kvLL * GIC per phase / 1000
    ///
    /// Mutually exclusive with using the VarCurve property and pu curves.If you specify this (default), VarCurve is ignored.
    ///
    /// DSS property name: K, DSS property index: 15
    ///
    double K()
    {
        return Obj_GetFloat64(ptr, Properties::K);
    }

    GICTransformer& K(double value)
    {
        Obj_SetFloat64(ptr, Properties::K, value);
        return *this;
    }

    /// Normal rated current.
    ///
    /// DSS property name: normamps, DSS property index: 16
    ///
    double normamps()
    {
        return Obj_GetFloat64(ptr, Properties::normamps);
    }

    GICTransformer& normamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::normamps, value);
        return *this;
    }

    /// Maximum or emerg current.
    ///
    /// DSS property name: emergamps, DSS property index: 17
    ///
    double emergamps()
    {
        return Obj_GetFloat64(ptr, Properties::emergamps);
    }

    GICTransformer& emergamps(double value)
    {
        Obj_SetFloat64(ptr, Properties::emergamps, value);
        return *this;
    }

    /// Failure rate per year.
    ///
    /// DSS property name: faultrate, DSS property index: 18
    ///
    double faultrate()
    {
        return Obj_GetFloat64(ptr, Properties::faultrate);
    }

    GICTransformer& faultrate(double value)
    {
        Obj_SetFloat64(ptr, Properties::faultrate, value);
        return *this;
    }

    /// Percent of failures that become permanent.
    ///
    /// DSS property name: pctperm, DSS property index: 19
    ///
    double pctperm()
    {
        return Obj_GetFloat64(ptr, Properties::pctperm);
    }

    GICTransformer& pctperm(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctperm, value);
        return *this;
    }

    /// Hours to repair.
    ///
    /// DSS property name: repair, DSS property index: 20
    ///
    double repair()
    {
        return Obj_GetFloat64(ptr, Properties::repair);
    }

    GICTransformer& repair(double value)
    {
        Obj_SetFloat64(ptr, Properties::repair, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 21
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    GICTransformer& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 22
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    GICTransformer& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 23
    ///
    GICTransformer& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 23
    ///
    GICTransformer& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class VSConverter: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 45;
    struct Properties
    {
        enum {
            phases = 1,
            Bus1 = 2,
            kVac = 3,
            kVdc = 4,
            kW = 5,
            Ndc = 6,
            Rac = 7,
            Xac = 8,
            m0 = 9,
            d0 = 10,
            Mmin = 11,
            Mmax = 12,
            Iacmax = 13,
            Idcmax = 14,
            Vacref = 15,
            Pacref = 16,
            Qacref = 17,
            Vdcref = 18,
            VscMode = 19,
            spectrum = 20,
            basefreq = 21,
            enabled = 22,
            like = 23,
        };
    };

    // Class-specific enumerations

        ///
        /// VSConverter: Control Mode (DSS enumeration for VSConverter)
        ///
    enum class VSConverterControlMode: int32_t
    {
        Fixed = 0, ///< Fixed
        PacVac = 1, ///< PacVac
        PacQac = 2, ///< PacQac
        VdcVac = 3, ///< VdcVac
        VdcQac = 4 ///< VdcQac
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    VSConverter(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    VSConverter(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the VSConverter element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    VSConverter(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the VSConverter element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    VSConverter& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    VSConverter& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Number of AC plus DC conductors. Default is 4. AC phases numbered before DC conductors.
    ///
    /// DSS property name: phases, DSS property index: 1
    ///
    int32_t phases()
    {
        return Obj_GetInt32(ptr, Properties::phases);
    }

    VSConverter& phases(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::phases, value);
        return *this;
    }

    /// Name of converter bus, containing both AC and DC conductors. Bus2 is always ground.
    ///
    /// DSS property name: Bus1, DSS property index: 2
    ///
    string Bus1()
    {
        return get_prop_string(Properties::Bus1);
    }

    VSConverter& Bus1(const string &value)
    {
        set_string(Properties::Bus1, value);
        return *this;
    }

    VSConverter& Bus1(const char* value)
    {
        set_string(Properties::Bus1, value);
        return *this;
    }

    /// Nominal AC line-neutral voltage in kV. Must be specified > 0.
    ///
    /// DSS property name: kVac, DSS property index: 3
    ///
    double kVac()
    {
        return Obj_GetFloat64(ptr, Properties::kVac);
    }

    VSConverter& kVac(double value)
    {
        Obj_SetFloat64(ptr, Properties::kVac, value);
        return *this;
    }

    /// Nominal DC voltage in kV. Must be specified > 0.
    ///
    /// DSS property name: kVdc, DSS property index: 4
    ///
    double kVdc()
    {
        return Obj_GetFloat64(ptr, Properties::kVdc);
    }

    VSConverter& kVdc(double value)
    {
        Obj_SetFloat64(ptr, Properties::kVdc, value);
        return *this;
    }

    /// Nominal converter power in kW. Must be specified > 0.
    ///
    /// DSS property name: kW, DSS property index: 5
    ///
    double kW()
    {
        return Obj_GetFloat64(ptr, Properties::kW);
    }

    VSConverter& kW(double value)
    {
        Obj_SetFloat64(ptr, Properties::kW, value);
        return *this;
    }

    /// Number of DC conductors. Default is 1. DC conductors numbered after AC phases.
    ///
    /// DSS property name: Ndc, DSS property index: 6
    ///
    int32_t Ndc()
    {
        return Obj_GetInt32(ptr, Properties::Ndc);
    }

    VSConverter& Ndc(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Ndc, value);
        return *this;
    }

    /// AC resistance (ohms) for the converter transformer, plus any series reactors. Default is 0.
    /// Must be 0 for Vac control mode.
    ///
    /// DSS property name: Rac, DSS property index: 7
    ///
    double Rac()
    {
        return Obj_GetFloat64(ptr, Properties::Rac);
    }

    VSConverter& Rac(double value)
    {
        Obj_SetFloat64(ptr, Properties::Rac, value);
        return *this;
    }

    /// AC reactance (ohms) for the converter transformer, plus any series reactors. Default is 0.
    /// Must be 0 for Vac control mode. Must be >0 for PacVac, PacQac or VacVdc control mode.
    ///
    /// DSS property name: Xac, DSS property index: 8
    ///
    double Xac()
    {
        return Obj_GetFloat64(ptr, Properties::Xac);
    }

    VSConverter& Xac(double value)
    {
        Obj_SetFloat64(ptr, Properties::Xac, value);
        return *this;
    }

    /// Fixed or initial value of the modulation index. Default is 0.5.
    ///
    /// DSS property name: m0, DSS property index: 9
    ///
    double m0()
    {
        return Obj_GetFloat64(ptr, Properties::m0);
    }

    VSConverter& m0(double value)
    {
        Obj_SetFloat64(ptr, Properties::m0, value);
        return *this;
    }

    /// Fixed or initial value of the power angle in degrees. Default is 0.
    ///
    /// DSS property name: d0, DSS property index: 10
    ///
    double d0()
    {
        return Obj_GetFloat64(ptr, Properties::d0);
    }

    VSConverter& d0(double value)
    {
        Obj_SetFloat64(ptr, Properties::d0, value);
        return *this;
    }

    /// Minimum value of modulation index. Default is 0.1.
    ///
    /// DSS property name: Mmin, DSS property index: 11
    ///
    double Mmin()
    {
        return Obj_GetFloat64(ptr, Properties::Mmin);
    }

    VSConverter& Mmin(double value)
    {
        Obj_SetFloat64(ptr, Properties::Mmin, value);
        return *this;
    }

    /// Maximum value of modulation index. Default is 0.9.
    ///
    /// DSS property name: Mmax, DSS property index: 12
    ///
    double Mmax()
    {
        return Obj_GetFloat64(ptr, Properties::Mmax);
    }

    VSConverter& Mmax(double value)
    {
        Obj_SetFloat64(ptr, Properties::Mmax, value);
        return *this;
    }

    /// Maximum value of AC line current, per-unit of nominal. Default is 2.
    ///
    /// DSS property name: Iacmax, DSS property index: 13
    ///
    double Iacmax()
    {
        return Obj_GetFloat64(ptr, Properties::Iacmax);
    }

    VSConverter& Iacmax(double value)
    {
        Obj_SetFloat64(ptr, Properties::Iacmax, value);
        return *this;
    }

    /// Maximum value of DC current, per-unit of nominal. Default is 2.
    ///
    /// DSS property name: Idcmax, DSS property index: 14
    ///
    double Idcmax()
    {
        return Obj_GetFloat64(ptr, Properties::Idcmax);
    }

    VSConverter& Idcmax(double value)
    {
        Obj_SetFloat64(ptr, Properties::Idcmax, value);
        return *this;
    }

    /// Reference AC line-to-neutral voltage, RMS Volts. Default is 0.
    /// Applies to PacVac and VdcVac control modes, influencing m.
    ///
    /// DSS property name: Vacref, DSS property index: 15
    ///
    double Vacref()
    {
        return Obj_GetFloat64(ptr, Properties::Vacref);
    }

    VSConverter& Vacref(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vacref, value);
        return *this;
    }

    /// Reference total AC real power, Watts. Default is 0.
    /// Applies to PacVac and PacQac control modes, influencing d.
    ///
    /// DSS property name: Pacref, DSS property index: 16
    ///
    double Pacref()
    {
        return Obj_GetFloat64(ptr, Properties::Pacref);
    }

    VSConverter& Pacref(double value)
    {
        Obj_SetFloat64(ptr, Properties::Pacref, value);
        return *this;
    }

    /// Reference total AC reactive power, Vars. Default is 0.
    /// Applies to PacQac and VdcQac control modes, influencing m.
    ///
    /// DSS property name: Qacref, DSS property index: 17
    ///
    double Qacref()
    {
        return Obj_GetFloat64(ptr, Properties::Qacref);
    }

    VSConverter& Qacref(double value)
    {
        Obj_SetFloat64(ptr, Properties::Qacref, value);
        return *this;
    }

    /// Reference DC voltage, Volts. Default is 0.
    /// Applies to VdcVac control mode, influencing d.
    ///
    /// DSS property name: Vdcref, DSS property index: 18
    ///
    double Vdcref()
    {
        return Obj_GetFloat64(ptr, Properties::Vdcref);
    }

    VSConverter& Vdcref(double value)
    {
        Obj_SetFloat64(ptr, Properties::Vdcref, value);
        return *this;
    }

    /// Control Mode (Fixed|PacVac|PacQac|VdcVac|VdcQac). Default is Fixed.
    ///
    /// DSS property name: VscMode, DSS property index: 19
    ///
    VSConverterControlMode VscMode()
    {
        return VSConverterControlMode(Obj_GetInt32(ptr, Properties::VscMode));
    }

    VSConverter& VscMode(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::VscMode, value);
        return *this;
    }

    VSConverter& VscMode(VSConverterControlMode value)
    {
        Obj_SetInt32(ptr, Properties::VscMode, int32_t(value));
        return *this;
    }

    VSConverter& VscMode(const string &value)
    {
        set_string(Properties::VscMode, value);
        return *this;
    }

    VSConverter& VscMode(const char *value)
    {
        set_string(Properties::VscMode, value);
        return *this;
    }

    /// Control Mode (Fixed|PacVac|PacQac|VdcVac|VdcQac). Default is Fixed.
    ///
    /// DSS property name: VscMode, DSS property index: 19
    ///
    string VscMode_str()
    {
        return get_prop_string(Properties::VscMode);
    }

    /// Control Mode (Fixed|PacVac|PacQac|VdcVac|VdcQac). Default is Fixed.
    ///
    /// DSS property name: VscMode, DSS property index: 19
    ///
    VSConverter& VscMode_str(const string &value)
    {
        set_string(Properties::VscMode, value);
        return *this;
    }

    /// Name of harmonic spectrum for this device.
    ///
    /// DSS property name: spectrum, DSS property index: 20
    ///
    string spectrum()
    {
        return get_prop_string(Properties::spectrum);
    }

    VSConverter& spectrum(const string &value)
    {
        set_string(Properties::spectrum, value);
        return *this;
    }

    VSConverter& spectrum(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Name of harmonic spectrum for this device.
    ///
    /// DSS property name: spectrum, DSS property index: 20
    ///
    dss::obj::Spectrum spectrum_obj()
    {
        return get_obj<dss::obj::Spectrum>(Properties::spectrum);
    }

    VSConverter& spectrum_obj(dss::obj::Spectrum &value)
    {
        set_obj(Properties::spectrum, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 21
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    VSConverter& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 22
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    VSConverter& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 23
    ///
    VSConverter& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 23
    ///
    VSConverter& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Monitor: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 46;
    struct Properties
    {
        enum {
            element = 1,
            terminal = 2,
            mode = 3,
            action = 4,
            residual = 5,
            VIPolar = 6,
            PPolar = 7,
            basefreq = 8,
            enabled = 9,
            like = 10,
        };
    };

    // Class-specific enumerations

        ///
        /// Monitor: Action (DSS enumeration for Monitor)
        ///
    enum class MonitorAction: int32_t
    {
        Clear = 0, ///< Clear
        Save = 1, ///< Save
        Take = 2, ///< Take
        Process = 3, ///< Process
        Reset = 0 ///< Reset
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Monitor(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Monitor(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Monitor element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Monitor(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Monitor element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Monitor& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Monitor& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name (Full Object name) of element to which the monitor is connected.
    ///
    /// DSS property name: element, DSS property index: 1
    ///
    string element()
    {
        return get_prop_string(Properties::element);
    }

    Monitor& element(const string &value)
    {
        set_string(Properties::element, value);
        return *this;
    }

    Monitor& element(dss::obj::DSSObj &value)
    {
        set_obj(Properties::element, value);
        return *this;
    }

    /// Name (Full Object name) of element to which the monitor is connected.
    ///
    /// DSS property name: element, DSS property index: 1
    ///
    dss::obj::DSSObj element_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::element);
    }

    Monitor& element_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::element, value);
        return *this;
    }

    /// Number of the terminal of the circuit element to which the monitor is connected. 1 or 2, typically. For monitoring states, attach monitor to terminal 1.
    ///
    /// DSS property name: terminal, DSS property index: 2
    ///
    int32_t terminal()
    {
        return Obj_GetInt32(ptr, Properties::terminal);
    }

    Monitor& terminal(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::terminal, value);
        return *this;
    }

    /// Bitmask integer designating the values the monitor is to capture:
    /// 0 = Voltages and currents at designated terminal
    /// 1 = Powers at designated terminal
    /// 2 = Tap Position (Transformer Device only)
    /// 3 = State Variables (PCElements only)
    /// 4 = Flicker level and severity index (Pst) for voltages. No adders apply.
    ///     Flicker level at simulation time step, Pst at 10-minute time step.
    /// 5 = Solution variables (Iterations, etc).
    /// Normally, these would be actual phasor quantities from solution.
    /// 6 = Capacitor Switching (Capacitors only)
    /// 7 = Storage state vars (Storage device only)
    /// 8 = All winding currents (Transformer device only)
    /// 9 = Losses, watts and var (of monitored device)
    /// 10 = All Winding voltages (Transformer device only)
    /// Normally, these would be actual phasor quantities from solution.
    /// 11 = All terminal node voltages and line currents of monitored device
    /// 12 = All terminal node voltages LL and line currents of monitored device
    /// Combine mode with adders below to achieve other results for terminal quantities:
    /// +16 = Sequence quantities
    /// +32 = Magnitude only
    /// +64 = Positive sequence only or avg of all phases
    ///
    /// Mix adder to obtain desired results. For example:
    /// Mode=112 will save positive sequence voltage and current magnitudes only
    /// Mode=48 will save all sequence voltages and currents, but magnitude only.
    ///
    /// DSS property name: mode, DSS property index: 3
    ///
    int32_t mode()
    {
        return Obj_GetInt32(ptr, Properties::mode);
    }

    Monitor& mode(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::mode, value);
        return *this;
    }

    /// {Clear | Save | Take | Process}
    /// (C)lears or (S)aves current buffer.
    /// (T)ake action takes a sample.
    /// (P)rocesses the data taken so far (e.g. Pst for mode 4).
    ///
    /// Note that monitors are automatically reset (cleared) when the Set Mode= command is issued. Otherwise, the user must explicitly reset all monitors (reset monitors command) or individual monitors with the Clear action.
    ///
    /// DSS property name: action, DSS property index: 4
    ///
    Monitor& action(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::action, value);
        return *this;
    }

    /// {Clear | Save | Take | Process}
    /// (C)lears or (S)aves current buffer.
    /// (T)ake action takes a sample.
    /// (P)rocesses the data taken so far (e.g. Pst for mode 4).
    ///
    /// Note that monitors are automatically reset (cleared) when the Set Mode= command is issued. Otherwise, the user must explicitly reset all monitors (reset monitors command) or individual monitors with the Clear action.
    ///
    /// DSS property name: action, DSS property index: 4
    ///
    Monitor& action(MonitorAction value)
    {
        Obj_SetInt32(ptr, Properties::action, int32_t(value));
        return *this;
    }

    /// {Clear | Save | Take | Process}
    /// (C)lears or (S)aves current buffer.
    /// (T)ake action takes a sample.
    /// (P)rocesses the data taken so far (e.g. Pst for mode 4).
    ///
    /// Note that monitors are automatically reset (cleared) when the Set Mode= command is issued. Otherwise, the user must explicitly reset all monitors (reset monitors command) or individual monitors with the Clear action.
    ///
    /// DSS property name: action, DSS property index: 4
    ///
    Monitor& action(const string &value)
    {
        set_string(Properties::action, value);
        return *this;
    }

    /// {Clear | Save | Take | Process}
    /// (C)lears or (S)aves current buffer.
    /// (T)ake action takes a sample.
    /// (P)rocesses the data taken so far (e.g. Pst for mode 4).
    ///
    /// Note that monitors are automatically reset (cleared) when the Set Mode= command is issued. Otherwise, the user must explicitly reset all monitors (reset monitors command) or individual monitors with the Clear action.
    ///
    /// DSS property name: action, DSS property index: 4
    ///
    Monitor& action(const char *value)
    {
        set_string(Properties::action, value);
        return *this;
    }

    /// {Yes/True | No/False} Default = No.  Include Residual cbannel (sum of all phases) for voltage and current. Does not apply to sequence quantity modes or power modes.
    ///
    /// DSS property name: residual, DSS property index: 5
    ///
    bool residual()
    {
        return Obj_GetInt32(ptr, Properties::residual) != 0;
    }

    Monitor& residual(bool value)
    {
        Obj_SetInt32(ptr, Properties::residual, value);
        return *this;
    }

    /// {Yes/True | No/False} Default = YES. Report voltage and current in polar form (Mag/Angle). (default)  Otherwise, it will be real and imaginary.
    ///
    /// DSS property name: VIPolar, DSS property index: 6
    ///
    bool VIPolar()
    {
        return Obj_GetInt32(ptr, Properties::VIPolar) != 0;
    }

    Monitor& VIPolar(bool value)
    {
        Obj_SetInt32(ptr, Properties::VIPolar, value);
        return *this;
    }

    /// {Yes/True | No/False} Default = YES. Report power in Apparent power, S, in polar form (Mag/Angle).(default)  Otherwise, is P and Q
    ///
    /// DSS property name: PPolar, DSS property index: 7
    ///
    bool PPolar()
    {
        return Obj_GetInt32(ptr, Properties::PPolar) != 0;
    }

    Monitor& PPolar(bool value)
    {
        Obj_SetInt32(ptr, Properties::PPolar, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 8
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Monitor& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 9
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Monitor& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 10
    ///
    Monitor& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 10
    ///
    Monitor& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class EnergyMeter: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 47;
    struct Properties
    {
        enum {
            element = 1,
            terminal = 2,
            action = 3,
            option = 4,
            kVAnormal = 5,
            kVAemerg = 6,
            peakcurrent = 7,
            Zonelist = 8,
            LocalOnly = 9,
            Mask = 10,
            Losses = 11,
            LineLosses = 12,
            XfmrLosses = 13,
            SeqLosses = 14,
            threePaseLosses = 15,
            VbaseLosses = 16,
            PhaseVoltageReport = 17,
            Int_Rate = 18,
            Int_Duration = 19,
            SAIFI = 20,
            SAIFIkW = 21,
            SAIDI = 22,
            CAIDI = 23,
            CustInterrupts = 24,
            basefreq = 25,
            enabled = 26,
            like = 27,
        };
    };

    // Class-specific enumerations

        ///
        /// EnergyMeter: Action (DSS enumeration for EnergyMeter)
        ///
    enum class EnergyMeterAction: int32_t
    {
        Allocate = 0, ///< Allocate
        Clear = 1, ///< Clear
        Reduce = 2, ///< Reduce
        Save = 3, ///< Save
        Take = 4, ///< Take
        ZoneDump = 5 ///< ZoneDump
    };



    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    EnergyMeter(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    EnergyMeter(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the EnergyMeter element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    EnergyMeter(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the EnergyMeter element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    EnergyMeter& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    EnergyMeter& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name (Full Object name) of element to which the monitor is connected.
    ///
    /// DSS property name: element, DSS property index: 1
    ///
    string element()
    {
        return get_prop_string(Properties::element);
    }

    EnergyMeter& element(const string &value)
    {
        set_string(Properties::element, value);
        return *this;
    }

    EnergyMeter& element(dss::obj::DSSObj &value)
    {
        set_obj(Properties::element, value);
        return *this;
    }

    /// Name (Full Object name) of element to which the monitor is connected.
    ///
    /// DSS property name: element, DSS property index: 1
    ///
    dss::obj::DSSObj element_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::element);
    }

    EnergyMeter& element_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::element, value);
        return *this;
    }

    /// Number of the terminal of the circuit element to which the monitor is connected. 1 or 2, typically.
    ///
    /// DSS property name: terminal, DSS property index: 2
    ///
    int32_t terminal()
    {
        return Obj_GetInt32(ptr, Properties::terminal);
    }

    EnergyMeter& terminal(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::terminal, value);
        return *this;
    }

    /// {Clear (reset) | Save | Take | Zonedump | Allocate | Reduce}
    ///
    /// (A)llocate = Allocate loads on the meter zone to match PeakCurrent.
    /// (C)lear = reset all registers to zero
    /// (R)educe = reduces zone by merging lines (see Set Keeplist & ReduceOption)
    /// (S)ave = saves the current register values to a file.
    ///    File name is "MTR_metername.CSV".
    /// (T)ake = Takes a sample at present solution
    /// (Z)onedump = Dump names of elements in meter zone to a file
    ///    File name is "Zone_metername.CSV".
    ///
    /// DSS property name: action, DSS property index: 3
    ///
    EnergyMeter& action(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::action, value);
        return *this;
    }

    /// {Clear (reset) | Save | Take | Zonedump | Allocate | Reduce}
    ///
    /// (A)llocate = Allocate loads on the meter zone to match PeakCurrent.
    /// (C)lear = reset all registers to zero
    /// (R)educe = reduces zone by merging lines (see Set Keeplist & ReduceOption)
    /// (S)ave = saves the current register values to a file.
    ///    File name is "MTR_metername.CSV".
    /// (T)ake = Takes a sample at present solution
    /// (Z)onedump = Dump names of elements in meter zone to a file
    ///    File name is "Zone_metername.CSV".
    ///
    /// DSS property name: action, DSS property index: 3
    ///
    EnergyMeter& action(EnergyMeterAction value)
    {
        Obj_SetInt32(ptr, Properties::action, int32_t(value));
        return *this;
    }

    /// {Clear (reset) | Save | Take | Zonedump | Allocate | Reduce}
    ///
    /// (A)llocate = Allocate loads on the meter zone to match PeakCurrent.
    /// (C)lear = reset all registers to zero
    /// (R)educe = reduces zone by merging lines (see Set Keeplist & ReduceOption)
    /// (S)ave = saves the current register values to a file.
    ///    File name is "MTR_metername.CSV".
    /// (T)ake = Takes a sample at present solution
    /// (Z)onedump = Dump names of elements in meter zone to a file
    ///    File name is "Zone_metername.CSV".
    ///
    /// DSS property name: action, DSS property index: 3
    ///
    EnergyMeter& action(const string &value)
    {
        set_string(Properties::action, value);
        return *this;
    }

    /// {Clear (reset) | Save | Take | Zonedump | Allocate | Reduce}
    ///
    /// (A)llocate = Allocate loads on the meter zone to match PeakCurrent.
    /// (C)lear = reset all registers to zero
    /// (R)educe = reduces zone by merging lines (see Set Keeplist & ReduceOption)
    /// (S)ave = saves the current register values to a file.
    ///    File name is "MTR_metername.CSV".
    /// (T)ake = Takes a sample at present solution
    /// (Z)onedump = Dump names of elements in meter zone to a file
    ///    File name is "Zone_metername.CSV".
    ///
    /// DSS property name: action, DSS property index: 3
    ///
    EnergyMeter& action(const char *value)
    {
        set_string(Properties::action, value);
        return *this;
    }

    /// Enter a string ARRAY of any combination of the following. Options processed left-to-right:
    ///
    /// (E)xcess : (default) UE/EEN is estimate of energy over capacity
    /// (T)otal : UE/EEN is total energy after capacity exceeded
    /// (R)adial : (default) Treats zone as a radial circuit
    /// (M)esh : Treats zone as meshed network (not radial).
    /// (C)ombined : (default) Load UE/EEN computed from combination of overload and undervoltage.
    /// (V)oltage : Load UE/EEN computed based on voltage only.
    ///
    /// Example: option=(E, R)
    ///
    /// DSS property name: option, DSS property index: 4
    ///
    strings option()
    {
        return get_array<strings>(Properties::option);
    }

    EnergyMeter& option(strings &value)
    {
        set_array<strings>(Properties::option, value);
        return *this;
    }

    /// Upper limit on kVA load in the zone, Normal configuration. Default is 0.0 (ignored). Overrides limits on individual lines for overload EEN. With "LocalOnly=Yes" option, uses only load in metered branch.
    ///
    /// DSS property name: kVAnormal, DSS property index: 5
    ///
    double kVAnormal()
    {
        return Obj_GetFloat64(ptr, Properties::kVAnormal);
    }

    EnergyMeter& kVAnormal(double value)
    {
        Obj_SetFloat64(ptr, Properties::kVAnormal, value);
        return *this;
    }

    /// Upper limit on kVA load in the zone, Emergency configuration. Default is 0.0 (ignored). Overrides limits on individual lines for overload UE. With "LocalOnly=Yes" option, uses only load in metered branch.
    ///
    /// DSS property name: kVAemerg, DSS property index: 6
    ///
    double kVAemerg()
    {
        return Obj_GetFloat64(ptr, Properties::kVAemerg);
    }

    EnergyMeter& kVAemerg(double value)
    {
        Obj_SetFloat64(ptr, Properties::kVAemerg, value);
        return *this;
    }

    /// ARRAY of current magnitudes representing the peak currents measured at this location for the load allocation function.  Default is (400, 400, 400). Enter one current for each phase
    ///
    /// DSS property name: peakcurrent, DSS property index: 7
    ///
    VectorXd peakcurrent()
    {
        return get_array<VectorXd>(Properties::peakcurrent);
    }

    EnergyMeter& peakcurrent(VectorXd &value)
    {
        set_array<VectorXd>(Properties::peakcurrent, value);
        return *this;
    }

    /// ARRAY of full element names for this meter's zone.  Default is for meter to find it's own zone. If specified, DSS uses this list instead.  Can access the names in a single-column text file.  Examples:
    ///
    /// zonelist=[line.L1, transformer.T1, Line.L3]
    /// zonelist=(file=branchlist.txt)
    ///
    /// DSS property name: Zonelist, DSS property index: 8
    ///
    strings Zonelist()
    {
        return get_array<strings>(Properties::Zonelist);
    }

    EnergyMeter& Zonelist(strings &value)
    {
        set_array<strings>(Properties::Zonelist, value);
        return *this;
    }

    /// {Yes | No}  Default is NO.  If Yes, meter considers only the monitored element for EEN and UE calcs.  Uses whole zone for losses.
    ///
    /// DSS property name: LocalOnly, DSS property index: 9
    ///
    bool LocalOnly()
    {
        return Obj_GetInt32(ptr, Properties::LocalOnly) != 0;
    }

    EnergyMeter& LocalOnly(bool value)
    {
        Obj_SetInt32(ptr, Properties::LocalOnly, value);
        return *this;
    }

    /// Mask for adding registers whenever all meters are totalized.  Array of floating point numbers representing the multiplier to be used for summing each register from this meter. Default = (1, 1, 1, 1, ... ).  You only have to enter as many as are changed (positional). Useful when two meters monitor same energy, etc.
    ///
    /// DSS property name: Mask, DSS property index: 10
    ///
    VectorXd Mask()
    {
        return get_array<VectorXd>(Properties::Mask);
    }

    EnergyMeter& Mask(VectorXd &value)
    {
        set_array<VectorXd>(Properties::Mask, value);
        return *this;
    }

    /// {Yes | No}  Default is YES. Compute Zone losses. If NO, then no losses at all are computed.
    ///
    /// DSS property name: Losses, DSS property index: 11
    ///
    bool Losses()
    {
        return Obj_GetInt32(ptr, Properties::Losses) != 0;
    }

    EnergyMeter& Losses(bool value)
    {
        Obj_SetInt32(ptr, Properties::Losses, value);
        return *this;
    }

    /// {Yes | No}  Default is YES. Compute Line losses. If NO, then none of the losses are computed.
    ///
    /// DSS property name: LineLosses, DSS property index: 12
    ///
    bool LineLosses()
    {
        return Obj_GetInt32(ptr, Properties::LineLosses) != 0;
    }

    EnergyMeter& LineLosses(bool value)
    {
        Obj_SetInt32(ptr, Properties::LineLosses, value);
        return *this;
    }

    /// {Yes | No}  Default is YES. Compute Transformer losses. If NO, transformers are ignored in loss calculations.
    ///
    /// DSS property name: XfmrLosses, DSS property index: 13
    ///
    bool XfmrLosses()
    {
        return Obj_GetInt32(ptr, Properties::XfmrLosses) != 0;
    }

    EnergyMeter& XfmrLosses(bool value)
    {
        Obj_SetInt32(ptr, Properties::XfmrLosses, value);
        return *this;
    }

    /// {Yes | No}  Default is YES. Compute Sequence losses in lines and segregate by line mode losses and zero mode losses.
    ///
    /// DSS property name: SeqLosses, DSS property index: 14
    ///
    bool SeqLosses()
    {
        return Obj_GetInt32(ptr, Properties::SeqLosses) != 0;
    }

    EnergyMeter& SeqLosses(bool value)
    {
        Obj_SetInt32(ptr, Properties::SeqLosses, value);
        return *this;
    }

    /// {Yes | No}  Default is YES. Compute Line losses and segregate by 3-phase and other (1- and 2-phase) line losses.
    ///
    /// DSS property name: 3phaseLosses, DSS property index: 15
    ///
    bool threePaseLosses()
    {
        return Obj_GetInt32(ptr, Properties::threePaseLosses) != 0;
    }

    EnergyMeter& threePaseLosses(bool value)
    {
        Obj_SetInt32(ptr, Properties::threePaseLosses, value);
        return *this;
    }

    /// {Yes | No}  Default is YES. Compute losses and segregate by voltage base. If NO, then voltage-based tabulation is not reported.
    ///
    /// DSS property name: VbaseLosses, DSS property index: 16
    ///
    bool VbaseLosses()
    {
        return Obj_GetInt32(ptr, Properties::VbaseLosses) != 0;
    }

    EnergyMeter& VbaseLosses(bool value)
    {
        Obj_SetInt32(ptr, Properties::VbaseLosses, value);
        return *this;
    }

    /// {Yes | No}  Default is NO.  Report min, max, and average phase voltages for the zone and tabulate by voltage base. Demand Intervals must be turned on (Set Demand=true) and voltage bases must be defined for this property to take effect. Result is in a separate report file.
    ///
    /// DSS property name: PhaseVoltageReport, DSS property index: 17
    ///
    bool PhaseVoltageReport()
    {
        return Obj_GetInt32(ptr, Properties::PhaseVoltageReport) != 0;
    }

    EnergyMeter& PhaseVoltageReport(bool value)
    {
        Obj_SetInt32(ptr, Properties::PhaseVoltageReport, value);
        return *this;
    }

    /// Average number of annual interruptions for head of the meter zone (source side of zone or feeder).
    ///
    /// DSS property name: Int_Rate, DSS property index: 18
    ///
    double Int_Rate()
    {
        return Obj_GetFloat64(ptr, Properties::Int_Rate);
    }

    EnergyMeter& Int_Rate(double value)
    {
        Obj_SetFloat64(ptr, Properties::Int_Rate, value);
        return *this;
    }

    /// Average annual duration, in hr, of interruptions for head of the meter zone (source side of zone or feeder).
    ///
    /// DSS property name: Int_Duration, DSS property index: 19
    ///
    double Int_Duration()
    {
        return Obj_GetFloat64(ptr, Properties::Int_Duration);
    }

    EnergyMeter& Int_Duration(double value)
    {
        Obj_SetFloat64(ptr, Properties::Int_Duration, value);
        return *this;
    }

    /// (Read only) Makes SAIFI result available via return on query (? energymeter.myMeter.SAIFI.
    ///
    /// DSS property name: SAIFI, DSS property index: 20
    ///
    double SAIFI()
    {
        return Obj_GetFloat64(ptr, Properties::SAIFI);
    }

    EnergyMeter& SAIFI(double value)
    {
        Obj_SetFloat64(ptr, Properties::SAIFI, value);
        return *this;
    }

    /// (Read only) Makes SAIFIkW result available via return on query (? energymeter.myMeter.SAIFIkW.
    ///
    /// DSS property name: SAIFIkW, DSS property index: 21
    ///
    double SAIFIkW()
    {
        return Obj_GetFloat64(ptr, Properties::SAIFIkW);
    }

    EnergyMeter& SAIFIkW(double value)
    {
        Obj_SetFloat64(ptr, Properties::SAIFIkW, value);
        return *this;
    }

    /// (Read only) Makes SAIDI result available via return on query (? energymeter.myMeter.SAIDI.
    ///
    /// DSS property name: SAIDI, DSS property index: 22
    ///
    double SAIDI()
    {
        return Obj_GetFloat64(ptr, Properties::SAIDI);
    }

    EnergyMeter& SAIDI(double value)
    {
        Obj_SetFloat64(ptr, Properties::SAIDI, value);
        return *this;
    }

    /// (Read only) Makes CAIDI result available via return on query (? energymeter.myMeter.CAIDI.
    ///
    /// DSS property name: CAIDI, DSS property index: 23
    ///
    double CAIDI()
    {
        return Obj_GetFloat64(ptr, Properties::CAIDI);
    }

    EnergyMeter& CAIDI(double value)
    {
        Obj_SetFloat64(ptr, Properties::CAIDI, value);
        return *this;
    }

    /// (Read only) Makes Total Customer Interrupts value result available via return on query (? energymeter.myMeter.CustInterrupts.
    ///
    /// DSS property name: CustInterrupts, DSS property index: 24
    ///
    double CustInterrupts()
    {
        return Obj_GetFloat64(ptr, Properties::CustInterrupts);
    }

    EnergyMeter& CustInterrupts(double value)
    {
        Obj_SetFloat64(ptr, Properties::CustInterrupts, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 25
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    EnergyMeter& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 26
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    EnergyMeter& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 27
    ///
    EnergyMeter& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 27
    ///
    EnergyMeter& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class Sensor: public DSSObj
{
public:
    const static char dss_cls_name[];
    const static int32_t dss_cls_idx = 48;
    struct Properties
    {
        enum {
            element = 1,
            terminal = 2,
            kvbase = 3,
            clear = 4,
            kVs = 5,
            currents = 6,
            kWs = 7,
            kvars = 8,
            conn = 9,
            Deltadirection = 10,
            pctError = 11,
            Weight = 12,
            basefreq = 13,
            enabled = 14,
            like = 15,
        };
    };

    ///
    /// Create wrapper directly by a given object handle/pointer
    ///
    Sensor(APIUtil *util=nullptr, void *ptr_=nullptr): DSSObj(util, ptr_)
    {
    }

    ///
    /// Create a wrapper for an element given by the integer index "idx"
    ///
    Sensor(APIUtil *util, int32_t idx): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByIdx(util->ctx, dss_cls_idx, idx);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Sensor element by the given index");
        }
    }

    ///
    /// Create a wrapper for an element given its name
    ///
    Sensor(APIUtil *util, char *name): DSSObj(util, nullptr)
    {
        ptr = Obj_GetHandleByName(util->ctx, dss_cls_idx, name);
        check_for_error();
        if (ptr == nullptr)
        {
            throw std::runtime_error("Could not find the Sensor element by the given name");
        }
    }

    ///
    /// Returns the object's name
    ///
    const char* name()
    {
        return Obj_GetName(ptr);
    }

    ///
    /// Marks an object for edition. Until `end_edit` is called, some actions are postponed.
    ///
    Sensor& begin_edit()
    {
        Obj_BeginEdit(ptr);
        return *this;
    }

    ///
    /// Finalizes an object edition. This will result more compelx side-effects like Yprim
    /// and other internal data recalculation, for example.
    ///
    Sensor& end_edit(int32_t num_edits=1)
    {
        Obj_EndEdit(ptr, num_edits);
        return *this;
    }

    /// Name (Full Object name) of element to which the Sensor is connected.
    ///
    /// DSS property name: element, DSS property index: 1
    ///
    string element()
    {
        return get_prop_string(Properties::element);
    }

    Sensor& element(const string &value)
    {
        set_string(Properties::element, value);
        return *this;
    }

    Sensor& element(dss::obj::DSSObj &value)
    {
        set_obj(Properties::element, value);
        return *this;
    }

    /// Name (Full Object name) of element to which the Sensor is connected.
    ///
    /// DSS property name: element, DSS property index: 1
    ///
    dss::obj::DSSObj element_obj()
    {
        return get_obj<dss::obj::DSSObj>(Properties::element);
    }

    Sensor& element_obj(dss::obj::DSSObj &value)
    {
        set_obj(Properties::element, value);
        return *this;
    }

    /// Number of the terminal of the circuit element to which the Sensor is connected. 1 or 2, typically. Default is 1.
    ///
    /// DSS property name: terminal, DSS property index: 2
    ///
    int32_t terminal()
    {
        return Obj_GetInt32(ptr, Properties::terminal);
    }

    Sensor& terminal(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::terminal, value);
        return *this;
    }

    /// Voltage base for the sensor, in kV. If connected to a 2- or 3-phase terminal,
    /// specify L-L voltage. For 1-phase devices specify L-N or actual 1-phase voltage. Like many other DSS devices, default is 12.47kV.
    ///
    /// DSS property name: kvbase, DSS property index: 3
    ///
    double kvbase()
    {
        return Obj_GetFloat64(ptr, Properties::kvbase);
    }

    Sensor& kvbase(double value)
    {
        Obj_SetFloat64(ptr, Properties::kvbase, value);
        return *this;
    }

    /// { Yes | No }. Clear=Yes clears sensor values. Should be issued before putting in a new set of measurements.
    ///
    /// DSS property name: clear, DSS property index: 4
    ///
    bool clear()
    {
        return Obj_GetInt32(ptr, Properties::clear) != 0;
    }

    Sensor& clear(bool value)
    {
        Obj_SetInt32(ptr, Properties::clear, value);
        return *this;
    }

    /// Array of Voltages (kV) measured by the voltage sensor. For Delta-connected sensors, Line-Line voltages are expected. For Wye, Line-Neutral are expected.
    ///
    /// DSS property name: kVs, DSS property index: 5
    ///
    VectorXd kVs()
    {
        return get_array<VectorXd>(Properties::kVs);
    }

    Sensor& kVs(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kVs, value);
        return *this;
    }

    /// Array of Currents (amps) measured by the current sensor. Specify this or power quantities; not both.
    ///
    /// DSS property name: currents, DSS property index: 6
    ///
    VectorXd currents()
    {
        return get_array<VectorXd>(Properties::currents);
    }

    Sensor& currents(VectorXd &value)
    {
        set_array<VectorXd>(Properties::currents, value);
        return *this;
    }

    /// Array of Active power (kW) measurements at the sensor. Is converted into Currents along with q=[...]
    /// Will override any currents=[...] specification.
    ///
    /// DSS property name: kWs, DSS property index: 7
    ///
    VectorXd kWs()
    {
        return get_array<VectorXd>(Properties::kWs);
    }

    Sensor& kWs(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kWs, value);
        return *this;
    }

    /// Array of Reactive power (kvar) measurements at the sensor. Is converted into Currents along with p=[...]
    ///
    /// DSS property name: kvars, DSS property index: 8
    ///
    VectorXd kvars()
    {
        return get_array<VectorXd>(Properties::kvars);
    }

    Sensor& kvars(VectorXd &value)
    {
        set_array<VectorXd>(Properties::kvars, value);
        return *this;
    }

    /// Voltage sensor Connection: { wye | delta | LN | LL }.  Default is wye. Applies to voltage measurement only.
    /// Currents are always assumed to be line currents.
    /// If wye or LN, voltage is assumed measured line-neutral; otherwise, line-line.
    ///
    /// DSS property name: conn, DSS property index: 9
    ///
    Connection conn()
    {
        return Connection(Obj_GetInt32(ptr, Properties::conn));
    }

    Sensor& conn(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::conn, value);
        return *this;
    }

    Sensor& conn(Connection value)
    {
        Obj_SetInt32(ptr, Properties::conn, int32_t(value));
        return *this;
    }

    Sensor& conn(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    Sensor& conn(const char *value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// Voltage sensor Connection: { wye | delta | LN | LL }.  Default is wye. Applies to voltage measurement only.
    /// Currents are always assumed to be line currents.
    /// If wye or LN, voltage is assumed measured line-neutral; otherwise, line-line.
    ///
    /// DSS property name: conn, DSS property index: 9
    ///
    string conn_str()
    {
        return get_prop_string(Properties::conn);
    }

    /// Voltage sensor Connection: { wye | delta | LN | LL }.  Default is wye. Applies to voltage measurement only.
    /// Currents are always assumed to be line currents.
    /// If wye or LN, voltage is assumed measured line-neutral; otherwise, line-line.
    ///
    /// DSS property name: conn, DSS property index: 9
    ///
    Sensor& conn_str(const string &value)
    {
        set_string(Properties::conn, value);
        return *this;
    }

    /// {1 or -1}  Default is 1:  1-2, 2-3, 3-1.  For reverse rotation, enter -1. Any positive or negative entry will suffice.
    ///
    /// DSS property name: Deltadirection, DSS property index: 10
    ///
    int32_t Deltadirection()
    {
        return Obj_GetInt32(ptr, Properties::Deltadirection);
    }

    Sensor& Deltadirection(int32_t value)
    {
        Obj_SetInt32(ptr, Properties::Deltadirection, value);
        return *this;
    }

    /// Assumed percent error in the measurement. Default is 1.
    ///
    /// DSS property name: %Error, DSS property index: 11
    ///
    double pctError()
    {
        return Obj_GetFloat64(ptr, Properties::pctError);
    }

    Sensor& pctError(double value)
    {
        Obj_SetFloat64(ptr, Properties::pctError, value);
        return *this;
    }

    /// Weighting factor: Default is 1.
    ///
    /// DSS property name: Weight, DSS property index: 12
    ///
    double Weight()
    {
        return Obj_GetFloat64(ptr, Properties::Weight);
    }

    Sensor& Weight(double value)
    {
        Obj_SetFloat64(ptr, Properties::Weight, value);
        return *this;
    }

    /// Base Frequency for ratings.
    ///
    /// DSS property name: basefreq, DSS property index: 13
    ///
    double basefreq()
    {
        return Obj_GetFloat64(ptr, Properties::basefreq);
    }

    Sensor& basefreq(double value)
    {
        Obj_SetFloat64(ptr, Properties::basefreq, value);
        return *this;
    }

    /// {Yes|No or True|False} Indicates whether this element is enabled.
    ///
    /// DSS property name: enabled, DSS property index: 14
    ///
    bool enabled()
    {
        return Obj_GetInt32(ptr, Properties::enabled) != 0;
    }

    Sensor& enabled(bool value)
    {
        Obj_SetInt32(ptr, Properties::enabled, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 15
    ///
    Sensor& like(const string &value)
    {
        set_string(Properties::like, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 15
    ///
    Sensor& like(const char *value)
    {
        set_string(Properties::like, value);
        return *this;
    }
};


class LineCodeBatch: public DSSBatch
{
public:
    typedef LineCode::Properties Properties;
    typedef LineCode BatchElementClass;

    ///
    /// Create a batch of all LineCode elements
    ///
    LineCodeBatch(APIUtil *util):
        DSSBatch(util, LineCode::dss_cls_idx)
    {
    }

    ///
    /// Create a batch of all LineCode elements that match an integer property value
    ///
    LineCodeBatch(APIUtil *util, int32_t prop_idx, int32_t prop_value):
        DSSBatch(util, LineCode::dss_cls_idx, prop_idx, prop_value)
    {
    }

    ///
    /// Create a batch of all LineCode elements that match a regular expression
    ///
    LineCodeBatch(APIUtil *util, const char* regexp):
        DSSBatch(util, LineCode::dss_cls_idx, regexp)
    {
    }


    LineCodeBatch& begin_edit()
    {
        Batch_BeginEdit(pointer, count[0]);
        return *this;
    }

    LineCodeBatch& end_edit(int32_t num_edits=1)
    {
        Batch_EndEdit(pointer, count[0], num_edits);
        return *this;
    }


    /// Number of phases in the line this line code data represents.  Setting this property reinitializes the line code.  Impedance matrix is reset for default symmetrical component.
    ///
    /// DSS property name: nphases, DSS property index: 1
    ///
    BatchInt32ArrayProxy nphases()
    {
        return BatchInt32ArrayProxy(*this, Properties::nphases);
    }

    LineCodeBatch& nphases(int32_t value)
    {
        set_batch_val(Properties::nphases, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& nphases(T &value)
    {
        set_batch_val_for_each<T>(Properties::nphases, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& nphases(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::nphases, it_begin, it_end);
        return *this;
    }

    /// Positive-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Rmatrix.
    ///
    /// DSS property name: r1, DSS property index: 2
    ///
    BatchFloat64ArrayProxy r1()
    {
        return BatchFloat64ArrayProxy(*this, Properties::r1);
    }

    LineCodeBatch& r1(double value)
    {
        set_batch_val<double>(Properties::r1, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& r1(T &value)
    {
        set_batch_val_for_each<T>(Properties::r1, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& r1(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::r1, it_begin, it_end);
        return *this;
    }

    /// Positive-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Xmatrix
    ///
    /// DSS property name: x1, DSS property index: 3
    ///
    BatchFloat64ArrayProxy x1()
    {
        return BatchFloat64ArrayProxy(*this, Properties::x1);
    }

    LineCodeBatch& x1(double value)
    {
        set_batch_val<double>(Properties::x1, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& x1(T &value)
    {
        set_batch_val_for_each<T>(Properties::x1, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& x1(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::x1, it_begin, it_end);
        return *this;
    }

    /// Zero-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.
    ///
    /// DSS property name: r0, DSS property index: 4
    ///
    BatchFloat64ArrayProxy r0()
    {
        return BatchFloat64ArrayProxy(*this, Properties::r0);
    }

    LineCodeBatch& r0(double value)
    {
        set_batch_val<double>(Properties::r0, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& r0(T &value)
    {
        set_batch_val_for_each<T>(Properties::r0, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& r0(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::r0, it_begin, it_end);
        return *this;
    }

    /// Zero-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.
    ///
    /// DSS property name: x0, DSS property index: 5
    ///
    BatchFloat64ArrayProxy x0()
    {
        return BatchFloat64ArrayProxy(*this, Properties::x0);
    }

    LineCodeBatch& x0(double value)
    {
        set_batch_val<double>(Properties::x0, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& x0(T &value)
    {
        set_batch_val_for_each<T>(Properties::x0, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& x0(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::x0, it_begin, it_end);
        return *this;
    }

    /// Positive-sequence capacitance, nf per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Cmatrix and B1.
    ///
    /// DSS property name: C1, DSS property index: 6
    ///
    BatchFloat64ArrayProxy C1()
    {
        return BatchFloat64ArrayProxy(*this, Properties::C1);
    }

    LineCodeBatch& C1(double value)
    {
        set_batch_val<double>(Properties::C1, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& C1(T &value)
    {
        set_batch_val_for_each<T>(Properties::C1, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& C1(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::C1, it_begin, it_end);
        return *this;
    }

    /// Zero-sequence capacitance, nf per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also B0.
    ///
    /// DSS property name: C0, DSS property index: 7
    ///
    BatchFloat64ArrayProxy C0()
    {
        return BatchFloat64ArrayProxy(*this, Properties::C0);
    }

    LineCodeBatch& C0(double value)
    {
        set_batch_val<double>(Properties::C0, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& C0(T &value)
    {
        set_batch_val_for_each<T>(Properties::C0, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& C0(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::C0, it_begin, it_end);
        return *this;
    }

    /// One of (ohms per ...) {none|mi|km|kft|m|me|ft|in|cm}.  Default is none; assumes units agree with length unitsgiven in Line object
    ///
    /// DSS property name: units, DSS property index: 8
    ///
    BatchInt32ArrayProxy units()
    {
        return BatchInt32ArrayProxy(*this, Properties::units);
    }

    LineCodeBatch& units(string &value)
    {
        set_batch_val(Properties::units, value);
        return *this;
    }

    LineCodeBatch& units(int32_t value)
    {
        set_batch_val(Properties::units, value);
        return *this;
    }

    LineCodeBatch& units(DimensionUnits value)
    {
        set_batch_val(Properties::units, int32_t(value));
        return *this;
    }

    LineCodeBatch& units(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::units, value.begin(), value.end());
        return *this;
    }

    LineCodeBatch& units(std::vector<int32_t> &value)
    {
        set_batch_val_for_each<std::vector<int32_t>>(Properties::units, value.begin(), value.end());
        return *this;
    }

    LineCodeBatch& units(std::vector<DimensionUnits> &value)
    {
        set_batch_val_for_each<std::vector<DimensionUnits>>(Properties::units, value.begin(), value.end());
        return *this;
    }

    /// One of (ohms per ...) {none|mi|km|kft|m|me|ft|in|cm}.  Default is none; assumes units agree with length unitsgiven in Line object
    ///
    /// DSS property name: units, DSS property index: 8
    ///
    strings units_str()
    {
        return get_batch_val<strings>(Properties::units);
    }

    LineCodeBatch& units_str(string &value)
    {
        units(value);
        return *this;
    }

    LineCodeBatch& units_str(strings &value)
    {
        units(value);
        return *this;
    }

    /// Resistance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration.  For balanced line models, you may use the standard symmetrical component data definition instead.
    ///
    /// DSS property name: rmatrix, DSS property index: 9
    ///
    std::vector<VectorXd> rmatrix()
    {
        return get_batch_valarray<VectorXd>(Properties::rmatrix);
    }

    LineCodeBatch& rmatrix(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::rmatrix, value);
        return *this;
    }

    /// Reactance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration.  For balanced line models, you may use the standard symmetrical component data definition instead.
    ///
    /// DSS property name: xmatrix, DSS property index: 10
    ///
    std::vector<VectorXd> xmatrix()
    {
        return get_batch_valarray<VectorXd>(Properties::xmatrix);
    }

    LineCodeBatch& xmatrix(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::xmatrix, value);
        return *this;
    }

    /// Nodal Capacitance matrix, lower triangle, nf per unit length.Order of the matrix is the number of phases. May be used to specify the shunt capacitance of any line configuration.  For balanced line models, you may use the standard symmetrical component data definition instead.
    ///
    /// DSS property name: cmatrix, DSS property index: 11
    ///
    std::vector<VectorXd> cmatrix()
    {
        return get_batch_valarray<VectorXd>(Properties::cmatrix);
    }

    LineCodeBatch& cmatrix(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::cmatrix, value);
        return *this;
    }

    /// Frequency at which impedances are specified.
    ///
    /// DSS property name: baseFreq, DSS property index: 12
    ///
    BatchFloat64ArrayProxy baseFreq()
    {
        return BatchFloat64ArrayProxy(*this, Properties::baseFreq);
    }

    LineCodeBatch& baseFreq(double value)
    {
        set_batch_val<double>(Properties::baseFreq, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& baseFreq(T &value)
    {
        set_batch_val_for_each<T>(Properties::baseFreq, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& baseFreq(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::baseFreq, it_begin, it_end);
        return *this;
    }

    /// Normal ampere limit on line.  This is the so-called Planning Limit. It may also be the value above which load will have to be dropped in a contingency.  Usually about 75% - 80% of the emergency (one-hour) rating.
    ///
    /// DSS property name: normamps, DSS property index: 13
    ///
    BatchFloat64ArrayProxy normamps()
    {
        return BatchFloat64ArrayProxy(*this, Properties::normamps);
    }

    LineCodeBatch& normamps(double value)
    {
        set_batch_val<double>(Properties::normamps, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& normamps(T &value)
    {
        set_batch_val_for_each<T>(Properties::normamps, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& normamps(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::normamps, it_begin, it_end);
        return *this;
    }

    /// Emergency ampere limit on line (usually one-hour rating).
    ///
    /// DSS property name: emergamps, DSS property index: 14
    ///
    BatchFloat64ArrayProxy emergamps()
    {
        return BatchFloat64ArrayProxy(*this, Properties::emergamps);
    }

    LineCodeBatch& emergamps(double value)
    {
        set_batch_val<double>(Properties::emergamps, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& emergamps(T &value)
    {
        set_batch_val_for_each<T>(Properties::emergamps, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& emergamps(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::emergamps, it_begin, it_end);
        return *this;
    }

    /// Number of faults per unit length per year.
    ///
    /// DSS property name: faultrate, DSS property index: 15
    ///
    BatchFloat64ArrayProxy faultrate()
    {
        return BatchFloat64ArrayProxy(*this, Properties::faultrate);
    }

    LineCodeBatch& faultrate(double value)
    {
        set_batch_val<double>(Properties::faultrate, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& faultrate(T &value)
    {
        set_batch_val_for_each<T>(Properties::faultrate, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& faultrate(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::faultrate, it_begin, it_end);
        return *this;
    }

    /// Percentage of the faults that become permanent.
    ///
    /// DSS property name: pctperm, DSS property index: 16
    ///
    BatchFloat64ArrayProxy pctperm()
    {
        return BatchFloat64ArrayProxy(*this, Properties::pctperm);
    }

    LineCodeBatch& pctperm(double value)
    {
        set_batch_val<double>(Properties::pctperm, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& pctperm(T &value)
    {
        set_batch_val_for_each<T>(Properties::pctperm, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& pctperm(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::pctperm, it_begin, it_end);
        return *this;
    }

    /// Hours to repair.
    ///
    /// DSS property name: repair, DSS property index: 17
    ///
    BatchFloat64ArrayProxy repair()
    {
        return BatchFloat64ArrayProxy(*this, Properties::repair);
    }

    LineCodeBatch& repair(double value)
    {
        set_batch_val<double>(Properties::repair, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& repair(T &value)
    {
        set_batch_val_for_each<T>(Properties::repair, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& repair(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::repair, it_begin, it_end);
        return *this;
    }

    /// Kron = Y/N. Default=N.  Perform Kron reduction on the impedance matrix after it is formed, reducing order by 1. Eliminates the conductor designated by the "Neutral=" property. Do this after the R, X, and C matrices are defined. Ignored for symmetrical components. May be issued more than once to eliminate more than one conductor by resetting the Neutral property after the previous invoking of this property. Generally, you do not want to do a Kron reduction on the matrix if you intend to solve at a frequency other than the base frequency and exploit the Rg and Xg values.
    ///
    /// DSS property name: Kron, DSS property index: 18
    ///
    LineCodeBatch& Kron(bool value)
    {
        set_batch_val(Properties::Kron, int32_t(value));
        return *this;
    }

    /// Carson earth return resistance per unit length used to compute impedance values at base frequency.  For making better frequency adjustments. Default is 0.01805 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.
    ///
    /// DSS property name: Rg, DSS property index: 19
    ///
    BatchFloat64ArrayProxy Rg()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Rg);
    }

    LineCodeBatch& Rg(double value)
    {
        set_batch_val<double>(Properties::Rg, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& Rg(T &value)
    {
        set_batch_val_for_each<T>(Properties::Rg, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& Rg(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Rg, it_begin, it_end);
        return *this;
    }

    /// Carson earth return reactance per unit length used to compute impedance values at base frequency.  For making better frequency adjustments. Default value is 0.155081 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.
    ///
    /// DSS property name: Xg, DSS property index: 20
    ///
    BatchFloat64ArrayProxy Xg()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Xg);
    }

    LineCodeBatch& Xg(double value)
    {
        set_batch_val<double>(Properties::Xg, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& Xg(T &value)
    {
        set_batch_val_for_each<T>(Properties::Xg, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& Xg(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Xg, it_begin, it_end);
        return *this;
    }

    /// Default=100 meter ohms.  Earth resitivity used to compute earth correction factor.
    ///
    /// DSS property name: rho, DSS property index: 21
    ///
    BatchFloat64ArrayProxy rho()
    {
        return BatchFloat64ArrayProxy(*this, Properties::rho);
    }

    LineCodeBatch& rho(double value)
    {
        set_batch_val<double>(Properties::rho, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& rho(T &value)
    {
        set_batch_val_for_each<T>(Properties::rho, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& rho(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::rho, it_begin, it_end);
        return *this;
    }

    /// Designates which conductor is the "neutral" conductor that will be eliminated by Kron reduction. Default is the last conductor (nphases value). After Kron reduction is set to 0. Subsequent issuing of Kron=Yes will not do anything until this property is set to a legal value. Applies only to LineCodes defined by R, X, and C matrix.
    ///
    /// DSS property name: neutral, DSS property index: 22
    ///
    BatchInt32ArrayProxy neutral()
    {
        return BatchInt32ArrayProxy(*this, Properties::neutral);
    }

    LineCodeBatch& neutral(int32_t value)
    {
        set_batch_val(Properties::neutral, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& neutral(T &value)
    {
        set_batch_val_for_each<T>(Properties::neutral, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& neutral(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::neutral, it_begin, it_end);
        return *this;
    }

    /// Alternate way to specify C1. MicroS per unit length
    ///
    /// DSS property name: B1, DSS property index: 23
    ///
    BatchFloat64ArrayProxy B1()
    {
        return BatchFloat64ArrayProxy(*this, Properties::B1);
    }

    LineCodeBatch& B1(double value)
    {
        set_batch_val<double>(Properties::B1, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& B1(T &value)
    {
        set_batch_val_for_each<T>(Properties::B1, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& B1(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::B1, it_begin, it_end);
        return *this;
    }

    /// Alternate way to specify C0. MicroS per unit length
    ///
    /// DSS property name: B0, DSS property index: 24
    ///
    BatchFloat64ArrayProxy B0()
    {
        return BatchFloat64ArrayProxy(*this, Properties::B0);
    }

    LineCodeBatch& B0(double value)
    {
        set_batch_val<double>(Properties::B0, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& B0(T &value)
    {
        set_batch_val_for_each<T>(Properties::B0, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& B0(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::B0, it_begin, it_end);
        return *this;
    }

    /// Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.
    ///
    /// DSS property name: Seasons, DSS property index: 25
    ///
    BatchInt32ArrayProxy Seasons()
    {
        return BatchInt32ArrayProxy(*this, Properties::Seasons);
    }

    LineCodeBatch& Seasons(int32_t value)
    {
        set_batch_val(Properties::Seasons, value);
        return *this;
    }

    template <typename T>
    LineCodeBatch& Seasons(T &value)
    {
        set_batch_val_for_each<T>(Properties::Seasons, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LineCodeBatch& Seasons(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Seasons, it_begin, it_end);
        return *this;
    }

    /// An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
    /// multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.
    ///
    /// DSS property name: Ratings, DSS property index: 26
    ///
    std::vector<VectorXd> Ratings()
    {
        return get_batch_valarray<VectorXd>(Properties::Ratings);
    }

    LineCodeBatch& Ratings(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::Ratings, value);
        return *this;
    }

    /// Code designating the type of line.
    /// One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW
    ///
    /// OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.
    ///
    /// DSS property name: LineType, DSS property index: 27
    ///
    BatchInt32ArrayProxy linetype()
    {
        return BatchInt32ArrayProxy(*this, Properties::LineType);
    }

    LineCodeBatch& linetype(string &value)
    {
        set_batch_val(Properties::LineType, value);
        return *this;
    }

    LineCodeBatch& linetype(int32_t value)
    {
        set_batch_val(Properties::LineType, value);
        return *this;
    }

    LineCodeBatch& linetype(LineType value)
    {
        set_batch_val(Properties::LineType, int32_t(value));
        return *this;
    }

    LineCodeBatch& linetype(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::LineType, value.begin(), value.end());
        return *this;
    }

    LineCodeBatch& linetype(std::vector<int32_t> &value)
    {
        set_batch_val_for_each<std::vector<int32_t>>(Properties::LineType, value.begin(), value.end());
        return *this;
    }

    LineCodeBatch& linetype(std::vector<LineType> &value)
    {
        set_batch_val_for_each<std::vector<LineType>>(Properties::LineType, value.begin(), value.end());
        return *this;
    }

    /// Code designating the type of line.
    /// One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW
    ///
    /// OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.
    ///
    /// DSS property name: LineType, DSS property index: 27
    ///
    strings linetype_str()
    {
        return get_batch_val<strings>(Properties::LineType);
    }

    LineCodeBatch& linetype_str(string &value)
    {
        linetype(value);
        return *this;
    }

    LineCodeBatch& linetype_str(strings &value)
    {
        linetype(value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 28
    ///
    LineCodeBatch& like(const string &value)
    {
        set_batch_val(Properties::like, value.c_str());
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 28
    ///
    LineCodeBatch& like(const char *value)
    {
        set_batch_val(Properties::like, value);
        return *this;
    }
};


class LoadShapeBatch: public DSSBatch
{
public:
    typedef LoadShape::Properties Properties;
    typedef LoadShape BatchElementClass;

    // Shortcuts to class-specific enumerations
    typedef LoadShape::LoadShapeAction LoadShapeAction;


    ///
    /// Create a batch of all LoadShape elements
    ///
    LoadShapeBatch(APIUtil *util):
        DSSBatch(util, LoadShape::dss_cls_idx)
    {
    }

    ///
    /// Create a batch of all LoadShape elements that match an integer property value
    ///
    LoadShapeBatch(APIUtil *util, int32_t prop_idx, int32_t prop_value):
        DSSBatch(util, LoadShape::dss_cls_idx, prop_idx, prop_value)
    {
    }

    ///
    /// Create a batch of all LoadShape elements that match a regular expression
    ///
    LoadShapeBatch(APIUtil *util, const char* regexp):
        DSSBatch(util, LoadShape::dss_cls_idx, regexp)
    {
    }


    LoadShapeBatch& begin_edit()
    {
        Batch_BeginEdit(pointer, count[0]);
        return *this;
    }

    LoadShapeBatch& end_edit(int32_t num_edits=1)
    {
        Batch_EndEdit(pointer, count[0], num_edits);
        return *this;
    }


    /// Max number of points to expect in load shape vectors. This gets reset to the number of multiplier values found (in files only) if less than specified.
    ///
    /// DSS property name: npts, DSS property index: 1
    ///
    BatchInt32ArrayProxy npts()
    {
        return BatchInt32ArrayProxy(*this, Properties::npts);
    }

    LoadShapeBatch& npts(int32_t value)
    {
        set_batch_val(Properties::npts, value);
        return *this;
    }

    template <typename T>
    LoadShapeBatch& npts(T &value)
    {
        set_batch_val_for_each<T>(Properties::npts, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LoadShapeBatch& npts(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::npts, it_begin, it_end);
        return *this;
    }

    /// Time interval for fixed interval data, hrs. Default = 1. If Interval = 0 then time data (in hours) may be at either regular or  irregular intervals and time value must be specified using either the Hour property or input files. Then values are interpolated when Interval=0, but not for fixed interval data.
    ///
    /// See also "sinterval" and "minterval".
    ///
    /// DSS property name: interval, DSS property index: 2
    ///
    BatchFloat64ArrayProxy interval()
    {
        return BatchFloat64ArrayProxy(*this, Properties::interval);
    }

    LoadShapeBatch& interval(double value)
    {
        set_batch_val<double>(Properties::interval, value);
        return *this;
    }

    template <typename T>
    LoadShapeBatch& interval(T &value)
    {
        set_batch_val_for_each<T>(Properties::interval, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LoadShapeBatch& interval(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::interval, it_begin, it_end);
        return *this;
    }

    /// Array of multiplier values for active power (P) or other key value (such as pu V for Vsource).
    ///
    /// You can also use the syntax:
    ///
    /// mult = (file=filename)     !for text file one value per line
    /// mult = (dblfile=filename)  !for packed file of doubles
    /// mult = (sngfile=filename)  !for packed file of singles
    /// mult = (file=MyCSVFile.CSV, col=3, header=yes)  !for multicolumn CSV files
    ///
    /// Note: this property will reset Npts if the  number of values in the files are fewer.
    ///
    /// Same as Pmult
    ///
    /// DSS property name: mult, DSS property index: 3
    ///
    std::vector<VectorXd> mult()
    {
        return get_batch_valarray<VectorXd>(Properties::mult);
    }

    LoadShapeBatch& mult(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::mult, value);
        return *this;
    }

    /// Array of hour values. Only necessary to define for variable interval data (Interval=0). If you set Interval>0 to denote fixed interval data, DO NOT USE THIS PROPERTY. You can also use the syntax:
    /// hour = (file=filename)     !for text file one value per line
    /// hour = (dblfile=filename)  !for packed file of doubles
    /// hour = (sngfile=filename)  !for packed file of singles
    ///
    /// DSS property name: hour, DSS property index: 4
    ///
    std::vector<VectorXd> hour()
    {
        return get_batch_valarray<VectorXd>(Properties::hour);
    }

    LoadShapeBatch& hour(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::hour, value);
        return *this;
    }

    /// Mean of the active power multipliers.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently. Used for Monte Carlo load simulations.
    ///
    /// DSS property name: mean, DSS property index: 5
    ///
    BatchFloat64ArrayProxy mean()
    {
        return BatchFloat64ArrayProxy(*this, Properties::mean);
    }

    LoadShapeBatch& mean(double value)
    {
        set_batch_val<double>(Properties::mean, value);
        return *this;
    }

    template <typename T>
    LoadShapeBatch& mean(T &value)
    {
        set_batch_val_for_each<T>(Properties::mean, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LoadShapeBatch& mean(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::mean, it_begin, it_end);
        return *this;
    }

    /// Standard deviation of active power multipliers.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently.Is overwritten if you subsequently read in a curve
    ///
    /// Used for Monte Carlo load simulations.
    ///
    /// DSS property name: stddev, DSS property index: 6
    ///
    BatchFloat64ArrayProxy stddev()
    {
        return BatchFloat64ArrayProxy(*this, Properties::stddev);
    }

    LoadShapeBatch& stddev(double value)
    {
        set_batch_val<double>(Properties::stddev, value);
        return *this;
    }

    template <typename T>
    LoadShapeBatch& stddev(T &value)
    {
        set_batch_val_for_each<T>(Properties::stddev, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LoadShapeBatch& stddev(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::stddev, it_begin, it_end);
        return *this;
    }

    /// Switch input of active power load curve data to a CSV text file containing (hour, mult) points, or simply (mult) values for fixed time interval data, one per line. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: csvfile, DSS property index: 7
    ///
    strings csvfile()
    {
        return get_batch_val<strings>(Properties::csvfile);
    }

    LoadShapeBatch& csvfile(const string &value)
    {
        set_batch_val(Properties::csvfile, value.c_str());
        return *this;
    }

    LoadShapeBatch& csvfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::csvfile, value.begin(), value.end());
        return *this;
    }

    /// Switch input of active power load curve data to a binary file of singles containing (hour, mult) points, or simply (mult) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: sngfile, DSS property index: 8
    ///
    strings sngfile()
    {
        return get_batch_val<strings>(Properties::sngfile);
    }

    LoadShapeBatch& sngfile(const string &value)
    {
        set_batch_val(Properties::sngfile, value.c_str());
        return *this;
    }

    LoadShapeBatch& sngfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::sngfile, value.begin(), value.end());
        return *this;
    }

    /// Switch input of active power load curve data to a binary file of doubles containing (hour, mult) points, or simply (mult) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: dblfile, DSS property index: 9
    ///
    strings dblfile()
    {
        return get_batch_val<strings>(Properties::dblfile);
    }

    LoadShapeBatch& dblfile(const string &value)
    {
        set_batch_val(Properties::dblfile, value.c_str());
        return *this;
    }

    LoadShapeBatch& dblfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::dblfile, value.begin(), value.end());
        return *this;
    }

    /// {NORMALIZE | DblSave | SngSave} After defining load curve data, setting action=normalize will modify the multipliers so that the peak is 1.0. The mean and std deviation are recomputed.
    ///
    /// Setting action=DblSave or SngSave will cause the present mult and qmult values to be written to either a packed file of double or single. The filename is the loadshape name. The mult array will have a "_P" appended on the file name and the qmult array, if it exists, will have "_Q" appended.
    ///
    /// DSS property name: action, DSS property index: 10
    ///
    LoadShapeBatch& action(int32_t value)
    {
        set_batch_val(Properties::action, value);
        return *this;
    }

    /// {NORMALIZE | DblSave | SngSave} After defining load curve data, setting action=normalize will modify the multipliers so that the peak is 1.0. The mean and std deviation are recomputed.
    ///
    /// Setting action=DblSave or SngSave will cause the present mult and qmult values to be written to either a packed file of double or single. The filename is the loadshape name. The mult array will have a "_P" appended on the file name and the qmult array, if it exists, will have "_Q" appended.
    ///
    /// DSS property name: action, DSS property index: 10
    ///
    LoadShapeBatch& action(LoadShape::LoadShapeAction value)
    {
        set_batch_val(Properties::action, int32_t(value));
        return *this;
    }

    /// {NORMALIZE | DblSave | SngSave} After defining load curve data, setting action=normalize will modify the multipliers so that the peak is 1.0. The mean and std deviation are recomputed.
    ///
    /// Setting action=DblSave or SngSave will cause the present mult and qmult values to be written to either a packed file of double or single. The filename is the loadshape name. The mult array will have a "_P" appended on the file name and the qmult array, if it exists, will have "_Q" appended.
    ///
    /// DSS property name: action, DSS property index: 10
    ///
    LoadShapeBatch& action(const string &value)
    {
        set_batch_val(Properties::action, value.c_str());
        return *this;
    }

    /// {NORMALIZE | DblSave | SngSave} After defining load curve data, setting action=normalize will modify the multipliers so that the peak is 1.0. The mean and std deviation are recomputed.
    ///
    /// Setting action=DblSave or SngSave will cause the present mult and qmult values to be written to either a packed file of double or single. The filename is the loadshape name. The mult array will have a "_P" appended on the file name and the qmult array, if it exists, will have "_Q" appended.
    ///
    /// DSS property name: action, DSS property index: 10
    ///
    LoadShapeBatch& action(const char *value)
    {
        set_batch_val(Properties::action, value);
        return *this;
    }

    /// Array of multiplier values for reactive power (Q).  You can also use the syntax:
    /// qmult = (file=filename)     !for text file one value per line
    /// qmult = (dblfile=filename)  !for packed file of doubles
    /// qmult = (sngfile=filename)  !for packed file of singles
    /// qmult = (file=MyCSVFile.CSV, col=4, header=yes)  !for multicolumn CSV files
    ///
    /// DSS property name: qmult, DSS property index: 11
    ///
    std::vector<VectorXd> qmult()
    {
        return get_batch_valarray<VectorXd>(Properties::qmult);
    }

    LoadShapeBatch& qmult(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::qmult, value);
        return *this;
    }

    /// {Yes | No* | True | False*} If true, signifies to Load, Generator, Vsource, or other objects to use the return value as the actual kW, kvar, kV, or other value rather than a multiplier. Nominally for AMI Load data but may be used for other functions.
    ///
    /// DSS property name: UseActual, DSS property index: 12
    ///
    bools UseActual()
    {
        return get_batch_val<bools>(Properties::UseActual);
    }

    LoadShapeBatch& UseActual(bool value)
    {
        set_batch_val(Properties::UseActual, int32_t(value));
        return *this;
    }

    LoadShapeBatch& UseActual(bools &value)
    {
        set_batch_val_for_each<std::vector<int32_t>>(Properties::UseActual, value.begin(), value.end());
        return *this;
    }

    /// kW value at the time of max power. Is automatically set upon reading in a loadshape. Use this property to override the value automatically computed or to retrieve the value computed.
    ///
    /// DSS property name: Pmax, DSS property index: 13
    ///
    BatchFloat64ArrayProxy Pmax()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Pmax);
    }

    LoadShapeBatch& Pmax(double value)
    {
        set_batch_val<double>(Properties::Pmax, value);
        return *this;
    }

    template <typename T>
    LoadShapeBatch& Pmax(T &value)
    {
        set_batch_val_for_each<T>(Properties::Pmax, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LoadShapeBatch& Pmax(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Pmax, it_begin, it_end);
        return *this;
    }

    /// kvar value at the time of max kW power. Is automatically set upon reading in a loadshape. Use this property to override the value automatically computed or to retrieve the value computed.
    ///
    /// DSS property name: Qmax, DSS property index: 14
    ///
    BatchFloat64ArrayProxy Qmax()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Qmax);
    }

    LoadShapeBatch& Qmax(double value)
    {
        set_batch_val<double>(Properties::Qmax, value);
        return *this;
    }

    template <typename T>
    LoadShapeBatch& Qmax(T &value)
    {
        set_batch_val_for_each<T>(Properties::Qmax, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LoadShapeBatch& Qmax(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Qmax, it_begin, it_end);
        return *this;
    }

    /// Specify fixed interval in SECONDS. Alternate way to specify Interval property.
    ///
    /// DSS property name: sinterval, DSS property index: 15
    ///
    BatchFloat64ArrayProxy sinterval()
    {
        return BatchFloat64ArrayProxy(*this, Properties::sinterval);
    }

    LoadShapeBatch& sinterval(double value)
    {
        set_batch_val<double>(Properties::sinterval, value);
        return *this;
    }

    template <typename T>
    LoadShapeBatch& sinterval(T &value)
    {
        set_batch_val_for_each<T>(Properties::sinterval, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LoadShapeBatch& sinterval(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::sinterval, it_begin, it_end);
        return *this;
    }

    /// Specify fixed interval in MINUTES. Alternate way to specify Interval property.
    ///
    /// DSS property name: minterval, DSS property index: 16
    ///
    BatchFloat64ArrayProxy minterval()
    {
        return BatchFloat64ArrayProxy(*this, Properties::minterval);
    }

    LoadShapeBatch& minterval(double value)
    {
        set_batch_val<double>(Properties::minterval, value);
        return *this;
    }

    template <typename T>
    LoadShapeBatch& minterval(T &value)
    {
        set_batch_val_for_each<T>(Properties::minterval, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LoadShapeBatch& minterval(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::minterval, it_begin, it_end);
        return *this;
    }

    /// Base P value for normalization. Default is zero, meaning the peak will be used.
    ///
    /// DSS property name: Pbase, DSS property index: 17
    ///
    BatchFloat64ArrayProxy Pbase()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Pbase);
    }

    LoadShapeBatch& Pbase(double value)
    {
        set_batch_val<double>(Properties::Pbase, value);
        return *this;
    }

    template <typename T>
    LoadShapeBatch& Pbase(T &value)
    {
        set_batch_val_for_each<T>(Properties::Pbase, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LoadShapeBatch& Pbase(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Pbase, it_begin, it_end);
        return *this;
    }

    /// Base Q value for normalization. Default is zero, meaning the peak will be used.
    ///
    /// DSS property name: Qbase, DSS property index: 18
    ///
    BatchFloat64ArrayProxy Qbase()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Qbase);
    }

    LoadShapeBatch& Qbase(double value)
    {
        set_batch_val<double>(Properties::Qbase, value);
        return *this;
    }

    template <typename T>
    LoadShapeBatch& Qbase(T &value)
    {
        set_batch_val_for_each<T>(Properties::Qbase, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    LoadShapeBatch& Qbase(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Qbase, it_begin, it_end);
        return *this;
    }

    /// Synonym for "mult".
    ///
    /// DSS property name: Pmult, DSS property index: 19
    ///
    std::vector<VectorXd> Pmult()
    {
        return get_batch_valarray<VectorXd>(Properties::Pmult);
    }

    LoadShapeBatch& Pmult(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::Pmult, value);
        return *this;
    }

    /// Switch input to a CSV text file containing (active, reactive) power (P, Q) multiplier pairs, one per row.
    /// If the interval=0, there should be 3 items on each line: (hour, Pmult, Qmult)
    ///
    /// DSS property name: PQCSVFile, DSS property index: 20
    ///
    strings PQCSVFile()
    {
        return get_batch_val<strings>(Properties::PQCSVFile);
    }

    LoadShapeBatch& PQCSVFile(const string &value)
    {
        set_batch_val(Properties::PQCSVFile, value.c_str());
        return *this;
    }

    LoadShapeBatch& PQCSVFile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::PQCSVFile, value.begin(), value.end());
        return *this;
    }

    /// {Yes | No* | True | False*} Enables the memory mapping functionality for dealing with large amounts of load shapes.
    /// By defaul is False. Use it to accelerate the model loading when the containing a large number of load shapes.
    ///
    /// DSS property name: MemoryMapping, DSS property index: 21
    ///
    bools MemoryMapping()
    {
        return get_batch_val<bools>(Properties::MemoryMapping);
    }

    LoadShapeBatch& MemoryMapping(bool value)
    {
        set_batch_val(Properties::MemoryMapping, int32_t(value));
        return *this;
    }

    LoadShapeBatch& MemoryMapping(bools &value)
    {
        set_batch_val_for_each<std::vector<int32_t>>(Properties::MemoryMapping, value.begin(), value.end());
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 22
    ///
    LoadShapeBatch& like(const string &value)
    {
        set_batch_val(Properties::like, value.c_str());
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 22
    ///
    LoadShapeBatch& like(const char *value)
    {
        set_batch_val(Properties::like, value);
        return *this;
    }
};


class TShapeBatch: public DSSBatch
{
public:
    typedef TShape::Properties Properties;
    typedef TShape BatchElementClass;

    // Shortcuts to class-specific enumerations
    typedef TShape::TShapeAction TShapeAction;


    ///
    /// Create a batch of all TShape elements
    ///
    TShapeBatch(APIUtil *util):
        DSSBatch(util, TShape::dss_cls_idx)
    {
    }

    ///
    /// Create a batch of all TShape elements that match an integer property value
    ///
    TShapeBatch(APIUtil *util, int32_t prop_idx, int32_t prop_value):
        DSSBatch(util, TShape::dss_cls_idx, prop_idx, prop_value)
    {
    }

    ///
    /// Create a batch of all TShape elements that match a regular expression
    ///
    TShapeBatch(APIUtil *util, const char* regexp):
        DSSBatch(util, TShape::dss_cls_idx, regexp)
    {
    }


    TShapeBatch& begin_edit()
    {
        Batch_BeginEdit(pointer, count[0]);
        return *this;
    }

    TShapeBatch& end_edit(int32_t num_edits=1)
    {
        Batch_EndEdit(pointer, count[0], num_edits);
        return *this;
    }


    /// Max number of points to expect in temperature shape vectors. This gets reset to the number of Temperature values found if less than specified.
    ///
    /// DSS property name: npts, DSS property index: 1
    ///
    BatchInt32ArrayProxy npts()
    {
        return BatchInt32ArrayProxy(*this, Properties::npts);
    }

    TShapeBatch& npts(int32_t value)
    {
        set_batch_val(Properties::npts, value);
        return *this;
    }

    template <typename T>
    TShapeBatch& npts(T &value)
    {
        set_batch_val_for_each<T>(Properties::npts, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    TShapeBatch& npts(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::npts, it_begin, it_end);
        return *this;
    }

    /// Time interval for fixed interval data, hrs. Default = 1. If Interval = 0 then time data (in hours) may be at irregular intervals and time value must be specified using either the Hour property or input files. Then values are interpolated when Interval=0, but not for fixed interval data.
    ///
    /// See also "sinterval" and "minterval".
    ///
    /// DSS property name: interval, DSS property index: 2
    ///
    BatchFloat64ArrayProxy interval()
    {
        return BatchFloat64ArrayProxy(*this, Properties::interval);
    }

    TShapeBatch& interval(double value)
    {
        set_batch_val<double>(Properties::interval, value);
        return *this;
    }

    template <typename T>
    TShapeBatch& interval(T &value)
    {
        set_batch_val_for_each<T>(Properties::interval, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    TShapeBatch& interval(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::interval, it_begin, it_end);
        return *this;
    }

    /// Array of temperature values.  Units should be compatible with the object using the data. You can also use the syntax:
    /// Temp = (file=filename)     !for text file one value per line
    /// Temp = (dblfile=filename)  !for packed file of doubles
    /// Temp = (sngfile=filename)  !for packed file of singles
    ///
    /// Note: this property will reset Npts if the  number of values in the files are fewer.
    ///
    /// DSS property name: temp, DSS property index: 3
    ///
    std::vector<VectorXd> temp()
    {
        return get_batch_valarray<VectorXd>(Properties::temp);
    }

    TShapeBatch& temp(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::temp, value);
        return *this;
    }

    /// Array of hour values. Only necessary to define this property for variable interval data. If the data are fixed interval, do not use this property. You can also use the syntax:
    /// hour = (file=filename)     !for text file one value per line
    /// hour = (dblfile=filename)  !for packed file of doubles
    /// hour = (sngfile=filename)  !for packed file of singles
    ///
    /// DSS property name: hour, DSS property index: 4
    ///
    std::vector<VectorXd> hour()
    {
        return get_batch_valarray<VectorXd>(Properties::hour);
    }

    TShapeBatch& hour(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::hour, value);
        return *this;
    }

    /// Mean of the temperature curve values.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently. Used for Monte Carlo load simulations.
    ///
    /// DSS property name: mean, DSS property index: 5
    ///
    BatchFloat64ArrayProxy mean()
    {
        return BatchFloat64ArrayProxy(*this, Properties::mean);
    }

    TShapeBatch& mean(double value)
    {
        set_batch_val<double>(Properties::mean, value);
        return *this;
    }

    template <typename T>
    TShapeBatch& mean(T &value)
    {
        set_batch_val_for_each<T>(Properties::mean, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    TShapeBatch& mean(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::mean, it_begin, it_end);
        return *this;
    }

    /// Standard deviation of the temperatures.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently.Is overwritten if you subsequently read in a curve
    ///
    /// Used for Monte Carlo load simulations.
    ///
    /// DSS property name: stddev, DSS property index: 6
    ///
    BatchFloat64ArrayProxy stddev()
    {
        return BatchFloat64ArrayProxy(*this, Properties::stddev);
    }

    TShapeBatch& stddev(double value)
    {
        set_batch_val<double>(Properties::stddev, value);
        return *this;
    }

    template <typename T>
    TShapeBatch& stddev(T &value)
    {
        set_batch_val_for_each<T>(Properties::stddev, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    TShapeBatch& stddev(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::stddev, it_begin, it_end);
        return *this;
    }

    /// Switch input of  temperature curve data to a csv file containing (hour, Temp) points, or simply (Temp) values for fixed time interval data, one per line. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: csvfile, DSS property index: 7
    ///
    strings csvfile()
    {
        return get_batch_val<strings>(Properties::csvfile);
    }

    TShapeBatch& csvfile(const string &value)
    {
        set_batch_val(Properties::csvfile, value.c_str());
        return *this;
    }

    TShapeBatch& csvfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::csvfile, value.begin(), value.end());
        return *this;
    }

    /// Switch input of  temperature curve data to a binary file of singles containing (hour, Temp) points, or simply (Temp) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: sngfile, DSS property index: 8
    ///
    strings sngfile()
    {
        return get_batch_val<strings>(Properties::sngfile);
    }

    TShapeBatch& sngfile(const string &value)
    {
        set_batch_val(Properties::sngfile, value.c_str());
        return *this;
    }

    TShapeBatch& sngfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::sngfile, value.begin(), value.end());
        return *this;
    }

    /// Switch input of  temperature curve data to a binary file of doubles containing (hour, Temp) points, or simply (Temp) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: dblfile, DSS property index: 9
    ///
    strings dblfile()
    {
        return get_batch_val<strings>(Properties::dblfile);
    }

    TShapeBatch& dblfile(const string &value)
    {
        set_batch_val(Properties::dblfile, value.c_str());
        return *this;
    }

    TShapeBatch& dblfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::dblfile, value.begin(), value.end());
        return *this;
    }

    /// Specify fixed interval in SECONDS. Alternate way to specify Interval property.
    ///
    /// DSS property name: sinterval, DSS property index: 10
    ///
    BatchFloat64ArrayProxy sinterval()
    {
        return BatchFloat64ArrayProxy(*this, Properties::sinterval);
    }

    TShapeBatch& sinterval(double value)
    {
        set_batch_val<double>(Properties::sinterval, value);
        return *this;
    }

    template <typename T>
    TShapeBatch& sinterval(T &value)
    {
        set_batch_val_for_each<T>(Properties::sinterval, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    TShapeBatch& sinterval(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::sinterval, it_begin, it_end);
        return *this;
    }

    /// Specify fixed interval in MINUTES. Alternate way to specify Interval property.
    ///
    /// DSS property name: minterval, DSS property index: 11
    ///
    BatchFloat64ArrayProxy minterval()
    {
        return BatchFloat64ArrayProxy(*this, Properties::minterval);
    }

    TShapeBatch& minterval(double value)
    {
        set_batch_val<double>(Properties::minterval, value);
        return *this;
    }

    template <typename T>
    TShapeBatch& minterval(T &value)
    {
        set_batch_val_for_each<T>(Properties::minterval, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    TShapeBatch& minterval(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::minterval, it_begin, it_end);
        return *this;
    }

    /// {DblSave | SngSave} After defining temperature curve data... Setting action=DblSave or SngSave will cause the present "Temp" values to be written to either a packed file of double or single. The filename is the Tshape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    TShapeBatch& action(int32_t value)
    {
        set_batch_val(Properties::action, value);
        return *this;
    }

    /// {DblSave | SngSave} After defining temperature curve data... Setting action=DblSave or SngSave will cause the present "Temp" values to be written to either a packed file of double or single. The filename is the Tshape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    TShapeBatch& action(TShape::TShapeAction value)
    {
        set_batch_val(Properties::action, int32_t(value));
        return *this;
    }

    /// {DblSave | SngSave} After defining temperature curve data... Setting action=DblSave or SngSave will cause the present "Temp" values to be written to either a packed file of double or single. The filename is the Tshape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    TShapeBatch& action(const string &value)
    {
        set_batch_val(Properties::action, value.c_str());
        return *this;
    }

    /// {DblSave | SngSave} After defining temperature curve data... Setting action=DblSave or SngSave will cause the present "Temp" values to be written to either a packed file of double or single. The filename is the Tshape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    TShapeBatch& action(const char *value)
    {
        set_batch_val(Properties::action, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 13
    ///
    TShapeBatch& like(const string &value)
    {
        set_batch_val(Properties::like, value.c_str());
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 13
    ///
    TShapeBatch& like(const char *value)
    {
        set_batch_val(Properties::like, value);
        return *this;
    }
};


class PriceShapeBatch: public DSSBatch
{
public:
    typedef PriceShape::Properties Properties;
    typedef PriceShape BatchElementClass;

    // Shortcuts to class-specific enumerations
    typedef PriceShape::PriceShapeAction PriceShapeAction;


    ///
    /// Create a batch of all PriceShape elements
    ///
    PriceShapeBatch(APIUtil *util):
        DSSBatch(util, PriceShape::dss_cls_idx)
    {
    }

    ///
    /// Create a batch of all PriceShape elements that match an integer property value
    ///
    PriceShapeBatch(APIUtil *util, int32_t prop_idx, int32_t prop_value):
        DSSBatch(util, PriceShape::dss_cls_idx, prop_idx, prop_value)
    {
    }

    ///
    /// Create a batch of all PriceShape elements that match a regular expression
    ///
    PriceShapeBatch(APIUtil *util, const char* regexp):
        DSSBatch(util, PriceShape::dss_cls_idx, regexp)
    {
    }


    PriceShapeBatch& begin_edit()
    {
        Batch_BeginEdit(pointer, count[0]);
        return *this;
    }

    PriceShapeBatch& end_edit(int32_t num_edits=1)
    {
        Batch_EndEdit(pointer, count[0], num_edits);
        return *this;
    }


    /// Max number of points to expect in price shape vectors. This gets reset to the number of Price values found if less than specified.
    ///
    /// DSS property name: npts, DSS property index: 1
    ///
    BatchInt32ArrayProxy npts()
    {
        return BatchInt32ArrayProxy(*this, Properties::npts);
    }

    PriceShapeBatch& npts(int32_t value)
    {
        set_batch_val(Properties::npts, value);
        return *this;
    }

    template <typename T>
    PriceShapeBatch& npts(T &value)
    {
        set_batch_val_for_each<T>(Properties::npts, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    PriceShapeBatch& npts(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::npts, it_begin, it_end);
        return *this;
    }

    /// Time interval for fixed interval data, hrs. Default = 1. If Interval = 0 then time data (in hours) may be at irregular intervals and time value must be specified using either the Hour property or input files. Then values are interpolated when Interval=0, but not for fixed interval data.
    ///
    /// See also "sinterval" and "minterval".
    ///
    /// DSS property name: interval, DSS property index: 2
    ///
    BatchFloat64ArrayProxy interval()
    {
        return BatchFloat64ArrayProxy(*this, Properties::interval);
    }

    PriceShapeBatch& interval(double value)
    {
        set_batch_val<double>(Properties::interval, value);
        return *this;
    }

    template <typename T>
    PriceShapeBatch& interval(T &value)
    {
        set_batch_val_for_each<T>(Properties::interval, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    PriceShapeBatch& interval(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::interval, it_begin, it_end);
        return *this;
    }

    /// Array of Price values.  Units should be compatible with the object using the data. You can also use the syntax:
    /// Price = (file=filename)     !for text file one value per line
    /// Price = (dblfile=filename)  !for packed file of doubles
    /// Price = (sngfile=filename)  !for packed file of singles
    ///
    /// Note: this property will reset Npts if the  number of values in the files are fewer.
    ///
    /// DSS property name: price, DSS property index: 3
    ///
    std::vector<VectorXd> price()
    {
        return get_batch_valarray<VectorXd>(Properties::price);
    }

    PriceShapeBatch& price(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::price, value);
        return *this;
    }

    /// Array of hour values. Only necessary to define this property for variable interval data. If the data are fixed interval, do not use this property. You can also use the syntax:
    /// hour = (file=filename)     !for text file one value per line
    /// hour = (dblfile=filename)  !for packed file of doubles
    /// hour = (sngfile=filename)  !for packed file of singles
    ///
    /// DSS property name: hour, DSS property index: 4
    ///
    std::vector<VectorXd> hour()
    {
        return get_batch_valarray<VectorXd>(Properties::hour);
    }

    PriceShapeBatch& hour(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::hour, value);
        return *this;
    }

    /// Mean of the Price curve values.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently. Used for Monte Carlo load simulations.
    ///
    /// DSS property name: mean, DSS property index: 5
    ///
    BatchFloat64ArrayProxy mean()
    {
        return BatchFloat64ArrayProxy(*this, Properties::mean);
    }

    PriceShapeBatch& mean(double value)
    {
        set_batch_val<double>(Properties::mean, value);
        return *this;
    }

    template <typename T>
    PriceShapeBatch& mean(T &value)
    {
        set_batch_val_for_each<T>(Properties::mean, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    PriceShapeBatch& mean(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::mean, it_begin, it_end);
        return *this;
    }

    /// Standard deviation of the Prices.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently.Is overwritten if you subsequently read in a curve
    ///
    /// Used for Monte Carlo load simulations.
    ///
    /// DSS property name: stddev, DSS property index: 6
    ///
    BatchFloat64ArrayProxy stddev()
    {
        return BatchFloat64ArrayProxy(*this, Properties::stddev);
    }

    PriceShapeBatch& stddev(double value)
    {
        set_batch_val<double>(Properties::stddev, value);
        return *this;
    }

    template <typename T>
    PriceShapeBatch& stddev(T &value)
    {
        set_batch_val_for_each<T>(Properties::stddev, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    PriceShapeBatch& stddev(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::stddev, it_begin, it_end);
        return *this;
    }

    /// Switch input of  Price curve data to a csv file containing (hour, Price) points, or simply (Price) values for fixed time interval data, one per line. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: csvfile, DSS property index: 7
    ///
    strings csvfile()
    {
        return get_batch_val<strings>(Properties::csvfile);
    }

    PriceShapeBatch& csvfile(const string &value)
    {
        set_batch_val(Properties::csvfile, value.c_str());
        return *this;
    }

    PriceShapeBatch& csvfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::csvfile, value.begin(), value.end());
        return *this;
    }

    /// Switch input of  Price curve data to a binary file of singles containing (hour, Price) points, or simply (Price) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: sngfile, DSS property index: 8
    ///
    strings sngfile()
    {
        return get_batch_val<strings>(Properties::sngfile);
    }

    PriceShapeBatch& sngfile(const string &value)
    {
        set_batch_val(Properties::sngfile, value.c_str());
        return *this;
    }

    PriceShapeBatch& sngfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::sngfile, value.begin(), value.end());
        return *this;
    }

    /// Switch input of  Price curve data to a binary file of doubles containing (hour, Price) points, or simply (Price) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: dblfile, DSS property index: 9
    ///
    strings dblfile()
    {
        return get_batch_val<strings>(Properties::dblfile);
    }

    PriceShapeBatch& dblfile(const string &value)
    {
        set_batch_val(Properties::dblfile, value.c_str());
        return *this;
    }

    PriceShapeBatch& dblfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::dblfile, value.begin(), value.end());
        return *this;
    }

    /// Specify fixed interval in SECONDS. Alternate way to specify Interval property.
    ///
    /// DSS property name: sinterval, DSS property index: 10
    ///
    BatchFloat64ArrayProxy sinterval()
    {
        return BatchFloat64ArrayProxy(*this, Properties::sinterval);
    }

    PriceShapeBatch& sinterval(double value)
    {
        set_batch_val<double>(Properties::sinterval, value);
        return *this;
    }

    template <typename T>
    PriceShapeBatch& sinterval(T &value)
    {
        set_batch_val_for_each<T>(Properties::sinterval, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    PriceShapeBatch& sinterval(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::sinterval, it_begin, it_end);
        return *this;
    }

    /// Specify fixed interval in MINUTES. Alternate way to specify Interval property.
    ///
    /// DSS property name: minterval, DSS property index: 11
    ///
    BatchFloat64ArrayProxy minterval()
    {
        return BatchFloat64ArrayProxy(*this, Properties::minterval);
    }

    PriceShapeBatch& minterval(double value)
    {
        set_batch_val<double>(Properties::minterval, value);
        return *this;
    }

    template <typename T>
    PriceShapeBatch& minterval(T &value)
    {
        set_batch_val_for_each<T>(Properties::minterval, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    PriceShapeBatch& minterval(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::minterval, it_begin, it_end);
        return *this;
    }

    /// {DblSave | SngSave} After defining Price curve data... Setting action=DblSave or SngSave will cause the present "Price" values to be written to either a packed file of double or single. The filename is the PriceShape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    PriceShapeBatch& action(int32_t value)
    {
        set_batch_val(Properties::action, value);
        return *this;
    }

    /// {DblSave | SngSave} After defining Price curve data... Setting action=DblSave or SngSave will cause the present "Price" values to be written to either a packed file of double or single. The filename is the PriceShape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    PriceShapeBatch& action(PriceShape::PriceShapeAction value)
    {
        set_batch_val(Properties::action, int32_t(value));
        return *this;
    }

    /// {DblSave | SngSave} After defining Price curve data... Setting action=DblSave or SngSave will cause the present "Price" values to be written to either a packed file of double or single. The filename is the PriceShape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    PriceShapeBatch& action(const string &value)
    {
        set_batch_val(Properties::action, value.c_str());
        return *this;
    }

    /// {DblSave | SngSave} After defining Price curve data... Setting action=DblSave or SngSave will cause the present "Price" values to be written to either a packed file of double or single. The filename is the PriceShape name.
    ///
    /// DSS property name: action, DSS property index: 12
    ///
    PriceShapeBatch& action(const char *value)
    {
        set_batch_val(Properties::action, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 13
    ///
    PriceShapeBatch& like(const string &value)
    {
        set_batch_val(Properties::like, value.c_str());
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 13
    ///
    PriceShapeBatch& like(const char *value)
    {
        set_batch_val(Properties::like, value);
        return *this;
    }
};


class XYcurveBatch: public DSSBatch
{
public:
    typedef XYcurve::Properties Properties;
    typedef XYcurve BatchElementClass;

    ///
    /// Create a batch of all XYcurve elements
    ///
    XYcurveBatch(APIUtil *util):
        DSSBatch(util, XYcurve::dss_cls_idx)
    {
    }

    ///
    /// Create a batch of all XYcurve elements that match an integer property value
    ///
    XYcurveBatch(APIUtil *util, int32_t prop_idx, int32_t prop_value):
        DSSBatch(util, XYcurve::dss_cls_idx, prop_idx, prop_value)
    {
    }

    ///
    /// Create a batch of all XYcurve elements that match a regular expression
    ///
    XYcurveBatch(APIUtil *util, const char* regexp):
        DSSBatch(util, XYcurve::dss_cls_idx, regexp)
    {
    }


    XYcurveBatch& begin_edit()
    {
        Batch_BeginEdit(pointer, count[0]);
        return *this;
    }

    XYcurveBatch& end_edit(int32_t num_edits=1)
    {
        Batch_EndEdit(pointer, count[0], num_edits);
        return *this;
    }


    /// Max number of points to expect in curve. This could get reset to the actual number of points defined if less than specified.
    ///
    /// DSS property name: npts, DSS property index: 1
    ///
    BatchInt32ArrayProxy npts()
    {
        return BatchInt32ArrayProxy(*this, Properties::npts);
    }

    XYcurveBatch& npts(int32_t value)
    {
        set_batch_val(Properties::npts, value);
        return *this;
    }

    template <typename T>
    XYcurveBatch& npts(T &value)
    {
        set_batch_val_for_each<T>(Properties::npts, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    XYcurveBatch& npts(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::npts, it_begin, it_end);
        return *this;
    }

    /// One way to enter the points in a curve. Enter x and y values as one array in the order [x1, y1, x2, y2, ...]. For example:
    ///
    /// Points=[1,100 2,200 3, 300]
    ///
    /// Values separated by commas or white space. Zero fills arrays if insufficient number of values.
    ///
    /// DSS property name: Points, DSS property index: 2
    ///
    std::vector<VectorXd> Points()
    {
        return get_batch_valarray<VectorXd>(Properties::Points);
    }

    XYcurveBatch& Points(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::Points, value);
        return *this;
    }

    /// Alternate way to enter Y values. Enter an array of Y values corresponding to the X values.  You can also use the syntax:
    /// Yarray = (file=filename)     !for text file one value per line
    /// Yarray = (dblfile=filename)  !for packed file of doubles
    /// Yarray = (sngfile=filename)  !for packed file of singles
    ///
    /// Note: this property will reset Npts to a smaller value if the  number of values in the files are fewer.
    ///
    /// DSS property name: Yarray, DSS property index: 3
    ///
    std::vector<VectorXd> Yarray()
    {
        return get_batch_valarray<VectorXd>(Properties::Yarray);
    }

    XYcurveBatch& Yarray(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::Yarray, value);
        return *this;
    }

    /// Alternate way to enter X values. Enter an array of X values corresponding to the Y values.  You can also use the syntax:
    /// Xarray = (file=filename)     !for text file one value per line
    /// Xarray = (dblfile=filename)  !for packed file of doubles
    /// Xarray = (sngfile=filename)  !for packed file of singles
    ///
    /// Note: this property will reset Npts to a smaller value if the  number of values in the files are fewer.
    ///
    /// DSS property name: Xarray, DSS property index: 4
    ///
    std::vector<VectorXd> Xarray()
    {
        return get_batch_valarray<VectorXd>(Properties::Xarray);
    }

    XYcurveBatch& Xarray(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::Xarray, value);
        return *this;
    }

    /// Switch input of  X-Y curve data to a CSV file containing X, Y points one per line. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: csvfile, DSS property index: 5
    ///
    strings csvfile()
    {
        return get_batch_val<strings>(Properties::csvfile);
    }

    XYcurveBatch& csvfile(const string &value)
    {
        set_batch_val(Properties::csvfile, value.c_str());
        return *this;
    }

    XYcurveBatch& csvfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::csvfile, value.begin(), value.end());
        return *this;
    }

    /// Switch input of  X-Y curve data to a binary file of SINGLES containing X, Y points packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: sngfile, DSS property index: 6
    ///
    strings sngfile()
    {
        return get_batch_val<strings>(Properties::sngfile);
    }

    XYcurveBatch& sngfile(const string &value)
    {
        set_batch_val(Properties::sngfile, value.c_str());
        return *this;
    }

    XYcurveBatch& sngfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::sngfile, value.begin(), value.end());
        return *this;
    }

    /// Switch input of  X-Y  curve data to a binary file of DOUBLES containing X, Y points packed one after another. NOTE: This action may reset the number of points to a lower value.
    ///
    /// DSS property name: dblfile, DSS property index: 7
    ///
    strings dblfile()
    {
        return get_batch_val<strings>(Properties::dblfile);
    }

    XYcurveBatch& dblfile(const string &value)
    {
        set_batch_val(Properties::dblfile, value.c_str());
        return *this;
    }

    XYcurveBatch& dblfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::dblfile, value.begin(), value.end());
        return *this;
    }

    /// Enter a value and then retrieve the interpolated Y value from the Y property. On input shifted then scaled to original curve. Scaled then shifted on output.
    ///
    /// DSS property name: x, DSS property index: 8
    ///
    BatchFloat64ArrayProxy x()
    {
        return BatchFloat64ArrayProxy(*this, Properties::x);
    }

    XYcurveBatch& x(double value)
    {
        set_batch_val<double>(Properties::x, value);
        return *this;
    }

    template <typename T>
    XYcurveBatch& x(T &value)
    {
        set_batch_val_for_each<T>(Properties::x, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    XYcurveBatch& x(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::x, it_begin, it_end);
        return *this;
    }

    /// Enter a value and then retrieve the interpolated X value from the X property. On input shifted then scaled to original curve. Scaled then shifted on output.
    ///
    /// DSS property name: y, DSS property index: 9
    ///
    BatchFloat64ArrayProxy y()
    {
        return BatchFloat64ArrayProxy(*this, Properties::y);
    }

    XYcurveBatch& y(double value)
    {
        set_batch_val<double>(Properties::y, value);
        return *this;
    }

    template <typename T>
    XYcurveBatch& y(T &value)
    {
        set_batch_val_for_each<T>(Properties::y, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    XYcurveBatch& y(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::y, it_begin, it_end);
        return *this;
    }

    /// Shift X property values (in/out) by this amount of offset. Default = 0. Does not change original definition of arrays.
    ///
    /// DSS property name: Xshift, DSS property index: 10
    ///
    BatchFloat64ArrayProxy Xshift()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Xshift);
    }

    XYcurveBatch& Xshift(double value)
    {
        set_batch_val<double>(Properties::Xshift, value);
        return *this;
    }

    template <typename T>
    XYcurveBatch& Xshift(T &value)
    {
        set_batch_val_for_each<T>(Properties::Xshift, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    XYcurveBatch& Xshift(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Xshift, it_begin, it_end);
        return *this;
    }

    /// Shift Y property values (in/out) by this amount of offset. Default = 0. Does not change original definition of arrays.
    ///
    /// DSS property name: Yshift, DSS property index: 11
    ///
    BatchFloat64ArrayProxy Yshift()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Yshift);
    }

    XYcurveBatch& Yshift(double value)
    {
        set_batch_val<double>(Properties::Yshift, value);
        return *this;
    }

    template <typename T>
    XYcurveBatch& Yshift(T &value)
    {
        set_batch_val_for_each<T>(Properties::Yshift, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    XYcurveBatch& Yshift(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Yshift, it_begin, it_end);
        return *this;
    }

    /// Scale X property values (in/out) by this factor. Default = 1.0. Does not change original definition of arrays.
    ///
    /// DSS property name: Xscale, DSS property index: 12
    ///
    BatchFloat64ArrayProxy Xscale()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Xscale);
    }

    XYcurveBatch& Xscale(double value)
    {
        set_batch_val<double>(Properties::Xscale, value);
        return *this;
    }

    template <typename T>
    XYcurveBatch& Xscale(T &value)
    {
        set_batch_val_for_each<T>(Properties::Xscale, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    XYcurveBatch& Xscale(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Xscale, it_begin, it_end);
        return *this;
    }

    /// Scale Y property values (in/out) by this factor. Default = 1.0. Does not change original definition of arrays.
    ///
    /// DSS property name: Yscale, DSS property index: 13
    ///
    BatchFloat64ArrayProxy Yscale()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Yscale);
    }

    XYcurveBatch& Yscale(double value)
    {
        set_batch_val<double>(Properties::Yscale, value);
        return *this;
    }

    template <typename T>
    XYcurveBatch& Yscale(T &value)
    {
        set_batch_val_for_each<T>(Properties::Yscale, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    XYcurveBatch& Yscale(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Yscale, it_begin, it_end);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 14
    ///
    XYcurveBatch& like(const string &value)
    {
        set_batch_val(Properties::like, value.c_str());
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 14
    ///
    XYcurveBatch& like(const char *value)
    {
        set_batch_val(Properties::like, value);
        return *this;
    }
};


class GrowthShapeBatch: public DSSBatch
{
public:
    typedef GrowthShape::Properties Properties;
    typedef GrowthShape BatchElementClass;

    ///
    /// Create a batch of all GrowthShape elements
    ///
    GrowthShapeBatch(APIUtil *util):
        DSSBatch(util, GrowthShape::dss_cls_idx)
    {
    }

    ///
    /// Create a batch of all GrowthShape elements that match an integer property value
    ///
    GrowthShapeBatch(APIUtil *util, int32_t prop_idx, int32_t prop_value):
        DSSBatch(util, GrowthShape::dss_cls_idx, prop_idx, prop_value)
    {
    }

    ///
    /// Create a batch of all GrowthShape elements that match a regular expression
    ///
    GrowthShapeBatch(APIUtil *util, const char* regexp):
        DSSBatch(util, GrowthShape::dss_cls_idx, regexp)
    {
    }


    GrowthShapeBatch& begin_edit()
    {
        Batch_BeginEdit(pointer, count[0]);
        return *this;
    }

    GrowthShapeBatch& end_edit(int32_t num_edits=1)
    {
        Batch_EndEdit(pointer, count[0], num_edits);
        return *this;
    }


    /// Number of points to expect in subsequent vector.
    ///
    /// DSS property name: npts, DSS property index: 1
    ///
    BatchInt32ArrayProxy npts()
    {
        return BatchInt32ArrayProxy(*this, Properties::npts);
    }

    GrowthShapeBatch& npts(int32_t value)
    {
        set_batch_val(Properties::npts, value);
        return *this;
    }

    template <typename T>
    GrowthShapeBatch& npts(T &value)
    {
        set_batch_val_for_each<T>(Properties::npts, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    GrowthShapeBatch& npts(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::npts, it_begin, it_end);
        return *this;
    }

    /// Array of year values, or a text file spec, corresponding to the multipliers. Enter only those years where the growth changes. May be any integer sequence -- just so it is consistent. See help on Mult.
    ///
    /// DSS property name: year, DSS property index: 2
    ///
    std::vector<VectorXd> year()
    {
        return get_batch_valarray<VectorXd>(Properties::year);
    }

    GrowthShapeBatch& year(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::year, value);
        return *this;
    }

    /// Array of growth multiplier values, or a text file spec, corresponding to the year values. Enter the multiplier by which you would multiply the previous year's load to get the present year's.
    ///
    /// Examples:
    ///
    ///   Year = [1, 2, 5]   Mult=[1.05, 1.025, 1.02].
    ///   Year= (File=years.txt) Mult= (file=mults.txt).
    ///
    /// Text files contain one value per line.
    ///
    /// DSS property name: mult, DSS property index: 3
    ///
    std::vector<VectorXd> mult()
    {
        return get_batch_valarray<VectorXd>(Properties::mult);
    }

    GrowthShapeBatch& mult(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::mult, value);
        return *this;
    }

    /// Switch input of growth curve data to a csv file containing (year, mult) points, one per line.
    ///
    /// DSS property name: csvfile, DSS property index: 4
    ///
    strings csvfile()
    {
        return get_batch_val<strings>(Properties::csvfile);
    }

    GrowthShapeBatch& csvfile(const string &value)
    {
        set_batch_val(Properties::csvfile, value.c_str());
        return *this;
    }

    GrowthShapeBatch& csvfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::csvfile, value.begin(), value.end());
        return *this;
    }

    /// Switch input of growth curve data to a binary file of singles containing (year, mult) points, packed one after another.
    ///
    /// DSS property name: sngfile, DSS property index: 5
    ///
    strings sngfile()
    {
        return get_batch_val<strings>(Properties::sngfile);
    }

    GrowthShapeBatch& sngfile(const string &value)
    {
        set_batch_val(Properties::sngfile, value.c_str());
        return *this;
    }

    GrowthShapeBatch& sngfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::sngfile, value.begin(), value.end());
        return *this;
    }

    /// Switch input of growth curve data to a binary file of doubles containing (year, mult) points, packed one after another.
    ///
    /// DSS property name: dblfile, DSS property index: 6
    ///
    strings dblfile()
    {
        return get_batch_val<strings>(Properties::dblfile);
    }

    GrowthShapeBatch& dblfile(const string &value)
    {
        set_batch_val(Properties::dblfile, value.c_str());
        return *this;
    }

    GrowthShapeBatch& dblfile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::dblfile, value.begin(), value.end());
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 7
    ///
    GrowthShapeBatch& like(const string &value)
    {
        set_batch_val(Properties::like, value.c_str());
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 7
    ///
    GrowthShapeBatch& like(const char *value)
    {
        set_batch_val(Properties::like, value);
        return *this;
    }
};


class TCC_CurveBatch: public DSSBatch
{
public:
    typedef TCC_Curve::Properties Properties;
    typedef TCC_Curve BatchElementClass;

    ///
    /// Create a batch of all TCC_Curve elements
    ///
    TCC_CurveBatch(APIUtil *util):
        DSSBatch(util, TCC_Curve::dss_cls_idx)
    {
    }

    ///
    /// Create a batch of all TCC_Curve elements that match an integer property value
    ///
    TCC_CurveBatch(APIUtil *util, int32_t prop_idx, int32_t prop_value):
        DSSBatch(util, TCC_Curve::dss_cls_idx, prop_idx, prop_value)
    {
    }

    ///
    /// Create a batch of all TCC_Curve elements that match a regular expression
    ///
    TCC_CurveBatch(APIUtil *util, const char* regexp):
        DSSBatch(util, TCC_Curve::dss_cls_idx, regexp)
    {
    }


    TCC_CurveBatch& begin_edit()
    {
        Batch_BeginEdit(pointer, count[0]);
        return *this;
    }

    TCC_CurveBatch& end_edit(int32_t num_edits=1)
    {
        Batch_EndEdit(pointer, count[0], num_edits);
        return *this;
    }


    /// Number of points to expect in time-current arrays.
    ///
    /// DSS property name: npts, DSS property index: 1
    ///
    BatchInt32ArrayProxy npts()
    {
        return BatchInt32ArrayProxy(*this, Properties::npts);
    }

    TCC_CurveBatch& npts(int32_t value)
    {
        set_batch_val(Properties::npts, value);
        return *this;
    }

    template <typename T>
    TCC_CurveBatch& npts(T &value)
    {
        set_batch_val_for_each<T>(Properties::npts, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    TCC_CurveBatch& npts(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::npts, it_begin, it_end);
        return *this;
    }

    /// Array of current (or voltage) values corresponding to time values (see help on T_Array).
    ///
    /// DSS property name: C_array, DSS property index: 2
    ///
    std::vector<VectorXd> C_array()
    {
        return get_batch_valarray<VectorXd>(Properties::C_array);
    }

    TCC_CurveBatch& C_array(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::C_array, value);
        return *this;
    }

    /// Array of time values in sec. Typical array syntax:
    /// t_array = (1, 2, 3, 4, ...)
    ///
    /// Can also substitute a file designation:
    /// t_array =  (file=filename)
    ///
    /// The specified file has one value per line.
    ///
    /// DSS property name: T_array, DSS property index: 3
    ///
    std::vector<VectorXd> T_array()
    {
        return get_batch_valarray<VectorXd>(Properties::T_array);
    }

    TCC_CurveBatch& T_array(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::T_array, value);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 4
    ///
    TCC_CurveBatch& like(const string &value)
    {
        set_batch_val(Properties::like, value.c_str());
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 4
    ///
    TCC_CurveBatch& like(const char *value)
    {
        set_batch_val(Properties::like, value);
        return *this;
    }
};


class SpectrumBatch: public DSSBatch
{
public:
    typedef Spectrum::Properties Properties;
    typedef Spectrum BatchElementClass;

    ///
    /// Create a batch of all Spectrum elements
    ///
    SpectrumBatch(APIUtil *util):
        DSSBatch(util, Spectrum::dss_cls_idx)
    {
    }

    ///
    /// Create a batch of all Spectrum elements that match an integer property value
    ///
    SpectrumBatch(APIUtil *util, int32_t prop_idx, int32_t prop_value):
        DSSBatch(util, Spectrum::dss_cls_idx, prop_idx, prop_value)
    {
    }

    ///
    /// Create a batch of all Spectrum elements that match a regular expression
    ///
    SpectrumBatch(APIUtil *util, const char* regexp):
        DSSBatch(util, Spectrum::dss_cls_idx, regexp)
    {
    }


    SpectrumBatch& begin_edit()
    {
        Batch_BeginEdit(pointer, count[0]);
        return *this;
    }

    SpectrumBatch& end_edit(int32_t num_edits=1)
    {
        Batch_EndEdit(pointer, count[0], num_edits);
        return *this;
    }


    /// Number of frequencies in this spectrum. (See CSVFile)
    ///
    /// DSS property name: NumHarm, DSS property index: 1
    ///
    BatchInt32ArrayProxy NumHarm()
    {
        return BatchInt32ArrayProxy(*this, Properties::NumHarm);
    }

    SpectrumBatch& NumHarm(int32_t value)
    {
        set_batch_val(Properties::NumHarm, value);
        return *this;
    }

    template <typename T>
    SpectrumBatch& NumHarm(T &value)
    {
        set_batch_val_for_each<T>(Properties::NumHarm, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    SpectrumBatch& NumHarm(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::NumHarm, it_begin, it_end);
        return *this;
    }

    /// Array of harmonic values. You can also use the syntax
    /// harmonic = (file=filename)     !for text file one value per line
    /// harmonic = (dblfile=filename)  !for packed file of doubles
    /// harmonic = (sngfile=filename)  !for packed file of singles
    ///
    /// DSS property name: harmonic, DSS property index: 2
    ///
    std::vector<VectorXd> harmonic()
    {
        return get_batch_valarray<VectorXd>(Properties::harmonic);
    }

    SpectrumBatch& harmonic(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::harmonic, value);
        return *this;
    }

    /// Array of magnitude values, assumed to be in PERCENT. You can also use the syntax
    /// %mag = (file=filename)     !for text file one value per line
    /// %mag = (dblfile=filename)  !for packed file of doubles
    /// %mag = (sngfile=filename)  !for packed file of singles
    ///
    /// DSS property name: %mag, DSS property index: 3
    ///
    std::vector<VectorXd> pctmag()
    {
        return get_batch_valarray<VectorXd>(Properties::pctmag);
    }

    SpectrumBatch& pctmag(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::pctmag, value);
        return *this;
    }

    /// Array of phase angle values, degrees.You can also use the syntax
    /// angle = (file=filename)     !for text file one value per line
    /// angle = (dblfile=filename)  !for packed file of doubles
    /// angle = (sngfile=filename)  !for packed file of singles
    ///
    /// DSS property name: angle, DSS property index: 4
    ///
    std::vector<VectorXd> angle()
    {
        return get_batch_valarray<VectorXd>(Properties::angle);
    }

    SpectrumBatch& angle(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::angle, value);
        return *this;
    }

    /// File of spectrum points with (harmonic, magnitude-percent, angle-degrees) values, one set of 3 per line, in CSV format. If fewer than NUMHARM frequencies found in the file, NUMHARM is set to the smaller value.
    ///
    /// DSS property name: CSVFile, DSS property index: 5
    ///
    strings CSVFile()
    {
        return get_batch_val<strings>(Properties::CSVFile);
    }

    SpectrumBatch& CSVFile(const string &value)
    {
        set_batch_val(Properties::CSVFile, value.c_str());
        return *this;
    }

    SpectrumBatch& CSVFile(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::CSVFile, value.begin(), value.end());
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 6
    ///
    SpectrumBatch& like(const string &value)
    {
        set_batch_val(Properties::like, value.c_str());
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 6
    ///
    SpectrumBatch& like(const char *value)
    {
        set_batch_val(Properties::like, value);
        return *this;
    }
};


class WireDataBatch: public DSSBatch
{
public:
    typedef WireData::Properties Properties;
    typedef WireData BatchElementClass;

    ///
    /// Create a batch of all WireData elements
    ///
    WireDataBatch(APIUtil *util):
        DSSBatch(util, WireData::dss_cls_idx)
    {
    }

    ///
    /// Create a batch of all WireData elements that match an integer property value
    ///
    WireDataBatch(APIUtil *util, int32_t prop_idx, int32_t prop_value):
        DSSBatch(util, WireData::dss_cls_idx, prop_idx, prop_value)
    {
    }

    ///
    /// Create a batch of all WireData elements that match a regular expression
    ///
    WireDataBatch(APIUtil *util, const char* regexp):
        DSSBatch(util, WireData::dss_cls_idx, regexp)
    {
    }


    WireDataBatch& begin_edit()
    {
        Batch_BeginEdit(pointer, count[0]);
        return *this;
    }

    WireDataBatch& end_edit(int32_t num_edits=1)
    {
        Batch_EndEdit(pointer, count[0], num_edits);
        return *this;
    }


    /// dc Resistance, ohms per unit length (see Runits). Defaults to Rac/1.02 if not specified.
    ///
    /// DSS property name: Rdc, DSS property index: 1
    ///
    BatchFloat64ArrayProxy Rdc()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Rdc);
    }

    WireDataBatch& Rdc(double value)
    {
        set_batch_val<double>(Properties::Rdc, value);
        return *this;
    }

    template <typename T>
    WireDataBatch& Rdc(T &value)
    {
        set_batch_val_for_each<T>(Properties::Rdc, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    WireDataBatch& Rdc(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Rdc, it_begin, it_end);
        return *this;
    }

    /// Resistance at 60 Hz per unit length. Defaults to 1.02*Rdc if not specified.
    ///
    /// DSS property name: Rac, DSS property index: 2
    ///
    BatchFloat64ArrayProxy Rac()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Rac);
    }

    WireDataBatch& Rac(double value)
    {
        set_batch_val<double>(Properties::Rac, value);
        return *this;
    }

    template <typename T>
    WireDataBatch& Rac(T &value)
    {
        set_batch_val_for_each<T>(Properties::Rac, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    WireDataBatch& Rac(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Rac, it_begin, it_end);
        return *this;
    }

    /// Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: Runits, DSS property index: 3
    ///
    BatchInt32ArrayProxy Runits()
    {
        return BatchInt32ArrayProxy(*this, Properties::Runits);
    }

    WireDataBatch& Runits(string &value)
    {
        set_batch_val(Properties::Runits, value);
        return *this;
    }

    WireDataBatch& Runits(int32_t value)
    {
        set_batch_val(Properties::Runits, value);
        return *this;
    }

    WireDataBatch& Runits(DimensionUnits value)
    {
        set_batch_val(Properties::Runits, int32_t(value));
        return *this;
    }

    WireDataBatch& Runits(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::Runits, value.begin(), value.end());
        return *this;
    }

    WireDataBatch& Runits(std::vector<int32_t> &value)
    {
        set_batch_val_for_each<std::vector<int32_t>>(Properties::Runits, value.begin(), value.end());
        return *this;
    }

    WireDataBatch& Runits(std::vector<DimensionUnits> &value)
    {
        set_batch_val_for_each<std::vector<DimensionUnits>>(Properties::Runits, value.begin(), value.end());
        return *this;
    }

    /// Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: Runits, DSS property index: 3
    ///
    strings Runits_str()
    {
        return get_batch_val<strings>(Properties::Runits);
    }

    WireDataBatch& Runits_str(string &value)
    {
        Runits(value);
        return *this;
    }

    WireDataBatch& Runits_str(strings &value)
    {
        Runits(value);
        return *this;
    }

    /// GMR at 60 Hz. Defaults to .7788*radius if not specified.
    ///
    /// DSS property name: GMRac, DSS property index: 4
    ///
    BatchFloat64ArrayProxy GMRac()
    {
        return BatchFloat64ArrayProxy(*this, Properties::GMRac);
    }

    WireDataBatch& GMRac(double value)
    {
        set_batch_val<double>(Properties::GMRac, value);
        return *this;
    }

    template <typename T>
    WireDataBatch& GMRac(T &value)
    {
        set_batch_val_for_each<T>(Properties::GMRac, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    WireDataBatch& GMRac(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::GMRac, it_begin, it_end);
        return *this;
    }

    /// Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: GMRunits, DSS property index: 5
    ///
    BatchInt32ArrayProxy GMRunits()
    {
        return BatchInt32ArrayProxy(*this, Properties::GMRunits);
    }

    WireDataBatch& GMRunits(string &value)
    {
        set_batch_val(Properties::GMRunits, value);
        return *this;
    }

    WireDataBatch& GMRunits(int32_t value)
    {
        set_batch_val(Properties::GMRunits, value);
        return *this;
    }

    WireDataBatch& GMRunits(DimensionUnits value)
    {
        set_batch_val(Properties::GMRunits, int32_t(value));
        return *this;
    }

    WireDataBatch& GMRunits(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::GMRunits, value.begin(), value.end());
        return *this;
    }

    WireDataBatch& GMRunits(std::vector<int32_t> &value)
    {
        set_batch_val_for_each<std::vector<int32_t>>(Properties::GMRunits, value.begin(), value.end());
        return *this;
    }

    WireDataBatch& GMRunits(std::vector<DimensionUnits> &value)
    {
        set_batch_val_for_each<std::vector<DimensionUnits>>(Properties::GMRunits, value.begin(), value.end());
        return *this;
    }

    /// Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: GMRunits, DSS property index: 5
    ///
    strings GMRunits_str()
    {
        return get_batch_val<strings>(Properties::GMRunits);
    }

    WireDataBatch& GMRunits_str(string &value)
    {
        GMRunits(value);
        return *this;
    }

    WireDataBatch& GMRunits_str(strings &value)
    {
        GMRunits(value);
        return *this;
    }

    /// Outside radius of conductor. Defaults to GMR/0.7788 if not specified.
    ///
    /// DSS property name: radius, DSS property index: 6
    ///
    BatchFloat64ArrayProxy radius()
    {
        return BatchFloat64ArrayProxy(*this, Properties::radius);
    }

    WireDataBatch& radius(double value)
    {
        set_batch_val<double>(Properties::radius, value);
        return *this;
    }

    template <typename T>
    WireDataBatch& radius(T &value)
    {
        set_batch_val_for_each<T>(Properties::radius, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    WireDataBatch& radius(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::radius, it_begin, it_end);
        return *this;
    }

    /// Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: radunits, DSS property index: 7
    ///
    BatchInt32ArrayProxy radunits()
    {
        return BatchInt32ArrayProxy(*this, Properties::radunits);
    }

    WireDataBatch& radunits(string &value)
    {
        set_batch_val(Properties::radunits, value);
        return *this;
    }

    WireDataBatch& radunits(int32_t value)
    {
        set_batch_val(Properties::radunits, value);
        return *this;
    }

    WireDataBatch& radunits(DimensionUnits value)
    {
        set_batch_val(Properties::radunits, int32_t(value));
        return *this;
    }

    WireDataBatch& radunits(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::radunits, value.begin(), value.end());
        return *this;
    }

    WireDataBatch& radunits(std::vector<int32_t> &value)
    {
        set_batch_val_for_each<std::vector<int32_t>>(Properties::radunits, value.begin(), value.end());
        return *this;
    }

    WireDataBatch& radunits(std::vector<DimensionUnits> &value)
    {
        set_batch_val_for_each<std::vector<DimensionUnits>>(Properties::radunits, value.begin(), value.end());
        return *this;
    }

    /// Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: radunits, DSS property index: 7
    ///
    strings radunits_str()
    {
        return get_batch_val<strings>(Properties::radunits);
    }

    WireDataBatch& radunits_str(string &value)
    {
        radunits(value);
        return *this;
    }

    WireDataBatch& radunits_str(strings &value)
    {
        radunits(value);
        return *this;
    }

    /// Normal ampacity, amperes. Defaults to Emergency amps/1.5 if not specified.
    ///
    /// DSS property name: normamps, DSS property index: 8
    ///
    BatchFloat64ArrayProxy normamps()
    {
        return BatchFloat64ArrayProxy(*this, Properties::normamps);
    }

    WireDataBatch& normamps(double value)
    {
        set_batch_val<double>(Properties::normamps, value);
        return *this;
    }

    template <typename T>
    WireDataBatch& normamps(T &value)
    {
        set_batch_val_for_each<T>(Properties::normamps, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    WireDataBatch& normamps(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::normamps, it_begin, it_end);
        return *this;
    }

    /// Emergency ampacity, amperes. Defaults to 1.5 * Normal Amps if not specified.
    ///
    /// DSS property name: emergamps, DSS property index: 9
    ///
    BatchFloat64ArrayProxy emergamps()
    {
        return BatchFloat64ArrayProxy(*this, Properties::emergamps);
    }

    WireDataBatch& emergamps(double value)
    {
        set_batch_val<double>(Properties::emergamps, value);
        return *this;
    }

    template <typename T>
    WireDataBatch& emergamps(T &value)
    {
        set_batch_val_for_each<T>(Properties::emergamps, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    WireDataBatch& emergamps(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::emergamps, it_begin, it_end);
        return *this;
    }

    /// Diameter; Alternative method for entering radius.
    ///
    /// DSS property name: diam, DSS property index: 10
    ///
    BatchFloat64ArrayProxy diam()
    {
        return BatchFloat64ArrayProxy(*this, Properties::diam);
    }

    WireDataBatch& diam(double value)
    {
        set_batch_val<double>(Properties::diam, value);
        return *this;
    }

    template <typename T>
    WireDataBatch& diam(T &value)
    {
        set_batch_val_for_each<T>(Properties::diam, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    WireDataBatch& diam(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::diam, it_begin, it_end);
        return *this;
    }

    /// Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.
    ///
    /// DSS property name: Seasons, DSS property index: 11
    ///
    BatchInt32ArrayProxy Seasons()
    {
        return BatchInt32ArrayProxy(*this, Properties::Seasons);
    }

    WireDataBatch& Seasons(int32_t value)
    {
        set_batch_val(Properties::Seasons, value);
        return *this;
    }

    template <typename T>
    WireDataBatch& Seasons(T &value)
    {
        set_batch_val_for_each<T>(Properties::Seasons, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    WireDataBatch& Seasons(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Seasons, it_begin, it_end);
        return *this;
    }

    /// An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
    /// multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.
    ///
    /// DSS property name: Ratings, DSS property index: 12
    ///
    std::vector<VectorXd> Ratings()
    {
        return get_batch_valarray<VectorXd>(Properties::Ratings);
    }

    WireDataBatch& Ratings(VectorXd &value)
    {
        set_batch_val<VectorXd>(Properties::Ratings, value);
        return *this;
    }

    /// Equivalent conductor radius for capacitance calcs. Specify this for bundled conductors. Defaults to same value as radius. Define Diam or Radius property first.
    ///
    /// DSS property name: Capradius, DSS property index: 13
    ///
    BatchFloat64ArrayProxy Capradius()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Capradius);
    }

    WireDataBatch& Capradius(double value)
    {
        set_batch_val<double>(Properties::Capradius, value);
        return *this;
    }

    template <typename T>
    WireDataBatch& Capradius(T &value)
    {
        set_batch_val_for_each<T>(Properties::Capradius, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    WireDataBatch& Capradius(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Capradius, it_begin, it_end);
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 14
    ///
    WireDataBatch& like(const string &value)
    {
        set_batch_val(Properties::like, value.c_str());
        return *this;
    }

    /// Make like another object, e.g.:
    ///
    /// New Capacitor.C2 like=c1  ...
    ///
    /// DSS property name: like, DSS property index: 14
    ///
    WireDataBatch& like(const char *value)
    {
        set_batch_val(Properties::like, value);
        return *this;
    }
};


class CNDataBatch: public DSSBatch
{
public:
    typedef CNData::Properties Properties;
    typedef CNData BatchElementClass;

    ///
    /// Create a batch of all CNData elements
    ///
    CNDataBatch(APIUtil *util):
        DSSBatch(util, CNData::dss_cls_idx)
    {
    }

    ///
    /// Create a batch of all CNData elements that match an integer property value
    ///
    CNDataBatch(APIUtil *util, int32_t prop_idx, int32_t prop_value):
        DSSBatch(util, CNData::dss_cls_idx, prop_idx, prop_value)
    {
    }

    ///
    /// Create a batch of all CNData elements that match a regular expression
    ///
    CNDataBatch(APIUtil *util, const char* regexp):
        DSSBatch(util, CNData::dss_cls_idx, regexp)
    {
    }


    CNDataBatch& begin_edit()
    {
        Batch_BeginEdit(pointer, count[0]);
        return *this;
    }

    CNDataBatch& end_edit(int32_t num_edits=1)
    {
        Batch_EndEdit(pointer, count[0], num_edits);
        return *this;
    }


    /// Number of concentric neutral strands; default is 2
    ///
    /// DSS property name: k, DSS property index: 1
    ///
    BatchInt32ArrayProxy k()
    {
        return BatchInt32ArrayProxy(*this, Properties::k);
    }

    CNDataBatch& k(int32_t value)
    {
        set_batch_val(Properties::k, value);
        return *this;
    }

    template <typename T>
    CNDataBatch& k(T &value)
    {
        set_batch_val_for_each<T>(Properties::k, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    CNDataBatch& k(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::k, it_begin, it_end);
        return *this;
    }

    /// Diameter of a concentric neutral strand; same units as core conductor radius; no default.
    ///
    /// DSS property name: DiaStrand, DSS property index: 2
    ///
    BatchFloat64ArrayProxy DiaStrand()
    {
        return BatchFloat64ArrayProxy(*this, Properties::DiaStrand);
    }

    CNDataBatch& DiaStrand(double value)
    {
        set_batch_val<double>(Properties::DiaStrand, value);
        return *this;
    }

    template <typename T>
    CNDataBatch& DiaStrand(T &value)
    {
        set_batch_val_for_each<T>(Properties::DiaStrand, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    CNDataBatch& DiaStrand(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::DiaStrand, it_begin, it_end);
        return *this;
    }

    /// Geometric mean radius of a concentric neutral strand; same units as core conductor GMR; defaults to 0.7788 * CN strand radius.
    ///
    /// DSS property name: GmrStrand, DSS property index: 3
    ///
    BatchFloat64ArrayProxy GmrStrand()
    {
        return BatchFloat64ArrayProxy(*this, Properties::GmrStrand);
    }

    CNDataBatch& GmrStrand(double value)
    {
        set_batch_val<double>(Properties::GmrStrand, value);
        return *this;
    }

    template <typename T>
    CNDataBatch& GmrStrand(T &value)
    {
        set_batch_val_for_each<T>(Properties::GmrStrand, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    CNDataBatch& GmrStrand(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::GmrStrand, it_begin, it_end);
        return *this;
    }

    /// AC resistance of a concentric neutral strand; same units as core conductor resistance; no default.
    ///
    /// DSS property name: Rstrand, DSS property index: 4
    ///
    BatchFloat64ArrayProxy Rstrand()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Rstrand);
    }

    CNDataBatch& Rstrand(double value)
    {
        set_batch_val<double>(Properties::Rstrand, value);
        return *this;
    }

    template <typename T>
    CNDataBatch& Rstrand(T &value)
    {
        set_batch_val_for_each<T>(Properties::Rstrand, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    CNDataBatch& Rstrand(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Rstrand, it_begin, it_end);
        return *this;
    }

    /// Insulation layer relative permittivity; default is 2.3.
    ///
    /// DSS property name: EpsR, DSS property index: 5
    ///
    BatchFloat64ArrayProxy EpsR()
    {
        return BatchFloat64ArrayProxy(*this, Properties::EpsR);
    }

    CNDataBatch& EpsR(double value)
    {
        set_batch_val<double>(Properties::EpsR, value);
        return *this;
    }

    template <typename T>
    CNDataBatch& EpsR(T &value)
    {
        set_batch_val_for_each<T>(Properties::EpsR, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    CNDataBatch& EpsR(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::EpsR, it_begin, it_end);
        return *this;
    }

    /// Insulation layer thickness; same units as radius; no default. With DiaIns, establishes inner radius for capacitance calculation.
    ///
    /// DSS property name: InsLayer, DSS property index: 6
    ///
    BatchFloat64ArrayProxy InsLayer()
    {
        return BatchFloat64ArrayProxy(*this, Properties::InsLayer);
    }

    CNDataBatch& InsLayer(double value)
    {
        set_batch_val<double>(Properties::InsLayer, value);
        return *this;
    }

    template <typename T>
    CNDataBatch& InsLayer(T &value)
    {
        set_batch_val_for_each<T>(Properties::InsLayer, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    CNDataBatch& InsLayer(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::InsLayer, it_begin, it_end);
        return *this;
    }

    /// Diameter over insulation layer; same units as radius; no default. Establishes outer radius for capacitance calculation.
    ///
    /// DSS property name: DiaIns, DSS property index: 7
    ///
    BatchFloat64ArrayProxy DiaIns()
    {
        return BatchFloat64ArrayProxy(*this, Properties::DiaIns);
    }

    CNDataBatch& DiaIns(double value)
    {
        set_batch_val<double>(Properties::DiaIns, value);
        return *this;
    }

    template <typename T>
    CNDataBatch& DiaIns(T &value)
    {
        set_batch_val_for_each<T>(Properties::DiaIns, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    CNDataBatch& DiaIns(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::DiaIns, it_begin, it_end);
        return *this;
    }

    /// Diameter over cable; same units as radius; no default.
    ///
    /// DSS property name: DiaCable, DSS property index: 8
    ///
    BatchFloat64ArrayProxy DiaCable()
    {
        return BatchFloat64ArrayProxy(*this, Properties::DiaCable);
    }

    CNDataBatch& DiaCable(double value)
    {
        set_batch_val<double>(Properties::DiaCable, value);
        return *this;
    }

    template <typename T>
    CNDataBatch& DiaCable(T &value)
    {
        set_batch_val_for_each<T>(Properties::DiaCable, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    CNDataBatch& DiaCable(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::DiaCable, it_begin, it_end);
        return *this;
    }

    /// dc Resistance, ohms per unit length (see Runits). Defaults to Rac/1.02 if not specified.
    ///
    /// DSS property name: Rdc, DSS property index: 9
    ///
    BatchFloat64ArrayProxy Rdc()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Rdc);
    }

    CNDataBatch& Rdc(double value)
    {
        set_batch_val<double>(Properties::Rdc, value);
        return *this;
    }

    template <typename T>
    CNDataBatch& Rdc(T &value)
    {
        set_batch_val_for_each<T>(Properties::Rdc, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    CNDataBatch& Rdc(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Rdc, it_begin, it_end);
        return *this;
    }

    /// Resistance at 60 Hz per unit length. Defaults to 1.02*Rdc if not specified.
    ///
    /// DSS property name: Rac, DSS property index: 10
    ///
    BatchFloat64ArrayProxy Rac()
    {
        return BatchFloat64ArrayProxy(*this, Properties::Rac);
    }

    CNDataBatch& Rac(double value)
    {
        set_batch_val<double>(Properties::Rac, value);
        return *this;
    }

    template <typename T>
    CNDataBatch& Rac(T &value)
    {
        set_batch_val_for_each<T>(Properties::Rac, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    CNDataBatch& Rac(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::Rac, it_begin, it_end);
        return *this;
    }

    /// Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: Runits, DSS property index: 11
    ///
    BatchInt32ArrayProxy Runits()
    {
        return BatchInt32ArrayProxy(*this, Properties::Runits);
    }

    CNDataBatch& Runits(string &value)
    {
        set_batch_val(Properties::Runits, value);
        return *this;
    }

    CNDataBatch& Runits(int32_t value)
    {
        set_batch_val(Properties::Runits, value);
        return *this;
    }

    CNDataBatch& Runits(DimensionUnits value)
    {
        set_batch_val(Properties::Runits, int32_t(value));
        return *this;
    }

    CNDataBatch& Runits(strings &value)
    {
        set_batch_val_for_each<strings>(Properties::Runits, value.begin(), value.end());
        return *this;
    }

    CNDataBatch& Runits(std::vector<int32_t> &value)
    {
        set_batch_val_for_each<std::vector<int32_t>>(Properties::Runits, value.begin(), value.end());
        return *this;
    }

    CNDataBatch& Runits(std::vector<DimensionUnits> &value)
    {
        set_batch_val_for_each<std::vector<DimensionUnits>>(Properties::Runits, value.begin(), value.end());
        return *this;
    }

    /// Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: Runits, DSS property index: 11
    ///
    strings Runits_str()
    {
        return get_batch_val<strings>(Properties::Runits);
    }

    CNDataBatch& Runits_str(string &value)
    {
        Runits(value);
        return *this;
    }

    CNDataBatch& Runits_str(strings &value)
    {
        Runits(value);
        return *this;
    }

    /// GMR at 60 Hz. Defaults to .7788*radius if not specified.
    ///
    /// DSS property name: GMRac, DSS property index: 12
    ///
    BatchFloat64ArrayProxy GMRac()
    {
        return BatchFloat64ArrayProxy(*this, Properties::GMRac);
    }

    CNDataBatch& GMRac(double value)
    {
        set_batch_val<double>(Properties::GMRac, value);
        return *this;
    }

    template <typename T>
    CNDataBatch& GMRac(T &value)
    {
        set_batch_val_for_each<T>(Properties::GMRac, value.begin(), value.end());
        return *this;
    }

    template <typename T>
    CNDataBatch& GMRac(typename T::iterator it_begin, typename T::iterator it_end)
    {
        set_batch_val_for_each<T>(Properties::GMRac, it_begin, it_end);
        return *this;
    }

    /// Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.
    ///
    /// DSS property name: GMRunits, DSS property index: 13
    ///
    BatchInt32ArrayProxy GMRunits()
    {
        return BatchInt32ArrayProxy(*this, Properties::GMRunits);
    }

    CNDataBatch& GMRunits(string &value)
    {
        set_batch_val(Properties::GMRunits, value);
        return *this;
    }

    CNDataBatch& GMRunits(int32_t value)
    {
        set_batch_val(Properties::GMRunits, value);
        return *this;
    }

    CNDataBatch& GMRun