'''
This module exposes base objects for the pythonic Obj and Batch interfaces from DSS C-API.
These interfaces are unique to DSS Extensions, they are not present in the official OpenDSS.

This is still under development and the final implementation might differ in some aspects.
For example, we plan to adjust the spelling of many properties; `nconds` will become `NConds`, 
`kv` will become `kV`, and so on.

Currently, we're already removed `wdg` and the indirect properties in favour of the array versions.
This will also be tweaked after more usage. `NConds`, for example, may also be removed if we can
develop a mechanism to ensure the dimensions match.

Copyright (c) 2021-2023 Paulo Meira
Copyright (c) 2021-2023 DSS Extensions contributors
'''
from typing import Union, List, AnyStr, Optional
from typing_extensions import TypedDict, Unpack
import numpy as np
from ._obj_bases import (
    BatchFloat64ArrayProxy,
    BatchInt32ArrayProxy,
    DSSObj,
    DSSBatch,
    IDSSObj,
    LIST_LIKE,
    # NotSet,
)

try:
    from typing import ForwardRef
except:
    from typing import _ForwardRef  as ForwardRef

try:
    from enum import IntEnum
except:
    from aenum import IntEnum

from ._types import Float64Array, Int32Array
from ._cffi_api_util import Base

PDElement = Union[
    ForwardRef('AutoTrans'),
    ForwardRef('Capacitor'),
    ForwardRef('Line'),
    ForwardRef('Reactor'),
    ForwardRef('Transformer'),
]

# Global enumerations
class EarthModel(IntEnum):
    """Earth Model (DSS enumeration)"""
    Carson = 1 # Carson
    FullCarson = 2 # FullCarson
    Deri = 3 # Deri

class LineType(IntEnum):
    """Line Type (DSS enumeration)"""
    oh = 1 # oh
    ug = 2 # ug
    ug_ts = 3 # ug_ts
    ug_cn = 4 # ug_cn
    swt_ldbrk = 5 # swt_ldbrk
    swt_fuse = 6 # swt_fuse
    swt_sect = 7 # swt_sect
    swt_rec = 8 # swt_rec
    swt_disc = 9 # swt_disc
    swt_brk = 10 # swt_brk
    swt_elbow = 11 # swt_elbow

class DimensionUnits(IntEnum):
    """Dimension Units (DSS enumeration)"""
    none = 0 # none
    mi = 1 # mi
    kft = 2 # kft
    km = 3 # km
    m = 4 # m
    ft = 5 # ft
    inch = 6 # in
    cm = 7 # cm
    mm = 8 # mm
    meter = 4 # meter
    miles = 1 # miles

class ScanType(IntEnum):
    """Scan Type (DSS enumeration)"""
    none = -1 # None
    Zero = 0 # Zero
    Positive = 1 # Positive

class SequenceType(IntEnum):
    """Sequence Type (DSS enumeration)"""
    Negative = -1 # Negative
    Zero = 0 # Zero
    Positive = 1 # Positive

class Connection(IntEnum):
    """Connection (DSS enumeration)"""
    wye = 0 # wye
    delta = 1 # delta
    y = 0 # y
    ln = 0 # ln
    ll = 1 # ll

class CoreType(IntEnum):
    """Core Type (DSS enumeration)"""
    shell = 0 # shell
    one_phase = 1 # 1-phase
    three_leg = 3 # 3-leg
    four_leg = 4 # 4-leg
    five_leg = 5 # 5-leg
    core_1_phase = 9 # core-1-phase

class PhaseSequence(IntEnum):
    """Phase Sequence (DSS enumeration)"""
    Lag = 0 # Lag
    Lead = 1 # Lead
    ANSI = 0 # ANSI
    Euro = 1 # Euro

class LoadSolutionModel(IntEnum):
    """Load Solution Model (DSS enumeration)"""
    PowerFlow = 1 # PowerFlow
    Admittance = 2 # Admittance

class RandomType(IntEnum):
    """Random Type (DSS enumeration)"""
    none = 0 # None
    Gaussian = 1 # Gaussian
    Uniform = 2 # Uniform
    LogNormal = 3 # LogNormal

class ControlMode(IntEnum):
    """Control Mode (DSS enumeration)"""
    Off = -1 # Off
    Static = 0 # Static
    Event = 1 # Event
    Time = 2 # Time
    MultiRate = 3 # MultiRate

class InverterControlMode(IntEnum):
    """Inverter Control Mode (DSS enumeration)"""
    GFL = 0 # GFL
    GFM = 1 # GFM

class SolutionMode(IntEnum):
    """Solution Mode (DSS enumeration)"""
    Snap = 0 # Snap
    Daily = 1 # Daily
    Yearly = 2 # Yearly
    M1 = 3 # M1
    LD1 = 4 # LD1
    PeakDay = 5 # PeakDay
    DutyCycle = 6 # DutyCycle
    Direct = 7 # Direct
    MF = 8 # MF
    FaultStudy = 9 # FaultStudy
    M2 = 10 # M2
    M3 = 11 # M3
    LD2 = 12 # LD2
    AutoAdd = 13 # AutoAdd
    Dynamic = 14 # Dynamic
    Harmonic = 15 # Harmonic
    Time = 16 # Time
    HarmonicT = 17 # HarmonicT
    Snapshot = 0 # Snapshot
    Dynamics = 14 # Dynamics
    Harmonics = 15 # Harmonics
    S = 0 # S
    Y = 2 # Y
    H = 15 # H
    T = 16 # T
    F = 9 # F

class SolutionAlgorithm(IntEnum):
    """Solution Algorithm (DSS enumeration)"""
    Normal = 0 # Normal
    Newton = 1 # Newton

class CircuitModel(IntEnum):
    """Circuit Model (DSS enumeration)"""
    Multiphase = 0 # Multiphase
    Positive = 1 # Positive

class AutoAddDeviceType(IntEnum):
    """AutoAdd Device Type (DSS enumeration)"""
    Generator = 1 # Generator
    Capacitor = 2 # Capacitor

class LoadShapeClass(IntEnum):
    """Load Shape Class (DSS enumeration)"""
    none = -1 # None
    Daily = 0 # Daily
    Yearly = 1 # Yearly
    Duty = 2 # Duty

class MonitoredPhase(IntEnum):
    """Monitored Phase (DSS enumeration)"""
    min = -3 # min
    max = -2 # max
    avg = -1 # avg


class LineCode(DSSObj):
    __slots__ = []
    _cls_name = 'LineCode'
    _cls_idx = 1
    _cls_prop_idx = {
        'nphases': 1,
        'r1': 2,
        'x1': 3,
        'r0': 4,
        'x0': 5,
        'c1': 6,
        'c0': 7,
        'units': 8,
        'rmatrix': 9,
        'xmatrix': 10,
        'cmatrix': 11,
        'basefreq': 12,
        'normamps': 13,
        'emergamps': 14,
        'faultrate': 15,
        'pctperm': 16,
        'repair': 17,
        'kron': 18,
        'rg': 19,
        'xg': 20,
        'rho': 21,
        'neutral': 22,
        'b1': 23,
        'b0': 24,
        'seasons': 25,
        'ratings': 26,
        'linetype': 27,
        'like': 28,
    }

    @property
    def nphases(self) -> int:
        """
        Number of phases in the line this line code data represents.  Setting this property reinitializes the line code.  Impedance matrix is reset for default symmetrical component.

        DSS property name: `nphases`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @nphases.setter
    def nphases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def r1(self) -> float:
        """
        Positive-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Rmatrix.

        DSS property name: `r1`, DSS property index: 2.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 2)

    @r1.setter
    def r1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 2, value)

    @property
    def x1(self) -> float:
        """
        Positive-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Xmatrix

        DSS property name: `x1`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @x1.setter
    def x1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def r0(self) -> float:
        """
        Zero-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.

        DSS property name: `r0`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @r0.setter
    def r0(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def x0(self) -> float:
        """
        Zero-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.

        DSS property name: `x0`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @x0.setter
    def x0(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def C1(self) -> float:
        """
        Positive-sequence capacitance, nf per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Cmatrix and B1.

        DSS property name: `C1`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @C1.setter
    def C1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def C0(self) -> float:
        """
        Zero-sequence capacitance, nf per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also B0.

        DSS property name: `C0`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @C0.setter
    def C0(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def units(self) -> DimensionUnits:
        """
        One of (ohms per ...) {none|mi|km|kft|m|me|ft|in|cm}.  Default is none; assumes units agree with length unitsgiven in Line object

        DSS property name: `units`, DSS property index: 8.
        """
        return DimensionUnits(self._lib.Obj_GetInt32(self._ptr, 8))

    @units.setter
    def units(self, value: Union[AnyStr, int, DimensionUnits]):
        if not isinstance(value, int):
            self._set_string_o(8, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 8, value)

    @property
    def units_str(self) -> str:
        """
        One of (ohms per ...) {none|mi|km|kft|m|me|ft|in|cm}.  Default is none; assumes units agree with length unitsgiven in Line object

        DSS property name: `units`, DSS property index: 8.
        """
        return self._get_prop_string(8)

    @units_str.setter
    def units_str(self, value: AnyStr):
        self.units = value

    @property
    def rmatrix(self) -> Float64Array:
        """
        Resistance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration.  For balanced line models, you may use the standard symmetrical component data definition instead.

        DSS property name: `rmatrix`, DSS property index: 9.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 9)

    @rmatrix.setter
    def rmatrix(self, value: Float64Array):
        self._set_float64_array_o(9, value)

    @property
    def xmatrix(self) -> Float64Array:
        """
        Reactance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration.  For balanced line models, you may use the standard symmetrical component data definition instead.

        DSS property name: `xmatrix`, DSS property index: 10.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 10)

    @xmatrix.setter
    def xmatrix(self, value: Float64Array):
        self._set_float64_array_o(10, value)

    @property
    def cmatrix(self) -> Float64Array:
        """
        Nodal Capacitance matrix, lower triangle, nf per unit length.Order of the matrix is the number of phases. May be used to specify the shunt capacitance of any line configuration.  For balanced line models, you may use the standard symmetrical component data definition instead.

        DSS property name: `cmatrix`, DSS property index: 11.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 11)

    @cmatrix.setter
    def cmatrix(self, value: Float64Array):
        self._set_float64_array_o(11, value)

    @property
    def baseFreq(self) -> float:
        """
        Frequency at which impedances are specified.

        DSS property name: `baseFreq`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @baseFreq.setter
    def baseFreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def normamps(self) -> float:
        """
        Normal ampere limit on line.  This is the so-called Planning Limit. It may also be the value above which load will have to be dropped in a contingency.  Usually about 75% - 80% of the emergency (one-hour) rating.

        DSS property name: `normamps`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @normamps.setter
    def normamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def emergamps(self) -> float:
        """
        Emergency ampere limit on line (usually one-hour rating).

        DSS property name: `emergamps`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @emergamps.setter
    def emergamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def faultrate(self) -> float:
        """
        Number of faults per unit length per year.

        DSS property name: `faultrate`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @faultrate.setter
    def faultrate(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def pctperm(self) -> float:
        """
        Percentage of the faults that become permanent.

        DSS property name: `pctperm`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @pctperm.setter
    def pctperm(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def repair(self) -> float:
        """
        Hours to repair.

        DSS property name: `repair`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @repair.setter
    def repair(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    def Kron(self, value: bool):
        """
        Kron = Y/N. Default=N.  Perform Kron reduction on the impedance matrix after it is formed, reducing order by 1. Eliminates the conductor designated by the "Neutral=" property. Do this after the R, X, and C matrices are defined. Ignored for symmetrical components. May be issued more than once to eliminate more than one conductor by resetting the Neutral property after the previous invoking of this property. Generally, you do not want to do a Kron reduction on the matrix if you intend to solve at a frequency other than the base frequency and exploit the Rg and Xg values.

        DSS property name: `Kron`, DSS property index: 18.
        """
        self._lib.Obj_SetInt32(self._ptr, 18, value)

    @property
    def Rg(self) -> float:
        """
        Carson earth return resistance per unit length used to compute impedance values at base frequency.  For making better frequency adjustments. Default is 0.01805 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.

        DSS property name: `Rg`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @Rg.setter
    def Rg(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def Xg(self) -> float:
        """
        Carson earth return reactance per unit length used to compute impedance values at base frequency.  For making better frequency adjustments. Default value is 0.155081 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.

        DSS property name: `Xg`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @Xg.setter
    def Xg(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    @property
    def rho(self) -> float:
        """
        Default=100 meter ohms.  Earth resitivity used to compute earth correction factor.

        DSS property name: `rho`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @rho.setter
    def rho(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def neutral(self) -> int:
        """
        Designates which conductor is the "neutral" conductor that will be eliminated by Kron reduction. Default is the last conductor (nphases value). After Kron reduction is set to 0. Subsequent issuing of Kron=Yes will not do anything until this property is set to a legal value. Applies only to LineCodes defined by R, X, and C matrix.

        DSS property name: `neutral`, DSS property index: 22.
        """
        return self._lib.Obj_GetInt32(self._ptr, 22)

    @neutral.setter
    def neutral(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 22, value)

    @property
    def B1(self) -> float:
        """
        Alternate way to specify C1. MicroS per unit length

        DSS property name: `B1`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @B1.setter
    def B1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def B0(self) -> float:
        """
        Alternate way to specify C0. MicroS per unit length

        DSS property name: `B0`, DSS property index: 24.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 24)

    @B0.setter
    def B0(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 24, value)

    @property
    def Seasons(self) -> int:
        """
        Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 25.
        """
        return self._lib.Obj_GetInt32(self._ptr, 25)

    @Seasons.setter
    def Seasons(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 25, value)

    @property
    def Ratings(self) -> Float64Array:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.

        DSS property name: `Ratings`, DSS property index: 26.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 26)

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(26, value)

    @property
    def linetype(self) -> LineType:
        """
        Code designating the type of line. 
        One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW

        OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.

        DSS property name: `LineType`, DSS property index: 27.
        """
        return LineType(self._lib.Obj_GetInt32(self._ptr, 27))

    @linetype.setter
    def linetype(self, value: Union[AnyStr, int, LineType]):
        if not isinstance(value, int):
            self._set_string_o(27, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 27, value)

    @property
    def linetype_str(self) -> str:
        """
        Code designating the type of line. 
        One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW

        OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.

        DSS property name: `LineType`, DSS property index: 27.
        """
        return self._get_prop_string(27)

    @linetype_str.setter
    def linetype_str(self, value: AnyStr):
        self.linetype = value

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 28.
        """
        self._set_string_o(28, value)

class LoadShape(DSSObj):
    __slots__ = []
    _cls_name = 'LoadShape'
    _cls_idx = 2
    _cls_prop_idx = {
        'npts': 1,
        'interval': 2,
        'mult': 3,
        'hour': 4,
        'mean': 5,
        'stddev': 6,
        'csvfile': 7,
        'sngfile': 8,
        'dblfile': 9,
        'action': 10,
        'qmult': 11,
        'useactual': 12,
        'pmax': 13,
        'qmax': 14,
        'sinterval': 15,
        'minterval': 16,
        'pbase': 17,
        'qbase': 18,
        'pmult': 19,
        'pqcsvfile': 20,
        'memorymapping': 21,
        'like': 22,
    }

    # Class-specific enumerations
    class LoadShapeAction(IntEnum):
        """LoadShape: Action (DSS enumeration for LoadShape)"""
        Normalize = 0 # Normalize
        DblSave = 1 # DblSave
        SngSave = 2 # SngSave


    @property
    def npts(self) -> int:
        """
        Max number of points to expect in load shape vectors. This gets reset to the number of multiplier values found (in files only) if less than specified.

        DSS property name: `npts`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @npts.setter
    def npts(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def interval(self) -> float:
        """
        Time interval for fixed interval data, hrs. Default = 1. If Interval = 0 then time data (in hours) may be at either regular or  irregular intervals and time value must be specified using either the Hour property or input files. Then values are interpolated when Interval=0, but not for fixed interval data.  

        See also "sinterval" and "minterval".

        DSS property name: `interval`, DSS property index: 2.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 2)

    @interval.setter
    def interval(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 2, value)

    @property
    def hour(self) -> Float64Array:
        """
        Array of hour values. Only necessary to define for variable interval data (Interval=0). If you set Interval>0 to denote fixed interval data, DO NOT USE THIS PROPERTY. You can also use the syntax: 
        hour = (file=filename)     !for text file one value per line
        hour = (dblfile=filename)  !for packed file of doubles
        hour = (sngfile=filename)  !for packed file of singles 

        DSS property name: `hour`, DSS property index: 4.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 4)

    @hour.setter
    def hour(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def mean(self) -> float:
        """
        Mean of the active power multipliers.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently. Used for Monte Carlo load simulations.

        DSS property name: `mean`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @mean.setter
    def mean(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def stddev(self) -> float:
        """
        Standard deviation of active power multipliers.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently.Is overwritten if you subsequently read in a curve

        Used for Monte Carlo load simulations.

        DSS property name: `stddev`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @stddev.setter
    def stddev(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def csvfile(self) -> str:
        """
        Switch input of active power load curve data to a CSV text file containing (hour, mult) points, or simply (mult) values for fixed time interval data, one per line. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `csvfile`, DSS property index: 7.
        """
        return self._get_prop_string(7)

    @csvfile.setter
    def csvfile(self, value: AnyStr):
        self._set_string_o(7, value)

    @property
    def sngfile(self) -> str:
        """
        Switch input of active power load curve data to a binary file of singles containing (hour, mult) points, or simply (mult) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `sngfile`, DSS property index: 8.
        """
        return self._get_prop_string(8)

    @sngfile.setter
    def sngfile(self, value: AnyStr):
        self._set_string_o(8, value)

    @property
    def dblfile(self) -> str:
        """
        Switch input of active power load curve data to a binary file of doubles containing (hour, mult) points, or simply (mult) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `dblfile`, DSS property index: 9.
        """
        return self._get_prop_string(9)

    @dblfile.setter
    def dblfile(self, value: AnyStr):
        self._set_string_o(9, value)

    def action(self, value: Union[str, bytes, int, LoadShapeAction]):
        """
        {NORMALIZE | DblSave | SngSave} After defining load curve data, setting action=normalize will modify the multipliers so that the peak is 1.0. The mean and std deviation are recomputed.

        Setting action=DblSave or SngSave will cause the present mult and qmult values to be written to either a packed file of double or single. The filename is the loadshape name. The mult array will have a "_P" appended on the file name and the qmult array, if it exists, will have "_Q" appended.

        DSS property name: `action`, DSS property index: 10.
        """
        if isinstance(value, int):
            self._lib.Obj_SetInt32(self._ptr, 10, value)
            return
    
        self._set_string_o(10, value)

    @property
    def qmult(self) -> Float64Array:
        """
        Array of multiplier values for reactive power (Q).  You can also use the syntax: 
        qmult = (file=filename)     !for text file one value per line
        qmult = (dblfile=filename)  !for packed file of doubles
        qmult = (sngfile=filename)  !for packed file of singles 
        qmult = (file=MyCSVFile.csv, col=4, header=yes)  !for multicolumn CSV files 

        DSS property name: `qmult`, DSS property index: 11.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 11)

    @qmult.setter
    def qmult(self, value: Float64Array):
        self._set_float64_array_o(11, value)

    @property
    def UseActual(self) -> bool:
        """
        {Yes | No* | True | False*} If true, signifies to Load, Generator, Vsource, or other objects to use the return value as the actual kW, kvar, kV, or other value rather than a multiplier. Nominally for AMI Load data but may be used for other functions.

        DSS property name: `UseActual`, DSS property index: 12.
        """
        return self._lib.Obj_GetInt32(self._ptr, 12) != 0

    @UseActual.setter
    def UseActual(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 12, value)

    @property
    def Pmax(self) -> float:
        """
        kW value at the time of max power. Is automatically set upon reading in a loadshape. Use this property to override the value automatically computed or to retrieve the value computed.

        DSS property name: `Pmax`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @Pmax.setter
    def Pmax(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def Qmax(self) -> float:
        """
        kvar value at the time of max kW power. Is automatically set upon reading in a loadshape. Use this property to override the value automatically computed or to retrieve the value computed.

        DSS property name: `Qmax`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @Qmax.setter
    def Qmax(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def sinterval(self) -> float:
        """
        Specify fixed interval in SECONDS. Alternate way to specify Interval property.

        DSS property name: `sinterval`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @sinterval.setter
    def sinterval(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def minterval(self) -> float:
        """
        Specify fixed interval in MINUTES. Alternate way to specify Interval property.

        DSS property name: `minterval`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @minterval.setter
    def minterval(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def Pbase(self) -> float:
        """
        Base P value for normalization. Default is zero, meaning the peak will be used.

        DSS property name: `Pbase`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @Pbase.setter
    def Pbase(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def Qbase(self) -> float:
        """
        Base Q value for normalization. Default is zero, meaning the peak will be used.

        DSS property name: `Qbase`, DSS property index: 18.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 18)

    @Qbase.setter
    def Qbase(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 18, value)

    @property
    def Pmult(self) -> Float64Array:
        """
        Synonym for "mult".

        DSS property name: `Pmult`, DSS property index: 19.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 19)

    @Pmult.setter
    def Pmult(self, value: Float64Array):
        self._set_float64_array_o(19, value)

    @property
    def PQCSVFile(self) -> str:
        """
        Switch input to a CSV text file containing (active, reactive) power (P, Q) multiplier pairs, one per row. 
        If the interval=0, there should be 3 items on each line: (hour, Pmult, Qmult)

        DSS property name: `PQCSVFile`, DSS property index: 20.
        """
        return self._get_prop_string(20)

    @PQCSVFile.setter
    def PQCSVFile(self, value: AnyStr):
        self._set_string_o(20, value)

    @property
    def MemoryMapping(self) -> bool:
        """
        {Yes | No* | True | False*} Enables the memory mapping functionality for dealing with large amounts of load shapes. 
        By defaul is False. Use it to accelerate the model loading when the containing a large number of load shapes.

        DSS property name: `MemoryMapping`, DSS property index: 21.
        """
        return self._lib.Obj_GetInt32(self._ptr, 21) != 0

    @MemoryMapping.setter
    def MemoryMapping(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 21, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 22.
        """
        self._set_string_o(22, value)

class TShape(DSSObj):
    __slots__ = []
    _cls_name = 'TShape'
    _cls_idx = 3
    _cls_prop_idx = {
        'npts': 1,
        'interval': 2,
        'temp': 3,
        'hour': 4,
        'mean': 5,
        'stddev': 6,
        'csvfile': 7,
        'sngfile': 8,
        'dblfile': 9,
        'sinterval': 10,
        'minterval': 11,
        'action': 12,
        'like': 13,
    }

    # Class-specific enumerations
    class TShapeAction(IntEnum):
        """TShape: Action (DSS enumeration for TShape)"""
        DblSave = 0 # DblSave
        SngSave = 1 # SngSave


    @property
    def npts(self) -> int:
        """
        Max number of points to expect in temperature shape vectors. This gets reset to the number of Temperature values found if less than specified.

        DSS property name: `npts`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @npts.setter
    def npts(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def interval(self) -> float:
        """
        Time interval for fixed interval data, hrs. Default = 1. If Interval = 0 then time data (in hours) may be at irregular intervals and time value must be specified using either the Hour property or input files. Then values are interpolated when Interval=0, but not for fixed interval data.  

        See also "sinterval" and "minterval".

        DSS property name: `interval`, DSS property index: 2.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 2)

    @interval.setter
    def interval(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 2, value)

    @property
    def temp(self) -> Float64Array:
        """
        Array of temperature values.  Units should be compatible with the object using the data. You can also use the syntax: 
        Temp = (file=filename)     !for text file one value per line
        Temp = (dblfile=filename)  !for packed file of doubles
        Temp = (sngfile=filename)  !for packed file of singles 

        Note: this property will reset Npts if the  number of values in the files are fewer.

        DSS property name: `temp`, DSS property index: 3.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 3)

    @temp.setter
    def temp(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    @property
    def hour(self) -> Float64Array:
        """
        Array of hour values. Only necessary to define this property for variable interval data. If the data are fixed interval, do not use this property. You can also use the syntax: 
        hour = (file=filename)     !for text file one value per line
        hour = (dblfile=filename)  !for packed file of doubles
        hour = (sngfile=filename)  !for packed file of singles 

        DSS property name: `hour`, DSS property index: 4.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 4)

    @hour.setter
    def hour(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def mean(self) -> float:
        """
        Mean of the temperature curve values.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently. Used for Monte Carlo load simulations.

        DSS property name: `mean`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @mean.setter
    def mean(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def stddev(self) -> float:
        """
        Standard deviation of the temperatures.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently.Is overwritten if you subsequently read in a curve

        Used for Monte Carlo load simulations.

        DSS property name: `stddev`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @stddev.setter
    def stddev(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def csvfile(self) -> str:
        """
        Switch input of  temperature curve data to a csv file containing (hour, Temp) points, or simply (Temp) values for fixed time interval data, one per line. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `csvfile`, DSS property index: 7.
        """
        return self._get_prop_string(7)

    @csvfile.setter
    def csvfile(self, value: AnyStr):
        self._set_string_o(7, value)

    @property
    def sngfile(self) -> str:
        """
        Switch input of  temperature curve data to a binary file of singles containing (hour, Temp) points, or simply (Temp) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `sngfile`, DSS property index: 8.
        """
        return self._get_prop_string(8)

    @sngfile.setter
    def sngfile(self, value: AnyStr):
        self._set_string_o(8, value)

    @property
    def dblfile(self) -> str:
        """
        Switch input of  temperature curve data to a binary file of doubles containing (hour, Temp) points, or simply (Temp) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `dblfile`, DSS property index: 9.
        """
        return self._get_prop_string(9)

    @dblfile.setter
    def dblfile(self, value: AnyStr):
        self._set_string_o(9, value)

    @property
    def sinterval(self) -> float:
        """
        Specify fixed interval in SECONDS. Alternate way to specify Interval property.

        DSS property name: `sinterval`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @sinterval.setter
    def sinterval(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def minterval(self) -> float:
        """
        Specify fixed interval in MINUTES. Alternate way to specify Interval property.

        DSS property name: `minterval`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @minterval.setter
    def minterval(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    def action(self, value: Union[str, bytes, int, TShapeAction]):
        """
        {DblSave | SngSave} After defining temperature curve data... Setting action=DblSave or SngSave will cause the present "Temp" values to be written to either a packed file of double or single. The filename is the Tshape name. 

        DSS property name: `action`, DSS property index: 12.
        """
        if isinstance(value, int):
            self._lib.Obj_SetInt32(self._ptr, 12, value)
            return
    
        self._set_string_o(12, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 13.
        """
        self._set_string_o(13, value)

class PriceShape(DSSObj):
    __slots__ = []
    _cls_name = 'PriceShape'
    _cls_idx = 4
    _cls_prop_idx = {
        'npts': 1,
        'interval': 2,
        'price': 3,
        'hour': 4,
        'mean': 5,
        'stddev': 6,
        'csvfile': 7,
        'sngfile': 8,
        'dblfile': 9,
        'sinterval': 10,
        'minterval': 11,
        'action': 12,
        'like': 13,
    }

    # Class-specific enumerations
    class PriceShapeAction(IntEnum):
        """PriceShape: Action (DSS enumeration for PriceShape)"""
        DblSave = 0 # DblSave
        SngSave = 1 # SngSave


    @property
    def npts(self) -> int:
        """
        Max number of points to expect in price shape vectors. This gets reset to the number of Price values found if less than specified.

        DSS property name: `npts`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @npts.setter
    def npts(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def interval(self) -> float:
        """
        Time interval for fixed interval data, hrs. Default = 1. If Interval = 0 then time data (in hours) may be at irregular intervals and time value must be specified using either the Hour property or input files. Then values are interpolated when Interval=0, but not for fixed interval data.  

        See also "sinterval" and "minterval".

        DSS property name: `interval`, DSS property index: 2.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 2)

    @interval.setter
    def interval(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 2, value)

    @property
    def price(self) -> Float64Array:
        """
        Array of Price values.  Units should be compatible with the object using the data. You can also use the syntax: 
        Price = (file=filename)     !for text file one value per line
        Price = (dblfile=filename)  !for packed file of doubles
        Price = (sngfile=filename)  !for packed file of singles 

        Note: this property will reset Npts if the  number of values in the files are fewer.

        DSS property name: `price`, DSS property index: 3.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 3)

    @price.setter
    def price(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    @property
    def hour(self) -> Float64Array:
        """
        Array of hour values. Only necessary to define this property for variable interval data. If the data are fixed interval, do not use this property. You can also use the syntax: 
        hour = (file=filename)     !for text file one value per line
        hour = (dblfile=filename)  !for packed file of doubles
        hour = (sngfile=filename)  !for packed file of singles 

        DSS property name: `hour`, DSS property index: 4.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 4)

    @hour.setter
    def hour(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def mean(self) -> float:
        """
        Mean of the Price curve values.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently. Used for Monte Carlo load simulations.

        DSS property name: `mean`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @mean.setter
    def mean(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def stddev(self) -> float:
        """
        Standard deviation of the Prices.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently.Is overwritten if you subsequently read in a curve

        Used for Monte Carlo load simulations.

        DSS property name: `stddev`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @stddev.setter
    def stddev(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def csvfile(self) -> str:
        """
        Switch input of  Price curve data to a csv file containing (hour, Price) points, or simply (Price) values for fixed time interval data, one per line. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `csvfile`, DSS property index: 7.
        """
        return self._get_prop_string(7)

    @csvfile.setter
    def csvfile(self, value: AnyStr):
        self._set_string_o(7, value)

    @property
    def sngfile(self) -> str:
        """
        Switch input of  Price curve data to a binary file of singles containing (hour, Price) points, or simply (Price) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `sngfile`, DSS property index: 8.
        """
        return self._get_prop_string(8)

    @sngfile.setter
    def sngfile(self, value: AnyStr):
        self._set_string_o(8, value)

    @property
    def dblfile(self) -> str:
        """
        Switch input of  Price curve data to a binary file of doubles containing (hour, Price) points, or simply (Price) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `dblfile`, DSS property index: 9.
        """
        return self._get_prop_string(9)

    @dblfile.setter
    def dblfile(self, value: AnyStr):
        self._set_string_o(9, value)

    @property
    def sinterval(self) -> float:
        """
        Specify fixed interval in SECONDS. Alternate way to specify Interval property.

        DSS property name: `sinterval`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @sinterval.setter
    def sinterval(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def minterval(self) -> float:
        """
        Specify fixed interval in MINUTES. Alternate way to specify Interval property.

        DSS property name: `minterval`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @minterval.setter
    def minterval(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    def action(self, value: Union[str, bytes, int, PriceShapeAction]):
        """
        {DblSave | SngSave} After defining Price curve data... Setting action=DblSave or SngSave will cause the present "Price" values to be written to either a packed file of double or single. The filename is the PriceShape name. 

        DSS property name: `action`, DSS property index: 12.
        """
        if isinstance(value, int):
            self._lib.Obj_SetInt32(self._ptr, 12, value)
            return
    
        self._set_string_o(12, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 13.
        """
        self._set_string_o(13, value)

class XYcurve(DSSObj):
    __slots__ = []
    _cls_name = 'XYcurve'
    _cls_idx = 5
    _cls_prop_idx = {
        'npts': 1,
        'points': 2,
        'yarray': 3,
        'xarray': 4,
        'csvfile': 5,
        'sngfile': 6,
        'dblfile': 7,
        'x': 8,
        'y': 9,
        'xshift': 10,
        'yshift': 11,
        'xscale': 12,
        'yscale': 13,
        'like': 14,
    }

    @property
    def npts(self) -> int:
        """
        Max number of points to expect in curve. This could get reset to the actual number of points defined if less than specified.

        DSS property name: `npts`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @npts.setter
    def npts(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def Yarray(self) -> Float64Array:
        """
        Alternate way to enter Y values. Enter an array of Y values corresponding to the X values.  You can also use the syntax: 
        Yarray = (file=filename)     !for text file one value per line
        Yarray = (dblfile=filename)  !for packed file of doubles
        Yarray = (sngfile=filename)  !for packed file of singles 

        Note: this property will reset Npts to a smaller value if the  number of values in the files are fewer.

        DSS property name: `Yarray`, DSS property index: 3.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 3)

    @Yarray.setter
    def Yarray(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    @property
    def Xarray(self) -> Float64Array:
        """
        Alternate way to enter X values. Enter an array of X values corresponding to the Y values.  You can also use the syntax: 
        Xarray = (file=filename)     !for text file one value per line
        Xarray = (dblfile=filename)  !for packed file of doubles
        Xarray = (sngfile=filename)  !for packed file of singles 

        Note: this property will reset Npts to a smaller value if the  number of values in the files are fewer.

        DSS property name: `Xarray`, DSS property index: 4.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 4)

    @Xarray.setter
    def Xarray(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def csvfile(self) -> str:
        """
        Switch input of  X-Y curve data to a CSV file containing X, Y points one per line. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `csvfile`, DSS property index: 5.
        """
        return self._get_prop_string(5)

    @csvfile.setter
    def csvfile(self, value: AnyStr):
        self._set_string_o(5, value)

    @property
    def sngfile(self) -> str:
        """
        Switch input of  X-Y curve data to a binary file of SINGLES containing X, Y points packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `sngfile`, DSS property index: 6.
        """
        return self._get_prop_string(6)

    @sngfile.setter
    def sngfile(self, value: AnyStr):
        self._set_string_o(6, value)

    @property
    def dblfile(self) -> str:
        """
        Switch input of  X-Y  curve data to a binary file of DOUBLES containing X, Y points packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `dblfile`, DSS property index: 7.
        """
        return self._get_prop_string(7)

    @dblfile.setter
    def dblfile(self, value: AnyStr):
        self._set_string_o(7, value)

    @property
    def x(self) -> float:
        """
        Enter a value and then retrieve the interpolated Y value from the Y property. On input shifted then scaled to original curve. Scaled then shifted on output.

        DSS property name: `x`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @x.setter
    def x(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def y(self) -> float:
        """
        Enter a value and then retrieve the interpolated X value from the X property. On input shifted then scaled to original curve. Scaled then shifted on output.

        DSS property name: `y`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @y.setter
    def y(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def Xshift(self) -> float:
        """
        Shift X property values (in/out) by this amount of offset. Default = 0. Does not change original definition of arrays.

        DSS property name: `Xshift`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @Xshift.setter
    def Xshift(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def Yshift(self) -> float:
        """
        Shift Y property values (in/out) by this amount of offset. Default = 0. Does not change original definition of arrays.

        DSS property name: `Yshift`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @Yshift.setter
    def Yshift(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def Xscale(self) -> float:
        """
        Scale X property values (in/out) by this factor. Default = 1.0. Does not change original definition of arrays.

        DSS property name: `Xscale`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @Xscale.setter
    def Xscale(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def Yscale(self) -> float:
        """
        Scale Y property values (in/out) by this factor. Default = 1.0. Does not change original definition of arrays.

        DSS property name: `Yscale`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @Yscale.setter
    def Yscale(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 14.
        """
        self._set_string_o(14, value)

class GrowthShape(DSSObj):
    __slots__ = []
    _cls_name = 'GrowthShape'
    _cls_idx = 6
    _cls_prop_idx = {
        'npts': 1,
        'year': 2,
        'mult': 3,
        'csvfile': 4,
        'sngfile': 5,
        'dblfile': 6,
        'like': 7,
    }

    @property
    def npts(self) -> int:
        """
        Number of points to expect in subsequent vector.

        DSS property name: `npts`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @npts.setter
    def npts(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def year(self) -> Float64Array:
        """
        Array of year values, or a text file spec, corresponding to the multipliers. Enter only those years where the growth changes. May be any integer sequence -- just so it is consistent. See help on Mult.

        DSS property name: `year`, DSS property index: 2.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 2)

    @year.setter
    def year(self, value: Float64Array):
        self._set_float64_array_o(2, value)

    @property
    def mult(self) -> Float64Array:
        """
        Array of growth multiplier values, or a text file spec, corresponding to the year values. Enter the multiplier by which you would multiply the previous year's load to get the present year's.

        Examples:

          Year = [1, 2, 5]   Mult=[1.05, 1.025, 1.02].
          Year= (File=years.txt) Mult= (file=mults.txt).

        Text files contain one value per line.

        DSS property name: `mult`, DSS property index: 3.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 3)

    @mult.setter
    def mult(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    @property
    def csvfile(self) -> str:
        """
        Switch input of growth curve data to a csv file containing (year, mult) points, one per line.

        DSS property name: `csvfile`, DSS property index: 4.
        """
        return self._get_prop_string(4)

    @csvfile.setter
    def csvfile(self, value: AnyStr):
        self._set_string_o(4, value)

    @property
    def sngfile(self) -> str:
        """
        Switch input of growth curve data to a binary file of singles containing (year, mult) points, packed one after another.

        DSS property name: `sngfile`, DSS property index: 5.
        """
        return self._get_prop_string(5)

    @sngfile.setter
    def sngfile(self, value: AnyStr):
        self._set_string_o(5, value)

    @property
    def dblfile(self) -> str:
        """
        Switch input of growth curve data to a binary file of doubles containing (year, mult) points, packed one after another.

        DSS property name: `dblfile`, DSS property index: 6.
        """
        return self._get_prop_string(6)

    @dblfile.setter
    def dblfile(self, value: AnyStr):
        self._set_string_o(6, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 7.
        """
        self._set_string_o(7, value)

class TCC_Curve(DSSObj):
    __slots__ = []
    _cls_name = 'TCC_Curve'
    _cls_idx = 7
    _cls_prop_idx = {
        'npts': 1,
        'c_array': 2,
        't_array': 3,
        'like': 4,
    }

    @property
    def npts(self) -> int:
        """
        Number of points to expect in time-current arrays.

        DSS property name: `npts`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @npts.setter
    def npts(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def C_array(self) -> Float64Array:
        """
        Array of current (or voltage) values corresponding to time values (see help on T_Array).

        DSS property name: `C_array`, DSS property index: 2.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 2)

    @C_array.setter
    def C_array(self, value: Float64Array):
        self._set_float64_array_o(2, value)

    @property
    def T_array(self) -> Float64Array:
        """
        Array of time values in sec. Typical array syntax: 
        t_array = (1, 2, 3, 4, ...)

        Can also substitute a file designation: 
        t_array =  (file=filename)

        The specified file has one value per line.

        DSS property name: `T_array`, DSS property index: 3.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 3)

    @T_array.setter
    def T_array(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 4.
        """
        self._set_string_o(4, value)

class Spectrum(DSSObj):
    __slots__ = []
    _cls_name = 'Spectrum'
    _cls_idx = 8
    _cls_prop_idx = {
        'numharm': 1,
        'harmonic': 2,
        'pctmag': 3,
        '%mag': 3,
        'angle': 4,
        'csvfile': 5,
        'like': 6,
    }

    @property
    def NumHarm(self) -> int:
        """
        Number of frequencies in this spectrum. (See CSVFile)

        DSS property name: `NumHarm`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @NumHarm.setter
    def NumHarm(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def harmonic(self) -> Float64Array:
        """
        Array of harmonic values. You can also use the syntax
        harmonic = (file=filename)     !for text file one value per line
        harmonic = (dblfile=filename)  !for packed file of doubles
        harmonic = (sngfile=filename)  !for packed file of singles 

        DSS property name: `harmonic`, DSS property index: 2.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 2)

    @harmonic.setter
    def harmonic(self, value: Float64Array):
        self._set_float64_array_o(2, value)

    @property
    def pctmag(self) -> Float64Array:
        """
        Array of magnitude values, assumed to be in PERCENT. You can also use the syntax
        %mag = (file=filename)     !for text file one value per line
        %mag = (dblfile=filename)  !for packed file of doubles
        %mag = (sngfile=filename)  !for packed file of singles 

        DSS property name: `%mag`, DSS property index: 3.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 3)

    @pctmag.setter
    def pctmag(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    @property
    def angle(self) -> Float64Array:
        """
        Array of phase angle values, degrees.You can also use the syntax
        angle = (file=filename)     !for text file one value per line
        angle = (dblfile=filename)  !for packed file of doubles
        angle = (sngfile=filename)  !for packed file of singles 

        DSS property name: `angle`, DSS property index: 4.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 4)

    @angle.setter
    def angle(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def CSVFile(self) -> str:
        """
        File of spectrum points with (harmonic, magnitude-percent, angle-degrees) values, one set of 3 per line, in CSV format. If fewer than NUMHARM frequencies found in the file, NUMHARM is set to the smaller value.

        DSS property name: `CSVFile`, DSS property index: 5.
        """
        return self._get_prop_string(5)

    @CSVFile.setter
    def CSVFile(self, value: AnyStr):
        self._set_string_o(5, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 6.
        """
        self._set_string_o(6, value)

class WireData(DSSObj):
    __slots__ = []
    _cls_name = 'WireData'
    _cls_idx = 9
    _cls_prop_idx = {
        'rdc': 1,
        'rac': 2,
        'runits': 3,
        'gmrac': 4,
        'gmrunits': 5,
        'radius': 6,
        'radunits': 7,
        'normamps': 8,
        'emergamps': 9,
        'diam': 10,
        'seasons': 11,
        'ratings': 12,
        'capradius': 13,
        'like': 14,
    }

    @property
    def Rdc(self) -> float:
        """
        dc Resistance, ohms per unit length (see Runits). Defaults to Rac/1.02 if not specified.

        DSS property name: `Rdc`, DSS property index: 1.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 1)

    @Rdc.setter
    def Rdc(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 1, value)

    @property
    def Rac(self) -> float:
        """
        Resistance at 60 Hz per unit length. Defaults to 1.02*Rdc if not specified.

        DSS property name: `Rac`, DSS property index: 2.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 2)

    @Rac.setter
    def Rac(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 2, value)

    @property
    def Runits(self) -> DimensionUnits:
        """
        Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `Runits`, DSS property index: 3.
        """
        return DimensionUnits(self._lib.Obj_GetInt32(self._ptr, 3))

    @Runits.setter
    def Runits(self, value: Union[AnyStr, int, DimensionUnits]):
        if not isinstance(value, int):
            self._set_string_o(3, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 3, value)

    @property
    def Runits_str(self) -> str:
        """
        Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `Runits`, DSS property index: 3.
        """
        return self._get_prop_string(3)

    @Runits_str.setter
    def Runits_str(self, value: AnyStr):
        self.Runits = value

    @property
    def GMRac(self) -> float:
        """
        GMR at 60 Hz. Defaults to .7788*radius if not specified.

        DSS property name: `GMRac`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @GMRac.setter
    def GMRac(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def GMRunits(self) -> DimensionUnits:
        """
        Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `GMRunits`, DSS property index: 5.
        """
        return DimensionUnits(self._lib.Obj_GetInt32(self._ptr, 5))

    @GMRunits.setter
    def GMRunits(self, value: Union[AnyStr, int, DimensionUnits]):
        if not isinstance(value, int):
            self._set_string_o(5, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 5, value)

    @property
    def GMRunits_str(self) -> str:
        """
        Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `GMRunits`, DSS property index: 5.
        """
        return self._get_prop_string(5)

    @GMRunits_str.setter
    def GMRunits_str(self, value: AnyStr):
        self.GMRunits = value

    @property
    def radius(self) -> float:
        """
        Outside radius of conductor. Defaults to GMR/0.7788 if not specified.

        DSS property name: `radius`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @radius.setter
    def radius(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def radunits(self) -> DimensionUnits:
        """
        Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `radunits`, DSS property index: 7.
        """
        return DimensionUnits(self._lib.Obj_GetInt32(self._ptr, 7))

    @radunits.setter
    def radunits(self, value: Union[AnyStr, int, DimensionUnits]):
        if not isinstance(value, int):
            self._set_string_o(7, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 7, value)

    @property
    def radunits_str(self) -> str:
        """
        Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `radunits`, DSS property index: 7.
        """
        return self._get_prop_string(7)

    @radunits_str.setter
    def radunits_str(self, value: AnyStr):
        self.radunits = value

    @property
    def normamps(self) -> float:
        """
        Normal ampacity, amperes. Defaults to Emergency amps/1.5 if not specified.

        DSS property name: `normamps`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @normamps.setter
    def normamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def emergamps(self) -> float:
        """
        Emergency ampacity, amperes. Defaults to 1.5 * Normal Amps if not specified.

        DSS property name: `emergamps`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @emergamps.setter
    def emergamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def diam(self) -> float:
        """
        Diameter; Alternative method for entering radius.

        DSS property name: `diam`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @diam.setter
    def diam(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def Seasons(self) -> int:
        """
        Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 11.
        """
        return self._lib.Obj_GetInt32(self._ptr, 11)

    @Seasons.setter
    def Seasons(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 11, value)

    @property
    def Ratings(self) -> Float64Array:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.

        DSS property name: `Ratings`, DSS property index: 12.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 12)

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(12, value)

    @property
    def Capradius(self) -> float:
        """
        Equivalent conductor radius for capacitance calcs. Specify this for bundled conductors. Defaults to same value as radius. Define Diam or Radius property first.

        DSS property name: `Capradius`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @Capradius.setter
    def Capradius(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 14.
        """
        self._set_string_o(14, value)

class CNData(DSSObj):
    __slots__ = []
    _cls_name = 'CNData'
    _cls_idx = 10
    _cls_prop_idx = {
        'k': 1,
        'diastrand': 2,
        'gmrstrand': 3,
        'rstrand': 4,
        'epsr': 5,
        'inslayer': 6,
        'diains': 7,
        'diacable': 8,
        'rdc': 9,
        'rac': 10,
        'runits': 11,
        'gmrac': 12,
        'gmrunits': 13,
        'radius': 14,
        'radunits': 15,
        'normamps': 16,
        'emergamps': 17,
        'diam': 18,
        'seasons': 19,
        'ratings': 20,
        'capradius': 21,
        'like': 22,
    }

    @property
    def k(self) -> int:
        """
        Number of concentric neutral strands; default is 2

        DSS property name: `k`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @k.setter
    def k(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def DiaStrand(self) -> float:
        """
        Diameter of a concentric neutral strand; same units as core conductor radius; no default.

        DSS property name: `DiaStrand`, DSS property index: 2.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 2)

    @DiaStrand.setter
    def DiaStrand(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 2, value)

    @property
    def GmrStrand(self) -> float:
        """
        Geometric mean radius of a concentric neutral strand; same units as core conductor GMR; defaults to 0.7788 * CN strand radius.

        DSS property name: `GmrStrand`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @GmrStrand.setter
    def GmrStrand(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def Rstrand(self) -> float:
        """
        AC resistance of a concentric neutral strand; same units as core conductor resistance; no default.

        DSS property name: `Rstrand`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @Rstrand.setter
    def Rstrand(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def EpsR(self) -> float:
        """
        Insulation layer relative permittivity; default is 2.3.

        DSS property name: `EpsR`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @EpsR.setter
    def EpsR(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def InsLayer(self) -> float:
        """
        Insulation layer thickness; same units as radius; no default. With DiaIns, establishes inner radius for capacitance calculation.

        DSS property name: `InsLayer`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @InsLayer.setter
    def InsLayer(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def DiaIns(self) -> float:
        """
        Diameter over insulation layer; same units as radius; no default. Establishes outer radius for capacitance calculation.

        DSS property name: `DiaIns`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @DiaIns.setter
    def DiaIns(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def DiaCable(self) -> float:
        """
        Diameter over cable; same units as radius; no default.

        DSS property name: `DiaCable`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @DiaCable.setter
    def DiaCable(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def Rdc(self) -> float:
        """
        dc Resistance, ohms per unit length (see Runits). Defaults to Rac/1.02 if not specified.

        DSS property name: `Rdc`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @Rdc.setter
    def Rdc(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def Rac(self) -> float:
        """
        Resistance at 60 Hz per unit length. Defaults to 1.02*Rdc if not specified.

        DSS property name: `Rac`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @Rac.setter
    def Rac(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def Runits(self) -> DimensionUnits:
        """
        Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `Runits`, DSS property index: 11.
        """
        return DimensionUnits(self._lib.Obj_GetInt32(self._ptr, 11))

    @Runits.setter
    def Runits(self, value: Union[AnyStr, int, DimensionUnits]):
        if not isinstance(value, int):
            self._set_string_o(11, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 11, value)

    @property
    def Runits_str(self) -> str:
        """
        Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `Runits`, DSS property index: 11.
        """
        return self._get_prop_string(11)

    @Runits_str.setter
    def Runits_str(self, value: AnyStr):
        self.Runits = value

    @property
    def GMRac(self) -> float:
        """
        GMR at 60 Hz. Defaults to .7788*radius if not specified.

        DSS property name: `GMRac`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @GMRac.setter
    def GMRac(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def GMRunits(self) -> DimensionUnits:
        """
        Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `GMRunits`, DSS property index: 13.
        """
        return DimensionUnits(self._lib.Obj_GetInt32(self._ptr, 13))

    @GMRunits.setter
    def GMRunits(self, value: Union[AnyStr, int, DimensionUnits]):
        if not isinstance(value, int):
            self._set_string_o(13, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 13, value)

    @property
    def GMRunits_str(self) -> str:
        """
        Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `GMRunits`, DSS property index: 13.
        """
        return self._get_prop_string(13)

    @GMRunits_str.setter
    def GMRunits_str(self, value: AnyStr):
        self.GMRunits = value

    @property
    def radius(self) -> float:
        """
        Outside radius of conductor. Defaults to GMR/0.7788 if not specified.

        DSS property name: `radius`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @radius.setter
    def radius(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def radunits(self) -> DimensionUnits:
        """
        Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `radunits`, DSS property index: 15.
        """
        return DimensionUnits(self._lib.Obj_GetInt32(self._ptr, 15))

    @radunits.setter
    def radunits(self, value: Union[AnyStr, int, DimensionUnits]):
        if not isinstance(value, int):
            self._set_string_o(15, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 15, value)

    @property
    def radunits_str(self) -> str:
        """
        Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `radunits`, DSS property index: 15.
        """
        return self._get_prop_string(15)

    @radunits_str.setter
    def radunits_str(self, value: AnyStr):
        self.radunits = value

    @property
    def normamps(self) -> float:
        """
        Normal ampacity, amperes. Defaults to Emergency amps/1.5 if not specified.

        DSS property name: `normamps`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @normamps.setter
    def normamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def emergamps(self) -> float:
        """
        Emergency ampacity, amperes. Defaults to 1.5 * Normal Amps if not specified.

        DSS property name: `emergamps`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @emergamps.setter
    def emergamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def diam(self) -> float:
        """
        Diameter; Alternative method for entering radius.

        DSS property name: `diam`, DSS property index: 18.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 18)

    @diam.setter
    def diam(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 18, value)

    @property
    def Seasons(self) -> int:
        """
        Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 19.
        """
        return self._lib.Obj_GetInt32(self._ptr, 19)

    @Seasons.setter
    def Seasons(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 19, value)

    @property
    def Ratings(self) -> Float64Array:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.

        DSS property name: `Ratings`, DSS property index: 20.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 20)

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(20, value)

    @property
    def Capradius(self) -> float:
        """
        Equivalent conductor radius for capacitance calcs. Specify this for bundled conductors. Defaults to same value as radius. Define Diam or Radius property first.

        DSS property name: `Capradius`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @Capradius.setter
    def Capradius(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 22.
        """
        self._set_string_o(22, value)

class TSData(DSSObj):
    __slots__ = []
    _cls_name = 'TSData'
    _cls_idx = 11
    _cls_prop_idx = {
        'diashield': 1,
        'tapelayer': 2,
        'tapelap': 3,
        'epsr': 4,
        'inslayer': 5,
        'diains': 6,
        'diacable': 7,
        'rdc': 8,
        'rac': 9,
        'runits': 10,
        'gmrac': 11,
        'gmrunits': 12,
        'radius': 13,
        'radunits': 14,
        'normamps': 15,
        'emergamps': 16,
        'diam': 17,
        'seasons': 18,
        'ratings': 19,
        'capradius': 20,
        'like': 21,
    }

    @property
    def DiaShield(self) -> float:
        """
        Diameter over tape shield; same units as radius; no default.

        DSS property name: `DiaShield`, DSS property index: 1.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 1)

    @DiaShield.setter
    def DiaShield(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 1, value)

    @property
    def TapeLayer(self) -> float:
        """
        Tape shield thickness; same units as radius; no default.

        DSS property name: `TapeLayer`, DSS property index: 2.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 2)

    @TapeLayer.setter
    def TapeLayer(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 2, value)

    @property
    def TapeLap(self) -> float:
        """
        Tape Lap in percent; default 20.0

        DSS property name: `TapeLap`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @TapeLap.setter
    def TapeLap(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def EpsR(self) -> float:
        """
        Insulation layer relative permittivity; default is 2.3.

        DSS property name: `EpsR`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @EpsR.setter
    def EpsR(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def InsLayer(self) -> float:
        """
        Insulation layer thickness; same units as radius; no default. With DiaIns, establishes inner radius for capacitance calculation.

        DSS property name: `InsLayer`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @InsLayer.setter
    def InsLayer(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def DiaIns(self) -> float:
        """
        Diameter over insulation layer; same units as radius; no default. Establishes outer radius for capacitance calculation.

        DSS property name: `DiaIns`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @DiaIns.setter
    def DiaIns(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def DiaCable(self) -> float:
        """
        Diameter over cable; same units as radius; no default.

        DSS property name: `DiaCable`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @DiaCable.setter
    def DiaCable(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def Rdc(self) -> float:
        """
        dc Resistance, ohms per unit length (see Runits). Defaults to Rac/1.02 if not specified.

        DSS property name: `Rdc`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @Rdc.setter
    def Rdc(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def Rac(self) -> float:
        """
        Resistance at 60 Hz per unit length. Defaults to 1.02*Rdc if not specified.

        DSS property name: `Rac`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @Rac.setter
    def Rac(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def Runits(self) -> DimensionUnits:
        """
        Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `Runits`, DSS property index: 10.
        """
        return DimensionUnits(self._lib.Obj_GetInt32(self._ptr, 10))

    @Runits.setter
    def Runits(self, value: Union[AnyStr, int, DimensionUnits]):
        if not isinstance(value, int):
            self._set_string_o(10, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 10, value)

    @property
    def Runits_str(self) -> str:
        """
        Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `Runits`, DSS property index: 10.
        """
        return self._get_prop_string(10)

    @Runits_str.setter
    def Runits_str(self, value: AnyStr):
        self.Runits = value

    @property
    def GMRac(self) -> float:
        """
        GMR at 60 Hz. Defaults to .7788*radius if not specified.

        DSS property name: `GMRac`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @GMRac.setter
    def GMRac(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def GMRunits(self) -> DimensionUnits:
        """
        Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `GMRunits`, DSS property index: 12.
        """
        return DimensionUnits(self._lib.Obj_GetInt32(self._ptr, 12))

    @GMRunits.setter
    def GMRunits(self, value: Union[AnyStr, int, DimensionUnits]):
        if not isinstance(value, int):
            self._set_string_o(12, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 12, value)

    @property
    def GMRunits_str(self) -> str:
        """
        Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `GMRunits`, DSS property index: 12.
        """
        return self._get_prop_string(12)

    @GMRunits_str.setter
    def GMRunits_str(self, value: AnyStr):
        self.GMRunits = value

    @property
    def radius(self) -> float:
        """
        Outside radius of conductor. Defaults to GMR/0.7788 if not specified.

        DSS property name: `radius`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @radius.setter
    def radius(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def radunits(self) -> DimensionUnits:
        """
        Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `radunits`, DSS property index: 14.
        """
        return DimensionUnits(self._lib.Obj_GetInt32(self._ptr, 14))

    @radunits.setter
    def radunits(self, value: Union[AnyStr, int, DimensionUnits]):
        if not isinstance(value, int):
            self._set_string_o(14, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 14, value)

    @property
    def radunits_str(self) -> str:
        """
        Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `radunits`, DSS property index: 14.
        """
        return self._get_prop_string(14)

    @radunits_str.setter
    def radunits_str(self, value: AnyStr):
        self.radunits = value

    @property
    def normamps(self) -> float:
        """
        Normal ampacity, amperes. Defaults to Emergency amps/1.5 if not specified.

        DSS property name: `normamps`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @normamps.setter
    def normamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def emergamps(self) -> float:
        """
        Emergency ampacity, amperes. Defaults to 1.5 * Normal Amps if not specified.

        DSS property name: `emergamps`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @emergamps.setter
    def emergamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def diam(self) -> float:
        """
        Diameter; Alternative method for entering radius.

        DSS property name: `diam`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @diam.setter
    def diam(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def Seasons(self) -> int:
        """
        Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 18.
        """
        return self._lib.Obj_GetInt32(self._ptr, 18)

    @Seasons.setter
    def Seasons(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 18, value)

    @property
    def Ratings(self) -> Float64Array:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.

        DSS property name: `Ratings`, DSS property index: 19.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 19)

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(19, value)

    @property
    def Capradius(self) -> float:
        """
        Equivalent conductor radius for capacitance calcs. Specify this for bundled conductors. Defaults to same value as radius. Define Diam or Radius property first.

        DSS property name: `Capradius`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @Capradius.setter
    def Capradius(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 21.
        """
        self._set_string_o(21, value)

class LineSpacing(DSSObj):
    __slots__ = []
    _cls_name = 'LineSpacing'
    _cls_idx = 12
    _cls_prop_idx = {
        'nconds': 1,
        'nphases': 2,
        'x': 3,
        'h': 4,
        'units': 5,
        'like': 6,
    }

    @property
    def nconds(self) -> int:
        """
        Number of wires in this geometry. Default is 3. Triggers memory allocations. Define first!

        DSS property name: `nconds`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @nconds.setter
    def nconds(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def nphases(self) -> int:
        """
        Number of retained phase conductors. If less than the number of wires, list the retained phase coordinates first.

        DSS property name: `nphases`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @nphases.setter
    def nphases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def x(self) -> Float64Array:
        """
        Array of wire X coordinates.

        DSS property name: `x`, DSS property index: 3.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 3)

    @x.setter
    def x(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    @property
    def h(self) -> Float64Array:
        """
        Array of wire Heights.

        DSS property name: `h`, DSS property index: 4.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 4)

    @h.setter
    def h(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def units(self) -> DimensionUnits:
        """
        Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined

        DSS property name: `units`, DSS property index: 5.
        """
        return DimensionUnits(self._lib.Obj_GetInt32(self._ptr, 5))

    @units.setter
    def units(self, value: Union[AnyStr, int, DimensionUnits]):
        if not isinstance(value, int):
            self._set_string_o(5, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 5, value)

    @property
    def units_str(self) -> str:
        """
        Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined

        DSS property name: `units`, DSS property index: 5.
        """
        return self._get_prop_string(5)

    @units_str.setter
    def units_str(self, value: AnyStr):
        self.units = value

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 6.
        """
        self._set_string_o(6, value)

class LineGeometry(DSSObj):
    __slots__ = []
    _cls_name = 'LineGeometry'
    _cls_idx = 13
    _cls_prop_idx = {
        'nconds': 1,
        'nphases': 2,
        'cond': 3,
        'wire': 4,
        'x': 5,
        'h': 6,
        'units': 7,
        'normamps': 8,
        'emergamps': 9,
        'reduce': 10,
        'spacing': 11,
        'wires': 12,
        'cncable': 13,
        'tscable': 14,
        'cncables': 15,
        'tscables': 16,
        'seasons': 17,
        'ratings': 18,
        'linetype': 19,
        'like': 20,
    }

    @property
    def nconds(self) -> int:
        """
        Number of conductors in this geometry. Default is 3. Triggers memory allocations. Define first!

        DSS property name: `nconds`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @nconds.setter
    def nconds(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def nphases(self) -> int:
        """
        Number of phases. Default =3; All other conductors are considered neutrals and might be reduced out.

        DSS property name: `nphases`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @nphases.setter
    def nphases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def x(self) -> Float64Array:
        """
        x coordinate.

        DSS property name: `x`, DSS property index: 5.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 5)

    @x.setter
    def x(self, value: Float64Array):
        self._set_float64_array_o(5, value)

    @property
    def h(self) -> Float64Array:
        """
        Height of conductor.

        DSS property name: `h`, DSS property index: 6.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 6)

    @h.setter
    def h(self, value: Float64Array):
        self._set_float64_array_o(6, value)

    @property
    def units(self) -> DimensionUnits:
        """
        Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined

        DSS property name: `units`, DSS property index: 7.
        """
        return DimensionUnits(self._lib.Obj_GetInt32(self._ptr, 7))

    @units.setter
    def units(self, value: Union[AnyStr, int, DimensionUnits]):
        if not isinstance(value, int):
            self._set_string_o(7, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 7, value)

    @property
    def units_str(self) -> str:
        """
        Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined

        DSS property name: `units`, DSS property index: 7.
        """
        return self._get_prop_string(7)

    @units_str.setter
    def units_str(self, value: AnyStr):
        self.units = value

    @property
    def normamps(self) -> float:
        """
        Normal ampacity, amperes for the line. Defaults to first conductor if not specified.

        DSS property name: `normamps`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @normamps.setter
    def normamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def emergamps(self) -> float:
        """
        Emergency ampacity, amperes. Defaults to first conductor if not specified.

        DSS property name: `emergamps`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @emergamps.setter
    def emergamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def reduce(self) -> bool:
        """
        {Yes | No} Default = no. Reduce to Nphases (Kron Reduction). Reduce out neutrals.

        DSS property name: `reduce`, DSS property index: 10.
        """
        return self._lib.Obj_GetInt32(self._ptr, 10) != 0

    @reduce.setter
    def reduce(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 10, value)

    @property
    def spacing(self) -> str:
        """
        Reference to a LineSpacing for use in a line constants calculation.
        Alternative to x, h, and units. MUST BE PREVIOUSLY DEFINED.
        Must match "nconds" as previously defined for this geometry.
        Must be used in conjunction with the Wires property.

        DSS property name: `spacing`, DSS property index: 11.
        """
        return self._get_prop_string(11)

    @spacing.setter
    def spacing(self, value: Union[AnyStr, LineSpacing]):
        if isinstance(value, DSSObj):
            self._set_obj(11, value)
            return

        self._set_string_o(11, value)

    @property
    def spacing_obj(self) -> LineSpacing:
        """
        Reference to a LineSpacing for use in a line constants calculation.
        Alternative to x, h, and units. MUST BE PREVIOUSLY DEFINED.
        Must match "nconds" as previously defined for this geometry.
        Must be used in conjunction with the Wires property.

        DSS property name: `spacing`, DSS property index: 11.
        """
        return self._get_obj(11, LineSpacing)

    @spacing_obj.setter
    def spacing_obj(self, value: LineSpacing):
        self._set_obj(11, value)

    @property
    def wires(self) -> List[str]:
        """
        Array of WireData names for use in a line constants calculation.
        Alternative to individual wire inputs. ALL MUST BE PREVIOUSLY DEFINED.
        Must match "nconds" as previously defined for this geometry,
        unless TSData or CNData were previously assigned to phases, and these wires are neutrals.
        Must be used in conjunction with the Spacing property.

        DSS property name: `wires`, DSS property index: 12.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 12)

    @wires.setter
    def wires(self, value: List[Union[AnyStr, WireData]]):
        if value is None or len(value) == 0 or not isinstance(value[0], DSSObj):
            self._set_string_array_o(12, value)
            return

        self._set_obj_array(12, value)

    @property
    def wires_obj(self) -> List[WireData]:
        """
        Array of WireData names for use in a line constants calculation.
        Alternative to individual wire inputs. ALL MUST BE PREVIOUSLY DEFINED.
        Must match "nconds" as previously defined for this geometry,
        unless TSData or CNData were previously assigned to phases, and these wires are neutrals.
        Must be used in conjunction with the Spacing property.

        DSS property name: `wires`, DSS property index: 12.
        """
        return self._get_obj_array(12, WireData)

    @wires_obj.setter
    def wires_obj(self, value: List[WireData]):
        self._set_obj_array(12, value)

    @property
    def cncables(self) -> List[str]:
        """
        Array of CNData names for cable parameter calculation.
        All must be previously defined, and match "nphases" for this geometry.
        You can later define "nconds-nphases" wires for bare neutral conductors.

        DSS property name: `cncables`, DSS property index: 15.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 15)

    @cncables.setter
    def cncables(self, value: List[Union[AnyStr, CNData]]):
        if value is None or len(value) == 0 or not isinstance(value[0], DSSObj):
            self._set_string_array_o(15, value)
            return

        self._set_obj_array(15, value)

    @property
    def cncables_obj(self) -> List[CNData]:
        """
        Array of CNData names for cable parameter calculation.
        All must be previously defined, and match "nphases" for this geometry.
        You can later define "nconds-nphases" wires for bare neutral conductors.

        DSS property name: `cncables`, DSS property index: 15.
        """
        return self._get_obj_array(15, CNData)

    @cncables_obj.setter
    def cncables_obj(self, value: List[CNData]):
        self._set_obj_array(15, value)

    @property
    def tscables(self) -> List[str]:
        """
        Array of TSData names for cable parameter calculation.
        All must be previously defined, and match "nphases" for this geometry.
        You can later define "nconds-nphases" wires for bare neutral conductors.

        DSS property name: `tscables`, DSS property index: 16.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 16)

    @tscables.setter
    def tscables(self, value: List[Union[AnyStr, TSData]]):
        if value is None or len(value) == 0 or not isinstance(value[0], DSSObj):
            self._set_string_array_o(16, value)
            return

        self._set_obj_array(16, value)

    @property
    def tscables_obj(self) -> List[TSData]:
        """
        Array of TSData names for cable parameter calculation.
        All must be previously defined, and match "nphases" for this geometry.
        You can later define "nconds-nphases" wires for bare neutral conductors.

        DSS property name: `tscables`, DSS property index: 16.
        """
        return self._get_obj_array(16, TSData)

    @tscables_obj.setter
    def tscables_obj(self, value: List[TSData]):
        self._set_obj_array(16, value)

    @property
    def Seasons(self) -> int:
        """
        Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property. Defaults to first conductor if not specified.

        DSS property name: `Seasons`, DSS property index: 17.
        """
        return self._lib.Obj_GetInt32(self._ptr, 17)

    @Seasons.setter
    def Seasons(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 17, value)

    @property
    def Ratings(self) -> Float64Array:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.Defaults to first conductor if not specified.

        DSS property name: `Ratings`, DSS property index: 18.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 18)

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(18, value)

    @property
    def linetype(self) -> LineType:
        """
        Code designating the type of line. 
        One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW

        OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.

        DSS property name: `LineType`, DSS property index: 19.
        """
        return LineType(self._lib.Obj_GetInt32(self._ptr, 19))

    @linetype.setter
    def linetype(self, value: Union[AnyStr, int, LineType]):
        if not isinstance(value, int):
            self._set_string_o(19, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 19, value)

    @property
    def linetype_str(self) -> str:
        """
        Code designating the type of line. 
        One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW

        OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.

        DSS property name: `LineType`, DSS property index: 19.
        """
        return self._get_prop_string(19)

    @linetype_str.setter
    def linetype_str(self, value: AnyStr):
        self.linetype = value

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 20.
        """
        self._set_string_o(20, value)

class XfmrCode(DSSObj):
    __slots__ = []
    _cls_name = 'XfmrCode'
    _cls_idx = 14
    _cls_prop_idx = {
        'phases': 1,
        'windings': 2,
        'wdg': 3,
        'conn': 4,
        'kv': 5,
        'kva': 6,
        'tap': 7,
        'pctr': 8,
        '%r': 8,
        'rneut': 9,
        'xneut': 10,
        'conns': 11,
        'kvs': 12,
        'kvas': 13,
        'taps': 14,
        'xhl': 15,
        'xht': 16,
        'xlt': 17,
        'xscarray': 18,
        'thermal': 19,
        'n': 20,
        'm': 21,
        'flrise': 22,
        'hsrise': 23,
        'pctloadloss': 24,
        '%loadloss': 24,
        'pctnoloadloss': 25,
        '%noloadloss': 25,
        'normhkva': 26,
        'emerghkva': 27,
        'maxtap': 28,
        'mintap': 29,
        'numtaps': 30,
        'pctimag': 31,
        '%imag': 31,
        'ppm_antifloat': 32,
        'pctrs': 33,
        '%rs': 33,
        'x12': 34,
        'x13': 35,
        'x23': 36,
        'rdcohms': 37,
        'seasons': 38,
        'ratings': 39,
        'like': 40,
    }

    @property
    def phases(self) -> int:
        """
        Number of phases this transformer. Default is 3.

        DSS property name: `phases`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def windings(self) -> int:
        """
        Number of windings, this transformers. (Also is the number of terminals) Default is 2. This property triggers memory allocation for the Transformer and will cause other properties to revert to default values.

        DSS property name: `windings`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @windings.setter
    def windings(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def pctR(self) -> Float64Array:
        """
        Percent resistance this winding.  (half of total for a 2-winding).

        DSS property name: `%R`, DSS property index: 8.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 8)

    @pctR.setter
    def pctR(self, value: Float64Array):
        self._set_float64_array_o(8, value)

    @property
    def Rneut(self) -> Float64Array:
        """
        Default = -1. Neutral resistance of wye (star)-connected winding in actual ohms.If entered as a negative value, the neutral is assumed to be open, or floating.

        DSS property name: `Rneut`, DSS property index: 9.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 9)

    @Rneut.setter
    def Rneut(self, value: Float64Array):
        self._set_float64_array_o(9, value)

    @property
    def Xneut(self) -> Float64Array:
        """
        Neutral reactance of wye(star)-connected winding in actual ohms.  May be + or -.

        DSS property name: `Xneut`, DSS property index: 10.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 10)

    @Xneut.setter
    def Xneut(self, value: Float64Array):
        self._set_float64_array_o(10, value)

    @property
    def conns(self) -> List[Connection]:
        """
        Use this to specify all the Winding connections at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus"
        ~ conns=(delta, wye)

        DSS property name: `conns`, DSS property index: 11.
        """
        return [Connection(val) for val in self._get_int32_list(self._lib.Obj_GetInt32Array, self._ptr, 11)]

    @conns.setter
    def conns(self, value: Union[List[Union[int,Connection]], List[AnyStr]]):
        if len(value) and not isinstance(value[0], int):
            self._set_string_array_o(11, value)
            return    
        self._set_int32_array_o(11, value)

    @property
    def conns_str(self) -> List[str]:
        """
        Use this to specify all the Winding connections at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus"
        ~ conns=(delta, wye)

        DSS property name: `conns`, DSS property index: 11.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 11)

    @conns_str.setter
    def conns_str(self, value: AnyStr):
        self.conns = value

    @property
    def kVs(self) -> Float64Array:
        """
        Use this to specify the kV ratings of all windings at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus" 
        ~ conns=(delta, wye)
        ~ kvs=(115, 12.47)

        See kV= property for voltage rules.

        DSS property name: `kVs`, DSS property index: 12.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 12)

    @kVs.setter
    def kVs(self, value: Float64Array):
        self._set_float64_array_o(12, value)

    @property
    def kVAs(self) -> Float64Array:
        """
        Use this to specify the kVA ratings of all windings at once using an array.

        DSS property name: `kVAs`, DSS property index: 13.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 13)

    @kVAs.setter
    def kVAs(self, value: Float64Array):
        self._set_float64_array_o(13, value)

    @property
    def taps(self) -> Float64Array:
        """
        Use this to specify the normal p.u. tap of all windings at once using an array.

        DSS property name: `taps`, DSS property index: 14.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 14)

    @taps.setter
    def taps(self, value: Float64Array):
        self._set_float64_array_o(14, value)

    @property
    def Xhl(self) -> float:
        """
        Use this to specify the percent reactance, H-L (winding 1 to winding 2).  Use for 2- or 3-winding transformers. On the kva base of winding 1.

        DSS property name: `Xhl`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @Xhl.setter
    def Xhl(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def Xht(self) -> float:
        """
        Use this to specify the percent reactance, H-T (winding 1 to winding 3).  Use for 3-winding transformers only. On the kVA base of winding 1.

        DSS property name: `Xht`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @Xht.setter
    def Xht(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def Xlt(self) -> float:
        """
        Use this to specify the percent reactance, L-T (winding 2 to winding 3).  Use for 3-winding transformers only. On the kVA base of winding 1.

        DSS property name: `Xlt`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @Xlt.setter
    def Xlt(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def Xscarray(self) -> Float64Array:
        """
        Use this to specify the percent reactance between all pairs of windings as an array. All values are on the kVA base of winding 1.  The order of the values is as follows:

        (x12 13 14... 23 24.. 34 ..)  

        There will be n(n-1)/2 values, where n=number of windings.

        DSS property name: `Xscarray`, DSS property index: 18.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 18)

    @Xscarray.setter
    def Xscarray(self, value: Float64Array):
        self._set_float64_array_o(18, value)

    @property
    def thermal(self) -> float:
        """
        Thermal time constant of the transformer in hours.  Typically about 2.

        DSS property name: `thermal`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @thermal.setter
    def thermal(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def n(self) -> float:
        """
        n Exponent for thermal properties in IEEE C57.  Typically 0.8.

        DSS property name: `n`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @n.setter
    def n(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    @property
    def m(self) -> float:
        """
        m Exponent for thermal properties in IEEE C57.  Typically 0.9 - 1.0

        DSS property name: `m`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @m.setter
    def m(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def flrise(self) -> float:
        """
        Temperature rise, deg C, for full load.  Default is 65.

        DSS property name: `flrise`, DSS property index: 22.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 22)

    @flrise.setter
    def flrise(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 22, value)

    @property
    def hsrise(self) -> float:
        """
        Hot spot temperature rise, deg C.  Default is 15.

        DSS property name: `hsrise`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @hsrise.setter
    def hsrise(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def pctloadloss(self) -> float:
        """
        Percent load loss at full load. The %R of the High and Low windings (1 and 2) are adjusted to agree at rated kVA loading.

        DSS property name: `%loadloss`, DSS property index: 24.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 24)

    @pctloadloss.setter
    def pctloadloss(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 24, value)

    @property
    def pctnoloadloss(self) -> float:
        """
        Percent no load losses at rated excitatation voltage. Default is 0. Converts to a resistance in parallel with the magnetizing impedance in each winding.

        DSS property name: `%noloadloss`, DSS property index: 25.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 25)

    @pctnoloadloss.setter
    def pctnoloadloss(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 25, value)

    @property
    def normhkVA(self) -> float:
        """
        Normal maximum kVA rating of H winding (winding 1).  Usually 100% - 110% ofmaximum nameplate rating, depending on load shape. Defaults to 110% of kVA rating of Winding 1.

        DSS property name: `normhkVA`, DSS property index: 26.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 26)

    @normhkVA.setter
    def normhkVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 26, value)

    @property
    def emerghkVA(self) -> float:
        """
        Emergency (contingency)  kVA rating of H winding (winding 1).  Usually 140% - 150% ofmaximum nameplate rating, depending on load shape. Defaults to 150% of kVA rating of Winding 1.

        DSS property name: `emerghkVA`, DSS property index: 27.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 27)

    @emerghkVA.setter
    def emerghkVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 27, value)

    @property
    def MaxTap(self) -> Float64Array:
        """
        Max per unit tap for the active winding.  Default is 1.10

        DSS property name: `MaxTap`, DSS property index: 28.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 28)

    @MaxTap.setter
    def MaxTap(self, value: Float64Array):
        self._set_float64_array_o(28, value)

    @property
    def MinTap(self) -> Float64Array:
        """
        Min per unit tap for the active winding.  Default is 0.90

        DSS property name: `MinTap`, DSS property index: 29.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 29)

    @MinTap.setter
    def MinTap(self, value: Float64Array):
        self._set_float64_array_o(29, value)

    @property
    def NumTaps(self) -> Int32Array:
        """
        Total number of taps between min and max tap.  Default is 32.

        DSS property name: `NumTaps`, DSS property index: 30.
        """
        return self._get_int32_array(self._lib.Obj_GetInt32Array, self._ptr, 30)

    @NumTaps.setter
    def NumTaps(self, value: Int32Array):
        self._set_int32_array_o(30, value)

    @property
    def pctimag(self) -> float:
        """
        Percent magnetizing current. Default=0.0. Magnetizing branch is in parallel with windings in each phase. Also, see "ppm_antifloat".

        DSS property name: `%imag`, DSS property index: 31.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 31)

    @pctimag.setter
    def pctimag(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 31, value)

    @property
    def ppm_antifloat(self) -> float:
        """
        Default=1 ppm.  Parts per million of transformer winding VA rating connected to ground to protect against accidentally floating a winding without a reference. If positive then the effect is adding a very large reactance to ground.  If negative, then a capacitor.

        DSS property name: `ppm_antifloat`, DSS property index: 32.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 32)

    @ppm_antifloat.setter
    def ppm_antifloat(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 32, value)

    @property
    def pctRs(self) -> Float64Array:
        """
        Use this property to specify all the winding %resistances using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus" ~ %Rs=(0.2  0.3)

        DSS property name: `%Rs`, DSS property index: 33.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 33)

    @pctRs.setter
    def pctRs(self, value: Float64Array):
        self._set_float64_array_o(33, value)

    @property
    def X12(self) -> float:
        """
        Alternative to XHL for specifying the percent reactance from winding 1 to winding 2.  Use for 2- or 3-winding transformers. Percent on the kVA base of winding 1. 

        DSS property name: `X12`, DSS property index: 34.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 34)

    @X12.setter
    def X12(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 34, value)

    @property
    def X13(self) -> float:
        """
        Alternative to XHT for specifying the percent reactance from winding 1 to winding 3.  Use for 3-winding transformers only. Percent on the kVA base of winding 1. 

        DSS property name: `X13`, DSS property index: 35.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 35)

    @X13.setter
    def X13(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 35, value)

    @property
    def X23(self) -> float:
        """
        Alternative to XLT for specifying the percent reactance from winding 2 to winding 3.Use for 3-winding transformers only. Percent on the kVA base of winding 1.  

        DSS property name: `X23`, DSS property index: 36.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 36)

    @X23.setter
    def X23(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 36, value)

    @property
    def RdcOhms(self) -> Float64Array:
        """
        Winding dc resistance in OHMS. Useful for GIC analysis. From transformer test report. Defaults to 85% of %R property

        DSS property name: `RdcOhms`, DSS property index: 37.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 37)

    @RdcOhms.setter
    def RdcOhms(self, value: Float64Array):
        self._set_float64_array_o(37, value)

    @property
    def Seasons(self) -> int:
        """
        Defines the number of ratings to be defined for the transfomer, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 38.
        """
        return self._lib.Obj_GetInt32(self._ptr, 38)

    @Seasons.setter
    def Seasons(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 38, value)

    @property
    def Ratings(self) -> Float64Array:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in transformers.

        DSS property name: `Ratings`, DSS property index: 39.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 39)

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(39, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 40.
        """
        self._set_string_o(40, value)

class Line(DSSObj):
    __slots__ = []
    _cls_name = 'Line'
    _cls_idx = 15
    _cls_prop_idx = {
        'bus1': 1,
        'bus2': 2,
        'linecode': 3,
        'length': 4,
        'phases': 5,
        'r1': 6,
        'x1': 7,
        'r0': 8,
        'x0': 9,
        'c1': 10,
        'c0': 11,
        'rmatrix': 12,
        'xmatrix': 13,
        'cmatrix': 14,
        'switch': 15,
        'rg': 16,
        'xg': 17,
        'rho': 18,
        'geometry': 19,
        'units': 20,
        'spacing': 21,
        'wires': 22,
        'earthmodel': 23,
        'cncables': 24,
        'tscables': 25,
        'b1': 26,
        'b0': 27,
        'seasons': 28,
        'ratings': 29,
        'linetype': 30,
        'normamps': 31,
        'emergamps': 32,
        'faultrate': 33,
        'pctperm': 34,
        'repair': 35,
        'basefreq': 36,
        'enabled': 37,
        'like': 38,
    }

    @property
    def bus1(self) -> str:
        """
        Name of bus to which first terminal is connected.
        Example:
        bus1=busname   (assumes all terminals connected in normal phase order)
        bus1=busname.3.1.2.0 (specify terminal to node connections explicitly)

        DSS property name: `bus1`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(1, value)

    @property
    def bus2(self) -> str:
        """
        Name of bus to which 2nd terminal is connected.

        DSS property name: `bus2`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @bus2.setter
    def bus2(self, value: AnyStr):
        self._set_string_o(2, value)

    @property
    def linecode(self) -> str:
        """
        Name of linecode object describing line impedances.
        If you use a line code, you do not need to specify the impedances here. The line code must have been PREVIOUSLY defined. The values specified last will prevail over those specified earlier (left-to-right sequence of properties).  You can subsequently change the number of phases if symmetrical component quantities are specified.If no line code or impedance data are specified, the line object defaults to 336 MCM ACSR on 4 ft spacing.

        DSS property name: `linecode`, DSS property index: 3.
        """
        return self._get_prop_string(3)

    @linecode.setter
    def linecode(self, value: Union[AnyStr, LineCode]):
        if isinstance(value, DSSObj):
            self._set_obj(3, value)
            return

        self._set_string_o(3, value)

    @property
    def linecode_obj(self) -> LineCode:
        """
        Name of linecode object describing line impedances.
        If you use a line code, you do not need to specify the impedances here. The line code must have been PREVIOUSLY defined. The values specified last will prevail over those specified earlier (left-to-right sequence of properties).  You can subsequently change the number of phases if symmetrical component quantities are specified.If no line code or impedance data are specified, the line object defaults to 336 MCM ACSR on 4 ft spacing.

        DSS property name: `linecode`, DSS property index: 3.
        """
        return self._get_obj(3, LineCode)

    @linecode_obj.setter
    def linecode_obj(self, value: LineCode):
        self._set_obj(3, value)

    @property
    def length(self) -> float:
        """
        Length of line. Default is 1.0. If units do not match the impedance data, specify "units" property. 

        DSS property name: `length`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @length.setter
    def length(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def phases(self) -> int:
        """
        Number of phases, this line.

        DSS property name: `phases`, DSS property index: 5.
        """
        return self._lib.Obj_GetInt32(self._ptr, 5)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 5, value)

    @property
    def r1(self) -> float:
        """
        Positive-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Rmatrix.

        DSS property name: `r1`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @r1.setter
    def r1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def x1(self) -> float:
        """
        Positive-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.  See also Xmatrix

        DSS property name: `x1`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @x1.setter
    def x1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def r0(self) -> float:
        """
        Zero-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.

        DSS property name: `r0`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @r0.setter
    def r0(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def x0(self) -> float:
        """
        Zero-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.

        DSS property name: `x0`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @x0.setter
    def x0(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def C1(self) -> float:
        """
        Positive-sequence capacitance, nf per unit length.  Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Cmatrix and B1.

        DSS property name: `C1`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @C1.setter
    def C1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def C0(self) -> float:
        """
        Zero-sequence capacitance, nf per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.See also B0.

        DSS property name: `C0`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @C0.setter
    def C0(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def rmatrix(self) -> Float64Array:
        """
        Resistance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration. Using any of Rmatrix, Xmatrix, Cmatrix forces program to use the matrix values for line impedance definition. For balanced line models, you may use the standard symmetrical component data definition instead.

        DSS property name: `rmatrix`, DSS property index: 12.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 12)

    @rmatrix.setter
    def rmatrix(self, value: Float64Array):
        self._set_float64_array_o(12, value)

    @property
    def xmatrix(self) -> Float64Array:
        """
        Reactance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration. Using any of Rmatrix, Xmatrix, Cmatrix forces program to use the matrix values for line impedance definition.  For balanced line models, you may use the standard symmetrical component data definition instead.

        DSS property name: `xmatrix`, DSS property index: 13.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 13)

    @xmatrix.setter
    def xmatrix(self, value: Float64Array):
        self._set_float64_array_o(13, value)

    @property
    def cmatrix(self) -> Float64Array:
        """
        Nodal Capacitance matrix, lower triangle, nf per unit length.Order of the matrix is the number of phases. May be used to specify the shunt capacitance of any line configuration. Using any of Rmatrix, Xmatrix, Cmatrix forces program to use the matrix values for line impedance definition.  For balanced line models, you may use the standard symmetrical component data definition instead.

        DSS property name: `cmatrix`, DSS property index: 14.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 14)

    @cmatrix.setter
    def cmatrix(self, value: Float64Array):
        self._set_float64_array_o(14, value)

    @property
    def Switch(self) -> bool:
        """
        {y/n | T/F}  Default= no/false.  Designates this line as a switch for graphics and algorithmic purposes. 
        SIDE EFFECT: Sets r1 = 1.0; x1 = 1.0; r0 = 1.0; x0 = 1.0; c1 = 1.1 ; c0 = 1.0;  length = 0.001; You must reset if you want something different.

        DSS property name: `Switch`, DSS property index: 15.
        """
        return self._lib.Obj_GetInt32(self._ptr, 15) != 0

    @Switch.setter
    def Switch(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 15, value)

    @property
    def Rg(self) -> float:
        """
        Carson earth return resistance per unit length used to compute impedance values at base frequency. Default is 0.01805 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.

        DSS property name: `Rg`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @Rg.setter
    def Rg(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def Xg(self) -> float:
        """
        Carson earth return reactance per unit length used to compute impedance values at base frequency.  For making better frequency adjustments. Default is 0.155081 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.

        DSS property name: `Xg`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @Xg.setter
    def Xg(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def rho(self) -> float:
        """
        Default=100 meter ohms.  Earth resitivity used to compute earth correction factor. Overrides Line geometry definition if specified.

        DSS property name: `rho`, DSS property index: 18.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 18)

    @rho.setter
    def rho(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 18, value)

    @property
    def geometry(self) -> str:
        """
        Geometry code for LineGeometry Object. Supercedes any previous definition of line impedance. Line constants are computed for each frequency change or rho change. CAUTION: may alter number of phases. You cannot subsequently change the number of phases unless you change how the line impedance is defined.

        DSS property name: `geometry`, DSS property index: 19.
        """
        return self._get_prop_string(19)

    @geometry.setter
    def geometry(self, value: Union[AnyStr, LineGeometry]):
        if isinstance(value, DSSObj):
            self._set_obj(19, value)
            return

        self._set_string_o(19, value)

    @property
    def geometry_obj(self) -> LineGeometry:
        """
        Geometry code for LineGeometry Object. Supercedes any previous definition of line impedance. Line constants are computed for each frequency change or rho change. CAUTION: may alter number of phases. You cannot subsequently change the number of phases unless you change how the line impedance is defined.

        DSS property name: `geometry`, DSS property index: 19.
        """
        return self._get_obj(19, LineGeometry)

    @geometry_obj.setter
    def geometry_obj(self, value: LineGeometry):
        self._set_obj(19, value)

    @property
    def units(self) -> DimensionUnits:
        """
        Length Units = {none | mi|kft|km|m|Ft|in|cm } Default is None - assumes length units match impedance units.

        DSS property name: `units`, DSS property index: 20.
        """
        return DimensionUnits(self._lib.Obj_GetInt32(self._ptr, 20))

    @units.setter
    def units(self, value: Union[AnyStr, int, DimensionUnits]):
        if not isinstance(value, int):
            self._set_string_o(20, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 20, value)

    @property
    def units_str(self) -> str:
        """
        Length Units = {none | mi|kft|km|m|Ft|in|cm } Default is None - assumes length units match impedance units.

        DSS property name: `units`, DSS property index: 20.
        """
        return self._get_prop_string(20)

    @units_str.setter
    def units_str(self, value: AnyStr):
        self.units = value

    @property
    def spacing(self) -> str:
        """
        Reference to a LineSpacing for use in a line constants calculation.
        Must be used in conjunction with the Wires property.
        Specify this before the wires property.

        DSS property name: `spacing`, DSS property index: 21.
        """
        return self._get_prop_string(21)

    @spacing.setter
    def spacing(self, value: Union[AnyStr, LineSpacing]):
        if isinstance(value, DSSObj):
            self._set_obj(21, value)
            return

        self._set_string_o(21, value)

    @property
    def spacing_obj(self) -> LineSpacing:
        """
        Reference to a LineSpacing for use in a line constants calculation.
        Must be used in conjunction with the Wires property.
        Specify this before the wires property.

        DSS property name: `spacing`, DSS property index: 21.
        """
        return self._get_obj(21, LineSpacing)

    @spacing_obj.setter
    def spacing_obj(self, value: LineSpacing):
        self._set_obj(21, value)

    @property
    def wires(self) -> List[str]:
        """
        Array of WireData names for use in an overhead line constants calculation.
        Must be used in conjunction with the Spacing property.
        Specify the Spacing first, and "ncond" wires.
        May also be used to specify bare neutrals with cables, using "ncond-nphase" wires.

        DSS property name: `wires`, DSS property index: 22.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 22)

    @wires.setter
    def wires(self, value: List[Union[AnyStr, WireData]]):
        if value is None or len(value) == 0 or not isinstance(value[0], DSSObj):
            self._set_string_array_o(22, value)
            return

        self._set_obj_array(22, value)

    @property
    def wires_obj(self) -> List[WireData]:
        """
        Array of WireData names for use in an overhead line constants calculation.
        Must be used in conjunction with the Spacing property.
        Specify the Spacing first, and "ncond" wires.
        May also be used to specify bare neutrals with cables, using "ncond-nphase" wires.

        DSS property name: `wires`, DSS property index: 22.
        """
        return self._get_obj_array(22, WireData)

    @wires_obj.setter
    def wires_obj(self, value: List[WireData]):
        self._set_obj_array(22, value)

    @property
    def earthmodel(self) -> EarthModel:
        """
        One of {Carson | FullCarson | Deri}. Default is the global value established with the Set EarthModel command. See the Options Help on EarthModel option. This is used to override the global value for this line. This option applies only when the "geometry" property is used.

        DSS property name: `EarthModel`, DSS property index: 23.
        """
        return EarthModel(self._lib.Obj_GetInt32(self._ptr, 23))

    @earthmodel.setter
    def earthmodel(self, value: Union[AnyStr, int, EarthModel]):
        if not isinstance(value, int):
            self._set_string_o(23, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 23, value)

    @property
    def earthmodel_str(self) -> str:
        """
        One of {Carson | FullCarson | Deri}. Default is the global value established with the Set EarthModel command. See the Options Help on EarthModel option. This is used to override the global value for this line. This option applies only when the "geometry" property is used.

        DSS property name: `EarthModel`, DSS property index: 23.
        """
        return self._get_prop_string(23)

    @earthmodel_str.setter
    def earthmodel_str(self, value: AnyStr):
        self.earthmodel = value

    @property
    def cncables(self) -> List[str]:
        """
        Array of CNData names for use in a cable constants calculation.
        Must be used in conjunction with the Spacing property.
        Specify the Spacing first, using "nphases" cncables.
        You may later specify "nconds-nphases" wires for separate neutrals

        DSS property name: `cncables`, DSS property index: 24.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 24)

    @cncables.setter
    def cncables(self, value: List[Union[AnyStr, CNData]]):
        if value is None or len(value) == 0 or not isinstance(value[0], DSSObj):
            self._set_string_array_o(24, value)
            return

        self._set_obj_array(24, value)

    @property
    def cncables_obj(self) -> List[CNData]:
        """
        Array of CNData names for use in a cable constants calculation.
        Must be used in conjunction with the Spacing property.
        Specify the Spacing first, using "nphases" cncables.
        You may later specify "nconds-nphases" wires for separate neutrals

        DSS property name: `cncables`, DSS property index: 24.
        """
        return self._get_obj_array(24, CNData)

    @cncables_obj.setter
    def cncables_obj(self, value: List[CNData]):
        self._set_obj_array(24, value)

    @property
    def tscables(self) -> List[str]:
        """
        Array of TSData names for use in a cable constants calculation.
        Must be used in conjunction with the Spacing property.
        Specify the Spacing first, using "nphases" tscables.
        You may later specify "nconds-nphases" wires for separate neutrals

        DSS property name: `tscables`, DSS property index: 25.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 25)

    @tscables.setter
    def tscables(self, value: List[Union[AnyStr, TSData]]):
        if value is None or len(value) == 0 or not isinstance(value[0], DSSObj):
            self._set_string_array_o(25, value)
            return

        self._set_obj_array(25, value)

    @property
    def tscables_obj(self) -> List[TSData]:
        """
        Array of TSData names for use in a cable constants calculation.
        Must be used in conjunction with the Spacing property.
        Specify the Spacing first, using "nphases" tscables.
        You may later specify "nconds-nphases" wires for separate neutrals

        DSS property name: `tscables`, DSS property index: 25.
        """
        return self._get_obj_array(25, TSData)

    @tscables_obj.setter
    def tscables_obj(self, value: List[TSData]):
        self._set_obj_array(25, value)

    @property
    def B1(self) -> float:
        """
        Alternate way to specify C1. MicroS per unit length

        DSS property name: `B1`, DSS property index: 26.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 26)

    @B1.setter
    def B1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 26, value)

    @property
    def B0(self) -> float:
        """
        Alternate way to specify C0. MicroS per unit length

        DSS property name: `B0`, DSS property index: 27.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 27)

    @B0.setter
    def B0(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 27, value)

    @property
    def Seasons(self) -> int:
        """
        Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 28.
        """
        return self._lib.Obj_GetInt32(self._ptr, 28)

    @Seasons.setter
    def Seasons(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 28, value)

    @property
    def Ratings(self) -> Float64Array:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.

        DSS property name: `Ratings`, DSS property index: 29.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 29)

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(29, value)

    @property
    def linetype(self) -> LineType:
        """
        Code designating the type of line. 
        One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW

        OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.

        DSS property name: `LineType`, DSS property index: 30.
        """
        return LineType(self._lib.Obj_GetInt32(self._ptr, 30))

    @linetype.setter
    def linetype(self, value: Union[AnyStr, int, LineType]):
        if not isinstance(value, int):
            self._set_string_o(30, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 30, value)

    @property
    def linetype_str(self) -> str:
        """
        Code designating the type of line. 
        One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW

        OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.

        DSS property name: `LineType`, DSS property index: 30.
        """
        return self._get_prop_string(30)

    @linetype_str.setter
    def linetype_str(self, value: AnyStr):
        self.linetype = value

    @property
    def normamps(self) -> float:
        """
        Normal rated current.

        DSS property name: `normamps`, DSS property index: 31.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 31)

    @normamps.setter
    def normamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 31, value)

    @property
    def emergamps(self) -> float:
        """
        Maximum or emerg current.

        DSS property name: `emergamps`, DSS property index: 32.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 32)

    @emergamps.setter
    def emergamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 32, value)

    @property
    def faultrate(self) -> float:
        """
        Failure rate PER UNIT LENGTH per year. Length must be same units as LENGTH property. Default is 0.1 fault per unit length per year.

        DSS property name: `faultrate`, DSS property index: 33.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 33)

    @faultrate.setter
    def faultrate(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 33, value)

    @property
    def pctperm(self) -> float:
        """
        Percent of failures that become permanent. Default is 20.

        DSS property name: `pctperm`, DSS property index: 34.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 34)

    @pctperm.setter
    def pctperm(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 34, value)

    @property
    def repair(self) -> float:
        """
        Hours to repair. Default is 3 hr.

        DSS property name: `repair`, DSS property index: 35.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 35)

    @repair.setter
    def repair(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 35, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 36.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 36)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 36, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 37.
        """
        return self._lib.Obj_GetInt32(self._ptr, 37) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 37, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 38.
        """
        self._set_string_o(38, value)

class Vsource(DSSObj):
    __slots__ = []
    _cls_name = 'Vsource'
    _cls_idx = 16
    _cls_prop_idx = {
        'bus1': 1,
        'basekv': 2,
        'pu': 3,
        'angle': 4,
        'frequency': 5,
        'phases': 6,
        'mvasc3': 7,
        'mvasc1': 8,
        'x1r1': 9,
        'x0r0': 10,
        'isc3': 11,
        'isc1': 12,
        'r1': 13,
        'x1': 14,
        'r0': 15,
        'x0': 16,
        'scantype': 17,
        'sequence': 18,
        'bus2': 19,
        'z1': 20,
        'z0': 21,
        'z2': 22,
        'puz1': 23,
        'puz0': 24,
        'puz2': 25,
        'basemva': 26,
        'yearly': 27,
        'daily': 28,
        'duty': 29,
        'model': 30,
        'puzideal': 31,
        'spectrum': 32,
        'basefreq': 33,
        'enabled': 34,
        'like': 35,
    }

    # Class-specific enumerations
    class VSourceModel(IntEnum):
        """VSource: Model (DSS enumeration for Vsource)"""
        Thevenin = 0 # Thevenin
        Ideal = 1 # Ideal


    @property
    def bus1(self) -> str:
        """
        Name of bus to which the main terminal (1) is connected.
        bus1=busname
        bus1=busname.1.2.3

        The VSOURCE object is a two-terminal voltage source (thevenin equivalent). Bus2 defaults to Bus1 with all phases connected to ground (node 0) unless previously specified. This is a Yg connection. If you want something different, define the Bus2 property ezplicitly.

        DSS property name: `bus1`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(1, value)

    @property
    def basekv(self) -> float:
        """
        Base Source kV, usually phase-phase (L-L) unless you are making a positive-sequence model or 1-phase modelin which case, it will be phase-neutral (L-N) kV.

        DSS property name: `basekv`, DSS property index: 2.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 2)

    @basekv.setter
    def basekv(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 2, value)

    @property
    def pu(self) -> float:
        """
        Per unit of the base voltage that the source is actually operating at.
        "pu=1.05"

        DSS property name: `pu`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @pu.setter
    def pu(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def angle(self) -> float:
        """
        Phase angle in degrees of first phase: e.g.,Angle=10.3

        DSS property name: `angle`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @angle.setter
    def angle(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def frequency(self) -> float:
        """
        Source frequency.  Defaults to system default base frequency.

        DSS property name: `frequency`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @frequency.setter
    def frequency(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def phases(self) -> int:
        """
        Number of phases.  Defaults to 3.

        DSS property name: `phases`, DSS property index: 6.
        """
        return self._lib.Obj_GetInt32(self._ptr, 6)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def MVAsc3(self) -> float:
        """
        MVA Short circuit, 3-phase fault. Default = 2000. Z1 is determined by squaring the base kv and dividing by this value. For single-phase source, this value is not used.

        DSS property name: `MVAsc3`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @MVAsc3.setter
    def MVAsc3(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def MVAsc1(self) -> float:
        """
        MVA Short Circuit, 1-phase fault. Default = 2100. The "single-phase impedance", Zs, is determined by squaring the base kV and dividing by this value. Then Z0 is determined by Z0 = 3Zs - 2Z1.  For 1-phase sources, Zs is used directly. Use X0R0 to define X/R ratio for 1-phase source.

        DSS property name: `MVAsc1`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @MVAsc1.setter
    def MVAsc1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def x1r1(self) -> float:
        """
        Positive-sequence  X/R ratio. Default = 4.

        DSS property name: `x1r1`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @x1r1.setter
    def x1r1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def x0r0(self) -> float:
        """
        Zero-sequence X/R ratio.Default = 3.

        DSS property name: `x0r0`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @x0r0.setter
    def x0r0(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def Isc3(self) -> float:
        """
        Alternate method of defining the source impedance. 
        3-phase short circuit current, amps.  Default is 10000.

        DSS property name: `Isc3`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @Isc3.setter
    def Isc3(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def Isc1(self) -> float:
        """
        Alternate method of defining the source impedance. 
        single-phase short circuit current, amps.  Default is 10500.

        DSS property name: `Isc1`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @Isc1.setter
    def Isc1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def R1(self) -> float:
        """
        Alternate method of defining the source impedance. 
        Positive-sequence resistance, ohms.  Default is 1.65.

        DSS property name: `R1`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @R1.setter
    def R1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def X1(self) -> float:
        """
        Alternate method of defining the source impedance. 
        Positive-sequence reactance, ohms.  Default is 6.6.

        DSS property name: `X1`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @X1.setter
    def X1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def R0(self) -> float:
        """
        Alternate method of defining the source impedance. 
        Zero-sequence resistance, ohms.  Default is 1.9.

        DSS property name: `R0`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @R0.setter
    def R0(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def X0(self) -> float:
        """
        Alternate method of defining the source impedance. 
        Zero-sequence reactance, ohms.  Default is 5.7.

        DSS property name: `X0`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @X0.setter
    def X0(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def scantype(self) -> ScanType:
        """
        {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.

        DSS property name: `ScanType`, DSS property index: 17.
        """
        return ScanType(self._lib.Obj_GetInt32(self._ptr, 17))

    @scantype.setter
    def scantype(self, value: Union[AnyStr, int, ScanType]):
        if not isinstance(value, int):
            self._set_string_o(17, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 17, value)

    @property
    def scantype_str(self) -> str:
        """
        {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.

        DSS property name: `ScanType`, DSS property index: 17.
        """
        return self._get_prop_string(17)

    @scantype_str.setter
    def scantype_str(self, value: AnyStr):
        self.scantype = value

    @property
    def Sequence(self) -> SequenceType:
        """
        {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence. 

        DSS property name: `Sequence`, DSS property index: 18.
        """
        return SequenceType(self._lib.Obj_GetInt32(self._ptr, 18))

    @Sequence.setter
    def Sequence(self, value: Union[AnyStr, int, SequenceType]):
        if not isinstance(value, int):
            self._set_string_o(18, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 18, value)

    @property
    def Sequence_str(self) -> str:
        """
        {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence. 

        DSS property name: `Sequence`, DSS property index: 18.
        """
        return self._get_prop_string(18)

    @Sequence_str.setter
    def Sequence_str(self, value: AnyStr):
        self.Sequence = value

    @property
    def bus2(self) -> str:
        """
        Name of bus to which 2nd terminal is connected.
        bus2=busname
        bus2=busname.1.2.3

        Default is Bus1.0.0.0 (grounded wye connection)

        DSS property name: `bus2`, DSS property index: 19.
        """
        return self._get_prop_string(19)

    @bus2.setter
    def bus2(self, value: AnyStr):
        self._set_string_o(19, value)

    @property
    def Z2(self) -> complex:
        """
        Negative-sequence equivalent source impedance, ohms, as a 2-element array representing a complex number. Example: 

        Z2=[1, 2]  ! represents 1 + j2 

        Used to define the impedance matrix of the VSOURCE if Z1 is also specified. 

        Note: Z2 defaults to Z1 if it is not specifically defined. If Z2 is not equal to Z1, the impedance matrix is asymmetrical.

        DSS property name: `Z2`, DSS property index: 22.
        """
        return self._get_complex(22)

    @Z2.setter
    def Z2(self, value: complex):
        self._set_complex(22, value)

    @property
    def puZ1(self) -> complex:
        """
        2-element array: e.g., [1  2]. An alternate way to specify Z1. See Z1 property. Per-unit positive-sequence impedance on base of Vsource BasekV and BaseMVA.

        DSS property name: `puZ1`, DSS property index: 23.
        """
        return self._get_complex(23)

    @puZ1.setter
    def puZ1(self, value: complex):
        self._set_complex(23, value)

    @property
    def puZ0(self) -> complex:
        """
        2-element array: e.g., [1  2]. An alternate way to specify Z0. See Z0 property. Per-unit zero-sequence impedance on base of Vsource BasekV and BaseMVA.

        DSS property name: `puZ0`, DSS property index: 24.
        """
        return self._get_complex(24)

    @puZ0.setter
    def puZ0(self, value: complex):
        self._set_complex(24, value)

    @property
    def puZ2(self) -> complex:
        """
        2-element array: e.g., [1  2]. An alternate way to specify Z2. See Z2 property. Per-unit negative-sequence impedance on base of Vsource BasekV and BaseMVA.

        DSS property name: `puZ2`, DSS property index: 25.
        """
        return self._get_complex(25)

    @puZ2.setter
    def puZ2(self, value: complex):
        self._set_complex(25, value)

    @property
    def baseMVA(self) -> float:
        """
        Default value is 100. Base used to convert values specifiied with puZ1, puZ0, and puZ2 properties to ohms on kV base specified by BasekV property.

        DSS property name: `baseMVA`, DSS property index: 26.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 26)

    @baseMVA.setter
    def baseMVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 26, value)

    @property
    def Yearly(self) -> str:
        """
        LOADSHAPE object to use for the per-unit voltage for YEARLY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.

        Must be previously defined as a LOADSHAPE object. 

        Is set to the Daily load shape when Daily is defined.  The daily load shape is repeated in this case. Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Yearly`, DSS property index: 27.
        """
        return self._get_prop_string(27)

    @Yearly.setter
    def Yearly(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(27, value)
            return

        self._set_string_o(27, value)

    @property
    def Yearly_obj(self) -> LoadShape:
        """
        LOADSHAPE object to use for the per-unit voltage for YEARLY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.

        Must be previously defined as a LOADSHAPE object. 

        Is set to the Daily load shape when Daily is defined.  The daily load shape is repeated in this case. Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Yearly`, DSS property index: 27.
        """
        return self._get_obj(27, LoadShape)

    @Yearly_obj.setter
    def Yearly_obj(self, value: LoadShape):
        self._set_obj(27, value)

    @property
    def Daily(self) -> str:
        """
        LOADSHAPE object to use for the per-unit voltage for DAILY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.

        Must be previously defined as a LOADSHAPE object. 

        Sets Yearly curve if it is not already defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Daily`, DSS property index: 28.
        """
        return self._get_prop_string(28)

    @Daily.setter
    def Daily(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(28, value)
            return

        self._set_string_o(28, value)

    @property
    def Daily_obj(self) -> LoadShape:
        """
        LOADSHAPE object to use for the per-unit voltage for DAILY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.

        Must be previously defined as a LOADSHAPE object. 

        Sets Yearly curve if it is not already defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Daily`, DSS property index: 28.
        """
        return self._get_obj(28, LoadShape)

    @Daily_obj.setter
    def Daily_obj(self, value: LoadShape):
        self._set_obj(28, value)

    @property
    def Duty(self) -> str:
        """
        LOADSHAPE object to use for the per-unit voltage for DUTYCYCLE-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.

        Must be previously defined as a LOADSHAPE object. 

        Defaults to Daily load shape when Daily is defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Duty`, DSS property index: 29.
        """
        return self._get_prop_string(29)

    @Duty.setter
    def Duty(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(29, value)
            return

        self._set_string_o(29, value)

    @property
    def Duty_obj(self) -> LoadShape:
        """
        LOADSHAPE object to use for the per-unit voltage for DUTYCYCLE-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.

        Must be previously defined as a LOADSHAPE object. 

        Defaults to Daily load shape when Daily is defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Duty`, DSS property index: 29.
        """
        return self._get_obj(29, LoadShape)

    @Duty_obj.setter
    def Duty_obj(self, value: LoadShape):
        self._set_obj(29, value)

    @property
    def Model(self) -> VSourceModel:
        """
        {Thevenin* | Ideal}  Specifies whether the Vsource is to be considered a Thevenin short circuit model or a quasi-ideal voltage source. If Thevenin, the Vsource uses the impedances defined for all calculations. If "Ideal", the model uses a small impedance on the diagonal of the impedance matrix for the fundamental base frequency power flow only. Then switches to actual Thevenin model for other frequencies. 

        DSS property name: `Model`, DSS property index: 30.
        """
        return Vsource.VSourceModel(self._lib.Obj_GetInt32(self._ptr, 30))

    @Model.setter
    def Model(self, value: Union[AnyStr, int, VSourceModel]):
        if not isinstance(value, int):
            self._set_string_o(30, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 30, value)

    @property
    def Model_str(self) -> str:
        """
        {Thevenin* | Ideal}  Specifies whether the Vsource is to be considered a Thevenin short circuit model or a quasi-ideal voltage source. If Thevenin, the Vsource uses the impedances defined for all calculations. If "Ideal", the model uses a small impedance on the diagonal of the impedance matrix for the fundamental base frequency power flow only. Then switches to actual Thevenin model for other frequencies. 

        DSS property name: `Model`, DSS property index: 30.
        """
        return self._get_prop_string(30)

    @Model_str.setter
    def Model_str(self, value: AnyStr):
        self.Model = value

    @property
    def puZideal(self) -> complex:
        """
        2-element array: e.g., [1  2]. The pu impedance to use for the quasi-ideal voltage source model. Should be a very small impedances. Default is [1e-6, 0.001]. Per-unit impedance on base of Vsource BasekV and BaseMVA. If too small, solution may not work. Be sure to check the voltage values and powers.

        DSS property name: `puZideal`, DSS property index: 31.
        """
        return self._get_complex(31)

    @puZideal.setter
    def puZideal(self, value: complex):
        self._set_complex(31, value)

    @property
    def spectrum(self) -> str:
        """
        Name of harmonic spectrum for this source.  Default is "defaultvsource", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 32.
        """
        return self._get_prop_string(32)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum]):
        if isinstance(value, DSSObj):
            self._set_obj(32, value)
            return

        self._set_string_o(32, value)

    @property
    def spectrum_obj(self) -> Spectrum:
        """
        Name of harmonic spectrum for this source.  Default is "defaultvsource", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 32.
        """
        return self._get_obj(32, Spectrum)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_obj(32, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 33.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 33)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 33, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 34.
        """
        return self._lib.Obj_GetInt32(self._ptr, 34) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 34, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 35.
        """
        self._set_string_o(35, value)

class Isource(DSSObj):
    __slots__ = []
    _cls_name = 'Isource'
    _cls_idx = 17
    _cls_prop_idx = {
        'bus1': 1,
        'amps': 2,
        'angle': 3,
        'frequency': 4,
        'phases': 5,
        'scantype': 6,
        'sequence': 7,
        'yearly': 8,
        'daily': 9,
        'duty': 10,
        'bus2': 11,
        'spectrum': 12,
        'basefreq': 13,
        'enabled': 14,
        'like': 15,
    }

    @property
    def bus1(self) -> str:
        """
        Name of bus to which source is connected.
        bus1=busname
        bus1=busname.1.2.3

        DSS property name: `bus1`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(1, value)

    @property
    def amps(self) -> float:
        """
        Magnitude of current source, each phase, in Amps.

        DSS property name: `amps`, DSS property index: 2.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 2)

    @amps.setter
    def amps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 2, value)

    @property
    def angle(self) -> float:
        """
        Phase angle in degrees of first phase: e.g.,Angle=10.3.
        Phase shift between phases is assumed 120 degrees when number of phases <= 3

        DSS property name: `angle`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @angle.setter
    def angle(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def frequency(self) -> float:
        """
        Source frequency.  Defaults to  circuit fundamental frequency.

        DSS property name: `frequency`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @frequency.setter
    def frequency(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def phases(self) -> int:
        """
        Number of phases.  Defaults to 3. For 3 or less, phase shift is 120 degrees.

        DSS property name: `phases`, DSS property index: 5.
        """
        return self._lib.Obj_GetInt32(self._ptr, 5)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 5, value)

    @property
    def scantype(self) -> ScanType:
        """
        {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.

        DSS property name: `scantype`, DSS property index: 6.
        """
        return ScanType(self._lib.Obj_GetInt32(self._ptr, 6))

    @scantype.setter
    def scantype(self, value: Union[AnyStr, int, ScanType]):
        if not isinstance(value, int):
            self._set_string_o(6, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def scantype_str(self) -> str:
        """
        {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.

        DSS property name: `scantype`, DSS property index: 6.
        """
        return self._get_prop_string(6)

    @scantype_str.setter
    def scantype_str(self, value: AnyStr):
        self.scantype = value

    @property
    def sequence(self) -> SequenceType:
        """
        {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence. 

        DSS property name: `sequence`, DSS property index: 7.
        """
        return SequenceType(self._lib.Obj_GetInt32(self._ptr, 7))

    @sequence.setter
    def sequence(self, value: Union[AnyStr, int, SequenceType]):
        if not isinstance(value, int):
            self._set_string_o(7, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 7, value)

    @property
    def sequence_str(self) -> str:
        """
        {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence. 

        DSS property name: `sequence`, DSS property index: 7.
        """
        return self._get_prop_string(7)

    @sequence_str.setter
    def sequence_str(self, value: AnyStr):
        self.sequence = value

    @property
    def Yearly(self) -> str:
        """
        LOADSHAPE object to use for the per-unit current for YEARLY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual Amp.

        Must be previously defined as a LOADSHAPE object. 

        Is set to the Daily load shape when Daily is defined.  The daily load shape is repeated in this case. Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Yearly`, DSS property index: 8.
        """
        return self._get_prop_string(8)

    @Yearly.setter
    def Yearly(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(8, value)
            return

        self._set_string_o(8, value)

    @property
    def Yearly_obj(self) -> LoadShape:
        """
        LOADSHAPE object to use for the per-unit current for YEARLY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual Amp.

        Must be previously defined as a LOADSHAPE object. 

        Is set to the Daily load shape when Daily is defined.  The daily load shape is repeated in this case. Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Yearly`, DSS property index: 8.
        """
        return self._get_obj(8, LoadShape)

    @Yearly_obj.setter
    def Yearly_obj(self, value: LoadShape):
        self._set_obj(8, value)

    @property
    def Daily(self) -> str:
        """
        LOADSHAPE object to use for the per-unit current for DAILY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual A.

        Must be previously defined as a LOADSHAPE object. 

        Sets Yearly curve if it is not already defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Daily`, DSS property index: 9.
        """
        return self._get_prop_string(9)

    @Daily.setter
    def Daily(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(9, value)
            return

        self._set_string_o(9, value)

    @property
    def Daily_obj(self) -> LoadShape:
        """
        LOADSHAPE object to use for the per-unit current for DAILY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual A.

        Must be previously defined as a LOADSHAPE object. 

        Sets Yearly curve if it is not already defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Daily`, DSS property index: 9.
        """
        return self._get_obj(9, LoadShape)

    @Daily_obj.setter
    def Daily_obj(self, value: LoadShape):
        self._set_obj(9, value)

    @property
    def Duty(self) -> str:
        """
        LOADSHAPE object to use for the per-unit current for DUTYCYCLE-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual A.

        Must be previously defined as a LOADSHAPE object. 

        Defaults to Daily load shape when Daily is defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Duty`, DSS property index: 10.
        """
        return self._get_prop_string(10)

    @Duty.setter
    def Duty(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(10, value)
            return

        self._set_string_o(10, value)

    @property
    def Duty_obj(self) -> LoadShape:
        """
        LOADSHAPE object to use for the per-unit current for DUTYCYCLE-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual A.

        Must be previously defined as a LOADSHAPE object. 

        Defaults to Daily load shape when Daily is defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Duty`, DSS property index: 10.
        """
        return self._get_obj(10, LoadShape)

    @Duty_obj.setter
    def Duty_obj(self, value: LoadShape):
        self._set_obj(10, value)

    @property
    def Bus2(self) -> str:
        """
        Name of bus to which 2nd terminal is connected.
        bus2=busname
        bus2=busname.1.2.3

        Default is Bus1.0.0.0 (grounded-wye connection)

        DSS property name: `Bus2`, DSS property index: 11.
        """
        return self._get_prop_string(11)

    @Bus2.setter
    def Bus2(self, value: AnyStr):
        self._set_string_o(11, value)

    @property
    def spectrum(self) -> str:
        """
        Harmonic spectrum assumed for this source.  Default is "default".

        DSS property name: `spectrum`, DSS property index: 12.
        """
        return self._get_prop_string(12)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum]):
        if isinstance(value, DSSObj):
            self._set_obj(12, value)
            return

        self._set_string_o(12, value)

    @property
    def spectrum_obj(self) -> Spectrum:
        """
        Harmonic spectrum assumed for this source.  Default is "default".

        DSS property name: `spectrum`, DSS property index: 12.
        """
        return self._get_obj(12, Spectrum)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_obj(12, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 14.
        """
        return self._lib.Obj_GetInt32(self._ptr, 14) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 14, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 15.
        """
        self._set_string_o(15, value)

class VCCS(DSSObj):
    __slots__ = []
    _cls_name = 'VCCS'
    _cls_idx = 18
    _cls_prop_idx = {
        'bus1': 1,
        'phases': 2,
        'prated': 3,
        'vrated': 4,
        'ppct': 5,
        'bp1': 6,
        'bp2': 7,
        'filter': 8,
        'fsample': 9,
        'rmsmode': 10,
        'imaxpu': 11,
        'vrmstau': 12,
        'irmstau': 13,
        'spectrum': 14,
        'basefreq': 15,
        'enabled': 16,
        'like': 17,
    }

    @property
    def bus1(self) -> str:
        """
        Name of bus to which source is connected.
        bus1=busname
        bus1=busname.1.2.3

        DSS property name: `bus1`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(1, value)

    @property
    def phases(self) -> int:
        """
        Number of phases.  Defaults to 1.

        DSS property name: `phases`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def prated(self) -> float:
        """
        Total rated power, in Watts.

        DSS property name: `prated`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @prated.setter
    def prated(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def vrated(self) -> float:
        """
        Rated line-to-line voltage, in Volts

        DSS property name: `vrated`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @vrated.setter
    def vrated(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def ppct(self) -> float:
        """
        Steady-state operating output, in percent of rated.

        DSS property name: `ppct`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @ppct.setter
    def ppct(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def bp1(self) -> str:
        """
        XYCurve defining the input piece-wise linear block.

        DSS property name: `bp1`, DSS property index: 6.
        """
        return self._get_prop_string(6)

    @bp1.setter
    def bp1(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(6, value)
            return

        self._set_string_o(6, value)

    @property
    def bp1_obj(self) -> XYcurve:
        """
        XYCurve defining the input piece-wise linear block.

        DSS property name: `bp1`, DSS property index: 6.
        """
        return self._get_obj(6, XYcurve)

    @bp1_obj.setter
    def bp1_obj(self, value: XYcurve):
        self._set_obj(6, value)

    @property
    def bp2(self) -> str:
        """
        XYCurve defining the output piece-wise linear block.

        DSS property name: `bp2`, DSS property index: 7.
        """
        return self._get_prop_string(7)

    @bp2.setter
    def bp2(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(7, value)
            return

        self._set_string_o(7, value)

    @property
    def bp2_obj(self) -> XYcurve:
        """
        XYCurve defining the output piece-wise linear block.

        DSS property name: `bp2`, DSS property index: 7.
        """
        return self._get_obj(7, XYcurve)

    @bp2_obj.setter
    def bp2_obj(self, value: XYcurve):
        self._set_obj(7, value)

    @property
    def filter(self) -> str:
        """
        XYCurve defining the digital filter coefficients (x numerator, y denominator).

        DSS property name: `filter`, DSS property index: 8.
        """
        return self._get_prop_string(8)

    @filter.setter
    def filter(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(8, value)
            return

        self._set_string_o(8, value)

    @property
    def filter_obj(self) -> XYcurve:
        """
        XYCurve defining the digital filter coefficients (x numerator, y denominator).

        DSS property name: `filter`, DSS property index: 8.
        """
        return self._get_obj(8, XYcurve)

    @filter_obj.setter
    def filter_obj(self, value: XYcurve):
        self._set_obj(8, value)

    @property
    def fsample(self) -> float:
        """
        Sample frequency [Hz} for the digital filter.

        DSS property name: `fsample`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @fsample.setter
    def fsample(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def rmsmode(self) -> bool:
        """
        True if only Hz is used to represent a phase-locked loop (PLL), ignoring the BP1, BP2 and time-domain transformations. Default is no.

        DSS property name: `rmsmode`, DSS property index: 10.
        """
        return self._lib.Obj_GetInt32(self._ptr, 10) != 0

    @rmsmode.setter
    def rmsmode(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 10, value)

    @property
    def imaxpu(self) -> float:
        """
        Maximum output current in per-unit of rated; defaults to 1.1

        DSS property name: `imaxpu`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @imaxpu.setter
    def imaxpu(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def vrmstau(self) -> float:
        """
        Time constant in sensing Vrms for the PLL; defaults to 0.0015

        DSS property name: `vrmstau`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @vrmstau.setter
    def vrmstau(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def irmstau(self) -> float:
        """
        Time constant in producing Irms from the PLL; defaults to 0.0015

        DSS property name: `irmstau`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @irmstau.setter
    def irmstau(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def spectrum(self) -> str:
        """
        Harmonic spectrum assumed for this source.  Default is "default".

        DSS property name: `spectrum`, DSS property index: 14.
        """
        return self._get_prop_string(14)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum]):
        if isinstance(value, DSSObj):
            self._set_obj(14, value)
            return

        self._set_string_o(14, value)

    @property
    def spectrum_obj(self) -> Spectrum:
        """
        Harmonic spectrum assumed for this source.  Default is "default".

        DSS property name: `spectrum`, DSS property index: 14.
        """
        return self._get_obj(14, Spectrum)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_obj(14, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 16.
        """
        return self._lib.Obj_GetInt32(self._ptr, 16) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 16, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 17.
        """
        self._set_string_o(17, value)

class Load(DSSObj):
    __slots__ = []
    _cls_name = 'Load'
    _cls_idx = 19
    _cls_prop_idx = {
        'phases': 1,
        'bus1': 2,
        'kv': 3,
        'kw': 4,
        'pf': 5,
        'model': 6,
        'yearly': 7,
        'daily': 8,
        'duty': 9,
        'growth': 10,
        'conn': 11,
        'kvar': 12,
        'rneut': 13,
        'xneut': 14,
        'status': 15,
        'cls': 16,
        'class': 16,
        'vminpu': 17,
        'vmaxpu': 18,
        'vminnorm': 19,
        'vminemerg': 20,
        'xfkva': 21,
        'allocationfactor': 22,
        'kva': 23,
        'pctmean': 24,
        '%mean': 24,
        'pctstddev': 25,
        '%stddev': 25,
        'cvrwatts': 26,
        'cvrvars': 27,
        'kwh': 28,
        'kwhdays': 29,
        'cfactor': 30,
        'cvrcurve': 31,
        'numcust': 32,
        'zipv': 33,
        'pctseriesrl': 34,
        '%seriesrl': 34,
        'relweight': 35,
        'vlowpu': 36,
        'puxharm': 37,
        'xrharm': 38,
        'spectrum': 39,
        'basefreq': 40,
        'enabled': 41,
        'like': 42,
    }

    # Class-specific enumerations
    class LoadModel(IntEnum):
        """Load: Model (DSS enumeration for Load)"""
        ConstantPQ = 1 # Constant PQ
        ConstantZ = 2 # Constant Z
        Motor = 3 # Motor (constant P, quadratic Q)
        CVR = 4 # CVR (linear P, quadratic Q)
        ConstantI = 5 # Constant I
        ConstantP_fixedQ = 6 # Constant P, fixed Q
        ConstantP_fixedX = 7 # Constant P, fixed X
        ZIPV = 8 # ZIPV

    class LoadStatus(IntEnum):
        """Load: Status (DSS enumeration for Load)"""
        Variable = 0 # Variable
        Fixed = 1 # Fixed
        Exempt = 2 # Exempt


    @property
    def phases(self) -> int:
        """
        Number of Phases, this load.  Load is evenly divided among phases.

        DSS property name: `phases`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def bus1(self) -> str:
        """
        Bus to which the load is connected.  May include specific node specification.

        DSS property name: `bus1`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(2, value)

    @property
    def kV(self) -> float:
        """
        Nominal rated (1.0 per unit) voltage, kV, for load. For 2- and 3-phase loads, specify phase-phase kV. Otherwise, specify actual kV across each branch of the load. If wye (star), specify phase-neutral kV. If delta or phase-phase connected, specify phase-phase kV.

        DSS property name: `kV`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @kV.setter
    def kV(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def kW(self) -> float:
        """
        Total base kW for the load.  Normally, you would enter the maximum kW for the load for the first year and allow it to be adjusted by the load shapes, growth shapes, and global load multiplier.

        Legal ways to define base load:
        kW, PF
        kW, kvar
        kVA, PF
        XFKVA * Allocationfactor, PF
        kWh/(kWhdays*24) * Cfactor, PF

        DSS property name: `kW`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @kW.setter
    def kW(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def pf(self) -> float:
        """
        Load power factor.  Enter negative for leading powerfactor (when kW and kvar have opposite signs.)

        DSS property name: `pf`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @pf.setter
    def pf(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def model(self) -> LoadModel:
        """
        Integer code for the model to use for load variation with voltage. Valid values are:

        1:Standard constant P+jQ load. (Default)
        2:Constant impedance load. 
        3:Const P, Quadratic Q (like a motor).
        4:Nominal Linear P, Quadratic Q (feeder mix). Use this with CVRfactor.
        5:Constant Current Magnitude
        6:Const P, Fixed Q
        7:Const P, Fixed Impedance Q
        8:ZIPV (7 values)

        For Types 6 and 7, only the P is modified by load multipliers.

        DSS property name: `model`, DSS property index: 6.
        """
        return Load.LoadModel(self._lib.Obj_GetInt32(self._ptr, 6))

    @model.setter
    def model(self, value: Union[int, LoadModel]):
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def yearly(self) -> str:
        """
        LOADSHAPE object to use for yearly simulations.  Must be previously defined as a Loadshape object. Is set to the Daily load shape  when Daily is defined.  The daily load shape is repeated in this case. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. The default is no variation.

        DSS property name: `yearly`, DSS property index: 7.
        """
        return self._get_prop_string(7)

    @yearly.setter
    def yearly(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(7, value)
            return

        self._set_string_o(7, value)

    @property
    def yearly_obj(self) -> LoadShape:
        """
        LOADSHAPE object to use for yearly simulations.  Must be previously defined as a Loadshape object. Is set to the Daily load shape  when Daily is defined.  The daily load shape is repeated in this case. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. The default is no variation.

        DSS property name: `yearly`, DSS property index: 7.
        """
        return self._get_obj(7, LoadShape)

    @yearly_obj.setter
    def yearly_obj(self, value: LoadShape):
        self._set_obj(7, value)

    @property
    def daily(self) -> str:
        """
        LOADSHAPE object to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Default is no variation (constant) if not defined. Side effect: Sets Yearly load shape if not already defined.

        DSS property name: `daily`, DSS property index: 8.
        """
        return self._get_prop_string(8)

    @daily.setter
    def daily(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(8, value)
            return

        self._set_string_o(8, value)

    @property
    def daily_obj(self) -> LoadShape:
        """
        LOADSHAPE object to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Default is no variation (constant) if not defined. Side effect: Sets Yearly load shape if not already defined.

        DSS property name: `daily`, DSS property index: 8.
        """
        return self._get_obj(8, LoadShape)

    @daily_obj.setter
    def daily_obj(self, value: LoadShape):
        self._set_obj(8, value)

    @property
    def duty(self) -> str:
        """
        LOADSHAPE object to use for duty cycle simulations.  Must be previously defined as a Loadshape object.  Typically would have time intervals less than 1 hr. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.Set to NONE to reset to no loadahape. Set Status=Fixed to ignore Loadshape designation.  Defaults to Daily curve If not specified.

        DSS property name: `duty`, DSS property index: 9.
        """
        return self._get_prop_string(9)

    @duty.setter
    def duty(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(9, value)
            return

        self._set_string_o(9, value)

    @property
    def duty_obj(self) -> LoadShape:
        """
        LOADSHAPE object to use for duty cycle simulations.  Must be previously defined as a Loadshape object.  Typically would have time intervals less than 1 hr. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.Set to NONE to reset to no loadahape. Set Status=Fixed to ignore Loadshape designation.  Defaults to Daily curve If not specified.

        DSS property name: `duty`, DSS property index: 9.
        """
        return self._get_obj(9, LoadShape)

    @duty_obj.setter
    def duty_obj(self, value: LoadShape):
        self._set_obj(9, value)

    @property
    def growth(self) -> str:
        """
        Characteristic  to use for growth factors by years.  Must be previously defined as a Growthshape object. Defaults to circuit default growth factor (see Set Growth command).

        DSS property name: `growth`, DSS property index: 10.
        """
        return self._get_prop_string(10)

    @growth.setter
    def growth(self, value: Union[AnyStr, GrowthShape]):
        if isinstance(value, DSSObj):
            self._set_obj(10, value)
            return

        self._set_string_o(10, value)

    @property
    def growth_obj(self) -> GrowthShape:
        """
        Characteristic  to use for growth factors by years.  Must be previously defined as a Growthshape object. Defaults to circuit default growth factor (see Set Growth command).

        DSS property name: `growth`, DSS property index: 10.
        """
        return self._get_obj(10, GrowthShape)

    @growth_obj.setter
    def growth_obj(self, value: GrowthShape):
        self._set_obj(10, value)

    @property
    def conn(self) -> Connection:
        """
        ={wye or LN | delta or LL}.  Default is wye.

        DSS property name: `conn`, DSS property index: 11.
        """
        return Connection(self._lib.Obj_GetInt32(self._ptr, 11))

    @conn.setter
    def conn(self, value: Union[AnyStr, int, Connection]):
        if not isinstance(value, int):
            self._set_string_o(11, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 11, value)

    @property
    def conn_str(self) -> str:
        """
        ={wye or LN | delta or LL}.  Default is wye.

        DSS property name: `conn`, DSS property index: 11.
        """
        return self._get_prop_string(11)

    @conn_str.setter
    def conn_str(self, value: AnyStr):
        self.conn = value

    @property
    def kvar(self) -> float:
        """
        Specify the base kvar for specifying load as kW & kvar.  Assumes kW has been already defined.  Alternative to specifying the power factor.  Side effect:  the power factor and kVA is altered to agree.

        DSS property name: `kvar`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @kvar.setter
    def kvar(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def Rneut(self) -> float:
        """
        Default is -1. Neutral resistance of wye (star)-connected load in actual ohms. If entered as a negative value, the neutral can be open, or floating, or it can be connected to node 0 (ground), which is the usual default. If >=0 be sure to explicitly specify the node connection for the neutral, or last, conductor. Otherwise, the neutral impedance will be shorted to ground.

        DSS property name: `Rneut`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @Rneut.setter
    def Rneut(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def Xneut(self) -> float:
        """
        Neutral reactance of wye(star)-connected load in actual ohms.  May be + or -.

        DSS property name: `Xneut`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @Xneut.setter
    def Xneut(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def status(self) -> LoadStatus:
        """
        ={Variable | Fixed | Exempt}.  Default is variable. If Fixed, no load multipliers apply;  however, growth multipliers do apply.  All multipliers apply to Variable loads.  Exempt loads are not modified by the global load multiplier, such as in load duration curves, etc.  Daily multipliers do apply, so setting this property to Exempt is a good way to represent industrial load that stays the same day-after-day for the period study.

        DSS property name: `status`, DSS property index: 15.
        """
        return Load.LoadStatus(self._lib.Obj_GetInt32(self._ptr, 15))

    @status.setter
    def status(self, value: Union[AnyStr, int, LoadStatus]):
        if not isinstance(value, int):
            self._set_string_o(15, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 15, value)

    @property
    def status_str(self) -> str:
        """
        ={Variable | Fixed | Exempt}.  Default is variable. If Fixed, no load multipliers apply;  however, growth multipliers do apply.  All multipliers apply to Variable loads.  Exempt loads are not modified by the global load multiplier, such as in load duration curves, etc.  Daily multipliers do apply, so setting this property to Exempt is a good way to represent industrial load that stays the same day-after-day for the period study.

        DSS property name: `status`, DSS property index: 15.
        """
        return self._get_prop_string(15)

    @status_str.setter
    def status_str(self, value: AnyStr):
        self.status = value

    @property
    def cls(self) -> int:
        """
        An arbitrary integer number representing the class of load so that load values may be segregated by load value. Default is 1; not used internally.

        DSS property name: `class`, DSS property index: 16.
        """
        return self._lib.Obj_GetInt32(self._ptr, 16)

    @cls.setter
    def cls(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 16, value)

    @property
    def Vminpu(self) -> float:
        """
        Default = 0.95.  Minimum per unit voltage for which the MODEL is assumed to apply. Lower end of normal voltage range.Below this value, the load model reverts to a constant impedance model that matches the model at the transition voltage. See also "Vlowpu" which causes the model to match Model=2 below the transition voltage.

        DSS property name: `Vminpu`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @Vminpu.setter
    def Vminpu(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def Vmaxpu(self) -> float:
        """
        Default = 1.05.  Maximum per unit voltage for which the MODEL is assumed to apply. Above this value, the load model reverts to a constant impedance model.

        DSS property name: `Vmaxpu`, DSS property index: 18.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 18)

    @Vmaxpu.setter
    def Vmaxpu(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 18, value)

    @property
    def Vminnorm(self) -> float:
        """
        Minimum per unit voltage for load EEN evaluations, Normal limit.  Default = 0, which defaults to system "vminnorm" property (see Set Command under Executive).  If this property is specified, it ALWAYS overrides the system specification. This allows you to have different criteria for different loads. Set to zero to revert to the default system value.

        DSS property name: `Vminnorm`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @Vminnorm.setter
    def Vminnorm(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def Vminemerg(self) -> float:
        """
        Minimum per unit voltage for load UE evaluations, Emergency limit.  Default = 0, which defaults to system "vminemerg" property (see Set Command under Executive).  If this property is specified, it ALWAYS overrides the system specification. This allows you to have different criteria for different loads. Set to zero to revert to the default system value.

        DSS property name: `Vminemerg`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @Vminemerg.setter
    def Vminemerg(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    @property
    def xfkVA(self) -> float:
        """
        Default = 0.0.  Rated kVA of service transformer for allocating loads based on connected kVA at a bus. Side effect:  kW, PF, and kvar are modified. See help on kVA.

        DSS property name: `xfkVA`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @xfkVA.setter
    def xfkVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def allocationfactor(self) -> float:
        """
        Default = 0.5.  Allocation factor for allocating loads based on connected kVA at a bus. Side effect:  kW, PF, and kvar are modified by multiplying this factor times the XFKVA (if > 0).

        DSS property name: `allocationfactor`, DSS property index: 22.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 22)

    @allocationfactor.setter
    def allocationfactor(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 22, value)

    @property
    def kVA(self) -> float:
        """
        Specify base Load in kVA (and power factor)

        Legal ways to define base load:
        kW, PF
        kW, kvar
        kVA, PF
        XFKVA * Allocationfactor, PF
        kWh/(kWhdays*24) * Cfactor, PF

        DSS property name: `kVA`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @kVA.setter
    def kVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def pctmean(self) -> float:
        """
        Percent mean value for load to use for monte carlo studies if no loadshape is assigned to this load. Default is 50.

        DSS property name: `%mean`, DSS property index: 24.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 24)

    @pctmean.setter
    def pctmean(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 24, value)

    @property
    def pctstddev(self) -> float:
        """
        Percent Std deviation value for load to use for monte carlo studies if no loadshape is assigned to this load. Default is 10.

        DSS property name: `%stddev`, DSS property index: 25.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 25)

    @pctstddev.setter
    def pctstddev(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 25, value)

    @property
    def CVRwatts(self) -> float:
        """
        Percent reduction in active power (watts) per 1% reduction in voltage from 100% rated. Default=1. 
         Typical values range from 0.4 to 0.8. Applies to Model=4 only.
         Intended to represent conservation voltage reduction or voltage optimization measures.

        DSS property name: `CVRwatts`, DSS property index: 26.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 26)

    @CVRwatts.setter
    def CVRwatts(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 26, value)

    @property
    def CVRvars(self) -> float:
        """
        Percent reduction in reactive power (vars) per 1% reduction in voltage from 100% rated. Default=2. 
         Typical values range from 2 to 3. Applies to Model=4 only.
         Intended to represent conservation voltage reduction or voltage optimization measures.

        DSS property name: `CVRvars`, DSS property index: 27.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 27)

    @CVRvars.setter
    def CVRvars(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 27, value)

    @property
    def kwh(self) -> float:
        """
        kWh billed for this period. Default is 0. See help on kVA and Cfactor and kWhDays.

        DSS property name: `kwh`, DSS property index: 28.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 28)

    @kwh.setter
    def kwh(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 28, value)

    @property
    def kwhdays(self) -> float:
        """
        Length of kWh billing period in days (24 hr days). Default is 30. Average demand is computed using this value.

        DSS property name: `kwhdays`, DSS property index: 29.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 29)

    @kwhdays.setter
    def kwhdays(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 29, value)

    @property
    def Cfactor(self) -> float:
        """
        Factor relating average kW to peak kW. Default is 4.0. See kWh and kWhdays. See kVA.

        DSS property name: `Cfactor`, DSS property index: 30.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 30)

    @Cfactor.setter
    def Cfactor(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 30, value)

    @property
    def CVRcurve(self) -> str:
        """
        Default is NONE. Curve describing both watt and var factors as a function of time. Refers to a LoadShape object with both Mult and Qmult defined. Define a Loadshape to agree with yearly or daily curve according to the type of analysis being done. If NONE, the CVRwatts and CVRvars factors are used and assumed constant.

        DSS property name: `CVRcurve`, DSS property index: 31.
        """
        return self._get_prop_string(31)

    @CVRcurve.setter
    def CVRcurve(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(31, value)
            return

        self._set_string_o(31, value)

    @property
    def CVRcurve_obj(self) -> LoadShape:
        """
        Default is NONE. Curve describing both watt and var factors as a function of time. Refers to a LoadShape object with both Mult and Qmult defined. Define a Loadshape to agree with yearly or daily curve according to the type of analysis being done. If NONE, the CVRwatts and CVRvars factors are used and assumed constant.

        DSS property name: `CVRcurve`, DSS property index: 31.
        """
        return self._get_obj(31, LoadShape)

    @CVRcurve_obj.setter
    def CVRcurve_obj(self, value: LoadShape):
        self._set_obj(31, value)

    @property
    def NumCust(self) -> int:
        """
        Number of customers, this load. Default is 1.

        DSS property name: `NumCust`, DSS property index: 32.
        """
        return self._lib.Obj_GetInt32(self._ptr, 32)

    @NumCust.setter
    def NumCust(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 32, value)

    @property
    def ZIPV(self) -> Float64Array:
        """
        Array of 7 coefficients:

         First 3 are ZIP weighting factors for real power (should sum to 1)
         Next 3 are ZIP weighting factors for reactive power (should sum to 1)
         Last 1 is cut-off voltage in p.u. of base kV; load is 0 below this cut-off
         No defaults; all coefficients must be specified if using model=8.

        DSS property name: `ZIPV`, DSS property index: 33.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 33)

    @ZIPV.setter
    def ZIPV(self, value: Float64Array):
        self._set_float64_array_o(33, value)

    @property
    def pctSeriesRL(self) -> float:
        """
        Percent of load that is series R-L for Harmonic studies. Default is 50. Remainder is assumed to be parallel R and L. This can have a significant impact on the amount of damping observed in Harmonics solutions.

        DSS property name: `%SeriesRL`, DSS property index: 34.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 34)

    @pctSeriesRL.setter
    def pctSeriesRL(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 34, value)

    @property
    def RelWeight(self) -> float:
        """
        Relative weighting factor for reliability calcs. Default = 1. Used to designate high priority loads such as hospitals, etc. 

        Is multiplied by number of customers and load kW during reliability calcs.

        DSS property name: `RelWeight`, DSS property index: 35.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 35)

    @RelWeight.setter
    def RelWeight(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 35, value)

    @property
    def Vlowpu(self) -> float:
        """
        Default = 0.50.  Per unit voltage at which the model switches to same as constant Z model (model=2). This allows more consistent convergence at very low voltaes due to opening switches or solving for fault situations.

        DSS property name: `Vlowpu`, DSS property index: 36.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 36)

    @Vlowpu.setter
    def Vlowpu(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 36, value)

    @property
    def puXharm(self) -> float:
        """
        Special reactance, pu (based on kVA, kV properties), for the series impedance branch in the load model for HARMONICS analysis. Generally used to represent motor load blocked rotor reactance. If not specified (that is, set =0, the default value), the series branch is computed from the percentage of the nominal load at fundamental frequency specified by the %SERIESRL property. 

        Applies to load model in HARMONICS mode only.

        A typical value would be approximately 0.20 pu based on kVA * %SeriesRL / 100.0.

        DSS property name: `puXharm`, DSS property index: 37.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 37)

    @puXharm.setter
    def puXharm(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 37, value)

    @property
    def XRharm(self) -> float:
        """
        X/R ratio of the special harmonics mode reactance specified by the puXHARM property at fundamental frequency. Default is 6. 

        DSS property name: `XRharm`, DSS property index: 38.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 38)

    @XRharm.setter
    def XRharm(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 38, value)

    @property
    def spectrum(self) -> str:
        """
        Name of harmonic current spectrum for this load.  Default is "defaultload", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 39.
        """
        return self._get_prop_string(39)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum]):
        if isinstance(value, DSSObj):
            self._set_obj(39, value)
            return

        self._set_string_o(39, value)

    @property
    def spectrum_obj(self) -> Spectrum:
        """
        Name of harmonic current spectrum for this load.  Default is "defaultload", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 39.
        """
        return self._get_obj(39, Spectrum)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_obj(39, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 40.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 40)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 40, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 41.
        """
        return self._lib.Obj_GetInt32(self._ptr, 41) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 41, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 42.
        """
        self._set_string_o(42, value)

class Transformer(DSSObj):
    __slots__ = []
    _cls_name = 'Transformer'
    _cls_idx = 20
    _cls_prop_idx = {
        'phases': 1,
        'windings': 2,
        'wdg': 3,
        'bus': 4,
        'conn': 5,
        'kv': 6,
        'kva': 7,
        'tap': 8,
        'pctr': 9,
        '%r': 9,
        'rneut': 10,
        'xneut': 11,
        'buses': 12,
        'conns': 13,
        'kvs': 14,
        'kvas': 15,
        'taps': 16,
        'xhl': 17,
        'xht': 18,
        'xlt': 19,
        'xscarray': 20,
        'thermal': 21,
        'n': 22,
        'm': 23,
        'flrise': 24,
        'hsrise': 25,
        'pctloadloss': 26,
        '%loadloss': 26,
        'pctnoloadloss': 27,
        '%noloadloss': 27,
        'normhkva': 28,
        'emerghkva': 29,
        'sub': 30,
        'maxtap': 31,
        'mintap': 32,
        'numtaps': 33,
        'subname': 34,
        'pctimag': 35,
        '%imag': 35,
        'ppm_antifloat': 36,
        'pctrs': 37,
        '%rs': 37,
        'bank': 38,
        'xfmrcode': 39,
        'xrconst': 40,
        'x12': 41,
        'x13': 42,
        'x23': 43,
        'leadlag': 44,
        'wdgcurrents': 45,
        'core': 46,
        'rdcohms': 47,
        'seasons': 48,
        'ratings': 49,
        'normamps': 50,
        'emergamps': 51,
        'faultrate': 52,
        'pctperm': 53,
        'repair': 54,
        'basefreq': 55,
        'enabled': 56,
        'like': 57,
    }

    @property
    def phases(self) -> int:
        """
        Number of phases this transformer. Default is 3.

        DSS property name: `phases`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def windings(self) -> int:
        """
        Number of windings, this transformers. (Also is the number of terminals) Default is 2. This property triggers memory allocation for the Transformer and will cause other properties to revert to default values.

        DSS property name: `windings`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @windings.setter
    def windings(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def pctR(self) -> Float64Array:
        """
        Percent resistance this winding.  (half of total for a 2-winding).

        DSS property name: `%R`, DSS property index: 9.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 9)

    @pctR.setter
    def pctR(self, value: Float64Array):
        self._set_float64_array_o(9, value)

    @property
    def Rneut(self) -> Float64Array:
        """
        Default = -1. Neutral resistance of wye (star)-connected winding in actual ohms. If entered as a negative value, the neutral is assumed to be open, or floating. To solidly ground the neutral, connect the neutral conductor to Node 0 in the Bus property spec for this winding. For example: Bus=MyBusName.1.2.3.0, which is generally the default connection.

        DSS property name: `Rneut`, DSS property index: 10.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 10)

    @Rneut.setter
    def Rneut(self, value: Float64Array):
        self._set_float64_array_o(10, value)

    @property
    def Xneut(self) -> Float64Array:
        """
        Neutral reactance of wye(star)-connected winding in actual ohms.  May be + or -.

        DSS property name: `Xneut`, DSS property index: 11.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 11)

    @Xneut.setter
    def Xneut(self, value: Float64Array):
        self._set_float64_array_o(11, value)

    @property
    def buses(self) -> List[str]:
        """
        Use this to specify all the bus connections at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus"

        DSS property name: `buses`, DSS property index: 12.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 12)

    @buses.setter
    def buses(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 12, value_ptr, value_count)
        self._check_for_error()

    @property
    def conns(self) -> List[Connection]:
        """
        Use this to specify all the Winding connections at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus" ~ conns=(delta, wye)

        DSS property name: `conns`, DSS property index: 13.
        """
        return [Connection(val) for val in self._get_int32_list(self._lib.Obj_GetInt32Array, self._ptr, 13)]

    @conns.setter
    def conns(self, value: Union[List[Union[int,Connection]], List[AnyStr]]):
        if len(value) and not isinstance(value[0], int):
            self._set_string_array_o(13, value)
            return    
        self._set_int32_array_o(13, value)

    @property
    def conns_str(self) -> List[str]:
        """
        Use this to specify all the Winding connections at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus" ~ conns=(delta, wye)

        DSS property name: `conns`, DSS property index: 13.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 13)

    @conns_str.setter
    def conns_str(self, value: AnyStr):
        self.conns = value

    @property
    def kVs(self) -> Float64Array:
        """
        Use this to specify the kV ratings of all windings at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus" 
        ~ conns=(delta, wye)
        ~ kvs=(115, 12.47)

        See kV= property for voltage rules.

        DSS property name: `kVs`, DSS property index: 14.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 14)

    @kVs.setter
    def kVs(self, value: Float64Array):
        self._set_float64_array_o(14, value)

    @property
    def kVAs(self) -> Float64Array:
        """
        Use this to specify the kVA ratings of all windings at once using an array.

        DSS property name: `kVAs`, DSS property index: 15.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 15)

    @kVAs.setter
    def kVAs(self, value: Float64Array):
        self._set_float64_array_o(15, value)

    @property
    def taps(self) -> Float64Array:
        """
        Use this to specify the p.u. tap of all windings at once using an array.

        DSS property name: `taps`, DSS property index: 16.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 16)

    @taps.setter
    def taps(self, value: Float64Array):
        self._set_float64_array_o(16, value)

    @property
    def XHL(self) -> float:
        """
        Use this to specify the percent reactance, H-L (winding 1 to winding 2).  Use for 2- or 3-winding transformers. On the kVA base of winding 1. See also X12.

        DSS property name: `XHL`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @XHL.setter
    def XHL(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def XHT(self) -> float:
        """
        Use this to specify the percent reactance, H-T (winding 1 to winding 3).  Use for 3-winding transformers only. On the kVA base of winding 1. See also X13.

        DSS property name: `XHT`, DSS property index: 18.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 18)

    @XHT.setter
    def XHT(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 18, value)

    @property
    def XLT(self) -> float:
        """
        Use this to specify the percent reactance, L-T (winding 2 to winding 3).  Use for 3-winding transformers only. On the kVA base of winding 1.  See also X23.

        DSS property name: `XLT`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @XLT.setter
    def XLT(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def Xscarray(self) -> Float64Array:
        """
        Use this to specify the percent reactance between all pairs of windings as an array. All values are on the kVA base of winding 1.  The order of the values is as follows:

        (x12 13 14... 23 24.. 34 ..)  

        There will be n(n-1)/2 values, where n=number of windings.

        DSS property name: `Xscarray`, DSS property index: 20.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 20)

    @Xscarray.setter
    def Xscarray(self, value: Float64Array):
        self._set_float64_array_o(20, value)

    @property
    def thermal(self) -> float:
        """
        Thermal time constant of the transformer in hours.  Typically about 2.

        DSS property name: `thermal`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @thermal.setter
    def thermal(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def n(self) -> float:
        """
        n Exponent for thermal properties in IEEE C57.  Typically 0.8.

        DSS property name: `n`, DSS property index: 22.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 22)

    @n.setter
    def n(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 22, value)

    @property
    def m(self) -> float:
        """
        m Exponent for thermal properties in IEEE C57.  Typically 0.9 - 1.0

        DSS property name: `m`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @m.setter
    def m(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def flrise(self) -> float:
        """
        Temperature rise, deg C, for full load.  Default is 65.

        DSS property name: `flrise`, DSS property index: 24.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 24)

    @flrise.setter
    def flrise(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 24, value)

    @property
    def hsrise(self) -> float:
        """
        Hot spot temperature rise, deg C.  Default is 15.

        DSS property name: `hsrise`, DSS property index: 25.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 25)

    @hsrise.setter
    def hsrise(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 25, value)

    @property
    def pctloadloss(self) -> float:
        """
        Percent load loss at full load. The %R of the High and Low windings (1 and 2) are adjusted to agree at rated kVA loading.

        DSS property name: `%loadloss`, DSS property index: 26.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 26)

    @pctloadloss.setter
    def pctloadloss(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 26, value)

    @property
    def pctnoloadloss(self) -> float:
        """
        Percent no load losses at rated excitatation voltage. Default is 0. Converts to a resistance in parallel with the magnetizing impedance in each winding.

        DSS property name: `%noloadloss`, DSS property index: 27.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 27)

    @pctnoloadloss.setter
    def pctnoloadloss(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 27, value)

    @property
    def normhkVA(self) -> float:
        """
        Normal maximum kVA rating of H winding (winding 1).  Usually 100% - 110% ofmaximum nameplate rating, depending on load shape. Defaults to 110% of kVA rating of Winding 1.

        DSS property name: `normhkVA`, DSS property index: 28.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 28)

    @normhkVA.setter
    def normhkVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 28, value)

    @property
    def emerghkVA(self) -> float:
        """
        Emergency (contingency)  kVA rating of H winding (winding 1).  Usually 140% - 150% ofmaximum nameplate rating, depending on load shape. Defaults to 150% of kVA rating of Winding 1.

        DSS property name: `emerghkVA`, DSS property index: 29.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 29)

    @emerghkVA.setter
    def emerghkVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 29, value)

    @property
    def sub(self) -> bool:
        """
        ={Yes|No}  Designates whether this transformer is to be considered a substation.Default is No.

        DSS property name: `sub`, DSS property index: 30.
        """
        return self._lib.Obj_GetInt32(self._ptr, 30) != 0

    @sub.setter
    def sub(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 30, value)

    @property
    def MaxTap(self) -> Float64Array:
        """
        Max per unit tap for the active winding.  Default is 1.10

        DSS property name: `MaxTap`, DSS property index: 31.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 31)

    @MaxTap.setter
    def MaxTap(self, value: Float64Array):
        self._set_float64_array_o(31, value)

    @property
    def MinTap(self) -> Float64Array:
        """
        Min per unit tap for the active winding.  Default is 0.90

        DSS property name: `MinTap`, DSS property index: 32.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 32)

    @MinTap.setter
    def MinTap(self, value: Float64Array):
        self._set_float64_array_o(32, value)

    @property
    def NumTaps(self) -> Int32Array:
        """
        Total number of taps between min and max tap.  Default is 32 (16 raise and 16 lower taps about the neutral position). The neutral position is not counted.

        DSS property name: `NumTaps`, DSS property index: 33.
        """
        return self._get_int32_array(self._lib.Obj_GetInt32Array, self._ptr, 33)

    @NumTaps.setter
    def NumTaps(self, value: Int32Array):
        self._set_int32_array_o(33, value)

    @property
    def subname(self) -> str:
        """
        Substation Name. Optional. Default is null. If specified, printed on plots

        DSS property name: `subname`, DSS property index: 34.
        """
        return self._get_prop_string(34)

    @subname.setter
    def subname(self, value: AnyStr):
        self._set_string_o(34, value)

    @property
    def pctimag(self) -> float:
        """
        Percent magnetizing current. Default=0.0. Magnetizing branch is in parallel with windings in each phase. Also, see "ppm_antifloat".

        DSS property name: `%imag`, DSS property index: 35.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 35)

    @pctimag.setter
    def pctimag(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 35, value)

    @property
    def ppm_antifloat(self) -> float:
        """
        Default=1 ppm.  Parts per million of transformer winding VA rating connected to ground to protect against accidentally floating a winding without a reference. If positive then the effect is adding a very large reactance to ground.  If negative, then a capacitor.

        DSS property name: `ppm_antifloat`, DSS property index: 36.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 36)

    @ppm_antifloat.setter
    def ppm_antifloat(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 36, value)

    @property
    def pctRs(self) -> Float64Array:
        """
        Use this property to specify all the winding %resistances using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus" ~ %Rs=(0.2  0.3)

        DSS property name: `%Rs`, DSS property index: 37.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 37)

    @pctRs.setter
    def pctRs(self, value: Float64Array):
        self._set_float64_array_o(37, value)

    @property
    def bank(self) -> str:
        """
        Name of the bank this transformer is part of, for CIM, MultiSpeak, and other interfaces.

        DSS property name: `bank`, DSS property index: 38.
        """
        return self._get_prop_string(38)

    @bank.setter
    def bank(self, value: AnyStr):
        self._set_string_o(38, value)

    @property
    def xfmrcode(self) -> str:
        """
        Name of a library entry for transformer properties. The named XfmrCode must already be defined.

        DSS property name: `XfmrCode`, DSS property index: 39.
        """
        return self._get_prop_string(39)

    @xfmrcode.setter
    def xfmrcode(self, value: Union[AnyStr, XfmrCode]):
        if isinstance(value, DSSObj):
            self._set_obj(39, value)
            return

        self._set_string_o(39, value)

    @property
    def xfmrcode_obj(self) -> XfmrCode:
        """
        Name of a library entry for transformer properties. The named XfmrCode must already be defined.

        DSS property name: `XfmrCode`, DSS property index: 39.
        """
        return self._get_obj(39, XfmrCode)

    @xfmrcode_obj.setter
    def xfmrcode_obj(self, value: XfmrCode):
        self._set_obj(39, value)

    @property
    def XRConst(self) -> bool:
        """
        ={Yes|No} Default is NO. Signifies whether or not the X/R is assumed contant for harmonic studies.

        DSS property name: `XRConst`, DSS property index: 40.
        """
        return self._lib.Obj_GetInt32(self._ptr, 40) != 0

    @XRConst.setter
    def XRConst(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 40, value)

    @property
    def X12(self) -> float:
        """
        Alternative to XHL for specifying the percent reactance from winding 1 to winding 2.  Use for 2- or 3-winding transformers. Percent on the kVA base of winding 1. 

        DSS property name: `X12`, DSS property index: 41.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 41)

    @X12.setter
    def X12(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 41, value)

    @property
    def X13(self) -> float:
        """
        Alternative to XHT for specifying the percent reactance from winding 1 to winding 3.  Use for 3-winding transformers only. Percent on the kVA base of winding 1. 

        DSS property name: `X13`, DSS property index: 42.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 42)

    @X13.setter
    def X13(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 42, value)

    @property
    def X23(self) -> float:
        """
        Alternative to XLT for specifying the percent reactance from winding 2 to winding 3.Use for 3-winding transformers only. Percent on the kVA base of winding 1.  

        DSS property name: `X23`, DSS property index: 43.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 43)

    @X23.setter
    def X23(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 43, value)

    @property
    def LeadLag(self) -> PhaseSequence:
        """
        {Lead | Lag (default) | ANSI (default) | Euro } Designation in mixed Delta-wye connections the relationship between HV to LV winding. Default is ANSI 30 deg lag, e.g., Dy1 of Yd1 vector group. To get typical European Dy11 connection, specify either "lead" or "Euro"

        DSS property name: `LeadLag`, DSS property index: 44.
        """
        return PhaseSequence(self._lib.Obj_GetInt32(self._ptr, 44))

    @LeadLag.setter
    def LeadLag(self, value: Union[AnyStr, int, PhaseSequence]):
        if not isinstance(value, int):
            self._set_string_o(44, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 44, value)

    @property
    def LeadLag_str(self) -> str:
        """
        {Lead | Lag (default) | ANSI (default) | Euro } Designation in mixed Delta-wye connections the relationship between HV to LV winding. Default is ANSI 30 deg lag, e.g., Dy1 of Yd1 vector group. To get typical European Dy11 connection, specify either "lead" or "Euro"

        DSS property name: `LeadLag`, DSS property index: 44.
        """
        return self._get_prop_string(44)

    @LeadLag_str.setter
    def LeadLag_str(self, value: AnyStr):
        self.LeadLag = value

    def WdgCurrents(self) -> str:
        """
        (Read only) Makes winding currents available via return on query (? Transformer.TX.WdgCurrents). Order: Phase 1, Wdg 1, Wdg 2, ..., Phase 2 ...

        DSS property name: `WdgCurrents`, DSS property index: 45.
        """
        # []
        # StringSilentROFunction
        return self._get_prop_string(self._lib.Obj_GetString(self._ptr, 45))

    @property
    def Core(self) -> CoreType:
        """
        {Shell*|5-leg|3-Leg|1-phase|core-1-phase|4-leg} Core Type. Used for GIC analysis

        DSS property name: `Core`, DSS property index: 46.
        """
        return CoreType(self._lib.Obj_GetInt32(self._ptr, 46))

    @Core.setter
    def Core(self, value: Union[AnyStr, int, CoreType]):
        if not isinstance(value, int):
            self._set_string_o(46, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 46, value)

    @property
    def Core_str(self) -> str:
        """
        {Shell*|5-leg|3-Leg|1-phase|core-1-phase|4-leg} Core Type. Used for GIC analysis

        DSS property name: `Core`, DSS property index: 46.
        """
        return self._get_prop_string(46)

    @Core_str.setter
    def Core_str(self, value: AnyStr):
        self.Core = value

    @property
    def RdcOhms(self) -> Float64Array:
        """
        Winding dc resistance in OHMS. Useful for GIC analysis. From transformer test report. Defaults to 85% of %R property

        DSS property name: `RdcOhms`, DSS property index: 47.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 47)

    @RdcOhms.setter
    def RdcOhms(self, value: Float64Array):
        self._set_float64_array_o(47, value)

    @property
    def Seasons(self) -> int:
        """
        Defines the number of ratings to be defined for the transfomer, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 48.
        """
        return self._lib.Obj_GetInt32(self._ptr, 48)

    @Seasons.setter
    def Seasons(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 48, value)

    @property
    def Ratings(self) -> Float64Array:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in transformers. Is given in kVA

        DSS property name: `Ratings`, DSS property index: 49.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 49)

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(49, value)

    @property
    def normamps(self) -> float:
        """
        Normal rated current.

        DSS property name: `normamps`, DSS property index: 50.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 50)

    @normamps.setter
    def normamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 50, value)

    @property
    def emergamps(self) -> float:
        """
        Maximum or emerg current.

        DSS property name: `emergamps`, DSS property index: 51.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 51)

    @emergamps.setter
    def emergamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 51, value)

    @property
    def faultrate(self) -> float:
        """
        Failure rate per year.

        DSS property name: `faultrate`, DSS property index: 52.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 52)

    @faultrate.setter
    def faultrate(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 52, value)

    @property
    def pctperm(self) -> float:
        """
        Percent of failures that become permanent.

        DSS property name: `pctperm`, DSS property index: 53.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 53)

    @pctperm.setter
    def pctperm(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 53, value)

    @property
    def repair(self) -> float:
        """
        Hours to repair.

        DSS property name: `repair`, DSS property index: 54.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 54)

    @repair.setter
    def repair(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 54, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 55.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 55)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 55, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 56.
        """
        return self._lib.Obj_GetInt32(self._ptr, 56) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 56, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 57.
        """
        self._set_string_o(57, value)

class Capacitor(DSSObj):
    __slots__ = []
    _cls_name = 'Capacitor'
    _cls_idx = 22
    _cls_prop_idx = {
        'bus1': 1,
        'bus2': 2,
        'phases': 3,
        'kvar': 4,
        'kv': 5,
        'conn': 6,
        'cmatrix': 7,
        'cuf': 8,
        'r': 9,
        'xl': 10,
        'harm': 11,
        'numsteps': 12,
        'states': 13,
        'normamps': 14,
        'emergamps': 15,
        'faultrate': 16,
        'pctperm': 17,
        'repair': 18,
        'basefreq': 19,
        'enabled': 20,
        'like': 21,
    }

    @property
    def bus1(self) -> str:
        """
        Name of first bus of 2-terminal capacitor. Examples:
        bus1=busname
        bus1=busname.1.2.3

        If only one bus specified, Bus2 will default to this bus, Node 0, and the capacitor will be a Yg shunt bank.

        DSS property name: `bus1`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(1, value)

    @property
    def bus2(self) -> str:
        """
        Name of 2nd bus. Defaults to all phases connected to first bus, node 0, (Shunt Wye Connection) except when Bus2 explicitly specified. 

        Not necessary to specify for delta (LL) connection.

        DSS property name: `bus2`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @bus2.setter
    def bus2(self, value: AnyStr):
        self._set_string_o(2, value)

    @property
    def phases(self) -> int:
        """
        Number of phases.

        DSS property name: `phases`, DSS property index: 3.
        """
        return self._lib.Obj_GetInt32(self._ptr, 3)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 3, value)

    @property
    def kvar(self) -> Float64Array:
        """
        Total kvar, if one step, or ARRAY of kvar ratings for each step.  Evenly divided among phases. See rules for NUMSTEPS.

        DSS property name: `kvar`, DSS property index: 4.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 4)

    @kvar.setter
    def kvar(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def kv(self) -> float:
        """
        For 2, 3-phase, kV phase-phase. Otherwise specify actual can rating.

        DSS property name: `kv`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @kv.setter
    def kv(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def conn(self) -> Connection:
        """
        ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN

        DSS property name: `conn`, DSS property index: 6.
        """
        return Connection(self._lib.Obj_GetInt32(self._ptr, 6))

    @conn.setter
    def conn(self, value: Union[AnyStr, int, Connection]):
        if not isinstance(value, int):
            self._set_string_o(6, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def conn_str(self) -> str:
        """
        ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN

        DSS property name: `conn`, DSS property index: 6.
        """
        return self._get_prop_string(6)

    @conn_str.setter
    def conn_str(self, value: AnyStr):
        self.conn = value

    @property
    def cmatrix(self) -> Float64Array:
        """
        Nodal cap. matrix, lower triangle, microfarads, of the following form:

        cmatrix="c11 | -c21 c22 | -c31 -c32 c33"

        All steps are assumed the same if this property is used.

        DSS property name: `cmatrix`, DSS property index: 7.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 7)

    @cmatrix.setter
    def cmatrix(self, value: Float64Array):
        self._set_float64_array_o(7, value)

    @property
    def cuf(self) -> Float64Array:
        """
        ARRAY of Capacitance, each phase, for each step, microfarads.
        See Rules for NumSteps.

        DSS property name: `cuf`, DSS property index: 8.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 8)

    @cuf.setter
    def cuf(self, value: Float64Array):
        self._set_float64_array_o(8, value)

    @property
    def R(self) -> Float64Array:
        """
        ARRAY of series resistance in each phase (line), ohms. Default is 0.0

        DSS property name: `R`, DSS property index: 9.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 9)

    @R.setter
    def R(self, value: Float64Array):
        self._set_float64_array_o(9, value)

    @property
    def XL(self) -> Float64Array:
        """
        ARRAY of series inductive reactance(s) in each phase (line) for filter, ohms at base frequency. Use this OR "h" property to define filter. Default is 0.0.

        DSS property name: `XL`, DSS property index: 10.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 10)

    @XL.setter
    def XL(self, value: Float64Array):
        self._set_float64_array_o(10, value)

    @property
    def Harm(self) -> Float64Array:
        """
        ARRAY of harmonics to which each step is tuned. Zero is interpreted as meaning zero reactance (no filter). Default is zero.

        DSS property name: `Harm`, DSS property index: 11.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 11)

    @Harm.setter
    def Harm(self, value: Float64Array):
        self._set_float64_array_o(11, value)

    @property
    def Numsteps(self) -> int:
        """
        Number of steps in this capacitor bank. Default = 1. Forces reallocation of the capacitance, reactor, and states array.  Rules: If this property was previously =1, the value in the kvar property is divided equally among the steps. The kvar property does not need to be reset if that is accurate.  If the Cuf or Cmatrix property was used previously, all steps are set to the value of the first step. The states property is set to all steps on. All filter steps are set to the same harmonic. If this property was previously >1, the arrays are reallocated, but no values are altered. You must SUBSEQUENTLY assign all array properties.

        DSS property name: `Numsteps`, DSS property index: 12.
        """
        return self._lib.Obj_GetInt32(self._ptr, 12)

    @Numsteps.setter
    def Numsteps(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 12, value)

    @property
    def states(self) -> Int32Array:
        """
        ARRAY of integers {1|0} states representing the state of each step (on|off). Defaults to 1 when reallocated (on). Capcontrol will modify this array as it turns steps on or off.

        DSS property name: `states`, DSS property index: 13.
        """
        return self._get_int32_array(self._lib.Obj_GetInt32Array, self._ptr, 13)

    @states.setter
    def states(self, value: Int32Array):
        self._set_int32_array_o(13, value)

    @property
    def normamps(self) -> float:
        """
        Normal rated current.

        DSS property name: `normamps`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @normamps.setter
    def normamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def emergamps(self) -> float:
        """
        Maximum or emerg current.

        DSS property name: `emergamps`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @emergamps.setter
    def emergamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def faultrate(self) -> float:
        """
        Failure rate per year.

        DSS property name: `faultrate`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @faultrate.setter
    def faultrate(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def pctperm(self) -> float:
        """
        Percent of failures that become permanent.

        DSS property name: `pctperm`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @pctperm.setter
    def pctperm(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def repair(self) -> float:
        """
        Hours to repair.

        DSS property name: `repair`, DSS property index: 18.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 18)

    @repair.setter
    def repair(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 18, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 20.
        """
        return self._lib.Obj_GetInt32(self._ptr, 20) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 20, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 21.
        """
        self._set_string_o(21, value)

class Reactor(DSSObj):
    __slots__ = []
    _cls_name = 'Reactor'
    _cls_idx = 23
    _cls_prop_idx = {
        'bus1': 1,
        'bus2': 2,
        'phases': 3,
        'kvar': 4,
        'kv': 5,
        'conn': 6,
        'rmatrix': 7,
        'xmatrix': 8,
        'parallel': 9,
        'r': 10,
        'x': 11,
        'rp': 12,
        'z1': 13,
        'z2': 14,
        'z0': 15,
        'z': 16,
        'rcurve': 17,
        'lcurve': 18,
        'lmh': 19,
        'normamps': 20,
        'emergamps': 21,
        'faultrate': 22,
        'pctperm': 23,
        'repair': 24,
        'basefreq': 25,
        'enabled': 26,
        'like': 27,
    }

    @property
    def bus1(self) -> str:
        """
        Name of first bus. Examples:
        bus1=busname
        bus1=busname.1.2.3

        Bus2 property will default to this bus, node 0, unless previously specified. Only Bus1 need be specified for a Yg shunt reactor.

        DSS property name: `bus1`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(1, value)

    @property
    def bus2(self) -> str:
        """
        Name of 2nd bus. Defaults to all phases connected to first bus, node 0, (Shunt Wye Connection) except when Bus2 is specifically defined.

        Not necessary to specify for delta (LL) connection

        DSS property name: `bus2`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @bus2.setter
    def bus2(self, value: AnyStr):
        self._set_string_o(2, value)

    @property
    def phases(self) -> int:
        """
        Number of phases.

        DSS property name: `phases`, DSS property index: 3.
        """
        return self._lib.Obj_GetInt32(self._ptr, 3)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 3, value)

    @property
    def kvar(self) -> float:
        """
        Total kvar, all phases.  Evenly divided among phases. Only determines X. Specify R separately

        DSS property name: `kvar`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @kvar.setter
    def kvar(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def kv(self) -> float:
        """
        For 2, 3-phase, kV phase-phase. Otherwise specify actual coil rating.

        DSS property name: `kv`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @kv.setter
    def kv(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def conn(self) -> Connection:
        """
        ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN. If Delta, then only one terminal.

        DSS property name: `conn`, DSS property index: 6.
        """
        return Connection(self._lib.Obj_GetInt32(self._ptr, 6))

    @conn.setter
    def conn(self, value: Union[AnyStr, int, Connection]):
        if not isinstance(value, int):
            self._set_string_o(6, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def conn_str(self) -> str:
        """
        ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN. If Delta, then only one terminal.

        DSS property name: `conn`, DSS property index: 6.
        """
        return self._get_prop_string(6)

    @conn_str.setter
    def conn_str(self, value: AnyStr):
        self.conn = value

    @property
    def Rmatrix(self) -> Float64Array:
        """
        Resistance matrix, lower triangle, ohms at base frequency. Order of the matrix is the number of phases. Mutually exclusive to specifying parameters by kvar or X.

        DSS property name: `Rmatrix`, DSS property index: 7.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 7)

    @Rmatrix.setter
    def Rmatrix(self, value: Float64Array):
        self._set_float64_array_o(7, value)

    @property
    def Xmatrix(self) -> Float64Array:
        """
        Reactance matrix, lower triangle, ohms at base frequency. Order of the matrix is the number of phases. Mutually exclusive to specifying parameters by kvar or X.

        DSS property name: `Xmatrix`, DSS property index: 8.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 8)

    @Xmatrix.setter
    def Xmatrix(self, value: Float64Array):
        self._set_float64_array_o(8, value)

    @property
    def Parallel(self) -> bool:
        """
        {Yes | No}  Default=No. Indicates whether Rmatrix and Xmatrix are to be considered in parallel. Default is series. For other models, specify R and Rp.

        DSS property name: `Parallel`, DSS property index: 9.
        """
        return self._lib.Obj_GetInt32(self._ptr, 9) != 0

    @Parallel.setter
    def Parallel(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 9, value)

    @property
    def R(self) -> float:
        """
        Resistance (in series with reactance), each phase, ohms. This property applies to REACTOR specified by either kvar or X. See also help on Z.

        DSS property name: `R`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @R.setter
    def R(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def X(self) -> float:
        """
        Reactance, each phase, ohms at base frequency. See also help on Z and LmH properties.

        DSS property name: `X`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @X.setter
    def X(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def Rp(self) -> float:
        """
        Resistance in parallel with R and X (the entire branch). Assumed infinite if not specified.

        DSS property name: `Rp`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @Rp.setter
    def Rp(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def Z1(self) -> complex:
        """
        Positive-sequence impedance, ohms, as a 2-element array representing a complex number. Example: 

        Z1=[1, 2]  ! represents 1 + j2 

        If defined, Z1, Z2, and Z0 are used to define the impedance matrix of the REACTOR. Z1 MUST BE DEFINED TO USE THIS OPTION FOR DEFINING THE MATRIX.

        Side Effect: Sets Z2 and Z0 to same values unless they were previously defined.

        DSS property name: `Z1`, DSS property index: 13.
        """
        return self._get_complex(13)

    @Z1.setter
    def Z1(self, value: complex):
        self._set_complex(13, value)

    @property
    def Z2(self) -> complex:
        """
        Negative-sequence impedance, ohms, as a 2-element array representing a complex number. Example: 

        Z2=[1, 2]  ! represents 1 + j2 

        Used to define the impedance matrix of the REACTOR if Z1 is also specified. 

        Note: Z2 defaults to Z1 if it is not specifically defined. If Z2 is not equal to Z1, the impedance matrix is asymmetrical.

        DSS property name: `Z2`, DSS property index: 14.
        """
        return self._get_complex(14)

    @Z2.setter
    def Z2(self, value: complex):
        self._set_complex(14, value)

    @property
    def Z0(self) -> complex:
        """
        Zer0-sequence impedance, ohms, as a 2-element array representing a complex number. Example: 

        Z0=[3, 4]  ! represents 3 + j4 

        Used to define the impedance matrix of the REACTOR if Z1 is also specified. 

        Note: Z0 defaults to Z1 if it is not specifically defined. 

        DSS property name: `Z0`, DSS property index: 15.
        """
        return self._get_complex(15)

    @Z0.setter
    def Z0(self, value: complex):
        self._set_complex(15, value)

    @property
    def RCurve(self) -> str:
        """
        Name of XYCurve object, previously defined, describing per-unit variation of phase resistance, R, vs. frequency. Applies to resistance specified by R or Z property. If actual values are not known, R often increases by approximately the square root of frequency.

        DSS property name: `RCurve`, DSS property index: 17.
        """
        return self._get_prop_string(17)

    @RCurve.setter
    def RCurve(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(17, value)
            return

        self._set_string_o(17, value)

    @property
    def RCurve_obj(self) -> XYcurve:
        """
        Name of XYCurve object, previously defined, describing per-unit variation of phase resistance, R, vs. frequency. Applies to resistance specified by R or Z property. If actual values are not known, R often increases by approximately the square root of frequency.

        DSS property name: `RCurve`, DSS property index: 17.
        """
        return self._get_obj(17, XYcurve)

    @RCurve_obj.setter
    def RCurve_obj(self, value: XYcurve):
        self._set_obj(17, value)

    @property
    def LCurve(self) -> str:
        """
        Name of XYCurve object, previously defined, describing per-unit variation of phase inductance, L=X/w, vs. frequency. Applies to reactance specified by X, LmH, Z, or kvar property.L generally decreases somewhat with frequency above the base frequency, approaching a limit at a few kHz.

        DSS property name: `LCurve`, DSS property index: 18.
        """
        return self._get_prop_string(18)

    @LCurve.setter
    def LCurve(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(18, value)
            return

        self._set_string_o(18, value)

    @property
    def LCurve_obj(self) -> XYcurve:
        """
        Name of XYCurve object, previously defined, describing per-unit variation of phase inductance, L=X/w, vs. frequency. Applies to reactance specified by X, LmH, Z, or kvar property.L generally decreases somewhat with frequency above the base frequency, approaching a limit at a few kHz.

        DSS property name: `LCurve`, DSS property index: 18.
        """
        return self._get_obj(18, XYcurve)

    @LCurve_obj.setter
    def LCurve_obj(self, value: XYcurve):
        self._set_obj(18, value)

    @property
    def LmH(self) -> float:
        """
        Inductance, mH. Alternate way to define the reactance, X, property.

        DSS property name: `LmH`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @LmH.setter
    def LmH(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def normamps(self) -> float:
        """
        Normal rated current.

        DSS property name: `normamps`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @normamps.setter
    def normamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    @property
    def emergamps(self) -> float:
        """
        Maximum or emerg current.

        DSS property name: `emergamps`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @emergamps.setter
    def emergamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def faultrate(self) -> float:
        """
        Failure rate per year.

        DSS property name: `faultrate`, DSS property index: 22.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 22)

    @faultrate.setter
    def faultrate(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 22, value)

    @property
    def pctperm(self) -> float:
        """
        Percent of failures that become permanent.

        DSS property name: `pctperm`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @pctperm.setter
    def pctperm(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def repair(self) -> float:
        """
        Hours to repair.

        DSS property name: `repair`, DSS property index: 24.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 24)

    @repair.setter
    def repair(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 24, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 25.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 25)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 25, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 26.
        """
        return self._lib.Obj_GetInt32(self._ptr, 26) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 26, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 27.
        """
        self._set_string_o(27, value)

class CapControl(DSSObj):
    __slots__ = []
    _cls_name = 'CapControl'
    _cls_idx = 24
    _cls_prop_idx = {
        'element': 1,
        'terminal': 2,
        'capacitor': 3,
        'type': 4,
        'ptratio': 5,
        'ctratio': 6,
        'onsetting': 7,
        'offsetting': 8,
        'delay': 9,
        'voltoverride': 10,
        'vmax': 11,
        'vmin': 12,
        'delayoff': 13,
        'deadtime': 14,
        'ctphase': 15,
        'ptphase': 16,
        'vbus': 17,
        'eventlog': 18,
        'usermodel': 19,
        'userdata': 20,
        'pctminkvar': 21,
        'reset': 22,
        'basefreq': 23,
        'enabled': 24,
        'like': 25,
    }

    # Class-specific enumerations
    class CapControlType(IntEnum):
        """CapControl: Type (DSS enumeration for CapControl)"""
        Current = 0 # Current
        Voltage = 1 # Voltage
        kvar = 2 # kvar
        Time = 3 # Time
        PowerFactor = 4 # PowerFactor


    @property
    def element(self) -> str:
        """
        Full object name of the circuit element, typically a line or transformer, to which the capacitor control's PT and/or CT are connected.There is no default; must be specified.

        DSS property name: `element`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @element.setter
    def element(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(1, value)
            return

        self._set_string_o(1, value)

    @property
    def element_obj(self) -> DSSObj:
        """
        Full object name of the circuit element, typically a line or transformer, to which the capacitor control's PT and/or CT are connected.There is no default; must be specified.

        DSS property name: `element`, DSS property index: 1.
        """
        return self._get_obj(1, None)

    @element_obj.setter
    def element_obj(self, value: DSSObj):
        self._set_obj(1, value)

    @property
    def terminal(self) -> int:
        """
        Number of the terminal of the circuit element to which the CapControl is connected. 1 or 2, typically.  Default is 1.

        DSS property name: `terminal`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @terminal.setter
    def terminal(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def capacitor(self) -> str:
        """
        Name of Capacitor element which the CapControl controls. No Default; Must be specified.Do not specify the full object name; "Capacitor" is assumed for the object class.  Example:

        Capacitor=cap1

        DSS property name: `capacitor`, DSS property index: 3.
        """
        return self._get_prop_string(3)

    @capacitor.setter
    def capacitor(self, value: Union[AnyStr, Capacitor]):
        if isinstance(value, DSSObj):
            self._set_obj(3, value)
            return

        self._set_string_o(3, value)

    @property
    def capacitor_obj(self) -> Capacitor:
        """
        Name of Capacitor element which the CapControl controls. No Default; Must be specified.Do not specify the full object name; "Capacitor" is assumed for the object class.  Example:

        Capacitor=cap1

        DSS property name: `capacitor`, DSS property index: 3.
        """
        return self._get_obj(3, Capacitor)

    @capacitor_obj.setter
    def capacitor_obj(self, value: Capacitor):
        self._set_obj(3, value)

    @property
    def type(self) -> CapControlType:
        """
        {Current | voltage | kvar | PF | time } Control type.  Specify the ONsetting and OFFsetting appropriately with the type of control. (See help for ONsetting)

        DSS property name: `type`, DSS property index: 4.
        """
        return CapControl.CapControlType(self._lib.Obj_GetInt32(self._ptr, 4))

    @type.setter
    def type(self, value: Union[AnyStr, int, CapControlType]):
        if not isinstance(value, int):
            self._set_string_o(4, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 4, value)

    @property
    def type_str(self) -> str:
        """
        {Current | voltage | kvar | PF | time } Control type.  Specify the ONsetting and OFFsetting appropriately with the type of control. (See help for ONsetting)

        DSS property name: `type`, DSS property index: 4.
        """
        return self._get_prop_string(4)

    @type_str.setter
    def type_str(self, value: AnyStr):
        self.type = value

    @property
    def PTratio(self) -> float:
        """
        Ratio of the PT that converts the monitored voltage to the control voltage. Default is 60.  If the capacitor is Wye, the 1st phase line-to-neutral voltage is monitored.  Else, the line-to-line voltage (1st - 2nd phase) is monitored.

        DSS property name: `PTratio`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @PTratio.setter
    def PTratio(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def CTratio(self) -> float:
        """
        Ratio of the CT from line amps to control ampere setting for current and kvar control types. 

        DSS property name: `CTratio`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @CTratio.setter
    def CTratio(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def ONsetting(self) -> float:
        """
        Value at which the control arms to switch the capacitor ON (or ratchet up a step).  

        Type of Control:

        Current: Line Amps / CTratio
        Voltage: Line-Neutral (or Line-Line for delta) Volts / PTratio
        kvar:    Total kvar, all phases (3-phase for pos seq model). This is directional. 
        PF:      Power Factor, Total power in monitored terminal. Negative for Leading. 
        Time:    Hrs from Midnight as a floating point number (decimal). 7:30am would be entered as 7.5.

        DSS property name: `ONsetting`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @ONsetting.setter
    def ONsetting(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def OFFsetting(self) -> float:
        """
        Value at which the control arms to switch the capacitor OFF. (See help for ONsetting)For Time control, is OK to have Off time the next day ( < On time)

        DSS property name: `OFFsetting`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @OFFsetting.setter
    def OFFsetting(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def Delay(self) -> float:
        """
        Time delay, in seconds, from when the control is armed before it sends out the switching command to turn ON.  The control may reset before the action actually occurs. This is used to determine which capacity control will act first. Default is 15.  You may specify any floating point number to achieve a model of whatever condition is necessary.

        DSS property name: `Delay`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @Delay.setter
    def Delay(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def VoltOverride(self) -> bool:
        """
        {Yes | No}  Default is No.  Switch to indicate whether VOLTAGE OVERRIDE is to be considered. Vmax and Vmin must be set to reasonable values if this property is Yes.

        DSS property name: `VoltOverride`, DSS property index: 10.
        """
        return self._lib.Obj_GetInt32(self._ptr, 10) != 0

    @VoltOverride.setter
    def VoltOverride(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 10, value)

    @property
    def Vmax(self) -> float:
        """
        Maximum voltage, in volts.  If the voltage across the capacitor divided by the PTRATIO is greater than this voltage, the capacitor will switch OFF regardless of other control settings. Default is 126 (goes with a PT ratio of 60 for 12.47 kV system).

        DSS property name: `Vmax`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @Vmax.setter
    def Vmax(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def Vmin(self) -> float:
        """
        Minimum voltage, in volts.  If the voltage across the capacitor divided by the PTRATIO is less than this voltage, the capacitor will switch ON regardless of other control settings. Default is 115 (goes with a PT ratio of 60 for 12.47 kV system).

        DSS property name: `Vmin`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @Vmin.setter
    def Vmin(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def DelayOFF(self) -> float:
        """
        Time delay, in seconds, for control to turn OFF when present state is ON. Default is 15.

        DSS property name: `DelayOFF`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @DelayOFF.setter
    def DelayOFF(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def DeadTime(self) -> float:
        """
        Dead time after capacitor is turned OFF before it can be turned back ON. Default is 300 sec.

        DSS property name: `DeadTime`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @DeadTime.setter
    def DeadTime(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def CTPhase(self) -> Union[MonitoredPhase, int]:
        """
        Number of the phase being monitored for CURRENT control or one of {AVG | MAX | MIN} for all phases. Default=1. If delta or L-L connection, enter the first or the two phases being monitored [1-2, 2-3, 3-1]. Must be less than the number of phases. Does not apply to kvar control which uses all phases by default.

        DSS property name: `CTPhase`, DSS property index: 15.
        """
        value = self._lib.Obj_GetInt32(self._ptr, 15)
        if value > 0:
            return value
    
        return MonitoredPhase(value)

    @CTPhase.setter
    def CTPhase(self, value: Union[AnyStr, int, MonitoredPhase]):
        if not isinstance(value, int):
            self._set_string_o(15, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 15, value)

    @property
    def CTPhase_str(self) -> str:
        """
        Number of the phase being monitored for CURRENT control or one of {AVG | MAX | MIN} for all phases. Default=1. If delta or L-L connection, enter the first or the two phases being monitored [1-2, 2-3, 3-1]. Must be less than the number of phases. Does not apply to kvar control which uses all phases by default.

        DSS property name: `CTPhase`, DSS property index: 15.
        """
        return self._get_prop_string(15)

    @CTPhase_str.setter
    def CTPhase_str(self, value: AnyStr):
        self.CTPhase = value

    @property
    def PTPhase(self) -> Union[MonitoredPhase, int]:
        """
        Number of the phase being monitored for VOLTAGE control or one of {AVG | MAX | MIN} for all phases. Default=1. If delta or L-L connection, enter the first or the two phases being monitored [1-2, 2-3, 3-1]. Must be less than the number of phases. Does not apply to kvar control which uses all phases by default.

        DSS property name: `PTPhase`, DSS property index: 16.
        """
        value = self._lib.Obj_GetInt32(self._ptr, 16)
        if value > 0:
            return value
    
        return MonitoredPhase(value)

    @PTPhase.setter
    def PTPhase(self, value: Union[AnyStr, int, MonitoredPhase]):
        if not isinstance(value, int):
            self._set_string_o(16, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 16, value)

    @property
    def PTPhase_str(self) -> str:
        """
        Number of the phase being monitored for VOLTAGE control or one of {AVG | MAX | MIN} for all phases. Default=1. If delta or L-L connection, enter the first or the two phases being monitored [1-2, 2-3, 3-1]. Must be less than the number of phases. Does not apply to kvar control which uses all phases by default.

        DSS property name: `PTPhase`, DSS property index: 16.
        """
        return self._get_prop_string(16)

    @PTPhase_str.setter
    def PTPhase_str(self, value: AnyStr):
        self.PTPhase = value

    @property
    def VBus(self) -> str:
        """
        Name of bus to use for voltage override function. Default is bus at monitored terminal. Sometimes it is useful to monitor a bus in another location to emulate various DMS control algorithms.

        DSS property name: `VBus`, DSS property index: 17.
        """
        return self._get_prop_string(17)

    @VBus.setter
    def VBus(self, value: AnyStr):
        self._set_string_o(17, value)

    @property
    def EventLog(self) -> bool:
        """
        {Yes/True* | No/False} Default is YES for CapControl. Log control actions to Eventlog.

        DSS property name: `EventLog`, DSS property index: 18.
        """
        return self._lib.Obj_GetInt32(self._ptr, 18) != 0

    @EventLog.setter
    def EventLog(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 18, value)

    @property
    def UserModel(self) -> str:
        """
        Name of DLL containing user-written CapControl model, overriding the default model.  Set to "none" to negate previous setting. 

        DSS property name: `UserModel`, DSS property index: 19.
        """
        return self._get_prop_string(19)

    @UserModel.setter
    def UserModel(self, value: AnyStr):
        self._set_string_o(19, value)

    @property
    def UserData(self) -> str:
        """
        String (in quotes or parentheses if necessary) that gets passed to the user-written CapControl model Edit function for defining the data required for that model. 

        DSS property name: `UserData`, DSS property index: 20.
        """
        return self._get_prop_string(20)

    @UserData.setter
    def UserData(self, value: AnyStr):
        self._set_string_o(20, value)

    @property
    def pctMinkvar(self) -> float:
        """
        For PF control option, min percent of total bank kvar at which control will close capacitor switch. Default = 50.

        DSS property name: `pctMinkvar`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @pctMinkvar.setter
    def pctMinkvar(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    def Reset(self, value: bool):
        """
        {Yes | No} If Yes, forces Reset of this CapControl.

        DSS property name: `Reset`, DSS property index: 22.
        """
        self._lib.Obj_SetInt32(self._ptr, 22, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 24.
        """
        return self._lib.Obj_GetInt32(self._ptr, 24) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 24, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 25.
        """
        self._set_string_o(25, value)

class Fault(DSSObj):
    __slots__ = []
    _cls_name = 'Fault'
    _cls_idx = 25
    _cls_prop_idx = {
        'bus1': 1,
        'bus2': 2,
        'phases': 3,
        'r': 4,
        'pctstddev': 5,
        '%stddev': 5,
        'gmatrix': 6,
        'ontime': 7,
        'temporary': 8,
        'minamps': 9,
        'normamps': 10,
        'emergamps': 11,
        'faultrate': 12,
        'pctperm': 13,
        'repair': 14,
        'basefreq': 15,
        'enabled': 16,
        'like': 17,
    }

    @property
    def bus1(self) -> str:
        """
        Name of first bus. Examples:

        bus1=busname
        bus1=busname.1.2.3

        Bus2 automatically defaults to busname.0,0,0 unless it was previously defined. 

        DSS property name: `bus1`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(1, value)

    @property
    def bus2(self) -> str:
        """
        Name of 2nd bus of the 2-terminal Fault object. Defaults to all phases connected to first bus, node 0, if not specified. (Shunt Wye Connection to ground reference)

        That is, the Fault defaults to a ground fault unless otherwise specified.

        DSS property name: `bus2`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @bus2.setter
    def bus2(self, value: AnyStr):
        self._set_string_o(2, value)

    @property
    def phases(self) -> int:
        """
        Number of Phases. Default is 1.

        DSS property name: `phases`, DSS property index: 3.
        """
        return self._lib.Obj_GetInt32(self._ptr, 3)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 3, value)

    @property
    def r(self) -> float:
        """
        Resistance, each phase, ohms. Default is 0.0001. Assumed to be Mean value if gaussian random mode.Max value if uniform mode.  A Fault is actually a series resistance that defaults to a wye connection to ground on the second terminal.  You may reconnect the 2nd terminal to achieve whatever connection.  Use the Gmatrix property to specify an arbitrary conductance matrix.

        DSS property name: `r`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @r.setter
    def r(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def pctstddev(self) -> float:
        """
        Percent standard deviation in resistance to assume for Monte Carlo fault (MF) solution mode for GAUSSIAN distribution. Default is 0 (no variation from mean).

        DSS property name: `%stddev`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @pctstddev.setter
    def pctstddev(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def Gmatrix(self) -> Float64Array:
        """
        Use this to specify a nodal conductance (G) matrix to represent some arbitrary resistance network. Specify in lower triangle form as usual for DSS matrices.

        DSS property name: `Gmatrix`, DSS property index: 6.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 6)

    @Gmatrix.setter
    def Gmatrix(self, value: Float64Array):
        self._set_float64_array_o(6, value)

    @property
    def ONtime(self) -> float:
        """
        Time (sec) at which the fault is established for time varying simulations. Default is 0.0 (on at the beginning of the simulation)

        DSS property name: `ONtime`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @ONtime.setter
    def ONtime(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def temporary(self) -> bool:
        """
        {Yes | No} Default is No.  Designate whether the fault is temporary.  For Time-varying simulations, the fault will be removed if the current through the fault drops below the MINAMPS criteria.

        DSS property name: `temporary`, DSS property index: 8.
        """
        return self._lib.Obj_GetInt32(self._ptr, 8) != 0

    @temporary.setter
    def temporary(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 8, value)

    @property
    def MinAmps(self) -> float:
        """
        Minimum amps that can sustain a temporary fault. Default is 5.

        DSS property name: `MinAmps`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @MinAmps.setter
    def MinAmps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def normamps(self) -> float:
        """
        Normal rated current.

        DSS property name: `normamps`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @normamps.setter
    def normamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def emergamps(self) -> float:
        """
        Maximum or emerg current.

        DSS property name: `emergamps`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @emergamps.setter
    def emergamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def faultrate(self) -> float:
        """
        Failure rate per year.

        DSS property name: `faultrate`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @faultrate.setter
    def faultrate(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def pctperm(self) -> float:
        """
        Percent of failures that become permanent.

        DSS property name: `pctperm`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @pctperm.setter
    def pctperm(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def repair(self) -> float:
        """
        Hours to repair.

        DSS property name: `repair`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @repair.setter
    def repair(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 16.
        """
        return self._lib.Obj_GetInt32(self._ptr, 16) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 16, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 17.
        """
        self._set_string_o(17, value)

class DynamicExp(DSSObj):
    __slots__ = []
    _cls_name = 'DynamicExp'
    _cls_idx = 26
    _cls_prop_idx = {
        'nvariables': 1,
        'varnames': 2,
        'var': 3,
        'varidx': 4,
        'expression': 5,
        'domain': 6,
        'like': 7,
    }

    # Class-specific enumerations
    class DynamicExpDomain(IntEnum):
        """DynamicExp: Domain (DSS enumeration for DynamicExp)"""
        Time = 0 # Time
        dq = 1 # dq


    @property
    def NVariables(self) -> int:
        """
        (Int) Number of state variables to be considered in the differential equation.

        DSS property name: `NVariables`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @NVariables.setter
    def NVariables(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def VarNames(self) -> List[str]:
        """
        ([String]) Array of strings with the names of the state variables.

        DSS property name: `VarNames`, DSS property index: 2.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 2)

    @VarNames.setter
    def VarNames(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 2, value_ptr, value_count)
        self._check_for_error()

    @property
    def var(self) -> str:
        """
        (String) Activates the state variable using the given name.

        DSS property name: `var`, DSS property index: 3.
        """
        return self._get_prop_string(3)

    @var.setter
    def var(self, value: AnyStr):
        self._set_string_o(3, value)

    @property
    def VarIdx(self) -> int:
        """
        (Int) read-only, returns the index of the active state variable.

        DSS property name: `VarIdx`, DSS property index: 4.
        """
        return self._lib.Obj_GetInt32(self._ptr, 4)

    @VarIdx.setter
    def VarIdx(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 4, value)

    @property
    def Expression(self) -> str:
        """
        It is the differential expression using OpenDSS RPN syntax. The expression must be contained within brackets in case of having multiple equations, for example:

        expression="[w dt = 1 M / (P_m D*w - P_e -) *]"

        DSS property name: `Expression`, DSS property index: 5.
        """
        return self._get_prop_string(5)

    @Expression.setter
    def Expression(self, value: AnyStr):
        self._set_string_o(5, value)

    @property
    def Domain(self) -> DynamicExpDomain:
        """
        It is the domain for which the equation is defined, it can be one of [time*, dq]. By deafult, dynamic epxressions are defined in the time domain.

        DSS property name: `Domain`, DSS property index: 6.
        """
        return DynamicExp.DynamicExpDomain(self._lib.Obj_GetInt32(self._ptr, 6))

    @Domain.setter
    def Domain(self, value: Union[AnyStr, int, DynamicExpDomain]):
        if not isinstance(value, int):
            self._set_string_o(6, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def Domain_str(self) -> str:
        """
        It is the domain for which the equation is defined, it can be one of [time*, dq]. By deafult, dynamic epxressions are defined in the time domain.

        DSS property name: `Domain`, DSS property index: 6.
        """
        return self._get_prop_string(6)

    @Domain_str.setter
    def Domain_str(self, value: AnyStr):
        self.Domain = value

    def like(self, value: AnyStr):
        """
        DynamicExp.like

        DSS property name: `like`, DSS property index: 7.
        """
        self._set_string_o(7, value)

class Generator(DSSObj):
    __slots__ = []
    _cls_name = 'Generator'
    _cls_idx = 27
    _cls_prop_idx = {
        'phases': 1,
        'bus1': 2,
        'kv': 3,
        'kw': 4,
        'pf': 5,
        'kvar': 6,
        'model': 7,
        'vminpu': 8,
        'vmaxpu': 9,
        'yearly': 10,
        'daily': 11,
        'duty': 12,
        'dispmode': 13,
        'dispvalue': 14,
        'conn': 15,
        'status': 16,
        'cls': 17,
        'class': 17,
        'vpu': 18,
        'maxkvar': 19,
        'minkvar': 20,
        'pvfactor': 21,
        'forceon': 22,
        'kva': 23,
        'mva': 24,
        'xd': 25,
        'xdp': 26,
        'xdpp': 27,
        'h': 28,
        'd': 29,
        'usermodel': 30,
        'userdata': 31,
        'shaftmodel': 32,
        'shaftdata': 33,
        'dutystart': 34,
        'debugtrace': 35,
        'balanced': 36,
        'xrdp': 37,
        'usefuel': 38,
        'fuelkwh': 39,
        'pctfuel': 40,
        '%fuel': 40,
        'pctreserve': 41,
        '%reserve': 41,
        'refuel': 42,
        'dynamiceq': 43,
        'dynout': 44,
        'spectrum': 45,
        'basefreq': 46,
        'enabled': 47,
        'like': 48,
    }

    # Class-specific enumerations
    class GeneratorDispatchMode(IntEnum):
        """Generator: Dispatch Mode (DSS enumeration for Generator)"""
        Default = 0 # Default
        LoadLevel = 1 # LoadLevel
        Price = 2 # Price

    class GeneratorStatus(IntEnum):
        """Generator: Status (DSS enumeration for Generator)"""
        Variable = 0 # Variable
        Fixed = 1 # Fixed


    @property
    def phases(self) -> int:
        """
        Number of Phases, this Generator.  Power is evenly divided among phases.

        DSS property name: `phases`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def bus1(self) -> str:
        """
        Bus to which the Generator is connected.  May include specific node specification.

        DSS property name: `bus1`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(2, value)

    @property
    def kv(self) -> float:
        """
        Nominal rated (1.0 per unit) voltage, kV, for Generator. For 2- and 3-phase Generators, specify phase-phase kV. Otherwise, for phases=1 or phases>3, specify actual kV across each branch of the Generator. If wye (star), specify phase-neutral kV. If delta or phase-phase connected, specify phase-phase kV.

        DSS property name: `kv`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @kv.setter
    def kv(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def kW(self) -> float:
        """
        Total base kW for the Generator.  A positive value denotes power coming OUT of the element, 
        which is the opposite of a load. This value is modified depending on the dispatch mode. Unaffected by the global load multiplier and growth curves. If you want there to be more generation, you must add more generators or change this value.

        DSS property name: `kW`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @kW.setter
    def kW(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def pf(self) -> float:
        """
        Generator power factor. Default is 0.80. Enter negative for leading powerfactor (when kW and kvar have opposite signs.)
        A positive power factor for a generator signifies that the generator produces vars 
        as is typical for a synchronous generator.  Induction machines would be 
        specified with a negative power factor.

        DSS property name: `pf`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @pf.setter
    def pf(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def kvar(self) -> float:
        """
        Specify the base kvar.  Alternative to specifying the power factor.  Side effect:  the power factor value is altered to agree based on present value of kW.

        DSS property name: `kvar`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @kvar.setter
    def kvar(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def model(self) -> int:
        """
        Integer code for the model to use for generation variation with voltage. Valid values are:

        1:Generator injects a constant kW at specified power factor.
        2:Generator is modeled as a constant admittance.
        3:Const kW, constant kV.  Somewhat like a conventional transmission power flow P-V generator.
        4:Const kW, Fixed Q (Q never varies)
        5:Const kW, Fixed Q(as a constant reactance)
        6:Compute load injection from User-written Model.(see usage of Xd, Xdp)
        7:Constant kW, kvar, but current-limited below Vminpu. Approximates a simple inverter. See also Balanced.

        DSS property name: `model`, DSS property index: 7.
        """
        return self._lib.Obj_GetInt32(self._ptr, 7)

    @model.setter
    def model(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 7, value)

    @property
    def Vminpu(self) -> float:
        """
        Default = 0.90.  Minimum per unit voltage for which the Model is assumed to apply. Below this value, the load model reverts to a constant impedance model. For model 7, the current is limited to the value computed for constant power at Vminpu.

        DSS property name: `Vminpu`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @Vminpu.setter
    def Vminpu(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def Vmaxpu(self) -> float:
        """
        Default = 1.10.  Maximum per unit voltage for which the Model is assumed to apply. Above this value, the load model reverts to a constant impedance model.

        DSS property name: `Vmaxpu`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @Vmaxpu.setter
    def Vmaxpu(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def yearly(self) -> str:
        """
        Dispatch shape to use for yearly simulations.  Must be previously defined as a Loadshape object. If this is not specified, a constant value is assumed (no variation). If the generator is assumed to be ON continuously, specify Status=FIXED, or designate a curve that is 1.0 per unit at all times. Set to NONE to reset to no loadahape. Nominally for 8760 simulations.  If there are fewer points in the designated shape than the number of points in the solution, the curve is repeated.

        DSS property name: `yearly`, DSS property index: 10.
        """
        return self._get_prop_string(10)

    @yearly.setter
    def yearly(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(10, value)
            return

        self._set_string_o(10, value)

    @property
    def yearly_obj(self) -> LoadShape:
        """
        Dispatch shape to use for yearly simulations.  Must be previously defined as a Loadshape object. If this is not specified, a constant value is assumed (no variation). If the generator is assumed to be ON continuously, specify Status=FIXED, or designate a curve that is 1.0 per unit at all times. Set to NONE to reset to no loadahape. Nominally for 8760 simulations.  If there are fewer points in the designated shape than the number of points in the solution, the curve is repeated.

        DSS property name: `yearly`, DSS property index: 10.
        """
        return self._get_obj(10, LoadShape)

    @yearly_obj.setter
    def yearly_obj(self, value: LoadShape):
        self._set_obj(10, value)

    @property
    def daily(self) -> str:
        """
        Dispatch shape to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically.  If generator is assumed to be ON continuously, specify Status=FIXED, or designate a Loadshape objectthat is 1.0 perunit for all hours. Set to NONE to reset to no loadahape. 

        DSS property name: `daily`, DSS property index: 11.
        """
        return self._get_prop_string(11)

    @daily.setter
    def daily(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(11, value)
            return

        self._set_string_o(11, value)

    @property
    def daily_obj(self) -> LoadShape:
        """
        Dispatch shape to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically.  If generator is assumed to be ON continuously, specify Status=FIXED, or designate a Loadshape objectthat is 1.0 perunit for all hours. Set to NONE to reset to no loadahape. 

        DSS property name: `daily`, DSS property index: 11.
        """
        return self._get_obj(11, LoadShape)

    @daily_obj.setter
    def daily_obj(self, value: LoadShape):
        self._set_obj(11, value)

    @property
    def duty(self) -> str:
        """
        Load shape to use for duty cycle dispatch simulations such as for wind generation. Must be previously defined as a Loadshape object. Typically would have time intervals less than 1 hr -- perhaps, in seconds. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.

        DSS property name: `duty`, DSS property index: 12.
        """
        return self._get_prop_string(12)

    @duty.setter
    def duty(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(12, value)
            return

        self._set_string_o(12, value)

    @property
    def duty_obj(self) -> LoadShape:
        """
        Load shape to use for duty cycle dispatch simulations such as for wind generation. Must be previously defined as a Loadshape object. Typically would have time intervals less than 1 hr -- perhaps, in seconds. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.

        DSS property name: `duty`, DSS property index: 12.
        """
        return self._get_obj(12, LoadShape)

    @duty_obj.setter
    def duty_obj(self, value: LoadShape):
        self._set_obj(12, value)

    @property
    def dispmode(self) -> GeneratorDispatchMode:
        """
        {Default* | Loadlevel | Price } Default = Default. Dispatch mode. In default mode, gen is either always on or follows dispatch curve as specified. Otherwise, the gen comes on when either the global default load level (Loadshape "default") or the price level exceeds the dispatch value.

        DSS property name: `dispmode`, DSS property index: 13.
        """
        return Generator.GeneratorDispatchMode(self._lib.Obj_GetInt32(self._ptr, 13))

    @dispmode.setter
    def dispmode(self, value: Union[AnyStr, int, GeneratorDispatchMode]):
        if not isinstance(value, int):
            self._set_string_o(13, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 13, value)

    @property
    def dispmode_str(self) -> str:
        """
        {Default* | Loadlevel | Price } Default = Default. Dispatch mode. In default mode, gen is either always on or follows dispatch curve as specified. Otherwise, the gen comes on when either the global default load level (Loadshape "default") or the price level exceeds the dispatch value.

        DSS property name: `dispmode`, DSS property index: 13.
        """
        return self._get_prop_string(13)

    @dispmode_str.setter
    def dispmode_str(self, value: AnyStr):
        self.dispmode = value

    @property
    def dispvalue(self) -> float:
        """
        Dispatch value. 
        If = 0.0 (default) then Generator follow dispatch curves, if any. 
        If > 0  then Generator is ON only when either the price signal (in Price dispatch mode) exceeds this value or the active circuit load multiplier * "default" loadshape value * the default yearly growth factor exceeds this value.  Then the generator follows dispatch curves (duty, daily, or yearly), if any (see also Status).

        DSS property name: `dispvalue`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @dispvalue.setter
    def dispvalue(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def conn(self) -> Connection:
        """
        ={wye|LN|delta|LL}.  Default is wye.

        DSS property name: `conn`, DSS property index: 15.
        """
        return Connection(self._lib.Obj_GetInt32(self._ptr, 15))

    @conn.setter
    def conn(self, value: Union[AnyStr, int, Connection]):
        if not isinstance(value, int):
            self._set_string_o(15, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 15, value)

    @property
    def conn_str(self) -> str:
        """
        ={wye|LN|delta|LL}.  Default is wye.

        DSS property name: `conn`, DSS property index: 15.
        """
        return self._get_prop_string(15)

    @conn_str.setter
    def conn_str(self, value: AnyStr):
        self.conn = value

    @property
    def status(self) -> GeneratorStatus:
        """
        ={Fixed | Variable*}.  If Fixed, then dispatch multipliers do not apply. The generator is alway at full power when it is ON.  Default is Variable  (follows curves).

        DSS property name: `status`, DSS property index: 16.
        """
        return Generator.GeneratorStatus(self._lib.Obj_GetInt32(self._ptr, 16))

    @status.setter
    def status(self, value: Union[AnyStr, int, GeneratorStatus]):
        if not isinstance(value, int):
            self._set_string_o(16, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 16, value)

    @property
    def status_str(self) -> str:
        """
        ={Fixed | Variable*}.  If Fixed, then dispatch multipliers do not apply. The generator is alway at full power when it is ON.  Default is Variable  (follows curves).

        DSS property name: `status`, DSS property index: 16.
        """
        return self._get_prop_string(16)

    @status_str.setter
    def status_str(self, value: AnyStr):
        self.status = value

    @property
    def cls(self) -> int:
        """
        An arbitrary integer number representing the class of Generator so that Generator values may be segregated by class.

        DSS property name: `class`, DSS property index: 17.
        """
        return self._lib.Obj_GetInt32(self._ptr, 17)

    @cls.setter
    def cls(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 17, value)

    @property
    def Vpu(self) -> float:
        """
        Per Unit voltage set point for Model = 3  (typical power flow model).  Default is 1.0. 

        DSS property name: `Vpu`, DSS property index: 18.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 18)

    @Vpu.setter
    def Vpu(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 18, value)

    @property
    def maxkvar(self) -> float:
        """
        Maximum kvar limit for Model = 3.  Defaults to twice the specified load kvar.  Always reset this if you change PF or kvar properties.

        DSS property name: `maxkvar`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @maxkvar.setter
    def maxkvar(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def minkvar(self) -> float:
        """
        Minimum kvar limit for Model = 3. Enter a negative number if generator can absorb vars. Defaults to negative of Maxkvar.  Always reset this if you change PF or kvar properties.

        DSS property name: `minkvar`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @minkvar.setter
    def minkvar(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    @property
    def pvfactor(self) -> float:
        """
        Deceleration factor for P-V generator model (Model=3).  Default is 0.1. If the circuit converges easily, you may want to use a higher number such as 1.0. Use a lower number if solution diverges. Use Debugtrace=yes to create a file that will trace the convergence of a generator model.

        DSS property name: `pvfactor`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @pvfactor.setter
    def pvfactor(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def forceon(self) -> bool:
        """
        {Yes | No}  Forces generator ON despite requirements of other dispatch modes. Stays ON until this property is set to NO, or an internal algorithm cancels the forced ON state.

        DSS property name: `forceon`, DSS property index: 22.
        """
        return self._lib.Obj_GetInt32(self._ptr, 22) != 0

    @forceon.setter
    def forceon(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 22, value)

    @property
    def kVA(self) -> float:
        """
        kVA rating of electrical machine. Defaults to 1.2* kW if not specified. Applied to machine or inverter definition for Dynamics mode solutions. 

        DSS property name: `kVA`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @kVA.setter
    def kVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def Xd(self) -> float:
        """
        Per unit synchronous reactance of machine. Presently used only for Thevinen impedance for power flow calcs of user models (model=6). Typically use a value 0.4 to 1.0. Default is 1.0

        DSS property name: `Xd`, DSS property index: 25.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 25)

    @Xd.setter
    def Xd(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 25, value)

    @property
    def Xdp(self) -> float:
        """
        Per unit transient reactance of the machine.  Used for Dynamics mode and Fault studies.  Default is 0.27.For user models, this value is used for the Thevinen/Norton impedance for Dynamics Mode.

        DSS property name: `Xdp`, DSS property index: 26.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 26)

    @Xdp.setter
    def Xdp(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 26, value)

    @property
    def Xdpp(self) -> float:
        """
        Per unit subtransient reactance of the machine.  Used for Harmonics. Default is 0.20.

        DSS property name: `Xdpp`, DSS property index: 27.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 27)

    @Xdpp.setter
    def Xdpp(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 27, value)

    @property
    def H(self) -> float:
        """
        Per unit mass constant of the machine.  MW-sec/MVA.  Default is 1.0.

        DSS property name: `H`, DSS property index: 28.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 28)

    @H.setter
    def H(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 28, value)

    @property
    def D(self) -> float:
        """
        Damping constant.  Usual range is 0 to 4. Default is 1.0.  Adjust to get damping

        DSS property name: `D`, DSS property index: 29.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 29)

    @D.setter
    def D(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 29, value)

    @property
    def UserModel(self) -> str:
        """
        Name of DLL containing user-written model, which computes the terminal currents for Dynamics studies, overriding the default model.  Set to "none" to negate previous setting.

        DSS property name: `UserModel`, DSS property index: 30.
        """
        return self._get_prop_string(30)

    @UserModel.setter
    def UserModel(self, value: AnyStr):
        self._set_string_o(30, value)

    @property
    def UserData(self) -> str:
        """
        String (in quotes or parentheses) that gets passed to user-written model for defining the data required for that model.

        DSS property name: `UserData`, DSS property index: 31.
        """
        return self._get_prop_string(31)

    @UserData.setter
    def UserData(self, value: AnyStr):
        self._set_string_o(31, value)

    @property
    def ShaftModel(self) -> str:
        """
        Name of user-written DLL containing a Shaft model, which models the prime mover and determines the power on the shaft for Dynamics studies. Models additional mass elements other than the single-mass model in the DSS default model. Set to "none" to negate previous setting.

        DSS property name: `ShaftModel`, DSS property index: 32.
        """
        return self._get_prop_string(32)

    @ShaftModel.setter
    def ShaftModel(self, value: AnyStr):
        self._set_string_o(32, value)

    @property
    def ShaftData(self) -> str:
        """
        String (in quotes or parentheses) that gets passed to user-written shaft dynamic model for defining the data for that model.

        DSS property name: `ShaftData`, DSS property index: 33.
        """
        return self._get_prop_string(33)

    @ShaftData.setter
    def ShaftData(self, value: AnyStr):
        self._set_string_o(33, value)

    @property
    def DutyStart(self) -> float:
        """
        Starting time offset [hours] into the duty cycle shape for this generator, defaults to 0

        DSS property name: `DutyStart`, DSS property index: 34.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 34)

    @DutyStart.setter
    def DutyStart(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 34, value)

    @property
    def debugtrace(self) -> bool:
        """
        {Yes | No }  Default is no.  Turn this on to capture the progress of the generator model for each iteration.  Creates a separate file for each generator named "GEN_name.csv".

        DSS property name: `debugtrace`, DSS property index: 35.
        """
        return self._lib.Obj_GetInt32(self._ptr, 35) != 0

    @debugtrace.setter
    def debugtrace(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 35, value)

    @property
    def Balanced(self) -> bool:
        """
        {Yes | No*} Default is No.  For Model=7, force balanced current only for 3-phase generators. Force zero- and negative-sequence to zero.

        DSS property name: `Balanced`, DSS property index: 36.
        """
        return self._lib.Obj_GetInt32(self._ptr, 36) != 0

    @Balanced.setter
    def Balanced(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 36, value)

    @property
    def XRdp(self) -> float:
        """
        Default is 20. X/R ratio for Xdp property for FaultStudy and Dynamic modes.

        DSS property name: `XRdp`, DSS property index: 37.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 37)

    @XRdp.setter
    def XRdp(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 37, value)

    @property
    def UseFuel(self) -> bool:
        """
        {Yes | *No}. Activates the use of fuel for the operation of the generator. When the fuel level reaches the reserve level, the generator stops until it gets refueled. By default, the generator is connected to a continuous fuel supply, Use this mode to mimic dependency on fuel level for different generation technologies.

        DSS property name: `UseFuel`, DSS property index: 38.
        """
        return self._lib.Obj_GetInt32(self._ptr, 38) != 0

    @UseFuel.setter
    def UseFuel(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 38, value)

    @property
    def FuelkWh(self) -> float:
        """
        {*0}Is the nominal level of fuel for the generator (kWh). It only applies if UseFuel = Yes/True

        DSS property name: `FuelkWh`, DSS property index: 39.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 39)

    @FuelkWh.setter
    def FuelkWh(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 39, value)

    @property
    def pctFuel(self) -> float:
        """
        It is a number between 0 and 100 representing the current amount of fuel avaiable in percentage of FuelkWh. It only applies if UseFuel = Yes/True

        DSS property name: `%Fuel`, DSS property index: 40.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 40)

    @pctFuel.setter
    def pctFuel(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 40, value)

    @property
    def pctReserve(self) -> float:
        """
        It is a number between 0 and 100 representing the reserve level in percentage of FuelkWh. It only applies if UseFuel = Yes/True

        DSS property name: `%Reserve`, DSS property index: 41.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 41)

    @pctReserve.setter
    def pctReserve(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 41, value)

    def Refuel(self, value: bool):
        """
        It is a boolean value (Yes/True, No/False) that can be used to manually refuel the generator when needed. It only applies if UseFuel = Yes/True

        DSS property name: `Refuel`, DSS property index: 42.
        """
        self._lib.Obj_SetInt32(self._ptr, 42, value)

    @property
    def DynamicEq(self) -> str:
        """
        The name of the dynamic equation (DynamicExp) that will be used for defining the dynamic behavior of the generator. if not defined, the generator dynamics will follow the built-in dynamic equation.

        DSS property name: `DynamicEq`, DSS property index: 43.
        """
        return self._get_prop_string(43)

    @DynamicEq.setter
    def DynamicEq(self, value: Union[AnyStr, DynamicExp]):
        if isinstance(value, DSSObj):
            self._set_obj(43, value)
            return

        self._set_string_o(43, value)

    @property
    def DynamicEq_obj(self) -> DynamicExp:
        """
        The name of the dynamic equation (DynamicExp) that will be used for defining the dynamic behavior of the generator. if not defined, the generator dynamics will follow the built-in dynamic equation.

        DSS property name: `DynamicEq`, DSS property index: 43.
        """
        return self._get_obj(43, DynamicExp)

    @DynamicEq_obj.setter
    def DynamicEq_obj(self, value: DynamicExp):
        self._set_obj(43, value)

    @property
    def DynOut(self) -> str:
        """
        The name of the variables within the Dynamic equation that will be used to govern the generator dynamics.This generator model requires 2 outputs from the dynamic equation: 

        1. Shaft speed (velocity) relative to synchronous speed.
        2. Shaft, or power, angle (relative to synchronous reference frame).

        The output variables need to be defined in tha strict order.

        DSS property name: `DynOut`, DSS property index: 44.
        """
        return self._get_prop_string(44)

    @DynOut.setter
    def DynOut(self, value: AnyStr):
        self._set_string_o(44, value)

    @property
    def spectrum(self) -> str:
        """
        Name of harmonic voltage or current spectrum for this generator. Voltage behind Xd" for machine - default. Current injection for inverter. Default value is "default", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 45.
        """
        return self._get_prop_string(45)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum]):
        if isinstance(value, DSSObj):
            self._set_obj(45, value)
            return

        self._set_string_o(45, value)

    @property
    def spectrum_obj(self) -> Spectrum:
        """
        Name of harmonic voltage or current spectrum for this generator. Voltage behind Xd" for machine - default. Current injection for inverter. Default value is "default", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 45.
        """
        return self._get_obj(45, Spectrum)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_obj(45, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 46.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 46)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 46, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 47.
        """
        return self._lib.Obj_GetInt32(self._ptr, 47) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 47, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 48.
        """
        self._set_string_o(48, value)

class GenDispatcher(DSSObj):
    __slots__ = []
    _cls_name = 'GenDispatcher'
    _cls_idx = 28
    _cls_prop_idx = {
        'element': 1,
        'terminal': 2,
        'kwlimit': 3,
        'kwband': 4,
        'kvarlimit': 5,
        'genlist': 6,
        'weights': 7,
        'basefreq': 8,
        'enabled': 9,
        'like': 10,
    }

    @property
    def Element(self) -> str:
        """
        Full object name of the circuit element, typically a line or transformer, which the control is monitoring. There is no default; must be specified.

        DSS property name: `Element`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @Element.setter
    def Element(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(1, value)
            return

        self._set_string_o(1, value)

    @property
    def Element_obj(self) -> DSSObj:
        """
        Full object name of the circuit element, typically a line or transformer, which the control is monitoring. There is no default; must be specified.

        DSS property name: `Element`, DSS property index: 1.
        """
        return self._get_obj(1, None)

    @Element_obj.setter
    def Element_obj(self, value: DSSObj):
        self._set_obj(1, value)

    @property
    def Terminal(self) -> int:
        """
        Number of the terminal of the circuit element to which the GenDispatcher control is connected. 1 or 2, typically.  Default is 1. Make sure you have the direction on the power matching the sign of kWLimit.

        DSS property name: `Terminal`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @Terminal.setter
    def Terminal(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def kWLimit(self) -> float:
        """
        kW Limit for the monitored element. The generators are dispatched to hold the power in band.

        DSS property name: `kWLimit`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @kWLimit.setter
    def kWLimit(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def kWBand(self) -> float:
        """
        Bandwidth (kW) of the dead band around the target limit.No dispatch changes are attempted if the power in the monitored terminal stays within this band.

        DSS property name: `kWBand`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @kWBand.setter
    def kWBand(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def kvarlimit(self) -> float:
        """
        Max kvar to be delivered through the element.  Uses same dead band as kW.

        DSS property name: `kvarlimit`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @kvarlimit.setter
    def kvarlimit(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def GenList(self) -> List[str]:
        """
        Array list of generators to be dispatched.  If not specified, all generators in the circuit are assumed dispatchable.

        DSS property name: `GenList`, DSS property index: 6.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 6)

    @GenList.setter
    def GenList(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 6, value_ptr, value_count)
        self._check_for_error()

    @property
    def Weights(self) -> Float64Array:
        """
        GenDispatcher.Weights

        DSS property name: `Weights`, DSS property index: 7.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 7)

    @Weights.setter
    def Weights(self, value: Float64Array):
        self._set_float64_array_o(7, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 9.
        """
        return self._lib.Obj_GetInt32(self._ptr, 9) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 9, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 10.
        """
        self._set_string_o(10, value)

class Storage(DSSObj):
    __slots__ = []
    _cls_name = 'Storage'
    _cls_idx = 29
    _cls_prop_idx = {
        'phases': 1,
        'bus1': 2,
        'kv': 3,
        'conn': 4,
        'kw': 5,
        'kvar': 6,
        'pf': 7,
        'kva': 8,
        'pctcutin': 9,
        '%cutin': 9,
        'pctcutout': 10,
        '%cutout': 10,
        'effcurve': 11,
        'varfollowinverter': 12,
        'kvarmax': 13,
        'kvarmaxabs': 14,
        'wattpriority': 15,
        'pfpriority': 16,
        'pctpminnovars': 17,
        '%pminnovars': 17,
        'pctpminkvarmax': 18,
        '%pminkvarmax': 18,
        'kwrated': 19,
        'pctkwrated': 20,
        '%kwrated': 20,
        'kwhrated': 21,
        'kwhstored': 22,
        'pctstored': 23,
        '%stored': 23,
        'pctreserve': 24,
        '%reserve': 24,
        'state': 25,
        'pctdischarge': 26,
        '%discharge': 26,
        'pctcharge': 27,
        '%charge': 27,
        'pcteffcharge': 28,
        '%effcharge': 28,
        'pcteffdischarge': 29,
        '%effdischarge': 29,
        'pctidlingkw': 30,
        '%idlingkw': 30,
        'pctidlingkvar': 31,
        '%idlingkvar': 31,
        'pctr': 32,
        '%r': 32,
        'pctx': 33,
        '%x': 33,
        'model': 34,
        'vminpu': 35,
        'vmaxpu': 36,
        'balanced': 37,
        'limitcurrent': 38,
        'yearly': 39,
        'daily': 40,
        'duty': 41,
        'dispmode': 42,
        'dischargetrigger': 43,
        'chargetrigger': 44,
        'timechargetrig': 45,
        'cls': 46,
        'class': 46,
        'dynadll': 47,
        'dynadata': 48,
        'usermodel': 49,
        'userdata': 50,
        'debugtrace': 51,
        'kvdc': 52,
        'kp': 53,
        'pitol': 54,
        'safevoltage': 55,
        'safemode': 56,
        'dynamiceq': 57,
        'dynout': 58,
        'controlmode': 59,
        'spectrum': 60,
        'basefreq': 61,
        'enabled': 62,
        'like': 63,
    }

    # Class-specific enumerations
    class StorageState(IntEnum):
        """Storage: State (DSS enumeration for Storage)"""
        Charging = -1 # Charging
        Idling = 0 # Idling
        Discharging = 1 # Discharging

    class StorageDispatchMode(IntEnum):
        """Storage: Dispatch Mode (DSS enumeration for Storage)"""
        Default = 0 # Default
        LoadLevel = 1 # LoadLevel
        Price = 2 # Price
        External = 3 # External
        Follow = 4 # Follow


    @property
    def phases(self) -> int:
        """
        Number of Phases, this Storage element.  Power is evenly divided among phases.

        DSS property name: `phases`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def bus1(self) -> str:
        """
        Bus to which the Storage element is connected.  May include specific node specification.

        DSS property name: `bus1`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(2, value)

    @property
    def kv(self) -> float:
        """
        Nominal rated (1.0 per unit) voltage, kV, for Storage element. For 2- and 3-phase Storage elements, specify phase-phase kV. Otherwise, specify actual kV across each branch of the Storage element. 

        If wye (star), specify phase-neutral kV. 

        If delta or phase-phase connected, specify phase-phase kV.

        DSS property name: `kv`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @kv.setter
    def kv(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def conn(self) -> Connection:
        """
        ={wye|LN|delta|LL}.  Default is wye.

        DSS property name: `conn`, DSS property index: 4.
        """
        return Connection(self._lib.Obj_GetInt32(self._ptr, 4))

    @conn.setter
    def conn(self, value: Union[AnyStr, int, Connection]):
        if not isinstance(value, int):
            self._set_string_o(4, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 4, value)

    @property
    def conn_str(self) -> str:
        """
        ={wye|LN|delta|LL}.  Default is wye.

        DSS property name: `conn`, DSS property index: 4.
        """
        return self._get_prop_string(4)

    @conn_str.setter
    def conn_str(self, value: AnyStr):
        self.conn = value

    @property
    def kW(self) -> float:
        """
        Get/set the requested kW value. Final kW is subjected to the inverter ratings. A positive value denotes power coming OUT of the element, which is the opposite of a Load element. A negative value indicates the Storage element is in Charging state. This value is modified internally depending on the dispatch mode.

        DSS property name: `kW`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @kW.setter
    def kW(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def kvar(self) -> float:
        """
        Get/set the requested kvar value. Final kvar is subjected to the inverter ratings. Sets inverter to operate in constant kvar mode.

        DSS property name: `kvar`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @kvar.setter
    def kvar(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def pf(self) -> float:
        """
        Get/set the requested PF value. Final PF is subjected to the inverter ratings. Sets inverter to operate in constant PF mode. Nominally, the power factor for discharging (acting as a generator). Default is 1.0. 

        Enter negative for leading power factor (when kW and kvar have opposite signs.)

        A positive power factor signifies kw and kvar at the same direction.

        DSS property name: `pf`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @pf.setter
    def pf(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def kVA(self) -> float:
        """
        Indicates the inverter nameplate capability (in kVA). Used as the base for Dynamics mode and Harmonics mode values.

        DSS property name: `kVA`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @kVA.setter
    def kVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def pctCutin(self) -> float:
        """
        Cut-in power as a percentage of inverter kVA rating. It is the minimum DC power necessary to turn the inverter ON when it is OFF. Must be greater than or equal to %CutOut. Defaults to 2 for PVSystems and 0 for Storage elements which means that the inverter state will be always ON for this element.

        DSS property name: `%Cutin`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @pctCutin.setter
    def pctCutin(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def pctCutout(self) -> float:
        """
        Cut-out power as a percentage of inverter kVA rating. It is the minimum DC power necessary to keep the inverter ON. Must be less than or equal to %CutIn. Defaults to 0, which means that, once ON, the inverter state will be always ON for this element.

        DSS property name: `%Cutout`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @pctCutout.setter
    def pctCutout(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def EffCurve(self) -> str:
        """
        An XYCurve object, previously defined, that describes the PER UNIT efficiency vs PER UNIT of rated kVA for the inverter. Power at the AC side of the inverter is discounted by the multiplier obtained from this curve.

        DSS property name: `EffCurve`, DSS property index: 11.
        """
        return self._get_prop_string(11)

    @EffCurve.setter
    def EffCurve(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(11, value)
            return

        self._set_string_o(11, value)

    @property
    def EffCurve_obj(self) -> XYcurve:
        """
        An XYCurve object, previously defined, that describes the PER UNIT efficiency vs PER UNIT of rated kVA for the inverter. Power at the AC side of the inverter is discounted by the multiplier obtained from this curve.

        DSS property name: `EffCurve`, DSS property index: 11.
        """
        return self._get_obj(11, XYcurve)

    @EffCurve_obj.setter
    def EffCurve_obj(self, value: XYcurve):
        self._set_obj(11, value)

    @property
    def VarFollowInverter(self) -> bool:
        """
        Boolean variable (Yes|No) or (True|False). Defaults to False, which indicates that the reactive power generation/absorption does not respect the inverter status.When set to True, the reactive power generation/absorption will cease when the inverter status is off, due to DC kW dropping below %CutOut.  The reactive power generation/absorption will begin again when the DC kW is above %CutIn.  When set to False, the Storage will generate/absorb reactive power regardless of the status of the inverter.

        DSS property name: `VarFollowInverter`, DSS property index: 12.
        """
        return self._lib.Obj_GetInt32(self._ptr, 12) != 0

    @VarFollowInverter.setter
    def VarFollowInverter(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 12, value)

    @property
    def kvarMax(self) -> float:
        """
        Indicates the maximum reactive power GENERATION (un-signed numerical variable in kvar) for the inverter. Defaults to kVA rating of the inverter.

        DSS property name: `kvarMax`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @kvarMax.setter
    def kvarMax(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def kvarMaxAbs(self) -> float:
        """
        Indicates the maximum reactive power ABSORPTION (un-signed numerical variable in kvar) for the inverter. Defaults to kvarMax.

        DSS property name: `kvarMaxAbs`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @kvarMaxAbs.setter
    def kvarMaxAbs(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def WattPriority(self) -> bool:
        """
        {Yes/No*/True/False} Set inverter to watt priority instead of the default var priority.

        DSS property name: `WattPriority`, DSS property index: 15.
        """
        return self._lib.Obj_GetInt32(self._ptr, 15) != 0

    @WattPriority.setter
    def WattPriority(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 15, value)

    @property
    def PFPriority(self) -> bool:
        """
        If set to true, priority is given to power factor and WattPriority is neglected. It works only if operating in either constant PF or constant kvar modes. Defaults to False.

        DSS property name: `PFPriority`, DSS property index: 16.
        """
        return self._lib.Obj_GetInt32(self._ptr, 16) != 0

    @PFPriority.setter
    def PFPriority(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 16, value)

    @property
    def pctPminNoVars(self) -> float:
        """
        Minimum active power as percentage of kWrated under which there is no vars production/absorption. Defaults to 0 (disabled).

        DSS property name: `%PminNoVars`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @pctPminNoVars.setter
    def pctPminNoVars(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def pctPminkvarMax(self) -> float:
        """
        Minimum active power as percentage of kWrated that allows the inverter to produce/absorb reactive power up to its maximum reactive power, which can be either kvarMax or kvarMaxAbs, depending on the current operation quadrant. Defaults to 0 (disabled).

        DSS property name: `%PminkvarMax`, DSS property index: 18.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 18)

    @pctPminkvarMax.setter
    def pctPminkvarMax(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 18, value)

    @property
    def kWrated(self) -> float:
        """
        kW rating of power output. Base for Loadshapes when DispMode=Follow. Sets kVA property if it has not been specified yet. Defaults to 25.

        DSS property name: `kWrated`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @kWrated.setter
    def kWrated(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def pctkWrated(self) -> float:
        """
        Upper limit on active power as a percentage of kWrated. Defaults to 100 (disabled).

        DSS property name: `%kWrated`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @pctkWrated.setter
    def pctkWrated(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    @property
    def kWhrated(self) -> float:
        """
        Rated Storage capacity in kWh. Default is 50.

        DSS property name: `kWhrated`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @kWhrated.setter
    def kWhrated(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def kWhstored(self) -> float:
        """
        Present amount of energy stored, kWh. Default is same as kWhrated.

        DSS property name: `kWhstored`, DSS property index: 22.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 22)

    @kWhstored.setter
    def kWhstored(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 22, value)

    @property
    def pctstored(self) -> float:
        """
        Present amount of energy stored, % of rated kWh. Default is 100.

        DSS property name: `%stored`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @pctstored.setter
    def pctstored(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def pctreserve(self) -> float:
        """
        Percentage of rated kWh Storage capacity to be held in reserve for normal operation. Default = 20. 
        This is treated as the minimum energy discharge level unless there is an emergency. For emergency operation set this property lower. Cannot be less than zero.

        DSS property name: `%reserve`, DSS property index: 24.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 24)

    @pctreserve.setter
    def pctreserve(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 24, value)

    @property
    def State(self) -> StorageState:
        """
        {IDLING | CHARGING | DISCHARGING}  Get/Set present operational state. In DISCHARGING mode, the Storage element acts as a generator and the kW property is positive. The element continues discharging at the scheduled output power level until the Storage reaches the reserve value. Then the state reverts to IDLING. In the CHARGING state, the Storage element behaves like a Load and the kW property is negative. The element continues to charge until the max Storage kWh is reached and then switches to IDLING state. In IDLING state, the element draws the idling losses plus the associated inverter losses.

        DSS property name: `State`, DSS property index: 25.
        """
        return Storage.StorageState(self._lib.Obj_GetInt32(self._ptr, 25))

    @State.setter
    def State(self, value: Union[AnyStr, int, StorageState]):
        if not isinstance(value, int):
            self._set_string_o(25, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 25, value)

    @property
    def State_str(self) -> str:
        """
        {IDLING | CHARGING | DISCHARGING}  Get/Set present operational state. In DISCHARGING mode, the Storage element acts as a generator and the kW property is positive. The element continues discharging at the scheduled output power level until the Storage reaches the reserve value. Then the state reverts to IDLING. In the CHARGING state, the Storage element behaves like a Load and the kW property is negative. The element continues to charge until the max Storage kWh is reached and then switches to IDLING state. In IDLING state, the element draws the idling losses plus the associated inverter losses.

        DSS property name: `State`, DSS property index: 25.
        """
        return self._get_prop_string(25)

    @State_str.setter
    def State_str(self, value: AnyStr):
        self.State = value

    @property
    def pctDischarge(self) -> float:
        """
        Discharge rate (output power) in percentage of rated kW. Default = 100.

        DSS property name: `%Discharge`, DSS property index: 26.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 26)

    @pctDischarge.setter
    def pctDischarge(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 26, value)

    @property
    def pctCharge(self) -> float:
        """
        Charging rate (input power) in percentage of rated kW. Default = 100.

        DSS property name: `%Charge`, DSS property index: 27.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 27)

    @pctCharge.setter
    def pctCharge(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 27, value)

    @property
    def pctEffCharge(self) -> float:
        """
        Percentage efficiency for CHARGING the Storage element. Default = 90.

        DSS property name: `%EffCharge`, DSS property index: 28.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 28)

    @pctEffCharge.setter
    def pctEffCharge(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 28, value)

    @property
    def pctEffDischarge(self) -> float:
        """
        Percentage efficiency for DISCHARGING the Storage element. Default = 90.

        DSS property name: `%EffDischarge`, DSS property index: 29.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 29)

    @pctEffDischarge.setter
    def pctEffDischarge(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 29, value)

    @property
    def pctIdlingkW(self) -> float:
        """
        Percentage of rated kW consumed by idling losses. Default = 1.

        DSS property name: `%IdlingkW`, DSS property index: 30.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 30)

    @pctIdlingkW.setter
    def pctIdlingkW(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 30, value)

    @property
    def pctR(self) -> float:
        """
        Equivalent percentage internal resistance, ohms. Default is 0. Placed in series with internal voltage source for harmonics and dynamics modes. Use a combination of %IdlingkW, %EffCharge and %EffDischarge to account for losses in power flow modes.

        DSS property name: `%R`, DSS property index: 32.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 32)

    @pctR.setter
    def pctR(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 32, value)

    @property
    def pctX(self) -> float:
        """
        Equivalent percentage internal reactance, ohms. Default is 50%. Placed in series with internal voltage source for harmonics and dynamics modes. (Limits fault current to 2 pu.

        DSS property name: `%X`, DSS property index: 33.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 33)

    @pctX.setter
    def pctX(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 33, value)

    @property
    def model(self) -> int:
        """
        Integer code (default=1) for the model to be used for power output variation with voltage. Valid values are:

        1:Storage element injects/absorbs a CONSTANT power.
        2:Storage element is modeled as a CONSTANT IMPEDANCE.
        3:Compute load injection from User-written Model.

        DSS property name: `model`, DSS property index: 34.
        """
        return self._lib.Obj_GetInt32(self._ptr, 34)

    @model.setter
    def model(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 34, value)

    @property
    def Vminpu(self) -> float:
        """
        Default = 0.90.  Minimum per unit voltage for which the Model is assumed to apply. Below this value, the load model reverts to a constant impedance model.

        DSS property name: `Vminpu`, DSS property index: 35.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 35)

    @Vminpu.setter
    def Vminpu(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 35, value)

    @property
    def Vmaxpu(self) -> float:
        """
        Default = 1.10.  Maximum per unit voltage for which the Model is assumed to apply. Above this value, the load model reverts to a constant impedance model.

        DSS property name: `Vmaxpu`, DSS property index: 36.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 36)

    @Vmaxpu.setter
    def Vmaxpu(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 36, value)

    @property
    def Balanced(self) -> bool:
        """
        {Yes | No*} Default is No. Force balanced current only for 3-phase Storage. Forces zero- and negative-sequence to zero. 

        DSS property name: `Balanced`, DSS property index: 37.
        """
        return self._lib.Obj_GetInt32(self._ptr, 37) != 0

    @Balanced.setter
    def Balanced(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 37, value)

    @property
    def LimitCurrent(self) -> bool:
        """
        Limits current magnitude to Vminpu value for both 1-phase and 3-phase Storage similar to Generator Model 7. For 3-phase, limits the positive-sequence current but not the negative-sequence.

        DSS property name: `LimitCurrent`, DSS property index: 38.
        """
        return self._lib.Obj_GetInt32(self._ptr, 38) != 0

    @LimitCurrent.setter
    def LimitCurrent(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 38, value)

    @property
    def yearly(self) -> str:
        """
        Dispatch shape to use for yearly simulations.  Must be previously defined as a Loadshape object. If this is not specified, the Daily dispatch shape, if any, is repeated during Yearly solution modes. In the default dispatch mode, the Storage element uses this loadshape to trigger State changes.

        DSS property name: `yearly`, DSS property index: 39.
        """
        return self._get_prop_string(39)

    @yearly.setter
    def yearly(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(39, value)
            return

        self._set_string_o(39, value)

    @property
    def yearly_obj(self) -> LoadShape:
        """
        Dispatch shape to use for yearly simulations.  Must be previously defined as a Loadshape object. If this is not specified, the Daily dispatch shape, if any, is repeated during Yearly solution modes. In the default dispatch mode, the Storage element uses this loadshape to trigger State changes.

        DSS property name: `yearly`, DSS property index: 39.
        """
        return self._get_obj(39, LoadShape)

    @yearly_obj.setter
    def yearly_obj(self, value: LoadShape):
        self._set_obj(39, value)

    @property
    def daily(self) -> str:
        """
        Dispatch shape to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically.  In the default dispatch mode, the Storage element uses this loadshape to trigger State changes.

        DSS property name: `daily`, DSS property index: 40.
        """
        return self._get_prop_string(40)

    @daily.setter
    def daily(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(40, value)
            return

        self._set_string_o(40, value)

    @property
    def daily_obj(self) -> LoadShape:
        """
        Dispatch shape to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically.  In the default dispatch mode, the Storage element uses this loadshape to trigger State changes.

        DSS property name: `daily`, DSS property index: 40.
        """
        return self._get_obj(40, LoadShape)

    @daily_obj.setter
    def daily_obj(self, value: LoadShape):
        self._set_obj(40, value)

    @property
    def duty(self) -> str:
        """
        Load shape to use for duty cycle dispatch simulations such as for solar ramp rate studies. Must be previously defined as a Loadshape object. 

        Typically would have time intervals of 1-5 seconds. 

        Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.

        DSS property name: `duty`, DSS property index: 41.
        """
        return self._get_prop_string(41)

    @duty.setter
    def duty(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(41, value)
            return

        self._set_string_o(41, value)

    @property
    def duty_obj(self) -> LoadShape:
        """
        Load shape to use for duty cycle dispatch simulations such as for solar ramp rate studies. Must be previously defined as a Loadshape object. 

        Typically would have time intervals of 1-5 seconds. 

        Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.

        DSS property name: `duty`, DSS property index: 41.
        """
        return self._get_obj(41, LoadShape)

    @duty_obj.setter
    def duty_obj(self, value: LoadShape):
        self._set_obj(41, value)

    @property
    def DispMode(self) -> StorageDispatchMode:
        """
        {DEFAULT | FOLLOW | EXTERNAL | LOADLEVEL | PRICE } Default = "DEFAULT". Dispatch mode. 

        In DEFAULT mode, Storage element state is triggered to discharge or charge at the specified rate by the loadshape curve corresponding to the solution mode. 

        In FOLLOW mode the kW output of the Storage element follows the active loadshape multiplier until Storage is either exhausted or full. The element discharges for positive values and charges for negative values.  The loadshape is based on rated kW. 

        In EXTERNAL mode, Storage element state is controlled by an external Storagecontroller. This mode is automatically set if this Storage element is included in the element list of a StorageController element. 

        For the other two dispatch modes, the Storage element state is controlled by either the global default Loadlevel value or the price level. 

        DSS property name: `DispMode`, DSS property index: 42.
        """
        return Storage.StorageDispatchMode(self._lib.Obj_GetInt32(self._ptr, 42))

    @DispMode.setter
    def DispMode(self, value: Union[AnyStr, int, StorageDispatchMode]):
        if not isinstance(value, int):
            self._set_string_o(42, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 42, value)

    @property
    def DispMode_str(self) -> str:
        """
        {DEFAULT | FOLLOW | EXTERNAL | LOADLEVEL | PRICE } Default = "DEFAULT". Dispatch mode. 

        In DEFAULT mode, Storage element state is triggered to discharge or charge at the specified rate by the loadshape curve corresponding to the solution mode. 

        In FOLLOW mode the kW output of the Storage element follows the active loadshape multiplier until Storage is either exhausted or full. The element discharges for positive values and charges for negative values.  The loadshape is based on rated kW. 

        In EXTERNAL mode, Storage element state is controlled by an external Storagecontroller. This mode is automatically set if this Storage element is included in the element list of a StorageController element. 

        For the other two dispatch modes, the Storage element state is controlled by either the global default Loadlevel value or the price level. 

        DSS property name: `DispMode`, DSS property index: 42.
        """
        return self._get_prop_string(42)

    @DispMode_str.setter
    def DispMode_str(self, value: AnyStr):
        self.DispMode = value

    @property
    def DischargeTrigger(self) -> float:
        """
        Dispatch trigger value for discharging the Storage. 
        If = 0.0 the Storage element state is changed by the State command or by a StorageController object. 
        If <> 0  the Storage element state is set to DISCHARGING when this trigger level is EXCEEDED by either the specified Loadshape curve value or the price signal or global Loadlevel value, depending on dispatch mode. See State property.

        DSS property name: `DischargeTrigger`, DSS property index: 43.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 43)

    @DischargeTrigger.setter
    def DischargeTrigger(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 43, value)

    @property
    def ChargeTrigger(self) -> float:
        """
        Dispatch trigger value for charging the Storage. 

        If = 0.0 the Storage element state is changed by the State command or StorageController object.  

        If <> 0  the Storage element state is set to CHARGING when this trigger level is GREATER than either the specified Loadshape curve value or the price signal or global Loadlevel value, depending on dispatch mode. See State property.

        DSS property name: `ChargeTrigger`, DSS property index: 44.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 44)

    @ChargeTrigger.setter
    def ChargeTrigger(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 44, value)

    @property
    def TimeChargeTrig(self) -> float:
        """
        Time of day in fractional hours (0230 = 2.5) at which Storage element will automatically go into charge state. Default is 2.0.  Enter a negative time value to disable this feature.

        DSS property name: `TimeChargeTrig`, DSS property index: 45.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 45)

    @TimeChargeTrig.setter
    def TimeChargeTrig(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 45, value)

    @property
    def cls(self) -> int:
        """
        An arbitrary integer number representing the class of Storage element so that Storage values may be segregated by class.

        DSS property name: `class`, DSS property index: 46.
        """
        return self._lib.Obj_GetInt32(self._ptr, 46)

    @cls.setter
    def cls(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 46, value)

    @property
    def DynaDLL(self) -> str:
        """
        Name of DLL containing user-written dynamics model, which computes the terminal currents for Dynamics-mode simulations, overriding the default model.  Set to "none" to negate previous setting. This DLL has a simpler interface than the UserModel DLL and is only used for Dynamics mode.

        DSS property name: `DynaDLL`, DSS property index: 47.
        """
        return self._get_prop_string(47)

    @DynaDLL.setter
    def DynaDLL(self, value: AnyStr):
        self._set_string_o(47, value)

    @property
    def DynaData(self) -> str:
        """
        String (in quotes or parentheses if necessary) that gets passed to the user-written dynamics model Edit function for defining the data required for that model.

        DSS property name: `DynaData`, DSS property index: 48.
        """
        return self._get_prop_string(48)

    @DynaData.setter
    def DynaData(self, value: AnyStr):
        self._set_string_o(48, value)

    @property
    def UserModel(self) -> str:
        """
        Name of DLL containing user-written model, which computes the terminal currents for both power flow and dynamics, overriding the default model.  Set to "none" to negate previous setting.

        DSS property name: `UserModel`, DSS property index: 49.
        """
        return self._get_prop_string(49)

    @UserModel.setter
    def UserModel(self, value: AnyStr):
        self._set_string_o(49, value)

    @property
    def UserData(self) -> str:
        """
        String (in quotes or parentheses) that gets passed to user-written model for defining the data required for that model.

        DSS property name: `UserData`, DSS property index: 50.
        """
        return self._get_prop_string(50)

    @UserData.setter
    def UserData(self, value: AnyStr):
        self._set_string_o(50, value)

    @property
    def debugtrace(self) -> bool:
        """
        {Yes | No }  Default is no.  Turn this on to capture the progress of the Storage model for each iteration.  Creates a separate file for each Storage element named "Storage_name.csv".

        DSS property name: `debugtrace`, DSS property index: 51.
        """
        return self._lib.Obj_GetInt32(self._ptr, 51) != 0

    @debugtrace.setter
    def debugtrace(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 51, value)

    @property
    def kVDC(self) -> float:
        """
        Indicates the rated voltage (kV) at the input of the inverter while the storage is discharging. The value is normally greater or equal to the kV base of the Storage device. It is used for dynamics simulation ONLY.

        DSS property name: `kVDC`, DSS property index: 52.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 52)

    @kVDC.setter
    def kVDC(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 52, value)

    @property
    def Kp(self) -> float:
        """
        It is the proportional gain for the PI controller within the inverter. Use it to modify the controller response in dynamics simulation mode.

        DSS property name: `Kp`, DSS property index: 53.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 53)

    @Kp.setter
    def Kp(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 53, value)

    @property
    def PITol(self) -> float:
        """
        It is the tolerance (%) for the closed loop controller of the inverter. For dynamics simulation mode.

        DSS property name: `PITol`, DSS property index: 54.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 54)

    @PITol.setter
    def PITol(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 54, value)

    @property
    def SafeVoltage(self) -> float:
        """
        Indicates the voltage level (%) respect to the base voltage level for which the Inverter will operate. If this threshold is violated, the Inverter will enter safe mode (OFF). For dynamic simulation. By default is 80%.

        DSS property name: `SafeVoltage`, DSS property index: 55.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 55)

    @SafeVoltage.setter
    def SafeVoltage(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 55, value)

    @property
    def SafeMode(self) -> bool:
        """
        (Read only) Indicates whether the inverter entered (Yes) or not (No) into Safe Mode.

        DSS property name: `SafeMode`, DSS property index: 56.
        """
        return self._lib.Obj_GetInt32(self._ptr, 56) != 0

    @SafeMode.setter
    def SafeMode(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 56, value)

    @property
    def DynamicEq(self) -> str:
        """
        The name of the dynamic equation (DynamicExp) that will be used for defining the dynamic behavior of the generator. If not defined, the generator dynamics will follow the built-in dynamic equation.

        DSS property name: `DynamicEq`, DSS property index: 57.
        """
        return self._get_prop_string(57)

    @DynamicEq.setter
    def DynamicEq(self, value: Union[AnyStr, DynamicExp]):
        if isinstance(value, DSSObj):
            self._set_obj(57, value)
            return

        self._set_string_o(57, value)

    @property
    def DynamicEq_obj(self) -> DynamicExp:
        """
        The name of the dynamic equation (DynamicExp) that will be used for defining the dynamic behavior of the generator. If not defined, the generator dynamics will follow the built-in dynamic equation.

        DSS property name: `DynamicEq`, DSS property index: 57.
        """
        return self._get_obj(57, DynamicExp)

    @DynamicEq_obj.setter
    def DynamicEq_obj(self, value: DynamicExp):
        self._set_obj(57, value)

    @property
    def DynOut(self) -> str:
        """
        The name of the variables within the Dynamic equation that will be used to govern the Storage dynamics. This Storage model requires 1 output from the dynamic equation:

            1. Current.

        The output variables need to be defined in the same order.

        DSS property name: `DynOut`, DSS property index: 58.
        """
        return self._get_prop_string(58)

    @DynOut.setter
    def DynOut(self, value: AnyStr):
        self._set_string_o(58, value)

    @property
    def ControlMode(self) -> InverterControlMode:
        """
        Defines the control mode for the inverter. It can be one of {GFM | GFL*}. By default it is GFL (Grid Following Inverter). Use GFM (Grid Forming Inverter) for energizing islanded microgrids, but, if the device is conencted to the grid, it is highly recommended to use GFL.

        GFM control mode disables any control action set by the InvControl device.

        DSS property name: `ControlMode`, DSS property index: 59.
        """
        return InverterControlMode(self._lib.Obj_GetInt32(self._ptr, 59))

    @ControlMode.setter
    def ControlMode(self, value: Union[AnyStr, int, InverterControlMode]):
        if not isinstance(value, int):
            self._set_string_o(59, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 59, value)

    @property
    def ControlMode_str(self) -> str:
        """
        Defines the control mode for the inverter. It can be one of {GFM | GFL*}. By default it is GFL (Grid Following Inverter). Use GFM (Grid Forming Inverter) for energizing islanded microgrids, but, if the device is conencted to the grid, it is highly recommended to use GFL.

        GFM control mode disables any control action set by the InvControl device.

        DSS property name: `ControlMode`, DSS property index: 59.
        """
        return self._get_prop_string(59)

    @ControlMode_str.setter
    def ControlMode_str(self, value: AnyStr):
        self.ControlMode = value

    @property
    def spectrum(self) -> str:
        """
        Name of harmonic voltage or current spectrum for this Storage element. Current injection is assumed for inverter. Default value is "default", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 60.
        """
        return self._get_prop_string(60)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum]):
        if isinstance(value, DSSObj):
            self._set_obj(60, value)
            return

        self._set_string_o(60, value)

    @property
    def spectrum_obj(self) -> Spectrum:
        """
        Name of harmonic voltage or current spectrum for this Storage element. Current injection is assumed for inverter. Default value is "default", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 60.
        """
        return self._get_obj(60, Spectrum)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_obj(60, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 61.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 61)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 61, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 62.
        """
        return self._lib.Obj_GetInt32(self._ptr, 62) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 62, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 63.
        """
        self._set_string_o(63, value)

class StorageController(DSSObj):
    __slots__ = []
    _cls_name = 'StorageController'
    _cls_idx = 30
    _cls_prop_idx = {
        'element': 1,
        'terminal': 2,
        'monphase': 3,
        'kwtarget': 4,
        'kwtargetlow': 5,
        'pctkwband': 6,
        '%kwband': 6,
        'kwband': 7,
        'pctkwbandlow': 8,
        '%kwbandlow': 8,
        'kwbandlow': 9,
        'elementlist': 10,
        'weights': 11,
        'modedischarge': 12,
        'modecharge': 13,
        'timedischargetrigger': 14,
        'timechargetrigger': 15,
        'pctratekw': 16,
        '%ratekw': 16,
        'pctratecharge': 17,
        '%ratecharge': 17,
        'pctreserve': 18,
        '%reserve': 18,
        'kwhtotal': 19,
        'kwtotal': 20,
        'kwhactual': 21,
        'kwactual': 22,
        'kwneed': 23,
        'yearly': 24,
        'daily': 25,
        'duty': 26,
        'eventlog': 27,
        'inhibittime': 28,
        'tup': 29,
        'tflat': 30,
        'tdn': 31,
        'kwthreshold': 32,
        'dispfactor': 33,
        'resetlevel': 34,
        'seasons': 35,
        'seasontargets': 36,
        'seasontargetslow': 37,
        'basefreq': 38,
        'enabled': 39,
        'like': 40,
    }

    # Class-specific enumerations
    class StorageControllerDischargemode(IntEnum):
        """StorageController: Discharge mode (DSS enumeration for StorageController)"""
        Peakshave = 5 # Peakshave
        Follow = 1 # Follow
        Support = 3 # Support
        Loadshape = 2 # Loadshape
        Time = 4 # Time
        Schedule = 6 # Schedule
        I_Peakshave = 8 # I-Peakshave

    class StorageControllerChargemode(IntEnum):
        """StorageController: Charge mode (DSS enumeration for StorageController)"""
        Loadshape = 2 # Loadshape
        Time = 4 # Time
        PeakshaveLow = 7 # PeakshaveLow
        I_PeakshaveLow = 9 # I-PeakshaveLow


    @property
    def Element(self) -> str:
        """
        Full object name of the circuit element, typically a line or transformer, which the control is monitoring. There is no default; Must be specified.

        DSS property name: `Element`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @Element.setter
    def Element(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(1, value)
            return

        self._set_string_o(1, value)

    @property
    def Element_obj(self) -> DSSObj:
        """
        Full object name of the circuit element, typically a line or transformer, which the control is monitoring. There is no default; Must be specified.

        DSS property name: `Element`, DSS property index: 1.
        """
        return self._get_obj(1, None)

    @Element_obj.setter
    def Element_obj(self, value: DSSObj):
        self._set_obj(1, value)

    @property
    def Terminal(self) -> int:
        """
        Number of the terminal of the circuit element to which the StorageController control is connected. 1 or 2, typically.  Default is 1. Make sure to select the proper direction on the power for the respective dispatch mode.

        DSS property name: `Terminal`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @Terminal.setter
    def Terminal(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def MonPhase(self) -> Union[MonitoredPhase, int]:
        """
        Number of the phase being monitored or one of {AVG | MAX | MIN} for all phases. Default=MAX. Must be less than the number of phases. Used in PeakShave, Follow, Support and I-PeakShave discharging modes and in PeakShaveLow, I-PeakShaveLow charging modes. For modes based on active power measurements, the value used by the control is the monitored one multiplied by the number of phases of the monitored element.

        DSS property name: `MonPhase`, DSS property index: 3.
        """
        value = self._lib.Obj_GetInt32(self._ptr, 3)
        if value > 0:
            return value
    
        return MonitoredPhase(value)

    @MonPhase.setter
    def MonPhase(self, value: Union[AnyStr, int, MonitoredPhase]):
        if not isinstance(value, int):
            self._set_string_o(3, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 3, value)

    @property
    def MonPhase_str(self) -> str:
        """
        Number of the phase being monitored or one of {AVG | MAX | MIN} for all phases. Default=MAX. Must be less than the number of phases. Used in PeakShave, Follow, Support and I-PeakShave discharging modes and in PeakShaveLow, I-PeakShaveLow charging modes. For modes based on active power measurements, the value used by the control is the monitored one multiplied by the number of phases of the monitored element.

        DSS property name: `MonPhase`, DSS property index: 3.
        """
        return self._get_prop_string(3)

    @MonPhase_str.setter
    def MonPhase_str(self, value: AnyStr):
        self.MonPhase = value

    @property
    def kWTarget(self) -> float:
        """
        kW/kamps target for Discharging. The Storage element fleet is dispatched to try to hold the power/current in band at least until the Storage is depleted. The selection of power or current depends on the Discharge mode (PeakShave->kW, I-PeakShave->kamps).

        DSS property name: `kWTarget`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @kWTarget.setter
    def kWTarget(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def kWTargetLow(self) -> float:
        """
        kW/kamps target for Charging. The Storage element fleet is dispatched to try to hold the power/current in band at least until the Storage is fully charged. The selection of power or current depends on the charge mode (PeakShavelow->kW, I-PeakShavelow->kamps).

        DSS property name: `kWTargetLow`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @kWTargetLow.setter
    def kWTargetLow(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def pctkWBand(self) -> float:
        """
        Bandwidth (% of Target kW/kamps) of the dead band around the kW/kamps target value. Default is 2% (+/-1%).No dispatch changes are attempted if the power in the monitored terminal stays within this band.

        DSS property name: `%kWBand`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @pctkWBand.setter
    def pctkWBand(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def kWBand(self) -> float:
        """
        Alternative way of specifying the bandwidth. (kW/kamps) of the dead band around the kW/kamps target value. Default is 2% of kWTarget (+/-1%).No dispatch changes are attempted if the power in the monitored terminal stays within this band.

        DSS property name: `kWBand`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @kWBand.setter
    def kWBand(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def pctkWBandLow(self) -> float:
        """
        Bandwidth (% of kWTargetLow) of the dead band around the kW/kamps low target value. Default is 2% (+/-1%).No charging is attempted if the power in the monitored terminal stays within this band.

        DSS property name: `%kWBandLow`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @pctkWBandLow.setter
    def pctkWBandLow(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def kWBandLow(self) -> float:
        """
        Alternative way of specifying the bandwidth. (kW/kamps) of the dead band around the kW/kamps low target value. Default is 2% of kWTargetLow (+/-1%).No charging is attempted if the power in the monitored terminal stays within this band.

        DSS property name: `kWBandLow`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @kWBandLow.setter
    def kWBandLow(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def ElementList(self) -> List[str]:
        """
        Array list of Storage elements to be controlled.  If not specified, all Storage elements in the circuit not presently dispatched by another controller are assumed dispatched by this controller.

        DSS property name: `ElementList`, DSS property index: 10.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 10)

    @ElementList.setter
    def ElementList(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 10, value_ptr, value_count)
        self._check_for_error()

    @property
    def Weights(self) -> Float64Array:
        """
        Array of proportional weights corresponding to each Storage element in the ElementList. The needed kW or kvar to get back to center band is dispatched to each Storage element according to these weights. Default is to set all weights to 1.0.

        DSS property name: `Weights`, DSS property index: 11.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 11)

    @Weights.setter
    def Weights(self, value: Float64Array):
        self._set_float64_array_o(11, value)

    @property
    def ModeDischarge(self) -> StorageControllerDischargemode:
        """
        {PeakShave* | Follow | Support | Loadshape | Time | Schedule | I-PeakShave} Mode of operation for the DISCHARGE FUNCTION of this controller. 

        In PeakShave mode (Default), the control attempts to discharge Storage to keep power in the monitored element below the kWTarget. 

        In Follow mode, the control is triggered by time and resets the kWTarget value to the present monitored element power. It then attempts to discharge Storage to keep power in the monitored element below the new kWTarget. See TimeDischargeTrigger.

        In Support mode, the control operates oppositely of PeakShave mode: Storage is discharged to keep kW power output up near the target. 

        In Loadshape mode, both charging and discharging precisely follows the per unit loadshape. Storage is discharged when the loadshape value is positive. 

        In Time mode, the Storage discharge is turned on at the specified %RatekW at the specified discharge trigger time in fractional hours.

        In Schedule mode, the Tup, TFlat, and Tdn properties specify the up ramp duration, flat duration, and down ramp duration for the schedule. The schedule start time is set by TimeDischargeTrigger and the rate of discharge for the flat part is determined by %RatekW.

        In I-PeakShave mode, the control attempts to discharge Storage to keep current in the monitored element below the target given in k-amps (thousands of amps), when this control mode is active, the property kWTarget will be expressed in k-amps. 

        DSS property name: `ModeDischarge`, DSS property index: 12.
        """
        return StorageController.StorageControllerDischargemode(self._lib.Obj_GetInt32(self._ptr, 12))

    @ModeDischarge.setter
    def ModeDischarge(self, value: Union[AnyStr, int, StorageControllerDischargemode]):
        if not isinstance(value, int):
            self._set_string_o(12, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 12, value)

    @property
    def ModeDischarge_str(self) -> str:
        """
        {PeakShave* | Follow | Support | Loadshape | Time | Schedule | I-PeakShave} Mode of operation for the DISCHARGE FUNCTION of this controller. 

        In PeakShave mode (Default), the control attempts to discharge Storage to keep power in the monitored element below the kWTarget. 

        In Follow mode, the control is triggered by time and resets the kWTarget value to the present monitored element power. It then attempts to discharge Storage to keep power in the monitored element below the new kWTarget. See TimeDischargeTrigger.

        In Support mode, the control operates oppositely of PeakShave mode: Storage is discharged to keep kW power output up near the target. 

        In Loadshape mode, both charging and discharging precisely follows the per unit loadshape. Storage is discharged when the loadshape value is positive. 

        In Time mode, the Storage discharge is turned on at the specified %RatekW at the specified discharge trigger time in fractional hours.

        In Schedule mode, the Tup, TFlat, and Tdn properties specify the up ramp duration, flat duration, and down ramp duration for the schedule. The schedule start time is set by TimeDischargeTrigger and the rate of discharge for the flat part is determined by %RatekW.

        In I-PeakShave mode, the control attempts to discharge Storage to keep current in the monitored element below the target given in k-amps (thousands of amps), when this control mode is active, the property kWTarget will be expressed in k-amps. 

        DSS property name: `ModeDischarge`, DSS property index: 12.
        """
        return self._get_prop_string(12)

    @ModeDischarge_str.setter
    def ModeDischarge_str(self, value: AnyStr):
        self.ModeDischarge = value

    @property
    def ModeCharge(self) -> StorageControllerChargemode:
        """
        {Loadshape | Time* | PeakShaveLow | I-PeakShaveLow} Mode of operation for the CHARGE FUNCTION of this controller. 

        In Loadshape mode, both charging and discharging precisely follows the per unit loadshape. Storage is charged when the loadshape value is negative. 

        In Time mode, the Storage charging FUNCTION is triggered at the specified %RateCharge at the specified charge trigger time in fractional hours.

        In PeakShaveLow mode, the charging operation will charge the Storage fleet when the power at amonitored element is below a specified KW target (kWTarget_low). The Storage will charge as much power as necessary to keep the power within the deadband around kWTarget_low.

        In I-PeakShaveLow mode, the charging operation will charge the Storage fleet when the current (Amps) at amonitored element is below a specified amps target (kWTarget_low). The Storage will charge as much power as necessary to keep the amps within the deadband around kWTarget_low. When this control mode is active, the property kWTarget_low will be expressed in k-amps and all the other parameters will be adjusted to match the amps (current) control criteria.

        DSS property name: `ModeCharge`, DSS property index: 13.
        """
        return StorageController.StorageControllerChargemode(self._lib.Obj_GetInt32(self._ptr, 13))

    @ModeCharge.setter
    def ModeCharge(self, value: Union[AnyStr, int, StorageControllerChargemode]):
        if not isinstance(value, int):
            self._set_string_o(13, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 13, value)

    @property
    def ModeCharge_str(self) -> str:
        """
        {Loadshape | Time* | PeakShaveLow | I-PeakShaveLow} Mode of operation for the CHARGE FUNCTION of this controller. 

        In Loadshape mode, both charging and discharging precisely follows the per unit loadshape. Storage is charged when the loadshape value is negative. 

        In Time mode, the Storage charging FUNCTION is triggered at the specified %RateCharge at the specified charge trigger time in fractional hours.

        In PeakShaveLow mode, the charging operation will charge the Storage fleet when the power at amonitored element is below a specified KW target (kWTarget_low). The Storage will charge as much power as necessary to keep the power within the deadband around kWTarget_low.

        In I-PeakShaveLow mode, the charging operation will charge the Storage fleet when the current (Amps) at amonitored element is below a specified amps target (kWTarget_low). The Storage will charge as much power as necessary to keep the amps within the deadband around kWTarget_low. When this control mode is active, the property kWTarget_low will be expressed in k-amps and all the other parameters will be adjusted to match the amps (current) control criteria.

        DSS property name: `ModeCharge`, DSS property index: 13.
        """
        return self._get_prop_string(13)

    @ModeCharge_str.setter
    def ModeCharge_str(self, value: AnyStr):
        self.ModeCharge = value

    @property
    def TimeDischargeTrigger(self) -> float:
        """
        Default time of day (hr) for initiating Discharging of the fleet. During Follow or Time mode discharging is triggered at a fixed time each day at this hour. If Follow mode, Storage will be discharged to attempt to hold the load at or below the power level at the time of triggering. In Time mode, the discharge is based on the %RatekW property value. Set this to a negative value to ignore. Default is 12.0 for Follow mode; otherwise it is -1 (ignored). 

        DSS property name: `TimeDischargeTrigger`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @TimeDischargeTrigger.setter
    def TimeDischargeTrigger(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def TimeChargeTrigger(self) -> float:
        """
        Default time of day (hr) for initiating charging in Time control mode. Set this to a negative value to ignore. Default is 2.0.  (0200).When this value is >0 the Storage fleet is set to charging at this time regardless of other control criteria to make sure Storage is topped off for the next discharge cycle.

        DSS property name: `TimeChargeTrigger`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @TimeChargeTrigger.setter
    def TimeChargeTrigger(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def pctRatekW(self) -> float:
        """
        Sets the kW discharge rate in % of rated capacity for each element of the fleet. Applies to TIME control mode, SCHEDULE mode, or anytime discharging is triggered by time.

        DSS property name: `%RatekW`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @pctRatekW.setter
    def pctRatekW(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def pctRateCharge(self) -> float:
        """
        Sets the kW charging rate in % of rated capacity for each element of the fleet. Applies to TIME control mode and anytime charging mode is entered due to a time trigger.

        DSS property name: `%RateCharge`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @pctRateCharge.setter
    def pctRateCharge(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def pctReserve(self) -> float:
        """
        Use this property to change the % reserve for each Storage element under control of this controller. This might be used, for example, to allow deeper discharges of Storage or in case of emergency operation to use the remainder of the Storage element.

        DSS property name: `%Reserve`, DSS property index: 18.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 18)

    @pctReserve.setter
    def pctReserve(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 18, value)

    @property
    def kWhTotal(self) -> float:
        """
        (Read only). Total rated kWh energy Storage capacity of Storage elements controlled by this controller.

        DSS property name: `kWhTotal`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @kWhTotal.setter
    def kWhTotal(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def kWTotal(self) -> float:
        """
        (Read only). Total rated kW power capacity of Storage elements controlled by this controller.

        DSS property name: `kWTotal`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @kWTotal.setter
    def kWTotal(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    @property
    def kWhActual(self) -> float:
        """
        (Read only). Actual kWh stored of all controlled Storage elements. 

        DSS property name: `kWhActual`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @kWhActual.setter
    def kWhActual(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def kWActual(self) -> float:
        """
        (Read only). Actual kW output of all controlled Storage elements. 

        DSS property name: `kWActual`, DSS property index: 22.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 22)

    @kWActual.setter
    def kWActual(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 22, value)

    @property
    def kWneed(self) -> float:
        """
        (Read only). KW needed to meet target.

        DSS property name: `kWneed`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @kWneed.setter
    def kWneed(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def Yearly(self) -> str:
        """
        Dispatch loadshape object, If any, for Yearly solution Mode.

        DSS property name: `Yearly`, DSS property index: 24.
        """
        return self._get_prop_string(24)

    @Yearly.setter
    def Yearly(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(24, value)
            return

        self._set_string_o(24, value)

    @property
    def Yearly_obj(self) -> LoadShape:
        """
        Dispatch loadshape object, If any, for Yearly solution Mode.

        DSS property name: `Yearly`, DSS property index: 24.
        """
        return self._get_obj(24, LoadShape)

    @Yearly_obj.setter
    def Yearly_obj(self, value: LoadShape):
        self._set_obj(24, value)

    @property
    def Daily(self) -> str:
        """
        Dispatch loadshape object, If any, for Daily solution mode.

        DSS property name: `Daily`, DSS property index: 25.
        """
        return self._get_prop_string(25)

    @Daily.setter
    def Daily(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(25, value)
            return

        self._set_string_o(25, value)

    @property
    def Daily_obj(self) -> LoadShape:
        """
        Dispatch loadshape object, If any, for Daily solution mode.

        DSS property name: `Daily`, DSS property index: 25.
        """
        return self._get_obj(25, LoadShape)

    @Daily_obj.setter
    def Daily_obj(self, value: LoadShape):
        self._set_obj(25, value)

    @property
    def Duty(self) -> str:
        """
        Dispatch loadshape object, If any, for Dutycycle solution mode.

        DSS property name: `Duty`, DSS property index: 26.
        """
        return self._get_prop_string(26)

    @Duty.setter
    def Duty(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(26, value)
            return

        self._set_string_o(26, value)

    @property
    def Duty_obj(self) -> LoadShape:
        """
        Dispatch loadshape object, If any, for Dutycycle solution mode.

        DSS property name: `Duty`, DSS property index: 26.
        """
        return self._get_obj(26, LoadShape)

    @Duty_obj.setter
    def Duty_obj(self, value: LoadShape):
        self._set_obj(26, value)

    @property
    def EventLog(self) -> bool:
        """
        {Yes/True | No/False} Default is No. Log control actions to Eventlog.

        DSS property name: `EventLog`, DSS property index: 27.
        """
        return self._lib.Obj_GetInt32(self._ptr, 27) != 0

    @EventLog.setter
    def EventLog(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 27, value)

    @property
    def InhibitTime(self) -> int:
        """
        Hours (integer) to inhibit Discharging after going into Charge mode. Default is 5.

        DSS property name: `InhibitTime`, DSS property index: 28.
        """
        return self._lib.Obj_GetInt32(self._ptr, 28)

    @InhibitTime.setter
    def InhibitTime(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 28, value)

    @property
    def Tup(self) -> float:
        """
        Duration, hrs, of upramp part for SCHEDULE mode. Default is 0.25.

        DSS property name: `Tup`, DSS property index: 29.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 29)

    @Tup.setter
    def Tup(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 29, value)

    @property
    def TFlat(self) -> float:
        """
        Duration, hrs, of flat part for SCHEDULE mode. Default is 2.0.

        DSS property name: `TFlat`, DSS property index: 30.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 30)

    @TFlat.setter
    def TFlat(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 30, value)

    @property
    def Tdn(self) -> float:
        """
        Duration, hrs, of downramp part for SCHEDULE mode. Default is 0.25.

        DSS property name: `Tdn`, DSS property index: 31.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 31)

    @Tdn.setter
    def Tdn(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 31, value)

    @property
    def kWThreshold(self) -> float:
        """
        Threshold, kW, for Follow mode. kW has to be above this value for the Storage element to be dispatched on. Defaults to 75% of the kWTarget value. Must reset this property after setting kWTarget if you want a different value.

        DSS property name: `kWThreshold`, DSS property index: 32.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 32)

    @kWThreshold.setter
    def kWThreshold(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 32, value)

    @property
    def DispFactor(self) -> float:
        """
        Defaults to 1 (disabled). Set to any value between 0 and 1 to enable this parameter.

        Use this parameter to reduce the amount of power requested by the controller in each control iteration. It can be useful when maximum control iterations are exceeded due to numerical instability such as fleet being set to charging and idling in subsequent control iterations (check the Eventlog). 

        DSS property name: `DispFactor`, DSS property index: 33.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 33)

    @DispFactor.setter
    def DispFactor(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 33, value)

    @property
    def ResetLevel(self) -> float:
        """
        The level of charge required for allowing the storage to discharge again after reaching the reserve storage level. After reaching this level, the storage control  will not allow the storage device to discharge, forcing the storage to charge. Once the storage reaches thislevel, the storage will be able to discharge again. This value is a number between 0.2 and 1

        DSS property name: `ResetLevel`, DSS property index: 34.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 34)

    @ResetLevel.setter
    def ResetLevel(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 34, value)

    @property
    def Seasons(self) -> int:
        """
        With this property the user can specify the number of targets to be used by the controller using the list given at "SeasonTargets"/"SeasonTargetsLow", which can be used to dynamically adjust the storage controller during a QSTS simulation. The default value is 1. This property needs to be defined before defining SeasonTargets/SeasonTargetsLow.

        DSS property name: `Seasons`, DSS property index: 35.
        """
        return self._lib.Obj_GetInt32(self._ptr, 35)

    @Seasons.setter
    def Seasons(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 35, value)

    @property
    def SeasonTargets(self) -> Float64Array:
        """
        An array of doubles specifying the targets to be used during a QSTS simulation. These targets will take effect only if SeasonRating=true. The number of targets cannot exceed the number of seasons defined at the SeasonSignal.The difference between the targets defined at SeasonTargets and SeasonTargetsLow is that SeasonTargets applies to discharging modes, while SeasonTargetsLow applies to charging modes.

        DSS property name: `SeasonTargets`, DSS property index: 36.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 36)

    @SeasonTargets.setter
    def SeasonTargets(self, value: Float64Array):
        self._set_float64_array_o(36, value)

    @property
    def SeasonTargetsLow(self) -> Float64Array:
        """
        An array of doubles specifying the targets to be used during a QSTS simulation. These targets will take effect only if SeasonRating=true. The number of targets cannot exceed the number of seasons defined at the SeasonSignal.The difference between the targets defined at SeasonTargets and SeasonTargetsLow is that SeasonTargets applies to discharging modes, while SeasonTargetsLow applies to charging modes.

        DSS property name: `SeasonTargetsLow`, DSS property index: 37.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 37)

    @SeasonTargetsLow.setter
    def SeasonTargetsLow(self, value: Float64Array):
        self._set_float64_array_o(37, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 38.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 38)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 38, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 39.
        """
        return self._lib.Obj_GetInt32(self._ptr, 39) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 39, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 40.
        """
        self._set_string_o(40, value)

class Relay(DSSObj):
    __slots__ = []
    _cls_name = 'Relay'
    _cls_idx = 31
    _cls_prop_idx = {
        'monitoredobj': 1,
        'monitoredterm': 2,
        'switchedobj': 3,
        'switchedterm': 4,
        'type': 5,
        'phasecurve': 6,
        'groundcurve': 7,
        'phasetrip': 8,
        'groundtrip': 9,
        'tdphase': 10,
        'tdground': 11,
        'phaseinst': 12,
        'groundinst': 13,
        'reset': 14,
        'shots': 15,
        'recloseintervals': 16,
        'delay': 17,
        'overvoltcurve': 18,
        'undervoltcurve': 19,
        'kvbase': 20,
        'pctpickup47': 21,
        '47%pickup': 21,
        'baseamps46': 22,
        '46baseamps': 22,
        'pctpickup46': 23,
        '46%pickup': 23,
        'isqt46': 24,
        '46isqt': 24,
        'variable': 25,
        'overtrip': 26,
        'undertrip': 27,
        'breakertime': 28,
        'action': 29,
        'z1mag': 30,
        'z1ang': 31,
        'z0mag': 32,
        'z0ang': 33,
        'mphase': 34,
        'mground': 35,
        'eventlog': 36,
        'debugtrace': 37,
        'distreverse': 38,
        'normal': 39,
        'state': 40,
        'doc_tiltanglelow': 41,
        'doc_tiltanglehigh': 42,
        'doc_tripsettinglow': 43,
        'doc_tripsettinghigh': 44,
        'doc_tripsettingmag': 45,
        'doc_delayinner': 46,
        'doc_phasecurveinner': 47,
        'doc_phasetripinner': 48,
        'doc_tdphaseinner': 49,
        'doc_p1blocking': 50,
        'basefreq': 51,
        'enabled': 52,
        'like': 53,
    }

    # Class-specific enumerations
    class RelayType(IntEnum):
        """Relay: Type (DSS enumeration for Relay)"""
        Current = 0 # Current
        Voltage = 1 # Voltage
        ReversePower = 3 # ReversePower
        relay46 = 4 # 46
        relay47 = 5 # 47
        Generic = 6 # Generic
        Distance = 7 # Distance
        TD21 = 8 # TD21
        DOC = 9 # DOC

    class RelayAction(IntEnum):
        """Relay: Action (DSS enumeration for Relay)"""
        close = 2 # close
        open = 1 # open
        trip = 1 # trip

    class RelayState(IntEnum):
        """Relay: State (DSS enumeration for Relay)"""
        closed = 2 # closed
        open = 1 # open
        trip = 1 # trip


    @property
    def MonitoredObj(self) -> str:
        """
        Full object name of the circuit element, typically a line, transformer, load, or generator, to which the relay's PT and/or CT are connected. This is the "monitored" element. There is no default; must be specified.

        DSS property name: `MonitoredObj`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @MonitoredObj.setter
    def MonitoredObj(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(1, value)
            return

        self._set_string_o(1, value)

    @property
    def MonitoredObj_obj(self) -> DSSObj:
        """
        Full object name of the circuit element, typically a line, transformer, load, or generator, to which the relay's PT and/or CT are connected. This is the "monitored" element. There is no default; must be specified.

        DSS property name: `MonitoredObj`, DSS property index: 1.
        """
        return self._get_obj(1, None)

    @MonitoredObj_obj.setter
    def MonitoredObj_obj(self, value: DSSObj):
        self._set_obj(1, value)

    @property
    def MonitoredTerm(self) -> int:
        """
        Number of the terminal of the circuit element to which the Relay is connected. 1 or 2, typically.  Default is 1.

        DSS property name: `MonitoredTerm`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @MonitoredTerm.setter
    def MonitoredTerm(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def SwitchedObj(self) -> str:
        """
        Name of circuit element switch that the Relay controls. Specify the full object name.Defaults to the same as the Monitored element. This is the "controlled" element.

        DSS property name: `SwitchedObj`, DSS property index: 3.
        """
        return self._get_prop_string(3)

    @SwitchedObj.setter
    def SwitchedObj(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(3, value)
            return

        self._set_string_o(3, value)

    @property
    def SwitchedObj_obj(self) -> DSSObj:
        """
        Name of circuit element switch that the Relay controls. Specify the full object name.Defaults to the same as the Monitored element. This is the "controlled" element.

        DSS property name: `SwitchedObj`, DSS property index: 3.
        """
        return self._get_obj(3, None)

    @SwitchedObj_obj.setter
    def SwitchedObj_obj(self, value: DSSObj):
        self._set_obj(3, value)

    @property
    def SwitchedTerm(self) -> int:
        """
        Number of the terminal of the controlled element in which the switch is controlled by the Relay. 1 or 2, typically.  Default is 1.

        DSS property name: `SwitchedTerm`, DSS property index: 4.
        """
        return self._lib.Obj_GetInt32(self._ptr, 4)

    @SwitchedTerm.setter
    def SwitchedTerm(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 4, value)

    @property
    def type(self) -> RelayType:
        """
        One of a legal relay type:
          Current
          Voltage
          Reversepower
          46 (neg seq current)
          47 (neg seq voltage)
          Generic (generic over/under relay)
          Distance
          TD21
          DOC (directional overcurrent)

        Default is overcurrent relay (Current) Specify the curve and pickup settings appropriate for each type. Generic relays monitor PC Element Control variables and trip on out of over/under range in definite time.

        DSS property name: `type`, DSS property index: 5.
        """
        return Relay.RelayType(self._lib.Obj_GetInt32(self._ptr, 5))

    @type.setter
    def type(self, value: Union[AnyStr, int, RelayType]):
        if not isinstance(value, int):
            self._set_string_o(5, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 5, value)

    @property
    def type_str(self) -> str:
        """
        One of a legal relay type:
          Current
          Voltage
          Reversepower
          46 (neg seq current)
          47 (neg seq voltage)
          Generic (generic over/under relay)
          Distance
          TD21
          DOC (directional overcurrent)

        Default is overcurrent relay (Current) Specify the curve and pickup settings appropriate for each type. Generic relays monitor PC Element Control variables and trip on out of over/under range in definite time.

        DSS property name: `type`, DSS property index: 5.
        """
        return self._get_prop_string(5)

    @type_str.setter
    def type_str(self, value: AnyStr):
        self.type = value

    @property
    def Phasecurve(self) -> str:
        """
        Name of the TCC Curve object that determines the phase trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored). For overcurrent relay, multiplying the current values in the curve by the "phasetrip" value gives the actual current.

        DSS property name: `Phasecurve`, DSS property index: 6.
        """
        return self._get_prop_string(6)

    @Phasecurve.setter
    def Phasecurve(self, value: Union[AnyStr, TCC_Curve]):
        if isinstance(value, DSSObj):
            self._set_obj(6, value)
            return

        self._set_string_o(6, value)

    @property
    def Phasecurve_obj(self) -> TCC_Curve:
        """
        Name of the TCC Curve object that determines the phase trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored). For overcurrent relay, multiplying the current values in the curve by the "phasetrip" value gives the actual current.

        DSS property name: `Phasecurve`, DSS property index: 6.
        """
        return self._get_obj(6, TCC_Curve)

    @Phasecurve_obj.setter
    def Phasecurve_obj(self, value: TCC_Curve):
        self._set_obj(6, value)

    @property
    def Groundcurve(self) -> str:
        """
        Name of the TCC Curve object that determines the ground trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored).For overcurrent relay, multiplying the current values in the curve by the "groundtrip" valuw gives the actual current.

        DSS property name: `Groundcurve`, DSS property index: 7.
        """
        return self._get_prop_string(7)

    @Groundcurve.setter
    def Groundcurve(self, value: Union[AnyStr, TCC_Curve]):
        if isinstance(value, DSSObj):
            self._set_obj(7, value)
            return

        self._set_string_o(7, value)

    @property
    def Groundcurve_obj(self) -> TCC_Curve:
        """
        Name of the TCC Curve object that determines the ground trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored).For overcurrent relay, multiplying the current values in the curve by the "groundtrip" valuw gives the actual current.

        DSS property name: `Groundcurve`, DSS property index: 7.
        """
        return self._get_obj(7, TCC_Curve)

    @Groundcurve_obj.setter
    def Groundcurve_obj(self, value: TCC_Curve):
        self._set_obj(7, value)

    @property
    def PhaseTrip(self) -> float:
        """
        Multiplier or actual phase amps for the phase TCC curve.  Defaults to 1.0.

        DSS property name: `PhaseTrip`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @PhaseTrip.setter
    def PhaseTrip(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def GroundTrip(self) -> float:
        """
        Multiplier or actual ground amps (3I0) for the ground TCC curve.  Defaults to 1.0.

        DSS property name: `GroundTrip`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @GroundTrip.setter
    def GroundTrip(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def TDPhase(self) -> float:
        """
        Time dial for Phase trip curve. Multiplier on time axis of specified curve. Default=1.0.

        DSS property name: `TDPhase`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @TDPhase.setter
    def TDPhase(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def TDGround(self) -> float:
        """
        Time dial for Ground trip curve. Multiplier on time axis of specified curve. Default=1.0.

        DSS property name: `TDGround`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @TDGround.setter
    def TDGround(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def PhaseInst(self) -> float:
        """
        Actual  amps (Current relay) or kW (reverse power relay) for instantaneous phase trip which is assumed to happen in 0.01 sec + Delay Time. Default is 0.0, which signifies no inst trip. Use this value for specifying the Reverse Power threshold (kW) for reverse power relays.

        DSS property name: `PhaseInst`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @PhaseInst.setter
    def PhaseInst(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def GroundInst(self) -> float:
        """
        Actual  amps for instantaneous ground trip which is assumed to happen in 0.01 sec + Delay Time.Default is 0.0, which signifies no inst trip.

        DSS property name: `GroundInst`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @GroundInst.setter
    def GroundInst(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def Reset(self) -> float:
        """
        Reset time in sec for relay.  Default is 15. If this much time passes between the last pickup event, and the relay has not locked out, the operation counter resets.

        DSS property name: `Reset`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @Reset.setter
    def Reset(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def Shots(self) -> int:
        """
        Number of shots to lockout.  Default is 4. This is one more than the number of reclose intervals.

        DSS property name: `Shots`, DSS property index: 15.
        """
        return self._lib.Obj_GetInt32(self._ptr, 15)

    @Shots.setter
    def Shots(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 15, value)

    @property
    def RecloseIntervals(self) -> Float64Array:
        """
        Array of reclose intervals. If none, specify "NONE". Default for overcurrent relay is (0.5, 2.0, 2.0) seconds. Default for a voltage relay is (5.0). In a voltage relay, this is  seconds after restoration of voltage that the reclose occurs. Reverse power relay is one shot to lockout, so this is ignored.  A locked out relay must be closed manually (set action=close).

        DSS property name: `RecloseIntervals`, DSS property index: 16.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 16)

    @RecloseIntervals.setter
    def RecloseIntervals(self, value: Float64Array):
        self._set_float64_array_o(16, value)

    @property
    def Delay(self) -> float:
        """
        Trip time delay (sec) for DEFINITE TIME relays. Default is 0.0 for current, voltage and DOC relays. If >0 then this value is used instead of curves. Used by Generic, RevPower, 46 and 47 relays. Defaults to 0.1 s for these relays.

        DSS property name: `Delay`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @Delay.setter
    def Delay(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def Overvoltcurve(self) -> str:
        """
        TCC Curve object to use for overvoltage relay.  Curve is assumed to be defined with per unit voltage values. Voltage base should be defined for the relay. Default is none (ignored).

        DSS property name: `Overvoltcurve`, DSS property index: 18.
        """
        return self._get_prop_string(18)

    @Overvoltcurve.setter
    def Overvoltcurve(self, value: Union[AnyStr, TCC_Curve]):
        if isinstance(value, DSSObj):
            self._set_obj(18, value)
            return

        self._set_string_o(18, value)

    @property
    def Overvoltcurve_obj(self) -> TCC_Curve:
        """
        TCC Curve object to use for overvoltage relay.  Curve is assumed to be defined with per unit voltage values. Voltage base should be defined for the relay. Default is none (ignored).

        DSS property name: `Overvoltcurve`, DSS property index: 18.
        """
        return self._get_obj(18, TCC_Curve)

    @Overvoltcurve_obj.setter
    def Overvoltcurve_obj(self, value: TCC_Curve):
        self._set_obj(18, value)

    @property
    def Undervoltcurve(self) -> str:
        """
        TCC Curve object to use for undervoltage relay.  Curve is assumed to be defined with per unit voltage values. Voltage base should be defined for the relay. Default is none (ignored).

        DSS property name: `Undervoltcurve`, DSS property index: 19.
        """
        return self._get_prop_string(19)

    @Undervoltcurve.setter
    def Undervoltcurve(self, value: Union[AnyStr, TCC_Curve]):
        if isinstance(value, DSSObj):
            self._set_obj(19, value)
            return

        self._set_string_o(19, value)

    @property
    def Undervoltcurve_obj(self) -> TCC_Curve:
        """
        TCC Curve object to use for undervoltage relay.  Curve is assumed to be defined with per unit voltage values. Voltage base should be defined for the relay. Default is none (ignored).

        DSS property name: `Undervoltcurve`, DSS property index: 19.
        """
        return self._get_obj(19, TCC_Curve)

    @Undervoltcurve_obj.setter
    def Undervoltcurve_obj(self, value: TCC_Curve):
        self._set_obj(19, value)

    @property
    def kvbase(self) -> float:
        """
        Voltage base (kV) for the relay. Specify line-line for 3 phase devices); line-neutral for 1-phase devices.  Relay assumes the number of phases of the monitored element.  Default is 0.0, which results in assuming the voltage values in the "TCC" curve are specified in actual line-to-neutral volts.

        DSS property name: `kvbase`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @kvbase.setter
    def kvbase(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    @property
    def pctPickup47(self) -> float:
        """
        Percent voltage pickup for 47 relay (Neg seq voltage). Default is 2. Specify also base voltage (kvbase) and delay time value.   

        DSS property name: `47%Pickup`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @pctPickup47.setter
    def pctPickup47(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def BaseAmps46(self) -> float:
        """
        Base current, Amps, for 46 relay (neg seq current).  Used for establishing pickup and per unit I-squared-t.

        DSS property name: `46BaseAmps`, DSS property index: 22.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 22)

    @BaseAmps46.setter
    def BaseAmps46(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 22, value)

    @property
    def pctPickup46(self) -> float:
        """
        Percent pickup current for 46 relay (neg seq current).  Default is 20.0.   When current exceeds this value * BaseAmps, I-squared-t calc starts.

        DSS property name: `46%Pickup`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @pctPickup46.setter
    def pctPickup46(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def isqt46(self) -> float:
        """
        Negative Sequence I-squared-t trip value for 46 relay (neg seq current).  Default is 1 (trips in 1 sec for 1 per unit neg seq current).  Should be 1 to 99.

        DSS property name: `46isqt`, DSS property index: 24.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 24)

    @isqt46.setter
    def isqt46(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 24, value)

    @property
    def Variable(self) -> str:
        """
        Name of variable in PC Elements being monitored.  Only applies to Generic relay.

        DSS property name: `Variable`, DSS property index: 25.
        """
        return self._get_prop_string(25)

    @Variable.setter
    def Variable(self, value: AnyStr):
        self._set_string_o(25, value)

    @property
    def overtrip(self) -> float:
        """
        Trip setting (high value) for Generic relay variable.  Relay trips in definite time if value of variable exceeds this value.

        DSS property name: `overtrip`, DSS property index: 26.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 26)

    @overtrip.setter
    def overtrip(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 26, value)

    @property
    def undertrip(self) -> float:
        """
        Trip setting (low value) for Generic relay variable.  Relay trips in definite time if value of variable is less than this value.

        DSS property name: `undertrip`, DSS property index: 27.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 27)

    @undertrip.setter
    def undertrip(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 27, value)

    @property
    def Breakertime(self) -> float:
        """
        Fixed delay time (sec) added to relay time. Default is 0.0. Designed to represent breaker time or some other delay after a trip decision is made.Use Delay property for setting a fixed trip time delay.Added to trip time of current and voltage relays. Could use in combination with inst trip value to obtain a definite time overcurrent relay.

        DSS property name: `Breakertime`, DSS property index: 28.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 28)

    @Breakertime.setter
    def Breakertime(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 28, value)

    @property
    def action(self) -> RelayAction:
        """
        DEPRECATED. See "State" property

        DSS property name: `action`, DSS property index: 29.
        """
        return Relay.RelayAction(self._lib.Obj_GetInt32(self._ptr, 29))

    @action.setter
    def action(self, value: Union[AnyStr, int, RelayAction]):
        if not isinstance(value, int):
            self._set_string_o(29, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 29, value)

    @property
    def action_str(self) -> str:
        """
        DEPRECATED. See "State" property

        DSS property name: `action`, DSS property index: 29.
        """
        return self._get_prop_string(29)

    @action_str.setter
    def action_str(self, value: AnyStr):
        self.action = value

    @property
    def Z1mag(self) -> float:
        """
        Positive sequence reach impedance in primary ohms for Distance and TD21 functions. Default=0.7

        DSS property name: `Z1mag`, DSS property index: 30.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 30)

    @Z1mag.setter
    def Z1mag(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 30, value)

    @property
    def Z1ang(self) -> float:
        """
        Positive sequence reach impedance angle in degrees for Distance and TD21 functions. Default=64.0

        DSS property name: `Z1ang`, DSS property index: 31.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 31)

    @Z1ang.setter
    def Z1ang(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 31, value)

    @property
    def Z0mag(self) -> float:
        """
        Zero sequence reach impedance in primary ohms for Distance and TD21 functions. Default=2.1

        DSS property name: `Z0mag`, DSS property index: 32.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 32)

    @Z0mag.setter
    def Z0mag(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 32, value)

    @property
    def Z0ang(self) -> float:
        """
        Zero sequence reach impedance angle in degrees for Distance and TD21 functions. Default=68.0

        DSS property name: `Z0ang`, DSS property index: 33.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 33)

    @Z0ang.setter
    def Z0ang(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 33, value)

    @property
    def Mphase(self) -> float:
        """
        Phase reach multiplier in per-unit for Distance and TD21 functions. Default=0.7

        DSS property name: `Mphase`, DSS property index: 34.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 34)

    @Mphase.setter
    def Mphase(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 34, value)

    @property
    def Mground(self) -> float:
        """
        Ground reach multiplier in per-unit for Distance and TD21 functions. Default=0.7

        DSS property name: `Mground`, DSS property index: 35.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 35)

    @Mground.setter
    def Mground(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 35, value)

    @property
    def EventLog(self) -> bool:
        """
        {Yes/True* | No/False} Default is Yes for Relay. Write trips, reclose and reset events to EventLog.

        DSS property name: `EventLog`, DSS property index: 36.
        """
        return self._lib.Obj_GetInt32(self._ptr, 36) != 0

    @EventLog.setter
    def EventLog(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 36, value)

    @property
    def DebugTrace(self) -> bool:
        """
        {Yes/True* | No/False} Default is No for Relay. Write extra details to Eventlog.

        DSS property name: `DebugTrace`, DSS property index: 37.
        """
        return self._lib.Obj_GetInt32(self._ptr, 37) != 0

    @DebugTrace.setter
    def DebugTrace(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 37, value)

    @property
    def DistReverse(self) -> bool:
        """
        {Yes/True* | No/False} Default is No; reverse direction for distance and td21 types.

        DSS property name: `DistReverse`, DSS property index: 38.
        """
        return self._lib.Obj_GetInt32(self._ptr, 38) != 0

    @DistReverse.setter
    def DistReverse(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 38, value)

    @property
    def Normal(self) -> RelayState:
        """
        {Open | Closed} Normal state of the relay. The relay reverts to this state for reset, change of mode, etc. Defaults to "State" if not specifically declared.

        DSS property name: `Normal`, DSS property index: 39.
        """
        return Relay.RelayState(self._lib.Obj_GetInt32(self._ptr, 39))

    @Normal.setter
    def Normal(self, value: Union[AnyStr, int, RelayState]):
        if not isinstance(value, int):
            self._set_string_o(39, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 39, value)

    @property
    def Normal_str(self) -> str:
        """
        {Open | Closed} Normal state of the relay. The relay reverts to this state for reset, change of mode, etc. Defaults to "State" if not specifically declared.

        DSS property name: `Normal`, DSS property index: 39.
        """
        return self._get_prop_string(39)

    @Normal_str.setter
    def Normal_str(self, value: AnyStr):
        self.Normal = value

    @property
    def State(self) -> RelayState:
        """
        {Open | Closed} Actual state of the relay. Upon setting, immediately forces state of the relay, overriding the Relay control. Simulates manual control on relay. Defaults to Closed. "Open" causes the controlled element to open and lock out. "Closed" causes the controlled element to close and the relay to reset to its first operation.

        DSS property name: `State`, DSS property index: 40.
        """
        return Relay.RelayState(self._lib.Obj_GetInt32(self._ptr, 40))

    @State.setter
    def State(self, value: Union[AnyStr, int, RelayState]):
        if not isinstance(value, int):
            self._set_string_o(40, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 40, value)

    @property
    def State_str(self) -> str:
        """
        {Open | Closed} Actual state of the relay. Upon setting, immediately forces state of the relay, overriding the Relay control. Simulates manual control on relay. Defaults to Closed. "Open" causes the controlled element to open and lock out. "Closed" causes the controlled element to close and the relay to reset to its first operation.

        DSS property name: `State`, DSS property index: 40.
        """
        return self._get_prop_string(40)

    @State_str.setter
    def State_str(self, value: AnyStr):
        self.State = value

    @property
    def DOC_TiltAngleLow(self) -> float:
        """
        Tilt angle for low-current trip line. Default is 90.

        DSS property name: `DOC_TiltAngleLow`, DSS property index: 41.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 41)

    @DOC_TiltAngleLow.setter
    def DOC_TiltAngleLow(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 41, value)

    @property
    def DOC_TiltAngleHigh(self) -> float:
        """
        Tilt angle for high-current trip line. Default is 90.

        DSS property name: `DOC_TiltAngleHigh`, DSS property index: 42.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 42)

    @DOC_TiltAngleHigh.setter
    def DOC_TiltAngleHigh(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 42, value)

    @property
    def DOC_TripSettingLow(self) -> float:
        """
        Resistive trip setting for low-current line. Default is 0.

        DSS property name: `DOC_TripSettingLow`, DSS property index: 43.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 43)

    @DOC_TripSettingLow.setter
    def DOC_TripSettingLow(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 43, value)

    @property
    def DOC_TripSettingHigh(self) -> float:
        """
        Resistive trip setting for high-current line.  Default is -1 (deactivated). To activate, set a positive value. Must be greater than "DOC_TripSettingLow".

        DSS property name: `DOC_TripSettingHigh`, DSS property index: 44.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 44)

    @DOC_TripSettingHigh.setter
    def DOC_TripSettingHigh(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 44, value)

    @property
    def DOC_TripSettingMag(self) -> float:
        """
        Trip setting for current magnitude (defines a circle in the relay characteristics). Default is -1 (deactivated). To activate, set a positive value.

        DSS property name: `DOC_TripSettingMag`, DSS property index: 45.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 45)

    @DOC_TripSettingMag.setter
    def DOC_TripSettingMag(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 45, value)

    @property
    def DOC_DelayInner(self) -> float:
        """
        Trip time delay (sec) for operation in inner region for DOC relay, defined when "DOC_TripSettingMag" or "DOC_TripSettingHigh" are activate. Default is -1.0 (deactivated), meaning that the relay characteristic is insensitive in the inner region (no trip). Set to 0 for instantaneous trip and >0 for a definite time delay. If "DOC_PhaseCurveInner" is specified, time delay from curve is utilized instead.

        DSS property name: `DOC_DelayInner`, DSS property index: 46.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 46)

    @DOC_DelayInner.setter
    def DOC_DelayInner(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 46, value)

    @property
    def DOC_PhaseCurveInner(self) -> float:
        """
        Name of the TCC Curve object that determines the phase trip for operation in inner region for DOC relay. Must have been previously defined as a TCC_Curve object. Default is none (ignored). Multiplying the current values in the curve by the "DOC_PhaseTripInner" value gives the actual current.

        DSS property name: `DOC_PhaseCurveInner`, DSS property index: 47.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 47)

    @DOC_PhaseCurveInner.setter
    def DOC_PhaseCurveInner(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 47, value)

    @property
    def DOC_PhaseTripInner(self) -> float:
        """
        Multiplier for the "DOC_PhaseCurveInner" TCC curve.  Defaults to 1.0.

        DSS property name: `DOC_PhaseTripInner`, DSS property index: 48.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 48)

    @DOC_PhaseTripInner.setter
    def DOC_PhaseTripInner(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 48, value)

    @property
    def DOC_TDPhaseInner(self) -> str:
        """
        Time dial for "DOC_PhaseCurveInner" TCC curve. Multiplier on time axis of specified curve. Default=1.0.

        DSS property name: `DOC_TDPhaseInner`, DSS property index: 49.
        """
        return self._get_prop_string(49)

    @DOC_TDPhaseInner.setter
    def DOC_TDPhaseInner(self, value: Union[AnyStr, TCC_Curve]):
        if isinstance(value, DSSObj):
            self._set_obj(49, value)
            return

        self._set_string_o(49, value)

    @property
    def DOC_TDPhaseInner_obj(self) -> TCC_Curve:
        """
        Time dial for "DOC_PhaseCurveInner" TCC curve. Multiplier on time axis of specified curve. Default=1.0.

        DSS property name: `DOC_TDPhaseInner`, DSS property index: 49.
        """
        return self._get_obj(49, TCC_Curve)

    @DOC_TDPhaseInner_obj.setter
    def DOC_TDPhaseInner_obj(self, value: TCC_Curve):
        self._set_obj(49, value)

    @property
    def DOC_P1Blocking(self) -> bool:
        """
        {Yes/True* | No/False} Blocking element that impedes relay from tripping if balanced net three-phase active power is in the forward direction (i.e., flowing into the monitored terminal). For a delayed trip, if at any given time the reverse power flow condition stops, the tripping is reset. Default=True.

        DSS property name: `DOC_P1Blocking`, DSS property index: 50.
        """
        return self._lib.Obj_GetInt32(self._ptr, 50) != 0

    @DOC_P1Blocking.setter
    def DOC_P1Blocking(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 50, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 51.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 51)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 51, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 52.
        """
        return self._lib.Obj_GetInt32(self._ptr, 52) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 52, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 53.
        """
        self._set_string_o(53, value)

class Recloser(DSSObj):
    __slots__ = []
    _cls_name = 'Recloser'
    _cls_idx = 32
    _cls_prop_idx = {
        'monitoredobj': 1,
        'monitoredterm': 2,
        'switchedobj': 3,
        'switchedterm': 4,
        'numfast': 5,
        'phasefast': 6,
        'phasedelayed': 7,
        'groundfast': 8,
        'grounddelayed': 9,
        'phasetrip': 10,
        'groundtrip': 11,
        'phaseinst': 12,
        'groundinst': 13,
        'reset': 14,
        'shots': 15,
        'recloseintervals': 16,
        'delay': 17,
        'action': 18,
        'tdphfast': 19,
        'tdgrfast': 20,
        'tdphdelayed': 21,
        'tdgrdelayed': 22,
        'normal': 23,
        'state': 24,
        'basefreq': 25,
        'enabled': 26,
        'like': 27,
    }

    # Class-specific enumerations
    class RecloserAction(IntEnum):
        """Recloser: Action (DSS enumeration for Recloser)"""
        close = 2 # close
        open = 1 # open
        trip = 1 # trip

    class RecloserState(IntEnum):
        """Recloser: State (DSS enumeration for Recloser)"""
        closed = 2 # closed
        open = 1 # open
        trip = 1 # trip


    @property
    def MonitoredObj(self) -> str:
        """
        Full object name of the circuit element, typically a line, transformer, load, or generator, to which the Recloser's PT and/or CT are connected. This is the "monitored" element. There is no default; must be specified.

        DSS property name: `MonitoredObj`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @MonitoredObj.setter
    def MonitoredObj(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(1, value)
            return

        self._set_string_o(1, value)

    @property
    def MonitoredObj_obj(self) -> DSSObj:
        """
        Full object name of the circuit element, typically a line, transformer, load, or generator, to which the Recloser's PT and/or CT are connected. This is the "monitored" element. There is no default; must be specified.

        DSS property name: `MonitoredObj`, DSS property index: 1.
        """
        return self._get_obj(1, None)

    @MonitoredObj_obj.setter
    def MonitoredObj_obj(self, value: DSSObj):
        self._set_obj(1, value)

    @property
    def MonitoredTerm(self) -> int:
        """
        Number of the terminal of the circuit element to which the Recloser is connected. 1 or 2, typically.  Default is 1.

        DSS property name: `MonitoredTerm`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @MonitoredTerm.setter
    def MonitoredTerm(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def SwitchedObj(self) -> str:
        """
        Name of circuit element switch that the Recloser controls. Specify the full object name.Defaults to the same as the Monitored element. This is the "controlled" element.

        DSS property name: `SwitchedObj`, DSS property index: 3.
        """
        return self._get_prop_string(3)

    @SwitchedObj.setter
    def SwitchedObj(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(3, value)
            return

        self._set_string_o(3, value)

    @property
    def SwitchedObj_obj(self) -> DSSObj:
        """
        Name of circuit element switch that the Recloser controls. Specify the full object name.Defaults to the same as the Monitored element. This is the "controlled" element.

        DSS property name: `SwitchedObj`, DSS property index: 3.
        """
        return self._get_obj(3, None)

    @SwitchedObj_obj.setter
    def SwitchedObj_obj(self, value: DSSObj):
        self._set_obj(3, value)

    @property
    def SwitchedTerm(self) -> int:
        """
        Number of the terminal of the controlled element in which the switch is controlled by the Recloser. 1 or 2, typically.  Default is 1.

        DSS property name: `SwitchedTerm`, DSS property index: 4.
        """
        return self._lib.Obj_GetInt32(self._ptr, 4)

    @SwitchedTerm.setter
    def SwitchedTerm(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 4, value)

    @property
    def NumFast(self) -> int:
        """
        Number of Fast (fuse saving) operations.  Default is 1. (See "Shots")

        DSS property name: `NumFast`, DSS property index: 5.
        """
        return self._lib.Obj_GetInt32(self._ptr, 5)

    @NumFast.setter
    def NumFast(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 5, value)

    @property
    def PhaseFast(self) -> str:
        """
        Name of the TCC Curve object that determines the Phase Fast trip.  Must have been previously defined as a TCC_Curve object. Default is "A". Multiplying the current values in the curve by the "phasetrip" value gives the actual current.

        DSS property name: `PhaseFast`, DSS property index: 6.
        """
        return self._get_prop_string(6)

    @PhaseFast.setter
    def PhaseFast(self, value: Union[AnyStr, TCC_Curve]):
        if isinstance(value, DSSObj):
            self._set_obj(6, value)
            return

        self._set_string_o(6, value)

    @property
    def PhaseFast_obj(self) -> TCC_Curve:
        """
        Name of the TCC Curve object that determines the Phase Fast trip.  Must have been previously defined as a TCC_Curve object. Default is "A". Multiplying the current values in the curve by the "phasetrip" value gives the actual current.

        DSS property name: `PhaseFast`, DSS property index: 6.
        """
        return self._get_obj(6, TCC_Curve)

    @PhaseFast_obj.setter
    def PhaseFast_obj(self, value: TCC_Curve):
        self._set_obj(6, value)

    @property
    def PhaseDelayed(self) -> str:
        """
        Name of the TCC Curve object that determines the Phase Delayed trip.  Must have been previously defined as a TCC_Curve object. Default is "D".Multiplying the current values in the curve by the "phasetrip" value gives the actual current.

        DSS property name: `PhaseDelayed`, DSS property index: 7.
        """
        return self._get_prop_string(7)

    @PhaseDelayed.setter
    def PhaseDelayed(self, value: Union[AnyStr, TCC_Curve]):
        if isinstance(value, DSSObj):
            self._set_obj(7, value)
            return

        self._set_string_o(7, value)

    @property
    def PhaseDelayed_obj(self) -> TCC_Curve:
        """
        Name of the TCC Curve object that determines the Phase Delayed trip.  Must have been previously defined as a TCC_Curve object. Default is "D".Multiplying the current values in the curve by the "phasetrip" value gives the actual current.

        DSS property name: `PhaseDelayed`, DSS property index: 7.
        """
        return self._get_obj(7, TCC_Curve)

    @PhaseDelayed_obj.setter
    def PhaseDelayed_obj(self, value: TCC_Curve):
        self._set_obj(7, value)

    @property
    def GroundFast(self) -> str:
        """
        Name of the TCC Curve object that determines the Ground Fast trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored). Multiplying the current values in the curve by the "groundtrip" value gives the actual current.

        DSS property name: `GroundFast`, DSS property index: 8.
        """
        return self._get_prop_string(8)

    @GroundFast.setter
    def GroundFast(self, value: Union[AnyStr, TCC_Curve]):
        if isinstance(value, DSSObj):
            self._set_obj(8, value)
            return

        self._set_string_o(8, value)

    @property
    def GroundFast_obj(self) -> TCC_Curve:
        """
        Name of the TCC Curve object that determines the Ground Fast trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored). Multiplying the current values in the curve by the "groundtrip" value gives the actual current.

        DSS property name: `GroundFast`, DSS property index: 8.
        """
        return self._get_obj(8, TCC_Curve)

    @GroundFast_obj.setter
    def GroundFast_obj(self, value: TCC_Curve):
        self._set_obj(8, value)

    @property
    def GroundDelayed(self) -> str:
        """
        Name of the TCC Curve object that determines the Ground Delayed trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored).Multiplying the current values in the curve by the "groundtrip" value gives the actual current.

        DSS property name: `GroundDelayed`, DSS property index: 9.
        """
        return self._get_prop_string(9)

    @GroundDelayed.setter
    def GroundDelayed(self, value: Union[AnyStr, TCC_Curve]):
        if isinstance(value, DSSObj):
            self._set_obj(9, value)
            return

        self._set_string_o(9, value)

    @property
    def GroundDelayed_obj(self) -> TCC_Curve:
        """
        Name of the TCC Curve object that determines the Ground Delayed trip.  Must have been previously defined as a TCC_Curve object. Default is none (ignored).Multiplying the current values in the curve by the "groundtrip" value gives the actual current.

        DSS property name: `GroundDelayed`, DSS property index: 9.
        """
        return self._get_obj(9, TCC_Curve)

    @GroundDelayed_obj.setter
    def GroundDelayed_obj(self, value: TCC_Curve):
        self._set_obj(9, value)

    @property
    def PhaseTrip(self) -> float:
        """
        Multiplier or actual phase amps for the phase TCC curve.  Defaults to 1.0.

        DSS property name: `PhaseTrip`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @PhaseTrip.setter
    def PhaseTrip(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def GroundTrip(self) -> float:
        """
        Multiplier or actual ground amps (3I0) for the ground TCC curve.  Defaults to 1.0.

        DSS property name: `GroundTrip`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @GroundTrip.setter
    def GroundTrip(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def PhaseInst(self) -> float:
        """
        Actual amps for instantaneous phase trip which is assumed to happen in 0.01 sec + Delay Time. Default is 0.0, which signifies no inst trip. 

        DSS property name: `PhaseInst`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @PhaseInst.setter
    def PhaseInst(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def GroundInst(self) -> float:
        """
        Actual amps for instantaneous ground trip which is assumed to happen in 0.01 sec + Delay Time.Default is 0.0, which signifies no inst trip.

        DSS property name: `GroundInst`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @GroundInst.setter
    def GroundInst(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def Reset(self) -> float:
        """
        Reset time in sec for Recloser.  Default is 15. 

        DSS property name: `Reset`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @Reset.setter
    def Reset(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def Shots(self) -> int:
        """
        Total Number of fast and delayed shots to lockout.  Default is 4. This is one more than the number of reclose intervals.

        DSS property name: `Shots`, DSS property index: 15.
        """
        return self._lib.Obj_GetInt32(self._ptr, 15)

    @Shots.setter
    def Shots(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 15, value)

    @property
    def RecloseIntervals(self) -> Float64Array:
        """
        Array of reclose intervals.  Default for Recloser is (0.5, 2.0, 2.0) seconds. A locked out Recloser must be closed manually (action=close).

        DSS property name: `RecloseIntervals`, DSS property index: 16.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 16)

    @RecloseIntervals.setter
    def RecloseIntervals(self, value: Float64Array):
        self._set_float64_array_o(16, value)

    @property
    def Delay(self) -> float:
        """
        Fixed delay time (sec) added to Recloser trip time. Default is 0.0. Used to represent breaker time or any other delay.

        DSS property name: `Delay`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @Delay.setter
    def Delay(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def TDPhFast(self) -> float:
        """
        Time dial for Phase Fast trip curve. Multiplier on time axis of specified curve. Default=1.0.

        DSS property name: `TDPhFast`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @TDPhFast.setter
    def TDPhFast(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def TDGrFast(self) -> float:
        """
        Time dial for Ground Fast trip curve. Multiplier on time axis of specified curve. Default=1.0.

        DSS property name: `TDGrFast`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @TDGrFast.setter
    def TDGrFast(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    @property
    def TDPhDelayed(self) -> float:
        """
        Time dial for Phase Delayed trip curve. Multiplier on time axis of specified curve. Default=1.0.

        DSS property name: `TDPhDelayed`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @TDPhDelayed.setter
    def TDPhDelayed(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def TDGrDelayed(self) -> float:
        """
        Time dial for Ground Delayed trip curve. Multiplier on time axis of specified curve. Default=1.0.

        DSS property name: `TDGrDelayed`, DSS property index: 22.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 22)

    @TDGrDelayed.setter
    def TDGrDelayed(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 22, value)

    @property
    def Normal(self) -> RecloserState:
        """
        {Open | Closed} Normal state of the recloser. The recloser reverts to this state for reset, change of mode, etc. Defaults to "State" if not specificallt declared.

        DSS property name: `Normal`, DSS property index: 23.
        """
        return Recloser.RecloserState(self._lib.Obj_GetInt32(self._ptr, 23))

    @Normal.setter
    def Normal(self, value: Union[AnyStr, int, RecloserState]):
        if not isinstance(value, int):
            self._set_string_o(23, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 23, value)

    @property
    def Normal_str(self) -> str:
        """
        {Open | Closed} Normal state of the recloser. The recloser reverts to this state for reset, change of mode, etc. Defaults to "State" if not specificallt declared.

        DSS property name: `Normal`, DSS property index: 23.
        """
        return self._get_prop_string(23)

    @Normal_str.setter
    def Normal_str(self, value: AnyStr):
        self.Normal = value

    @property
    def State(self) -> RecloserState:
        """
        {Open | Closed} Actual state of the recloser. Upon setting, immediately forces state of the recloser, overriding the Recloser control. Simulates manual control on recloser. Defaults to Closed. "Open" causes the controlled element to open and lock out. "Closed" causes the controlled element to close and the recloser to reset to its first operation.

        DSS property name: `State`, DSS property index: 24.
        """
        return Recloser.RecloserState(self._lib.Obj_GetInt32(self._ptr, 24))

    @State.setter
    def State(self, value: Union[AnyStr, int, RecloserState]):
        if not isinstance(value, int):
            self._set_string_o(24, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 24, value)

    @property
    def State_str(self) -> str:
        """
        {Open | Closed} Actual state of the recloser. Upon setting, immediately forces state of the recloser, overriding the Recloser control. Simulates manual control on recloser. Defaults to Closed. "Open" causes the controlled element to open and lock out. "Closed" causes the controlled element to close and the recloser to reset to its first operation.

        DSS property name: `State`, DSS property index: 24.
        """
        return self._get_prop_string(24)

    @State_str.setter
    def State_str(self, value: AnyStr):
        self.State = value

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 25.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 25)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 25, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 26.
        """
        return self._lib.Obj_GetInt32(self._ptr, 26) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 26, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 27.
        """
        self._set_string_o(27, value)

class Fuse(DSSObj):
    __slots__ = []
    _cls_name = 'Fuse'
    _cls_idx = 33
    _cls_prop_idx = {
        'monitoredobj': 1,
        'monitoredterm': 2,
        'switchedobj': 3,
        'switchedterm': 4,
        'fusecurve': 5,
        'ratedcurrent': 6,
        'delay': 7,
        'action': 8,
        'normal': 9,
        'state': 10,
        'basefreq': 11,
        'enabled': 12,
        'like': 13,
    }

    # Class-specific enumerations
    class FuseAction(IntEnum):
        """Fuse: Action (DSS enumeration for Fuse)"""
        close = 2 # close
        open = 1 # open

    class FuseState(IntEnum):
        """Fuse: State (DSS enumeration for Fuse)"""
        closed = 2 # closed
        open = 1 # open


    @property
    def MonitoredObj(self) -> str:
        """
        Full object name of the circuit element, typically a line, transformer, load, or generator, to which the Fuse is connected. This is the "monitored" element. There is no default; must be specified.

        DSS property name: `MonitoredObj`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @MonitoredObj.setter
    def MonitoredObj(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(1, value)
            return

        self._set_string_o(1, value)

    @property
    def MonitoredObj_obj(self) -> DSSObj:
        """
        Full object name of the circuit element, typically a line, transformer, load, or generator, to which the Fuse is connected. This is the "monitored" element. There is no default; must be specified.

        DSS property name: `MonitoredObj`, DSS property index: 1.
        """
        return self._get_obj(1, None)

    @MonitoredObj_obj.setter
    def MonitoredObj_obj(self, value: DSSObj):
        self._set_obj(1, value)

    @property
    def MonitoredTerm(self) -> int:
        """
        Number of the terminal of the circuit element to which the Fuse is connected. 1 or 2, typically.  Default is 1.

        DSS property name: `MonitoredTerm`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @MonitoredTerm.setter
    def MonitoredTerm(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def SwitchedObj(self) -> str:
        """
        Name of circuit element switch that the Fuse controls. Specify the full object name.Defaults to the same as the Monitored element. This is the "controlled" element.

        DSS property name: `SwitchedObj`, DSS property index: 3.
        """
        return self._get_prop_string(3)

    @SwitchedObj.setter
    def SwitchedObj(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(3, value)
            return

        self._set_string_o(3, value)

    @property
    def SwitchedObj_obj(self) -> DSSObj:
        """
        Name of circuit element switch that the Fuse controls. Specify the full object name.Defaults to the same as the Monitored element. This is the "controlled" element.

        DSS property name: `SwitchedObj`, DSS property index: 3.
        """
        return self._get_obj(3, None)

    @SwitchedObj_obj.setter
    def SwitchedObj_obj(self, value: DSSObj):
        self._set_obj(3, value)

    @property
    def SwitchedTerm(self) -> int:
        """
        Number of the terminal of the controlled element in which the switch is controlled by the Fuse. 1 or 2, typically.  Default is 1.  Assumes all phases of the element have a fuse of this type.

        DSS property name: `SwitchedTerm`, DSS property index: 4.
        """
        return self._lib.Obj_GetInt32(self._ptr, 4)

    @SwitchedTerm.setter
    def SwitchedTerm(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 4, value)

    @property
    def FuseCurve(self) -> str:
        """
        Name of the TCC Curve object that determines the fuse blowing.  Must have been previously defined as a TCC_Curve object. Default is "Tlink". Multiplying the current values in the curve by the "RatedCurrent" value gives the actual current.

        DSS property name: `FuseCurve`, DSS property index: 5.
        """
        return self._get_prop_string(5)

    @FuseCurve.setter
    def FuseCurve(self, value: Union[AnyStr, TCC_Curve]):
        if isinstance(value, DSSObj):
            self._set_obj(5, value)
            return

        self._set_string_o(5, value)

    @property
    def FuseCurve_obj(self) -> TCC_Curve:
        """
        Name of the TCC Curve object that determines the fuse blowing.  Must have been previously defined as a TCC_Curve object. Default is "Tlink". Multiplying the current values in the curve by the "RatedCurrent" value gives the actual current.

        DSS property name: `FuseCurve`, DSS property index: 5.
        """
        return self._get_obj(5, TCC_Curve)

    @FuseCurve_obj.setter
    def FuseCurve_obj(self, value: TCC_Curve):
        self._set_obj(5, value)

    @property
    def RatedCurrent(self) -> float:
        """
        Multiplier or actual phase amps for the phase TCC curve.  Defaults to 1.0.

        DSS property name: `RatedCurrent`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @RatedCurrent.setter
    def RatedCurrent(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def Delay(self) -> float:
        """
        Fixed delay time (sec) added to Fuse blowing time determined from the TCC curve. Default is 0.0. Used to represent fuse clearing time or any other delay.

        DSS property name: `Delay`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @Delay.setter
    def Delay(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    def Action(self, value: Union[str, bytes, int, FuseAction]):
        """
        DEPRECATED. See "State" property.

        DSS property name: `Action`, DSS property index: 8.
        """
        if isinstance(value, int):
            self._lib.Obj_SetInt32(self._ptr, 8, value)
            return
    
        self._set_string_o(8, value)

    @property
    def Normal(self) -> List[FuseState]:
        """
        ARRAY of strings {Open | Closed} representing the Normal state of the fuse in each phase of the controlled element. The fuse reverts to this state for reset, change of mode, etc. Defaults to "State" if not specifically declared.

        DSS property name: `Normal`, DSS property index: 9.
        """
        return [Fuse.FuseState(val) for val in self._get_int32_list(self._lib.Obj_GetInt32Array, self._ptr, 9)]

    @Normal.setter
    def Normal(self, value: Union[List[Union[int,FuseState]], List[AnyStr]]):
        if len(value) and not isinstance(value[0], int):
            self._set_string_array_o(9, value)
            return    
        self._set_int32_array_o(9, value)

    @property
    def Normal_str(self) -> List[str]:
        """
        ARRAY of strings {Open | Closed} representing the Normal state of the fuse in each phase of the controlled element. The fuse reverts to this state for reset, change of mode, etc. Defaults to "State" if not specifically declared.

        DSS property name: `Normal`, DSS property index: 9.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 9)

    @Normal_str.setter
    def Normal_str(self, value: AnyStr):
        self.Normal = value

    @property
    def State(self) -> List[FuseState]:
        """
        ARRAY of strings {Open | Closed} representing the Actual state of the fuse in each phase of the controlled element. Upon setting, immediately forces state of fuse(s). Simulates manual control on Fuse. Defaults to Closed for all phases.

        DSS property name: `State`, DSS property index: 10.
        """
        return [Fuse.FuseState(val) for val in self._get_int32_list(self._lib.Obj_GetInt32Array, self._ptr, 10)]

    @State.setter
    def State(self, value: Union[List[Union[int,FuseState]], List[AnyStr]]):
        if len(value) and not isinstance(value[0], int):
            self._set_string_array_o(10, value)
            return    
        self._set_int32_array_o(10, value)

    @property
    def State_str(self) -> List[str]:
        """
        ARRAY of strings {Open | Closed} representing the Actual state of the fuse in each phase of the controlled element. Upon setting, immediately forces state of fuse(s). Simulates manual control on Fuse. Defaults to Closed for all phases.

        DSS property name: `State`, DSS property index: 10.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 10)

    @State_str.setter
    def State_str(self, value: AnyStr):
        self.State = value

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 12.
        """
        return self._lib.Obj_GetInt32(self._ptr, 12) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 12, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 13.
        """
        self._set_string_o(13, value)

class SwtControl(DSSObj):
    __slots__ = []
    _cls_name = 'SwtControl'
    _cls_idx = 34
    _cls_prop_idx = {
        'switchedobj': 1,
        'switchedterm': 2,
        'action': 3,
        'lock': 4,
        'delay': 5,
        'normal': 6,
        'state': 7,
        'reset': 8,
        'basefreq': 9,
        'enabled': 10,
        'like': 11,
    }

    # Class-specific enumerations
    class SwtControlAction(IntEnum):
        """SwtControl: Action (DSS enumeration for SwtControl)"""
        close = 2 # close
        open = 1 # open

    class SwtControlState(IntEnum):
        """SwtControl: State (DSS enumeration for SwtControl)"""
        closed = 2 # closed
        open = 1 # open


    @property
    def SwitchedObj(self) -> str:
        """
        Name of circuit element switch that the SwtControl operates. Specify the full object class and name.

        DSS property name: `SwitchedObj`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @SwitchedObj.setter
    def SwitchedObj(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(1, value)
            return

        self._set_string_o(1, value)

    @property
    def SwitchedObj_obj(self) -> DSSObj:
        """
        Name of circuit element switch that the SwtControl operates. Specify the full object class and name.

        DSS property name: `SwitchedObj`, DSS property index: 1.
        """
        return self._get_obj(1, None)

    @SwitchedObj_obj.setter
    def SwitchedObj_obj(self, value: DSSObj):
        self._set_obj(1, value)

    @property
    def SwitchedTerm(self) -> int:
        """
        Terminal number of the controlled element switch. 1 or 2, typically.  Default is 1.

        DSS property name: `SwitchedTerm`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @SwitchedTerm.setter
    def SwitchedTerm(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def Lock(self) -> bool:
        """
        {Yes | No} Delayed action. Sends CTRL_LOCK or CTRL_UNLOCK message to control queue. After delay time, controlled switch is locked in its present open / close state or unlocked. Switch will not respond to either manual (Action) or automatic (COM interface) control or internal OpenDSS Reset when locked.

        DSS property name: `Lock`, DSS property index: 4.
        """
        return self._lib.Obj_GetInt32(self._ptr, 4) != 0

    @Lock.setter
    def Lock(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 4, value)

    @property
    def Delay(self) -> float:
        """
        Operating time delay (sec) of the switch. Defaults to 120.

        DSS property name: `Delay`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @Delay.setter
    def Delay(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def Normal(self) -> SwtControlState:
        """
        {Open | Closed] Normal state of the switch. If not Locked, the switch reverts to this state for reset, change of mode, etc. Defaults to first Action or State specified if not specifically declared.

        DSS property name: `Normal`, DSS property index: 6.
        """
        return SwtControl.SwtControlState(self._lib.Obj_GetInt32(self._ptr, 6))

    @Normal.setter
    def Normal(self, value: Union[AnyStr, int, SwtControlState]):
        if not isinstance(value, int):
            self._set_string_o(6, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def Normal_str(self) -> str:
        """
        {Open | Closed] Normal state of the switch. If not Locked, the switch reverts to this state for reset, change of mode, etc. Defaults to first Action or State specified if not specifically declared.

        DSS property name: `Normal`, DSS property index: 6.
        """
        return self._get_prop_string(6)

    @Normal_str.setter
    def Normal_str(self, value: AnyStr):
        self.Normal = value

    @property
    def State(self) -> SwtControlState:
        """
        {Open | Closed] Present state of the switch. Upon setting, immediately forces state of switch.

        DSS property name: `State`, DSS property index: 7.
        """
        return SwtControl.SwtControlState(self._lib.Obj_GetInt32(self._ptr, 7))

    @State.setter
    def State(self, value: Union[AnyStr, int, SwtControlState]):
        if not isinstance(value, int):
            self._set_string_o(7, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 7, value)

    @property
    def State_str(self) -> str:
        """
        {Open | Closed] Present state of the switch. Upon setting, immediately forces state of switch.

        DSS property name: `State`, DSS property index: 7.
        """
        return self._get_prop_string(7)

    @State_str.setter
    def State_str(self, value: AnyStr):
        self.State = value

    def Reset(self, value: bool):
        """
        {Yes | No} If Yes, forces Reset of switch to Normal state and removes Lock independently of any internal reset command for mode change, etc.

        DSS property name: `Reset`, DSS property index: 8.
        """
        self._lib.Obj_SetInt32(self._ptr, 8, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 10.
        """
        return self._lib.Obj_GetInt32(self._ptr, 10) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 10, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 11.
        """
        self._set_string_o(11, value)

class PVSystem(DSSObj):
    __slots__ = []
    _cls_name = 'PVSystem'
    _cls_idx = 35
    _cls_prop_idx = {
        'phases': 1,
        'bus1': 2,
        'kv': 3,
        'irradiance': 4,
        'pmpp': 5,
        'pctpmpp': 6,
        '%pmpp': 6,
        'temperature': 7,
        'pf': 8,
        'conn': 9,
        'kvar': 10,
        'kva': 11,
        'pctcutin': 12,
        '%cutin': 12,
        'pctcutout': 13,
        '%cutout': 13,
        'effcurve': 14,
        'ptcurve': 15,
        'p-tcurve': 15,
        'pctr': 16,
        '%r': 16,
        'pctx': 17,
        '%x': 17,
        'model': 18,
        'vminpu': 19,
        'vmaxpu': 20,
        'balanced': 21,
        'limitcurrent': 22,
        'yearly': 23,
        'daily': 24,
        'duty': 25,
        'tyearly': 26,
        'tdaily': 27,
        'tduty': 28,
        'cls': 29,
        'class': 29,
        'usermodel': 30,
        'userdata': 31,
        'debugtrace': 32,
        'varfollowinverter': 33,
        'dutystart': 34,
        'wattpriority': 35,
        'pfpriority': 36,
        'pctpminnovars': 37,
        '%pminnovars': 37,
        'pctpminkvarmax': 38,
        '%pminkvarmax': 38,
        'kvarmax': 39,
        'kvarmaxabs': 40,
        'kvdc': 41,
        'kp': 42,
        'pitol': 43,
        'safevoltage': 44,
        'safemode': 45,
        'dynamiceq': 46,
        'dynout': 47,
        'controlmode': 48,
        'spectrum': 49,
        'basefreq': 50,
        'enabled': 51,
        'like': 52,
    }

    @property
    def phases(self) -> int:
        """
        Number of Phases, this PVSystem element.  Power is evenly divided among phases.

        DSS property name: `phases`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def bus1(self) -> str:
        """
        Bus to which the PVSystem element is connected.  May include specific node specification.

        DSS property name: `bus1`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(2, value)

    @property
    def kv(self) -> float:
        """
        Nominal rated (1.0 per unit) voltage, kV, for PVSystem element. For 2- and 3-phase PVSystem elements, specify phase-phase kV. Otherwise, specify actual kV across each branch of the PVSystem element. If 1-phase wye (star or LN), specify phase-neutral kV. If 1-phase delta or phase-phase connected, specify phase-phase kV.

        DSS property name: `kv`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @kv.setter
    def kv(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def irradiance(self) -> float:
        """
        Get/set the present irradiance value in kW/sq-m. Used as base value for shape multipliers. Generally entered as peak value for the time period of interest and the yearly, daily, and duty load shape objects are defined as per unit multipliers (just like Loads/Generators).

        DSS property name: `irradiance`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @irradiance.setter
    def irradiance(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def Pmpp(self) -> float:
        """
        Get/set the rated max power of the PV array for 1.0 kW/sq-m irradiance and a user-selected array temperature. The P-TCurve should be defined relative to the selected array temperature.

        DSS property name: `Pmpp`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @Pmpp.setter
    def Pmpp(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def pctPmpp(self) -> float:
        """
        Upper limit on active power as a percentage of Pmpp.

        DSS property name: `%Pmpp`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @pctPmpp.setter
    def pctPmpp(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def Temperature(self) -> float:
        """
        Get/set the present Temperature. Used as fixed value corresponding to PTCurve property. A multiplier is obtained from the Pmpp-Temp curve and applied to the nominal Pmpp from the irradiance to determine the net array output.

        DSS property name: `Temperature`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @Temperature.setter
    def Temperature(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def pf(self) -> float:
        """
        Nominally, the power factor for the output power. Default is 1.0. Setting this property will cause the inverter to operate in constant power factor mode.Enter negative when kW and kvar have opposite signs.
        A positive power factor signifies that the PVSystem element produces vars 
        as is typical for a generator.  

        DSS property name: `pf`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @pf.setter
    def pf(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def conn(self) -> Connection:
        """
        ={wye|LN|delta|LL}.  Default is wye.

        DSS property name: `conn`, DSS property index: 9.
        """
        return Connection(self._lib.Obj_GetInt32(self._ptr, 9))

    @conn.setter
    def conn(self, value: Union[AnyStr, int, Connection]):
        if not isinstance(value, int):
            self._set_string_o(9, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 9, value)

    @property
    def conn_str(self) -> str:
        """
        ={wye|LN|delta|LL}.  Default is wye.

        DSS property name: `conn`, DSS property index: 9.
        """
        return self._get_prop_string(9)

    @conn_str.setter
    def conn_str(self, value: AnyStr):
        self.conn = value

    @property
    def kvar(self) -> float:
        """
        Get/set the present kvar value.  Setting this property forces the inverter to operate in constant kvar mode.

        DSS property name: `kvar`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @kvar.setter
    def kvar(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def kVA(self) -> float:
        """
        kVA rating of inverter. Used as the base for Dynamics mode and Harmonics mode values.

        DSS property name: `kVA`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @kVA.setter
    def kVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def pctCutin(self) -> float:
        """
        % cut-in power -- % of kVA rating of inverter. When the inverter is OFF, the power from the array must be greater than this for the inverter to turn on.

        DSS property name: `%Cutin`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @pctCutin.setter
    def pctCutin(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def pctCutout(self) -> float:
        """
        % cut-out power -- % of kVA rating of inverter. When the inverter is ON, the inverter turns OFF when the power from the array drops below this value.

        DSS property name: `%Cutout`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @pctCutout.setter
    def pctCutout(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def EffCurve(self) -> str:
        """
        An XYCurve object, previously defined, that describes the PER UNIT efficiency vs PER UNIT of rated kVA for the inverter. Inverter output power is discounted by the multiplier obtained from this curve.

        DSS property name: `EffCurve`, DSS property index: 14.
        """
        return self._get_prop_string(14)

    @EffCurve.setter
    def EffCurve(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(14, value)
            return

        self._set_string_o(14, value)

    @property
    def EffCurve_obj(self) -> XYcurve:
        """
        An XYCurve object, previously defined, that describes the PER UNIT efficiency vs PER UNIT of rated kVA for the inverter. Inverter output power is discounted by the multiplier obtained from this curve.

        DSS property name: `EffCurve`, DSS property index: 14.
        """
        return self._get_obj(14, XYcurve)

    @EffCurve_obj.setter
    def EffCurve_obj(self, value: XYcurve):
        self._set_obj(14, value)

    @property
    def PTCurve(self) -> str:
        """
        An XYCurve object, previously defined, that describes the PV array PER UNIT Pmpp vs Temperature curve. Temperature units must agree with the Temperature property and the Temperature shapes used for simulations. The Pmpp values are specified in per unit of the Pmpp value for 1 kW/sq-m irradiance. The value for the temperature at which Pmpp is defined should be 1.0. The net array power is determined by the irradiance * Pmpp * f(Temperature)

        DSS property name: `P-TCurve`, DSS property index: 15.
        """
        return self._get_prop_string(15)

    @PTCurve.setter
    def PTCurve(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(15, value)
            return

        self._set_string_o(15, value)

    @property
    def PTCurve_obj(self) -> XYcurve:
        """
        An XYCurve object, previously defined, that describes the PV array PER UNIT Pmpp vs Temperature curve. Temperature units must agree with the Temperature property and the Temperature shapes used for simulations. The Pmpp values are specified in per unit of the Pmpp value for 1 kW/sq-m irradiance. The value for the temperature at which Pmpp is defined should be 1.0. The net array power is determined by the irradiance * Pmpp * f(Temperature)

        DSS property name: `P-TCurve`, DSS property index: 15.
        """
        return self._get_obj(15, XYcurve)

    @PTCurve_obj.setter
    def PTCurve_obj(self, value: XYcurve):
        self._set_obj(15, value)

    @property
    def pctR(self) -> float:
        """
        Equivalent percent internal resistance, ohms. Default is 50%. Placed in series with internal voltage source for harmonics and dynamics modes. (Limits fault current to about 2 pu if not current limited -- see LimitCurrent) 

        DSS property name: `%R`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @pctR.setter
    def pctR(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def pctX(self) -> float:
        """
        Equivalent percent internal reactance, ohms. Default is 0%. Placed in series with internal voltage source for harmonics and dynamics modes. 

        DSS property name: `%X`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @pctX.setter
    def pctX(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def model(self) -> int:
        """
        Integer code (default=1) for the model to use for power output variation with voltage. Valid values are:

        1:PVSystem element injects a CONSTANT kW at specified power factor.
        2:PVSystem element is modeled as a CONSTANT ADMITTANCE.
        3:Compute load injection from User-written Model.

        DSS property name: `model`, DSS property index: 18.
        """
        return self._lib.Obj_GetInt32(self._ptr, 18)

    @model.setter
    def model(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 18, value)

    @property
    def Vminpu(self) -> float:
        """
        Default = 0.90.  Minimum per unit voltage for which the Model is assumed to apply. Below this value, the load model reverts to a constant impedance model except for Dynamics model. In Dynamics mode, the current magnitude is limited to the value the power flow would compute for this voltage.

        DSS property name: `Vminpu`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @Vminpu.setter
    def Vminpu(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def Vmaxpu(self) -> float:
        """
        Default = 1.10.  Maximum per unit voltage for which the Model is assumed to apply. Above this value, the load model reverts to a constant impedance model.

        DSS property name: `Vmaxpu`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @Vmaxpu.setter
    def Vmaxpu(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    @property
    def Balanced(self) -> bool:
        """
        {Yes | No*} Default is No.  Force balanced current only for 3-phase PVSystems. Forces zero- and negative-sequence to zero. 

        DSS property name: `Balanced`, DSS property index: 21.
        """
        return self._lib.Obj_GetInt32(self._ptr, 21) != 0

    @Balanced.setter
    def Balanced(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 21, value)

    @property
    def LimitCurrent(self) -> bool:
        """
        Limits current magnitude to Vminpu value for both 1-phase and 3-phase PVSystems similar to Generator Model 7. For 3-phase, limits the positive-sequence current but not the negative-sequence.

        DSS property name: `LimitCurrent`, DSS property index: 22.
        """
        return self._lib.Obj_GetInt32(self._ptr, 22) != 0

    @LimitCurrent.setter
    def LimitCurrent(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 22, value)

    @property
    def yearly(self) -> str:
        """
        Dispatch shape to use for yearly simulations.  Must be previously defined as a Loadshape object. If this is not specified, the Daily dispatch shape, if any, is repeated during Yearly solution modes. In the default dispatch mode, the PVSystem element uses this loadshape to trigger State changes.

        DSS property name: `yearly`, DSS property index: 23.
        """
        return self._get_prop_string(23)

    @yearly.setter
    def yearly(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(23, value)
            return

        self._set_string_o(23, value)

    @property
    def yearly_obj(self) -> LoadShape:
        """
        Dispatch shape to use for yearly simulations.  Must be previously defined as a Loadshape object. If this is not specified, the Daily dispatch shape, if any, is repeated during Yearly solution modes. In the default dispatch mode, the PVSystem element uses this loadshape to trigger State changes.

        DSS property name: `yearly`, DSS property index: 23.
        """
        return self._get_obj(23, LoadShape)

    @yearly_obj.setter
    def yearly_obj(self, value: LoadShape):
        self._set_obj(23, value)

    @property
    def daily(self) -> str:
        """
        Dispatch shape to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically.  In the default dispatch mode, the PVSystem element uses this loadshape to trigger State changes.

        DSS property name: `daily`, DSS property index: 24.
        """
        return self._get_prop_string(24)

    @daily.setter
    def daily(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(24, value)
            return

        self._set_string_o(24, value)

    @property
    def daily_obj(self) -> LoadShape:
        """
        Dispatch shape to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically.  In the default dispatch mode, the PVSystem element uses this loadshape to trigger State changes.

        DSS property name: `daily`, DSS property index: 24.
        """
        return self._get_obj(24, LoadShape)

    @daily_obj.setter
    def daily_obj(self, value: LoadShape):
        self._set_obj(24, value)

    @property
    def duty(self) -> str:
        """
        Load shape to use for duty cycle dispatch simulations such as for solar ramp rate studies. Must be previously defined as a Loadshape object. Typically would have time intervals of 1-5 seconds. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.

        DSS property name: `duty`, DSS property index: 25.
        """
        return self._get_prop_string(25)

    @duty.setter
    def duty(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(25, value)
            return

        self._set_string_o(25, value)

    @property
    def duty_obj(self) -> LoadShape:
        """
        Load shape to use for duty cycle dispatch simulations such as for solar ramp rate studies. Must be previously defined as a Loadshape object. Typically would have time intervals of 1-5 seconds. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.

        DSS property name: `duty`, DSS property index: 25.
        """
        return self._get_obj(25, LoadShape)

    @duty_obj.setter
    def duty_obj(self, value: LoadShape):
        self._set_obj(25, value)

    @property
    def Tyearly(self) -> str:
        """
        Temperature shape to use for yearly simulations.  Must be previously defined as a TShape object. If this is not specified, the Daily dispatch shape, if any, is repeated during Yearly solution modes. The PVSystem element uses this TShape to determine the Pmpp from the Pmpp vs T curve. Units must agree with the Pmpp vs T curve.

        DSS property name: `Tyearly`, DSS property index: 26.
        """
        return self._get_prop_string(26)

    @Tyearly.setter
    def Tyearly(self, value: Union[AnyStr, TShape]):
        if isinstance(value, DSSObj):
            self._set_obj(26, value)
            return

        self._set_string_o(26, value)

    @property
    def Tyearly_obj(self) -> TShape:
        """
        Temperature shape to use for yearly simulations.  Must be previously defined as a TShape object. If this is not specified, the Daily dispatch shape, if any, is repeated during Yearly solution modes. The PVSystem element uses this TShape to determine the Pmpp from the Pmpp vs T curve. Units must agree with the Pmpp vs T curve.

        DSS property name: `Tyearly`, DSS property index: 26.
        """
        return self._get_obj(26, TShape)

    @Tyearly_obj.setter
    def Tyearly_obj(self, value: TShape):
        self._set_obj(26, value)

    @property
    def Tdaily(self) -> str:
        """
        Temperature shape to use for daily simulations.  Must be previously defined as a TShape object of 24 hrs, typically.  The PVSystem element uses this TShape to determine the Pmpp from the Pmpp vs T curve. Units must agree with the Pmpp vs T curve.

        DSS property name: `Tdaily`, DSS property index: 27.
        """
        return self._get_prop_string(27)

    @Tdaily.setter
    def Tdaily(self, value: Union[AnyStr, TShape]):
        if isinstance(value, DSSObj):
            self._set_obj(27, value)
            return

        self._set_string_o(27, value)

    @property
    def Tdaily_obj(self) -> TShape:
        """
        Temperature shape to use for daily simulations.  Must be previously defined as a TShape object of 24 hrs, typically.  The PVSystem element uses this TShape to determine the Pmpp from the Pmpp vs T curve. Units must agree with the Pmpp vs T curve.

        DSS property name: `Tdaily`, DSS property index: 27.
        """
        return self._get_obj(27, TShape)

    @Tdaily_obj.setter
    def Tdaily_obj(self, value: TShape):
        self._set_obj(27, value)

    @property
    def Tduty(self) -> str:
        """
        Temperature shape to use for duty cycle dispatch simulations such as for solar ramp rate studies. Must be previously defined as a TShape object. Typically would have time intervals of 1-5 seconds. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat. The PVSystem model uses this TShape to determine the Pmpp from the Pmpp vs T curve. Units must agree with the Pmpp vs T curve.

        DSS property name: `Tduty`, DSS property index: 28.
        """
        return self._get_prop_string(28)

    @Tduty.setter
    def Tduty(self, value: Union[AnyStr, TShape]):
        if isinstance(value, DSSObj):
            self._set_obj(28, value)
            return

        self._set_string_o(28, value)

    @property
    def Tduty_obj(self) -> TShape:
        """
        Temperature shape to use for duty cycle dispatch simulations such as for solar ramp rate studies. Must be previously defined as a TShape object. Typically would have time intervals of 1-5 seconds. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat. The PVSystem model uses this TShape to determine the Pmpp from the Pmpp vs T curve. Units must agree with the Pmpp vs T curve.

        DSS property name: `Tduty`, DSS property index: 28.
        """
        return self._get_obj(28, TShape)

    @Tduty_obj.setter
    def Tduty_obj(self, value: TShape):
        self._set_obj(28, value)

    @property
    def cls(self) -> int:
        """
        An arbitrary integer number representing the class of PVSystem element so that PVSystem values may be segregated by class.

        DSS property name: `class`, DSS property index: 29.
        """
        return self._lib.Obj_GetInt32(self._ptr, 29)

    @cls.setter
    def cls(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 29, value)

    @property
    def UserModel(self) -> str:
        """
        Name of DLL containing user-written model, which computes the terminal currents for Dynamics studies, overriding the default model.  Set to "none" to negate previous setting.

        DSS property name: `UserModel`, DSS property index: 30.
        """
        return self._get_prop_string(30)

    @UserModel.setter
    def UserModel(self, value: AnyStr):
        self._set_string_o(30, value)

    @property
    def UserData(self) -> str:
        """
        String (in quotes or parentheses) that gets passed to user-written model for defining the data required for that model.

        DSS property name: `UserData`, DSS property index: 31.
        """
        return self._get_prop_string(31)

    @UserData.setter
    def UserData(self, value: AnyStr):
        self._set_string_o(31, value)

    @property
    def debugtrace(self) -> bool:
        """
        {Yes | No }  Default is no.  Turn this on to capture the progress of the PVSystem model for each iteration.  Creates a separate file for each PVSystem element named "PVSystem_name.csv".

        DSS property name: `debugtrace`, DSS property index: 32.
        """
        return self._lib.Obj_GetInt32(self._ptr, 32) != 0

    @debugtrace.setter
    def debugtrace(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 32, value)

    @property
    def VarFollowInverter(self) -> bool:
        """
        Boolean variable (Yes|No) or (True|False). Defaults to False which indicates that the reactive power generation/absorption does not respect the inverter status.When set to True, the PVSystem reactive power generation/absorption will cease when the inverter status is off, due to panel kW dropping below %Cutout.  The reactive power generation/absorption will begin again when the panel kW is above %Cutin.  When set to False, the PVSystem will generate/absorb reactive power regardless of the status of the inverter.

        DSS property name: `VarFollowInverter`, DSS property index: 33.
        """
        return self._lib.Obj_GetInt32(self._ptr, 33) != 0

    @VarFollowInverter.setter
    def VarFollowInverter(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 33, value)

    @property
    def DutyStart(self) -> float:
        """
        Starting time offset [hours] into the duty cycle shape for this PVSystem, defaults to 0

        DSS property name: `DutyStart`, DSS property index: 34.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 34)

    @DutyStart.setter
    def DutyStart(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 34, value)

    @property
    def WattPriority(self) -> bool:
        """
        {Yes/No*/True/False} Set inverter to watt priority instead of the default var priority

        DSS property name: `WattPriority`, DSS property index: 35.
        """
        return self._lib.Obj_GetInt32(self._ptr, 35) != 0

    @WattPriority.setter
    def WattPriority(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 35, value)

    @property
    def PFPriority(self) -> bool:
        """
        {Yes/No*/True/False} Set inverter to operate with PF priority when in constant PF mode. If "Yes", value assigned to "WattPriority" is neglected. If controlled by an InvControl with either Volt-Var or DRC or both functions activated, PF priority is neglected and "WattPriority" is considered. Default = No.

        DSS property name: `PFPriority`, DSS property index: 36.
        """
        return self._lib.Obj_GetInt32(self._ptr, 36) != 0

    @PFPriority.setter
    def PFPriority(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 36, value)

    @property
    def pctPminNoVars(self) -> float:
        """
        Minimum active power as percentage of Pmpp under which there is no vars production/absorption.

        DSS property name: `%PminNoVars`, DSS property index: 37.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 37)

    @pctPminNoVars.setter
    def pctPminNoVars(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 37, value)

    @property
    def pctPminkvarMax(self) -> float:
        """
        Minimum active power as percentage of Pmpp that allows the inverter to produce/absorb reactive power up to its kvarMax or kvarMaxAbs.

        DSS property name: `%PminkvarMax`, DSS property index: 38.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 38)

    @pctPminkvarMax.setter
    def pctPminkvarMax(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 38, value)

    @property
    def kvarMax(self) -> float:
        """
        Indicates the maximum reactive power GENERATION (un-signed numerical variable in kvar) for the inverter (as an un-signed value). Defaults to kVA rating of the inverter.

        DSS property name: `kvarMax`, DSS property index: 39.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 39)

    @kvarMax.setter
    def kvarMax(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 39, value)

    @property
    def kvarMaxAbs(self) -> float:
        """
        Indicates the maximum reactive power ABSORPTION (un-signed numerical variable in kvar) for the inverter (as an un-signed value). Defaults to kVA rating of the inverter.

        DSS property name: `kvarMaxAbs`, DSS property index: 40.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 40)

    @kvarMaxAbs.setter
    def kvarMaxAbs(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 40, value)

    @property
    def kVDC(self) -> float:
        """
        Indicates the rated voltage (kV) at the input of the inverter at the peak of PV energy production. The value is normally greater or equal to the kV base of the PV system. It is used for dynamics simulation ONLY.

        DSS property name: `kVDC`, DSS property index: 41.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 41)

    @kVDC.setter
    def kVDC(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 41, value)

    @property
    def Kp(self) -> float:
        """
        It is the proportional gain for the PI controller within the inverter. Use it to modify the controller response in dynamics simulation mode.

        DSS property name: `Kp`, DSS property index: 42.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 42)

    @Kp.setter
    def Kp(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 42, value)

    @property
    def PITol(self) -> float:
        """
        It is the tolerance (%) for the closed loop controller of the inverter. For dynamics simulation mode.

        DSS property name: `PITol`, DSS property index: 43.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 43)

    @PITol.setter
    def PITol(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 43, value)

    @property
    def SafeVoltage(self) -> float:
        """
        Indicates the voltage level (%) respect to the base voltage level for which the Inverter will operate. If this threshold is violated, the Inverter will enter safe mode (OFF). For dynamic simulation. By default is 80%

        DSS property name: `SafeVoltage`, DSS property index: 44.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 44)

    @SafeVoltage.setter
    def SafeVoltage(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 44, value)

    @property
    def SafeMode(self) -> bool:
        """
        (Read only) Indicates whether the inverter entered (Yes) or not (No) into Safe Mode.

        DSS property name: `SafeMode`, DSS property index: 45.
        """
        return self._lib.Obj_GetInt32(self._ptr, 45) != 0

    @SafeMode.setter
    def SafeMode(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 45, value)

    @property
    def DynamicEq(self) -> str:
        """
        The name of the dynamic equation (DynamicExp) that will be used for defining the dynamic behavior of the generator. If not defined, the generator dynamics will follow the built-in dynamic equation.

        DSS property name: `DynamicEq`, DSS property index: 46.
        """
        return self._get_prop_string(46)

    @DynamicEq.setter
    def DynamicEq(self, value: Union[AnyStr, DynamicExp]):
        if isinstance(value, DSSObj):
            self._set_obj(46, value)
            return

        self._set_string_o(46, value)

    @property
    def DynamicEq_obj(self) -> DynamicExp:
        """
        The name of the dynamic equation (DynamicExp) that will be used for defining the dynamic behavior of the generator. If not defined, the generator dynamics will follow the built-in dynamic equation.

        DSS property name: `DynamicEq`, DSS property index: 46.
        """
        return self._get_obj(46, DynamicExp)

    @DynamicEq_obj.setter
    def DynamicEq_obj(self, value: DynamicExp):
        self._set_obj(46, value)

    @property
    def DynOut(self) -> str:
        """
        The name of the variables within the Dynamic equation that will be used to govern the PVSystem dynamics. This PVsystem model requires 1 output from the dynamic equation:

            1. Current.

        The output variables need to be defined in the same order.

        DSS property name: `DynOut`, DSS property index: 47.
        """
        return self._get_prop_string(47)

    @DynOut.setter
    def DynOut(self, value: AnyStr):
        self._set_string_o(47, value)

    @property
    def ControlMode(self) -> InverterControlMode:
        """
        Defines the control mode for the inverter. It can be one of {GFM | GFL*}. By default it is GFL (Grid Following Inverter). Use GFM (Grid Forming Inverter) for energizing islanded microgrids, but, if the device is conencted to the grid, it is highly recommended to use GFL.

        GFM control mode disables any control action set by the InvControl device.

        DSS property name: `ControlMode`, DSS property index: 48.
        """
        return InverterControlMode(self._lib.Obj_GetInt32(self._ptr, 48))

    @ControlMode.setter
    def ControlMode(self, value: Union[AnyStr, int, InverterControlMode]):
        if not isinstance(value, int):
            self._set_string_o(48, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 48, value)

    @property
    def ControlMode_str(self) -> str:
        """
        Defines the control mode for the inverter. It can be one of {GFM | GFL*}. By default it is GFL (Grid Following Inverter). Use GFM (Grid Forming Inverter) for energizing islanded microgrids, but, if the device is conencted to the grid, it is highly recommended to use GFL.

        GFM control mode disables any control action set by the InvControl device.

        DSS property name: `ControlMode`, DSS property index: 48.
        """
        return self._get_prop_string(48)

    @ControlMode_str.setter
    def ControlMode_str(self, value: AnyStr):
        self.ControlMode = value

    @property
    def spectrum(self) -> str:
        """
        Name of harmonic voltage or current spectrum for this PVSystem element. A harmonic voltage source is assumed for the inverter. Default value is "default", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 49.
        """
        return self._get_prop_string(49)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum]):
        if isinstance(value, DSSObj):
            self._set_obj(49, value)
            return

        self._set_string_o(49, value)

    @property
    def spectrum_obj(self) -> Spectrum:
        """
        Name of harmonic voltage or current spectrum for this PVSystem element. A harmonic voltage source is assumed for the inverter. Default value is "default", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 49.
        """
        return self._get_obj(49, Spectrum)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_obj(49, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 50.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 50)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 50, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 51.
        """
        return self._lib.Obj_GetInt32(self._ptr, 51) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 51, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 52.
        """
        self._set_string_o(52, value)

class UPFC(DSSObj):
    __slots__ = []
    _cls_name = 'UPFC'
    _cls_idx = 36
    _cls_prop_idx = {
        'bus1': 1,
        'bus2': 2,
        'refkv': 3,
        'pf': 4,
        'frequency': 5,
        'phases': 6,
        'xs': 7,
        'tol1': 8,
        'mode': 9,
        'vpqmax': 10,
        'losscurve': 11,
        'vhlimit': 12,
        'vllimit': 13,
        'climit': 14,
        'refkv2': 15,
        'kvarlimit': 16,
        'element': 17,
        'spectrum': 18,
        'basefreq': 19,
        'enabled': 20,
        'like': 21,
    }

    @property
    def bus1(self) -> str:
        """
        Name of bus to which the input terminal (1) is connected.
        bus1=busname.1.3
        bus1=busname.1.2.3

        DSS property name: `bus1`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(1, value)

    @property
    def bus2(self) -> str:
        """
        Name of bus to which the output terminal (2) is connected.
        bus2=busname.1.2
        bus2=busname.1.2.3

        DSS property name: `bus2`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @bus2.setter
    def bus2(self, value: AnyStr):
        self._set_string_o(2, value)

    @property
    def refkV(self) -> float:
        """
        UPFC.refkV

        DSS property name: `refkV`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @refkV.setter
    def refkV(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def PF(self) -> float:
        """
        UPFC.PF

        DSS property name: `PF`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @PF.setter
    def PF(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def Frequency(self) -> float:
        """
        UPFC.Frequency

        DSS property name: `Frequency`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @Frequency.setter
    def Frequency(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def Phases(self) -> int:
        """
        UPFC.Phases

        DSS property name: `Phases`, DSS property index: 6.
        """
        return self._lib.Obj_GetInt32(self._ptr, 6)

    @Phases.setter
    def Phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def Xs(self) -> float:
        """
        Reactance of the series transformer of the UPFC, ohms (default=0.7540 ... 2 mH)

        DSS property name: `Xs`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @Xs.setter
    def Xs(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def Tol1(self) -> float:
        """
        Tolerance in pu for the series PI controller
        Tol1=0.02 is the format used to define 2% tolerance (Default=2%)

        DSS property name: `Tol1`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @Tol1.setter
    def Tol1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def Mode(self) -> int:
        """
        Integer used to define the control mode of the UPFC: 

        0 = Off, 
        1 = Voltage regulator, 
        2 = Phase angle regulator, 
        3 = Dual mode
        4 = It is a control mode where the user can set two different set points to create a secure GAP, these references must be defined in the parameters RefkV and RefkV2. The only restriction when setting these values is that RefkV must be higher than RefkV2. 
        5 = In this mode the user can define the same GAP using two set points as in control mode 4. The only difference between mode 5 and mode 4 is that in mode 5, the UPFC controller performs dual control actions just as in control mode 3

        DSS property name: `Mode`, DSS property index: 9.
        """
        return self._lib.Obj_GetInt32(self._ptr, 9)

    @Mode.setter
    def Mode(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 9, value)

    @property
    def VpqMax(self) -> float:
        """
        Maximum voltage (in volts) delivered by the series voltage source (Default = 24 V)

        DSS property name: `VpqMax`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @VpqMax.setter
    def VpqMax(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def LossCurve(self) -> str:
        """
        Name of the XYCurve for describing the losses behavior as a function of the voltage at the input of the UPFC

        DSS property name: `LossCurve`, DSS property index: 11.
        """
        return self._get_prop_string(11)

    @LossCurve.setter
    def LossCurve(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(11, value)
            return

        self._set_string_o(11, value)

    @property
    def LossCurve_obj(self) -> XYcurve:
        """
        Name of the XYCurve for describing the losses behavior as a function of the voltage at the input of the UPFC

        DSS property name: `LossCurve`, DSS property index: 11.
        """
        return self._get_obj(11, XYcurve)

    @LossCurve_obj.setter
    def LossCurve_obj(self, value: XYcurve):
        self._set_obj(11, value)

    @property
    def VHLimit(self) -> float:
        """
        High limit for the voltage at the input of the UPFC, if the voltage is above this value the UPFC turns off. This value is specified in Volts (default 300 V)

        DSS property name: `VHLimit`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @VHLimit.setter
    def VHLimit(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def VLLimit(self) -> float:
        """
        low limit for the voltage at the input of the UPFC, if voltage is below this value the UPFC turns off. This value is specified in Volts (default 125 V)

        DSS property name: `VLLimit`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @VLLimit.setter
    def VLLimit(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def CLimit(self) -> float:
        """
        Current Limit for the UPFC, if the current passing through the UPFC is higher than this value the UPFC turns off. This value is specified in Amps (Default 265 A)

        DSS property name: `CLimit`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @CLimit.setter
    def CLimit(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def refkV2(self) -> float:
        """
        UPFC.refkV2

        DSS property name: `refkV2`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @refkV2.setter
    def refkV2(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def kvarLimit(self) -> float:
        """
        Maximum amount of reactive power (kvar) that can be absorved by the UPFC (Default = 5)

        DSS property name: `kvarLimit`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @kvarLimit.setter
    def kvarLimit(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def Element(self) -> str:
        """
        The name of the PD element monitored when operating with reactive power compensation. Normally, it should be the PD element immediately upstream the UPFC. The element must be defined including the class, e.g. Line.myline.

        DSS property name: `Element`, DSS property index: 17.
        """
        return self._get_prop_string(17)

    @Element.setter
    def Element(self, value: Union[AnyStr, PDElement]):
        if isinstance(value, DSSObj):
            self._set_obj(17, value)
            return

        self._set_string_o(17, value)

    @property
    def Element_obj(self) -> PDElement:
        """
        The name of the PD element monitored when operating with reactive power compensation. Normally, it should be the PD element immediately upstream the UPFC. The element must be defined including the class, e.g. Line.myline.

        DSS property name: `Element`, DSS property index: 17.
        """
        return self._get_obj(17, PDElement)

    @Element_obj.setter
    def Element_obj(self, value: PDElement):
        self._set_obj(17, value)

    @property
    def spectrum(self) -> str:
        """
        Name of harmonic spectrum for this source.  Default is "defaultUPFC", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 18.
        """
        return self._get_prop_string(18)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum]):
        if isinstance(value, DSSObj):
            self._set_obj(18, value)
            return

        self._set_string_o(18, value)

    @property
    def spectrum_obj(self) -> Spectrum:
        """
        Name of harmonic spectrum for this source.  Default is "defaultUPFC", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 18.
        """
        return self._get_obj(18, Spectrum)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_obj(18, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 20.
        """
        return self._lib.Obj_GetInt32(self._ptr, 20) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 20, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 21.
        """
        self._set_string_o(21, value)

class UPFCControl(DSSObj):
    __slots__ = []
    _cls_name = 'UPFCControl'
    _cls_idx = 37
    _cls_prop_idx = {
        'upfclist': 1,
        'basefreq': 2,
        'enabled': 3,
        'like': 4,
    }

    @property
    def UPFCList(self) -> List[str]:
        """
        The list of all the UPFC devices to be controlled by this controller, If left empty, this control will apply for all UPFCs in the model.

        DSS property name: `UPFCList`, DSS property index: 1.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 1)

    @UPFCList.setter
    def UPFCList(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 1, value_ptr, value_count)
        self._check_for_error()

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 2.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 2)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 2, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 3.
        """
        return self._lib.Obj_GetInt32(self._ptr, 3) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 3, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 4.
        """
        self._set_string_o(4, value)

class ESPVLControl(DSSObj):
    __slots__ = []
    _cls_name = 'ESPVLControl'
    _cls_idx = 38
    _cls_prop_idx = {
        'element': 1,
        'terminal': 2,
        'type': 3,
        'kwband': 4,
        'kvarlimit': 5,
        'localcontrollist': 6,
        'localcontrolweights': 7,
        'pvsystemlist': 8,
        'pvsystemweights': 9,
        'storagelist': 10,
        'storageweights': 11,
        'basefreq': 12,
        'enabled': 13,
        'like': 14,
    }

    # Class-specific enumerations
    class ESPVLControlType(IntEnum):
        """ESPVLControl: Type (DSS enumeration for ESPVLControl)"""
        SystemController = 1 # SystemController
        LocalController = 2 # LocalController


    @property
    def Element(self) -> str:
        """
        Full object name of the circuit element, typically a line or transformer, which the control is monitoring. There is no default; must be specified.

        DSS property name: `Element`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @Element.setter
    def Element(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(1, value)
            return

        self._set_string_o(1, value)

    @property
    def Element_obj(self) -> DSSObj:
        """
        Full object name of the circuit element, typically a line or transformer, which the control is monitoring. There is no default; must be specified.

        DSS property name: `Element`, DSS property index: 1.
        """
        return self._get_obj(1, None)

    @Element_obj.setter
    def Element_obj(self, value: DSSObj):
        self._set_obj(1, value)

    @property
    def Terminal(self) -> int:
        """
        Number of the terminal of the circuit element to which the ESPVLControl control is connected. 1 or 2, typically.  Default is 1. Make sure you have the direction on the power matching the sign of kWLimit.

        DSS property name: `Terminal`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @Terminal.setter
    def Terminal(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def Type(self) -> ESPVLControlType:
        """
        Type of controller.  1= System Controller; 2= Local controller. 

        DSS property name: `Type`, DSS property index: 3.
        """
        return ESPVLControl.ESPVLControlType(self._lib.Obj_GetInt32(self._ptr, 3))

    @Type.setter
    def Type(self, value: Union[AnyStr, int, ESPVLControlType]):
        if not isinstance(value, int):
            self._set_string_o(3, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 3, value)

    @property
    def Type_str(self) -> str:
        """
        Type of controller.  1= System Controller; 2= Local controller. 

        DSS property name: `Type`, DSS property index: 3.
        """
        return self._get_prop_string(3)

    @Type_str.setter
    def Type_str(self, value: AnyStr):
        self.Type = value

    @property
    def kWBand(self) -> float:
        """
        Bandwidth (kW) of the dead band around the target limit.No dispatch changes are attempted if the power in the monitored terminal stays within this band.

        DSS property name: `kWBand`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @kWBand.setter
    def kWBand(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def kvarlimit(self) -> float:
        """
        Max kvar to be delivered through the element.  Uses same dead band as kW.

        DSS property name: `kvarlimit`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @kvarlimit.setter
    def kvarlimit(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def LocalControlList(self) -> List[str]:
        """
        Array list of ESPVLControl local controller objects to be dispatched by System Controller. If not specified, all ESPVLControl devices with type=local in the circuit not attached to another controller are assumed to be part of this controller's fleet.

        DSS property name: `LocalControlList`, DSS property index: 6.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 6)

    @LocalControlList.setter
    def LocalControlList(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 6, value_ptr, value_count)
        self._check_for_error()

    @property
    def LocalControlWeights(self) -> Float64Array:
        """
        Array of proportional weights corresponding to each ESPVLControl local controller in the LocalControlList.

        DSS property name: `LocalControlWeights`, DSS property index: 7.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 7)

    @LocalControlWeights.setter
    def LocalControlWeights(self, value: Float64Array):
        self._set_float64_array_o(7, value)

    @property
    def PVSystemList(self) -> List[str]:
        """
        Array list of PVSystem objects to be dispatched by a Local Controller. 

        DSS property name: `PVSystemList`, DSS property index: 8.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 8)

    @PVSystemList.setter
    def PVSystemList(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 8, value_ptr, value_count)
        self._check_for_error()

    @property
    def PVSystemWeights(self) -> Float64Array:
        """
        Array of proportional weights corresponding to each PVSystem in the PVSystemList.

        DSS property name: `PVSystemWeights`, DSS property index: 9.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 9)

    @PVSystemWeights.setter
    def PVSystemWeights(self, value: Float64Array):
        self._set_float64_array_o(9, value)

    @property
    def StorageList(self) -> List[str]:
        """
        Array list of Storage objects to be dispatched by Local Controller. 

        DSS property name: `StorageList`, DSS property index: 10.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 10)

    @StorageList.setter
    def StorageList(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 10, value_ptr, value_count)
        self._check_for_error()

    @property
    def StorageWeights(self) -> Float64Array:
        """
        Array of proportional weights corresponding to each Storage object in the StorageControlList.

        DSS property name: `StorageWeights`, DSS property index: 11.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 11)

    @StorageWeights.setter
    def StorageWeights(self, value: Float64Array):
        self._set_float64_array_o(11, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 13.
        """
        return self._lib.Obj_GetInt32(self._ptr, 13) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 13, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 14.
        """
        self._set_string_o(14, value)

class IndMach012(DSSObj):
    __slots__ = []
    _cls_name = 'IndMach012'
    _cls_idx = 39
    _cls_prop_idx = {
        'phases': 1,
        'bus1': 2,
        'kv': 3,
        'kw': 4,
        'pf': 5,
        'conn': 6,
        'kva': 7,
        'h': 8,
        'd': 9,
        'purs': 10,
        'puxs': 11,
        'purr': 12,
        'puxr': 13,
        'puxm': 14,
        'slip': 15,
        'maxslip': 16,
        'slipoption': 17,
        'yearly': 18,
        'daily': 19,
        'duty': 20,
        'debugtrace': 21,
        'spectrum': 22,
        'basefreq': 23,
        'enabled': 24,
        'like': 25,
    }

    # Class-specific enumerations
    class IndMach012SlipOption(IntEnum):
        """IndMach012: Slip Option (DSS enumeration for IndMach012)"""
        VariableSlip = 0 # VariableSlip
        FixedSlip = 1 # FixedSlip


    @property
    def phases(self) -> int:
        """
        Number of Phases, this Induction Machine.  

        DSS property name: `phases`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def bus1(self) -> str:
        """
        Bus to which the Induction Machine is connected.  May include specific node specification.

        DSS property name: `bus1`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(2, value)

    @property
    def kv(self) -> float:
        """
        Nominal rated (1.0 per unit) voltage, kV. For 2- and 3-phase machines, specify phase-phase kV. Otherwise, specify actual kV across each branch of the machine. If wye (star), specify phase-neutral kV. If delta or phase-phase connected, specify phase-phase kV.

        DSS property name: `kv`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @kv.setter
    def kv(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def kW(self) -> float:
        """
        Shaft Power, kW, for the Induction Machine.  A positive value denotes power for a load. 
        Negative value denotes an induction generator. 

        DSS property name: `kW`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @kW.setter
    def kW(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def pf(self) -> float:
        """
        [Read Only] Present power factor for the machine. 

        DSS property name: `pf`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @pf.setter
    def pf(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def conn(self) -> Connection:
        """
        Connection of stator: Delta or Wye. Default is Delta.

        DSS property name: `conn`, DSS property index: 6.
        """
        return Connection(self._lib.Obj_GetInt32(self._ptr, 6))

    @conn.setter
    def conn(self, value: Union[AnyStr, int, Connection]):
        if not isinstance(value, int):
            self._set_string_o(6, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def conn_str(self) -> str:
        """
        Connection of stator: Delta or Wye. Default is Delta.

        DSS property name: `conn`, DSS property index: 6.
        """
        return self._get_prop_string(6)

    @conn_str.setter
    def conn_str(self, value: AnyStr):
        self.conn = value

    @property
    def kVA(self) -> float:
        """
        Rated kVA for the machine.

        DSS property name: `kVA`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @kVA.setter
    def kVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def H(self) -> float:
        """
        Per unit mass constant of the machine.  MW-sec/MVA.  Default is 1.0.

        DSS property name: `H`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @H.setter
    def H(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def D(self) -> float:
        """
        Damping constant.  Usual range is 0 to 4. Default is 1.0.  Adjust to get damping in Dynamics mode,

        DSS property name: `D`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @D.setter
    def D(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def puRs(self) -> float:
        """
        Per unit stator resistance. Default is 0.0053.

        DSS property name: `puRs`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @puRs.setter
    def puRs(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def puXs(self) -> float:
        """
        Per unit stator leakage reactance. Default is 0.106.

        DSS property name: `puXs`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @puXs.setter
    def puXs(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def puRr(self) -> float:
        """
        Per unit rotor  resistance. Default is 0.007.

        DSS property name: `puRr`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @puRr.setter
    def puRr(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def puXr(self) -> float:
        """
        Per unit rotor leakage reactance. Default is 0.12.

        DSS property name: `puXr`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @puXr.setter
    def puXr(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def puXm(self) -> float:
        """
        Per unit magnetizing reactance.Default is 4.0.

        DSS property name: `puXm`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @puXm.setter
    def puXm(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def Slip(self) -> float:
        """
        Initial slip value. Default is 0.007

        DSS property name: `Slip`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @Slip.setter
    def Slip(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def MaxSlip(self) -> float:
        """
        Max slip value to allow. Default is 0.1. Set this before setting slip.

        DSS property name: `MaxSlip`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @MaxSlip.setter
    def MaxSlip(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def SlipOption(self) -> IndMach012SlipOption:
        """
        Option for slip model. One of {fixedslip | variableslip*  }

        DSS property name: `SlipOption`, DSS property index: 17.
        """
        return IndMach012.IndMach012SlipOption(self._lib.Obj_GetInt32(self._ptr, 17))

    @SlipOption.setter
    def SlipOption(self, value: Union[AnyStr, int, IndMach012SlipOption]):
        if not isinstance(value, int):
            self._set_string_o(17, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 17, value)

    @property
    def SlipOption_str(self) -> str:
        """
        Option for slip model. One of {fixedslip | variableslip*  }

        DSS property name: `SlipOption`, DSS property index: 17.
        """
        return self._get_prop_string(17)

    @SlipOption_str.setter
    def SlipOption_str(self, value: AnyStr):
        self.SlipOption = value

    @property
    def Yearly(self) -> str:
        """
        LOADSHAPE object to use for yearly simulations.  Must be previously defined as a Loadshape object. Is set to the Daily load shape  when Daily is defined.  The daily load shape is repeated in this case. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. The default is no variation.

        DSS property name: `Yearly`, DSS property index: 18.
        """
        return self._get_prop_string(18)

    @Yearly.setter
    def Yearly(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(18, value)
            return

        self._set_string_o(18, value)

    @property
    def Yearly_obj(self) -> LoadShape:
        """
        LOADSHAPE object to use for yearly simulations.  Must be previously defined as a Loadshape object. Is set to the Daily load shape  when Daily is defined.  The daily load shape is repeated in this case. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. The default is no variation.

        DSS property name: `Yearly`, DSS property index: 18.
        """
        return self._get_obj(18, LoadShape)

    @Yearly_obj.setter
    def Yearly_obj(self, value: LoadShape):
        self._set_obj(18, value)

    @property
    def Daily(self) -> str:
        """
        LOADSHAPE object to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Default is no variation (constant) if not defined. Side effect: Sets Yearly load shape if not already defined.

        DSS property name: `Daily`, DSS property index: 19.
        """
        return self._get_prop_string(19)

    @Daily.setter
    def Daily(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(19, value)
            return

        self._set_string_o(19, value)

    @property
    def Daily_obj(self) -> LoadShape:
        """
        LOADSHAPE object to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Default is no variation (constant) if not defined. Side effect: Sets Yearly load shape if not already defined.

        DSS property name: `Daily`, DSS property index: 19.
        """
        return self._get_obj(19, LoadShape)

    @Daily_obj.setter
    def Daily_obj(self, value: LoadShape):
        self._set_obj(19, value)

    @property
    def Duty(self) -> str:
        """
        LOADSHAPE object to use for duty cycle simulations.  Must be previously defined as a Loadshape object.  Typically would have time intervals less than 1 hr. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.Set to NONE to reset to no loadahape. Set Status=Fixed to ignore Loadshape designation.  Defaults to Daily curve If not specified.

        DSS property name: `Duty`, DSS property index: 20.
        """
        return self._get_prop_string(20)

    @Duty.setter
    def Duty(self, value: Union[AnyStr, LoadShape]):
        if isinstance(value, DSSObj):
            self._set_obj(20, value)
            return

        self._set_string_o(20, value)

    @property
    def Duty_obj(self) -> LoadShape:
        """
        LOADSHAPE object to use for duty cycle simulations.  Must be previously defined as a Loadshape object.  Typically would have time intervals less than 1 hr. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.Set to NONE to reset to no loadahape. Set Status=Fixed to ignore Loadshape designation.  Defaults to Daily curve If not specified.

        DSS property name: `Duty`, DSS property index: 20.
        """
        return self._get_obj(20, LoadShape)

    @Duty_obj.setter
    def Duty_obj(self, value: LoadShape):
        self._set_obj(20, value)

    @property
    def Debugtrace(self) -> bool:
        """
        [Yes | No*] Write DebugTrace file.

        DSS property name: `Debugtrace`, DSS property index: 21.
        """
        return self._lib.Obj_GetInt32(self._ptr, 21) != 0

    @Debugtrace.setter
    def Debugtrace(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 21, value)

    @property
    def spectrum(self) -> str:
        """
        Name of harmonic voltage or current spectrum for this IndMach012. Voltage behind Xd" for machine - default. Current injection for inverter. Default value is "default", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 22.
        """
        return self._get_prop_string(22)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum]):
        if isinstance(value, DSSObj):
            self._set_obj(22, value)
            return

        self._set_string_o(22, value)

    @property
    def spectrum_obj(self) -> Spectrum:
        """
        Name of harmonic voltage or current spectrum for this IndMach012. Voltage behind Xd" for machine - default. Current injection for inverter. Default value is "default", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 22.
        """
        return self._get_obj(22, Spectrum)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_obj(22, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 24.
        """
        return self._lib.Obj_GetInt32(self._ptr, 24) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 24, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 25.
        """
        self._set_string_o(25, value)

class GICsource(DSSObj):
    __slots__ = []
    _cls_name = 'GICsource'
    _cls_idx = 40
    _cls_prop_idx = {
        'volts': 1,
        'angle': 2,
        'frequency': 3,
        'phases': 4,
        'en': 5,
        'ee': 6,
        'lat1': 7,
        'lon1': 8,
        'lat2': 9,
        'lon2': 10,
        'spectrum': 11,
        'basefreq': 12,
        'enabled': 13,
        'like': 14,
    }

    @property
    def Volts(self) -> float:
        """
        Voltage magnitude, in volts, of the GIC voltage induced across the associated line. When specified, induced voltage is assumed defined by Voltage and Angle properties. 

        Specify this value

        OR

        EN, EE, lat1, lon1, lat2, lon2. 

        Not both!!  Last one entered will take precedence. Assumed identical in each phase of the Line object.

        DSS property name: `Volts`, DSS property index: 1.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 1)

    @Volts.setter
    def Volts(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 1, value)

    @property
    def angle(self) -> float:
        """
        Phase angle in degrees of first phase. Default=0.0.  See Voltage property

        DSS property name: `angle`, DSS property index: 2.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 2)

    @angle.setter
    def angle(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 2, value)

    @property
    def frequency(self) -> float:
        """
        Source frequency.  Defaults to  0.1 Hz. So GICSource=0 at power frequency.

        DSS property name: `frequency`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @frequency.setter
    def frequency(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def phases(self) -> int:
        """
        Number of phases.  Defaults to 3. All three phases are assumed in phase (zero sequence)

        DSS property name: `phases`, DSS property index: 4.
        """
        return self._lib.Obj_GetInt32(self._ptr, 4)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 4, value)

    @property
    def EN(self) -> float:
        """
        Northward Electric field (V/km). If specified, Voltage and Angle are computed from EN, EE, lat and lon values.

        DSS property name: `EN`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @EN.setter
    def EN(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def EE(self) -> float:
        """
        Eastward Electric field (V/km).  If specified, Voltage and Angle are computed from EN, EE, lat and lon values.

        DSS property name: `EE`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @EE.setter
    def EE(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def Lat1(self) -> float:
        """
        Latitude of Bus1 of the line(degrees)

        DSS property name: `Lat1`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @Lat1.setter
    def Lat1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def Lon1(self) -> float:
        """
        Longitude of Bus1 of the line (degrees)

        DSS property name: `Lon1`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @Lon1.setter
    def Lon1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def Lat2(self) -> float:
        """
        Latitude of Bus2 of the line (degrees)

        DSS property name: `Lat2`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @Lat2.setter
    def Lat2(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def Lon2(self) -> float:
        """
        Longitude of Bus2 of the line (degrees)

        DSS property name: `Lon2`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @Lon2.setter
    def Lon2(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def spectrum(self) -> str:
        """
        Not used.

        DSS property name: `spectrum`, DSS property index: 11.
        """
        return self._get_prop_string(11)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum]):
        if isinstance(value, DSSObj):
            self._set_obj(11, value)
            return

        self._set_string_o(11, value)

    @property
    def spectrum_obj(self) -> Spectrum:
        """
        Not used.

        DSS property name: `spectrum`, DSS property index: 11.
        """
        return self._get_obj(11, Spectrum)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_obj(11, value)

    @property
    def basefreq(self) -> float:
        """
        Not used.

        DSS property name: `basefreq`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 13.
        """
        return self._lib.Obj_GetInt32(self._ptr, 13) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 13, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 14.
        """
        self._set_string_o(14, value)

class AutoTrans(DSSObj):
    __slots__ = []
    _cls_name = 'AutoTrans'
    _cls_idx = 41
    _cls_prop_idx = {
        'phases': 1,
        'windings': 2,
        'wdg': 3,
        'bus': 4,
        'conn': 5,
        'kv': 6,
        'kva': 7,
        'tap': 8,
        'pctr': 9,
        '%r': 9,
        'rdcohms': 10,
        'core': 11,
        'buses': 12,
        'conns': 13,
        'kvs': 14,
        'kvas': 15,
        'taps': 16,
        'xhx': 17,
        'xht': 18,
        'xxt': 19,
        'xscarray': 20,
        'thermal': 21,
        'n': 22,
        'm': 23,
        'flrise': 24,
        'hsrise': 25,
        'pctloadloss': 26,
        '%loadloss': 26,
        'pctnoloadloss': 27,
        '%noloadloss': 27,
        'normhkva': 28,
        'emerghkva': 29,
        'sub': 30,
        'maxtap': 31,
        'mintap': 32,
        'numtaps': 33,
        'subname': 34,
        'pctimag': 35,
        '%imag': 35,
        'ppm_antifloat': 36,
        'pctrs': 37,
        '%rs': 37,
        'xrconst': 38,
        'leadlag': 39,
        'wdgcurrents': 40,
        'normamps': 41,
        'emergamps': 42,
        'faultrate': 43,
        'pctperm': 44,
        'repair': 45,
        'basefreq': 46,
        'enabled': 47,
        'like': 48,
    }

    # Class-specific enumerations
    class AutoTransConnection(IntEnum):
        """AutoTrans: Connection (DSS enumeration for AutoTrans)"""
        wye = 0 # wye
        delta = 1 # delta
        series = 2 # series
        y = 0 # y
        ln = 0 # ln
        ll = 1 # ll


    @property
    def phases(self) -> int:
        """
        Number of phases this AutoTrans. Default is 3.

        DSS property name: `phases`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def windings(self) -> int:
        """
        Number of windings, this AutoTranss. (Also is the number of terminals) Default is 2. This property triggers memory allocation for the AutoTrans and will cause other properties to revert to default values.

        DSS property name: `windings`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @windings.setter
    def windings(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def pctR(self) -> Float64Array:
        """
        Percent ac resistance this winding.  This value is for the power flow model.Is derived from the full load losses in the transformer test report.

        DSS property name: `%R`, DSS property index: 9.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 9)

    @pctR.setter
    def pctR(self, value: Float64Array):
        self._set_float64_array_o(9, value)

    @property
    def Rdcohms(self) -> Float64Array:
        """
        Winding dc resistance in OHMS. Specify this for GIC analysis. From transformer test report (divide by number of phases). Defaults to 85% of %R property (the ac value that includes stray losses).

        DSS property name: `Rdcohms`, DSS property index: 10.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 10)

    @Rdcohms.setter
    def Rdcohms(self, value: Float64Array):
        self._set_float64_array_o(10, value)

    @property
    def Core(self) -> CoreType:
        """
        {Shell*|5-leg|3-Leg|1-phase|core-1-phase|4-leg} Core Type. Used for GIC analysis in auxiliary programs. Not used inside OpenDSS.

        DSS property name: `Core`, DSS property index: 11.
        """
        return CoreType(self._lib.Obj_GetInt32(self._ptr, 11))

    @Core.setter
    def Core(self, value: Union[AnyStr, int, CoreType]):
        if not isinstance(value, int):
            self._set_string_o(11, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 11, value)

    @property
    def Core_str(self) -> str:
        """
        {Shell*|5-leg|3-Leg|1-phase|core-1-phase|4-leg} Core Type. Used for GIC analysis in auxiliary programs. Not used inside OpenDSS.

        DSS property name: `Core`, DSS property index: 11.
        """
        return self._get_prop_string(11)

    @Core_str.setter
    def Core_str(self, value: AnyStr):
        self.Core = value

    @property
    def buses(self) -> List[str]:
        """
        Use this to specify all the bus connections at once using an array. Example:

        New AutoTrans.T1 buses=[Hbus, Xbus]

        DSS property name: `buses`, DSS property index: 12.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 12)

    @buses.setter
    def buses(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 12, value_ptr, value_count)
        self._check_for_error()

    @property
    def conns(self) -> List[AutoTransConnection]:
        """
        Use this to specify all the Winding connections at once using an array. Example:

        New AutoTrans.T1 buses=[Hbus, Xbus] ~ conns=(series, wye)

        DSS property name: `conns`, DSS property index: 13.
        """
        return [AutoTrans.AutoTransConnection(val) for val in self._get_int32_list(self._lib.Obj_GetInt32Array, self._ptr, 13)]

    @conns.setter
    def conns(self, value: Union[List[Union[int,AutoTransConnection]], List[AnyStr]]):
        if len(value) and not isinstance(value[0], int):
            self._set_string_array_o(13, value)
            return    
        self._set_int32_array_o(13, value)

    @property
    def conns_str(self) -> List[str]:
        """
        Use this to specify all the Winding connections at once using an array. Example:

        New AutoTrans.T1 buses=[Hbus, Xbus] ~ conns=(series, wye)

        DSS property name: `conns`, DSS property index: 13.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 13)

    @conns_str.setter
    def conns_str(self, value: AnyStr):
        self.conns = value

    @property
    def kVs(self) -> Float64Array:
        """
        Use this to specify the kV ratings of all windings at once using an array. Example:

        New AutoTrans.T1 buses=[Hbus, Xbus] 
        ~ conns=(series, wye)
        ~ kvs=(115, 12.47)

        See kV= property for voltage rules.

        DSS property name: `kVs`, DSS property index: 14.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 14)

    @kVs.setter
    def kVs(self, value: Float64Array):
        self._set_float64_array_o(14, value)

    @property
    def kVAs(self) -> Float64Array:
        """
        Use this to specify the kVA ratings of all windings at once using an array.

        DSS property name: `kVAs`, DSS property index: 15.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 15)

    @kVAs.setter
    def kVAs(self, value: Float64Array):
        self._set_float64_array_o(15, value)

    @property
    def taps(self) -> Float64Array:
        """
        Use this to specify the p.u. tap of all windings at once using an array.

        DSS property name: `taps`, DSS property index: 16.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 16)

    @taps.setter
    def taps(self, value: Float64Array):
        self._set_float64_array_o(16, value)

    @property
    def XHX(self) -> float:
        """
        Use this to specify the percent reactance, H-L (winding 1 to winding 2).  Use for 2- or 3-winding AutoTranss. On the kVA base of winding 1(H-X). 

        DSS property name: `XHX`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @XHX.setter
    def XHX(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def XHT(self) -> float:
        """
        Use this to specify the percent reactance, H-T (winding 1 to winding 3).  Use for 3-winding AutoTranss only. On the kVA base of winding 1(H-X). 

        DSS property name: `XHT`, DSS property index: 18.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 18)

    @XHT.setter
    def XHT(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 18, value)

    @property
    def XXT(self) -> float:
        """
        Use this to specify the percent reactance, L-T (winding 2 to winding 3).  Use for 3-winding AutoTranss only. On the kVA base of winding 1(H-X).  

        DSS property name: `XXT`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @XXT.setter
    def XXT(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def XSCarray(self) -> Float64Array:
        """
        Use this to specify the percent reactance between all pairs of windings as an array. All values are on the kVA base of winding 1.  The order of the values is as follows:

        (x12 13 14... 23 24.. 34 ..)  

        There will be n(n-1)/2 values, where n=number of windings.

        DSS property name: `XSCarray`, DSS property index: 20.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 20)

    @XSCarray.setter
    def XSCarray(self, value: Float64Array):
        self._set_float64_array_o(20, value)

    @property
    def thermal(self) -> float:
        """
        Thermal time constant of the AutoTrans in hours.  Typically about 2.

        DSS property name: `thermal`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @thermal.setter
    def thermal(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def n(self) -> float:
        """
        n Exponent for thermal properties in IEEE C57.  Typically 0.8.

        DSS property name: `n`, DSS property index: 22.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 22)

    @n.setter
    def n(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 22, value)

    @property
    def m(self) -> float:
        """
        m Exponent for thermal properties in IEEE C57.  Typically 0.9 - 1.0

        DSS property name: `m`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @m.setter
    def m(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def flrise(self) -> float:
        """
        Temperature rise, deg C, for full load.  Default is 65.

        DSS property name: `flrise`, DSS property index: 24.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 24)

    @flrise.setter
    def flrise(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 24, value)

    @property
    def hsrise(self) -> float:
        """
        Hot spot temperature rise, deg C.  Default is 15.

        DSS property name: `hsrise`, DSS property index: 25.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 25)

    @hsrise.setter
    def hsrise(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 25, value)

    @property
    def pctloadloss(self) -> float:
        """
        Percent load loss at full load. The %R of the High and Low windings (1 and 2) are adjusted to agree at rated kVA loading.

        DSS property name: `%loadloss`, DSS property index: 26.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 26)

    @pctloadloss.setter
    def pctloadloss(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 26, value)

    @property
    def pctnoloadloss(self) -> float:
        """
        Percent no load losses at rated excitatation voltage. Default is 0. Converts to a resistance in parallel with the magnetizing impedance in each winding.

        DSS property name: `%noloadloss`, DSS property index: 27.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 27)

    @pctnoloadloss.setter
    def pctnoloadloss(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 27, value)

    @property
    def normhkVA(self) -> float:
        """
        Normal maximum kVA rating of H winding (winding 1+2).  Usually 100% - 110% ofmaximum nameplate rating, depending on load shape. Defaults to 110% of kVA rating of Winding 1.

        DSS property name: `normhkVA`, DSS property index: 28.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 28)

    @normhkVA.setter
    def normhkVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 28, value)

    @property
    def emerghkVA(self) -> float:
        """
        Emergency (contingency)  kVA rating of H winding (winding 1+2).  Usually 140% - 150% ofmaximum nameplate rating, depending on load shape. Defaults to 150% of kVA rating of Winding 1.

        DSS property name: `emerghkVA`, DSS property index: 29.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 29)

    @emerghkVA.setter
    def emerghkVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 29, value)

    @property
    def sub(self) -> bool:
        """
        ={Yes|No}  Designates whether this AutoTrans is to be considered a substation.Default is No.

        DSS property name: `sub`, DSS property index: 30.
        """
        return self._lib.Obj_GetInt32(self._ptr, 30) != 0

    @sub.setter
    def sub(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 30, value)

    @property
    def MaxTap(self) -> Float64Array:
        """
        Max per unit tap for the active winding.  Default is 1.10

        DSS property name: `MaxTap`, DSS property index: 31.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 31)

    @MaxTap.setter
    def MaxTap(self, value: Float64Array):
        self._set_float64_array_o(31, value)

    @property
    def MinTap(self) -> Float64Array:
        """
        Min per unit tap for the active winding.  Default is 0.90

        DSS property name: `MinTap`, DSS property index: 32.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 32)

    @MinTap.setter
    def MinTap(self, value: Float64Array):
        self._set_float64_array_o(32, value)

    @property
    def NumTaps(self) -> Int32Array:
        """
        Total number of taps between min and max tap.  Default is 32 (16 raise and 16 lower taps about the neutral position). The neutral position is not counted.

        DSS property name: `NumTaps`, DSS property index: 33.
        """
        return self._get_int32_array(self._lib.Obj_GetInt32Array, self._ptr, 33)

    @NumTaps.setter
    def NumTaps(self, value: Int32Array):
        self._set_int32_array_o(33, value)

    @property
    def subname(self) -> str:
        """
        Substation Name. Optional. Default is null. If specified, printed on plots

        DSS property name: `subname`, DSS property index: 34.
        """
        return self._get_prop_string(34)

    @subname.setter
    def subname(self, value: AnyStr):
        self._set_string_o(34, value)

    @property
    def pctimag(self) -> float:
        """
        Percent magnetizing current. Default=0.0. Magnetizing branch is in parallel with windings in each phase. Also, see "ppm_antifloat".

        DSS property name: `%imag`, DSS property index: 35.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 35)

    @pctimag.setter
    def pctimag(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 35, value)

    @property
    def ppm_antifloat(self) -> float:
        """
        Default=1 ppm.  Parts per million of AutoTrans winding VA rating connected to ground to protect against accidentally floating a winding without a reference. If positive then the effect is adding a very large reactance to ground.  If negative, then a capacitor.

        DSS property name: `ppm_antifloat`, DSS property index: 36.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 36)

    @ppm_antifloat.setter
    def ppm_antifloat(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 36, value)

    @property
    def pctRs(self) -> Float64Array:
        """
        Use this property to specify all the winding ac %resistances using an array. Example:

        New AutoTrans.T1 buses=[Hibus, lowbus] ~ %Rs=(0.2  0.3)

        DSS property name: `%Rs`, DSS property index: 37.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 37)

    @pctRs.setter
    def pctRs(self, value: Float64Array):
        self._set_float64_array_o(37, value)

    @property
    def XRConst(self) -> bool:
        """
        ={Yes|No} Default is NO. Signifies whether or not the X/R is assumed contant for harmonic studies.

        DSS property name: `XRConst`, DSS property index: 38.
        """
        return self._lib.Obj_GetInt32(self._ptr, 38) != 0

    @XRConst.setter
    def XRConst(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 38, value)

    @property
    def LeadLag(self) -> PhaseSequence:
        """
        {Lead | Lag (default) | ANSI (default) | Euro } Designation in mixed Delta-wye connections the relationship between HV to LV winding. Default is ANSI 30 deg lag, e.g., Dy1 of Yd1 vector group. To get typical European Dy11 connection, specify either "lead" or "Euro"

        DSS property name: `LeadLag`, DSS property index: 39.
        """
        return PhaseSequence(self._lib.Obj_GetInt32(self._ptr, 39))

    @LeadLag.setter
    def LeadLag(self, value: Union[AnyStr, int, PhaseSequence]):
        if not isinstance(value, int):
            self._set_string_o(39, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 39, value)

    @property
    def LeadLag_str(self) -> str:
        """
        {Lead | Lag (default) | ANSI (default) | Euro } Designation in mixed Delta-wye connections the relationship between HV to LV winding. Default is ANSI 30 deg lag, e.g., Dy1 of Yd1 vector group. To get typical European Dy11 connection, specify either "lead" or "Euro"

        DSS property name: `LeadLag`, DSS property index: 39.
        """
        return self._get_prop_string(39)

    @LeadLag_str.setter
    def LeadLag_str(self, value: AnyStr):
        self.LeadLag = value

    def WdgCurrents(self) -> str:
        """
        (Read only) Makes winding currents available via return on query (? AutoTrans.TX.WdgCurrents). Order: Phase 1, Wdg 1, Wdg 2, ..., Phase 2 ...

        DSS property name: `WdgCurrents`, DSS property index: 40.
        """
        # []
        # StringSilentROFunction
        return self._get_prop_string(self._lib.Obj_GetString(self._ptr, 40))

    @property
    def normamps(self) -> float:
        """
        Normal rated current.

        DSS property name: `normamps`, DSS property index: 41.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 41)

    @normamps.setter
    def normamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 41, value)

    @property
    def emergamps(self) -> float:
        """
        Maximum or emerg current.

        DSS property name: `emergamps`, DSS property index: 42.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 42)

    @emergamps.setter
    def emergamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 42, value)

    @property
    def faultrate(self) -> float:
        """
        Failure rate per year.

        DSS property name: `faultrate`, DSS property index: 43.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 43)

    @faultrate.setter
    def faultrate(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 43, value)

    @property
    def pctperm(self) -> float:
        """
        Percent of failures that become permanent.

        DSS property name: `pctperm`, DSS property index: 44.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 44)

    @pctperm.setter
    def pctperm(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 44, value)

    @property
    def repair(self) -> float:
        """
        Hours to repair.

        DSS property name: `repair`, DSS property index: 45.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 45)

    @repair.setter
    def repair(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 45, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 46.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 46)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 46, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 47.
        """
        return self._lib.Obj_GetInt32(self._ptr, 47) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 47, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 48.
        """
        self._set_string_o(48, value)

class RegControl(DSSObj):
    __slots__ = []
    _cls_name = 'RegControl'
    _cls_idx = 21
    _cls_prop_idx = {
        'transformer': 1,
        'winding': 2,
        'vreg': 3,
        'band': 4,
        'ptratio': 5,
        'ctprim': 6,
        'r': 7,
        'x': 8,
        'bus': 9,
        'delay': 10,
        'reversible': 11,
        'revvreg': 12,
        'revband': 13,
        'revr': 14,
        'revx': 15,
        'tapdelay': 16,
        'debugtrace': 17,
        'maxtapchange': 18,
        'inversetime': 19,
        'tapwinding': 20,
        'vlimit': 21,
        'ptphase': 22,
        'revthreshold': 23,
        'revdelay': 24,
        'revneutral': 25,
        'eventlog': 26,
        'remoteptratio': 27,
        'tapnum': 28,
        'reset': 29,
        'ldc_z': 30,
        'rev_z': 31,
        'cogen': 32,
        'basefreq': 33,
        'enabled': 34,
        'like': 35,
    }

    # Class-specific enumerations
    class RegControlPhaseSelection(IntEnum):
        """RegControl: Phase Selection (DSS enumeration for RegControl)"""
        min = -3 # min
        max = -2 # max


    @property
    def transformer(self) -> str:
        """
        Name of Transformer or AutoTrans element to which the RegControl is connected. Do not specify the full object name; "Transformer" or "AutoTrans" is assumed for the object class.  Example:

        Transformer=Xfmr1

        DSS property name: `transformer`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @transformer.setter
    def transformer(self, value: Union[AnyStr, Transformer, AutoTrans]):
        if isinstance(value, DSSObj):
            self._set_obj(1, value)
            return

        self._set_string_o(1, value)

    @property
    def transformer_obj(self) -> DSSObj:
        """
        Name of Transformer or AutoTrans element to which the RegControl is connected. Do not specify the full object name; "Transformer" or "AutoTrans" is assumed for the object class.  Example:

        Transformer=Xfmr1

        DSS property name: `transformer`, DSS property index: 1.
        """
        return self._get_obj(1, None)

    @transformer_obj.setter
    def transformer_obj(self, value: Union[Transformer, AutoTrans]):
        self._set_obj(1, value)

    @property
    def winding(self) -> int:
        """
        Number of the winding of the transformer element that the RegControl is monitoring. 1 or 2, typically.  Side Effect: Sets TAPWINDING property to the same winding.

        DSS property name: `winding`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @winding.setter
    def winding(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def vreg(self) -> float:
        """
        Voltage regulator setting, in VOLTS, for the winding being controlled.  Multiplying this value times the ptratio should yield the voltage across the WINDING of the controlled transformer. Default is 120.0

        DSS property name: `vreg`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @vreg.setter
    def vreg(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def band(self) -> float:
        """
        Bandwidth in VOLTS for the controlled bus (see help for ptratio property).  Default is 3.0

        DSS property name: `band`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @band.setter
    def band(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def ptratio(self) -> float:
        """
        Ratio of the PT that converts the controlled winding voltage to the regulator control voltage. Default is 60.  If the winding is Wye, the line-to-neutral voltage is used.  Else, the line-to-line voltage is used. SIDE EFFECT: Also sets RemotePTRatio property.

        DSS property name: `ptratio`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @ptratio.setter
    def ptratio(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def CTprim(self) -> float:
        """
        Rating, in Amperes, of the primary CT rating for which the line amps convert to control rated amps.The typical default secondary ampere rating is 0.2 Amps (check with manufacturer specs). Current at which the LDC voltages match the R and X settings.

        DSS property name: `CTprim`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @CTprim.setter
    def CTprim(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def R(self) -> float:
        """
        R setting on the line drop compensator in the regulator, expressed in VOLTS.

        DSS property name: `R`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @R.setter
    def R(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def X(self) -> float:
        """
        X setting on the line drop compensator in the regulator, expressed in VOLTS.

        DSS property name: `X`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @X.setter
    def X(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def bus(self) -> str:
        """
        Name of a bus (busname.nodename) in the system to use as the controlled bus instead of the bus to which the transformer winding is connected or the R and X line drop compensator settings.  Do not specify this value if you wish to use the line drop compensator settings.  Default is null string. Assumes the base voltage for this bus is the same as the transformer winding base specified above. Note: This bus (1-phase) WILL BE CREATED by the regulator control upon SOLVE if not defined by some other device. You can specify the node of the bus you wish to sample (defaults to 1). If specified, the RegControl is redefined as a 1-phase device since only one voltage is used.

        DSS property name: `bus`, DSS property index: 9.
        """
        return self._get_prop_string(9)

    @bus.setter
    def bus(self, value: AnyStr):
        self._set_string_o(9, value)

    @property
    def delay(self) -> float:
        """
        Time delay, in seconds, from when the voltage goes out of band to when the tap changing begins. This is used to determine which regulator control will act first. Default is 15.  You may specify any floating point number to achieve a model of whatever condition is necessary.

        DSS property name: `delay`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @delay.setter
    def delay(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def reversible(self) -> bool:
        """
        {Yes |No*} Indicates whether or not the regulator can be switched to regulate in the reverse direction. Default is No.Typically applies only to line regulators and not to LTC on a substation transformer.

        DSS property name: `reversible`, DSS property index: 11.
        """
        return self._lib.Obj_GetInt32(self._ptr, 11) != 0

    @reversible.setter
    def reversible(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 11, value)

    @property
    def revvreg(self) -> float:
        """
        Voltage setting in volts for operation in the reverse direction.

        DSS property name: `revvreg`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @revvreg.setter
    def revvreg(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def revband(self) -> float:
        """
        Bandwidth for operating in the reverse direction.

        DSS property name: `revband`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @revband.setter
    def revband(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def revR(self) -> float:
        """
        R line drop compensator setting for reverse direction.

        DSS property name: `revR`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @revR.setter
    def revR(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def revX(self) -> float:
        """
        X line drop compensator setting for reverse direction.

        DSS property name: `revX`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @revX.setter
    def revX(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def tapdelay(self) -> float:
        """
        Delay in sec between tap changes. Default is 2. This is how long it takes between changes after the first change.

        DSS property name: `tapdelay`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @tapdelay.setter
    def tapdelay(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def debugtrace(self) -> bool:
        """
        {Yes | No* }  Default is no.  Turn this on to capture the progress of the regulator model for each control iteration.  Creates a separate file for each RegControl named "REG_name.csv".

        DSS property name: `debugtrace`, DSS property index: 17.
        """
        return self._lib.Obj_GetInt32(self._ptr, 17) != 0

    @debugtrace.setter
    def debugtrace(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 17, value)

    @property
    def maxtapchange(self) -> int:
        """
        Maximum allowable tap change per control iteration in STATIC control mode.  Default is 16. 

        Set this to 1 to better approximate actual control action. 

        Set this to 0 to fix the tap in the current position.

        DSS property name: `maxtapchange`, DSS property index: 18.
        """
        return self._lib.Obj_GetInt32(self._ptr, 18)

    @maxtapchange.setter
    def maxtapchange(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 18, value)

    @property
    def inversetime(self) -> bool:
        """
        {Yes | No* } Default is no.  The time delay is adjusted inversely proportional to the amount the voltage is outside the band down to 10%.

        DSS property name: `inversetime`, DSS property index: 19.
        """
        return self._lib.Obj_GetInt32(self._ptr, 19) != 0

    @inversetime.setter
    def inversetime(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 19, value)

    @property
    def tapwinding(self) -> int:
        """
        Winding containing the actual taps, if different than the WINDING property. Defaults to the same winding as specified by the WINDING property.

        DSS property name: `tapwinding`, DSS property index: 20.
        """
        return self._lib.Obj_GetInt32(self._ptr, 20)

    @tapwinding.setter
    def tapwinding(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 20, value)

    @property
    def vlimit(self) -> float:
        """
        Voltage Limit for bus to which regulated winding is connected (e.g. first customer). Default is 0.0. Set to a value greater then zero to activate this function.

        DSS property name: `vlimit`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @vlimit.setter
    def vlimit(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def PTphase(self) -> Union[RegControlPhaseSelection, int]:
        """
        For multi-phase transformers, the number of the phase being monitored or one of { MAX | MIN} for all phases. Default=1. Must be less than or equal to the number of phases. Ignored for regulated bus.

        DSS property name: `PTphase`, DSS property index: 22.
        """
        value = self._lib.Obj_GetInt32(self._ptr, 22)
        if value > 0:
            return value
    
        return RegControl.RegControlPhaseSelection(value)

    @PTphase.setter
    def PTphase(self, value: Union[AnyStr, int, RegControlPhaseSelection]):
        if not isinstance(value, int):
            self._set_string_o(22, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 22, value)

    @property
    def PTphase_str(self) -> str:
        """
        For multi-phase transformers, the number of the phase being monitored or one of { MAX | MIN} for all phases. Default=1. Must be less than or equal to the number of phases. Ignored for regulated bus.

        DSS property name: `PTphase`, DSS property index: 22.
        """
        return self._get_prop_string(22)

    @PTphase_str.setter
    def PTphase_str(self, value: AnyStr):
        self.PTphase = value

    @property
    def revThreshold(self) -> float:
        """
        kW reverse power threshold for reversing the direction of the regulator. Default is 100.0 kw.

        DSS property name: `revThreshold`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @revThreshold.setter
    def revThreshold(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def revDelay(self) -> float:
        """
        Time Delay in seconds (s) for executing the reversing action once the threshold for reversing has been exceeded. Default is 60 s.

        DSS property name: `revDelay`, DSS property index: 24.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 24)

    @revDelay.setter
    def revDelay(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 24, value)

    @property
    def revNeutral(self) -> bool:
        """
        {Yes | No*} Default is no. Set this to Yes if you want the regulator to go to neutral in the reverse direction or in cogen operation.

        DSS property name: `revNeutral`, DSS property index: 25.
        """
        return self._lib.Obj_GetInt32(self._ptr, 25) != 0

    @revNeutral.setter
    def revNeutral(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 25, value)

    @property
    def EventLog(self) -> bool:
        """
        {Yes/True* | No/False} Default is YES for regulator control. Log control actions to Eventlog.

        DSS property name: `EventLog`, DSS property index: 26.
        """
        return self._lib.Obj_GetInt32(self._ptr, 26) != 0

    @EventLog.setter
    def EventLog(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 26, value)

    @property
    def RemotePTRatio(self) -> float:
        """
        When regulating a bus (the Bus= property is set), the PT ratio required to convert actual voltage at the remote bus to control voltage. Is initialized to PTratio property. Set this property after setting PTratio.

        DSS property name: `RemotePTRatio`, DSS property index: 27.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 27)

    @RemotePTRatio.setter
    def RemotePTRatio(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 27, value)

    @property
    def TapNum(self) -> int:
        """
        An integer number indicating the tap position that the controlled transformer winding tap position is currently at, or is being set to.  If being set, and the value is outside the range of the transformer min or max tap, then set to the min or max tap position as appropriate. Default is 0

        DSS property name: `TapNum`, DSS property index: 28.
        """
        return self._lib.Obj_GetInt32(self._ptr, 28)

    @TapNum.setter
    def TapNum(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 28, value)

    def Reset(self, value: bool):
        """
        {Yes | No} If Yes, forces Reset of this RegControl.

        DSS property name: `Reset`, DSS property index: 29.
        """
        self._lib.Obj_SetInt32(self._ptr, 29, value)

    @property
    def LDC_Z(self) -> float:
        """
        Z value for Beckwith LDC_Z control option. Volts adjustment at rated control current.

        DSS property name: `LDC_Z`, DSS property index: 30.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 30)

    @LDC_Z.setter
    def LDC_Z(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 30, value)

    @property
    def rev_Z(self) -> float:
        """
        Reverse Z value for Beckwith LDC_Z control option.

        DSS property name: `rev_Z`, DSS property index: 31.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 31)

    @rev_Z.setter
    def rev_Z(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 31, value)

    @property
    def Cogen(self) -> bool:
        """
        {Yes|No*} Default is No. The Cogen feature is activated. Continues looking forward if power reverses, but switches to reverse-mode LDC, vreg and band values.

        DSS property name: `Cogen`, DSS property index: 32.
        """
        return self._lib.Obj_GetInt32(self._ptr, 32) != 0

    @Cogen.setter
    def Cogen(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 32, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 33.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 33)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 33, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 34.
        """
        return self._lib.Obj_GetInt32(self._ptr, 34) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 34, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 35.
        """
        self._set_string_o(35, value)

class InvControl(DSSObj):
    __slots__ = []
    _cls_name = 'InvControl'
    _cls_idx = 42
    _cls_prop_idx = {
        'derlist': 1,
        'mode': 2,
        'combimode': 3,
        'vvc_curve1': 4,
        'hysteresis_offset': 5,
        'voltage_curvex_ref': 6,
        'avgwindowlen': 7,
        'voltwatt_curve': 8,
        'dbvmin': 9,
        'dbvmax': 10,
        'argralowv': 11,
        'argrahiv': 12,
        'dynreacavgwindowlen': 13,
        'deltaq_factor': 14,
        'voltagechangetolerance': 15,
        'varchangetolerance': 16,
        'voltwattyaxis': 17,
        'rateofchangemode': 18,
        'lpftau': 19,
        'risefalllimit': 20,
        'deltap_factor': 21,
        'eventlog': 22,
        'refreactivepower': 23,
        'activepchangetolerance': 24,
        'monvoltagecalc': 25,
        'monbus': 26,
        'monbusesvbase': 27,
        'voltwattch_curve': 28,
        'wattpf_curve': 29,
        'wattvar_curve': 30,
        'vv_refreactivepower': 31,
        'pvsystemlist': 32,
        'vsetpoint': 33,
        'controlmodel': 34,
        'basefreq': 35,
        'enabled': 36,
        'like': 37,
    }

    # Class-specific enumerations
    class InvControlControlMode(IntEnum):
        """InvControl: Control Mode (DSS enumeration for InvControl)"""
        Voltvar = 1 # Voltvar
        VoltWatt = 2 # VoltWatt
        DynamicReaccurr = 3 # DynamicReaccurr
        WattPF = 4 # WattPF
        Wattvar = 5 # Wattvar
        AVR = 6 # AVR
        GFM = 7 # GFM

    class InvControlCombiMode(IntEnum):
        """InvControl: Combi Mode (DSS enumeration for InvControl)"""
        VV_VW = 1 # VV_VW
        VV_DRC = 2 # VV_DRC

    class InvControlVoltageCurveXRef(IntEnum):
        """InvControl: Voltage Curve X Ref (DSS enumeration for InvControl)"""
        Rated = 0 # Rated
        Avg = 1 # Avg
        RAvg = 2 # RAvg

    class InvControlVoltWattYAxis(IntEnum):
        """InvControl: Volt-watt Y-Axis (DSS enumeration for InvControl)"""
        PAvailablePU = 0 # PAvailablePU
        PMPPPU = 1 # PMPPPU
        PctPMPPPU = 2 # PctPMPPPU
        KVARatingPU = 3 # KVARatingPU

    class InvControlRateOfChangeMode(IntEnum):
        """InvControl: Rate-of-change Mode (DSS enumeration for InvControl)"""
        Inactive = 0 # Inactive
        LPF = 1 # LPF
        RiseFall = 2 # RiseFall

    class InvControlReactivePowerReference(IntEnum):
        """InvControl: Reactive Power Reference (DSS enumeration for InvControl)"""
        VARAVAL = 0 # VARAVAL
        VARMAX = 1 # VARMAX

    class InvControlControlModel(IntEnum):
        """InvControl: Control Model (DSS enumeration for InvControl)"""
        Linear = 0 # Linear
        Exponential = 1 # Exponential


    @property
    def DERList(self) -> List[str]:
        """
        Array list of PVSystem and/or Storage elements to be controlled. If not specified, all PVSystem and Storage in the circuit are assumed to be controlled by this control. 

        No capability of hierarchical control between two controls for a single element is implemented at this time.

        DSS property name: `DERList`, DSS property index: 1.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 1)

    @DERList.setter
    def DERList(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 1, value_ptr, value_count)
        self._check_for_error()

    @property
    def Mode(self) -> InvControlControlMode:
        """
        Smart inverter function in which the InvControl will control the PC elements specified in DERList, according to the options below:

        Must be one of: {VOLTVAR* | VOLTWATT | DYNAMICREACCURR | WATTPF | WATTVAR | GFM} 
        if the user desires to use modes simultaneously, then set the CombiMode property. Setting the Mode to any valid value disables combination mode.

        In volt-var mode (Default). This mode attempts to CONTROL the vars, according to one or two volt-var curves, depending on the monitored voltages, present active power output, and the capabilities of the PVSystem/Storage. 

        In volt-watt mode. This mode attempts to LIMIT the watts, according to one defined volt-watt curve, depending on the monitored voltages and the capabilities of the PVSystem/Storage. 

        In dynamic reactive current mode. This mode attempts to increasingly counter deviations by CONTROLLING vars, depending on the monitored voltages, present active power output, and the capabilities of the of the PVSystem/Storage.

        In watt-pf mode. This mode attempts to CONTROL the vars, according to a watt-pf curve, depending on the present active power output, and the capabilities of the PVSystem/Storage. 

        In watt-var mode. This mode attempts to CONTROL the vars, according to a watt-var curve, depending on the present active power output, and the capabilities of the PVSystem/Storage. 

        In GFM mode this control will trigger the GFM control routine for the DERs within the DERList. The GFM actiosn will only take place if the pointed DERs are in GFM mode. The controller parameters are locally setup at the DER.

        DSS property name: `Mode`, DSS property index: 2.
        """
        return InvControl.InvControlControlMode(self._lib.Obj_GetInt32(self._ptr, 2))

    @Mode.setter
    def Mode(self, value: Union[AnyStr, int, InvControlControlMode]):
        if not isinstance(value, int):
            self._set_string_o(2, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def Mode_str(self) -> str:
        """
        Smart inverter function in which the InvControl will control the PC elements specified in DERList, according to the options below:

        Must be one of: {VOLTVAR* | VOLTWATT | DYNAMICREACCURR | WATTPF | WATTVAR | GFM} 
        if the user desires to use modes simultaneously, then set the CombiMode property. Setting the Mode to any valid value disables combination mode.

        In volt-var mode (Default). This mode attempts to CONTROL the vars, according to one or two volt-var curves, depending on the monitored voltages, present active power output, and the capabilities of the PVSystem/Storage. 

        In volt-watt mode. This mode attempts to LIMIT the watts, according to one defined volt-watt curve, depending on the monitored voltages and the capabilities of the PVSystem/Storage. 

        In dynamic reactive current mode. This mode attempts to increasingly counter deviations by CONTROLLING vars, depending on the monitored voltages, present active power output, and the capabilities of the of the PVSystem/Storage.

        In watt-pf mode. This mode attempts to CONTROL the vars, according to a watt-pf curve, depending on the present active power output, and the capabilities of the PVSystem/Storage. 

        In watt-var mode. This mode attempts to CONTROL the vars, according to a watt-var curve, depending on the present active power output, and the capabilities of the PVSystem/Storage. 

        In GFM mode this control will trigger the GFM control routine for the DERs within the DERList. The GFM actiosn will only take place if the pointed DERs are in GFM mode. The controller parameters are locally setup at the DER.

        DSS property name: `Mode`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @Mode_str.setter
    def Mode_str(self, value: AnyStr):
        self.Mode = value

    @property
    def CombiMode(self) -> InvControlCombiMode:
        """
        Combination of smart inverter functions in which the InvControl will control the PC elements in DERList, according to the options below: 

        Must be a combination of the following: {VV_VW | VV_DRC}. Default is to not set this property, in which case the single control mode in Mode is active.  

        In combined VV_VW mode, both volt-var and volt-watt control modes are active simultaneously.  See help individually for volt-var mode and volt-watt mode in Mode property.
        Note that the PVSystem/Storage will attempt to achieve both the volt-watt and volt-var set-points based on the capabilities of the inverter in the PVSystem/Storage (kVA rating, etc), any limits set on maximum active power,

        In combined VV_DRC, both the volt-var and the dynamic reactive current modes are simultaneously active.

        DSS property name: `CombiMode`, DSS property index: 3.
        """
        return InvControl.InvControlCombiMode(self._lib.Obj_GetInt32(self._ptr, 3))

    @CombiMode.setter
    def CombiMode(self, value: Union[AnyStr, int, InvControlCombiMode]):
        if not isinstance(value, int):
            self._set_string_o(3, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 3, value)

    @property
    def CombiMode_str(self) -> str:
        """
        Combination of smart inverter functions in which the InvControl will control the PC elements in DERList, according to the options below: 

        Must be a combination of the following: {VV_VW | VV_DRC}. Default is to not set this property, in which case the single control mode in Mode is active.  

        In combined VV_VW mode, both volt-var and volt-watt control modes are active simultaneously.  See help individually for volt-var mode and volt-watt mode in Mode property.
        Note that the PVSystem/Storage will attempt to achieve both the volt-watt and volt-var set-points based on the capabilities of the inverter in the PVSystem/Storage (kVA rating, etc), any limits set on maximum active power,

        In combined VV_DRC, both the volt-var and the dynamic reactive current modes are simultaneously active.

        DSS property name: `CombiMode`, DSS property index: 3.
        """
        return self._get_prop_string(3)

    @CombiMode_str.setter
    def CombiMode_str(self, value: AnyStr):
        self.CombiMode = value

    @property
    def vvc_curve1(self) -> str:
        """
        Required for VOLTVAR mode. 

        Name of the XYCurve object containing the volt-var curve. The positive values of the y-axis of the volt-var curve represent values in pu of the provided base reactive power. The negative values of the y-axis are values in pu of the absorbed base reactive power. 
        Provided and absorbed base reactive power values are defined in the RefReactivePower property

        Units for the x-axis are per-unit voltage, which may be in per unit of the rated voltage for the PVSystem/Storage, or may be in per unit of the average voltage at the terminals over a user-defined number of prior solutions. 

        DSS property name: `vvc_curve1`, DSS property index: 4.
        """
        return self._get_prop_string(4)

    @vvc_curve1.setter
    def vvc_curve1(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(4, value)
            return

        self._set_string_o(4, value)

    @property
    def vvc_curve1_obj(self) -> XYcurve:
        """
        Required for VOLTVAR mode. 

        Name of the XYCurve object containing the volt-var curve. The positive values of the y-axis of the volt-var curve represent values in pu of the provided base reactive power. The negative values of the y-axis are values in pu of the absorbed base reactive power. 
        Provided and absorbed base reactive power values are defined in the RefReactivePower property

        Units for the x-axis are per-unit voltage, which may be in per unit of the rated voltage for the PVSystem/Storage, or may be in per unit of the average voltage at the terminals over a user-defined number of prior solutions. 

        DSS property name: `vvc_curve1`, DSS property index: 4.
        """
        return self._get_obj(4, XYcurve)

    @vvc_curve1_obj.setter
    def vvc_curve1_obj(self, value: XYcurve):
        self._set_obj(4, value)

    @property
    def hysteresis_offset(self) -> float:
        """
        Required for VOLTVAR mode, and defaults to 0. 

        for the times when the terminal voltage is decreasing, this is the off-set in per-unit voltage of a curve whose shape is the same as vvc_curve. It is offset by a certain negative value of per-unit voltage, which is defined by the base quantity for the x-axis of the volt-var curve (see help for voltage_curvex_ref)

        if the PVSystem/Storage terminal voltage has been increasing, and has not changed directions, utilize vvc_curve1 for the volt-var response. 

        if the PVSystem/Storage terminal voltage has been increasing and changes directions and begins to decrease, then move from utilizing vvc_curve1 to a volt-var curve of the same shape, but offset by a certain per-unit voltage value. 

        Maintain the same per-unit available var output level (unless head-room has changed due to change in active power or kva rating of PVSystem/Storage).  Per-unit var values remain the same for this internally constructed second curve (hysteresis curve). 

        if the terminal voltage has been decreasing and changes directions and begins to increase , then move from utilizing the offset curve, back to the vvc_curve1 for volt-var response, but stay at the same per-unit available vars output level.

        DSS property name: `hysteresis_offset`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @hysteresis_offset.setter
    def hysteresis_offset(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def voltage_curvex_ref(self) -> InvControlVoltageCurveXRef:
        """
        Required for VOLTVAR and VOLTWATT modes, and defaults to rated.  Possible values are: {rated|avg|ravg}.  

        Defines whether the x-axis values (voltage in per unit) for vvc_curve1 and the volt-watt curve corresponds to:

        rated. The rated voltage for the PVSystem/Storage object (1.0 in the volt-var curve equals rated voltage).

        avg. The average terminal voltage recorded over a certain number of prior power-flow solutions.
        with the avg setting, 1.0 per unit on the x-axis of the volt-var curve(s) corresponds to the average voltage.
        from a certain number of prior intervals.  See avgwindowlen parameter.

        ravg. Same as avg, with the exception that the avgerage terminal voltage is divided by the rated voltage.

        DSS property name: `voltage_curvex_ref`, DSS property index: 6.
        """
        return InvControl.InvControlVoltageCurveXRef(self._lib.Obj_GetInt32(self._ptr, 6))

    @voltage_curvex_ref.setter
    def voltage_curvex_ref(self, value: Union[AnyStr, int, InvControlVoltageCurveXRef]):
        if not isinstance(value, int):
            self._set_string_o(6, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def voltage_curvex_ref_str(self) -> str:
        """
        Required for VOLTVAR and VOLTWATT modes, and defaults to rated.  Possible values are: {rated|avg|ravg}.  

        Defines whether the x-axis values (voltage in per unit) for vvc_curve1 and the volt-watt curve corresponds to:

        rated. The rated voltage for the PVSystem/Storage object (1.0 in the volt-var curve equals rated voltage).

        avg. The average terminal voltage recorded over a certain number of prior power-flow solutions.
        with the avg setting, 1.0 per unit on the x-axis of the volt-var curve(s) corresponds to the average voltage.
        from a certain number of prior intervals.  See avgwindowlen parameter.

        ravg. Same as avg, with the exception that the avgerage terminal voltage is divided by the rated voltage.

        DSS property name: `voltage_curvex_ref`, DSS property index: 6.
        """
        return self._get_prop_string(6)

    @voltage_curvex_ref_str.setter
    def voltage_curvex_ref_str(self, value: AnyStr):
        self.voltage_curvex_ref = value

    @property
    def avgwindowlen(self) -> int:
        """
        Required for VOLTVAR mode and VOLTWATT mode, and defaults to 0 seconds (0s). 

        Sets the length of the averaging window over which the average PVSystem/Storage terminal voltage is calculated. 

        Units are indicated by appending s, m, or h to the integer value. 

        The averaging window will calculate the average PVSystem/Storage terminal voltage over the specified period of time, up to and including the last power flow solution. 

        Note, if the solution stepsize is larger than the window length, then the voltage will be assumed to have been constant over the time-frame specified by the window length.

        DSS property name: `avgwindowlen`, DSS property index: 7.
        """
        return self._lib.Obj_GetInt32(self._ptr, 7)

    @avgwindowlen.setter
    def avgwindowlen(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 7, value)

    @property
    def voltwatt_curve(self) -> str:
        """
        Required for VOLTWATT mode. 

        Name of the XYCurve object containing the volt-watt curve. 

        Units for the x-axis are per-unit voltage, which may be in per unit of the rated voltage for the PVSystem/Storage, or may be in per unit of the average voltage at the terminals over a user-defined number of prior solutions. 

        Units for the y-axis are either in one of the options described in the VoltwattYAxis property. 

        DSS property name: `voltwatt_curve`, DSS property index: 8.
        """
        return self._get_prop_string(8)

    @voltwatt_curve.setter
    def voltwatt_curve(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(8, value)
            return

        self._set_string_o(8, value)

    @property
    def voltwatt_curve_obj(self) -> XYcurve:
        """
        Required for VOLTWATT mode. 

        Name of the XYCurve object containing the volt-watt curve. 

        Units for the x-axis are per-unit voltage, which may be in per unit of the rated voltage for the PVSystem/Storage, or may be in per unit of the average voltage at the terminals over a user-defined number of prior solutions. 

        Units for the y-axis are either in one of the options described in the VoltwattYAxis property. 

        DSS property name: `voltwatt_curve`, DSS property index: 8.
        """
        return self._get_obj(8, XYcurve)

    @voltwatt_curve_obj.setter
    def voltwatt_curve_obj(self, value: XYcurve):
        self._set_obj(8, value)

    @property
    def DbVMin(self) -> float:
        """
        Required for the dynamic reactive current mode (DYNAMICREACCURR), and defaults to 0.95 per-unit voltage (referenced to the PVSystem/Storage object rated voltage or a windowed average value). 

        This parameter is the minimum voltage that defines the voltage dead-band within which no reactive power is allowed to be generated. 

        DSS property name: `DbVMin`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @DbVMin.setter
    def DbVMin(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def DbVMax(self) -> float:
        """
        Required for the dynamic reactive current mode (DYNAMICREACCURR), and defaults to 1.05 per-unit voltage (referenced to the PVSystem object rated voltage or a windowed average value). 

        This parameter is the maximum voltage that defines the voltage dead-band within which no reactive power is allowed to be generated. 

        DSS property name: `DbVMax`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @DbVMax.setter
    def DbVMax(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def ArGraLowV(self) -> float:
        """
        Required for the dynamic reactive current mode (DYNAMICREACCURR), and defaults to 0.1  

        This is a gradient, expressed in unit-less terms of %/%, to establish the ratio by which percentage capacitive reactive power production is increased as the  percent delta-voltage decreases below DbVMin. 

        Percent delta-voltage is defined as the present PVSystem/Storage terminal voltage minus the moving average voltage, expressed as a percentage of the rated voltage for the PVSystem/Storage object. 

        Note, the moving average voltage for the dynamic reactive current mode is different than the moving average voltage for the volt-watt and volt-var modes.

        DSS property name: `ArGraLowV`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @ArGraLowV.setter
    def ArGraLowV(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def ArGraHiV(self) -> float:
        """
        Required for the dynamic reactive current mode (DYNAMICREACCURR), and defaults to 0.1  

        This is a gradient, expressed in unit-less terms of %/%, to establish the ratio by which percentage inductive reactive power production is increased as the  percent delta-voltage decreases above DbVMax. 

        Percent delta-voltage is defined as the present PVSystem/Storage terminal voltage minus the moving average voltage, expressed as a percentage of the rated voltage for the PVSystem/Storage object. 

        Note, the moving average voltage for the dynamic reactive current mode is different than the mmoving average voltage for the volt-watt and volt-var modes.

        DSS property name: `ArGraHiV`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @ArGraHiV.setter
    def ArGraHiV(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def DynReacavgwindowlen(self) -> int:
        """
        Required for the dynamic reactive current mode (DYNAMICREACCURR), and defaults to 1 seconds (1s). do not use a value smaller than 1.0 

        Sets the length of the averaging window over which the average PVSystem/Storage terminal voltage is calculated for the dynamic reactive current mode. 

        Units are indicated by appending s, m, or h to the integer value. 

        Typically this will be a shorter averaging window than the volt-var and volt-watt averaging window.

        The averaging window will calculate the average PVSystem/Storage terminal voltage over the specified period of time, up to and including the last power flow solution.  Note, if the solution stepsize is larger than the window length, then the voltage will be assumed to have been constant over the time-frame specified by the window length.

        DSS property name: `DynReacavgwindowlen`, DSS property index: 13.
        """
        return self._lib.Obj_GetInt32(self._ptr, 13)

    @DynReacavgwindowlen.setter
    def DynReacavgwindowlen(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 13, value)

    @property
    def deltaQ_Factor(self) -> float:
        """
        Required for the VOLTVAR and DYNAMICREACCURR modes.  Defaults to -1.0. 

        Defining -1.0, OpenDSS takes care internally of delta_Q itself. It tries to improve convergence as well as speed up process

        Sets the maximum change (in per unit) from the prior var output level to the desired var output level during each control iteration. 


        if numerical instability is noticed in solutions such as var sign changing from one control iteration to the next and voltages oscillating between two values with some separation, this is an indication of numerical instability (use the EventLog to diagnose). 

        if the maximum control iterations are exceeded, and no numerical instability is seen in the EventLog of via monitors, then try increasing the value of this parameter to reduce the number of control iterations needed to achieve the control criteria, and move to the power flow solution. 

        When operating the controller using expoenential control model (see CtrlModel), this parameter represents the sampling time gain of the controller, which is used for accelrating the controller response in terms of control iterations required.

        DSS property name: `deltaQ_Factor`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @deltaQ_Factor.setter
    def deltaQ_Factor(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def VoltageChangeTolerance(self) -> float:
        """
        Defaults to 0.0001 per-unit voltage.  This parameter should only be modified by advanced users of the InvControl.  

        Tolerance in pu of the control loop convergence associated to the monitored voltage in pu. This value is compared with the difference of the monitored voltage in pu of the current and previous control iterations of the control loop

        This voltage tolerance value plus the var/watt tolerance value (VarChangeTolerance/ActivePChangeTolerance) determine, together, when to stop control iterations by the InvControl. 

        If an InvControl is controlling more than one PVSystem/Storage, each PVSystem/Storage has this quantity calculated independently, and so an individual PVSystem/Storage may reach the tolerance within different numbers of control iterations.

        DSS property name: `VoltageChangeTolerance`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @VoltageChangeTolerance.setter
    def VoltageChangeTolerance(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def VarChangeTolerance(self) -> float:
        """
        Required for VOLTVAR and DYNAMICREACCURR modes.  Defaults to 0.025 per unit of the base provided or absorbed reactive power described in the RefReactivePower property This parameter should only be modified by advanced users of the InvControl. 

        Tolerance in pu of the convergence of the control loop associated with reactive power. For the same control iteration, this value is compared to the difference, as an absolute value (without sign), between the desired reactive power value in pu and the output reactive power in pu of the controlled element.

        This reactive power tolerance value plus the voltage tolerance value (VoltageChangeTolerance) determine, together, when to stop control iterations by the InvControl.  

        If an InvControl is controlling more than one PVSystem/Storage, each PVSystem/Storage has this quantity calculated independently, and so an individual PVSystem/Storage may reach the tolerance within different numbers of control iterations.

        DSS property name: `VarChangeTolerance`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @VarChangeTolerance.setter
    def VarChangeTolerance(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def VoltwattYAxis(self) -> InvControlVoltWattYAxis:
        """
        Required for VOLTWATT mode.  Must be one of: {PMPPPU* | PAVAILABLEPU| PCTPMPPPU | KVARATINGPU}.  The default is PMPPPU.  

        Units for the y-axis of the volt-watt curve while in volt-watt mode. 

        When set to PMPPPU. The y-axis corresponds to the value in pu of Pmpp property of the PVSystem. 

        When set to PAVAILABLEPU. The y-axis corresponds to the value in pu of the available active power of the PVSystem. 

        When set to PCTPMPPPU. The y-axis corresponds to the value in pu of the power Pmpp multiplied by 1/100 of the %Pmpp property of the PVSystem.

        When set to KVARATINGPU. The y-axis corresponds to the value in pu of the kVA property of the PVSystem.

        DSS property name: `VoltwattYAxis`, DSS property index: 17.
        """
        return InvControl.InvControlVoltWattYAxis(self._lib.Obj_GetInt32(self._ptr, 17))

    @VoltwattYAxis.setter
    def VoltwattYAxis(self, value: Union[AnyStr, int, InvControlVoltWattYAxis]):
        if not isinstance(value, int):
            self._set_string_o(17, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 17, value)

    @property
    def VoltwattYAxis_str(self) -> str:
        """
        Required for VOLTWATT mode.  Must be one of: {PMPPPU* | PAVAILABLEPU| PCTPMPPPU | KVARATINGPU}.  The default is PMPPPU.  

        Units for the y-axis of the volt-watt curve while in volt-watt mode. 

        When set to PMPPPU. The y-axis corresponds to the value in pu of Pmpp property of the PVSystem. 

        When set to PAVAILABLEPU. The y-axis corresponds to the value in pu of the available active power of the PVSystem. 

        When set to PCTPMPPPU. The y-axis corresponds to the value in pu of the power Pmpp multiplied by 1/100 of the %Pmpp property of the PVSystem.

        When set to KVARATINGPU. The y-axis corresponds to the value in pu of the kVA property of the PVSystem.

        DSS property name: `VoltwattYAxis`, DSS property index: 17.
        """
        return self._get_prop_string(17)

    @VoltwattYAxis_str.setter
    def VoltwattYAxis_str(self, value: AnyStr):
        self.VoltwattYAxis = value

    @property
    def RateofChangeMode(self) -> InvControlRateOfChangeMode:
        """
        Required for VOLTWATT and VOLTVAR mode.  Must be one of: {INACTIVE* | LPF | RISEFALL }.  The default is INACTIVE.  

        Auxiliary option that aims to limit the changes of the desired reactive power and the active power limit between time steps, the alternatives are listed below: 

        INACTIVE. It indicates there is no limit on rate of change imposed for either active or reactive power output. 

        LPF. A low-pass RC filter is applied to the desired reactive power and/or the active power limit to determine the output power as a function of a time constant defined in the LPFTau property. 

        RISEFALL. A rise and fall limit in the change of active and/or reactive power expressed in terms of pu power per second, defined in the RiseFallLimit, is applied to the desired reactive power and/or the active power limit. 

        DSS property name: `RateofChangeMode`, DSS property index: 18.
        """
        return InvControl.InvControlRateOfChangeMode(self._lib.Obj_GetInt32(self._ptr, 18))

    @RateofChangeMode.setter
    def RateofChangeMode(self, value: Union[AnyStr, int, InvControlRateOfChangeMode]):
        if not isinstance(value, int):
            self._set_string_o(18, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 18, value)

    @property
    def RateofChangeMode_str(self) -> str:
        """
        Required for VOLTWATT and VOLTVAR mode.  Must be one of: {INACTIVE* | LPF | RISEFALL }.  The default is INACTIVE.  

        Auxiliary option that aims to limit the changes of the desired reactive power and the active power limit between time steps, the alternatives are listed below: 

        INACTIVE. It indicates there is no limit on rate of change imposed for either active or reactive power output. 

        LPF. A low-pass RC filter is applied to the desired reactive power and/or the active power limit to determine the output power as a function of a time constant defined in the LPFTau property. 

        RISEFALL. A rise and fall limit in the change of active and/or reactive power expressed in terms of pu power per second, defined in the RiseFallLimit, is applied to the desired reactive power and/or the active power limit. 

        DSS property name: `RateofChangeMode`, DSS property index: 18.
        """
        return self._get_prop_string(18)

    @RateofChangeMode_str.setter
    def RateofChangeMode_str(self, value: AnyStr):
        self.RateofChangeMode = value

    @property
    def LPFTau(self) -> float:
        """
        Not required. Defaults to 0 seconds. 

        Filter time constant of the LPF option of the RateofChangeMode property. The time constant will cause the low-pass filter to achieve 95% of the target value in 3 time constants.

        DSS property name: `LPFTau`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @LPFTau.setter
    def LPFTau(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def RiseFallLimit(self) -> float:
        """
        Not required.  Defaults to no limit (-1). Must be -1 (no limit) or a positive value.  

        Limit in power in pu per second used by the RISEFALL option of the RateofChangeMode property.The base value for this ramp is defined in the RefReactivePower property and/or in VoltwattYAxis.

        DSS property name: `RiseFallLimit`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @RiseFallLimit.setter
    def RiseFallLimit(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    @property
    def deltaP_Factor(self) -> float:
        """
        Required for the VOLTWATT modes.  Defaults to -1.0. 

        Defining -1.0, OpenDSS takes care internally of delta_P itself. It tries to improve convergence as well as speed up process

        Defining between 0.05 and 1.0, it sets the maximum change (in unit of the y-axis) from the prior active power output level to the desired active power output level during each control iteration. 


        If numerical instability is noticed in solutions such as active power changing substantially from one control iteration to the next and/or voltages oscillating between two values with some separation, this is an indication of numerical instability (use the EventLog to diagnose). 

        If the maximum control iterations are exceeded, and no numerical instability is seen in the EventLog of via monitors, then try increasing the value of this parameter to reduce the number of control iterations needed to achieve the control criteria, and move to the power flow solution.

        DSS property name: `deltaP_Factor`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @deltaP_Factor.setter
    def deltaP_Factor(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def EventLog(self) -> bool:
        """
        {Yes/True | No/False*} Default is NO for InvControl. Log control actions to Eventlog.

        DSS property name: `EventLog`, DSS property index: 22.
        """
        return self._lib.Obj_GetInt32(self._ptr, 22) != 0

    @EventLog.setter
    def EventLog(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 22, value)

    @property
    def RefReactivePower(self) -> InvControlReactivePowerReference:
        """
        Required for any mode that has VOLTVAR, DYNAMICREACCURR and WATTVAR. Defaults to VARAVAL.

        Defines the base reactive power for both the provided and absorbed reactive power, according to one of the following options: 

        VARAVAL. The base values for the provided and absorbed reactive power are equal to the available reactive power.

        VARMAX: The base values of the provided and absorbed reactive power are equal to the value defined in the kvarMax and kvarMaxAbs properties, respectively.

        DSS property name: `RefReactivePower`, DSS property index: 23.
        """
        return InvControl.InvControlReactivePowerReference(self._lib.Obj_GetInt32(self._ptr, 23))

    @RefReactivePower.setter
    def RefReactivePower(self, value: Union[AnyStr, int, InvControlReactivePowerReference]):
        if not isinstance(value, int):
            self._set_string_o(23, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 23, value)

    @property
    def RefReactivePower_str(self) -> str:
        """
        Required for any mode that has VOLTVAR, DYNAMICREACCURR and WATTVAR. Defaults to VARAVAL.

        Defines the base reactive power for both the provided and absorbed reactive power, according to one of the following options: 

        VARAVAL. The base values for the provided and absorbed reactive power are equal to the available reactive power.

        VARMAX: The base values of the provided and absorbed reactive power are equal to the value defined in the kvarMax and kvarMaxAbs properties, respectively.

        DSS property name: `RefReactivePower`, DSS property index: 23.
        """
        return self._get_prop_string(23)

    @RefReactivePower_str.setter
    def RefReactivePower_str(self, value: AnyStr):
        self.RefReactivePower = value

    @property
    def ActivePChangeTolerance(self) -> float:
        """
        Required for VOLTWATT. Default is 0.01

        Tolerance in pu of the convergence of the control loop associated with active power. For the same control iteration, this value is compared to the difference between the active power limit in pu resulted from the convergence process and the one resulted from the volt-watt function.

        This reactive power tolerance value plus the voltage tolerance value (VoltageChangeTolerance) determine, together, when to stop control iterations by the InvControl.  

        If an InvControl is controlling more than one PVSystem/Storage, each PVSystem/Storage has this quantity calculated independently, and so an individual PVSystem/Storage may reach the tolerance within different numbers of control iterations.

        DSS property name: `ActivePChangeTolerance`, DSS property index: 24.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 24)

    @ActivePChangeTolerance.setter
    def ActivePChangeTolerance(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 24, value)

    @property
    def monVoltageCalc(self) -> Union[MonitoredPhase, int]:
        """
        Number of the phase being monitored or one of {AVG | MAX | MIN} for all phases. Default=AVG. 

        DSS property name: `monVoltageCalc`, DSS property index: 25.
        """
        value = self._lib.Obj_GetInt32(self._ptr, 25)
        if value > 0:
            return value
    
        return MonitoredPhase(value)

    @monVoltageCalc.setter
    def monVoltageCalc(self, value: Union[AnyStr, int, MonitoredPhase]):
        if not isinstance(value, int):
            self._set_string_o(25, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 25, value)

    @property
    def monVoltageCalc_str(self) -> str:
        """
        Number of the phase being monitored or one of {AVG | MAX | MIN} for all phases. Default=AVG. 

        DSS property name: `monVoltageCalc`, DSS property index: 25.
        """
        return self._get_prop_string(25)

    @monVoltageCalc_str.setter
    def monVoltageCalc_str(self, value: AnyStr):
        self.monVoltageCalc = value

    @property
    def monBus(self) -> List[str]:
        """
        Name of monitored bus used by the voltage-dependente control modes. Default is bus of the controlled PVSystem/Storage or Storage.

        DSS property name: `monBus`, DSS property index: 26.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 26)

    @monBus.setter
    def monBus(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 26, value_ptr, value_count)
        self._check_for_error()

    @property
    def MonBusesVbase(self) -> Float64Array:
        """
        Array list of rated voltages of the buses and their nodes presented in the monBus property. This list may have different line-to-line and/or line-to-ground voltages.

        DSS property name: `MonBusesVbase`, DSS property index: 27.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 27)

    @MonBusesVbase.setter
    def MonBusesVbase(self, value: Float64Array):
        self._set_float64_array_o(27, value)

    @property
    def voltwattCH_curve(self) -> str:
        """
        Required for VOLTWATT mode for Storage element in CHARGING state. 

        The name of an XYCurve object that describes the variation in active power output (in per unit of maximum active power outut for the Storage). 

        Units for the x-axis are per-unit voltage, which may be in per unit of the rated voltage for the Storage, or may be in per unit of the average voltage at the terminals over a user-defined number of prior solutions. 

        Units for the y-axis are either in: (1) per unit of maximum active power output capability of the Storage, or (2) maximum available active power output capability (defined by the parameter: VoltwattYAxis), corresponding to the terminal voltage (x-axis value in per unit). 

        No default -- must be specified for VOLTWATT mode for Storage element in CHARGING state.

        DSS property name: `voltwattCH_curve`, DSS property index: 28.
        """
        return self._get_prop_string(28)

    @voltwattCH_curve.setter
    def voltwattCH_curve(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(28, value)
            return

        self._set_string_o(28, value)

    @property
    def voltwattCH_curve_obj(self) -> XYcurve:
        """
        Required for VOLTWATT mode for Storage element in CHARGING state. 

        The name of an XYCurve object that describes the variation in active power output (in per unit of maximum active power outut for the Storage). 

        Units for the x-axis are per-unit voltage, which may be in per unit of the rated voltage for the Storage, or may be in per unit of the average voltage at the terminals over a user-defined number of prior solutions. 

        Units for the y-axis are either in: (1) per unit of maximum active power output capability of the Storage, or (2) maximum available active power output capability (defined by the parameter: VoltwattYAxis), corresponding to the terminal voltage (x-axis value in per unit). 

        No default -- must be specified for VOLTWATT mode for Storage element in CHARGING state.

        DSS property name: `voltwattCH_curve`, DSS property index: 28.
        """
        return self._get_obj(28, XYcurve)

    @voltwattCH_curve_obj.setter
    def voltwattCH_curve_obj(self, value: XYcurve):
        self._set_obj(28, value)

    @property
    def wattpf_curve(self) -> str:
        """
        Required for WATTPF mode.

        Name of the XYCurve object containing the watt-pf curve.
        The positive values of the y-axis are positive power factor values. The negative values of the the y-axis are negative power factor values. When positive, the output reactive power has the same direction of the output active power, and when negative, it has the opposite direction.
        Units for the x-axis are per-unit output active power, and the base active power is the Pmpp for PVSystem and kWrated for Storage.

        The y-axis represents the power factor and the reference is power factor equal to 0. 

        For example, if the user wants to define the following XY coordinates: (0, 0.9); (0.2, 0.9); (0.5, -0.9); (1, -0.9).
        Try to plot them considering the y-axis reference equal to unity power factor.

        The user needs to translate this curve into a plot in which the y-axis reference is equal to 0 power factor.It means that two new XY coordinates need to be included, in this case they are: (0.35, 1); (0.35, -1).
        Try to plot them considering the y-axis reference equal to 0 power factor.
        The discontinity in 0.35pu is not a problem since var is zero for either power factor equal to 1 or -1.

        DSS property name: `wattpf_curve`, DSS property index: 29.
        """
        return self._get_prop_string(29)

    @wattpf_curve.setter
    def wattpf_curve(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(29, value)
            return

        self._set_string_o(29, value)

    @property
    def wattpf_curve_obj(self) -> XYcurve:
        """
        Required for WATTPF mode.

        Name of the XYCurve object containing the watt-pf curve.
        The positive values of the y-axis are positive power factor values. The negative values of the the y-axis are negative power factor values. When positive, the output reactive power has the same direction of the output active power, and when negative, it has the opposite direction.
        Units for the x-axis are per-unit output active power, and the base active power is the Pmpp for PVSystem and kWrated for Storage.

        The y-axis represents the power factor and the reference is power factor equal to 0. 

        For example, if the user wants to define the following XY coordinates: (0, 0.9); (0.2, 0.9); (0.5, -0.9); (1, -0.9).
        Try to plot them considering the y-axis reference equal to unity power factor.

        The user needs to translate this curve into a plot in which the y-axis reference is equal to 0 power factor.It means that two new XY coordinates need to be included, in this case they are: (0.35, 1); (0.35, -1).
        Try to plot them considering the y-axis reference equal to 0 power factor.
        The discontinity in 0.35pu is not a problem since var is zero for either power factor equal to 1 or -1.

        DSS property name: `wattpf_curve`, DSS property index: 29.
        """
        return self._get_obj(29, XYcurve)

    @wattpf_curve_obj.setter
    def wattpf_curve_obj(self, value: XYcurve):
        self._set_obj(29, value)

    @property
    def wattvar_curve(self) -> str:
        """
        Required for WATTVAR mode. 

        Name of the XYCurve object containing the watt-var curve. The positive values of the y-axis of the watt-var curve represent values in pu of the provided base reactive power. The negative values of the y-axis are values in pu of the absorbed base reactive power. 
        Provided and absorbed base reactive power values are defined in the RefReactivePower property.

        Units for the x-axis are per-unit output active power, and the base active power is the Pmpp for PVSystem and kWrated for Storage.

        DSS property name: `wattvar_curve`, DSS property index: 30.
        """
        return self._get_prop_string(30)

    @wattvar_curve.setter
    def wattvar_curve(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(30, value)
            return

        self._set_string_o(30, value)

    @property
    def wattvar_curve_obj(self) -> XYcurve:
        """
        Required for WATTVAR mode. 

        Name of the XYCurve object containing the watt-var curve. The positive values of the y-axis of the watt-var curve represent values in pu of the provided base reactive power. The negative values of the y-axis are values in pu of the absorbed base reactive power. 
        Provided and absorbed base reactive power values are defined in the RefReactivePower property.

        Units for the x-axis are per-unit output active power, and the base active power is the Pmpp for PVSystem and kWrated for Storage.

        DSS property name: `wattvar_curve`, DSS property index: 30.
        """
        return self._get_obj(30, XYcurve)

    @wattvar_curve_obj.setter
    def wattvar_curve_obj(self, value: XYcurve):
        self._set_obj(30, value)

    @property
    def Vsetpoint(self) -> float:
        """
        Required for Active Voltage Regulation (AVR).

        DSS property name: `Vsetpoint`, DSS property index: 33.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 33)

    @Vsetpoint.setter
    def Vsetpoint(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 33, value)

    @property
    def ControlModel(self) -> InvControlControlModel:
        """
        Integer defining the method for moving across the control curve. It can be one of the following:

        0 = Linear mode (default)
        1 = Exponential

        Use this property for better tunning your controller and improve the controller response in terms of control iterations needed to reach the target.
        This property alters the meaning of deltaQ_factor and deltaP_factor properties accroding to its value (Check help). The method can also be combined with the controller tolerance for improving performance.

        DSS property name: `ControlModel`, DSS property index: 34.
        """
        return InvControl.InvControlControlModel(self._lib.Obj_GetInt32(self._ptr, 34))

    @ControlModel.setter
    def ControlModel(self, value: Union[int, InvControlControlModel]):
        self._lib.Obj_SetInt32(self._ptr, 34, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 35.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 35)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 35, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 36.
        """
        return self._lib.Obj_GetInt32(self._ptr, 36) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 36, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 37.
        """
        self._set_string_o(37, value)

class ExpControl(DSSObj):
    __slots__ = []
    _cls_name = 'ExpControl'
    _cls_idx = 43
    _cls_prop_idx = {
        'pvsystemlist': 1,
        'vreg': 2,
        'slope': 3,
        'vregtau': 4,
        'qbias': 5,
        'vregmin': 6,
        'vregmax': 7,
        'qmaxlead': 8,
        'qmaxlag': 9,
        'eventlog': 10,
        'deltaq_factor': 11,
        'preferq': 12,
        'tresponse': 13,
        'derlist': 14,
        'basefreq': 15,
        'enabled': 16,
        'like': 17,
    }

    @property
    def PVSystemList(self) -> List[str]:
        """
        Array list of PVSystems to be controlled.

        If not specified, all PVSystems in the circuit are assumed to be controlled by this ExpControl.

        DSS property name: `PVSystemList`, DSS property index: 1.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 1)

    @PVSystemList.setter
    def PVSystemList(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 1, value_ptr, value_count)
        self._check_for_error()

    @property
    def Vreg(self) -> float:
        """
        Per-unit voltage at which reactive power is zero; defaults to 1.0.

        This may dynamically self-adjust when VregTau > 0, limited by VregMin and VregMax.If imput as 0, Vreg will be initialized from a snapshot solution with no inverter Q.The equilibrium point of reactive power is also affected by Qbias

        DSS property name: `Vreg`, DSS property index: 2.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 2)

    @Vreg.setter
    def Vreg(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 2, value)

    @property
    def Slope(self) -> float:
        """
        Per-unit reactive power injection / per-unit voltage deviation from Vreg; defaults to 50.

        Unlike InvControl, base reactive power is constant at the inverter kva rating.

        DSS property name: `Slope`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @Slope.setter
    def Slope(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def VregTau(self) -> float:
        """
        Time constant for adaptive Vreg. Defaults to 1200 seconds.

        When the control injects or absorbs reactive power due to a voltage deviation from the Q=0 crossing of the volt-var curve, the Q=0 crossing will move toward the actual terminal voltage with this time constant. Over time, the effect is to gradually bring inverter reactive power to zero as the grid voltage changes due to non-solar effects. If zero, then Vreg stays fixed. IEEE1547-2018 requires adjustability from 300s to 5000s

        DSS property name: `VregTau`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @VregTau.setter
    def VregTau(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def Qbias(self) -> float:
        """
        Equilibrium per-unit reactive power when V=Vreg; defaults to 0.

        Enter > 0 for lagging (capacitive) bias, < 0 for leading (inductive) bias.

        DSS property name: `Qbias`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @Qbias.setter
    def Qbias(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def VregMin(self) -> float:
        """
        Lower limit on adaptive Vreg; defaults to 0.95 per-unit

        DSS property name: `VregMin`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @VregMin.setter
    def VregMin(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def VregMax(self) -> float:
        """
        Upper limit on adaptive Vreg; defaults to 1.05 per-unit

        DSS property name: `VregMax`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @VregMax.setter
    def VregMax(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def QmaxLead(self) -> float:
        """
        Limit on leading (inductive) reactive power injection, in per-unit of base kva; defaults to 0.44.For Category A inverters per P1547/D7, set this value to 0.25.

        Regardless of QmaxLead, the reactive power injection is still limited by dynamic headroom when actual real power output exceeds 0%

        DSS property name: `QmaxLead`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @QmaxLead.setter
    def QmaxLead(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def QmaxLag(self) -> float:
        """
        Limit on lagging (capacitive) reactive power injection, in per-unit of base kva; defaults to 0.44.

        For Category A inverters per P1547/D7, set this value to 0.25.Regardless of QmaxLag, the reactive power injection is still limited by dynamic headroom when actual real power output exceeds 0%

        DSS property name: `QmaxLag`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @QmaxLag.setter
    def QmaxLag(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def EventLog(self) -> bool:
        """
        {Yes/True* | No/False} Default is No for ExpControl. Log control actions to Eventlog.

        DSS property name: `EventLog`, DSS property index: 10.
        """
        return self._lib.Obj_GetInt32(self._ptr, 10) != 0

    @EventLog.setter
    def EventLog(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 10, value)

    @property
    def DeltaQ_factor(self) -> float:
        """
        Convergence parameter; Defaults to 0.7. 

        Sets the maximum change (in per unit) from the prior var output level to the desired var output level during each control iteration. If numerical instability is noticed in solutions such as var sign changing from one control iteration to the next and voltages oscillating between two values with some separation, this is an indication of numerical instability (use the EventLog to diagnose). If the maximum control iterations are exceeded, and no numerical instability is seen in the EventLog of via monitors, then try increasing the value of this parameter to reduce the number of control iterations needed to achieve the control criteria, and move to the power flow solution.

        DSS property name: `DeltaQ_factor`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @DeltaQ_factor.setter
    def DeltaQ_factor(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def PreferQ(self) -> bool:
        """
        {Yes/True* | No/False} Default is No for ExpControl.

        Curtails real power output as needed to meet the reactive power requirement. IEEE1547-2018 requires Yes, but the default is No for backward compatibility of OpenDSS models.

        DSS property name: `PreferQ`, DSS property index: 12.
        """
        return self._lib.Obj_GetInt32(self._ptr, 12) != 0

    @PreferQ.setter
    def PreferQ(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 12, value)

    @property
    def Tresponse(self) -> float:
        """
        Open-loop response time for changes in Q.

        The value of Q reaches 90% of the target change within Tresponse, which corresponds to a low-pass filter having tau = Tresponse / 2.3026. The behavior is similar to LPFTAU in InvControl, but here the response time is input instead of the time constant. IEEE1547-2018 default is 10s for Catagory A and 5s for Category B, adjustable from 1s to 90s for both categories. However, the default is 0 for backward compatibility of OpenDSS models.

        DSS property name: `Tresponse`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @Tresponse.setter
    def Tresponse(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def DERList(self) -> List[str]:
        """
        Alternative to PVSystemList for CIM export and import.

        However, storage is not actually implemented yet. Use fully qualified PVSystem names.

        DSS property name: `DERList`, DSS property index: 14.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 14)

    @DERList.setter
    def DERList(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 14, value_ptr, value_count)
        self._check_for_error()

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 16.
        """
        return self._lib.Obj_GetInt32(self._ptr, 16) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 16, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 17.
        """
        self._set_string_o(17, value)

class GICLine(DSSObj):
    __slots__ = []
    _cls_name = 'GICLine'
    _cls_idx = 44
    _cls_prop_idx = {
        'bus1': 1,
        'bus2': 2,
        'volts': 3,
        'angle': 4,
        'frequency': 5,
        'phases': 6,
        'r': 7,
        'x': 8,
        'c': 9,
        'en': 10,
        'ee': 11,
        'lat1': 12,
        'lon1': 13,
        'lat2': 14,
        'lon2': 15,
        'spectrum': 16,
        'basefreq': 17,
        'enabled': 18,
        'like': 19,
    }

    @property
    def bus1(self) -> str:
        """
        Name of bus to which the main terminal (1) is connected.
        bus1=busname
        bus1=busname.1.2.3

        DSS property name: `bus1`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @bus1.setter
    def bus1(self, value: AnyStr):
        self._set_string_o(1, value)

    @property
    def bus2(self) -> str:
        """
        Name of bus to which 2nd terminal is connected.
        bus2=busname
        bus2=busname.1.2.3

        No Default; must be specified.

        DSS property name: `bus2`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @bus2.setter
    def bus2(self, value: AnyStr):
        self._set_string_o(2, value)

    @property
    def Volts(self) -> float:
        """
        Voltage magnitude, in volts, of the GIC voltage induced across this line. When spedified, voltage source is assumed defined by Voltage and Angle properties. 

        Specify this value

        OR

        EN, EE, lat1, lon1, lat2, lon2. 

        Not both!!  Last one entered will take precedence. Assumed identical in each phase of the Line object.

        DSS property name: `Volts`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @Volts.setter
    def Volts(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def Angle(self) -> float:
        """
        Phase angle in degrees of first phase. Default=0.0.  See Voltage property

        DSS property name: `Angle`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @Angle.setter
    def Angle(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def frequency(self) -> float:
        """
        Source frequency.  Defaults to 0.1 Hz.

        DSS property name: `frequency`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @frequency.setter
    def frequency(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def phases(self) -> int:
        """
        Number of phases.  Defaults to 3.

        DSS property name: `phases`, DSS property index: 6.
        """
        return self._lib.Obj_GetInt32(self._ptr, 6)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def R(self) -> float:
        """
        Resistance of line, ohms of impedance in series with GIC voltage source. 

        DSS property name: `R`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @R.setter
    def R(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def X(self) -> float:
        """
        Reactance at base frequency, ohms. Default = 0.0. This value is generally not important for GIC studies but may be used if desired.

        DSS property name: `X`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @X.setter
    def X(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def C(self) -> float:
        """
        Value of line blocking capacitance in microfarads. Default = 0.0, implying that there is no line blocking capacitor.

        DSS property name: `C`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @C.setter
    def C(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def EN(self) -> float:
        """
        Northward Electric field (V/km). If specified, Voltage and Angle are computed from EN, EE, lat and lon values.

        DSS property name: `EN`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @EN.setter
    def EN(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def EE(self) -> float:
        """
        Eastward Electric field (V/km).  If specified, Voltage and Angle are computed from EN, EE, lat and lon values.

        DSS property name: `EE`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @EE.setter
    def EE(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def Lat1(self) -> float:
        """
        Latitude of Bus1 (degrees)

        DSS property name: `Lat1`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @Lat1.setter
    def Lat1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def Lon1(self) -> float:
        """
        Longitude of Bus1 (degrees)

        DSS property name: `Lon1`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @Lon1.setter
    def Lon1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def Lat2(self) -> float:
        """
        Latitude of Bus2 (degrees)

        DSS property name: `Lat2`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @Lat2.setter
    def Lat2(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def Lon2(self) -> float:
        """
        Longitude of Bus2 (degrees)

        DSS property name: `Lon2`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @Lon2.setter
    def Lon2(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def spectrum(self) -> str:
        """
        Inherited Property for all PCElements. Name of harmonic spectrum for this source.  Default is "defaultvsource", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 16.
        """
        return self._get_prop_string(16)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum]):
        if isinstance(value, DSSObj):
            self._set_obj(16, value)
            return

        self._set_string_o(16, value)

    @property
    def spectrum_obj(self) -> Spectrum:
        """
        Inherited Property for all PCElements. Name of harmonic spectrum for this source.  Default is "defaultvsource", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 16.
        """
        return self._get_obj(16, Spectrum)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_obj(16, value)

    @property
    def basefreq(self) -> float:
        """
        Inherited Property for all PCElements. Base frequency for specification of reactance value.

        DSS property name: `basefreq`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 18.
        """
        return self._lib.Obj_GetInt32(self._ptr, 18) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 18, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 19.
        """
        self._set_string_o(19, value)

class GICTransformer(DSSObj):
    __slots__ = []
    _cls_name = 'GICTransformer'
    _cls_idx = 45
    _cls_prop_idx = {
        'bush': 1,
        'busnh': 2,
        'busx': 3,
        'busnx': 4,
        'phases': 5,
        'type': 6,
        'r1': 7,
        'r2': 8,
        'kvll1': 9,
        'kvll2': 10,
        'mva': 11,
        'varcurve': 12,
        'pctr1': 13,
        '%r1': 13,
        'pctr2': 14,
        '%r2': 14,
        'k': 15,
        'normamps': 16,
        'emergamps': 17,
        'faultrate': 18,
        'pctperm': 19,
        'repair': 20,
        'basefreq': 21,
        'enabled': 22,
        'like': 23,
    }

    # Class-specific enumerations
    class GICTransformerType(IntEnum):
        """GICTransformer: Type (DSS enumeration for GICTransformer)"""
        GSU = 1 # GSU
        Auto = 2 # Auto
        YY = 3 # YY


    @property
    def BusH(self) -> str:
        """
        Name of High-side(H) bus. Examples:
        BusH=busname
        BusH=busname.1.2.3

        DSS property name: `BusH`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @BusH.setter
    def BusH(self, value: AnyStr):
        self._set_string_o(1, value)

    @property
    def BusNH(self) -> str:
        """
        Name of Neutral bus for H, or first, winding. Defaults to all phases connected to H-side bus, node 0, if not specified and transformer type is either GSU or YY. (Shunt Wye Connection to ground reference)For Auto, this is automatically set to the X bus.

        DSS property name: `BusNH`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @BusNH.setter
    def BusNH(self, value: AnyStr):
        self._set_string_o(2, value)

    @property
    def BusX(self) -> str:
        """
        Name of Low-side(X) bus, if type=Auto or YY. 

        DSS property name: `BusX`, DSS property index: 3.
        """
        return self._get_prop_string(3)

    @BusX.setter
    def BusX(self, value: AnyStr):
        self._set_string_o(3, value)

    @property
    def BusNX(self) -> str:
        """
        Name of Neutral bus for X, or Second, winding. Defaults to all phases connected to X-side bus, node 0, if not specified. (Shunt Wye Connection to ground reference)

        DSS property name: `BusNX`, DSS property index: 4.
        """
        return self._get_prop_string(4)

    @BusNX.setter
    def BusNX(self, value: AnyStr):
        self._set_string_o(4, value)

    @property
    def phases(self) -> int:
        """
        Number of Phases. Default is 3.

        DSS property name: `phases`, DSS property index: 5.
        """
        return self._lib.Obj_GetInt32(self._ptr, 5)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 5, value)

    @property
    def Type(self) -> GICTransformerType:
        """
        Type of transformer: {GSU* | Auto | YY}. Default is GSU.

        DSS property name: `Type`, DSS property index: 6.
        """
        return GICTransformer.GICTransformerType(self._lib.Obj_GetInt32(self._ptr, 6))

    @Type.setter
    def Type(self, value: Union[AnyStr, int, GICTransformerType]):
        if not isinstance(value, int):
            self._set_string_o(6, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def Type_str(self) -> str:
        """
        Type of transformer: {GSU* | Auto | YY}. Default is GSU.

        DSS property name: `Type`, DSS property index: 6.
        """
        return self._get_prop_string(6)

    @Type_str.setter
    def Type_str(self, value: AnyStr):
        self.Type = value

    @property
    def R1(self) -> float:
        """
        Resistance, each phase, ohms for H winding, (Series winding, if Auto). Default is 0.0001. If 

        DSS property name: `R1`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @R1.setter
    def R1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def R2(self) -> float:
        """
        Resistance, each phase, ohms for X winding, (Common winding, if Auto). Default is 0.0001. 

        DSS property name: `R2`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @R2.setter
    def R2(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def KVLL1(self) -> float:
        """
        Optional. kV LL rating for H winding (winding 1). Default is 500. Required if you are going to export vars for power flow analysis or enter winding resistances in percent.

        DSS property name: `KVLL1`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @KVLL1.setter
    def KVLL1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def KVLL2(self) -> float:
        """
        Optional. kV LL rating for X winding (winding 2). Default is 138. Required if you are going to export vars for power flow analysis or enter winding resistances in percent..

        DSS property name: `KVLL2`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @KVLL2.setter
    def KVLL2(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def MVA(self) -> float:
        """
        Optional. MVA Rating assumed Transformer. Default is 100. Used for computing vars due to GIC and winding resistances if kV and MVA ratings are specified.

        DSS property name: `MVA`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @MVA.setter
    def MVA(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def VarCurve(self) -> str:
        """
        Optional. XYCurve object name. Curve is expected as TOTAL pu vars vs pu GIC amps/phase. Vars are in pu of the MVA property. No Default value. Required only if you are going to export vars for power flow analysis. See K property.

        DSS property name: `VarCurve`, DSS property index: 12.
        """
        return self._get_prop_string(12)

    @VarCurve.setter
    def VarCurve(self, value: Union[AnyStr, XYcurve]):
        if isinstance(value, DSSObj):
            self._set_obj(12, value)
            return

        self._set_string_o(12, value)

    @property
    def VarCurve_obj(self) -> XYcurve:
        """
        Optional. XYCurve object name. Curve is expected as TOTAL pu vars vs pu GIC amps/phase. Vars are in pu of the MVA property. No Default value. Required only if you are going to export vars for power flow analysis. See K property.

        DSS property name: `VarCurve`, DSS property index: 12.
        """
        return self._get_obj(12, XYcurve)

    @VarCurve_obj.setter
    def VarCurve_obj(self, value: XYcurve):
        self._set_obj(12, value)

    @property
    def pctR1(self) -> float:
        """
        Optional. Percent Resistance, each phase, for H winding (1), (Series winding, if Auto). Default is 0.2. 

        Alternative way to enter R1 value. It is the actual resistances in ohmns that matter. MVA and kV should be specified.

        DSS property name: `%R1`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @pctR1.setter
    def pctR1(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def pctR2(self) -> float:
        """
        Optional. Percent Resistance, each phase, for X winding (2), (Common winding, if Auto). Default is 0.2. 

        Alternative way to enter R2 value. It is the actual resistances in ohms that matter. MVA and kV should be specified.

        DSS property name: `%R2`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @pctR2.setter
    def pctR2(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def K(self) -> float:
        """
        Mvar K factor. Default way to convert GIC Amps in H winding (winding 1) to Mvar. Default is 2.2. Commonly-used simple multiplier for estimating Mvar losses for power flow analysis. 

        Mvar = K * kvLL * GIC per phase / 1000 

        Mutually exclusive with using the VarCurve property and pu curves.If you specify this (default), VarCurve is ignored.

        DSS property name: `K`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @K.setter
    def K(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def normamps(self) -> float:
        """
        Normal rated current.

        DSS property name: `normamps`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @normamps.setter
    def normamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def emergamps(self) -> float:
        """
        Maximum or emerg current.

        DSS property name: `emergamps`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @emergamps.setter
    def emergamps(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def faultrate(self) -> float:
        """
        Failure rate per year.

        DSS property name: `faultrate`, DSS property index: 18.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 18)

    @faultrate.setter
    def faultrate(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 18, value)

    @property
    def pctperm(self) -> float:
        """
        Percent of failures that become permanent.

        DSS property name: `pctperm`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @pctperm.setter
    def pctperm(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def repair(self) -> float:
        """
        Hours to repair.

        DSS property name: `repair`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @repair.setter
    def repair(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 22.
        """
        return self._lib.Obj_GetInt32(self._ptr, 22) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 22, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 23.
        """
        self._set_string_o(23, value)

class VSConverter(DSSObj):
    __slots__ = []
    _cls_name = 'VSConverter'
    _cls_idx = 46
    _cls_prop_idx = {
        'phases': 1,
        'bus1': 2,
        'kvac': 3,
        'kvdc': 4,
        'kw': 5,
        'ndc': 6,
        'rac': 7,
        'xac': 8,
        'm0': 9,
        'd0': 10,
        'mmin': 11,
        'mmax': 12,
        'iacmax': 13,
        'idcmax': 14,
        'vacref': 15,
        'pacref': 16,
        'qacref': 17,
        'vdcref': 18,
        'vscmode': 19,
        'spectrum': 20,
        'basefreq': 21,
        'enabled': 22,
        'like': 23,
    }

    # Class-specific enumerations
    class VSConverterControlMode(IntEnum):
        """VSConverter: Control Mode (DSS enumeration for VSConverter)"""
        Fixed = 0 # Fixed
        PacVac = 1 # PacVac
        PacQac = 2 # PacQac
        VdcVac = 3 # VdcVac
        VdcQac = 4 # VdcQac


    @property
    def phases(self) -> int:
        """
        Number of AC plus DC conductors. Default is 4. AC phases numbered before DC conductors.

        DSS property name: `phases`, DSS property index: 1.
        """
        return self._lib.Obj_GetInt32(self._ptr, 1)

    @phases.setter
    def phases(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 1, value)

    @property
    def Bus1(self) -> str:
        """
        Name of converter bus, containing both AC and DC conductors. Bus2 is always ground.

        DSS property name: `Bus1`, DSS property index: 2.
        """
        return self._get_prop_string(2)

    @Bus1.setter
    def Bus1(self, value: AnyStr):
        self._set_string_o(2, value)

    @property
    def kVac(self) -> float:
        """
        Nominal AC line-neutral voltage in kV. Must be specified > 0.

        DSS property name: `kVac`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @kVac.setter
    def kVac(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def kVdc(self) -> float:
        """
        Nominal DC voltage in kV. Must be specified > 0.

        DSS property name: `kVdc`, DSS property index: 4.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 4)

    @kVdc.setter
    def kVdc(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 4, value)

    @property
    def kW(self) -> float:
        """
        Nominal converter power in kW. Must be specified > 0.

        DSS property name: `kW`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @kW.setter
    def kW(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def Ndc(self) -> int:
        """
        Number of DC conductors. Default is 1. DC conductors numbered after AC phases.

        DSS property name: `Ndc`, DSS property index: 6.
        """
        return self._lib.Obj_GetInt32(self._ptr, 6)

    @Ndc.setter
    def Ndc(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def Rac(self) -> float:
        """
        AC resistance (ohms) for the converter transformer, plus any series reactors. Default is 0.
        Must be 0 for Vac control mode.

        DSS property name: `Rac`, DSS property index: 7.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 7)

    @Rac.setter
    def Rac(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 7, value)

    @property
    def Xac(self) -> float:
        """
        AC reactance (ohms) for the converter transformer, plus any series reactors. Default is 0.
        Must be 0 for Vac control mode. Must be >0 for PacVac, PacQac or VacVdc control mode.

        DSS property name: `Xac`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @Xac.setter
    def Xac(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def m0(self) -> float:
        """
        Fixed or initial value of the modulation index. Default is 0.5.

        DSS property name: `m0`, DSS property index: 9.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 9)

    @m0.setter
    def m0(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 9, value)

    @property
    def d0(self) -> float:
        """
        Fixed or initial value of the power angle in degrees. Default is 0.

        DSS property name: `d0`, DSS property index: 10.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 10)

    @d0.setter
    def d0(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 10, value)

    @property
    def Mmin(self) -> float:
        """
        Minimum value of modulation index. Default is 0.1.

        DSS property name: `Mmin`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @Mmin.setter
    def Mmin(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def Mmax(self) -> float:
        """
        Maximum value of modulation index. Default is 0.9.

        DSS property name: `Mmax`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @Mmax.setter
    def Mmax(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def Iacmax(self) -> float:
        """
        Maximum value of AC line current, per-unit of nominal. Default is 2.

        DSS property name: `Iacmax`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @Iacmax.setter
    def Iacmax(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def Idcmax(self) -> float:
        """
        Maximum value of DC current, per-unit of nominal. Default is 2.

        DSS property name: `Idcmax`, DSS property index: 14.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 14)

    @Idcmax.setter
    def Idcmax(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 14, value)

    @property
    def Vacref(self) -> float:
        """
        Reference AC line-to-neutral voltage, RMS Volts. Default is 0.
        Applies to PacVac and VdcVac control modes, influencing m.

        DSS property name: `Vacref`, DSS property index: 15.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 15)

    @Vacref.setter
    def Vacref(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 15, value)

    @property
    def Pacref(self) -> float:
        """
        Reference total AC real power, Watts. Default is 0.
        Applies to PacVac and PacQac control modes, influencing d.

        DSS property name: `Pacref`, DSS property index: 16.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 16)

    @Pacref.setter
    def Pacref(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 16, value)

    @property
    def Qacref(self) -> float:
        """
        Reference total AC reactive power, Vars. Default is 0.
        Applies to PacQac and VdcQac control modes, influencing m.

        DSS property name: `Qacref`, DSS property index: 17.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 17)

    @Qacref.setter
    def Qacref(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 17, value)

    @property
    def Vdcref(self) -> float:
        """
        Reference DC voltage, Volts. Default is 0.
        Applies to VdcVac control mode, influencing d.

        DSS property name: `Vdcref`, DSS property index: 18.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 18)

    @Vdcref.setter
    def Vdcref(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 18, value)

    @property
    def VscMode(self) -> VSConverterControlMode:
        """
        Control Mode (Fixed|PacVac|PacQac|VdcVac|VdcQac). Default is Fixed.

        DSS property name: `VscMode`, DSS property index: 19.
        """
        return VSConverter.VSConverterControlMode(self._lib.Obj_GetInt32(self._ptr, 19))

    @VscMode.setter
    def VscMode(self, value: Union[AnyStr, int, VSConverterControlMode]):
        if not isinstance(value, int):
            self._set_string_o(19, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 19, value)

    @property
    def VscMode_str(self) -> str:
        """
        Control Mode (Fixed|PacVac|PacQac|VdcVac|VdcQac). Default is Fixed.

        DSS property name: `VscMode`, DSS property index: 19.
        """
        return self._get_prop_string(19)

    @VscMode_str.setter
    def VscMode_str(self, value: AnyStr):
        self.VscMode = value

    @property
    def spectrum(self) -> str:
        """
        Name of harmonic spectrum for this device.

        DSS property name: `spectrum`, DSS property index: 20.
        """
        return self._get_prop_string(20)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum]):
        if isinstance(value, DSSObj):
            self._set_obj(20, value)
            return

        self._set_string_o(20, value)

    @property
    def spectrum_obj(self) -> Spectrum:
        """
        Name of harmonic spectrum for this device.

        DSS property name: `spectrum`, DSS property index: 20.
        """
        return self._get_obj(20, Spectrum)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_obj(20, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 22.
        """
        return self._lib.Obj_GetInt32(self._ptr, 22) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 22, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 23.
        """
        self._set_string_o(23, value)

class Monitor(DSSObj):
    __slots__ = []
    _cls_name = 'Monitor'
    _cls_idx = 47
    _cls_prop_idx = {
        'element': 1,
        'terminal': 2,
        'mode': 3,
        'action': 4,
        'residual': 5,
        'vipolar': 6,
        'ppolar': 7,
        'basefreq': 8,
        'enabled': 9,
        'like': 10,
    }

    # Class-specific enumerations
    class MonitorAction(IntEnum):
        """Monitor: Action (DSS enumeration for Monitor)"""
        Clear = 0 # Clear
        Save = 1 # Save
        Take = 2 # Take
        Process = 3 # Process
        Reset = 0 # Reset


    @property
    def element(self) -> str:
        """
        Name (Full Object name) of element to which the monitor is connected.

        DSS property name: `element`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @element.setter
    def element(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(1, value)
            return

        self._set_string_o(1, value)

    @property
    def element_obj(self) -> DSSObj:
        """
        Name (Full Object name) of element to which the monitor is connected.

        DSS property name: `element`, DSS property index: 1.
        """
        return self._get_obj(1, None)

    @element_obj.setter
    def element_obj(self, value: DSSObj):
        self._set_obj(1, value)

    @property
    def terminal(self) -> int:
        """
        Number of the terminal of the circuit element to which the monitor is connected. 1 or 2, typically. For monitoring states, attach monitor to terminal 1.

        DSS property name: `terminal`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @terminal.setter
    def terminal(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def mode(self) -> int:
        """
        Bitmask integer designating the values the monitor is to capture: 
        0 = Voltages and currents at designated terminal
        1 = Powers at designated terminal
        2 = Tap Position (Transformer Device only)
        3 = State Variables (PCElements only)
        4 = Flicker level and severity index (Pst) for voltages. No adders apply.
            Flicker level at simulation time step, Pst at 10-minute time step.
        5 = Solution variables (Iterations, etc).
        Normally, these would be actual phasor quantities from solution.
        6 = Capacitor Switching (Capacitors only)
        7 = Storage state vars (Storage device only)
        8 = All winding currents (Transformer device only)
        9 = Losses, watts and var (of monitored device)
        10 = All Winding voltages (Transformer device only)
        Normally, these would be actual phasor quantities from solution.
        11 = All terminal node voltages and line currents of monitored device
        12 = All terminal node voltages LL and line currents of monitored device
        Combine mode with adders below to achieve other results for terminal quantities:
        +16 = Sequence quantities
        +32 = Magnitude only
        +64 = Positive sequence only or avg of all phases

        Mix adder to obtain desired results. For example:
        Mode=112 will save positive sequence voltage and current magnitudes only
        Mode=48 will save all sequence voltages and currents, but magnitude only.

        DSS property name: `mode`, DSS property index: 3.
        """
        return self._lib.Obj_GetInt32(self._ptr, 3)

    @mode.setter
    def mode(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 3, value)

    def action(self, value: Union[str, bytes, int, MonitorAction]):
        """
        {Clear | Save | Take | Process}
        (C)lears or (S)aves current buffer.
        (T)ake action takes a sample.
        (P)rocesses the data taken so far (e.g. Pst for mode 4).

        Note that monitors are automatically reset (cleared) when the Set Mode= command is issued. Otherwise, the user must explicitly reset all monitors (reset monitors command) or individual monitors with the Clear action.

        DSS property name: `action`, DSS property index: 4.
        """
        if isinstance(value, int):
            self._lib.Obj_SetInt32(self._ptr, 4, value)
            return
    
        self._set_string_o(4, value)

    @property
    def residual(self) -> bool:
        """
        {Yes/True | No/False} Default = No.  Include Residual cbannel (sum of all phases) for voltage and current. Does not apply to sequence quantity modes or power modes.

        DSS property name: `residual`, DSS property index: 5.
        """
        return self._lib.Obj_GetInt32(self._ptr, 5) != 0

    @residual.setter
    def residual(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 5, value)

    @property
    def VIPolar(self) -> bool:
        """
        {Yes/True | No/False} Default = YES. Report voltage and current in polar form (Mag/Angle). (default)  Otherwise, it will be real and imaginary.

        DSS property name: `VIPolar`, DSS property index: 6.
        """
        return self._lib.Obj_GetInt32(self._ptr, 6) != 0

    @VIPolar.setter
    def VIPolar(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 6, value)

    @property
    def PPolar(self) -> bool:
        """
        {Yes/True | No/False} Default = YES. Report power in Apparent power, S, in polar form (Mag/Angle).(default)  Otherwise, is P and Q

        DSS property name: `PPolar`, DSS property index: 7.
        """
        return self._lib.Obj_GetInt32(self._ptr, 7) != 0

    @PPolar.setter
    def PPolar(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 7, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 8.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 8)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 8, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 9.
        """
        return self._lib.Obj_GetInt32(self._ptr, 9) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 9, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 10.
        """
        self._set_string_o(10, value)

class EnergyMeter(DSSObj):
    __slots__ = []
    _cls_name = 'EnergyMeter'
    _cls_idx = 48
    _cls_prop_idx = {
        'element': 1,
        'terminal': 2,
        'action': 3,
        'option': 4,
        'kvanormal': 5,
        'kvaemerg': 6,
        'peakcurrent': 7,
        'zonelist': 8,
        'localonly': 9,
        'mask': 10,
        'losses': 11,
        'linelosses': 12,
        'xfmrlosses': 13,
        'seqlosses': 14,
        'threepaselosses': 15,
        '3phaselosses': 15,
        'vbaselosses': 16,
        'phasevoltagereport': 17,
        'int_rate': 18,
        'int_duration': 19,
        'saifi': 20,
        'saifikw': 21,
        'saidi': 22,
        'caidi': 23,
        'custinterrupts': 24,
        'basefreq': 25,
        'enabled': 26,
        'like': 27,
    }

    # Class-specific enumerations
    class EnergyMeterAction(IntEnum):
        """EnergyMeter: Action (DSS enumeration for EnergyMeter)"""
        Allocate = 0 # Allocate
        Clear = 1 # Clear
        Reduce = 2 # Reduce
        Save = 3 # Save
        Take = 4 # Take
        ZoneDump = 5 # ZoneDump


    @property
    def element(self) -> str:
        """
        Name (Full Object name) of element to which the monitor is connected.

        DSS property name: `element`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @element.setter
    def element(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(1, value)
            return

        self._set_string_o(1, value)

    @property
    def element_obj(self) -> DSSObj:
        """
        Name (Full Object name) of element to which the monitor is connected.

        DSS property name: `element`, DSS property index: 1.
        """
        return self._get_obj(1, None)

    @element_obj.setter
    def element_obj(self, value: DSSObj):
        self._set_obj(1, value)

    @property
    def terminal(self) -> int:
        """
        Number of the terminal of the circuit element to which the monitor is connected. 1 or 2, typically.

        DSS property name: `terminal`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @terminal.setter
    def terminal(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    def action(self, value: Union[str, bytes, int, EnergyMeterAction]):
        """
        {Clear (reset) | Save | Take | Zonedump | Allocate | Reduce} 

        (A)llocate = Allocate loads on the meter zone to match PeakCurrent.
        (C)lear = reset all registers to zero
        (R)educe = reduces zone by merging lines (see Set Keeplist & ReduceOption)
        (S)ave = saves the current register values to a file.
           File name is "MTR_metername.csv".
        (T)ake = Takes a sample at present solution
        (Z)onedump = Dump names of elements in meter zone to a file
           File name is "Zone_metername.csv".

        DSS property name: `action`, DSS property index: 3.
        """
        if isinstance(value, int):
            self._lib.Obj_SetInt32(self._ptr, 3, value)
            return
    
        self._set_string_o(3, value)

    @property
    def option(self) -> List[str]:
        """
        Enter a string ARRAY of any combination of the following. Options processed left-to-right:

        (E)xcess : (default) UE/EEN is estimate of energy over capacity 
        (T)otal : UE/EEN is total energy after capacity exceeded
        (R)adial : (default) Treats zone as a radial circuit
        (M)esh : Treats zone as meshed network (not radial).
        (C)ombined : (default) Load UE/EEN computed from combination of overload and undervoltage.
        (V)oltage : Load UE/EEN computed based on voltage only.

        Example: option=(E, R)

        DSS property name: `option`, DSS property index: 4.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 4)

    @option.setter
    def option(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 4, value_ptr, value_count)
        self._check_for_error()

    @property
    def kVAnormal(self) -> float:
        """
        Upper limit on kVA load in the zone, Normal configuration. Default is 0.0 (ignored). Overrides limits on individual lines for overload EEN. With "LocalOnly=Yes" option, uses only load in metered branch.

        DSS property name: `kVAnormal`, DSS property index: 5.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 5)

    @kVAnormal.setter
    def kVAnormal(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 5, value)

    @property
    def kVAemerg(self) -> float:
        """
        Upper limit on kVA load in the zone, Emergency configuration. Default is 0.0 (ignored). Overrides limits on individual lines for overload UE. With "LocalOnly=Yes" option, uses only load in metered branch.

        DSS property name: `kVAemerg`, DSS property index: 6.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 6)

    @kVAemerg.setter
    def kVAemerg(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 6, value)

    @property
    def peakcurrent(self) -> Float64Array:
        """
        ARRAY of current magnitudes representing the peak currents measured at this location for the load allocation function.  Default is (400, 400, 400). Enter one current for each phase

        DSS property name: `peakcurrent`, DSS property index: 7.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 7)

    @peakcurrent.setter
    def peakcurrent(self, value: Float64Array):
        self._set_float64_array_o(7, value)

    @property
    def Zonelist(self) -> List[str]:
        """
        ARRAY of full element names for this meter's zone.  Default is for meter to find it's own zone. If specified, DSS uses this list instead.  Can access the names in a single-column text file.  Examples: 

        zonelist=[line.L1, transformer.T1, Line.L3] 
        zonelist=(file=branchlist.txt)

        DSS property name: `Zonelist`, DSS property index: 8.
        """
        return self._get_string_array(self._lib.Obj_GetStringArray, self._ptr, 8)

    @Zonelist.setter
    def Zonelist(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        self._lib.Obj_SetStringArray(self._ptr, 8, value_ptr, value_count)
        self._check_for_error()

    @property
    def LocalOnly(self) -> bool:
        """
        {Yes | No}  Default is NO.  If Yes, meter considers only the monitored element for EEN and UE calcs.  Uses whole zone for losses.

        DSS property name: `LocalOnly`, DSS property index: 9.
        """
        return self._lib.Obj_GetInt32(self._ptr, 9) != 0

    @LocalOnly.setter
    def LocalOnly(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 9, value)

    @property
    def Mask(self) -> Float64Array:
        """
        Mask for adding registers whenever all meters are totalized.  Array of floating point numbers representing the multiplier to be used for summing each register from this meter. Default = (1, 1, 1, 1, ... ).  You only have to enter as many as are changed (positional). Useful when two meters monitor same energy, etc.

        DSS property name: `Mask`, DSS property index: 10.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 10)

    @Mask.setter
    def Mask(self, value: Float64Array):
        self._set_float64_array_o(10, value)

    @property
    def Losses(self) -> bool:
        """
        {Yes | No}  Default is YES. Compute Zone losses. If NO, then no losses at all are computed.

        DSS property name: `Losses`, DSS property index: 11.
        """
        return self._lib.Obj_GetInt32(self._ptr, 11) != 0

    @Losses.setter
    def Losses(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 11, value)

    @property
    def LineLosses(self) -> bool:
        """
        {Yes | No}  Default is YES. Compute Line losses. If NO, then none of the losses are computed.

        DSS property name: `LineLosses`, DSS property index: 12.
        """
        return self._lib.Obj_GetInt32(self._ptr, 12) != 0

    @LineLosses.setter
    def LineLosses(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 12, value)

    @property
    def XfmrLosses(self) -> bool:
        """
        {Yes | No}  Default is YES. Compute Transformer losses. If NO, transformers are ignored in loss calculations.

        DSS property name: `XfmrLosses`, DSS property index: 13.
        """
        return self._lib.Obj_GetInt32(self._ptr, 13) != 0

    @XfmrLosses.setter
    def XfmrLosses(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 13, value)

    @property
    def SeqLosses(self) -> bool:
        """
        {Yes | No}  Default is YES. Compute Sequence losses in lines and segregate by line mode losses and zero mode losses.

        DSS property name: `SeqLosses`, DSS property index: 14.
        """
        return self._lib.Obj_GetInt32(self._ptr, 14) != 0

    @SeqLosses.setter
    def SeqLosses(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 14, value)

    @property
    def threePaseLosses(self) -> bool:
        """
        {Yes | No}  Default is YES. Compute Line losses and segregate by 3-phase and other (1- and 2-phase) line losses. 

        DSS property name: `3phaseLosses`, DSS property index: 15.
        """
        return self._lib.Obj_GetInt32(self._ptr, 15) != 0

    @threePaseLosses.setter
    def threePaseLosses(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 15, value)

    @property
    def VbaseLosses(self) -> bool:
        """
        {Yes | No}  Default is YES. Compute losses and segregate by voltage base. If NO, then voltage-based tabulation is not reported.

        DSS property name: `VbaseLosses`, DSS property index: 16.
        """
        return self._lib.Obj_GetInt32(self._ptr, 16) != 0

    @VbaseLosses.setter
    def VbaseLosses(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 16, value)

    @property
    def PhaseVoltageReport(self) -> bool:
        """
        {Yes | No}  Default is NO.  Report min, max, and average phase voltages for the zone and tabulate by voltage base. Demand Intervals must be turned on (Set Demand=true) and voltage bases must be defined for this property to take effect. Result is in a separate report file.

        DSS property name: `PhaseVoltageReport`, DSS property index: 17.
        """
        return self._lib.Obj_GetInt32(self._ptr, 17) != 0

    @PhaseVoltageReport.setter
    def PhaseVoltageReport(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 17, value)

    @property
    def Int_Rate(self) -> float:
        """
        Average number of annual interruptions for head of the meter zone (source side of zone or feeder).

        DSS property name: `Int_Rate`, DSS property index: 18.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 18)

    @Int_Rate.setter
    def Int_Rate(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 18, value)

    @property
    def Int_Duration(self) -> float:
        """
        Average annual duration, in hr, of interruptions for head of the meter zone (source side of zone or feeder).

        DSS property name: `Int_Duration`, DSS property index: 19.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 19)

    @Int_Duration.setter
    def Int_Duration(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 19, value)

    @property
    def SAIFI(self) -> float:
        """
        (Read only) Makes SAIFI result available via return on query (? energymeter.myMeter.SAIFI.

        DSS property name: `SAIFI`, DSS property index: 20.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 20)

    @SAIFI.setter
    def SAIFI(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 20, value)

    @property
    def SAIFIkW(self) -> float:
        """
        (Read only) Makes SAIFIkW result available via return on query (? energymeter.myMeter.SAIFIkW.

        DSS property name: `SAIFIkW`, DSS property index: 21.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 21)

    @SAIFIkW.setter
    def SAIFIkW(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 21, value)

    @property
    def SAIDI(self) -> float:
        """
        (Read only) Makes SAIDI result available via return on query (? energymeter.myMeter.SAIDI.

        DSS property name: `SAIDI`, DSS property index: 22.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 22)

    @SAIDI.setter
    def SAIDI(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 22, value)

    @property
    def CAIDI(self) -> float:
        """
        (Read only) Makes CAIDI result available via return on query (? energymeter.myMeter.CAIDI.

        DSS property name: `CAIDI`, DSS property index: 23.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 23)

    @CAIDI.setter
    def CAIDI(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 23, value)

    @property
    def CustInterrupts(self) -> float:
        """
        (Read only) Makes Total Customer Interrupts value result available via return on query (? energymeter.myMeter.CustInterrupts.

        DSS property name: `CustInterrupts`, DSS property index: 24.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 24)

    @CustInterrupts.setter
    def CustInterrupts(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 24, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 25.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 25)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 25, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 26.
        """
        return self._lib.Obj_GetInt32(self._ptr, 26) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 26, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 27.
        """
        self._set_string_o(27, value)

class Sensor(DSSObj):
    __slots__ = []
    _cls_name = 'Sensor'
    _cls_idx = 49
    _cls_prop_idx = {
        'element': 1,
        'terminal': 2,
        'kvbase': 3,
        'clear': 4,
        'kvs': 5,
        'currents': 6,
        'kws': 7,
        'kvars': 8,
        'conn': 9,
        'deltadirection': 10,
        'pcterror': 11,
        '%error': 11,
        'weight': 12,
        'basefreq': 13,
        'enabled': 14,
        'like': 15,
    }

    @property
    def element(self) -> str:
        """
        Name (Full Object name) of element to which the Sensor is connected.

        DSS property name: `element`, DSS property index: 1.
        """
        return self._get_prop_string(1)

    @element.setter
    def element(self, value: Union[AnyStr, DSSObj]):
        if isinstance(value, DSSObj):
            self._set_obj(1, value)
            return

        self._set_string_o(1, value)

    @property
    def element_obj(self) -> DSSObj:
        """
        Name (Full Object name) of element to which the Sensor is connected.

        DSS property name: `element`, DSS property index: 1.
        """
        return self._get_obj(1, None)

    @element_obj.setter
    def element_obj(self, value: DSSObj):
        self._set_obj(1, value)

    @property
    def terminal(self) -> int:
        """
        Number of the terminal of the circuit element to which the Sensor is connected. 1 or 2, typically. Default is 1.

        DSS property name: `terminal`, DSS property index: 2.
        """
        return self._lib.Obj_GetInt32(self._ptr, 2)

    @terminal.setter
    def terminal(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 2, value)

    @property
    def kvbase(self) -> float:
        """
        Voltage base for the sensor, in kV. If connected to a 2- or 3-phase terminal, 
        specify L-L voltage. For 1-phase devices specify L-N or actual 1-phase voltage. Like many other DSS devices, default is 12.47kV.

        DSS property name: `kvbase`, DSS property index: 3.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 3)

    @kvbase.setter
    def kvbase(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 3, value)

    @property
    def clear(self) -> bool:
        """
        { Yes | No }. Clear=Yes clears sensor values. Should be issued before putting in a new set of measurements.

        DSS property name: `clear`, DSS property index: 4.
        """
        return self._lib.Obj_GetInt32(self._ptr, 4) != 0

    @clear.setter
    def clear(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 4, value)

    @property
    def kVs(self) -> Float64Array:
        """
        Array of Voltages (kV) measured by the voltage sensor. For Delta-connected sensors, Line-Line voltages are expected. For Wye, Line-Neutral are expected.

        DSS property name: `kVs`, DSS property index: 5.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 5)

    @kVs.setter
    def kVs(self, value: Float64Array):
        self._set_float64_array_o(5, value)

    @property
    def currents(self) -> Float64Array:
        """
        Array of Currents (amps) measured by the current sensor. Specify this or power quantities; not both.

        DSS property name: `currents`, DSS property index: 6.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 6)

    @currents.setter
    def currents(self, value: Float64Array):
        self._set_float64_array_o(6, value)

    @property
    def kWs(self) -> Float64Array:
        """
        Array of Active power (kW) measurements at the sensor. Is converted into Currents along with q=[...]
        Will override any currents=[...] specification.

        DSS property name: `kWs`, DSS property index: 7.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 7)

    @kWs.setter
    def kWs(self, value: Float64Array):
        self._set_float64_array_o(7, value)

    @property
    def kvars(self) -> Float64Array:
        """
        Array of Reactive power (kvar) measurements at the sensor. Is converted into Currents along with p=[...]

        DSS property name: `kvars`, DSS property index: 8.
        """
        return self._get_float64_array(self._lib.Obj_GetFloat64Array, self._ptr, 8)

    @kvars.setter
    def kvars(self, value: Float64Array):
        self._set_float64_array_o(8, value)

    @property
    def conn(self) -> Connection:
        """
        Voltage sensor Connection: { wye | delta | LN | LL }.  Default is wye. Applies to voltage measurement only. 
        Currents are always assumed to be line currents.
        If wye or LN, voltage is assumed measured line-neutral; otherwise, line-line.

        DSS property name: `conn`, DSS property index: 9.
        """
        return Connection(self._lib.Obj_GetInt32(self._ptr, 9))

    @conn.setter
    def conn(self, value: Union[AnyStr, int, Connection]):
        if not isinstance(value, int):
            self._set_string_o(9, value)
            return
        self._lib.Obj_SetInt32(self._ptr, 9, value)

    @property
    def conn_str(self) -> str:
        """
        Voltage sensor Connection: { wye | delta | LN | LL }.  Default is wye. Applies to voltage measurement only. 
        Currents are always assumed to be line currents.
        If wye or LN, voltage is assumed measured line-neutral; otherwise, line-line.

        DSS property name: `conn`, DSS property index: 9.
        """
        return self._get_prop_string(9)

    @conn_str.setter
    def conn_str(self, value: AnyStr):
        self.conn = value

    @property
    def Deltadirection(self) -> int:
        """
        {1 or -1}  Default is 1:  1-2, 2-3, 3-1.  For reverse rotation, enter -1. Any positive or negative entry will suffice.

        DSS property name: `Deltadirection`, DSS property index: 10.
        """
        return self._lib.Obj_GetInt32(self._ptr, 10)

    @Deltadirection.setter
    def Deltadirection(self, value: int):
        self._lib.Obj_SetInt32(self._ptr, 10, value)

    @property
    def pctError(self) -> float:
        """
        Assumed percent error in the measurement. Default is 1.

        DSS property name: `%Error`, DSS property index: 11.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 11)

    @pctError.setter
    def pctError(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 11, value)

    @property
    def Weight(self) -> float:
        """
        Weighting factor: Default is 1.

        DSS property name: `Weight`, DSS property index: 12.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 12)

    @Weight.setter
    def Weight(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 12, value)

    @property
    def basefreq(self) -> float:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 13.
        """
        return self._lib.Obj_GetFloat64(self._ptr, 13)

    @basefreq.setter
    def basefreq(self, value: float):
        self._lib.Obj_SetFloat64(self._ptr, 13, value)

    @property
    def enabled(self) -> bool:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 14.
        """
        return self._lib.Obj_GetInt32(self._ptr, 14) != 0

    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Obj_SetInt32(self._ptr, 14, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 15.
        """
        self._set_string_o(15, value)

class LineCodeProperties(TypedDict):
    nphases: int
    r1: float
    x1: float
    r0: float
    x0: float
    C1: float
    C0: float
    units: Union[AnyStr, int, DimensionUnits]
    rmatrix: Float64Array
    xmatrix: Float64Array
    cmatrix: Float64Array
    baseFreq: float
    normamps: float
    emergamps: float
    faultrate: float
    pctperm: float
    repair: float
    Kron: bool
    Rg: float
    Xg: float
    rho: float
    neutral: int
    B1: float
    B0: float
    Seasons: int
    Ratings: Float64Array
    linetype: Union[AnyStr, int, LineType]
    like: AnyStr

class LoadShapeProperties(TypedDict):
    npts: int
    interval: float
    hour: Float64Array
    mean: float
    stddev: float
    csvfile: AnyStr
    sngfile: AnyStr
    dblfile: AnyStr
    action: Union[str, bytes, int, LoadShape.LoadShapeAction]
    qmult: Float64Array
    UseActual: bool
    Pmax: float
    Qmax: float
    sinterval: float
    minterval: float
    Pbase: float
    Qbase: float
    Pmult: Float64Array
    PQCSVFile: AnyStr
    MemoryMapping: bool
    like: AnyStr

class TShapeProperties(TypedDict):
    npts: int
    interval: float
    temp: Float64Array
    hour: Float64Array
    mean: float
    stddev: float
    csvfile: AnyStr
    sngfile: AnyStr
    dblfile: AnyStr
    sinterval: float
    minterval: float
    action: Union[str, bytes, int, TShape.TShapeAction]
    like: AnyStr

class PriceShapeProperties(TypedDict):
    npts: int
    interval: float
    price: Float64Array
    hour: Float64Array
    mean: float
    stddev: float
    csvfile: AnyStr
    sngfile: AnyStr
    dblfile: AnyStr
    sinterval: float
    minterval: float
    action: Union[str, bytes, int, PriceShape.PriceShapeAction]
    like: AnyStr

class XYcurveProperties(TypedDict):
    npts: int
    Yarray: Float64Array
    Xarray: Float64Array
    csvfile: AnyStr
    sngfile: AnyStr
    dblfile: AnyStr
    x: float
    y: float
    Xshift: float
    Yshift: float
    Xscale: float
    Yscale: float
    like: AnyStr

class GrowthShapeProperties(TypedDict):
    npts: int
    year: Float64Array
    mult: Float64Array
    csvfile: AnyStr
    sngfile: AnyStr
    dblfile: AnyStr
    like: AnyStr

class TCC_CurveProperties(TypedDict):
    npts: int
    C_array: Float64Array
    T_array: Float64Array
    like: AnyStr

class SpectrumProperties(TypedDict):
    NumHarm: int
    harmonic: Float64Array
    pctmag: Float64Array
    angle: Float64Array
    CSVFile: AnyStr
    like: AnyStr

class WireDataProperties(TypedDict):
    Rdc: float
    Rac: float
    Runits: Union[AnyStr, int, DimensionUnits]
    GMRac: float
    GMRunits: Union[AnyStr, int, DimensionUnits]
    radius: float
    radunits: Union[AnyStr, int, DimensionUnits]
    normamps: float
    emergamps: float
    diam: float
    Seasons: int
    Ratings: Float64Array
    Capradius: float
    like: AnyStr

class CNDataProperties(TypedDict):
    k: int
    DiaStrand: float
    GmrStrand: float
    Rstrand: float
    EpsR: float
    InsLayer: float
    DiaIns: float
    DiaCable: float
    Rdc: float
    Rac: float
    Runits: Union[AnyStr, int, DimensionUnits]
    GMRac: float
    GMRunits: Union[AnyStr, int, DimensionUnits]
    radius: float
    radunits: Union[AnyStr, int, DimensionUnits]
    normamps: float
    emergamps: float
    diam: float
    Seasons: int
    Ratings: Float64Array
    Capradius: float
    like: AnyStr

class TSDataProperties(TypedDict):
    DiaShield: float
    TapeLayer: float
    TapeLap: float
    EpsR: float
    InsLayer: float
    DiaIns: float
    DiaCable: float
    Rdc: float
    Rac: float
    Runits: Union[AnyStr, int, DimensionUnits]
    GMRac: float
    GMRunits: Union[AnyStr, int, DimensionUnits]
    radius: float
    radunits: Union[AnyStr, int, DimensionUnits]
    normamps: float
    emergamps: float
    diam: float
    Seasons: int
    Ratings: Float64Array
    Capradius: float
    like: AnyStr

class LineSpacingProperties(TypedDict):
    nconds: int
    nphases: int
    x: Float64Array
    h: Float64Array
    units: Union[AnyStr, int, DimensionUnits]
    like: AnyStr

class LineGeometryProperties(TypedDict):
    nconds: int
    nphases: int
    x: Float64Array
    h: Float64Array
    units: Union[AnyStr, int, DimensionUnits]
    normamps: float
    emergamps: float
    reduce: bool
    spacing: Union[AnyStr, LineSpacing]
    wires: List[Union[AnyStr, WireData]]
    cncables: List[Union[AnyStr, CNData]]
    tscables: List[Union[AnyStr, TSData]]
    Seasons: int
    Ratings: Float64Array
    linetype: Union[AnyStr, int, LineType]
    like: AnyStr

class XfmrCodeProperties(TypedDict):
    phases: int
    windings: int
    pctR: Float64Array
    Rneut: Float64Array
    Xneut: Float64Array
    conns: Union[List[Union[int,Connection]], List[AnyStr]]
    kVs: Float64Array
    kVAs: Float64Array
    taps: Float64Array
    Xhl: float
    Xht: float
    Xlt: float
    Xscarray: Float64Array
    thermal: float
    n: float
    m: float
    flrise: float
    hsrise: float
    pctloadloss: float
    pctnoloadloss: float
    normhkVA: float
    emerghkVA: float
    MaxTap: Float64Array
    MinTap: Float64Array
    NumTaps: Int32Array
    pctimag: float
    ppm_antifloat: float
    pctRs: Float64Array
    X12: float
    X13: float
    X23: float
    RdcOhms: Float64Array
    Seasons: int
    Ratings: Float64Array
    like: AnyStr

class LineProperties(TypedDict):
    bus1: AnyStr
    bus2: AnyStr
    linecode: Union[AnyStr, LineCode]
    length: float
    phases: int
    r1: float
    x1: float
    r0: float
    x0: float
    C1: float
    C0: float
    rmatrix: Float64Array
    xmatrix: Float64Array
    cmatrix: Float64Array
    Switch: bool
    Rg: float
    Xg: float
    rho: float
    geometry: Union[AnyStr, LineGeometry]
    units: Union[AnyStr, int, DimensionUnits]
    spacing: Union[AnyStr, LineSpacing]
    wires: List[Union[AnyStr, WireData]]
    earthmodel: Union[AnyStr, int, EarthModel]
    cncables: List[Union[AnyStr, CNData]]
    tscables: List[Union[AnyStr, TSData]]
    B1: float
    B0: float
    Seasons: int
    Ratings: Float64Array
    linetype: Union[AnyStr, int, LineType]
    normamps: float
    emergamps: float
    faultrate: float
    pctperm: float
    repair: float
    basefreq: float
    enabled: bool
    like: AnyStr

class VsourceProperties(TypedDict):
    bus1: AnyStr
    basekv: float
    pu: float
    angle: float
    frequency: float
    phases: int
    MVAsc3: float
    MVAsc1: float
    x1r1: float
    x0r0: float
    Isc3: float
    Isc1: float
    R1: float
    X1: float
    R0: float
    X0: float
    scantype: Union[AnyStr, int, ScanType]
    Sequence: Union[AnyStr, int, SequenceType]
    bus2: AnyStr
    Z2: complex
    puZ1: complex
    puZ0: complex
    puZ2: complex
    baseMVA: float
    Yearly: Union[AnyStr, LoadShape]
    Daily: Union[AnyStr, LoadShape]
    Duty: Union[AnyStr, LoadShape]
    Model: Union[AnyStr, int, Vsource.VSourceModel]
    puZideal: complex
    spectrum: Union[AnyStr, Spectrum]
    basefreq: float
    enabled: bool
    like: AnyStr

class IsourceProperties(TypedDict):
    bus1: AnyStr
    amps: float
    angle: float
    frequency: float
    phases: int
    scantype: Union[AnyStr, int, ScanType]
    sequence: Union[AnyStr, int, SequenceType]
    Yearly: Union[AnyStr, LoadShape]
    Daily: Union[AnyStr, LoadShape]
    Duty: Union[AnyStr, LoadShape]
    Bus2: AnyStr
    spectrum: Union[AnyStr, Spectrum]
    basefreq: float
    enabled: bool
    like: AnyStr

class VCCSProperties(TypedDict):
    bus1: AnyStr
    phases: int
    prated: float
    vrated: float
    ppct: float
    bp1: Union[AnyStr, XYcurve]
    bp2: Union[AnyStr, XYcurve]
    filter: Union[AnyStr, XYcurve]
    fsample: float
    rmsmode: bool
    imaxpu: float
    vrmstau: float
    irmstau: float
    spectrum: Union[AnyStr, Spectrum]
    basefreq: float
    enabled: bool
    like: AnyStr

class LoadProperties(TypedDict):
    phases: int
    bus1: AnyStr
    kV: float
    kW: float
    pf: float
    model: Union[int, Load.LoadModel]
    yearly: Union[AnyStr, LoadShape]
    daily: Union[AnyStr, LoadShape]
    duty: Union[AnyStr, LoadShape]
    growth: Union[AnyStr, GrowthShape]
    conn: Union[AnyStr, int, Connection]
    kvar: float
    Rneut: float
    Xneut: float
    status: Union[AnyStr, int, Load.LoadStatus]
    cls: int
    Vminpu: float
    Vmaxpu: float
    Vminnorm: float
    Vminemerg: float
    xfkVA: float
    allocationfactor: float
    kVA: float
    pctmean: float
    pctstddev: float
    CVRwatts: float
    CVRvars: float
    kwh: float
    kwhdays: float
    Cfactor: float
    CVRcurve: Union[AnyStr, LoadShape]
    NumCust: int
    ZIPV: Float64Array
    pctSeriesRL: float
    RelWeight: float
    Vlowpu: float
    puXharm: float
    XRharm: float
    spectrum: Union[AnyStr, Spectrum]
    basefreq: float
    enabled: bool
    like: AnyStr

class TransformerProperties(TypedDict):
    phases: int
    windings: int
    pctR: Float64Array
    Rneut: Float64Array
    Xneut: Float64Array
    buses: List[AnyStr]
    conns: Union[List[Union[int,Connection]], List[AnyStr]]
    kVs: Float64Array
    kVAs: Float64Array
    taps: Float64Array
    XHL: float
    XHT: float
    XLT: float
    Xscarray: Float64Array
    thermal: float
    n: float
    m: float
    flrise: float
    hsrise: float
    pctloadloss: float
    pctnoloadloss: float
    normhkVA: float
    emerghkVA: float
    sub: bool
    MaxTap: Float64Array
    MinTap: Float64Array
    NumTaps: Int32Array
    subname: AnyStr
    pctimag: float
    ppm_antifloat: float
    pctRs: Float64Array
    bank: AnyStr
    xfmrcode: Union[AnyStr, XfmrCode]
    XRConst: bool
    X12: float
    X13: float
    X23: float
    LeadLag: Union[AnyStr, int, PhaseSequence]
    Core: Union[AnyStr, int, CoreType]
    RdcOhms: Float64Array
    Seasons: int
    Ratings: Float64Array
    normamps: float
    emergamps: float
    faultrate: float
    pctperm: float
    repair: float
    basefreq: float
    enabled: bool
    like: AnyStr

class CapacitorProperties(TypedDict):
    bus1: AnyStr
    bus2: AnyStr
    phases: int
    kvar: Float64Array
    kv: float
    conn: Union[AnyStr, int, Connection]
    cmatrix: Float64Array
    cuf: Float64Array
    R: Float64Array
    XL: Float64Array
    Harm: Float64Array
    Numsteps: int
    states: Int32Array
    normamps: float
    emergamps: float
    faultrate: float
    pctperm: float
    repair: float
    basefreq: float
    enabled: bool
    like: AnyStr

class ReactorProperties(TypedDict):
    bus1: AnyStr
    bus2: AnyStr
    phases: int
    kvar: float
    kv: float
    conn: Union[AnyStr, int, Connection]
    Rmatrix: Float64Array
    Xmatrix: Float64Array
    Parallel: bool
    R: float
    X: float
    Rp: float
    Z1: complex
    Z2: complex
    Z0: complex
    RCurve: Union[AnyStr, XYcurve]
    LCurve: Union[AnyStr, XYcurve]
    LmH: float
    normamps: float
    emergamps: float
    faultrate: float
    pctperm: float
    repair: float
    basefreq: float
    enabled: bool
    like: AnyStr

class CapControlProperties(TypedDict):
    element: Union[AnyStr, DSSObj]
    terminal: int
    capacitor: Union[AnyStr, Capacitor]
    type: Union[AnyStr, int, CapControl.CapControlType]
    PTratio: float
    CTratio: float
    ONsetting: float
    OFFsetting: float
    Delay: float
    VoltOverride: bool
    Vmax: float
    Vmin: float
    DelayOFF: float
    DeadTime: float
    CTPhase: Union[AnyStr, int, MonitoredPhase]
    PTPhase: Union[AnyStr, int, MonitoredPhase]
    VBus: AnyStr
    EventLog: bool
    UserModel: AnyStr
    UserData: AnyStr
    pctMinkvar: float
    Reset: bool
    basefreq: float
    enabled: bool
    like: AnyStr

class FaultProperties(TypedDict):
    bus1: AnyStr
    bus2: AnyStr
    phases: int
    r: float
    pctstddev: float
    Gmatrix: Float64Array
    ONtime: float
    temporary: bool
    MinAmps: float
    normamps: float
    emergamps: float
    faultrate: float
    pctperm: float
    repair: float
    basefreq: float
    enabled: bool
    like: AnyStr

class DynamicExpProperties(TypedDict):
    NVariables: int
    VarNames: List[AnyStr]
    var: AnyStr
    VarIdx: int
    Expression: AnyStr
    Domain: Union[AnyStr, int, DynamicExp.DynamicExpDomain]
    like: AnyStr

class GeneratorProperties(TypedDict):
    phases: int
    bus1: AnyStr
    kv: float
    kW: float
    pf: float
    kvar: float
    model: int
    Vminpu: float
    Vmaxpu: float
    yearly: Union[AnyStr, LoadShape]
    daily: Union[AnyStr, LoadShape]
    duty: Union[AnyStr, LoadShape]
    dispmode: Union[AnyStr, int, Generator.GeneratorDispatchMode]
    dispvalue: float
    conn: Union[AnyStr, int, Connection]
    status: Union[AnyStr, int, Generator.GeneratorStatus]
    cls: int
    Vpu: float
    maxkvar: float
    minkvar: float
    pvfactor: float
    forceon: bool
    kVA: float
    Xd: float
    Xdp: float
    Xdpp: float
    H: float
    D: float
    UserModel: AnyStr
    UserData: AnyStr
    ShaftModel: AnyStr
    ShaftData: AnyStr
    DutyStart: float
    debugtrace: bool
    Balanced: bool
    XRdp: float
    UseFuel: bool
    FuelkWh: float
    pctFuel: float
    pctReserve: float
    Refuel: bool
    DynamicEq: Union[AnyStr, DynamicExp]
    DynOut: AnyStr
    spectrum: Union[AnyStr, Spectrum]
    basefreq: float
    enabled: bool
    like: AnyStr

class GenDispatcherProperties(TypedDict):
    Element: Union[AnyStr, DSSObj]
    Terminal: int
    kWLimit: float
    kWBand: float
    kvarlimit: float
    GenList: List[AnyStr]
    Weights: Float64Array
    basefreq: float
    enabled: bool
    like: AnyStr

class StorageProperties(TypedDict):
    phases: int
    bus1: AnyStr
    kv: float
    conn: Union[AnyStr, int, Connection]
    kW: float
    kvar: float
    pf: float
    kVA: float
    pctCutin: float
    pctCutout: float
    EffCurve: Union[AnyStr, XYcurve]
    VarFollowInverter: bool
    kvarMax: float
    kvarMaxAbs: float
    WattPriority: bool
    PFPriority: bool
    pctPminNoVars: float
    pctPminkvarMax: float
    kWrated: float
    pctkWrated: float
    kWhrated: float
    kWhstored: float
    pctstored: float
    pctreserve: float
    State: Union[AnyStr, int, Storage.StorageState]
    pctDischarge: float
    pctCharge: float
    pctEffCharge: float
    pctEffDischarge: float
    pctIdlingkW: float
    pctR: float
    pctX: float
    model: int
    Vminpu: float
    Vmaxpu: float
    Balanced: bool
    LimitCurrent: bool
    yearly: Union[AnyStr, LoadShape]
    daily: Union[AnyStr, LoadShape]
    duty: Union[AnyStr, LoadShape]
    DispMode: Union[AnyStr, int, Storage.StorageDispatchMode]
    DischargeTrigger: float
    ChargeTrigger: float
    TimeChargeTrig: float
    cls: int
    DynaDLL: AnyStr
    DynaData: AnyStr
    UserModel: AnyStr
    UserData: AnyStr
    debugtrace: bool
    kVDC: float
    Kp: float
    PITol: float
    SafeVoltage: float
    SafeMode: bool
    DynamicEq: Union[AnyStr, DynamicExp]
    DynOut: AnyStr
    ControlMode: Union[AnyStr, int, InverterControlMode]
    spectrum: Union[AnyStr, Spectrum]
    basefreq: float
    enabled: bool
    like: AnyStr

class StorageControllerProperties(TypedDict):
    Element: Union[AnyStr, DSSObj]
    Terminal: int
    MonPhase: Union[AnyStr, int, MonitoredPhase]
    kWTarget: float
    kWTargetLow: float
    pctkWBand: float
    kWBand: float
    pctkWBandLow: float
    kWBandLow: float
    ElementList: List[AnyStr]
    Weights: Float64Array
    ModeDischarge: Union[AnyStr, int, StorageController.StorageControllerDischargemode]
    ModeCharge: Union[AnyStr, int, StorageController.StorageControllerChargemode]
    TimeDischargeTrigger: float
    TimeChargeTrigger: float
    pctRatekW: float
    pctRateCharge: float
    pctReserve: float
    kWhTotal: float
    kWTotal: float
    kWhActual: float
    kWActual: float
    kWneed: float
    Yearly: Union[AnyStr, LoadShape]
    Daily: Union[AnyStr, LoadShape]
    Duty: Union[AnyStr, LoadShape]
    EventLog: bool
    InhibitTime: int
    Tup: float
    TFlat: float
    Tdn: float
    kWThreshold: float
    DispFactor: float
    ResetLevel: float
    Seasons: int
    SeasonTargets: Float64Array
    SeasonTargetsLow: Float64Array
    basefreq: float
    enabled: bool
    like: AnyStr

class RelayProperties(TypedDict):
    MonitoredObj: Union[AnyStr, DSSObj]
    MonitoredTerm: int
    SwitchedObj: Union[AnyStr, DSSObj]
    SwitchedTerm: int
    type: Union[AnyStr, int, Relay.RelayType]
    Phasecurve: Union[AnyStr, TCC_Curve]
    Groundcurve: Union[AnyStr, TCC_Curve]
    PhaseTrip: float
    GroundTrip: float
    TDPhase: float
    TDGround: float
    PhaseInst: float
    GroundInst: float
    Reset: float
    Shots: int
    RecloseIntervals: Float64Array
    Delay: float
    Overvoltcurve: Union[AnyStr, TCC_Curve]
    Undervoltcurve: Union[AnyStr, TCC_Curve]
    kvbase: float
    pctPickup47: float
    BaseAmps46: float
    pctPickup46: float
    isqt46: float
    Variable: AnyStr
    overtrip: float
    undertrip: float
    Breakertime: float
    action: Union[AnyStr, int, Relay.RelayAction]
    Z1mag: float
    Z1ang: float
    Z0mag: float
    Z0ang: float
    Mphase: float
    Mground: float
    EventLog: bool
    DebugTrace: bool
    DistReverse: bool
    Normal: Union[AnyStr, int, Relay.RelayState]
    State: Union[AnyStr, int, Relay.RelayState]
    DOC_TiltAngleLow: float
    DOC_TiltAngleHigh: float
    DOC_TripSettingLow: float
    DOC_TripSettingHigh: float
    DOC_TripSettingMag: float
    DOC_DelayInner: float
    DOC_PhaseCurveInner: float
    DOC_PhaseTripInner: float
    DOC_TDPhaseInner: Union[AnyStr, TCC_Curve]
    DOC_P1Blocking: bool
    basefreq: float
    enabled: bool
    like: AnyStr

class RecloserProperties(TypedDict):
    MonitoredObj: Union[AnyStr, DSSObj]
    MonitoredTerm: int
    SwitchedObj: Union[AnyStr, DSSObj]
    SwitchedTerm: int
    NumFast: int
    PhaseFast: Union[AnyStr, TCC_Curve]
    PhaseDelayed: Union[AnyStr, TCC_Curve]
    GroundFast: Union[AnyStr, TCC_Curve]
    GroundDelayed: Union[AnyStr, TCC_Curve]
    PhaseTrip: float
    GroundTrip: float
    PhaseInst: float
    GroundInst: float
    Reset: float
    Shots: int
    RecloseIntervals: Float64Array
    Delay: float
    TDPhFast: float
    TDGrFast: float
    TDPhDelayed: float
    TDGrDelayed: float
    Normal: Union[AnyStr, int, Recloser.RecloserState]
    State: Union[AnyStr, int, Recloser.RecloserState]
    basefreq: float
    enabled: bool
    like: AnyStr

class FuseProperties(TypedDict):
    MonitoredObj: Union[AnyStr, DSSObj]
    MonitoredTerm: int
    SwitchedObj: Union[AnyStr, DSSObj]
    SwitchedTerm: int
    FuseCurve: Union[AnyStr, TCC_Curve]
    RatedCurrent: float
    Delay: float
    Action: Union[str, bytes, int, Fuse.FuseAction]
    Normal: Union[List[Union[int,Fuse.FuseState]], List[AnyStr]]
    State: Union[List[Union[int,Fuse.FuseState]], List[AnyStr]]
    basefreq: float
    enabled: bool
    like: AnyStr

class SwtControlProperties(TypedDict):
    SwitchedObj: Union[AnyStr, DSSObj]
    SwitchedTerm: int
    Lock: bool
    Delay: float
    Normal: Union[AnyStr, int, SwtControl.SwtControlState]
    State: Union[AnyStr, int, SwtControl.SwtControlState]
    Reset: bool
    basefreq: float
    enabled: bool
    like: AnyStr

class PVSystemProperties(TypedDict):
    phases: int
    bus1: AnyStr
    kv: float
    irradiance: float
    Pmpp: float
    pctPmpp: float
    Temperature: float
    pf: float
    conn: Union[AnyStr, int, Connection]
    kvar: float
    kVA: float
    pctCutin: float
    pctCutout: float
    EffCurve: Union[AnyStr, XYcurve]
    PTCurve: Union[AnyStr, XYcurve]
    pctR: float
    pctX: float
    model: int
    Vminpu: float
    Vmaxpu: float
    Balanced: bool
    LimitCurrent: bool
    yearly: Union[AnyStr, LoadShape]
    daily: Union[AnyStr, LoadShape]
    duty: Union[AnyStr, LoadShape]
    Tyearly: Union[AnyStr, TShape]
    Tdaily: Union[AnyStr, TShape]
    Tduty: Union[AnyStr, TShape]
    cls: int
    UserModel: AnyStr
    UserData: AnyStr
    debugtrace: bool
    VarFollowInverter: bool
    DutyStart: float
    WattPriority: bool
    PFPriority: bool
    pctPminNoVars: float
    pctPminkvarMax: float
    kvarMax: float
    kvarMaxAbs: float
    kVDC: float
    Kp: float
    PITol: float
    SafeVoltage: float
    SafeMode: bool
    DynamicEq: Union[AnyStr, DynamicExp]
    DynOut: AnyStr
    ControlMode: Union[AnyStr, int, InverterControlMode]
    spectrum: Union[AnyStr, Spectrum]
    basefreq: float
    enabled: bool
    like: AnyStr

class UPFCProperties(TypedDict):
    bus1: AnyStr
    bus2: AnyStr
    refkV: float
    PF: float
    Frequency: float
    Phases: int
    Xs: float
    Tol1: float
    Mode: int
    VpqMax: float
    LossCurve: Union[AnyStr, XYcurve]
    VHLimit: float
    VLLimit: float
    CLimit: float
    refkV2: float
    kvarLimit: float
    Element: Union[AnyStr, PDElement]
    spectrum: Union[AnyStr, Spectrum]
    basefreq: float
    enabled: bool
    like: AnyStr

class UPFCControlProperties(TypedDict):
    UPFCList: List[AnyStr]
    basefreq: float
    enabled: bool
    like: AnyStr

class ESPVLControlProperties(TypedDict):
    Element: Union[AnyStr, DSSObj]
    Terminal: int
    Type: Union[AnyStr, int, ESPVLControl.ESPVLControlType]
    kWBand: float
    kvarlimit: float
    LocalControlList: List[AnyStr]
    LocalControlWeights: Float64Array
    PVSystemList: List[AnyStr]
    PVSystemWeights: Float64Array
    StorageList: List[AnyStr]
    StorageWeights: Float64Array
    basefreq: float
    enabled: bool
    like: AnyStr

class IndMach012Properties(TypedDict):
    phases: int
    bus1: AnyStr
    kv: float
    kW: float
    pf: float
    conn: Union[AnyStr, int, Connection]
    kVA: float
    H: float
    D: float
    puRs: float
    puXs: float
    puRr: float
    puXr: float
    puXm: float
    Slip: float
    MaxSlip: float
    SlipOption: Union[AnyStr, int, IndMach012.IndMach012SlipOption]
    Yearly: Union[AnyStr, LoadShape]
    Daily: Union[AnyStr, LoadShape]
    Duty: Union[AnyStr, LoadShape]
    Debugtrace: bool
    spectrum: Union[AnyStr, Spectrum]
    basefreq: float
    enabled: bool
    like: AnyStr

class GICsourceProperties(TypedDict):
    Volts: float
    angle: float
    frequency: float
    phases: int
    EN: float
    EE: float
    Lat1: float
    Lon1: float
    Lat2: float
    Lon2: float
    spectrum: Union[AnyStr, Spectrum]
    basefreq: float
    enabled: bool
    like: AnyStr

class AutoTransProperties(TypedDict):
    phases: int
    windings: int
    pctR: Float64Array
    Rdcohms: Float64Array
    Core: Union[AnyStr, int, CoreType]
    buses: List[AnyStr]
    conns: Union[List[Union[int,AutoTrans.AutoTransConnection]], List[AnyStr]]
    kVs: Float64Array
    kVAs: Float64Array
    taps: Float64Array
    XHX: float
    XHT: float
    XXT: float
    XSCarray: Float64Array
    thermal: float
    n: float
    m: float
    flrise: float
    hsrise: float
    pctloadloss: float
    pctnoloadloss: float
    normhkVA: float
    emerghkVA: float
    sub: bool
    MaxTap: Float64Array
    MinTap: Float64Array
    NumTaps: Int32Array
    subname: AnyStr
    pctimag: float
    ppm_antifloat: float
    pctRs: Float64Array
    XRConst: bool
    LeadLag: Union[AnyStr, int, PhaseSequence]
    normamps: float
    emergamps: float
    faultrate: float
    pctperm: float
    repair: float
    basefreq: float
    enabled: bool
    like: AnyStr

class RegControlProperties(TypedDict):
    transformer: Union[AnyStr, Transformer, AutoTrans]
    winding: int
    vreg: float
    band: float
    ptratio: float
    CTprim: float
    R: float
    X: float
    bus: AnyStr
    delay: float
    reversible: bool
    revvreg: float
    revband: float
    revR: float
    revX: float
    tapdelay: float
    debugtrace: bool
    maxtapchange: int
    inversetime: bool
    tapwinding: int
    vlimit: float
    PTphase: Union[AnyStr, int, RegControl.RegControlPhaseSelection]
    revThreshold: float
    revDelay: float
    revNeutral: bool
    EventLog: bool
    RemotePTRatio: float
    TapNum: int
    Reset: bool
    LDC_Z: float
    rev_Z: float
    Cogen: bool
    basefreq: float
    enabled: bool
    like: AnyStr

class InvControlProperties(TypedDict):
    DERList: List[AnyStr]
    Mode: Union[AnyStr, int, InvControl.InvControlControlMode]
    CombiMode: Union[AnyStr, int, InvControl.InvControlCombiMode]
    vvc_curve1: Union[AnyStr, XYcurve]
    hysteresis_offset: float
    voltage_curvex_ref: Union[AnyStr, int, InvControl.InvControlVoltageCurveXRef]
    avgwindowlen: int
    voltwatt_curve: Union[AnyStr, XYcurve]
    DbVMin: float
    DbVMax: float
    ArGraLowV: float
    ArGraHiV: float
    DynReacavgwindowlen: int
    deltaQ_Factor: float
    VoltageChangeTolerance: float
    VarChangeTolerance: float
    VoltwattYAxis: Union[AnyStr, int, InvControl.InvControlVoltWattYAxis]
    RateofChangeMode: Union[AnyStr, int, InvControl.InvControlRateOfChangeMode]
    LPFTau: float
    RiseFallLimit: float
    deltaP_Factor: float
    EventLog: bool
    RefReactivePower: Union[AnyStr, int, InvControl.InvControlReactivePowerReference]
    ActivePChangeTolerance: float
    monVoltageCalc: Union[AnyStr, int, MonitoredPhase]
    monBus: List[AnyStr]
    MonBusesVbase: Float64Array
    voltwattCH_curve: Union[AnyStr, XYcurve]
    wattpf_curve: Union[AnyStr, XYcurve]
    wattvar_curve: Union[AnyStr, XYcurve]
    Vsetpoint: float
    ControlModel: Union[int, InvControl.InvControlControlModel]
    basefreq: float
    enabled: bool
    like: AnyStr

class ExpControlProperties(TypedDict):
    PVSystemList: List[AnyStr]
    Vreg: float
    Slope: float
    VregTau: float
    Qbias: float
    VregMin: float
    VregMax: float
    QmaxLead: float
    QmaxLag: float
    EventLog: bool
    DeltaQ_factor: float
    PreferQ: bool
    Tresponse: float
    DERList: List[AnyStr]
    basefreq: float
    enabled: bool
    like: AnyStr

class GICLineProperties(TypedDict):
    bus1: AnyStr
    bus2: AnyStr
    Volts: float
    Angle: float
    frequency: float
    phases: int
    R: float
    X: float
    C: float
    EN: float
    EE: float
    Lat1: float
    Lon1: float
    Lat2: float
    Lon2: float
    spectrum: Union[AnyStr, Spectrum]
    basefreq: float
    enabled: bool
    like: AnyStr

class GICTransformerProperties(TypedDict):
    BusH: AnyStr
    BusNH: AnyStr
    BusX: AnyStr
    BusNX: AnyStr
    phases: int
    Type: Union[AnyStr, int, GICTransformer.GICTransformerType]
    R1: float
    R2: float
    KVLL1: float
    KVLL2: float
    MVA: float
    VarCurve: Union[AnyStr, XYcurve]
    pctR1: float
    pctR2: float
    K: float
    normamps: float
    emergamps: float
    faultrate: float
    pctperm: float
    repair: float
    basefreq: float
    enabled: bool
    like: AnyStr

class VSConverterProperties(TypedDict):
    phases: int
    Bus1: AnyStr
    kVac: float
    kVdc: float
    kW: float
    Ndc: int
    Rac: float
    Xac: float
    m0: float
    d0: float
    Mmin: float
    Mmax: float
    Iacmax: float
    Idcmax: float
    Vacref: float
    Pacref: float
    Qacref: float
    Vdcref: float
    VscMode: Union[AnyStr, int, VSConverter.VSConverterControlMode]
    spectrum: Union[AnyStr, Spectrum]
    basefreq: float
    enabled: bool
    like: AnyStr

class MonitorProperties(TypedDict):
    element: Union[AnyStr, DSSObj]
    terminal: int
    mode: int
    action: Union[str, bytes, int, Monitor.MonitorAction]
    residual: bool
    VIPolar: bool
    PPolar: bool
    basefreq: float
    enabled: bool
    like: AnyStr

class EnergyMeterProperties(TypedDict):
    element: Union[AnyStr, DSSObj]
    terminal: int
    action: Union[str, bytes, int, EnergyMeter.EnergyMeterAction]
    option: List[AnyStr]
    kVAnormal: float
    kVAemerg: float
    peakcurrent: Float64Array
    Zonelist: List[AnyStr]
    LocalOnly: bool
    Mask: Float64Array
    Losses: bool
    LineLosses: bool
    XfmrLosses: bool
    SeqLosses: bool
    threePaseLosses: bool
    VbaseLosses: bool
    PhaseVoltageReport: bool
    Int_Rate: float
    Int_Duration: float
    SAIFI: float
    SAIFIkW: float
    SAIDI: float
    CAIDI: float
    CustInterrupts: float
    basefreq: float
    enabled: bool
    like: AnyStr

class SensorProperties(TypedDict):
    element: Union[AnyStr, DSSObj]
    terminal: int
    kvbase: float
    clear: bool
    kVs: Float64Array
    currents: Float64Array
    kWs: Float64Array
    kvars: Float64Array
    conn: Union[AnyStr, int, Connection]
    Deltadirection: int
    pctError: float
    Weight: float
    basefreq: float
    enabled: bool
    like: AnyStr

class LineCodeBatch(DSSBatch):
    _cls_name = 'LineCode'
    _obj_cls = LineCode
    _cls_idx = 1


    @property
    def nphases(self) -> BatchInt32ArrayProxy:
        """
        Number of phases in the line this line code data represents.  Setting this property reinitializes the line code.  Impedance matrix is reset for default symmetrical component.

        DSS property name: `nphases`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @nphases.setter
    def nphases(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def r1(self) -> BatchFloat64ArrayProxy:
        """
        Positive-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Rmatrix.

        DSS property name: `r1`, DSS property index: 2.
        """
        return BatchFloat64ArrayProxy(self, 2)

    @r1.setter
    def r1(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(2, value)

    @property
    def x1(self) -> BatchFloat64ArrayProxy:
        """
        Positive-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Xmatrix

        DSS property name: `x1`, DSS property index: 3.
        """
        return BatchFloat64ArrayProxy(self, 3)

    @x1.setter
    def x1(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(3, value)

    @property
    def r0(self) -> BatchFloat64ArrayProxy:
        """
        Zero-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.

        DSS property name: `r0`, DSS property index: 4.
        """
        return BatchFloat64ArrayProxy(self, 4)

    @r0.setter
    def r0(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(4, value)

    @property
    def x0(self) -> BatchFloat64ArrayProxy:
        """
        Zero-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.

        DSS property name: `x0`, DSS property index: 5.
        """
        return BatchFloat64ArrayProxy(self, 5)

    @x0.setter
    def x0(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(5, value)

    @property
    def C1(self) -> BatchFloat64ArrayProxy:
        """
        Positive-sequence capacitance, nf per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Cmatrix and B1.

        DSS property name: `C1`, DSS property index: 6.
        """
        return BatchFloat64ArrayProxy(self, 6)

    @C1.setter
    def C1(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(6, value)

    @property
    def C0(self) -> BatchFloat64ArrayProxy:
        """
        Zero-sequence capacitance, nf per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also B0.

        DSS property name: `C0`, DSS property index: 7.
        """
        return BatchFloat64ArrayProxy(self, 7)

    @C0.setter
    def C0(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(7, value)

    @property
    def units(self) -> BatchInt32ArrayProxy:
        """
        One of (ohms per ...) {none|mi|km|kft|m|me|ft|in|cm}.  Default is none; assumes units agree with length unitsgiven in Line object

        DSS property name: `units`, DSS property index: 8.
        """
        return BatchInt32ArrayProxy(self, 8)

    @units.setter
    def units(self, value: Union[AnyStr, int, DimensionUnits, List[AnyStr], List[int], List[DimensionUnits], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(8, value)
            return
    
        self._set_batch_int32_array(8, value)

    @property
    def units_str(self) -> str:
        """
        One of (ohms per ...) {none|mi|km|kft|m|me|ft|in|cm}.  Default is none; assumes units agree with length unitsgiven in Line object

        DSS property name: `units`, DSS property index: 8.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 8)

    @units_str.setter
    def units_str(self, value: AnyStr):
        self.units = value

    @property
    def rmatrix(self) -> List[Float64Array]:
        """
        Resistance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration.  For balanced line models, you may use the standard symmetrical component data definition instead.

        DSS property name: `rmatrix`, DSS property index: 9.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 9)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @rmatrix.setter
    def rmatrix(self, value: Float64Array):
        self._set_float64_array_o(9, value)

    @property
    def xmatrix(self) -> List[Float64Array]:
        """
        Reactance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration.  For balanced line models, you may use the standard symmetrical component data definition instead.

        DSS property name: `xmatrix`, DSS property index: 10.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 10)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @xmatrix.setter
    def xmatrix(self, value: Float64Array):
        self._set_float64_array_o(10, value)

    @property
    def cmatrix(self) -> List[Float64Array]:
        """
        Nodal Capacitance matrix, lower triangle, nf per unit length.Order of the matrix is the number of phases. May be used to specify the shunt capacitance of any line configuration.  For balanced line models, you may use the standard symmetrical component data definition instead.

        DSS property name: `cmatrix`, DSS property index: 11.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 11)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @cmatrix.setter
    def cmatrix(self, value: Float64Array):
        self._set_float64_array_o(11, value)

    @property
    def baseFreq(self) -> BatchFloat64ArrayProxy:
        """
        Frequency at which impedances are specified.

        DSS property name: `baseFreq`, DSS property index: 12.
        """
        return BatchFloat64ArrayProxy(self, 12)

    @baseFreq.setter
    def baseFreq(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(12, value)

    @property
    def normamps(self) -> BatchFloat64ArrayProxy:
        """
        Normal ampere limit on line.  This is the so-called Planning Limit. It may also be the value above which load will have to be dropped in a contingency.  Usually about 75% - 80% of the emergency (one-hour) rating.

        DSS property name: `normamps`, DSS property index: 13.
        """
        return BatchFloat64ArrayProxy(self, 13)

    @normamps.setter
    def normamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(13, value)

    @property
    def emergamps(self) -> BatchFloat64ArrayProxy:
        """
        Emergency ampere limit on line (usually one-hour rating).

        DSS property name: `emergamps`, DSS property index: 14.
        """
        return BatchFloat64ArrayProxy(self, 14)

    @emergamps.setter
    def emergamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(14, value)

    @property
    def faultrate(self) -> BatchFloat64ArrayProxy:
        """
        Number of faults per unit length per year.

        DSS property name: `faultrate`, DSS property index: 15.
        """
        return BatchFloat64ArrayProxy(self, 15)

    @faultrate.setter
    def faultrate(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(15, value)

    @property
    def pctperm(self) -> BatchFloat64ArrayProxy:
        """
        Percentage of the faults that become permanent.

        DSS property name: `pctperm`, DSS property index: 16.
        """
        return BatchFloat64ArrayProxy(self, 16)

    @pctperm.setter
    def pctperm(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(16, value)

    @property
    def repair(self) -> BatchFloat64ArrayProxy:
        """
        Hours to repair.

        DSS property name: `repair`, DSS property index: 17.
        """
        return BatchFloat64ArrayProxy(self, 17)

    @repair.setter
    def repair(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(17, value)

    def Kron(self, value: bool):
        """
        Kron = Y/N. Default=N.  Perform Kron reduction on the impedance matrix after it is formed, reducing order by 1. Eliminates the conductor designated by the "Neutral=" property. Do this after the R, X, and C matrices are defined. Ignored for symmetrical components. May be issued more than once to eliminate more than one conductor by resetting the Neutral property after the previous invoking of this property. Generally, you do not want to do a Kron reduction on the matrix if you intend to solve at a frequency other than the base frequency and exploit the Rg and Xg values.

        DSS property name: `Kron`, DSS property index: 18.
        """
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 18, value)

    @property
    def Rg(self) -> BatchFloat64ArrayProxy:
        """
        Carson earth return resistance per unit length used to compute impedance values at base frequency.  For making better frequency adjustments. Default is 0.01805 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.

        DSS property name: `Rg`, DSS property index: 19.
        """
        return BatchFloat64ArrayProxy(self, 19)

    @Rg.setter
    def Rg(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(19, value)

    @property
    def Xg(self) -> BatchFloat64ArrayProxy:
        """
        Carson earth return reactance per unit length used to compute impedance values at base frequency.  For making better frequency adjustments. Default value is 0.155081 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.

        DSS property name: `Xg`, DSS property index: 20.
        """
        return BatchFloat64ArrayProxy(self, 20)

    @Xg.setter
    def Xg(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(20, value)

    @property
    def rho(self) -> BatchFloat64ArrayProxy:
        """
        Default=100 meter ohms.  Earth resitivity used to compute earth correction factor.

        DSS property name: `rho`, DSS property index: 21.
        """
        return BatchFloat64ArrayProxy(self, 21)

    @rho.setter
    def rho(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(21, value)

    @property
    def neutral(self) -> BatchInt32ArrayProxy:
        """
        Designates which conductor is the "neutral" conductor that will be eliminated by Kron reduction. Default is the last conductor (nphases value). After Kron reduction is set to 0. Subsequent issuing of Kron=Yes will not do anything until this property is set to a legal value. Applies only to LineCodes defined by R, X, and C matrix.

        DSS property name: `neutral`, DSS property index: 22.
        """
        return BatchInt32ArrayProxy(self, 22)

    @neutral.setter
    def neutral(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(22, value)

    @property
    def B1(self) -> BatchFloat64ArrayProxy:
        """
        Alternate way to specify C1. MicroS per unit length

        DSS property name: `B1`, DSS property index: 23.
        """
        return BatchFloat64ArrayProxy(self, 23)

    @B1.setter
    def B1(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(23, value)

    @property
    def B0(self) -> BatchFloat64ArrayProxy:
        """
        Alternate way to specify C0. MicroS per unit length

        DSS property name: `B0`, DSS property index: 24.
        """
        return BatchFloat64ArrayProxy(self, 24)

    @B0.setter
    def B0(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(24, value)

    @property
    def Seasons(self) -> BatchInt32ArrayProxy:
        """
        Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 25.
        """
        return BatchInt32ArrayProxy(self, 25)

    @Seasons.setter
    def Seasons(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(25, value)

    @property
    def Ratings(self) -> List[Float64Array]:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.

        DSS property name: `Ratings`, DSS property index: 26.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 26)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(26, value)

    @property
    def linetype(self) -> BatchInt32ArrayProxy:
        """
        Code designating the type of line. 
        One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW

        OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.

        DSS property name: `LineType`, DSS property index: 27.
        """
        return BatchInt32ArrayProxy(self, 27)

    @linetype.setter
    def linetype(self, value: Union[AnyStr, int, LineType, List[AnyStr], List[int], List[LineType], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(27, value)
            return
    
        self._set_batch_int32_array(27, value)

    @property
    def linetype_str(self) -> str:
        """
        Code designating the type of line. 
        One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW

        OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.

        DSS property name: `LineType`, DSS property index: 27.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 27)

    @linetype_str.setter
    def linetype_str(self, value: AnyStr):
        self.linetype = value

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 28.
        """
        self._set_batch_string(28, value)

class LoadShapeBatch(DSSBatch):
    _cls_name = 'LoadShape'
    _obj_cls = LoadShape
    _cls_idx = 2


    @property
    def npts(self) -> BatchInt32ArrayProxy:
        """
        Max number of points to expect in load shape vectors. This gets reset to the number of multiplier values found (in files only) if less than specified.

        DSS property name: `npts`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @npts.setter
    def npts(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def interval(self) -> BatchFloat64ArrayProxy:
        """
        Time interval for fixed interval data, hrs. Default = 1. If Interval = 0 then time data (in hours) may be at either regular or  irregular intervals and time value must be specified using either the Hour property or input files. Then values are interpolated when Interval=0, but not for fixed interval data.  

        See also "sinterval" and "minterval".

        DSS property name: `interval`, DSS property index: 2.
        """
        return BatchFloat64ArrayProxy(self, 2)

    @interval.setter
    def interval(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(2, value)

    @property
    def hour(self) -> List[Float64Array]:
        """
        Array of hour values. Only necessary to define for variable interval data (Interval=0). If you set Interval>0 to denote fixed interval data, DO NOT USE THIS PROPERTY. You can also use the syntax: 
        hour = (file=filename)     !for text file one value per line
        hour = (dblfile=filename)  !for packed file of doubles
        hour = (sngfile=filename)  !for packed file of singles 

        DSS property name: `hour`, DSS property index: 4.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 4)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @hour.setter
    def hour(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def mean(self) -> BatchFloat64ArrayProxy:
        """
        Mean of the active power multipliers.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently. Used for Monte Carlo load simulations.

        DSS property name: `mean`, DSS property index: 5.
        """
        return BatchFloat64ArrayProxy(self, 5)

    @mean.setter
    def mean(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(5, value)

    @property
    def stddev(self) -> BatchFloat64ArrayProxy:
        """
        Standard deviation of active power multipliers.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently.Is overwritten if you subsequently read in a curve

        Used for Monte Carlo load simulations.

        DSS property name: `stddev`, DSS property index: 6.
        """
        return BatchFloat64ArrayProxy(self, 6)

    @stddev.setter
    def stddev(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(6, value)

    @property
    def csvfile(self) -> List[str]:
        """
        Switch input of active power load curve data to a CSV text file containing (hour, mult) points, or simply (mult) values for fixed time interval data, one per line. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `csvfile`, DSS property index: 7.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 7) 

    @csvfile.setter
    def csvfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(7, value)

    @property
    def sngfile(self) -> List[str]:
        """
        Switch input of active power load curve data to a binary file of singles containing (hour, mult) points, or simply (mult) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `sngfile`, DSS property index: 8.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 8) 

    @sngfile.setter
    def sngfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(8, value)

    @property
    def dblfile(self) -> List[str]:
        """
        Switch input of active power load curve data to a binary file of doubles containing (hour, mult) points, or simply (mult) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `dblfile`, DSS property index: 9.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 9) 

    @dblfile.setter
    def dblfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(9, value)

    def action(self, value: Union[str, bytes, int, LoadShape.LoadShapeAction]):
        """
        {NORMALIZE | DblSave | SngSave} After defining load curve data, setting action=normalize will modify the multipliers so that the peak is 1.0. The mean and std deviation are recomputed.

        Setting action=DblSave or SngSave will cause the present mult and qmult values to be written to either a packed file of double or single. The filename is the loadshape name. The mult array will have a "_P" appended on the file name and the qmult array, if it exists, will have "_Q" appended.

        DSS property name: `action`, DSS property index: 10.
        """
        if isinstance(value, int):
            self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 10, value)
        else:
            self._set_batch_string(10, value)

    @property
    def qmult(self) -> List[Float64Array]:
        """
        Array of multiplier values for reactive power (Q).  You can also use the syntax: 
        qmult = (file=filename)     !for text file one value per line
        qmult = (dblfile=filename)  !for packed file of doubles
        qmult = (sngfile=filename)  !for packed file of singles 
        qmult = (file=MyCSVFile.csv, col=4, header=yes)  !for multicolumn CSV files 

        DSS property name: `qmult`, DSS property index: 11.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 11)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @qmult.setter
    def qmult(self, value: Float64Array):
        self._set_float64_array_o(11, value)

    @property
    def UseActual(self) -> List[bool]:
        """
        {Yes | No* | True | False*} If true, signifies to Load, Generator, Vsource, or other objects to use the return value as the actual kW, kvar, kV, or other value rather than a multiplier. Nominally for AMI Load data but may be used for other functions.

        DSS property name: `UseActual`, DSS property index: 12.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 12)
        ]
    @UseActual.setter
    def UseActual(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 12, value)

    @property
    def Pmax(self) -> BatchFloat64ArrayProxy:
        """
        kW value at the time of max power. Is automatically set upon reading in a loadshape. Use this property to override the value automatically computed or to retrieve the value computed.

        DSS property name: `Pmax`, DSS property index: 13.
        """
        return BatchFloat64ArrayProxy(self, 13)

    @Pmax.setter
    def Pmax(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(13, value)

    @property
    def Qmax(self) -> BatchFloat64ArrayProxy:
        """
        kvar value at the time of max kW power. Is automatically set upon reading in a loadshape. Use this property to override the value automatically computed or to retrieve the value computed.

        DSS property name: `Qmax`, DSS property index: 14.
        """
        return BatchFloat64ArrayProxy(self, 14)

    @Qmax.setter
    def Qmax(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(14, value)

    @property
    def sinterval(self) -> BatchFloat64ArrayProxy:
        """
        Specify fixed interval in SECONDS. Alternate way to specify Interval property.

        DSS property name: `sinterval`, DSS property index: 15.
        """
        return BatchFloat64ArrayProxy(self, 15)

    @sinterval.setter
    def sinterval(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(15, value)

    @property
    def minterval(self) -> BatchFloat64ArrayProxy:
        """
        Specify fixed interval in MINUTES. Alternate way to specify Interval property.

        DSS property name: `minterval`, DSS property index: 16.
        """
        return BatchFloat64ArrayProxy(self, 16)

    @minterval.setter
    def minterval(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(16, value)

    @property
    def Pbase(self) -> BatchFloat64ArrayProxy:
        """
        Base P value for normalization. Default is zero, meaning the peak will be used.

        DSS property name: `Pbase`, DSS property index: 17.
        """
        return BatchFloat64ArrayProxy(self, 17)

    @Pbase.setter
    def Pbase(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(17, value)

    @property
    def Qbase(self) -> BatchFloat64ArrayProxy:
        """
        Base Q value for normalization. Default is zero, meaning the peak will be used.

        DSS property name: `Qbase`, DSS property index: 18.
        """
        return BatchFloat64ArrayProxy(self, 18)

    @Qbase.setter
    def Qbase(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(18, value)

    @property
    def Pmult(self) -> List[Float64Array]:
        """
        Synonym for "mult".

        DSS property name: `Pmult`, DSS property index: 19.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 19)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Pmult.setter
    def Pmult(self, value: Float64Array):
        self._set_float64_array_o(19, value)

    @property
    def PQCSVFile(self) -> List[str]:
        """
        Switch input to a CSV text file containing (active, reactive) power (P, Q) multiplier pairs, one per row. 
        If the interval=0, there should be 3 items on each line: (hour, Pmult, Qmult)

        DSS property name: `PQCSVFile`, DSS property index: 20.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 20) 

    @PQCSVFile.setter
    def PQCSVFile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(20, value)

    @property
    def MemoryMapping(self) -> List[bool]:
        """
        {Yes | No* | True | False*} Enables the memory mapping functionality for dealing with large amounts of load shapes. 
        By defaul is False. Use it to accelerate the model loading when the containing a large number of load shapes.

        DSS property name: `MemoryMapping`, DSS property index: 21.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 21)
        ]
    @MemoryMapping.setter
    def MemoryMapping(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 21, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 22.
        """
        self._set_batch_string(22, value)

class TShapeBatch(DSSBatch):
    _cls_name = 'TShape'
    _obj_cls = TShape
    _cls_idx = 3


    @property
    def npts(self) -> BatchInt32ArrayProxy:
        """
        Max number of points to expect in temperature shape vectors. This gets reset to the number of Temperature values found if less than specified.

        DSS property name: `npts`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @npts.setter
    def npts(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def interval(self) -> BatchFloat64ArrayProxy:
        """
        Time interval for fixed interval data, hrs. Default = 1. If Interval = 0 then time data (in hours) may be at irregular intervals and time value must be specified using either the Hour property or input files. Then values are interpolated when Interval=0, but not for fixed interval data.  

        See also "sinterval" and "minterval".

        DSS property name: `interval`, DSS property index: 2.
        """
        return BatchFloat64ArrayProxy(self, 2)

    @interval.setter
    def interval(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(2, value)

    @property
    def temp(self) -> List[Float64Array]:
        """
        Array of temperature values.  Units should be compatible with the object using the data. You can also use the syntax: 
        Temp = (file=filename)     !for text file one value per line
        Temp = (dblfile=filename)  !for packed file of doubles
        Temp = (sngfile=filename)  !for packed file of singles 

        Note: this property will reset Npts if the  number of values in the files are fewer.

        DSS property name: `temp`, DSS property index: 3.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 3)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @temp.setter
    def temp(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    @property
    def hour(self) -> List[Float64Array]:
        """
        Array of hour values. Only necessary to define this property for variable interval data. If the data are fixed interval, do not use this property. You can also use the syntax: 
        hour = (file=filename)     !for text file one value per line
        hour = (dblfile=filename)  !for packed file of doubles
        hour = (sngfile=filename)  !for packed file of singles 

        DSS property name: `hour`, DSS property index: 4.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 4)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @hour.setter
    def hour(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def mean(self) -> BatchFloat64ArrayProxy:
        """
        Mean of the temperature curve values.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently. Used for Monte Carlo load simulations.

        DSS property name: `mean`, DSS property index: 5.
        """
        return BatchFloat64ArrayProxy(self, 5)

    @mean.setter
    def mean(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(5, value)

    @property
    def stddev(self) -> BatchFloat64ArrayProxy:
        """
        Standard deviation of the temperatures.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently.Is overwritten if you subsequently read in a curve

        Used for Monte Carlo load simulations.

        DSS property name: `stddev`, DSS property index: 6.
        """
        return BatchFloat64ArrayProxy(self, 6)

    @stddev.setter
    def stddev(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(6, value)

    @property
    def csvfile(self) -> List[str]:
        """
        Switch input of  temperature curve data to a csv file containing (hour, Temp) points, or simply (Temp) values for fixed time interval data, one per line. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `csvfile`, DSS property index: 7.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 7) 

    @csvfile.setter
    def csvfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(7, value)

    @property
    def sngfile(self) -> List[str]:
        """
        Switch input of  temperature curve data to a binary file of singles containing (hour, Temp) points, or simply (Temp) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `sngfile`, DSS property index: 8.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 8) 

    @sngfile.setter
    def sngfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(8, value)

    @property
    def dblfile(self) -> List[str]:
        """
        Switch input of  temperature curve data to a binary file of doubles containing (hour, Temp) points, or simply (Temp) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `dblfile`, DSS property index: 9.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 9) 

    @dblfile.setter
    def dblfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(9, value)

    @property
    def sinterval(self) -> BatchFloat64ArrayProxy:
        """
        Specify fixed interval in SECONDS. Alternate way to specify Interval property.

        DSS property name: `sinterval`, DSS property index: 10.
        """
        return BatchFloat64ArrayProxy(self, 10)

    @sinterval.setter
    def sinterval(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(10, value)

    @property
    def minterval(self) -> BatchFloat64ArrayProxy:
        """
        Specify fixed interval in MINUTES. Alternate way to specify Interval property.

        DSS property name: `minterval`, DSS property index: 11.
        """
        return BatchFloat64ArrayProxy(self, 11)

    @minterval.setter
    def minterval(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(11, value)

    def action(self, value: Union[str, bytes, int, TShape.TShapeAction]):
        """
        {DblSave | SngSave} After defining temperature curve data... Setting action=DblSave or SngSave will cause the present "Temp" values to be written to either a packed file of double or single. The filename is the Tshape name. 

        DSS property name: `action`, DSS property index: 12.
        """
        if isinstance(value, int):
            self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 12, value)
        else:
            self._set_batch_string(12, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 13.
        """
        self._set_batch_string(13, value)

class PriceShapeBatch(DSSBatch):
    _cls_name = 'PriceShape'
    _obj_cls = PriceShape
    _cls_idx = 4


    @property
    def npts(self) -> BatchInt32ArrayProxy:
        """
        Max number of points to expect in price shape vectors. This gets reset to the number of Price values found if less than specified.

        DSS property name: `npts`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @npts.setter
    def npts(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def interval(self) -> BatchFloat64ArrayProxy:
        """
        Time interval for fixed interval data, hrs. Default = 1. If Interval = 0 then time data (in hours) may be at irregular intervals and time value must be specified using either the Hour property or input files. Then values are interpolated when Interval=0, but not for fixed interval data.  

        See also "sinterval" and "minterval".

        DSS property name: `interval`, DSS property index: 2.
        """
        return BatchFloat64ArrayProxy(self, 2)

    @interval.setter
    def interval(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(2, value)

    @property
    def price(self) -> List[Float64Array]:
        """
        Array of Price values.  Units should be compatible with the object using the data. You can also use the syntax: 
        Price = (file=filename)     !for text file one value per line
        Price = (dblfile=filename)  !for packed file of doubles
        Price = (sngfile=filename)  !for packed file of singles 

        Note: this property will reset Npts if the  number of values in the files are fewer.

        DSS property name: `price`, DSS property index: 3.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 3)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @price.setter
    def price(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    @property
    def hour(self) -> List[Float64Array]:
        """
        Array of hour values. Only necessary to define this property for variable interval data. If the data are fixed interval, do not use this property. You can also use the syntax: 
        hour = (file=filename)     !for text file one value per line
        hour = (dblfile=filename)  !for packed file of doubles
        hour = (sngfile=filename)  !for packed file of singles 

        DSS property name: `hour`, DSS property index: 4.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 4)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @hour.setter
    def hour(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def mean(self) -> BatchFloat64ArrayProxy:
        """
        Mean of the Price curve values.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently. Used for Monte Carlo load simulations.

        DSS property name: `mean`, DSS property index: 5.
        """
        return BatchFloat64ArrayProxy(self, 5)

    @mean.setter
    def mean(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(5, value)

    @property
    def stddev(self) -> BatchFloat64ArrayProxy:
        """
        Standard deviation of the Prices.  This is computed on demand the first time a value is needed.  However, you may set it to another value independently.Is overwritten if you subsequently read in a curve

        Used for Monte Carlo load simulations.

        DSS property name: `stddev`, DSS property index: 6.
        """
        return BatchFloat64ArrayProxy(self, 6)

    @stddev.setter
    def stddev(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(6, value)

    @property
    def csvfile(self) -> List[str]:
        """
        Switch input of  Price curve data to a csv file containing (hour, Price) points, or simply (Price) values for fixed time interval data, one per line. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `csvfile`, DSS property index: 7.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 7) 

    @csvfile.setter
    def csvfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(7, value)

    @property
    def sngfile(self) -> List[str]:
        """
        Switch input of  Price curve data to a binary file of singles containing (hour, Price) points, or simply (Price) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `sngfile`, DSS property index: 8.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 8) 

    @sngfile.setter
    def sngfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(8, value)

    @property
    def dblfile(self) -> List[str]:
        """
        Switch input of  Price curve data to a binary file of doubles containing (hour, Price) points, or simply (Price) values for fixed time interval data, packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `dblfile`, DSS property index: 9.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 9) 

    @dblfile.setter
    def dblfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(9, value)

    @property
    def sinterval(self) -> BatchFloat64ArrayProxy:
        """
        Specify fixed interval in SECONDS. Alternate way to specify Interval property.

        DSS property name: `sinterval`, DSS property index: 10.
        """
        return BatchFloat64ArrayProxy(self, 10)

    @sinterval.setter
    def sinterval(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(10, value)

    @property
    def minterval(self) -> BatchFloat64ArrayProxy:
        """
        Specify fixed interval in MINUTES. Alternate way to specify Interval property.

        DSS property name: `minterval`, DSS property index: 11.
        """
        return BatchFloat64ArrayProxy(self, 11)

    @minterval.setter
    def minterval(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(11, value)

    def action(self, value: Union[str, bytes, int, PriceShape.PriceShapeAction]):
        """
        {DblSave | SngSave} After defining Price curve data... Setting action=DblSave or SngSave will cause the present "Price" values to be written to either a packed file of double or single. The filename is the PriceShape name. 

        DSS property name: `action`, DSS property index: 12.
        """
        if isinstance(value, int):
            self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 12, value)
        else:
            self._set_batch_string(12, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 13.
        """
        self._set_batch_string(13, value)

class XYcurveBatch(DSSBatch):
    _cls_name = 'XYcurve'
    _obj_cls = XYcurve
    _cls_idx = 5


    @property
    def npts(self) -> BatchInt32ArrayProxy:
        """
        Max number of points to expect in curve. This could get reset to the actual number of points defined if less than specified.

        DSS property name: `npts`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @npts.setter
    def npts(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def Yarray(self) -> List[Float64Array]:
        """
        Alternate way to enter Y values. Enter an array of Y values corresponding to the X values.  You can also use the syntax: 
        Yarray = (file=filename)     !for text file one value per line
        Yarray = (dblfile=filename)  !for packed file of doubles
        Yarray = (sngfile=filename)  !for packed file of singles 

        Note: this property will reset Npts to a smaller value if the  number of values in the files are fewer.

        DSS property name: `Yarray`, DSS property index: 3.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 3)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Yarray.setter
    def Yarray(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    @property
    def Xarray(self) -> List[Float64Array]:
        """
        Alternate way to enter X values. Enter an array of X values corresponding to the Y values.  You can also use the syntax: 
        Xarray = (file=filename)     !for text file one value per line
        Xarray = (dblfile=filename)  !for packed file of doubles
        Xarray = (sngfile=filename)  !for packed file of singles 

        Note: this property will reset Npts to a smaller value if the  number of values in the files are fewer.

        DSS property name: `Xarray`, DSS property index: 4.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 4)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Xarray.setter
    def Xarray(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def csvfile(self) -> List[str]:
        """
        Switch input of  X-Y curve data to a CSV file containing X, Y points one per line. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `csvfile`, DSS property index: 5.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 5) 

    @csvfile.setter
    def csvfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(5, value)

    @property
    def sngfile(self) -> List[str]:
        """
        Switch input of  X-Y curve data to a binary file of SINGLES containing X, Y points packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `sngfile`, DSS property index: 6.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 6) 

    @sngfile.setter
    def sngfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(6, value)

    @property
    def dblfile(self) -> List[str]:
        """
        Switch input of  X-Y  curve data to a binary file of DOUBLES containing X, Y points packed one after another. NOTE: This action may reset the number of points to a lower value.

        DSS property name: `dblfile`, DSS property index: 7.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 7) 

    @dblfile.setter
    def dblfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(7, value)

    @property
    def x(self) -> BatchFloat64ArrayProxy:
        """
        Enter a value and then retrieve the interpolated Y value from the Y property. On input shifted then scaled to original curve. Scaled then shifted on output.

        DSS property name: `x`, DSS property index: 8.
        """
        return BatchFloat64ArrayProxy(self, 8)

    @x.setter
    def x(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(8, value)

    @property
    def y(self) -> BatchFloat64ArrayProxy:
        """
        Enter a value and then retrieve the interpolated X value from the X property. On input shifted then scaled to original curve. Scaled then shifted on output.

        DSS property name: `y`, DSS property index: 9.
        """
        return BatchFloat64ArrayProxy(self, 9)

    @y.setter
    def y(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(9, value)

    @property
    def Xshift(self) -> BatchFloat64ArrayProxy:
        """
        Shift X property values (in/out) by this amount of offset. Default = 0. Does not change original definition of arrays.

        DSS property name: `Xshift`, DSS property index: 10.
        """
        return BatchFloat64ArrayProxy(self, 10)

    @Xshift.setter
    def Xshift(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(10, value)

    @property
    def Yshift(self) -> BatchFloat64ArrayProxy:
        """
        Shift Y property values (in/out) by this amount of offset. Default = 0. Does not change original definition of arrays.

        DSS property name: `Yshift`, DSS property index: 11.
        """
        return BatchFloat64ArrayProxy(self, 11)

    @Yshift.setter
    def Yshift(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(11, value)

    @property
    def Xscale(self) -> BatchFloat64ArrayProxy:
        """
        Scale X property values (in/out) by this factor. Default = 1.0. Does not change original definition of arrays.

        DSS property name: `Xscale`, DSS property index: 12.
        """
        return BatchFloat64ArrayProxy(self, 12)

    @Xscale.setter
    def Xscale(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(12, value)

    @property
    def Yscale(self) -> BatchFloat64ArrayProxy:
        """
        Scale Y property values (in/out) by this factor. Default = 1.0. Does not change original definition of arrays.

        DSS property name: `Yscale`, DSS property index: 13.
        """
        return BatchFloat64ArrayProxy(self, 13)

    @Yscale.setter
    def Yscale(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(13, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 14.
        """
        self._set_batch_string(14, value)

class GrowthShapeBatch(DSSBatch):
    _cls_name = 'GrowthShape'
    _obj_cls = GrowthShape
    _cls_idx = 6


    @property
    def npts(self) -> BatchInt32ArrayProxy:
        """
        Number of points to expect in subsequent vector.

        DSS property name: `npts`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @npts.setter
    def npts(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def year(self) -> List[Float64Array]:
        """
        Array of year values, or a text file spec, corresponding to the multipliers. Enter only those years where the growth changes. May be any integer sequence -- just so it is consistent. See help on Mult.

        DSS property name: `year`, DSS property index: 2.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 2)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @year.setter
    def year(self, value: Float64Array):
        self._set_float64_array_o(2, value)

    @property
    def mult(self) -> List[Float64Array]:
        """
        Array of growth multiplier values, or a text file spec, corresponding to the year values. Enter the multiplier by which you would multiply the previous year's load to get the present year's.

        Examples:

          Year = [1, 2, 5]   Mult=[1.05, 1.025, 1.02].
          Year= (File=years.txt) Mult= (file=mults.txt).

        Text files contain one value per line.

        DSS property name: `mult`, DSS property index: 3.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 3)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @mult.setter
    def mult(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    @property
    def csvfile(self) -> List[str]:
        """
        Switch input of growth curve data to a csv file containing (year, mult) points, one per line.

        DSS property name: `csvfile`, DSS property index: 4.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 4) 

    @csvfile.setter
    def csvfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(4, value)

    @property
    def sngfile(self) -> List[str]:
        """
        Switch input of growth curve data to a binary file of singles containing (year, mult) points, packed one after another.

        DSS property name: `sngfile`, DSS property index: 5.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 5) 

    @sngfile.setter
    def sngfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(5, value)

    @property
    def dblfile(self) -> List[str]:
        """
        Switch input of growth curve data to a binary file of doubles containing (year, mult) points, packed one after another.

        DSS property name: `dblfile`, DSS property index: 6.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 6) 

    @dblfile.setter
    def dblfile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(6, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 7.
        """
        self._set_batch_string(7, value)

class TCC_CurveBatch(DSSBatch):
    _cls_name = 'TCC_Curve'
    _obj_cls = TCC_Curve
    _cls_idx = 7


    @property
    def npts(self) -> BatchInt32ArrayProxy:
        """
        Number of points to expect in time-current arrays.

        DSS property name: `npts`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @npts.setter
    def npts(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def C_array(self) -> List[Float64Array]:
        """
        Array of current (or voltage) values corresponding to time values (see help on T_Array).

        DSS property name: `C_array`, DSS property index: 2.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 2)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @C_array.setter
    def C_array(self, value: Float64Array):
        self._set_float64_array_o(2, value)

    @property
    def T_array(self) -> List[Float64Array]:
        """
        Array of time values in sec. Typical array syntax: 
        t_array = (1, 2, 3, 4, ...)

        Can also substitute a file designation: 
        t_array =  (file=filename)

        The specified file has one value per line.

        DSS property name: `T_array`, DSS property index: 3.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 3)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @T_array.setter
    def T_array(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 4.
        """
        self._set_batch_string(4, value)

class SpectrumBatch(DSSBatch):
    _cls_name = 'Spectrum'
    _obj_cls = Spectrum
    _cls_idx = 8


    @property
    def NumHarm(self) -> BatchInt32ArrayProxy:
        """
        Number of frequencies in this spectrum. (See CSVFile)

        DSS property name: `NumHarm`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @NumHarm.setter
    def NumHarm(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def harmonic(self) -> List[Float64Array]:
        """
        Array of harmonic values. You can also use the syntax
        harmonic = (file=filename)     !for text file one value per line
        harmonic = (dblfile=filename)  !for packed file of doubles
        harmonic = (sngfile=filename)  !for packed file of singles 

        DSS property name: `harmonic`, DSS property index: 2.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 2)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @harmonic.setter
    def harmonic(self, value: Float64Array):
        self._set_float64_array_o(2, value)

    @property
    def pctmag(self) -> List[Float64Array]:
        """
        Array of magnitude values, assumed to be in PERCENT. You can also use the syntax
        %mag = (file=filename)     !for text file one value per line
        %mag = (dblfile=filename)  !for packed file of doubles
        %mag = (sngfile=filename)  !for packed file of singles 

        DSS property name: `%mag`, DSS property index: 3.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 3)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @pctmag.setter
    def pctmag(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    @property
    def angle(self) -> List[Float64Array]:
        """
        Array of phase angle values, degrees.You can also use the syntax
        angle = (file=filename)     !for text file one value per line
        angle = (dblfile=filename)  !for packed file of doubles
        angle = (sngfile=filename)  !for packed file of singles 

        DSS property name: `angle`, DSS property index: 4.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 4)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @angle.setter
    def angle(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def CSVFile(self) -> List[str]:
        """
        File of spectrum points with (harmonic, magnitude-percent, angle-degrees) values, one set of 3 per line, in CSV format. If fewer than NUMHARM frequencies found in the file, NUMHARM is set to the smaller value.

        DSS property name: `CSVFile`, DSS property index: 5.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 5) 

    @CSVFile.setter
    def CSVFile(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(5, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 6.
        """
        self._set_batch_string(6, value)

class WireDataBatch(DSSBatch):
    _cls_name = 'WireData'
    _obj_cls = WireData
    _cls_idx = 9


    @property
    def Rdc(self) -> BatchFloat64ArrayProxy:
        """
        dc Resistance, ohms per unit length (see Runits). Defaults to Rac/1.02 if not specified.

        DSS property name: `Rdc`, DSS property index: 1.
        """
        return BatchFloat64ArrayProxy(self, 1)

    @Rdc.setter
    def Rdc(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(1, value)

    @property
    def Rac(self) -> BatchFloat64ArrayProxy:
        """
        Resistance at 60 Hz per unit length. Defaults to 1.02*Rdc if not specified.

        DSS property name: `Rac`, DSS property index: 2.
        """
        return BatchFloat64ArrayProxy(self, 2)

    @Rac.setter
    def Rac(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(2, value)

    @property
    def Runits(self) -> BatchInt32ArrayProxy:
        """
        Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `Runits`, DSS property index: 3.
        """
        return BatchInt32ArrayProxy(self, 3)

    @Runits.setter
    def Runits(self, value: Union[AnyStr, int, DimensionUnits, List[AnyStr], List[int], List[DimensionUnits], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(3, value)
            return
    
        self._set_batch_int32_array(3, value)

    @property
    def Runits_str(self) -> str:
        """
        Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `Runits`, DSS property index: 3.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 3)

    @Runits_str.setter
    def Runits_str(self, value: AnyStr):
        self.Runits = value

    @property
    def GMRac(self) -> BatchFloat64ArrayProxy:
        """
        GMR at 60 Hz. Defaults to .7788*radius if not specified.

        DSS property name: `GMRac`, DSS property index: 4.
        """
        return BatchFloat64ArrayProxy(self, 4)

    @GMRac.setter
    def GMRac(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(4, value)

    @property
    def GMRunits(self) -> BatchInt32ArrayProxy:
        """
        Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `GMRunits`, DSS property index: 5.
        """
        return BatchInt32ArrayProxy(self, 5)

    @GMRunits.setter
    def GMRunits(self, value: Union[AnyStr, int, DimensionUnits, List[AnyStr], List[int], List[DimensionUnits], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(5, value)
            return
    
        self._set_batch_int32_array(5, value)

    @property
    def GMRunits_str(self) -> str:
        """
        Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `GMRunits`, DSS property index: 5.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 5)

    @GMRunits_str.setter
    def GMRunits_str(self, value: AnyStr):
        self.GMRunits = value

    @property
    def radius(self) -> BatchFloat64ArrayProxy:
        """
        Outside radius of conductor. Defaults to GMR/0.7788 if not specified.

        DSS property name: `radius`, DSS property index: 6.
        """
        return BatchFloat64ArrayProxy(self, 6)

    @radius.setter
    def radius(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(6, value)

    @property
    def radunits(self) -> BatchInt32ArrayProxy:
        """
        Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `radunits`, DSS property index: 7.
        """
        return BatchInt32ArrayProxy(self, 7)

    @radunits.setter
    def radunits(self, value: Union[AnyStr, int, DimensionUnits, List[AnyStr], List[int], List[DimensionUnits], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(7, value)
            return
    
        self._set_batch_int32_array(7, value)

    @property
    def radunits_str(self) -> str:
        """
        Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `radunits`, DSS property index: 7.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 7)

    @radunits_str.setter
    def radunits_str(self, value: AnyStr):
        self.radunits = value

    @property
    def normamps(self) -> BatchFloat64ArrayProxy:
        """
        Normal ampacity, amperes. Defaults to Emergency amps/1.5 if not specified.

        DSS property name: `normamps`, DSS property index: 8.
        """
        return BatchFloat64ArrayProxy(self, 8)

    @normamps.setter
    def normamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(8, value)

    @property
    def emergamps(self) -> BatchFloat64ArrayProxy:
        """
        Emergency ampacity, amperes. Defaults to 1.5 * Normal Amps if not specified.

        DSS property name: `emergamps`, DSS property index: 9.
        """
        return BatchFloat64ArrayProxy(self, 9)

    @emergamps.setter
    def emergamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(9, value)

    @property
    def diam(self) -> BatchFloat64ArrayProxy:
        """
        Diameter; Alternative method for entering radius.

        DSS property name: `diam`, DSS property index: 10.
        """
        return BatchFloat64ArrayProxy(self, 10)

    @diam.setter
    def diam(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(10, value)

    @property
    def Seasons(self) -> BatchInt32ArrayProxy:
        """
        Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 11.
        """
        return BatchInt32ArrayProxy(self, 11)

    @Seasons.setter
    def Seasons(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(11, value)

    @property
    def Ratings(self) -> List[Float64Array]:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.

        DSS property name: `Ratings`, DSS property index: 12.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 12)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(12, value)

    @property
    def Capradius(self) -> BatchFloat64ArrayProxy:
        """
        Equivalent conductor radius for capacitance calcs. Specify this for bundled conductors. Defaults to same value as radius. Define Diam or Radius property first.

        DSS property name: `Capradius`, DSS property index: 13.
        """
        return BatchFloat64ArrayProxy(self, 13)

    @Capradius.setter
    def Capradius(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(13, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 14.
        """
        self._set_batch_string(14, value)

class CNDataBatch(DSSBatch):
    _cls_name = 'CNData'
    _obj_cls = CNData
    _cls_idx = 10


    @property
    def k(self) -> BatchInt32ArrayProxy:
        """
        Number of concentric neutral strands; default is 2

        DSS property name: `k`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @k.setter
    def k(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def DiaStrand(self) -> BatchFloat64ArrayProxy:
        """
        Diameter of a concentric neutral strand; same units as core conductor radius; no default.

        DSS property name: `DiaStrand`, DSS property index: 2.
        """
        return BatchFloat64ArrayProxy(self, 2)

    @DiaStrand.setter
    def DiaStrand(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(2, value)

    @property
    def GmrStrand(self) -> BatchFloat64ArrayProxy:
        """
        Geometric mean radius of a concentric neutral strand; same units as core conductor GMR; defaults to 0.7788 * CN strand radius.

        DSS property name: `GmrStrand`, DSS property index: 3.
        """
        return BatchFloat64ArrayProxy(self, 3)

    @GmrStrand.setter
    def GmrStrand(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(3, value)

    @property
    def Rstrand(self) -> BatchFloat64ArrayProxy:
        """
        AC resistance of a concentric neutral strand; same units as core conductor resistance; no default.

        DSS property name: `Rstrand`, DSS property index: 4.
        """
        return BatchFloat64ArrayProxy(self, 4)

    @Rstrand.setter
    def Rstrand(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(4, value)

    @property
    def EpsR(self) -> BatchFloat64ArrayProxy:
        """
        Insulation layer relative permittivity; default is 2.3.

        DSS property name: `EpsR`, DSS property index: 5.
        """
        return BatchFloat64ArrayProxy(self, 5)

    @EpsR.setter
    def EpsR(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(5, value)

    @property
    def InsLayer(self) -> BatchFloat64ArrayProxy:
        """
        Insulation layer thickness; same units as radius; no default. With DiaIns, establishes inner radius for capacitance calculation.

        DSS property name: `InsLayer`, DSS property index: 6.
        """
        return BatchFloat64ArrayProxy(self, 6)

    @InsLayer.setter
    def InsLayer(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(6, value)

    @property
    def DiaIns(self) -> BatchFloat64ArrayProxy:
        """
        Diameter over insulation layer; same units as radius; no default. Establishes outer radius for capacitance calculation.

        DSS property name: `DiaIns`, DSS property index: 7.
        """
        return BatchFloat64ArrayProxy(self, 7)

    @DiaIns.setter
    def DiaIns(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(7, value)

    @property
    def DiaCable(self) -> BatchFloat64ArrayProxy:
        """
        Diameter over cable; same units as radius; no default.

        DSS property name: `DiaCable`, DSS property index: 8.
        """
        return BatchFloat64ArrayProxy(self, 8)

    @DiaCable.setter
    def DiaCable(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(8, value)

    @property
    def Rdc(self) -> BatchFloat64ArrayProxy:
        """
        dc Resistance, ohms per unit length (see Runits). Defaults to Rac/1.02 if not specified.

        DSS property name: `Rdc`, DSS property index: 9.
        """
        return BatchFloat64ArrayProxy(self, 9)

    @Rdc.setter
    def Rdc(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(9, value)

    @property
    def Rac(self) -> BatchFloat64ArrayProxy:
        """
        Resistance at 60 Hz per unit length. Defaults to 1.02*Rdc if not specified.

        DSS property name: `Rac`, DSS property index: 10.
        """
        return BatchFloat64ArrayProxy(self, 10)

    @Rac.setter
    def Rac(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(10, value)

    @property
    def Runits(self) -> BatchInt32ArrayProxy:
        """
        Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `Runits`, DSS property index: 11.
        """
        return BatchInt32ArrayProxy(self, 11)

    @Runits.setter
    def Runits(self, value: Union[AnyStr, int, DimensionUnits, List[AnyStr], List[int], List[DimensionUnits], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(11, value)
            return
    
        self._set_batch_int32_array(11, value)

    @property
    def Runits_str(self) -> str:
        """
        Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `Runits`, DSS property index: 11.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 11)

    @Runits_str.setter
    def Runits_str(self, value: AnyStr):
        self.Runits = value

    @property
    def GMRac(self) -> BatchFloat64ArrayProxy:
        """
        GMR at 60 Hz. Defaults to .7788*radius if not specified.

        DSS property name: `GMRac`, DSS property index: 12.
        """
        return BatchFloat64ArrayProxy(self, 12)

    @GMRac.setter
    def GMRac(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(12, value)

    @property
    def GMRunits(self) -> BatchInt32ArrayProxy:
        """
        Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `GMRunits`, DSS property index: 13.
        """
        return BatchInt32ArrayProxy(self, 13)

    @GMRunits.setter
    def GMRunits(self, value: Union[AnyStr, int, DimensionUnits, List[AnyStr], List[int], List[DimensionUnits], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(13, value)
            return
    
        self._set_batch_int32_array(13, value)

    @property
    def GMRunits_str(self) -> str:
        """
        Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `GMRunits`, DSS property index: 13.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 13)

    @GMRunits_str.setter
    def GMRunits_str(self, value: AnyStr):
        self.GMRunits = value

    @property
    def radius(self) -> BatchFloat64ArrayProxy:
        """
        Outside radius of conductor. Defaults to GMR/0.7788 if not specified.

        DSS property name: `radius`, DSS property index: 14.
        """
        return BatchFloat64ArrayProxy(self, 14)

    @radius.setter
    def radius(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(14, value)

    @property
    def radunits(self) -> BatchInt32ArrayProxy:
        """
        Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `radunits`, DSS property index: 15.
        """
        return BatchInt32ArrayProxy(self, 15)

    @radunits.setter
    def radunits(self, value: Union[AnyStr, int, DimensionUnits, List[AnyStr], List[int], List[DimensionUnits], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(15, value)
            return
    
        self._set_batch_int32_array(15, value)

    @property
    def radunits_str(self) -> str:
        """
        Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `radunits`, DSS property index: 15.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 15)

    @radunits_str.setter
    def radunits_str(self, value: AnyStr):
        self.radunits = value

    @property
    def normamps(self) -> BatchFloat64ArrayProxy:
        """
        Normal ampacity, amperes. Defaults to Emergency amps/1.5 if not specified.

        DSS property name: `normamps`, DSS property index: 16.
        """
        return BatchFloat64ArrayProxy(self, 16)

    @normamps.setter
    def normamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(16, value)

    @property
    def emergamps(self) -> BatchFloat64ArrayProxy:
        """
        Emergency ampacity, amperes. Defaults to 1.5 * Normal Amps if not specified.

        DSS property name: `emergamps`, DSS property index: 17.
        """
        return BatchFloat64ArrayProxy(self, 17)

    @emergamps.setter
    def emergamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(17, value)

    @property
    def diam(self) -> BatchFloat64ArrayProxy:
        """
        Diameter; Alternative method for entering radius.

        DSS property name: `diam`, DSS property index: 18.
        """
        return BatchFloat64ArrayProxy(self, 18)

    @diam.setter
    def diam(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(18, value)

    @property
    def Seasons(self) -> BatchInt32ArrayProxy:
        """
        Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 19.
        """
        return BatchInt32ArrayProxy(self, 19)

    @Seasons.setter
    def Seasons(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(19, value)

    @property
    def Ratings(self) -> List[Float64Array]:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.

        DSS property name: `Ratings`, DSS property index: 20.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 20)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(20, value)

    @property
    def Capradius(self) -> BatchFloat64ArrayProxy:
        """
        Equivalent conductor radius for capacitance calcs. Specify this for bundled conductors. Defaults to same value as radius. Define Diam or Radius property first.

        DSS property name: `Capradius`, DSS property index: 21.
        """
        return BatchFloat64ArrayProxy(self, 21)

    @Capradius.setter
    def Capradius(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(21, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 22.
        """
        self._set_batch_string(22, value)

class TSDataBatch(DSSBatch):
    _cls_name = 'TSData'
    _obj_cls = TSData
    _cls_idx = 11


    @property
    def DiaShield(self) -> BatchFloat64ArrayProxy:
        """
        Diameter over tape shield; same units as radius; no default.

        DSS property name: `DiaShield`, DSS property index: 1.
        """
        return BatchFloat64ArrayProxy(self, 1)

    @DiaShield.setter
    def DiaShield(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(1, value)

    @property
    def TapeLayer(self) -> BatchFloat64ArrayProxy:
        """
        Tape shield thickness; same units as radius; no default.

        DSS property name: `TapeLayer`, DSS property index: 2.
        """
        return BatchFloat64ArrayProxy(self, 2)

    @TapeLayer.setter
    def TapeLayer(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(2, value)

    @property
    def TapeLap(self) -> BatchFloat64ArrayProxy:
        """
        Tape Lap in percent; default 20.0

        DSS property name: `TapeLap`, DSS property index: 3.
        """
        return BatchFloat64ArrayProxy(self, 3)

    @TapeLap.setter
    def TapeLap(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(3, value)

    @property
    def EpsR(self) -> BatchFloat64ArrayProxy:
        """
        Insulation layer relative permittivity; default is 2.3.

        DSS property name: `EpsR`, DSS property index: 4.
        """
        return BatchFloat64ArrayProxy(self, 4)

    @EpsR.setter
    def EpsR(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(4, value)

    @property
    def InsLayer(self) -> BatchFloat64ArrayProxy:
        """
        Insulation layer thickness; same units as radius; no default. With DiaIns, establishes inner radius for capacitance calculation.

        DSS property name: `InsLayer`, DSS property index: 5.
        """
        return BatchFloat64ArrayProxy(self, 5)

    @InsLayer.setter
    def InsLayer(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(5, value)

    @property
    def DiaIns(self) -> BatchFloat64ArrayProxy:
        """
        Diameter over insulation layer; same units as radius; no default. Establishes outer radius for capacitance calculation.

        DSS property name: `DiaIns`, DSS property index: 6.
        """
        return BatchFloat64ArrayProxy(self, 6)

    @DiaIns.setter
    def DiaIns(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(6, value)

    @property
    def DiaCable(self) -> BatchFloat64ArrayProxy:
        """
        Diameter over cable; same units as radius; no default.

        DSS property name: `DiaCable`, DSS property index: 7.
        """
        return BatchFloat64ArrayProxy(self, 7)

    @DiaCable.setter
    def DiaCable(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(7, value)

    @property
    def Rdc(self) -> BatchFloat64ArrayProxy:
        """
        dc Resistance, ohms per unit length (see Runits). Defaults to Rac/1.02 if not specified.

        DSS property name: `Rdc`, DSS property index: 8.
        """
        return BatchFloat64ArrayProxy(self, 8)

    @Rdc.setter
    def Rdc(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(8, value)

    @property
    def Rac(self) -> BatchFloat64ArrayProxy:
        """
        Resistance at 60 Hz per unit length. Defaults to 1.02*Rdc if not specified.

        DSS property name: `Rac`, DSS property index: 9.
        """
        return BatchFloat64ArrayProxy(self, 9)

    @Rac.setter
    def Rac(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(9, value)

    @property
    def Runits(self) -> BatchInt32ArrayProxy:
        """
        Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `Runits`, DSS property index: 10.
        """
        return BatchInt32ArrayProxy(self, 10)

    @Runits.setter
    def Runits(self, value: Union[AnyStr, int, DimensionUnits, List[AnyStr], List[int], List[DimensionUnits], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(10, value)
            return
    
        self._set_batch_int32_array(10, value)

    @property
    def Runits_str(self) -> str:
        """
        Length units for resistance: ohms per {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `Runits`, DSS property index: 10.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 10)

    @Runits_str.setter
    def Runits_str(self, value: AnyStr):
        self.Runits = value

    @property
    def GMRac(self) -> BatchFloat64ArrayProxy:
        """
        GMR at 60 Hz. Defaults to .7788*radius if not specified.

        DSS property name: `GMRac`, DSS property index: 11.
        """
        return BatchFloat64ArrayProxy(self, 11)

    @GMRac.setter
    def GMRac(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(11, value)

    @property
    def GMRunits(self) -> BatchInt32ArrayProxy:
        """
        Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `GMRunits`, DSS property index: 12.
        """
        return BatchInt32ArrayProxy(self, 12)

    @GMRunits.setter
    def GMRunits(self, value: Union[AnyStr, int, DimensionUnits, List[AnyStr], List[int], List[DimensionUnits], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(12, value)
            return
    
        self._set_batch_int32_array(12, value)

    @property
    def GMRunits_str(self) -> str:
        """
        Units for GMR: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `GMRunits`, DSS property index: 12.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 12)

    @GMRunits_str.setter
    def GMRunits_str(self, value: AnyStr):
        self.GMRunits = value

    @property
    def radius(self) -> BatchFloat64ArrayProxy:
        """
        Outside radius of conductor. Defaults to GMR/0.7788 if not specified.

        DSS property name: `radius`, DSS property index: 13.
        """
        return BatchFloat64ArrayProxy(self, 13)

    @radius.setter
    def radius(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(13, value)

    @property
    def radunits(self) -> BatchInt32ArrayProxy:
        """
        Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `radunits`, DSS property index: 14.
        """
        return BatchInt32ArrayProxy(self, 14)

    @radunits.setter
    def radunits(self, value: Union[AnyStr, int, DimensionUnits, List[AnyStr], List[int], List[DimensionUnits], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(14, value)
            return
    
        self._set_batch_int32_array(14, value)

    @property
    def radunits_str(self) -> str:
        """
        Units for outside radius: {mi|kft|km|m|Ft|in|cm|mm} Default=none.

        DSS property name: `radunits`, DSS property index: 14.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 14)

    @radunits_str.setter
    def radunits_str(self, value: AnyStr):
        self.radunits = value

    @property
    def normamps(self) -> BatchFloat64ArrayProxy:
        """
        Normal ampacity, amperes. Defaults to Emergency amps/1.5 if not specified.

        DSS property name: `normamps`, DSS property index: 15.
        """
        return BatchFloat64ArrayProxy(self, 15)

    @normamps.setter
    def normamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(15, value)

    @property
    def emergamps(self) -> BatchFloat64ArrayProxy:
        """
        Emergency ampacity, amperes. Defaults to 1.5 * Normal Amps if not specified.

        DSS property name: `emergamps`, DSS property index: 16.
        """
        return BatchFloat64ArrayProxy(self, 16)

    @emergamps.setter
    def emergamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(16, value)

    @property
    def diam(self) -> BatchFloat64ArrayProxy:
        """
        Diameter; Alternative method for entering radius.

        DSS property name: `diam`, DSS property index: 17.
        """
        return BatchFloat64ArrayProxy(self, 17)

    @diam.setter
    def diam(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(17, value)

    @property
    def Seasons(self) -> BatchInt32ArrayProxy:
        """
        Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 18.
        """
        return BatchInt32ArrayProxy(self, 18)

    @Seasons.setter
    def Seasons(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(18, value)

    @property
    def Ratings(self) -> List[Float64Array]:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.

        DSS property name: `Ratings`, DSS property index: 19.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 19)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(19, value)

    @property
    def Capradius(self) -> BatchFloat64ArrayProxy:
        """
        Equivalent conductor radius for capacitance calcs. Specify this for bundled conductors. Defaults to same value as radius. Define Diam or Radius property first.

        DSS property name: `Capradius`, DSS property index: 20.
        """
        return BatchFloat64ArrayProxy(self, 20)

    @Capradius.setter
    def Capradius(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(20, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 21.
        """
        self._set_batch_string(21, value)

class LineSpacingBatch(DSSBatch):
    _cls_name = 'LineSpacing'
    _obj_cls = LineSpacing
    _cls_idx = 12


    @property
    def nconds(self) -> BatchInt32ArrayProxy:
        """
        Number of wires in this geometry. Default is 3. Triggers memory allocations. Define first!

        DSS property name: `nconds`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @nconds.setter
    def nconds(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def nphases(self) -> BatchInt32ArrayProxy:
        """
        Number of retained phase conductors. If less than the number of wires, list the retained phase coordinates first.

        DSS property name: `nphases`, DSS property index: 2.
        """
        return BatchInt32ArrayProxy(self, 2)

    @nphases.setter
    def nphases(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(2, value)

    @property
    def x(self) -> List[Float64Array]:
        """
        Array of wire X coordinates.

        DSS property name: `x`, DSS property index: 3.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 3)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @x.setter
    def x(self, value: Float64Array):
        self._set_float64_array_o(3, value)

    @property
    def h(self) -> List[Float64Array]:
        """
        Array of wire Heights.

        DSS property name: `h`, DSS property index: 4.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 4)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @h.setter
    def h(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def units(self) -> BatchInt32ArrayProxy:
        """
        Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined

        DSS property name: `units`, DSS property index: 5.
        """
        return BatchInt32ArrayProxy(self, 5)

    @units.setter
    def units(self, value: Union[AnyStr, int, DimensionUnits, List[AnyStr], List[int], List[DimensionUnits], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(5, value)
            return
    
        self._set_batch_int32_array(5, value)

    @property
    def units_str(self) -> str:
        """
        Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined

        DSS property name: `units`, DSS property index: 5.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 5)

    @units_str.setter
    def units_str(self, value: AnyStr):
        self.units = value

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 6.
        """
        self._set_batch_string(6, value)

class LineGeometryBatch(DSSBatch):
    _cls_name = 'LineGeometry'
    _obj_cls = LineGeometry
    _cls_idx = 13


    @property
    def nconds(self) -> BatchInt32ArrayProxy:
        """
        Number of conductors in this geometry. Default is 3. Triggers memory allocations. Define first!

        DSS property name: `nconds`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @nconds.setter
    def nconds(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def nphases(self) -> BatchInt32ArrayProxy:
        """
        Number of phases. Default =3; All other conductors are considered neutrals and might be reduced out.

        DSS property name: `nphases`, DSS property index: 2.
        """
        return BatchInt32ArrayProxy(self, 2)

    @nphases.setter
    def nphases(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(2, value)

    @property
    def x(self) -> List[Float64Array]:
        """
        x coordinate.

        DSS property name: `x`, DSS property index: 5.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 5)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @x.setter
    def x(self, value: Float64Array):
        self._set_float64_array_o(5, value)

    @property
    def h(self) -> List[Float64Array]:
        """
        Height of conductor.

        DSS property name: `h`, DSS property index: 6.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 6)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @h.setter
    def h(self, value: Float64Array):
        self._set_float64_array_o(6, value)

    @property
    def units(self) -> BatchInt32ArrayProxy:
        """
        Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined

        DSS property name: `units`, DSS property index: 7.
        """
        return BatchInt32ArrayProxy(self, 7)

    @units.setter
    def units(self, value: Union[AnyStr, int, DimensionUnits, List[AnyStr], List[int], List[DimensionUnits], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(7, value)
            return
    
        self._set_batch_int32_array(7, value)

    @property
    def units_str(self) -> str:
        """
        Units for x and h: {mi|kft|km|m|Ft|in|cm } Initial default is "ft", but defaults to last unit defined

        DSS property name: `units`, DSS property index: 7.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 7)

    @units_str.setter
    def units_str(self, value: AnyStr):
        self.units = value

    @property
    def normamps(self) -> BatchFloat64ArrayProxy:
        """
        Normal ampacity, amperes for the line. Defaults to first conductor if not specified.

        DSS property name: `normamps`, DSS property index: 8.
        """
        return BatchFloat64ArrayProxy(self, 8)

    @normamps.setter
    def normamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(8, value)

    @property
    def emergamps(self) -> BatchFloat64ArrayProxy:
        """
        Emergency ampacity, amperes. Defaults to first conductor if not specified.

        DSS property name: `emergamps`, DSS property index: 9.
        """
        return BatchFloat64ArrayProxy(self, 9)

    @emergamps.setter
    def emergamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(9, value)

    @property
    def reduce(self) -> List[bool]:
        """
        {Yes | No} Default = no. Reduce to Nphases (Kron Reduction). Reduce out neutrals.

        DSS property name: `reduce`, DSS property index: 10.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 10)
        ]
    @reduce.setter
    def reduce(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 10, value)

    @property
    def spacing(self) -> List[str]:
        """
        Reference to a LineSpacing for use in a line constants calculation.
        Alternative to x, h, and units. MUST BE PREVIOUSLY DEFINED.
        Must match "nconds" as previously defined for this geometry.
        Must be used in conjunction with the Wires property.

        DSS property name: `spacing`, DSS property index: 11.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 11)

    @spacing.setter
    def spacing(self, value: Union[AnyStr, LineSpacing, List[AnyStr], List[LineSpacing]]):
        self._set_batch_obj_prop(11, value)

    @property
    def spacing_obj(self) -> List[LineSpacing]:
        """
        Reference to a LineSpacing for use in a line constants calculation.
        Alternative to x, h, and units. MUST BE PREVIOUSLY DEFINED.
        Must match "nconds" as previously defined for this geometry.
        Must be used in conjunction with the Wires property.

        DSS property name: `spacing`, DSS property index: 11.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 11)

    @spacing_obj.setter
    def spacing_obj(self, value: LineSpacing):
        self._set_batch_string(11, value)

    @property
    def wires(self) -> List[List[str]]:
        """
        Array of WireData names for use in a line constants calculation.
        Alternative to individual wire inputs. ALL MUST BE PREVIOUSLY DEFINED.
        Must match "nconds" as previously defined for this geometry,
        unless TSData or CNData were previously assigned to phases, and these wires are neutrals.
        Must be used in conjunction with the Spacing property.

        DSS property name: `wires`, DSS property index: 12.
        """
        return self._get_string_ll(12)

    @wires.setter
    def wires(self, value: Union[List[AnyStr], List[WireData]]):
        if (not len(value)) or isinstance(value[0], (bytes, str)) or (len(value[0]) and isinstance(value[0][0], (bytes, str))):
            self._set_batch_stringlist_prop(12, value)
            return

        self._set_batch_objlist_prop(12, value)

    @property
    def wires_obj(self) -> List[List[WireData]]:
        """
        Array of WireData names for use in a line constants calculation.
        Alternative to individual wire inputs. ALL MUST BE PREVIOUSLY DEFINED.
        Must match "nconds" as previously defined for this geometry,
        unless TSData or CNData were previously assigned to phases, and these wires are neutrals.
        Must be used in conjunction with the Spacing property.

        DSS property name: `wires`, DSS property index: 12.
        """
        return self._get_obj_ll(12, WireData)

    @wires_obj.setter
    def wires_obj(self, value: List[WireData]):
        self._set_batch_objlist_prop(12, value)

    @property
    def cncables(self) -> List[List[str]]:
        """
        Array of CNData names for cable parameter calculation.
        All must be previously defined, and match "nphases" for this geometry.
        You can later define "nconds-nphases" wires for bare neutral conductors.

        DSS property name: `cncables`, DSS property index: 15.
        """
        return self._get_string_ll(15)

    @cncables.setter
    def cncables(self, value: Union[List[AnyStr], List[CNData]]):
        if (not len(value)) or isinstance(value[0], (bytes, str)) or (len(value[0]) and isinstance(value[0][0], (bytes, str))):
            self._set_batch_stringlist_prop(15, value)
            return

        self._set_batch_objlist_prop(15, value)

    @property
    def cncables_obj(self) -> List[List[CNData]]:
        """
        Array of CNData names for cable parameter calculation.
        All must be previously defined, and match "nphases" for this geometry.
        You can later define "nconds-nphases" wires for bare neutral conductors.

        DSS property name: `cncables`, DSS property index: 15.
        """
        return self._get_obj_ll(15, CNData)

    @cncables_obj.setter
    def cncables_obj(self, value: List[CNData]):
        self._set_batch_objlist_prop(15, value)

    @property
    def tscables(self) -> List[List[str]]:
        """
        Array of TSData names for cable parameter calculation.
        All must be previously defined, and match "nphases" for this geometry.
        You can later define "nconds-nphases" wires for bare neutral conductors.

        DSS property name: `tscables`, DSS property index: 16.
        """
        return self._get_string_ll(16)

    @tscables.setter
    def tscables(self, value: Union[List[AnyStr], List[TSData]]):
        if (not len(value)) or isinstance(value[0], (bytes, str)) or (len(value[0]) and isinstance(value[0][0], (bytes, str))):
            self._set_batch_stringlist_prop(16, value)
            return

        self._set_batch_objlist_prop(16, value)

    @property
    def tscables_obj(self) -> List[List[TSData]]:
        """
        Array of TSData names for cable parameter calculation.
        All must be previously defined, and match "nphases" for this geometry.
        You can later define "nconds-nphases" wires for bare neutral conductors.

        DSS property name: `tscables`, DSS property index: 16.
        """
        return self._get_obj_ll(16, TSData)

    @tscables_obj.setter
    def tscables_obj(self, value: List[TSData]):
        self._set_batch_objlist_prop(16, value)

    @property
    def Seasons(self) -> BatchInt32ArrayProxy:
        """
        Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property. Defaults to first conductor if not specified.

        DSS property name: `Seasons`, DSS property index: 17.
        """
        return BatchInt32ArrayProxy(self, 17)

    @Seasons.setter
    def Seasons(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(17, value)

    @property
    def Ratings(self) -> List[Float64Array]:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.Defaults to first conductor if not specified.

        DSS property name: `Ratings`, DSS property index: 18.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 18)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(18, value)

    @property
    def linetype(self) -> BatchInt32ArrayProxy:
        """
        Code designating the type of line. 
        One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW

        OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.

        DSS property name: `LineType`, DSS property index: 19.
        """
        return BatchInt32ArrayProxy(self, 19)

    @linetype.setter
    def linetype(self, value: Union[AnyStr, int, LineType, List[AnyStr], List[int], List[LineType], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(19, value)
            return
    
        self._set_batch_int32_array(19, value)

    @property
    def linetype_str(self) -> str:
        """
        Code designating the type of line. 
        One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW

        OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.

        DSS property name: `LineType`, DSS property index: 19.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 19)

    @linetype_str.setter
    def linetype_str(self, value: AnyStr):
        self.linetype = value

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 20.
        """
        self._set_batch_string(20, value)

class XfmrCodeBatch(DSSBatch):
    _cls_name = 'XfmrCode'
    _obj_cls = XfmrCode
    _cls_idx = 14


    @property
    def phases(self) -> BatchInt32ArrayProxy:
        """
        Number of phases this transformer. Default is 3.

        DSS property name: `phases`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @phases.setter
    def phases(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def windings(self) -> BatchInt32ArrayProxy:
        """
        Number of windings, this transformers. (Also is the number of terminals) Default is 2. This property triggers memory allocation for the Transformer and will cause other properties to revert to default values.

        DSS property name: `windings`, DSS property index: 2.
        """
        return BatchInt32ArrayProxy(self, 2)

    @windings.setter
    def windings(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(2, value)

    @property
    def pctR(self) -> List[Float64Array]:
        """
        Percent resistance this winding.  (half of total for a 2-winding).

        DSS property name: `%R`, DSS property index: 8.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 8)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @pctR.setter
    def pctR(self, value: Float64Array):
        self._set_float64_array_o(8, value)

    @property
    def Rneut(self) -> List[Float64Array]:
        """
        Default = -1. Neutral resistance of wye (star)-connected winding in actual ohms.If entered as a negative value, the neutral is assumed to be open, or floating.

        DSS property name: `Rneut`, DSS property index: 9.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 9)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Rneut.setter
    def Rneut(self, value: Float64Array):
        self._set_float64_array_o(9, value)

    @property
    def Xneut(self) -> List[Float64Array]:
        """
        Neutral reactance of wye(star)-connected winding in actual ohms.  May be + or -.

        DSS property name: `Xneut`, DSS property index: 10.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 10)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Xneut.setter
    def Xneut(self, value: Float64Array):
        self._set_float64_array_o(10, value)

    @property
    def conns(self) -> List[Int32Array]:
        """
        Use this to specify all the Winding connections at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus"
        ~ conns=(delta, wye)

        DSS property name: `conns`, DSS property index: 11.
        """
        return [
            self._get_int32_array(self._lib.Obj_GetInt32Array, x, 11)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @conns.setter
    def conns(self, value: Union[List[Union[int,Connection]], List[AnyStr]]): #TODO: list of lists
        if len(value) and not isinstance(value[0], int):
            value, value_ptr, value_count = self._prepare_string_array(value)
            for x in self._ffi.unpack(self.pointer[0], self.count[0]):
                self._lib.Obj_SetStringArray(x, 11, value_ptr, value_count)

            self._check_for_error()
            return

        self._set_batch_int32_array(11, value)

    @property
    def conns_str(self) -> List[List[str]]:
        """
        Use this to specify all the Winding connections at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus"
        ~ conns=(delta, wye)

        DSS property name: `conns`, DSS property index: 11.
        """
        return self._get_string_ll(11)

    @conns_str.setter
    def conns_str(self, value: AnyStr):
        self.conns = value

    @property
    def kVs(self) -> List[Float64Array]:
        """
        Use this to specify the kV ratings of all windings at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus" 
        ~ conns=(delta, wye)
        ~ kvs=(115, 12.47)

        See kV= property for voltage rules.

        DSS property name: `kVs`, DSS property index: 12.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 12)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @kVs.setter
    def kVs(self, value: Float64Array):
        self._set_float64_array_o(12, value)

    @property
    def kVAs(self) -> List[Float64Array]:
        """
        Use this to specify the kVA ratings of all windings at once using an array.

        DSS property name: `kVAs`, DSS property index: 13.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 13)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @kVAs.setter
    def kVAs(self, value: Float64Array):
        self._set_float64_array_o(13, value)

    @property
    def taps(self) -> List[Float64Array]:
        """
        Use this to specify the normal p.u. tap of all windings at once using an array.

        DSS property name: `taps`, DSS property index: 14.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 14)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @taps.setter
    def taps(self, value: Float64Array):
        self._set_float64_array_o(14, value)

    @property
    def Xhl(self) -> BatchFloat64ArrayProxy:
        """
        Use this to specify the percent reactance, H-L (winding 1 to winding 2).  Use for 2- or 3-winding transformers. On the kva base of winding 1.

        DSS property name: `Xhl`, DSS property index: 15.
        """
        return BatchFloat64ArrayProxy(self, 15)

    @Xhl.setter
    def Xhl(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(15, value)

    @property
    def Xht(self) -> BatchFloat64ArrayProxy:
        """
        Use this to specify the percent reactance, H-T (winding 1 to winding 3).  Use for 3-winding transformers only. On the kVA base of winding 1.

        DSS property name: `Xht`, DSS property index: 16.
        """
        return BatchFloat64ArrayProxy(self, 16)

    @Xht.setter
    def Xht(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(16, value)

    @property
    def Xlt(self) -> BatchFloat64ArrayProxy:
        """
        Use this to specify the percent reactance, L-T (winding 2 to winding 3).  Use for 3-winding transformers only. On the kVA base of winding 1.

        DSS property name: `Xlt`, DSS property index: 17.
        """
        return BatchFloat64ArrayProxy(self, 17)

    @Xlt.setter
    def Xlt(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(17, value)

    @property
    def Xscarray(self) -> List[Float64Array]:
        """
        Use this to specify the percent reactance between all pairs of windings as an array. All values are on the kVA base of winding 1.  The order of the values is as follows:

        (x12 13 14... 23 24.. 34 ..)  

        There will be n(n-1)/2 values, where n=number of windings.

        DSS property name: `Xscarray`, DSS property index: 18.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 18)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Xscarray.setter
    def Xscarray(self, value: Float64Array):
        self._set_float64_array_o(18, value)

    @property
    def thermal(self) -> BatchFloat64ArrayProxy:
        """
        Thermal time constant of the transformer in hours.  Typically about 2.

        DSS property name: `thermal`, DSS property index: 19.
        """
        return BatchFloat64ArrayProxy(self, 19)

    @thermal.setter
    def thermal(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(19, value)

    @property
    def n(self) -> BatchFloat64ArrayProxy:
        """
        n Exponent for thermal properties in IEEE C57.  Typically 0.8.

        DSS property name: `n`, DSS property index: 20.
        """
        return BatchFloat64ArrayProxy(self, 20)

    @n.setter
    def n(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(20, value)

    @property
    def m(self) -> BatchFloat64ArrayProxy:
        """
        m Exponent for thermal properties in IEEE C57.  Typically 0.9 - 1.0

        DSS property name: `m`, DSS property index: 21.
        """
        return BatchFloat64ArrayProxy(self, 21)

    @m.setter
    def m(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(21, value)

    @property
    def flrise(self) -> BatchFloat64ArrayProxy:
        """
        Temperature rise, deg C, for full load.  Default is 65.

        DSS property name: `flrise`, DSS property index: 22.
        """
        return BatchFloat64ArrayProxy(self, 22)

    @flrise.setter
    def flrise(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(22, value)

    @property
    def hsrise(self) -> BatchFloat64ArrayProxy:
        """
        Hot spot temperature rise, deg C.  Default is 15.

        DSS property name: `hsrise`, DSS property index: 23.
        """
        return BatchFloat64ArrayProxy(self, 23)

    @hsrise.setter
    def hsrise(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(23, value)

    @property
    def pctloadloss(self) -> BatchFloat64ArrayProxy:
        """
        Percent load loss at full load. The %R of the High and Low windings (1 and 2) are adjusted to agree at rated kVA loading.

        DSS property name: `%loadloss`, DSS property index: 24.
        """
        return BatchFloat64ArrayProxy(self, 24)

    @pctloadloss.setter
    def pctloadloss(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(24, value)

    @property
    def pctnoloadloss(self) -> BatchFloat64ArrayProxy:
        """
        Percent no load losses at rated excitatation voltage. Default is 0. Converts to a resistance in parallel with the magnetizing impedance in each winding.

        DSS property name: `%noloadloss`, DSS property index: 25.
        """
        return BatchFloat64ArrayProxy(self, 25)

    @pctnoloadloss.setter
    def pctnoloadloss(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(25, value)

    @property
    def normhkVA(self) -> BatchFloat64ArrayProxy:
        """
        Normal maximum kVA rating of H winding (winding 1).  Usually 100% - 110% ofmaximum nameplate rating, depending on load shape. Defaults to 110% of kVA rating of Winding 1.

        DSS property name: `normhkVA`, DSS property index: 26.
        """
        return BatchFloat64ArrayProxy(self, 26)

    @normhkVA.setter
    def normhkVA(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(26, value)

    @property
    def emerghkVA(self) -> BatchFloat64ArrayProxy:
        """
        Emergency (contingency)  kVA rating of H winding (winding 1).  Usually 140% - 150% ofmaximum nameplate rating, depending on load shape. Defaults to 150% of kVA rating of Winding 1.

        DSS property name: `emerghkVA`, DSS property index: 27.
        """
        return BatchFloat64ArrayProxy(self, 27)

    @emerghkVA.setter
    def emerghkVA(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(27, value)

    @property
    def MaxTap(self) -> List[Float64Array]:
        """
        Max per unit tap for the active winding.  Default is 1.10

        DSS property name: `MaxTap`, DSS property index: 28.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 28)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @MaxTap.setter
    def MaxTap(self, value: Float64Array):
        self._set_float64_array_o(28, value)

    @property
    def MinTap(self) -> List[Float64Array]:
        """
        Min per unit tap for the active winding.  Default is 0.90

        DSS property name: `MinTap`, DSS property index: 29.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 29)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @MinTap.setter
    def MinTap(self, value: Float64Array):
        self._set_float64_array_o(29, value)

    @property
    def NumTaps(self) -> List[Int32Array]:
        """
        Total number of taps between min and max tap.  Default is 32.

        DSS property name: `NumTaps`, DSS property index: 30.
        """
        return [
            self._get_int32_array(self._lib.Obj_GetInt32Array, x, 30)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @NumTaps.setter
    def NumTaps(self, value: Int32Array): #TODO: list of arrays, matrix
        self._set_batch_int32_array(30, value)

    @property
    def pctimag(self) -> BatchFloat64ArrayProxy:
        """
        Percent magnetizing current. Default=0.0. Magnetizing branch is in parallel with windings in each phase. Also, see "ppm_antifloat".

        DSS property name: `%imag`, DSS property index: 31.
        """
        return BatchFloat64ArrayProxy(self, 31)

    @pctimag.setter
    def pctimag(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(31, value)

    @property
    def ppm_antifloat(self) -> BatchFloat64ArrayProxy:
        """
        Default=1 ppm.  Parts per million of transformer winding VA rating connected to ground to protect against accidentally floating a winding without a reference. If positive then the effect is adding a very large reactance to ground.  If negative, then a capacitor.

        DSS property name: `ppm_antifloat`, DSS property index: 32.
        """
        return BatchFloat64ArrayProxy(self, 32)

    @ppm_antifloat.setter
    def ppm_antifloat(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(32, value)

    @property
    def pctRs(self) -> List[Float64Array]:
        """
        Use this property to specify all the winding %resistances using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus" ~ %Rs=(0.2  0.3)

        DSS property name: `%Rs`, DSS property index: 33.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 33)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @pctRs.setter
    def pctRs(self, value: Float64Array):
        self._set_float64_array_o(33, value)

    @property
    def X12(self) -> BatchFloat64ArrayProxy:
        """
        Alternative to XHL for specifying the percent reactance from winding 1 to winding 2.  Use for 2- or 3-winding transformers. Percent on the kVA base of winding 1. 

        DSS property name: `X12`, DSS property index: 34.
        """
        return BatchFloat64ArrayProxy(self, 34)

    @X12.setter
    def X12(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(34, value)

    @property
    def X13(self) -> BatchFloat64ArrayProxy:
        """
        Alternative to XHT for specifying the percent reactance from winding 1 to winding 3.  Use for 3-winding transformers only. Percent on the kVA base of winding 1. 

        DSS property name: `X13`, DSS property index: 35.
        """
        return BatchFloat64ArrayProxy(self, 35)

    @X13.setter
    def X13(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(35, value)

    @property
    def X23(self) -> BatchFloat64ArrayProxy:
        """
        Alternative to XLT for specifying the percent reactance from winding 2 to winding 3.Use for 3-winding transformers only. Percent on the kVA base of winding 1.  

        DSS property name: `X23`, DSS property index: 36.
        """
        return BatchFloat64ArrayProxy(self, 36)

    @X23.setter
    def X23(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(36, value)

    @property
    def RdcOhms(self) -> List[Float64Array]:
        """
        Winding dc resistance in OHMS. Useful for GIC analysis. From transformer test report. Defaults to 85% of %R property

        DSS property name: `RdcOhms`, DSS property index: 37.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 37)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @RdcOhms.setter
    def RdcOhms(self, value: Float64Array):
        self._set_float64_array_o(37, value)

    @property
    def Seasons(self) -> BatchInt32ArrayProxy:
        """
        Defines the number of ratings to be defined for the transfomer, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 38.
        """
        return BatchInt32ArrayProxy(self, 38)

    @Seasons.setter
    def Seasons(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(38, value)

    @property
    def Ratings(self) -> List[Float64Array]:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in transformers.

        DSS property name: `Ratings`, DSS property index: 39.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 39)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(39, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 40.
        """
        self._set_batch_string(40, value)

class LineBatch(DSSBatch):
    _cls_name = 'Line'
    _obj_cls = Line
    _cls_idx = 15


    @property
    def bus1(self) -> List[str]:
        """
        Name of bus to which first terminal is connected.
        Example:
        bus1=busname   (assumes all terminals connected in normal phase order)
        bus1=busname.3.1.2.0 (specify terminal to node connections explicitly)

        DSS property name: `bus1`, DSS property index: 1.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 1) 

    @bus1.setter
    def bus1(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(1, value)

    @property
    def bus2(self) -> List[str]:
        """
        Name of bus to which 2nd terminal is connected.

        DSS property name: `bus2`, DSS property index: 2.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 2) 

    @bus2.setter
    def bus2(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(2, value)

    @property
    def linecode(self) -> List[str]:
        """
        Name of linecode object describing line impedances.
        If you use a line code, you do not need to specify the impedances here. The line code must have been PREVIOUSLY defined. The values specified last will prevail over those specified earlier (left-to-right sequence of properties).  You can subsequently change the number of phases if symmetrical component quantities are specified.If no line code or impedance data are specified, the line object defaults to 336 MCM ACSR on 4 ft spacing.

        DSS property name: `linecode`, DSS property index: 3.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 3)

    @linecode.setter
    def linecode(self, value: Union[AnyStr, LineCode, List[AnyStr], List[LineCode]]):
        self._set_batch_obj_prop(3, value)

    @property
    def linecode_obj(self) -> List[LineCode]:
        """
        Name of linecode object describing line impedances.
        If you use a line code, you do not need to specify the impedances here. The line code must have been PREVIOUSLY defined. The values specified last will prevail over those specified earlier (left-to-right sequence of properties).  You can subsequently change the number of phases if symmetrical component quantities are specified.If no line code or impedance data are specified, the line object defaults to 336 MCM ACSR on 4 ft spacing.

        DSS property name: `linecode`, DSS property index: 3.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 3)

    @linecode_obj.setter
    def linecode_obj(self, value: LineCode):
        self._set_batch_string(3, value)

    @property
    def length(self) -> BatchFloat64ArrayProxy:
        """
        Length of line. Default is 1.0. If units do not match the impedance data, specify "units" property. 

        DSS property name: `length`, DSS property index: 4.
        """
        return BatchFloat64ArrayProxy(self, 4)

    @length.setter
    def length(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(4, value)

    @property
    def phases(self) -> BatchInt32ArrayProxy:
        """
        Number of phases, this line.

        DSS property name: `phases`, DSS property index: 5.
        """
        return BatchInt32ArrayProxy(self, 5)

    @phases.setter
    def phases(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(5, value)

    @property
    def r1(self) -> BatchFloat64ArrayProxy:
        """
        Positive-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Rmatrix.

        DSS property name: `r1`, DSS property index: 6.
        """
        return BatchFloat64ArrayProxy(self, 6)

    @r1.setter
    def r1(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(6, value)

    @property
    def x1(self) -> BatchFloat64ArrayProxy:
        """
        Positive-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.  See also Xmatrix

        DSS property name: `x1`, DSS property index: 7.
        """
        return BatchFloat64ArrayProxy(self, 7)

    @x1.setter
    def x1(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(7, value)

    @property
    def r0(self) -> BatchFloat64ArrayProxy:
        """
        Zero-sequence Resistance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.

        DSS property name: `r0`, DSS property index: 8.
        """
        return BatchFloat64ArrayProxy(self, 8)

    @r0.setter
    def r0(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(8, value)

    @property
    def x0(self) -> BatchFloat64ArrayProxy:
        """
        Zero-sequence Reactance, ohms per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.

        DSS property name: `x0`, DSS property index: 9.
        """
        return BatchFloat64ArrayProxy(self, 9)

    @x0.setter
    def x0(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(9, value)

    @property
    def C1(self) -> BatchFloat64ArrayProxy:
        """
        Positive-sequence capacitance, nf per unit length.  Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition. See also Cmatrix and B1.

        DSS property name: `C1`, DSS property index: 10.
        """
        return BatchFloat64ArrayProxy(self, 10)

    @C1.setter
    def C1(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(10, value)

    @property
    def C0(self) -> BatchFloat64ArrayProxy:
        """
        Zero-sequence capacitance, nf per unit length. Setting any of R1, R0, X1, X0, C1, C0 forces the program to use the symmetrical component line definition.See also B0.

        DSS property name: `C0`, DSS property index: 11.
        """
        return BatchFloat64ArrayProxy(self, 11)

    @C0.setter
    def C0(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(11, value)

    @property
    def rmatrix(self) -> List[Float64Array]:
        """
        Resistance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration. Using any of Rmatrix, Xmatrix, Cmatrix forces program to use the matrix values for line impedance definition. For balanced line models, you may use the standard symmetrical component data definition instead.

        DSS property name: `rmatrix`, DSS property index: 12.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 12)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @rmatrix.setter
    def rmatrix(self, value: Float64Array):
        self._set_float64_array_o(12, value)

    @property
    def xmatrix(self) -> List[Float64Array]:
        """
        Reactance matrix, lower triangle, ohms per unit length. Order of the matrix is the number of phases. May be used to specify the impedance of any line configuration. Using any of Rmatrix, Xmatrix, Cmatrix forces program to use the matrix values for line impedance definition.  For balanced line models, you may use the standard symmetrical component data definition instead.

        DSS property name: `xmatrix`, DSS property index: 13.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 13)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @xmatrix.setter
    def xmatrix(self, value: Float64Array):
        self._set_float64_array_o(13, value)

    @property
    def cmatrix(self) -> List[Float64Array]:
        """
        Nodal Capacitance matrix, lower triangle, nf per unit length.Order of the matrix is the number of phases. May be used to specify the shunt capacitance of any line configuration. Using any of Rmatrix, Xmatrix, Cmatrix forces program to use the matrix values for line impedance definition.  For balanced line models, you may use the standard symmetrical component data definition instead.

        DSS property name: `cmatrix`, DSS property index: 14.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 14)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @cmatrix.setter
    def cmatrix(self, value: Float64Array):
        self._set_float64_array_o(14, value)

    @property
    def Switch(self) -> List[bool]:
        """
        {y/n | T/F}  Default= no/false.  Designates this line as a switch for graphics and algorithmic purposes. 
        SIDE EFFECT: Sets r1 = 1.0; x1 = 1.0; r0 = 1.0; x0 = 1.0; c1 = 1.1 ; c0 = 1.0;  length = 0.001; You must reset if you want something different.

        DSS property name: `Switch`, DSS property index: 15.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 15)
        ]
    @Switch.setter
    def Switch(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 15, value)

    @property
    def Rg(self) -> BatchFloat64ArrayProxy:
        """
        Carson earth return resistance per unit length used to compute impedance values at base frequency. Default is 0.01805 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.

        DSS property name: `Rg`, DSS property index: 16.
        """
        return BatchFloat64ArrayProxy(self, 16)

    @Rg.setter
    def Rg(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(16, value)

    @property
    def Xg(self) -> BatchFloat64ArrayProxy:
        """
        Carson earth return reactance per unit length used to compute impedance values at base frequency.  For making better frequency adjustments. Default is 0.155081 = 60 Hz value in ohms per kft (matches default line impedances). This value is required for harmonic solutions if you wish to adjust the earth return impedances for frequency. If not, set both Rg and Xg = 0.

        DSS property name: `Xg`, DSS property index: 17.
        """
        return BatchFloat64ArrayProxy(self, 17)

    @Xg.setter
    def Xg(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(17, value)

    @property
    def rho(self) -> BatchFloat64ArrayProxy:
        """
        Default=100 meter ohms.  Earth resitivity used to compute earth correction factor. Overrides Line geometry definition if specified.

        DSS property name: `rho`, DSS property index: 18.
        """
        return BatchFloat64ArrayProxy(self, 18)

    @rho.setter
    def rho(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(18, value)

    @property
    def geometry(self) -> List[str]:
        """
        Geometry code for LineGeometry Object. Supercedes any previous definition of line impedance. Line constants are computed for each frequency change or rho change. CAUTION: may alter number of phases. You cannot subsequently change the number of phases unless you change how the line impedance is defined.

        DSS property name: `geometry`, DSS property index: 19.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 19)

    @geometry.setter
    def geometry(self, value: Union[AnyStr, LineGeometry, List[AnyStr], List[LineGeometry]]):
        self._set_batch_obj_prop(19, value)

    @property
    def geometry_obj(self) -> List[LineGeometry]:
        """
        Geometry code for LineGeometry Object. Supercedes any previous definition of line impedance. Line constants are computed for each frequency change or rho change. CAUTION: may alter number of phases. You cannot subsequently change the number of phases unless you change how the line impedance is defined.

        DSS property name: `geometry`, DSS property index: 19.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 19)

    @geometry_obj.setter
    def geometry_obj(self, value: LineGeometry):
        self._set_batch_string(19, value)

    @property
    def units(self) -> BatchInt32ArrayProxy:
        """
        Length Units = {none | mi|kft|km|m|Ft|in|cm } Default is None - assumes length units match impedance units.

        DSS property name: `units`, DSS property index: 20.
        """
        return BatchInt32ArrayProxy(self, 20)

    @units.setter
    def units(self, value: Union[AnyStr, int, DimensionUnits, List[AnyStr], List[int], List[DimensionUnits], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(20, value)
            return
    
        self._set_batch_int32_array(20, value)

    @property
    def units_str(self) -> str:
        """
        Length Units = {none | mi|kft|km|m|Ft|in|cm } Default is None - assumes length units match impedance units.

        DSS property name: `units`, DSS property index: 20.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 20)

    @units_str.setter
    def units_str(self, value: AnyStr):
        self.units = value

    @property
    def spacing(self) -> List[str]:
        """
        Reference to a LineSpacing for use in a line constants calculation.
        Must be used in conjunction with the Wires property.
        Specify this before the wires property.

        DSS property name: `spacing`, DSS property index: 21.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 21)

    @spacing.setter
    def spacing(self, value: Union[AnyStr, LineSpacing, List[AnyStr], List[LineSpacing]]):
        self._set_batch_obj_prop(21, value)

    @property
    def spacing_obj(self) -> List[LineSpacing]:
        """
        Reference to a LineSpacing for use in a line constants calculation.
        Must be used in conjunction with the Wires property.
        Specify this before the wires property.

        DSS property name: `spacing`, DSS property index: 21.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 21)

    @spacing_obj.setter
    def spacing_obj(self, value: LineSpacing):
        self._set_batch_string(21, value)

    @property
    def wires(self) -> List[List[str]]:
        """
        Array of WireData names for use in an overhead line constants calculation.
        Must be used in conjunction with the Spacing property.
        Specify the Spacing first, and "ncond" wires.
        May also be used to specify bare neutrals with cables, using "ncond-nphase" wires.

        DSS property name: `wires`, DSS property index: 22.
        """
        return self._get_string_ll(22)

    @wires.setter
    def wires(self, value: Union[List[AnyStr], List[WireData]]):
        if (not len(value)) or isinstance(value[0], (bytes, str)) or (len(value[0]) and isinstance(value[0][0], (bytes, str))):
            self._set_batch_stringlist_prop(22, value)
            return

        self._set_batch_objlist_prop(22, value)

    @property
    def wires_obj(self) -> List[List[WireData]]:
        """
        Array of WireData names for use in an overhead line constants calculation.
        Must be used in conjunction with the Spacing property.
        Specify the Spacing first, and "ncond" wires.
        May also be used to specify bare neutrals with cables, using "ncond-nphase" wires.

        DSS property name: `wires`, DSS property index: 22.
        """
        return self._get_obj_ll(22, WireData)

    @wires_obj.setter
    def wires_obj(self, value: List[WireData]):
        self._set_batch_objlist_prop(22, value)

    @property
    def earthmodel(self) -> BatchInt32ArrayProxy:
        """
        One of {Carson | FullCarson | Deri}. Default is the global value established with the Set EarthModel command. See the Options Help on EarthModel option. This is used to override the global value for this line. This option applies only when the "geometry" property is used.

        DSS property name: `EarthModel`, DSS property index: 23.
        """
        return BatchInt32ArrayProxy(self, 23)

    @earthmodel.setter
    def earthmodel(self, value: Union[AnyStr, int, EarthModel, List[AnyStr], List[int], List[EarthModel], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(23, value)
            return
    
        self._set_batch_int32_array(23, value)

    @property
    def earthmodel_str(self) -> str:
        """
        One of {Carson | FullCarson | Deri}. Default is the global value established with the Set EarthModel command. See the Options Help on EarthModel option. This is used to override the global value for this line. This option applies only when the "geometry" property is used.

        DSS property name: `EarthModel`, DSS property index: 23.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 23)

    @earthmodel_str.setter
    def earthmodel_str(self, value: AnyStr):
        self.earthmodel = value

    @property
    def cncables(self) -> List[List[str]]:
        """
        Array of CNData names for use in a cable constants calculation.
        Must be used in conjunction with the Spacing property.
        Specify the Spacing first, using "nphases" cncables.
        You may later specify "nconds-nphases" wires for separate neutrals

        DSS property name: `cncables`, DSS property index: 24.
        """
        return self._get_string_ll(24)

    @cncables.setter
    def cncables(self, value: Union[List[AnyStr], List[CNData]]):
        if (not len(value)) or isinstance(value[0], (bytes, str)) or (len(value[0]) and isinstance(value[0][0], (bytes, str))):
            self._set_batch_stringlist_prop(24, value)
            return

        self._set_batch_objlist_prop(24, value)

    @property
    def cncables_obj(self) -> List[List[CNData]]:
        """
        Array of CNData names for use in a cable constants calculation.
        Must be used in conjunction with the Spacing property.
        Specify the Spacing first, using "nphases" cncables.
        You may later specify "nconds-nphases" wires for separate neutrals

        DSS property name: `cncables`, DSS property index: 24.
        """
        return self._get_obj_ll(24, CNData)

    @cncables_obj.setter
    def cncables_obj(self, value: List[CNData]):
        self._set_batch_objlist_prop(24, value)

    @property
    def tscables(self) -> List[List[str]]:
        """
        Array of TSData names for use in a cable constants calculation.
        Must be used in conjunction with the Spacing property.
        Specify the Spacing first, using "nphases" tscables.
        You may later specify "nconds-nphases" wires for separate neutrals

        DSS property name: `tscables`, DSS property index: 25.
        """
        return self._get_string_ll(25)

    @tscables.setter
    def tscables(self, value: Union[List[AnyStr], List[TSData]]):
        if (not len(value)) or isinstance(value[0], (bytes, str)) or (len(value[0]) and isinstance(value[0][0], (bytes, str))):
            self._set_batch_stringlist_prop(25, value)
            return

        self._set_batch_objlist_prop(25, value)

    @property
    def tscables_obj(self) -> List[List[TSData]]:
        """
        Array of TSData names for use in a cable constants calculation.
        Must be used in conjunction with the Spacing property.
        Specify the Spacing first, using "nphases" tscables.
        You may later specify "nconds-nphases" wires for separate neutrals

        DSS property name: `tscables`, DSS property index: 25.
        """
        return self._get_obj_ll(25, TSData)

    @tscables_obj.setter
    def tscables_obj(self, value: List[TSData]):
        self._set_batch_objlist_prop(25, value)

    @property
    def B1(self) -> BatchFloat64ArrayProxy:
        """
        Alternate way to specify C1. MicroS per unit length

        DSS property name: `B1`, DSS property index: 26.
        """
        return BatchFloat64ArrayProxy(self, 26)

    @B1.setter
    def B1(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(26, value)

    @property
    def B0(self) -> BatchFloat64ArrayProxy:
        """
        Alternate way to specify C0. MicroS per unit length

        DSS property name: `B0`, DSS property index: 27.
        """
        return BatchFloat64ArrayProxy(self, 27)

    @B0.setter
    def B0(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(27, value)

    @property
    def Seasons(self) -> BatchInt32ArrayProxy:
        """
        Defines the number of ratings to be defined for the wire, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 28.
        """
        return BatchInt32ArrayProxy(self, 28)

    @Seasons.setter
    def Seasons(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(28, value)

    @property
    def Ratings(self) -> List[Float64Array]:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in lines.

        DSS property name: `Ratings`, DSS property index: 29.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 29)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(29, value)

    @property
    def linetype(self) -> BatchInt32ArrayProxy:
        """
        Code designating the type of line. 
        One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW

        OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.

        DSS property name: `LineType`, DSS property index: 30.
        """
        return BatchInt32ArrayProxy(self, 30)

    @linetype.setter
    def linetype(self, value: Union[AnyStr, int, LineType, List[AnyStr], List[int], List[LineType], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(30, value)
            return
    
        self._set_batch_int32_array(30, value)

    @property
    def linetype_str(self) -> str:
        """
        Code designating the type of line. 
        One of: OH, UG, UG_TS, UG_CN, SWT_LDBRK, SWT_FUSE, SWT_SECT, SWT_REC, SWT_DISC, SWT_BRK, SWT_ELBOW

        OpenDSS currently does not use this internally. For whatever purpose the user defines. Default is OH.

        DSS property name: `LineType`, DSS property index: 30.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 30)

    @linetype_str.setter
    def linetype_str(self, value: AnyStr):
        self.linetype = value

    @property
    def normamps(self) -> BatchFloat64ArrayProxy:
        """
        Normal rated current.

        DSS property name: `normamps`, DSS property index: 31.
        """
        return BatchFloat64ArrayProxy(self, 31)

    @normamps.setter
    def normamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(31, value)

    @property
    def emergamps(self) -> BatchFloat64ArrayProxy:
        """
        Maximum or emerg current.

        DSS property name: `emergamps`, DSS property index: 32.
        """
        return BatchFloat64ArrayProxy(self, 32)

    @emergamps.setter
    def emergamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(32, value)

    @property
    def faultrate(self) -> BatchFloat64ArrayProxy:
        """
        Failure rate PER UNIT LENGTH per year. Length must be same units as LENGTH property. Default is 0.1 fault per unit length per year.

        DSS property name: `faultrate`, DSS property index: 33.
        """
        return BatchFloat64ArrayProxy(self, 33)

    @faultrate.setter
    def faultrate(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(33, value)

    @property
    def pctperm(self) -> BatchFloat64ArrayProxy:
        """
        Percent of failures that become permanent. Default is 20.

        DSS property name: `pctperm`, DSS property index: 34.
        """
        return BatchFloat64ArrayProxy(self, 34)

    @pctperm.setter
    def pctperm(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(34, value)

    @property
    def repair(self) -> BatchFloat64ArrayProxy:
        """
        Hours to repair. Default is 3 hr.

        DSS property name: `repair`, DSS property index: 35.
        """
        return BatchFloat64ArrayProxy(self, 35)

    @repair.setter
    def repair(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(35, value)

    @property
    def basefreq(self) -> BatchFloat64ArrayProxy:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 36.
        """
        return BatchFloat64ArrayProxy(self, 36)

    @basefreq.setter
    def basefreq(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(36, value)

    @property
    def enabled(self) -> List[bool]:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 37.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 37)
        ]
    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 37, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 38.
        """
        self._set_batch_string(38, value)

class VsourceBatch(DSSBatch):
    _cls_name = 'Vsource'
    _obj_cls = Vsource
    _cls_idx = 16


    @property
    def bus1(self) -> List[str]:
        """
        Name of bus to which the main terminal (1) is connected.
        bus1=busname
        bus1=busname.1.2.3

        The VSOURCE object is a two-terminal voltage source (thevenin equivalent). Bus2 defaults to Bus1 with all phases connected to ground (node 0) unless previously specified. This is a Yg connection. If you want something different, define the Bus2 property ezplicitly.

        DSS property name: `bus1`, DSS property index: 1.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 1) 

    @bus1.setter
    def bus1(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(1, value)

    @property
    def basekv(self) -> BatchFloat64ArrayProxy:
        """
        Base Source kV, usually phase-phase (L-L) unless you are making a positive-sequence model or 1-phase modelin which case, it will be phase-neutral (L-N) kV.

        DSS property name: `basekv`, DSS property index: 2.
        """
        return BatchFloat64ArrayProxy(self, 2)

    @basekv.setter
    def basekv(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(2, value)

    @property
    def pu(self) -> BatchFloat64ArrayProxy:
        """
        Per unit of the base voltage that the source is actually operating at.
        "pu=1.05"

        DSS property name: `pu`, DSS property index: 3.
        """
        return BatchFloat64ArrayProxy(self, 3)

    @pu.setter
    def pu(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(3, value)

    @property
    def angle(self) -> BatchFloat64ArrayProxy:
        """
        Phase angle in degrees of first phase: e.g.,Angle=10.3

        DSS property name: `angle`, DSS property index: 4.
        """
        return BatchFloat64ArrayProxy(self, 4)

    @angle.setter
    def angle(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(4, value)

    @property
    def frequency(self) -> BatchFloat64ArrayProxy:
        """
        Source frequency.  Defaults to system default base frequency.

        DSS property name: `frequency`, DSS property index: 5.
        """
        return BatchFloat64ArrayProxy(self, 5)

    @frequency.setter
    def frequency(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(5, value)

    @property
    def phases(self) -> BatchInt32ArrayProxy:
        """
        Number of phases.  Defaults to 3.

        DSS property name: `phases`, DSS property index: 6.
        """
        return BatchInt32ArrayProxy(self, 6)

    @phases.setter
    def phases(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(6, value)

    @property
    def MVAsc3(self) -> BatchFloat64ArrayProxy:
        """
        MVA Short circuit, 3-phase fault. Default = 2000. Z1 is determined by squaring the base kv and dividing by this value. For single-phase source, this value is not used.

        DSS property name: `MVAsc3`, DSS property index: 7.
        """
        return BatchFloat64ArrayProxy(self, 7)

    @MVAsc3.setter
    def MVAsc3(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(7, value)

    @property
    def MVAsc1(self) -> BatchFloat64ArrayProxy:
        """
        MVA Short Circuit, 1-phase fault. Default = 2100. The "single-phase impedance", Zs, is determined by squaring the base kV and dividing by this value. Then Z0 is determined by Z0 = 3Zs - 2Z1.  For 1-phase sources, Zs is used directly. Use X0R0 to define X/R ratio for 1-phase source.

        DSS property name: `MVAsc1`, DSS property index: 8.
        """
        return BatchFloat64ArrayProxy(self, 8)

    @MVAsc1.setter
    def MVAsc1(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(8, value)

    @property
    def x1r1(self) -> BatchFloat64ArrayProxy:
        """
        Positive-sequence  X/R ratio. Default = 4.

        DSS property name: `x1r1`, DSS property index: 9.
        """
        return BatchFloat64ArrayProxy(self, 9)

    @x1r1.setter
    def x1r1(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(9, value)

    @property
    def x0r0(self) -> BatchFloat64ArrayProxy:
        """
        Zero-sequence X/R ratio.Default = 3.

        DSS property name: `x0r0`, DSS property index: 10.
        """
        return BatchFloat64ArrayProxy(self, 10)

    @x0r0.setter
    def x0r0(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(10, value)

    @property
    def Isc3(self) -> BatchFloat64ArrayProxy:
        """
        Alternate method of defining the source impedance. 
        3-phase short circuit current, amps.  Default is 10000.

        DSS property name: `Isc3`, DSS property index: 11.
        """
        return BatchFloat64ArrayProxy(self, 11)

    @Isc3.setter
    def Isc3(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(11, value)

    @property
    def Isc1(self) -> BatchFloat64ArrayProxy:
        """
        Alternate method of defining the source impedance. 
        single-phase short circuit current, amps.  Default is 10500.

        DSS property name: `Isc1`, DSS property index: 12.
        """
        return BatchFloat64ArrayProxy(self, 12)

    @Isc1.setter
    def Isc1(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(12, value)

    @property
    def R1(self) -> BatchFloat64ArrayProxy:
        """
        Alternate method of defining the source impedance. 
        Positive-sequence resistance, ohms.  Default is 1.65.

        DSS property name: `R1`, DSS property index: 13.
        """
        return BatchFloat64ArrayProxy(self, 13)

    @R1.setter
    def R1(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(13, value)

    @property
    def X1(self) -> BatchFloat64ArrayProxy:
        """
        Alternate method of defining the source impedance. 
        Positive-sequence reactance, ohms.  Default is 6.6.

        DSS property name: `X1`, DSS property index: 14.
        """
        return BatchFloat64ArrayProxy(self, 14)

    @X1.setter
    def X1(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(14, value)

    @property
    def R0(self) -> BatchFloat64ArrayProxy:
        """
        Alternate method of defining the source impedance. 
        Zero-sequence resistance, ohms.  Default is 1.9.

        DSS property name: `R0`, DSS property index: 15.
        """
        return BatchFloat64ArrayProxy(self, 15)

    @R0.setter
    def R0(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(15, value)

    @property
    def X0(self) -> BatchFloat64ArrayProxy:
        """
        Alternate method of defining the source impedance. 
        Zero-sequence reactance, ohms.  Default is 5.7.

        DSS property name: `X0`, DSS property index: 16.
        """
        return BatchFloat64ArrayProxy(self, 16)

    @X0.setter
    def X0(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(16, value)

    @property
    def scantype(self) -> BatchInt32ArrayProxy:
        """
        {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.

        DSS property name: `ScanType`, DSS property index: 17.
        """
        return BatchInt32ArrayProxy(self, 17)

    @scantype.setter
    def scantype(self, value: Union[AnyStr, int, ScanType, List[AnyStr], List[int], List[ScanType], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(17, value)
            return
    
        self._set_batch_int32_array(17, value)

    @property
    def scantype_str(self) -> str:
        """
        {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.

        DSS property name: `ScanType`, DSS property index: 17.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 17)

    @scantype_str.setter
    def scantype_str(self, value: AnyStr):
        self.scantype = value

    @property
    def Sequence(self) -> BatchInt32ArrayProxy:
        """
        {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence. 

        DSS property name: `Sequence`, DSS property index: 18.
        """
        return BatchInt32ArrayProxy(self, 18)

    @Sequence.setter
    def Sequence(self, value: Union[AnyStr, int, SequenceType, List[AnyStr], List[int], List[SequenceType], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(18, value)
            return
    
        self._set_batch_int32_array(18, value)

    @property
    def Sequence_str(self) -> str:
        """
        {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence. 

        DSS property name: `Sequence`, DSS property index: 18.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 18)

    @Sequence_str.setter
    def Sequence_str(self, value: AnyStr):
        self.Sequence = value

    @property
    def bus2(self) -> List[str]:
        """
        Name of bus to which 2nd terminal is connected.
        bus2=busname
        bus2=busname.1.2.3

        Default is Bus1.0.0.0 (grounded wye connection)

        DSS property name: `bus2`, DSS property index: 19.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 19) 

    @bus2.setter
    def bus2(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(19, value)

    @property
    def Z2(self) -> List[complex]:
        """
        Negative-sequence equivalent source impedance, ohms, as a 2-element array representing a complex number. Example: 

        Z2=[1, 2]  ! represents 1 + j2 

        Used to define the impedance matrix of the VSOURCE if Z1 is also specified. 

        Note: Z2 defaults to Z1 if it is not specifically defined. If Z2 is not equal to Z1, the impedance matrix is asymmetrical.

        DSS property name: `Z2`, DSS property index: 22.
        """
        return [   
            self._get_float64_array(
                self._lib.Obj_GetFloat64Array, 
                x,
                22,
            ).astype(complex)[0]
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Z2.setter
    def Z2(self, value: Union[complex, List[complex]]):
        if isinstance(value, complex):
            value, value_ptr, value_count = self._prepare_float64_array([value.real, value.imag])
            for x in self._ffi.unpack(self.pointer[0], self.count[0]):
                self._lib.Obj_SetFloat64Array(x, 22, value_ptr, value_count)
            return

        values = value
        if len(values) != self.count[0]:
            raise ValueError('Number of elements provided must match the number of objects in the batch.')

        value, value_ptr, value_count = self._prepare_float64_array([0, 0])
        for v, x in zip(values, self._ffi.unpack(self.pointer[0], self.count[0])):
            value[0] = v.real
            value[1] = v.imag
            self._lib.Obj_SetFloat64Array(x, 22, value_ptr, value_count)

    @property
    def puZ1(self) -> List[complex]:
        """
        2-element array: e.g., [1  2]. An alternate way to specify Z1. See Z1 property. Per-unit positive-sequence impedance on base of Vsource BasekV and BaseMVA.

        DSS property name: `puZ1`, DSS property index: 23.
        """
        return [   
            self._get_float64_array(
                self._lib.Obj_GetFloat64Array, 
                x,
                23,
            ).astype(complex)[0]
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @puZ1.setter
    def puZ1(self, value: Union[complex, List[complex]]):
        if isinstance(value, complex):
            value, value_ptr, value_count = self._prepare_float64_array([value.real, value.imag])
            for x in self._ffi.unpack(self.pointer[0], self.count[0]):
                self._lib.Obj_SetFloat64Array(x, 23, value_ptr, value_count)
            return

        values = value
        if len(values) != self.count[0]:
            raise ValueError('Number of elements provided must match the number of objects in the batch.')

        value, value_ptr, value_count = self._prepare_float64_array([0, 0])
        for v, x in zip(values, self._ffi.unpack(self.pointer[0], self.count[0])):
            value[0] = v.real
            value[1] = v.imag
            self._lib.Obj_SetFloat64Array(x, 23, value_ptr, value_count)

    @property
    def puZ0(self) -> List[complex]:
        """
        2-element array: e.g., [1  2]. An alternate way to specify Z0. See Z0 property. Per-unit zero-sequence impedance on base of Vsource BasekV and BaseMVA.

        DSS property name: `puZ0`, DSS property index: 24.
        """
        return [   
            self._get_float64_array(
                self._lib.Obj_GetFloat64Array, 
                x,
                24,
            ).astype(complex)[0]
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @puZ0.setter
    def puZ0(self, value: Union[complex, List[complex]]):
        if isinstance(value, complex):
            value, value_ptr, value_count = self._prepare_float64_array([value.real, value.imag])
            for x in self._ffi.unpack(self.pointer[0], self.count[0]):
                self._lib.Obj_SetFloat64Array(x, 24, value_ptr, value_count)
            return

        values = value
        if len(values) != self.count[0]:
            raise ValueError('Number of elements provided must match the number of objects in the batch.')

        value, value_ptr, value_count = self._prepare_float64_array([0, 0])
        for v, x in zip(values, self._ffi.unpack(self.pointer[0], self.count[0])):
            value[0] = v.real
            value[1] = v.imag
            self._lib.Obj_SetFloat64Array(x, 24, value_ptr, value_count)

    @property
    def puZ2(self) -> List[complex]:
        """
        2-element array: e.g., [1  2]. An alternate way to specify Z2. See Z2 property. Per-unit negative-sequence impedance on base of Vsource BasekV and BaseMVA.

        DSS property name: `puZ2`, DSS property index: 25.
        """
        return [   
            self._get_float64_array(
                self._lib.Obj_GetFloat64Array, 
                x,
                25,
            ).astype(complex)[0]
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @puZ2.setter
    def puZ2(self, value: Union[complex, List[complex]]):
        if isinstance(value, complex):
            value, value_ptr, value_count = self._prepare_float64_array([value.real, value.imag])
            for x in self._ffi.unpack(self.pointer[0], self.count[0]):
                self._lib.Obj_SetFloat64Array(x, 25, value_ptr, value_count)
            return

        values = value
        if len(values) != self.count[0]:
            raise ValueError('Number of elements provided must match the number of objects in the batch.')

        value, value_ptr, value_count = self._prepare_float64_array([0, 0])
        for v, x in zip(values, self._ffi.unpack(self.pointer[0], self.count[0])):
            value[0] = v.real
            value[1] = v.imag
            self._lib.Obj_SetFloat64Array(x, 25, value_ptr, value_count)

    @property
    def baseMVA(self) -> BatchFloat64ArrayProxy:
        """
        Default value is 100. Base used to convert values specifiied with puZ1, puZ0, and puZ2 properties to ohms on kV base specified by BasekV property.

        DSS property name: `baseMVA`, DSS property index: 26.
        """
        return BatchFloat64ArrayProxy(self, 26)

    @baseMVA.setter
    def baseMVA(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(26, value)

    @property
    def Yearly(self) -> List[str]:
        """
        LOADSHAPE object to use for the per-unit voltage for YEARLY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.

        Must be previously defined as a LOADSHAPE object. 

        Is set to the Daily load shape when Daily is defined.  The daily load shape is repeated in this case. Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Yearly`, DSS property index: 27.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 27)

    @Yearly.setter
    def Yearly(self, value: Union[AnyStr, LoadShape, List[AnyStr], List[LoadShape]]):
        self._set_batch_obj_prop(27, value)

    @property
    def Yearly_obj(self) -> List[LoadShape]:
        """
        LOADSHAPE object to use for the per-unit voltage for YEARLY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.

        Must be previously defined as a LOADSHAPE object. 

        Is set to the Daily load shape when Daily is defined.  The daily load shape is repeated in this case. Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Yearly`, DSS property index: 27.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 27)

    @Yearly_obj.setter
    def Yearly_obj(self, value: LoadShape):
        self._set_batch_string(27, value)

    @property
    def Daily(self) -> List[str]:
        """
        LOADSHAPE object to use for the per-unit voltage for DAILY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.

        Must be previously defined as a LOADSHAPE object. 

        Sets Yearly curve if it is not already defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Daily`, DSS property index: 28.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 28)

    @Daily.setter
    def Daily(self, value: Union[AnyStr, LoadShape, List[AnyStr], List[LoadShape]]):
        self._set_batch_obj_prop(28, value)

    @property
    def Daily_obj(self) -> List[LoadShape]:
        """
        LOADSHAPE object to use for the per-unit voltage for DAILY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.

        Must be previously defined as a LOADSHAPE object. 

        Sets Yearly curve if it is not already defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Daily`, DSS property index: 28.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 28)

    @Daily_obj.setter
    def Daily_obj(self, value: LoadShape):
        self._set_batch_string(28, value)

    @property
    def Duty(self) -> List[str]:
        """
        LOADSHAPE object to use for the per-unit voltage for DUTYCYCLE-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.

        Must be previously defined as a LOADSHAPE object. 

        Defaults to Daily load shape when Daily is defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Duty`, DSS property index: 29.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 29)

    @Duty.setter
    def Duty(self, value: Union[AnyStr, LoadShape, List[AnyStr], List[LoadShape]]):
        self._set_batch_obj_prop(29, value)

    @property
    def Duty_obj(self) -> List[LoadShape]:
        """
        LOADSHAPE object to use for the per-unit voltage for DUTYCYCLE-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual L-N kV.

        Must be previously defined as a LOADSHAPE object. 

        Defaults to Daily load shape when Daily is defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Duty`, DSS property index: 29.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 29)

    @Duty_obj.setter
    def Duty_obj(self, value: LoadShape):
        self._set_batch_string(29, value)

    @property
    def Model(self) -> BatchInt32ArrayProxy:
        """
        {Thevenin* | Ideal}  Specifies whether the Vsource is to be considered a Thevenin short circuit model or a quasi-ideal voltage source. If Thevenin, the Vsource uses the impedances defined for all calculations. If "Ideal", the model uses a small impedance on the diagonal of the impedance matrix for the fundamental base frequency power flow only. Then switches to actual Thevenin model for other frequencies. 

        DSS property name: `Model`, DSS property index: 30.
        """
        return BatchInt32ArrayProxy(self, 30)

    @Model.setter
    def Model(self, value: Union[AnyStr, int, Vsource.VSourceModel, List[AnyStr], List[int], List[Vsource.VSourceModel], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(30, value)
            return
    
        self._set_batch_int32_array(30, value)

    @property
    def Model_str(self) -> str:
        """
        {Thevenin* | Ideal}  Specifies whether the Vsource is to be considered a Thevenin short circuit model or a quasi-ideal voltage source. If Thevenin, the Vsource uses the impedances defined for all calculations. If "Ideal", the model uses a small impedance on the diagonal of the impedance matrix for the fundamental base frequency power flow only. Then switches to actual Thevenin model for other frequencies. 

        DSS property name: `Model`, DSS property index: 30.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 30)

    @Model_str.setter
    def Model_str(self, value: AnyStr):
        self.Model = value

    @property
    def puZideal(self) -> List[complex]:
        """
        2-element array: e.g., [1  2]. The pu impedance to use for the quasi-ideal voltage source model. Should be a very small impedances. Default is [1e-6, 0.001]. Per-unit impedance on base of Vsource BasekV and BaseMVA. If too small, solution may not work. Be sure to check the voltage values and powers.

        DSS property name: `puZideal`, DSS property index: 31.
        """
        return [   
            self._get_float64_array(
                self._lib.Obj_GetFloat64Array, 
                x,
                31,
            ).astype(complex)[0]
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @puZideal.setter
    def puZideal(self, value: Union[complex, List[complex]]):
        if isinstance(value, complex):
            value, value_ptr, value_count = self._prepare_float64_array([value.real, value.imag])
            for x in self._ffi.unpack(self.pointer[0], self.count[0]):
                self._lib.Obj_SetFloat64Array(x, 31, value_ptr, value_count)
            return

        values = value
        if len(values) != self.count[0]:
            raise ValueError('Number of elements provided must match the number of objects in the batch.')

        value, value_ptr, value_count = self._prepare_float64_array([0, 0])
        for v, x in zip(values, self._ffi.unpack(self.pointer[0], self.count[0])):
            value[0] = v.real
            value[1] = v.imag
            self._lib.Obj_SetFloat64Array(x, 31, value_ptr, value_count)

    @property
    def spectrum(self) -> List[str]:
        """
        Name of harmonic spectrum for this source.  Default is "defaultvsource", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 32.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 32)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum, List[AnyStr], List[Spectrum]]):
        self._set_batch_obj_prop(32, value)

    @property
    def spectrum_obj(self) -> List[Spectrum]:
        """
        Name of harmonic spectrum for this source.  Default is "defaultvsource", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 32.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 32)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_batch_string(32, value)

    @property
    def basefreq(self) -> BatchFloat64ArrayProxy:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 33.
        """
        return BatchFloat64ArrayProxy(self, 33)

    @basefreq.setter
    def basefreq(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(33, value)

    @property
    def enabled(self) -> List[bool]:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 34.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 34)
        ]
    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 34, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 35.
        """
        self._set_batch_string(35, value)

class IsourceBatch(DSSBatch):
    _cls_name = 'Isource'
    _obj_cls = Isource
    _cls_idx = 17


    @property
    def bus1(self) -> List[str]:
        """
        Name of bus to which source is connected.
        bus1=busname
        bus1=busname.1.2.3

        DSS property name: `bus1`, DSS property index: 1.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 1) 

    @bus1.setter
    def bus1(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(1, value)

    @property
    def amps(self) -> BatchFloat64ArrayProxy:
        """
        Magnitude of current source, each phase, in Amps.

        DSS property name: `amps`, DSS property index: 2.
        """
        return BatchFloat64ArrayProxy(self, 2)

    @amps.setter
    def amps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(2, value)

    @property
    def angle(self) -> BatchFloat64ArrayProxy:
        """
        Phase angle in degrees of first phase: e.g.,Angle=10.3.
        Phase shift between phases is assumed 120 degrees when number of phases <= 3

        DSS property name: `angle`, DSS property index: 3.
        """
        return BatchFloat64ArrayProxy(self, 3)

    @angle.setter
    def angle(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(3, value)

    @property
    def frequency(self) -> BatchFloat64ArrayProxy:
        """
        Source frequency.  Defaults to  circuit fundamental frequency.

        DSS property name: `frequency`, DSS property index: 4.
        """
        return BatchFloat64ArrayProxy(self, 4)

    @frequency.setter
    def frequency(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(4, value)

    @property
    def phases(self) -> BatchInt32ArrayProxy:
        """
        Number of phases.  Defaults to 3. For 3 or less, phase shift is 120 degrees.

        DSS property name: `phases`, DSS property index: 5.
        """
        return BatchInt32ArrayProxy(self, 5)

    @phases.setter
    def phases(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(5, value)

    @property
    def scantype(self) -> BatchInt32ArrayProxy:
        """
        {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.

        DSS property name: `scantype`, DSS property index: 6.
        """
        return BatchInt32ArrayProxy(self, 6)

    @scantype.setter
    def scantype(self, value: Union[AnyStr, int, ScanType, List[AnyStr], List[int], List[ScanType], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(6, value)
            return
    
        self._set_batch_int32_array(6, value)

    @property
    def scantype_str(self) -> str:
        """
        {pos*| zero | none} Maintain specified sequence for harmonic solution. Default is positive sequence. Otherwise, angle between phases rotates with harmonic.

        DSS property name: `scantype`, DSS property index: 6.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 6)

    @scantype_str.setter
    def scantype_str(self, value: AnyStr):
        self.scantype = value

    @property
    def sequence(self) -> BatchInt32ArrayProxy:
        """
        {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence. 

        DSS property name: `sequence`, DSS property index: 7.
        """
        return BatchInt32ArrayProxy(self, 7)

    @sequence.setter
    def sequence(self, value: Union[AnyStr, int, SequenceType, List[AnyStr], List[int], List[SequenceType], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(7, value)
            return
    
        self._set_batch_int32_array(7, value)

    @property
    def sequence_str(self) -> str:
        """
        {pos*| neg | zero} Set the phase angles for the specified symmetrical component sequence for non-harmonic solution modes. Default is positive sequence. 

        DSS property name: `sequence`, DSS property index: 7.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 7)

    @sequence_str.setter
    def sequence_str(self, value: AnyStr):
        self.sequence = value

    @property
    def Yearly(self) -> List[str]:
        """
        LOADSHAPE object to use for the per-unit current for YEARLY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual Amp.

        Must be previously defined as a LOADSHAPE object. 

        Is set to the Daily load shape when Daily is defined.  The daily load shape is repeated in this case. Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Yearly`, DSS property index: 8.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 8)

    @Yearly.setter
    def Yearly(self, value: Union[AnyStr, LoadShape, List[AnyStr], List[LoadShape]]):
        self._set_batch_obj_prop(8, value)

    @property
    def Yearly_obj(self) -> List[LoadShape]:
        """
        LOADSHAPE object to use for the per-unit current for YEARLY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual Amp.

        Must be previously defined as a LOADSHAPE object. 

        Is set to the Daily load shape when Daily is defined.  The daily load shape is repeated in this case. Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Yearly`, DSS property index: 8.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 8)

    @Yearly_obj.setter
    def Yearly_obj(self, value: LoadShape):
        self._set_batch_string(8, value)

    @property
    def Daily(self) -> List[str]:
        """
        LOADSHAPE object to use for the per-unit current for DAILY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual A.

        Must be previously defined as a LOADSHAPE object. 

        Sets Yearly curve if it is not already defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Daily`, DSS property index: 9.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 9)

    @Daily.setter
    def Daily(self, value: Union[AnyStr, LoadShape, List[AnyStr], List[LoadShape]]):
        self._set_batch_obj_prop(9, value)

    @property
    def Daily_obj(self) -> List[LoadShape]:
        """
        LOADSHAPE object to use for the per-unit current for DAILY-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual A.

        Must be previously defined as a LOADSHAPE object. 

        Sets Yearly curve if it is not already defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Daily`, DSS property index: 9.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 9)

    @Daily_obj.setter
    def Daily_obj(self, value: LoadShape):
        self._set_batch_string(9, value)

    @property
    def Duty(self) -> List[str]:
        """
        LOADSHAPE object to use for the per-unit current for DUTYCYCLE-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual A.

        Must be previously defined as a LOADSHAPE object. 

        Defaults to Daily load shape when Daily is defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Duty`, DSS property index: 10.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 10)

    @Duty.setter
    def Duty(self, value: Union[AnyStr, LoadShape, List[AnyStr], List[LoadShape]]):
        self._set_batch_obj_prop(10, value)

    @property
    def Duty_obj(self) -> List[LoadShape]:
        """
        LOADSHAPE object to use for the per-unit current for DUTYCYCLE-mode simulations. Set the Mult property of the LOADSHAPE to the pu curve. Qmult is not used. If UseActual=Yes then the Mult curve should be actual A.

        Must be previously defined as a LOADSHAPE object. 

        Defaults to Daily load shape when Daily is defined.   Set to NONE to reset to no loadahape for Yearly mode. The default is no variation.

        DSS property name: `Duty`, DSS property index: 10.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 10)

    @Duty_obj.setter
    def Duty_obj(self, value: LoadShape):
        self._set_batch_string(10, value)

    @property
    def Bus2(self) -> List[str]:
        """
        Name of bus to which 2nd terminal is connected.
        bus2=busname
        bus2=busname.1.2.3

        Default is Bus1.0.0.0 (grounded-wye connection)

        DSS property name: `Bus2`, DSS property index: 11.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 11) 

    @Bus2.setter
    def Bus2(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(11, value)

    @property
    def spectrum(self) -> List[str]:
        """
        Harmonic spectrum assumed for this source.  Default is "default".

        DSS property name: `spectrum`, DSS property index: 12.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 12)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum, List[AnyStr], List[Spectrum]]):
        self._set_batch_obj_prop(12, value)

    @property
    def spectrum_obj(self) -> List[Spectrum]:
        """
        Harmonic spectrum assumed for this source.  Default is "default".

        DSS property name: `spectrum`, DSS property index: 12.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 12)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_batch_string(12, value)

    @property
    def basefreq(self) -> BatchFloat64ArrayProxy:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 13.
        """
        return BatchFloat64ArrayProxy(self, 13)

    @basefreq.setter
    def basefreq(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(13, value)

    @property
    def enabled(self) -> List[bool]:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 14.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 14)
        ]
    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 14, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 15.
        """
        self._set_batch_string(15, value)

class VCCSBatch(DSSBatch):
    _cls_name = 'VCCS'
    _obj_cls = VCCS
    _cls_idx = 18


    @property
    def bus1(self) -> List[str]:
        """
        Name of bus to which source is connected.
        bus1=busname
        bus1=busname.1.2.3

        DSS property name: `bus1`, DSS property index: 1.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 1) 

    @bus1.setter
    def bus1(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(1, value)

    @property
    def phases(self) -> BatchInt32ArrayProxy:
        """
        Number of phases.  Defaults to 1.

        DSS property name: `phases`, DSS property index: 2.
        """
        return BatchInt32ArrayProxy(self, 2)

    @phases.setter
    def phases(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(2, value)

    @property
    def prated(self) -> BatchFloat64ArrayProxy:
        """
        Total rated power, in Watts.

        DSS property name: `prated`, DSS property index: 3.
        """
        return BatchFloat64ArrayProxy(self, 3)

    @prated.setter
    def prated(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(3, value)

    @property
    def vrated(self) -> BatchFloat64ArrayProxy:
        """
        Rated line-to-line voltage, in Volts

        DSS property name: `vrated`, DSS property index: 4.
        """
        return BatchFloat64ArrayProxy(self, 4)

    @vrated.setter
    def vrated(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(4, value)

    @property
    def ppct(self) -> BatchFloat64ArrayProxy:
        """
        Steady-state operating output, in percent of rated.

        DSS property name: `ppct`, DSS property index: 5.
        """
        return BatchFloat64ArrayProxy(self, 5)

    @ppct.setter
    def ppct(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(5, value)

    @property
    def bp1(self) -> List[str]:
        """
        XYCurve defining the input piece-wise linear block.

        DSS property name: `bp1`, DSS property index: 6.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 6)

    @bp1.setter
    def bp1(self, value: Union[AnyStr, XYcurve, List[AnyStr], List[XYcurve]]):
        self._set_batch_obj_prop(6, value)

    @property
    def bp1_obj(self) -> List[XYcurve]:
        """
        XYCurve defining the input piece-wise linear block.

        DSS property name: `bp1`, DSS property index: 6.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 6)

    @bp1_obj.setter
    def bp1_obj(self, value: XYcurve):
        self._set_batch_string(6, value)

    @property
    def bp2(self) -> List[str]:
        """
        XYCurve defining the output piece-wise linear block.

        DSS property name: `bp2`, DSS property index: 7.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 7)

    @bp2.setter
    def bp2(self, value: Union[AnyStr, XYcurve, List[AnyStr], List[XYcurve]]):
        self._set_batch_obj_prop(7, value)

    @property
    def bp2_obj(self) -> List[XYcurve]:
        """
        XYCurve defining the output piece-wise linear block.

        DSS property name: `bp2`, DSS property index: 7.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 7)

    @bp2_obj.setter
    def bp2_obj(self, value: XYcurve):
        self._set_batch_string(7, value)

    @property
    def filter(self) -> List[str]:
        """
        XYCurve defining the digital filter coefficients (x numerator, y denominator).

        DSS property name: `filter`, DSS property index: 8.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 8)

    @filter.setter
    def filter(self, value: Union[AnyStr, XYcurve, List[AnyStr], List[XYcurve]]):
        self._set_batch_obj_prop(8, value)

    @property
    def filter_obj(self) -> List[XYcurve]:
        """
        XYCurve defining the digital filter coefficients (x numerator, y denominator).

        DSS property name: `filter`, DSS property index: 8.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 8)

    @filter_obj.setter
    def filter_obj(self, value: XYcurve):
        self._set_batch_string(8, value)

    @property
    def fsample(self) -> BatchFloat64ArrayProxy:
        """
        Sample frequency [Hz} for the digital filter.

        DSS property name: `fsample`, DSS property index: 9.
        """
        return BatchFloat64ArrayProxy(self, 9)

    @fsample.setter
    def fsample(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(9, value)

    @property
    def rmsmode(self) -> List[bool]:
        """
        True if only Hz is used to represent a phase-locked loop (PLL), ignoring the BP1, BP2 and time-domain transformations. Default is no.

        DSS property name: `rmsmode`, DSS property index: 10.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 10)
        ]
    @rmsmode.setter
    def rmsmode(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 10, value)

    @property
    def imaxpu(self) -> BatchFloat64ArrayProxy:
        """
        Maximum output current in per-unit of rated; defaults to 1.1

        DSS property name: `imaxpu`, DSS property index: 11.
        """
        return BatchFloat64ArrayProxy(self, 11)

    @imaxpu.setter
    def imaxpu(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(11, value)

    @property
    def vrmstau(self) -> BatchFloat64ArrayProxy:
        """
        Time constant in sensing Vrms for the PLL; defaults to 0.0015

        DSS property name: `vrmstau`, DSS property index: 12.
        """
        return BatchFloat64ArrayProxy(self, 12)

    @vrmstau.setter
    def vrmstau(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(12, value)

    @property
    def irmstau(self) -> BatchFloat64ArrayProxy:
        """
        Time constant in producing Irms from the PLL; defaults to 0.0015

        DSS property name: `irmstau`, DSS property index: 13.
        """
        return BatchFloat64ArrayProxy(self, 13)

    @irmstau.setter
    def irmstau(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(13, value)

    @property
    def spectrum(self) -> List[str]:
        """
        Harmonic spectrum assumed for this source.  Default is "default".

        DSS property name: `spectrum`, DSS property index: 14.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 14)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum, List[AnyStr], List[Spectrum]]):
        self._set_batch_obj_prop(14, value)

    @property
    def spectrum_obj(self) -> List[Spectrum]:
        """
        Harmonic spectrum assumed for this source.  Default is "default".

        DSS property name: `spectrum`, DSS property index: 14.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 14)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_batch_string(14, value)

    @property
    def basefreq(self) -> BatchFloat64ArrayProxy:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 15.
        """
        return BatchFloat64ArrayProxy(self, 15)

    @basefreq.setter
    def basefreq(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(15, value)

    @property
    def enabled(self) -> List[bool]:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 16.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 16)
        ]
    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 16, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 17.
        """
        self._set_batch_string(17, value)

class LoadBatch(DSSBatch):
    _cls_name = 'Load'
    _obj_cls = Load
    _cls_idx = 19


    @property
    def phases(self) -> BatchInt32ArrayProxy:
        """
        Number of Phases, this load.  Load is evenly divided among phases.

        DSS property name: `phases`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @phases.setter
    def phases(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def bus1(self) -> List[str]:
        """
        Bus to which the load is connected.  May include specific node specification.

        DSS property name: `bus1`, DSS property index: 2.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 2) 

    @bus1.setter
    def bus1(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(2, value)

    @property
    def kV(self) -> BatchFloat64ArrayProxy:
        """
        Nominal rated (1.0 per unit) voltage, kV, for load. For 2- and 3-phase loads, specify phase-phase kV. Otherwise, specify actual kV across each branch of the load. If wye (star), specify phase-neutral kV. If delta or phase-phase connected, specify phase-phase kV.

        DSS property name: `kV`, DSS property index: 3.
        """
        return BatchFloat64ArrayProxy(self, 3)

    @kV.setter
    def kV(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(3, value)

    @property
    def kW(self) -> BatchFloat64ArrayProxy:
        """
        Total base kW for the load.  Normally, you would enter the maximum kW for the load for the first year and allow it to be adjusted by the load shapes, growth shapes, and global load multiplier.

        Legal ways to define base load:
        kW, PF
        kW, kvar
        kVA, PF
        XFKVA * Allocationfactor, PF
        kWh/(kWhdays*24) * Cfactor, PF

        DSS property name: `kW`, DSS property index: 4.
        """
        return BatchFloat64ArrayProxy(self, 4)

    @kW.setter
    def kW(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(4, value)

    @property
    def pf(self) -> BatchFloat64ArrayProxy:
        """
        Load power factor.  Enter negative for leading powerfactor (when kW and kvar have opposite signs.)

        DSS property name: `pf`, DSS property index: 5.
        """
        return BatchFloat64ArrayProxy(self, 5)

    @pf.setter
    def pf(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(5, value)

    @property
    def model(self) -> BatchInt32ArrayProxy:
        """
        Integer code for the model to use for load variation with voltage. Valid values are:

        1:Standard constant P+jQ load. (Default)
        2:Constant impedance load. 
        3:Const P, Quadratic Q (like a motor).
        4:Nominal Linear P, Quadratic Q (feeder mix). Use this with CVRfactor.
        5:Constant Current Magnitude
        6:Const P, Fixed Q
        7:Const P, Fixed Impedance Q
        8:ZIPV (7 values)

        For Types 6 and 7, only the P is modified by load multipliers.

        DSS property name: `model`, DSS property index: 6.
        """
        return BatchInt32ArrayProxy(self, 6)

    @model.setter
    def model(self, value: Union[int, Load.LoadModel, Int32Array]):
        self._set_batch_int32_array(6, value)

    @property
    def yearly(self) -> List[str]:
        """
        LOADSHAPE object to use for yearly simulations.  Must be previously defined as a Loadshape object. Is set to the Daily load shape  when Daily is defined.  The daily load shape is repeated in this case. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. The default is no variation.

        DSS property name: `yearly`, DSS property index: 7.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 7)

    @yearly.setter
    def yearly(self, value: Union[AnyStr, LoadShape, List[AnyStr], List[LoadShape]]):
        self._set_batch_obj_prop(7, value)

    @property
    def yearly_obj(self) -> List[LoadShape]:
        """
        LOADSHAPE object to use for yearly simulations.  Must be previously defined as a Loadshape object. Is set to the Daily load shape  when Daily is defined.  The daily load shape is repeated in this case. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. The default is no variation.

        DSS property name: `yearly`, DSS property index: 7.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 7)

    @yearly_obj.setter
    def yearly_obj(self, value: LoadShape):
        self._set_batch_string(7, value)

    @property
    def daily(self) -> List[str]:
        """
        LOADSHAPE object to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Default is no variation (constant) if not defined. Side effect: Sets Yearly load shape if not already defined.

        DSS property name: `daily`, DSS property index: 8.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 8)

    @daily.setter
    def daily(self, value: Union[AnyStr, LoadShape, List[AnyStr], List[LoadShape]]):
        self._set_batch_obj_prop(8, value)

    @property
    def daily_obj(self) -> List[LoadShape]:
        """
        LOADSHAPE object to use for daily simulations.  Must be previously defined as a Loadshape object of 24 hrs, typically. Set Status=Fixed to ignore Loadshape designation. Set to NONE to reset to no loadahape. Default is no variation (constant) if not defined. Side effect: Sets Yearly load shape if not already defined.

        DSS property name: `daily`, DSS property index: 8.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 8)

    @daily_obj.setter
    def daily_obj(self, value: LoadShape):
        self._set_batch_string(8, value)

    @property
    def duty(self) -> List[str]:
        """
        LOADSHAPE object to use for duty cycle simulations.  Must be previously defined as a Loadshape object.  Typically would have time intervals less than 1 hr. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.Set to NONE to reset to no loadahape. Set Status=Fixed to ignore Loadshape designation.  Defaults to Daily curve If not specified.

        DSS property name: `duty`, DSS property index: 9.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 9)

    @duty.setter
    def duty(self, value: Union[AnyStr, LoadShape, List[AnyStr], List[LoadShape]]):
        self._set_batch_obj_prop(9, value)

    @property
    def duty_obj(self) -> List[LoadShape]:
        """
        LOADSHAPE object to use for duty cycle simulations.  Must be previously defined as a Loadshape object.  Typically would have time intervals less than 1 hr. Designate the number of points to solve using the Set Number=xxxx command. If there are fewer points in the actual shape, the shape is assumed to repeat.Set to NONE to reset to no loadahape. Set Status=Fixed to ignore Loadshape designation.  Defaults to Daily curve If not specified.

        DSS property name: `duty`, DSS property index: 9.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 9)

    @duty_obj.setter
    def duty_obj(self, value: LoadShape):
        self._set_batch_string(9, value)

    @property
    def growth(self) -> List[str]:
        """
        Characteristic  to use for growth factors by years.  Must be previously defined as a Growthshape object. Defaults to circuit default growth factor (see Set Growth command).

        DSS property name: `growth`, DSS property index: 10.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 10)

    @growth.setter
    def growth(self, value: Union[AnyStr, GrowthShape, List[AnyStr], List[GrowthShape]]):
        self._set_batch_obj_prop(10, value)

    @property
    def growth_obj(self) -> List[GrowthShape]:
        """
        Characteristic  to use for growth factors by years.  Must be previously defined as a Growthshape object. Defaults to circuit default growth factor (see Set Growth command).

        DSS property name: `growth`, DSS property index: 10.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 10)

    @growth_obj.setter
    def growth_obj(self, value: GrowthShape):
        self._set_batch_string(10, value)

    @property
    def conn(self) -> BatchInt32ArrayProxy:
        """
        ={wye or LN | delta or LL}.  Default is wye.

        DSS property name: `conn`, DSS property index: 11.
        """
        return BatchInt32ArrayProxy(self, 11)

    @conn.setter
    def conn(self, value: Union[AnyStr, int, Connection, List[AnyStr], List[int], List[Connection], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(11, value)
            return
    
        self._set_batch_int32_array(11, value)

    @property
    def conn_str(self) -> str:
        """
        ={wye or LN | delta or LL}.  Default is wye.

        DSS property name: `conn`, DSS property index: 11.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 11)

    @conn_str.setter
    def conn_str(self, value: AnyStr):
        self.conn = value

    @property
    def kvar(self) -> BatchFloat64ArrayProxy:
        """
        Specify the base kvar for specifying load as kW & kvar.  Assumes kW has been already defined.  Alternative to specifying the power factor.  Side effect:  the power factor and kVA is altered to agree.

        DSS property name: `kvar`, DSS property index: 12.
        """
        return BatchFloat64ArrayProxy(self, 12)

    @kvar.setter
    def kvar(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(12, value)

    @property
    def Rneut(self) -> BatchFloat64ArrayProxy:
        """
        Default is -1. Neutral resistance of wye (star)-connected load in actual ohms. If entered as a negative value, the neutral can be open, or floating, or it can be connected to node 0 (ground), which is the usual default. If >=0 be sure to explicitly specify the node connection for the neutral, or last, conductor. Otherwise, the neutral impedance will be shorted to ground.

        DSS property name: `Rneut`, DSS property index: 13.
        """
        return BatchFloat64ArrayProxy(self, 13)

    @Rneut.setter
    def Rneut(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(13, value)

    @property
    def Xneut(self) -> BatchFloat64ArrayProxy:
        """
        Neutral reactance of wye(star)-connected load in actual ohms.  May be + or -.

        DSS property name: `Xneut`, DSS property index: 14.
        """
        return BatchFloat64ArrayProxy(self, 14)

    @Xneut.setter
    def Xneut(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(14, value)

    @property
    def status(self) -> BatchInt32ArrayProxy:
        """
        ={Variable | Fixed | Exempt}.  Default is variable. If Fixed, no load multipliers apply;  however, growth multipliers do apply.  All multipliers apply to Variable loads.  Exempt loads are not modified by the global load multiplier, such as in load duration curves, etc.  Daily multipliers do apply, so setting this property to Exempt is a good way to represent industrial load that stays the same day-after-day for the period study.

        DSS property name: `status`, DSS property index: 15.
        """
        return BatchInt32ArrayProxy(self, 15)

    @status.setter
    def status(self, value: Union[AnyStr, int, Load.LoadStatus, List[AnyStr], List[int], List[Load.LoadStatus], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(15, value)
            return
    
        self._set_batch_int32_array(15, value)

    @property
    def status_str(self) -> str:
        """
        ={Variable | Fixed | Exempt}.  Default is variable. If Fixed, no load multipliers apply;  however, growth multipliers do apply.  All multipliers apply to Variable loads.  Exempt loads are not modified by the global load multiplier, such as in load duration curves, etc.  Daily multipliers do apply, so setting this property to Exempt is a good way to represent industrial load that stays the same day-after-day for the period study.

        DSS property name: `status`, DSS property index: 15.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 15)

    @status_str.setter
    def status_str(self, value: AnyStr):
        self.status = value

    @property
    def cls(self) -> BatchInt32ArrayProxy:
        """
        An arbitrary integer number representing the class of load so that load values may be segregated by load value. Default is 1; not used internally.

        DSS property name: `class`, DSS property index: 16.
        """
        return BatchInt32ArrayProxy(self, 16)

    @cls.setter
    def cls(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(16, value)

    @property
    def Vminpu(self) -> BatchFloat64ArrayProxy:
        """
        Default = 0.95.  Minimum per unit voltage for which the MODEL is assumed to apply. Lower end of normal voltage range.Below this value, the load model reverts to a constant impedance model that matches the model at the transition voltage. See also "Vlowpu" which causes the model to match Model=2 below the transition voltage.

        DSS property name: `Vminpu`, DSS property index: 17.
        """
        return BatchFloat64ArrayProxy(self, 17)

    @Vminpu.setter
    def Vminpu(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(17, value)

    @property
    def Vmaxpu(self) -> BatchFloat64ArrayProxy:
        """
        Default = 1.05.  Maximum per unit voltage for which the MODEL is assumed to apply. Above this value, the load model reverts to a constant impedance model.

        DSS property name: `Vmaxpu`, DSS property index: 18.
        """
        return BatchFloat64ArrayProxy(self, 18)

    @Vmaxpu.setter
    def Vmaxpu(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(18, value)

    @property
    def Vminnorm(self) -> BatchFloat64ArrayProxy:
        """
        Minimum per unit voltage for load EEN evaluations, Normal limit.  Default = 0, which defaults to system "vminnorm" property (see Set Command under Executive).  If this property is specified, it ALWAYS overrides the system specification. This allows you to have different criteria for different loads. Set to zero to revert to the default system value.

        DSS property name: `Vminnorm`, DSS property index: 19.
        """
        return BatchFloat64ArrayProxy(self, 19)

    @Vminnorm.setter
    def Vminnorm(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(19, value)

    @property
    def Vminemerg(self) -> BatchFloat64ArrayProxy:
        """
        Minimum per unit voltage for load UE evaluations, Emergency limit.  Default = 0, which defaults to system "vminemerg" property (see Set Command under Executive).  If this property is specified, it ALWAYS overrides the system specification. This allows you to have different criteria for different loads. Set to zero to revert to the default system value.

        DSS property name: `Vminemerg`, DSS property index: 20.
        """
        return BatchFloat64ArrayProxy(self, 20)

    @Vminemerg.setter
    def Vminemerg(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(20, value)

    @property
    def xfkVA(self) -> BatchFloat64ArrayProxy:
        """
        Default = 0.0.  Rated kVA of service transformer for allocating loads based on connected kVA at a bus. Side effect:  kW, PF, and kvar are modified. See help on kVA.

        DSS property name: `xfkVA`, DSS property index: 21.
        """
        return BatchFloat64ArrayProxy(self, 21)

    @xfkVA.setter
    def xfkVA(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(21, value)

    @property
    def allocationfactor(self) -> BatchFloat64ArrayProxy:
        """
        Default = 0.5.  Allocation factor for allocating loads based on connected kVA at a bus. Side effect:  kW, PF, and kvar are modified by multiplying this factor times the XFKVA (if > 0).

        DSS property name: `allocationfactor`, DSS property index: 22.
        """
        return BatchFloat64ArrayProxy(self, 22)

    @allocationfactor.setter
    def allocationfactor(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(22, value)

    @property
    def kVA(self) -> BatchFloat64ArrayProxy:
        """
        Specify base Load in kVA (and power factor)

        Legal ways to define base load:
        kW, PF
        kW, kvar
        kVA, PF
        XFKVA * Allocationfactor, PF
        kWh/(kWhdays*24) * Cfactor, PF

        DSS property name: `kVA`, DSS property index: 23.
        """
        return BatchFloat64ArrayProxy(self, 23)

    @kVA.setter
    def kVA(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(23, value)

    @property
    def pctmean(self) -> BatchFloat64ArrayProxy:
        """
        Percent mean value for load to use for monte carlo studies if no loadshape is assigned to this load. Default is 50.

        DSS property name: `%mean`, DSS property index: 24.
        """
        return BatchFloat64ArrayProxy(self, 24)

    @pctmean.setter
    def pctmean(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(24, value)

    @property
    def pctstddev(self) -> BatchFloat64ArrayProxy:
        """
        Percent Std deviation value for load to use for monte carlo studies if no loadshape is assigned to this load. Default is 10.

        DSS property name: `%stddev`, DSS property index: 25.
        """
        return BatchFloat64ArrayProxy(self, 25)

    @pctstddev.setter
    def pctstddev(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(25, value)

    @property
    def CVRwatts(self) -> BatchFloat64ArrayProxy:
        """
        Percent reduction in active power (watts) per 1% reduction in voltage from 100% rated. Default=1. 
         Typical values range from 0.4 to 0.8. Applies to Model=4 only.
         Intended to represent conservation voltage reduction or voltage optimization measures.

        DSS property name: `CVRwatts`, DSS property index: 26.
        """
        return BatchFloat64ArrayProxy(self, 26)

    @CVRwatts.setter
    def CVRwatts(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(26, value)

    @property
    def CVRvars(self) -> BatchFloat64ArrayProxy:
        """
        Percent reduction in reactive power (vars) per 1% reduction in voltage from 100% rated. Default=2. 
         Typical values range from 2 to 3. Applies to Model=4 only.
         Intended to represent conservation voltage reduction or voltage optimization measures.

        DSS property name: `CVRvars`, DSS property index: 27.
        """
        return BatchFloat64ArrayProxy(self, 27)

    @CVRvars.setter
    def CVRvars(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(27, value)

    @property
    def kwh(self) -> BatchFloat64ArrayProxy:
        """
        kWh billed for this period. Default is 0. See help on kVA and Cfactor and kWhDays.

        DSS property name: `kwh`, DSS property index: 28.
        """
        return BatchFloat64ArrayProxy(self, 28)

    @kwh.setter
    def kwh(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(28, value)

    @property
    def kwhdays(self) -> BatchFloat64ArrayProxy:
        """
        Length of kWh billing period in days (24 hr days). Default is 30. Average demand is computed using this value.

        DSS property name: `kwhdays`, DSS property index: 29.
        """
        return BatchFloat64ArrayProxy(self, 29)

    @kwhdays.setter
    def kwhdays(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(29, value)

    @property
    def Cfactor(self) -> BatchFloat64ArrayProxy:
        """
        Factor relating average kW to peak kW. Default is 4.0. See kWh and kWhdays. See kVA.

        DSS property name: `Cfactor`, DSS property index: 30.
        """
        return BatchFloat64ArrayProxy(self, 30)

    @Cfactor.setter
    def Cfactor(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(30, value)

    @property
    def CVRcurve(self) -> List[str]:
        """
        Default is NONE. Curve describing both watt and var factors as a function of time. Refers to a LoadShape object with both Mult and Qmult defined. Define a Loadshape to agree with yearly or daily curve according to the type of analysis being done. If NONE, the CVRwatts and CVRvars factors are used and assumed constant.

        DSS property name: `CVRcurve`, DSS property index: 31.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 31)

    @CVRcurve.setter
    def CVRcurve(self, value: Union[AnyStr, LoadShape, List[AnyStr], List[LoadShape]]):
        self._set_batch_obj_prop(31, value)

    @property
    def CVRcurve_obj(self) -> List[LoadShape]:
        """
        Default is NONE. Curve describing both watt and var factors as a function of time. Refers to a LoadShape object with both Mult and Qmult defined. Define a Loadshape to agree with yearly or daily curve according to the type of analysis being done. If NONE, the CVRwatts and CVRvars factors are used and assumed constant.

        DSS property name: `CVRcurve`, DSS property index: 31.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 31)

    @CVRcurve_obj.setter
    def CVRcurve_obj(self, value: LoadShape):
        self._set_batch_string(31, value)

    @property
    def NumCust(self) -> BatchInt32ArrayProxy:
        """
        Number of customers, this load. Default is 1.

        DSS property name: `NumCust`, DSS property index: 32.
        """
        return BatchInt32ArrayProxy(self, 32)

    @NumCust.setter
    def NumCust(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(32, value)

    @property
    def ZIPV(self) -> List[Float64Array]:
        """
        Array of 7 coefficients:

         First 3 are ZIP weighting factors for real power (should sum to 1)
         Next 3 are ZIP weighting factors for reactive power (should sum to 1)
         Last 1 is cut-off voltage in p.u. of base kV; load is 0 below this cut-off
         No defaults; all coefficients must be specified if using model=8.

        DSS property name: `ZIPV`, DSS property index: 33.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 33)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @ZIPV.setter
    def ZIPV(self, value: Float64Array):
        self._set_float64_array_o(33, value)

    @property
    def pctSeriesRL(self) -> BatchFloat64ArrayProxy:
        """
        Percent of load that is series R-L for Harmonic studies. Default is 50. Remainder is assumed to be parallel R and L. This can have a significant impact on the amount of damping observed in Harmonics solutions.

        DSS property name: `%SeriesRL`, DSS property index: 34.
        """
        return BatchFloat64ArrayProxy(self, 34)

    @pctSeriesRL.setter
    def pctSeriesRL(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(34, value)

    @property
    def RelWeight(self) -> BatchFloat64ArrayProxy:
        """
        Relative weighting factor for reliability calcs. Default = 1. Used to designate high priority loads such as hospitals, etc. 

        Is multiplied by number of customers and load kW during reliability calcs.

        DSS property name: `RelWeight`, DSS property index: 35.
        """
        return BatchFloat64ArrayProxy(self, 35)

    @RelWeight.setter
    def RelWeight(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(35, value)

    @property
    def Vlowpu(self) -> BatchFloat64ArrayProxy:
        """
        Default = 0.50.  Per unit voltage at which the model switches to same as constant Z model (model=2). This allows more consistent convergence at very low voltaes due to opening switches or solving for fault situations.

        DSS property name: `Vlowpu`, DSS property index: 36.
        """
        return BatchFloat64ArrayProxy(self, 36)

    @Vlowpu.setter
    def Vlowpu(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(36, value)

    @property
    def puXharm(self) -> BatchFloat64ArrayProxy:
        """
        Special reactance, pu (based on kVA, kV properties), for the series impedance branch in the load model for HARMONICS analysis. Generally used to represent motor load blocked rotor reactance. If not specified (that is, set =0, the default value), the series branch is computed from the percentage of the nominal load at fundamental frequency specified by the %SERIESRL property. 

        Applies to load model in HARMONICS mode only.

        A typical value would be approximately 0.20 pu based on kVA * %SeriesRL / 100.0.

        DSS property name: `puXharm`, DSS property index: 37.
        """
        return BatchFloat64ArrayProxy(self, 37)

    @puXharm.setter
    def puXharm(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(37, value)

    @property
    def XRharm(self) -> BatchFloat64ArrayProxy:
        """
        X/R ratio of the special harmonics mode reactance specified by the puXHARM property at fundamental frequency. Default is 6. 

        DSS property name: `XRharm`, DSS property index: 38.
        """
        return BatchFloat64ArrayProxy(self, 38)

    @XRharm.setter
    def XRharm(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(38, value)

    @property
    def spectrum(self) -> List[str]:
        """
        Name of harmonic current spectrum for this load.  Default is "defaultload", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 39.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 39)

    @spectrum.setter
    def spectrum(self, value: Union[AnyStr, Spectrum, List[AnyStr], List[Spectrum]]):
        self._set_batch_obj_prop(39, value)

    @property
    def spectrum_obj(self) -> List[Spectrum]:
        """
        Name of harmonic current spectrum for this load.  Default is "defaultload", which is defined when the DSS starts.

        DSS property name: `spectrum`, DSS property index: 39.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 39)

    @spectrum_obj.setter
    def spectrum_obj(self, value: Spectrum):
        self._set_batch_string(39, value)

    @property
    def basefreq(self) -> BatchFloat64ArrayProxy:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 40.
        """
        return BatchFloat64ArrayProxy(self, 40)

    @basefreq.setter
    def basefreq(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(40, value)

    @property
    def enabled(self) -> List[bool]:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 41.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 41)
        ]
    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 41, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 42.
        """
        self._set_batch_string(42, value)

class TransformerBatch(DSSBatch):
    _cls_name = 'Transformer'
    _obj_cls = Transformer
    _cls_idx = 20


    @property
    def phases(self) -> BatchInt32ArrayProxy:
        """
        Number of phases this transformer. Default is 3.

        DSS property name: `phases`, DSS property index: 1.
        """
        return BatchInt32ArrayProxy(self, 1)

    @phases.setter
    def phases(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(1, value)

    @property
    def windings(self) -> BatchInt32ArrayProxy:
        """
        Number of windings, this transformers. (Also is the number of terminals) Default is 2. This property triggers memory allocation for the Transformer and will cause other properties to revert to default values.

        DSS property name: `windings`, DSS property index: 2.
        """
        return BatchInt32ArrayProxy(self, 2)

    @windings.setter
    def windings(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(2, value)

    @property
    def pctR(self) -> List[Float64Array]:
        """
        Percent resistance this winding.  (half of total for a 2-winding).

        DSS property name: `%R`, DSS property index: 9.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 9)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @pctR.setter
    def pctR(self, value: Float64Array):
        self._set_float64_array_o(9, value)

    @property
    def Rneut(self) -> List[Float64Array]:
        """
        Default = -1. Neutral resistance of wye (star)-connected winding in actual ohms. If entered as a negative value, the neutral is assumed to be open, or floating. To solidly ground the neutral, connect the neutral conductor to Node 0 in the Bus property spec for this winding. For example: Bus=MyBusName.1.2.3.0, which is generally the default connection.

        DSS property name: `Rneut`, DSS property index: 10.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 10)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Rneut.setter
    def Rneut(self, value: Float64Array):
        self._set_float64_array_o(10, value)

    @property
    def Xneut(self) -> List[Float64Array]:
        """
        Neutral reactance of wye(star)-connected winding in actual ohms.  May be + or -.

        DSS property name: `Xneut`, DSS property index: 11.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 11)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Xneut.setter
    def Xneut(self, value: Float64Array):
        self._set_float64_array_o(11, value)

    @property
    def buses(self) -> List[List[str]]:
        """
        Use this to specify all the bus connections at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus"

        DSS property name: `buses`, DSS property index: 12.
        """
        return self._get_string_ll(12)

    @buses.setter
    def buses(self, value: List[AnyStr]):
        value, value_ptr, value_count = self._prepare_string_array(value)
        for x in self._ffi.unpack(self.pointer[0], self.count[0]):
            self._lib.Obj_SetStringArray(x, 12, value_ptr, value_count)
    
        self._check_for_error()

    @property
    def conns(self) -> List[Int32Array]:
        """
        Use this to specify all the Winding connections at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus" ~ conns=(delta, wye)

        DSS property name: `conns`, DSS property index: 13.
        """
        return [
            self._get_int32_array(self._lib.Obj_GetInt32Array, x, 13)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @conns.setter
    def conns(self, value: Union[List[Union[int,Connection]], List[AnyStr]]): #TODO: list of lists
        if len(value) and not isinstance(value[0], int):
            value, value_ptr, value_count = self._prepare_string_array(value)
            for x in self._ffi.unpack(self.pointer[0], self.count[0]):
                self._lib.Obj_SetStringArray(x, 13, value_ptr, value_count)

            self._check_for_error()
            return

        self._set_batch_int32_array(13, value)

    @property
    def conns_str(self) -> List[List[str]]:
        """
        Use this to specify all the Winding connections at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus" ~ conns=(delta, wye)

        DSS property name: `conns`, DSS property index: 13.
        """
        return self._get_string_ll(13)

    @conns_str.setter
    def conns_str(self, value: AnyStr):
        self.conns = value

    @property
    def kVs(self) -> List[Float64Array]:
        """
        Use this to specify the kV ratings of all windings at once using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus" 
        ~ conns=(delta, wye)
        ~ kvs=(115, 12.47)

        See kV= property for voltage rules.

        DSS property name: `kVs`, DSS property index: 14.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 14)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @kVs.setter
    def kVs(self, value: Float64Array):
        self._set_float64_array_o(14, value)

    @property
    def kVAs(self) -> List[Float64Array]:
        """
        Use this to specify the kVA ratings of all windings at once using an array.

        DSS property name: `kVAs`, DSS property index: 15.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 15)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @kVAs.setter
    def kVAs(self, value: Float64Array):
        self._set_float64_array_o(15, value)

    @property
    def taps(self) -> List[Float64Array]:
        """
        Use this to specify the p.u. tap of all windings at once using an array.

        DSS property name: `taps`, DSS property index: 16.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 16)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @taps.setter
    def taps(self, value: Float64Array):
        self._set_float64_array_o(16, value)

    @property
    def XHL(self) -> BatchFloat64ArrayProxy:
        """
        Use this to specify the percent reactance, H-L (winding 1 to winding 2).  Use for 2- or 3-winding transformers. On the kVA base of winding 1. See also X12.

        DSS property name: `XHL`, DSS property index: 17.
        """
        return BatchFloat64ArrayProxy(self, 17)

    @XHL.setter
    def XHL(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(17, value)

    @property
    def XHT(self) -> BatchFloat64ArrayProxy:
        """
        Use this to specify the percent reactance, H-T (winding 1 to winding 3).  Use for 3-winding transformers only. On the kVA base of winding 1. See also X13.

        DSS property name: `XHT`, DSS property index: 18.
        """
        return BatchFloat64ArrayProxy(self, 18)

    @XHT.setter
    def XHT(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(18, value)

    @property
    def XLT(self) -> BatchFloat64ArrayProxy:
        """
        Use this to specify the percent reactance, L-T (winding 2 to winding 3).  Use for 3-winding transformers only. On the kVA base of winding 1.  See also X23.

        DSS property name: `XLT`, DSS property index: 19.
        """
        return BatchFloat64ArrayProxy(self, 19)

    @XLT.setter
    def XLT(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(19, value)

    @property
    def Xscarray(self) -> List[Float64Array]:
        """
        Use this to specify the percent reactance between all pairs of windings as an array. All values are on the kVA base of winding 1.  The order of the values is as follows:

        (x12 13 14... 23 24.. 34 ..)  

        There will be n(n-1)/2 values, where n=number of windings.

        DSS property name: `Xscarray`, DSS property index: 20.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 20)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Xscarray.setter
    def Xscarray(self, value: Float64Array):
        self._set_float64_array_o(20, value)

    @property
    def thermal(self) -> BatchFloat64ArrayProxy:
        """
        Thermal time constant of the transformer in hours.  Typically about 2.

        DSS property name: `thermal`, DSS property index: 21.
        """
        return BatchFloat64ArrayProxy(self, 21)

    @thermal.setter
    def thermal(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(21, value)

    @property
    def n(self) -> BatchFloat64ArrayProxy:
        """
        n Exponent for thermal properties in IEEE C57.  Typically 0.8.

        DSS property name: `n`, DSS property index: 22.
        """
        return BatchFloat64ArrayProxy(self, 22)

    @n.setter
    def n(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(22, value)

    @property
    def m(self) -> BatchFloat64ArrayProxy:
        """
        m Exponent for thermal properties in IEEE C57.  Typically 0.9 - 1.0

        DSS property name: `m`, DSS property index: 23.
        """
        return BatchFloat64ArrayProxy(self, 23)

    @m.setter
    def m(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(23, value)

    @property
    def flrise(self) -> BatchFloat64ArrayProxy:
        """
        Temperature rise, deg C, for full load.  Default is 65.

        DSS property name: `flrise`, DSS property index: 24.
        """
        return BatchFloat64ArrayProxy(self, 24)

    @flrise.setter
    def flrise(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(24, value)

    @property
    def hsrise(self) -> BatchFloat64ArrayProxy:
        """
        Hot spot temperature rise, deg C.  Default is 15.

        DSS property name: `hsrise`, DSS property index: 25.
        """
        return BatchFloat64ArrayProxy(self, 25)

    @hsrise.setter
    def hsrise(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(25, value)

    @property
    def pctloadloss(self) -> BatchFloat64ArrayProxy:
        """
        Percent load loss at full load. The %R of the High and Low windings (1 and 2) are adjusted to agree at rated kVA loading.

        DSS property name: `%loadloss`, DSS property index: 26.
        """
        return BatchFloat64ArrayProxy(self, 26)

    @pctloadloss.setter
    def pctloadloss(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(26, value)

    @property
    def pctnoloadloss(self) -> BatchFloat64ArrayProxy:
        """
        Percent no load losses at rated excitatation voltage. Default is 0. Converts to a resistance in parallel with the magnetizing impedance in each winding.

        DSS property name: `%noloadloss`, DSS property index: 27.
        """
        return BatchFloat64ArrayProxy(self, 27)

    @pctnoloadloss.setter
    def pctnoloadloss(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(27, value)

    @property
    def normhkVA(self) -> BatchFloat64ArrayProxy:
        """
        Normal maximum kVA rating of H winding (winding 1).  Usually 100% - 110% ofmaximum nameplate rating, depending on load shape. Defaults to 110% of kVA rating of Winding 1.

        DSS property name: `normhkVA`, DSS property index: 28.
        """
        return BatchFloat64ArrayProxy(self, 28)

    @normhkVA.setter
    def normhkVA(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(28, value)

    @property
    def emerghkVA(self) -> BatchFloat64ArrayProxy:
        """
        Emergency (contingency)  kVA rating of H winding (winding 1).  Usually 140% - 150% ofmaximum nameplate rating, depending on load shape. Defaults to 150% of kVA rating of Winding 1.

        DSS property name: `emerghkVA`, DSS property index: 29.
        """
        return BatchFloat64ArrayProxy(self, 29)

    @emerghkVA.setter
    def emerghkVA(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(29, value)

    @property
    def sub(self) -> List[bool]:
        """
        ={Yes|No}  Designates whether this transformer is to be considered a substation.Default is No.

        DSS property name: `sub`, DSS property index: 30.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 30)
        ]
    @sub.setter
    def sub(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 30, value)

    @property
    def MaxTap(self) -> List[Float64Array]:
        """
        Max per unit tap for the active winding.  Default is 1.10

        DSS property name: `MaxTap`, DSS property index: 31.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 31)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @MaxTap.setter
    def MaxTap(self, value: Float64Array):
        self._set_float64_array_o(31, value)

    @property
    def MinTap(self) -> List[Float64Array]:
        """
        Min per unit tap for the active winding.  Default is 0.90

        DSS property name: `MinTap`, DSS property index: 32.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 32)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @MinTap.setter
    def MinTap(self, value: Float64Array):
        self._set_float64_array_o(32, value)

    @property
    def NumTaps(self) -> List[Int32Array]:
        """
        Total number of taps between min and max tap.  Default is 32 (16 raise and 16 lower taps about the neutral position). The neutral position is not counted.

        DSS property name: `NumTaps`, DSS property index: 33.
        """
        return [
            self._get_int32_array(self._lib.Obj_GetInt32Array, x, 33)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @NumTaps.setter
    def NumTaps(self, value: Int32Array): #TODO: list of arrays, matrix
        self._set_batch_int32_array(33, value)

    @property
    def subname(self) -> List[str]:
        """
        Substation Name. Optional. Default is null. If specified, printed on plots

        DSS property name: `subname`, DSS property index: 34.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 34) 

    @subname.setter
    def subname(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(34, value)

    @property
    def pctimag(self) -> BatchFloat64ArrayProxy:
        """
        Percent magnetizing current. Default=0.0. Magnetizing branch is in parallel with windings in each phase. Also, see "ppm_antifloat".

        DSS property name: `%imag`, DSS property index: 35.
        """
        return BatchFloat64ArrayProxy(self, 35)

    @pctimag.setter
    def pctimag(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(35, value)

    @property
    def ppm_antifloat(self) -> BatchFloat64ArrayProxy:
        """
        Default=1 ppm.  Parts per million of transformer winding VA rating connected to ground to protect against accidentally floating a winding without a reference. If positive then the effect is adding a very large reactance to ground.  If negative, then a capacitor.

        DSS property name: `ppm_antifloat`, DSS property index: 36.
        """
        return BatchFloat64ArrayProxy(self, 36)

    @ppm_antifloat.setter
    def ppm_antifloat(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(36, value)

    @property
    def pctRs(self) -> List[Float64Array]:
        """
        Use this property to specify all the winding %resistances using an array. Example:

        New Transformer.T1 buses="Hibus, lowbus" ~ %Rs=(0.2  0.3)

        DSS property name: `%Rs`, DSS property index: 37.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 37)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @pctRs.setter
    def pctRs(self, value: Float64Array):
        self._set_float64_array_o(37, value)

    @property
    def bank(self) -> List[str]:
        """
        Name of the bank this transformer is part of, for CIM, MultiSpeak, and other interfaces.

        DSS property name: `bank`, DSS property index: 38.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 38) 

    @bank.setter
    def bank(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(38, value)

    @property
    def xfmrcode(self) -> List[str]:
        """
        Name of a library entry for transformer properties. The named XfmrCode must already be defined.

        DSS property name: `XfmrCode`, DSS property index: 39.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 39)

    @xfmrcode.setter
    def xfmrcode(self, value: Union[AnyStr, XfmrCode, List[AnyStr], List[XfmrCode]]):
        self._set_batch_obj_prop(39, value)

    @property
    def xfmrcode_obj(self) -> List[XfmrCode]:
        """
        Name of a library entry for transformer properties. The named XfmrCode must already be defined.

        DSS property name: `XfmrCode`, DSS property index: 39.
        """
        return self._get_obj_array(self._lib.Batch_GetObject, self.pointer[0], self.count[0], 39)

    @xfmrcode_obj.setter
    def xfmrcode_obj(self, value: XfmrCode):
        self._set_batch_string(39, value)

    @property
    def XRConst(self) -> List[bool]:
        """
        ={Yes|No} Default is NO. Signifies whether or not the X/R is assumed contant for harmonic studies.

        DSS property name: `XRConst`, DSS property index: 40.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 40)
        ]
    @XRConst.setter
    def XRConst(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 40, value)

    @property
    def X12(self) -> BatchFloat64ArrayProxy:
        """
        Alternative to XHL for specifying the percent reactance from winding 1 to winding 2.  Use for 2- or 3-winding transformers. Percent on the kVA base of winding 1. 

        DSS property name: `X12`, DSS property index: 41.
        """
        return BatchFloat64ArrayProxy(self, 41)

    @X12.setter
    def X12(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(41, value)

    @property
    def X13(self) -> BatchFloat64ArrayProxy:
        """
        Alternative to XHT for specifying the percent reactance from winding 1 to winding 3.  Use for 3-winding transformers only. Percent on the kVA base of winding 1. 

        DSS property name: `X13`, DSS property index: 42.
        """
        return BatchFloat64ArrayProxy(self, 42)

    @X13.setter
    def X13(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(42, value)

    @property
    def X23(self) -> BatchFloat64ArrayProxy:
        """
        Alternative to XLT for specifying the percent reactance from winding 2 to winding 3.Use for 3-winding transformers only. Percent on the kVA base of winding 1.  

        DSS property name: `X23`, DSS property index: 43.
        """
        return BatchFloat64ArrayProxy(self, 43)

    @X23.setter
    def X23(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(43, value)

    @property
    def LeadLag(self) -> BatchInt32ArrayProxy:
        """
        {Lead | Lag (default) | ANSI (default) | Euro } Designation in mixed Delta-wye connections the relationship between HV to LV winding. Default is ANSI 30 deg lag, e.g., Dy1 of Yd1 vector group. To get typical European Dy11 connection, specify either "lead" or "Euro"

        DSS property name: `LeadLag`, DSS property index: 44.
        """
        return BatchInt32ArrayProxy(self, 44)

    @LeadLag.setter
    def LeadLag(self, value: Union[AnyStr, int, PhaseSequence, List[AnyStr], List[int], List[PhaseSequence], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(44, value)
            return
    
        self._set_batch_int32_array(44, value)

    @property
    def LeadLag_str(self) -> str:
        """
        {Lead | Lag (default) | ANSI (default) | Euro } Designation in mixed Delta-wye connections the relationship between HV to LV winding. Default is ANSI 30 deg lag, e.g., Dy1 of Yd1 vector group. To get typical European Dy11 connection, specify either "lead" or "Euro"

        DSS property name: `LeadLag`, DSS property index: 44.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 44)

    @LeadLag_str.setter
    def LeadLag_str(self, value: AnyStr):
        self.LeadLag = value

    def WdgCurrents(self) -> List[str]:
        """
        (Read only) Makes winding currents available via return on query (? Transformer.TX.WdgCurrents). Order: Phase 1, Wdg 1, Wdg 2, ..., Phase 2 ...

        DSS property name: `WdgCurrents`, DSS property index: 45.
        """
        # []
        # StringSilentROFunction
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 45)

    @property
    def Core(self) -> BatchInt32ArrayProxy:
        """
        {Shell*|5-leg|3-Leg|1-phase|core-1-phase|4-leg} Core Type. Used for GIC analysis

        DSS property name: `Core`, DSS property index: 46.
        """
        return BatchInt32ArrayProxy(self, 46)

    @Core.setter
    def Core(self, value: Union[AnyStr, int, CoreType, List[AnyStr], List[int], List[CoreType], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(46, value)
            return
    
        self._set_batch_int32_array(46, value)

    @property
    def Core_str(self) -> str:
        """
        {Shell*|5-leg|3-Leg|1-phase|core-1-phase|4-leg} Core Type. Used for GIC analysis

        DSS property name: `Core`, DSS property index: 46.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 46)

    @Core_str.setter
    def Core_str(self, value: AnyStr):
        self.Core = value

    @property
    def RdcOhms(self) -> List[Float64Array]:
        """
        Winding dc resistance in OHMS. Useful for GIC analysis. From transformer test report. Defaults to 85% of %R property

        DSS property name: `RdcOhms`, DSS property index: 47.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 47)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @RdcOhms.setter
    def RdcOhms(self, value: Float64Array):
        self._set_float64_array_o(47, value)

    @property
    def Seasons(self) -> BatchInt32ArrayProxy:
        """
        Defines the number of ratings to be defined for the transfomer, to be used only when defining seasonal ratings using the "Ratings" property.

        DSS property name: `Seasons`, DSS property index: 48.
        """
        return BatchInt32ArrayProxy(self, 48)

    @Seasons.setter
    def Seasons(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(48, value)

    @property
    def Ratings(self) -> List[Float64Array]:
        """
        An array of ratings to be used when the seasonal ratings flag is True. It can be used to insert
        multiple ratings to change during a QSTS simulation to evaluate different ratings in transformers. Is given in kVA

        DSS property name: `Ratings`, DSS property index: 49.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 49)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Ratings.setter
    def Ratings(self, value: Float64Array):
        self._set_float64_array_o(49, value)

    @property
    def normamps(self) -> BatchFloat64ArrayProxy:
        """
        Normal rated current.

        DSS property name: `normamps`, DSS property index: 50.
        """
        return BatchFloat64ArrayProxy(self, 50)

    @normamps.setter
    def normamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(50, value)

    @property
    def emergamps(self) -> BatchFloat64ArrayProxy:
        """
        Maximum or emerg current.

        DSS property name: `emergamps`, DSS property index: 51.
        """
        return BatchFloat64ArrayProxy(self, 51)

    @emergamps.setter
    def emergamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(51, value)

    @property
    def faultrate(self) -> BatchFloat64ArrayProxy:
        """
        Failure rate per year.

        DSS property name: `faultrate`, DSS property index: 52.
        """
        return BatchFloat64ArrayProxy(self, 52)

    @faultrate.setter
    def faultrate(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(52, value)

    @property
    def pctperm(self) -> BatchFloat64ArrayProxy:
        """
        Percent of failures that become permanent.

        DSS property name: `pctperm`, DSS property index: 53.
        """
        return BatchFloat64ArrayProxy(self, 53)

    @pctperm.setter
    def pctperm(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(53, value)

    @property
    def repair(self) -> BatchFloat64ArrayProxy:
        """
        Hours to repair.

        DSS property name: `repair`, DSS property index: 54.
        """
        return BatchFloat64ArrayProxy(self, 54)

    @repair.setter
    def repair(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(54, value)

    @property
    def basefreq(self) -> BatchFloat64ArrayProxy:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 55.
        """
        return BatchFloat64ArrayProxy(self, 55)

    @basefreq.setter
    def basefreq(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(55, value)

    @property
    def enabled(self) -> List[bool]:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 56.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 56)
        ]
    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 56, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 57.
        """
        self._set_batch_string(57, value)

class CapacitorBatch(DSSBatch):
    _cls_name = 'Capacitor'
    _obj_cls = Capacitor
    _cls_idx = 22


    @property
    def bus1(self) -> List[str]:
        """
        Name of first bus of 2-terminal capacitor. Examples:
        bus1=busname
        bus1=busname.1.2.3

        If only one bus specified, Bus2 will default to this bus, Node 0, and the capacitor will be a Yg shunt bank.

        DSS property name: `bus1`, DSS property index: 1.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 1) 

    @bus1.setter
    def bus1(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(1, value)

    @property
    def bus2(self) -> List[str]:
        """
        Name of 2nd bus. Defaults to all phases connected to first bus, node 0, (Shunt Wye Connection) except when Bus2 explicitly specified. 

        Not necessary to specify for delta (LL) connection.

        DSS property name: `bus2`, DSS property index: 2.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 2) 

    @bus2.setter
    def bus2(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(2, value)

    @property
    def phases(self) -> BatchInt32ArrayProxy:
        """
        Number of phases.

        DSS property name: `phases`, DSS property index: 3.
        """
        return BatchInt32ArrayProxy(self, 3)

    @phases.setter
    def phases(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(3, value)

    @property
    def kvar(self) -> List[Float64Array]:
        """
        Total kvar, if one step, or ARRAY of kvar ratings for each step.  Evenly divided among phases. See rules for NUMSTEPS.

        DSS property name: `kvar`, DSS property index: 4.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 4)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @kvar.setter
    def kvar(self, value: Float64Array):
        self._set_float64_array_o(4, value)

    @property
    def kv(self) -> BatchFloat64ArrayProxy:
        """
        For 2, 3-phase, kV phase-phase. Otherwise specify actual can rating.

        DSS property name: `kv`, DSS property index: 5.
        """
        return BatchFloat64ArrayProxy(self, 5)

    @kv.setter
    def kv(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(5, value)

    @property
    def conn(self) -> BatchInt32ArrayProxy:
        """
        ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN

        DSS property name: `conn`, DSS property index: 6.
        """
        return BatchInt32ArrayProxy(self, 6)

    @conn.setter
    def conn(self, value: Union[AnyStr, int, Connection, List[AnyStr], List[int], List[Connection], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(6, value)
            return
    
        self._set_batch_int32_array(6, value)

    @property
    def conn_str(self) -> str:
        """
        ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN

        DSS property name: `conn`, DSS property index: 6.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 6)

    @conn_str.setter
    def conn_str(self, value: AnyStr):
        self.conn = value

    @property
    def cmatrix(self) -> List[Float64Array]:
        """
        Nodal cap. matrix, lower triangle, microfarads, of the following form:

        cmatrix="c11 | -c21 c22 | -c31 -c32 c33"

        All steps are assumed the same if this property is used.

        DSS property name: `cmatrix`, DSS property index: 7.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 7)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @cmatrix.setter
    def cmatrix(self, value: Float64Array):
        self._set_float64_array_o(7, value)

    @property
    def cuf(self) -> List[Float64Array]:
        """
        ARRAY of Capacitance, each phase, for each step, microfarads.
        See Rules for NumSteps.

        DSS property name: `cuf`, DSS property index: 8.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 8)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @cuf.setter
    def cuf(self, value: Float64Array):
        self._set_float64_array_o(8, value)

    @property
    def R(self) -> List[Float64Array]:
        """
        ARRAY of series resistance in each phase (line), ohms. Default is 0.0

        DSS property name: `R`, DSS property index: 9.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 9)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @R.setter
    def R(self, value: Float64Array):
        self._set_float64_array_o(9, value)

    @property
    def XL(self) -> List[Float64Array]:
        """
        ARRAY of series inductive reactance(s) in each phase (line) for filter, ohms at base frequency. Use this OR "h" property to define filter. Default is 0.0.

        DSS property name: `XL`, DSS property index: 10.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 10)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @XL.setter
    def XL(self, value: Float64Array):
        self._set_float64_array_o(10, value)

    @property
    def Harm(self) -> List[Float64Array]:
        """
        ARRAY of harmonics to which each step is tuned. Zero is interpreted as meaning zero reactance (no filter). Default is zero.

        DSS property name: `Harm`, DSS property index: 11.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 11)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Harm.setter
    def Harm(self, value: Float64Array):
        self._set_float64_array_o(11, value)

    @property
    def Numsteps(self) -> BatchInt32ArrayProxy:
        """
        Number of steps in this capacitor bank. Default = 1. Forces reallocation of the capacitance, reactor, and states array.  Rules: If this property was previously =1, the value in the kvar property is divided equally among the steps. The kvar property does not need to be reset if that is accurate.  If the Cuf or Cmatrix property was used previously, all steps are set to the value of the first step. The states property is set to all steps on. All filter steps are set to the same harmonic. If this property was previously >1, the arrays are reallocated, but no values are altered. You must SUBSEQUENTLY assign all array properties.

        DSS property name: `Numsteps`, DSS property index: 12.
        """
        return BatchInt32ArrayProxy(self, 12)

    @Numsteps.setter
    def Numsteps(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(12, value)

    @property
    def states(self) -> List[Int32Array]:
        """
        ARRAY of integers {1|0} states representing the state of each step (on|off). Defaults to 1 when reallocated (on). Capcontrol will modify this array as it turns steps on or off.

        DSS property name: `states`, DSS property index: 13.
        """
        return [
            self._get_int32_array(self._lib.Obj_GetInt32Array, x, 13)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @states.setter
    def states(self, value: Int32Array): #TODO: list of arrays, matrix
        self._set_batch_int32_array(13, value)

    @property
    def normamps(self) -> BatchFloat64ArrayProxy:
        """
        Normal rated current.

        DSS property name: `normamps`, DSS property index: 14.
        """
        return BatchFloat64ArrayProxy(self, 14)

    @normamps.setter
    def normamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(14, value)

    @property
    def emergamps(self) -> BatchFloat64ArrayProxy:
        """
        Maximum or emerg current.

        DSS property name: `emergamps`, DSS property index: 15.
        """
        return BatchFloat64ArrayProxy(self, 15)

    @emergamps.setter
    def emergamps(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(15, value)

    @property
    def faultrate(self) -> BatchFloat64ArrayProxy:
        """
        Failure rate per year.

        DSS property name: `faultrate`, DSS property index: 16.
        """
        return BatchFloat64ArrayProxy(self, 16)

    @faultrate.setter
    def faultrate(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(16, value)

    @property
    def pctperm(self) -> BatchFloat64ArrayProxy:
        """
        Percent of failures that become permanent.

        DSS property name: `pctperm`, DSS property index: 17.
        """
        return BatchFloat64ArrayProxy(self, 17)

    @pctperm.setter
    def pctperm(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(17, value)

    @property
    def repair(self) -> BatchFloat64ArrayProxy:
        """
        Hours to repair.

        DSS property name: `repair`, DSS property index: 18.
        """
        return BatchFloat64ArrayProxy(self, 18)

    @repair.setter
    def repair(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(18, value)

    @property
    def basefreq(self) -> BatchFloat64ArrayProxy:
        """
        Base Frequency for ratings.

        DSS property name: `basefreq`, DSS property index: 19.
        """
        return BatchFloat64ArrayProxy(self, 19)

    @basefreq.setter
    def basefreq(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(19, value)

    @property
    def enabled(self) -> List[bool]:
        """
        {Yes|No or True|False} Indicates whether this element is enabled.

        DSS property name: `enabled`, DSS property index: 20.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 20)
        ]
    @enabled.setter
    def enabled(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0], 20, value)

    def like(self, value: AnyStr):
        """
        Make like another object, e.g.:

        New Capacitor.C2 like=c1  ...

        DSS property name: `like`, DSS property index: 21.
        """
        self._set_batch_string(21, value)

class ReactorBatch(DSSBatch):
    _cls_name = 'Reactor'
    _obj_cls = Reactor
    _cls_idx = 23


    @property
    def bus1(self) -> List[str]:
        """
        Name of first bus. Examples:
        bus1=busname
        bus1=busname.1.2.3

        Bus2 property will default to this bus, node 0, unless previously specified. Only Bus1 need be specified for a Yg shunt reactor.

        DSS property name: `bus1`, DSS property index: 1.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 1) 

    @bus1.setter
    def bus1(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(1, value)

    @property
    def bus2(self) -> List[str]:
        """
        Name of 2nd bus. Defaults to all phases connected to first bus, node 0, (Shunt Wye Connection) except when Bus2 is specifically defined.

        Not necessary to specify for delta (LL) connection

        DSS property name: `bus2`, DSS property index: 2.
        """

        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 2) 

    @bus2.setter
    def bus2(self, value: Union[AnyStr, List[AnyStr]]):
        self._set_batch_string(2, value)

    @property
    def phases(self) -> BatchInt32ArrayProxy:
        """
        Number of phases.

        DSS property name: `phases`, DSS property index: 3.
        """
        return BatchInt32ArrayProxy(self, 3)

    @phases.setter
    def phases(self, value: Union[int, Int32Array]):
        self._set_batch_int32_array(3, value)

    @property
    def kvar(self) -> BatchFloat64ArrayProxy:
        """
        Total kvar, all phases.  Evenly divided among phases. Only determines X. Specify R separately

        DSS property name: `kvar`, DSS property index: 4.
        """
        return BatchFloat64ArrayProxy(self, 4)

    @kvar.setter
    def kvar(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(4, value)

    @property
    def kv(self) -> BatchFloat64ArrayProxy:
        """
        For 2, 3-phase, kV phase-phase. Otherwise specify actual coil rating.

        DSS property name: `kv`, DSS property index: 5.
        """
        return BatchFloat64ArrayProxy(self, 5)

    @kv.setter
    def kv(self, value: Union[float, Float64Array]):
        self._set_batch_float64_array(5, value)

    @property
    def conn(self) -> BatchInt32ArrayProxy:
        """
        ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN. If Delta, then only one terminal.

        DSS property name: `conn`, DSS property index: 6.
        """
        return BatchInt32ArrayProxy(self, 6)

    @conn.setter
    def conn(self, value: Union[AnyStr, int, Connection, List[AnyStr], List[int], List[Connection], Int32Array]):
        if isinstance(value, (str, bytes)) or (isinstance(value, LIST_LIKE) and isinstance(value[0], (str, bytes))):
            self._set_batch_string(6, value)
            return
    
        self._set_batch_int32_array(6, value)

    @property
    def conn_str(self) -> str:
        """
        ={wye | delta |LN |LL}  Default is wye, which is equivalent to LN. If Delta, then only one terminal.

        DSS property name: `conn`, DSS property index: 6.
        """
        return self._get_string_array(self._lib.Batch_GetString, self.pointer[0], self.count[0], 6)

    @conn_str.setter
    def conn_str(self, value: AnyStr):
        self.conn = value

    @property
    def Rmatrix(self) -> List[Float64Array]:
        """
        Resistance matrix, lower triangle, ohms at base frequency. Order of the matrix is the number of phases. Mutually exclusive to specifying parameters by kvar or X.

        DSS property name: `Rmatrix`, DSS property index: 7.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 7)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Rmatrix.setter
    def Rmatrix(self, value: Float64Array):
        self._set_float64_array_o(7, value)

    @property
    def Xmatrix(self) -> List[Float64Array]:
        """
        Reactance matrix, lower triangle, ohms at base frequency. Order of the matrix is the number of phases. Mutually exclusive to specifying parameters by kvar or X.

        DSS property name: `Xmatrix`, DSS property index: 8.
        """
        return [
            self._get_float64_array(self._lib.Obj_GetFloat64Array, x, 8)
            for x in self._ffi.unpack(self.pointer[0], self.count[0])
        ]

    @Xmatrix.setter
    def Xmatrix(self, value: Float64Array):
        self._set_float64_array_o(8, value)

    @property
    def Parallel(self) -> List[bool]:
        """
        {Yes | No}  Default=No. Indicates whether Rmatrix and Xmatrix are to be considered in parallel. Default is series. For other models, specify R and Rp.

        DSS property name: `Parallel`, DSS property index: 9.
        """
        return [v != 0 for v in 
            self._get_int32_array(self._lib.Batch_GetInt32, self.pointer[0], self.count[0], 9)
        ]
    @Parallel.setter
    def Parallel(self, value: bool):
        self._lib.Batch_SetInt32(self.pointer[0], self.count[0