//! Types for interacting with Duper's abstract syntax tree.

use std::{
    borrow::Cow,
    collections::HashMap,
    fmt::{Debug, Display},
};

use crate::{DuperParser, DuperRule, visitor::DuperVisitor};

/// A Duper identifier: `MyIdentifier(...)`
#[derive(Debug, Clone, Hash, PartialEq, Eq)]
pub struct DuperIdentifier<'a>(pub(crate) Cow<'a, str>);

/// A Duper value.
#[derive(Debug, Clone)]
pub struct DuperValue<'a> {
    /// The identifier of this value.
    pub identifier: Option<DuperIdentifier<'a>>,
    /// The actual value contained here.
    pub inner: DuperInner<'a>,
}

/// The value contained within a [`DuperValue`].
#[derive(Debug, Clone, PartialEq)]
pub enum DuperInner<'a> {
    /// A Duper object: `{...}`
    Object(DuperObject<'a>),
    /// A Duper array: `[...]`
    Array(DuperArray<'a>),
    /// A Duper tuple: `(...)`
    Tuple(DuperTuple<'a>),
    /// A Duper string: `"..."`
    String(DuperString<'a>),
    /// A Duper bytestring: `b"..."`
    Bytes(DuperBytes<'a>),
    /// A Duper integer.
    Integer(i64),
    /// A Duper float.
    Float(f64),
    /// A Duper boolean.
    Boolean(bool),
    /// A Duper null.
    Null,
}

/// A key in a [`DuperObject`].
#[derive(Debug, Clone, Hash, PartialEq, Eq)]
pub struct DuperKey<'a>(pub(crate) Cow<'a, str>);

/// An object (or map) from [`DuperKey`]s to [`DuperValue`]s.
#[derive(Debug, Clone)]
pub struct DuperObject<'a>(pub(crate) Vec<(DuperKey<'a>, DuperValue<'a>)>);

/// An array (or list) of [`DuperValue`]s.
#[derive(Debug, Clone, PartialEq)]
pub struct DuperArray<'a>(pub(crate) Vec<DuperValue<'a>>);

/// An tuple of [`DuperValue`]s.
#[derive(Debug, Clone, PartialEq)]
pub struct DuperTuple<'a>(pub(crate) Vec<DuperValue<'a>>);

/// A string, which may be borrowed or owned.
#[derive(Debug, Clone, Hash, PartialEq, Eq)]
pub struct DuperString<'a>(pub(crate) Cow<'a, str>);

/// A byte sequence, which may be borrowed or owned.
#[derive(Debug, Clone, Hash, PartialEq, Eq)]
pub struct DuperBytes<'a>(pub(crate) Cow<'a, [u8]>);

/// Possible errors generated by [`DuperIdentifier::try_from()`].
#[derive(Debug, Clone)]
pub enum DuperIdentifierTryFromError<'a> {
    /// The identifier was empty.
    EmptyIdentifier,
    /// The identifier contained an invalid character.
    InvalidChar(Cow<'a, str>, usize),
}

/// Possible errors generated by [`DuperObject::try_from()`].
#[derive(Debug, Clone)]
pub enum DuperObjectTryFromError<'a> {
    /// The key was duplicated.
    DuplicateKey(Cow<'a, str>),
}

impl<'a> DuperIdentifier<'a> {
    /// Consume this identifier and return the underlying [`Cow<'_, str>`].
    pub fn into_inner(self) -> Cow<'a, str> {
        self.0
    }

    /// Create a valid identifier from the provided [`Cow<'_, str>`], discarding
    /// any invalid characters if necessary.
    pub fn try_from_lossy(value: Cow<'a, str>) -> Result<Self, DuperIdentifierTryFromError<'a>> {
        let mut new_value = None;
        let mut chars = value.char_indices();
        match chars.next() {
            None => return Err(DuperIdentifierTryFromError::EmptyIdentifier),
            Some((_, c)) if c.is_alphabetic() && !c.is_uppercase() => {
                new_value = Some(c.to_uppercase().to_string());
            }
            Some((_, c)) if c.is_alphabetic() && c.is_uppercase() => (),
            _ => return Err(DuperIdentifierTryFromError::InvalidChar(value, 0)),
        }
        let mut last_char_was_separator = false;
        for (pos, char) in chars {
            match char {
                '-' | '_' => {
                    match new_value.as_mut() {
                        Some(new_value) if !last_char_was_separator => {
                            new_value.push(char);
                        }
                        _ => (),
                    }
                    last_char_was_separator = true;
                }
                char if char.is_alphanumeric() => {
                    if let Some(new_value) = new_value.as_mut() {
                        new_value.push(char);
                    }
                    last_char_was_separator = false;
                }
                _ => match new_value.as_mut() {
                    Some(_) => (),
                    None => new_value = Some(value.split_at(pos).0.to_owned()),
                },
            }
        }
        Ok(Self(match new_value {
            Some(new_value) if new_value.is_empty() => {
                return Err(DuperIdentifierTryFromError::EmptyIdentifier);
            }
            Some(new_value) => Cow::Owned(new_value),
            None => value,
        }))
    }

    /// Create a clone of this DuperIdentifier with a static lifetime.
    pub fn static_clone(&self) -> DuperIdentifier<'static> {
        DuperIdentifier(Cow::Owned(self.0.clone().into_owned()))
    }
}

impl<'a> Display for DuperIdentifier<'a> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.write_str(&self.0)
    }
}

impl<'a> AsRef<str> for DuperIdentifier<'a> {
    fn as_ref(&self) -> &str {
        &self.0
    }
}

impl<'a> TryFrom<Cow<'a, str>> for DuperIdentifier<'a> {
    type Error = DuperIdentifierTryFromError<'a>;

    /// Create a valid identifier from the provided [`Cow<'_, str>`], returning
    /// an error if there are invalid characters.
    fn try_from(value: Cow<'a, str>) -> Result<Self, Self::Error> {
        let mut chars = value.char_indices();
        match chars.next() {
            None => return Err(DuperIdentifierTryFromError::EmptyIdentifier),
            Some((_, c)) if !c.is_uppercase() => {
                return Err(DuperIdentifierTryFromError::InvalidChar(value, 0));
            }
            _ => (),
        }
        let mut last_char_was_separator = false;
        for (pos, char) in chars {
            match char {
                '-' | '_' => {
                    if last_char_was_separator {
                        return Err(DuperIdentifierTryFromError::InvalidChar(value, pos));
                    } else {
                        last_char_was_separator = true;
                    }
                }
                char if char.is_alphanumeric() => {
                    last_char_was_separator = false;
                }
                _ => return Err(DuperIdentifierTryFromError::InvalidChar(value, pos)),
            }
        }
        Ok(Self(value))
    }
}

impl<'a> TryFrom<&'a str> for DuperIdentifier<'a> {
    type Error = DuperIdentifierTryFromError<'a>;

    /// Create a valid identifier from the provided `&str`, returning
    /// an error if there are invalid characters.
    fn try_from(value: &'a str) -> Result<Self, Self::Error> {
        Self::try_from(Cow::Borrowed(value))
    }
}

impl TryFrom<String> for DuperIdentifier<'static> {
    type Error = DuperIdentifierTryFromError<'static>;

    /// Create a valid identifier from the provided [`String`], returning
    /// an error if there are invalid characters.
    fn try_from(value: String) -> Result<Self, Self::Error> {
        Self::try_from(Cow::Owned(value))
    }
}

impl Display for DuperIdentifierTryFromError<'_> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            DuperIdentifierTryFromError::EmptyIdentifier => f.write_str("empty identifier"),
            DuperIdentifierTryFromError::InvalidChar(identifier, pos) => f.write_fmt(format_args!(
                "invalid character in position {pos} of identifier {identifier}"
            )),
        }
    }
}

impl std::error::Error for DuperIdentifierTryFromError<'_> {}

impl<'a> DuperKey<'a> {
    /// Consume this key and return the underlying [`Cow<'_, str>`].
    pub fn into_inner(self) -> Cow<'a, str> {
        self.0
    }
}

impl<'a> AsRef<str> for DuperKey<'a> {
    fn as_ref(&self) -> &str {
        &self.0
    }
}

impl<'a> From<Cow<'a, str>> for DuperKey<'a> {
    fn from(value: Cow<'a, str>) -> Self {
        Self(value)
    }
}

impl<'a> From<&'a str> for DuperKey<'a> {
    fn from(value: &'a str) -> Self {
        Self(Cow::Borrowed(value))
    }
}

impl From<String> for DuperKey<'static> {
    fn from(value: String) -> Self {
        Self(Cow::Owned(value))
    }
}

impl<'a> DuperValue<'a> {
    /// Accepts a [`DuperVisitor`] and visits it with the current value.
    pub fn accept<V: DuperVisitor>(&self, visitor: &mut V) -> V::Value {
        match &self.inner {
            DuperInner::Object(object) => visitor.visit_object(self.identifier.as_ref(), object),
            DuperInner::Array(array) => visitor.visit_array(self.identifier.as_ref(), array),
            DuperInner::Tuple(tuple) => visitor.visit_tuple(self.identifier.as_ref(), tuple),
            DuperInner::String(string) => visitor.visit_string(self.identifier.as_ref(), string),
            DuperInner::Bytes(bytes) => visitor.visit_bytes(self.identifier.as_ref(), bytes),
            DuperInner::Integer(integer) => {
                visitor.visit_integer(self.identifier.as_ref(), *integer)
            }
            DuperInner::Float(float) => visitor.visit_float(self.identifier.as_ref(), *float),
            DuperInner::Boolean(boolean) => {
                visitor.visit_boolean(self.identifier.as_ref(), *boolean)
            }
            DuperInner::Null => visitor.visit_null(self.identifier.as_ref()),
        }
    }
}

impl<'a> TryFrom<&'a str> for DuperValue<'a> {
    type Error = Box<pest::error::Error<DuperRule>>;

    fn try_from(value: &'a str) -> Result<Self, Self::Error> {
        DuperParser::parse_duper_value(value)
    }
}

impl<'a> PartialEq for DuperValue<'a> {
    fn eq(&self, other: &Self) -> bool {
        self.inner == other.inner
    }
}

impl<'a> DuperObject<'a> {
    /// Consume this object and return the underlying [`Vec<(DuperKey<'_>, DuperValue<'_>)>`].
    pub fn into_inner(self) -> Vec<(DuperKey<'a>, DuperValue<'a>)> {
        self.0
    }

    /// Returns `true` if the object contains no elements.
    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }

    /// Returns the amount of elements in this object.
    pub fn len(&self) -> usize {
        self.0.len()
    }

    /// Returns an iterator over references to the (key, value) pairs in this
    /// object.
    pub fn iter(&self) -> impl Iterator<Item = &(DuperKey<'a>, DuperValue<'a>)> {
        self.0.iter()
    }
}

impl<'a> TryFrom<Vec<(DuperKey<'a>, DuperValue<'a>)>> for DuperObject<'a> {
    type Error = DuperObjectTryFromError<'a>;

    /// Create a valid object from the provided [`Vec`], returning an error if
    /// a duplicate key is found.
    fn try_from(value: Vec<(DuperKey<'a>, DuperValue<'a>)>) -> Result<Self, Self::Error> {
        let mut keys = std::collections::HashSet::with_capacity(value.len());
        for (key, _) in value.iter() {
            if keys.contains(key) {
                return Err(DuperObjectTryFromError::DuplicateKey(key.0.clone()));
            }
            keys.insert(key);
        }
        Ok(Self(value))
    }
}

impl<'a> PartialEq for DuperObject<'a> {
    fn eq(&self, other: &Self) -> bool {
        if self.0.len() != other.0.len() {
            return false;
        }
        let other_map: HashMap<_, _> = other.0.iter().map(|(k, v)| (k.clone(), v)).collect();
        for (k, v) in self.0.iter() {
            match other_map.get(k) {
                Some(v2) => {
                    if v != *v2 {
                        return false;
                    }
                }
                None => return false,
            }
        }
        true
    }
}

impl Display for DuperObjectTryFromError<'_> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            DuperObjectTryFromError::DuplicateKey(key) => {
                f.write_fmt(format_args!("duplicate key {key} in object"))
            }
        }
    }
}

impl std::error::Error for DuperObjectTryFromError<'_> {}

impl<'a> DuperArray<'a> {
    /// Consume this array and return the underlying [`Vec<DuperValue<'_>>`].
    pub fn into_inner(self) -> Vec<DuperValue<'a>> {
        self.0
    }

    /// Returns `true` if the array contains no elements.
    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }

    /// Returns the amount of elements in this array.
    pub fn len(&self) -> usize {
        self.0.len()
    }

    /// Returns an iterator over references to the values in this array.
    pub fn iter(&self) -> impl Iterator<Item = &DuperValue<'a>> {
        self.0.iter()
    }

    /// Returns a reference to the value at the given position.
    pub fn get(&self, index: usize) -> Option<&DuperValue<'a>> {
        self.0.get(index)
    }
}

impl<'a> From<Vec<DuperValue<'a>>> for DuperArray<'a> {
    fn from(value: Vec<DuperValue<'a>>) -> Self {
        Self(value)
    }
}

impl<'a> DuperTuple<'a> {
    /// Consume this tuple and return the underlying [`Vec<DuperValue<'_>>`].
    pub fn into_inner(self) -> Vec<DuperValue<'a>> {
        self.0
    }

    /// Returns `true` if the tuple contains no elements.
    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }

    /// Returns the amount of elements in this tuple.
    pub fn len(&self) -> usize {
        self.0.len()
    }

    /// Returns an iterator over references to the values in this tuple.
    pub fn iter(&self) -> impl Iterator<Item = &DuperValue<'a>> {
        self.0.iter()
    }

    /// Returns a reference to the value at the given position.
    pub fn get(&self, index: usize) -> Option<&DuperValue<'a>> {
        self.0.get(index)
    }
}

impl<'a> From<Vec<DuperValue<'a>>> for DuperTuple<'a> {
    fn from(value: Vec<DuperValue<'a>>) -> Self {
        Self(value)
    }
}

impl<'a> DuperString<'a> {
    /// Consume this string and return the underlying [`Cow<'_, str>`].
    pub fn into_inner(self) -> Cow<'a, str> {
        self.0
    }
}

impl<'a> From<Cow<'a, str>> for DuperString<'a> {
    fn from(value: Cow<'a, str>) -> Self {
        Self(value)
    }
}

impl<'a> From<&'a str> for DuperString<'a> {
    fn from(value: &'a str) -> Self {
        Self(Cow::Borrowed(value))
    }
}

impl From<String> for DuperString<'static> {
    fn from(value: String) -> Self {
        Self(Cow::Owned(value))
    }
}

impl<'a> AsRef<str> for DuperString<'a> {
    fn as_ref(&self) -> &str {
        &self.0
    }
}

impl<'a> DuperBytes<'a> {
    /// Consume these bytes and return the underlying [`Cow<'_, [u8]>`].
    pub fn into_inner(self) -> Cow<'a, [u8]> {
        self.0
    }
}

impl<'a> From<Cow<'a, [u8]>> for DuperBytes<'a> {
    fn from(value: Cow<'a, [u8]>) -> Self {
        Self(value)
    }
}

impl<'a> From<&'a [u8]> for DuperBytes<'a> {
    fn from(value: &'a [u8]) -> Self {
        Self(Cow::Borrowed(value))
    }
}

impl From<Vec<u8>> for DuperBytes<'static> {
    fn from(value: Vec<u8>) -> Self {
        Self(Cow::Owned(value))
    }
}

impl<'a> AsRef<[u8]> for DuperBytes<'a> {
    fn as_ref(&self) -> &[u8] {
        &self.0
    }
}
