use std::borrow::Cow;

use chumsky::prelude::*;

use crate::{
    DuperBytes, DuperString,
    ast::{DuperArray, DuperIdentifier, DuperInner, DuperKey, DuperObject, DuperTuple, DuperValue},
    escape::{unescape_bytes, unescape_str},
};

pub struct DuperParser;

impl DuperParser {
    /// Parse a Duper trunk, i.e. only an array, tuple, or object at the top level.
    ///
    /// A pretty printed version of the error can be obtained from the [`prettify_error`] method.
    pub fn parse_duper_trunk<'a>(input: &'a str) -> Result<DuperValue<'a>, Vec<Rich<'a, char>>> {
        // duper_trunk().parse(input).into_result()
        let value = duper_trunk().parse(input).into_result()?;
        match &value.inner {
            DuperInner::Object(_) | DuperInner::Array(_) | DuperInner::Tuple(_) => Ok(value),
            _ => todo!(),
        }
    }

    /// Parse a Duper value at the top level.
    ///
    /// A pretty printed version of the error can be obtained from the [`prettify_error`] method.
    pub fn parse_duper_value<'a>(input: &'a str) -> Result<DuperValue<'a>, Vec<Rich<'a, char>>> {
        duper_value().parse(input).into_result()
    }

    /// Prettifies the errors generated by this parser.
    pub fn prettify_error<'a>(
        input: &'a str,
        errors: &[Rich<'a, char>],
        filename: Option<&str>,
    ) -> String {
        let filename = filename.unwrap_or("input");
        let mut message = Vec::new();
        for rich in errors.iter() {
            let filename = filename.to_owned();
            ariadne::Report::build(
                ariadne::ReportKind::Error,
                (filename.clone(), rich.span().into_range()),
            )
            .with_config(ariadne::Config::new().with_index_type(ariadne::IndexType::Byte))
            .with_code(3)
            .with_message(rich.to_string())
            .with_label(
                ariadne::Label::new((filename.clone(), rich.span().into_range()))
                    .with_message(rich.reason().to_string())
                    .with_color(ariadne::Color::Red),
            )
            .finish()
            .write(ariadne::sources([(filename, input)]), &mut message)
            .unwrap();
        }
        // String::from_utf8(message).unwrap()
        String::from_utf8_lossy(&message).into_owned()
    }
}

pub(crate) fn duper_trunk<'a>()
-> impl Parser<'a, &'a str, DuperValue<'a>, extra::Err<Rich<'a, char>>> {
    identified_trunk().then_ignore(end())
}

pub(crate) fn duper_value<'a>()
-> impl Parser<'a, &'a str, DuperValue<'a>, extra::Err<Rich<'a, char>>> {
    identified_value().then_ignore(end())
}

pub(crate) fn whitespace_and_comments<'a>()
-> impl Parser<'a, &'a str, (), extra::Err<Rich<'a, char>>> + Clone {
    choice((
        just("//")
            .then(none_of("\r\n").repeated())
            .padded()
            .ignored(),
        just("/*")
            .then(any().and_is(just("*/").not()).repeated())
            .then(just("*/"))
            .padded()
            .ignored(),
        text::whitespace().ignored(),
    ))
    .ignored()
}

pub(crate) fn identifier<'a>()
-> impl Parser<'a, &'a str, DuperIdentifier<'a>, extra::Err<Rich<'a, char>>> + Clone {
    one_of('A'..='Z')
        .labelled("ASCII uppercase letter")
        .then(
            just('-')
                .or(just('_'))
                .or_not()
                .then(
                    any()
                        .filter(|c: &char| c.is_ascii_alphanumeric())
                        .labelled("ASCII alphanumeric"),
                )
                .repeated(),
        )
        .to_slice()
        .map(|identifier| DuperIdentifier(Cow::Borrowed(identifier)))
}

pub(crate) fn identified_trunk<'a>()
-> impl Parser<'a, &'a str, DuperValue<'a>, extra::Err<Rich<'a, char>>> + Clone {
    let inner_trunk = choice((
        object(identified_value()).map(DuperInner::Object),
        array(identified_value()).map(DuperInner::Array),
        tuple(identified_value()).map(DuperInner::Tuple),
    ))
    .padded_by(whitespace_and_comments());

    identifier()
        .then(inner_trunk.clone().delimited_by(just('('), just(')')))
        .map(|(identifier, inner)| DuperValue {
            identifier: Some(identifier),
            inner,
        })
        .or(inner_trunk.map(|inner| DuperValue {
            identifier: None,
            inner,
        }))
        .padded_by(whitespace_and_comments())
}

pub(crate) fn identified_value<'a>()
-> impl Parser<'a, &'a str, DuperValue<'a>, extra::Err<Rich<'a, char>>> + Clone {
    recursive(move |identified_value| {
        let inner_value = choice((
            object(identified_value.clone()).map(DuperInner::Object),
            array(identified_value.clone()).map(DuperInner::Array),
            tuple(identified_value).map(DuperInner::Tuple),
            quoted_string().map(|cow_str| DuperInner::String(DuperString(cow_str))),
            raw_string().map(|str| DuperInner::String(DuperString(Cow::Borrowed(str)))),
            quoted_bytes().map(|cow_bytes| DuperInner::Bytes(DuperBytes(cow_bytes))),
            raw_bytes().map(|bytes| DuperInner::Bytes(DuperBytes(Cow::Borrowed(bytes)))),
            float().map(DuperInner::Float),
            integer().map(DuperInner::Integer),
            boolean().map(DuperInner::Boolean),
            null().map(|_| DuperInner::Null),
        ))
        .padded_by(whitespace_and_comments());

        identifier()
            .then(inner_value.clone().delimited_by(just('('), just(')')))
            .map(|(identifier, inner)| DuperValue {
                identifier: Some(identifier),
                inner,
            })
            .or(inner_value.map(|inner| DuperValue {
                identifier: None,
                inner,
            }))
            .padded_by(whitespace_and_comments())
    })
}

pub(crate) fn object<'a>(
    identified_value: impl Parser<'a, &'a str, DuperValue<'a>, extra::Err<Rich<'a, char>>> + Clone,
) -> impl Parser<'a, &'a str, DuperObject<'a>, extra::Err<Rich<'a, char>>> + Clone {
    object_key()
        .then_ignore(
            just(':')
                .ignored()
                .recover_with(via_parser(value_recovery()))
                .padded_by(whitespace_and_comments()),
        )
        .then(identified_value)
        .padded_by(whitespace_and_comments())
        .separated_by(
            just(',')
                .ignored()
                .recover_with(via_parser(key_recovery()))
                .padded_by(whitespace_and_comments()),
        )
        .allow_trailing()
        .collect::<Vec<_>>()
        .try_map(|object, span| {
            DuperObject::try_from(object).map_err(|err| Rich::custom(span, err))
        })
        .delimited_by(
            just('{').padded_by(whitespace_and_comments()),
            just('}').padded_by(whitespace_and_comments()),
        )
}

pub(crate) fn object_key<'a>()
-> impl Parser<'a, &'a str, DuperKey<'a>, extra::Err<Rich<'a, char>>> + Clone {
    let plain_key = any()
        .filter(|c: &char| c.is_ascii_alphabetic())
        .labelled("ASCII letter")
        .to_slice()
        .or(just('_')
            .then(
                any()
                    .filter(|c: &char| c.is_ascii_alphanumeric())
                    .labelled("ASCII alphanumeric"),
            )
            .to_slice())
        .then(
            just('-')
                .or(just('_'))
                .or_not()
                .then(
                    any()
                        .filter(|c: &char| c.is_ascii_alphanumeric())
                        .labelled("ASCII alphanumeric"),
                )
                .repeated(),
        )
        .to_slice()
        .map(|str| DuperKey(Cow::Borrowed(str)));

    quoted_string()
        .map(DuperKey)
        .or(raw_string().map(|str| DuperKey(Cow::Borrowed(str))))
        .or(plain_key)
}

pub(crate) fn array<'a>(
    identified_value: impl Parser<'a, &'a str, DuperValue<'a>, extra::Err<Rich<'a, char>>> + Clone,
) -> impl Parser<'a, &'a str, DuperArray<'a>, extra::Err<Rich<'a, char>>> + Clone {
    identified_value
        .padded_by(whitespace_and_comments())
        .separated_by(
            just(',')
                .ignored()
                .recover_with(via_parser(value_recovery()))
                .padded_by(whitespace_and_comments()),
        )
        .allow_trailing()
        .collect::<Vec<_>>()
        .map(DuperArray)
        .delimited_by(
            just('[').padded_by(whitespace_and_comments()),
            just(']').padded_by(whitespace_and_comments()),
        )
        .or(just(',')
            .padded_by(whitespace_and_comments())
            .delimited_by(just('['), just(']'))
            .map(|_| DuperArray(vec![])))
}

pub(crate) fn tuple<'a>(
    identified_value: impl Parser<'a, &'a str, DuperValue<'a>, extra::Err<Rich<'a, char>>> + Clone,
) -> impl Parser<'a, &'a str, DuperTuple<'a>, extra::Err<Rich<'a, char>>> + Clone {
    identified_value
        .padded_by(whitespace_and_comments())
        .separated_by(
            just(',')
                .ignored()
                .recover_with(via_parser(value_recovery()))
                .padded_by(whitespace_and_comments()),
        )
        .allow_trailing()
        .collect::<Vec<_>>()
        .map(DuperTuple)
        .delimited_by(
            just('(').padded_by(whitespace_and_comments()),
            just(')').padded_by(whitespace_and_comments()),
        )
        .or(just(',')
            .padded_by(whitespace_and_comments())
            .delimited_by(just('('), just(')'))
            .map(|_| DuperTuple(vec![])))
}

pub(crate) fn key_recovery<'a>() -> impl Parser<'a, &'a str, (), extra::Err<Rich<'a, char>>> + Clone
{
    choice((one_of('a'..='z'), one_of('A'..='Z'), one_of("_\""))).ignored()
}

pub(crate) fn value_recovery<'a>()
-> impl Parser<'a, &'a str, (), extra::Err<Rich<'a, char>>> + Clone {
    choice((
        one_of('0'..='9'),
        one_of('A'..='Z'),
        one_of(r#"_"brntf-+{[("#),
    ))
    .ignored()
}

pub(crate) fn quoted_string<'a>()
-> impl Parser<'a, &'a str, Cow<'a, str>, extra::Err<Rich<'a, char>>> + Clone {
    quoted_inner()
        .delimited_by(just('"'), just('"'))
        .try_map(|str, span| unescape_str(str).map_err(|err| Rich::custom(span, err)))
}

pub(crate) fn quoted_bytes<'a>()
-> impl Parser<'a, &'a str, Cow<'a, [u8]>, extra::Err<Rich<'a, char>>> + Clone {
    quoted_inner()
        .delimited_by(just("b\""), just('"'))
        .try_map(|bytes, span| unescape_bytes(bytes).map_err(|err| Rich::custom(span, err)))
}

pub(crate) fn quoted_inner<'a>()
-> impl Parser<'a, &'a str, &'a str, extra::Err<Rich<'a, char>>> + Clone {
    let escaped_characters = just('\\')
        .then(choice((
            one_of("\"\\/bfnrt0").to_slice(),
            just('x')
                .then(
                    any()
                        .filter(|char: &char| char.is_ascii_hexdigit())
                        .labelled("a hexadecimal digit")
                        .repeated()
                        .exactly(2),
                )
                .to_slice(),
            just('u')
                .then(
                    any()
                        .filter(|char: &char| char.is_ascii_hexdigit())
                        .labelled("a hexadecimal digit")
                        .repeated()
                        .exactly(4),
                )
                .to_slice(),
        )))
        .to_slice();

    none_of("\"\\")
        .to_slice()
        .or(escaped_characters)
        .repeated()
        .to_slice()
}

pub(crate) fn raw_string<'a>()
-> impl Parser<'a, &'a str, &'a str, extra::Err<Rich<'a, char>>> + Clone {
    let hashtags = just('#')
        .repeated()
        .to_slice()
        .map(|slice: &str| slice.len())
        .delimited_by(just('r'), just('"'));

    hashtags.ignore_with_ctx(
        any()
            .and_is(
                just('"')
                    .then(
                        just('#')
                            .repeated()
                            .configure(|repeated, ctx| repeated.exactly(*ctx)),
                    )
                    .not(),
            )
            .repeated()
            .to_slice()
            .then_ignore(
                just('"').then(
                    just('#')
                        .repeated()
                        .configure(|repeated, ctx| repeated.exactly(*ctx)),
                ),
            ),
    )
}

pub(crate) fn raw_bytes<'a>()
-> impl Parser<'a, &'a str, &'a [u8], extra::Err<Rich<'a, char>>> + Clone {
    let hashtags = just('#')
        .repeated()
        .to_slice()
        .map(|slice: &str| slice.len())
        .delimited_by(just("br"), just('"'));

    hashtags.ignore_with_ctx(
        any()
            .and_is(
                just('"')
                    .then(
                        just('#')
                            .repeated()
                            .configure(|repeated, ctx| repeated.exactly(*ctx)),
                    )
                    .not(),
            )
            .repeated()
            .to_slice()
            .map(|slice: &str| slice.as_bytes())
            .then_ignore(
                just('"').then(
                    just('#')
                        .repeated()
                        .configure(|repeated, ctx| repeated.exactly(*ctx)),
                ),
            ),
    )
}

pub(crate) fn integer_digits<'a>()
-> impl Parser<'a, &'a str, &'a str, extra::Err<Rich<'a, char>>> + Clone {
    one_of('1'..='9')
        .labelled("a digit 1 through 9")
        .then(
            just('_')
                .or_not()
                .then(one_of('0'..='9').labelled("a digit"))
                .repeated(),
        )
        .to_slice()
        .or(one_of('0'..='9').labelled("a digit").to_slice())
}

pub(crate) fn float<'a>() -> impl Parser<'a, &'a str, f64, extra::Err<Rich<'a, char>>> + Clone {
    let decimal = just('+')
        .or(just('-'))
        .or_not()
        .then(integer_digits())
        .to_slice();

    let fractional = just('.').then(
        one_of('0'..='9').labelled("a digit").then(
            just('_')
                .or_not()
                .then(one_of('0'..='9').labelled("a digit"))
                .repeated(),
        ),
    );

    let exponent = one_of("eE")
        .then(just('+').or(just('-')).or_not())
        .then(integer_digits())
        .to_slice();

    choice((
        decimal.clone().then(exponent.clone()).to_slice(),
        decimal.then(fractional.then(exponent.or_not())).to_slice(),
    ))
    .to_slice()
    .try_map(|float: &str, span| {
        let float: f64 = float
            .replace('_', "")
            .parse()
            .map_err(|err| Rich::custom(span, err))?;
        if float.is_infinite() {
            Err(Rich::custom(span, "float cannot be represented in f64"))
        } else {
            Ok(float)
        }
    })
}

pub(crate) fn integer<'a>() -> impl Parser<'a, &'a str, i64, extra::Err<Rich<'a, char>>> + Clone {
    let decimal_integer = just('+')
        .or(just('-'))
        .or_not()
        .then(integer_digits())
        .to_slice()
        .try_map(|integer: &str, span| {
            integer
                .replace('_', "")
                .parse()
                .map_err(|err| Rich::custom(span, err))
        });

    let hex_integer = just("0x")
        .ignore_then(
            any()
                .filter(|char: &char| char.is_ascii_hexdigit())
                .labelled("a hexadecimal digit")
                .then(
                    just('_')
                        .or_not()
                        .then(
                            any()
                                .filter(|char: &char| char.is_ascii_hexdigit())
                                .labelled("a hexadecimal digit"),
                        )
                        .repeated(),
                )
                .to_slice(),
        )
        .try_map(|integer: &str, span| {
            i64::from_str_radix(&integer.replace('_', ""), 16)
                .map_err(|err| Rich::custom(span, err))
        });

    let octal_integer = just("0o")
        .ignore_then(
            any()
                .filter(|char: &char| char.is_digit(8))
                .labelled("an octal digit")
                .then(
                    just('_')
                        .or_not()
                        .then(
                            any()
                                .filter(|char: &char| char.is_digit(8))
                                .labelled("an octal digit"),
                        )
                        .repeated(),
                )
                .to_slice(),
        )
        .try_map(|integer: &str, span| {
            i64::from_str_radix(&integer.replace('_', ""), 8).map_err(|err| Rich::custom(span, err))
        });

    let binary_integer = just("0b")
        .ignore_then(
            one_of("01")
                .then(just('_').or_not().then(one_of("01")).repeated())
                .to_slice(),
        )
        .try_map(|integer: &str, span| {
            i64::from_str_radix(&integer.replace('_', ""), 2).map_err(|err| Rich::custom(span, err))
        });

    choice((hex_integer, octal_integer, binary_integer, decimal_integer))
}

pub(crate) fn boolean<'a>() -> impl Parser<'a, &'a str, bool, extra::Err<Rich<'a, char>>> + Clone {
    choice((just("true").to(true), just("false").to(false)))
}

pub(crate) fn null<'a>() -> impl Parser<'a, &'a str, (), extra::Err<Rich<'a, char>>> + Clone {
    just("null").to(())
}

#[cfg(test)]
mod duper_parser_tests {
    use crate::{
        DuperArray, DuperBytes, DuperIdentifier, DuperInner, DuperKey, DuperObject, DuperParser,
        DuperString, DuperTuple, DuperValue,
    };

    #[test]
    fn duper_trunk() {
        let input = r#"
            "hello"
        "#;
        assert!(DuperParser::parse_duper_trunk(input).is_err());

        let input = r#"
            br"¯\_(ツ)_/¯"
        "#;
        assert!(DuperParser::parse_duper_trunk(input).is_err());

        let input = r#"
            9001
        "#;
        assert!(DuperParser::parse_duper_trunk(input).is_err());

        let input = r#"
            3.14
        "#;
        assert!(DuperParser::parse_duper_trunk(input).is_err());

        let input = r#"
            true
        "#;
        assert!(DuperParser::parse_duper_trunk(input).is_err());

        let input = r#"
            null
        "#;
        assert!(DuperParser::parse_duper_trunk(input).is_err());

        let input = r#"
            (,)
        "#;
        let duper = DuperParser::parse_duper_trunk(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Tuple(_)));

        let input = r#"
            {duper: 1337}
        "#;
        let duper = DuperParser::parse_duper_trunk(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Object(_)));

        let input = r#"
            [1, 2.2, null]
        "#;
        let duper = DuperParser::parse_duper_trunk(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Array(_)));
    }

    #[test]
    fn duper_value() {
        let input = r#"
            "hello"
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::String(_)));

        let input = r#"
            br"¯\_(ツ)_/¯"
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Bytes(_)));

        let input = r#"
            9001
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Integer(_)));

        let input = r#"
            0x2001
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Integer(_)));

        let input = r#"
            0o755
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Integer(_)));

        let input = r#"
            0b0101_0101
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Integer(_)));

        let input = r#"
            3.14
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Float(_)));

        let input = r#"
            12e-10
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Float(_)));

        let input = r#"
            1_2.3_4E5_6
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Float(_)));

        let input = r#"
            true
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Boolean(_)));

        let input = r#"
            null
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Null));

        let input = r#"
            (,)
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Tuple(_)));

        let input = r#"
            {duper: 1337}
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Object(_)));

        let input = r#"
            [1, 2.2, null]
        "#;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert!(matches!(duper.inner, DuperInner::Array(_)));
    }

    #[test]
    fn parsing_errors() {
        // No values
        let input = "";
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            // Only a line comment
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            /*
                Only a multiline comment
            */
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());

        // Strings and bytes
        let input = r#"
            'single quotes'
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            "unclosed
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            unopened"
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            b"unclosed bytes
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            r"unclosed raw
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            br"unclosed raw bytes
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            "invalid UTF-8 \xF0\x90\x80"
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            "unknown escape \e"
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());

        // Floats and decimal
        let input = r#"
            +
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            -
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            01  // No trailing zero
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            _
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            _12
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12_
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            1__2
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12f
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12.
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12e
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12e+
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"12e-"#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12.34.56
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12e34e56
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12.3a4
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            e10
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            .
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            -.5
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            +.5
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            _.123
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12._34
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12e_34
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12e+_34
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            1__2.34
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12.3__4
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12e3__4
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12.34_
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12e34_
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12e
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            12e3.4
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            1e1000  // Too big to represent
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());

        // Hexadecimal
        let input = r#"
            0x
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0xg
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0x_
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0x1_
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0x_1
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0x1.2
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0x-1
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            -0x1
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());

        // Octal
        let input = r#"
            0o
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0o8
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0o_
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0o1_
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0o_1
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0o1.2
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0o-1
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            -0o1
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());

        // Binary
        let input = r#"
            0b
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0b2
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0b_
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0b1_
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0b_1
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0b1.1
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            0b-1
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            -0b1
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());

        // Objects
        let input = r#"
            {,}
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            {,,}
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            {1234: "invalid key"}
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            {b"value as key": null}
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            {"missing" "colon"}
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            {"missing_value"}
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            {
                name: "duplicated",
                "n\x61me": "duplicated",
            }
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());

        // Arrays and tuples
        let input = r#"
            [,,]
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            (,,)
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());

        // Identifiers and values
        let input = r#"
            ÑInvalidIdentifier({})
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            NoParenthesis[]
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            NoValue()
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            null,
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            ,null
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
        let input = r#"
            true,false
        "#;
        assert!(DuperParser::parse_duper_value(input).is_err());
    }

    #[test]
    fn example() {
        use std::borrow::Cow;

        let input = r##"
            Product({
                product_id: Uuid("1dd7b7aa-515e-405f-85a9-8ac812242609"),
                name: "Wireless Bluetooth Headphones",
                brand: "AudioTech",
                price: Decimal("129.99"),
                dimensions: (18.5, 15.2, 7.8),  // In centimeters
                weight: Kilograms(0.285),
                in_stock: true,
                specifications: {
                    battery_life: Duration("30h"),
                    noise_cancellation: true,
                    connectivity: ["Bluetooth 5.0", "3.5mm Jack"],
                },
                image_thumbnail: Png(b"\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x64"),
                tags: ["electronics", "audio", "wireless"],
                release_date: Date("2023-11-15"),
                /* Warranty is optional */
                warranty_period: null,
                customer_ratings: {
                    latest_review: r#"Absolutely ""astounding""!! 😎"#,
                    average: 4.5,
                    count: 127,
                },
                created_at: DateTime("2023-11-17T21:50:43+00:00"),
            })
        "##;
        let duper = DuperParser::parse_duper_value(input).unwrap();
        assert_eq!(
            duper,
            DuperValue {
                identifier: Some(DuperIdentifier(Cow::Borrowed("Product"))),
                inner: DuperInner::Object(DuperObject(vec![
                    (
                        DuperKey(Cow::Borrowed("product_id")),
                        DuperValue {
                            identifier: Some(DuperIdentifier(Cow::Borrowed("Uuid"))),
                            inner: DuperInner::String(DuperString(Cow::Borrowed(
                                "1dd7b7aa-515e-405f-85a9-8ac812242609"
                            ))),
                        }
                    ),
                    (
                        DuperKey(Cow::Borrowed("name")),
                        DuperValue {
                            identifier: None,
                            inner: DuperInner::String(DuperString(Cow::Borrowed(
                                "Wireless Bluetooth Headphones"
                            ))),
                        }
                    ),
                    (
                        DuperKey(Cow::Borrowed("brand")),
                        DuperValue {
                            identifier: None,
                            inner: DuperInner::String(DuperString(Cow::Borrowed("AudioTech"))),
                        }
                    ),
                    (
                        DuperKey(Cow::Borrowed("price")),
                        DuperValue {
                            identifier: Some(DuperIdentifier(Cow::Borrowed("Decimal"))),
                            inner: DuperInner::String(DuperString(Cow::Borrowed("129.99"))),
                        }
                    ),
                    (
                        DuperKey(Cow::Borrowed("dimensions")),
                        DuperValue {
                            identifier: None,
                            inner: DuperInner::Tuple(DuperTuple(vec![
                                DuperValue {
                                    identifier: None,
                                    inner: DuperInner::Float(18.5)
                                },
                                DuperValue {
                                    identifier: None,
                                    inner: DuperInner::Float(15.2)
                                },
                                DuperValue {
                                    identifier: None,
                                    inner: DuperInner::Float(7.8)
                                },
                            ])),
                        }
                    ),
                    (
                        DuperKey(Cow::Borrowed("weight")),
                        DuperValue {
                            identifier: Some(DuperIdentifier(Cow::Borrowed("Weight"))),
                            inner: DuperInner::Float(0.285)
                        }
                    ),
                    (
                        DuperKey(Cow::Borrowed("in_stock")),
                        DuperValue {
                            identifier: None,
                            inner: DuperInner::Boolean(true),
                        }
                    ),
                    (
                        DuperKey(Cow::Borrowed("specifications")),
                        DuperValue {
                            identifier: None,
                            inner: DuperInner::Object(DuperObject(vec![
                                (
                                    DuperKey(Cow::Borrowed("battery_life")),
                                    DuperValue {
                                        identifier: Some(DuperIdentifier(Cow::Borrowed(
                                            "Duration"
                                        ))),
                                        inner: DuperInner::String(DuperString(Cow::Borrowed(
                                            "30h"
                                        ))),
                                    }
                                ),
                                (
                                    DuperKey(Cow::Borrowed("noise_cancellation")),
                                    DuperValue {
                                        identifier: None,
                                        inner: DuperInner::Boolean(true),
                                    }
                                ),
                                (
                                    DuperKey(Cow::Borrowed("connectivity")),
                                    DuperValue {
                                        identifier: None,
                                        inner: DuperInner::Array(DuperArray(vec![
                                            DuperValue {
                                                identifier: None,
                                                inner: DuperInner::String(DuperString(
                                                    Cow::Borrowed("Bluetooth 5.0")
                                                ))
                                            },
                                            DuperValue {
                                                identifier: None,
                                                inner: DuperInner::String(DuperString(
                                                    Cow::Borrowed("3.5mm Jack")
                                                ))
                                            },
                                        ])),
                                    }
                                ),
                            ])),
                        }
                    ),
                    (
                        DuperKey(Cow::Borrowed("image_thumbnail")),
                        DuperValue {
                            identifier: Some(DuperIdentifier(Cow::Borrowed("Png"))),
                            inner: DuperInner::Bytes(DuperBytes(Cow::Borrowed(
                                b"\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x64"
                            ))),
                        }
                    ),
                    (
                        DuperKey(Cow::Borrowed("tags")),
                        DuperValue {
                            identifier: None,
                            inner: DuperInner::Array(DuperArray(vec![
                                DuperValue {
                                    identifier: None,
                                    inner: DuperInner::String(DuperString(Cow::Borrowed(
                                        "electronics"
                                    )))
                                },
                                DuperValue {
                                    identifier: None,
                                    inner: DuperInner::String(DuperString(Cow::Borrowed("audio")))
                                },
                                DuperValue {
                                    identifier: None,
                                    inner: DuperInner::String(DuperString(Cow::Borrowed(
                                        "wireless"
                                    )))
                                },
                            ])),
                        }
                    ),
                    (
                        DuperKey(Cow::Borrowed("release_date")),
                        DuperValue {
                            identifier: Some(DuperIdentifier(Cow::Borrowed("Date"))),
                            inner: DuperInner::String(DuperString(Cow::Borrowed("2023-11-15"))),
                        }
                    ),
                    (
                        DuperKey(Cow::Borrowed("warranty_period")),
                        DuperValue {
                            identifier: None,
                            inner: DuperInner::Null,
                        }
                    ),
                    (
                        DuperKey(Cow::Borrowed("customer_ratings")),
                        DuperValue {
                            identifier: None,
                            inner: DuperInner::Object(DuperObject(vec![
                                (
                                    DuperKey(Cow::Borrowed("latest_review")),
                                    DuperValue {
                                        identifier: None,
                                        inner: DuperInner::String(DuperString(Cow::Borrowed(
                                            r#"Absolutely ""astounding""!! 😎"#
                                        ))),
                                    }
                                ),
                                (
                                    DuperKey(Cow::Borrowed("average")),
                                    DuperValue {
                                        identifier: None,
                                        inner: DuperInner::Float(4.5)
                                    }
                                ),
                                (
                                    DuperKey(Cow::Borrowed("count")),
                                    DuperValue {
                                        identifier: None,
                                        inner: DuperInner::Integer(127)
                                    }
                                ),
                            ])),
                        }
                    ),
                    (
                        DuperKey(Cow::Borrowed("created_at")),
                        DuperValue {
                            identifier: Some(DuperIdentifier(Cow::Borrowed("DateTime"))),
                            inner: DuperInner::String(DuperString(Cow::Borrowed(
                                "2023-11-17T21:50:43+00:00"
                            ))),
                        }
                    ),
                ])),
            }
        );
    }
}
