"""
Integration tests based on t4.py to verify code optimizations work correctly.

Tests real-world usage patterns including:
- SSL connections with custom certificates
- Debug mode
- Query execution and cursor operations
- Connection cleanup
- Error handling
- Concurrent operations
"""

import unittest
import os
import tempfile
from unittest.mock import Mock, patch, MagicMock, call
import grpc

from e6data_python_connector import Connection
from e6data_python_connector.constants import (
    MAX_RETRY_ATTEMPTS,
    RETRY_SLEEP_SECONDS,
    GRPC_ERROR_STRATEGY_MISMATCH,
    GRPC_ERROR_ACCESS_DENIED,
    GRPC_ERROR_SERVICE_UNAVAILABLE,
    STRATEGY_BLUE,
    STRATEGY_GREEN,
    VALID_STRATEGIES
)


class TestConnectionWithSSL(unittest.TestCase):
    """Test SSL connection functionality based on t4.py patterns."""

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_secure_connection_uses_ssl_credentials_utility(self, mock_stub, mock_secure_channel):
        """Test that secure connections use the get_ssl_credentials utility function."""
        # Create a temporary cert file
        cert_content = b"-----BEGIN CERTIFICATE-----\ntest\n-----END CERTIFICATE-----"

        with tempfile.NamedTemporaryFile(mode='wb', delete=False, suffix='.pem') as f:
            f.write(cert_content)
            cert_path = f.name

        try:
            # Create connection with SSL cert
            conn = Connection(
                host='test.example.com',
                port=443,
                username='test@example.com',
                password='test_token',
                database='test_db',
                catalog='test_catalog',
                secure=True,
                ssl_cert=cert_path,
                cluster_name='test-cluster'
            )

            # Verify secure_channel was called (which means get_ssl_credentials was used)
            self.assertTrue(mock_secure_channel.called)

            # Clean up
            conn.close()
        finally:
            os.unlink(cert_path)

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_secure_connection_with_bytes_cert(self, mock_stub, mock_secure_channel):
        """Test SSL connection with certificate as bytes."""
        cert_content = b"-----BEGIN CERTIFICATE-----\ntest\n-----END CERTIFICATE-----"

        # Create connection with cert as bytes
        conn = Connection(
            host='test.example.com',
            port=443,
            username='test@example.com',
            password='test_token',
            database='test_db',
            catalog='test_catalog',
            secure=True,
            ssl_cert=cert_content,
            cluster_name='test-cluster'
        )

        # Verify secure_channel was called
        self.assertTrue(mock_secure_channel.called)

        conn.close()


class TestDebugMode(unittest.TestCase):
    """Test debug mode functionality."""

    @patch('e6data_python_connector.e6data_grpc.grpc.insecure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    @patch('e6data_python_connector.e6data_grpc.logging.basicConfig')
    def test_debug_mode_enables_logging(self, mock_basicConfig, mock_stub, mock_channel):
        """Test that debug=True enables comprehensive logging."""
        conn = Connection(
            host='test.example.com',
            port=80,
            username='test@example.com',
            password='test_token',
            database='test_db',
            catalog='test_catalog',
            debug=True,
            cluster_name='test-cluster'
        )

        # Verify logging was configured
        mock_basicConfig.assert_called_once()
        call_kwargs = mock_basicConfig.call_args[1]
        self.assertEqual(call_kwargs['level'], 10)  # DEBUG level = 10

        conn.close()


class TestQueryExecution(unittest.TestCase):
    """Test query execution patterns from t4.py."""

    def setUp(self):
        """Set up mock connection and cursor."""
        self.mock_channel = Mock()
        self.mock_client = Mock()
        self.mock_auth_response = Mock()
        self.mock_auth_response.sessionId = 'test-session-123'
        self.mock_auth_response.new_strategy = None
        self.mock_client.authenticate.return_value = self.mock_auth_response

    @patch('e6data_python_connector.e6data_grpc.grpc.insecure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_query_execution_flow(self, mock_stub, mock_channel):
        """Test the complete query execution flow from t4.py."""
        # Mock responses
        mock_client = Mock()
        mock_stub.return_value = mock_client

        mock_auth_response = Mock()
        mock_auth_response.sessionId = 'test-session'
        mock_auth_response.new_strategy = None
        mock_client.authenticate.return_value = mock_auth_response

        mock_prepare_response = Mock()
        mock_prepare_response.queryId = 'query-123'
        mock_prepare_response.engineIP = '192.168.1.1'
        mock_prepare_response.new_strategy = None
        mock_client.prepareStatementV2.return_value = mock_prepare_response

        mock_execute_response = Mock()
        mock_execute_response.new_strategy = None
        mock_client.executeStatementV2.return_value = mock_execute_response

        # Create connection
        conn = Connection(
            host='test.example.com',
            port=80,
            username='test@example.com',
            password='test_token',
            database='test_db',
            catalog='test_catalog',
            cluster_name='test-cluster'
        )

        cursor = conn.cursor()

        # Verify cursor was created
        self.assertIsNotNone(cursor)

        # Clean up
        conn.close()


class TestErrorHandling(unittest.TestCase):
    """Test error handling with constants."""

    def test_access_denied_error_constant_exists(self):
        """Test that GRPC_ERROR_ACCESS_DENIED constant is defined and used."""
        # Verify the constant exists and has the right value
        self.assertEqual(GRPC_ERROR_ACCESS_DENIED, 'Access denied')

        # Verify it's used in the re_auth decorator
        from e6data_python_connector.e6data_grpc import re_auth
        import inspect

        source = inspect.getsource(re_auth)
        self.assertIn('GRPC_ERROR_ACCESS_DENIED', source)

    def test_strategy_mismatch_error_constant_exists(self):
        """Test that GRPC_ERROR_STRATEGY_MISMATCH constant is defined and used."""
        # Verify the constant exists and has the right value
        self.assertEqual(GRPC_ERROR_STRATEGY_MISMATCH, 'status: 456')

        # Verify it's used in the re_auth decorator
        from e6data_python_connector.e6data_grpc import re_auth
        import inspect

        source = inspect.getsource(re_auth)
        self.assertIn('GRPC_ERROR_STRATEGY_MISMATCH', source)


class TestConnectionCleanup(unittest.TestCase):
    """Test connection cleanup patterns from t4.py."""

    @patch('e6data_python_connector.e6data_grpc.grpc.insecure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_connection_close_cleanup(self, mock_stub, mock_channel):
        """Test that connection.close() properly cleans up resources."""
        mock_client = Mock()
        mock_stub.return_value = mock_client

        mock_auth_response = Mock()
        mock_auth_response.sessionId = 'test-session'
        mock_auth_response.new_strategy = None
        mock_client.authenticate.return_value = mock_auth_response

        conn = Connection(
            host='test.example.com',
            port=80,
            username='test@example.com',
            password='test_token',
            database='test_db',
            catalog='test_catalog',
            cluster_name='test-cluster'
        )

        # Close connection
        conn.close()

        # Verify channel is None after close
        self.assertIsNone(conn._channel)
        self.assertIsNone(conn._session_id)

    @patch('e6data_python_connector.e6data_grpc.grpc.insecure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_context_manager_cleanup(self, mock_stub, mock_channel):
        """Test that using connection as context manager cleans up properly."""
        mock_client = Mock()
        mock_stub.return_value = mock_client

        mock_auth_response = Mock()
        mock_auth_response.sessionId = 'test-session'
        mock_auth_response.new_strategy = None
        mock_client.authenticate.return_value = mock_auth_response

        with Connection(
            host='test.example.com',
            port=80,
            username='test@example.com',
            password='test_token',
            database='test_db',
            catalog='test_catalog',
            cluster_name='test-cluster'
        ) as conn:
            # Connection should be active
            self.assertIsNotNone(conn)

        # After exiting context, channel should be closed
        self.assertIsNone(conn._channel)


class TestStrategyConstants(unittest.TestCase):
    """Test that strategy constants are used correctly."""

    def test_valid_strategies_contains_blue_and_green(self):
        """Test that VALID_STRATEGIES contains both blue and green."""
        self.assertIn(STRATEGY_BLUE, VALID_STRATEGIES)
        self.assertIn(STRATEGY_GREEN, VALID_STRATEGIES)
        self.assertEqual(STRATEGY_BLUE, 'blue')
        self.assertEqual(STRATEGY_GREEN, 'green')

    def test_strategy_constants_are_lowercase(self):
        """Test that strategy constants are lowercase for comparison."""
        # This is important because the code normalizes strategies to lowercase
        self.assertEqual(STRATEGY_BLUE, STRATEGY_BLUE.lower())
        self.assertEqual(STRATEGY_GREEN, STRATEGY_GREEN.lower())


class TestGrpcOptions(unittest.TestCase):
    """Test gRPC options configuration from t4.py."""

    @patch('e6data_python_connector.e6data_grpc.grpc.insecure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_custom_grpc_options(self, mock_stub, mock_channel):
        """Test that custom gRPC options are applied."""
        custom_options = {
            'max_receive_message_length': 100 * 1024 * 1024,
            'keepalive_time_ms': 60000
        }

        conn = Connection(
            host='test.example.com',
            port=80,
            username='test@example.com',
            password='test_token',
            database='test_db',
            catalog='test_catalog',
            cluster_name='test-cluster',
            grpc_options=custom_options
        )

        # Verify options are set
        grpc_opts = conn._get_grpc_options
        self.assertIsInstance(grpc_opts, list)

        # Convert to dict for easier checking
        opts_dict = dict(grpc_opts)
        self.assertIn('grpc.max_receive_message_length', opts_dict)

        conn.close()


class TestAutoResume(unittest.TestCase):
    """Test auto_resume functionality from t4.py."""

    @patch('e6data_python_connector.e6data_grpc.grpc.insecure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_auto_resume_disabled(self, mock_stub, mock_channel):
        """Test that auto_resume can be disabled like in t4.py."""
        conn = Connection(
            host='test.example.com',
            port=80,
            username='test@example.com',
            password='test_token',
            database='test_db',
            catalog='test_catalog',
            cluster_name='test-cluster',
            auto_resume=False
        )

        # Verify auto_resume is disabled
        self.assertFalse(conn._auto_resume)

        conn.close()

    def test_service_unavailable_error_constant_exists(self):
        """Test that GRPC_ERROR_SERVICE_UNAVAILABLE constant is defined."""
        # Verify the constant exists and has the right value
        self.assertEqual(GRPC_ERROR_SERVICE_UNAVAILABLE, 'status: 503')

        # This constant is used for auto-resume logic
        from e6data_python_connector.e6data_grpc import Connection
        import inspect

        # Check that the constant is available in the module
        source = inspect.getsource(Connection)
        self.assertIn('503', source)  # The 503 error is mentioned in the code


class TestConcurrentPatterns(unittest.TestCase):
    """Test patterns for concurrent query execution from t4.py (commented code)."""

    @patch('e6data_python_connector.e6data_grpc.grpc.insecure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_multiple_connections_isolated(self, mock_stub, mock_channel):
        """Test that multiple connections can be created independently."""
        mock_client = Mock()
        mock_stub.return_value = mock_client

        # Create multiple connections
        conn1 = Connection(
            host='test.example.com',
            port=80,
            username='user1@example.com',
            password='token1',
            database='test_db',
            catalog='test_catalog',
            cluster_name='test-cluster'
        )

        conn2 = Connection(
            host='test.example.com',
            port=80,
            username='user2@example.com',
            password='token2',
            database='test_db',
            catalog='test_catalog',
            cluster_name='test-cluster'
        )

        # Each should be independent objects
        self.assertIsNot(conn1, conn2)
        self.assertNotEqual(conn1._Connection__username, conn2._Connection__username)

        # Clean up
        conn1.close()
        conn2.close()


class TestOptimizationsIntegration(unittest.TestCase):
    """Integration tests to verify all optimizations work together."""

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_all_optimizations_work_together(self, mock_stub, mock_channel):
        """Test that all optimizations (SSL, constants, no duplication) work together."""
        # Create cert file
        cert_content = b"-----BEGIN CERTIFICATE-----\ntest\n-----END CERTIFICATE-----"

        with tempfile.NamedTemporaryFile(mode='wb', delete=False, suffix='.pem') as f:
            f.write(cert_content)
            cert_path = f.name

        try:
            # This tests:
            # 1. get_ssl_credentials utility function (from common.py)
            # 2. Constants usage (STRATEGY_BLUE, etc.)
            # 3. Direct import of _get_grpc_header (no wrapper)
            # 4. Debug mode
            conn = Connection(
                host='test.example.com',
                port=443,
                username='test@example.com',
                password='test_token',
                database='test_db',
                catalog='test_catalog',
                cluster_name='test-cluster',
                secure=True,
                ssl_cert=cert_path,
                debug=True,
                auto_resume=False,
                grpc_options={
                    'max_receive_message_length': 100 * 1024 * 1024
                }
            )

            # Verify secure_channel was called (SSL credentials utility used)
            self.assertTrue(mock_channel.called)

            # Verify connection is valid
            self.assertIsNotNone(conn)
            self.assertTrue(conn._secure_channel)
            self.assertFalse(conn._auto_resume)
            self.assertTrue(conn._debug)

            conn.close()
        finally:
            os.unlink(cert_path)


if __name__ == '__main__':
    unittest.main()