from __future__ import annotations

import pyrogram
from pyrogram import enums, raw, types, utils

from .inline_query_result import InlineQueryResult


class InlineQueryResultAnimation(InlineQueryResult):
    """Link to an animated GIF file.

    By default, this animated GIF file will be sent by the user with optional caption.
    Alternatively, you can use *input_message_content* to send a message with the specified content instead of the
    animation.

    Parameters:
        animation_url (``str``):
            A valid URL for the animated GIF file.
            File size must not exceed 1 MB.

        animation_width (``int``, *optional*)
            Width of the animation.

        animation_height (``int``, *optional*)
            Height of the animation.

        animation_duration (``int``, *optional*)
            Duration of the animation in seconds.

        thumb_url (``str``, *optional*):
            URL of the static thumbnail for the result (jpeg or gif)
            Defaults to the value passed in *animation_url*.

        thumb_mime_type (``str``, *optional*)
            MIME type of the thumbnail, must be one of "image/jpeg", "image/gif", or "video/mp4".
            Defaults to "image/jpeg".

        id (``str``, *optional*):
            Unique identifier for this result, 1-64 bytes.
            Defaults to a randomly generated UUID4.

        title (``str``, *optional*):
            Title for the result.

        caption (``str``, *optional*):
            Caption of the animation to be sent, 0-1024 characters.

        parse_mode (:obj:`~pyrogram.enums.ParseMode`, *optional*):
            By default, texts are parsed using both Markdown and HTML styles.
            You can combine both syntaxes together.

        caption_entities (List of :obj:`~pyrogram.types.MessageEntity`):
            List of special entities that appear in the caption, which can be specified instead of *parse_mode*.

        reply_markup (:obj:`~pyrogram.types.InlineKeyboardMarkup`, *optional*):
            An InlineKeyboardMarkup object.

        input_message_content (:obj:`~pyrogram.types.InputMessageContent`):
            Content of the message to be sent instead of the photo.
    """

    def __init__(
        self,
        animation_url: str,
        animation_width: int = 0,
        animation_height: int = 0,
        animation_duration: int = 0,
        thumb_url: str | None = None,
        thumb_mime_type: str = "image/jpeg",
        id: str | None = None,
        title: str | None = None,
        description: str | None = None,
        caption: str = "",
        parse_mode: enums.ParseMode | None = None,
        caption_entities: list[types.MessageEntity] | None = None,
        reply_markup: types.InlineKeyboardMarkup = None,
        input_message_content: types.InputMessageContent = None,
    ) -> None:
        super().__init__("gif", id, input_message_content, reply_markup)

        self.animation_url = animation_url
        self.animation_width = animation_width
        self.animation_height = animation_height
        self.animation_duration = animation_duration
        self.thumb_url = thumb_url
        self.thumb_mime_type = thumb_mime_type
        self.title = title
        self.description = description
        self.caption = caption
        self.parse_mode = parse_mode
        self.caption_entities = caption_entities
        self.reply_markup = reply_markup
        self.input_message_content = input_message_content

    async def write(self, client: pyrogram.Client):
        animation = raw.types.InputWebDocument(
            url=self.animation_url,
            size=0,
            mime_type="image/gif",
            attributes=[
                raw.types.DocumentAttributeVideo(
                    w=self.animation_width,
                    h=self.animation_height,
                    duration=self.animation_duration,
                ),
            ],
        )

        if self.thumb_url is None:
            thumb = animation
        else:
            thumb = raw.types.InputWebDocument(
                url=self.thumb_url,
                size=0,
                mime_type=self.thumb_mime_type,
                attributes=[],
            )

        message, entities = (
            await utils.parse_text_entities(
                client,
                self.caption,
                self.parse_mode,
                self.caption_entities,
            )
        ).values()

        return raw.types.InputBotInlineResult(
            id=self.id,
            type=self.type,
            title=self.title,
            thumb=thumb,
            content=animation,
            send_message=(
                await self.input_message_content.write(client, self.reply_markup)
                if self.input_message_content
                else raw.types.InputBotInlineMessageMediaAuto(
                    reply_markup=await self.reply_markup.write(client)
                    if self.reply_markup
                    else None,
                    message=message,
                    entities=entities,
                )
            ),
        )
