"""
`build_log_parser_test`
=======================================================================
Unit tests for parsing of build logs generated by compilers
* Author(s): Jimmy Gomez
"""
import os
import json
import pytest
import unittest
import embedops_cli.eotools.parse
from embedops_cli.eotools.parse import (
    parse_storage_sizes,
    main as parse_main,
)
import embedops_cli.utilities
from embedops_cli.eotools.iar_parser.ewp import (
    IARMapFileNotFoundException,
    get_configuration_mapfile_path
)
from unittest import mock
from click.testing import CliRunner

file_dir = os.path.dirname(os.path.realpath(__file__))


def test_parsing_of_iar_build_log():
    """Test parsing of build log generated by IAR"""
    input_filename = file_dir + "/iar-files/componentA.ewp"
    storage_sizes = parse_storage_sizes(input_filename, "IAR", iar_config="Debug")
    print(storage_sizes)
    assert storage_sizes[0]["dimensions"]["build_target_name"] == "componentA"
    assert storage_sizes[0]["dimensions"]["build_target_group"] == "Debug"
    assert storage_sizes[0]["dimensions"]["build_target"] == "componentA - Debug"
    assert storage_sizes[0]["flash_code_size"] == 7636
    assert storage_sizes[0]["flash_data_size"] == 136
    assert storage_sizes[0]["ram_size"] == 1036


def test_parsing_ewp_file_exists_but_mapfile_doesnt_log():
    """Test parsing of build log generated by IAR"""
    input_filename = file_dir + "/iar-files/componentB.ewp"
    with pytest.raises(SystemExit) as pytest_wrapped_e:
        parse_storage_sizes(input_filename, "IAR", iar_config="Debug")
    assert pytest_wrapped_e.type == SystemExit
    assert pytest_wrapped_e.value.code == 1


def test_get_mapfile_path_from_ilinkoutfile_element():
    with pytest.raises(IARMapFileNotFoundException) as pytest_wrapped_e:
        get_configuration_mapfile_path(os.path.join(file_dir, 'iar-files', 'ilinkoutfile', 'componentA.ewp'), 'Debug')
    assert pytest_wrapped_e.type == IARMapFileNotFoundException
    assert "Debug/List/MyProjectHasAFunnyName.map" in pytest_wrapped_e.value.message


def test_parsing_of_gcc_build_log():
    """Test parsing of build log generated by GCC"""
    input_filename = file_dir + "/gcc_example_build.log"
    storage_sizes = parse_storage_sizes(input_filename, "GCC")
    assert storage_sizes[0]["flash_code_size"] == 36080
    assert storage_sizes[0]["flash_data_size"] == 664 + 50960
    assert storage_sizes[0]["ram_size"] == 50960
    assert storage_sizes[0]["dimensions"]["build_target_name"] == "application.elf"

    input_filename = file_dir + "/gcc_example_build_2.log"
    storage_sizes = parse_storage_sizes(input_filename, "GCC")
    assert storage_sizes[0]["flash_code_size"] == 4634921
    assert storage_sizes[0]["flash_data_size"] == 41808 + 2631216
    assert storage_sizes[0]["ram_size"] == 2631216
    assert storage_sizes[0]["dimensions"]["build_target_name"] == "fastfare_app"


def test_parsing_of_iar_multiple_map_files():
    """Test parsing of build log generated by IAR"""
    input_filename = file_dir + "/iar-files/multiple-configurations/OttoGUIMicrocontroller.ewp"
    assert os.path.exists(input_filename)
    storage_sizes = parse_storage_sizes(input_filename, "IAR", iar_config="Debug")
    print(json.dumps(storage_sizes, indent=2))
    assert len(storage_sizes) == 1
    assert storage_sizes[0]["dimensions"]["build_target_name"] == "OttoGUIMicrocontroller"
    assert storage_sizes[0]["dimensions"]["build_target_group"] == "Debug"
    assert storage_sizes[0]["flash_code_size"] == 337642
    assert storage_sizes[0]["flash_data_size"] == 31974173
    assert storage_sizes[0]["ram_size"] == 7394959

    storage_sizes = parse_storage_sizes(input_filename, "IAR", iar_config="Production")
    print(json.dumps(storage_sizes, indent=2))
    assert len(storage_sizes) == 1
    assert storage_sizes[0]["dimensions"]["build_target_name"] == "OttoGUIMicrocontroller"
    assert storage_sizes[0]["dimensions"]["build_target_group"] == "Production"
    assert storage_sizes[0]["flash_code_size"] == 334656
    assert storage_sizes[0]["flash_data_size"] == 31967050
    assert storage_sizes[0]["ram_size"] == 7394961

    storage_sizes = parse_storage_sizes(input_filename, "IAR", iar_config="Production (Test)")
    print(json.dumps(storage_sizes, indent=2))
    assert len(storage_sizes) == 1
    assert storage_sizes[0]["dimensions"]["build_target_name"] == "OttoGUIMicrocontroller"
    assert storage_sizes[0]["dimensions"]["build_target_group"] == "Production (Test)"
    assert storage_sizes[0]["flash_code_size"] == 340092
    assert storage_sizes[0]["flash_data_size"] == 31967039
    assert storage_sizes[0]["ram_size"] == 7394961


class test_exit_main_if_unsupported_compiler(unittest.TestCase):
    def setUp(self):
        self.runner = CliRunner()

    @mock.patch.dict(os.environ, {"EMBEDOPS_COMPILER": "FOO"})
    def test_command(self):
        result = self.runner.invoke(parse_main, ["-d"])
        self.assertEqual(result.exit_code, 1)
        self.assertEqual(result.output.strip(), "EMBEDOPS_COMPILER not set")


class test_exit_zero_if_supported_compiler_with_build_log_no_debug_flag(unittest.TestCase):
    def setUp(self):
        self.runner = CliRunner()

    @mock.patch.dict('os.environ', {"EMBEDOPS_COMPILER": "GCC"})
    @mock.patch.object(embedops_cli.eotools.parse, 'settings', create=True)
    def test_command(self, mocker):
        setattr(embedops_cli.eotools.parse.settings, 'input_file', f"{file_dir}/gcc_example_build.log")
        setattr(embedops_cli.eotools.parse.settings, 'metrics_endpoint', "https://app.embedops.io/metrics")
        delattr(embedops_cli.eotools.parse.settings, 'api_repo_key')
        result = self.runner.invoke(parse_main)
        self.assertEqual(result.exit_code, 0)


def test_exit_parse_storage_sizes_if_unsupported_compiler():
    input_filename = file_dir + "/gcc_example_build.log"
    with pytest.raises(SystemExit) as pytest_wrapped_e:
        parse_storage_sizes(input_filename, "FOO")
    assert pytest_wrapped_e.type == SystemExit
    assert pytest_wrapped_e.value.code == 1


def test_exit_parse_storage_sizes_if_logfile_dne():
    input_filename = file_dir + "/does_not_exist.log"
    with pytest.raises(SystemExit) as pytest_wrapped_e:
        parse_storage_sizes(input_filename, "GCC")
    assert pytest_wrapped_e.type == SystemExit
    assert pytest_wrapped_e.value.code == 1


def test_exit_parse_storage_sizes_if_ewp_file_dne():
    input_filename = file_dir + "/does_not_exist.ewp"
    with pytest.raises(SystemExit) as pytest_wrapped_e:
        parse_storage_sizes(input_filename, "IAR")
        assert pytest_wrapped_e.value.code == 1
