"""Common functions used by the HIL commands"""
import sys
import json
import os.path
import tempfile
from pathlib import Path
import traceback
import subprocess
from datetime import datetime
from os import getcwd
from typing import Optional
import click
from dynaconf import Dynaconf
from embedops_cli.version import __version__ as cli_version
from embedops_cli.hil.hil_types import get_hil_config_path
from embedops_cli.sse import eo_sse
from embedops_cli import config
from embedops_cli.api.rest import ApiException
from embedops_cli.eo_types import (
    EmbedOpsException,
    NoAvailableHilDevice,
    NoRepoIdException,
    NoCIRunIdException,
    NetworkException,
    UnauthorizedUserException,
    UnknownShellException,
)
from embedops_cli.hil.hil_types import (
    NoHILRootPathException,
    HILPackageCreationException,
    NoHILArtifactsPathException,
    HILFilesInvalidException,
    HILExternalModulesPathInvalidException,
    NoHILResultsPathException,
)
from embedops_cli.hil.hil_package import create_hil_package
from embedops_cli import embedops_authorization
from embedops_cli.sse.sse_api import SSEApi
from embedops_cli.utilities import (
    echo_error_and_fix,
    get_client as util_get_client,
)
from embedops_cli.utilities import logging_setup
from embedops_cli.hil import hil_package
from embedops_cli.api.models import HilRunCreateProps

HIL_GATEWAY_IMAGE_NAME = "embedops-gateway-image.img"
HIL_EXTERNAL_MODULES_MAX_SIZE = (
    5 * 1024 * 1024
)  # 5MB max limit to external modules folder

_logger = logging_setup(__name__)


def get_hil_root_path():
    """get hil_root_path from .embedops/hil/config.yml

    Returns:
        str: the value stored in <repo_root>/.embedops/hil/config.yml:hil_root_path or None
    """
    dot_eo = Dynaconf(
        load_dotenv=False,
        settings_files=[get_hil_config_path()],
        silent_errors=True,
    )
    dot_eo.configure(
        LOADERS_FOR_DYNACONF=[
            "dynaconf.loaders.yaml_loader",
        ]
    )
    return dot_eo.get("hil_root_path")


def get_hil_artifacts_path():
    """get hil_artifacts from .embedops/hil/config.yml

    Returns:
        str: the value stored in <repo_root>/.embedops/hil/config.yml:hil_artifacts or None
    """
    dot_eo = Dynaconf(
        load_dotenv=False,
        settings_files=[get_hil_config_path()],
        silent_errors=True,
    )
    dot_eo.configure(
        LOADERS_FOR_DYNACONF=[
            "dynaconf.loaders.yaml_loader",
        ]
    )
    return dot_eo.get("hil_artifacts")


def get_hil_external_modules_path():
    """get hil_external_modules from .embedops/hil/config.yml

    Returns:
        str: the value stored in <repo_root>/.embedops/hil/config.yml:hil_external_modules or None
    """
    dot_eo = Dynaconf(
        load_dotenv=False,
        settings_files=[get_hil_config_path()],
        silent_errors=True,
    )
    dot_eo.configure(
        LOADERS_FOR_DYNACONF=[
            "dynaconf.loaders.yaml_loader",
        ]
    )
    return dot_eo.get("hil_external_modules")


def get_hil_results_base_path():
    """get hil_results_base from .embedops/hil/config.yml

    Returns:
        str: the value stored in <repo_root>/.embedops/hil/config.yml:hil_results_base or None
    """
    dot_eo = Dynaconf(
        load_dotenv=False,
        settings_files=[get_hil_config_path()],
        silent_errors=True,
    )
    dot_eo.configure(
        LOADERS_FOR_DYNACONF=[
            "dynaconf.loaders.yaml_loader",
        ]
    )
    return dot_eo.get("hil_results_base")


def get_hil_results_paths():

    """
    Return both the base path for results,
    and a specific folder for this exact time for results to go.
    """

    base_path = get_hil_results_base_path()

    if base_path is None:
        base_path = os.getcwd()  # Use the repo root as a default

    now = datetime.now()
    hil_results_name = now.strftime("hil_results_%Y_%m_%d_%HH_%MM_%SS")
    hil_results_path = os.path.join(base_path, hil_results_name)

    return base_path, hil_results_path


def get_hil_artifacts_path_from_ci_artifacts_dir(
    hil_artifacts_path: str,
) -> Optional[str]:
    """get hil_artifacts from ./artifacts directory generated by CI

    Returns:
        str?: path of file in artifacts directory that matches filename
            of hil_artifacts_path or None
    """
    artifacts_dir = os.path.join(os.getcwd(), "artifacts")
    file_name = os.path.basename(hil_artifacts_path)

    for root, _, files in os.walk(artifacts_dir):
        if file_name in files:
            absolute_path = os.path.join(root, file_name)
            relative_path = os.path.relpath(absolute_path)
            return relative_path
    return None


def hil_get_git_short_sha():
    """Helper function to get the git short sha"""
    cmd = ["git", "rev-parse", "--short", "HEAD"]

    try:
        git_hash_run = subprocess.run(
            cmd,
            capture_output=True,
            text=True,
            check=False,
        )
    except subprocess.CalledProcessError:
        _logger.debug(f"cmd {cmd} caused an exception")

        echo_error_and_fix(UnknownShellException(cmd))
        sys.exit(1)

    if git_hash_run.returncode != 0:
        git_hash = "Not Available"
    else:
        git_hash = git_hash_run.stdout.strip()

    return git_hash


def hil_echo_run_banner(ex_pkg_id: str, sdk_path: str) -> None:
    """print the embedops job banner to the console (local and pipeline)

    Args:
        ex_pkg_id (str): repo_id or ci_run_id
        sdk_path (str): path to hil sdk from the repo root.
                        set in .embedops/hil/config.yml:hil_root_path
    """

    git_hash = hil_get_git_short_sha()

    current_time = datetime.utcnow().strftime("%Y-%m-%d %H:%M:%S UTC")
    click.secho("-" * 80, fg="magenta")
    click.secho("Running job      'hil'", err=False, fg="magenta")
    click.secho(f"-> cli version   '{cli_version}'", err=False, fg="white")
    click.secho(f"-> repo id       '{ex_pkg_id}'", err=False, fg="white")
    click.secho(f"-> directory     '{getcwd()}'", err=False, fg="white")
    click.secho(f"-> hil sdk path  '{sdk_path}'", err=False, fg="white")
    click.secho(f"-> git sha       '{git_hash}'", err=False, fg="white")
    click.secho(f"-> timestamp     '{current_time}'", err=False, fg="white")
    click.secho("-" * 80, fg="magenta")
    click.secho("\n")


def hil_run(  # pylint: disable=R0912,R0914,R0915
    verbosity: str = "info", local: bool = False, devices=None
) -> int:
    """
    Run hil in either local or CI mode,
    using the current repository as a source.
    """

    try:
        ex_pkg_id = config.settings.run_id
        if not ex_pkg_id and not local:
            echo_error_and_fix(NoCIRunIdException())
        if local:
            ex_pkg_id = config.get_repo_id()
            if not ex_pkg_id:
                echo_error_and_fix(NoRepoIdException())

        hil_root_path = get_hil_root_path()
        if not hil_root_path:
            echo_error_and_fix(NoHILRootPathException())

        # Clear the current results directory, if it exists
        hil_results_base, hil_results_dir = get_hil_results_paths()
        if not os.path.isdir(hil_results_base):
            echo_error_and_fix(NoHILResultsPathException(hil_results_base))

        hil_artifacts = get_hil_artifacts_path()

        if not local and hil_artifacts:
            hil_artifacts = get_hil_artifacts_path_from_ci_artifacts_dir(hil_artifacts)

        if not hil_artifacts or not os.path.isfile(
            os.path.join(os.path.curdir, hil_artifacts)
        ):
            echo_error_and_fix(NoHILArtifactsPathException())

        hil_sdk_full_path = os.path.join(os.path.curdir, hil_root_path)
        if not os.path.isdir(hil_sdk_full_path):
            echo_error_and_fix(NoHILRootPathException())

        _logger.debug(f"using host: {config.settings.host}")
        hil_echo_run_banner(ex_pkg_id, hil_sdk_full_path)

        validate_hil_root(hil_root_path)

        # Validate external modules directory, if it exists
        external_modules_path = get_hil_external_modules_path()
        if external_modules_path is not None:
            validate_external_modules_path(external_modules_path)

        # Compile the package in a temporary folder that is deleted after uploading
        with tempfile.TemporaryDirectory() as tmp_dir:
            hil_ep_file = os.path.join(tmp_dir, "hil_ep.zip")
            # TODO: Manifest data is being left blank intentionally
            if not create_hil_package(
                [os.path.join(os.path.curdir, hil_artifacts)],
                hil_root_path,
                external_modules_path,
                {},
                hil_ep_file,
            ):
                echo_error_and_fix(HILPackageCreationException())

            # Get the upload URL
            _logger.debug(
                f"[local={local}] getting presigned url for hil execution package upload"
            )
            if local:
                api_client = embedops_authorization.get_user_client()
                upload_url_response = api_client.get_pre_signed_url_for_upload(
                    ex_pkg_id
                )
            else:
                api_client = util_get_client()
                upload_url_response = api_client.get_pre_signed_url_for_upload_0(
                    ex_pkg_id
                )
            upload_url = upload_url_response.url
            try:
                upload_status = hil_package.upload_hil_package(hil_ep_file, upload_url)
                if upload_status != 200:
                    echo_error_and_fix(
                        NetworkException(
                            upload_status,
                            # message="network exception during execution package upload",
                        )
                    )
            except ApiException:
                echo_error_and_fix(
                    EmbedOpsException(
                        # fix_message=f"Uploading HIL execution package failed: {exc.body.decode()}"
                    )
                )

        if local:
            # After package is uploaded, call into SSE and print any further events
            sse_api = SSEApi()
            return_code = 2

            for event in sse_api.sse_hil_run(ex_pkg_id, verbosity, devices):
                if event.event == eo_sse.SSE_TEXT_EVENT:
                    eo_sse.sse_print_command_text(event)
                elif event.event == eo_sse.SSE_RESULT_EVENT:
                    result_event_obj = json.loads(event.data)
                    return_code = result_event_obj["exitCode"]
                    break
                elif event.event == eo_sse.SSE_FILE_EVENT:
                    eo_sse.sse_process_file_event(event, hil_results_dir)
                else:
                    _logger.debug(f"got unhandled SSE Event: {event.event}")

            # At end of run, check if results were generated
            if os.path.isdir(hil_results_dir):
                click.secho(f"Results were downloaded to {hil_results_dir}")

            # If the command hasn't returned anything yet, exit here
            return return_code

        print("starting hil run...", file=sys.stderr)
        try:
            response = api_client.init_hil_run_from_ci(
                body=HilRunCreateProps(ci_run_id=ex_pkg_id)
            )
            print(f"HIL run started successfully. View progress here: {response.url}")
            return 0
        except ApiException as exc:
            if exc.status == 424:  #
                echo_error_and_fix(
                    NoAvailableHilDevice(
                        # f"{exc.body.decode()}"
                    )
                )
            raise exc

    except (
        NoRepoIdException,
        NoHILRootPathException,
        HILPackageCreationException,
        NetworkException,
        UnauthorizedUserException,
        EmbedOpsException,
        NoAvailableHilDevice,
        NoHILArtifactsPathException,
        HILFilesInvalidException,
    ) as exc:
        echo_error_and_fix(exc)
        return 2
    except ApiException as exc:
        echo_error_and_fix(
            EmbedOpsException(
                # TODO: Fix how custom messages are handled and tested
                # fix_message=f"Uploading HIL execution package failed: {exc.body.decode()}"
            )
        )
    except Exception:  # pylint: disable=broad-except
        traceback.print_exc()
    return -1


def validate_hil_root(hil_root_path: str) -> bool:
    """Inspect a HIL root path, ensuring that it has a non-zero number of python files,
    and those python files compile"""

    non_sdk_py_files = []

    if not os.path.exists(hil_root_path):
        echo_error_and_fix(
            HILFilesInvalidException(additional_message="HIL Root path does not exist.")
        )

    for path, _, files in os.walk(hil_root_path):
        if "hil_sdk" not in path:
            py_files = [f for f in files if f.endswith(".py")]
            for py_file in py_files:
                non_sdk_py_files.append(os.path.join(path, py_file))

    if len(non_sdk_py_files) == 0:
        echo_error_and_fix(
            HILFilesInvalidException(
                additional_message=f"No valid Python files found under {hil_root_path}."
            )
        )

    for py_file in non_sdk_py_files:
        try:
            with open(py_file, "r", encoding="utf-8") as source_file:
                source = source_file.read() + "\n"
                compile(source, py_file, "exec")
        except (SyntaxError, ValueError) as err:
            echo_error_and_fix(HILFilesInvalidException(additional_message=str(err)))


def validate_external_modules_path(external_modules_path: str):

    """Ensure that the external modules path exists, contains at least one .py file,
    and that its size is not over the maximum allowed"""

    if not os.path.exists(external_modules_path):
        echo_error_and_fix(
            HILExternalModulesPathInvalidException("The path does not exist.")
        )

    if not os.path.isdir(external_modules_path):
        echo_error_and_fix(
            HILExternalModulesPathInvalidException("The path must be a directory.")
        )

    py_files = list(Path(external_modules_path).glob("*.py"))
    if len(py_files) == 0:
        echo_error_and_fix(
            HILExternalModulesPathInvalidException(
                "The path does not contain any Python files."
            )
        )

    total_size = sum(
        file.stat().st_size for file in Path(external_modules_path).rglob("*")
    )
    if total_size > HIL_EXTERNAL_MODULES_MAX_SIZE:
        echo_error_and_fix(
            HILExternalModulesPathInvalidException(
                f"The path is too large ({total_size} total bytes)."
            )
        )


def validate_verbosity(ctx, param, value):

    # pylint: disable=unused-argument

    """
    Validate the value passed in for verbosity
    """

    if value in ["error", "warning", "info", "debug"]:
        return value

    raise click.BadParameter("Verbosity must be one of [error, warning, info, debug]")
