from .encapsulation import start
from .validator.validator import validate
from .utils.input_provider import StaticInputProvider

import os
import json
import logging
import argparse
from itertools import accumulate

"""
Main file of algorithm encapsulation package.
Do not edit this file if you don't know what you are doing.
Do not integrate your own algorithm into this,
 instead, use a separate python file and define it in manifest.json.

For development, use: `cd /src ; py ./main.py -i cli --patient-id 123 --case-id 456 --file-context somejobcontext --job-reference "abc#a234" --skip-reports --skip-download-upload`
"""

required_name = "__main__"
logging.basicConfig()
logger = logging.getLogger()
logger.setLevel(logging.DEBUG)

parser = argparse.ArgumentParser(prog = "Algorithm Encapsulation",
    description="Executes underlying algorithm",
    epilog="By Jurjen Verbruggen")
parser.add_argument("-i", "--input", 
                    choices=['environment', 'cli'], 
                    default='environment')
parser.add_argument("--skip-reports", action="store_true")
parser.add_argument("--skip-download-upload", action="store_true")
cli_input_options = parser.add_argument_group('CLI input options')
cli_input_options.add_argument("--mode", choices=['run', 'validate'], default='run')
cli_input_options.add_argument("--patient-id")
cli_input_options.add_argument("--case-id")
cli_input_options.add_argument("--file-context")
cli_input_options.add_argument("--input-contexts")
cli_input_options.add_argument("--job-reference")

def parse_input():
    args = parser.parse_args()
    input = args.input

    if input == "environment":
        logger.info("Using Environment variables for input")
        keys = ["EP_MODE", "EP_INSTRUCTION_PATIENT_ID", "EP_INSTRUCTION_CASE_ID", "EP_INSTRUCTION_SUB_JOB_REFERENCE", "EP_INSTRUCTION_INPUT_CONTEXT"]

        all_keys_available = accumulate(keys, lambda total, element: total and (element in os.environ), initial=True)
        if not all_keys_available:
            raise Exception("Missing environment keys!")

        mode=os.environ["EP_MODE"]
        if mode == "VALIDATE":
            return None

        input_contexts = str(os.environ["EP_INSTRUCTION_INPUT_CONTEXT"]).replace("'", "\"")

        input_provider = StaticInputProvider(
            mode=os.environ["EP_MODE"],
            input_type="environment",
            patient_id=os.environ["EP_INSTRUCTION_PATIENT_ID"],
            case_id=os.environ["EP_INSTRUCTION_CASE_ID"],
            reference=os.environ["EP_INSTRUCTION_SUB_JOB_REFERENCE"],
            input_resources_context=json.loads(input_contexts),
            skip_reports=False,
            skip_download_upload=False,
            development_mode="EP_ENVIRONMENT" not in os.environ
        )
    elif input == "cli":
        logger.info("Using CLI for input")

        if args.mode is None or args.patient_id is None \
                or args.case_id is None or args.job_reference is None or args.file_context is None \
                or args.input_contexts is None is None:
            raise KeyError("Missing configuration for CLI mode, [mode, patient_id, case_id, file_context, job-reference, input-contexts]")

        mode=str.upper(args.mode)
        if mode == "VALIDATE":
            return None

        input_contexts = str(args.input_contexts).replace("'", "\"")

        input_provider = StaticInputProvider(
            mode=mode, 
            input_type="cli",
            patient_id=args.patient_id, 
            case_id=args.case_id,
            reference=args.job_reference,
            input_resources_context=json.loads(input_contexts),
            skip_reports=args.skip_reports,
            skip_download_upload=args.skip_download_upload,
            development_mode=True)
    else:
        raise KeyError("Input type not known")
    
    logger.info(f"InputProvider: {input_provider}")

    return input_provider
    
def main():
    input_provider = parse_input()
    if input_provider is None:
        mode = "VALIDATE"
    else:
        mode = input_provider.get_mode()

    if mode == "RUN":
        logger.info("Running in normal mode!")
        start(logger, input_provider)
    elif mode == "VALIDATE":
        logger.info("Running in validation mode!")
        validate()
    else:
        raise KeyError("EP_MODE not in environment!")
