"""Multi-signature related classes implemented according to `EIP-11 <https://github.com/ergoplatform/eips/blob/9906f403f61d8592ab955494cc4310563f4067c6/eip-0011.md>`_

In Ergo multi-signature requires two stages: the commit stage and signing

Commit Stage
_____________
Signers generate commitments (randomly generated), and share it with other signers until enough commitments are gathered to meet the required number of signers.
When using :py:meth:`ergo_lib_python.wallet.Wallet.generate_commitments` two hints are generated for each input:

- OwnCommitment, which includes private randomness (should not be shared with other signers)
- :py:class:`RealCommitment` which includes the public image of randomness (r * G), where G is the base point of the elliptic curve group

Once enough commitments are gathered (t commitments for a t-of-n signature), signing can begin

Signing Stage
---------------
If the signature is a threshold signature (t-of-n), then for signers who are not present :py:class:`SimulatedCommitment`'s are generated.
Each signer now signs the message/transaction and produces a :py:class:`RealSecretProof`, and :py:class:`SimulatedSecretProof` for signers who are not present
"""
import typing
from typing import Any, Self
from ergo_lib_python.sigma_protocol import ProveDlog
from ergo_lib_python.transaction import Transaction
from ergo_lib_python.chain import ErgoBox, ErgoStateContext
__all__ = ['HintsBag', 'TransactionHintsBag', 'RealCommitment', 'SimulatedCommitment', 'RealSecretProof', 'SimulatedSecretProof', 'extract_hints']
@typing.final
class RealCommitment:
    @property
    def image(self) -> Any:
        """Public image of secret"""
        ...
    @property
    def commitment(self) -> Any:
        """Return public image of commitment generated by signer"""
        ...
    @property
    def position(self) -> list[int]:
        """Position in tree of public image"""
        ...
@typing.final
class SimulatedCommitment:
    @property
    def image(self) -> Any:
        """Public image of secret"""
        ...
    @property
    def commitment(self) -> Any:
        """Return public image of commitment generated by signer"""
        ...
    @property
    def position(self) -> list[int]:
        """Position in tree of public image"""
        ...
@typing.final
class RealSecretProof:
    @property
    def image(self) -> Any:
        """Public image of secret"""
        ...
    @property
    def challenge(self) -> bytes:
        """Challenge used for a proof"""
        ...
    @property
    def position(self) -> list[int]:
        """Position in tree of public image"""
        ...

@typing.final
class SimulatedSecretProof:
    @property
    def image(self) -> Any:
        """Public image of secret"""
        ...
    @property
    def challenge(self) -> bytes:
        """Challenge used for a proof"""
        ...
    @property
    def position(self) -> list[int]:
        """Position in tree of public image"""
        ...

@typing.final
class HintsBag:
    """Collection of hints to be used by a prover. This also includes private hints (OwnCommitment) that should not be shared, see :py:meth:`HintsBag.json`"""
    def __new__(cls):
        ...
    def add_commitment(self, commitment: RealCommitment | SimulatedCommitment):
        """Add a commitment to bag"""
        ...
    def add_proof(self, proof: RealSecretProof | SimulatedSecretProof):
        """Add a proof (signature) to bag"""
        ...
    @property
    def real_commitments(self) -> list[RealCommitment]:
        """Return list of real commitments in bag"""
    @property
    def simulated_commitments(self) -> list[RealCommitment]:
        """Return list of real commitments in bag"""
    @property
    def real_proofs(self) -> list[RealSecretProof]:
        """Return list of real proofs in bag"""
    @property
    def simulated_proofs(self) -> list[SimulatedSecretProof]:
        """Return list of simulated proofs in bag"""

    def without_secrets(self) -> Self:
        """Create new hints bag with private data removed. This is suitable for sending to other signers"""
        ...
    def json(self) -> str:
        """Return JSON with private data removed. See :py:meth:`private_json` which includes private data"""
    def private_json(self) -> str:
     """Return all hints in bag (including OwnCommitment, which should be kept private). This is suitable for storing hints but should not be shared,
     as sharing private data + a valid signature would lead to private key leakage"""
     ...

@typing.final
class TransactionHintsBag:
    """Represents hints for all inputs of a Transaction"""
    @classmethod
    def from_json(cls, s: str) -> Self:
        """Parse TransactionHintsBag from JSON"""
        ...
    def add_hints_for_input(self, index: int, hints_bag: HintsBag):
        """Add hints for input at `index`

           :param index: index of input
           :param hints_bag: Hints for input
        """
        ...
    def all_hints_for_input(self, index: int) -> HintsBag:
        """Return all hints for input, including private hints (if present)"""
        ...
    def json(self) -> str:
        """Convert to JSON. By default this will also include private hints that should not be shared publicly"""
        ...

def extract_hints(tx: Transaction,
    boxes_to_spend: list[ErgoBox],
    data_boxes: list[ErgoBox],
    state_context: ErgoStateContext,
    real_propositions: list[ProveDlog],
    simulated_propositions: list[ProveDlog]) -> TransactionHintsBag:
    """Extract hints (commitments and signatures) from `tx`

       :param tx: Partially signed transaction to extract commitments from
       :param boxes_to_spend: boxes being spent by tx
       :param data_boxes: boxes for data inputs
       :param state_context: current blockchain context
       :param real_propositions: list of propositions that are not simulated (real signature, commitment)
       :param simulated_propositions: list of propositions that are simulated
    """
    ...
