from devtools import utils
from devtools.config import as_config
from devtools.exc import InvalidField
from devtools.utils.exc import panic

from . import drivers
from .typedef import Driver


@as_config
class DatabaseConfig:
    driver: Driver
    host: str
    port: int = -1
    user: str = ""
    password: str = ""
    name: str = ""
    pool_size: int = 20
    pool_recycle: int = 3600
    max_overflow: int = 0
    autotransaction: bool = False

    @utils.lazyfield
    def effective_port(self) -> int:
        return self.port if self.port != -1 else self.dialect.default_port

    @utils.lazyfield
    def dialect(self) -> drivers.Dialect:
        if self.driver is Driver.CUSTOM and not self.dialect_overriden:
            raise panic(
                InvalidField,
                'Field driver of value "custom" missing custom dialect,'
                "dont forget to call .override_dialect(dialect)",
            )
        return drivers.resolve_driver(self.driver)

    @utils.lazyfield
    def dialect_overriden(self):
        return False

    def override_dialect(self, dialect: drivers.Dialect) -> None:
        DatabaseConfig.dialect.manual_set(self, dialect)
        DatabaseConfig.dialect_overriden.manual_set(self, True)