// Copyright (c) 2023, INRAE.
// Distributed under the terms of the GPL-3 Licence.
// The full licence is in the file LICENCE, distributed with this software.

#include <unordered_map>
#include <gtest/gtest.h>

#include <xtensor/xtensor.hpp>
#include <xtensor/xrandom.hpp>

#include "detail/uncertainty.hpp"

TEST(UncertaintyTests, TestBootstrapGenerator)
{
    // three hydrological years (2012, 2013, 2014)
    std::vector<std::string> datetimes = {
        "2011-10-01 00:00:00", "2011-10-02 00:00:00", "2011-10-03 00:00:00", "2011-10-04 00:00:00", "2011-10-05 00:00:00", "2011-10-06 00:00:00", "2011-10-07 00:00:00", "2011-10-08 00:00:00", "2011-10-09 00:00:00", "2011-10-10 00:00:00", "2011-10-11 00:00:00", "2011-10-12 00:00:00", "2011-10-13 00:00:00", "2011-10-14 00:00:00", "2011-10-15 00:00:00", "2011-10-16 00:00:00", "2011-10-17 00:00:00", "2011-10-18 00:00:00", "2011-10-19 00:00:00", "2011-10-20 00:00:00", "2011-10-21 00:00:00", "2011-10-22 00:00:00", "2011-10-23 00:00:00", "2011-10-24 00:00:00", "2011-10-25 00:00:00", "2011-10-26 00:00:00", "2011-10-27 00:00:00", "2011-10-28 00:00:00", "2011-10-29 00:00:00", "2011-10-30 00:00:00", "2011-10-31 00:00:00", "2011-11-01 00:00:00", "2011-11-02 00:00:00", "2011-11-03 00:00:00", "2011-11-04 00:00:00", "2011-11-05 00:00:00", "2011-11-06 00:00:00", "2011-11-07 00:00:00", "2011-11-08 00:00:00", "2011-11-09 00:00:00", "2011-11-10 00:00:00", "2011-11-11 00:00:00", "2011-11-12 00:00:00", "2011-11-13 00:00:00", "2011-11-14 00:00:00", "2011-11-15 00:00:00", "2011-11-16 00:00:00", "2011-11-17 00:00:00", "2011-11-18 00:00:00", "2011-11-19 00:00:00", "2011-11-20 00:00:00", "2011-11-21 00:00:00", "2011-11-22 00:00:00", "2011-11-23 00:00:00", "2011-11-24 00:00:00", "2011-11-25 00:00:00", "2011-11-26 00:00:00", "2011-11-27 00:00:00", "2011-11-28 00:00:00", "2011-11-29 00:00:00", "2011-11-30 00:00:00", "2011-12-01 00:00:00", "2011-12-02 00:00:00", "2011-12-03 00:00:00", "2011-12-04 00:00:00", "2011-12-05 00:00:00", "2011-12-06 00:00:00", "2011-12-07 00:00:00", "2011-12-08 00:00:00", "2011-12-09 00:00:00", "2011-12-10 00:00:00", "2011-12-11 00:00:00", "2011-12-12 00:00:00", "2011-12-13 00:00:00", "2011-12-14 00:00:00", "2011-12-15 00:00:00", "2011-12-16 00:00:00", "2011-12-17 00:00:00", "2011-12-18 00:00:00", "2011-12-19 00:00:00", "2011-12-20 00:00:00", "2011-12-21 00:00:00", "2011-12-22 00:00:00", "2011-12-23 00:00:00", "2011-12-24 00:00:00", "2011-12-25 00:00:00", "2011-12-26 00:00:00", "2011-12-27 00:00:00", "2011-12-28 00:00:00", "2011-12-29 00:00:00", "2011-12-30 00:00:00", "2011-12-31 00:00:00", "2012-01-01 00:00:00", "2012-01-02 00:00:00", "2012-01-03 00:00:00", "2012-01-04 00:00:00", "2012-01-05 00:00:00", "2012-01-06 00:00:00", "2012-01-07 00:00:00", "2012-01-08 00:00:00", "2012-01-09 00:00:00", "2012-01-10 00:00:00", "2012-01-11 00:00:00", "2012-01-12 00:00:00", "2012-01-13 00:00:00", "2012-01-14 00:00:00", "2012-01-15 00:00:00", "2012-01-16 00:00:00", "2012-01-17 00:00:00", "2012-01-18 00:00:00", "2012-01-19 00:00:00", "2012-01-20 00:00:00", "2012-01-21 00:00:00", "2012-01-22 00:00:00", "2012-01-23 00:00:00", "2012-01-24 00:00:00", "2012-01-25 00:00:00", "2012-01-26 00:00:00", "2012-01-27 00:00:00", "2012-01-28 00:00:00", "2012-01-29 00:00:00", "2012-01-30 00:00:00", "2012-01-31 00:00:00", "2012-02-01 00:00:00", "2012-02-02 00:00:00", "2012-02-03 00:00:00", "2012-02-04 00:00:00", "2012-02-05 00:00:00", "2012-02-06 00:00:00", "2012-02-07 00:00:00", "2012-02-08 00:00:00", "2012-02-09 00:00:00", "2012-02-10 00:00:00", "2012-02-11 00:00:00", "2012-02-12 00:00:00", "2012-02-13 00:00:00", "2012-02-14 00:00:00", "2012-02-15 00:00:00", "2012-02-16 00:00:00", "2012-02-17 00:00:00", "2012-02-18 00:00:00", "2012-02-19 00:00:00", "2012-02-20 00:00:00", "2012-02-21 00:00:00", "2012-02-22 00:00:00", "2012-02-23 00:00:00", "2012-02-24 00:00:00", "2012-02-25 00:00:00", "2012-02-26 00:00:00", "2012-02-27 00:00:00", "2012-02-28 00:00:00", "2012-02-29 00:00:00", "2012-03-01 00:00:00", "2012-03-02 00:00:00", "2012-03-03 00:00:00", "2012-03-04 00:00:00", "2012-03-05 00:00:00", "2012-03-06 00:00:00", "2012-03-07 00:00:00", "2012-03-08 00:00:00", "2012-03-09 00:00:00", "2012-03-10 00:00:00", "2012-03-11 00:00:00", "2012-03-12 00:00:00", "2012-03-13 00:00:00", "2012-03-14 00:00:00", "2012-03-15 00:00:00", "2012-03-16 00:00:00", "2012-03-17 00:00:00", "2012-03-18 00:00:00", "2012-03-19 00:00:00", "2012-03-20 00:00:00", "2012-03-21 00:00:00", "2012-03-22 00:00:00", "2012-03-23 00:00:00", "2012-03-24 00:00:00", "2012-03-25 00:00:00", "2012-03-26 00:00:00", "2012-03-27 00:00:00", "2012-03-28 00:00:00", "2012-03-29 00:00:00", "2012-03-30 00:00:00", "2012-03-31 00:00:00", "2012-04-01 00:00:00", "2012-04-02 00:00:00", "2012-04-03 00:00:00", "2012-04-04 00:00:00", "2012-04-05 00:00:00", "2012-04-06 00:00:00", "2012-04-07 00:00:00", "2012-04-08 00:00:00", "2012-04-09 00:00:00", "2012-04-10 00:00:00", "2012-04-11 00:00:00", "2012-04-12 00:00:00", "2012-04-13 00:00:00", "2012-04-14 00:00:00", "2012-04-15 00:00:00", "2012-04-16 00:00:00", "2012-04-17 00:00:00", "2012-04-18 00:00:00", "2012-04-19 00:00:00", "2012-04-20 00:00:00", "2012-04-21 00:00:00", "2012-04-22 00:00:00", "2012-04-23 00:00:00", "2012-04-24 00:00:00", "2012-04-25 00:00:00", "2012-04-26 00:00:00", "2012-04-27 00:00:00", "2012-04-28 00:00:00", "2012-04-29 00:00:00", "2012-04-30 00:00:00", "2012-05-01 00:00:00", "2012-05-02 00:00:00", "2012-05-03 00:00:00", "2012-05-04 00:00:00", "2012-05-05 00:00:00", "2012-05-06 00:00:00", "2012-05-07 00:00:00", "2012-05-08 00:00:00", "2012-05-09 00:00:00", "2012-05-10 00:00:00", "2012-05-11 00:00:00", "2012-05-12 00:00:00", "2012-05-13 00:00:00", "2012-05-14 00:00:00", "2012-05-15 00:00:00", "2012-05-16 00:00:00", "2012-05-17 00:00:00", "2012-05-18 00:00:00", "2012-05-19 00:00:00", "2012-05-20 00:00:00", "2012-05-21 00:00:00", "2012-05-22 00:00:00", "2012-05-23 00:00:00", "2012-05-24 00:00:00", "2012-05-25 00:00:00", "2012-05-26 00:00:00", "2012-05-27 00:00:00", "2012-05-28 00:00:00", "2012-05-29 00:00:00", "2012-05-30 00:00:00", "2012-05-31 00:00:00", "2012-06-01 00:00:00", "2012-06-02 00:00:00", "2012-06-03 00:00:00", "2012-06-04 00:00:00", "2012-06-05 00:00:00", "2012-06-06 00:00:00", "2012-06-07 00:00:00", "2012-06-08 00:00:00", "2012-06-09 00:00:00", "2012-06-10 00:00:00", "2012-06-11 00:00:00", "2012-06-12 00:00:00", "2012-06-13 00:00:00", "2012-06-14 00:00:00", "2012-06-15 00:00:00", "2012-06-16 00:00:00", "2012-06-17 00:00:00", "2012-06-18 00:00:00", "2012-06-19 00:00:00", "2012-06-20 00:00:00", "2012-06-21 00:00:00", "2012-06-22 00:00:00", "2012-06-23 00:00:00", "2012-06-24 00:00:00", "2012-06-25 00:00:00", "2012-06-26 00:00:00", "2012-06-27 00:00:00", "2012-06-28 00:00:00", "2012-06-29 00:00:00", "2012-06-30 00:00:00", "2012-07-01 00:00:00", "2012-07-02 00:00:00", "2012-07-03 00:00:00", "2012-07-04 00:00:00", "2012-07-05 00:00:00", "2012-07-06 00:00:00", "2012-07-07 00:00:00", "2012-07-08 00:00:00", "2012-07-09 00:00:00", "2012-07-10 00:00:00", "2012-07-11 00:00:00", "2012-07-12 00:00:00", "2012-07-13 00:00:00", "2012-07-14 00:00:00", "2012-07-15 00:00:00", "2012-07-16 00:00:00", "2012-07-17 00:00:00", "2012-07-18 00:00:00", "2012-07-19 00:00:00", "2012-07-20 00:00:00", "2012-07-21 00:00:00", "2012-07-22 00:00:00", "2012-07-23 00:00:00", "2012-07-24 00:00:00", "2012-07-25 00:00:00", "2012-07-26 00:00:00", "2012-07-27 00:00:00", "2012-07-28 00:00:00", "2012-07-29 00:00:00", "2012-07-30 00:00:00", "2012-07-31 00:00:00", "2012-08-01 00:00:00", "2012-08-02 00:00:00", "2012-08-03 00:00:00", "2012-08-04 00:00:00", "2012-08-05 00:00:00", "2012-08-06 00:00:00", "2012-08-07 00:00:00", "2012-08-08 00:00:00", "2012-08-09 00:00:00", "2012-08-10 00:00:00", "2012-08-11 00:00:00", "2012-08-12 00:00:00", "2012-08-13 00:00:00", "2012-08-14 00:00:00", "2012-08-15 00:00:00", "2012-08-16 00:00:00", "2012-08-17 00:00:00", "2012-08-18 00:00:00", "2012-08-19 00:00:00", "2012-08-20 00:00:00", "2012-08-21 00:00:00", "2012-08-22 00:00:00", "2012-08-23 00:00:00", "2012-08-24 00:00:00", "2012-08-25 00:00:00", "2012-08-26 00:00:00", "2012-08-27 00:00:00", "2012-08-28 00:00:00", "2012-08-29 00:00:00", "2012-08-30 00:00:00", "2012-08-31 00:00:00", "2012-09-01 00:00:00", "2012-09-02 00:00:00", "2012-09-03 00:00:00", "2012-09-04 00:00:00", "2012-09-05 00:00:00", "2012-09-06 00:00:00", "2012-09-07 00:00:00", "2012-09-08 00:00:00", "2012-09-09 00:00:00", "2012-09-10 00:00:00", "2012-09-11 00:00:00", "2012-09-12 00:00:00", "2012-09-13 00:00:00", "2012-09-14 00:00:00", "2012-09-15 00:00:00", "2012-09-16 00:00:00", "2012-09-17 00:00:00", "2012-09-18 00:00:00", "2012-09-19 00:00:00", "2012-09-20 00:00:00", "2012-09-21 00:00:00", "2012-09-22 00:00:00", "2012-09-23 00:00:00", "2012-09-24 00:00:00", "2012-09-25 00:00:00", "2012-09-26 00:00:00", "2012-09-27 00:00:00", "2012-09-28 00:00:00", "2012-09-29 00:00:00", "2012-09-30 00:00:00",
        "2012-10-01 00:00:00", "2012-10-02 00:00:00", "2012-10-03 00:00:00", "2012-10-04 00:00:00", "2012-10-05 00:00:00", "2012-10-06 00:00:00", "2012-10-07 00:00:00", "2012-10-08 00:00:00", "2012-10-09 00:00:00", "2012-10-10 00:00:00", "2012-10-11 00:00:00", "2012-10-12 00:00:00", "2012-10-13 00:00:00", "2012-10-14 00:00:00", "2012-10-15 00:00:00", "2012-10-16 00:00:00", "2012-10-17 00:00:00", "2012-10-18 00:00:00", "2012-10-19 00:00:00", "2012-10-20 00:00:00", "2012-10-21 00:00:00", "2012-10-22 00:00:00", "2012-10-23 00:00:00", "2012-10-24 00:00:00", "2012-10-25 00:00:00", "2012-10-26 00:00:00", "2012-10-27 00:00:00", "2012-10-28 00:00:00", "2012-10-29 00:00:00", "2012-10-30 00:00:00", "2012-10-31 00:00:00", "2012-11-01 00:00:00", "2012-11-02 00:00:00", "2012-11-03 00:00:00", "2012-11-04 00:00:00", "2012-11-05 00:00:00", "2012-11-06 00:00:00", "2012-11-07 00:00:00", "2012-11-08 00:00:00", "2012-11-09 00:00:00", "2012-11-10 00:00:00", "2012-11-11 00:00:00", "2012-11-12 00:00:00", "2012-11-13 00:00:00", "2012-11-14 00:00:00", "2012-11-15 00:00:00", "2012-11-16 00:00:00", "2012-11-17 00:00:00", "2012-11-18 00:00:00", "2012-11-19 00:00:00", "2012-11-20 00:00:00", "2012-11-21 00:00:00", "2012-11-22 00:00:00", "2012-11-23 00:00:00", "2012-11-24 00:00:00", "2012-11-25 00:00:00", "2012-11-26 00:00:00", "2012-11-27 00:00:00", "2012-11-28 00:00:00", "2012-11-29 00:00:00", "2012-11-30 00:00:00", "2012-12-01 00:00:00", "2012-12-02 00:00:00", "2012-12-03 00:00:00", "2012-12-04 00:00:00", "2012-12-05 00:00:00", "2012-12-06 00:00:00", "2012-12-07 00:00:00", "2012-12-08 00:00:00", "2012-12-09 00:00:00", "2012-12-10 00:00:00", "2012-12-11 00:00:00", "2012-12-12 00:00:00", "2012-12-13 00:00:00", "2012-12-14 00:00:00", "2012-12-15 00:00:00", "2012-12-16 00:00:00", "2012-12-17 00:00:00", "2012-12-18 00:00:00", "2012-12-19 00:00:00", "2012-12-20 00:00:00", "2012-12-21 00:00:00", "2012-12-22 00:00:00", "2012-12-23 00:00:00", "2012-12-24 00:00:00", "2012-12-25 00:00:00", "2012-12-26 00:00:00", "2012-12-27 00:00:00", "2012-12-28 00:00:00", "2012-12-29 00:00:00", "2012-12-30 00:00:00", "2012-12-31 00:00:00", "2013-01-01 00:00:00", "2013-01-02 00:00:00", "2013-01-03 00:00:00", "2013-01-04 00:00:00", "2013-01-05 00:00:00", "2013-01-06 00:00:00", "2013-01-07 00:00:00", "2013-01-08 00:00:00", "2013-01-09 00:00:00", "2013-01-10 00:00:00", "2013-01-11 00:00:00", "2013-01-12 00:00:00", "2013-01-13 00:00:00", "2013-01-14 00:00:00", "2013-01-15 00:00:00", "2013-01-16 00:00:00", "2013-01-17 00:00:00", "2013-01-18 00:00:00", "2013-01-19 00:00:00", "2013-01-20 00:00:00", "2013-01-21 00:00:00", "2013-01-22 00:00:00", "2013-01-23 00:00:00", "2013-01-24 00:00:00", "2013-01-25 00:00:00", "2013-01-26 00:00:00", "2013-01-27 00:00:00", "2013-01-28 00:00:00", "2013-01-29 00:00:00", "2013-01-30 00:00:00", "2013-01-31 00:00:00", "2013-02-01 00:00:00", "2013-02-02 00:00:00", "2013-02-03 00:00:00", "2013-02-04 00:00:00", "2013-02-05 00:00:00", "2013-02-06 00:00:00", "2013-02-07 00:00:00", "2013-02-08 00:00:00", "2013-02-09 00:00:00", "2013-02-10 00:00:00", "2013-02-11 00:00:00", "2013-02-12 00:00:00", "2013-02-13 00:00:00", "2013-02-14 00:00:00", "2013-02-15 00:00:00", "2013-02-16 00:00:00", "2013-02-17 00:00:00", "2013-02-18 00:00:00", "2013-02-19 00:00:00", "2013-02-20 00:00:00", "2013-02-21 00:00:00", "2013-02-22 00:00:00", "2013-02-23 00:00:00", "2013-02-24 00:00:00", "2013-02-25 00:00:00", "2013-02-26 00:00:00", "2013-02-27 00:00:00", "2013-02-28 00:00:00", "2013-03-01 00:00:00", "2013-03-02 00:00:00", "2013-03-03 00:00:00", "2013-03-04 00:00:00", "2013-03-05 00:00:00", "2013-03-06 00:00:00", "2013-03-07 00:00:00", "2013-03-08 00:00:00", "2013-03-09 00:00:00", "2013-03-10 00:00:00", "2013-03-11 00:00:00", "2013-03-12 00:00:00", "2013-03-13 00:00:00", "2013-03-14 00:00:00", "2013-03-15 00:00:00", "2013-03-16 00:00:00", "2013-03-17 00:00:00", "2013-03-18 00:00:00", "2013-03-19 00:00:00", "2013-03-20 00:00:00", "2013-03-21 00:00:00", "2013-03-22 00:00:00", "2013-03-23 00:00:00", "2013-03-24 00:00:00", "2013-03-25 00:00:00", "2013-03-26 00:00:00", "2013-03-27 00:00:00", "2013-03-28 00:00:00", "2013-03-29 00:00:00", "2013-03-30 00:00:00", "2013-03-31 00:00:00", "2013-04-01 00:00:00", "2013-04-02 00:00:00", "2013-04-03 00:00:00", "2013-04-04 00:00:00", "2013-04-05 00:00:00", "2013-04-06 00:00:00", "2013-04-07 00:00:00", "2013-04-08 00:00:00", "2013-04-09 00:00:00", "2013-04-10 00:00:00", "2013-04-11 00:00:00", "2013-04-12 00:00:00", "2013-04-13 00:00:00", "2013-04-14 00:00:00", "2013-04-15 00:00:00", "2013-04-16 00:00:00", "2013-04-17 00:00:00", "2013-04-18 00:00:00", "2013-04-19 00:00:00", "2013-04-20 00:00:00", "2013-04-21 00:00:00", "2013-04-22 00:00:00", "2013-04-23 00:00:00", "2013-04-24 00:00:00", "2013-04-25 00:00:00", "2013-04-26 00:00:00", "2013-04-27 00:00:00", "2013-04-28 00:00:00", "2013-04-29 00:00:00", "2013-04-30 00:00:00", "2013-05-01 00:00:00", "2013-05-02 00:00:00", "2013-05-03 00:00:00", "2013-05-04 00:00:00", "2013-05-05 00:00:00", "2013-05-06 00:00:00", "2013-05-07 00:00:00", "2013-05-08 00:00:00", "2013-05-09 00:00:00", "2013-05-10 00:00:00", "2013-05-11 00:00:00", "2013-05-12 00:00:00", "2013-05-13 00:00:00", "2013-05-14 00:00:00", "2013-05-15 00:00:00", "2013-05-16 00:00:00", "2013-05-17 00:00:00", "2013-05-18 00:00:00", "2013-05-19 00:00:00", "2013-05-20 00:00:00", "2013-05-21 00:00:00", "2013-05-22 00:00:00", "2013-05-23 00:00:00", "2013-05-24 00:00:00", "2013-05-25 00:00:00", "2013-05-26 00:00:00", "2013-05-27 00:00:00", "2013-05-28 00:00:00", "2013-05-29 00:00:00", "2013-05-30 00:00:00", "2013-05-31 00:00:00", "2013-06-01 00:00:00", "2013-06-02 00:00:00", "2013-06-03 00:00:00", "2013-06-04 00:00:00", "2013-06-05 00:00:00", "2013-06-06 00:00:00", "2013-06-07 00:00:00", "2013-06-08 00:00:00", "2013-06-09 00:00:00", "2013-06-10 00:00:00", "2013-06-11 00:00:00", "2013-06-12 00:00:00", "2013-06-13 00:00:00", "2013-06-14 00:00:00", "2013-06-15 00:00:00", "2013-06-16 00:00:00", "2013-06-17 00:00:00", "2013-06-18 00:00:00", "2013-06-19 00:00:00", "2013-06-20 00:00:00", "2013-06-21 00:00:00", "2013-06-22 00:00:00", "2013-06-23 00:00:00", "2013-06-24 00:00:00", "2013-06-25 00:00:00", "2013-06-26 00:00:00", "2013-06-27 00:00:00", "2013-06-28 00:00:00", "2013-06-29 00:00:00", "2013-06-30 00:00:00", "2013-07-01 00:00:00", "2013-07-02 00:00:00", "2013-07-03 00:00:00", "2013-07-04 00:00:00", "2013-07-05 00:00:00", "2013-07-06 00:00:00", "2013-07-07 00:00:00", "2013-07-08 00:00:00", "2013-07-09 00:00:00", "2013-07-10 00:00:00", "2013-07-11 00:00:00", "2013-07-12 00:00:00", "2013-07-13 00:00:00", "2013-07-14 00:00:00", "2013-07-15 00:00:00", "2013-07-16 00:00:00", "2013-07-17 00:00:00", "2013-07-18 00:00:00", "2013-07-19 00:00:00", "2013-07-20 00:00:00", "2013-07-21 00:00:00", "2013-07-22 00:00:00", "2013-07-23 00:00:00", "2013-07-24 00:00:00", "2013-07-25 00:00:00", "2013-07-26 00:00:00", "2013-07-27 00:00:00", "2013-07-28 00:00:00", "2013-07-29 00:00:00", "2013-07-30 00:00:00", "2013-07-31 00:00:00", "2013-08-01 00:00:00", "2013-08-02 00:00:00", "2013-08-03 00:00:00", "2013-08-04 00:00:00", "2013-08-05 00:00:00", "2013-08-06 00:00:00", "2013-08-07 00:00:00", "2013-08-08 00:00:00", "2013-08-09 00:00:00", "2013-08-10 00:00:00", "2013-08-11 00:00:00", "2013-08-12 00:00:00", "2013-08-13 00:00:00", "2013-08-14 00:00:00", "2013-08-15 00:00:00", "2013-08-16 00:00:00", "2013-08-17 00:00:00", "2013-08-18 00:00:00", "2013-08-19 00:00:00", "2013-08-20 00:00:00", "2013-08-21 00:00:00", "2013-08-22 00:00:00", "2013-08-23 00:00:00", "2013-08-24 00:00:00", "2013-08-25 00:00:00", "2013-08-26 00:00:00", "2013-08-27 00:00:00", "2013-08-28 00:00:00", "2013-08-29 00:00:00", "2013-08-30 00:00:00", "2013-08-31 00:00:00", "2013-09-01 00:00:00", "2013-09-02 00:00:00", "2013-09-03 00:00:00", "2013-09-04 00:00:00", "2013-09-05 00:00:00", "2013-09-06 00:00:00", "2013-09-07 00:00:00", "2013-09-08 00:00:00", "2013-09-09 00:00:00", "2013-09-10 00:00:00", "2013-09-11 00:00:00", "2013-09-12 00:00:00", "2013-09-13 00:00:00", "2013-09-14 00:00:00", "2013-09-15 00:00:00", "2013-09-16 00:00:00", "2013-09-17 00:00:00", "2013-09-18 00:00:00", "2013-09-19 00:00:00", "2013-09-20 00:00:00", "2013-09-21 00:00:00", "2013-09-22 00:00:00", "2013-09-23 00:00:00", "2013-09-24 00:00:00", "2013-09-25 00:00:00", "2013-09-26 00:00:00", "2013-09-27 00:00:00", "2013-09-28 00:00:00", "2013-09-29 00:00:00", "2013-09-30 00:00:00",
        "2013-10-01 00:00:00", "2013-10-02 00:00:00", "2013-10-03 00:00:00", "2013-10-04 00:00:00", "2013-10-05 00:00:00", "2013-10-06 00:00:00", "2013-10-07 00:00:00", "2013-10-08 00:00:00", "2013-10-09 00:00:00", "2013-10-10 00:00:00", "2013-10-11 00:00:00", "2013-10-12 00:00:00", "2013-10-13 00:00:00", "2013-10-14 00:00:00", "2013-10-15 00:00:00", "2013-10-16 00:00:00", "2013-10-17 00:00:00", "2013-10-18 00:00:00", "2013-10-19 00:00:00", "2013-10-20 00:00:00", "2013-10-21 00:00:00", "2013-10-22 00:00:00", "2013-10-23 00:00:00", "2013-10-24 00:00:00", "2013-10-25 00:00:00", "2013-10-26 00:00:00", "2013-10-27 00:00:00", "2013-10-28 00:00:00", "2013-10-29 00:00:00", "2013-10-30 00:00:00", "2013-10-31 00:00:00", "2013-11-01 00:00:00", "2013-11-02 00:00:00", "2013-11-03 00:00:00", "2013-11-04 00:00:00", "2013-11-05 00:00:00", "2013-11-06 00:00:00", "2013-11-07 00:00:00", "2013-11-08 00:00:00", "2013-11-09 00:00:00", "2013-11-10 00:00:00", "2013-11-11 00:00:00", "2013-11-12 00:00:00", "2013-11-13 00:00:00", "2013-11-14 00:00:00", "2013-11-15 00:00:00", "2013-11-16 00:00:00", "2013-11-17 00:00:00", "2013-11-18 00:00:00", "2013-11-19 00:00:00", "2013-11-20 00:00:00", "2013-11-21 00:00:00", "2013-11-22 00:00:00", "2013-11-23 00:00:00", "2013-11-24 00:00:00", "2013-11-25 00:00:00", "2013-11-26 00:00:00", "2013-11-27 00:00:00", "2013-11-28 00:00:00", "2013-11-29 00:00:00", "2013-11-30 00:00:00", "2013-12-01 00:00:00", "2013-12-02 00:00:00", "2013-12-03 00:00:00", "2013-12-04 00:00:00", "2013-12-05 00:00:00", "2013-12-06 00:00:00", "2013-12-07 00:00:00", "2013-12-08 00:00:00", "2013-12-09 00:00:00", "2013-12-10 00:00:00", "2013-12-11 00:00:00", "2013-12-12 00:00:00", "2013-12-13 00:00:00", "2013-12-14 00:00:00", "2013-12-15 00:00:00", "2013-12-16 00:00:00", "2013-12-17 00:00:00", "2013-12-18 00:00:00", "2013-12-19 00:00:00", "2013-12-20 00:00:00", "2013-12-21 00:00:00", "2013-12-22 00:00:00", "2013-12-23 00:00:00", "2013-12-24 00:00:00", "2013-12-25 00:00:00", "2013-12-26 00:00:00", "2013-12-27 00:00:00", "2013-12-28 00:00:00", "2013-12-29 00:00:00", "2013-12-30 00:00:00", "2013-12-31 00:00:00", "2014-01-01 00:00:00", "2014-01-02 00:00:00", "2014-01-03 00:00:00", "2014-01-04 00:00:00", "2014-01-05 00:00:00", "2014-01-06 00:00:00", "2014-01-07 00:00:00", "2014-01-08 00:00:00", "2014-01-09 00:00:00", "2014-01-10 00:00:00", "2014-01-11 00:00:00", "2014-01-12 00:00:00", "2014-01-13 00:00:00", "2014-01-14 00:00:00", "2014-01-15 00:00:00", "2014-01-16 00:00:00", "2014-01-17 00:00:00", "2014-01-18 00:00:00", "2014-01-19 00:00:00", "2014-01-20 00:00:00", "2014-01-21 00:00:00", "2014-01-22 00:00:00", "2014-01-23 00:00:00", "2014-01-24 00:00:00", "2014-01-25 00:00:00", "2014-01-26 00:00:00", "2014-01-27 00:00:00", "2014-01-28 00:00:00", "2014-01-29 00:00:00", "2014-01-30 00:00:00", "2014-01-31 00:00:00", "2014-02-01 00:00:00", "2014-02-02 00:00:00", "2014-02-03 00:00:00", "2014-02-04 00:00:00", "2014-02-05 00:00:00", "2014-02-06 00:00:00", "2014-02-07 00:00:00", "2014-02-08 00:00:00", "2014-02-09 00:00:00", "2014-02-10 00:00:00", "2014-02-11 00:00:00", "2014-02-12 00:00:00", "2014-02-13 00:00:00", "2014-02-14 00:00:00", "2014-02-15 00:00:00", "2014-02-16 00:00:00", "2014-02-17 00:00:00", "2014-02-18 00:00:00", "2014-02-19 00:00:00", "2014-02-20 00:00:00", "2014-02-21 00:00:00", "2014-02-22 00:00:00", "2014-02-23 00:00:00", "2014-02-24 00:00:00", "2014-02-25 00:00:00", "2014-02-26 00:00:00", "2014-02-27 00:00:00", "2014-02-28 00:00:00", "2014-03-01 00:00:00", "2014-03-02 00:00:00", "2014-03-03 00:00:00", "2014-03-04 00:00:00", "2014-03-05 00:00:00", "2014-03-06 00:00:00", "2014-03-07 00:00:00", "2014-03-08 00:00:00", "2014-03-09 00:00:00", "2014-03-10 00:00:00", "2014-03-11 00:00:00", "2014-03-12 00:00:00", "2014-03-13 00:00:00", "2014-03-14 00:00:00", "2014-03-15 00:00:00", "2014-03-16 00:00:00", "2014-03-17 00:00:00", "2014-03-18 00:00:00", "2014-03-19 00:00:00", "2014-03-20 00:00:00", "2014-03-21 00:00:00", "2014-03-22 00:00:00", "2014-03-23 00:00:00", "2014-03-24 00:00:00", "2014-03-25 00:00:00", "2014-03-26 00:00:00", "2014-03-27 00:00:00", "2014-03-28 00:00:00", "2014-03-29 00:00:00", "2014-03-30 00:00:00", "2014-03-31 00:00:00", "2014-04-01 00:00:00", "2014-04-02 00:00:00", "2014-04-03 00:00:00", "2014-04-04 00:00:00", "2014-04-05 00:00:00", "2014-04-06 00:00:00", "2014-04-07 00:00:00", "2014-04-08 00:00:00", "2014-04-09 00:00:00", "2014-04-10 00:00:00", "2014-04-11 00:00:00", "2014-04-12 00:00:00", "2014-04-13 00:00:00", "2014-04-14 00:00:00", "2014-04-15 00:00:00", "2014-04-16 00:00:00", "2014-04-17 00:00:00", "2014-04-18 00:00:00", "2014-04-19 00:00:00", "2014-04-20 00:00:00", "2014-04-21 00:00:00", "2014-04-22 00:00:00", "2014-04-23 00:00:00", "2014-04-24 00:00:00", "2014-04-25 00:00:00", "2014-04-26 00:00:00", "2014-04-27 00:00:00", "2014-04-28 00:00:00", "2014-04-29 00:00:00", "2014-04-30 00:00:00", "2014-05-01 00:00:00", "2014-05-02 00:00:00", "2014-05-03 00:00:00", "2014-05-04 00:00:00", "2014-05-05 00:00:00", "2014-05-06 00:00:00", "2014-05-07 00:00:00", "2014-05-08 00:00:00", "2014-05-09 00:00:00", "2014-05-10 00:00:00", "2014-05-11 00:00:00", "2014-05-12 00:00:00", "2014-05-13 00:00:00", "2014-05-14 00:00:00", "2014-05-15 00:00:00", "2014-05-16 00:00:00", "2014-05-17 00:00:00", "2014-05-18 00:00:00", "2014-05-19 00:00:00", "2014-05-20 00:00:00", "2014-05-21 00:00:00", "2014-05-22 00:00:00", "2014-05-23 00:00:00", "2014-05-24 00:00:00", "2014-05-25 00:00:00", "2014-05-26 00:00:00", "2014-05-27 00:00:00", "2014-05-28 00:00:00", "2014-05-29 00:00:00", "2014-05-30 00:00:00", "2014-05-31 00:00:00", "2014-06-01 00:00:00", "2014-06-02 00:00:00", "2014-06-03 00:00:00", "2014-06-04 00:00:00", "2014-06-05 00:00:00", "2014-06-06 00:00:00", "2014-06-07 00:00:00", "2014-06-08 00:00:00", "2014-06-09 00:00:00", "2014-06-10 00:00:00", "2014-06-11 00:00:00", "2014-06-12 00:00:00", "2014-06-13 00:00:00", "2014-06-14 00:00:00", "2014-06-15 00:00:00", "2014-06-16 00:00:00", "2014-06-17 00:00:00", "2014-06-18 00:00:00", "2014-06-19 00:00:00", "2014-06-20 00:00:00", "2014-06-21 00:00:00", "2014-06-22 00:00:00", "2014-06-23 00:00:00", "2014-06-24 00:00:00", "2014-06-25 00:00:00", "2014-06-26 00:00:00", "2014-06-27 00:00:00", "2014-06-28 00:00:00", "2014-06-29 00:00:00", "2014-06-30 00:00:00", "2014-07-01 00:00:00", "2014-07-02 00:00:00", "2014-07-03 00:00:00", "2014-07-04 00:00:00", "2014-07-05 00:00:00", "2014-07-06 00:00:00", "2014-07-07 00:00:00", "2014-07-08 00:00:00", "2014-07-09 00:00:00", "2014-07-10 00:00:00", "2014-07-11 00:00:00", "2014-07-12 00:00:00", "2014-07-13 00:00:00", "2014-07-14 00:00:00", "2014-07-15 00:00:00", "2014-07-16 00:00:00", "2014-07-17 00:00:00", "2014-07-18 00:00:00", "2014-07-19 00:00:00", "2014-07-20 00:00:00", "2014-07-21 00:00:00", "2014-07-22 00:00:00", "2014-07-23 00:00:00", "2014-07-24 00:00:00", "2014-07-25 00:00:00", "2014-07-26 00:00:00", "2014-07-27 00:00:00", "2014-07-28 00:00:00", "2014-07-29 00:00:00", "2014-07-30 00:00:00", "2014-07-31 00:00:00", "2014-08-01 00:00:00", "2014-08-02 00:00:00", "2014-08-03 00:00:00", "2014-08-04 00:00:00", "2014-08-05 00:00:00", "2014-08-06 00:00:00", "2014-08-07 00:00:00", "2014-08-08 00:00:00", "2014-08-09 00:00:00", "2014-08-10 00:00:00", "2014-08-11 00:00:00", "2014-08-12 00:00:00", "2014-08-13 00:00:00", "2014-08-14 00:00:00", "2014-08-15 00:00:00", "2014-08-16 00:00:00", "2014-08-17 00:00:00", "2014-08-18 00:00:00", "2014-08-19 00:00:00", "2014-08-20 00:00:00", "2014-08-21 00:00:00", "2014-08-22 00:00:00", "2014-08-23 00:00:00", "2014-08-24 00:00:00", "2014-08-25 00:00:00", "2014-08-26 00:00:00", "2014-08-27 00:00:00", "2014-08-28 00:00:00", "2014-08-29 00:00:00", "2014-08-30 00:00:00", "2014-08-31 00:00:00", "2014-09-01 00:00:00", "2014-09-02 00:00:00", "2014-09-03 00:00:00", "2014-09-04 00:00:00", "2014-09-05 00:00:00", "2014-09-06 00:00:00", "2014-09-07 00:00:00", "2014-09-08 00:00:00", "2014-09-09 00:00:00", "2014-09-10 00:00:00", "2014-09-11 00:00:00", "2014-09-12 00:00:00", "2014-09-13 00:00:00", "2014-09-14 00:00:00", "2014-09-15 00:00:00", "2014-09-16 00:00:00", "2014-09-17 00:00:00", "2014-09-18 00:00:00", "2014-09-19 00:00:00", "2014-09-20 00:00:00", "2014-09-21 00:00:00", "2014-09-22 00:00:00", "2014-09-23 00:00:00", "2014-09-24 00:00:00", "2014-09-25 00:00:00", "2014-09-26 00:00:00", "2014-09-27 00:00:00", "2014-09-28 00:00:00", "2014-09-29 00:00:00", "2014-09-30 00:00:00"
    };

    // use bootstrap generator to get samples
    // experiment (5 samples of 4 year length, set seed to fixed value 7):
    auto samples = evalhyd::uncertainty::bootstrap(datetimes, 5, 4, 7);

    // manually generate results expected from random generator
    std::vector<std::vector<int>> results;

    std::unordered_map<int, std::vector<int>> year_to_indices;
    xt::xtensor<int, 1> hy_2012 = xt::arange(0, 366); // 0-365
    year_to_indices[2011] = std::vector<int>(hy_2012.begin(), hy_2012.end());
    xt::xtensor<int, 1> hy_2013 = xt::arange(366, 731);  // 366-730
    year_to_indices[2012] = std::vector<int>(hy_2013.begin(), hy_2013.end());
    xt::xtensor<int, 1> hy_2014 = xt::arange(731, 1096);  // 731-1095
    year_to_indices[2013] = std::vector<int>(hy_2014.begin(), hy_2014.end());

    std::vector<std::vector<int>> sampled_years;
#if EVALHYD_TESTING_OS == WINDOWS
    sampled_years = {{2011, 2012, 2012, 2013},
                     {2012, 2011, 2013, 2013},
                     {2012, 2013, 2012, 2012},
                     {2013, 2011, 2013, 2012},
                     {2011, 2012, 2011, 2011}};
#elif EVALHYD_TESTING_OS == MACOS
    sampled_years = {{2011, 2012, 2013, 2011},
                     {2012, 2013, 2011, 2012},
                     {2013, 2013, 2013, 2011},
                     {2013, 2011, 2011, 2011},
                     {2011, 2013, 2013, 2011}};
#elif EVALHYD_TESTING_OS == LINUX
    sampled_years = {{2011, 2011, 2013, 2011},
                     {2012, 2013, 2013, 2012},
                     {2013, 2011, 2012, 2011},
                     {2012, 2011, 2011, 2012},
                     {2011, 2011, 2012, 2012}};
#endif

    for (std::size_t s = 0; s < samples.size(); s++)
    {
        std::vector<int> indices;
        for (std::size_t y = 0; y < 4; y++)
        {
            auto i = year_to_indices[sampled_years[s][y]];
            indices.insert(indices.end(), i.begin(), i.end());
        }
        results.push_back(indices);
    }

    // check that sampled indices are as expected by applying them on some data
    for (std::size_t s = 0; s < samples.size(); s++)
    {
        auto data = xt::arange(datetimes.size());
        EXPECT_TRUE(
                xt::allclose(
                        xt::view(data, samples[s]),
                        xt::view(data, xt::keep(results[s]))
                )
        ) << "Failure for ( sample " << s << ")";
    }
}
