from socket import gethostname

import cloudpickle
import zmq


class SocketInterface:
    """
    The SocketInterface is an abstraction layer on top of the zero message queue.

    Args:
        spawner (executorlib.shared.spawner.BaseSpawner): Interface for starting the parallel process
    """

    def __init__(self, spawner=None):
        """
        Initialize the SocketInterface.

        Args:
            spawner (executorlib.shared.spawner.BaseSpawner): Interface for starting the parallel process
        """
        self._context = zmq.Context()
        self._socket = self._context.socket(zmq.PAIR)
        self._process = None
        self._spawner = spawner

    def send_dict(self, input_dict: dict):
        """
        Send a dictionary with instructions to a connected client process.

        Args:
            input_dict (dict): dictionary of commands to be communicated. The key "shutdown" is reserved to stop the
                connected client from listening.
        """
        self._socket.send(cloudpickle.dumps(input_dict))

    def receive_dict(self):
        """
        Receive a dictionary from a connected client process.

        Returns:
            dict: dictionary with response received from the connected client
        """
        output = cloudpickle.loads(self._socket.recv())
        if "result" in output.keys():
            return output["result"]
        else:
            error_type = output["error_type"].split("'")[1]
            raise eval(error_type)(output["error"])

    def send_and_receive_dict(self, input_dict: dict) -> dict:
        """
        Combine both the send_dict() and receive_dict() function in a single call.

        Args:
            input_dict (dict): dictionary of commands to be communicated. The key "shutdown" is reserved to stop the
                connected client from listening.

        Returns:
            dict: dictionary with response received from the connected client
        """
        self.send_dict(input_dict=input_dict)
        return self.receive_dict()

    def bind_to_random_port(self):
        """
        Identify a random port typically in the range from 49152 to 65536 to bind the SocketInterface instance to. Other
        processes can then connect to this port to receive instructions and send results.

        Returns:
            int: port the SocketInterface instance is bound to.
        """
        return self._socket.bind_to_random_port("tcp://*")

    def bootup(
        self,
        command_lst: list[str],
    ):
        """
        Boot up the client process to connect to the SocketInterface.

        Args:
            command_lst (list): list of strings to start the client process
        """
        self._spawner.bootup(
            command_lst=command_lst,
        )

    def shutdown(self, wait: bool = True):
        """
        Shutdown the SocketInterface and the connected client process.

        Args:
            wait (bool): Whether to wait for the client process to finish before returning. Default is True.
        """
        result = None
        if self._spawner.poll():
            result = self.send_and_receive_dict(
                input_dict={"shutdown": True, "wait": wait}
            )
            self._spawner.shutdown(wait=wait)
        if self._socket is not None:
            self._socket.close()
        if self._context is not None:
            self._context.term()
        self._process = None
        self._socket = None
        self._context = None
        return result

    def __del__(self):
        """
        Destructor for the SocketInterface class.
        Calls the shutdown method with wait=True to ensure proper cleanup.
        """
        self.shutdown(wait=True)


def interface_bootup(
    command_lst: list[str],
    connections,
    hostname_localhost: bool = False,
):
    """
    Start interface for ZMQ communication

    Args:
        command_lst (list): List of commands as strings
        connections (executorlib.shared.spawner.BaseSpawner): Interface to start parallel process, like MPI, SLURM
                                                                  or Flux
        hostname_localhost (boolean): use localhost instead of the hostname to establish the zmq connection. In the
                                      context of an HPC cluster this essential to be able to communicate to an
                                      Executor running on a different compute node within the same allocation. And
                                      in principle any computer should be able to resolve that their own hostname
                                      points to the same address as localhost. Still MacOS >= 12 seems to disable
                                      this look up for security reasons. So on MacOS it is required to set this
                                      option to true

    Returns:
         executorlib.shared.communication.SocketInterface: socket interface for zmq communication
    """
    if not hostname_localhost:
        command_lst += [
            "--host",
            gethostname(),
        ]
    interface = SocketInterface(spawner=connections)
    command_lst += [
        "--zmqport",
        str(interface.bind_to_random_port()),
    ]
    interface.bootup(
        command_lst=command_lst,
    )
    return interface


def interface_connect(host: str, port: str):
    """
    Connect to an existing SocketInterface instance by providing the hostname and the port as strings.

    Args:
        host (str): hostname of the host running the SocketInterface instance to connect to.
        port (str): port on the host the SocketInterface instance is running on.
    """
    context = zmq.Context()
    socket = context.socket(zmq.PAIR)
    socket.connect("tcp://" + host + ":" + port)
    return context, socket


def interface_send(socket: zmq.Socket, result_dict: dict):
    """
    Send results to a SocketInterface instance.

    Args:
        socket (zmq.Socket): socket for the connection
        result_dict (dict): dictionary to be sent, supported keys are result, error and error_type.
    """
    socket.send(cloudpickle.dumps(result_dict))


def interface_receive(socket: zmq.Socket):
    """
    Receive instructions from a SocketInterface instance.

    Args:
        socket (zmq.Socket): socket for the connection
    """
    return cloudpickle.loads(socket.recv())


def interface_shutdown(socket: zmq.Socket, context: zmq.Context):
    """
    Close the connection to a SocketInterface instance.

    Args:
        socket (zmq.Socket): socket for the connection
        context (zmq.sugar.context.Context): context for the connection
    """
    socket.close()
    context.term()
