# fathom-python

<!-- Start Summary [summary] -->
## Summary

Fathom External API: The Fathom External API lets you poll meetings, teams, and team members, and
optionally receive webhooks when content from a new meeting is ready.
<!-- End Summary [summary] -->

<!-- Start SDK Installation [installation] -->
## SDK Installation

> [!NOTE]
> **Python version upgrade policy**
>
> Once a Python version reaches its [official end of life date](https://devguide.python.org/versions/), a 3-month grace period is provided for users to upgrade. Following this grace period, the minimum python version supported in the SDK will be updated.

The SDK can be installed with *uv*, *pip*, or *poetry* package managers.

### uv

*uv* is a fast Python package installer and resolver, designed as a drop-in replacement for pip and pip-tools. It's recommended for its speed and modern Python tooling capabilities.

```bash
uv add fathom-python
```

### PIP

*PIP* is the default package installer for Python, enabling easy installation and management of packages from PyPI via the command line.

```bash
pip install fathom-python
```

### Poetry

*Poetry* is a modern tool that simplifies dependency management and package publishing by using a single `pyproject.toml` file to handle project metadata and dependencies.

```bash
poetry add fathom-python
```

### Shell and script usage with `uv`

You can use this SDK in a Python shell with [uv](https://docs.astral.sh/uv/) and the `uvx` command that comes with it like so:

```shell
uvx --from fathom-python python
```

It's also possible to write a standalone Python script without needing to set up a whole project like so:

```python
#!/usr/bin/env -S uv run --script
# /// script
# requires-python = ">=3.9"
# dependencies = [
#     "fathom-python",
# ]
# ///

from fathom_python import Fathom

sdk = Fathom(
  # SDK arguments
)

# Rest of script here...
```

Once that is saved to a file, you can run it with `uv run script.py` where
`script.py` can be replaced with the actual file name.
<!-- End SDK Installation [installation] -->

<!-- Start IDE Support [idesupport] -->
## IDE Support

### PyCharm

Generally, the SDK will work well with most IDEs out of the box. However, when using PyCharm, you can enjoy much better integration with Pydantic by installing an additional plugin.

- [PyCharm Pydantic Plugin](https://docs.pydantic.dev/latest/integrations/pycharm/)
<!-- End IDE Support [idesupport] -->

<!-- Start SDK Example Usage [usage] -->
## SDK Example Usage

### Example

```python
# Synchronous Example
from fathom_python import Fathom, models
import os


with Fathom(
    security=models.Security(
        api_key_auth=os.getenv("FATHOM_API_KEY_AUTH", ""),
    ),
) as fathom:

    res = fathom.list_meetings(calendar_invitees=[
        "cfo@acme.com",
        "legal@acme.com",
    ], calendar_invitees_domains=[
        "acme.com",
        "client.com",
    ], calendar_invitees_domains_type=models.ListMeetingsCalendarInviteesDomainsType.ALL, include_crm_matches=False, include_summary=False, include_transcript=False, meeting_type=models.MeetingType.ALL, recorded_by=[
        "ceo@acme.com",
        "pm@acme.com",
    ], teams=[
        "Sales",
        "Engineering",
    ])

    while res is not None:
        # Handle items

        res = res.next()
```

</br>

The same SDK client can also be used to make asynchronous requests by importing asyncio.
```python
# Asynchronous Example
import asyncio
from fathom_python import Fathom, models
import os

async def main():

    async with Fathom(
        security=models.Security(
            api_key_auth=os.getenv("FATHOM_API_KEY_AUTH", ""),
        ),
    ) as fathom:

        res = await fathom.list_meetings_async(calendar_invitees=[
            "cfo@acme.com",
            "legal@acme.com",
        ], calendar_invitees_domains=[
            "acme.com",
            "client.com",
        ], calendar_invitees_domains_type=models.ListMeetingsCalendarInviteesDomainsType.ALL, include_crm_matches=False, include_summary=False, include_transcript=False, meeting_type=models.MeetingType.ALL, recorded_by=[
            "ceo@acme.com",
            "pm@acme.com",
        ], teams=[
            "Sales",
            "Engineering",
        ])

        while res is not None:
            # Handle items

            res = res.next()

asyncio.run(main())
```
<!-- End SDK Example Usage [usage] -->

<!-- Start Authentication [security] -->
## Authentication

### Per-Client Security Schemes

This SDK supports the following security schemes globally:

| Name           | Type   | Scheme      | Environment Variable  |
| -------------- | ------ | ----------- | --------------------- |
| `api_key_auth` | apiKey | API key     | `FATHOM_API_KEY_AUTH` |
| `bearer_auth`  | http   | HTTP Bearer | `FATHOM_BEARER_AUTH`  |

You can set the security parameters through the `security` optional parameter when initializing the SDK client instance. The selected scheme will be used by default to authenticate with the API for all operations that support it. For example:
```python
from fathom_python import Fathom, models
import os


with Fathom(
    security=models.Security(
        api_key_auth=os.getenv("FATHOM_API_KEY_AUTH", ""),
    ),
) as fathom:

    res = fathom.list_meetings(calendar_invitees=[
        "cfo@acme.com",
        "legal@acme.com",
    ], calendar_invitees_domains=[
        "acme.com",
        "client.com",
    ], calendar_invitees_domains_type=models.ListMeetingsCalendarInviteesDomainsType.ALL, include_crm_matches=False, include_summary=False, include_transcript=False, meeting_type=models.MeetingType.ALL, recorded_by=[
        "ceo@acme.com",
        "pm@acme.com",
    ], teams=[
        "Sales",
        "Engineering",
    ])

    while res is not None:
        # Handle items

        res = res.next()

```
<!-- End Authentication [security] -->

<!-- Start Available Resources and Operations [operations] -->
## Available Resources and Operations

<details open>
<summary>Available methods</summary>

### [Fathom SDK](docs/sdks/fathom/README.md)

* [list_meetings](docs/sdks/fathom/README.md#list_meetings) - List meetings
* [get_recording_summary](docs/sdks/fathom/README.md#get_recording_summary) - Get summary
* [get_recording_transcript](docs/sdks/fathom/README.md#get_recording_transcript) - Get transcript
* [list_teams](docs/sdks/fathom/README.md#list_teams) - List teams
* [list_team_members](docs/sdks/fathom/README.md#list_team_members) - List team members
* [create_webhook](docs/sdks/fathom/README.md#create_webhook) - Create a webhook
* [delete_webhook](docs/sdks/fathom/README.md#delete_webhook) - Delete a webhook

</details>
<!-- End Available Resources and Operations [operations] -->

<!-- Start Pagination [pagination] -->
## Pagination

Some of the endpoints in this SDK support pagination. To use pagination, you make your SDK calls as usual, but the
returned response object will have a `Next` method that can be called to pull down the next group of results. If the
return value of `Next` is `None`, then there are no more pages to be fetched.

Here's an example of one such pagination call:
```python
from fathom_python import Fathom, models
import os


with Fathom(
    security=models.Security(
        api_key_auth=os.getenv("FATHOM_API_KEY_AUTH", ""),
    ),
) as fathom:

    res = fathom.list_meetings(calendar_invitees=[
        "cfo@acme.com",
        "legal@acme.com",
    ], calendar_invitees_domains=[
        "acme.com",
        "client.com",
    ], calendar_invitees_domains_type=models.ListMeetingsCalendarInviteesDomainsType.ALL, include_crm_matches=False, include_summary=False, include_transcript=False, meeting_type=models.MeetingType.ALL, recorded_by=[
        "ceo@acme.com",
        "pm@acme.com",
    ], teams=[
        "Sales",
        "Engineering",
    ])

    while res is not None:
        # Handle items

        res = res.next()

```
<!-- End Pagination [pagination] -->

<!-- Start Retries [retries] -->
## Retries

Some of the endpoints in this SDK support retries. If you use the SDK without any configuration, it will fall back to the default retry strategy provided by the API. However, the default retry strategy can be overridden on a per-operation basis, or across the entire SDK.

To change the default retry strategy for a single API call, simply provide a `RetryConfig` object to the call:
```python
from fathom_python import Fathom, models
from fathom_python.utils import BackoffStrategy, RetryConfig
import os


with Fathom(
    security=models.Security(
        api_key_auth=os.getenv("FATHOM_API_KEY_AUTH", ""),
    ),
) as fathom:

    res = fathom.list_meetings(calendar_invitees=[
        "cfo@acme.com",
        "legal@acme.com",
    ], calendar_invitees_domains=[
        "acme.com",
        "client.com",
    ], calendar_invitees_domains_type=models.ListMeetingsCalendarInviteesDomainsType.ALL, include_crm_matches=False, include_summary=False, include_transcript=False, meeting_type=models.MeetingType.ALL, recorded_by=[
        "ceo@acme.com",
        "pm@acme.com",
    ], teams=[
        "Sales",
        "Engineering",
    ],
        RetryConfig("backoff", BackoffStrategy(1, 50, 1.1, 100), False))

    while res is not None:
        # Handle items

        res = res.next()

```

If you'd like to override the default retry strategy for all operations that support retries, you can use the `retry_config` optional parameter when initializing the SDK:
```python
from fathom_python import Fathom, models
from fathom_python.utils import BackoffStrategy, RetryConfig
import os


with Fathom(
    retry_config=RetryConfig("backoff", BackoffStrategy(1, 50, 1.1, 100), False),
    security=models.Security(
        api_key_auth=os.getenv("FATHOM_API_KEY_AUTH", ""),
    ),
) as fathom:

    res = fathom.list_meetings(calendar_invitees=[
        "cfo@acme.com",
        "legal@acme.com",
    ], calendar_invitees_domains=[
        "acme.com",
        "client.com",
    ], calendar_invitees_domains_type=models.ListMeetingsCalendarInviteesDomainsType.ALL, include_crm_matches=False, include_summary=False, include_transcript=False, meeting_type=models.MeetingType.ALL, recorded_by=[
        "ceo@acme.com",
        "pm@acme.com",
    ], teams=[
        "Sales",
        "Engineering",
    ])

    while res is not None:
        # Handle items

        res = res.next()

```
<!-- End Retries [retries] -->

<!-- Start Error Handling [errors] -->
## Error Handling

[`FathomError`](./src/fathom_python/errors/fathomerror.py) is the base class for all HTTP error responses. It has the following properties:

| Property           | Type             | Description                                            |
| ------------------ | ---------------- | ------------------------------------------------------ |
| `err.message`      | `str`            | Error message                                          |
| `err.status_code`  | `int`            | HTTP response status code eg `404`                     |
| `err.headers`      | `httpx.Headers`  | HTTP response headers                                  |
| `err.body`         | `str`            | HTTP body. Can be empty string if no body is returned. |
| `err.raw_response` | `httpx.Response` | Raw HTTP response                                      |

### Example
```python
from fathom_python import Fathom, errors, models
import os


with Fathom(
    security=models.Security(
        api_key_auth=os.getenv("FATHOM_API_KEY_AUTH", ""),
    ),
) as fathom:
    res = None
    try:

        res = fathom.list_meetings(calendar_invitees=[
            "cfo@acme.com",
            "legal@acme.com",
        ], calendar_invitees_domains=[
            "acme.com",
            "client.com",
        ], calendar_invitees_domains_type=models.ListMeetingsCalendarInviteesDomainsType.ALL, include_crm_matches=False, include_summary=False, include_transcript=False, meeting_type=models.MeetingType.ALL, recorded_by=[
            "ceo@acme.com",
            "pm@acme.com",
        ], teams=[
            "Sales",
            "Engineering",
        ])

        while res is not None:
            # Handle items

            res = res.next()


    except errors.FathomError as e:
        # The base class for HTTP error responses
        print(e.message)
        print(e.status_code)
        print(e.body)
        print(e.headers)
        print(e.raw_response)

```

### Error Classes
**Primary error:**
* [`FathomError`](./src/fathom_python/errors/fathomerror.py): The base class for HTTP error responses.

<details><summary>Less common errors (5)</summary>

<br />

**Network errors:**
* [`httpx.RequestError`](https://www.python-httpx.org/exceptions/#httpx.RequestError): Base class for request errors.
    * [`httpx.ConnectError`](https://www.python-httpx.org/exceptions/#httpx.ConnectError): HTTP client was unable to make a request to a server.
    * [`httpx.TimeoutException`](https://www.python-httpx.org/exceptions/#httpx.TimeoutException): HTTP request timed out.


**Inherit from [`FathomError`](./src/fathom_python/errors/fathomerror.py)**:
* [`ResponseValidationError`](./src/fathom_python/errors/responsevalidationerror.py): Type mismatch between the response data and the expected Pydantic model. Provides access to the Pydantic validation error via the `cause` attribute.

</details>
<!-- End Error Handling [errors] -->

<!-- Start Server Selection [server] -->
## Server Selection

### Override Server URL Per-Client

The default server can be overridden globally by passing a URL to the `server_url: str` optional parameter when initializing the SDK client instance. For example:
```python
from fathom_python import Fathom, models
import os


with Fathom(
    server_url="https://api.fathom.ai/external/v1",
    security=models.Security(
        api_key_auth=os.getenv("FATHOM_API_KEY_AUTH", ""),
    ),
) as fathom:

    res = fathom.list_meetings(calendar_invitees=[
        "cfo@acme.com",
        "legal@acme.com",
    ], calendar_invitees_domains=[
        "acme.com",
        "client.com",
    ], calendar_invitees_domains_type=models.ListMeetingsCalendarInviteesDomainsType.ALL, include_crm_matches=False, include_summary=False, include_transcript=False, meeting_type=models.MeetingType.ALL, recorded_by=[
        "ceo@acme.com",
        "pm@acme.com",
    ], teams=[
        "Sales",
        "Engineering",
    ])

    while res is not None:
        # Handle items

        res = res.next()

```
<!-- End Server Selection [server] -->

<!-- Start Custom HTTP Client [http-client] -->
## Custom HTTP Client

The Python SDK makes API calls using the [httpx](https://www.python-httpx.org/) HTTP library.  In order to provide a convenient way to configure timeouts, cookies, proxies, custom headers, and other low-level configuration, you can initialize the SDK client with your own HTTP client instance.
Depending on whether you are using the sync or async version of the SDK, you can pass an instance of `HttpClient` or `AsyncHttpClient` respectively, which are Protocol's ensuring that the client has the necessary methods to make API calls.
This allows you to wrap the client with your own custom logic, such as adding custom headers, logging, or error handling, or you can just pass an instance of `httpx.Client` or `httpx.AsyncClient` directly.

For example, you could specify a header for every request that this sdk makes as follows:
```python
from fathom_python import Fathom
import httpx

http_client = httpx.Client(headers={"x-custom-header": "someValue"})
s = Fathom(client=http_client)
```

or you could wrap the client with your own custom logic:
```python
from fathom_python import Fathom
from fathom_python.httpclient import AsyncHttpClient
import httpx

class CustomClient(AsyncHttpClient):
    client: AsyncHttpClient

    def __init__(self, client: AsyncHttpClient):
        self.client = client

    async def send(
        self,
        request: httpx.Request,
        *,
        stream: bool = False,
        auth: Union[
            httpx._types.AuthTypes, httpx._client.UseClientDefault, None
        ] = httpx.USE_CLIENT_DEFAULT,
        follow_redirects: Union[
            bool, httpx._client.UseClientDefault
        ] = httpx.USE_CLIENT_DEFAULT,
    ) -> httpx.Response:
        request.headers["Client-Level-Header"] = "added by client"

        return await self.client.send(
            request, stream=stream, auth=auth, follow_redirects=follow_redirects
        )

    def build_request(
        self,
        method: str,
        url: httpx._types.URLTypes,
        *,
        content: Optional[httpx._types.RequestContent] = None,
        data: Optional[httpx._types.RequestData] = None,
        files: Optional[httpx._types.RequestFiles] = None,
        json: Optional[Any] = None,
        params: Optional[httpx._types.QueryParamTypes] = None,
        headers: Optional[httpx._types.HeaderTypes] = None,
        cookies: Optional[httpx._types.CookieTypes] = None,
        timeout: Union[
            httpx._types.TimeoutTypes, httpx._client.UseClientDefault
        ] = httpx.USE_CLIENT_DEFAULT,
        extensions: Optional[httpx._types.RequestExtensions] = None,
    ) -> httpx.Request:
        return self.client.build_request(
            method,
            url,
            content=content,
            data=data,
            files=files,
            json=json,
            params=params,
            headers=headers,
            cookies=cookies,
            timeout=timeout,
            extensions=extensions,
        )

s = Fathom(async_client=CustomClient(httpx.AsyncClient()))
```
<!-- End Custom HTTP Client [http-client] -->

<!-- Start Resource Management [resource-management] -->
## Resource Management

The `Fathom` class implements the context manager protocol and registers a finalizer function to close the underlying sync and async HTTPX clients it uses under the hood. This will close HTTP connections, release memory and free up other resources held by the SDK. In short-lived Python programs and notebooks that make a few SDK method calls, resource management may not be a concern. However, in longer-lived programs, it is beneficial to create a single SDK instance via a [context manager][context-manager] and reuse it across the application.

[context-manager]: https://docs.python.org/3/reference/datamodel.html#context-managers

```python
from fathom_python import Fathom, models
import os
def main():

    with Fathom(
        security=models.Security(
            api_key_auth=os.getenv("FATHOM_API_KEY_AUTH", ""),
        ),
    ) as fathom:
        # Rest of application here...


# Or when using async:
async def amain():

    async with Fathom(
        security=models.Security(
            api_key_auth=os.getenv("FATHOM_API_KEY_AUTH", ""),
        ),
    ) as fathom:
        # Rest of application here...
```
<!-- End Resource Management [resource-management] -->

<!-- Start Debugging [debug] -->
## Debugging

You can setup your SDK to emit debug logs for SDK requests and responses.

You can pass your own logger class directly into your SDK.
```python
from fathom_python import Fathom
import logging

logging.basicConfig(level=logging.DEBUG)
s = Fathom(debug_logger=logging.getLogger("fathom_python"))
```

You can also enable a default debug logger by setting an environment variable `FATHOM_DEBUG` to true.
<!-- End Debugging [debug] -->

<!-- Placeholder for Future Speakeasy SDK Sections -->

# Development

## Maturity

This SDK is in beta, and there may be breaking changes between versions without a major version update. Therefore, we recommend pinning usage
to a specific package version. This way, you can install the same version each time without breaking changes unless you are intentionally
looking for the latest version.
