"""Canvas model extension for the ASC Framing Decision List (FDL).

This module provides the `Canvas` model extension used by the FDL. It supplies a post-initialisation
hook that helps certain calculations like effective_anchor_point when appropriate, and creates
validators such as uniqueness of the FramingDecision IDs, and convenience methods creating
subsequent models such as FramingDecisions.
"""

from typing import Any

from pydantic import field_validator

from .schema import (
    _Canvas,
    FdlIdFramingDecision,
    FramingIntent,
    FramingDecision,
)


class Canvas(_Canvas):
    """Canvas model with FDL-specific validation and helpers.

    The class extends the `_Canvas` schema that is generated by the official FDL schema, with
    helpers needed in FDL workflows.
    """

    def model_post_init(self, __context: Any | None) -> None:
        """Post-initialisation hook.

        This validates and helps to automatically get appropriate `effective_anchor_point` when
        applicable. So users can only specify effective_dimensions and have a center-based anchor
        point calculated for them unless a specific, non-central position needs to be specified.

        Raises:
            ValueError: if `effective_anchor_point` is set but `effective_dimensions` is missing.
        """
        from .calc import get_anchor_point

        if self.effective_dimensions is None and self.effective_anchor_point:
            raise ValueError("effective_anchor_point can only be set with effective_dimensions")

        # Calculate the effective anchor point if not provided
        if not self.effective_anchor_point and self.effective_dimensions and self.dimensions:
            self.effective_anchor_point = get_anchor_point(
                canvas_width=self.dimensions.width,
                canvas_height=self.dimensions.height,
                target_width=self.effective_dimensions.width,
                target_height=self.effective_dimensions.height,
            )

    @field_validator("framing_decisions")
    @classmethod
    def validate_uniqueness_framing_decisions(
        cls, framing_decisions: list[FramingDecision] | None
    ) -> list[FramingDecision] | None:
        """Ensure FramingDecision IDs are unique.

        Raises:
            ValueError: when a duplicate FramingDecision ID is detected.
        """
        if framing_decisions is None:
            return None

        seen = set()
        for framing_decision in framing_decisions:
            framing_decision_id = framing_decision.id
            if str(framing_decision_id) in seen:
                raise ValueError(f"FramingDecision ID {framing_decision_id!r} already exists")

            seen.add(str(framing_decision_id))

        return framing_decisions

    def add_framing_decision(
        self,
        framing_intent: FramingIntent,
        label: str | None = None,
    ) -> FramingDecision:
        """Create and append a FramingDecision for framing_intent.

        The method computes dimensions, anchors and protection regions and appends the constructed
        FramingDecision to this canvas. If the resulting FramingDecision ID duplicates an existing
        ID a `ValueError` will be raised.
        """
        from .calc import calculate_framing_decision

        # Generate a framing decision ID
        framing_decision_id = FdlIdFramingDecision.model_validate(
            f"{str(self.id)}-{str(framing_intent.id)}"
        )

        # Calculate dimensions and protection based on the framing intent and canvas
        dimensions, anchor, protection_dimensions, protection_anchor = calculate_framing_decision(
            canvas=self, framing_intent=framing_intent
        )

        # Add to canvas
        if not self.framing_decisions:
            self.framing_decisions = []

        # Create the framing decision
        framing_decision = FramingDecision(
            id=framing_decision_id,
            framing_intent_id=framing_intent.id,
            dimensions=dimensions,
            anchor_point=anchor,
            protection_dimensions=protection_dimensions,
            protection_anchor_point=protection_anchor,
            label=label,
        )

        # Use variable assignment to trigger the pydantic validator
        self.framing_decisions = self.framing_decisions + [framing_decision]

        return framing_decision
