"""Context model extension for the ASC Framing Decision List (FDL).

This module provides the `Context` model extension used by the FDL. It supplies a
post-initialisation hook that serves validators and convenience methods creating subsequent models
such as Canvases.
"""

from typing import Iterator, Optional, Union

from pydantic import field_validator

from .canvas import Canvas
from .schema import FdlId, _Context


class Context(_Context):
    """Context model with FDL-specific validation and helpers.

    The class extends the `_Context` schema that is generated by the official FDL schema, with
    helpers needed in FDL workflows.
    """

    @field_validator("canvases")
    @classmethod
    def validate_uniqueness_canvases(
        cls, canvases: Optional[list[Canvas]]
    ) -> Optional[list[Canvas]]:
        """Ensure each canvas in the context has a unique ID.

        Raises:
            ValueError: when a duplicate Canvas ID is detected.
        """
        if canvases is None:
            return None

        seen = set()
        for canvas in canvases:
            canvas_id = canvas.id
            if str(canvas_id) in seen:
                raise ValueError(f"Canvas ID {canvas_id!r} already exists")

            seen.add(str(canvas_id))

        return canvases

    def add_canvas(self, canvas: Canvas) -> Canvas:
        """Append a canvas to this context.

        The implementation uses assignment to trigger the pydantic validator
        (so duplicates will raise ValueError).
        """
        if not self.canvases:
            self.canvases = []

        # Use variable assignment to trigger the pydantic validator
        self.canvases = self.canvases + [canvas]

        return canvas

    def get_canvas_by_id(self, canvas_id: Union[str, FdlId]) -> Canvas:
        """Return the canvas with `canvas_id` from this context.

        Raises:
            ValueError: if the canvas ID is not present in this context.
        """
        canvas_id = FdlId.model_validate(str(canvas_id))

        for canvas in self.canvases or []:
            if canvas.id == canvas_id:
                return canvas

        raise ValueError(f"Canvas ID {canvas_id!r} not found in this context")

    def iter_canvases(self) -> Iterator[Canvas]:
        """Iterator over all canvases in this context."""
        if not self.canvases:
            return

        for canvas in self.canvases or []:
            yield canvas
