import json
from datetime import datetime
from http import HTTPStatus
from uuid import UUID

import pytest
import responses
from responses import matchers

from fiddler3.entities.project import Project
from fiddler3.exceptions import Conflict, NotFound
from fiddler3.tests.constants import ORG_ID, ORG_NAME, PROJECT_ID, PROJECT_NAME, URL

API_RESPONSE_200 = {
    'data': {
        'id': PROJECT_ID,
        'name': PROJECT_NAME,
        'organization': {
            'id': ORG_ID,
            'name': ORG_NAME,
        },
        'created_at': '2023-11-22 16:50:57.705784',
        'updated_at': '2023-11-22 16:50:57.705784',
    }
}

API_RESPONSE_409 = {
    'error': {
        'code': 409,
        'message': 'Project already exists',
        'errors': [
            {
                'reason': 'Conflict',
                'message': 'Project already exists',
                'help': '',
            }
        ],
    }
}

API_RESPONSE_404 = {
    'error': {
        'code': 404,
        'message': 'Project not found for the given identifier',
        'errors': [
            {
                'reason': 'ObjectNotFound',
                'message': 'Project not found for the given identifier',
                'help': '',
            }
        ],
    }
}

API_RESPONSE_FROM_NAME = {
    'data': {
        'page_size': 100,
        'total': 1,
        'item_count': 1,
        'page_count': 1,
        'page_index': 1,
        'offset': 0,
        'items': [API_RESPONSE_200['data']],
    }
}

LIST_API_RESPONSE = {
    'data': {
        'page_size': 100,
        'total': 2,
        'item_count': 2,
        'page_count': 1,
        'page_index': 1,
        'offset': 0,
        'items': [
            API_RESPONSE_200['data'],
            {
                'id': '2531bfd9-2ca2-4a7b-bb5a-136c8da09ca1',
                'name': 'project2',
                'organization': {
                    'id': ORG_ID,
                    'name': ORG_NAME,
                },
                'created_at': '2023-11-22 16:50:57.705784',
                'updated_at': '2023-11-22 16:50:57.705784',
            },
        ],
    }
}


@responses.activate
def test_add_project_success() -> None:
    responses.post(
        url=f'{URL}/v3/projects',
        json=API_RESPONSE_200,
    )
    project = Project(name=PROJECT_NAME).create()
    assert isinstance(project, Project)
    assert project.id == UUID(PROJECT_ID)
    assert project.name == PROJECT_NAME
    assert project.created_at == datetime.fromisoformat(
        API_RESPONSE_200['data']['created_at']
    )
    assert project.updated_at == datetime.fromisoformat(
        API_RESPONSE_200['data']['updated_at']
    )

    assert json.loads(responses.calls[0].request.body) == {'name': PROJECT_NAME}


@responses.activate
def test_add_project_conflict() -> None:
    responses.post(
        url=f'{URL}/v3/projects', json=API_RESPONSE_409, status=HTTPStatus.CONFLICT
    )

    with pytest.raises(Conflict):
        Project(name=PROJECT_NAME).create()


@responses.activate
def test_get_project_success() -> None:
    responses.get(
        url=f'{URL}/v3/projects/{PROJECT_ID}',
        json=API_RESPONSE_200,
    )
    project = Project.get(id_=PROJECT_ID)
    assert isinstance(project, Project)


@responses.activate
def test_get_project_not_found() -> None:
    responses.get(
        url=f'{URL}/v3/projects/{PROJECT_ID}',
        json=API_RESPONSE_404,
        status=HTTPStatus.NOT_FOUND,
    )

    with pytest.raises(NotFound):
        Project.get(id_=PROJECT_ID)


@responses.activate
def test_project_from_name_success() -> None:
    params = {'name': PROJECT_NAME}
    responses.get(
        url=f'{URL}/v3/projects',
        json=API_RESPONSE_FROM_NAME,
        match=[matchers.query_param_matcher(params)],
    )
    project = Project.from_name(name=PROJECT_NAME)
    assert isinstance(project, Project)


@responses.activate
def test_project_list_success() -> None:
    responses.get(
        url=f'{URL}/v3/projects',
        json=LIST_API_RESPONSE,
    )
    for project in Project.list():
        assert isinstance(project, Project)


@responses.activate
def test_project_list_empty() -> None:
    resp = API_RESPONSE_FROM_NAME.copy()
    resp['data']['total'] = 0
    resp['data']['item_count'] = 0
    resp['data']['items'] = []

    responses.get(
        url=f'{URL}/v3/projects',
        json=resp,
    )
    assert len(list(Project.list())) == 0


@responses.activate
def test_project_from_name_not_found() -> None:
    resp = API_RESPONSE_FROM_NAME.copy()
    resp['data']['total'] = 0
    resp['data']['item_count'] = 0
    resp['data']['items'] = []

    params = {'name': PROJECT_NAME}
    responses.get(
        url=f'{URL}/v3/projects',
        json=resp,
        match=[matchers.query_param_matcher(params)],
    )

    with pytest.raises(NotFound):
        Project.from_name(name=PROJECT_NAME)


@responses.activate
def test_delete_project_success() -> None:
    responses.delete(
        url=f'{URL}/v2/projects/{PROJECT_ID}',
        json={},
    )
    project = Project(name=PROJECT_NAME)
    project.id = PROJECT_ID

    project.delete()


@responses.activate
def test_delete_project_not_found() -> None:
    responses.delete(
        url=f'{URL}/v2/projects/{PROJECT_ID}',
        json=API_RESPONSE_404,
        status=HTTPStatus.NOT_FOUND,
    )
    project = Project(name=PROJECT_NAME)
    project.id = PROJECT_ID

    with pytest.raises(NotFound):
        project.delete()
