from pathlib import Path
from uuid import uuid4

import numpy as np
import pandas as pd
import responses

from fiddler.constants.dataset import EnvType
from fiddler.entities.job import Job
from fiddler.entities.model import Model
from fiddler.tests.apis.test_files import SINGLE_UPLOAD_200_RESPONSE
from fiddler.tests.apis.test_model import API_RESPONSE_200 as MODEL_API_RESPONSE_200
from fiddler.tests.constants import JOB_ID, MODEL_ID, URL

FILE_PUBLISH_202_API_RESPONSE = {
    'data': {
        'source_type': 'FILE',
        'job': {'id': JOB_ID, 'name': 'Upload dataset'},
    },
    'api_version': '3.0',
    'kind': 'NORMAL',
}

PUBLISH_JOB_RESPONSE = {
    'data': {
        'id': 'e5784edf-2361-43e1-b67f-3dec90039d9b',
        'name': 'Publish events',
        'info': {
            'env_name': 'production',
            'env_type': 'PRODUCTION',
            'model_name': 'bank_churn',
            'model_uuid': 'd30bc065-a2ca-461e-bdcc-56d47c85d2a3',
            'project_name': 'test_project_5',
            'resource_name': 'bank_churn',
            'resource_type': 'EVENT',
            '__tracker_info': {
                'progress': 0,
                'error_reason': None,
                'error_message': None,
            },
        },
        'status': 'PENDING',
        'progress': 0,
        'error_message': None,
        'error_reason': None,
    },
    'api_version': '3.0',
    'kind': 'NORMAL',
}

df = pd.DataFrame(np.random.randint(0, 100, size=(10, 4)), columns=list('ABCD'))

STREAM_PUBLISH_202_API_RESPONSE = {
    'data': {
        'source_type': 'EVENTS',
        'fiddler_ids': [str(uuid4()) for i in range(df.shape[0])],
    },
    'api_version': '3.0',
    'kind': 'NORMAL',
}

BASE_TEST_DIR = Path(__file__).resolve().parent.parent
FILE_PATH = BASE_TEST_DIR / 'artifact_test_dir' / 'model.yaml'


@responses.activate
def test_publish_file() -> None:
    responses.get(
        url=f'{URL}/v3/models/{MODEL_ID}',
        json=MODEL_API_RESPONSE_200,
    )
    model = Model.get(id_=MODEL_ID)

    responses.post(
        url=f'{URL}/v3/files/upload',
        json=SINGLE_UPLOAD_200_RESPONSE,
    )
    responses.get(
        url=f'{URL}/v3/jobs/{JOB_ID}',
        json=PUBLISH_JOB_RESPONSE,
    )

    responses.post(
        url=f'{URL}/v3/events',
        json=FILE_PUBLISH_202_API_RESPONSE,
    )
    publish_response = model.publish(
        source=FILE_PATH, environment=EnvType.PRE_PRODUCTION, dataset_name='dataset_1'
    )

    assert isinstance(publish_response, Job)

    FILE_PUBLISH_202_API_RESPONSE['data']['job']['name'] = 'Publish events'

    publish_response = model.publish(
        source=FILE_PATH, environment=EnvType.PRODUCTION, dataset_name='dataset_2'
    )

    assert isinstance(publish_response, Job)


@responses.activate
def test_publish_stream() -> None:
    responses.get(
        url=f'{URL}/v3/models/{MODEL_ID}',
        json=MODEL_API_RESPONSE_200,
    )
    model = Model.get(id_=MODEL_ID)

    responses.post(
        url=f'{URL}/v3/events',
        json=STREAM_PUBLISH_202_API_RESPONSE,
    )
    publish_response = model.publish(
        source=df,
        environment=EnvType.PRODUCTION,
    )

    assert isinstance(publish_response, list)
    assert len(publish_response) == df.shape[0]

    publish_response = model.publish(
        source=df.to_dict('records'),
        environment=EnvType.PRODUCTION,
    )

    assert isinstance(publish_response, list)
    assert len(publish_response) == df.shape[0]


@responses.activate
def test_publish_update() -> None:
    responses.get(
        url=f'{URL}/v3/models/{MODEL_ID}',
        json=MODEL_API_RESPONSE_200,
    )
    model = Model.get(id_=MODEL_ID)

    responses.patch(
        url=f'{URL}/v3/events',
        json=STREAM_PUBLISH_202_API_RESPONSE,
    )
    publish_response = model.publish(
        source=df, environment=EnvType.PRODUCTION, update=True
    )
    assert isinstance(publish_response, list)
    assert len(publish_response) == df.shape[0]

    responses.post(
        url=f'{URL}/v3/files/upload',
        json=SINGLE_UPLOAD_200_RESPONSE,
    )
    responses.get(
        url=f'{URL}/v3/jobs/{JOB_ID}',
        json=PUBLISH_JOB_RESPONSE,
    )

    responses.patch(
        url=f'{URL}/v3/events',
        json=FILE_PUBLISH_202_API_RESPONSE,
    )
    publish_response = model.publish(
        source=FILE_PATH,
        environment=EnvType.PRODUCTION,
        dataset_name='dataset_1',
        update=True,
    )

    assert isinstance(publish_response, Job)
