from __future__ import annotations

from collections.abc import Mapping
from typing import Any, TypeVar, cast
from uuid import UUID

from attrs import define as _attrs_define
from attrs import field as _attrs_field

from ..models.standard_type import StandardType
from ..models.web_map_service_level import WebMapServiceLevel
from ..models.web_map_service_type import WebMapServiceType
from ..types import UNSET, Unset

T = TypeVar("T", bound="WebMapService")


@_attrs_define
class WebMapService:
    """
    Attributes:
        web_map_service_id (UUID):
        name (str):
        url (str):
        service_type (WebMapServiceType):
        organization_id (None | UUID): The ID for the organization that owns this web map service.
        project_id (None | UUID): The ID for the project that owns this web map service.
        level (WebMapServiceLevel):
        available_standard_ids (list[StandardType] | Unset):
        description (None | str | Unset):
    """

    web_map_service_id: UUID
    name: str
    url: str
    service_type: WebMapServiceType
    organization_id: None | UUID
    project_id: None | UUID
    level: WebMapServiceLevel
    available_standard_ids: list[StandardType] | Unset = UNSET
    description: None | str | Unset = UNSET
    additional_properties: dict[str, Any] = _attrs_field(init=False, factory=dict)

    def to_dict(self) -> dict[str, Any]:
        web_map_service_id = str(self.web_map_service_id)

        name = self.name

        url = self.url

        service_type = self.service_type.value

        organization_id: None | str
        if isinstance(self.organization_id, UUID):
            organization_id = str(self.organization_id)
        else:
            organization_id = self.organization_id

        project_id: None | str
        if isinstance(self.project_id, UUID):
            project_id = str(self.project_id)
        else:
            project_id = self.project_id

        level = self.level.value

        available_standard_ids: list[str] | Unset = UNSET
        if not isinstance(self.available_standard_ids, Unset):
            available_standard_ids = []
            for available_standard_ids_item_data in self.available_standard_ids:
                available_standard_ids_item = available_standard_ids_item_data.value
                available_standard_ids.append(available_standard_ids_item)

        description: None | str | Unset
        if isinstance(self.description, Unset):
            description = UNSET
        else:
            description = self.description

        field_dict: dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "web_map_service_id": web_map_service_id,
                "name": name,
                "url": url,
                "service_type": service_type,
                "organization_id": organization_id,
                "project_id": project_id,
                "level": level,
            }
        )
        if available_standard_ids is not UNSET:
            field_dict["available_standard_ids"] = available_standard_ids
        if description is not UNSET:
            field_dict["description"] = description

        return field_dict

    @classmethod
    def from_dict(cls: type[T], src_dict: Mapping[str, Any]) -> T:
        d = dict(src_dict)
        web_map_service_id = UUID(d.pop("web_map_service_id"))

        name = d.pop("name")

        url = d.pop("url")

        service_type = WebMapServiceType(d.pop("service_type"))

        def _parse_organization_id(data: object) -> None | UUID:
            if data is None:
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                organization_id_type_0 = UUID(data)

                return organization_id_type_0
            except (TypeError, ValueError, AttributeError, KeyError):
                pass
            return cast(None | UUID, data)

        organization_id = _parse_organization_id(d.pop("organization_id"))

        def _parse_project_id(data: object) -> None | UUID:
            if data is None:
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                project_id_type_0 = UUID(data)

                return project_id_type_0
            except (TypeError, ValueError, AttributeError, KeyError):
                pass
            return cast(None | UUID, data)

        project_id = _parse_project_id(d.pop("project_id"))

        level = WebMapServiceLevel(d.pop("level"))

        _available_standard_ids = d.pop("available_standard_ids", UNSET)
        available_standard_ids: list[StandardType] | Unset = UNSET
        if _available_standard_ids is not UNSET:
            available_standard_ids = []
            for available_standard_ids_item_data in _available_standard_ids:
                available_standard_ids_item = StandardType(available_standard_ids_item_data)

                available_standard_ids.append(available_standard_ids_item)

        def _parse_description(data: object) -> None | str | Unset:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(None | str | Unset, data)

        description = _parse_description(d.pop("description", UNSET))

        web_map_service = cls(
            web_map_service_id=web_map_service_id,
            name=name,
            url=url,
            service_type=service_type,
            organization_id=organization_id,
            project_id=project_id,
            level=level,
            available_standard_ids=available_standard_ids,
            description=description,
        )

        web_map_service.additional_properties = d
        return web_map_service

    @property
    def additional_keys(self) -> list[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties
