/*
 * SPDX-FileCopyrightText: Copyright (c) 1993-2024 NVIDIA CORPORATION &
 * AFFILIATES. All rights reserved. SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#pragma once

#include <flashinfer/trtllm/fmha/kernelParams.h>
// Helper to print Data_type
inline const char* dataTypeToString(Data_type dt) {
  switch (dt) {
    case DATA_TYPE_FP16:
      return "FP16";
    case DATA_TYPE_BF16:
      return "BF16";
    case DATA_TYPE_FP32:
      return "FP32";
    case DATA_TYPE_E4M3:
      return "E4M3";
    default:
      return "UNKNOWN";
  }
}

static const struct TllmGenFmhaKernelMetaInfo {
  Data_type mDataTypeQ;
  Data_type mDataTypeKv;
  Data_type mDataTypeO;
  int mTileSizeQ;
  int mTileSizeKv;
  int mStepQ;
  int mStepKv;
  int mHeadDimPerCtaV;
  int mHeadDimQk;
  int mHeadDimV;
  int mSM;
  const char* mFuncName;
  int mSharedMemBytes;
  int mThreadsPerCTA;
  int mQkvLayout;
  int mNumTokensPerPage;
  int mMaskType;
  int mKernelType;
  int mMaxNumHeadsQPerKvInCta;
  int mTileScheduler;
  int mMultiCtasKvMode;
  bool mGroupsHeadsQ;
  bool mReuseSmemKForV;
  bool m2CtaMma;
  const char* sha256;

  void print() const {
    std::cout << "TllmGenFmhaKernelMetaInfo {\n";
    std::cout << "  mDataTypeQ: " << dataTypeToString(mDataTypeQ) << "\n";
    std::cout << "  mDataTypeKv: " << dataTypeToString(mDataTypeKv) << "\n";
    std::cout << "  mDataTypeO: " << dataTypeToString(mDataTypeO) << "\n";
    std::cout << "  mTileSizeQ: " << mTileSizeQ << "\n";
    std::cout << "  mTileSizeKv: " << mTileSizeKv << "\n";
    std::cout << "  mStepQ: " << mStepQ << "\n";
    std::cout << "  mStepKv: " << mStepKv << "\n";
    std::cout << "  mHeadDimPerCtaV: " << mHeadDimPerCtaV << "\n";
    std::cout << "  mHeadDimQk: " << mHeadDimQk << "\n";
    std::cout << "  mHeadDimV: " << mHeadDimV << "\n";
    std::cout << "  mSM: " << mSM << "\n";
    std::cout << "  mFuncName: " << (mFuncName ? mFuncName : "null") << "\n";
    std::cout << "  mSharedMemBytes: " << mSharedMemBytes << "\n";
    std::cout << "  mThreadsPerCTA: " << mThreadsPerCTA << "\n";
    std::cout << "  mQkvLayout: " << mQkvLayout << "\n";
    std::cout << "  mNumTokensPerPage: " << mNumTokensPerPage << "\n";
    std::cout << "  mMaskType: " << mMaskType << "\n";
    std::cout << "  mKernelType: " << mKernelType << "\n";
    std::cout << "  mMaxNumHeadsQPerKvInCta: " << mMaxNumHeadsQPerKvInCta << "\n";
    std::cout << "  mTileScheduler: " << mTileScheduler << "\n";
    std::cout << "  mGroupsHeadsQ: " << std::boolalpha << mGroupsHeadsQ << "\n";
    std::cout << "  mMultiCtasKvMode: " << std::boolalpha << mMultiCtasKvMode << "\n";
    std::cout << "  mReuseSmemKForV: " << std::boolalpha << mReuseSmemKForV << "\n";
    std::cout << "  m2CtaMma: " << std::boolalpha << m2CtaMma << "\n";
    std::cout << "  sha256: " << (sha256 ? sha256 : "null") << "\n";
    std::cout << "}\n";
  }
} sTllmGenFmhaKernelMetaInfos[] = {
    // clang-format off
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "06fa6e5978cb0d5ed2ec654591b5bc7fe3fabfbc5bd29690dc039f8d2746a15f"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "c20701b2701dad7c0c8b10a5ad85d28f701ee1d432fc7298649fe1f2856cad92"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "31fa72526523245bdfb409115f3e8fb64f5b4670387336efd58bd69c17d83817"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "a19266f6b619a6ab1a92d6befe1b65283d56da8aa188470bc5f5a66549ff0e52"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "569e3f91b1c6e66981270a3ffc8e02432abc03d3ce12d888b2a0f297bd8e5738"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 77456, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "0ef26a988cc53f3106f13d8c36aaee35d8add36ad3e6a9f8073715c20a9e6ac3"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "2618c62778666aa1ca523220ad434c43b8bcf0c9343be958c962e4dfebbd68f3"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 77456, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "bc6c1a3d676a3eeebe82a6e3e510ac1a551e264027c9b962616cb22d762c31cb"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 77456, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "1fee3d9a5184482a262ff77a3f4e1a7fa0467d5c4978d058e0fb3bf0da51ed93"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "34a4520e5e7fa4b5db84378e68e55ae14e5c27b8f6d3bfc57543565aceaf13cc"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 146064, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "811d23e7802b145f4aae906c29365d17a81286d51ea7cea3d704570d69aff342"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "2d1b065c378e3199e727941b89b16ae3a4982e3d66a7ee072848cb57c76a6aa1"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "8590e436d65c2499a9d5840b3fa50d8b465669de94e6b2073f88a12be91c5c28"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 146064, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "a978aed0b8623015b746062470309ecbc42c3a2eed59c44a66b850f0ff39565f"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 110288, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "9b7ea05f7c454ba6c5f04e54cd1572f8b2cf86499fb344ded5eafb7b0f3fc47d"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "b14b1940167f1ed3a261fc529297be5c9b53776f493e69f03e05080148da300e"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "d7f6165de6ccfef24ce3a93c7a1b0c03413a0efbec264263235fbf7e03bbfa5a"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 146064, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "f80baa13ec6d429c5e0c0f1289f5a30d0a63c3eb11a59d6518fcad9a537477f9"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 110288, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "e897500a6b7143a596044fa367c78fe9535fd532941c20b76461587286ddd07d"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "57bd4edd182324bb2b1c49582388776162ef84c995466020be5065d81ca08470"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 110288, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "dc3a6a8270d3a04893b8cb8b74a46f6a9a93cd75598fb5789390a7a0840a2d76"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 175824, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "fd8f382ecec44ae071e091436338ffe584d29c84531792022e3b1780bb231940"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 175824, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "239e86ba1f0baeeb0f3bf3b11bc6928ef79d572bd8827e7a3a3c12dfa144e80a"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 175824, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "d158111df93a0238c159bba98606720590115ea52878591c2d7b292d749915d7"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "812f2a59cdb5cae44190c48b942d236d69cf8339b773988377678ca4a4a92cb2"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "bc39aeaae11aa020c83287160b1b6014e2b4fe021257675c572fdbfca05fcf10"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "9fd0b3aa97803ef16f22cbb81e3420674ff4ee615b3b598a27928b0f44f3b624"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 78480, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "bbe9bc7e879587792db1853cf3fdf1948255aa8747b0ad44ce6ea68d5e7ca104"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 78480, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "4cb2153897823b532f902716ffbef308a1ebe715f2492b91b7df569ecc6c29b6"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 78480, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "5fc48366e358e41d2338ee9aef7ee6e9f6cf511c6acaf991471efdc2bae7cb28"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "fdbfff28d8edb6879b67c0ee57934d470385ab457a6530f05a6f410ad0ab2600"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "30cedef28f8f209bfafbceac05afacfec41edfd3440b9439aeb3064fc93c3bc2"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "1a414868ec41132c612339d8c74dc2fad35ebdd1f11b84eaecf80739c13ba022"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 147088, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "037ca837ef5999fe9008971d0212e89b60db9ecddf7c87dce397400da1593953"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "27f2d359228edd8d7c2bfce7e27894e2ec61beccdfe02a781bf911f5ebe503a6"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 147088, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "f74802234df6267c462b1297ab22f87548e2b6618f63cf9da55f5847307c122d"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "54f3ee9a91c0af8e1469e386ec38b873576d94f3bb562b3968ded615ea834e0c"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "5dbeb51a0b7b5ee98ce85ff99751319302eecd5082af15cf1d89dfdc41183920"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 111312, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "59a27801752534cd8a36895bca7fabd0c51812f8ac9870e73d40a539aa692ca7"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 111312, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "ee9764e7b108694342d43af8e7db6a55d92bf395df82fd5d1c5f313c6fcbe723"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 111312, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "ff1e596a4f7d6a672564b55148a5a8b7734a4a05d3dcd601d78ccbe816102910"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 147088, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "a9f93efd08025cd91d7c4de6a2ad65a9d4d922b05d73397df5b7dae0be7bb71e"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "6d11f5c93487b1942825441ab47c4002bc1eea3edf241d61b11fd74de3fa4d01"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "93064e75f33c39099644afe3f24a0eff33ab03e5046bcf699d8a68303e48727d"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "cb255fdd13369a7036afe2c64e9c0a3851e41475be5887f3bc8c8f544664a004"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 176848, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "16c296b54189d02a2ca17035a0056127b2edf55e61f4e0107c50bb1f1695ea1a"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 176848, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "da7336b69ec7528a4ff3a18315d883e1dd48a307837bf524781ea5c6e76ae3c6"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 176848, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "245cf667f2b50c95b01c57af3c2fb007b9b4bb6badeb3b527e06add6c5b9ef69"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "f54bab2171824957f922a45e7b315410b53f3c06321a501c66f073c974b8cede"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "8152a76471a3482423e3b3c6447bfd35097edad393fd7f8a93ba7c18f318b46b"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "41cb62a923db56efa85eec333ff910b987949dc1f092ca58baaf7387a369e236"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 78480, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "ab02d00ba50c0eb34d099a9ac675297b921b69625440af033124ed3cc10042d3"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 78480, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "42908561922c0291982793623cb6c99ecbaa212e97de9c00036f496aa9ef7acd"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 78480, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "99a8ced65f46dbdf8d13cf8b78624aabfeef644c65dc58c69a0bec000e3ce765"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "22aae8fdb51f2dab97480eef3d0b9f93eb9b5aaa2c66b46fbfaecd140c585428"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "4ba686ce4aebf97834158a730730ad801a6baa9cc97bc4ede3250c3009ca0b14"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "76e6237eff56ef4c4ddb62ee03031cf36097fe2ab60d7214dfe11c30ded2a3bb"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 147088, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "ddd3574d8c9fb759eb2d386167917905d83e005ce2fc15f5159b54dbff6bd32e"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 147088, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "0b0a711eae8380ea977c20a6f445c975be45a9ecc9aaed6b82191329c4de457f"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "c46ae569b8cdc1e19a7b2cdcaaf98efae89d46ac10de3a8dcaba7b7d3b7e1754"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "5337c4c3425402075c6b489a9d6fd7ae286132abbce1a1b642b38de2d3e34974"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 76432, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "49eb2b6574ae6ed4e3d86ce8cd8a305e5e96a6d8a4e2bfe33a8248ebe2d2b91e"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 147088, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "04e1c3a299cee66abd7cd89ab68dbd160b65937fd35c6ba662e42a485f03d051"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "e8932d29b53b01529bc212e9c4e7c9d9620f01630161b3782585882c53f03eaf"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "d2416b17fbaa2c795b60c16b062166bb82e3dca6998b2fdd3d16773ef48c466c"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 142992, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "84b59bbb7e13f0905dcabea7e64a75762e011c5166c088eff3b342dfbae748ac"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 111312, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "2ce2741355f5de6308fe5d61abd2d26eb40e09491233708dee0ec3987cec8ad0"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 111312, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "5a1beb9c12500de35b4260cdd20412afcba66709c3da6083d47ad1a7b527aef7"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 111312, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "bb0db052ff88bfbbf5074599d82ac594c954bc2668068b77260fbab578f7749d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "76d207a71b38034b8511a1644d4ee2f9432ac6b86ad1280caf7bd005852ff80e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "a3e47fe3064740984c9483c7e3f11b5f1c52b2b185b69ca14668d1a102ee2e11"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "1ac300079334d3858fceddad505e6557e6c35bee09b5ac5606af48ab1c1d4554"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64VarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 64, 0, 2, 8, 0, 0, true, false, false, "376893eb34eb3319bd37d049859311cd08dbd71de1cded07a9106342245586da"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32VarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 32, 0, 2, 8, 0, 0, true, false, false, "c6269026936043c4041130bb579dc31972ec36fa8c9af9b545d16aa2b303b996"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 77456, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "cbec9b6543025e5a0cc0918f8fe1e644f1d4e97bb5d5b8b1c30e80a98db2251d"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 176848, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "616cdc6d1f0ee5fa81f3aee7ec2fe2baab6effacca1844313a016f926431813e"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 176848, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "5677403cfe8fb7d1223be9a0aadd03849f7863c2fb72b4708f5166a46d24b0db"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 77456, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "cebfab6399c2feb9082278c1f9348135b8a33666b474a71f39c302233e985ffc"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 176848, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "224321b048f3e34ca4d91645756dee72638971c228c711bd876a250487d6154e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16VarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 16, 0, 2, 8, 0, 0, true, false, false, "dc01a8706a732f184fe4a138231874b9b4a92d891b641a1be84c6688aa2a6919"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 77456, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "4323c6f5b428600b37e84333616cb768c3f27b17fcd60f263ebdc767783fb4c7"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64VarSeqQ8Kv128PersistentSwapsAbForGen", 146064, 512, 2, 64, 0, 2, 8, 1, 0, true, false, false, "c80e5d3524eabcdec602d001501ae63eb3399e2190b2a774b245f7e0baf30f55"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32VarSeqQ8Kv128PersistentSwapsAbForGen", 146064, 512, 2, 32, 0, 2, 8, 1, 0, true, false, false, "109cf87c1d8b31d3bdf8880a56c1cd01f68054cb04befba635ffb707310c57be"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16VarSeqQ8Kv128PersistentSwapsAbForGen", 146064, 512, 2, 16, 0, 2, 8, 1, 0, true, false, false, "afd97e66299276a460826f9edf95cd01d086d220ac87dcf72552c71bc10271e5"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "d41a34801c44eaf0bf7c8b3489555d437a500dfd4d0b0543fcf61f8ce7b68c72"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "1018267bfafcb91b06e678fc6d42724d4ef79f1669b76720b09bfeda0c950329"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 75408, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "d903fa968e37fa519f84fccca20b3f85899cfb4b2dd108ca82dba741b91b719e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 16, 0, 2, 8, 0, 1, true, false, false, "51c5c6a79e3b69e15ca03df8b6a9da24b9a4bea700538d0443e6f684d6cc6838"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 32, 0, 2, 8, 0, 1, true, false, false, "4c40df655f763a1b7182b15417f99540d5cbc0b2722b782bb5b1debc4214afdd"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64MultiCtasKvVarSeqQ8Kv128StaticSwapsAbForGen", 141968, 512, 2, 64, 0, 2, 8, 0, 1, true, false, false, "7853f10fb0aea65f847c050deceac0d9e049730ea4246484778b5fd1a2f0ae27"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 110288, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "fbeb4f3d9cf50389f2093ef9e00604f358ed84917446b8d8609f3bfacd323f8b"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 110288, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "87d72bc1fe3b3391ecf32a2bddb579bd7ababe289791fbd700ac3014a4533875"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 110288, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "5515784b3699722c476354af6909050a665b0549179cee0b1687971ec4457b14"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 175824, 512, 2, 32, 0, 2, 8, 0, 2, true, false, false, "daaaa9801978721a9b10386cef5a6ef28a2396dc3e74441855400864b0751add"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 175824, 512, 2, 16, 0, 2, 8, 0, 2, true, false, false, "4c647114210cd1a71bf204107f68958b1cbd29f3d7f688d4ed032771830ae856"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 8, 128, 8, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ8Kv128StaticSwapsAbForGen", 175824, 512, 2, 64, 0, 2, 8, 0, 2, true, false, false, "5adb016db46ff70f0f7c06ea0ffb4c0da188f331bb975747dc3e422fe0faca11"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 86032, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "e25d44ada0b07f42a34bffcb41784ba2b3d5569125654c12b7e4310c172c31ea"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 86032, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "d3cfe16e0d0f70f7e69786ff451035e567bd8c11fa2ce214c9ae137bad43a98b"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 86032, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "ac22a5b75a4134364f258012fbe5eeb71cedcc93646cb96937cb9b323699e0c7"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 167952, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "0b722badeddb2186c97f563c6e84d1fe382394c0fec9f01ac95badf35f11b16d"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 102416, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "b5b606dd330b92c13f2eb80376c76fbf9562244c90603737f54d233ccebe0b6d"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 167952, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "7e723de9604b803cc9af1ecdbf739596c8bc10c3a9d228436d320c662f643086"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 120912, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "469f817a1679bc7f86b7aedf299924747a914e252b1e5b31c41c3225d9c0ea49"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 86048, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "1966fa6aad45352c8a027cbe399b5c2b60e69fc27814bbb182154ac9071ae0f8"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 102416, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "d919ee41ad1ffd27b32155c11bfb3cfb7facdd38ceecb04d7cb612002dfebf53"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 102416, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "eabe31d3fb4187ff327194456be8c529047d1fe58572f82dc114f51e509fb3d5"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 200720, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "31604f918bb0341785e2fd980dc14d9778bf8dbd525022871d342e23729697d5"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 86048, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "6df3e0e258056b31b0828b458c59a552d4efe8089f2aa7d4cd787bd852925fc1"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 120912, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "8dc74c2981afbb88657524168d139b2d12417130fb24a8c241c9a6407e6764be"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 167952, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "cd4ae09ed4ba9091714390bc72aaef59c23fb31ba8fcf1d9f477c7ca5c49e54a"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 200720, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "52cc71524f96b6194c71fc3462a8e71ecbb45491ee963d80d082765e327add39"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 120912, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "d8ad4d8a28df85aa17d77a05e27d4b4214210b56b15a7cbd15e8c165353f5a67"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 200720, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "4676c28ab436b44947e123fec189f1a0519dacf59f30b82b9e744caf3d693440"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H64PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 86048, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "d6df9b36e4ae2d7ab533ebd699bac6eb14a6bc2d39cd8492fe9721c46e5f6b55"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 167968, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "eb6f03014cafca66df265fa6da2d4af7b9186f12c5c358adcdd0f05a94b6480e"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 231376, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "7753c22b2fed9f0c5cd32fc9c236fac24fb593e1a3804e84fe6257d51e699305"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 167968, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "48a6ee86ffcedacaf6a47b236383bcea0f42a8c5729ede9eab53db9e956224c0"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 231376, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "8575ccf5f47dc296a0fa5bf3d74323fb2efee06f97600eb6cc3f6564d7f1896e"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 167968, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "c9bd39581f7127b795a71cd9feb4899cb7f6677e58c62cb8cea5f73b1c825df5"},
{ DATA_TYPE_FP16, DATA_TYPE_FP16, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvFp16OFp16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 231376, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "b97ca758b697395aec4dfe45e365d5fe4f8c2fd3e4636a79c5db37b95f032486"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 87056, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "2c754f3963b65524c2e51eb50c2b2345ca6de98cda062b464e43afe6f7eb54da"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 87056, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "dd98e99786e6d829ae4c8fc9c4228c317bb637fdd4d7bdadfce2ac17b5dc6814"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 87056, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "a3d158dbcbe95620440664edf17211a933c8be60ce8a90581c856c8d502ec7bf"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 103440, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "0f6c06bb6f5b63928a7ca10302a4352b859aadab16945ee7c8ecd1f0c722bd94"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 103440, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "6907631a161c313ad949407f218bfd40f63aee655a56cbb735bfe3ef1672f053"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 103440, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "0b12536d03e6ef1429e23c00814c56ac146d3230dd9961541b3337b7536acf32"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 168976, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "b311daf4d5c0626f2396bab6b2f983797f0607e4f9a133b069999e570e0eddf7"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 168976, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "8207d12cb7655670d6ce190b41212fd1206f94c18dd524394fbecf45b6bb127c"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 87072, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "af30fb7e994784b473f4136413d80fdf1b7303e30083c9015f719e9c4ab24df5"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 168976, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "9fa93a60cb97a6260334ea5e48e9f5a1df5f007cb42c5bf95541386025714ce5"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 121936, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "e02ec9f9fa2e5ebbb1f1a7a83b664bcad4396f0945d05b4a5dca3317bf2458c9"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 87072, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "ab0aeae1e1ab9f51f3c0acf0d8d5ed0267f688d31f801c9d68d0e924416ad837"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 201744, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "0f1120046c9d4f9e10dd0a2133bcb8223f5599f1fc079492770bd00033c50c4f"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 121936, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "6118efac6b515008b2229ec40449e9cd4170e45a6505f8b46fdd8c944eb90098"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 201744, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "6efa59d389bd4d4b6870f9ede5145d5f1a5b27d28b98cc4346fce14143f357d7"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 87072, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "40592346bd2c7d0e7667e599ecb96813910154d97d2465fa92dd637c2a10e6ba"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 121936, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "9a7622127a8f6dea8da48d6b3480343d8204ea566dfb95b9270791a237aca0fb"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 87056, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "2b031c3df4b674b8ef3a3b8fa099b99680599853d46b6cbb65fd030e4585ebb9"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 201744, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "a35a3356faf0c17b8d485361e03c47c90dfe3253073b92d39bfd525a69d2e8db"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 87056, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "e784850c655ab4d9ab73b109b5d957737e590f4f8b289c0b2436f9ec6a498f92"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 87056, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "726ebd60573a00de132636b77b0a24ff6608276c824a8f9e3cadc5581fca28aa"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 103440, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "867f72233b2cf60ce0df46db4d57a7c8e363fe94a36609b210eba65448d753d6"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 103440, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "78daf628e83527b9c50d1432ac236b6b8c9c0f3e65e94530f7fa31995e8cc5cb"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 168992, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "0203068d68437fe8100f0ca89bed6b4710f11aa74771e8e7d7c1abe1810645f1"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 168992, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "658cc34edf9eb2657ae7a18f72f91e2d4f381b31e9ea7a3bfa42b04812136a12"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 232400, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "661be578c1d4195e0263251523f001ad02572e7aed3d69a7bd4b68ee63c95828"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 232400, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "36a36ce05c6966a2d5d729ce723970a5080e917e2775079dd9c465f755a5e4ad"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 168992, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "b1eb727007f9b23172bfdf122925e6dac628e687759ec73ee261481990e07299"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 103440, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "ef29ee7957ab0c00f4e8a5f9a16cf4ca92e68f615d84dc2180562446778f9e7b"},
{ DATA_TYPE_FP16, DATA_TYPE_E4M3, DATA_TYPE_FP16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QFp16KvE4m3OFp16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 232400, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "71ae7b217736a329abf992fbc8eb48766677d2ad605ed296bc0a20a217f0664c"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 168976, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "54e8462156fad09311a25c5327635dc04e1425450355c4feb827e506c740f508"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 168976, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "df50c9d738798db93b24cdf0850a6a463816b5c352f63abbcbe2d99819ed0505"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 168976, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "e85a6781698f206d3d75398109fea670437a4c6828247810294edc8bbe18ad37"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 87072, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "55af4d3f4a33d1262341e2cf986030aaf6fafc485f408a954e798af6811d6d64"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 87072, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "8837c31dce89517d475af4804a67f1743ffe0eb5b072158bb116940ab10c6c6e"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 87072, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "098e901827f616d82f319ea48fb12de23744498f91f0619bb421582779caf4ee"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 201744, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "371059b4c0003fe5b11a2f43310d65c2c393ba45906f9cacede1698f86bc3182"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 201744, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "532baf768c2f815153297c0355544e6f01304ab8c5f8b2956a2b207902feab59"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 201744, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "536bc8a4569067f7a363a4ab1c2ee2033a3fc0fe942b1bac945bad102cf7fdef"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 168992, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "129c6adbb74be751067a5388ed3c58d2fa17fce87f5f6440f3305e3049e32354"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 86032, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "a51bda68bed8ad9d6d31163493519c5e238be00102b6e9f73288903c3bdcd61b"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 86032, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "0f1b1b50722b89b9189da966c758c20cedbc1ed8e5035eb16bc712bbd62f5f3b"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 86032, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "122f121da3fa5549ebbf7a7f57eaa6766fcc70194527ae89fc945604e142977f"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64VarSeqQ128Kv128StaticKeepsAbForGen", 167952, 512, 2, 64, 0, 3, 128, 0, 0, true, false, false, "79a6b1c79c7f0c965233155c1912bb76b9e09ef358483c878dc5a55786d8b3a3"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 102416, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "7f427cda2b677acc7ff200db4c23a352ef86005f09ed7ea1c9f388ed4305994c"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 102416, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "f9030525e973c8c2eb57b7081757350158cdd96db878b529765c4736f9d7c1c1"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 102416, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "952bab02f4f3a78956ef906baf83173c6093f14e1ad7cd01145df689bbdbfa0d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32VarSeqQ128Kv128StaticKeepsAbForGen", 167952, 512, 2, 32, 0, 3, 128, 0, 0, true, false, false, "2f2635860e42a69d67b74bb6eb5ea83cafbe13d719623ccc4e6e0334ed35a954"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 121936, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "6a67e260208c21c5b374c631f25ffbfdf622e339674660fbcb43e29a1ac71d58"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 121936, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "cec2ab69bac9adc0f0cf598daf5068474e186b1799a3693c3b8791357db56b8f"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 121936, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "50bcf18d55e5f67075c10e5bb6ab20a92ca866bd601526eb17f5dd2f0c338920"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16VarSeqQ128Kv128StaticKeepsAbForGen", 167952, 512, 2, 16, 0, 3, 128, 0, 0, true, false, false, "39c38f44c38b94e9b328d767cab82fa7a37860966ccbcb9c35ae63ee2d610673"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 86048, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "95d2320ae2bb255a520188fadf4fa3408107f3f6f67729a67e2f5393f51a5b50"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64VarSeqQ128Kv128PersistentKeepsAbForGen", 200720, 512, 2, 64, 0, 3, 128, 1, 0, true, false, false, "61704184c62ea32b2d0b317a83375ababdcca1d247ab978163d5551c4dd46eae"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32VarSeqQ128Kv128PersistentKeepsAbForGen", 200720, 512, 2, 32, 0, 3, 128, 1, 0, true, false, false, "19ac93434ad4ab6319183be977349039ab89a5757f894c87bc21b34f355d2aa7"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 86048, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "e5a4102c8f14d7299babc8b42566c149f516a54b08f9f03e7990d0ff214db7ec"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16VarSeqQ128Kv128PersistentKeepsAbForGen", 200720, 512, 2, 16, 0, 3, 128, 1, 0, true, false, false, "9f7ec23a4782b050ac0702a068cac2a0e42523ebc43cef5b0f8f5eb9d0d6c510"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 168992, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "72b0851859f97c81d47f1bbba6932b272688095ad20cd9d014702b744b03dea0"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 86048, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "55aeb28b36cb8574e0dac1be0295f018c443e51e621b6776588ed068ed7de2cf"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 168992, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "38f2138c379600e22a56cfaec2b3903ccd4e5a6bb4b7a58fbea0694d2fe81c52"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 232400, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "bd70a01eed4e207e76d60283bc59506596e76065239b43b21e9d9d800963f9a7"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 232400, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "6fa03f57459c973ceb130acca022e9be3c88ea79d600a53178efdbf40cd5eddf"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 167968, 512, 2, 16, 0, 3, 128, 0, 1, true, false, false, "67990c64f07e01da376da901d9b7674842eeb6d18fba35ccb3892b154e02e10b"},
{ DATA_TYPE_BF16, DATA_TYPE_E4M3, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QBfloat16KvE4m3OBfloat16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 232400, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "996c63ee13c2979173b5b755ec592b87dbcf8be4b7ff32234f2e95788b0c389e"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 120912, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "5cc1730f6ca989dd5ea88604eb58e1761bb2be8d56d7ce763bc25a5306ea453d"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 167968, 512, 2, 64, 0, 3, 128, 0, 1, true, false, false, "4c1e7272bfaef0528c4f3b40a1a94fad9a44c11fcff0202f9a8e2aac515c06fe"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32MultiCtasKvVarSeqQ128Kv128StaticKeepsAbForGen", 167968, 512, 2, 32, 0, 3, 128, 0, 1, true, false, false, "cd33dda14afbb9bdd59474638f52eb38dbc20fc416cdae21d506be37df2f4034"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 120912, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "043156f29a7cb05c3f1cbc511eb5072fb431e03cb5c6e525685be4193f47c169"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 64, 64, 64, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H64PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 120912, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "5bf7a80fda35614ffb5ea4cbbbf814fd0cec0846a29920495335fe465110185c"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP16MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 231376, 512, 2, 16, 0, 3, 128, 0, 2, true, false, false, "20244b8ed7a9c026f70c61788858cb83964add8eb7867717e2cbe4b835ec1067"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP32MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 231376, 512, 2, 32, 0, 3, 128, 0, 2, true, false, false, "98a9c0b2ed4fdacde0ff66e6be9c45c048e7f0ec4cb50af2c4a5b1181c5b3611"},
{ DATA_TYPE_BF16, DATA_TYPE_BF16, DATA_TYPE_BF16, 128, 128, 128, 256, 128, 128, 128, kSM_100, "fmhaSm100Kernel_QkvBfloat16OBfloat16H128PagedKvDenseP64MultiCtasKvCgaVarSeqQ128Kv128StaticKeepsAbForGen", 231376, 512, 2, 64, 0, 3, 128, 0, 2, true, false, false, "4f409bd224bdd24f95caec2e74a5dcfdf56b3e310375c093313d947c7beb9428"},

    // clang-format on
};
