#!/usr/bin/env python3
#------------------------------------------------------------------------------#
#  fortnet-python: Python Tools for the Fortnet Software Package               #
#  Copyright (C) 2021 T. W. van der Heide                                      #
#                                                                              #
#  See the LICENSE file for terms of usage and distribution.                   #
#------------------------------------------------------------------------------#


'''
Regression tests covering the Fnetout class of Fortformat.
'''


import os
import pytest
import numpy as np

from common import compare_fnetout_references


REFPATH = os.path.join(os.getcwd(), 'test', 'references', 'Fnetout')


def test_predict_atomic():
    '''Test extraction capabilities for a prediction run
       with a network that was trained on atomic targets.
    '''

    fname = 'predict_atomic.hdf5'

    ref = {}
    ref['mode'] = 'predict'
    ref['ndatapoints'] = 5
    ref['targettype'] = 'atomic'
    ref['targets'] = None
    ref['tforces'] = False
    ref['forces'] = None
    ref['predictions'] = [
        np.array([[1.961575401201565427e-01, 9.168128808877051839e-01],
                  [1.325239781646761206e-01, 7.994346410064820940e-01],
                  [1.826092611054506987e-01, 8.918864627286081648e-01],
                  [1.951603716977679814e-01, 9.149779051068115399e-01],
                  [1.963975544054146483e-01, 9.172546297234291934e-01],
                  [1.365085697599923986e-01, 8.068187835637852245e-01],
                  [1.937271428648690563e-01, 9.123404738385268997e-01],
                  [1.963833753374974733e-01, 9.172283491672438283e-01],
                  [-2.963259061179163711e-01, 6.622931487753776381e+00],
                  [-3.116645694102148090e-01, 6.341542248977436458e+00],
                  [-2.954852994924470622e-01, 6.639489278084699464e+00],
                  [-3.046303752343871851e-01, 6.455384967114186523e+00]],
                 dtype=float),
        np.array([[1.811418904020697107e-01, 8.890399580545689240e-01],
                  [1.286134726005213336e-01, 7.921870956352004001e-01],
                  [1.287072680065694807e-01, 7.923610013248644224e-01],
                  [1.285878019428332852e-01, 7.921394561667119971e-01],
                  [-3.205833278148639831e-01, 6.199868006587744951e+00],
                  [-3.205832449473826062e-01, 6.199870243635043465e+00]],
                 dtype=float),
        np.array([[1.508316035937055932e-01, 8.333084902706219266e-01],
                  [1.963987299989748136e-01, 9.172568038424152581e-01],
                  [1.963985352644728455e-01, 9.172564425915140651e-01],
                  [1.314458979434688091e-01, 7.974318952109518133e-01],
                  [1.959840207934034628e-01, 9.164924149116437935e-01],
                  [1.962475111339566924e-01, 9.169785285430018806e-01],
                  [1.963735428400687211e-01, 9.172103673056410944e-01],
                  [1.692361060177546561e-01, 8.672524620359242098e-01],
                  [-2.953595347026437556e-01, 6.642087650077651340e+00],
                  [-3.151594350113108844e-01, 6.282255421963240494e+00],
                  [-2.991868120084945071e-01, 6.559077847747195378e+00],
                  [-3.170787084631181418e-01, 6.252835565560094011e+00]],
                 dtype=float),
        np.array([[1.304479687184249281e-01, 7.955871276861898878e-01],
                  [1.297462265528342706e-01, 7.942881684589961910e-01],
                  [1.298443617239196379e-01, 7.944708584405727470e-01],
                  [1.961872820312715870e-01, 9.168651269507970270e-01],
                  [-3.205789586106497779e-01, 6.199943703977714549e+00],
                  [-3.205781729831197469e-01, 6.199947713843369179e+00]],
                 dtype=float),
        np.array([[1.288099388080513885e-01, 7.925517780736619500e-01],
                  [1.286199169387698682e-01, 7.921996037242402533e-01],
                  [1.286878255987483899e-01, 7.923246429757131448e-01],
                  [1.312376406171068266e-01, 7.970445915261700209e-01],
                  [-3.205835576648750629e-01, 6.199865084107108792e+00],
                  [-3.205822580166140523e-01, 6.199887555086769808e+00]],
                 dtype=float)]

    equal = compare_fnetout_references(ref, os.path.join(REFPATH, '_' + fname))

    assert equal


def test_predict_global():
    '''Test extraction capabilities for a prediction run
       with a network that was trained on global targets.
    '''

    fname = 'predict_global.hdf5'

    ref = {}
    ref['mode'] = 'predict'
    ref['ndatapoints'] = 5
    ref['targettype'] = 'global'
    ref['targets'] = None
    ref['tforces'] = False
    ref['forces'] = None
    ref['predictions'] = [
        np.array([-1.526436789762218496e+02], dtype=float),
        np.array([-4.585193773117663341e+02], dtype=float),
        np.array([-2.290754290677185736e+02], dtype=float),
        np.array([-6.877477714671086915e+02], dtype=float),
        np.array([-5.349057545062817098e+02], dtype=float)]

    equal = compare_fnetout_references(ref, os.path.join(REFPATH, '_' + fname))

    assert equal


def test_predict_global_singleforces():
    '''Test extraction capabilities for a prediction run with a network
       that was trained on global targets and calculates atomic forces.
    '''

    fname = 'predict_global_singleforces.hdf5'

    ref = {}
    ref['mode'] = 'predict'
    ref['ndatapoints'] = 2
    ref['targettype'] = 'global'
    ref['targets'] = None
    ref['tforces'] = True
    ref['forces'] = []
    ref['forces'].append([])
    ref['forces'].append([])
    ref['forces'][0].append(np.array([
        [-1.129280561189105470e+00, 0.000000000000000000e+00,
         0.000000000000000000e+00],
        [1.129280561189105470e+00, 0.000000000000000000e+00,
         0.000000000000000000e+00]], dtype=float))
    ref['forces'][1].append(np.array([
        [-8.464270111301352983e-01, 0.000000000000000000e+00,
         0.000000000000000000e+00],
        [8.464270111301352983e-01, 0.000000000000000000e+00,
         0.000000000000000000e+00]], dtype=float))
    ref['predictions'] = [
        np.array([-4.301790810131604914e-01], dtype=float),
        np.array([-5.025593389423121948e-01], dtype=float)]

    equal = compare_fnetout_references(ref, os.path.join(REFPATH, '_' + fname))

    assert equal


def test_predict_global_multiforces():
    '''Test extraction capabilities for a prediction run with a network
       that was trained on global targets and calculates atomic forces.
    '''

    fname = 'predict_global_multiforces.hdf5'

    ref = {}
    ref['mode'] = 'predict'
    ref['ndatapoints'] = 2
    ref['targettype'] = 'global'
    ref['targets'] = None
    ref['tforces'] = True
    ref['forces'] = []
    ref['forces'].append([])
    ref['forces'].append([])
    ref['forces'][0].append(np.array([
        [-1.113504383113195217e+00, 0.000000000000000000e+00,
         0.000000000000000000e+00],
        [1.113504383113195217e+00, 0.000000000000000000e+00,
         0.000000000000000000e+00]], dtype=float))
    ref['forces'][0].append(np.array([
        [-1.117387033151562292e+00, 0.000000000000000000e+00,
         0.000000000000000000e+00],
        [1.117387033151562292e+00, 0.000000000000000000e+00,
         0.000000000000000000e+00]], dtype=float))
    ref['forces'][0].append(np.array([
        [-1.110108965167277972e+00, 0.000000000000000000e+00,
         0.000000000000000000e+00],
        [1.110108965167277972e+00, 0.000000000000000000e+00,
         0.000000000000000000e+00]], dtype=float))
    ref['forces'][1].append(np.array([
        [-8.450938994823964379e-01, 0.000000000000000000e+00,
         0.000000000000000000e+00],
        [8.450938994823964379e-01, 0.000000000000000000e+00,
         0.000000000000000000e+00]], dtype=float))
    ref['forces'][1].append(np.array([
        [-8.465140042623886529e-01, 0.000000000000000000e+00,
         0.000000000000000000e+00],
        [8.465140042623886529e-01, 0.000000000000000000e+00,
         0.000000000000000000e+00]], dtype=float))
    ref['forces'][1].append(np.array([
        [-8.438788427604926312e-01, 0.000000000000000000e+00,
         0.000000000000000000e+00],
        [8.438788427604926312e-01, 0.000000000000000000e+00,
         0.000000000000000000e+00]], dtype=float))
    ref['predictions'] = [
        np.array([-4.304246998683396441e-01, -4.302864774322330277e-01,
                  -4.305433861504512905e-01], dtype=float),
        np.array([-5.022394949529731534e-01, -5.022869347972704901e-01,
                  -5.021969559503443037e-01], dtype=float)]

    equal = compare_fnetout_references(ref, os.path.join(REFPATH, '_' + fname))

    assert equal


def test_validate_atomic():
    '''Test extraction capabilities for a validation run
       with a network that was trained on atomic targets.
    '''

    fname = 'validate_atomic.hdf5'

    ref = {}
    ref['mode'] = 'validate'
    ref['ndatapoints'] = 5
    ref['targettype'] = 'atomic'
    ref['tforces'] = False
    ref['forces'] = None
    ref['targets'] = [
        np.array([
            [1.540549993515014648e-01, 8.459450006484985352e-01],
            [1.883080005645751953e-01, 8.116919994354248047e-01],
            [1.595949977636337280e-01, 8.404050022363662720e-01],
            [1.432220041751861572e-01, 8.567779958248138428e-01],
            [1.232710033655166626e-01, 8.767289966344833374e-01],
            [1.735100001096725464e-01, 8.264899998903274536e-01],
            [1.588409990072250366e-01, 8.411590009927749634e-01],
            [1.403059959411621094e-01, 8.596940040588378906e-01],
            [-2.634609937667846680e-01, 6.263460993766784668e+00],
            [-3.214380145072937012e-01, 6.321438014507293701e+00],
            [-3.043099939823150635e-01, 6.304309993982315063e+00],
            [-3.519429862499237061e-01, 6.351942986249923706e+00]],
                 dtype=float),
        np.array([
            [1.272429972887039185e-01, 8.727570027112960815e-01],
            [1.549790054559707642e-01, 8.450209945440292358e-01],
            [1.774729937314987183e-01, 8.225270062685012817e-01],
            [1.796700060367584229e-01, 8.203299939632415771e-01],
            [-3.525030016899108887e-01, 6.352503001689910889e+00],
            [-2.868520021438598633e-01, 6.286852002143859863e+00]],
                 dtype=float),
        np.array([
            [1.852180063724517822e-01, 8.147819936275482178e-01],
            [1.311800032854080200e-01, 8.688199967145919800e-01],
            [1.232030019164085388e-01, 8.767969980835914612e-01],
            [1.774370074272155762e-01, 8.225629925727844238e-01],
            [1.587480008602142334e-01, 8.412519991397857666e-01],
            [1.444180011749267578e-01, 8.555819988250732422e-01],
            [1.365029960870742798e-01, 8.634970039129257202e-01],
            [1.802569925785064697e-01, 8.197430074214935303e-01],
            [-2.689329981803894043e-01, 6.268932998180389404e+00],
            [-3.368290066719055176e-01, 6.336829006671905518e+00],
            [-3.142969906330108643e-01, 6.314296990633010864e+00],
            [-3.169249892234802246e-01, 6.316924989223480225e+00]],
                 dtype=float),
        np.array([
            [1.770180016756057739e-01, 8.229819983243942261e-01],
            [1.812230050563812256e-01, 8.187769949436187744e-01],
            [1.482979953289031982e-01, 8.517020046710968018e-01],
            [9.460300207138061523e-02, 9.053969979286193848e-01],
            [-2.429430037736892700e-01, 6.242943003773689270e+00],
            [-3.581880033016204834e-01, 6.358188003301620483e+00]],
                 dtype=float),
        np.array([
            [1.596090048551559448e-01, 8.403909951448440552e-01],
            [1.659840047359466553e-01, 8.340159952640533447e-01],
            [1.713179945945739746e-01, 8.286820054054260254e-01],
            [1.658540070056915283e-01, 8.341459929943084717e-01],
            [-3.264440000057220459e-01, 6.326444000005722046e+00],
            [-3.363139927387237549e-01, 6.336313992738723755e+00]],
                 dtype=float)]
    ref['predictions'] = [
        np.array([
            [1.961575401201565427e-01, 9.168128808877051839e-01],
            [1.325239781646761206e-01, 7.994346410064820940e-01],
            [1.826092611054506987e-01, 8.918864627286081648e-01],
            [1.951603716977679814e-01, 9.149779051068115399e-01],
            [1.963975544054146483e-01, 9.172546297234291934e-01],
            [1.365085697599923986e-01, 8.068187835637852245e-01],
            [1.937271428648690563e-01, 9.123404738385268997e-01],
            [1.963833753374974733e-01, 9.172283491672438283e-01],
            [-2.963259061179163711e-01, 6.622931487753776381e+00],
            [-3.116645694102148090e-01, 6.341542248977436458e+00],
            [-2.954852994924470622e-01, 6.639489278084699464e+00],
            [-3.046303752343871851e-01, 6.455384967114186523e+00]],
                 dtype=float),
        np.array([
            [1.811418904020697107e-01, 8.890399580545689240e-01],
            [1.286134726005213336e-01, 7.921870956352004001e-01],
            [1.287072680065694807e-01, 7.923610013248644224e-01],
            [1.285878019428332852e-01, 7.921394561667119971e-01],
            [-3.205833278148639831e-01, 6.199868006587744951e+00],
            [-3.205832449473826062e-01, 6.199870243635043465e+00]],
                 dtype=float),
        np.array([
            [1.508316035937055932e-01, 8.333084902706219266e-01],
            [1.963987299989748136e-01, 9.172568038424152581e-01],
            [1.963985352644728455e-01, 9.172564425915140651e-01],
            [1.314458979434688091e-01, 7.974318952109518133e-01],
            [1.959840207934034628e-01, 9.164924149116437935e-01],
            [1.962475111339566924e-01, 9.169785285430018806e-01],
            [1.963735428400687211e-01, 9.172103673056410944e-01],
            [1.692361060177546561e-01, 8.672524620359242098e-01],
            [-2.953595347026437556e-01, 6.642087650077651340e+00],
            [-3.151594350113108844e-01, 6.282255421963240494e+00],
            [-2.991868120084945071e-01, 6.559077847747195378e+00],
            [-3.170787084631181418e-01, 6.252835565560094011e+00]],
                 dtype=float),
        np.array([
            [1.304479687184249281e-01, 7.955871276861898878e-01],
            [1.297462265528342706e-01, 7.942881684589961910e-01],
            [1.298443617239196379e-01, 7.944708584405727470e-01],
            [1.961872820312715870e-01, 9.168651269507970270e-01],
            [-3.205789586106497779e-01, 6.199943703977714549e+00],
            [-3.205781729831197469e-01, 6.199947713843369179e+00]],
                 dtype=float),
        np.array([
            [1.288099388080513885e-01, 7.925517780736619500e-01],
            [1.286199169387698682e-01, 7.921996037242402533e-01],
            [1.286878255987483899e-01, 7.923246429757131448e-01],
            [1.312376406171068266e-01, 7.970445915261700209e-01],
            [-3.205835576648750629e-01, 6.199865084107108792e+00],
            [-3.205822580166140523e-01, 6.199887555086769808e+00]],
                 dtype=float)]

    equal = compare_fnetout_references(ref, os.path.join(REFPATH, '_' + fname))

    assert equal


def test_validate_global():
    '''Test extraction capabilities for a validation run
       with a network that was trained on global targets.
    '''

    fname = 'validate_global.hdf5'

    ref = {}
    ref['mode'] = 'validate'
    ref['ndatapoints'] = 5
    ref['targettype'] = 'global'
    ref['targets'] = True
    ref['tforces'] = False
    ref['forces'] = None
    ref['targets'] = [
        np.array([-1.527736989418316114e+02], dtype=float),
        np.array([-4.584216715420000128e+02], dtype=float),
        np.array([-2.291870019319999869e+02], dtype=float),
        np.array([-6.876760346160000381e+02], dtype=float),
        np.array([-5.348338707069999600e+02], dtype=float)]
    ref['predictions'] = [
        np.array([-1.526436789762218496e+02], dtype=float),
        np.array([-4.585193773117663341e+02], dtype=float),
        np.array([-2.290754290677185736e+02], dtype=float),
        np.array([-6.877477714671086915e+02], dtype=float),
        np.array([-5.349057545062817098e+02], dtype=float)]

    equal = compare_fnetout_references(ref, os.path.join(REFPATH, '_' + fname))

    assert equal


if __name__ == '__main__':
    pytest.main()
