import json

import requests

from frinx.common.frinx_rest import CREATE_SNAPSHOT_URL
from frinx.common.frinx_rest import DELETE_SNAPSHOT_URL
from frinx.common.frinx_rest import REPLACE_CONFIG_WITH_SNAPSHOT_URL
from frinx.common.frinx_rest import UNICONFIG_HEADERS
from frinx.common.frinx_rest import UNICONFIG_URL_BASE
from frinx.common.type_aliases import DictAny
from frinx.common.util import normalize_base_url


def create_snapshot(
    node_ids: list[str],
    snapshot_name: str,
    transaction_id: str,
    uniconfig_server_id: str | None = None,
    uniconfig_url_base: str | None = None
) -> requests.Response:
    """
    Create Uniconfig snapshot.
    https://docs.frinx.io/frinx-uniconfig/user-guide/uniconfig-operations/snapshot-manager/rpc_create-snapshot/
    Args:
        node_ids: List of node ids to create snapshot from.
        snapshot_name: Name of Uniconfig snapshot.
        transaction_id: Transaction ID generated by Uniconfig.
        uniconfig_server_id: Uniconfig server id is used by load balancer to forward request to the correct Uniconfig
         node. It is required when using multi node deployment of Uniconfig.
        uniconfig_url_base: Override default Uniconfig url.

    Returns:
        Http response.
    """
    base_url = UNICONFIG_URL_BASE
    if uniconfig_url_base is not None:
        base_url = uniconfig_url_base

    url = normalize_base_url(base_url) + CREATE_SNAPSHOT_URL
    cookies: DictAny = {'UNICONFIGTXID': transaction_id, 'uniconfig_server_id': uniconfig_server_id}
    response = requests.post(
        url,
        data=json.dumps(
            {
                'input': {
                    'name': snapshot_name,
                    'target-nodes': {
                        'node': node_ids
                    }
                }
            }
        ),
        cookies=cookies,
        headers=UNICONFIG_HEADERS
    )
    response.raise_for_status()
    return response


def delete_snapshot(
    snapshot_name: str,
    transaction_id: str,
    uniconfig_server_id: str | None = None,
    uniconfig_url_base: str | None = None
) -> requests.Response:
    """
    Delete Uniconfig snapshot.
    https://docs.frinx.io/frinx-uniconfig/user-guide/uniconfig-operations/snapshot-manager/rpc_delete-snapshot/
    Args:
        snapshot_name: Name of Uniconfig snapshot.
        transaction_id: Transaction ID generated by Uniconfig.
        uniconfig_server_id: Uniconfig server id is used by load balancer to forward request to the correct Uniconfig
         node. It is required when using multi node deployment of Uniconfig.
        uniconfig_url_base: Override default Uniconfig url.

    Returns:
        Http response.
    """
    base_url = UNICONFIG_URL_BASE
    if uniconfig_url_base is not None:
        base_url = uniconfig_url_base

    url = normalize_base_url(base_url) + DELETE_SNAPSHOT_URL
    cookies: DictAny = {'UNICONFIGTXID': transaction_id, 'uniconfig_server_id': uniconfig_server_id}
    response = requests.post(
        url,
        data=json.dumps(
            {
                'input': {
                    'name': snapshot_name
                }
            }
        ),
        cookies=cookies,
        headers=UNICONFIG_HEADERS
    )

    response.raise_for_status()
    return response


def replace_config_with_snapshot(
    snapshot_name: str,
    node_ids: list[str],
    transaction_id: str,
    uniconfig_server_id: str | None = None,
    uniconfig_url_base: str | None = None
) -> requests.Response:
    """
    Replace Uniconfig CONFIG datastore with a snapshot.
    https://docs.frinx.io/frinx-uniconfig/user-guide/uniconfig-operations/snapshot-manager/rpc_replace-config-with-snapshot/
    Args:
        snapshot_name: Name of Uniconfig snapshot.
        node_ids: Target nodes, which will replace nodes in CONFIG datastore.
        transaction_id: Transaction ID generated by Uniconfig.
        uniconfig_server_id: Uniconfig server id is used by load balancer to forward request to the correct Uniconfig
         node. It is required when using multi node deployment of Uniconfig.
        uniconfig_url_base: Override default Uniconfig url.

    Returns:
        Http response.
    """
    base_url = UNICONFIG_URL_BASE
    if uniconfig_url_base is not None:
        base_url = uniconfig_url_base

    url = normalize_base_url(base_url) + REPLACE_CONFIG_WITH_SNAPSHOT_URL
    cookies: DictAny = {'UNICONFIGTXID': transaction_id, 'uniconfig_server_id': uniconfig_server_id}
    response = requests.post(
        url,
        data=json.dumps(
            {
                'input': {
                    'name': snapshot_name,
                    'target-nodes': {
                        'node': node_ids
                    }
                }
            }
        ),
        cookies=cookies,
        headers=UNICONFIG_HEADERS
    )

    response.raise_for_status()
    return response
