import json

import requests

from frinx.common.frinx_rest import CLOSE_TRANSACTION_URL
from frinx.common.frinx_rest import COMMIT_TRANSACTION_URL
from frinx.common.frinx_rest import CREATE_TRANSACTION_URL
from frinx.common.frinx_rest import REPLACE_CONFIG_WITH_OPERATIONAL_URL
from frinx.common.frinx_rest import SYNC_FROM_NETWORK_URL
from frinx.common.frinx_rest import UNICONFIG_HEADERS
from frinx.common.frinx_rest import UNICONFIG_URL_BASE
from frinx.common.type_aliases import DictAny
from frinx.common.util import normalize_base_url


def create_transaction(
    transaction_timeout: int | None = None,
    use_dedicated_session: bool = False,
    uniconfig_url_base: str = UNICONFIG_URL_BASE
) -> requests.Response:
    """
    Create Uniconfig transaction.
    https://docs.frinx.io/frinx-uniconfig/user-guide/uniconfig-operations/build-and-commit-model/#creation-of-transaction
    Args:
        transaction_timeout: Transaction timeout in seconds. This parameter overrides global timeout for Uniconfig
         transactions.
        use_dedicated_session: Create dedicated session for this transaction.
        uniconfig_url_base: Uniconfig url.

    Returns:
        Http response.
    """
    params: DictAny = {}
    if transaction_timeout is not None:
        params['timeout'] = transaction_timeout

    if use_dedicated_session:
        params['dedicatedDeviceSession'] = 'true'

    url = normalize_base_url(uniconfig_url_base) + CREATE_TRANSACTION_URL
    response = requests.post(url=url, params=params, headers=UNICONFIG_HEADERS)
    response.raise_for_status()
    return response


def close_transaction(
    transaction_id: str,
    uniconfig_server_id: str | None = None,
    uniconfig_url_base: str = UNICONFIG_URL_BASE
) -> requests.Response:
    """
    Close Uniconfig transaction.
    https://docs.frinx.io/frinx-uniconfig/user-guide/uniconfig-operations/build-and-commit-model/#closing-transaction
    Args:
        transaction_id: Transaction ID generated by Uniconfig.
        uniconfig_server_id: Uniconfig server id is used by load balancer to forward request to the correct Uniconfig
         node. It is required when using multi node deployment of Uniconfig.
        uniconfig_url_base: Uniconfig url.

    Returns:
        Http response.
    """
    url = normalize_base_url(uniconfig_url_base) + CLOSE_TRANSACTION_URL
    cookies: DictAny = {'UNICONFIGTXID': transaction_id, 'uniconfig_server_id': uniconfig_server_id}
    response = requests.post(
        url=url,
        cookies=cookies,
        headers=UNICONFIG_HEADERS
    )
    response.raise_for_status()
    return response


def commit_transaction(
    transaction_id: str,
    uniconfig_server_id: str | None = None,
    confirmed_commit: bool = False,
    validate_commit: bool = True,
    uniconfig_url_base: str = UNICONFIG_URL_BASE
) -> requests.Response:
    """
    Commit Uniconfig transaction.
    https://docs.frinx.io/frinx-uniconfig/user-guide/uniconfig-operations/uniconfig-node-manager/rpc_commit/
    Args:
        transaction_id: Transaction ID generated by Uniconfig.
        uniconfig_server_id: Uniconfig server id is used by load balancer to forward request to the correct Uniconfig
         node. It is required when using multi node deployment of Uniconfig.
        confirmed_commit: Lock configuration of a node, so no other transaction can make changes on the node.
        validate_commit: Validate constraints and consistency of the transaction.
        uniconfig_url_base: Uniconfig url.

    Returns:
        Http response.
    """
    url = normalize_base_url(uniconfig_url_base) + COMMIT_TRANSACTION_URL
    # Input with target nodes will be deprecated in the future, Uniconfig is able to track modified nodes by itself
    data = json.dumps({
            'input': {
                'do-confirmed-commit': confirmed_commit,
                'do-validate': validate_commit,
                'target-nodes': {
                    'node': []
                }
            }
        })
    cookies: DictAny = {'UNICONFIGTXID': transaction_id, 'uniconfig_server_id': uniconfig_server_id}
    response = requests.post(
        url=url,
        data=data,
        cookies=cookies,
        headers=UNICONFIG_HEADERS)

    response.raise_for_status()
    return response


def replace_config_with_operational(
    node_ids: list[str],
    transaction_id: str,
    uniconfig_server_id: str | None = None,
    uniconfig_url_base: str = UNICONFIG_URL_BASE
) -> requests.Response:
    """
    Replace Uniconfig CONFIG datastore with OPER datastore.
    https://docs.frinx.io/frinx-uniconfig/user-guide/uniconfig-operations/uniconfig-node-manager/rpc_replace-config-with-oper/
    Args:
        node_ids: Target nodes, which will replace nodes in CONFIG datastore.
        transaction_id: Transaction ID generated by Uniconfig.
        uniconfig_server_id: Uniconfig server id is used by load balancer to forward request to the correct Uniconfig
         node. It is required when using multi node deployment of Uniconfig.
        uniconfig_url_base: Uniconfig url.

    Returns:
        Http response.
    """
    url = normalize_base_url(uniconfig_url_base) + REPLACE_CONFIG_WITH_OPERATIONAL_URL
    cookies: DictAny = {'UNICONFIGTXID': transaction_id, 'uniconfig_server_id': uniconfig_server_id}
    response = requests.post(
        url,
        data=json.dumps(
            {
                'input': {
                    'target-nodes': {
                        'node': node_ids
                    }
                }
            }
        ),
        cookies=cookies,
        headers=UNICONFIG_HEADERS
    )

    response.raise_for_status()
    return response


def sync_from_network(
    node_ids: list[str],
    transaction_id: str,
    uniconfig_server_id: str | None = None,
    uniconfig_url_base: str = UNICONFIG_URL_BASE
) -> requests.Response:
    """
    Synchronize configuration from network nodes to the UniConfig nodes
     in the Operational datastore of UniConfig transaction.
    https://docs.frinx.io/frinx-uniconfig/user-guide/uniconfig-operations/uniconfig-node-manager/rpc_sync-from-network/
    Args:
        node_ids: Target nodes, which will be synced from network.
        uniconfig_server_id: Uniconfig server id is used by load balancer to forward request to the correct Uniconfig
         node. It is required when using multi node deployment of Uniconfig.
        transaction_id: Transaction ID generated by Uniconfig.
        uniconfig_url_base: Override default Uniconfig url.

    Returns:
        Http response.
    """
    url = normalize_base_url(uniconfig_url_base) + SYNC_FROM_NETWORK_URL
    cookies: DictAny = {'UNICONFIGTXID': transaction_id, 'uniconfig_server_id': uniconfig_server_id}
    response = requests.post(
        url,
        data=json.dumps(
            {
                'input': {
                    'target-nodes': {
                        'node': node_ids
                    }
                }
            }
        ),
        cookies=cookies,
        headers=UNICONFIG_HEADERS
    )

    response.raise_for_status()
    return response
