# :coding: utf-8
# :copyright: Copyright (c) 2014 ftrack

from .base import Location

from ..ftrackerror import LocationError


class UnmanagedLocation(Location):
    '''Represent location for un-managed components.

    An un-managed component is one that is only tracked by the system
    without taking any ownership of the data or its structure. This is similar
    to the previous behaviour when handling data.

    May also prove useful for those who want to manage data themselves 
    and ensure that users of the API cannot accidentally manage the data
    through the system.

    '''

    def addComponent(self, component, recursive=True, manageData=False):
        '''Add *component* to this location.

        If *component* is a container and *recursive* is True then also
        add each member of the container to this location.

        .. note::

            *manageData* must be False as this type of location guarantees to
            never attempt management of data. Will raise 
            :py:exc:`ftrack.LocationError` if a non false value is passed in.

        .. note::

            Externally managed data must still match the path generated by this
            location's structure interface. No existence check is currently
            performed.

        Return component in this location.

        '''
        if manageData:
            raise LocationError(
                'UnmanagedLocation cannot manage data for components.'
            )

        return super(UnmanagedLocation, self).addComponent(
            component,
            recursive=recursive,
            manageData=manageData
        )

    def removeComponent(self, componentId, recursive=True, manageData=False):
        '''Remove component with *componentId* from this location.

        If *component* is a container and *recursive* is True then also
        remove each member of the container from this location.

        .. note::

            *manageData* must be False as this type of location guarantees to
            never attempt management of data. Will raise 
            :py:exc:`ftrack.LocationError` if a non false value is passed in.

        '''
        if manageData:
            raise LocationError(
                'UnmanagedLocation cannot manage data for components.'
            )

        return super(UnmanagedLocation, self).removeComponent(
            componentId,
            recursive=recursive,
            manageData=manageData
        )
