# 两阶段识别最佳实践指南

## 问题描述

在使用 FunASR 的两阶段识别模式 (`RecognitionMode.TWO_PASS`) 时,您可能会遇到**一阶段(在线)有识别结果,但二阶段(离线)结果为空**的情况。

## 根本原因

这是 FunASR 的**设计特性**,而非 bug:

### 工作原理

两阶段识别使用不同的队列处理音频:
- **在线队列** (`asr_online_queue`): 处理实时流式识别,追求**低延迟**
- **离线队列** (`asr_offline_queue`): 处理高精度识别,追求**高准确度**

### 触发条件

当出现以下情况时,二阶段结果会为空:

1. **VAD 未检测到语音结束点**
   - 语音正在进行中,VAD 只检测到开始 `[start, -1]`
   - 离线模式**等待完整的语音段** `[start, end]`

2. **音频流未正确结束**
   - 客户端没有发送 `is_speaking=false` 信号
   - 服务端认为语音还在继续

3. **音频块太短**
   - VAD 没有足够的数据判断语音边界
   - 静音检测未触发结束事件

## 解决方案

### ✅ 方案1: 确保发送结束信号 (推荐)

在每次语音识别完成后,确保正确结束 session:

```python
import asyncio
from funasr_client import AsyncFunASRClient, SimpleCallback

async def recognize_with_proper_ending():
    """正确的两阶段识别使用示例"""
    client = AsyncFunASRClient(
        mode="2pass",
        enable_vad=True,
    )
    
    try:
        await client.start()
        
        # 创建回调
        results = []
        
        def on_partial(result):
            print(f"实时结果: {result.text}")
        
        def on_final(result):
            print(f"优化结果: {result.text}")  # ✅ 这里会有结果
            results.append(result.text)
        
        callback = SimpleCallback(
            on_partial=on_partial,
            on_final=on_final,
        )
        
        # 识别音频文件
        result = await client.recognize_file(
            "audio.wav",
            callback=callback
        )
        
        # ✅ recognize_file 会自动发送结束信号
        print(f"最终结果: {result.text}")
        
    finally:
        await client.close()

asyncio.run(recognize_with_proper_ending())
```

### ✅ 方案2: 实时流式识别 - 正确的句子分割

对于麦克风或实时流,需要**明确标记每个句子的结束**:

```python
import asyncio
from funasr_client import AsyncFunASRClient, SimpleCallback
import numpy as np

async def streaming_with_sentence_detection():
    """带句子检测的流式识别"""
    client = AsyncFunASRClient(
        mode="2pass",
        enable_vad=True,
        chunk_interval=10,  # 10ms chunks
    )
    
    # 静音检测参数
    SILENCE_THRESHOLD = 0.01  # 音量阈值
    SILENCE_DURATION = 1.5    # 1.5秒静音视为句子结束
    
    silence_start = None
    current_sentence_active = False
    
    def on_partial(result):
        nonlocal current_sentence_active
        if result.text.strip():
            print(f"🔄 {result.text}")
            current_sentence_active = True
    
    def on_final(result):
        nonlocal current_sentence_active
        if result.text.strip():
            print(f"✅ {result.text}")  # ✅ 会有完整的高精度结果
            current_sentence_active = False
    
    callback = SimpleCallback(
        on_partial=on_partial,
        on_final=on_final,
    )
    
    try:
        await client.start()
        session = await client.start_realtime(callback)
        
        # 模拟音频流
        async def audio_stream_with_silence_detection():
            nonlocal silence_start, current_sentence_active
            
            while session.is_active:
                # 读取音频块 (这里用随机数模拟)
                audio_chunk = np.random.randn(1600).astype(np.int16).tobytes()
                
                # 计算音量
                audio_array = np.frombuffer(audio_chunk, dtype=np.int16)
                volume = np.abs(audio_array).mean() / 32768.0
                
                if volume < SILENCE_THRESHOLD:
                    # 检测到静音
                    if silence_start is None:
                        silence_start = asyncio.get_event_loop().time()
                    elif (asyncio.get_event_loop().time() - silence_start) > SILENCE_DURATION:
                        if current_sentence_active:
                            # ✅ 关键: 句子结束,停止并重启session
                            print("🔇 检测到句子结束")
                            await client.end_realtime_session(session)
                            
                            # 短暂等待,让服务端处理完成
                            await asyncio.sleep(0.3)
                            
                            # 重新开始新的session
                            session = await client.start_realtime(callback)
                            current_sentence_active = False
                            silence_start = None
                            print("🎤 准备下一句...")
                            continue
                else:
                    # 有声音,重置静音计时
                    silence_start = None
                
                # 发送音频
                await session.send_audio(audio_chunk)
                await asyncio.sleep(0.01)
        
        await audio_stream_with_silence_detection()
        
    finally:
        if session.is_active:
            await client.end_realtime_session(session)
        await client.close()

asyncio.run(streaming_with_sentence_detection())
```

### ✅ 方案3: 批量文件处理

处理多个文件时,确保每个文件都完整处理:

```python
import asyncio
from pathlib import Path
from funasr_client import AsyncFunASRClient

async def batch_process_files():
    """批量处理音频文件"""
    client = AsyncFunASRClient(
        mode="2pass",
        enable_vad=True,
    )
    
    try:
        await client.start()
        
        audio_files = Path("audio_dir").glob("*.wav")
        
        for audio_file in audio_files:
            print(f"\n处理: {audio_file.name}")
            
            # ✅ 每个文件独立处理,自动发送结束信号
            result = await client.recognize_file(audio_file)
            
            print(f"  一阶段: {result.text[:50]}...")
            print(f"  二阶段: {result.text}")  # ✅ 会有完整结果
            print(f"  置信度: {result.confidence:.2f}")
            
    finally:
        await client.close()

asyncio.run(batch_process_files())
```

### ✅ 方案4: 调整 VAD 参数

如果 VAD 太敏感或不敏感,可以调整服务端配置:

```yaml
# 服务端配置文件
vad:
  max_end_silence_time: 500    # 降低结束静音时长 (默认800ms)
  speech_noise_thres: 0.6      # 降低语音检测阈值 (默认0.8)
  max_single_segment_time: 10000  # 单段最大时长 (ms)
```

或通过客户端配置:

```python
client = AsyncFunASRClient(
    mode="2pass",
    enable_vad=True,
    # 可以通过 hotwords 参数影响识别
    hotwords={"专业术语": 20},  # 提高特定词的权重
)
```

## 最佳实践总结

### ✅ DO - 推荐做法

1. **使用 `recognize_file()` 处理完整音频文件**
   - 自动处理开始和结束信号
   - 保证获得完整的二阶段结果

2. **实时流式场景,按句子分割**
   - 检测句子结束后,调用 `end_realtime_session()`
   - 为新句子创建新的 session

3. **启用 VAD** (`enable_vad=True`)
   - 更准确的语音段检测
   - 减少无效音频处理

4. **合理的 chunk_interval**
   - 推荐 10ms (平衡性能和准确度)
   - 低延迟场景可用 5ms

5. **监控 callback 输出**
   ```python
   def on_partial(result):
       print(f"[在线] {result.text}")
   
   def on_final(result):
       print(f"[离线] {result.text}")  # 检查是否有输出
   ```

### ❌ DON'T - 避免做法

1. **不要无限发送音频而不结束 session**
   ```python
   # ❌ 错误示例
   while True:
       audio = get_audio()
       await session.send_audio(audio)
       # 永远不结束,二阶段永远不触发
   ```

2. **不要忽略静音检测**
   ```python
   # ❌ 错误示例
   while recording:
       await session.send_audio(audio_chunk)
       # 没有检测静音,句子边界不清晰
   ```

3. **不要使用过小的音频块**
   ```python
   # ❌ 错误示例
   chunk_interval=1  # 太小,VAD 难以判断
   ```

4. **不要在 session 结束前断开连接**
   ```python
   # ❌ 错误示例
   session = await client.start_realtime(callback)
   await client.send_audio(audio)
   await client.close()  # 直接关闭,没有正确结束
   ```

## 理解两阶段模式的设计

### 为什么会这样设计?

两阶段模式是为了平衡**实时性**和**准确性**:

| 阶段 | 目标 | 特点 | 使用场景 |
|------|------|------|----------|
| 一阶段(在线) | 低延迟 | 实时输出,即使语音未结束 | 实时字幕,语音交互 |
| 二阶段(离线) | 高精度 | 等待完整语音段,综合优化 | 最终结果,文档转写 |

### 何时触发二阶段?

二阶段触发需要以下条件之一:

1. **VAD 检测到完整语音段**: `[start_time, end_time]`
2. **客户端发送结束信号**: `is_speaking=false`
3. **Session 正常结束**: `end_realtime_session()`

## 故障排查

### 问题: 二阶段一直为空

**检查清单:**

1. ✅ 是否启用了 VAD?
   ```python
   client = AsyncFunASRClient(enable_vad=True)
   ```

2. ✅ 是否正确结束了 session?
   ```python
   await client.end_realtime_session(session)
   ```

3. ✅ 音频是否足够长?
   - 至少 0.5 秒的有效语音

4. ✅ 是否有足够的静音?
   - 至少 0.5 秒的结束静音

5. ✅ 检查服务端日志
   ```bash
   # 查看 VAD 输出
   grep "vad_segments" /path/to/server.log
   ```

### 问题: 识别延迟高

**解决方案:**

```python
# 使用低延迟配置
from funasr_client import ConfigPresets

client = AsyncFunASRClient(
    config=ConfigPresets.low_latency()
)
```

### 问题: 准确度不够

**解决方案:**

```python
# 使用高精度配置
from funasr_client import ConfigPresets

client = AsyncFunASRClient(
    config=ConfigPresets.high_accuracy()
)
```

## 高级用法

### 自定义句子结束检测

```python
class SmartSentenceDetector:
    """智能句子结束检测器"""
    
    def __init__(self, 
                 silence_threshold=0.01,
                 min_silence_duration=1.0,
                 max_silence_duration=3.0):
        self.silence_threshold = silence_threshold
        self.min_silence_duration = min_silence_duration
        self.max_silence_duration = max_silence_duration
        self.silence_start = None
        self.last_speech_time = None
    
    def detect(self, audio_chunk: bytes) -> bool:
        """检测是否应该结束当前句子"""
        import numpy as np
        
        audio_array = np.frombuffer(audio_chunk, dtype=np.int16)
        volume = np.abs(audio_array).mean() / 32768.0
        current_time = time.time()
        
        if volume < self.silence_threshold:
            # 静音
            if self.silence_start is None:
                self.silence_start = current_time
            
            silence_duration = current_time - self.silence_start
            
            # 超过最大静音时长,强制结束
            if silence_duration > self.max_silence_duration:
                self.reset()
                return True
            
            # 达到最小静音时长,且之前有语音
            if (silence_duration > self.min_silence_duration and 
                self.last_speech_time is not None):
                self.reset()
                return True
        else:
            # 有声音
            self.silence_start = None
            self.last_speech_time = current_time
        
        return False
    
    def reset(self):
        """重置检测器"""
        self.silence_start = None
        self.last_speech_time = None
```

### 使用示例

```python
async def advanced_streaming():
    """高级流式识别"""
    detector = SmartSentenceDetector(
        silence_threshold=0.01,
        min_silence_duration=1.5,
        max_silence_duration=4.0,
    )
    
    client = AsyncFunASRClient(mode="2pass", enable_vad=True)
    
    try:
        await client.start()
        session = await client.start_realtime(callback)
        
        while active:
            audio_chunk = await get_audio_chunk()
            
            # 发送音频
            await session.send_audio(audio_chunk)
            
            # 检测句子结束
            if detector.detect(audio_chunk):
                print("句子结束,等待最终结果...")
                await client.end_realtime_session(session)
                await asyncio.sleep(0.5)  # 等待处理
                
                # 开始新句子
                session = await client.start_realtime(callback)
                print("开始新句子...")
    
    finally:
        await client.close()
```

## 参考资料

- [FunASR 官方文档](https://github.com/alibaba-damo-academy/FunASR)
- [WebSocket 协议规范](../../../runtime/docs/websocket_protocol.md)
- [两阶段架构设计](../../../runtime/docs/funasr-wss-server-2pass-architecture.puml)

## 常见问题 FAQ

**Q: 为什么不直接在一阶段输出高精度结果?**

A: 一阶段需要实时响应,没有完整的上下文信息。二阶段等待完整语音段后,可以进行全局优化(语言模型、标点预测、ITN等)。

**Q: 可以禁用一阶段,只用二阶段吗?**

A: 可以,使用 `mode="offline"`,但会失去实时反馈能力。

**Q: VAD 和音频结束信号,哪个优先级更高?**

A: 音频结束信号优先级更高。即使 VAD 未检测到结束,发送 `is_speaking=false` 也会触发二阶段处理。

**Q: 二阶段结果一定比一阶段准确吗?**

A: 通常是的。二阶段使用更大的模型、语言模型、标点模型等,准确度通常提升 5-10%。
