"""Integration tests for FunASR client components."""

import asyncio
from unittest.mock import MagicMock, patch

import pytest

from funasr_client import (
    AsyncFunASRClient,
    FunASRClient,
    SimpleCallback,
    create_async_client,
)
from funasr_client.models import ClientConfig, FinalResult, PartialResult


class TestEndToEndIntegration:
    """Test end-to-end integration scenarios."""

    def test_complete_recognition_workflow(self):
        """Test complete recognition workflow from file to result."""
        # Load .env file for integration test
        try:
            from dotenv import load_dotenv
            load_dotenv()
        except ImportError:
            pass  # dotenv is optional

        # Use real audio file for integration test
        audio_file = "examples/audio/asr_example.wav"

        # Create client using .env configuration
        client = FunASRClient()        # Test file recognition
        result = client.recognize_file(audio_file)

        # Verify result structure
        assert isinstance(result, FinalResult)
        assert result.text is not None
        assert len(result.text.strip()) > 0
        assert result.is_final is True
        assert result.confidence >= 0.0
        assert result.session_id is not None

    async def test_async_complete_recognition_workflow(self):
        """Test async complete recognition workflow."""
        # Load .env file for integration test
        try:
            from dotenv import load_dotenv
            load_dotenv()
        except ImportError:
            pass  # dotenv is optional

        # Use real audio file for integration test
        audio_file = "examples/audio/asr_example.wav"

        # Create async client using .env configuration
        client = AsyncFunASRClient()
        await client.start()

        try:
            # Test file recognition
            result = await client.recognize_file(audio_file)

            # Verify result structure
            assert isinstance(result, FinalResult)
            assert result.text is not None
            assert len(result.text.strip()) > 0
            assert result.is_final is True
            assert result.confidence >= 0.0
            assert result.session_id is not None
        finally:
            await client.close()

    async def test_realtime_recognition_workflow(self):
        """Test real-time recognition workflow."""
        # Load .env file for integration test
        try:
            from dotenv import load_dotenv
            load_dotenv()
        except ImportError:
            pass  # dotenv is optional

        # Create async client using .env configuration
        client = AsyncFunASRClient()
        await client.start()

        try:
            # Create a callback for real-time results
            from funasr_client.callbacks import AsyncSimpleCallback

            partial_results = []
            final_results = []

            async def on_partial(result):
                partial_results.append(result)

            async def on_final(result):
                final_results.append(result)

            callback = AsyncSimpleCallback(on_partial=on_partial, on_final=on_final)

            # Start real-time session
            session = await client.start_realtime(callback)

            # Verify session was created
            assert session is not None
            assert session.session_id is not None

            # End session
            await client.end_realtime_session(session)
        finally:
            await client.close()


class TestCallbackIntegration:
    """Test integration with callback systems."""

    def test_callback_integration(self):
        """Test callback integration during recognition."""
        from dotenv import load_dotenv
        load_dotenv()

        # Use a simple callback
        callback = SimpleCallback()

        # Recognize with callback using real audio file
        client = FunASRClient()
        result = client.recognize_file("examples/audio/asr_example.wav", callback)

        # Verify basic functionality works
        assert isinstance(result, FinalResult)
        assert result.text is not None

    async def test_async_callback_integration(self):
        """Test async callback integration."""
        from dotenv import load_dotenv
        load_dotenv()

        # Setup callback to capture results
        partial_results = []
        final_results = []

        def on_partial(result: PartialResult):
            partial_results.append(result)

        def on_final(result: FinalResult):
            final_results.append(result)

        callback = SimpleCallback(
            on_partial=on_partial,
            on_final=on_final,
        )

        client = AsyncFunASRClient()
        await client.start()

        # Recognize with callback using real audio file
        result = await client.recognize_file("examples/audio/asr_example.wav", callback)

        # Verify result and callback
        assert isinstance(result, FinalResult)
        assert result.text is not None
        assert len(final_results) == 1

        await client.close()


class TestConfigurationIntegration:
    """Test integration with different configurations."""

    def test_preset_configuration_integration(self):
        """Test integration with configuration presets."""
        # Test low latency preset
        client = create_async_client(preset="low_latency")
        assert client.config.timeout == 10.0  # Low latency timeout
        assert client.config.chunk_interval == 5

        # Test high accuracy preset
        client = create_async_client(preset="high_accuracy")
        assert client.config.timeout == 60.0  # High accuracy timeout
        assert client.config.enable_itn is True

        # Test balanced preset
        client = create_async_client(preset="balanced")
        assert client.config.timeout == 30.0  # Balanced timeout
        assert client.config.chunk_interval == 8

    def test_custom_configuration_integration(self):
        """Test integration with custom configuration."""
        config = ClientConfig(
            server_url="ws://integration.test.com:8080",
            timeout=45.0,
            mode="online",
            enable_vad=True,
            hotwords={"test": 10, "integration": 15},
        )

        client = create_async_client(config=config)

        assert client.config.server_url == "ws://integration.test.com:8080"
        assert client.config.timeout == 45.0
        assert client.config.enable_vad is True
        assert client.config.hotwords == {"test": 10, "integration": 15}


class TestErrorHandlingIntegration:
    """Test error handling across components."""

    def test_connection_error_propagation(self):
        """Test error propagation from protocol to client."""
        from dotenv import load_dotenv
        load_dotenv()

        # Use a real client to test error handling with actual file
        client = FunASRClient()

        # Try to recognize a non-existent file to trigger an error
        with pytest.raises(Exception):  # Should raise some kind of error
            client.recognize_file("non_existent_file.wav")

    async def test_async_error_propagation(self):
        """Test async error propagation."""
        from dotenv import load_dotenv
        load_dotenv()

        client = AsyncFunASRClient()
        await client.start()

        # Try to recognize a non-existent file to trigger an error
        with pytest.raises(Exception):  # Should raise some kind of error
            await client.recognize_file("non_existent_file.wav")

        await client.close()

    def test_recovery_integration(self):
        """Test error recovery integration with real service."""
        from dotenv import load_dotenv
        load_dotenv()

        # Test simple recovery - using real client should work
        client = FunASRClient()
        result = client.recognize_file("examples/audio/asr_example.wav")

        # Verify basic functionality works
        assert isinstance(result, FinalResult)
        assert result.text is not None


class TestAudioProcessingIntegration:
    """Test integration with audio processing components."""

    def test_audio_processing_pipeline(self):
        """Test audio processing pipeline integration."""
        from dotenv import load_dotenv
        load_dotenv()

        client = FunASRClient()
        result = client.recognize_file("examples/audio/asr_example.wav")

        # Verify result
        assert isinstance(result, FinalResult)
        assert result.text is not None

    @patch("funasr_client.audio.AudioFileStreamer")
    def test_streaming_integration(self, mock_streamer_class):
        """Test streaming integration."""
        mock_streamer = MagicMock()

        # Mock streaming chunks
        test_chunks = [b"audio_chunk_1", b"audio_chunk_2", b"audio_chunk_3"]
        mock_streamer.stream_chunks.return_value = test_chunks
        mock_streamer.get_duration.return_value = 1.5
        mock_streamer_class.return_value = mock_streamer

        # Test that streamer is properly initialized and used
        from funasr_client.audio import AudioFileStreamer

        streamer = AudioFileStreamer("streaming_test.wav", chunk_size=1024)

        chunks = list(streamer.stream_chunks())
        duration = streamer.get_duration()

        assert len(chunks) == 3
        assert duration == 1.5


class TestConcurrencyIntegration:
    """Test concurrent operations integration."""

    async def test_concurrent_recognition(self):
        """Test concurrent recognition requests."""
        from dotenv import load_dotenv
        load_dotenv()

        client = AsyncFunASRClient()
        await client.start()

        # Run concurrent recognition tasks using same file
        tasks = [
            client.recognize_file("examples/audio/asr_example.wav"),
            client.recognize_file("examples/audio/asr_example.wav"),
        ]

        results = await asyncio.gather(*tasks)

        # Verify both results
        assert len(results) == 2
        assert isinstance(results[0], FinalResult)
        assert isinstance(results[1], FinalResult)
        assert results[0].text is not None
        assert results[1].text is not None

        await client.close()

    async def test_concurrent_realtime_sessions(self):
        """Test concurrent real-time sessions."""
        from dotenv import load_dotenv

        from funasr_client.callbacks import SimpleCallback
        load_dotenv()

        client = AsyncFunASRClient()
        await client.start()

        # Create simple callbacks for testing
        callback1 = SimpleCallback()
        callback2 = SimpleCallback()

        # Start multiple concurrent real-time sessions
        session1 = await client.start_realtime(callback1)
        session2 = await client.start_realtime(callback2)

        # Verify sessions have different IDs
        assert session1.session_id != session2.session_id

        # End both sessions
        await client.end_realtime_session(session1)
        await client.end_realtime_session(session2)

        await client.close()


if __name__ == "__main__":
    pytest.main([__file__])
