import datetime
import inspect
import logging
from functools import wraps
from typing import Any, Callable, Iterable, Mapping, Optional


def log_this(
    log_level: int = logging.INFO,
    file_path: Optional[str] = None,
    log_format: Optional[str] = None,
    param_attrs: Optional[Mapping[str, Callable[[Any], Any]]] = None,
    discard_params: Optional[Iterable[str]] = None,
):
    """Flexibly log every invocation of your application's functions.

    If you want to know what option(s) your users select from a certain dropdown, or
    maybe the length of a string they entered, this is function decorator for you.

    The invocations will be logged using the options you set, and can later be parsed
    and analyzed to understand how your users are using your application.

    Parameters
    ----------
    log_level
        Logging level (``logging.INFO`` by default).
    file_path
        Path to a log file. If *None*, output goes to *stderr*.
    log_format
        ``{}``-style format string for the emitted records.
    param_attrs
        Mapping whose *keys* are parameter names and whose *values* are
        callables that receive the parameter value and return **what should be
        logged** under *Attrs*.
    discard_params
        Iterable of parameter names whose *values* **must not appear in the
        Args/Kwargs sections** of the log line. These parameters are still
        eligible for inclusion in *Attrs* via *param_attrs*.

    Examples
    --------
    **Basic usage**::

        @log_this()
        def add(a, b):
            return a + b


        add(1, 2)
        # → Calling: __main__.add [2025-05-19T17:25:21.780733+00:00  2025-05-19T17:25:21.781115+00:00] Args: [10, 20] Kwargs: {} Attrs: {}

    **Redacting a secret token**::

        @log_this(discard_params={"api_token"})
        def fetch_data(url, api_token): ...

    **Summarising large inputs**::

        @log_this(param_attrs={"payload": len}, discard_params={"payload"})
        # It's generally good to discard parameter that are large, so you don't clutter
        # your logs with huge objects. You can still log their attributes like length,
        # though.
        def send(payload: bytes): ...

    See Also
    --------
    You can use advertools.logs_to_df() to parse, compress and analyze the logs
    generated by this decorator.
    """

    handler: logging.Handler
    if file_path:
        handler = logging.FileHandler(file_path)
    else:
        handler = logging.StreamHandler()

    if log_format:
        handler.setFormatter(logging.Formatter(log_format, style="{"))

    discard_set = set(discard_params or ())

    def decorator(func):
        logger_name = f"{func.__module__}.{func.__qualname__}"
        logger = logging.getLogger(logger_name)
        logger.setLevel(log_level)

        if handler not in logger.handlers:
            logger.addHandler(handler)

        sig = inspect.signature(func)

        @wraps(func)
        def wrapper(*args, **kwargs):
            bound = sig.bind_partial(*args, **kwargs)
            bound.apply_defaults()

            positional_param_names = list(sig.parameters)[: len(args)]
            args_repr = [
                value
                for name, value in zip(positional_param_names, args)
                if name not in discard_set
            ]
            kwargs_repr = {k: v for k, v in kwargs.items() if k not in discard_set}

            attrs_repr = {}
            if param_attrs:
                for name, transformer in param_attrs.items():
                    if name not in bound.arguments:
                        continue
                    try:
                        attrs_repr[name] = transformer(bound.arguments[name])
                    except Exception as exc:
                        attrs_repr[name] = f"<transform error: {exc}>"

            try:
                utc = datetime.UTC
            except AttributeError:
                # Fallback for Python versions < 3.11 which didn't have datetime.UTC
                utc = datetime.timezone.utc

            t0 = datetime.datetime.now(utc)
            result = func(*args, **kwargs)
            t1 = datetime.datetime.now(utc)

            logger.log(
                log_level,
                (
                    f"Calling: {logger_name} "
                    f"[{t0.isoformat()}  {t1.isoformat()}] "
                    f"Args: {args_repr} Kwargs: {kwargs_repr} Attrs: {attrs_repr}"
                ),
            )
            return result

        return wrapper

    return decorator
