"""
functioncache

functioncache is a decorator which saves the return value of functions even
after the interpreter dies. For example this is useful on functions that download
and parse webpages. All you need to do is specify how long
the return values should be cached (use seconds, like time.sleep).

USAGE:

    from functioncache import functioncache

    @functioncache(24 * 60 * 60)
    def time_consuming_function(args):
        # etc

    @functioncache(functioncache.YEAR)
    def another_function(args):
        # etc

USAGE FOR *INSTANCE AGNOSTIC* CLASS METHODS:

Normally, you wouldn't want to cache class method results, because
they may be affected by changes to the state of the object (self).

*If you know what you're doing* and are sure that a class method
would not be affected by the object's state (for example - object
is a proxy to a remote service), you'll need to make sure the first
argument (self) doesn't get cached.

In that case, you should use the ignore_instance=True keyword:

    class db:
        @functioncache(functioncache.HOUR,ignore_instance=True)
        def query(...):
           ...

NOTE: All arguments of the decorated function and the return value need to be
    picklable for this to work.

NOTE: The cache isn't automatically cleaned, it is only overwritten. If your
    function can receive many different arguments that rarely repeat, your
    cache may forever grow. One day I might add a feature that once in every
    100 calls scans the db for outdated stuff and erases.

Tested on python 2.7 and 3.1
(ignore_instance tested on python 2.7 and it rocks)

License: BSD, do what you wish with this. Could be awesome to hear if you found
it useful and/or you have suggestions. ubershmekel at gmail


A trick to invalidate a single value:

    @functioncache.functioncache
    def somefunc(x, y, z):
        return x * y * z

    del somefunc._db[functioncache._args_key(somefunc, (1,2,3), {})]
    # or just iterate of somefunc._db (it's a shelve, like a dict) to find the right key.


"""


from decorator import decorate
import collections as _collections
import datetime as _datetime
import inspect as _inspect
import os as _os
import cPickle as _pickle
import shelve as _shelve
import sys as _sys
import time as _time
import traceback as _traceback
import errno as _errno
import types as _types

_retval = _collections.namedtuple('_retval', 'timesig data')
_SRC_DIR = _os.path.dirname(_os.path.abspath(__file__))
_CACHE_ROOT = _os.path.expanduser("~/.functioncache")

SECOND = 1
MINUTE = 60 * SECOND
HOUR = 60 * MINUTE
DAY = 24 * HOUR
WEEK = 7 * DAY
MONTH = 30 * DAY
YEAR = 365 * DAY
FOREVER = None

OPEN_DBS = dict()


def _mkdir_p(path):
    try:
        _os.makedirs(path)
    except OSError as exc:  # Python >2.5
        if exc.errno == _errno.EEXIST and _os.path.isdir(path):
            pass
        else:
            raise


def _get_cache_name(function):
    """
    returns a name for the module's cache db.
    """
    module_name = _inspect.getfile(function)
    cache_name = module_name

    # fix for '<string>' or '<stdin>' in exec or interpreter usage.
    cache_name = cache_name.replace('<', '_lt_')
    cache_name = cache_name.replace('>', '_gt_')

    cache_name += '.cache'
    cache_name = _CACHE_ROOT + _os.path.abspath(cache_name)
    cache_dir = _os.path.dirname(cache_name)
    if not _os.path.exists(cache_dir):
        _mkdir_p(cache_dir)
    return cache_name


def _log_error(error_str):
    try:
        error_log_fname = _os.path.join(_SRC_DIR, 'functioncache.err.log')
        if _os.path.isfile(error_log_fname):
            fhand = open(error_log_fname, 'a')
        else:
            fhand = open(error_log_fname, 'w')
        fhand.write('[%s] %s\r\n' %
                    (_datetime.datetime.now().isoformat(), error_str))
        fhand.close()
    except Exception:
        pass


class PicklingError(TypeError):
    pass


def function_name(fn):
    return fn.__name__


def _args_key(function, args, kwargs, function_key=function_name):
    arguments = (args, kwargs)
    # Check if you have a valid, cached answer, and return it.
    # Sadly this is python version dependant
    try:
        if _sys.version_info[0] == 2:
            arguments_pickle = _pickle.dumps(arguments)
        else:
            # NOTE: protocol=0 so it's ascii, this is crucial for py3k
            #       because shelve only works with proper strings.
            #       Otherwise, we'd get an exception because
            #       function.__name__ is str but dumps returns bytes.
            arguments_pickle = _pickle.dumps(
                arguments, protocol=0).decode('ascii')
    except TypeError as e:
        raise PicklingError(str(e))

    key = function_key(function) + arguments_pickle
    return key


class ShelveBackend(object):

    """
    store cache data in a file based on the filename of the file which
    contains the function being cached.  One cache file can store multiple
    functions' data.
    """

    def setup(self, function):
        self.shelve = _shelve.open(_get_cache_name(function))

    def __contains__(self, key):
        return key in self.shelve

    def __getitem__(self, key):
        return self.shelve[key]

    def __setitem__(self, key, value):
        # NOTE: no need to _db.sync() because there was no mutation
        # NOTE: it's importatnt to do _db.sync() because otherwise the cache
        # doesn't survive Ctrl-Break!
        self.shelve[key] = value
        self.shelve.sync()

import hashlib


class FileBackend(object):

    """
    provide a backend to functioncache which stores each function argument
    combination in a different file.  This works around the problem caused
    by multiple python processes/threads trying to use the same file for
    caching.  This will still fail if the same function with the same arguments
    try to write simultaneously.  The ShelveBackend will fail often
    if any function in the same python file tries to write simultaneously.
    """

    def setup(self, function):
        self.dir_name = _get_cache_name(function) + 'd'
        _mkdir_p(self.dir_name)

    def __contains__(self, key):
        return _os.path.isfile(self._get_filename(key))

    def __getitem__(self, key):
        return _pickle.load(open(self._get_filename(key)))

    def __setitem__(self, key, value):
        # first-write wins semantics.  if someone else already cached this
        # value while we were off computing it, don't bother writing.  If we do
        # write, and the other process is still writing, we may corrupt the
        # file
        import portalocker

        try:
            file = open(self._get_filename(key), 'w')
            portalocker.Lock(file)
            try:
                _pickle.dump(value, file, _pickle.HIGHEST_PROTOCOL)
            except TypeError as e:
                raise PicklingError(str(e))
        except portalocker.LockException as e:
            # someone else had the lock, thats ok, we don't have to
            # write the value if someone else already is
            pass
        except Exception as e:
            # delete the file in the event of an exception during saving
            # to protect from corrupted files causing problems later
            try:
                _os.remove(self._get_filename(key))
            except OSError:
                pass

            raise

    def _get_filename(self, key):
        # hash the key and use as a filename
        return self.dir_name + '/' + hashlib.sha512(key).hexdigest()


class DictBackend(dict):

    """ cached values won't persist outside of this thread """

    def setup(self, function):
        pass


class MemcacheBackend(object):

    """ simple wrapper around memcache """

    def setup(self, function):
        pass

    def __init__(self, mc=None):
        if mc:
            self.mc = mc
        else:
            import memcache
            self.mc = memcache.Client(['127.0.0.1:11211'], debug=0)

    def __contains__(self, key):
        return self[key] != None

    def __getitem__(self, key):
        return self.mc.get(self._hash_key(key))

    def __setitem__(self, key, value):
        if not self.mc.set(self._hash_key(key), value):
            raise Exception("memcache set failed")

    def _hash_key(self, key):
        # the key needs hashed to remove the invalid characters from the
        # pickled data and avoid problems with key length
        return hashlib.sha512(key).hexdigest()


class S3Backend(object):

    """ wrapper around s3 - requires you to pass in an s3pool """

    def setup(self, function):
        self.data_set = _get_cache_name(function)

    def __init__(self, s3pool):
        self.s3pool = s3pool

    def __contains__(self, key):
        return bool(self.s3pool.list(self.data_set, key))

    def __getitem__(self, key):
        return self.s3pool.get_contents_as_string(self.data_set, key)

    def __setitem__(self, key, value):
        return self.s3pool.set_contents_from_string(self.data_set, key, value)


class SkipCache(Exception):

    """
    Sometimes something goes wrong (e.g. communication with a server times out), and your function wants to return a
    "fail gracefully" fallback value, but you don't want it cached. The SkipCache exception lets you do just that.
    instead of doing:
      return some_fallback_value
    you do:
      raise SkipCache,"Server timeout",some_fallback_value
    The caller of your function will get the value, but it won't get cached.
    The error text ("Server timeout") will appear in functioncache's log.
    """

    def __init__(self, message, retval=None):
        Exception.__init__(self, message)
        self.retval = retval


def function_with_cache(function, *args, **kwargs):
    try:
        key = _args_key(
            function, args[1:] if function._ignore_instance else args, kwargs,
            function_key=function._function_key
        )

        if key in function._db:
            rv = function._db[key]
            if function._seconds_of_validity is None or _time.time(
            ) - rv.timesig < function._seconds_of_validity:
                return rv.data
    except:
        # in any case of failure, don't let functioncache break the
        # program
        error_str = _traceback.format_exc()
        _log_error(error_str)
        if not function._fail_silently:
            raise

    try:
        retval = function(*args, **kwargs)
    # Log the error, return the value, don't cache it.
    except SkipCache as e:
        error_str = _traceback.format_exc()
        _log_error(error_str)
        return e.retval

    # store in cache
    try:
        function._db[key] = _retval(_time.time(), retval)
    except:
        # in any case of failure, don't let functioncache break the
        # program
        error_str = _traceback.format_exc()
        _log_error(error_str)
        if not function._fail_silently:
            raise

    return retval


def functioncache(
        seconds_of_validity=None,
        fail_silently=True,
        backend=ShelveBackend(),
        ignore_instance=False,
        function_key=function_name):
    '''
    functioncache is called and the decorator should be returned.

    function_key allows you to introduce cache invalidation on implementation changes
    '''
    # if a class is passed in, create an instance of that class as the backend
    if isinstance(backend, _types.ClassType) or isinstance(backend, type):
        backend = backend()

    def functioncache_decorator(function):
        # make sure cache is loaded
        if not hasattr(function, '_db'):
            cache_name = (_get_cache_name(function), type(backend))
            if cache_name in OPEN_DBS:
                function._db = OPEN_DBS[cache_name]
            else:
                backend.setup(function)
                function._db = backend
                OPEN_DBS[cache_name] = function._db

            function_with_cache._db = function._db

        function._seconds_of_validity = seconds_of_validity
        function._fail_silently = fail_silently
        function._ignore_instance = ignore_instance
        function._function_key = function_key

        return decorate(function, function_with_cache)

    if isinstance(seconds_of_validity, _types.FunctionType):
        # support for when people use '@functioncache.functioncache' instead of
        # '@functioncache.functioncache()'
        func = seconds_of_validity
        return functioncache_decorator(func)

    return functioncache_decorator


def dictcache(seconds_of_validity=None, fail_silently=False):
    return functioncache(seconds_of_validity, fail_silently, DictBackend())


def filecache(seconds_of_validity=None, fail_silently=False):
    return functioncache(seconds_of_validity, fail_silently, FileBackend())


def shelvecache(seconds_of_validity=None, fail_silently=False):
    return functioncache(seconds_of_validity, fail_silently, ShelveBackend())


def memcachecache(seconds_of_validity=None, fail_silently=False, mc=None):
    return functioncache(
        seconds_of_validity, fail_silently, MemcacheBackend(mc)
    )

if _os.path.exists(_os.path.expanduser("~/.disable_functioncache")):
    print 'disabled functioncache'

    def functioncache(*_, **__):
        def nop_decorator(function):
            return function
        return nop_decorator
