# from types import NoneType
import warnings

import torch
from abc import abstractmethod
from typing import Tuple, Union

from FunctionEncoder.Model.InnerProduct import INNER_PRODUCTS


class BaseDataset:
    """ Base class for all datasets. Follow this interface to interact with FunctionEncoder.py"""
    def __init__(self,
                 input_size:Tuple[int],
                 output_size:Tuple[int],
                 data_type:str,
                 n_examples:int=None,
                 n_queries:int=None,
                 dtype: torch.dtype = torch.float32,
                 device: str = "auto",
                 ):
        """ Constructor for BaseDataset
        
        Args:
        input_size (Tuple[int]): Dimensionality of the input to the function space. IE (5,) for a 5D input or (3, 32, 32) for a 3 channel 32x32 image.
        output_size (Tuple[int]): Dimensionality of the output of the function space. IE (5,) for a 5D output.
        data_type (str): Type of data generated by the dataset. One of ["deterministic", "stochastic", "categorical"]
        n_functions (int): Number of functions to sample from the dataset per gradient step.
        n_examples (int): Number of examples to sample from the dataset per function per gradient step. These points are used to compute the coefficients of the basis functions.
        n_queries (int): Number of queries to sample from the dataset per function per gradient step. These points are used to evaluate the function approximation.
        dtype (torch.dtype): Data type of the tensors generated by the dataset. Default is torch.float32.
        device (str): Device to generate the tensors on. Default is "auto" which will use "cuda" if available, else "cpu".

        total_n_functions (int): Deprecated. Number of functions to sample from the dataset per gradient step. No longer needed.
        total_n_samples_per_function (int): Deprecated. Total number of points for a function including examples and queries. No longer needed.
        n_functions_per_sample (int): Deprecated. Number of functions to sample from the dataset per gradient step. Use n_functions instead.
        n_examples_per_sample (int): Deprecated. Number of examples to sample from the dataset per function per gradient step. Use n_examples instead.
        n_points_per_sample (int): Deprecated. Number of queries to sample from the dataset per function per gradient step. Use n_queries instead.
        """
        assert len(input_size) >= 1, "input_size must be a tuple of at least one element"
        assert len(output_size) >= 1, "output_size must be a tuple of at least one element"
        assert 1 <= n_queries <= float("inf"), "n_queries must be a positive integer"
        assert 1 <= n_examples <= float("inf"), "n_examples must be a positive integer"
        assert data_type in INNER_PRODUCTS.keys(), f"data_type must be one of {list(INNER_PRODUCTS.keys())}, got {data_type}"
        self.input_size = input_size
        self.output_size = output_size
        self.n_examples = n_examples
        self.n_queries = n_queries
        self.data_type = data_type.lower()
        self.device = device
        if self.device == "auto":
            self.device = "cuda" if torch.cuda.is_available() else "cpu"
        self.dtype = dtype


    @abstractmethod
    def __getitem__(self, index: int) -> Tuple[torch.Tensor, torch.Tensor, torch.Tensor, torch.Tensor, dict]:
        """
        Retrieve a batch of data from the dataset.

        Returns:
            example_xs (torch.Tensor): (n_functions, n_examples, *input_size)
            example_ys (torch.Tensor): (n_functions, n_examples, *output_size)
            query_xs (torch.Tensor): (n_functions, n_queries, *input_size)
            query_ys (torch.Tensor): (n_functions, n_queries, *output_size)
            info (dict): Additional information
        """
        pass

    @abstractmethod
    def __len__(self) -> int:
        """Return the number of batches or functions available in the dataset."""
        pass


    def check_dataset(self):
        """ Verify that a dataset is correctly implemented. Throws error if violated. 
        I would advise against overriding this method, as it is used to verify that the dataset is implemented correctly.
        However, if your use case is very different, you may need to."""
        out = self.__getitem__(0)
        assert len(out) == 5, f"Expected 5 outputs, got {len(out)}"
        
        example_xs, example_ys, query_xs, query_ys, info = out
        assert type(example_xs) == torch.Tensor, f"Expected example_xs to be a torch.Tensor, got {type(example_xs)}"
        assert type(example_ys) == torch.Tensor, f"Expected example_ys to be a torch.Tensor, got {type(example_ys)}"
        assert type(query_xs) == torch.Tensor, f"Expected xs to be a torch.Tensor, got {type(query_ys)}"
        assert type(query_ys) == torch.Tensor, f"Expected ys to be a torch.Tensor, got {type(query_ys)}"
        assert example_xs.shape == (self.n_examples, *self.input_size), f"Expected example_xs shape to be {( self.n_examples, *self.input_size)}, got {example_xs.shape}"
        assert example_ys.shape == (self.n_examples, *self.output_size), f"Expected example_ys shape to be {(self.n_examples, *self.output_size)}, got {example_ys.shape}"
        assert query_xs.shape == (self.n_queries, *self.input_size), f"Expected xs shape to be {(self.n_queries, *self.input_size)}, got {query_xs.shape}"
        assert query_ys.shape == (self.n_queries, *self.output_size), f"Expected ys shape to be {(self.n_queries, *self.output_size)}, got {query_ys.shape}"
        assert type(info) == dict, f"Expected info to be a dict, got {type(info)}"
        assert example_xs.dtype == example_ys.dtype == query_xs.dtype == query_ys.dtype == self.dtype, f"Expected all tensors to have dtype {self.dtype}, got {example_xs.dtype}, {example_ys.dtype}, {query_xs.dtype}, {query_ys.dtype}"