# GeneratePDFs Python SDK

Python SDK for the [GeneratePDFs.com](https://generatepdfs.com) API, your go-to place for HTML to PDF.

Upload your HTML files, along with any CSS files and images to generate a PDF. Alternatively provide a URL to generate a PDF from it's contents.

## Installation

```bash
pip install generatepdfs-python-sdk
```

## Get your API Token

Sign up for an account on [GeneratePDFs.com](https://generatepdfs.com) and head to the API Tokens section and create a new token.

## Usage

### Basic Setup

```python
from generatepdfs import GeneratePDFs

client = GeneratePDFs.connect('YOUR_API_TOKEN')
```

### Generate PDF from HTML File

```python
from generatepdfs import GeneratePDFs

# Simple HTML file
pdf = client.generate_from_html('/path/to/file.html')

# HTML file with CSS
pdf = client.generate_from_html(
    '/path/to/file.html',
    '/path/to/file.css'
)

# HTML file with CSS and images
pdf = client.generate_from_html(
    '/path/to/file.html',
    '/path/to/file.css',
    [
        {
            'name': 'logo.png',
            'path': '/path/to/logo.png',
            'mime_type': 'image/png'  # Optional, will be auto-detected
        },
        {
            'name': 'photo.jpg',
            'path': '/path/to/photo.jpg'
        }
    ]
)
```

### Generate PDF from URL

```python
pdf = client.generate_from_url('https://example.com')
```

### Get PDF by ID

```python
# Retrieve a PDF by its ID
pdf = client.get_pdf(123)
```

### Working with PDF Objects

The SDK returns `Pdf` objects that provide easy access to PDF information and downloading:

```python
# Access PDF properties
pdf_id = pdf.get_id()
pdf_name = pdf.get_name()
status = pdf.get_status()
download_url = pdf.get_download_url()
created_at = pdf.get_created_at()

# Check if PDF is ready
if pdf.is_ready():
    # Download PDF content as bytes
    pdf_content = pdf.download()
    
    # Or save directly to file
    pdf.download_to_file('/path/to/save/output.pdf')
```

### Refresh PDF data from the API (useful for checking status updates)
```python
refreshed_pdf = pdf.refresh()
if refreshed_pdf.is_ready():
    pdf_content = refreshed_pdf.download()
```

### Client Methods

- `generate_from_html(html_path: str, css_path: Optional[str] = None, images: Optional[List[Dict[str, str]]] = None) -> Pdf` - Generate a PDF from HTML file(s)
- `generate_from_url(url: str) -> Pdf` - Generate a PDF from a URL
- `get_pdf(pdf_id: int) -> Pdf` - Retrieve a PDF by its ID
- `download_pdf(download_url: str) -> bytes` - Download PDF binary content from a download URL

### PDF Object Methods

- `get_id() -> int` - Get the PDF ID
- `get_name() -> str` - Get the PDF filename
- `get_status() -> str` - Get the current status (pending, processing, completed, failed)
- `get_download_url() -> str` - Get the download URL
- `get_created_at() -> datetime` - Get the creation date
- `is_ready() -> bool` - Check if the PDF is ready for download
- `download() -> bytes` - Download and return PDF binary content
- `download_to_file(file_path: str) -> bool` - Download and save PDF to a file
- `refresh() -> Pdf` - Refresh PDF data from the API and return a new Pdf instance with updated information

## Requirements

- Python 3.8 or higher
- requests library

## Development Setup

To set up the development environment:

1. Clone the repository:
```bash
git clone https://github.com/GeneratePDFs/python-sdk.git
cd python-sdk
```

2. Create a virtual environment (recommended):
```bash
python3 -m venv venv
source venv/bin/activate  # On Windows: venv\Scripts\activate
```

3. Install development dependencies:
```bash
pip install -r requirements-dev.txt
```

Or install the package in development mode:
```bash
pip install -e ".[dev]"
```

## Testing

To run the test suite, execute:

```bash
pytest
```

To run tests with coverage:

```bash
pytest --cov=generatepdfs --cov-report=html
```

To run tests in verbose mode:

```bash
pytest -v
```

## Contributing

Contributions and suggestions are **welcome** and will be fully **credited**.

We accept contributions via Pull Requests on [GitHub](https://github.com/GeneratePDFs/python-sdk).

### Pull Requests

- **Follow PEP 8** - Use a code formatter like `black` or `autopep8` to ensure code style consistency
- **Add tests!** - Your patch won't be accepted if it doesn't have tests.
- **Document any change in behaviour** - Make sure the README / CHANGELOG and any other relevant documentation are kept up-to-date.
- **Consider our release cycle** - We try to follow semver. Randomly breaking public APIs is not an option.
- **Create topic branches** - Don't ask us to pull from your master branch.
- **One pull request per feature** - If you want to do more than one thing, send multiple pull requests.
- **Send coherent history** - Make sure each individual commit in your pull request is meaningful. If you had to make multiple intermediate commits while developing, please squash them before submitting.

## Changelog

See [CHANGELOG.md](CHANGELOG.md) for a history of changes.

## License

This project is licensed under the MIT License. See the [LICENSE](LICENSE) file for details.
