# -*- coding: utf-8 -*-
import json
import requests
from gentelsdkcore import Configuration

# --- Request Models ---

class CheckLLMTextStreamRequest(object):
    def __init__(self, content="", host=None):
        self.content = content
        self.host = host

class CheckLLMTextTrunkRequest(object):
    def __init__(self, content="", host=None):
        self.content = content
        self.host = host

class CheckLLMPhotoRequest(object):
    def __init__(self, content="", upload_type="URL", host=None):
        self.content = content
        self.upload_type = upload_type
        self.host = host

# --- API Class ---

class SHDApi(object):
    def __init__(self):
        self.config = Configuration.get_default()
        if self.config is None:
            raise ValueError("Configuration not set. Please call gentelsdkcore.Configuration.set_default(config).")

    def _call_api(self, modal, payload, host_override=None):
        """Helper function to make API calls."""
        
        host = host_override if host_override else self.config.host
        url = f"{self.config.schema}://{host}/api/shield/v1/{modal}"
        
        headers = {
            'Shield-Api-Key': self.config.ak,
            'Content-Type': 'application/json',
            'User-Agent': self.config.user_agent
        }

        try:
            response = requests.post(
                url,
                headers=headers,
                json=payload,
                timeout=self.config.timeout
            )
            response.raise_for_status()  # Raises an exception for bad status codes (4xx or 5xx)
            return response.json()
        except requests.exceptions.RequestException as e:
            # You might want to add more sophisticated error handling or logging here
            raise Exception(f"API request failed: {e}") from e

    def check_llm_text_stream(self, request: CheckLLMTextStreamRequest):
        """
        API for streaming text content moderation.
        Note: The underlying API call is the same as trunk for now, based on the docs.
              This can be adapted if a true streaming endpoint is provided.
        """
        if not isinstance(request, CheckLLMTextStreamRequest):
            raise TypeError("request must be an instance of CheckLLMTextStreamRequest")

        payload = {
            "text": request.content,
            "role": "user" 
        }
        return self._call_api("text", payload, request.host)

    def check_llm_text_trunk(self, request: CheckLLMTextTrunkRequest):
        """
        API for non-streaming text content moderation.
        """
        if not isinstance(request, CheckLLMTextTrunkRequest):
            raise TypeError("request must be an instance of CheckLLMTextTrunkRequest")
        
        payload = {
            "text": request.content,
            "role": "user"
        }
        return self._call_api("text", payload, request.host)

    def check_llm_photo(self, request: CheckLLMPhotoRequest):
        """
        API for image content moderation.
        """
        if not isinstance(request, CheckLLMPhotoRequest):
            raise TypeError("request must be an instance of CheckLLMPhotoRequest")
            
        payload = {
            "image": request.content,
            "upload_type": request.upload_type, # URL or BASE64
            "role": "user"
        }
        return self._call_api("image", payload, request.host) 