"""Global Fishing Watch (GFW) API Python Client - Bulk Download API Base Response Models.

This module defines base Pydantic response models, parameters and enumerations for
various Bulk Download API endpoints.
"""

import datetime

from enum import Enum
from typing import Any, List, Optional, Union

from pydantic import Field, field_validator

from gfwapiclient.base.models import BaseModel
from gfwapiclient.http.models import ResultItem
from gfwapiclient.resources.bulk_downloads.base.models.request import BulkReportFormat


__all__ = ["BulkReportItem"]


class BulkReportStatus(str, Enum):
    """Bulk report current generation process status.

    For more details on the Bulk Download API supported report statuses, please refer
    to the official Global Fishing Watch API documentation:

    See: https://globalfishingwatch.org/our-apis/documentation#bulk-report-response

    See: https://globalfishingwatch.org/our-apis/documentation#bulk-reports-get-http-response

    Attributes:
        PENDING (str):
            Bulk report has been created but processing has not yet started.

        PROCESSING (str):
           Bulk report generation is currently in progress.

        DONE (str):
            Bulk report has been successfully generated and is ready for use.

        FAILED (str):
            Bulk report generation encountered an error or was unable to complete.
    """

    PENDING = "pending"
    PROCESSING = "processing"
    DONE = "done"
    FAILED = "failed"


class BulkReportGeography(BaseModel):
    """Bulk report geography.

    For more details on the Bulk Download API supported geographies, please refer to
    the official Global Fishing Watch API documentation:

    See: https://globalfishingwatch.org/our-apis/documentation#bulk-report-response

    See: https://globalfishingwatch.org/our-apis/documentation#bulk-reports-get-http-response

    Attributes:
        type (Optional[str]):
            Type of geometry input (e.g. `"dataset"` or `"custom"`).

        dataset (Optional[str]):
            Dataset associated with the bulk report region of interest, if using
            reference geometry (e.g. `"public-eez-areas"`).

        id (Optional[Union[str, int]]):
            Identifier (ID) of the geometry used to define the bulk report region of
            interest (e.g. `8466`).
    """

    type: Optional[str] = Field(None, alias="type")
    dataset: Optional[str] = Field(None, alias="dataset")
    id: Optional[Union[str, int]] = Field(None, alias="id")


class BulkReportItem(ResultItem):
    """Bulk report entry.

    Represents a single entry in the bulk report result. Each entry captures metadata
    and status of the previously created bulk report.

    For more details on the Bulk Download API supported response bodies, please refer
    to the official Global Fishing Watch API documentation:

    See: https://globalfishingwatch.org/our-apis/documentation#bulk-report-response

    See: https://globalfishingwatch.org/our-apis/documentation#bulk-reports-get-http-response

    Attributes:
        id (Optional[str]):
            Unique identifier (ID) of the bulk report (e.g.,
            `"adbb9b62-5c08-4142-82e0-b2b575f3e058"`).

        name (Optional[str]):
            Human-readable name of the bulk report (e.g.,
            `"sar-fixed-infrastructure-data-202409"`).

        file_path (Optional[str]):
            Name of the output file generated by the bulk report (e.g.,
            `"sar_fixed_infrastructure_data_202409.json"` or
            `"sar_fixed_infrastructure_data_202409.csv"`).

        format (Optional[BulkReportFormat]):
            Format of the generated bulk report file (e.g., `"JSON"` or `"CSV"`).

        filters (Optional[List[str]]):
            List of applied filters used when generating the bulk report
            (e.g., `["label = 'oil'"]`).

        geom (Optional[BulkReportGeography]):
            Geography used when generating the bulk report (e.g.,
            `{"type": "dataset", "dataset": "public-eez-areas", "id": 8466}`).

        status (Optional[BulkReportStatus]):
            Current status of the bulk report generation process (e.g., `"done"`).

        owner_id (Optional[Union[str, int]]):
            Identifier (ID) of the entity that created the bulk report (e.g., `509`).

        owner_type (Optional[str]):
            Type of entity that created the bulk report (e.g., `"user-application"`).

        created_at (Optional[datetime.datetime]):
            Timestamp when the bulk report was created in ISO-8601 format
            (e.g., `"2025-06-24T14:21:27.517Z"`).

        updated_at (Optional[datetime.datetime]):
            Timestamp when the bulk report was last updated in ISO-8601 format
            (e.g., `"2025-06-24T14:21:27.517Z"`).

        file_size (Optional[float]):
            Size of the bulk report output file in bytes (e.g., `1207`).
    """

    id: Optional[str] = Field(None, alias="id")
    name: Optional[str] = Field(None, alias="name")
    file_path: Optional[str] = Field(None, alias="filepath")
    format: Optional[BulkReportFormat] = Field(None, alias="format")
    filters: Optional[List[str]] = Field(None, alias="filters")
    geom: Optional[BulkReportGeography] = Field(None, alias="geom")
    status: Optional[BulkReportStatus] = Field(None, alias="status")
    owner_id: Optional[Union[str, int]] = Field(None, alias="ownerId")
    owner_type: Optional[str] = Field(None, alias="ownerType")
    created_at: Optional[datetime.datetime] = Field(None, alias="createdAt")
    updated_at: Optional[datetime.datetime] = Field(None, alias="updatedAt")
    file_size: Optional[float] = Field(None, alias="fileSize")

    @field_validator(
        "created_at",
        "updated_at",
        mode="before",
    )
    @classmethod
    def empty_datetime_str_to_none(cls, value: Any) -> Optional[Any]:
        """Convert any empty datetime string to `None`.

        Args:
            value (Any):
                The value to validate.

        Returns:
            Optional[Any]:
                The validated datetime object or `None` if input is empty.
        """
        if isinstance(value, str) and value.strip() == "":
            return None
        return value
